;*******************************************************************************
; File: plotastro.pro
;
; Description:
; ------------
; Container of all IDL astro plotting scripts related to plotting in OYSTER. 
;
; Block directory:
; ----------------
; Block 1: plotjacoby,ww_loadjacoby,ww_jacobylines,ww_jacobycommand,
;	   ww_plotdevice,
; Block 2: plottlusty,ww_tlustyteff,ww_tlustylogg,ww_tlustyvsini,ww_tlustydm,
;	   ww_loadtlusty,ww_tlustycommand,tlusty_prepare,
; Block 3: plotpollux,pollux_obs_event,pollux_t_event,pollux_g_event,
;	   pollux_models_event,pollux_models,ww_polluxcommand,pollux_prepare
; Block 4: plotbinary,plotbinary_event
; Block 5: plotiso,plotiso_event,plot_iso,ww_isogetz,ww_isoutilities,
;	   ww_isogetdata,makeisotables,
;
;
;************************************************************************Block 1
pro plotjacoby
;
; Plot visual spectra of stars selected from the Library of Stellar Spectra.
; (JACOBY, HUNTER AND CHRISTIAN 1984; National Space Science Data Center)
;
common PlotJacoby,jacoby_wid,stars,fluxes,lambda
;
window,0,xsize=1024,ysize=500
fluxes=0
lambda=0
;
headers=''
status=dc_read_fixed(!atmospheres_dir+'jacoby/headers.dat',headers,resize=[1], $
	format='(a160)',/col)
stars=strmid(headers,0,17)
jacoby_wid=widget_base(title=' Jacoby ',resource_name='oyster', $
	/column)
row_wid=widget_base(jacoby_wid,/row)
list_wid=widget_list(row_wid,value=stars, $
	event_pro='ww_loadjacoby',ysize=12)
buttons=['He II','He I','H','Sun','Ca I','Ca II','Na I','Fe I']
button1_wid=cw_bgroup(row_wid,buttons,event_funct='ww_jacobylines', $
	/column,/return_name)
buttons=['CH','CN','Mg I','Mg II','Ti II','Sr II','TiO','O_2']
button2_wid=cw_bgroup(row_wid,buttons,event_funct='ww_jacobylines', $
	/column,/return_name)
devices=['Screen','File','PostScript']
device_wid=widget_droplist(jacoby_wid,title='Device:', $
	value=devices,uvalue=devices,event_pro='ww_plotdevice')
combox_wid=cw_bgroup(jacoby_wid,['Range','Quit'],event_funct='ww_jacobycommand', $
	/row,/return_name)
widget_control,jacoby_wid,/realize
xmanager,'plot_jacoby',jacoby_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_loadjacoby,event
;
common PlotJacoby,jacoby_wid,stars,fluxes,lambda
;
num_fluxes=2799
;
star=stars(event.index)
index=where(stars eq star(0),count)
if count eq 1 then begin
	fluxes=fltarr(num_fluxes)
	status=linknload(!external_lib,'jacoby',index,fluxes)
	lambda=findgen(num_fluxes)*1.4+3510.0
        fluxes=fluxes/avg(fluxes)
;	fluxes=fluxes*1.0e12
	if !d.name eq 'NULL' then begin
		openw,unit,'fluxes.dat',/get_lun
		for i=0,num_fluxes-1 do printf,unit,lambda(i),fluxes(i)
		free_lun,unit
	endif else begin
	plot,lambda,fluxes,charsize=1.5,xstyle=1,xticklen=0.05, $
		title=star(0),xtitle='lambda [Angstroems]',ytitle='Flux', $
		psym=0,linestyle=0
		f_u=total(johnson_u(lambda/10)*fluxes) $
		   /total(johnson_u(lambda/10))
		f_b=total(johnson_b(lambda/10)*fluxes) $
		   /total(johnson_b(lambda/10))
		f_v=total(johnson_v(lambda/10)*fluxes) $
		   /total(johnson_v(lambda/10))
		f_r=total(johnson_r(lambda/10)*fluxes) $
		   /total(johnson_r(lambda/10))
		f_i=total(johnson_i(lambda/10)*fluxes) $
		   /total(johnson_i(lambda/10))
; 		m_u=-2.5*alog10(f_u)-magfilter('U') ; not far enough to U band
		m_b=-2.5*alog10(f_b)-magfilter('B')
		m_v=-2.5*alog10(f_v)-magfilter('V')
		m_r=-2.5*alog10(f_r)-magfilter('R')
		m_i=-2.5*alog10(f_i)-magfilter('I')
; 		print,'(U-B) = ',m_u-m_b
		print,'(B-V) = ',m_b-m_v
		print,'(R-I) = ',m_r-m_i
;
;	Overplot stellar atmosphere (testing purposes)
	w=nameparse(star)
	if n_elements(w) eq 3 and strlen(w(0)) gt 5 then sptype=w(1)+w(2)
	if n_elements(w) eq 3 and strlen(w(0)) le 5 then sptype=w(2)
	if n_elements(w) eq 4 then sptype=w(2)+w(3)
	r=limbgrid(teff_star(sptype),logg_star(sptype),l,u,f) & l=l*10
	y=spline(l,f,lambda)
	i=where(l ge min(lambda) and l le max(lambda))
	r=total(fluxes)/total(y)
	oplot,lambda,y*r,color=tci(3)
	endelse
endif
;
end
;-------------------------------------------------------------------------------
function ww_jacobylines,event
;
common PlotJacoby,jacoby_wid,stars,fluxes,lambda
;
max_flux=max(fluxes)
;
case event.value of
	'H':	lines=[6563,4861,4340,4102,3970,3889,3835,3798,3771]
	'Sun':	lines=[3968,3934,4102,4227,4300,4384,4861,5167,5173,5184, $
		       5270,5890,5896,6563]
	'Mg I':	lines=[5167,5173,5184]
	'Mg II':lines=[4481]
	'Na I':	lines=[5890,5896]
	'Fe I':	lines=[5270,4405,4384,4325,4045]
	'Ca I':	lines=[4227,4425,4435,4455]
	'Ca II':lines=[3968,3934]
	'CH':	lines=[4300]
	'CN':	lines=[3888,4215]
	'He I':	lines=[3819,4009,4026,4144,4387,4471,6678]
	'He II':lines=[4100,4200,4340,4542,4686]
	'Ti II':lines=[4444]
	'Sr II':lines=[4077]
	'Cr I': lines=[4254,4275]
	'TiO':	lines=[4584,4761,4954,5167,5448,5595,6159,6651,7054]
	'O_2':	lines=[6867]
		else:	begin
			print,'***Error(WW_LINES): invalid choice!'
			return,-1
			end
endcase
for i=0,n_elements(lines)-1 do begin
	dlambda=abs(lambda-lines(i))
	index=where(dlambda eq min(dlambda))
	max_y=fluxes(index(0))*0.9
	oplot,[lines(i),lines(i)],[0,max_y],psym=0,linestyle=0
endfor
;
end
;-------------------------------------------------------------------------------
pro ww_plotdevice,event
;
widget_control,event.id,get_uvalue=devices
case devices(event.index) of
	'Screen':set_plot,!display
	'File': set_plot,'NULL'
	'PostScript': begin
		set_plot,'ps'
		device,/landscape
		end
	else: print,'***Error(WW_PLOTDEVICES): invalid option'
endcase
;
end
;-------------------------------------------------------------------------------
function ww_jacobycommand,event
;
common PlotJacoby,jacoby_wid,stars,fluxes,lambda
;
case event.value of
'Range'   :begin
	   icom=set_boxes(box_x,box_y)
	   if icom eq 4 then begin
                icom=set_boxes(box_x,box_y,clear)
  		!x.range=0
		!y.range=0
	   endif else begin
		!x.range=[box_x(0),box_x(1)]
		!y.range=[box_y(3),box_y(4)]
	   endelse
	   end
'Quit':    widget_control,jacoby_wid,/destroy
'Hardcopy':begin
           end
else:	   print,'***Error(WW_COMMAND): option not implemented!'
endcase
;
end
;************************************************************************Block 2
pro plottlusty
;
; Plot visual spectra downloaded from TLUSTY homepage.
;
common PlotTlusty,files_wid,files,teffs,loggs,teff,logg,vsini,deltam
;
; window_slide,xsize=8000,xv=1000
;
; Get filenames
files=specname(findfile(!atmospheres_dir+'tlusty/*.vis.7.gz'))
n=n_elements(files)
teffs=strarr(n)
loggs=strarr(n)
for i=0,n-1 do begin
	words=nameparse(files(i),['B','G','g','v'])
	teffs(i)=words(1)
	loggs(i)=words(2)
endfor
teffs=unique(teffs)
loggs=unique(loggs)
nt=n_elements(teffs)
nl=n_elements(loggs)
values=strarr(nt,nl)
for i=0,nt-1 do begin
for j=0,nl-1 do begin
index=where(strpos(files,teffs(i)) eq 1 and strpos(files,loggs(j)) eq 7,count)
if count gt 0 then values(i,j)='X'
endfor
endfor
teff=teffs(0)
logg=loggs(0)
tlabels=string(float(teffs)/1000,format='(f4.1)')
;
tlusty_wid=widget_base(title=' TLUSTY ',resource_name='oyster', $
	/column)
if 0 then begin
row_wid=widget_base(tlusty_wid,/row)
list_wid=widget_list(row_wid,value=teffs, $
	event_pro='ww_tlustyteff',ysize=12)
list_wid=widget_list(row_wid,value=loggs, $
	event_pro='ww_tlustylogg',ysize=12)
files_wid=widget_list(row_wid,value=files, $
	event_pro='ww_loadtlusty',ysize=12)
endif
draw_wid=widget_draw(tlusty_wid, $
	scr_xsize=1800,scr_ysize=200)
table_wid=widget_table(tlusty_wid,value=values, $
	/all_events,event_pro='ww_loadtlusty', $
	row_labels=loggs,column_labels=teffs)
row_wid=widget_base(tlusty_wid,/row)
combox_wid=cw_bgroup(row_wid,/row,/return_name, $
	['H-Line','V-Line','Window', $
	 'CII 4267','OII 4414','HeII 4542','OII 4943','CIII 5696','CIV 5812'], $
	event_funct='ww_tlustycommand')
vsinis=string(indgen(6)*20,format='(i3)')
vsini=0
opmenuv_wid=widget_droplist(row_wid,title='V*sin(i) [km/s]:', $
		event_pro='ww_tlustyvsini',value=vsinis,uvalue=vsinis)
deltams=string((findgen(9)-4)*0.5,format='(f4.1)')
deltam=deltams(0)
opmenum_wid=widget_droplist(row_wid,title='Delta-m:', $
		event_pro='ww_tlustydm',value=deltams,uvalue=deltams)
widget_control,tlusty_wid,/realize
xmanager,'plot_tlusty',tlusty_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_tlustyteff,event
;
common PlotTlusty,files_wid,files,teffs,loggs,teff,logg,vsini,deltam
;
teff=teffs(event.index)
;
files=specname(findfile(!atmospheres_dir+'tlusty/*.vis.7.gz'))
index=where(strpos(files,teff) eq 1 and strpos(files,logg) eq 7,count)
if count gt 0 then files=files(index) else files=''
widget_control,files_wid,set_value=files
;
end
;-------------------------------------------------------------------------------
pro ww_tlustylogg,event
;
common PlotTlusty,files_wid,files,teffs,loggs,teff,logg,vsini,deltam
;
logg=loggs(event.index)
;
files=specname(findfile(!atmospheres_dir+'tlusty/*.vis.7.gz'))
index=where(strpos(files,teff) eq 1 and strpos(files,logg) eq 7,count)
if count gt 0 then files=files(index) else files=''
widget_control,files_wid,set_value=files
;
end
;-------------------------------------------------------------------------------
pro ww_tlustyvsini,event
;
common PlotTlusty,files_wid,files,teffs,loggs,teff,logg,vsini,deltam
;
widget_control,event.id,get_uvalue=vsinis
vsini=vsinis(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_tlustydm,event
;
common PlotTlusty,files_wid,files,teffs,loggs,teff,logg,vsini,deltam
;
widget_control,event.id,get_uvalue=deltams
deltam=deltams(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_loadtlusty,event
;
common PlotTlusty,files_wid,files,teffs,loggs,teff,logg,vsini,deltam
;
forward_function lsf_rotate
;
if event.sel_left lt 0 then return
;
logg=loggs(event.sel_top)
teff=teffs(event.sel_left)
files=findfile(!atmospheres_dir+'tlusty/*.vis.7.gz')
index=where(strpos(specname(files),teff) eq 1 $
	and strpos(specname(files),logg) eq 7,count)
if count eq 1 then file=files(index(0)) else return
xdr_file=findfile(strmid(file,0,strpos(file,'.gz'))+'.xdr')
if strlen(xdr_file(0)) gt 0 then restore,xdr_file(0) else $
			status=dc_read_free(file,lambda,flux,/col)
r=poly_fit(lambda,flux,2,yfit)
x=lambda
y=flux/yfit
if float(vsini) gt 0 and (!x.range(1)-!x.range(0)) le 200 then begin
	index=where(x gt !x.range(0) and x lt !x.range(1))
	velgrid=3e5*(x(index)/(total(!x.range)/2)-1)
	deltav=median(velgrid-shift(velgrid,1))
	kernel=lsf_rotate(deltav,float(vsini),velgrid=vel)
;	wgrid=(total(!x.range)/2)*(1+vel/3e5)
	y=1-convol(1-y,kernel)/n_elements(kernel)
endif
f1=1.0
f2=10^(-deltam/2.5)
plot,x,(f1+f2*y)/(f1+f2),psym=0,title=file
;
return ; The code following is obsolete
;
file=files(event.index)
xdr_file=findfile(file+'.xdr')
if strlen(xdr_file(0)) gt 0 then restore, xdr_file(0) else $
			status=dc_read_free(file,lambda,flux,/col)
r=poly_fit(lambda,flux,2,yfit)
plot,lambda,flux/yfit,psym=0,title=file
;
end
;-------------------------------------------------------------------------------
function ww_tlustycommand,event
;
common PlotTlusty,files_wid,files,teffs,loggs,teff,logg,vsini,deltam
;
case event.value of
'Window':begin
	icom=set_boxes(box_x,box_y)
	if icom eq 4 then begin
                icom=set_boxes(box_x,box_y,clear)
  		!x.range=0
		!y.range=0
	endif else begin
		!x.range=[box_x(0),box_x(1)]
		!y.range=[box_y(3),box_y(4)]
	endelse
	end
'H-Line'  :begin
	   if !d.window eq -1 then begin
		print,'***Error(WW_PLOTUTIL): no plot window open!'
		return,-1
	   endif
	   print,'___________________________________________'
	   print,'Click left mouse button on plot window.'
	   print,'Click middle or right mouse button to exit!'
	   print,'__________________***______________________'
	   cursor,x,y,/down & icom=!err
	   device,set_graphics_function=6
	   x1=!x.crange(0) & x2=!x.crange(1)
	   xrange=x2-x1
	   x1=x1-0.1*xrange & x2=x2+0.1*xrange
	   oplot,[x1,x2],[y,y],psym=0
	   repeat begin
		cursor,x,yn,/change & icom=!err
		oplot,[x1,x2],[y,y],psym=0
		oplot,[x1,x2],[yn,yn],psym=0
		print,format='(%"Value: %s\r",$)', $
			string(yn,format='(f7.3)')
		y=yn
	   endrep until (icom eq 4) or (icom eq 2)
	   if icom eq 4 then oplot,[x1,x2],[y,y],psym=0
	   device,set_graphics_function=3
	   end
'V-Line':begin
        if !d.window eq -1 then begin
             print,'***Error(WW_PLOTUTIL): no plot window open!'
             return,-1
        endif
        print,'___________________________________________'
        print,'Click left mouse button on plot window.'
        print,'Click middle or right mouse button to exit!'
        print,'__________________***______________________'
        cursor,x,y,/down & icom=!err
        device,set_graphics_function=6
        y1=!y.crange(0) & y2=!y.crange(1)
        yrange=y2-y1
        y1=y1-0.1*yrange & y2=y2+0.1*yrange
        oplot,[x,x],[y1,y2],psym=0
        repeat begin
             cursor,xn,y,/change & icom=!err
             oplot,[x,x],[y1,y2],psym=0
             oplot,[xn,xn],[y1,y2],psym=0
;		print,format='(%"Wavelength: %d\r",$)',fix(xn)
		print,format='(%"Wavelength: %s\r",$)', $
			string(xn,format='(f7.2)')
             x=xn
        endrep until (icom eq 4) or (icom eq 2)
        if icom eq 4 then oplot,[x,x],[y1,y2],psym=0
        device,set_graphics_function=3
	print,''
	end
'Hardcopy':begin
        end
else:	begin
	words=nameparse(event.value)
	wl=float(words(1))
	!x.range=[wl-50,wl+50]
	!y.range=[0.95,1.005]
	end
endcase
;
end
;-------------------------------------------------------------------------------
pro tlusty_prepare
;
files=specname(findfile(!atmospheres_dir+'tlusty/*.vis.7'))
for i=0,n_elements(files)-1 do begin
	print,'Reading file: ',files(i)
	status=dc_read_free(files(i),lambda,flux,/col)
	index=where(lambda gt 4000 and lambda lt 6000)
	lambda=lambda(index)
	flux=flux(index)
	save,lambda,flux,filename=files(i)+'.xdr'
endfor
;
end
;************************************************************************Block 3
pro plotpollux
;
; Plot OPTIM3D-STAGGER (Chiavassa+ 2018) NIR spectra from POLLUX homepage.
; Paramaters are: Teff, g, [Fe/H], and [alpha/Fe].
; The spectra have R=20000, delta-lambda=0.0949 @ 1900 nm & 0.125 @ 2500 nm.
; GRAVITY spectra sampling: delta-lambda=0.246 nm.
;
forward_function pollux_models
;
common PlotPolluxObs,fits_file,wave,spec,error,qual,header,h_ext1,h_ext2
common PlotPolluxData,files,regions,r_label,t_grid,g_grid,t_eff,log_g
common PlotPolluxWids,opmenu_t_wid,opmenu_g_wid,m_list_wid
;
wave=0
spec=0
!x.range=[1900,2500]
!y.range=[0,1.2]
;
; STAGGER grid based on 31 regions
regions=strarr(56)
r_label=strarr(56)
t_grid=7000-findgen(7)*500
g_grid=5-findgen(8)*0.5
;
t_grid_str=string(t_grid,format='(i4)')
g_grid_str=string(g_grid,format='(f3.1)')
;
t_tol=240
l_tol=0.25
;
; Get filenames and parameter values
files=specname(findfile(!atmospheres_dir+'pollux/*.xdr'))
n=n_elements(files)
teffs_str=strmid(files,1,4) & teffs=float(teffs_str)
loggs_str=strmid(files,6,4) & loggs=float(loggs_str)
feovh_str=strmid(files,11,5) & feovh=float(feovh_str)
alpha_str=strmid(files,17,4) & alpha=float(alpha_str)
;
; Obtain indices of models for each region
tcount=0
for i=0,6 do begin
for j=0,7 do begin
	index=where(teffs lt (t_grid(i)+t_tol) and teffs gt (t_grid(i)-t_tol) $
		and loggs lt (g_grid(j)+l_tol) and loggs gt (g_grid(j)-l_tol),$
		count)
	tcount=tcount+count
	if count gt 0 then begin
		k=strarr(count)+'K '
		regions(i*6+j)=strjoin(string(index))
		r_label(i*6+j)=t_grid_str(i)+' '+g_grid_str(j)
	endif
endfor
endfor
;
; tcount must be equal the number of models
if tcount ne n then begin
	print,'Error: total models in regions not the same as #models!'
	return
endif
index=where(strlen(regions) gt 0)
regions=regions(index)
r_label=r_label(index)
;
pollux_wid=widget_base(/column,title='POLLUX spectra',resource_name='oyster')
draw_wid=widget_draw(pollux_wid,scr_xsize=1300,scr_ysize=400)
row_wid=widget_base(pollux_wid,/row)
;
t_grid=strtrim(string(fix(t_grid)),1)
g_grid=strtrim(string(g_grid,format='(f3.1)'),1)
opmenu_t_wid=widget_droplist(row_wid,title='T_eff', $
	event_func='pollux_t_event', $
	value=t_grid)
t_eff=t_grid(0)
;
index=where(strmid(r_label,0,4) eq t_eff)
r_logg=strmid(r_label(index),5,3)
opmenu_g_wid=widget_droplist(row_wid,title='log(g)', $
	event_func='pollux_g_event', $
	value=r_logg)
log_g=r_logg(0)
;
models=pollux_models()
m_list_wid=widget_list(row_wid,value=models,uvalue=models, $
	event_func='pollux_models_event',ysize=5)
;
fits_files=findfile('*.fits')
fits_files=['None',fits_files]
fits_files=fits_files(where(strlen(fits_files) gt 0))
opmenu_d_wid=widget_droplist(row_wid,title='Observed data:', $
	event_func='pollux_obs_event', $
	uvalue=fits_files,value=fits_files)
;
; row_wid=widget_base(pollux_wid,/row)
combox_wid=cw_bgroup(row_wid,/row,/return_name, $
	['H-Line','V-Line','Window'], $
	event_funct='ww_polluxcommand')
;
widget_control,pollux_wid,/realize
xmanager,'pollux',pollux_wid,/no_block
end
;-------------------------------------------------------------------------------
function pollux_obs_event,event
;
common PlotPolluxObs,fits_file,wave,spec,error,qual,header,h_ext1,h_ext2
common PlotPolluxExt,e,q,header_ext1,header_ext2
;
widget_control,event.id,get_uvalue=fits_files
fits_file=fits_files(event.index)
;
wave=0
spec=0
;
if fits_file ne 'None' then begin
	rddat,fits_file,wave,spec,error,qual,header
	wave=wave/10	; convert to nm
	e=mrdfits(fits_file,1,h_ext1)
	q=mrdfits(fits_file,2,h_ext2)
endif
;
end
;-------------------------------------------------------------------------------
function pollux_t_event,event
;
common PlotPolluxData,files,regions,r_label,t_grid,g_grid,t_eff,log_g
common PlotPolluxWids,opmenu_t_wid,opmenu_g_wid,m_list_wid
;
widget_control,event.id,get_value=t
t_eff=t(event.index)
index=where(strmid(r_label,0,4) eq t_eff)
;
r_logg=strmid(r_label(index),5,3)
widget_control,opmenu_g_wid,set_value=r_logg
;
log_g=r_logg(0)
widget_control,m_list_wid,set_value=pollux_models(),set_uvalue=pollux_models()
;
end
;-------------------------------------------------------------------------------
function pollux_g_event,event
;
common PlotPolluxData,files,regions,r_label,t_grid,g_grid,t_eff,log_g
common PlotPolluxWids,opmenu_t_wid,opmenu_g_wid,m_list_wid
;
widget_control,event.id,get_value=g
log_g=g(event.index)
;`
widget_control,m_list_wid,set_value=pollux_models(),set_uvalue=pollux_models()
;
end
;-------------------------------------------------------------------------------
function pollux_models_event,event
;
common PlotPolluxData,files,regions,r_label,t_grid,g_grid,t_eff,log_g
common PlotPolluxObs,fits_file,wave,spec,error,qual,header,h_ext1,h_ext2
common LocalPolluxModelsEvent,model,flux,lamn
;
; Restore Pollux spectrum (lambda_aa -> lamn,flux)
if size(event,/type) eq 8 then begin
	widget_control,event.id,get_uvalue=models
	model=models(event.index)
	restore,!atmospheres_dir+'pollux/'+model; Restores lambda_aa and flux
; 	Interpolate onto finer grid
	l=findgen(50001)*0.1+19500
	f=interpol(flux,lambda_aa,l)
	lamn=l/10
	flux=f
endif
;
; Normalize spectrum
r=polyfit(lamn,flux,3,yfit)
flux=flux/yfit
;
; Find best-fit Gaussian smoothing kernel and wavelength shift
j=0	; shift [pixels]
i=0	; Gaussian smoothing width [pixels]
if wave(0) ne 0 then begin
	l_min=2160.
	l_max=2170.
	l_min=2445.
	l_max=2449.
	wave_brg=wave(where(wave ge l_min and wave le l_max))
	spec_brg=spec(where(wave ge l_min and wave le l_max))
	lamn_brg=lamn(where(lamn ge l_min and lamn le l_max))
	flux_brg=flux(where(lamn ge l_min and lamn le l_max))
;	Loop over Gaussian smoothing width (in units of Pollux pixels)
	imin=4
	imax=20
;	Loop of wavelength shift (in units of Pollux pixels)
	jmin=20
	jmax=120
;	Compute fit quality for each combination
	chisq=fltarr(jmax-jmin+1,imax-imin+1)
	for i=imin,imax do begin
	flux_c=gauss_smooth(flux_brg,i,/edge_truncate)
	for j=jmin,jmax do chisq(j-jmin,i-imin)= $
		total((spec_brg-interpol(shift(flux_c,j),lamn_brg,wave_brg))^2)
	endfor
	print,model+', minimum chi^2 =',min(chisq)
	r=whereindex(where(chisq eq min(chisq)),chisq)
	j=r(0)+jmin
	i=r(1)
endif
;
; Plot Pollux spectrum...
if n_elements(wave) le 1 then begin
	plot,lamn,flux,xtitle=greek('lambda')+' [nm]', $
		xstyle=1,xticks=12,xminor=5,charsize=1.5
; ...or observed spectrum and selected Pollux spectrum (shifted, convolved)
endif else begin
	flux_w=interpol(shift(gauss_smooth(flux,i,/edge_truncate),j),lamn,wave)
	plot,wave,flux_w, $
		xtitle=greek('lambda')+' [nm]', $
		xstyle=1,xticks=12,xminor=5,charsize=1.5
	oplot,wave,spec,psym=0,color=tci(3)
;	Write the line-corrected spectrum, adding "_n" to the name
	fitsfile=file_basename(fits_file,'.fits')+'_n.fits'
	writefits,fitsfile,spec/flux_w,header
	writefits,fitsfile,error,h_ext1,/append
	writefits,fitsfile,qual,h_ext2,/append
endelse
;
end
;-------------------------------------------------------------------------------
function pollux_models
;
common PlotPolluxData,files,regions,r_label,t_grid,g_grid,t_eff,log_g
;
index=where(strmid(r_label,0,4) eq t_eff and strmid(r_label,5,3) eq log_g)
models=nameparse(regions(index))
return,files(fix(models))
;
end
;-------------------------------------------------------------------------------
function ww_polluxcommand,event
;
common PlotPolluxData,files,regions,r_label,t_grid,g_grid,t_eff,log_g
;
case event.value of
'Window':begin
	icom=set_boxes(box_x,box_y)
	if icom eq 4 then begin
                icom=set_boxes(box_x,box_y,/clear)
  		!x.range=[1900,2500]
		!y.range=[0,1.2]
	endif else begin
		!x.range=[box_x(0),box_x(1)]
		!y.range=[box_y(3),box_y(4)]
	endelse
	r=pollux_models_event()
	end
'H-Line'  :begin
	   if !d.window eq -1 then begin
		print,'***Error(WW_PLOTUTIL): no plot window open!'
		return,-1
	   endif
	   print,'___________________________________________'
	   print,'Click left mouse button on plot window.'
	   print,'Click middle or right mouse button to exit!'
	   print,'__________________***______________________'
	   cursor,x,y,/down & icom=!err
	   device,set_graphics_function=6
	   x1=!x.crange(0) & x2=!x.crange(1)
	   xrange=x2-x1
	   x1=x1-0.1*xrange & x2=x2+0.1*xrange
	   oplot,[x1,x2],[y,y],psym=0
	   repeat begin
		cursor,x,yn,/change & icom=!err
		oplot,[x1,x2],[y,y],psym=0
		oplot,[x1,x2],[yn,yn],psym=0
		print,format='(%"Value: %s\r",$)', $
			string(yn,format='(f7.3)')
		y=yn
	   endrep until (icom eq 4) or (icom eq 2)
	   if icom eq 4 then oplot,[x1,x2],[y,y],psym=0
	   device,set_graphics_function=3
	   end
'V-Line':begin
        if !d.window eq -1 then begin
             print,'***Error(WW_PLOTUTIL): no plot window open!'
             return,-1
        endif
        print,'___________________________________________'
        print,'Click left mouse button on plot window.'
        print,'Click middle or right mouse button to exit!'
        print,'__________________***______________________'
        cursor,x,y,/down & icom=!err
        device,set_graphics_function=6
        y1=!y.crange(0) & y2=!y.crange(1)
        yrange=y2-y1
        y1=y1-0.1*yrange & y2=y2+0.1*yrange
        oplot,[x,x],[y1,y2],psym=0
        repeat begin
             cursor,xn,y,/change & icom=!err
             oplot,[x,x],[y1,y2],psym=0
             oplot,[xn,xn],[y1,y2],psym=0
;		print,format='(%"Wavelength: %d\r",$)',fix(xn)
		print,format='(%"Wavelength: %s\r",$)', $
			string(xn,format='(f7.2)')
             x=xn
        endrep until (icom eq 4) or (icom eq 2)
        if icom eq 4 then oplot,[x,x],[y1,y2],psym=0
        device,set_graphics_function=3
	print,''
	end
'Hardcopy':begin
        end
else:	begin
	words=nameparse(event.value)
	wl=float(words(1))
	!x.range=[wl-50,wl+50]
	!y.range=[0.95,1.005]
	end
endcase
;
end
;-------------------------------------------------------------------------------
pro pollux_prepare
;
files=specname(findfile(!atmospheres_dir+'pollux/*.spec'))
for i=0,n_elements(files)-1 do begin
	print,'Reading file: ',files(i)
	status=dc_read_free(files(i),lambda_aa,flux,/col)
	index=where(lambda_aa ge 19000 and lambda_aa le 25000)
	lambda_aa=lambda_aa(index)
	flux=flux(index)
	save,lambda_aa,flux,filename=files(i)+'.xdr'
endfor
;
end
;************************************************************************Block 4
pro plotbinary
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(binary_model) eq 0 then begin
	print,'Please load binary model first!'
	return
endif
if n_elements(positions) eq 0 then begin
	print,'Please read position data before!'
	return
endif
;
status=init_plotorbit()
;
a=binary_model(0).semimajoraxis
e=binary_model(0).eccentricity
i=binary_model(0).inclination
o=binary_model(0).periastron
n=binary_model(0).ascendingnode
;
base_wid=widget_base(/col, title='Binary orbit',resource_name='oyster')
;
d_wid=widget_draw(base_wid,scr_ysize=400,scr_xsize=400)
a_wid=cw_fslider(base_wid,title='Semi-major axis',uvalue='a',value=a, $
	minimum=a/10,maximum=a,/drag,xsize=400)
n_wid=cw_fslider(base_wid,title='Ascending node',uvalue='n',value=n, $
	minimum=0,maximum=360,/drag,xsize=400)
i_wid=cw_fslider(base_wid,title='Inclination',uvalue='i',value=i, $
	minimum=0,maximum=180,/drag,xsize=400)
e_wid=cw_fslider(base_wid,title='Eccentricity',uvalue='e',value=e, $
	minimum=-0.95,maximum=0.95,/drag,xsize=400)
o_wid=cw_fslider(base_wid,title='Periastron',uvalue='o',value=o, $
	minimum=0,maximum=360,/drag,xsize=400)
;
widget_control,base_wid,/realize
xmanager,'plotbinary',base_wid,/no_block,event_handler='plotbinary_event'
;
plot,findgen(10),findgen(10),/nodata, $
	xrange=2*[a,-a],yrange=2*[-a,a],ticklen=-0.02
;
; Plot periastron vector
plotthiele,'A-B'
p=modelpos(binary_model(0).epoch)
plots,[[fltarr(2)],[reform(p(0,0,*)-p(0,1,*))]],linestyle=0
;
; Plot line of nodes
RAD=180/!pi
e=binary_model(0).eccentricity
ea=2*atan(tan((-binary_model(0).periastron/RAD)/2)/sqrt((1+e)/(1-e)))
ma=ea-e*sin(ea)
p1=modelpos((ma/(2*!pi))*binary_model(0).period+binary_model(0).epoch)
ea=2*atan(tan((!pi-binary_model(0).periastron/RAD)/2)/sqrt((1+e)/(1-e)))
ma=ea-e*sin(ea)
p2=modelpos((ma/(2*!pi))*binary_model(0).period+binary_model(0).epoch)
plots,[[reform(p1(0,0,*)-p1(0,1,*))],[reform(p2(0,0,*)-p2(0,1,*))]],linestyle=1
;
end
;-------------------------------------------------------------------------------
pro plotbinary_event,event
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
r=0.95
x=fix(poly(!x.crange,!x.s)*400) & xr=x(1)-x(0)
y=fix(poly(!y.crange,!y.s)*400) & yr=y(1)-y(0)
nx=xr-2
ny=yr-2
tv,bytarr(nx,ny),x(0)+1,y(0)+1
;
widget_control,event.id,get_uvalue=slider,get_value=value
case slider of
	'a':binary_model(0).semimajoraxis=value
	'n':binary_model(0).ascendingnode=value
	'i':binary_model(0).inclination=value
	'e':binary_model(0).eccentricity=value
	'o':binary_model(0).periastron=value
endcase
;
; Plot periastron vector
plotthiele,'A-B'
p=modelpos(binary_model(0).epoch)
plots,[[fltarr(2)],[reform(p(0,0,*)-p(0,1,*))]],linestyle=0
;
; Plot line of nodes
RAD=180/!pi
e=binary_model(0).eccentricity
ea=2*atan(tan((-binary_model(0).periastron/RAD)/2)/sqrt((1+e)/(1-e)))
ma=ea-e*sin(ea)
p1=modelpos((ma/(2*!pi))*binary_model(0).period+binary_model(0).epoch)
ea=2*atan(tan((!pi-binary_model(0).periastron/RAD)/2)/sqrt((1+e)/(1-e)))
ma=ea-e*sin(ea)
p2=modelpos((ma/(2*!pi))*binary_model(0).period+binary_model(0).epoch)
plots,[[reform(p1(0,0,*)-p1(0,1,*))],[reform(p2(0,0,*)-p2(0,1,*))]],linestyle=1
;
end
;************************************************************************Block 5
pro plotiso
;
common iso_data,z,age,table,current_data,current_age
common aux_data,dotted_data,dashed_data,do_dotted,do_dashed
common obs_data,ot,ote,ol,ole,do_obs
;
if n_elements(do_obs) eq 0 then do_obs=0
if n_elements(do_dotted) eq 0 then do_dotted=0
if n_elements(do_dashed) eq 0 then do_dashed=0
;
z_list=['0004','0010','0040','0080','0200','0500']
ages=float(indgen(350)+660)/100
buttons_1=['Range','Data','Hardcopy','Info','Interpolate','Dotted','Dashed']
;
top_wid=widget_base(/row,title=isochrones,resource_name='oyster')
list_wid=widget_list(top_wid,value=z_list,uvalue=z_list, $
	event_pro='ww_isogetz', $
	ysize=n_elements(z_list))
buttons_1_wid=cw_bgroup(top_wid,buttons_1,/col, $
	event_func='ww_isoutilities',/return_name)
controls_wid=cw_fslider(top_wid,title='log(age)', $
	minimum=min(ages),maximum=max(ages), $
	/drag,/vertical,format='(f5.2)')
widget_control,top_wid,/realize
xmanager,'plotiso',top_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro plotiso_event,event
;
common iso_data,z,age,table,current_data,current_age
;
widget_control,event.id,get_value=age
plot_iso,'x'
end
;-------------------------------------------------------------------------------
pro plot_iso,dev
;
common iso_data,z,age,table,current_data,current_age
common obs_data,ot,ote,ol,ole,do_obs
common aux_data,dotted_data,dashed_data,do_dotted,do_dashed
common interpol,l_d,c_d,l_i,c_i,l_a,c_a
common local,last_age
;
if n_elements(table) eq 0 then begin
	print,'***Error: no table loaded!'
	return
endif
;
set_plot,dev
if dev eq 'PS' then device,/encapsulated
;
if n_elements(last_age) eq 0 then last_age=0.0
if dev eq 'PS' then last_age=-1
;
if age ne last_age then begin
	if dev eq 'x' then begin
	 index=where(abs(table.age-age) eq min(abs(table.age-age)),count) 
	endif else begin
	 index=where(fix(table.age*10+0.5) eq fix(current_age*10+0.5),count) 
	endelse
	if count ne 0 then begin
		last_age=age
		if total(abs(!x.range)) eq 0 then begin
			range_auto=1 
			t_min=min(table(index).teff)
			t_max=max(table(index).teff)
			l_max=max(table(index).lum)
			l_min=min(table(index).lum)
			!x.range=[t_max,t_min]
			!y.range=[l_min,l_max]
		endif else range_auto=0
		!p.charsize=1.5
		if dev eq 'x' then begin
		plot,table(index).teff,table(index).lum, $
			xtitle='log(effective temperature/K)', $
			ytitle='log(luminosity/solar luminosity)',psym=0
		endif else begin
		print,'Sending PostScript plot to file wave.ps...'
		plot,table(index).teff,table(index).lum, $
			xtitle='log(effective temperature/K)', $
			ytitle='log(luminosity/solar luminosity)'
		if not do_dotted and not do_dashed then begin
		xyouts,0.25,0.2,'log(age/yr)='+string(age,format='(f4.2)'), $
			/normal,charsize=1.5
		xyouts,0.25,0.25,'Z='+string(float(z)/10000,format='(f5.3)'), $
			/normal,charsize=1.5
		endif
		endelse
		if range_auto then begin
			!x.range=0
			!y.range=0
		endif
		if do_dotted then $
			oplot,dotted_data(*,0),dotted_data(*,1),linestyle=1
		if do_dashed then $
			oplot,dashed_data(*,0),dashed_data(*,1),linestyle=2
;
		current_data=fltarr(count,2)
		current_data(*,0)=table(index).teff
		current_data(*,1)=table(index).lum
		current_age=age
;
do_obs=1
		if do_obs then begin
ot=[3.892,3.892]
ol=[1.685,1.246]
ote=[0.026,0.026]
ole=[0.089,0.097]
			px1=[ot(0)-ote(0),ot(0)+ote(0)]
			py1=[ol(0),ol(0)]
			px2=[ot(0),ot(0)]
			py2=[ol(0)-ole(0),ol(0)+ole(0)]
			sx1=[ot(1)-ote(1),ot(1)+ote(1)]
			sy1=[ol(1),ol(1)]
			sx2=[ot(1),ot(1)]
			sy2=[ol(1)-ole(1),ol(1)+ole(1)]
			oplot,px1,py1,thick=3
			oplot,px2,py2,thick=3
			oplot,sx1,sy1,thick=3
			oplot,sx2,sy2,thick=3
		endif
	endif else begin
;		print,'***Error: invalid age: log(age)=',age
	endelse
endif
;
if dev eq 'PS' then device,/close
set_plot,'x'
;
end
;-------------------------------------------------------------------------------
pro ww_isogetz,event
;
common iso_data,z,age,table,current_data,current_age
;
widget_control,event.id,get_uvalue=z_list
z=z_list(event.index)
if z eq '' then return
case z of
	'0004':restore,'z0004_b.tbl'
	'0010':restore,'z0010_b.tbl'
	'0040':restore,'z0040_b.tbl'
	'0080':restore,'z0080_b.tbl'
	'0200':restore,'z0200_b.tbl'
	'0500':restore,'z0500_b.tbl'
	  else:begin
	       print,'***Error: value not allowed: z=',z
	       return
	       end
endcase
;
end
;-------------------------------------------------------------------------------
function ww_isoutilities,event
;
common obs_data,ot,ote,ol,ole,do_obs
common iso_data,z,age,table,current_data,current_age
common aux_data,dotted_data,dashed_data,do_dotted,do_dashed
common interpol,l_d,c_d,l_i,c_i,l_a,c_a
;
case event.value of
	'Range':begin
	   	icom=set_boxes(box_x,box_y)
	   	if icom eq 4 then begin
  			!x.range=0
			!y.range=0
	   	endif else begin
			!x.range=[box_x(0),box_x(1)]
			!y.range=[box_y(3),box_y(4)]
	   	endelse
	   	end
	'Data' :begin
		if n_elements(do_obs) eq 0 then do_obs=0
		end
	'Hardcopy':begin
		plot_iso,'PS'
		end
	'Info' :begin
		if n_elements(table) eq 0 then begin
			print,'***Error: no table loaded!'
			return,-1
		endif
		index=where(fix(table.age*10+0.5) eq fix(current_age*10+0.5), $
			count)
		if count gt 0 then begin
			teff=table(index).teff
			lum =table(index).lum
;			mbol=table(index).mbol
			mass=table(index).mass
			mv=table(index).mv
			bv=table(index).bv
			x=double(teff*!x.s(1)+!x.s(0))*640
			y=double(lum *!y.s(1)+!y.s(0))*512
			cursor,xc,yc,/down,/device
			xc=double(xc)
			yc=double(yc)
			dist=sqrt((x-xc)^2+(y-yc)^2)
			i=where(dist eq min(dist))
			i=i(0)
			j=i+1
			label1:
			v1=[x(j)-x(i),y(j)-y(i)]
			v2=[xc-x(i),yc-y(i)]
			v1l=sqrt(v1(0)^2+v1(1)^2)
			dl=(v1(0)*v2(0)+v1(1)*v2(1))/v1l
			if dl lt 0 and j eq i+1 then begin
				j=i-1
				goto,label1
			endif
			dx=v1(0)/v1l*dl
			dy=v1(1)/v1l*dl
;
			if n_elements(c_i) eq 0 then c_i=0
			l_i=c_i
			c_i=i
			if n_elements(c_d) eq 0 then c_d=0
			l_d=c_d
			c_d=current_data
			if n_elements(c_a) eq 0 then c_a=0
			l_a=c_a
			c_a=current_age
;
			tvx=x(i)+dx
			tvy=y(i)+dy
			tvcrs,tvx,tvy,/device
			teff_int=(teff(j)-teff(i))/v1l*dl+teff(i)
			lum_int=(lum(j)-lum(i))/v1l*dl+lum(i)
;			mbol_int=(mbol(j)-mbol(i))/v1l*dl+mbol(i)
			mv_int=(mv(j)-mv(i))/v1l*dl+mv(i)
			bv_int=(bv(j)-bv(i))/v1l*dl+bv(i)
			mass_int=(mass(j)-mass(i))/v1l*dl+mass(i)
			lum_ml=-0.021828336 $
                               +4.5336284*alog10(mass_int) $
                               -0.62450534*alog10(mass_int)^2
mbol_int=0
			print,'T=',10^teff_int,', L=',10^lum_int, $
				', M_b=',mbol_int, $
				', M_V=',mv_int,', B-V=',bv_int, $
				', M=',mass_int,', L_ML=',10^lum_ml, $
			format='(a,f6.0,a,f8.1,a,f5.2,a,f5.2,a,f5.2,a,f5.2,a,f8.1)'
		endif
		end
	'Interpolate' :begin
		il=where(table.age eq l_a,count_l)
		table_l=table(il)
		ic=where(table.age eq c_a,count_c)
		table_c=table(ic)
		num_1=min([l_i,c_i])
		num_2=min([count_l-l_i,count_c-c_i])
		num=num_1+num_2

		teff_l=table_l(l_i-num_1:l_i+num_2-1).teff
		lum_l=table_l(l_i-num_1:l_i+num_2-1).lum
		mbol_l=table_l(l_i-num_1:l_i+num_2-1).mbol
		mass_l=table_l(l_i-num_1:l_i+num_2-1).mass
		mv_l=table_l(l_i-num_1:l_i+num_2-1).mv
		bv_l=table_l(l_i-num_1:l_i+num_2-1).bv
		teff_c=table_c(c_i-num_1:c_i+num_2-1).teff
		lum_c=table_c(c_i-num_1:c_i+num_2-1).lum
		mbol_c=table_c(c_i-num_1:c_i+num_2-1).mbol
		mass_c=table_c(c_i-num_1:c_i+num_2-1).mass
		mv_c=table_c(c_i-num_1:c_i+num_2-1).mv
		bv_c=table_c(c_i-num_1:c_i+num_2-1).bv

		num_i=9
		num_t=n_elements(table)
		num_n=num_t+num_i*num
		aget=fltarr(num_n) & aget(0:num_t-1)=table.age
		teff=fltarr(num_n) & teff(0:num_t-1)=table.teff
		lum=fltarr(num_n) & lum(0:num_t-1)=table.lum
		mbol=fltarr(num_n) & mbol(0:num_t-1)=table.mbol
		mass=fltarr(num_n) & mass(0:num_t-1)=table.mass
		mv=fltarr(num_n) & mv(0:num_t-1)=table.mv
		bv=fltarr(num_n) & bv(0:num_t-1)=table.bv
		age_step=float(fix((c_a-l_a)*100+0.5))/100/(num_i+1)
		for i=0,num_i-1 do begin
			aget(num_t+i*num:num_t+(i+1)*num-1)= $
			 (i+1)*age_step+l_a
			teff(num_t+i*num:num_t+(i+1)*num-1)= $
			 (teff_c-teff_l)/(c_a-l_a)*(i+1)*age_step+teff_l
			lum(num_t+i*num:num_t+(i+1)*num-1)= $
			 (lum_c-lum_l)/(c_a-l_a)*(i+1)*age_step+lum_l
			mbol(num_t+i*num:num_t+(i+1)*num-1)= $
			 (mbol_c-mbol_l)/(c_a-l_a)*(i+1)*age_step+mbol_l
			mass(num_t+i*num:num_t+(i+1)*num-1)= $
			 (mass_c-mass_l)/(c_a-l_a)*(i+1)*age_step+mass_l
			mv(num_t+i*num:num_t+(i+1)*num-1)= $
			 (mv_c-mv_l)/(c_a-l_a)*(i+1)*age_step+mv_l
			bv(num_t+i*num:num_t+(i+1)*num-1)= $
			 (bv_c-bv_l)/(c_a-l_a)*(i+1)*age_step+bv_l
		endfor
		table=build_table('aget age, teff, lum, mbol, mass, mv, bv')
		end
	'Dotted':begin
		dotted_data=current_data
		if do_dotted then do_dotted=0 else do_dotted=1
		end
	'Dashed':begin
		dashed_data=current_data
		if do_dashed then do_dashed=0 else do_dashed=1
		end
endcase
end
;-------------------------------------------------------------------------------
pro ww_isogetdata,event
;
common obs_data,ot,ote,ol,ole,do_obs
;
if n_elements(do_obs) eq 0 then do_obs=1
if do_obs eq 1 then do_obs=0 else do_obs=1
;
;if do_obs then status=dc_read_free(wwgetvalue(wid),ot,ote,ol,ole,/col)
;
end
;-------------------------------------------------------------------------------
