;*******************************************************************************
; File: plotwidget.pro
;
; Description:
; ------------
; Container of all IDL widget scripts related to plotting in OYSTER. 
;
; Block directory:
; ----------------
; Block 1: ww_plot,ww_plotrefresh,ww_plotdestroyed,
;	   ww_plotoptions,ww_plothelp,ww_plotok,
;          ww_3D,ww_3Ddestroyed,ww_3Drotate,
;	   ww_base,ww_basedestroyed,ww_plotutil,ww_fitplotdata,ww_listdefault
; Block 2: ww_setxaxis,ww_setyaxis,ww_setzaxis,ww_setslice,ww_setwsize,
;          ww_setntdir,ww_setstdir,ww_setnightsel,ww_setstarsel
; Block 3: ww_indexsel,ww_indexseldestroyed,
;	   ww_indexib,ww_indexob,ww_indextr,
;          ww_setinbeam,ww_setoutbeam,ww_settriple,
;	   ww_putdir,ww_getdir,
;	   ww_setchdir,ww_setbldir,ww_setptdir,
;	   ww_setchsel,ww_setblsel,ww_setptsel
;
; Astrometry
; Block 4: ww_plotorbit,ww_plotorbitdestroyed,ww_plotorbitoptions,
;	   ww_plotorbithelp,ww_plotorbitok,ww_plotorbitutil
;
; Spectroscopy (RV)
; Block 5: ww_plotvel,ww_plotveldestroyed,ww_plotveloptions,
;	   ww_plotvelhelp,ww_plotvelok,ww_plotvelutil
;
; Photometry
; Block 6: ww_plotmag,ww_plotmagdestroyed,ww_plotmagoptions,
;	   ww_plotmaghelp,ww_plotmagok,ww_plotmagutil
;
;************************************************************************Block 1
pro ww_plot
;
; Displays the main interferometry plot widget.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PlotWids1,dropy_wid,dropx_wid,dropsl_wid
common PlotWids2,dropst_wid,listst_wid,dropnt_wid,option_wid
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common DataSelWids,x_wid,x_shown,y_wid,y_shown,z_wid,z_shown
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common StarBase,StarTable,Notes
common LocalWwPlotRefresh,offsety,offseti,offsetd
;
; Check status of currently displayed widgets
if n_elements(x_shown) eq 0 then x_shown=0
if n_elements(y_shown) eq 0 then y_shown=0
if n_elements(z_shown) eq 0 then z_shown=0
if x_shown then widget_control,x_wid,/destroy
if y_shown then widget_control,y_wid,/destroy
if z_shown then widget_control,z_wid,/destroy
;
if n_elements(plot_wid) eq 0 then plot_wid=0L
if plot_wid ne 0 then widget_control,plot_wid,/destroy
;
; Check status of class
if n_elements(class) eq 0 then begin
	print,'***Error(WW_PLOT): no class specified!'
	return
endif
;
; Check data
case class of
	'point' :if checkdata([1,11]) ne 0 then return
	'delay' :if checkdata([1,11]) ne 0 then return
	'bg'    :if checkdata([1,2,3,10]) ne 0 then return
	'scan'  :if checkdata([1,9]) ne 0 then return
	'ncal'  :if checkdata([1,9]) ne 0 then return
	'phot'  :if checkdata([1,9]) ne 0 then return
	'astrom':if checkdata([1,9]) ne 0 then return
	'seeing':if checkdata([1]) ne 0 then return
	'metro' :if checkdata([16]) ne 0 then return
	'uv'    :if checkdata([9]) ne 0 then return
	'amoeba':if checkdata([13,14]) ne 0 then return
	'volvox':if checkdata([12,13,14]) ne 0 then return
	    else:begin
		 print,'***Error(WW_PLOT): unknown class: ',class
		 return
		 end
endcase
;
; Create the widget
case class of
	'point' :title='PointPlot'
	'delay' :title='DelayPlot'
	'bg'    :title='BgPlot'
	'scan'  :title='VisibilityPlot'
	'ncal'  :title='CalibrateSystem'
	'phot'  :title='Photometry'
	'astrom':title='AstrometryPlot'
	'seeing':title='SeeingPlot'
	'metro' :title='MetrologyPlot'
	'uv'    :title='AmoebaUV'
	'amoeba':title='AmoebaXY'
	'volvox':title='AmoebaAstrom'
	    else:begin
		 print,'***Error(WW_PLOT): unknown class: ',class
		 return
		 end
endcase
offsety=10
;plot_wid=widget_base(/column,title=title,kill_notify='ww_plotdestroyed', $
;			resource_name='oyster', $
;			xoffset=130,yoffset=offsety,/kbrd_focus_events)
plot_wid=widget_base(/column,title=title,kill_notify='ww_plotdestroyed', $
			resource_name='oyster', $
			xoffset=130,/kbrd_focus_events)
; 
; Axis selection
classes=['point','delay','bg','uv','scan','astrom','seeing','phot','amoeba','metro','volvox']
if long(where(classes eq class),0) ne -1 then begin 
 	ax_items=set_axitems(item_ids)
 	dropy_wid=widget_droplist(plot_wid,title='Y-axis:', $
		event_pro='ww_setyaxis',value=ax_items)
 	dropx_wid=widget_droplist(plot_wid,title='X-axis:', $
		event_pro='ww_setxaxis',value=ax_items)
 	if class ne 'uv' then $
 	dropz_wid=widget_droplist(plot_wid,title='Z-axis:', $
		event_pro='ww_setzaxis',value=ax_items)
endif
;
; Type selection
classes=['ncal']
if long(where(classes eq class),0) ne -1 then begin
	types=['V2Bias','TABias','APDFlux','Response','TrackJitter','TrackOffset']
	dropty_wid=widget_droplist(plot_wid,title='Type: ', $
		event_pro='ww_settype',value=types,uvalue=types)
	widget_control,dropty_wid,set_droplist_select=0
	type=types(0)
endif
;
; Slice and Preset selection
classes=['delay','scan','amoeba']
if long(where(classes eq class),0) ne -1 then begin
	row_wid=widget_base(plot_wid,/row)
 	slices=['ib','ob','tr','ch','bl','pt']
 	dropsl_wid=widget_droplist(row_wid,title='Slice: ', $
		event_pro='ww_setslice',value=slices,uvalue=slices)
 	widget_control,dropsl_wid,set_droplist_select=5
	presets=['Preset','V_cal','V_pt','V_ch','TA_ch','TP_ch']
 	droppr_wid=widget_droplist(row_wid, $
		event_pro='ww_setpreset',value=presets,uvalue=presets)
endif
;
; Star selection
classes=['point','delay','bg','scan','uv','astrom','seeing','ncal','phot','amoeba','metro','volvox']
if long(where(classes eq class),0) ne -1 then begin 
 	stars=['All','Cal','Sci','Sel']
	row_wid=widget_base(plot_wid,/row)
 	dropst_wid=widget_droplist(row_wid,title='Stars: ', $
		event_pro='ww_setstdir',value=stars,uvalue=stars)
 	sh_wid=widget_button(row_wid,value='Show',event_pro='ww_indexst')
 	up_wid=widget_button(row_wid,value='Update',event_pro='ww_updatest')
 	if class eq 'amoeba' or class eq 'uv' then stars=startable.starid $
					      else list_stars,stars
	names=strarr(n_elements(stars))
;	Special feature for IR/MIR observations: list IRAS targets
	if !owner eq 'chummel' then begin
		iras_stars=strarr(n_elements(stars))
		ra=dblarr(n_elements(stars))
		dec=dblarr(n_elements(stars))
		for i=0,n_elements(stars)-1 do begin
			j=where(startable.starid eq stars(i))
			ra(i)=startable(j).ra
			dec(i)=startable(j).dec
		endfor
		iras_stars='IRAS'+irasid(ra,dec)
	endif
	for i=0,n_elements(stars)-1 do begin
		j=where(stars(i) eq startable.starid) & j=j(0)
		names(i)=strtrim(startable(j).name,2)
		if !owner eq 'chummel' and $
			strpos(instrument_id(systemid),'MATISSE') ge 0 $
		then begin
			names(i)=iras_stars(j)
			if strlen(names(i)) eq 0 then $
				names(i)=strtrim(startable(j).name,2)
		endif
	endfor
 	listst_wid=widget_list(plot_wid,/multiple,ysize=3, $
		event_pro='ww_setstarsel', $
		value=stars+' ('+names+')',uvalue=stars)
	widget_control,up_wid,set_uvalue=listst_wid
endif
;
; Night selection
classes=['amoeba','uv','volvox']
if long(where(classes eq class),0) ne -1 then begin
 	nights=['All','Sel']
 	dropnt_wid=widget_droplist(plot_wid,title='Nights:         ', $
		event_pro='ww_setntdir',value=nights,uvalue=nights)
 	widget_control,dropnt_wid,set_droplist_select=0
 	nights=GenInfo.date+' '+GeoInfo.systemid+' '+GenInfo.configid
 	listnt_wid=widget_list(plot_wid,value=nights,uvalue=nights, $
		event_pro='ww_setnightsel',/multiple,ysize=3,xsize=30)
endif
;
; Window size selection
classes=['point','delay','scan','amoeba']
if long(where(classes eq class),0) ne -1 then begin 
 	wsizes=['500x400','640x512','800x640','1024x512','1792x1024','800x1024']
 	dropsz_wid=widget_droplist(plot_wid,title='Window:', $
		event_pro='ww_setwsize',value=wsizes,uvalue=wsizes)
 	widget_control,dropsz_wid,set_droplist_select=2
endif
!xsize=640
!ysize=512
;
; Menu selection
panel=widget_base(plot_wid,/row)
utils=widget_base(panel,/column)
plot_menu=['1\Plot\ww_plotok', $
	   '0\Screen', $
	   '0\File'];, $
;	   '4\Auto', $
;	   '2\Zero']
plot_menu_wid=cw_pdmenu(utils,plot_menu)
edit_menu=['1\Edit\ww_plotok', $
	   '0\Auto', $
	   '2\Zero']
edit_menu_wid=cw_pdmenu(utils,edit_menu)
util_menu=['1\Util\ww_plotutil', $
           '0\Edit', $
           '4\Window', $
           '0\Range', $
           '0\Mean', $
           '0\Identify', $
           '0\H-Line', $
           '0\V-Line', $
           '6\FIT']
util_menu_wid=cw_pdmenu(utils,util_menu)
clear_button_wid=widget_button(utils,value='Clear',event_pro='ww_plothelp')
help_button_wid=widget_button(utils,value='HELP',event_pro='ww_plothelp')
;
options=['Errors', $
	 'Flagged', $
	 'Lines', $
	 'All in 1', $
	 '3D','Image', $
	 'NoFIR', $
	 'NoSort', $
	 'NoTrace', $
	 'All OBs', $
	 'All IBs', $
	 'Model', $
	 'Color', $
	 'Smooth', $
	 'Paper', $
	 'Custom']
option_wid=cw_bgroup(panel,column=2,options,/nonexclusive, $
 			event_funct='ww_plotoptions')
;
; Display plot widget
widget_control,plot_wid,/realize
widget_control,plot_wid,yoffset=offsety+1
xmanager,'ww_plot',plot_wid,event_handler='ww_plotrefresh',/no_block
;
; Reset plot ranges
!x.range=0
!y.range=0
!z.range=0
;
; Call ww_indexsel for classes which require data selection in any case
if class eq 'uv' then ww_indexsel,{top:plot_wid},'y'
if class eq 'ncal' then ww_indexsel,{top:plot_wid},'y'
;
end
;-------------------------------------------------------------------------------
pro ww_plotrefresh,event
;
; Helper procedure to force refresh of widget when window manager doesn't.
; This fixes a problem which can occur when running IDL in virtual machines.
;
common LocalWwPlotRefresh,offsety,offseti,offsetd
;
; if !owner ne 'chummel' then return
;
g=widget_info(event.id,/geometry)
;
if n_elements(offseti) eq 0 then begin
	offseti=0
	offsetd=g.yoffset-offsety
endif
;
if abs(g.yoffset-offsety-offsetd) gt 2 then offsety=g.yoffset-offsetd
widget_control,event.id,xoffset=g.xoffset,yoffset=offsety+(offseti mod 2)
offseti=offseti+1
;
end
;-------------------------------------------------------------------------------
pro ww_plotdestroyed,wid
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
plot_wid=0L
;
end
;-------------------------------------------------------------------------------
function ww_plotoptions,event
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common DataSelWids,x_wid,x_shown,y_wid,y_shown,z_wid,z_shown
;
case event.value of
	0: ps_options.e=event.select	; Errors
	1: ps_options.f=event.select	; Flagged
	2: ps_options.l=event.select	; Lines
	3: ps_options.a=event.select	; All in 1
	4: ps_options.d=event.select	; 3D
	5: ps_options.g=event.select	; imaGe
	6: ps_options.r=event.select	; No fit in Range (NoFIR)
	7: ps_options.n=event.select	; Nosort
	8: ps_options.t=event.select	; noTrace
        9: ps_options.o=event.select	; all Output beams
       10: ps_options.i=event.select	; all Input beams
       11: ps_options.m=event.select	; Model
       12: ps_options.c=event.select	; Color print
       13: ps_options.s=event.select	; "Smooth" spectral model line
       14: ps_options.p=event.select	; use Publication labels
       15: ps_options.v=event.select	; normalize amplitude ratio to 1
endcase
if not ps_options.t and x_shown then widget_control,x_wid,/destroy
end
;-------------------------------------------------------------------------------
pro ww_plothelp,event
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common PlotUtilWids,base_wid,list_wids
;
widget_control,event.id,get_value=command
case command of
	'Clear':begin
	      	if n_elements(td_wid) eq 0 then td_wid=0L
	      	if td_wid ne 0 then widget_control,td_wid,/destroy
	      	if n_elements(pivot_wid) eq 0 then pivot_wid=0L
	      	if pivot_wid ne 0 then widget_control,pivot_wid,/destroy
	      	if n_elements(base_wid) eq 0 then base_wid=0L
	      	if base_wid ne 0 then widget_control,base_wid,/destroy
;	      	while !d.window ge 0 do wdelete,!d.window
		repeat begin
			wset,-1
			if !d.window ne -1 then wdelete
		endrep until !d.window eq -1
	      	end
	'HELP':	begin
	      	print,'_____________________________________________________'
	      	print,'Note: -1- refers to the first index in a selection!'
	      	print,'For buttons SELECTED, NEXT, and PREVIOUS, you have to'
      	      	print,'enter a selection (e.g. "1,2,4-7", which is parsed'
	      	print,'into "1,2,4,5,6,7"). NEXT will increment the current'
	      	print,'selection by the NUMBER of selected elements! You'
	      	print,'should not have more than one NEXT button selected.'
	      	print,'Note that options <All OB> and <All IB> disable the'
	      	print,'OutputBeam and InputBeam selections, respectively!'
	      	print,'Not all of the options are available for every class!'
	      	print,'______________________***____________________________'
	      	end
endcase
end
;-------------------------------------------------------------------------------
pro ww_plotok,event
;
; Checks data selection when PLOT button is pressed in plot selection widget.
; Calls plot or edit routines if checks come out ok. Data type specifies
; which particular routines are called. Also dispatches 3D plot.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common PlotUtilWids,base_wid,list_wids
common LocalWwPlotOk,i0
;
; Complete data selection, classes amoeba and uv checked separately
if class ne 'amoeba' and class ne 'uv' and class ne 'volvox' then begin
	if set_starsel() ne 0 then begin
		print,'***Error(WW_PLOTOK): invalid star selection!'
		return
	endif
	if set_indexsel('x') ne 0 then begin
		print,'***Error(WW_PLOTOK): invalid x-data selection!'
		return
	endif
	if set_indexsel('y') ne 0 then begin
		print,'***Error(WW_PLOTOK): invalid y-data selection!'
		return
	endif
	if set_indexsel('z') ne 0 then begin
		print,'***Error(WW_PLOTOK): invalid y-data selection!'
		return
	endif
endif else if set_nightsel() ne 0 then begin
		print,'***Error(WW_PLOTOK): invalid night selection!'
		return
endif
;
; Remove any plot utility widgets
if n_elements(base_wid) eq 0 then base_wid=0L
if base_wid ne 0 then widget_control,base_wid,/destroy
;
; Deactivate 3D widget if present
if n_elements(td_wid) eq 0 then td_wid=0L
if td_wid ne 0 then widget_control,td_wid,sensitive=0
;
; Extract calling command
widget_control,event.id,get_value=command
;
; 3D plots
if ps_options.d and (not ps_options.m) and $
   (command ne 'Auto') and (command ne 'Zero') then ww_3D,event
;
; All other plotting/automatic editing
case command of
'Screen':begin
;	 set_plot,!display
	 set_screen
	 command='S/P'
	 end
'File'  :begin
	 words=nameparse(systime())
	 filename=strjoin(words(1:3),'-')+'.eps'
	 print,'Printing to file '+filename+'...'
	 set_ps,filename,color=ps_options.c
	 command='S/P'
	 end
else    :
endcase
;
case class of
'point' :begin
	 case command of
	 'Auto'  :plotdata,1,'AUTO'
	 'Zero'  :plotdata,2,'ZERO'
	 'S/P'   :plotdata
	 else    :print,'***Error(WW_PLOTOK): '+ $
				'choice not available!'
	 endcase
	 end
'delay' :begin
	 case command of
	 'Auto'  :plotdata,1,'AUTO'
	 'Zero'  :plotdata,2,'ZERO'
	 'S/P'   :plotdata
	 else    :print,'***Error(WW_PLOTOK): '+ $
	 		'choice not available!'
	 endcase
	 end
'bg'    :begin
	 case command of
	 'Zero'  :plotdata,2,'ZERO'
	 'S/P'   :plotdata
	 else    :print,'***Error(WW_PLOTOK): '+ $
	 		'choice not available!'
	 endcase	       
	 end
'scan'  :begin
	 case command of
	 'S/P'   :plotdata
	 else    :print,'***Error(WW_PLOTOK): '+ $
	 		'choice not available!'
	 endcase	       
	 end
'phot'  :begin
	 case command of
	 'S/P'   :plotdata
	 else    :print,'***Error(WW_PLOTOK): '+ $
	 		'choice not available!'
	 endcase	       
	 end
'ncal'  :begin
	 case command of
	 'S/P'   :plotncal
	 else    :print,'***Error(WW_PLOTOK): '+ $
	 		'choice not available!'
	 endcase
	 end
'astrom':begin
	 case command of
	 'S/P'   :plotdata
	 else    :print,'***Error(WW_PLOTOK): '+ $
	 		'choice not available!'
	 endcase	       
	 end
'seeing':begin
	 case command of
	 'S/P'   :plotdata
	 else    :print,'***Error(WW_PLOTOK): '+ $
	 		'choice not available!'
	 endcase	       
	 end
'uv0'    :begin
	 case command of
	 'S/P'   :plotuv
	 else    :print,'***Error(WW_PLOTOK): '+ $
	 		'choice not available!'
	 endcase
	 end
'metro' :begin
	 case command of
 	'Auto'  :plotdata,1,'AUTO'
	'S/P'   :plotdata
	else    :print,'***Error(WW_PLOTOK): '+ $
			'choice not available!'
	 endcase
	 end
'uv':begin
	 case command of
	 'S/P' :begin
		num_nights=n_elements(ds_nights)
		nob_mc_mb=size(scans(0).vissqc)
		if num_nights eq 1 and nob_mc_mb(1) eq 1 then do1=1 else do1=0
		if n_elements(i0) eq 0 then i0=0
		if num_nights gt 1 and $
			!d.name ne 'PS' and $
			not ps_options.a then $
		print,'More than one plot! Left click to continue, '+ $
			'middle to pause, right to exit!'
		overplot=0
;
		for i=i0,num_nights-1 do begin
			if strupcase(!version.os) ne 'WIN32' then $
			if n_elements(bufferinfo) gt 1 then $
				loadnight,ds_nights(i)
			flag=0
			!quiet=1
			if set_starsel() ne 0 then flag=1
			if set_indexsel('x',/quiet) ne 0 then flag=1
			if set_indexsel('y',/quiet) ne 0 then flag=1
			if set_indexsel('z',/quiet) ne 0 then flag=1
			!quiet=0
			if flag eq 0 then begin
			   if !d.name ne 'PS' then set_screen
			   if do1 then plotuv1,overplot=ps_options.a*overplot $
				  else plotuv,overplot=ps_options.a*overplot
			   overplot=overplot+1
			endif
			if num_nights gt 1 and i lt num_nights-1 $
				   	and !d.name ne 'PS' $
				   	and not ps_options.a then begin
				i0=0
				cursor,x,y
				wait,1
				if !mouse.button eq 2 then i0=i
				if !mouse.button ne 1 then return
			endif
		endfor
		end
	 else   :print,'***Error(WW_PLOTOK): '+ $
			'choice not available!'
	 endcase	       
	 end
'amoeba':begin
	 case command of
	 'S/P' :begin
		num_nights=n_elements(ds_nights)
		if n_elements(i0) eq 0 then i0=0
		if num_nights gt 1 and !d.name ne 'PS' and not ps_options.a then $
		print,'More than one plot! Left click to continue, '+ $
			'middle to pause, right to exit!'
		previous_window=!d.window
;
		for i=i0,num_nights-1 do begin
		if i eq i0 then overplot=1
		if strupcase(!version.os) ne 'WIN32' then $
		if n_elements(bufferinfo) gt 1 then loadnight,ds_nights(i)
		flag=0
		!quiet=1
		if set_starsel() ne 0 then flag=1
		if set_indexsel('x') ne 0 then flag=1
		if set_indexsel('y') ne 0 then flag=1
		if set_indexsel('z') ne 0 then flag=1
		!quiet=0
		if flag eq 0 then begin
		if !d.name ne 'PS' then set_screen
		if not ps_options.a then begin
			if !d.name ne 'PS' then wdall
			plotdata,overplot=0,status=status 
			if !d.window ne -1 and !owner eq 'chummel' then begin
;			"Refresh" needed if running in virtual machine
;				wshow,iconic=1
;				wait,1
;				wshow,iconic=0
			endif
		endif else begin
			if !d.window eq -1 $
			or !d.window eq previous_window then plotdata $
							else plotdata,overplot=i
			status=0
		endelse
		if num_nights gt 1 and i lt num_nights-1 $
				   and status eq 0 $
				   and !d.window ne -1 $
				   and !d.name ne 'PS' $
				   and not ps_options.a then begin
			cursor,x,y,/down
;			wait,1
			if !mouse.button eq 2 then i0=i
			if !mouse.button ne 1 then return
			continue
		endif
		endif
		endfor
		end
'volvox':begin
	 case command of
	 'S/P' :begin
		num_nights=n_elements(ds_nights)
		if n_elements(i0) eq 0 then i0=0
		if num_nights gt 1 and !d.name ne 'PS' and not ps_options.a then $
		print,'More than one plot! Left click to continue, '+ $
			'middle to pause, right to exit!'
		for i=i0,num_nights-1 do begin
			if strupcase(!version.os) ne 'WIN32' then $
			if n_elements(bufferinfo) gt 1 then loadnight,ds_nights(i)
			flag=0
			if set_starsel() ne 0 then flag=1
			if set_indexsel('x') ne 0 then flag=1
			if set_indexsel('y') ne 0 then flag=1
			if set_indexsel('z') ne 0 then flag=1
			if flag eq 0 then begin
				if !d.name ne 'PS' then set_screen
				plotdata,overplot=ps_options.a*(i<1)
			endif
			if num_nights gt 1 and i lt num_nights-1 $
					   and !d.name ne 'PS' $
					   and not ps_options.a then begin
				i0=0
				cursor,x,y
				wait,1
				if !mouse.button eq 2 then i0=i
				if !mouse.button ne 1 then return
			endif
		endfor
		end
	 else   :print,'***Error(WW_PLOTOK): '+ $
			'choice not available!'
	 endcase	       
	 end
	 else   :print,'***Error(WW_PLOTOK): '+ $
			'choice not available!'
	 endcase	       
	 end
else    :print,'***Error(WW_PLOTOK): class unknown: ',class
endcase	       
;
if !d.name eq 'PS' then begin
	print,'Printed to file '+filename
	device,/close_file
	device,/portrait
	set_plot,!display
;	Deactivated in 2019, ps_options.s=1 now means "smoothing"
	if 0 then begin
	if ps_options.s then begin
	   if strpos(strupcase(getenv('HOST')),'tucana') ne -1 then begin
	      if ps_options.c then spawn,'lpr -h '+filename $
			      else spawn,'lpr -h '+filename
	   endif else begin
		if ps_options.c then spawn,'lpr -h -Pqmscolor '+filename $
		                else spawn,'lpr -h '+filename
	   endelse
	   print,'Plot submitted.'
	   wait,3
	   spawn,'rm -f '+filename
	endif else print,'Plot saved to: '+filename
	endif
endif
;
end
;-------------------------------------------------------------------------------
pro ww_3D,event
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
; 3D plots
;
if td_wid ne 0 then widget_control,td_wid,/destroy
xs=long(!xsize)
ys=long(!ysize)
td_wid=widget_base(group_leader=event.top,/column, $
			resource_name='oyster', $
			title='3D, Class:'+class, $
			kill_notify='ww_3Ddestroyed')
row1_wid=widget_base(td_wid,/row)
controlx_wid=widget_slider(row1_wid,minimum=-90,maximum=90,ysize=ys, $
	event_pro='ww_3Drotate',/vertical,value=0,uvalue='x')
drawing_wid=widget_draw(row1_wid,notify_realize='ww_3Drotate', $
	event_pro='ww_3Drotate',scr_xsize=xs,scr_ysize=ys)
controlz_wid=widget_slider(row1_wid,minimum=-90,maximum=90,ysize=ys, $
	event_pro='ww_3Drotate',/vertical,value=0,uvalue='z')
controly_wid=widget_slider(td_wid,minimum=-90,maximum=90,xsize=xs, $
	event_pro='ww_3Drotate',value=0,uvalue='y')
widget_control,td_wid,/realize
xmanager,'t3d',td_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_3Ddestroyed,wid
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
td_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_3Drotate,event
;
; Callback for 3D option.
;
common Ww3Drotate,x_rotation,y_rotation,z_rotation
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
result=size(event)
if result(n_elements(result)-2) ne 8 then begin
	x_rotation=0
	y_rotation=0
	z_rotation=0
endif else begin
	widget_control,event.id,get_uvalue=xyz
	case xyz of
		'x':x_rotation=event.value
		'y':y_rotation=event.value
		'z':z_rotation=event.value
	endcase
endelse
t3d,/reset,rotate=[-x_rotation,y_rotation,z_rotation]
scale3d
if class eq 'ncal' then plotncal else plotdata
;
end
;-------------------------------------------------------------------------------
function ww_base,title,list_index,callback_pro,list_wids,xpos,ypos, $
	group_leader=group_leader
;
common BaseFunctions,numlist,functions
;
if n_elements(functions) eq 0 then create_calentries
;
if n_elements(xpos) eq 0 then xpos=0
if n_elements(ypos) eq 0 then ypos=0
;
base_wid=widget_base(group_leader=group_leader,/floating, $
	/row,title=title,resource_name='oyster', $
	kill_notify='ww_basedestroyed',xoffset=xpos,yoffset=ypos)
;
num_list=n_elements(list_index)
list_wids=lonarr(num_list)
;
for i=0,num_list-1 do begin
	if list_index(i) eq 1 then j=0 else $
	j=long(total(numlist(0:list_index(i)-2)))
	k=j+numlist(list_index(i)-1)-1
	list_wids(i)=widget_list(base_wid,/multiple,ysize=5, $
		value=functions(j:k),uvalue=functions(j:k), $
		event_pro=callback_pro)
endfor
widget_control,base_wid,/realize
xmanager,'ww_base',base_wid,/no_block
;
return,base_wid
;
end
;-------------------------------------------------------------------------------
pro ww_basedestroyed,wid
;
common PlotUtilWids,base_wid,list_wids
;
base_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_plotutil,event
;
; Callback procedure for tasks working with data in PlotDataInfo.
; These are the buttons listed as utilities in the plot widget.
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common PlotDataInfo,plotscans,plotdata_x,plotdata_y,plotdata_y_bck
common FitPlotData,x_mid,x_scl,s
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common PlotUtilWids,base_wid,list_wids
;
widget_control,event.id,get_value=command
;
if ps_options.d then begin
	print,'***Error(WW_PLOTUTIL): action not available for 3D plots!'
	return
endif
if ps_options.g then begin
	print,'***Error(WW_PLOTUTIL): action not available for Image plots!'
	return
endif
;
case command of
;
'Identify':begin
	   if !p.multi(1)*!p.multi(2) gt 1 then begin
	   	print,'***Error(WW_PLOTUTIL): more than one plot in window!'
	   	return
	   endif
	   if class eq 'uv' or class eq 'ncal' then begin
		print,'***Error(WW_PLOTUTIL): action not available!'
		return
	   endif
	   if !d.window eq -1 then begin
		print,'***Error(WW_PLOTUTIL): no plot window open!'
		return
	   endif
	   if !p.multi(1)*!p.multi(2) gt 1 then begin
	   	print,'***Error(WW_PLOTUTIL): more than one plot in window!'
	   	return
	   endif
 	   print,'___________________________'
	   print,'Click left button on point.'
	   print,'Click right button to exit!'
	   print,'___________***_____________'
	   repeat begin
		icom=set_points(plotdata_x,plotdata_y,index)
		t=plotscans.time(index)/3600
		print,'_______________________________________'
		print,'Star is: ',plotscans.star(index)
		print,'Time is:',hms(t),' (',t,')',format='(a,a,a,f10.7,a)'
		print,'IB=',plotscans.ib(index)+1, $
		    ', OB=',plotscans.ob(index)+1, $
		    ', TR=',plotscans.tr(index)+1, $
		    ', CH=',plotscans.ch(index)+1, $
		    ', BL=',plotscans.bl(index)+1, $
		    ', PT=',plotscans.pt(index)+1, $
		    format='(a,i1,a,i1,a,i1,a,i3,a,i1,a,i4)'
		print,'__________________***__________________'
	   endrep until icom eq 4
	   end
'H-Line'  :begin
	   if !d.window eq -1 then begin
		print,'***Error(WW_PLOTUTIL): no plot window open!'
		return
	   endif
	   print,'___________________________________________'
	   print,'Click left mouse button on plot window.'
	   print,'Click middle or right mouse button to exit!'
	   print,'__________________***______________________'
	   cursor,x,y,/down & icom=!err
	   device,set_graphics_function=6
	   x1=!x.crange(0) & x2=!x.crange(1)
	   xrange=x2-x1
	   x1=x1-0.1*xrange & x2=x2+0.1*xrange
	   oplot,[x1,x2],[y,y],psym=0
	   repeat begin
		cursor,x,yn,/change & icom=!err
		oplot,[x1,x2],[y,y],psym=0
		oplot,[x1,x2],[yn,yn],psym=0
		y=yn
	   endrep until (icom eq 4) or (icom eq 2)
	   if icom eq 4 then oplot,[x1,x2],[y,y],psym=0
	   device,set_graphics_function=3
	   end
'V-Line'  :begin
	   if !d.window eq -1 then begin
		print,'***Error(WW_PLOTUTIL): no plot window open!'
		return
	   endif
	   print,'___________________________________________'
	   print,'Click left mouse button on plot window.'
	   print,'Click middle or right mouse button to exit!'
	   print,'__________________***______________________'
	   cursor,x,y,/down & icom=!err
	   device,set_graphics_function=6
	   y1=!y.crange(0) & y2=!y.crange(1)
	   yrange=y2-y1
	   y1=y1-0.1*yrange & y2=y2+0.1*yrange
	   oplot,[x,x],[y1,y2],psym=0
	   repeat begin
		cursor,xn,y,/change & icom=!err
		oplot,[x,x],[y1,y2],psym=0
		oplot,[xn,xn],[y1,y2],psym=0
	   	print,string(13b),'x='+string(xn),format='(a,a,$)'
		x=xn
	   endrep until (icom eq 4) or (icom eq 2)
	   print,''
	   if icom eq 4 then oplot,[x,x],[y1,y2],psym=0
	   device,set_graphics_function=3
	   end
'FIT'     :begin
	   if class eq 'uv' or class eq 'ncal' then begin
		print,'***Error(WW_PLOTUTIL): action not available!'
		return
	   endif
	   if !d.window eq -1 then begin
		print,'***Error(WW_PLOTUTIL): no plot window open!'
		return
	   endif
	   if !p.multi(1)*!p.multi(2) gt 1 then begin
	   	print,'***Error(WW_PLOTUTIL): more than one plot in window!'
	   	return
	   endif
	   if n_elements(base_wid) eq 0 then base_wid=0L
	   if base_wid ne 0 then widget_control,base_wid,/destroy
; 	   Save original plotdata
	   plotdata_y_bck=plotscans.y
	   base_wid=ww_base('Base functions',[1,2,3,4],'ww_fitplotdata', $
				list_wids,group_leader=plot_wid)
           end
'Window'  :begin
	   if !d.window eq -1 then begin
		print,'***Error(WW_PLOTUTIL): no plot window open!'
		return
	   endif
	   if !p.multi(1)*!p.multi(2) gt 1 then begin
	   	print,'***Error(WW_PLOTUTIL): more than one plot in window!'
	   	return
	   endif
	   icom=set_boxes(box_x,box_y)
	   if icom eq 4 then begin
		icom=set_boxes(box_x,box_y,/clear)
  		!x.range=0 & !x.style=0
		!y.range=0 & !y.style=0
		print,'Window reset.'
	   endif else begin
		!x.range=[box_x(0),box_x(1)]
		!y.range=[box_y(3),box_y(4)]
		!x.style=1
		!y.style=1
		print,'Window set.'
	   endelse
	   end
'Range'	  :begin
	   read,ymin,ymax,prompt='Enter Y min,max (auto=0,0): '
	   !y.range=[ymin,ymax]
	   if total(abs(!y.range)) ne 0 then !y.style=1 else !y.style=0
	   read,xmin,xmax,prompt='Enter X min,max (auto=0,0): '
           !x.range=[xmin,xmax]
	   if total(abs(!x.range)) ne 0 then !x.style=1 else !x.style=0
	   print,'Range set.'
           end
'Mean':	   begin
	   if !d.window eq -1 then begin
                print,'***Error(WW_PLOTUTIL): no plot window open!'
                return
           endif
           if !p.multi(1)*!p.multi(2) gt 1 then begin
                print,'***Error(WW_PLOTUTIL): more than one plot in window!'
                return
           endif
	   if set_boxes(box_x,box_y) ne 4 then begin
	   	x=plotscans.x
	   	y=plotscans.y
	   	x_err=plotscans.xe
	   	y_err=plotscans.ye
	   	index=where((x gt box_x(0)) and (x lt box_x(1)) and $
		       	    (y gt box_y(2)) and (y lt box_y(1)),count)
	   	if count gt 0 then begin
	   		jndex=where(x_err(index) gt 0 $
			        and y_err(index) gt 0,count)
	   		if count gt 0 then begin
				x=x(index(jndex))
				y=y(index(jndex))
				e=y_err(index(jndex))
				if ps_options.e then begin
					ym=total(y/e^2)/total(1/e^2)
					ye=sqrt(1/total(1/e^2))
					estring=' +/- '
				endif else begin
					ym=avg(y)
					ye=stddev(y)
					estring=' RMS='
				endelse
				print,'Average(y) = ',ym,estring,ye
				print,'Average(x) = ',avg(x)
			endif
	   	endif
	   endif else icom=set_boxes(box_x,box_y,/clear)
	   end
'Edit':    begin
	   if class eq 'uv' or class eq 'ncal' then begin
		print,'***Error(WW_PLOTUTIL): action not available!'
		return
	   endif
	   if !d.window eq -1 then begin
		print,'***Error(WW_PLOTUTIL): no plot window open!'
		return
	   endif
	   if !p.multi(1)*!p.multi(2) gt 1 then begin
		print,'***Error(WW_PLOTUTIL): more than one plot in window!'
		return
	   endif
	   if ps_options.m and ps_options.v then begin
		print,'***Error(WW_PLOTUTIL): please deselect Custom option!'
		return
	   endif
	   case class of
	   'point':begin
		if n_elements(pointflagtable) eq 0 then create_pointflagtable
		index=where(pointflagtable.time ge 0,count)
		end
	   'delay':begin
		if n_elements(pointflagtable) eq 0 then create_pointflagtable
		index=where(pointflagtable.time ge 0,count)
		end
	   'metro':begin
		if n_elements(inchflagtable) eq 0 then create_inchflagtable
		index=where(inchflagtable.time ge 0,count)
		end
	   'scan':begin
		if n_elements(scanflagtable) eq 0 then create_scanflagtable
		index=where(scanflagtable.time ge 0,count)
		end
	   'amoeba':begin
		if n_elements(scanflagtable) eq 0 then create_scanflagtable
		index=where(scanflagtable.time ge 0,count)
		end
	   'astrom':begin
		if n_elements(scanflagtable) eq 0 then create_scanflagtable
		index=where(scanflagtable.time ge 0,count)
		end
	   'bg':begin
		if n_elements(bgflagtable) eq 0 then create_bgflagtable
		index=where(bgflagtable.time ge 0,count)
		end
	   else:begin
		print,'***Error(WW_PLOTUTIL): no editing allowed in this class!'
		return
		end
	   endcase
	   id0=count
	   id=id0
	   timestamp=strmid(systime(),0,24)
	   ax_items=set_axitems(item_ids,units,abbrs)
	   reason=timestamp+' '+'MANUAL'+'_'+abbrs(plotscans.item(0)-1)
	   print,'Begin editing, REASON='+reason
	   icom=set_boxes(box_x,box_y,nbox)
	   icom=cw_form('0,LIST,Delete inside boxes|Delete outside boxes|Cancel,set_value=0,QUIT',title='Question?')
;
	   if icom.tag0 ne 2 then begin
;
;	   Start editing
	   t=plotscans.time
	   x=plotscans.x
	   y=plotscans.y
	   x_err=plotscans.xe
	   y_err=plotscans.ye
;
;	   Loop over boxes
	   for i=0,nbox-1 do begin
	   case icom.tag0 of
	      0:index=where((x gt box_x(0,i)) and (x lt box_x(1,i)) and $
		 (y gt box_y(2,i)) and (y lt box_y(1,i)),count)
	      1:index=where((x lt box_x(0,i)) or (x gt box_x(1,i)) or $
		 (y gt box_y(1,i)) or (y lt box_y(2,i)),count)
	   endcase
	   if count gt 0 then begin
	   	jndex=where(x_err(index) gt 0 and y_err(index) gt 0,count)
	   	if count gt 0 then index=index(jndex)
	   endif
	   if count gt 0 then begin
		n=id+count-1
		if count eq 1 then index=index(0)
		case class of
		'point':begin
                        set_pointflagtable,reason,plotscans.item(index), $
                          plotscans.ib(index),plotscans.ob(index), $
			  plotscans.tr(index),plotscans.ch(index), $
			  plotscans.bl(index),abs(plotscans.time(index))
			end
		'delay':begin
                        set_pointflagtable,reason,plotscans.item(index), $
                          plotscans.ib(index),plotscans.ob(index), $
			  plotscans.tr(index),plotscans.ch(index), $
			  plotscans.bl(index),abs(plotscans.time(index))
			end
		'metro':begin
                        set_inchflagtable,reason,plotscans.item(index), $
                          plotscans.ib(index),abs(plotscans.time(index))
			end
		'scan' :begin
			scanflagtable(id:n).reason=reason
			scanflagtable(id:n).item=plotscans.item(index)
			scanflagtable(id:n).baseline=plotscans.bl(index)
			scanflagtable(id:n).triple=plotscans.tr(index)
			scanflagtable(id:n).channel=plotscans.ch(index)
			scanflagtable(id:n).inbeam=plotscans.ib(index)
			scanflagtable(id:n).outbeam=plotscans.ob(index)
			scanflagtable(id:n).time=abs(plotscans.time(index))
			end
		'amoeba' :begin
			scanflagtable(id:n).reason=reason
			scanflagtable(id:n).item=plotscans.item(index)
			scanflagtable(id:n).baseline=plotscans.bl(index)
			scanflagtable(id:n).triple=plotscans.tr(index)
			scanflagtable(id:n).channel=plotscans.ch(index)
			scanflagtable(id:n).inbeam=plotscans.ib(index)
			scanflagtable(id:n).outbeam=plotscans.ob(index)
			scanflagtable(id:n).time=abs(plotscans.time(index))
			end
	       'astrom':begin
			scanflagtable(id:n).reason=reason
			scanflagtable(id:n).item=plotscans.item(index)
			scanflagtable(id:n).baseline=plotscans.bl(index)
			scanflagtable(id:n).triple=plotscans.tr(index)
			scanflagtable(id:n).channel=plotscans.ch(index)
			scanflagtable(id:n).inbeam=plotscans.ib(index)
			scanflagtable(id:n).outbeam=plotscans.ob(index)
			scanflagtable(id:n).time=abs(plotscans.time(index))
			end
		'bg'   :begin
			bgflagtable(id:n).reason=reason
			bgflagtable(id:n).item=plotscans.item(index)
			bgflagtable(id:n).channel=plotscans.ch(index)
			bgflagtable(id:n).outbeam=plotscans.ob(index)
			bgflagtable(id:n).time=abs(plotscans.time(index))
			end
		endcase
		id=n+1
		pcolor=!p.color
		!p.color=tci(2)
		oplot,x(index(0:count-1)),y(index(0:count-1))
		if ps_options.e then begin
			if count gt 1 then $
				oploterr,x(index),y(index),y_err(index),3 $
			else begin
				x1=x(index(0))
				y1=y(index(0))
				e1=abs(y_err(index(0)))
				oploterr,[x1,x1],[y1,y1],[e1,e1],3
			endelse
		endif
		!p.color=pcolor
	   endif else icom=set_boxes(box_x,box_y,nbox,/clear)
	   endfor
;
	   endif
	   if id gt id0 then begin
		case class of
			'point' :flagpointdata,pointflagtable(id0:id-1),flag=1
			'delay' :flagpointdata,pointflagtable(id0:id-1),flag=1
			'metro' :flaginchdata,inchflagtable(id0:id-1),flag=1
			'scan'  :flagscandata,scanflagtable(id0:id-1),flag=1
			'amoeba':flagscandata,scanflagtable(id0:id-1),flag=1
			'astrom':flagscandata,scanflagtable(id0:id-1),flag=1
			'bg'    :flagbgdata,bgflagtable(id0:id-1),flag=1
		endcase
	   endif
	   end
;
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_fitplotdata,event
;
; Fits selected functions to data in FitPlotData and plots the residuals.
;
common PlotUtilWids,base_wid,list_wids
common BaseFunctions,numlist,functions
common PlotDataInfo,plotscans,plotdata_x,plotdata_y,plotdata_y_bck
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common FitPlotData,x_mid,x_scl,s
;
; If error bars requested, use dot as plot symbol
if ps_options.e then psym=3 else psym=abs(!p.psym)
if ps_options.l then psym=-psym
;
; Get the selected base functions
i=where(list_wids eq event.id) & i=i(0)
sel_ids=widget_info(event.id,/list_select)
if min(sel_ids) ge 0 then list_functions=functions(i*5+sel_ids) $
		     else list_functions=''
num_functions=n_elements(list_functions)
;
; If all deselected, plot original data
if list_functions(0) eq '' then begin
	plotscans.y=plotdata_y_bck
	index=where(plotscans.xe gt 0 and plotscans.ye gt 0,count)
	if count gt 0 then begin
	plot,plotscans.x(index),plotscans.y(index),psym=psym, $
		xtype=!x.type,ytype=!y.type
	if ps_options.e then $
	oploterr,plotscans.x(index),plotscans.y(index),plotscans.ye(index),3
	endif
	return
endif
;
; Clear selection of all lists other then the one from which selected
index=where(list_wids eq event.id)
case index(0) of
	0:begin
	  widget_control,list_wids(1),set_value=functions(5:9)
	  widget_control,list_wids(2),set_value=functions(10:14)
	  widget_control,list_wids(3),set_value=functions(15:19)
	  end
	1:begin
	  widget_control,list_wids(0),set_value=functions(0:4)
	  widget_control,list_wids(2),set_value=functions(10:14)
	  widget_control,list_wids(3),set_value=functions(15:19)
	  end
	2:begin
	  widget_control,list_wids(0),set_value=functions(0:4)
	  widget_control,list_wids(1),set_value=functions(5:9)
	  widget_control,list_wids(3),set_value=functions(15:19)
	  end
	3:begin
	  widget_control,list_wids(0),set_value=functions(0:4)
	  widget_control,list_wids(1),set_value=functions(5:9)
	  widget_control,list_wids(2),set_value=functions(10:14)
	  end
endcase
;
; Set data
if not ps_options.r and total(abs(!x.range)+abs(!y.range)) ne 0 then begin
	xmin=!x.range(0)
	xmax=!x.range(1)
	ymin=!y.range(0)
	ymax=!y.range(1)
	if total(abs(!x.range)) eq 0 then begin
		xmin=min(plotdata_x)
		xmax=max(plotdata_x)
	endif
	if total(abs(!y.range)) eq 0 then begin
		ymin=min(plotdata_y_bck)
		ymax=max(plotdata_y_bck)
	endif
	index=where((plotdata_x ge xmin) and $
		    (plotdata_x le xmax) and $
		    (plotdata_y_bck ge ymin) and $
		    (plotdata_y_bck le ymax))
endif else index=indgen(n_elements(plotscans.x))
if index(0) eq -1 then begin
	print,'Warning(WW_FITPLOTDATA): no data in range!'
	index=indgen(n_elements(plotscans.x))
endif
x=plotscans.x(index)
y=plotdata_y_bck(index)
xe=plotscans.xe(index)
ye=plotscans.ye(index)
;
; Edit the data
index=where(xe gt 0 and ye gt 0,num_scans)
if num_scans eq 0 then begin
	print,'Warning(WW_FITPLOTDATA): no valid data!'
	return
endif
x=x(index) & y=y(index) & y_error=ye(index)
;
; Special logarithmic case
if !x.type eq 1 then x=alog10(x)
if !y.type eq 1 then begin
		     y_error=y_error/y
		     y=alog10(y)
endif
;
; Optional shifting and scaling for numerical stability
x_mid=(min(x)+max(x))/2 ; With scaling, fit coeffs are changed
x_scl=2/(max(x)-min(x)) 
;x_mid=0
;x_scl=1
;
; Weighted fit?
if ps_options.e then do_e=1 else do_e=0
;
; Note that wt is NOT 1/error^2 because of the following normalization!
if do_e then wt=1/y_error else wt=dblarr(num_scans)+1
;
if strpos(list_functions(0),'S_') ge 0 then begin
;	Smoothing
	smooth_time=float(strmid(list_functions(0),2,2))/60
	y_fit=y
	for i=0,num_scans-1 do begin
		twt=exp(-((x(i)-x)/smooth_time)^2)*wt
		y_fit(i)=total(y*twt)/total(twt)
	endfor
	y_res=y-y_fit
	if do_e then print,'Chi^2 = ',total((y_res/y_error)^2) $
		/(num_scans-(max(x)-min(x))/smooth_time)
endif else begin
; 	Analytical solution
	m=dblarr(num_scans,num_functions,/nozero)
	for i=0,num_functions-1 do $
		m(*,i)=evalfunction(list_functions(i),x,x_mid,x_scl)*wt
	y=y*wt
	tm=transpose(m)
	n=tm#m
	r=tm#y
	if num_functions gt 1 then begin
		svd,n,w
;		Full SVD call is: svd,n,w,u,v. v(*,i) is eigenvector to w_i
		if max(w)/min(w) gt 1e12 then begin
			print,'***Error(WW_FITPLOTDATA): equations ill-defined!'
			return
		endif
		s=invert(n)#r
	endif else s=1/n*r
	for i=0,num_functions-1 do $
		m(*,i)=evalfunction(list_functions(i),x,x_mid,x_scl)
	y=y/wt
	y_fit=m#s
	y_res=y-y_fit
;	if n_elements(unique(y_fit)) eq 1 then print,'Mean = ',y_fit(0)
	if do_e then print,'Chi^2 = ',total((y_res/y_error)^2) $
		/(num_scans-num_functions)
endelse
;
; Special logarithmic case
if !x.type eq 1 then x=10^x
if !y.type eq 1 then begin
		     y=10^y
		     y_error=y_error*y
		     y_res=10^y_res
endif
;
; Plot the residuals
plot,x,y_res,title='Residuals from fit', $
	yrange=[0,0],xmargin=[14,3],psym=psym,xtype=!x.type,ytype=!y.type
print,'RMS =',stddev(y_res)
if strpos(list_functions(0),'S_') lt 0 then print,'Coeffs =',s
if ps_options.e then oploterr,x,y_res,y_error,3
;
; Update plotdata
if strpos(list_functions(0),'S_') ge 0 then begin
	plotscans.y=plotdata_y_bck-spline(x,y_res,plotscans.x-x_mid)
endif else begin
	m=dblarr(n_elements(plotscans.y),num_functions,/nozero)
	for i=0,num_functions-1 do $
		m(*,i)=evalfunction(list_functions(i),plotscans.x,x_mid,x_scl)
	plotscans.y=plotdata_y_bck-m#s
endelse
;
end
;************************************************************************Block 2
pro ww_setxaxis,event
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common DataSelWids,x_wid,x_shown,y_wid,y_shown,z_wid,z_shown
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
pflag=ps_options.p
ps_options.p=0
items=set_axitems(item_ids)
ps_options.p=pflag
ds_x.item=fix(item_ids(event.index),0)
!x.range=0
check=[3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,21,24,25,26,27,28,29,30,31,32, $
       33,34,35,36,37,38,39,40,50,51,52,53,54,55,56,61,62,63,64,65,66,67,68, $
       69,70,71,72,73,74,75,76,77,80,82,83,84,85,86,87,88,90,91,93,94]
i=where(check eq ds_x.item,count)
if count ne 0 and (not x_shown) and ps_options.t then begin
	if init_indexsel('x') ne 0 then return
	ww_indexsel,event,'x'
endif
end
;-------------------------------------------------------------------------------
pro ww_setyaxis,event
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common DataSelWids,x_wid,x_shown,y_wid,y_shown,z_wid,z_shown
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
pflag=ps_options.p
ps_options.p=0
items=set_axitems(item_ids)
ps_options.p=pflag
yitem=fix(item_ids(event.index),0)
if (yitem lt 24 or yitem gt 31 or ds_y.item lt 24 or ds_y.item gt 31) $
then !y.range=0
ds_y.item=yitem
check=[3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,21,24,25,26,27,28,29,30,31,32, $
       33,34,35,36,37,38,39,40,50,51,52,53,54,55,56,61,62,63,64,65,66,67,68, $
       69,70,71,72,73,74,75,76,77,80,82,83,84,85,86,87,88,90,91,93,94]
i=where(check eq ds_y.item,count)
if count ne 0 and (not y_shown) then begin
	if init_indexsel('y') ne 0 then return
	ww_indexsel,event,'y'
endif
end
;-------------------------------------------------------------------------------
pro ww_setzaxis,event
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common DataSelWids,x_wid,x_shown,y_wid,y_shown,z_wid,z_shown
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
pflag=ps_options.p
ps_options.p=0
items=set_axitems(item_ids)
ps_options.p=pflag
ds_z.item=fix(item_ids(event.index),0)
!z.range=0
check=[3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,21,24,25,26,27,28,29,30,31,32, $
       33,34,35,36,37,38,39,40,50,51,52,53,54,55,56,61,62,63,64,65,66,67,68, $
       69,70,71,72,73,74,75,76,77,80,82,83,84,85,86,87,88,90,91,93,94]
i=where(check eq ds_z.item,count)
if count ne 0 and (not z_shown) and ps_options.t then begin
	if init_indexsel('z') ne 0 then return
	ww_indexsel,event,'z'
endif
end
;-------------------------------------------------------------------------------
pro ww_settype,event
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
widget_control,event.id,get_uvalue=types
type=types(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_setslice,event
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
widget_control,event.id,get_uvalue=slices
slice=slices(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_setpreset,event
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common PlotWids1,dropy_wid,dropx_wid,dropsl_wid
common PlotWids2,dropst_wid,listst_wid,dropnt_wid,option_wid
common IndexSelDropWids,dropch_wid,dropbl_wid,droptr_wid,droppt_wid
common IndexSelTextWids,textch_wid,textbl_wid,textpt_wid
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common StarBase,StarTable,Notes
;
widget_control,event.id,get_uvalue=presets
preset=presets(event.index)
;
; Using preset, the user can create plot widgets with pre-defined settings
case preset of
'Preset':begin
	print,'Select a pre-defined'
	print,'plot selection.'
	end
'V_cal':begin
	widget_control,dropx_wid,set_droplist_select=0
	ww_setxaxis,{index:0,top:plot_wid}
	widget_control,dropy_wid,set_droplist_select=11
	ww_setyaxis,{index:11,top:plot_wid}
	if class eq 'amoeba' then begin
		widget_control,dropy_wid,set_droplist_select=6
		ww_setyaxis,{index:6,top:plot_wid}
	endif
	widget_control,dropsl_wid,set_droplist_select=5 & slice='pt'
	widget_control,dropst_wid,set_droplist_select=1 & st_dir='Cal'
	widget_control,dropch_wid,set_droplist_select=1
	y_dir.ch='Selected'
	y_sel.ch=0
	y_sel.ch(0)=1
	widget_control,textch_wid,set_value='1'
	mch=n_elements(y_sel.ch)/2 ; choose middle channel
	y_sel.ch(0)=mch
	widget_control,textch_wid,set_value=strtrim(string(mch),1)
	widget_control,dropbl_wid,set_droplist_select=0
	y_dir.bl='All'
	widget_control,droppt_wid,set_droplist_select=0
	y_dir.pt='All'
	!y.range=[0,1.2]
	widget_control,option_wid,set_value=[1,0,1,0,0,0,0,0,0,0,0,0]
	ps_options.e=1	; Errors
	ps_options.l=1	; Lines
	ps_options.a=0	; All-in-one
	ps_options.m=0	; Model (#11)
	end
'V_pt':	begin
	widget_control,dropx_wid,set_droplist_select=0
	ww_setxaxis,{index:0,top:plot_wid}
	widget_control,dropy_wid,set_droplist_select=9
	ww_setyaxis,{index:9,top:plot_wid}
	if class eq 'amoeba' then begin
		widget_control,dropy_wid,set_droplist_select=6
		ww_setyaxis,{index:6,top:plot_wid}
	endif
	widget_control,dropsl_wid,set_droplist_select=5 & slice='pt'
	widget_control,dropst_wid,set_droplist_select=3 & st_dir='Sel'
 	if class eq 'amoeba' then stars=startable.starid $
			     else list_stars,stars
	if n_elements(gen_model) ne 0 then star=gen_model.starid $
				      else star=stars(0)
	st_sel=star
	index=where(stars eq star)
	widget_control,listst_wid,set_list_sel=index(0)
	widget_control,dropch_wid,set_droplist_select=1
	y_dir.ch='Selected'
	y_sel.ch=0
	y_sel.ch(0)=1
	widget_control,textch_wid,set_value='1'
	widget_control,dropbl_wid,set_droplist_select=0
	y_dir.bl='All'
	widget_control,droppt_wid,set_droplist_select=0
	y_dir.pt='All'
	!y.range=[0,1.2]
	widget_control,option_wid,set_value=[1,0,1,0,0,0,0,0,0,0,0,0]
	ps_options.e=1	; Errors
	ps_options.l=1	; Lines
	ps_options.a=0	; All-in-one
	ps_options.m=0	; Model (#11)
	if n_elements(gen_model) ne 0 then begin
		ps_options.m=1	; Model
		widget_control,option_wid,set_value=[1,0,0,0,0,0,0,0,0,0,0,1]
	endif
	end
'V_ch':	begin
	widget_control,dropx_wid,set_droplist_select=2
	ww_setxaxis,{index:2,top:plot_wid}
	if class eq 'amoeba' then begin
		widget_control,dropy_wid,set_droplist_select=6
		ww_setyaxis,{index:6,top:plot_wid}
	endif else begin
		widget_control,dropy_wid,set_droplist_select=9
		ww_setyaxis,{index:9,top:plot_wid}
	endelse
	widget_control,dropsl_wid,set_droplist_select=3 & slice='ch'
	widget_control,dropst_wid,set_droplist_select=3 & st_dir='Sel'
 	if class eq 'amoeba' then stars=startable.starid $
			     else list_stars,stars
	if n_elements(gen_model) ne 0 then star=gen_model.starid $
				      else star=stars(0)
	st_sel=star
	index=where(stars eq star)
	widget_control,listst_wid,set_list_sel=index(0)
	widget_control,dropch_wid,set_droplist_select=0
	y_dir.ch='All'
;	Initialize for first baseline
	widget_control,dropbl_wid,set_droplist_select=1
	y_dir.bl='Selected'
	y_sel.bl=0
	y_sel.bl(0)=1
	widget_control,textbl_wid,set_value='1'
;	Initialize for all baselines
	widget_control,dropbl_wid,set_droplist_select=0
	y_dir.bl='All'
	y_sel.bl=0
	y_sel.bl(*)=1
	widget_control,textbl_wid,set_value='1'
	widget_control,droppt_wid,set_droplist_select=0
;	Initialize for all data points
	y_dir.pt='All'
	!y.range=[0,1.2]
	widget_control,option_wid,set_value=[1,0,1,0,0,0,0,0,0,0,0,0]
	ps_options.e=1	; Errors
	ps_options.l=1	; Lines
	ps_options.a=0	; All-in-one
	ps_options.m=0	; Model (#11)
	if n_elements(gen_model) ne 0 then begin
		ps_options.m=1	; Model
		ps_options.l=0	; Lines
		widget_control,option_wid,set_value=[1,0,0,0,0,0,0,0,0,0,0,1]
	endif
	end
'TA_ch':begin
	widget_control,dropx_wid,set_droplist_select=2
	ww_setxaxis,{index:2,top:plot_wid}
	if class eq 'amoeba' then begin
		widget_control,dropy_wid,set_droplist_select=9
		ww_setyaxis,{index:9,top:plot_wid}
	endif else begin
		if strpos(instrument_id(systemid),'GRAVITY') ge 0 $
		or strpos(instrument_id(systemid),'MATISSE') ge 0 $
			then o=3 else o=0
		widget_control,dropy_wid,set_droplist_select=16+o
		ww_setyaxis,{index:16+o,top:plot_wid}
	endelse
	widget_control,dropsl_wid,set_droplist_select=3 & slice='ch'
	widget_control,dropst_wid,set_droplist_select=3 & st_dir='Sel'
 	if class eq 'amoeba' then stars=startable.starid $
			     else list_stars,stars
	if n_elements(gen_model) ne 0 then star=gen_model.starid $
				      else star=stars(0)
	st_sel=star
	index=where(stars eq star)
	widget_control,listst_wid,set_list_sel=index(0)
	widget_control,dropch_wid,set_droplist_select=0
	y_dir.ch='All'
;	widget_control,dropbl_wid,set_droplist_select=1
;	y_dir.bl='Selected'
;	y_sel.bl=0
;	y_sel.bl(0)=1
;	widget_control,textbl_wid,set_value='1'
	widget_control,droppt_wid,set_droplist_select=0
	y_dir.pt='All'
	!y.range=[0,1.2]
	widget_control,option_wid,set_value=[1,0,1,0,0,0,0,0,0,0,0,0]
	ps_options.e=1	; Errors
	ps_options.l=1	; Lines
	ps_options.a=0	; All-in-one
	ps_options.m=0	; Model (#11)
	if n_elements(gen_model) ne 0 then begin
		ps_options.m=1	; Model
		ps_options.l=0	; Lines
		widget_control,option_wid,set_value=[1,0,0,0,0,0,0,0,0,0,0,1]
	endif
	end
'TP_ch':begin
	widget_control,dropx_wid,set_droplist_select=2
	ww_setxaxis,{index:2,top:plot_wid}
	if class eq 'amoeba' then begin
		widget_control,dropy_wid,set_droplist_select=10
		ww_setyaxis,{index:10,top:plot_wid}
	endif else begin
		if strpos(instrument_id(systemid),'GRAVITY') ge 0 $
		or strpos(instrument_id(systemid),'MATISSE') ge 0 $
			then o=3 else o=0
		widget_control,dropy_wid,set_droplist_select=20+o
		ww_setyaxis,{index:20+o,top:plot_wid}
	endelse
	widget_control,dropsl_wid,set_droplist_select=3 & slice='ch'
	widget_control,dropst_wid,set_droplist_select=3 & st_dir='Sel'
 	if class eq 'amoeba' then stars=startable.starid $
			     else list_stars,stars
	if n_elements(gen_model) ne 0 then star=gen_model.starid $
				      else star=stars(0)
	st_sel=star
	index=where(stars eq star)
	widget_control,listst_wid,set_list_sel=index(0)
	widget_control,dropch_wid,set_droplist_select=0
	y_dir.ch='All'
;	widget_control,dropbl_wid,set_droplist_select=1
;	y_dir.bl='Selected'
;	y_sel.bl=0
;	y_sel.bl(0)=1
;	widget_control,textbl_wid,set_value='1'
	widget_control,droppt_wid,set_droplist_select=0
	y_dir.pt='All'
	!y.range=[-180,180]
	widget_control,option_wid,set_value=[1,0,1,0,0,0,0,0,0,0,0,0]
	ps_options.e=1	; Errors
	ps_options.l=1	; Lines
	ps_options.a=0	; All-in-one
	ps_options.m=0	; Model (#11)
	if n_elements(gen_model) ne 0 then begin
		ps_options.m=1	; Model
		ps_options.l=0	; Lines
		widget_control,option_wid,set_value=[1,0,0,0,0,0,0,0,0,0,0,1]
	endif
	end
else:	return
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setwsize,event
;
widget_control,event.id,get_uvalue=sizes
size=sizes(event.index)
x_pos=strpos(size,'x')
!xsize=long(strmid(size,0,x_pos))
!ysize=long(strmid(size,x_pos+1,strlen(size)-x_pos-1))
;
end
;-------------------------------------------------------------------------------
pro ww_setntdir,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
widget_control,event.id,get_uvalue=dirs
nt_dir=dirs(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_setstdir,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common StarBase,StarTable,Notes
;
widget_control,event.id,get_uvalue=dirs
st_dir=dirs(event.index)
;
index=where(startable.bflag eq 'C',count)
if st_dir eq 'Cal' then begin
	print,'Calibrators: ',strjoin(startable(index).starid,' ')
	print,'Calibrators: ',strjoin(startable(index).name,' ')
endif
;
index=where(startable.bflag ne 'C',count)
if st_dir eq 'Sci' then print,'Targets: ',strjoin(startable(index).name,' ')
;
end
;-------------------------------------------------------------------------------
pro ww_setnightsel,event
;
common PlotWids2,dropst_wid,listst_wid,dropnt_wid,option_wid
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
widget_control,event.id,get_uvalue=nights
index=widget_info(event.id,/list_select)
if index(0) ge 0 then nt_sel=nights(index) else nt_sel=''
;
widget_control,dropnt_wid,set_droplist_select=1
nt_dir='Sel'
;
end
;-------------------------------------------------------------------------------
pro ww_setstarsel,event
;
common PlotWids2,dropst_wid,listst_wid,dropnt_wid,option_wid
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
widget_control,event.id,get_uvalue=stars
index=widget_info(event.id,/list_select)
if index(0) ge 0 then st_sel=stars(index) else st_sel=''
;
widget_control,dropst_wid,set_droplist_select=3
st_dir='Sel'
;
end
;-------------------------------------------------------------------------------
pro ww_indexst,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common Tables,ScanTable,BGTable,StationTable
common StarBase,StarTable,Notes
;
case st_dir of
	'All':index=indgen(n_elements(scantable))
	'Cal':begin
	      index=where(startable.bflag eq 'C',count)
	      if count gt 0 then begin
	      cal_stars=startable(index).starid
	      mask=bytarr(n_elements(scantable))
	      for i=0,n_elements(cal_stars)-1 do begin
		index=where(scantable.starid eq cal_stars(i),count)
		if count gt 0 then mask(index)=1
	      endfor
	      index=where(mask eq 1)
	      endif else index=-1
	      end
	'Sci':begin
	      index=where(startable.bflag ne 'C',count)
	      if count gt 0 then begin
	      sci_stars=startable(index).starid
	      mask=bytarr(n_elements(scantable))
	      for i=0,n_elements(sci_stars)-1 do begin
		index=where(scantable.starid eq sci_stars(i),count)
		if count gt 0 then mask(index)=1
	      endfor
	      index=where(mask eq 1)
	      endif else index=-1
	      end
	'Sel':begin
	      mask=bytarr(n_elements(scantable))
	      for i=0,n_elements(st_sel)-1 do begin
		index=where(scantable.starid eq st_sel(i),count)
		if count gt 0 then mask(index)=1
	      endfor
	      index=where(mask eq 1)
	      end
endcase
if index(0) ne -1 then list_scans,index+1
;
end
;-------------------------------------------------------------------------------
pro ww_updatest,event
;
common StarBase,StarTable,Notes
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
if class eq 'amoeba' then stars=startable.starid else list_stars,stars
widget_control,event.id,get_uvalue=wid
widget_control,wid,set_value=stars
widget_control,wid,set_uvalue=stars
;
end
;************************************************************************Block 3
pro ww_indexsel,event,stream
;
; Displays widget for the selection of data.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common DataSelWids,x_wid,x_shown,y_wid,y_shown,z_wid,z_shown
common IndexSelDropWids,dropch_wid,dropbl_wid,droptr_wid,droppt_wid
common IndexSelTextWids,textch_wid,textbl_wid,textpt_wid
common LocalWwSetDir,ch_dirs,bl_dirs,pt_dirs
common LocalWwIndexselRefresh,offsety,offseti,offsetd
;
title='IndexSelection, '+stream
;
; Create the widget
if n_elements(class) eq 0 then begin
	print,'***Error(WW_DATASEL): no class specified!'
	return
endif
;
offsety=20
datasel_wid=widget_base(group_leader=event.top,/floating, $
			resource_name='oyster', $
			kill_notify='ww_indexseldestroyed', $
			xoffset=380,yoffset=offsety, $
			title=title,/column,uvalue=stream,/kbrd_focus_events)
case stream of
	'x':x_wid=datasel_wid
	'y':y_wid=datasel_wid
	'z':z_wid=datasel_wid
endcase
;
; Input beam selection
classes=['point','delay','scan','astrom','amoeba','metro','seeing','phot','volvox']
if long(where(classes eq class),0) ne -1 and GenConfig.NumSid gt 0 then begin 
 inbeams=string(indgen(max(GenInfo.NumSid))+1)
 row_wid=widget_base(datasel_wid,/row)
 ib_wid=widget_button(row_wid,value='InputBeam: ',event_pro='ww_indexib', $
	uvalue=stream)
 opmenuib_wid=widget_droplist(row_wid,title='', $
	event_pro='ww_setinbeam',value=inbeams,uvalue=stream)
endif
;
; Output beam selection
classes=['point','delay','bg','scan','uv','ncal','amoeba','seeing']
if long(where(classes eq class),0) ne -1 and GenConfig.NumOutBeam gt 0 then begin 
 outbeams=string(indgen(max(GenInfo.NumOutBeam))+1)
 row_wid=widget_base(datasel_wid,/row)
 ob_wid=widget_button(row_wid,value='OutputBeam:',event_pro='ww_indexob', $
	uvalue=stream)
 opmenuob_wid=widget_droplist(row_wid,title='', $
	event_pro='ww_setoutbeam',value=outbeams,uvalue=stream)
endif
;
; Triple selection
classes=['point','scan','amoeba','ncal']
if long(where(classes eq class),0) ne -1 $
and max(GenInfo.NumTriple) gt 0 then begin
 triples=string(indgen(max(GenInfo.NumTriple))+1)
 count=0
 if class ne 'amoeba' then $
 index=where(total(scantable.station,fix(size(scantable.station),0)) eq 0,count)
 if count gt 0 then begin
  for i=0,genconfig.numtriple-1 do begin
   for j=0,count-1 do begin
    if strpos(strjoin(genconfig.baselineid(genconfig.triplebase(*,i), $
					   genconfig.triplebeam(*,i)),' '), $
	      genconfig.stationid(index(j))) ge 0 then triples(i)=''
   endfor
  endfor
  index=where(strlen(triples) gt 0,count)
  if count gt 0 then triples=triples(index)
 endif
 if total(strlen(triples)) ne 0 then begin
 row_wid=widget_base(datasel_wid,/row)
 tr_wid=widget_button(row_wid,value='Triple:    ',event_pro='ww_indextr', $
	uvalue=stream)
 opmenutr_wid=widget_droplist(row_wid,title='', $
	event_pro='ww_settriple',value=triples,uvalue=stream+triples)
 case stream of
	'x':ds_x.tr=triples(0)-1
	'y':ds_y.tr=triples(0)-1
	'z':ds_z.tr=triples(0)-1
 endcase
 endif
endif
;
dirs=['All','Selected','Next','Current','Previous']
;
; Channel selection
classes=['point','delay','bg','scan','uv','ncal','amoeba','seeing']
if long(where(classes eq class),0) ne -1 then begin 
 dropch_wid=widget_droplist(datasel_wid,title='Channels:  ', $
	event_pro='ww_setchdir',value=dirs,uvalue=stream)
 widget_control,dropch_wid,set_droplist_select=3
 textch_wid=widget_text(datasel_wid,event_pro='ww_setchsel', $
	/editable,uvalue=stream)
 ch_dirs=dirs
endif
;
; Baseline selection
classes=['point','delay','scan','uv','amoeba','ncal','seeing']
if long(where(classes eq class),0) ne -1 then begin 
 dropbl_wid=widget_droplist(datasel_wid,title='Baselines: ', $
	event_pro='ww_setbldir',value=dirs,uvalue=stream)
 widget_control,dropbl_wid,set_droplist_select=3
 textbl_wid=widget_text(datasel_wid,event_pro='ww_setblsel', $
	/editable,uvalue=stream)
 bl_dirs=dirs
endif
;
if 0 then begin	; work to allow multiple triple selection in progress
; Triple selection
classes=['point','scan','amoeba','ncal']
if long(where(classes eq class),0) ne -1 $
and max(GenInfo.NumTriple) gt 0 then begin
 triples=string(indgen(max(GenInfo.NumTriple))+1)
 count=0
 if class ne 'amoeba' then $
 index=where(total(scantable.station,fix(size(scantable.station),0)) eq 0,count)
 if count gt 0 then begin
  for i=0,genconfig.numtriple-1 do begin
   for j=0,count-1 do begin
    if strpos(strjoin(genconfig.baselineid(genconfig.triplebase(*,i), $
					   genconfig.triplebeam(*,i)),' '), $
	      genconfig.stationid(index(j))) ge 0 then triples(i)=''
   endfor
  endfor
  index=where(strlen(triples) gt 0,count)
  if count gt 0 then triples=triples(index)
 endif
 if total(strlen(triples)) ne 0 then begin
 droptr_wid=widget_droplist(datasel_wid,title='Triples: ', $
	event_pro='ww_settrdir',value=dirs,uvalue=stream)
 widget_control,droptr_wid,set_droplist_select=3
 texttr_wid=widget_text(datasel_wid,event_pro='ww_settrsel', $
	/editable,uvalue=stream)
 tr_dirs=dirs
 endif
endif
endif
;
; Point/scan selection
classes=['delay','scan','uv','amoeba','metro','ncal','phot','seeing','volvox']
if long(where(classes eq class),0) ne -1 then begin
 values=dirs
 if class eq 'ncal' then values=[dirs,scanconfig()] 
 if class eq 'scan' then values=[dirs,scanconfig(/coh)]
 if class eq 'seeing' then values=[dirs,scanconfig(/coh)] 
 droppt_wid=widget_droplist(datasel_wid,title='Points:    ', $
	event_pro='ww_setptdir',value=values,uvalue=stream)
 widget_control,droppt_wid,set_droplist_select=0
 textpt_wid=widget_text(datasel_wid,event_pro='ww_setptsel', $
	/editable,uvalue=stream)
 pt_dirs=values
endif
;
widget_control,datasel_wid,/realize
xmanager,'ww_indexsel',datasel_wid,event_handler='ww_indexsel_refresh',/no_block
case stream of
	'x':x_shown=x_shown+1
	'y':y_shown=y_shown+1
	'z':z_shown=z_shown+1
endcase
end
;-------------------------------------------------------------------------------
pro ww_indexsel_refresh,event
;
; Helper procedure to force refresh of widget when window manager doesn't.
; This fixes a problem which can occur when running IDL in virtual machines.
;
common LocalWwIndexselRefresh,offsety,offseti,offsetd
;
; if !owner ne 'chummel' then return
;
g=widget_info(event.id,/geometry)
;
if n_elements(offseti) eq 0 then begin
	offseti=0
	offsetd=g.yoffset-offsety
endif
;
if abs(g.yoffset-offsety-offsetd) gt 2 then offsety=g.yoffset-offsetd
widget_control,event.id,xoffset=g.xoffset,yoffset=offsety+(offseti mod 2)
offseti=offseti+1
;
end
;-------------------------------------------------------------------------------
pro ww_indexseldestroyed,wid
;
common DataSelWids,x_wid,x_shown,y_wid,y_shown,z_wid,z_shown
;
widget_control,wid,get_uvalue=stream
case stream of
	'x':x_shown=x_shown-1
	'y':y_shown=y_shown-1
	'z':z_shown=z_shown-1
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_indexib,event
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
widget_control,event.id,get_uvalue=stream
case stream of
	'x':i=ds_x.ib
	'y':i=ds_y.ib
	'z':i=ds_z.ib
endcase
print,genconfig.stationid(i)+' ('+strjoin(genconfig.stationid,', ')+')'
;
end
;-------------------------------------------------------------------------------
pro ww_indexob,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
widget_control,event.id,get_uvalue=stream
case stream of
	'x':i=ds_x.ob
	'y':i=ds_y.ob
	'z':i=ds_z.ob
endcase
print,genconfig.baselineid(0:genconfig.numbaseline(i)-1,i)
;
; With CalibrateSystem GUI, display selected fit results
if strpos(type,'APDFlux') ge 0 then begin
	print,'genconfig.apdflux for this output beam:'
	print,genconfig.apdflux(*,*,i)
endif
if strpos(type,'V2Bias') ge 0 then begin
	configs=strarr(n_elements(scanconfig(/starlist)))
	stars=configs
	for j=0,n_elements(scanconfig(/starlist))-1 do begin
        	words=nameparse(genconfig.config(j))
        	configs(j)=words(0)
		stars(j)=words(1)
	endfor
	index=where(configs eq y_dir.pt,count)
	if count gt 0 then begin
	print,'Non-zero values of genconfig.v2bias for this output beam:'
	for j=0,count-1 do begin
		values=transpose(genconfig.v2bias(*,*,*,i,index(j)))
		if total(abs(values)) gt 0 then begin
		print,stars(index(j))+' ('+configs(index(j))+')'
		print,values
		endif
	endfor
	endif
endif
if strpos(type,'TrackOffset') ge 0 then begin
	print,'genconfig.offsetcal for this output beam:'
	for bl=0,genconfig.numbaseline(i)-1 do begin
		print,genconfig.baselineid(bl,i)
		for j=0,genconfig.numspecchan(i)-1 do $
			print,genconfig.offsetcal(*,bl,j,i)
	endfor
endif
;
end
;-------------------------------------------------------------------------------
pro ww_indextr,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
widget_control,event.id,get_uvalue=stream
case stream of
	'x':i=ds_x.tr
	'y':i=ds_y.tr
	'z':i=ds_z.tr
endcase
print,genconfig.baselineid(genconfig.triplebase(*,i),genconfig.triplebeam(*,i)) $
	+'('+string(genconfig.triplebeam(*,i)+1,format='(i1)')+')'
;
; With CalibrateSystem GUI, display selected fit results
if strpos(type,'TABias') ge 0 then begin
	configs=strarr(n_elements(scanconfig(/starlist)))
	for j=0,n_elements(scanconfig(/starlist))-1 do begin
        	words=nameparse(genconfig.config(j))
        	configs(j)=words(0)
	endfor
	index=where(configs eq y_dir.pt,count)
	if count gt 0 then begin
	print,'genconfig.tabias for this triple:'
	for j=0,count-1 do print,transpose(genconfig.tabias(*,*,i,index(j)))
	endif
endif
;
end
;-------------------------------------------------------------------------------
pro ww_setinbeam,event
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
widget_control,event.id,get_uvalue=stream
case stream of
	'x':ds_x.ib=event.index
	'y':ds_y.ib=event.index
	'z':ds_z.ib=event.index
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setoutbeam,event
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
widget_control,event.id,get_uvalue=stream
case stream of
	'x':ds_x.ob=event.index
	'y':ds_y.ob=event.index
	'z':ds_z.ob=event.index
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_settriple,event
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
widget_control,event.id,get_uvalue=s_t
stream=strmid(s_t(0),0,1)
triples=long(strmid(s_t,1,max(strlen(s_t))))-1
case stream of
	'x':ds_x.tr=triples(event.index)
	'y':ds_y.tr=triples(event.index)
	'z':ds_z.tr=triples(event.index)
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setchdir,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common LocalWwSetDir,ch_dirs,bl_dirs,pt_dirs
;
widget_control,event.id,get_uvalue=stream
case stream of
	'x':x_dir.ch=ch_dirs(event.index)
	'y':y_dir.ch=ch_dirs(event.index)
	'z':z_dir.ch=ch_dirs(event.index)
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setbldir,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common LocalWwSetDir,ch_dirs,bl_dirs,pt_dirs
;
widget_control,event.id,get_uvalue=stream
case stream of
	'x':x_dir.bl=bl_dirs(event.index)
	'y':y_dir.bl=bl_dirs(event.index)
	'z':z_dir.bl=bl_dirs(event.index)
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setptdir,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common LocalWwSetDir,ch_dirs,bl_dirs,pt_dirs
;
widget_control,event.id,get_uvalue=stream
case stream of
	'x':x_dir.pt=pt_dirs(event.index)
	'y':y_dir.pt=pt_dirs(event.index)
	'z':z_dir.pt=pt_dirs(event.index)
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setchsel,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common IndexSelDropWids,dropch_wid,dropbl_wid,droptr_wid,droppt_wid
;
widget_control,event.id,get_value=command,get_uvalue=stream
s=stringparse(command(0))
n=n_elements(s)
case stream of
	'x': begin
	     m=n_elements(x_sel.ch) & n=min([n,m])
	     x_sel.ch=0
	     x_sel.ch(0:n-1)=s(0:n-1)
	     end
	'y': begin
	     m=n_elements(y_sel.ch) & n=min([n,m])
	     y_sel.ch=0
	     y_sel.ch(0:n-1)=s(0:n-1)
	     end
	'z': begin
	     m=n_elements(z_sel.ch) & n=min([n,m])
	     z_sel.ch=0
	     z_sel.ch(0:n-1)=s(0:n-1)
	     end
endcase
if n lt n_elements(s) then print,'Warning: channel selection truncated!'
print,'Channel selection set.'
;
widget_control,dropch_wid,set_droplist_select=1
y_dir.ch='Selected'	; Assume for now that user selects for Y stream
;
end
;-------------------------------------------------------------------------------
pro ww_setblsel,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common IndexSelDropWids,dropch_wid,dropbl_wid,droptr_wid,droppt_wid
;
widget_control,event.id,get_value=command,get_uvalue=stream
s=stringparse(command(0))
n=n_elements(s)
case stream of
	'x': begin
	     m=n_elements(x_sel.bl) & n=min([n,m])
	     x_sel.bl=0
	     x_sel.bl(0:n-1)=s(0:n-1)
	     end
	'y': begin
	     m=n_elements(y_sel.bl) & n=min([n,m])
	     y_sel.bl=0
	     y_sel.bl(0:n-1)=s(0:n-1)
	     end
	'z': begin
	     m=n_elements(z_sel.bl) & n=min([n,m])
	     z_sel.bl=0
	     z_sel.bl(0:n-1)=s(0:n-1)
	     end
endcase
if n lt n_elements(s) then print,'Warning: baseline selection truncated!'
print,'Baseline selection set.'
;
widget_control,dropbl_wid,set_droplist_select=1
y_dir.bl='Selected'	; Assume for now that user selects for Y stream
;
end
;-------------------------------------------------------------------------------
pro ww_settrsel,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common IndexSelDropWids,dropch_wid,dropbl_wid,droptr_wid,droppt_wid
;
widget_control,event.id,get_value=command,get_uvalue=stream
s=stringparse(command(0))
n=n_elements(s)
case stream of
	'x': begin
	     m=n_elements(x_sel.tr) & n=min([n,m])
	     x_sel.tr=0
	     x_sel.tr(0:n-1)=s(0:n-1)
	     end
	'y': begin
	     m=n_elements(y_sel.tr) & n=min([n,m])
	     y_sel.tr=0
	     y_sel.tr(0:n-1)=s(0:n-1)
	     end
	'z': begin
	     m=n_elements(z_sel.tr) & n=min([n,m])
	     z_sel.tr=0
	     z_sel.tr(0:n-1)=s(0:n-1)
	     end
endcase
if n lt n_elements(s) then print,'Warning: baseline selection truncated!'
print,'Triple selection set.'
;
widget_control,droptr_wid,set_droplist_select=1
y_dir.tr='Selected'	; Assume for now that user selects for Y stream
;
end
;-------------------------------------------------------------------------------
pro ww_setptsel,event
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common IndexSelDropWids,dropch_wid,dropbl_wid,droptr_wid,droppt_wid
;
widget_control,event.id,get_value=command,get_uvalue=stream
s=stringparse(command(0))
n=n_elements(s)
case stream of
	'x': begin
	     m=n_elements(x_sel.pt) & n=min([n,m])
	     x_sel.pt=0
	     x_sel.pt(0:n-1)=s(0:n-1)
	     end
	'y': begin
	     m=n_elements(y_sel.pt) & n=min([n,m])
	     y_sel.pt=0
	     y_sel.pt(0:n-1)=s(0:n-1)
	     end
	'z': begin
	     m=n_elements(z_sel.pt) & n=min([n,m])
	     z_sel.pt=0
	     z_sel.pt(0:n-1)=s(0:n-1)
	     end
endcase
if n lt n_elements(s) then print,'Warning: point/scan selection truncated!'
print,'Point/scan selection set.'
;
widget_control,droppt_wid,set_droplist_select=1
y_dir.pt='Selected'	; Assume for now that user selects for Y stream
;
end
;************************************************************************Block 4
pro ww_plotorbit
;
; Plots orbit positions from interferometric or speckle data.
;
common WwPlotOrbit,plotorbit_wid,plotorbit_shown
common PlotOrbit,plotorbit_options
;
; Check status of currently displayed widgets
if n_elements(plotorbit_shown) eq 0 then plotorbit_shown=0
if plotorbit_shown then return
;
; Initialize options
if init_plotorbit() ne 0 then return
;
; Check data
list_poscomps,components
if strlen(components(0)) eq 0 then return
;
; Create the widget
plotorbit_wid=widget_base(kill_notify='ww_plotorbitdestroyed', $
	resource_name='oyster', $
	/column,title='PlotAstrometry',xoffset=!dxsize-471,yoffset=!dysize-293)
;
; Component selection
opmenu_wid=widget_droplist(plotorbit_wid,title='Component: ', $
	event_pro='ww_plotorbitcomp',value=components,uvalue=components)
plotorbit_options.component=components(0)
;
; Window size selection
wsizes=['400x400','600x600','800x800','1000x1000']
opmenusz_wid=widget_droplist(plotorbit_wid,title='Window:    ', $
	event_pro='ww_setwsize',value=wsizes,uvalue=wsizes)
!xsize=400
!ysize=400
;
; Menu selection
panel=widget_base(plotorbit_wid,/row)
utils=widget_base(panel,/column)
plot_menu=['1\Plot\ww_plotorbitok', $
	   '0\Screen', $
	   '0\File']
plot_menu_wid=cw_pdmenu(utils,plot_menu)
util_menu=['1\Util\ww_plotorbitutil', $
	   '0\Identify', $
	   '0\Window', $
	   '0\Range', $
	   '0\Ellipse', $
	   '0\Thiele', $
	   '0\Innes', $
	   '0\Date', $
	   '2\Edit']
util_menu_wid=cw_pdmenu(utils,util_menu)
clear_button_wid=widget_button(utils,value='Clear',event_pro='ww_plotorbithelp')
help_button_wid=widget_button(utils,value='HELP',event_pro='ww_plotorbithelp')
;
options=['Errors', $
	 'Flagged', $
	 'Ellipse', $
	 'Orbit', $
	 'Lines', $
	 'J2000', $
	 'Color']
option_wid=cw_bgroup(panel,column=2,options,/nonexclusive, $
 			event_funct='ww_plotorbitoptions')
;
; Display orbit widget
widget_control,plotorbit_wid,/realize
xmanager,'ww_plotorbit',plotorbit_wid,/no_block
plotorbit_shown=plotorbit_shown+1
;
; Reset plot ranges
!x.range=0
!y.range=0
!z.range=0
;
end
;-------------------------------------------------------------------------------
pro ww_plotorbitdestroyed,wid
;
common WwPlotorbit,plotorbit_wid,plotorbit_shown
;
plotorbit_shown=plotorbit_shown-1
;
end
;-------------------------------------------------------------------------------
pro ww_plotorbitcomp,event
;
common PlotOrbit,plotorbit_options
;
widget_control,event.id,get_uvalue=components
plotorbit_options.component=components(event.index)
;
end
;-------------------------------------------------------------------------------
function ww_plotorbitoptions,event
;
common PlotOrbit,plotorbit_options
;
case event.value of
	0: plotorbit_options.e=event.select
	1: plotorbit_options.f=event.select
	2: plotorbit_options.l=event.select
	3: plotorbit_options.o=event.select
	4: plotorbit_options.t=event.select
	5: plotorbit_options.j=event.select
	6: plotorbit_options.c=event.select
	7: plotorbit_options.s=event.select
endcase
end
;-------------------------------------------------------------------------------
pro ww_plotorbithelp,event
;
widget_control,event.id,get_value=command
case command of
	'Clear':begin
	      	while !d.window ge 0 do wdelete,!d.window
		end
	'HELP':	begin
		print,'___________________________________________________'
		print,'Identify: print information about measurements.'
		print,'Window  : define plot region in window display'
		print,'Range   : define plot region limits manually'
		print,'Ellipse : interactively define an ellipse'
		print,'Thiele  : Estimate orbital parameters from ellipse'
		print,'Innes   : Estimate orbital parameters from ellipse'
		print,'Date    : Display secondary position given a date'
		print,'Edit    : Edit data and update plot'
		print,'_______________________***_________________________'
		end
endcase
end
;-------------------------------------------------------------------------------
pro ww_plotorbitok,event
;
common PlotOrbit,plotorbit_options
;
widget_control,event.id,get_value=command
;
case command of
	'Screen':begin
		 set_screen
		 plotorbit
		 end
	'File'  :begin
;		 set_ps
	 	 words=nameparse(systime())
	 	 filename='Orbit_'+strjoin(words(1:3),'-')+'.eps'
	 	 print,'Printing to file '+filename+'...'
	 	 set_ps,filename
		 plotorbit
		 device,/close_file
		 device,/portrait
        	 set_plot,!display
		 if plotorbit_options.s then begin
		 if strpos(strupcase(getenv('HOST')),'FERMION') ne -1 then begin
			if plotorbit_options.c $
				then spawn,'lpr -h -Plp0 idl.eps' $
				else spawn,'lpr -h idl.eps'
		 endif else begin
			if plotorbit_options.c $
				then spawn,'lpr -h -Pqmscolor idl.eps' $
				else spawn,'lpr -h idl.eps'
		 endelse
		 print,'Plot submitted.'
		 wait,3
		 spawn,'rm -f idl.eps'
       	  	 endif else print,'Plot saved.'
		 end
	else    :print,'***Error(WW_PLOTORBOK): choice not available!'
endcase
;
end		
;-------------------------------------------------------------------------------
pro ww_plotorbitutil,event
;
common PlotOrbitDataInfo,plotscans,plotdata_x,plotdata_y
common PlotOrbit,plotorbit_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
widget_control,event.id,get_value=command
case command of
'Identify':begin
           if !d.window eq -1 then begin
		marquardtdata,y,ysig,ymod
		chisq=((y-ymod)/ysig)^2
		if n_elements(chisq) eq n_elements(positions) then begin
		   i=where(chisq gt -3,count)
		   for j=0,count-1 do print,positions(i(j)).date,chisq(i(j))
		endif
		return
	   endif
	   if n_elements(binary_model) ne 0 then begin
	   index=where(binary_model.component eq plotorbit_options.component)
	   epoch=binary_model(index).epoch+2440000.d0
	   period=binary_model(index).period
	   endif
           repeat begin
		icom=set_points(plotdata_x,plotdata_y,index)
		if n_elements(binary_model) ne 0 then begin
			phase=(jy2jd(plotscans(index).jy)-epoch)/period mod 1
			if phase lt 0 then phase=phase+1
		endif else phase=-1
                print,'_____________________________'
                print,'NPOI Date : ',jd2date(2440000.d0+plotscans(index).jd $
			+double(system_config('NPOI','MIDNIGHT'))/24)
                print,'UT Date is: ',plotscans(index).date
		print,'Julian Year: ',plotscans(index).jy
		if phase ne -1 then $
		print,'Orbital phase: ',string(phase,format='(f14.4)')
                print,'____________***______________'
           endrep until icom eq 4
           end
'Window'  :begin
           if !d.window eq -1 then begin
                print,'***Error(WW_PLOTORBITUTIL): no plot window open!'
                return
           endif
           icom=set_boxes(box_x,box_y)
           if icom eq 4 then begin
                icom=set_boxes(box_x,box_y,/clear)
                !x.range=0
                !y.range=0
		print,'Window reset.'
           endif else begin
                !x.range=[box_x(0),box_x(1)]
                !y.range=[box_y(3),box_y(4)]
		print,'Window set.'
           endelse
           end
'Range'   :begin
	   read,ymin,ymax,prompt='Enter Y min,max (auto=0,0): '
           !y.range=[ymin,ymax]
	   read,xmin,xmax,prompt='Enter X min,max (auto=0,0): '
           !x.range=[xmin,xmax]
           print,'Range set.'
           end
'Ellipse' :begin
	   if !d.window eq -1 then begin
                print,'***Error(WW_PLOTORBITUTIL): no plot window currently open!'
                return
           endif
	   e_parms=set_ellipse()
	   end
'Thiele'  :o_parms=thiele()
'Innes'   :o_parms=innes()
'Date'	  :begin
           if !d.window eq -1 then begin
               print,'***Error(WW_PLOTORBITUTIL): no plot window open!'
               return
           endif
	   if n_elements(binary_model) eq 0 then begin
		   print,'No binary model loaded!'
		   return
	   endif
 	   date=''
	   read,date,prompt='Enter date (YYYY-MM-DD.f): '
	   parsedate,date,y,m,d,h
	   RAD=180/pi_circle
	   rt=binarypos(julian(y,m,d)+h/24,plotorbit_options.component)
	   oplot,[0,rt(0)*sin(rt(1)/RAD)],[0,rt(0)*cos(rt(1)/RAD)],psym=-3
	   end
'Edit'    :begin
	   repeat begin
        	icom=set_points(plotdata_x,plotdata_y,i)
        	positions(i).rho=-positions(i).rho
        	if positions(i).rho gt 0 then !p.color=tci(1) else !p.color=tci(2)
        	if plotorbit_options.e then begin
                	r=ellipse(positions(i).emajor,positions(i).eminor, $
                        	p+positions(i).pa)
                	x=r*cos(p)+ra(i)+positions(i).ex
                	y=r*sin(p)+dec(i)+positions(i).ey
                	oplot,x,y,psym=3
        	endif else begin
                	x=l*sin(p)
                	y=l*cos(p)
                	usersym,x,y,/fill
                	oplot,[ra(i),ra(i)],[dec(i),dec(i)],psym=8
        	endelse
 	    endrep until icom eq 4
	    end
;
endcase
;
end
;************************************************************************Block 5
pro ww_plotvel
;
; Plots radial velocities from spectroscopy.
;
common WwPlotVel,plotvel_wid,plotvel_shown
common PlotVel,plotvel_options
;
; Check status of currently displayed widgets
if n_elements(plotvel_shown) eq 0 then plotvel_shown=0
if plotvel_shown then return
;
; Initialize options
if init_plotvel() ne 0 then return
;
; Check data
list_velcomps,components
if strlen(components(0)) eq 0 then return
;
; Create the widget
plotvel_wid=widget_base(kill_notify='ww_plotveldestroyed', $
	resource_name='oyster', $
	/column,title='PlotSpectroscopy',xoffset=!dxsize-236,yoffset=!dysize-293)
;
; Component selection
opmenu_wid=widget_droplist(plotvel_wid,title='Component: ', $
	event_pro='ww_plotvelcomp',value=components,uvalue=components)
plotvel_options.component=components(0)
;
; Window size selection
wsizes=['400x400','600x600','800x800','1000x1000']
opmenusz_wid=widget_droplist(plotvel_wid,title='Window:    ', $
	event_pro='ww_setwsize',value=wsizes,uvalue=wsizes)
!xsize=400
!ysize=400
;
; Menu selection
panel=widget_base(plotvel_wid,/row)
utils=widget_base(panel,/column)
plot_menu=['1\Plot\ww_plotvelok', $
	   '0\Screen', $
	   '0\File']
plot_menu_wid=cw_pdmenu(utils,plot_menu)
util_menu=['1\Util\ww_plotvelutil', $
	   '0\Identify', $
	   '0\Window', $
	   '0\Range', $
	   '0\Date', $
	   '2\Edit']
util_menu_wid=cw_pdmenu(utils,util_menu)
clear_button_wid=widget_button(utils,value='Clear',event_pro='ww_plotvelhelp')
help_button_wid=widget_button(utils,value='HELP',event_pro='ww_plotvelhelp')
;
options=['Errors', $
	 'Flagged', $
	 'Phase', $
	 'Orbit', $
	 'Color', $
	 'Paper', $
	 'All in 1', $
	 'Residuals']
option_wid=cw_bgroup(panel,column=2,options,/nonexclusive, $
 			event_funct='ww_plotveloptions')
;
; Display vel widget
widget_control,plotvel_wid,/realize
xmanager,'ww_plotorbit',plotvel_wid,/no_block
plotvel_shown=plotvel_shown+1
;
; Reset plot ranges
!x.range=0
!y.range=0
!z.range=0
;
end
;-------------------------------------------------------------------------------
pro ww_plotveldestroyed,wid
;
common WwPlotVel,plotvel_wid,plotvel_shown
;
plotvel_shown=plotvel_shown-1
;
end
;-------------------------------------------------------------------------------
pro ww_plotvelcomp,event
;
common PlotVel,plotvel_options
;
widget_control,event.id,get_uvalue=components
plotvel_options.component=components(event.index)
;
end
;-------------------------------------------------------------------------------
function ww_plotveloptions,event
;
common PlotVel,plotvel_options
;
case event.value of
	0: plotvel_options.e=event.select
	1: plotvel_options.f=event.select
	2: plotvel_options.p=event.select
	3: plotvel_options.o=event.select
	4: plotvel_options.c=event.select
	5: plotvel_options.b=event.select
	6: plotvel_options.a=event.select
	7: plotvel_options.r=event.select
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_plotvelhelp,event
;
widget_control,event.id,get_value=command
case command of
	'Clear':begin
	      	while !d.window ge 0 do wdelete,!d.window
		end
	'HELP':	begin
		print,'___________________________________________________'
		print,'_______________________***_________________________'
		end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_plotvelok,event
;
common PlotVel,plotvel_options
;
widget_control,event.id,get_value=command
;
case command of
	'Screen':begin
;		 set_plot,!display
		 set_screen
		 plotvel
		 end
	'File'  :begin
;		 set_ps
	 	 words=nameparse(systime())
	 	 filename='RVels_'+strjoin(words(1:3),'-')+'.eps'
	 	 print,'Printing to file '+filename+'...'
	 	 set_ps,filename
		 plotvel
		 device,/close_file
		 device,/portrait
        	 set_plot,!display
		 if plotvel_options.s then begin
		 if strpos(strupcase(getenv('HOST')),'FERMION') ne -1 then begin
			if plotvel_options.c $
				then spawn,'lpr -h -Plp0 idl.eps' $
				else spawn,'lpr -h idl.eps'
		 endif else begin
			if plotvel_options.c $
				then spawn,'lpr -h -Pqmscolor idl.eps' $
				else spawn,'lpr -h idl.eps'
		 endelse
		 print,'Plot submitted.'
		 wait,3
		 spawn,'rm -f idl.eps'
       	  	 endif else print,'Plot saved.
		 end
	else    :print,'***Error(WW_PLOTVELOK): choice not available!'
endcase
;
end		
;-------------------------------------------------------------------------------
pro ww_plotvelutil,event
;
common PlotVel,plotvel_options
common PlotVelDataInfo,plotscans,plotdata_x,plotdata_y
common ScanData,scans,bgscans,bufferinfo,positions,velocities
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common FitAstrometry,ellipse_options,vel_options,e_parms,o_parms
;
widget_control,event.id,get_value=command
;
case command of
'Identify':begin
           repeat begin
		icom=set_points(plotdata_x,plotdata_y,index)
                print,'_____________________________'
                print,'Date is: ',plotscans(index).date
		print,'JD: ',plotscans(index).jd+2440000l,format='(a,f12.4)'
                print,'____________***______________'
           endrep until icom eq 4
           end
'Window'  :begin
           icom=set_boxes(box_x,box_y)
           if icom eq 4 then begin
                icom=set_boxes(box_x,box_y,/clear)
                !x.range=0
                !y.range=0
           endif else begin
                !x.range=[box_x(0),box_x(1)]
                !y.range=[box_y(3),box_y(4)]
           endelse
           end
'Date'	  :begin
           if !d.window eq -1 then begin
                print,'***Error(WW_PLOTVELITUTIL): no plot window open!'
                return
           endif
	   if plotvel_options.p then begin
 	   date=''
	   read,date,prompt='Enter date (YYYY-MM-DD): '
	   parsedate,date,y,m,d
           ci=strpos(binary_model.component,plotvel_options.component)
           i=where(ci ge 0)
           components=binary_model(i).component
           periods=binary_model(i).period
           si=sort(strlen(components))
           components=components(si)
           periods=periods(si)
           j=where(periods ne 0) & j=j(0)
           i=where(binary_model.component eq components(j)) & i=i(0)
           x=(julian(y,m,d)-binary_model(i).epoch-2440000.d0)/binary_model(i).period $
                   mod 1
	   oplot,[x,x],[-200,200],psym=-1,color=tci(3)
	   endif
	   end
'Range'   :begin
	   read,ymin,ymax,prompt='Enter Y min,max (auto=0,0): '
           !y.range=[ymin,ymax]
	   read,xmin,xmax,prompt='Enter X min,max (auto=0,0): '
           !x.range=[xmin,xmax]
           print,'Range set.'
           end
else	  :print,'Choice not available!'
;
endcase
;
end
;************************************************************************Block 6
pro ww_plotmag
;
; Plots magnitudes from photometry.
;
common WwPlotMag,plotmag_wid,plotmag_shown
common PlotMag,plotmag_options
;
; Check status of currently displayed widgets
if n_elements(plotmag_shown) eq 0 then plotmag_shown=0
if plotmag_shown then return
;
; Initialize options
if init_plotmag() ne 0 then return
;
; Check data
list_magcomps,components
if strlen(components(0)) eq 0 then return
;
; Create the widget
plotmag_wid=widget_base(kill_notify='ww_plotmagdestroyed', $
	resource_name='oyster', $
	/column,title='PlotPhotometry',xoffset=0,yoffset=0)
;
; Component selection
opmenu_wid=widget_droplist(plotmag_wid,title='Component: ', $
	event_pro='ww_plotmagcomp',value=components,uvalue=components)
plotmag_options.component=components(0)
;
; Filter selection
list_filters,filters
opmenu_wid=widget_droplist(plotmag_wid,title='Filters:', $
	event_pro='ww_plotmagfilter',value=filters,uvalue=filters)
plotmag_options.filter=filters(0)
;
; Window size selection
wsizes=['400x400','600x600','800x800','1000x1000']
opmenusz_wid=widget_droplist(plotmag_wid,title='Window:    ', $
	event_pro='ww_setwsize',value=wsizes,uvalue=wsizes)
!xsize=400
!ysize=400
;
; Menu selection
panel=widget_base(plotmag_wid,/row)
utils=widget_base(panel,/column)
plot_menu=['1\Plot\ww_plotmagok', $
	   '0\Screen', $
	   '0\File']
plot_menu_wid=cw_pdmenu(utils,plot_menu)
util_menu=['1\Util\ww_plotmagutil', $
	   '0\Identify', $
	   '0\Window', $
	   '0\Range', $
	   '2\Edit']
util_menu_wid=cw_pdmenu(utils,util_menu)
clear_button_wid=widget_button(utils,value='Clear',event_pro='ww_plotmaghelp')
help_button_wid=widget_button(utils,value='HELP',event_pro='ww_plotmaghelp')
;
options=['Errors', $
	 'Flagged', $
	 'Phase', $
	 'Orbit', $
	 'Color', $
	 'Paper', $
	 'All in 1', $
	 'Residuals']
option_wid=cw_bgroup(panel,column=2,options,/nonexclusive, $
 			event_funct='ww_plotmagoptions')
;
; Display mag widget
widget_control,plotmag_wid,/realize
xmanager,'ww_plotmag',plotmag_wid,/no_block
plotmag_shown=plotmag_shown+1
;
; Reset plot ranges
!x.range=0
!y.range=0
!z.range=0
;
end
;-------------------------------------------------------------------------------
pro ww_plotmagdestroyed,wid
;
common WwPlotMag,plotmag_wid,plotmag_shown
;
plotmag_shown=plotmag_shown-1
;
end
;-------------------------------------------------------------------------------
pro ww_plotmagcomp,event
;
common PlotMag,plotmag_options
;
widget_control,event.id,get_uvalue=components
plotmag_options.component=components(event.index)
;
end
;-------------------------------------------------------------------------------
function ww_plotmagoptions,event
;
common PlotMag,plotmag_options
;
case event.value of
	0: plotmag_options.e=event.select
	1: plotmag_options.f=event.select
	2: plotmag_options.p=event.select
	3: plotmag_options.o=event.select
	4: plotmag_options.c=event.select
	5: plotmag_options.b=event.select
	6: plotmag_options.a=event.select
	7: plotmag_options.r=event.select
endcase
end
;-------------------------------------------------------------------------------
pro ww_plotmagfilter,event
;
common PlotMag,plotmag_options
;
widget_control,event.id,get_uvalue=filters
plotmag_options.filter=filters(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_plotmaghelp,event
;
widget_control,event.id,get_value=command
case command of
	'Clear':begin
		wdall
		end
	'HELP':	begin
		print,'___________________________________________________'
		print,'_______________________***_________________________'
		end
endcase
end
;-------------------------------------------------------------------------------
pro ww_plotmagok,event
;
common PlotMag,plotmag_options
;
widget_control,event.id,get_value=command
case command of
	'Screen':begin
;		 set_plot,!display
		 set_screen
		 plotmag
		 end
	'File'  :begin
;		 set_ps
	 	 words=nameparse(systime())
	 	 filename='Photo_'+strjoin(words(1:3),'-')+'.eps'
	 	 print,'Printing to file '+filename+'...'
	 	 set_ps,filename
		 plotmag
		 device,/close_file
		 device,/portrait
        	 set_plot,!display
		 if plotmag_options.s then begin
		 if strpos(strupcase(getenv('HOST')),'FERMION') ne -1 then begin
			if plotmag_options.c $
				then spawn,'lpr -h -Plp0 idl.eps' $
				else spawn,'lpr -h idl.eps'
		 endif else begin
			if plotvel_options.c $
				then spawn,'lpr -h -Pqmscolor idl.eps' $
				else spawn,'lpr -h idl.eps'
		 endelse
		 print,'Plot submitted.'
		 wait,3
		 spawn,'rm -f idl.eps'
       	  	 endif else print,'Plot saved.'
		 end
	else    :print,'***Error(WW_PLOTVELOK): choice not available!'
endcase
;
end		
;-------------------------------------------------------------------------------
pro ww_plotmagutil,event
;
common PlotMagDataInfo,plotscans,plotdata_x,plotdata_y
common PlotMag,plotmag_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities
common FitAstrometry,ellipse_options,mag_options,e_parms,o_parms
;
widget_control,event.id,get_value=command
case command of
'Identify':begin
           repeat begin
		icom=set_points(plotdata_x,plotdata_y,index)
                print,'_____________________________'
                print,'Date is: ',plotscans(index).date
		print,'JD: ',plotscans(index).jd+2440000l,format='(a,f12.4)'
                print,'____________***______________'
           endrep until icom eq 4
           end
'Window'  :begin
           icom=set_boxes(box_x,box_y)
           if icom eq 4 then begin
                icom=set_boxes(box_x,box_y,clear)
                !x.range=0
                !y.range=0
           endif else begin
                !x.range=[box_x(0),box_x(1)]
                !y.range=[box_y(3),box_y(4)]
           endelse
           end
'Range'   :begin
	   read,ymin,ymax,prompt='Enter Y min,max (auto=0,0): '
           !y.range=[ymin,ymax]
	   read,xmin,xmax,prompt='Enter X min,max (auto=0,0): '
           !x.range=[xmin,xmax]
           print,'Range set.'
           end
else	  :print,'Choice not available!'
;
endcase
;
end
;-------------------------------------------------------------------------------
