;*******************************************************************************
; File: mainwidget.pro
;
; Description:
; ------------
; Container of IDL main widget scripts for oyster. 
;
; Block directory:
; ----------------
;
; Block 1: oyster,iconify_oyster
;
; CHAMELEON:
;
; Block 2: ww_file,ww_fileselect,ww_fileselectfilter,ww_fileselectdir,
;	   ww_fileselectfile,ww_fileselectenter,ww_fileselectok
; Block 3: ww_access,ww_load,ww_write,
;	   ww_browse,ww_browsedestroyed,ww_browsefind,
;	   ww_loadstar,ww_loadstardestroyed,ww_loadstarsel,ww_loadstarok,
;          ww_loadmetrosel,ww_loadmetrodestroyed,ww_loadmetro
; Block 4: ww_reduce,ww_reducept,ww_reducebg,ww_reducedl,
;	   ww_listflagreasons,ww_unflag,
;          ww_triplesel,ww_tripledestroyed,ww_triple
; Block 5: ww_calibrate,ww_scans,ww_astrom,
;	   ww_astromsolve,ww_astromdestroyed,
;	   ww_setstations,ww_setfitstars,ww_setfitnights,ww_setastrodata,
;	   ww_setastroparms,ww_setrefstation,
;	   ww_astromfit
;	   ww_pivot,ww_pivotdestroyed,ww_pivotcorr
; Block 6: ww_utilities,ww_starinfo,ww_listdefault
;
; Block 7: ww_cal,ww_setcalvar,ww_caldestroyed,ww_calindicator,ww_calfunction,
;	   ww_caloptions,ww_calok,ww_listcalreasons,ww_editcal
;
; AMOEBA:
;
; Block 2: ww_amoebadata
; Block 3: ww_amoebamodel,ww_modelfile
; Block 4: ww_amoebaplot
; Block 5: ww_amoebafit
; Block 6: ww_volvox
; Block 7: ww_oyster,ww_quit,
; Block 8: ww_datasel,ww_dataseldestroyed,
;	   ww_setfilter,ww_setfileselfile,ww_setfiles,ww_setfiledir,
;	   ww_datweight,ww_dattype,
;          ww_chaselok,ww_psnselok,ww_velselok,ww_magselok,
;	   ww_obsselok,ww_calselok,c2osel_ok
;
; VOLVOX:
;
; Block 1: ww_volvoxsolve,
;	   ww_volvoxfit
;
;
;************************************************************************Block 1
pro oyster
;
; Main widget procedure for OYSTER software.
;
common OysterWids,top_wid
;
if n_elements(top_wid) eq 0 then top_wid=0L
if top_wid ne 0 then begin
	print,'Widget already displayed!'
	return
endif
top_wid=widget_base(title='oisdr',resource_name='oyster',/column, $
	kill_notify='ww_quit', $
	xoffset=1,yoffset=1)
;
button_wid=widget_button(top_wid,event_pro='ww_oyster',uvalue='Help', $
	/bitmap,value=!oyster_dir+'oisdr.bmp',/frame)
row_wid=widget_base(top_wid,/row)	; optional
chameleon_wid=widget_base(top_wid,/column,/frame,/base_align_center)
amoeba_wid=widget_base(top_wid,/column,/frame,/base_align_center)
; oyster_wid=widget_base(top_wid,/column,/frame,/base_align_center)
;
; CHAMELEON panel
file_menu=[ $
	'1\File\ww_file', $
			'0\OPEN', $
			'0\Close', $
			'2\ClrSt' $
	]
access_menu=[ $
	'1\Access\ww_access', $
			'0\Tree', $
			'0\BROWSE', $
			'1\Load\ww_access', $
					'0\INTERFEROMETRY', $
					'2\METROLOGY', $
			'1\Write\ww_write', $
					'0\HDS', $
					'0\XDR', $
					'0\OI-FITS', $
					'2\UV-FITS' $
	]
reduce_menu=[ $
	'1\Reduce\ww_reduce', $
			'1\BG Data\ww_reducebg', $
				'0\PLOT', $ 
				'0\Process BG', $
				'0\Median BG', $
				'0\Expand BG', $
				'0\Default BG', $
				'7\Flagtable\ww_reducebg', $
						'0\Save', $
						'0\Load', $
						'0\Apply', $
						'0\Clear', $
						'4\UNFLAG', $
						'2\Reset', $
			'1\Point Data', $
				'1\Imaging\ww_reducept', $
					'0\PLOT', $
					'0\Unwrap triple', $
					'2\Clear bias', $
				'1\Astrometry\ww_reducedl', $
					'0\PLOT', $
					'0\Init', $
					'0\InitW', $
					'0\PhaseEdit', $
					'0\PhaseWrap', $
					'2\DispCorr', $
				'7\Flagtable\ww_reducept', $
						'0\Save', $
						'0\Load', $
						'0\Apply', $
						'0\Clear', $
						'4\UNFLAG', $
						'2\Reset', $
			'1\Inch Data\ww_reducemt', $
				'0\PLOT', $
				'0\Init', $
				'0\MetroCorr', $
				'0\AstroCorr', $
				'7\Flagtable\ww_reducemt', $
						'0\Save', $
						'0\Load', $
						'0\Apply', $
						'0\Clear', $
						'4\UNFLAG', $
						'2\Reset', $
			'6\Average' $
	]
calibrate_menu=[ $
	'1\Calibrate\ww_calibrate', $
			'0\SEEING', $
			'0\SYSTEM', $
			'0\PHOTOMETRY', $
			'1\Visibility\ww_scans', $
					'0\PLOT', $
					'0\CALIBRATE', $
					'0\FlagIncoh', $
					'0\SetErrors', $
					'0\SetComplex', $
					'3\Triple\ww_scans', $
							'0\RewrapTriple', $
							'0\DewrapTriple', $
							'2\ManualUnwrap', $
			'1\Astrometry\ww_astrom', $
					'0\PLOT', $
					'0\Calc', $
					'0\MCorr', $
					'2\SOLVE', $
			'7\Flagtable\ww_scans', $
					'0\Save', $
					'0\Load', $
					'0\Apply', $
					'0\Clear', $
					'4\UNFLAG', $
					'2\Reset' $
	]
utilities_menu=[ $
	'1\Utilities\ww_utilities', $
			'0\MEMO', $
			'0\OBSBASE', $
			'0\.CAL->.CHA', $
			'0\.CHA->.OIFITS', $
			'0\Load Night', $
			'0\Load StarTable', $
			'1\List\ww_utilities', $
					'0\Summary', $
					'0\Stars', $
					'0\Scans', $
					'0\BGScans', $
					'4\ScnLog', $
					'0\ObsLog', $
					'0\ConLog', $
					'4\STARS', $
					'2\Star', $
			'3\Forms\ww_utilities', $
					'0\ScanForms', $
					'2\SpecForms' $
	]
;
file_wid=cw_pdmenu(chameleon_wid,file_menu)
access_wid=cw_pdmenu(chameleon_wid,access_menu)
reduce_wid=cw_pdmenu(chameleon_wid,reduce_menu)
calibrate_wid=cw_pdmenu(chameleon_wid,calibrate_menu)
utilities_wid=cw_pdmenu(chameleon_wid,utilities_menu)
;
; AMOEBA panel
data_menu=[ $
	'1\Data\ww_amoebadata', $
			'0\INTERFEROMETRY', $
			'0\ASTROMETRY', $
			'0\SPECTROSCOPY', $
			'0\PHOTOMETRY', $
			'6\Summary' $
	]
model_menu=[ $
	'1\Model\ww_amoebamodel', $
			'0\READ', $
			'0\Info', $
			'0\Calc', $
			'6\FIT' $
	]
plot_menu=[ $
	'1\Plot\ww_amoebaplot', $
			'1\Interferometry\ww_amoebaplot', $
				'0\UV Plots', $
				'0\XY Plots', $
				'2\Astrometry', $
			'0\ASTROMETRY', $
			'0\SPECTROSCOPY', $
			'2\PHOTOMETRY' $
	]
fit_menu=[ $
	'1\Fit\ww_amoebafit', $
			'0\ASTROMETRY', $
			'0\INTERFEROMETRY', $
			'6\CONTROL' $
	]
astrometry_menu=[ $
	'1\Astrometry\ww_volvox', $
			'0\Plot', $
			'0\Calc', $
			'0\MCorr', $
			'0\WCorr', $
			'0\Get PMs', $
			'0\Get HIP', $
			'6\SOLVE' $
	]
;
data_wid=cw_pdmenu(amoeba_wid,data_menu)
model_wid=cw_pdmenu(amoeba_wid,model_menu)
plot_wid=cw_pdmenu(amoeba_wid,plot_menu)
fit_wid=cw_pdmenu(amoeba_wid,fit_menu)
astrometry_wid=cw_pdmenu(amoeba_wid,astrometry_menu)
;
; OYSTER panel
oyster_menu=[ $
	'1\OYSTER\ww_oyster', $
			'0\Help', $
			'0\Save session', $
			'0\Restore session', $
			'0\Help,/memory', $
			'2\Exit widget' $
	]
;
; Display main widget
widget_control,top_wid,/realize
xmanager,'oyster',top_wid,/no_block
;
; Set the screen size variables
r=get_screen_size()
!dxsize=r[0]
!dysize=r[1]
;
end
;-------------------------------------------------------------------------------
pro iconify_oyster,iconify
;
if n_elements(iconify) eq 0 then iconify=1
;
common OysterWids,top_wid
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common AmoebaWids,data_wid,mfit_wid,ifit_wid,afit_wid,control_wid
;
if n_elements(top_wid) eq 0 then top_wid=0L
if n_elements(plot_wid) eq 0 then plot_wid=0L
if n_elements(td_wid) eq 0 then td_wid=0L
if n_elements(cal_wid) eq 0 then cal_wid=0L
if n_elements(astrom_wid) eq 0 then astrom_wid=0L
if n_elements(pivot_wid) eq 0 then pivot_wid=0L
if n_elements(triple_wid) eq 0 then triple_wid=0L
if n_elements(data_wid) eq 0 then data_wid=0L
if n_elements(mfit_wid) eq 0 then mfit_wid=0L
if n_elements(ifit_wid) eq 0 then ifit_wid=0L
if n_elements(afit_wid) eq 0 then afit_wid=0L
if n_elements(control_wid) eq 0 then control_wid=0L
;
if widget_info(top_wid,/valid) ne 0 then widget_control,top_wid,iconify=iconify
if widget_info(plot_wid,/valid) ne 0 then widget_control,plot_wid,iconify=iconify
if widget_info(td_wid,/valid) ne 0 then widget_control,td_wid,iconify=iconify
if widget_info(cal_wid,/valid) ne 0 then widget_control,cal_wid,iconify=iconify
if widget_info(astrom_wid,/valid) ne 0 then widget_control,astrom_wid,iconify=iconify
if widget_info(pivot_wid,/valid) ne 0 then widget_control,pivot_wid,iconify=iconify
if widget_info(triple_wid,/valid) ne 0 then widget_control,triple_wid,iconify=iconify
if widget_info(data_wid,/valid) ne 0 then widget_control,data_wid,iconify=iconify
if widget_info(mfit_wid,/valid) ne 0 then widget_control,mfit_wid,iconify=iconify
if widget_info(ifit_wid,/valid) ne 0 then widget_control,ifit_wid,iconify=iconify
if widget_info(afit_wid,/valid) ne 0 then widget_control,afit_wid,iconify=iconify
if widget_info(control_wid,/valid) ne 0 then widget_control,control_wid,iconify=iconify
;
end
;************************************************************************Block 2
pro ww_file,event
;
; Dispatch form FILE button.
;
common Hds,path,hds_file_stub
;
; This is to tell the compiler that ww_fileselect is a function
forward_function ww_fileselect
;
widget_control,event.id,get_value=command
;
if n_elements(path) eq 0 then path=''
;
case command of
	'OPEN': begin
	   	if hds_state() ne 0 then begin
	     		print,'Please close current file first!'
	     		return
	   	endif
		r=dialog_message('Single-click only, please!', $
		  	resource_name='oyster')
		hds_file=ww_fileselect(title='HDS file selection', $
			path=path,filter='*.c??')
		mode='READ'
		if strlen(hds_file) ne 0 then begin
			hds_open,hds_file,mode,status 
			hds_file_stub=basename(hds_file)
		endif
	   	end
	'Close':begin
	   	if hds_state() eq 0 then begin
			print,'File already closed!'
			return
	   	endif
	   	toplevel
	   	dat_annul
	   	end
	'ClrSt':begin
	   	clearstatus
	   	print,'Status cleared.'
	   	toplevel
	   	end
endcase
;
end
;-------------------------------------------------------------------------------
function ww_fileselect,title=title,path=path,filter=filter
;
common WwFileSelect,listdir_wid,listfiles_wid,filefilter,file
common OysterWids,top_wid
;
if not keyword_set(title) then title=''
if not keyword_set(path) then path=pathname('')
if not keyword_set(filter) then filefilter='*' else filefilter=filter
;
old_dir=pathname('')
cd,path
;
file_wid=widget_base(/column,title=title, $
	resource_name='oyster',xoffset=0,yoffset=50)
;
label_wid=widget_label(file_wid,value='Filter')
text_wid=widget_text(file_wid,value=filefilter,xsize=20, $
	event_pro='ww_fileselectfilter',/editable)
;
row_wid=widget_base(file_wid,/row)
;
column1_wid=widget_base(row_wid,/column)
label_wid=widget_label(column1_wid,value='Directories')
directories=[!oyster_dir+'model',finddir(path)]
listdir_wid=widget_list(column1_wid,value=directories,uvalue=directories, $
	event_pro='ww_fileselectdir',xsize=30,ysize=6)
;
column2_wid=widget_base(row_wid,/column)
label_wid=widget_label(column2_wid,value='Files')
files=findfile(filter)
if strlen(files[0]) eq 0 then files=' '
listfiles_wid=widget_list(column2_wid,value=files,uvalue=files, $
	event_pro='ww_fileselectfile',xsize=15,ysize=6)
;
label_wid=widget_label(file_wid,value='Manual input, hit Enter to select')
text_wid=widget_text(file_wid,value=path,event_pro='ww_fileselectenter', $
	xsize=20,/editable)
;
button_wid=cw_bgroup(file_wid,/row,['OK','HELP','Cancel'], $
	event_funct='ww_fileselectok',/return_name)
;
widget_control,top_wid,sensitive=0
widget_control,file_wid,/realize
xmanager,'ww_fileselect',file_wid
;
path=pathname()
if strlen(strcompress(file,/remove_all)) ne 0 then $
	file=pathname('')+specname(file)
cd,old_dir
return,file
;
end
;-------------------------------------------------------------------------------
pro ww_fileselectfilter,event
;
common WwFileSelect,listdir_wid,listfiles_wid,filefilter,file
;
widget_control,event.id,get_value=filefilter & filefilter=filefilter[0]
;
path=pathname(filefilter)
if strlen(path) ne 0 then cd,path
;
widget_control,listdir_wid,set_value=finddir(), $
		           set_uvalue=finddir()
widget_control,listfiles_wid,set_value=findfile(specname(filefilter)), $
			     set_uvalue=findfile(specname(filefilter))
;
end
;-------------------------------------------------------------------------------
pro ww_fileselectdir,event
;
common WwFileSelect,listdir_wid,listfiles_wid,filefilter,file
;
if event.clicks ne 1 then return
widget_control,event.id,get_uvalue=dirs
cd,dirs[event.index]
wait,1	; Avoid segmentation fault if second button click occurs while updating
;
widget_control,listdir_wid,set_value=finddir(), $
		           set_uvalue=finddir()
widget_control,listfiles_wid,set_value=findfile(specname(filefilter)), $
			     set_uvalue=findfile(specname(filefilter))
;
end
;-------------------------------------------------------------------------------
pro ww_fileselectfile,event
;
common WwFileSelect,listdir_wid,listfiles_wid,filefilter,file
;
if event.clicks ne 1 then return
widget_control,event.id,get_uvalue=files
file=files[event.index]
;
end
;-------------------------------------------------------------------------------
pro ww_fileselectenter,event
;
common OysterWids,top_wid
common WwFileSelect,listdir_wid,listfiles_wid,filefilter,file
;
widget_control,event.id,get_value=file & file=file[0]
path=pathname(file)
if strlen(path) ne 0 then cd,path
widget_control,event.top,/destroy
wait,1	; To address a likelyhood for KDE crash on toucan
widget_control,top_wid,sensitive=1
;
end
;-------------------------------------------------------------------------------
function ww_fileselectok,event
;
common OysterWids,top_wid
common WwFileSelect,listdir_wid,listfiles_wid,filefilter,file
;
case event.value of
	'OK':	begin
		widget_control,event.top,/destroy
		widget_control,top_wid,sensitive=1
		end
	'HELP':	begin
		print,'_______________________________________________'
		print,'There are two ways to select a model file:'
		print,'1: Choose directory, then select file, click OK'
	        print,'2: Enter selection (incl. path) and hit Enter'	
		print,'___________________***_________________________'
		end
	'Cancel':begin
		file=''
		widget_control,event.top,/destroy
		widget_control,top_wid,sensitive=1
		end
endcase
;
end
;************************************************************************Block 3
pro ww_access,event
;
; Dispatch for ACCESS button.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common Starbase,StarTable,Notes
;
widget_control,event.id,get_value=command
;
case command of
'Tree'     	:begin
	    	 list_tree 
	    	 print,'Listing complete.'
	    	 end
'BROWSE'   	:ww_browse
'INTERFEROMETRY':begin
	    	 get_format,format
	    	 case format of
	    		'CONSTRICTOR':	begin 
	    	 			get_sysconfig
	    	 			get_scantable
	    				get_bgdata
					fixearlydays
					get_startable
	    	 			get_constrictorlog
					get_observerlog
					ww_loadstar,event
					end
	    		'CHAMELEON':	begin
					get_scandata
					end
			else:		begin
					print,'***Error(WW_ACCESS): '+ $
						'wrong format!'
					return
					end
	    	 endcase
	    	 read_notes
	    	 end
'METROLOGY'     :begin
		 get_format,format
		 case format of
			'INCHWORM':	begin
					if n_elements(GenConfig) eq 0 then $
					get_genconfig
					if n_elements(GeoParms) eq 0 then $
					get_geoparms
					if n_elements(ScanTable) eq 0 then $
					get_scantable
					if n_elements(StarTable) eq 0 then $
					get_startable
		 			ww_loadmetrosel,event
					end
			else:		begin
					print,'***Error(WW_ACCESS): '+ $
						'wrong format!'
					return
					end
		 endcase
		 end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_load,event
;
; Dispatch for Load(ACCESS) button.
;
widget_control,event.id,get_value=item
;
case item of
	'GeoParms' :get_geoparms
	'GenConfig':get_genconfig
	'Scantable':get_scantable
	'BG table' :get_BGTable
	'BG Scans' :get_bgscans
	'Scans'    :get_scans
	'ObsLog'   :get_observerlog
	'ConLog'   :get_constrictorlog
	'POINTS'   :ww_loadstar,event
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_write,event
;
; Dispatch for Write(ACCESS) button.
;
common Hds,path,hds_file_stub
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([1,5,6,7,8,9]) ne 0 then return
;
widget_control,event.id,get_value=command
;
if strlen(genconfig.configid) ne 0 $
	then configtag='_'+system_id(systemid)+'_'+genconfig.configid $
	else configtag=''
;
file_stub=genconfig.date+configtag	; as of April 10, 2017
;
if command eq 'OI-FITS' then begin
	put_oifits,file_stub
	return
endif
;
if command eq 'UV-FITS' then begin
	put_uvfits,file_stub
	return
endif
;
if command eq 'XDR' then begin
	put_xdr,file_stub
	return
endif
;
; Close current file
hds_close
;
filename=hds_file_stub+'.cha'
result=findfile(filename,count=fcount)
; unix_command='ls '+filename+'; echo $status'
; spawn,unix_command,result	; result(0)=0 if file found
print,'Open file '+filename
if fcount ne 0 then hds_open,filename,'UPDATE',status else $
	hds_new,filename,'DataSet','CHAMELEON',status
if status ne 0 then begin
	print,'***Error(WW_WRITE): file problem!'
	clearstatus
	return
endif
;
case command of
	'GeoParms' :put_geoparms
	'GenConfig':put_genconfig
	'ObsLog'   :put_observerlog
	'ConLog'   :put_constrictorlog
	'Scans'    :put_scans
	'HDS'	   :begin
		    put_scanlog
		    put_observerlog
		    put_constrictorlog
		    put_sysconfig
		    put_scans
		    end
endcase
hds_close
;
end
;-------------------------------------------------------------------------------
pro ww_browse
;
; Create list widget displaying the names of the components in current level.
;
common OysterWids,top_wid
common BrowseInfo,list_wids,item_wids,item_names,loc_id
common Hds,path,hds_file_stub
;
maxlevel=10
list_wids=lonarr(maxlevel)		; BROWSE list widget ids
item_wids=lonarr(maxlevel)		; BROWSE list item wids
item_names=sindgen(maxlevel)		; BROWSE list item values
for i=0,maxlevel-1 do item_names[i]='123456789012345'	
;
; Check if objects are present
list_names,items
if n_elements(items) eq 0 then return
;
; Create widget base
dat_name,title
window_wid=widget_base(/column,resource_name='oyster',title=title, $
	kill_notify='ww_browsedestroyed',xoffset=122,yoffset=0)
;
loc_id=0
list_wids[loc_id]=window_wid
list_wid=widget_list(window_wid,event_pro='ww_browsefind', $
		value=items,uvalue=items,ysize=6)
;
widget_control,window_wid,/realize
xmanager,'ww_browse',window_wid,/no_block
;
if n_elements(top_wid) eq 0 then top_wid=0L
if top_wid ne 0 then widget_control,top_wid,sensitive=0
;
end
;-------------------------------------------------------------------------------
pro ww_browsedestroyed,wid
;
common OysterWids,top_wid
;
if n_elements(top_wid) eq 0 then top_wid=0L
if top_wid ne 0 then widget_control,top_wid,sensitive=1
;
end
;-------------------------------------------------------------------------------
pro ww_browsefind,event
;
; Create list widget displaying the names of the components in an object 
; selected from a list of objects (components) contained in an object.
; When necessary, annul active list widgets of higher order.
;
common Hds,path,hds_file_stub
common BrowseInfo,list_wids,item_wids,item_names,loc_id
;
; Return on second click of double-click
if event.clicks ne 1 then return
;
; Check if list widgets of higher order are active
i=0
while (item_wids[i] ne event.id) and (i le loc_id) do i=i+1	
if i lt loc_id then begin
	widget_control,list_wids[i+1],/destroy
	l=loc_id
	for j=i+1,l do begin
		dat_annul
		loc_id=loc_id-1
	endfor
endif
;
; Get selected item name
widget_control,event.id,get_uvalue=names
name=names[event.index]
if strlen(name[0]) eq 0 then return
item_names[loc_id]=name
item_wids[loc_id]=event.id
;
dat_shape,ndim,dims
if ndim eq 0 then begin
	dat_find,name
	loc_id=loc_id+1
endif else if ndim eq 1 then begin
	cell=long(name)
	dat_cell,ndim,cell
	loc_id=loc_id+1
endif else begin
	cell=long(stringparse(name[0]))
	dat_cell,ndim,cell
	loc_id=loc_id+1
endelse
;
list_names,items
if n_elements(items) eq 0 then begin
	dat_annul
	loc_id=loc_id-1
	return
endif
;
window_wid=widget_base(event.top,title=name,group_leader=list_wids[loc_id-1])
list_wid=widget_list(window_wid,event_pro='ww_browsefind', $
	value=items,uvalue=items,ysize=6)
widget_control,window_wid,/realize
list_wids[loc_id]=window_wid
;
end
;-------------------------------------------------------------------------------
pro ww_loadstar,event
;
; Callback for Load star button in ACCESS button. Creates window with widget
; listing the stars contained in data base. 
; Widget is intended to stay active throughout the loading/editing
; /averaging stage of CHAMELEON.
; 
common LoadStarInfo,selstars,loadstar_wid
;
get_datum,date
if date[0] eq '' then return
selstars=''
list_stars,stars
if stars[0] eq '' then return
;
if n_elements(loadstar_wid) eq 0 then loadstar_wid=0L
if loadstar_wid ne 0 then widget_control,loadstar_wid,/destroy
;
loadstar_wid=widget_base(/row,kill_notify='ww_loadstardestroyed', $
			title=date,xoffset=568,yoffset=0,resource='oyster')
list_wid=widget_list(loadstar_wid,value=stars,uvalue=stars, $
	event_pro='ww_loadstarsel', $
	ysize=n_elements(stars),/multiple)
button_wid=cw_bgroup(loadstar_wid,['All stars','HELP','Load'],/column, $
	event_funct='ww_loadstarok',uvalue=list_wid,/return_name)
;
widget_control,loadstar_wid,/realize
xmanager,'ww_loadstar',loadstar_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_loadstardestroyed,wid
;
common LoadStarInfo,selstars,loadstar_wid
;
loadstar_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_loadstarsel,event
;
common LoadStarInfo,selstars,loadstar_wid
;
widget_control,event.id,get_uvalue=stars
i=widget_info(event.id,/list_select)
if i[0] ge 0 then selstars=stars[i] else selstars=''
;
end
;-------------------------------------------------------------------------------
function ww_loadstarok,event
;
; Callback procedure for procedure ww_loadstarsel: Load pointdata.
;
common LoadStarInfo,selstars,loadstar_wid
common Tables,ScanTable,BGTable,StationTable
common Starbase,StarTable,Notes
;
widget_control,event.id,get_uvalue=list_wid
command=event.value
;
case command of
	'All stars': begin
		widget_control,list_wid, $
			set_list_select=indgen(n_elements(StarTable))
		list_stars,selstars
		end
	'HELP': begin
		print,'___________________________________'
		print,'Select a single star (click), group'
		print,'of stars (drag mouse) or all stars.'
		print,'Not all data might fit into memory!'
		print,'Note: action will be immediate and'
		print,'data loaded will overwrite existing'
		print,'point data.'
		print,'_______________***_________________'
		end
	'Load':	begin
		if strlen(selstars[0]) eq 0 then return,0
		for n=0,n_elements(selstars)-1 do begin
			index=where(ScanTable.starid eq selstars[n])
			if n eq 0 then jndex=index else jndex=[jndex,index]
		endfor
		get_points,jndex+1
;		quack,/point
		end
	endcase
;
return,0
;
end
;-------------------------------------------------------------------------------
pro ww_loadmetrosel,event
;
; Callback for Load metrology button in ACCESS button.
; 
common LoadMetroInfo,motiongroup,astromgroup,loadmetrosel_wid
;
get_datum,date
if date[0] eq '' then return
;
; At least one motion group must be present
get_nummotiongroup,n
if n eq 0 then begin
	print,'***Error(WW_LOADMETROSEL): no data!'
	return
endif
motiongroups=string(indgen(n)+1)
;
; An AstromCorGrp is not necessarily present
get_numastromcorgrp,n
if n gt 0 then astromgroups=string(indgen(n)+1)
;
; Close any existing metro widget
if n_elements(loadmetrosel_wid) eq 0 then loadmetrosel_wid=0L
if loadmetrosel_wid ne 0 then widget_control,loadmetrosel_wid,/destroy
;
loadmetrosel_wid=widget_base(title=date,kill_notify='ww_loadmetrodestroyed', $
			xoffset=580,yoffset=10,/column,resource='oyster')
;
text_wid=widget_text(loadmetrosel_wid,value=' Motion group selection: ')
motn_wid=widget_list(loadmetrosel_wid,value=motiongroups,uvalue='motion', $
	event_pro='ww_loadmetro',ysize=n_elements(motiongroups))
;
if n_elements(astromgroups) gt 0 then begin
text_wid=widget_text(loadmetrosel_wid,value=' Astrom group selection: ')
astm_wid=widget_list(loadmetrosel_wid,value=astromgroups,uvalue='astrom', $
	event_pro='ww_loadmetro',ysize=n_elements(astromgroups))
endif
;
widget_control,loadmetrosel_wid,/realize
xmanager,'ww_loadmetrosel',loadmetrosel_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_loadmetrodestroyed,wid
;
common LoadMetroInfo,motiongroup,astromgroup,loadmetrosel_wid
;
loadmetrosel_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_loadmetro,event
;
; Callback procedure for procedure ww_loadmetrosel: Load metrology data.
; Note that if an AstromCorGrp is selected, the corresponding MotionGroup
; will be loaded too (i.e. call get_inchgroup, instead of get_astromcorgrp).
;
common LoadMetroInfo,motiongroup,astromgroup,loadmetrosel_wid
;
; Return on second click of double-click
if event.clicks ne 1 then return
;
widget_control,event.id,get_uvalue=type
case type of
	'motion':	get_motiongroup,event.index+1
	'astrom':	get_inchgroup,event.index+1
endcase
;
end
;************************************************************************Block 4
pro ww_reduce,event
;
; Dispatch for REDUCE button
;
widget_control,event.id,get_value=command
case command of
	'Average': 	average
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_reducept,event
;
; Dispatch for REDUCE PointData button.
;
common Hds,path,hds_file_stub
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
; Set the class
class='point'
;
widget_control,event.id,get_value=command
case command of
       'PLOT'  :begin
		if init_plot(class,'pt') ne 0 then return
		ww_plot
		end
       'A/R triple':begin
		ww_triplesel
		end
       'Unwrap triple':begin
		unwraptriple
		end
       'Clear data':begin
		clearpointdata
		end
       'Clear bias':begin
		clearbiasdata
		end
       'Load'  :begin
		filename=hds_file_stub+'.flg'
		result=findfile(filename,count=fcount)
		if fcount ne 0 then restore,filename else begin
     		      print,'***Error(WW_REDUCEPT):',filename,' does not exist!'
     		      return
		endelse
		result=size(point_table) & n=n_elements(result)
		if result[n-2] ne 8 then begin
		      print,'***Error(WW_REDUCEPT): ',filename, $
		      ' does not contain table!'
	      	      return
		endif
		create_pointflagtable,n_elements(point_table)
		for i=0,n_elements(pointflagtable)-1 do begin
			t=pointflagtable[i]
			struct_assign,point_table[i],t
			pointflagtable[i]=t
		endfor
;		pointflagtable=point_table
		print,'Restored pointflagtable.'
		end
       'Apply' :begin
		if checkdata([17]) ne 0 then return
		index=where(pointflagtable.time ge 0,count)
		if count ne 0 then flagpointdata,pointflagtable[index],flag=1 $
     			else print,'Warning(WW_REDUCEPT): Flagtable empty!'
		end
       'UNFLAG':begin
		ww_listflagreasons,class
		end
       'Save'  :begin
		if checkdata([17]) ne 0 then return
		filename=hds_file_stub+'.flg'
		result=findfile(filename,count=fcount)
		if fcount ne 0 then restore,filename
		if n_elements(inch_table) eq 0 then inch_table=0
		if n_elements(bg_table) eq 0 then bg_table=0
		if n_elements(scan_table) eq 0 then scan_table=0
		point_table=pointflagtable
		save,filename=filename,point_table,inch_table,bg_table,scan_table
		print,'Saved pointflagtable.'
		end
       'Clear' :begin
		if checkdata([17]) ne 0 then return
		pointflagtable[*].time=-1
		print,'Cleared pointflagtable.'
		end
       'Reset' :begin
		resetflag,'point'
		print,'Flags removed.'
		end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_reducebg,event
;
; Dispatch for REDUCE BG Data button
;
common Hds,path,hds_file_stub
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
; Set the class
class='bg'

widget_control,event.id,get_value=command
case command of
       'PLOT'  :begin
		if init_plot(class,'pt') ne 0 then return
		ww_plot
		end
       'Expand BG':begin
		expandbg
		end
       'Process BG':begin
		processbg
		end
       'Median BG':begin
		medianbg
		end
       'Default BG':begin
		defaultbg
		end
       'Load'  :begin
		if n_elements(hds_file_stub) eq 0 then return
		filename=hds_file_stub+'.flg'
		result=findfile(filename,count=fcount)
		if fcount ne 0 then restore,filename else begin
		      print,'***Error(WW_REDUCEBG):',filename,' does not exist!'
		      return
		endelse
		result=size(bg_table) & n=n_elements(result)
		if result[n-2] ne 8 then begin
			print,'***Error(WW_REDUCEBG): ',filename, $
				' does not contain table!'
			return
		endif
		create_bgflagtable
		for i=0,n_elements(bgflagtable)-1 do begin
			t=bgflagtable[i]
			struct_assign,bg_table[i],t
			bgflagtable[i]=t
		endfor
;		bgflagtable=bg_table
		print,'Restored bgflagtable.'
		end
       'Apply' :begin
		if checkdata([18]) ne 0 then return
		index=where(bgflagtable.time ge 0,count)
		if count ne 0 then flagbgdata,bgflagtable[index],flag=1 $
			else print,'Flagtable empty!'
		end
       'UNFLAG':begin
		ww_listflagreasons,class
		end
       'Save'  :begin
		if checkdata([18]) ne 0 then return
		filename=hds_file_stub+'.flg'
		result=findfile(filename,count=fcount)
		if fcount ne 0 then restore,filename
		if n_elements(point_table) eq 0 then point_table=0
		if n_elements(inch_table) eq 0 then inch_table=0
		if n_elements(scan_table) eq 0 then scan_table=0
		bg_table=bgflagtable
		save,filename=filename,point_table,inch_table,bg_table,scan_table
		print,'Saved bgflagtable.'
		end
       'Clear' :begin
		if checkdata([18]) ne 0 then return
		bgflagtable[*].time=-1
		print,'Cleared bgflagtable.'
		end
       'Reset' :begin
		resetflag,'bg'
		print,'Flags removed.'
		end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_reducedl,event
;
; Dispatch for Reduce|Point Data|Astrometry button.
;
; Set the class
class='delay'
;
widget_control,event.id,get_value=command
case command of
	'PLOT':         begin
		    	if init_plot(class,'pt') ne 0 then return
		    	ww_plot
		    	end
	'Init':         dispinit,class='point'
	'InitW':   	dispinit,'FKV0000',class='point'
	'PhaseEdit':	flagphase
	'PhaseWrap':	unwrapphase
	'DispCorr':	dispcorr
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_reducemt,event
;
; Dispatch for METROLOGY button.
;
common Hds,path,hds_file_stub
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
; Set the class
class='metro'
;
widget_control,event.id,get_value=command
case command of
       'PLOT'  :begin
		if checkdata([16]) ne 0 then return
	 	if init_plot(class,'pt') ne 0 then return
	 	ww_plot
	 	end
       'MetroCorr':begin
		if checkdata([16]) ne 0 then return
		metrocorr
		end
       'AstroCorr':begin
		if checkdata([16]) ne 0 then return
		astrocorr
		end
       'Init'  :metroinit
       'Load'  :begin
		if n_elements(hds_file_stub) eq 0 then return
		filename=file+'.flg'
		result=findfile(filename,count=fcount)
		if fcount ne 0 then restore,filename else begin
     		      print,'***Error(WW_REDUCEMT):',filename,' does not exist!'
     		      return
		endelse
		result=size(inch_table) & n=n_elements(result)
		if result[n-2] ne 8 then begin
		      print,'***Error(WW_REDUCEMT): ',filename, $
		      ' does not contain table!'
	      	      return
		endif
		inchflagtable=inch_table
		print,'Restored inchflagtable.'
		end
       'Apply' :begin
		if checkdata([16,20]) ne 0 then return
		index=where(inchflagtable.time ge 0,count)
		if count ne 0 then flaginchdata,inchflagtable[index],flag=1 $
     			else print,'Warning(WW_REDUCEMT): Flagtable empty!'
		end
       'UNFLAG':begin
		ww_listflagreasons,class
		end
       'Save'  :begin
		if checkdata([20]) ne 0 then return
		filename=hds_file_stub+'.flg'
		result=findfile(filename,count=fcount)
		if fcount ne 0 then restore,filename
		if n_elements(point_table) eq 0 then point_table=0
		if n_elements(bg_table) eq 0 then bg_table=0
		if n_elements(scan_table) eq 0 then scan_table=0
		inch_table=inchflagtable
		save,filename=filename,point_table,inch_table,bg_table,scan_table
		print,'Saved inchflagtable.'
		end
       'Clear' :begin
		if checkdata([20]) ne 0 then return
		inchflagtable[*].time=-1
		print,'Cleared inchflagtable.'
		end
       'Reset' :begin
		resetflag,'metro'
		print,'Flags removed.'
		end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_listflagreasons,class
;
if n_elements(class) eq 0 then begin
	print,'***Error(WW_LISTFLAGREASONS): class not defined!'
	return
endif
;
list_flagreasons,class,reasons
if reasons[0] eq '' then begin
	print,'***Error(WW_LISTFLAGREASONS): table empty!'
	return
endif
;
window_wid=widget_base(/column,title='Flagreasons('+class+')', $
	resource_name='oyster',uvalue=class)
list_wid=widget_list(window_wid,value=reasons,uvalue=reasons, $
	event_pro='ww_unflag',ysize=6)
;
widget_control,window_wid,/realize
xmanager,'ww_listflagreasons',window_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_unflag,event
;
widget_control,event.top,get_uvalue=class
widget_control,event.id,get_uvalue=reasons
reason=reasons[event.index]
unflag,class,reason
print,'Removed flags in class ',class,', REASON=',strtrim(reason),'.', $
	format='(a,a,a,a,a)'
widget_control,event.top,/destroy
;
end
;-------------------------------------------------------------------------------
pro ww_triplesel
;
common TripleInfo,triple,channels,action
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
if n_elements(triple_wid) eq 0 then triple_wid=0L
if triple_wid ne 0 then widget_control,triple_wid,/destroy
;
if checkdata([8]) ne 0 then return
;
triple=intarr(3,2)
channels=intarr(max(GenConfig.NumSpecChan),3)
;
triple_wid=widget_base(kill_notify='ww_tripledestroyed', $
	title='Add/Delete Triple',/column,resource_name='oyster')
;
row_wid=widget_base(triple_wid,/row)
actions=['Add','Delete']
action='Add'
opmenu_wid=widget_droplist(row_wid,value=actions,uvalue=actions, $
	event_pro='ww_tripleaction')
button_wid=cw_bgroup(row_wid,['List','HELP','OK'],event_funct='ww_tripleok', $
			/row,/return_name)
clabels=['OB','Bl','Ch']
rlabels=['1','2','3']
table_wid=widget_table(triple_wid,event_pro='ww_triple',value=strarr(3,3), $
	column_labels=clabels,row_labels=rlabels,/editable)
;
widget_control,triple_wid,/realize
xmanager,'ww_triplesel',triple_wid,/no_block
;
end		
;-------------------------------------------------------------------------------
pro ww_tripledestroyed,wid
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
triple_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_tripleaction,event
;
common TripleInfo,triple,channels,action
;
widget_control,event.id,get_uvalue=actions
action=actions[event.index]
;
end
;-------------------------------------------------------------------------------
function ww_tripleok,event
;
common TripleInfo,triple,channels,action
;
case event.value of
	'OK':		begin
			triple=triple-1
			channels=channels-1
			tripleproduct
			triple=triple+1
			channels=channels+1
			end
	'HELP':		begin
			print,'________________________________________'
			print,'For each of the three baselines, specify'
			print,'OB and BL numbers. For CH, you can enter'
			print,'a range too, e.g. 1-4, or ALL for all'
			print,'channels. After selection, click OK.'
			print,'_________________***____________________'
			end
	'List':		begin
			list_triples
			end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_triple,event
;
common TripleInfo,triple,channels,action
;
widget_control,event.id,get_value=command
;
for i=0,2 do begin
	triple[i,*]=long(command[0:1,i])
	if strupcase(command[2,i]) eq 'ALL' then begin
	    	c=indgen(n_elements(channels[*,i]))+1
	endif else c=stringparse(string(command[2,i]))
	n=n_elements(c)
	channels[*,i]=0
	channels[0:n-1,i]=c
endfor
print,triple
print,channels
;
end
;************************************************************************Block 5
pro ww_calibrate,event
;
; Dispatch for CALIBRATE button.
;
if checkdata([1,3,5,6,7,8,9]) ne 0 then return
;
widget_control,event.id,get_value=command
case command of
	'SEEING'       :begin
			if init_plot('seeing','pt') ne 0 then return
			ww_plot
			end
	'SYSTEM'       :begin
			if init_plot('ncal','pt') ne 0 then return
			ww_plot
			end
	'PHOTOMETRY'   :begin
			if init_plot('phot','pt') ne 0 then return
			ww_plot
			end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_scans,event
;
; Dispatch for scan calibration.
;
common Hds,path,hds_file_stub
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
if checkdata([1,3,5,6,7,8,9]) ne 0 then return
;
; Set the class
class='scan'
;
widget_control,event.id,get_value=command
case command of
	'PLOT'		   :begin
			    if init_plot(class,'pt') ne 0 then return
			    ww_plot
			    end
	'CALIBRATE'        :ww_cal
	'RewrapTriple'	   :rewraptriple
	'DewrapTriple'	   :dewraptriple
	'ManualUnwrap'	   :manualunwrap
	'FlagIncoh'	   :begin
			    print,'Flagging incoherent scans...'
			    index=where(scantable.code eq 0,count)
			    if count gt 0 then flagscandata,sc=index+1,item=24,reason='incoh'
			    end
	'SetErrors'	   :set_viserrors
	'SetComplex'	   :set_complexvis
	'Load'             :begin
			    if n_elements(hds_file_stub) eq 0 then return
			    filename=file+'.flg'
			    result=findfile(filename,count=fcount)
			    if fcount ne 0 then restore,filename else begin
			      print,'***Error(WW_CALIBRATE):',filename, $
					' does not exist!'
			      return
			    endelse
			    result=size(scan_table) & n=n_elements(result)
			    if result[n-2] ne 8 then begin
			      print,'***Error(WW_CALIBRATE): ',filename, $
					' does not contain table!'
			      return
			    endif
			    create_scanflagtable
			    for i=0,n_elements(scanflagtable)-1 do begin
				t=scanflagtable[i]
				struct_assign,scan_table[i],t
				scanflagtable[i]=t
			    endfor
;			    scanflagtable=scan_table
			    print,'Restored scanflagtable.'
			    end
	'Apply'            :begin
			    if n_elements(scanflagtable) ne 0 then begin
			    	index=where(scanflagtable.time ge 0,count)
			    	if count ne 0 then $
					flagscandata,scanflagtable[index],flag=1 $
					else print,'Flagtable empty!'
			    endif else print,'***Error(WW_CALIBRATE): '+ $
						'table does not exist!'
			    end
        'UNFLAG'           :ww_listflagreasons,class
	'Save'             :begin
			    if n_elements(scanflagtable) eq 0 then begin
				print,'***Error(WW_CALIBRATE): '+ $
						'table does not exist!'
				return
			    endif
			    filename=hds_file_stub+'.flg'
			    result=findfile(filename,count=fcount)
			    if fcount ne 0 then restore,filename
			    if n_elements(point_table) eq 0 then point_table=0
			    if n_elements(inch_table) eq 0 then inch_table=0
			    if n_elements(bg_table) eq 0 then bg_table=0
			    scan_table=scanflagtable
			    save,filename=filename, $
				point_table,inch_table,bg_table,scan_table
			    print,'Saved scanflagtable.'
			    end
	'Clear'            :begin
			    if n_elements(scanflagtable) eq 0 then begin
				print,'***Error(WW_CALIBRATE): '+ $
						'table does not exist!'
				return
			    endif
			    scanflagtable[*].time=-1
			    print,'Cleared scanflagtable.'
			    end
	'Reset'		   :begin
			    resetflag,class
			    print,'Flags removed.'
			    end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_astrom,event
;
; Dispatch for CALIBRATE/ASTROMETRY button.
;
if checkdata([8,9]) ne 0 then return
;
; Set the class
class='astrom'
;
widget_control,event.id,get_value=command
case command of
	'PLOT' :begin
		if init_plot(class,'pt') ne 0 then return
		ww_plot
		end
	'Calc' :begin
		calcastrom
		calcvis
		end
	'MCorr':begin
		pivotcorr
		whitecorr
		end
	'SOLVE':ww_astromsolve
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_astromsolve
;
; Display a widget to compute single-night solutions for station coordinates
; and star positions.
;
common SetFitStarsInfo,sel_stars,stars_directive	; Local common block (2)
common SetStationsInfo,sel_stations,stations_directive	; Local common block (2)
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
;
; This is to tell the compiler that ww_astromfit is a function
forward_function ww_astromfit
;
; Remove existing widget
if n_elements(astrom_wid) eq 0 then astrom_wid=0L
if astrom_wid ne 0 then widget_control,astrom_wid,/destroy
;
astrom_wid=widget_base(title='AstrometrySolve',resource_name='oyster',/column, $
	kill_notify='ww_astromdestroyed',uvalue='Astrom', $
	xoffset=!dxsize-504,yoffset=!dysize-307)
;
; First two widget elements are in a row
row1_wid=widget_base(astrom_wid,/row)
;
; Station selection
station_dirs=['All','Sel']
stations_directive='All'
list_stations,fit_stations
index=where(total(scantable.station[0:GenConfig.NumSid-1,*],2) ne 0,count)
if count eq 0 then begin
	print,'***Error(WW_ASTROM): no data!'
	return
endif
fit_stations=fit_stations[index]
sel_stations=''
column1_wid=widget_base(row1_wid,/column)
opmenustn_wid=widget_droplist(column1_wid,title='Stns:', $
	value=station_dirs,uvalue=station_dirs, $
	event_pro='ww_setstations')
list1_wid=widget_list(column1_wid,value=fit_stations,uvalue=fit_stations,/multiple, $
	event_pro='ww_setstations',xsize=3,ysize=3)
;
; Star selection
star_dirs=['All','Sel']
stars_directive='Sel'
fit_stars=''
sel_stars=''
column2_wid=widget_base(row1_wid,/column)
opmenustr_wid=widget_droplist(column2_wid,title='Stars:', $
	value=star_dirs,uvalue=star_dirs, $
	event_pro='ww_setfitstars')
widget_control,opmenustr_wid,set_droplist_select=1
list_stars,stars
list2_wid=widget_list(column2_wid,value=stars,uvalue=stars,/multiple, $
	event_pro='ww_setfitstars',ysize=3)
;
row2_wid=widget_base(astrom_wid,/row)
;
; Use uncorrected or corrected delays
data_dirs=['FDL','GRP','DRY','WET']
fit_data='FDL'
opmenuda_wid=widget_droplist(row2_wid,title='Data:', $
		value=data_dirs,uvalue=data_dirs, $
		event_pro='ww_setastrodata')
;
; Reference station
j=where(fit_stations eq GenConfig.StationId[GenConfig.RefStation-1])
stations=shift(fit_stations,-j[0])
ref_station=stations
opmenurs_wid=widget_droplist(row2_wid,title='Reference:', $
		value=ref_station,uvalue=ref_station, $
		event_pro='ww_setrefstation')
;
; Button box
button_wid=cw_bgroup(astrom_wid,/row,['Solve','Save','Load','HELP'], $
	event_funct='ww_astromfit',/return_name)
;
; Display widget
widget_control,astrom_wid,/realize
xmanager,'ww_astromsolve',astrom_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_astromdestroyed,wid
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
astrom_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_setstations,event
;
common SetStationsInfo,sel_stations,stations_directive	; Local common block (2)
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
;
widget_control,event.id,get_uvalue=dirs
widget_control,event.top,get_uvalue=origin
;
case dirs[event.index] of
	'All':begin
	      case origin of
	      'Astrom': list_stations,fit_stations
	      'Volvox': fit_stations=stationtable.stationid
	      endcase
	      stations_directive=dirs[event.index]
	      end
	'Sel':begin
	      if n_elements(sel_stations) ne 0 then fit_stations=sel_stations
	      stations_directive=dirs[event.index]
	      end
	 else:begin
	      i=widget_info(event.id,/list_select)
	      if i[0] ge 0 then sel_stations=dirs[i] else sel_stations=''
	      if stations_directive eq 'Sel' then fit_stations=sel_stations
	      end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setfitstars,event
;
common SetFitStarsInfo,sel_stars,stars_directive	; Local common block (2)
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
;
widget_control,event.id,get_uvalue=dirs
widget_control,event.top,get_uvalue=origin
;
case dirs[event.index] of
	'All':begin
	      case origin of
	      'Astrom': list_stars,fit_stars
	      'Volvox': fit_stars=startable.starid
	      endcase
	      stars_directive=dirs[event.index]
	      end
	'Sel':begin
	      if n_elements(sel_stars) ne 0 then fit_stars=sel_stars
	      stars_directive=dirs[event.index]
	      end
	 else:begin
	      i=widget_info(event.id,/list_select)
	      if i[0] ge 0 then sel_stars=dirs[i] else sel_stars=''
	      if stars_directive eq 'Sel' then fit_stars=sel_stars
	      end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setfitnights,event
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
common SetFitNightsInfo,sel_nights,nights_directive	; Local common block (3)
;
widget_control,event.id,get_uvalue=dirs
;
case dirs[event.index] of
	'All':	begin
		fit_nights=geninfo.date
		nights_directive=dirs[event.index]
		end
	'Sel':	begin
	      	if n_elements(sel_nights) ne 0 then fit_nights=sel_nights
	      	nights_directive=dirs[event.index]
		end
	 else:	begin
	      	i=widget_info(event.id,/list_select)
	      	if i[0] ge 0 then sel_nights=dirs[i] else sel_nights=''
	      	if nights_directive eq 'Sel' then fit_nights=sel_nights
	      	end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setastrodata,event
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
;
widget_control,event.id,get_uvalue=data
fit_data=data[event.index]
;
end
;-------------------------------------------------------------------------------
pro ww_setastroparms,event
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
;
widget_control,event.id,get_uvalue=data
fit_parms=data[event.index]
;
end
;-------------------------------------------------------------------------------
pro ww_setrefstation,event
;
widget_control,event.id,get_uvalue=stations
referencestation,stations[event.index]
;
end
;-------------------------------------------------------------------------------
function ww_astromfit,event
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
case event.value of
	'Solve':solveastrom
	'Save' :put_StationTable
	'Load' :get_StationTable,Date+'.stn'
	'HELP' :begin
		print,'_______________________________________________'
		print,'SOLVE will calculate new coordinates for the'
		print,'selected stations and stars in a simultaneous'
		print,'fit to the selected data. All data will be used'
		print,'if either no star or no station was selected.'
		print,'SAVE will write a file stations.npoi with the'
		print,'updated coordinates. LOAD will read this file'
		print,'to update current coordinates.'
		print,'_____________________***_______________________'
		end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_pivot
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
; Create a widget tool to play with catseye pivot point offsets.
; Last values obtained are: p_C = -4 mu, p_E = +12 mu, p_W = +43 mu
;
if checkdata([8]) ne 0 then return
;
; Create this widget if necessary to set astrometry fit parameters
if n_elements(astrom_wid) eq 0 then astrom_wid=0L
if astrom_wid eq 0 then ww_astromsolve
;
; Plot control is through this widget
if n_elements(plot_wid) eq 0 then plot_wid=0L
if plot_wid eq 0 then begin
	if init_plot('astrom','pt') ne 0 then return
endif
;
; Remove any existing widget
if n_elements(pivot_wid) eq 0 then pivot_wid=0L
if pivot_wid ne 0 then widget_control,pivot_wid,/destroy
;
xs=long(!xsize)
ys=long(!ysize)
pivot_wid=widget_base(title='Pivot tool',/column, $
			resource_name='oyster', $
                       	kill_notify='ww_pivotdestroyed')
;
drawing_wid=widget_draw(pivot_wid,notify_realize='ww_pivotcorr', $
		scr_xsize=xs,scr_ysize=ys)
;
for i=0,GenConfig.NumSid-1 do begin
	label=GenConfig.StationId[i] $
     	     +' Catseye center - mirror offset [mu]'
	control_wid=widget_slider(pivot_wid,title=label, $
		minimum=-50,maximum=50,event_pro='ww_pivotcorr', $
        	xsize=xs,value=0,uvalue=string(i+1))
endfor
;
widget_control,pivot_wid,/realize
xmanager,'ww_pivot',pivot_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_pivotdestroyed,wid
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
pivot_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_pivotcorr,event
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
result=size(event)
if result[n_elements(result)-2] ne 8 then begin
	MetroConfig.SidModel.CatsEyeOff=0
endif else begin
;	Remove existing correction
	whitecorr
	pivotcorr
; 	Read new selected catseye center - mirror surface offset
	widget_control,event.id,get_uvalue=sid
	MetroConfig.SidModel.CatsEyeOff[sid-1]=event.value
endelse
;
; Apply new corrections
pivotcorr
whitecorr
;
; Compute and solve astrometry
calcastrom
solveastrom
;
; Update the plot selection
if set_datasel['x'] ne 0 then return
if set_datasel['y'] ne 0 then return
if set_datasel['z'] ne 0 then return
;
; Plot the data, w/out creating a new window
plotdata,/window
;
end
;************************************************************************Block 6
pro ww_utilities,event
;
; Dispatch for UTILITIES button.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common Starbase,startable,notes
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
widget_control,event.id,get_value=command
case command of
	'MEMO':	begin
		help_wid=widget_base(/col,resource_name='oyster', $
			title='Memo search')
		text_wid=widget_text(help_wid,event_pro='ww_memo', $
			/editable,xsize=40)
		list_wid=widget_list(help_wid,value=' ', $
			ysize=6,event_pro='ww_memolist')
		widget_control,text_wid,set_uvalue=list_wid
		widget_control,help_wid,/realize
		xmanager,'memo',help_wid,/no_block
		end
	'OBSBASE':	begin
			spawn,'whoami',userid
			if userid[0] ne 'chummel' then begin
			   print,'***Error(WW_UTILITIES): permission denied!'
			   return
			endif
			ww_obssel
			end
	'.CAL->.CHA':	ww_calsel
	'.CHA->.OIFITS':ww_chasel
	'Load Night':	begin
			if n_elements(GenInfo) gt 0 then begin
			nights=GenInfo.Date+' '+GeoInfo.SystemId+' '+GenInfo.ConfigId
			window_wid=widget_base(title='Nights in buffer', $
				resource_name='oyster')
		   	list_wid=widget_list(window_wid,value=nights,uvalue=nights, $
			event_pro='ww_loadnight',ysize=n_elements(nights))
		   	widget_control,window_wid,/realize
		   	xmanager,'ww_loadnight',window_wid,/no_block
			endif
			end	
	'Load StarTable'  :begin
			   get_startable
			   read_notes
			   end
	'Summary':begin
		  list_summary_chameleon
		  print,'Summary complete.'
		  end
	'Stars'  :list_stars
	'Scans'  :list_scans
	'BGScans':list_bgscans
	'ScnLog' :begin
		  if n_elements(ScanLog) gt 0 then begin
		  if strlen(strcompress(ScanLog,/remove)) gt 0 then begin
			index=where(byte(ScanLog) eq 10)
		  	xdisplayfile,'',title='ScanLog', $
				text=ScanLog+' ', $
				width=max(index-shift(index,1))
		  endif else print,'ScanLog empty!'
		  endif else print,'ScanLog nonexistent!'
		  end
	'ObsLog' :begin
		  if n_elements(ObserverLog) gt 0 then begin
		  if strlen(strcompress(ObserverLog,/remove)) gt 0 then begin
			index=where(byte(ObserverLog) eq 10)
		  	xdisplayfile,'',title='ObserverLog', $
				text=ObserverLog+' ', $
				width=max(index-shift(index,1))
		  endif else print,'ObserverLog empty!'
		  endif else print,'ObserverLog nonexistent!'
		  end
	'ConLog' :begin
		  if n_elements(ConstrictorLog) gt 0 then begin
		  if strlen(strcompress(ConstrictorLog,/remove)) gt 0 then begin
			index=where(byte(ConstrictorLog) eq 10)
		  	xdisplayfile,'',title='ConstrictorLog', $
				text=ConstrictorLog+' ', $
				width=max(index-shift(index,1))
		  endif else print,'ConstrictorLog empty!'
		  endif else print,'ConstrictorLog nonexistent!'
		  end
	'STARS':begin
		list_stars,stars
		if stars[0] ne '' then begin
		   window_wid=widget_base(title='StarInfo', $
				resource_name='oyster')
		   list_wid=widget_list(window_wid,value=stars,uvalue=stars, $
			event_pro='ww_starinfo',ysize=5)
		   widget_control,window_wid,/realize
		   xmanager,'ww_starinfo',window_wid,/no_block
		endif
		end
	'Star':begin
	       if n_elements(startable) ne 0 then startable_save=startable
	       star=strarr(1)
	       read,star,prompt='Enter star name: '
	       star=strupcase(star)
	       get_startable,star
	       list_star,star
	       if n_elements(startable_save) ne 0 then startable=startable_save
	       end
	'ScanForms':print_scanforms
	'SpecForms':print_specforms
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_memo,event
;
widget_control,event.id,get_value=keyword,get_uvalue=list_wid
keyword=keyword[0]
if strlen(strcompress(keyword,/remove)) eq 0 then return
if strupcase(keyword) eq 'ALL' then keyword='Re:'
words=nameparse(keyword)
result=''
for i=0,n_elements(words)-1 do begin
	spawn,'grep -Hi '+words[i]+' '+!oyster_dir+'docs/memo.*',r
	result=[result,r]
endfor
files=strjoin(unique(strmid(result,strlen(!oyster_dir+'docs/'),13)),' ')
if strlen(files) eq 0 then begin
	widget_control,list_wid,set_value=' ',set_uvalue=' '
	return
endif else begin
	files=nameparse(files)
	items=files
	for i=0,n_elements(files)-1 do begin
		spawn,'grep -Hi Re: '+!oyster_dir+'docs/'+files[i],r
		j=strpos(r,'Re:')+4
		items[i]=' "'+strmid(r[0],j,strlen(r)-j-1)+'"'
	endfor
	widget_control,list_wid,set_value=files+items,set_uvalue=files
endelse
;
end
;-------------------------------------------------------------------------------
pro ww_memolist,event
;
widget_control,event.id,get_uvalue=files
file=strcompress(files[event.index],/remove)
if strlen(file) eq 0 then return
spawn,'cat '+!oyster_dir+'docs/'+file[0],r
xdisplayfile,'',title='OYSTER memo',text=r
;
end
;-------------------------------------------------------------------------------
pro ww_loadnight,event
;
widget_control,event.id,get_uvalue=nights
night=nights[event.index]
night0=night
loadnight,night
; widget_control,event.top,/destroy
print,night0+' loaded.'
;
end
;-------------------------------------------------------------------------------
pro ww_starinfo,event
;
widget_control,event.id,get_uvalue=stars
star=stars[event.index]
list_star,star
list_note,star
widget_control,event.top,/destroy
;
end
;************************************************************************Block 7
pro ww_cal
;
; Display scan calibration selection widget.
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common CalibrateWids,indicator_wids,list_wids,num_list
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common SelCalStarsInfo,sel_stars,directive
;
; Remove cal and plot widgets if present
if n_elements(cal_wid) eq 0 then cal_wid=0L
if cal_wid ne 0 then widget_control,cal_wid,/destroy
;
; Initialize calibration control variables
initcal
num_indicators=n_elements(indicators)
indicator_wids=lonarr(num_indicators)
num_list=4 	; Number of function groups
list_wids=lonarr(num_indicators*num_list)
;
cal_wid=widget_base(/column,kill_notify='ww_caldestroyed', $
	resource_name='oyster', $
	xoffset=!dxsize-250,yoffset=!dysize-277,title='VisibilityCalibrate')
;
row1_wid=widget_base(cal_wid,/row)
;
list_indicators_wid=widget_list(row1_wid,value=indicators, $
	event_pro='ww_calindicator',/multiple,ysize=7)
;
column12_wid=widget_base(row1_wid,/column)
star_dirs=['All','Cal','Sel']
opmenu_wid=widget_droplist(column12_wid,title='Use:', $
	value=star_dirs,uvalue=star_dirs, $
	event_pro='ww_calstardir')
widget_control,opmenu_wid,set_droplist_select=2
directive='Sel'
;
list_stars,stars
list_stars_wid=widget_list(column12_wid,value=stars,uvalue=stars, $
	event_pro='ww_calstars',/multiple,ysize=4)
;
variables=[cal_options.n[0],cal_options.n[1],cal_options.n[2]]
opmenuvb_wid=widget_droplist(cal_wid,title='Variable:',value=variables, $
	event_pro='ww_setcalvar')
;
row2_wid=widget_base(cal_wid,/row)
;
button1_wid=cw_bgroup(row2_wid,['Calibrate','HELP'], $
		event_funct='ww_calok',/column,/return_name)
button2_wid=cw_bgroup(row2_wid,['Loop','Self'],/nonexclusive, $
		event_funct='ww_caloptions',/column)
;
column22_wid=widget_base(row2_wid,/column)
menu1=['1\Undo\ww_calutils', $
	'0\UNDO','0\Uncal','2\Reset']
menu2=['1\Entries\ww_calutils', $
	'0\Save','0\Load','0\Apply','0\Clear','0\List','6\EDIT']
menu1_wid=cw_pdmenu(column22_wid,menu1)
menu2_wid=cw_pdmenu(column22_wid,menu2)
;
widget_control,cal_wid,/realize
xmanager,'ww_cal',cal_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_caldestroyed,wid
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
;
cal_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_setcalvar,event
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
;
cal_options.v=event.index
;
end
;-------------------------------------------------------------------------------
function ww_caloptions,event
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
;
case event.value of
0:begin
 cal_options.l=event.select
 if cal_options.l eq 0 then print,'All sel. data will be combined in fit!' 
 if cal_options.l eq 1 then print,'Data will fit per baseline and channel.'
 end
1: begin
 cal_options.s=event.select
 if cal_options.s eq 1 then print,'Self-calibration of phases selected!'
 if cal_options.s eq 0 then print,'Self-calibration of phases cancelled!'
 end
endcase
;
;
end
;-------------------------------------------------------------------------------
pro ww_calstardir,event
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common Starbase,startable,notes
common SelCalStarsInfo,sel_stars,directive
;
widget_control,event.id,get_uvalue=dirs
directive=dirs[event.index]
case directive of
	'All':list_stars,cal_stars
	'Cal':begin
	      index=where(startable.bflag eq 'C',count)
	      if count gt 0 then begin
		cal_stars=startable[index].starid
		print,'Found',count,' listed calibrators:',format='(a,i3,a)'
		print,cal_stars
	      endif else print,'Warning: no calibrators listed in startable!'
	      end
	'Sel':if n_elements(sel_stars) ne 0 then cal_stars=sel_stars
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_calstars,event
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common SelCalStarsInfo,sel_stars,directive
;
widget_control,event.id,get_uvalue=stars
i=widget_info(event.id,/list_select)
if i[0] ge 0 then sel_stars=stars[i] else sel_stars=''
if directive eq 'Sel' then cal_stars=sel_stars
;
end
;-------------------------------------------------------------------------------
pro ww_calindicator,event
;
; Widget procedure to display and manage list widgets for the various 
; indicators. The lists contain the base functions.
;
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common CalibrateWids,indicator_wids,list_wids,num_list
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common BaseFunctions,numlist,functions
;
forward_function ww_base
;
xpos=!dxsize-[1005,835,715,595,475,355,160]
ypos=!dysize-[447,447,447,447,447,447,447]
;
; Get names of indicators. These are in an array.
i=widget_info(event.id,/list_select)
if i[0] ge 0 then new_indicators=indicators[i] else new_indicators=''
if new_indicators[0] eq '' then num_indicators=0 else $
	num_indicators=n_elements(new_indicators)
;
; Find out which indicator has been added to the list
for i=0,num_indicators-1 do begin
	hit=0
;
;	These are the indicators already selected
	if (sel_indct[0] eq '') then num_sel=0 else $
		num_sel=n_elements(sel_indct)
	for j=0,num_sel-1 do $
		if new_indicators[i] eq sel_indct[j] then hit=1
;
;	If no match was found: a new indicator was selected
	index=where(strpos(sel_funcn[*,0],'S_') ge 0,do_smooth)
	if hit eq 0 then begin
;
;		Add new indicator to array through copying
		new_sel_indct=strarr(num_sel+1)
		if num_sel gt 0 then new_sel_indct[0:num_sel-1]=sel_indct
		new_sel_indct[num_sel]=new_indicators[i]
		sel_indct=new_sel_indct
;
;		Expand array sel_funcn to hold base functions for new indct.
		new_functions=strarr(num_sel+1,n_elements(sel_funcn[0,*]))
		if num_sel gt 0 then new_functions[0:num_sel-1,*]=sel_funcn
		sel_funcn=new_functions
;
;		Create new base function widget
		index=where(indicators eq new_indicators[i])
		list_index=[1,2]
		if new_indicators[i] eq 'Wavelength' then list_index=[1,2,3]
                if new_indicators[i] eq 'Time' $
                or new_indicators[i] eq 'HourAngle' then list_index=[1,2,4]
;
		indicator_wids[index[0]]=ww_base(new_indicators[i],list_index, $
			'ww_calfunction',listwids, $
			xpos[index[0]],ypos[index[0]],group_leader=cal_wid)
		widget_control,indicator_wids[index[0]], $
			set_uvalue=new_indicators[i]
;
;		Store the new list widget IDs
		list_wids[index[0]*num_list+0]=listwids[0]
		list_wids[index[0]*num_list+1]=listwids[1]
		if new_indicators[i] eq 'Wavelength' then $
		list_wids[index[0]*num_list+2]=listwids[2]
		if new_indicators[i] eq 'Time' $
		or new_indicators[i] eq 'HourAngle' then $
		list_wids[index[0]*num_list+3]=listwids[2]
;			
;		Smoothing cannot be combined with any other function
		if do_smooth gt 0 then $
			widget_control,indicator_wids[index[0]],sensitive=0 $
		   else widget_control,indicator_wids[index[0]],sensitive=1
	endif
endfor
;
; Find out which indicator has been removed from the list
old_indicators=sel_indct
for i=0,n_elements(sel_indct)-1 do begin
	hit=0
	for j=0,num_indicators-1 do $
		if new_indicators[j] eq sel_indct[i] then hit=1
	if hit eq 0 then begin
;		sel_indct(i) was deselected and the funcn list has to be removed
;		Is this a list blocking others? If so, make others sensitive
		index=where(old_indicators eq sel_indct[i])
		blocking=strpos(sel_funcn[index,*],'S_') & blocking=blocking[0]
		if blocking ne -1 then begin
			for j=0,n_elements(indicators)-1 do begin
				if indicator_wids[j] ne 0 $
			       and indicators[j] ne sel_indct[i] then $
				widget_control,indicator_wids[j],sensitive=1
			endfor
		endif
		index=where(old_indicators ne sel_indct[i],count)
		if count gt 0 then begin 
			old_indicators=old_indicators[index]
			sel_funcn=sel_funcn[index,*]
		endif else begin
			old_indicators=''
			sel_funcn=strarr(1,n_elements(functions))
		endelse
		index=where(indicators eq sel_indct[i])
		if indicator_wids[index[0]] ne 0 then $
			widget_control,indicator_wids[index[0]],/destroy
		indicator_wids[index[0]]=0 
		list_wids[index[0]*num_list+0]=0
		list_wids[index[0]*num_list+1]=0
		list_wids[index[0]*num_list+2]=0
		list_wids[index[0]*num_list+3]=0
	endif
endfor
sel_indct=old_indicators
;
end
;-------------------------------------------------------------------------------
pro ww_calfunction,event
;
common CalibrateWids,indicator_wids,list_wids,num_list
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common BaseFunctions,numlist,functions
;
; Identify indicator for which functions were selected
widget_control,event.top,get_uvalue=indicator
i=where(sel_indct eq indicator)
widget_control,event.id,get_uvalue=new_functions
j=widget_info(event.id,/list_select)
if j[0] ge 0 then new_functions=new_functions[j] else new_functions=''
num_functions=n_elements(new_functions)
sel_funcn[i[0],*]=''
sel_funcn[i[0],0:num_functions-1]=new_functions 
;
; Clear the lists other than the one from which functions were selected
index=where(list_wids eq event.id)
i=index[0] mod num_list
for j=0,num_list-1 do begin
	k=index[0]-i+j
	if list_wids[k] ne event.id and list_wids[k] ne 0 then begin
		case j of
			0:widget_control,list_wids[k],set_value=functions[0:4]
			1:widget_control,list_wids[k],set_value=functions[5:9]
			2:widget_control,list_wids[k],set_value=functions[10:14]
			3:widget_control,list_wids[k],set_value=functions[15:19]
		endcase
	endif
endfor
;
; Set the sensitivity of the list widgets
index=strpos(new_functions[0],'S_')
for j=0,n_elements(indicators)-1 do begin
if indicator_wids[j] ne 0 and indicators[j] ne indicator then begin
	if index eq -1 then widget_control,indicator_wids[j],sensitive=1 $
		       else widget_control,indicator_wids[j],sensitive=0
endif
endfor
;
end
;-------------------------------------------------------------------------------
pro ww_calutils,event
;
common Hds,path,hds_file_stub
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
;
widget_control,event.id,get_value=command
case command of
	'UNDO' :ww_listcalreasons,'Undo'
	'Uncal':uncal
	'Reset':resetcal
;
	'Save' :begin
		save,filename=hds_file_stub+'.cal',cal_entries
		print,'Saved cal_entries.'
		end
	'Load' :begin
		if n_elements(hds_file_stub) eq 0 then return
		filename=hds_file_stub+'.cal'
		result=findfile(filename,count=fcount)
		if fcount ne 0 then restore,filename else begin
			print,'***Error(WW_OKCAL):',filename,' does not exist!'
			return
		endelse
		entries=cal_entries
		set_calentries,entries
		print,'Cal_entries loaded.'
		end
	'Apply':begin
		v=cal_options.v
		for i=0,2 do begin
			cal_options.v=i
			list_calreasons,reasons
			if reasons[0] ne '' then begin
				for j=0,n_elements(reasons)-1 do begin
					calibrate,reasons[j]
				endfor
			endif else begin
				print,'Warning(WW_CALOK): '+ $
				      'no entries for this variable: ',i,'!', $
				      format='(a,i1,a)'
			endelse
		endfor
		cal_options.v=v
		end
	'Clear':begin
		create_calentries
		print,'Cleared CalTable.'
		end
	'List' :list_calentries
	'EDIT' :ww_listcalreasons,'Edit'
endcase
;
end
;-------------------------------------------------------------------------------
function ww_calok,event
;
case event.value of
	'Calibrate':begin
		reason=strmid(systime(),0,24)
		calcoeffs,reason
		calibrate,reason
		end
	'HELP' :begin
	 	print,'________________________________________________'
		print,'Select/deselect calibrator stars and visibility'
		print,'reduction indicators. For each indicator, select'
		print,'base functions used in the fit. Calibration is  '
		print,'based on selected data, with one modification:  '
		print,'the star selection refers to stars to which ca- '
		print,'libration is applicable! "Calibrate" when ready!'
		print,'Self option: divide by visibility for line cals.'
		print,'_____________________***________________________'
		end
;
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_listcalreasons,command
;
; Widget to display list of cal reasons from which is to be selected a 
; calibration for editing or removal. (Not a callback function.)
;
common WwListCalReasons,listcalreasons_wid
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
;
if n_elements(listcalreasons_wid) eq 0 then listcalreasons_wid=0L
if listcalreasons_wid ne 0 then widget_control,listcalreasons_wid,/destroy
;
list_calreasons,reasons
if reasons[0] eq '' then begin
	print,'***Error(WW_LISTCALREASONS): no entries found!'
	return
endif
if n_elements(command) eq 0 then return
case command of
	'Edit':callback='ww_editcal'
	'Undo':callback='ww_undocal'
	else:return
endcase
variable=cal_options.n[cal_options.v]
listcalreasons_wid=widget_base(kill_notify='ww_listcaldestroyed', $
	resource_name='oyster', $
	title='Calreasons('+variable+')',xoffset=0,yoffset=0)
list_wid=widget_list(listcalreasons_wid,value=reasons,uvalue=reasons, $
	event_pro=callback,ysize=6)
;
widget_control,listcalreasons_wid,/realize
xmanager,'ww_listcalreasons',listcalreasons_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_listcaldestroyed,wid
;
common WwListCalReasons,listcalreasons_wid
;
listcalreasons_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_editcal,event
;
widget_control,event.id,get_uvalue=reasons
reason=reasons[event.index]
editcal,reason[0],'Edit'
widget_control,event.top,/destroy
;
end
;-------------------------------------------------------------------------------
pro ww_undocal,event
;
widget_control,event.id,get_uvalue=reasons
reason=reasons[event.index]
editcal,reason[0],'Undo'
widget_control,event.top,/destroy
;
end
;************************************************************************Block 2
pro ww_amoebadata,event
;
; Dispatch for DATA button.
;
widget_control,event.id,get_value=command
case command of
	'INTERFEROMETRY':	ww_datasel,'cha'
	'ASTROMETRY':		ww_datasel,'psn'
	'SPECTROSCOPY': 	ww_datasel,'vel'
	'PHOTOMETRY':		ww_datasel,'mag'
	'Summary':		list_summary_amoeba
endcase
;
end
;************************************************************************Block 3
pro ww_amoebamodel,event
;
; Dispatch for MODEL button.
;
widget_control,event.id,get_value=command
case command of
	'READ': ww_modelfile
	'Info': begin
		list_modcomps
		print,'Listing complete.'
		end
	'Calc': begin
		calcmodel
		print,'CalcModel complete.'
		end
	'FIT':  ww_modelfit
	 else:	print,'***Error(WW_AMOEBAMODEL): choice not yet supported.'
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_modelfile
;
modelfile=ww_fileselect(title='Model file selection', $
	filter='*.model')
if strlen(modelfile) eq 0 then return
if total(strlen(findfile(modelfile))) eq 0 then return
readmodel,modelfile
;
end
;************************************************************************Block 4
pro ww_amoebaplot,event
;
; Dispatch for PLOT button.
;
widget_control,event.id,get_value=command
case command of
	'UV Plots':begin
		if init_plot('uv','pt') ne 0 then return
		ww_plot
		end
	'XY Plots':begin
		if init_plot('amoeba','pt') ne 0 then return
		ww_plot
		end
	'Astrometry':begin
		if init_plot('volvox','pt') ne 0 then return
		ww_plot
		end
	'SPECTROSCOPY':begin
		ww_plotvel
		end
	'PHOTOMETRY':begin
		ww_plotmag
		end
	'ASTROMETRY':begin
		ww_plotorbit
		end
	else:print,'***Error(WW_AMOEBAPLOT): choice not yet supported.'
endcase
;
end
;************************************************************************Block 5
pro ww_amoebafit,event
;
; Dispatch for FIT button. Listed here are all specialized fits to specific
; data sets.
;
widget_control,event.id,get_value=command
case command of
	'CONTROL':		ww_marquardtfit
	'ASTROMETRY': 		ww_fitastrometry
	'INTERFEROMETRY': 	ww_fitinterferometry
	else:print,'***Error(WW_AMOEBAFIT): choice not yet supported.'
endcase
;
end
;************************************************************************Block 6
pro ww_volvox,event
;
; Dispatch for ASTROMETRY button.
;
widget_control,event.id,get_value=command
case command of
	'Plot':		plotvolvox
	'WCorr':	volvoxcorr,metro=0
	'MCorr':	volvoxcorr
	'Calc':		calcvolvox
	'Get PMs':	initvolvox,'pm'
	'Get HIP':	initvolvox,'hip'
	'SOLVE':	ww_volvoxsolve
	else:print,'***Error(WW_VOLVOX): choice not yet supported.'
endcase
;
end
;************************************************************************Block 7
pro ww_oyster,event
;
; Dispatch for OYSTER button.
;
common Hds,path,hds_file_stub
;
command=''
widget_control,event.id,get_uvalue=command
if string(command) ne 'Help' then widget_control,event.id,get_value=command
;
case command of
'Help':		begin
		case strupcase(!version.os) of
		'WIN32':begin
		  	print,'Starting Internet Explorer on OYSTER homepage...'
			spawn,'c:\"Program Files"\"Internet Explorer"\iexplore.exe ' $
				+!oyster_dir+'oyster\oyster.html',result
			end
		'LINUX':begin
	  		print,'Starting Firefox on OYSTER homepage...'
			spawn,'firefox ' $
				+!oyster_dir+'oyster/oyster.html &',result
			end
		'DARWIN':begin
	  		print,'Starting Safari on OYSTER homepage...'
			spawn,'unalias Safari;/Applications/Safari.app/Contents/MacOS/Safari ' $
				+!oyster_dir+'oyster/oyster.html &',result
			end
		else   :print,'Sorry, the location of a web browser is unknown on your system!'
		endcase
	  	end
'Save session':	begin
	  	if n_elements(hds_file_stub) eq 0 then begin
			print,'***Error(WW_OYSTER): file name undefined!'
			return
	  	endif
	  	filename=hds_file_stub+'.rst'
	  	saverestore,filename,'save'
	  	end
'Restore session':begin
	  	modelfile=ww_fileselect(title='Restore file selection', $
			filter='*.rst')
	  	saverestore,filename,'restore'
	  	end
'Help,/memory':	help,/memory
'Exit widget':	ww_quit
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_quit,event
;
common Hds,path,hds_file_stub
common OysterWids,top_wid
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common AmoebaWids,data_wid,mfit_wid,ifit_wid,afit_wid,control_wid
common CalibrateWids,indicator_wids,list_wids,num_list
;
top_wid=0L
;
; Chameleon widgets
if n_elements(plot_wid) eq 0 then plot_wid=0L
if plot_wid ne 0 then widget_control,plot_wid,/destroy
if n_elements(td_wid) eq 0 then td_wid=0L
if td_wid ne 0 then widget_control,td_wid,/destroy
if n_elements(cal_wid) eq 0 then cal_wid=0L 
if cal_wid ne 0 then widget_control,cal_wid,/destroy
if n_elements(astrom_wid) eq 0 then astrom_wid=0L
if astrom_wid ne 0 then widget_control,astrom_wid,/destroy
if n_elements(pivot_wid) eq 0 then pivot_wid=0L
if pivot_wid ne 0 then widget_control,pivot_wid,/destroy
if n_elements(triple_wid) eq 0 then triple_wid=0L 
if triple_wid ne 0 then widget_control,triple_wid,/destroy
;
; Amoeba widgets
if n_elements(data_wid) eq 0 then data_wid=0L
if data_wid ne 0 then widget_control,data_wid,/destroy
if n_elements(mfit_wid) eq 0 then mfit_wid=0L
if mfit_wid ne 0 then widget_control,mfit_wid,/destroy
if n_elements(ifit_wid) eq 0 then ifit_wid=0L
if ifit_wid ne 0 then widget_control,ifit_wid,/destroy
if n_elements(afit_wid) eq 0 then afit_wid=0L
if afit_wid ne 0 then widget_control,afit_wid,/destroy
if n_elements(control_wid) eq 0 then control_wid=0L
if control_wid ne 0 then widget_control,control_wid,/destroy
;
; These have no destroy-callbacks either
if n_elements(indicator_wids) ne 0 then indicator_wids[*]=0L
if n_elements(list_wids) ne 0 then list_wids[*]=0L
;
; Reset plot parameters
!x.range=0
!y.range=0
!z.range=0
!p.title=''
!x.title=''
!y.title=''
!z.title=''
;
end
;************************************************************************Block 8
pro ww_datasel,datatype
;
; Displays GUI to select data files of a given type, e.g. CHA, VEL, PSN, MAG.
; The DataInterferometry widget allows to read CHA or FITS files.
;
common ModelFit,parameters,ds_options
common AmoebaWids,data_wid,mfit_wid,ifit_wid,afit_wid,control_wid
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
;
; opmenu: 'l'=load, 'w'=weight', 'a'=amp, 'p'=phase, 'f'=flux
; file_dir: 'All' (in the curr. dir.), 'List' (sel. in GUI), or from 'File' 
; file_sel: the data files selected with the GUI
; sel_file: the file with the list of files
; file_ext: extension of files to be listed 
;
if n_elements(data_wid) eq 0 then data_wid=0L
if data_wid ne 0 then widget_control,data_wid,/destroy
opmenua_wid=0
opmenup_wid=0
;
case datatype of
	'cha':	begin
		datatitle='DataInterferometry'
		callback='ww_chaselok'
		buttons=['OK','HELP']
		d=1
		weight=ds_options.i
		end
	'psn':	begin
		datatitle='DataAstrometry'
		callback='ww_psnselok'
		buttons=['OK','HELP']
		d=2
		weight=ds_options.a
		end
	'vel':	begin
		datatitle='DataSpectroscopy'
		callback='ww_velselok'
		buttons=['OK','HELP']
		d=3
		weight=ds_options.s
		end
	'mag':	begin
		datatitle='DataPhotometry'
		callback='ww_magselok'
		buttons=['OK','HELP']
		d=4
		weight=ds_options.p
		end
endcase
data_wid=widget_base(kill_notify='ww_dataseldestroyed', $
		xoffset=568,yoffset=0, $
		title=datatitle,/column,resource_name='oyster')
row_wid=widget_base(data_wid,/row)
;
file_ext='*.'+datatype
if datatype eq 'cha' then file_ext=file_ext+' *.*fits'
files=findfile(file_ext)
if strlen(files[0]) eq 0 then begin
	print,'Warning(WW_DATASEL): no files found!'
	file_dir='File'
	current=3
endif else begin
	column1_wid=widget_base(row_wid,/column)
	label_wid=widget_label(column1_wid,value='Filter:')
	text_wid=widget_text(column1_wid,event_pro='ww_setfilter', $
		/editable,xsize=14,value=file_ext)
	list_files_wid=widget_list(column1_wid,value=files,uvalue=files, $
		event_pro='ww_setfiles',/multiple,ysize=8)
	widget_control,text_wid,set_uvalue=list_files_wid
	file_dir='All'
	current=1
endelse
;
column2_wid=widget_base(row_wid,/column)
file_dirs=['All','List','File']
opmenul_wid=widget_droplist(column2_wid,title='Load: ', $
	event_pro='ww_setfiledir',value=file_dirs,uvalue=file_dirs)
widget_control,opmenul_wid,set_droplist_select=current-1
file_sel=''
;
label_wid=widget_label(column2_wid,value='FileSpec:')
text_wid=widget_text(column2_wid,event_pro='ww_setfileselfile', $
	/editable,xsize=14)
;
if datatype eq 'cha' or datatype eq 'psn' $
or datatype eq 'vel' or datatype eq 'mag' then begin
	weights=['10','1','0.1','0']
	opmenuw_wid=widget_droplist(column2_wid,title='Weight:', $
		event_pro='ww_datweight',value=weights,uvalue={d:d,w:weights})
	widget_control,opmenuw_wid, $
		set_droplist_select=where(float(weights)-weight eq 0)
endif
;
; Additional options for some data types
if datatype eq 'cha' then begin
	types=['Vsqr','TAmp','Both','None']
	opmenua_wid=widget_droplist(column2_wid,title='Amp:  ', $
		event_pro='ww_dattype',value=types,uvalue='AMP '+types)
	sds=3
	if ds_options.v2 then sds=0
	if ds_options.ta then sds=1
	if ds_options.v2 and ds_options.ta then sds=2
	widget_control,opmenua_wid,set_droplist_select=sds
;
	types=['VPhs','TPhs','Both','None']
	opmenup_wid=widget_droplist(column2_wid,title='Phase:', $
		event_pro='ww_dattype',value=types,uvalue='PHS '+types)
	sds=3
	if ds_options.vp then sds=0
	if ds_options.tp then sds=1
	if ds_options.vp and ds_options.tp then sds=2
	widget_control,opmenup_wid,set_droplist_select=sds
;
	types=['Flux','None']
	opmenuf_wid=widget_droplist(column2_wid,title='Flux:', $
		event_pro='ww_dattype',value=types,uvalue='FLX '+types)
	sds=1
	if ds_options.fl then sds=0
	widget_control,opmenuf_wid,set_droplist_select=sds
endif
;
if datatype eq 'psn' then begin
	types=['Ell','Cir','Uni']
	opmenut_wid=widget_droplist(column2_wid,title='Type:  ', $
		event_pro='ww_psntype',value=types,uvalue=types)
	widget_control,opmenut_wid,set_droplist_select=0
endif
;
if datatype eq 'mag' then begin
	types=['Phot','Model']
	opmenut_wid=widget_droplist(column2_wid,title='Type:  ', $
		event_pro='ww_magtype',value=types,uvalue=types)
endif
;
button_wid=cw_bgroup(data_wid,buttons, $
		event_funct=callback,/row,/return_name)
;
widget_control,data_wid,/realize
xmanager,'ww_datasel',data_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_obssel
;
; Select .con or .cha files for updating the obsbase data base.
;
common ModelFit,parameters,ds_options
common AmoebaWids,data_wid,mfit_wid,ifit_wid,afit_wid,control_wid
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
;
if n_elements(data_wid) eq 0 then data_wid=0L
if data_wid ne 0 then widget_control,data_wid,/destroy
;
data_wid=widget_base(kill_notify='ww_dataseldestroyed', $
	xoffset=568,yoffset=0, $
	title='Observations database',/column,resource_name='oyster')
;
row_wid=widget_base(data_wid,/row)
file_ext='*.cha *.con'
files=findfile(file_ext)
if strlen(files[0]) eq 0 then begin
	print,'***Error(WW_OBSSEL): no files found!'
	file_dir='File'
	current=3
endif else begin
	column1_wid=widget_base(row_wid,/column)
	label_wid=widget_label(column1_wid,value='Filter:')
	text_wid=widget_text(column1_wid,event_pro='ww_setfilter', $
		/editable,xsize=14,value=file_ext)
	list_files_wid=widget_list(column1_wid,value=files,uvalue=files, $
		event_pro='ww_setfiles',/multiple,ysize=6)
	widget_control,text_wid,set_uvalue=list_files_wid
	file_dir='List'
	current=2
endelse
column2_wid=widget_base(row_wid,/column)
file_dirs=['All','List','File']
opmenul_wid=widget_droplist(column2_wid,title='Load: ', $
	event_pro='ww_setfiledir',value=file_dirs,uvalue=file_dirs)
widget_control,opmenul_wid,set_droplist_select=current-1
file_sel=''
;
label_wid=widget_label(column2_wid,value='FileSpec:')
text_wid=widget_text(column2_wid,event_pro='ww_setfileselfile', $
	/editable,xsize=14)
;
button_wid=cw_bgroup(data_wid,['BUILD','UPDATE','HELP'], $
		event_funct='ww_obsselok',/row,/return_name)
;
widget_control,data_wid,/realize
xmanager,'ww_obssel',data_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_calsel
;
; List .cal-files in current directory and let user select the files to be
; loaded.
;
common AmoebaWids,data_wid,mfit_wid,ifit_wid,afit_wid,control_wid
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
;
if n_elements(data_wid) eq 0 then data_wid=0L
if data_wid ne 0 then widget_control,data_wid,/destroy
;
data_wid=widget_base(kill_notify='ww_dataseldestroyed', $
	xoffset=568,yoffset=0, $
	title='.cal->.cha',/column,resource_name='oyster')
;
row_wid=widget_base(data_wid,/row)
file_ext='*.cal'
files=findfile(file_ext)
if strlen(files[0]) eq 0 then begin
	print,'Warning(WW_CALSEL): no files found!'
	file_dir='File'
	current=3
endif else begin
	column1_wid=widget_base(row_wid,/column)
	label_wid=widget_label(column1_wid,value='Filter:')
        text_wid=widget_text(column1_wid,event_pro='ww_setfilter', $
                /editable,xsize=14,value=file_ext)
	list_files_wid=widget_list(column1_wid,value=files,uvalue=files, $
		event_pro='ww_setfiles',/multiple,ysize=6)
	widget_control,text_wid,set_uvalue=list_files_wid
	file_dir='List'
	current=2
endelse
column2_wid=widget_base(row_wid,/column)
file_dirs=['All','List','File'] 
opmenul_wid=widget_droplist(column2_wid,title='Load:', $
	event_pro='ww_setfiledir',value=file_dirs,uvalue=file_dirs)
widget_control,opmenul_wid,set_droplist_select=current-1
file_sel=''
;
text_wid=widget_text(column2_wid,event_pro='ww_setfileselfile', $
	/editable,xsize=14)
;
button_wid=cw_bgroup(data_wid,['OK','HELP'], $
		event_funct='ww_calselok',/row,/return_name)
;
widget_control,data_wid,/realize
xmanager,'ww_calsel',data_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_chasel
;
; List .cha-files in current directory and let user select the files to be
; loaded.
;
common AmoebaWids,data_wid,mfit_wid,ifit_wid,afit_wid,control_wid
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
;
if n_elements(data_wid) eq 0 then data_wid=0L
if data_wid ne 0 then widget_control,data_wid,/destroy
;
data_wid=widget_base(kill_notify='ww_dataseldestroyed', $
	xoffset=568,yoffset=0, $
	title='.cha->.oifits',/column,resource_name='oyster')
;
row_wid=widget_base(data_wid,/row)
file_ext='*.cha'
files=findfile(file_ext)
if strlen(files[0]) eq 0 then begin
	print,'Warning(WW_CHASEL): no files found!'
	file_dir='File'
	current=3
endif else begin
	column1_wid=widget_base(row_wid,/column)
	label_wid=widget_label(column1_wid,value='Filter:')
        text_wid=widget_text(column1_wid,event_pro='ww_setfilter', $
                /editable,xsize=14,value=file_ext)
	list_files_wid=widget_list(column1_wid,value=files,uvalue=files, $
		event_pro='ww_setfiles',/multiple,ysize=6)
	widget_control,text_wid,set_uvalue=list_files_wid
	file_dir='List'
	current=2
endelse
column2_wid=widget_base(row_wid,/column)
file_dirs=['All','List','File'] 
opmenul_wid=widget_droplist(column2_wid,title='Load:', $
	event_pro='ww_setfiledir',value=file_dirs,uvalue=file_dirs)
widget_control,opmenul_wid,set_droplist_select=current-1
file_sel=''
;
text_wid=widget_text(column2_wid,event_pro='ww_setfileselfile', $
	/editable,xsize=14)
;
button_wid=cw_bgroup(data_wid,['OK','HELP'], $
		event_funct='ww_c2oselok',/row,/return_name)
;
widget_control,data_wid,/realize
xmanager,'ww_chasel',data_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_dataseldestroyed,wid
;
common AmoebaWids,data_wid,mfit_wid,ifit_wid,afit_wid,control_wid
;
data_wid=0L
;
end
;-------------------------------------------------------------------------------
pro ww_setfilter,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
;
widget_control,event.id,get_value=file_ext,get_uvalue=list_files_wid
files=findfile(file_ext[0])
widget_control,list_files_wid,set_value=files,set_uvalue=files
;
end
;-------------------------------------------------------------------------------
pro ww_setfileselfile,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
;
widget_control,event.id,get_value=sel_file
sel_file=sel_file[0]
print,'File name selected: '+sel_file
widget_control,opmenul_wid,set_droplist_select=2
;
end
;-------------------------------------------------------------------------------
pro ww_setfiles,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
;
widget_control,event.id,get_uvalue=files
i=widget_info(event.id,/list_select)
if i[0] ge 0 then file_sel=files[i]
;
widget_control,opmenul_wid,set_droplist_select=1
file_dir='List'
sel_file=files[i]
;
end
;-------------------------------------------------------------------------------
pro ww_setfiledir,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
;
widget_control,event.id,get_uvalue=dirs
file_dir=dirs[event.index]
;
end
;-------------------------------------------------------------------------------
pro ww_datweight,event
;
common ModelFit,parameters,ds_options
;
widget_control,event.id,get_uvalue=weights
;
case weights.d of
	1:item=9
	2:item=22
	3:item=23
	4:item=24
endcase
if checkdata(item) ne 0 then begin
	widget_control,event.id,set_droplist_select=3
	return
endif
case weights.d of
	1:ds_options.i=float(weights.w[event.index])
	2:ds_options.a=float(weights.w[event.index])
	3:ds_options.s=float(weights.w[event.index])
	4:begin
	  if ds_options.pm then print,'Note: for type Model weight is ignored!'
	  ds_options.p=float(weights.w[event.index])
	  end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_dattype,event
;
common ModelFit,parameters,ds_options
;
widget_control,event.id,get_uvalue=in_types
types=in_types
;
for i=0,n_elements(types)-1 do begin
	words=nameparse(types[i])
	type=words[0]
	types[i]=words[1]
endfor	
case type of
'AMP':	begin
	ds_options.v2=0
	ds_options.ta=0
	case types[event.index] of
	'Vsqr': ds_options.v2=1
	'TAmp': ds_options.ta=1
	'Both': begin
		ds_options.v2=1
		ds_options.ta=1
		end
	'None': 
	endcase
;	help,type,ds_options.v2
;	help,type,ds_options.ta
	end
'PHS':	begin
	ds_options.vp=0
	ds_options.tp=0
	case types[event.index] of
	'VPhs':	ds_options.vp=1
	'TPhs':	ds_options.tp=1
	'Both': begin
		ds_options.vp=1
		ds_options.tp=1
		end
	'None': 
	endcase
;	help,type,ds_options.vp
;	help,type,ds_options.tp
	end
'FLX':	begin
	ds_options.fl=0
	case types[event.index] of
	'Flux':	ds_options.fl=1
	'None':	ds_options.fl=0
	endcase
;	help,type,ds_options.fl
	end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_psntype,event
;
common ModelFit,parameters,ds_options
;
widget_control,event.id,get_uvalue=types
case types[event.index] of
	'Ell':  ds_options.c=0
	'Cir':  ds_options.c=1
	'Uni':  ds_options.c=2
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_magtype,event
;
common ModelFit,parameters,ds_options
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
;
widget_control,event.id,get_uvalue=types
case types[event.index] of
	'Phot' :begin
		ds_options.pm=0
		widget_control,opmenuw_wid,set_droplist_select=3
		print,'Weight set to zero, please change if needed!'
		end
	'Model':begin
		ds_options.p=0
		ds_options.pm=1
		widget_control,opmenuw_wid,set_droplist_select=3
		print,'Single component only!'
		print,'Band and wavelength sort order'
		print,'must match information in model!'
		end
endcase
;
end
;-------------------------------------------------------------------------------
function ww_chaselok,event
;
; Despite its name, this function also loads interferometry data in other
; format such as OIFITS, see load_interferometry.
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common ModelFit,parameters,ds_options
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
;
case event.value of
'OK':	begin
	if opmenua_wid eq 0 or opmenup_wid eq 0 then return,-1
	files=''
;	We should remove an old widget since it will be out of sync
	if n_elements(plot_wid) eq 0 then plot_wid=0L
	if plot_wid ne 0 then widget_control,plot_wid,/destroy
	case file_dir of
		'All'	:files=findfile(file_ext)
		'List'  :if n_elements(file_sel) ne 0 then files=file_sel
		'File'  :begin
			 if n_elements(sel_file) eq 0 then begin
			  	print,'***Error(WW_CHASELOK): file unspecified!'
			 	return,-1
			 endif
			 result=findfile(sel_file,count=fcount)
			 if fcount eq 0 then begin
			  	print,'***Error(WW_CHASELOK): file not found!'
			  	return,-1
			 endif
			 if strpos(sel_file,'.cha') ne -1 then $
				files=sel_file else begin
;			 Read file names from specified ("FileSpec") file
			 status=dc_read_fixed(sel_file,files,resize=[1], $
			  	format='(a)',/col)
			 if status ne 0 then begin
				print,'***Error(WW_CHASELOK): error reading file!'
				return,-1
			 endif
			 endelse
			 end	
	endcase
	if strlen(strcompress(files[0],/remove_all)) eq 0 then begin
		print,'***Error(WW_CHASELOK): no files specified!'
		return,-1
	endif
;	IDL version 6.0.3 in a virtual machine can hang if widget is exposed
	if !version.release eq '6.0.3' then iconify_oyster
	load_interferometry,files
	if !version.release eq '6.0.3' then iconify_oyster,0
	if strlen(files[0]) eq 0 then return,0
;	Display plot widget
	ww_plot
;	Update the menu options in the DataInterferometry GUI
	types=['Vsqr','TAmp','Both','None']
	sds=3
	if ds_options.v2 then sds=0
	if ds_options.ta then sds=1
	if ds_options.v2 and ds_options.ta then sds=2
	widget_control,opmenua_wid,set_droplist_select=sds
	types=['VPhs','TPhs','Both','None']
	sds=3
	if ds_options.vp then sds=0
	if ds_options.tp then sds=1
	if ds_options.vp and ds_options.tp then sds=2
	widget_control,opmenup_wid,set_droplist_select=sds
;	widget_control,opmenuw_wid,set_droplist_select=1
	print,'Interferometric data type selection updated.'
     	end
'HELP':	begin
	print,'_____________________________________________'
	print,'Select from files displayed in the list.'
	print,'Files should contain calibrated visibilities.'
	print,'Load options:                           '
	print,'All:  read all files listed.            '
	print,'File: read specified file for file list.'
	print,'List: read files selected in list.      '
	print,'the specified file for a list of files.'
	print,'___________________***_______________________'
	end
endcase
;
end
;-------------------------------------------------------------------------------
function ww_psnselok,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
common LocalPsnSel,psnfiles
;
case event.value of
'OK':	begin
	files=''
	case file_dir of
		'All'	:files=findfile('*.psn')
		'File'  :begin
			 if n_elements(sel_file) eq 0 then begin
			  	print,'***Error(WW_PSNSELOK): file unspecified!'
			 	return,-1
			 endif
			 result=findfile(sel_file,count=fcount)
			 if fcount eq 0 then begin
			  	print,'***Error(WW_PSNSELOK): file not found!'
			  	return,-1
			 endif
			 if strpos(sel_file,'.psn') ne -1 then $
				files=sel_file else begin
			 status=dc_read_fixed(sel_file,files,resize=[1], $
			  	format='(a)',/col)
			 if status ne 0 then begin
				print,'***Error(WW_PSNSELOK): error reading file!'
				return,-1
			 endif
			 endelse
			 end	
		'List'  :if n_elements(file_sel) ne 0 then files=file_sel
	endcase
	if strlen(strcompress(files[0],/remove_all)) eq 0 then begin
		print,'***Error(WW_PSNSELOK): no files specified!'
		return,-1
	endif
	load_astrometry,files
	psnfiles=files
	widget_control,opmenuw_wid,set_droplist_select=1
     	end
'HELP':	begin
	print,'________________________________________'
	print,'Select from files displayed in the list.'
	print,'These should contain measured positions '
	print,'for multiple star systems.              '
	print,'Load options:                           '
	print,'All:  read all files listed.            '
	print,'File: read specified file for file list.'
	print,'List: read files selected in list.      '
	print,'_________________***____________________'
	end
endcase
;
end
;-------------------------------------------------------------------------------
function ww_velselok,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
common LocalVelSel,velfiles
;
case event.value of
'OK':	begin
	files=''
	case file_dir of
		'All'	:files=findfile('*.vel')
		'File'  :begin
			 if n_elements(sel_file) eq 0 then begin
			  	print,'***Error(WW_VELSELOK): file unspecified!'
			 return,-1
			 endif
			 result=findfile(sel_file,count=fcount)
			 if fcount eq 0 then begin
			  	print,'***Error(WW_VELSELOK): file not found!'
			  	return,-1
			 endif
			 if strpos(sel_file,'.vel') ne -1 then $
				files=sel_file else begin
			 status=dc_read_fixed(sel_file,files,resize=[1], $
			  	format='(a)',/col)
			 if status ne 0 then begin
				print,'***Error(WW_VELSELOK): error reading file!'
				return,-1
			 endif
			 endelse
			 end	
		'List'  :if n_elements(file_sel) ne 0 then files=file_sel
	endcase
	if strlen(strcompress(files[0],/remove_all)) eq 0 then begin
		print,'***Error(WW_VELSELOK): no files specified!'
		return,-1
	endif
	load_spectroscopy,files
	velfiles=files
	widget_control,opmenuw_wid,set_droplist_select=1
     	end
'HELP':	begin
	print,'________________________________________'
	print,'Select from files displayed in the list.'
	print,'These should contain measured velocities'
	print,'for multiple star systems.              '
	print,'Load options:                           '
	print,'All:  read all files listed.            '
	print,'File: read specified file for file list.'
	print,'List: read files selected in list.      '
	print,'_________________***____________________'
	end
endcase
;
end
;-------------------------------------------------------------------------------
function ww_magselok,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common ModelFit,parameters,ds_options
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
common LocalMagSel,magfiles
;
case event.value of
'OK':	begin
	files=''
	case file_dir of
		'All'	:files=findfile('*.mag')
		'File'  :begin
			 if n_elements(sel_file) eq 0 then begin
			  	print,'***Error(WW_MAGSELOK): file unspecified!'
			 	return,-1
			 endif
			 result=findfile(sel_file,count=fcount)
			 if fcount eq 0 then begin
			  	print,'***Error(WW_MAGSELOK): file not found!'
			  	return,-1
			 endif
			 if strpos(sel_file,'.mag') ne -1 then $
				files=sel_file else begin
			 status=dc_read_fixed(sel_file,files,resize=[1], $
			  	format='(a)',/col)
			 if status ne 0 then begin
				print,'***Error(WW_MAGSELOK): error reading file!'
				return,-1
			 endif
			 endelse
			 end	
		'List'  :if n_elements(file_sel) ne 0 then files=file_sel
	endcase
	if strlen(strcompress(files[0],/remove_all)) eq 0 then begin
		print,'***Error(WW_MAGSELOK): no files specified!'
		return,-1
	endif
	load_photometry,files
	magfiles=files
	if ds_options.pm eq 0 then $
	widget_control,opmenuw_wid,set_droplist_select=1
	if ds_options.pm eq 1 then $
	widget_control,opmenuw_wid,set_droplist_select=3
     	end
'HELP':	begin
	print,'________________________________________'
	print,'Select from files displayed in the list.'
	print,'These should contain measured magnitudes'
	print,'for multiple star systems. Load options:'
	print,'All:  read all files listed.            '
	print,'File: read specified file for file list.'
	print,'List: read files selected in list.      '
	print,'Type: default is "Phot" for light curves'
	print,'  "Model" for variable binary components'
	print,'_________________***____________________'
	end
endcase
;
end
;-------------------------------------------------------------------------------
function ww_obsselok,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
;
files=''
case file_dir of
	'All'	:files=findfile('*.cha')
	'List'  :if n_elements(file_sel) ne 0 then files=file_sel
	'Sel'	:files=sel_file
	   else : 
endcase
case event.value of
'HELP':	begin
	print,'________________________________________'
	print,'Select from files displayed in the list.'
	print,'Load options:                           '
	print,'All:  read all files listed.            '
	print,'File: read specified file for file list.'
	print,'List: read files selected in list.      '
	print,'_________________***____________________'
	end
else:	begin
	files=''
	case file_dir of
		'All'	:files=findfile('*.con *.cha')
		'File'  :begin
			 if n_elements(sel_file) eq 0 then begin
			  	print,'***Error(WW_OBSSELOK): file unspecified!'
			 	return,-1
			 endif
			 result=findfile(sel_file,count=fcount)
			 if fcount eq 0 then begin
			  	print,'***Error(WW_OBSSELOK): file not found!'
			  	return,-1
			 endif
			 if strpos(sel_file,'.cha') ne -1 $
			 or strpos(sel_file,'.con') ne -1 then $
				files=sel_file else begin
			 status=dc_read_fixed(sel_file,files,resize=[1], $
			  	format='(a)',/col)
			 if status ne 0 then begin
				print,'***Error(WW_OBSSELOK): error reading file!'
				return,-1
			 endif
			 endelse
			 end	
		'List'  :if n_elements(file_sel) ne 0 then files=file_sel
	endcase
	if strlen(strcompress(files[0],/remove_all)) eq 0 then begin
		print,'***Error(WW_OBSSELOK): no files specified!'
		return,-1
	endif
	case event.value of
		'BUILD':	begin
				spawn,'rm -f '+!oyster_dir+'npoi/obsbase.xdr'
				obsbase,files
				end
		'UPDATE':	obsbase,files
	endcase
     	end
endcase
;
end
;-------------------------------------------------------------------------------
function ww_calselok,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common ModelFit,parameters,ds_options
;
case event.value of
'OK':	begin
	files=''
	case file_dir of
		'All'	:files=findfile('*.cal')
		'File'  :begin
			 if n_elements(sel_file) eq 0 then begin
			  	print,'***Error(WW_CALSELOK): file unspecified!'
			 return,-1
			 endif
			 result=findfile(sel_file,count=fcount)
			 if fcount eq 0 then begin
			  	print,'***Error(WW_CALSELOK): file not found!'
			  	return,-1
			 endif
			 if strpos(sel_file,'.cal') ne -1 then $
				files=sel_file else begin
			 status=dc_read_fixed(sel_file,files,resize=[1], $
			  	format='(a)',/col)
			 if status ne 0 then begin
				print,'***Error(WW_CALSELOK): error reading file!'
				return,-1
			 endif
			 endelse
			 end	
		'List'  :if n_elements(file_sel) ne 0 then files=file_sel
	endcase
	if strlen(strcompress(files[0],/remove_all)) eq 0 then begin
		print,'***Error(WW_CALSELOK): no files specified!'
		return,-1
	endif
	mark32cha,files
     	end
'HELP':	begin
	print,'____________________________________________'
	print,'Select from files displayed in the list.    '
	print,'These should contain calibrated visibilities'
	print,'from MarkIII. The procedure will load the '
	print,'MarkIII data and store it into .cha files.'
	print,'Load options:                           '
	print,'All:  read all files listed.            '
	print,'File: read specified file for file list.'
	print,'List: read files selected in list.      '
	print,'__________________***_______________________'
	end
endcase
end
;-------------------------------------------------------------------------------
function ww_c2oselok,event
;
common FileSelInfo,file_dir,file_sel,sel_file,file_ext
common ModelFit,parameters,ds_options
;
case event.value of
'OK':	begin
	files=''
	case file_dir of
		'All'	:files=findfile('*.cha')
		'File'  :begin
			 if n_elements(sel_file) eq 0 then begin
			  	print,'***Error(WW_CHASELOK): file unspecified!'
			 return,-1
			 endif
			 result=findfile(sel_file,count=fcount)
			 if fcount eq 0 then begin
			  	print,'***Error(WW_CHASELOK): file not found!'
			  	return,-1
			 endif
			 if strpos(sel_file,'.cha') ne -1 then $
				files=sel_file else begin
			 status=dc_read_fixed(sel_file,files,resize=[1], $
			  	format='(a)',/col)
			 if status ne 0 then begin
				print,'***Error(WW_CHASELOK): error reading file!'
				return,-1
			 endif
			 endelse
			 end	
		'List'  :if n_elements(file_sel) ne 0 then files=file_sel
	endcase
	if strlen(strcompress(files[0],/remove_all)) eq 0 then begin
		print,'***Error(WW_C2OSELOK): no files specified!'
		return,-1
	endif
	cha2oifits,files
     	end
'HELP':	begin
	print,'____________________________________________'
	print,'Select from files displayed in the list.    '
	print,'These should contain calibrated visibilities'
	print,'from NPOI. The procedure will load the '
	print,'NPOI data and store it into .oifits files.'
	print,'Load options:                           '
	print,'All:  read all files listed.            '
	print,'File: read specified file for file list.'
	print,'List: read files selected in list.      '
	print,'__________________***_______________________'
	end
endcase
end
;************************************************************************Block 1
pro ww_volvoxsolve
;
; Display a widget to compute multiple-night solutions for star positions and
; proper motions. This widget shares resources with the other astrometry
; widget.
;
common SetFitStarsInfo,sel_stars,stars_directive	; Local common block (2)
common SetStationsInfo,sel_stations,stations_directive	; Local common block (2)
common SetFitNightsInfo,sel_nights,nights_directive	; Local common block (3)
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common ChameleonWids,plot_wid,td_wid,cal_wid,astrom_wid,pivot_wid,triple_wid
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
;
; This is to tell the compiler that ww_astromfit is a function
forward_function ww_volvoxfit
;
; Remove existing widget
if n_elements(astrom_wid) eq 0 then astrom_wid=0L
if astrom_wid ne 0 then widget_control,astrom_wid,/destroy
;
astrom_wid=widget_base(title='VolvoxSolve',resource_name='oyster',/column, $
	kill_notify='ww_astromdestroyed',uvalue='Volvox', $
	xoffset=!dxsize-504,yoffset=!dysize-307)
;
; First two widget elements are in a row
row_wid=widget_base(astrom_wid,/row)
;
; Station selection
if n_elements(stationtable) eq 0 then get_stationtable,update=0
station_dirs=['All','Sel']
stations_directive='All'
fit_stations=stationtable.stationid
sel_stations=''
column1_wid=widget_base(row_wid,/column)
opmenustn_wid=widget_droplist(column1_wid,title='Stns:', $
	value=station_dirs,uvalue=station_dirs, $
	event_pro='ww_setstations')
stations=stationtable.stationid
list1_wid=widget_list(column1_wid,value=stations,uvalue=stations,/multiple, $
	event_pro='ww_setstations',xsize=3,ysize=3)
;
; Star selection
star_dirs=['All','Sel']
stars_directive=star_dirs[0]
fit_stars=startable.starid
sel_stars=''
column2_wid=widget_base(row_wid,/column)
opmenustr_wid=widget_droplist(column2_wid,title='Stars:', $
	value=star_dirs,uvalue=star_dirs, $
	event_pro='ww_setfitstars')
widget_control,opmenustr_wid,set_droplist_select=0
stars=startable.starid
list2_wid=widget_list(column2_wid,value=stars,uvalue=stars,/multiple, $
	event_pro='ww_setfitstars',ysize=3)
;
row2_wid=widget_base(astrom_wid,/row)
;
; Use uncorrected or corrected delays
data_dirs=['FDL','GRP','DRY','WET']
fit_data=data_dirs[0]
opmenuda_wid=widget_droplist(row2_wid,title='Data:', $
		value=data_dirs,uvalue=data_dirs, $
		event_pro='ww_setastrodata')
;
; Star parameters to solve for 
fit_dirs=['RA/Dec','P&PMs','PMs']
fit_parms=fit_dirs[0]
opmenupa_wid=widget_droplist(row2_wid,title='Parm:', $
 		value=fit_dirs,uvalue=fit_dirs, $
 		event_pro='ww_setastroparms')
;
; Night selection
night_dirs=['All','Sel']
nights_directive=night_dirs[0]
fit_nights=geninfo.date+' '+geninfo.configid
sel_nights=''
opmenunt_wid=widget_droplist(astrom_wid,title='Nights:         ', $
	value=night_dirs,uvalue=night_dirs, $
	event_pro='ww_setfitnights')
widget_control,opmenunt_wid,set_droplist_select=0
nights=fit_nights
list3_wid=widget_list(astrom_wid,value=nights,uvalue=nights,/multiple, $
	event_pro='ww_setfitnights',ysize=3)
;
; Button box
button_wid=cw_bgroup(astrom_wid,/row,['Solve','Fit','Save','HELP'], $
	event_funct='ww_volvoxfit',/return_name)
;
; Display widget
widget_control,astrom_wid,/realize
xmanager,'ww_volvoxsolve',astrom_wid,/no_block
;
end
;-------------------------------------------------------------------------------
function ww_volvoxfit,event
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
case event.value of
	'Solve':solvevolvox
	'Fit'  :fitvolvox
	'Save' :put_stationtable,/table
	'HELP' :begin
		print,'_______________________________________________'
		print,'SOLVE will calculate new coordinates for the'
		print,'selected stations and stars in a simultaneous'
		print,'fit to the selected data. All data will be used'
		print,'if either no star or no station was selected.'
		print,'If more then one night is selected, solution'
		print,'will be computed using sparse matrix Cholesky'
		print,'algorithm (HICHOL).'
		print,'FIT will just fit a common set of station '
		print,'coordinates to the selected data.'
		print,'_____________________***_______________________'
		end
	else: print,'Choice not yet supported.'
endcase
;
end
;-------------------------------------------------------------------------------
