;-------------------------------------------------------------------------------
; Container of scripts to prepare the v5 and gdl OYSTER code directories.
; v5 code is compatible with "new" IDL syntax, i.e. ()=function, []=array index
; gdl: v5 code split into one file for each procedure or function for GDL.
;
; Code here uses Landsman's idlv4_to_v5 converter, modified by C. Hummel (CAH)
; to read continuation lines. Also note that despite Landsman's comment to 
; "never change an intrinsic IDL function to square brackets, even if it is 
;  also a variable name", we are making an exception for the frequently used
; IDL intrinsic function LAMBDA (see note in code).

;-------------------------------------------------------------------------------
pro v5gdl
;
; Compound procedure to prepare v5 and gdl code for common and idl directories.
;
cd,!oyster_dir+'source/common',current=old_dir
if file_test('v5') then file_delete,'v5',/recursive
file_mkdir,'v5'
spawn,'ls *.pro',files
for i=0,n_elements(files)-1 do idlv4_to_v5,files(i),'v5'
cd,'v5'
split
;
cd,!oyster_dir+'source/idl'
if file_test('v5') then file_delete,'v5',/recursive
file_mkdir,'v5'
;
spawn,'cp -f c_gdl.pro ../gdl'
;
spawn,'ls *widget*.pro plot*.pro idlfunctions.pro',files
files=unique(files)
;
for i=0,n_elements(files)-1 do idlv4_to_v5,files(i),'v5'
cd,'v5'
split
;
cd,old_dir
;
end
;------------------------------------------------------------------------------
pro split,outdir
;
; Split IDL code containers into files, one for each function or procedure.
; Write files into folder outdir. Does not process c.pro.
;
if n_elements(outdir) eq 0 then outdir=!oyster_dir+'source/gdl'
;
files=file_search('*.pro')
if n_elements(files) eq 0 then return
;
; Do not split c.pro
index=where(files ne 'c.pro',n)
if n gt 0 then files=files(index) else return
;
for n=0,n_elements(files)-1 do begin
;
print,'Splitting container: ',files[n],'..'
;
status=dc_read_fixed(files[n],lines,/col,format='(a80)')
lines=lines(where(strlen(strcompress(lines,/remove_all)) gt 0))
l=n_elements(lines)
;
i=0L
;
while i lt l-1 do begin
repeat begin
divider1=strpos(lines(i),';****')
divider2=strpos(lines(i),';----')
proname1=strpos(strupcase(lines(i+1)),'PRO')
proname2=strpos(strupcase(lines(i+1)),'FUNCTION')
comment=strpos(strmid(lines(i+1),0,1),';')
i=i+1
start_code=(divider1 ge 0 or divider2 ge 0) $
	and (proname1 ge 0 or proname2 ge 0) $
	and (comment lt 0)
endrep until start_code or i eq l-1
;
words=nameparse(lines(i),',')
words=nameparse(words(0))
openw,unit,outdir+'/'+strlowcase(words(1))+'.pro',/get_lun
;
repeat begin
printf,unit,strtrim(lines(i))
if start_code then begin
	dpos=strpos(lines(i),'$')
	if dpos(0) lt 0 then begin
		printf,unit,'COMPILE_OPT STRICTARR,STRICTARRSUBS'
		start_code=0
	endif
endif
i=i+1
divider1=strpos(lines(i),';****')
divider2=strpos(lines(i),';----')
end_code=(divider1 ge 0 or divider2 ge 0)
endrep until end_code or i eq l-1
free_lun,unit
;
endwhile
;
endfor
;
end
;-------------------------------------------------------------------------------
function read_all_continuation_lines,inlun,n_cont,d_pos,kk
;
; Enable Landman's procedures to read IDL continuation lines
;
line=''
readf,inlun,line
kk=kk+1
len = strlen(line)
pos=strpos(line,'$') & n_pos=n_elements(pos)
if pos ge 0 then begin
	if pos(n_pos-1) eq len-1 then begin
		n_cont=n_cont+1	; Number of lines following the first one
		dpos=intarr(n_cont)
		dpos(n_cont-1)=len	; First character of continuation line
		if n_cont gt 1 then begin
			dpos(0:n_cont-2)=d_pos
			dpos(n_cont-1)=dpos(n_cont-1)+d_pos(n_cont-2)
		endif
		d_pos=dpos
		line=line+read_all_continuation_lines(inlun,n_cont,d_pos,kk)
	endif
endif
;
return,line
;
end
;------------------------------------------------------------------------------
pro write_all_continuation_lines,outlun,line,n_cont,d_pos
;
; Enable Landman's procedures to write IDL continuation lines
;
if n_cont eq 0 then begin
	printf,outlun,line
endif else begin
	printf,outlun,strmid(line,0,d_pos(0))
	d_pos=[d_pos,strlen(line)]
	for i=1,n_cont do $
		printf,outlun,strmid(line,d_pos(i-1),d_pos(i)-d_pos(i-1))
endelse
;
end
;-------------------------------------------------------------------------------
function isfunction,proname, outnames, numline
;+
; NAME:
;	ISFUNCTION()
; PURPOSE:
;	Determine whether the IDL program(s) in a file are procedures or 
;	functions.    Needed because the intrinisc RESOLVE_ROUTINE and 
;	ROUTINE_INFO() procedures require the user to know beforehand whether 
;	to supply the /IS_FUNCTION or /FUNCTION keywords.
;
; CALLING SEQUENCE:
;	status = ISFUNCTION( filename, [ outnames, numlines]
; INPUT:
;	filename = scalar string giving complete specification if file name
;		(include .pro extension)
;
; OUTPUT:
;	status - integer vector with number of elements equal to the number 
;	of routines in the file.    Each status value consists of 0 or 1
;	 1 - routine is an IDL function
;	 0 - routine is an IDL procedure
;	 If no valid IDL functions or procedures are found in the file, then
;		ISFUNCTION() returns a scalar value of -1 
;
; OPTIONAL OUTPUTS:
;	outnames - vector string, giving name of each IDL procedure or function
;		in the file
;	numlines - integer vector, giving the number of lines in each IDL
;		procedure or function in the file
; PROCEDURE CALLS:
;	FDECOMP
; REVISION HISTORY:
;	Written, W. Landsman                  June, 1995

 openr,lun,proname,/get_lun
 line = ''
 FDECOMP, proname, disk, dir, name, ext
 pname = strtrim(strlowcase(name),2)
 outnames = ''
 status = -1
 numline = 0
 nprog = 0
 step = 0
 while not eof(lun) do begin
	readf,lun,line
	comment = strpos(line,';')
	if comment EQ 0 then goto, Next_line
	if comment GT 0 then line = strmid(line,0,comment-1)
	tabpos = strpos(line,string(9b))    ;Remove all tabs
	while tabpos ne -1 do begin
		strput,line,' ',tabpos
		tabpos = strpos(line,string(9b) )
	endwhile
	line = ' ' + strlowcase(line)
		
	pos = strpos(line, ' pro ')

	if ( pos EQ 0 ) then begin	; CAH: GE 0 -> EQ 0
		comma = strpos(line,',')
		if comma GT 0 then line = strmid(line,0,comma)
		procname = strmid(line,pos+5,80)
		outnames = [outnames,procname]
		numline = [numline,step]
		status = [status,0]
                step = 0
       endif else begin
		
        pos = strpos(line,' function ')
	if ( pos EQ 0 ) then begin	; CAH: GE 0 -> EQ 0
		comma = strpos(line,',')
		if comma GT 0 then line = strmid(line,0,comma)
		procname = strmid(line,pos+10,80)
		outnames = [outnames,procname]
		numline = [numline, step]
		status = [status,1]
		step = 0
	endif
	endelse 
NEXT_LINE:
	step  = step +1

 endwhile
 free_lun,lun

; Any comments at the top of the file are included in the line count for the
; first procedure.

 if N_elements(status) GT 1 then begin
      numline = [numline,step]
      numline[2] = numline[1] + numline[2]    
      nprog = N_elements(status) - 1
      If N_elements(status) GT 1 then begin 
	outnames = strtrim( strlowcase(outnames[1:*]),2)
 	status = status[1:*]
	numline = numline[2:*]
      endif
 endif
 
 return,status
 end
;-------------------------------------------------------------------------------
pro idlv4_to_v5,infiles,outdir
;+
; NAME:
;	IDLV4_TO_V5
; PURPOSE:
;	Modify an IDL V4.0 (or earlier) procedure such that variables are 
;	indexed using square brackets, as allowed (and suggested) 
;	within IDL V5.0 and later
;
; CALLING SEQUENCE:
;	IDLV4_TO_V5, infiles, outdir 
;
; INPUTS:
;	infiles - scalar string specifying IDL procedure name(s), wild card 
;		values allowed
;	outdir - scalar string giving directory to contain output file.
;
; EXAMPLES:
;	Convert the procedure curvefit.pro in the current directory to a
;	IDL V5 version in the (Unix) idlv5 directory
;
;	IDL> idlv4_to_v5,'curvefit.pro','idlv5/'
;
;	Convert all the procedures in the current directory to IDL V5 versions
;	in the /share/idlv5 directory
;
;	IDL> idlv4_to_v5, '*.pro', '/share/idlv5/'
;
; METHOD:
;	ISFUNCTION() is used to determine all the routine names in the file,
;	and then ROUTINE_INFO() is used to determine the names of all variables
;	in the procedure.    Each (non-commented) line is scanned for
;	parentheses, and converted to square brackets if the token to the left
;	of the left parenthesis matches a variable name.
; 
; NOTES:
;	(1) Only runs under IDL V5.0 (since it calls ROUTINE_INFO())
;	(2) May possibly get confused by parenthesis within strings.
;	(3) May get confused by IDL statements that extend over multiple lines
;	    idlv4_to_v5 will supply a warning when it becomes confused by
;	    unmatched parenthesis.
;	(4) Do not include this procedure 'idlv4_to_v5' in the directory that 
;	    you are trying to convert (since it will compile the procedure 
;	    while executing it, and do a retall.)
;	(5) Conversions cannot be performed unless specified procedure(s) 
;	    already compile properly
;	(6) Will not work on IDL main programs
;	(7) May get confused by gaps between array name and parenthesis
;
; PROCEDURES CALLED:
;	FDECOMP, MATCH, REMOVE, ISFUNCTION()
; REVISION HISTORY:
;	Written  W. Landsman   Hughes STX     June 1997 
;	Variable names can have numerals      August 1997
;	Never change an intrinsic IDL function to square brackets, even if it
;	is also a variable name.
;-

 if N_params() LT 2 then begin
	print,'Syntax  - idlv4_to_v5, infiles, outdir'
	return
 endif

;Modified by CAH: findfile is obsolete
;a = findfile(infiles,count=n)
 a = file_search(infiles)
 if strlen(a(0)) eq 0 then n=0 else n=n_elements(a) 
 if n EQ 0 then message,'No files found ' + infiles
 get_lun,inlun
 get_lun,outlun

funcnames = routine_names(S_functions=-1)

line = ''

;Loop variables
;i - loop over all filenames (if wildcard value of 'infile' supplied)
;k - loop over all routines in the current filename
;kk -loop over all lines in the current routine
;j - loop over all left parentheses in the current line
;jj- loop over all right parentheses in the current line

for i=0,n-1 do begin            ;loop over each procedure name

	print,'Working on file: ',a[i]
	fdecomp,a[i],disk,dir,name,ext    ;Decompose file name
        status = isfunction(a[i], outnames,numline) 
;Resolve main procedure first, even if it means compiling twice
	g = where(outnames EQ strtrim(strlowcase(name),2),Ng)
	if Ng GT 0 then begin
		g = g[0] 
		if status[g] then resolve_routine,outnames[g],/is_function $
		else resolve_routine, outnames[g]
	endif
	

	for k = 0, N_elements(status)-1 do begin

	print,'Working on function ',outnames(k)

	case status[k] of
	 1: begin
		resolve_routine,outnames[k],/is_function
		variables = routine_info(/variables,outnames[k],/functions)
	    end
	 0: begin
		resolve_routine,outnames[k]
		variables = routine_info(/variables,outnames[k])
		end
	-1: begin
	    message,a[i] + ' will not be modified',/INF
	    goto, Done_pro 
	    end
	endcase
        
 	match, variables, funcnames, subv, Count = Nfunc
;	Code added by CAH to allow use of "lambda" as variable name
	if Nfunc gt 0 then begin
		index=where(variables(subv) ne 'LAMBDA',Nfunc)
		subv=subv(index)
	endif
;	End of added code
 	if Nfunc GT 0 then remove,subv,variables

 	if k EQ 0 then begin
		openr,inlun,a[i]
		openw,outlun, outdir + '/' + name + '.pro'
 	endif

	kk=0
	while kk lt numline[k] do begin
;	for kk=0,numline[k]-1 do begin
; CAH: read line and all continuation lines
		n_cont=0
		line=read_all_continuation_lines(inlun,n_cont,d_pos,kk)
;		readf,inlun,line
		len = strlen(line)
		pos = strpos( line, ';')
		if pos EQ -1 then begin
			goodline = line
			comment = ''
		endif else begin
			goodline = strmid(line,0, pos)
			comment = strmid(line,pos,len-pos)
		endelse
		if goodline EQ '' then goto,Done_line
	    	
		bchar = byte(goodline)

		leftparen = where(bchar EQ 40b, Nparen)
		if Nparen EQ 0 then goto, Done_line

;Variable names can contain letters, digits, underscore or a dollar sign.
;To allow structure tags and system variables, we include a period and a !
		n = strlen( goodline )

		mask = bytarr(n)
		ii = WHERE( ((bchar GE 65B) and (bchar LE 90b)) OR $
	        ((bchar GE 97B) and (bchar LE 122B)) OR $
                ((bchar GE 48B) and (bchar LE 57B)) OR $
              (bchar EQ 46B) or (bchar EQ 36B) OR $
              (bchar EQ 41B) OR $
              (bchar EQ 95B) or (bchar EQ 33B), count)
		if count GT 0 then mask[ii] = 1b else goto, Done_Line
 		pconvert = bytarr(Nparen)  ;Keep track of which paren to convert

; Now we step backward from the left parenthesis until we find the first 
; character that cannot be part of a variable name

		for j = 0, Nparen - 1 do begin
			mark =  leftparen[j] - 1
			if mark EQ -1 then goto,Done_paren
			while mask[mark] do begin
				mark = mark - 1b
				if mark EQ -1 then goto, Done_search
			endwhile
		done_search:

			if mark EQ leftparen[j]-1 then goto, Done_paren
			varname = strtrim(bchar[mark+1:leftparen[j]-1],2)
			if varname EQ '' then goto, Done_paren

; Test for structure name.   Note that for a structure x, that x.tag[3] is 
; legal in V5.0 but x.[3] is not (although x.(3) is).

			dot = strpos(varname,'.')    ;Structure name
			if dot EQ strlen(varname)-1 then goto,Done_paren
			if dot GT 0 then varname = strmid(varname,0,dot)
			g = where(variables EQ strupcase(varname), Ng)
			if Ng GT 0 then pconvert[j] = 1b
			if strmid(strtrim(varname,2),0,1) EQ '!' then $
				pconvert[j] = 1b				
		Done_paren:
		endfor
		convert = where(pconvert, Nconvert)

		if Nconvert GT 0 then begin
			bchar[leftparen[convert]] = 91b    ;byte('[')=91b
			rparen = where(bchar EQ 41b, Nrparen)
			if Nrparen EQ 0 then begin 
				message, 'Warning - no right parenthesis',/INF
				print,goodline
				goto,done_line
			endif

			for jj = 0, Nrparen - 1 do begin
			g = where(leftparen LT rparen[jj], Ng)
			if Ng EQ 0 then begin
				message,'Warning - missing left parenthesis',/INF
				print,goodline
				goto, done_line
			endif 
			leftindex = max(g)
			if pconvert[leftindex] then  bchar[rparen[jj]] = 93b 
			if N_elements(leftparen) GT 1 then $ 
			remove,leftindex,leftparen,pconvert $
			else goto, Done_rparen
			endfor
		endif
		done_rparen:
			goodline = string(bchar)
   		Done_line:  
;			printf,outlun,goodline + comment
;			Replace [] with (), for functions without parameters
			ipos=strpos(goodline,'[]')
			if ipos ge 0 then strput,goodline,'()',ipos
			write_all_continuation_lines,outlun,goodline+comment,n_cont,d_pos
	endwhile
	endfor
close,inlun
close,outlun
Done_pro:
endfor
free_lun,inlun,outlun
return
end
;------------------------------------------------------------------------------
;+
; NAME:
;     FDECOMP
; PURPOSE:
;     Routine to decompose file name(s) for any operating system
; EXPLANATION:
;     A faster version of this procedure for V5.3 or later is available in 
;     http://idlastro.gsfc.nasa.gov/ftp/v53/fdecomp.pro
;
; CALLING SEQUENCE:
;     FDECOMP, filename, disk, dir, name, qual, version, [OSFamily = ]
;
; INPUT:
;     filename - string file name(s), scalar or vector
;
; OUTPUTS:
;     All the output parameters will have the same number of elements as 
;       input filename 
;
;       disk - disk name, always '' on a Unix machine, scalar or vector string
;       dir - directory name, scalar or vector string
;       name - file name, scalar or vector string 
;       qual - qualifier, set equal to the characters beyond the last "."
;       version - version number, always '' on a non-VMS machine
;
; OPTIONAL INPUT KEYWORD:
;     OSFamily - one of the four scalar strings specifying the operating 
;             system:  'vms','Windows','MacOS' or 'unix'.    If not supplied,
;             then !VERSION.OS_FAMILY is used to determine the OS.
; EXAMPLES:
;     Consider the following file names 
;
;     Unix:    file = '/rsi/idl40/avg.pro' 
;     VMS:     file = '$1$dua5:[rsi.idl40]avg.pro;3
;     Mac:     file = 'Macintosh HD:Programs:avg.pro'
;     Windows: file =  'd:\rsi\idl40\avg.pro'
;       
;     then IDL> FDECOMP,  file, disk, dir, name, qual, version
;       will return the following
;
;                 Disk             Dir          Name        Qual     Version
;       Unix:      ''            '/rsi/idl40/'  'avg'       'pro'       ''
;       VMS:     '$1$dua5'       '[RSI.IDL40]'  'avg'       'pro'       '3'
;       Mac:     'Macintosh HD'  ':Programs:'   'avg'       'pro'       ''
;       Windows:    'd:'         \rsi\idl40\    'avg'       'pro'       ''
;
; NOTES:
;     (1) All tokens are removed between
;           1) name and qual  (i.e period is removed)
;           2) qual and ver   (i.e. VMS semicolon is removed)
;     (2) On VMS the filenames "MOTD" and "MOTD." are distinguished by the 
;         fact that qual = '' for the former and qual = ' ' for the latter.
;
;     A faster version of this procedure for V5.3 or later is available in 
;     http://idlastro.gsfc.nasa.gov/ftp/v53/fdecomp.pro
; ROUTINES CALLED:
;     Function GETTOK()
; HISTORY
;     version 1  D. Lindler  Oct 1986
;     Include VMS DECNET machine name in disk    W. Landsman  HSTX  Feb. 94
;     Converted to Mac IDL, I. Freedman HSTX March 1994          
;     Converted to IDL V5.0   W. Landsman   September 1997
;     Allow vector file name input    W. Landsman   October 2002
;     Fixed bug in version parameter introduced October 2002 W. Landsman April03
;-
;--------------------------------------------------------
;

  pro fdecomp_scalar, filename, disk, dir, name, qual, version, $
                      OSfamily = osfamily
  On_error,2                            ;Return to caller


; Find out what machine you're on, and take appropriate action.
 if not keyword_set(OSFAMILY) then osfamily = !VERSION.OS_FAMILY

 case OSFAMILY of

  "MacOS": begin

; disk name is all characters up to the first colon
; directory is string of folders         
; file name+qualifier is all characters after the last colon
; version   is null string
   
  st = filename
  if strpos(st,':') GE 0 then disk = gettok(st,':')  else disk = ''
         
     dir = ':' & tok = ''
     REPEAT BEGIN
        oldtok = tok
        tok = gettok(st,':')
        dir = dir + oldtok + ':'   
     ENDREP UNTIL tok EQ ''

       dir = strmid(dir,1,strpos(dir,oldtok)-1)   
         
     fname = oldtok     & qual = ''
     pos = strpos(fname,'.')
     if pos GE 0 then begin
       name = gettok(fname,'.')
       qual   = fname
     endif 

    version = ''
        
        end

 "vms":  begin                     ; begin VMS version

    st = filename

; get disk

    nodepos = strpos(st,'::')          ; Node name included in directory?
    if nodepos GE 0 then begin
        disk = strmid(st,0,nodepos+2) 
        st = strmid(st,nodepos+2, 999 )
    endif else disk = ''
    if strpos(st,':') GE 0 then disk = disk + gettok(st,':') + ':' else $
                                disk = disk + ''

; get dir

    if strpos( st, ']' ) GE 0 then dir = gettok( st, ']' ) + ']' else dir=''
    if strpos( st, ']' ) GE 0 then dir = dir + gettok( st, ']' ) + ']' 

; get name

    sv_name = st
    name = gettok(st,'.')

; get qualifier

    if (name + '.') EQ sv_name then qual = ' ' else $
    qual = gettok(st,';')

; get version

    version = st

  end   ;  end VMS version

 "Windows": begin

     st = filename
     pos = strpos( st, ':')                 ; DOS diskdrive (i.e. c:)
     if (pos gt 0) then disk = gettok(st,':') + ':' else disk=''

;  Search the path name (i.e. \dos\idl\) and locate all backslashes

     lpos = -1  ; directory position path (i.e. \dos\idl\)
     pos = -1
     repeat begin
        pos = strpos(st, '\',pos+1)
        if (pos GE 0) then lpos = pos
     endrep until pos lt 0

     ;  Parse off the directory path 

     if lpos ge 0 then begin
        dir = strmid(st, 0, lpos+1)
        len = strlen(st)
        if lpos eq (len-1) then $
                st = '' else st = strmid(st,lpos+1,len-lpos-1)
     endif else dir=''

; get Windows name and qualifier (extension)...qual is optional

     lpos=-1
     repeat begin                               
        pos = strpos(st,'.',pos+1)
        if (pos ge 0) then lpos = pos
    endrep until pos lt 0

    ; Parse name and qual (if a qual was found )

     if lpos ge 0 then begin
        len = strlen(st)
        name = strmid(st,0,lpos)
        qual = strmid(st,lpos+1,len-lpos-1)
     endif else begin
        name = st
        qual = '' 
     endelse

     version = ''               ; no version numbers in Windows         
     end

 ELSE: begin

    st = filename

; get disk

    disk = ''

; get dir

    lpos = -1
    pos = -1
    repeat begin
            pos = strpos(st, '/', pos+1)
            if (pos GE 0) then lpos = pos
    endrep until pos LT 0

    if lpos GE 0 then begin
            dir = strmid(st, 0, lpos+1)
            len = strlen(st)
            if lpos eq (len-1) then st = '' else $
                                    st = strmid(st,lpos+1,len-lpos-1)
    endif else dir = ''

; get name and qual

    pos = -1
    lpos = -1
    repeat begin
             pos = strpos(st,'.',pos+1)
             if (pos GE 0) then lpos = pos
    endrep until pos LT 0

    if lpos GE 0 then begin
             len = strlen(st)
             name = strmid(st,0,lpos)
             qual = strmid(st,lpos+1,len-lpos-1)
     endif else begin
         name = st
         qual = '' 
     endelse

    version = ''

 end 

ENDCASE                         ; end OTHER version
 return
 end

  pro fdecomp, filename, disk, dir, name, qual, ver, OSfamily = osfamily

  if N_params() LT 2 then begin
     print, 'Syntax - FDECOMP, filename, disk, [dir, name, qual, ver ] '
     return
  endif
  
  scalar = size(filename,/N_dimen)  eq 0
  if scalar then $
       fdecomp_scalar, filename, disk, dir, name, qual, ver, OSfamily=osfamily $
            else begin
       N = N_elements(filename)
       if N EQ 0 then message,'ERROR - Filename (first parameter) not defined'
       dir = strarr(n) & name = dir & qual = dir & disk = dir & ver = dir
       for i=0,n-1 do begin
           fdecomp_scalar, filename[i], xdisk, xdir, xname, xqual, xver, $
                  OSfamily=osfamily  
           disk[i] = xdisk & dir[i] = xdir & name[i] = xname 
           qual[i] = xqual & ver[i] = xver
       endfor
       endelse
  return
  end
;-------------------------------------------------------------------------------
