;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;fbtEwsError
;#description#
;Error handling/message printing routines for the whole EWS
;#end_class#
;*******************************************************
FUNCTION ewsErrMsg
; common for storing ews error messages, traceback stacks, and
; debugging flags
;
   COMMON ewsErrComm, ewsErrMsgs, ewsNewError, ewsTrace, ewsDebug, $
      ewsErrUnit
RETURN,ewsErrMsgs
END

PRO ewsAddErrMsg, msg, trace=trace
; add a single line of text to the ews error stack
; optionally store the current call stack (excluding ewsAddErrMsg itself).
; Normally you would do the latter when the calling routine is the
; "original" source of error, i.e. the first subroutine to detect an
; error, rather than adding comments at a higher level
;
;  INPUTS
;   msg   string      local error message
;   trace bool        if true, add trace information to common
;
   COMMON ewsErrComm, ewsErrMsgs, ewsTrace, ewsDebug, ewsErrUnit
   if(n_elements(ewsErrMsgs) le 0) then begin
      ewsErrMsgs = ''
   endif
;  ewsErrMsgs = [msg, ewsErrMsgs]
   if (KEYWORD_SET(trace)) then begin
      help, /last_message, out=ewsTrace
   endif
   RETURN
END

PRO ewsClearErrMsg
;
; blank out the ews error message stack and trace stack
;
; INPUTS/OUTPUTS   none
;
   COMMON ewsErrComm, ewsErrMsgs, ewsNewError, ewsTrace, ewsDebug, $
      ewsErrUnit
   ewsErrMsgs = ''
   ewsTrace = ''
   ewsNewError = -1
   RETURN
END

PRO ewsPrintErrMsg, noClear=noClear
; print out the EWS error stack and call stack.
; Normally these stacks are cleared after printing, unless the
; noClear keyword is set.
;
; INPUTS:
;   noClear   bool     do not clear error stack after print
;
; If ewsErrUnit is set in the ewsErrComm, the output will be
; sent to this unit, allowing output to a file.
; Otherwise it will be sent to unit 2 (syserr)
; In practice this is usually the user's screen
;
   COMMON ewsErrComm, ewsErrMsgs, ewsNewError, ewsTrace, ewsDebug, $
      ewsErrUnit
   if N_ELEMENTS(ewsErrUnit) LE 0 then eUnit = -2 else eUnit = ewsErrUnit
   if (N_ELEMENTS(ewsErrMsgs) GT 0) then begin
      outMessage = ewsErrMsgs[0:*]
      printf,eUnit, 'EWS ERROR MESSAGES'
      if (total (ewsErrMsgs NE '') gt 0) then $
         printf,eUnit, outMessage(where(outMessage NE '')),format='(a)'
   endif
   if (N_ELEMENTS(ewsTrace) gt 0) then begin
      printf, eUnit, 'Trace'
      for iT = 0, N_ELEMENTS(ewsTrace) -1 do begin 
         lTrace = ewsTrace[iT]
	 mLen = 22
	 if (STRLEN(lTrace) GE mLEN) then printf, eUnit, $
	    STRMID(lTrace, mLen, STRLEN(lTrace)-mLen)
      endfor
   endif
   if (NOT KEYWORD_SET(noClear)) then ewsClearErrMsg
   RETURN
END

PRO ewsPrintErrMain, noClear=noClear, midi=midi
;  This routine checks the call stack and calls ewsPrintErrMsg
;  if we are one level under MAIN.
;
;  INPUTS:
;   noClear  bool     do not clear messages after printing
;
;  The typical error handling code in a subroutine "ewsB(..., ierr)" will
;  look something like this:
;      ierr = 0
;      ....  innocuous code ....
;       z = ewsC(...., ierr)
;       if (ierr NE 0) then goto, err_ret
;       ... innocuous code ...
;       code that discovers a problem for the first time
;       ewsAddErrMsg, 'We made a bobo ',/trace
;       ierr = 1
;       goto, err_ret
;       .... innocuous code ....
;       return,    ;normal return
;err_ret: 
;       ewsPrintErrMain
;       return,    ;error return
;       end
   COMMON ewsErrComm, ewsErrMsgs, ewsNewError, ewsTrace, ewsDebug, $
      ewsErrUnit
;   print error messages if we are one level below main
   help,calls = callStack
   if (KEYWORD_SET(midi)) then minStack = 5 else minStack = 4
   if (N_ELEMENTS(callStack) LE minStack) then ewsPrintErrMsg, noClear=noClear
   RETURN
END

PRO ewsSetError,errMsg,notInitial=notInitial
; procedure to trigger/continue error handling in ews package
; By default (/notInitial is not set) we execute IDL MESSAGE
; routine which sets IDL error message to errMsg, saves a traceback list
; and generates a jump to error handling routines (usually a catch routine,
; can also be specified by ON_ERROR).
;
; if /notInitial is specified, errMsg is prepended to the existing
; message, and no new trace is generated
   COMMON ewsErrComm, ewsErrMsgs, ewsNewError, ewsTrace, ewsDebug, $
         ewsErrUnit
;if this is not the underlying error, set flag that we dont update trace
   ewsNewError = NOT KEYWORD_SET(notInitial)
   MESSAGE,errMsg,/NOPRINT
RETURN
END

PRO ewsCatchError, midi=midi
; standard procedure to process ews errors generated
; by other program sections
   COMMON ewsErrComm, ewsErrMsgs, ewsNewError, ewsTrace, ewsDebug, $
         ewsErrUnit
   help, /last_message, output=idlMessage
;find beginning of trace statement by existence of 'Execution halted'
   exeStatement = STRPOS(idlMessage, 'Execution halted')
   exeStatement = MIN((WHERE(exeStatement GE 0)))
   if (N_ELEMENTS(ewsNewError) LE 0) then ewsNewError = -1
   if (ewsNewError) then begin
      ewsErrMsgs = idlMessage[0:exeStatement-1]
      ewsTrace   = idlMessage[exeStatement:*]
   endif else begin
      ewsErrMsgs = [idlMessage[0:exeStatement-1], ewsErrMsgs]
   endelse
   ewsNewError = -1
   ewsPrintErrMain, midi=midi
RETURN
END

