;*******************************************************************************
; File: mymatissegui.pro
; Front-end for Reflex MATISSE data reduction
; Part of OYSTER (Christian Hummel)
;
; Block directory:
; ----------------
;
; Block 1: matisse_photscale,matisse_cal_response,matisse_response
;	   matisse_zerocounts,compile_zerocounts,qc_zerocounts,
;	   mdfc_cal,matisse_removebcd,matisse_listbcd,matisse_add_history
; Block 2: alloc_matisse_options,dfits_matisse,rm_unwanted,tartyp,
;	   rename2arc_matisse,rename2cal_matisse
; Block 3: analyzematisseendproducts,analyzematisseconfigids,matisseconfigids,
;	   selectmatissefiles, matissestarids,merge_intphot
; Block 4: matissegui_event,matissegui_productdirs,matissegui_configid,
;	   files_event,matissegui_options,create_all_cal_xdr,
;	   mymatissegui,mymatissepipe,mymatisseplots,
; Block 5: linknightstoendproducts,renamesciopsendproducts
;
;************************************************************************Block 1
pro matisse_photscale,calstar,scistar,template=template,noplot=noplot
;
; Given a calibrator, obtain information on absolute infrared fluxes and
; combine with measured spectra to derive scaling factor which convert
; counts into fluxes in Jansky.
;
; e.g. photscale,'HDN107446','USR0001',template='hd163588_K2III.tem'
;
; This procedure is based on midi/photscale
;
common StarBase,StarTable,Notes
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(noplot) eq 0 then noplot=0
;
; Get 12 micron fluxes, i.e. color-corrected IRAS fluxes
get_iras
;
; Get template fluxes, function cohen is in atmospheres.pro.
f=cohen(calstar,l,template=template)	; W/cm^2/micron, lambda [m]
; hnu=6.63e-27*3e8/l 	; This is the energy (ergs) of a photon of lambda
; f=f*hnu*1e-4		; W/cm^2/micron
nl=n_elements(l)
index=where(l(1:nl-1)-l(0:nl-2) gt 0)
index=[0,index+1]	; remove duplicates
l=l(index)
f=f(index)
; Get transmission function of IRAS filter
t_iras=iras_f12(l*1e9)
f_iras=total(f*t_iras)/total(t_iras)
l_iras=total(l*t_iras)/total(t_iras)
f_iras=f_iras*(l_iras*1e6)^2/3e-16	; convert to Jy
; Compute IRAS flux of template
print,'Predicted IRAS flux for template [Jy]: ',f_iras
; Convert W/cm^2/micron to Jy
f=f*(l*1e6)^2/3e-16	; Jy
lambda=genconfig.wavelength(0:genconfig.numspecchan(0)-1,0)
; Interpolate template fluxes for spectrometer
fluxes=spline(l,f,reverse(lambda))
fluxes=reverse(fluxes)
index=where(genconfig.wavelength(0:genconfig.numspecchan(0)-1,0) gt 11.5e-6 $
        and genconfig.wavelength(0:genconfig.numspecchan(0)-1,0) lt 12.5e-6)
i=where(startable.starid eq calstar)
; print,'12 micron flux for template [Jy]: ',avg(fluxes(index))
print,'IRAS flux for calibrator [Jy]: ',startable(i).f12*1.4
f12tem=f_iras
; Normalize template to calibrator flux using the ratio of IRAS fluxes
fluxes=fluxes/f12tem*(startable(i).f12*1.4)
;
; Initialize scale factors with airmass correction
for i=0,n_elements(scans)-1 do $
        scans(i).photometryscale=1./cos(scans(i).za/(180/!pi))
;
index=where(scans.starid eq calstar,count)
counts=(scans(index).photometry*scans(index).photometryscale) > 0
;
for k=0,genconfig.numsid-1 do begin
for l=0,genconfig.numoutbeam-1 do begin
for j=0,genconfig.numspecchan(l)-1 do begin
index=where(counts(k,l,j,*) ne 0,count)
scale=0
scale_err=0
if count gt 1 then begin
	mcount=mean(counts(k,l,j,index))
	ecount=stddev(counts(k,l,j,index))
	scale=fluxes(j)/mcount
	scale_err=scale*(ecount/mcount)
endif else if count eq 1 then begin
	mcount=mean(counts(k,l,j,index))
	scale=fluxes(j)/mcount
endif
for i=0,n_elements(scans)-1 do begin
        scans(i).photometryscale(k,l,j)=scans(i).photometryscale(k,l,j)*scale
        if scans(i).starid eq calstar then scans(i).photometryerr(k,l,j)=scale_err
endfor
endfor
endfor
endfor
;
; Some diagnostic plotting...
;
; Conversion factor, converts counts into fluxes
if not noplot then begin
index=where(scans.starid eq calstar,count)
f=scans(index).photometryscale
l=genconfig.wavelength(*,0)*1e6
!p.multi=[0,2,2]
!y.title='Conversion factor'
!x.title='Wavelength ['+greek('mu')+'m]'
!y.range=[0,0.2]
if strmid(genconfig.stationid(0),0,2) eq 'UT' then !y.range=[0,0.002]
if !d.name ne 'PS' then window,/free else device,/color
for k=0,1 do begin
for j=0,1 do begin
	!p.title=calstar+' (IB '+string(k+1,format='(i1)') $
			+', OB '+string(j+1,format='(i1)')+')'
	plot,lambda*1e6,lambda*0,/nodata,xrange=[8,13]
	for i=0,count-1 do begin
		x=l
		y=reform(f(k,j,*,i))
		oplot,x,y,color=tci(3)
	endfor
endfor
endfor
endif
;
; Calibrator
if not noplot then begin
index=where(scans.starid eq calstar,count)
f=scans(index).photometry*scans(index).photometryscale
e=scans(index).photometryerr
l=genconfig.wavelength(*,0)*1e6
!p.multi=[0,2,2]
!y.title='F!D'+greek('nu')+'!N [Jy]'
!x.title='Wavelength ['+greek('mu')+'m]'
!y.range=0
if !d.name ne 'PS' then window,/free else device,/color
for k=0,1 do begin
for j=0,1 do begin
	!p.title=calstar+' (IB '+string(k+1,format='(i1)') $
			+', OB '+string(j+1,format='(i1)')+')'
	plot,lambda*1e6,fluxes,psym=0,xrange=[8,13],thick=3
	for i=0,count-1 do begin
		x=l
		y=reform(f(k,j,*,i))
		z=reform(e(k,j,*,i))
		jndex=where(y gt 0 and z gt 0,n)
		if n gt 0 then oplot,x(jndex),y(jndex),color=tci(3)
	endfor
endfor
endfor
endif
;
; Science target
if n_elements(scistar) eq 1 then begin
index=where(scans.starid eq scistar,count)
f=scans(index).photometry*scans(index).photometryscale
e=scans(index).photometryerr
l=genconfig.wavelength(*,0)*1e6
lambda=l(where(l gt 7 and l lt 13)) & si=sort(lambda) & lambda=lambda(si)
fluxjy=lambda*0
fluxsq=fluxjy
numspectra=0
!p.multi=0
!x.range=[7.5,13]
!p.title=scistar
!p.title=''
!p.thick=2
!p.charsize=2
if !d.name ne 'PS' then window,/free else device,/color
for k=0,1 do begin
for j=0,1 do begin
	for i=0,count-1 do begin
		x=l
		y=reform(f(k,j,*,i))
		z=reform(e(k,j,*,i))
		index=where(y gt 0 and z gt 0,n)
		x=x(index)
		y=y(index)
		si=sort(x) & x=x(si) & y=y(si)
		if n gt 0 then begin
			if numspectra eq 0 then begin
				index=where(x gt 7 and x lt 13)
				plot,x,y,/nodata, $
				yrange=[min(y(index))*0,max(y(index))*1.2]
			endif
;			oplot,x,y,color=tci(3),psym=0
			fluxkj=spline(x,y,lambda)
			fluxjy=fluxjy+fluxkj
			fluxsq=fluxsq+fluxkj^2
			numspectra=numspectra+1
		endif
	endfor
endfor
endfor
fluxjy=fluxjy/numspectra
fluxsq=fluxsq/numspectra
oplot,lambda,fluxjy
fluxsg=sqrt(fluxsq-fluxjy^2)
oploterr,lambda,fluxjy,fluxsg,3
; xyouts,12.4,22.5,'[NeII]',charsize=1.5
openw,unit,scistar+'.txt',/get_lun
for i=0,n_elements(lambda)-1 do printf,unit,lambda(i),fluxjy(i),fluxsg(i)
free_lun,unit
endif
;
if !d.name ne 'PS' then set_screen else device,/close
;
end
;-------------------------------------------------------------------------------
pro matisse_cal_response,starid
;
; For a star labeled as calibrator (bflag='C') in the StarTable,
; compute the response function from the observed flux (O) and the
; stellar template flux (T) as T/O. Save with name corresponding to setup.
;
; T is derived from the NextGen spectrum closest in Teff and log(g).
; This the startable must have the info for teff and logg.
;
; Set up grid of wavelengths for band pass integrations
; Note that all functions in filter.pro use [nm] as units for lambda
;
; The data selected for this calibration should all have the same station
; configuration, and all OUT-OUT BCD combination, or BCD "removed."
; To check:
; dfits *.fits | fitsort INS.BCD1.NAME INS.BCD2.NAME
; dfits *.fits | fitsort TELESCOP DET.NAME INS.DIL.ID SEQ.DIL.WL0  INS.DIN.ID
;
; The procedure will average over several scans on the same star.
; Only one star can be processed at a time, with results labeled with StarID.
;
; First version using data in irs9a/2019-12-10/mat_raw_estimates_L/N_noBCD
; Recommended LOW-L/N data (HDN123139):
; get_oifits,'CAL_HD123139_CALIB_RAW_INT_2019-05-11T02:21:28.1063.fits'
; get_oifits,'CAL_HD123139_CALIB_RAW_INT_2019-05-11T02:21:28.5709.fits'
; get_oifits,'CAL_HD123139_CALIB_RAW_INT_2019-05-11T02:22:51.7036.fits'
; get_oifits,'CAL_HD123139_CALIB_RAW_INT_2019-05-11T02:22:52.1624.fits'
; get_oifits,'CAL_HD123139_CALIB_RAW_INT_2019-05-11T02:24:12.8549.fits'
; get_oifits,'CAL_HD123139_CALIB_RAW_INT_2019-05-11T02:24:13.3474.fits'
; get_oifits,'CAL_HD123139_CALIB_RAW_INT_2019-05-11T02:25:32.8374.fits'
; get_oifits,'CAL_HD123139_CALIB_RAW_INT_2019-05-11T02:25:33.3330.fits'
; get_oifits,'CAL_N-epssco-medium02_CALIB_RAW_INT_2019-04-28T09:05:09.5818.fits'
; get_oifits,'CAL_N-epssco-medium02_CALIB_RAW_INT_2019-05-01T08:51:17.5398.fits'
; get_oifits,'CAL_N-epssco-medium04_CALIB_RAW_INT_2019-04-28T10:05:18.8995.fits'
; get_oifits,'CAL_N-epssco-medium06_CALIB_RAW_INT_2019-04-30T07:32:53.2511.fits'
; Recommended LOW-L/N data with central L wavelength setting 4.1:
; e_Cen, eps_Cru, hd104646, q_Car, r_Car
; Recommended MED-L data (HDN146686):
; get_oifits,'CAL_L-gam2Nor_CALIB_RAW_INT_2019-05-04T06:10:00.1391.fits'
; get_oifits,'CAL_L-gam2Nor_CALIB_RAW_INT_2019-05-04T06:11:21.2035.fits'
; get_oifits,'CAL_L-gam2Nor_CALIB_RAW_INT_2019-05-04T06:12:43.5918.fits'
; get_oifits,'CAL_L-gam2Nor_CALIB_RAW_INT_2019-05-04T06:14:04.4674.fits'
; Recommended HIGH-L data (HDN151680):
; get_oifits,'CAL_N-epssco-medium02_CALIB_RAW_INT_2019-04-28T09:05:09.1451.fits'
; get_oifits,'CAL_N-epssco-medium02_CALIB_RAW_INT_2019-05-01T08:51:17.1351.fits'
; get_oifits,'CAL_N-epssco-medium04_CALIB_RAW_INT_2019-04-28T10:05:18.4300.fits'
; get_oifits,'CAL_N-epssco-medium06_CALIB_RAW_INT_2019-04-30T07:32:52.8285.fits'
; Recommended LOW-L/HIGH-N data, with SEQ.DIL.WL0 added to data base
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(starid) eq 0 then begin
	starid=startable(0).starid
	print,'StarId not specified, computing response for '+starid
endif
;
is=where(startable.starid eq starid,n)
if n eq 0 then begin
	print,'Star not found!'
	print,'Please select from: '
	print,strjoin(startable.starid)
	return
endif
;
; Make sure we have the spectral information
if startable(is).spectrum eq 'UNKNOWN' $
 or strlen(startable(is).spectrum) eq 0 then begin
 	print,'Spectrum of star unknown! Trying to retrieve from catalog...'
 	startable_tmp=startable
	startable=startable(is)
	read_catalogs
	startable_tmp(is)=startable
	startable=startable_tmp
	print,'Spectrum retrieved from catalog: '+startable.spectrum
endif
;
teff=teff_star(startable(is).spectrum)
logg=logg_star(startable(is).spectrum)
startable(is).teff=teff
startable(is).logg=logg
r=limbgrid(teff,logg,lambda_atm,limbdu,flux_atm)
;
index_sc=where(scans.starid eq startable(is).starid,count_sc)
print,'Number of scans on this star:',count_sc

;
cf=fltarr(genconfig.numsid,genconfig.numoutbeam,max(genconfig.numspecchan))
lf=fltarr(genconfig.numoutbeam,max(genconfig.numspecchan))
;
for ob=0,GenConfig.NumOutBeam-1 do begin
;
; Compute stellar template fluxes
lambda_grid=system_config(SystemId,'GRID')
if string(lambda_grid(0)) eq 'SPECTROMETER' then begin
	grid_type='SPECTROMETER'
	lambda_grid=genconfig.wavelength(0:genconfig.numspecchan(ob)-1,ob)*1e9
	fluxes=interpol(flux_atm,lambda_atm,lambda_grid)
endif else begin
	grid_type='GRID'
	lambda_grid=lambda_atm
	ch=indgen(genconfig.numspecchan(ob))
	jndex=where(lambda_grid*1e-9 le max(genconfig.wavelength(ch,ob) $
					   +genconfig.chanwidth(ch,ob)/2) $
		and lambda_grid*1e-9 ge min(genconfig.wavelength(ch,ob) $
					   -genconfig.chanwidth(ch,ob)/2),count)
	if count gt 0 then begin
		lambda_grid=lambda_grid(jndex)
		fluxes=flux_atm(jndex)
	endif
endelse
;
nc=GenConfig.NumSpecChan(ob)
lamda=genconfig.wavelength(*,ob)
;
insmode=instrument_id(systemid) & wl0=''
;
; Define wavelength ranges (index j) with good throughput for matching
; These ranges may be affected by lines, however...
if strpos(insmode,'N-LR') ge 0 then begin
	filestub='N-LR'
	j=where(lamda ge 8.5e-6 and lamda le 9.0e-6)
	!y.range=[0.1,10.]
endif else if strpos(insmode,'N-HR') ge 0 then begin
	filestub='N-HR'
	j=where(lamda ge 8.0e-6 and lamda le 9.0e-6)
	!y.range=[0.1,10.]
endif else if strpos(insmode,'L-LR') ge 0 then begin
	filestub='L-LR'
	j=where(lamda ge 3.5e-6 and lamda le 3.9e-6)
	wl0=string(median(genconfig.wavelength)*1e6,format='(f4.2)')
endif else if strpos(insmode,'L-MR') ge 0 then begin
	filestub='L-MR'
	j=where(lamda ge (min(lamda)+0.02e-6) and $
		lamda le (max(lamda)-0.02e-6))
	wl0=string(median(genconfig.wavelength)*1e6,format='(f4.2)')
endif else if strpos(insmode,'L-HR') ge 0 then begin
	filestub='L-HR'
	j=where(lamda ge (min(lamda)+0.01e-6) and $
		lamda le (max(lamda)-0.01e-6))
	wl0=string(median(genconfig.wavelength)*1e6,format='(f5.3)')
endif else if strpos(insmode,'L-HP') ge 0 then begin
	filestub='L-HP'
	j=where(lamda ge (min(lamda)+0.01e-6) and $
		lamda le (max(lamda)-0.01e-6))
	wl0=string(median(genconfig.wavelength)*1e6,format='(f6.4)')
endif else begin
	print,'Insmode '+insmode+' not known!'
	return
endelse
;
!p.multi=[0,2,2]
!p.psym=0
;
for id=0,GenConfig.NumSid-1 do begin
;
	is=genconfig.siderostatid(id)-1	; store by siderostat ID
	stationid=genconfig.stationid(where(genconfig.siderostatid eq is))
;
;	Bandpass integration
	if grid_type eq 'SPECTROMETER' then begin
;		Store stellar template flux
		for i=0,count_sc-1 do $
		scans(index_sc(i)).photometrym(is,ob,0:nc-1)=fluxes
	endif else begin
;		Grid
		for ch=0,nc-1 do begin
			filters=system_config(SystemId,'FILTERS')
			if n_elements(filters) gt 1 then filter=filters(ch,ob) $
						    else filter=filters(0)
			if filter(0) eq 'generic_c' then $
				tm=generic_c(lambda_grid,ch,ob) $
						 else $
				tm=call_function(filter,lambda_grid)
			if total(tm) eq 0 then begin
				print,'***Error(MATISSE_FLUX): grid too coarse!'
				return
			endif
;			Store stellar template flux
			for i=0,count_sc-1 do begin
			scans(index_sc(i)).photometrym(is,ob,ch)= $
				total(tm*fluxes)/total(tm)
			endfor
		endfor
	endelse
;
;	Compute and save response function...
;	First, normalize templates with "calibrated" flux in sel. ranges
	for i=0,count_sc-1 do begin
		mf=median(scans(index_sc(i)).photometrym(is,ob,j) $
			 /scans(index_sc(i)).photometry(is,ob,j))
		scans(index_sc(i)).photometrym(is,ob,*)= $
		scans(index_sc(i)).photometrym(is,ob,*)/mf
	endfor
;	Now compute the conversion factor for the whole wavelength range 
	lf(ob,0:nc-1)=lamda
	!y.range=0
	for i=0,count_sc-1 do begin
		case filestub of
		'N-HR':	begin
			r=polyfit(lamda,reform( $
			scans(index_sc(i)).photometry(is,ob,0:nc-1)),2)
			cf(is,ob,0:nc-1)=cf(is,ob,0:nc-1)+ $
			 	scans(index_sc(i)).photometrym(is,ob,0:nc-1) $
				/polynom(lamda,r)
;			plot,lamda,scans(index_sc(i)).photometry(is,ob,0:nc-1)
;			oplot,lamda,polynom(lamda,r)
			end
		else: 	cf(is,ob,0:nc-1)=cf(is,ob,0:nc-1)+ $
			 	scans(index_sc(i)).photometrym(is,ob,0:nc-1) $
				/scans(index_sc(i)).photometry(is,ob,0:nc-1)
		endcase
	endfor
	cf=cf/count_sc
;	Check for negative flux values
	index=where(cf(is,ob,*) lt 0,count)
	if count gt 0 then begin
		print,'Warning: negative chopped fluxes: '+stationid
		cf=abs(cf)
		color=tci(2)
	endif else color=tci(1)
	plot,lf*1e6,cf(is,ob,*),title=starid+', '+stationid+ $
		', SidID='+string(is+1,format='(i2)'),/ylog,color=color, $
		xtitle='Response (template/observed) versus lambda [mu]'
;
endfor	; Telescope
;
endfor	; Spectrometer
;
if strlen(wl0) ne 0 then insmode=insmode+'_'+wl0
filename=insmode+'_'+starid+'_response.xdr'
save,lf,cf,filename=filename
print,'File saved to '+filename
print,'Please remember to move file to '+!oyster_dir+'source/matisse'
print,'and rename verified file to '+insmode+'_response.xdr'
;
end
;-------------------------------------------------------------------------------
pro matisse_response,skip=skip
;
; Calibrate Photometry of science targets response functions computed by
; matisse_response: scans.photometryc=scans.photometry*response
;
common Starbase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; For debugging purposes
if keyword_set(skip) then begin
	scans.photometryc=scans.photometry
	scans.photometrycerr=scans.photometryerr
	return
endif
;
response=!oyster_dir+'source/matisse/'+instrument_id(systemid)+'_response.xdr'
if file_test(response) eq 0 then begin
	print,'Flux calibration file not found ('+response+')'
	return
endif
restore,response	; restores cf and lf
;
for ob=0,GenConfig.NumOutBeam-1 do begin
for ib=0,GenConfig.NumSid-1 do begin
;
lamda=genconfig.wavelength(*,ob)
si_lamda=sort(lamda)
;
; Interpolate response data on this wavelength scale
sigma=0	; for cubic spline fit (works better for values outside range)
;
lf_s=reform(lf(ob,*))
cf_s=reform(cf(ib,ob,*))
si=sort(lf_s)
cf_i=spline(lf_s(si),cf_s(si),lamda(si_lamda),sigma) & cf_i=cf_i(si_lamda)
;
for i=0,n_elements(scans)-1 do begin
	scans(i).photometry(ib,ob,*)=scans(i).photometry(ib,ob,*)*cf_i
	scans(i).photometryerr(ib,ob,*)=scans(i).photometryerr(ib,ob,*)*cf_i
endfor
;
endfor
endfor
;
end
;-------------------------------------------------------------------------------
function matisse_zerocounts,ob,zcs=zcs,tm=tm
;
; Compute the "zerocounts", i.e., the integrated fluxes recorded in Photometry 
; per unit of time [s] and source flux [Jy]. Do this for each telescope, and 
; compute the median over all targets with known brightness.
; To use: zcs=matisse_zerocounts() for N or L bands.
;
; The value should be around 0.17 (L) and close to 0.08 (N) and for each AT.
;
; The function can also be used to "predict" fluxes based on the photometry,
; adopting the zeropoints passed in zcs. zcs values are ordered by telescope ID,
; not StationID. The zcs calibration is mode dependent, examples below:
; To use for LM-band: print,matisse_zerocounts(zcs=[fltarr(4)+0.17])
; To use for N-band:  print,matisse_zerocounts(zcs=[fltarr(4)+0.08])
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
;
; Get FLP and F12 fluxes but don't overwrite existing values
index_f12=where(startable.f12 gt 0,count_f12)
index_flp=where(startable.flp gt 0,count_flp)
if count_f12 gt 0 then f12=startable(index_f12).f12
if count_flp gt 0 then flp=startable(index_flp).flp
; Read L and N band fluxes from the MDFC catalog (Cruzalebes et al.)
get_mdfc
if count_f12 gt 0 then startable(index_f12).f12=f12
if count_flp gt 0 then startable(index_flp).flp=flp
;
zerocounts=fltarr(n_elements(startable),genconfig.numsid)
valid_scans=intarr(n_elements(scans),genconfig.numsid)
;
; For the time being, we can process only a single output beam
if n_elements(ob) eq 0 then ob=0
;
photometry=reform(scans.photometry(*,ob,*))
photometryerr=reform(scans.photometryerr(*,ob,*))
for i=0,n_elements(scans)-1 do begin
	photometry(*,*,i)=photometry(*,*,i)/scans(i).int_time
	for j=0,genconfig.numsid-1 do begin
		index=where(photometryerr(j,*,i) gt 0,count)
		if count gt 10 then valid_scans(i,j)=1
	endfor
endfor
;
; Compute filter function (L or N), adopting the Johnson system
if strpos(instrument_id(systemid),'MATISSE-N') ge 0 then begin
	tm=call_function('johnson_n',genconfig.wavelength(*,ob)*1e9)
	ref_flux=startable.f12
endif else begin
	tm=call_function('johnson_lp',genconfig.wavelength(*,ob)*1e9)
	ref_flux=startable.flp
endelse
;
total_flux=fltarr(n_elements(startable),genconfig.numsid)
mean_tflux=fltarr(n_elements(startable),genconfig.numsid)
;
; Integrate photometry over the band weighted with the filter function
for i=0,n_elements(startable)-1 do begin
	for j=0,genconfig.numsid-1 do begin
;		Average of all observations of this star
		index=where(scans.starid eq startable(i).starid $
			and valid_scans(*,j) eq 1,count)
		for k=0,count-1 do begin
			total_flux(i,j)=total_flux(i,j)+ $
			total(median(reform(photometry(j,*,index(k))),5)*tm) $
				/total(tm)
		endfor
		if count gt 0 then begin
		   if not keyword_set(zcs) and ref_flux(i) ne 0 then begin
			zerocounts(i,j)=(total_flux(i,j)/count)/ref_flux(i)
			total_flux(*)=0
		   endif else if keyword_set(zcs) then begin
			total_flux(i,j)=total_flux(i,j)/count
		   endif
		endif

	endfor
endfor
;
if keyword_set(zcs) then begin
;	Predict fluxes based on values in zcs
	fln=fltarr(n_elements(startable),genconfig.numsid)
	for j=0,genconfig.numsid-1 do begin
	for i=0,n_elements(startable)-1 do begin
		fln(i,j)=total_flux(i,j)/zcs(j)
	endfor
	endfor
	return,fln
endif else begin
;	Inform of zero counts computed based on known fluxes 
	print,'Zero counts:'
	print,'    '+strjoin(startable.name,'    ')
	print,zerocounts
endelse
;
zcs=fltarr(genconfig.numsid)
for i=0,genconfig.numsid-1 do begin
	index=where(zerocounts(*,i) gt 0,count)
	if count gt 0 then zcs(i)=median(zerocounts(index,i))
endfor
;
; Return zero counts in the order of telescope ID, *not* station ID
si=sort(genconfig.siderostatid)
return,zcs(si)
;
end
;-------------------------------------------------------------------------------
pro matisse_add_history,files=files,record=record
;
; MATISSE reduced data files "merged" by mat_mergeAllOiFits.py (mat_tools)
; have OI_FLUX data values 4 times higher than before. This procedure should
; be used to add a history record to a file indicating it having been "merged"
; so that the photometric calibration using zerocounts based on "non-merged"
; files does not over-estimate the fluxes in units of Jy.
;
if not keyword_set(record) then begin
	record='Processed by mat_mergeAllOiFits'
	print,'Will add history record: ',record
endif
;
if not keyword_set(files) then begin
	files='*.fits'
	print,'Will process all FITS files'
endif
;
f=file_search(files)
if strlen(f(0)) eq 0 then begin
	print,'Error: files not found!'
	return
endif
;
for i=0,n_elements(f)-1 do begin
	header = headfits(f(i))
	sxaddhist,record,header
	modfits,f(i),0,header
endfor
;
print,'File headers updated.'
;
end
;-------------------------------------------------------------------------------
pro compile_zerocounts,update=update
;
; Run in a directory of reduced data to compile data base of zerocounts
; for all instrument modes encountered. Save in file "zerocounts.xdr".
; Only process calibrator data, e.g., in /iot/matisse/sv/sv_raw_estimates
; Before computing the zerocounts, the spectra are corrected for response.
;
; Use only data with INS.BCD1.NAME=OUT and INS.BCD2.NAME=OUT!
; For faster processing, the other files will be deleted!
; Use only data *not* processed by mat_mergeAllOiFits!
;
; Use code below to manually add other fields before running this procedure:
; restore,'zerocounts.xdr'
; z=zerocounts
; n=n_elements(z)
; zerocounts=replicate({ $
;  starid:'', $
;  target:'', $
;  band:'', $
;  ins_mode:'', $
;  det_mode:'', $
;  det_dit:'', $
;  tpl_nexp:'', $
;  array:'', $
;  zcs:fltarr(4)},n)
; zerocounts.starid=z.starid
; zerocounts.target=z.target
; zerocounts.band=z.band    
; zerocounts.ins_mode=z.ins_mode
; zerocounts.det_mode=z.det_mode
; zerocounts.det_dit=z.det_dit  
; zerocounts.tpl_nexp=z.tpl_nexp
; zerocounts.array='AT'
; zerocounts.zcs=z.zcs
; save,zerocounts,file='zerocounts.xdr'
;
; If update=1, append/replace data with new data after creating backup copy
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Starbase,StarTable,Notes
;
files=file_search('CAL_*_CALIB_RAW_INT_*.fits')
n=n_elements(files)
;
; Check for the application of mat_mergeAllOiFits
; d=mrdfits(files(0),0,h)
; if fitshparse(h,'mat_mergeAllOiFits') 
;
; Use character string for floating-point variable det_dit for equality tests
; ...and same for dil_wl0
z=replicate({ $
	starid:'', $
	target:'', $
	band:'', $
	ins_mode:'', $
	det_mode:'', $
	dil_wl0:blanks(4), $
	det_dit:blanks(10), $
	tpl_nexp:'', $
	array:'', $
	zcs:fltarr(4)},n)
bcd_ok=intarr(n)
;
for i=0,n-1 do begin
;	Check BCD combination
	d=mrdfits(files(i),0,h,/silent)
	if fitshparse(h,'INS.BCD1.NAME') ne 'OUT' or $
	   fitshparse(h,'INS.BCD2.NAME') ne 'OUT' then file_delete,files(i)
	if fitshparse(h,'INS.BCD1.NAME') ne 'OUT' or $
	   fitshparse(h,'INS.BCD2.NAME') ne 'OUT' then continue
   	bcd_ok(i)=1
	get_oifits,files(i)	; One OB only (L or N band)
	if strpos(genconfig.stationid(0),'A') eq 0 then z(i).array='AT'
	if strpos(genconfig.stationid(0),'U') eq 0 then z(i).array='UT'
	z(i).starid=startable(0).starid
	z(i).target=startable(0).name
	d=mrdfits(files(i),0,h,/silent)
	if strpos(genconfig.configid,'_L_') ge 0 then begin
		z(i).band='L'
		z(i).ins_mode=fitshparse(h,'INS.DIL.ID')
		z(i).dil_wl0=fitshparse(h,'SEQ.DIL.WL0')
	endif else begin
		z(i).band='N'
		z(i).ins_mode=fitshparse(h,'INS.DIN.ID')
	endelse
	z(i).det_mode=fitshparse(h,'DET.READ.CURNAME')
;	Make sure z.det_dit remains a 10 character string!
	det_dit=blanks(10)
	det_dit='0000000000'
	strput,det_dit,strtrim(string(fitshparse(h,'DET.SEQ1.DIT')),1)
	z(i).det_dit=det_dit
;	Make sure z.dil_wl0 remains a 4 character string!
	dil_wl0=blanks(4)
	dil_wl0='0000'
	strput,dil_wl0,strtrim(string(fitshparse(h,'SEQ.DIL.WL0')),1)
	z(i).det_dit=det_dit
	z(i).tpl_nexp=strtrim(string(fix(fitshparse(h,'TPL.NEXP'))),1)
;	Compute zerocounts
	zcs=matisse_zerocounts(0); One OB only (L or N band)
	z(i).zcs= $	; Reorder by SiderostatID
		zcs(uniq(genconfig.siderostatid,sort(genconfig.siderostatid)))
	if keyword_set(update) then print_zerocount,z(i)
endfor
;
z=z(where(bcd_ok eq 1))
;
index=where(total(z.zcs,1) eq 0,count)
if count gt 0 then begin
	print,'Targets not found in MDFC:'
	print,unique(z(index).starid)
endif
index=where(total(z.zcs,1) gt 0)
z=z(index)
;
z_file=!oyster_dir+'source/matisse/zerocounts.xdr'
; Make backup before updates
b_file=basename(z_file)+'_'+strjoin(nameparse(systime()),'-')+'.xdr'
if file_test(z_file) then spawn,'cp '+z_file+' '+b_file
print,'Saved previous data base to:'
print,b_file
if keyword_set(update) and file_test(z_file) then begin
	restore,z_file
	zerocounts=[zerocounts,z]
endif else zerocounts=z
;
save,zerocounts,file=z_file
;
end
;-------------------------------------------------------------------------------
pro print_zerocount,z
l=''
for i=1,n_tags(z)-2 do l=l+' '+z.(i)
l=l+strjoin(z.(i))
print,strcompress(l)
;
end
;-------------------------------------------------------------------------------
pro qc_zerocounts
;
; Print summary for QC
;
restore,!oyster_dir+'source/matisse/zerocounts.xdr'
z0=zerocounts
;
setups='Band '+z0.band+' '+z0.det_mode+' DIT '+ $
	z0.det_dit+' NEXP '+z0.tpl_nexp
usetups=unique(setups)
n=n_elements(usetups)
;
for i=0,n-1 do begin
	z1=z0(where(setups eq usetups(i)))
	starids=unique(z1.starid)
	print,'Results for setup: '+usetups(i)
	print,'--------------------------------------------------------------'
	for j=0,n_elements(starids)-1 do begin
		print,starids(j)
		index=where(z1.starid eq starids(j))
		print,z1(index).zcs
	endfor
	print,''
endfor
;
end
;-------------------------------------------------------------------------------
pro mdfc_cal
;
; Based on the L and N-band fluxes tabulated in the MDFC catalog,
; establish calibrations of the flux as a function of the stellar
; eff. temperatures, diameters, and distances.
;
common StarBase,StarTable,Notes
;
; Read SB9 catalog of spectroscopic binaries
get_sb9
hic_sb9=startable(where(startable.hic gt 0)).hic
;
; Get Hipparcos catalog with GAIA parallaxes
get_hip,/gaia
startable=startable(where(startable.px ne 0))
;
; Remove spectroscopic binaries
index=whereequal(startable.hic,hic_sb9)
startable(index).hic=0
startable=startable(where(startable.hic gt 0))
;
; Only use stars with both HIP and HDN IDs (MDFC does not have HIP IDs)
startable_hip=startable(where(startable.hdn gt 0))
;
; The following command was alread executed and the result saved in mdfc.xdr
; get_mdfc
; Restore calibration tables: rlg,rld,rng,rnd
restore,!oyster_dir+'starbase/MDFC/mdfc.xdr'
px=startable.px
;
i=where(startable.type gt 0 and $
	startable.type lt 60 and $
	startable.class ge 3 and $
	startable.f12 gt 0 and $
	startable.flp gt 0)
startable=startable(i)
;
; Update parallaxes with GAIA values
index=whereequal(startable.hdn,startable_hip.hdn)
for i=0l,n_elements(index)-1 do begin
	j=where(startable_hip.hdn eq startable(index(i)).hdn) & j=j(0)
	startable(index(i)).px=startable_hip(j).px
	startable(index(i)).pxe=startable_hip(j).pxe
	startable(index(i)).hic=startable_hip(j).hic
endfor
;
; Remove entries with "exotic" spectral types
startable=startable(where(strpos(startable.spectrum,'C7') lt 0))
startable=startable(where(strpos(startable.spectrum,'C6') lt 0))
startable=startable(where(strpos(startable.spectrum,'C5') lt 0))
;
; Select stars not too far away (px > 10 mas)
startable=startable(where(startable.px gt 0.010))
;
; Scale fluxes to a distance of 1 pc
startable.flp=startable.flp/startable.px^2
startable.f12=startable.f12/startable.px^2
;
; We could already compute Teff from spectral type...
teff_stars
;
; Get more precise V and (B-V) values from Tycho catalog
get_tycho
;
; ...but we compute more precise values of Teff from Tycho (B-V)
index=where(startable.class le 2)
startable(index).teff=teff_bv(startable(index).bv,/sg)
index=where(startable.class gt 2)
startable(index).teff=teff_bv(startable(index).bv)
;
; Compute diameters from spectral type (but not needed here...)
diam_stars
;
; Print some diagnostic messages
;index=where(startable.class ge 1 and startable.class le 2)
;median_teff=medianve(startable(index).teff,e)
;print,'Median Teff  (s.-g.) and error:',median_teff,e
index=where(startable.class ge 2.5 and startable.class le 3.5)
median_teff=medianve(startable(index).teff,e)
print,'Median Teff (giants) and error:',median_teff,e
index=where(startable.class ge 4.5 and startable.class le 5)
median_teff=medianve(startable(index).teff,e)
print,'Median Teff (dwarfs) and error:',median_teff,e
;
; Prepare relations for the luminosity classes 3 and 5
!p.multi=[0,2,2]
;
; L-band giants
index=where(startable.class eq 3)
st=startable(index)
index=where(st.flp lt 10000)
st=st(index)
si=sort(st.teff)
st=st(si)
plot,st.teff,st.flp,psym=1,xrange=[3000,10000],yrange=[0,15e3], $
	title='giants',xtitle='Teff',ytitle='L-band flux [Jy] at 1pc'
rlg=poly_fit(st.teff,st.flp,2,yfit)
oplot,st.teff,yfit,psym=-3,color=tci(2),thick=2
xyouts,0.1,0.9,strjoin(string(rlg(*)),' '),/norm,color=tci(3)
;
; L-band dwarfs
index=where(startable.class eq 5)
st=startable(index)
index=where(st.flp lt 10000)
st=st(index)
si=sort(st.teff)
st=st(si)
plot,st.teff,st.flp,psym=1,xrange=[3000,10000],yrange=[0,12e3], $
	title='dwarfs',xtitle='Teff',ytitle='L-band flux [Jy] at 1pc'
index=where(st.flp lt 6000 and st.teff lt 7000)
rld=poly_fit(st(index).teff,st(index).flp,4,yfit)
oplot,st(index).teff,yfit,psym=-3,color=tci(2),thick=2
xyouts,0.55,0.9,strjoin(string(rld(*)),' '),/norm,color=tci(3)
;
; N-band giants
index=where(startable.class eq 3)
st=startable(index)
index=where(st.f12 lt 10000)
st=st(index)
si=sort(st.teff)
st=st(si)
plot,st.teff,st.f12,psym=1,xrange=[3000,10000],yrange=[0,15e3], $
	title='giants',xtitle='Teff',ytitle='N-band flux [Jy] at 1pc'
rng=poly_fit(st.teff,st.f12,2,yfit)
oplot,st.teff,yfit,psym=-3,color=tci(2),thick=2
xyouts,0.1,0.4,strjoin(string(rng(*)),' '),/norm,color=tci(3)
;
; N-band dwarfs
index=where(startable.class eq 5)
st=startable(index)
index=where(st.f12 lt 10000)
st=st(index)
si=sort(st.teff)
st=st(si)
plot,st.teff,st.f12,psym=1,xrange=[3000,10000],yrange=[0,6e3], $
	title='dwarfs',xtitle='Teff',ytitle='N-band flux [Jy] at 1pc'
index=where(st.f12 lt 6000 and st.teff lt 7000)
rnd=poly_fit(st(index).teff,st(index).f12,4,yfit)
oplot,st(index).teff,yfit,psym=-3,color=tci(2),thick=2
xyouts,0.55,0.4,strjoin(string(rnd(*)),' '),/norm,color=tci(3)
;
save,rlg,rld,rng,rnd,filename=!oyster_dir+'starbase/MDFC/mdfc_ln_coeffs.xdr'
;
end
;-------------------------------------------------------------------------------
pro matisse_listbcd,files
;
; Run python script adaped from script with the same name in mat_tools.
; In-file changes, it is recommended to make a backup copy before!
;
if n_elements(files) eq 0 then files='*.fits'
if n_elements(files) eq 1 then f=file_search(files) else f=files
nf=n_elements(f)
;
baseline_configs=strarr(nf)
;
for i=0,nf-1 do begin
	exten=find_oiextn(f(i),'OI_ARRAY')
	oi_array=mrdfits(f(i),exten,h,/silent)
	tel_names=strarr(max(oi_array.sta_index))
	tel_names(oi_array.sta_index)=oi_array.tel_name
;
	exten=find_oiextn(f(i),'OI_VIS2')
	oi_vis2=mrdfits(f(i),exten,h,/silent)
	nm=size(oi_vis2.sta_index,/dim)
	bl=strarr(nm(1))
	for j=0,nm(1)-1 do bl(j)=strjoin(tel_names(oi_vis2(j).sta_index),'-')
	baseline_configs(i)=strjoin(bl,'_')
endfor
;
u_configs=unique(baseline_configs)
if n_elements(u_configs) eq 1 $
	then print,'Data are from one unique configuration!' $
	else print,'Data are from multiple BCD configurations!'

;
end
;-------------------------------------------------------------------------------
pro matisse_removebcd,files
;
; Run python script adaped from script with the same name in mat_tools.
; In-file changes, it is recommended to make a backup copy before!
;
if n_elements(files) eq 0 then files='*.fits'
if n_elements(files) eq 1 then f=file_search(files) else f=files
;
python='/usr/bin/python '
python='/usr/local/miniconda2/bin/python '
for i=0,n_elements(f)-1 do spawn,python+!oyster_dir+ $
	"source/python/mat_removeBCD.py -i "+f(i)
;
end
;************************************************************************Block 2
function alloc_matisse_options
;
return,{$
	pipeoptions:'Unc' $
	}
;
end
;-------------------------------------------------------------------------------
function dfits_matisse,r
;
ipos=strpos(r(1),'MATISSE')
jpos=strpos(r(0),'OBS.NAME')
return,strmid(r,ipos,29)+' '+strmid(r,jpos,strlen(r(1))-jpos)
return,strmid(r,ipos,strlen(r(1))-ipos)
;
end
;-------------------------------------------------------------------------------
pro rm_rmnrec
;
; Delete RMNREC files as detected by the small file size
;
f=file_search('MATIS*.fits*')
r=file_info(file_search(f))
i=where(r.size lt 400000,n)
if n gt 0 then file_delete,f(i)
;
end
;-------------------------------------------------------------------------------
pro rm_rmnrec
;
; Delete RMNREC files as detected by the value of the OBJECT keyword 
;
f=file_search('MATIS*.fits')
if strlen(f(0)) eq 0 then return
spawn,'dfits MATIS*.fits | fitsort OBJECT',result
result=result(1:n_elements(result)-1) ; dfits displays a header line
n=n_elements(result)
for i=0,n_elements(result)-1 do begin
        words=nameparse(result(i))
        if words(1) eq 'STD,RMNREC' then file_delete,f(i)
endfor
;
end
;-------------------------------------------------------------------------------
pro rm_unwanted,file_type
;
; Re/Move unwanted RMNREC, SEARCH, and OPEN files.
; Also remove processed calibrators and raw calibrations.
;
if n_elements(file_type) eq 0 then file_type='all'
;
spawn,"find . -name '*.fits'",f
if strlen(f(0)) eq 0 then return
;
if file_type eq 'all' or file_type eq 'rmnrec' then begin
ob=obj_new('fitsfilelist',f)
hk=ob->getHeaderKeys('OCS DET IMGNAME')
if size(hk.OCSDETIMGNAME,/type) eq 7 then begin
	index=where(hk.OCSDETIMGNAME eq 'MATISSE_OBS_RMNREC',count)
	if count gt 0 then for i=0,count-1 do spawn,'rm '+f(index(i))
endif
obj_destroy,ob
endif
;
if file_type eq 'all' or file_type eq 'search' then begin
spawn,"find . -name '*.fits'",f
ob=obj_new('fitsfilelist',f)
hk=ob->getHeaderKeys('DPR TYPE')
if size(hk.DPRTYPE,/type) eq 7 then begin
	index=where(strpos(hk.DPRTYPE,'SEARCH') ge 0,count)
	if count gt 0 then for i=0,count-1 do spawn,'rm '+f(index(i))
endif
obj_destroy,ob
endif
;
if file_type eq 'all' or file_type eq 'open' then begin
spawn,"find . -name '*.fits'",f
ob=obj_new('fitsfilelist',f)
hk=ob->getHeaderKeys('INS DIL ID')
if size(hk.INSDILID,/type) eq 7 then begin
	index=where(strpos(hk.INSDILID,'OPEN') ge 0,count)
	if count gt 0 then for i=0,count-1 do spawn,'rm '+f(index(i))
endif
obj_destroy,ob
endif
;
; Remove processed calibrators
if file_type eq 'all' or file_type eq 'proc' then begin
spawn,"find . -name 'M.*.fits'",f
ob=obj_new('fitsfilelist',f)
hk=ob->getHeaderKeys('PRO CATG')
if size(hk.PROCATG,/type) eq 7 then begin
	index=where(strpos(hk.PROCATG,'CALIB_RAW_INT') ge 0,count)
	if count gt 0 then for i=0,count-1 do spawn,'rm '+f(index(i))
endif
obj_destroy,ob
endif
;
; Remove raw calibrations
if file_type eq 'all' or file_type eq 'raw' then begin
spawn,"find . -name 'MATIS.*.fits'",f
ob=obj_new('fitsfilelist',f)
hk=ob->getHeaderKeys('OBS NAME')
if size(hk.OBSNAME,/type) eq 7 then begin
	index=where(strpos(hk.OBSNAME,'Calibrations') ge 0,count)
	if count gt 0 then for i=0,count-1 do spawn,'rm '+f(index(i))
endif
obj_destroy,ob
endif
;
; hk=ob->getHeaderKeys('INS DIN ID')
; if size(hk.INSDINID,/type) eq 7 then begin
; index=where(strpos(hk.INSDINID,'OPEN') ge 0,count)
; if count gt 0 then for i=0,count-1 do spawn,'rm '+f(index(i))
; endif
;
end
;-------------------------------------------------------------------------------
pro tartyp
;
; Analyze imaging data and display the names of files with only sky frames.
;
files=findfile('MATIS.*.fits')
;
spawn,'/scisoft/bin/dfits MATIS.*.fits | /scisoft/bin/fitsort DET.NAME OBJECT',r
;
index=where(strpos(r,'SKY') lt 0 and $
	    strpos(r,'FILE') lt 0 and $
	    strpos(r,'WAVE') lt 0 and $
	    strpos(r,'FLAT') lt 0 and $
	    strpos(r,'DARK') lt 0 and $
	    strpos(r,'KAPPA') lt 0,n)
;
files=files(index)
;
print,'Analyzing files...'
;
for i=0,n-1 do begin
	fxbopen,unit,files(i),2,header
	extname = strtrim(fxpar(header,'EXTNAME'),2)
	if extname ne 'IMAGING_DATA' then begin
		print,'Extension is not IMAGING_DATA, stopping...'
		return
	endif
;
	fxbread,unit,tartyp,'TARTYP'
;
	index=where(tartyp eq 'S',count)
	if count eq n_elements(tartyp) then begin
		print,'File '+files(i)+' has only sky frames!'
	endif else begin
		print,'File '+files(i)+' OK!'
	endelse
	fxbclose,unit
;
endfor
;
end
;-------------------------------------------------------------------------------
pro rename2arc_matisse,fitsfiles
;
if n_elements(fitsfiles) eq 0 then fitsfiles='MATISSE_GEN_???*.fits'
;
files=file_search(fitsfiles)
;
for i=0,n_elements(files)-1 do begin
;
	words=nameparse(files(i),['.','_'])
	instrument=words(0)
	shortinstr=strmid(instrument,0,5)
;
	fitsfile=obj_new('fitsfile',files(i))
	prihead=fitsfile->prihead()
	arcfile=prihead->getpar('ARCFILE')
	pipefile=string(prihead->getpar('PIPEFILE'))
	obj_destroy,fitsfile
	if strpos(pipefile,instrument) eq 0 and $
	   strpos(arcfile,shortinstr) eq 0 then begin
		ipos=25
		jpos=strpos(files(i),'.fits')
		suffix=strmid(files(i),ipos,jpos-ipos)
		destfile=instrument+'_'+strmid(arcfile,6,19)+'_'+suffix+'.fits'
	command='mv '+files(i)+' '+destfile
	spawn,command
	endif
;
endfor
;
end
;-------------------------------------------------------------------------------
pro rename2cal,fitsfiles,estimates=estimates,rectified=rectified
;
; Rename pipeline output files to more human-readable names.
;
if n_elements(fitsfiles) eq 0 then fitsfiles='*.fits'
;
files=file_search(fitsfiles)
;
if keyword_set(estimates) then begin
for i=0,n_elements(files)-1 do begin
	words1=nameparse(files(i),'_TARGET_RAW_INT')
	time_stamp=strmid(words1(1),0,25)
	d=mrdfits(files(i),0,h,/silent)
	words2=nameparse(fitshparse(h,'HIERARCH ESO PRO REC5 RAW2 NAME'),'_')
	detector=fitshparse(h,'HIERARCH ESO DET CHIP NAME')
	band=''
	if strpos(detector,'HAWAII') ge 0 then band='L'
	if strpos(detector,'AQUARIUS') ge 0 then band='N'
	newname=words1(0)+time_stamp+'-'+strjoin(words2(0:1),'_')+'_'+band+'.fits'
	print,newname
	spawn,'mv '+files(i)+' '+newname
endfor
return
endif
;
if keyword_set(rectified) then begin
for i=0,n_elements(files)-1 do begin
	words1=nameparse(files(i),'-SCI')
	d=mrdfits(files(i),0,h,/silent)
	words2=nameparse(fitshparse(h,'HIERARCH ESO PRO REC5 RAW2 NAME'),'_')
	detector=fitshparse(h,'HIERARCH ESO DET CHIP NAME')
	band=''
	if strpos(detector,'HAWAII') ge 0 then band='L'
	if strpos(detector,'AQUARIUS') ge 0 then band='N'
	newname=words1(0)+'-'+strjoin(words2(0:1),'_')+'_'+band+'.fits'
	print,newname
	spawn,'mv '+files(i)+' '+newname
endfor
return
endif
;
print,'Please specify /estimates or /rectified!'
;
end
;************************************************************************Block 3
function analyzematisseendproducts,udate,write=write,all=all
;
; Called by analyzematisseconfigids. Return dates for which pipeline
; products were found (using searchstrng). If given a date, return
; all files available for that date.
;
common MatissePipeline,rootdatadir,productsdir,searchstrng,configid,recipes
;
; Directory/file tree of reflex_end_products:
; First level  down: YYYY-MM-DDTHH:MM:SS (date of running the recipe)
; Second level down: MATIS.YYYY-MM-DDTHH:MM:SS.FFF_tpl
; Third level down: CAL/SCI_<target>_CALIB/TARGET_RAW_INT_YYYY-MM-DDTHH:MM:SS
;
cd,productsdir,current=old_dir
words=nameparse(productsdir,'/') & n=n_elements(words)
; SV reduced data directory (YYYY-MM-DD, created by linknightstoendproducts)
if strlen(words(n-1)) eq 10 then begin 
	wildcard='???_*_????-??-??T??:??:??.????*'
	spawn,'find        -L '+wildcard+' -name "*.fits"',f1
	f1=productsdir+'/'+f1
	spawn,'dfits `find -L '+wildcard+' -name "*.fits"` '+ $
					'| fitsort DATE-OBS',f2
	f2=f2(1:n_elements(f2)-1)
endif else if strlen(words(n-1)) eq strlen('reflex_end_products') then begin
; Regular reflex_end_products tree
	if file_test('????-??-??T??:??:??',/dir) eq 0 then begin
		print,'Not in a reflex_end_products tree!'
		retall
	endif
	wildcard='*.????-??-??T??:??:??.???*_tpl*'
	spawn,'find        ????-??-??T??:??:??/'+wildcard+' -name "*.fits"',f1
	f1='reflex_end_products/'+f1
	spawn,'dfits `find ????-??-??T??:??:??/'+wildcard+' -name "*.fits"` '+ $
					'| fitsort DATE-OBS',f2
	f2=f2(1:n_elements(f2)-1)
endif else begin
; Local directory
	spawn,'find -name "*.fits"',f1
	spawn,'dfits `find -name "*.fits"` | fitsort DATE-OBS',f2
	f2=f2(1:n_elements(f2)-1)
endelse
cd,old_dir
for i=0,n_elements(f2)-1 do begin
	words=nameparse(f2(i))
	f2(i)=words(1)
endfor
f=f1+' '+f2
n=n_elements(f)
dates=strarr(n)
times=strarr(n)
;
for i=0,n-1 do begin
	words=nameparse(f(i))
	f(i)=words(0)
	dates(i)=strmid(words(1),0,10)
	times(i)=strmid(words(1),11,8)
	if hms2h(times(i)) lt 12 then dates(i)=previousdate(dates(i))
endfor
;
udates=unique(dates)
;
if n_elements(write) eq 0 then write=0
;
if n_elements(all) eq 0 then all=0
if all then udate=udates
files=''
;
if n_elements(udate) ne 0 then begin
	for i=0,n_elements(udate)-1 do begin
		index=where(dates eq udate(i),count)
		if count gt 0 then files=[files,f(index)]
		if write then begin
			openw,unit,'product_files.txt',/get_lun
			for i=0,count-1 do printf,unit,f(index(i))
			free_lun,unit
		endif
	endfor
	return,files(1:n_elements(files)-1)
endif else return,udates
;
end
;-------------------------------------------------------------------------------
function analyzematisseconfigids
;
; Determine how many different configurations are present in the data
; of the select products directory. Configurations are different for different
; baselines, read-out windows, and spectral setups.
;
dates=analyzematisseendproducts()
files=analyzematisseendproducts(dates(0))
for i=1,n_elements(dates)-1 do files=[files,analyzematisseendproducts(dates(i))]
;
if strlen(files(0)) eq 0 then return,'No MATISSE reduced files'
;
n=n_elements(files)
configurations=strarr(n)
;
for i=0,n-1 do begin
;
	fitsfile=obj_new('fitsfile',files(i))
	prihead=fitsfile->prihead()
;
	obsstart=strtrim(prihead->getpar('OBS START'))
	insmode1=strmid(strtrim(prihead->getpar('DET CHIP NAME')),0,8)
	if strpos(insmode1,'HAWAII') eq 0 $
		then insmode2=strtrim(prihead->getpar('INS DIL ID')) $
		else insmode2=strtrim(prihead->getpar('INS DIN ID'))
	insmode3=strmid(strtrim(prihead->getpar('DET READ CURNAME')),4,4)
	insmode3=strtrim(string(long(prihead->getpar('DET SEQ1 DIT')*1e7)),1)
;	dprtype can be used to identify the calibrators
	dprtype=strtrim(prihead->getpar('PRO CATG'))
	j=strpos(dprtype,'TARGET')
	if j lt 0 then dprtype='CAL' else dprtype='SCI'
;
	obstimes=strmid(obsstart,11,8)
	obsstart=strmid(obsstart,0,10)
;	Reduced file date stamps should refer to UT date, not civil date
	if hms2h(obstimes) gt 12 then obsstart=nextdate(obsstart)
;	
	insmode=insmode1+'-'+insmode2+'-'+insmode3
;
	station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
	station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
	station_3=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION3')))
	station_4=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION4')))
;
	configurations(i)=obsstart+'_'+insmode $
				 +'_'+station_1 $
				 +'-'+station_2 $
				 +'-'+station_3 $
				 +'-'+station_4
;
	obj_destroy,fitsfile
;
endfor
;
return,unique(configurations)
;
end
;-------------------------------------------------------------------------------
function matisseconfigids,obht
;
; Return a unique set of configids currently loaded.
;
n=n_elements(obht)
configurations=strarr(n)
;
for i=0,n-1 do begin

	obsstart=obht(i).OBSSTART
	insmode1=obht(i).DETCHIPNAME
	if strpos(insmode1,'HAWAII') eq 0 $
		then insmode2=obht(i).INSDILID $
		else insmode2=obht(i).INSDINID
;	insmode3=obht(i).DETREADCURNAME
	insmode3=strtrim(string(long(float(obht(i).DETSEQ1DIT)*1e7)),1)
;
	obstimes=strmid(obsstart,11,8)
	obsstart=strmid(obsstart,0,10)
;	Reduced file date stamps should refer to UT date, not civil date
	if hms2h(obstimes) gt 12 then obsstart=nextdate(obsstart)
	words=nameparse(insmode1,'_')
	insmode1=words(0)
;
	insmode=insmode1+'-'+insmode2+'-'+insmode3
;
	station_1=vlti_stationid(strtrim(obht(i).ISSCONFSTATION1,2))
	station_2=vlti_stationid(strtrim(obht(i).ISSCONFSTATION2,2))
	station_3=vlti_stationid(strtrim(obht(i).ISSCONFSTATION3,2))
	station_4=vlti_stationid(strtrim(obht(i).ISSCONFSTATION4,2))
;
	configurations(i)=strcompress(obsstart+'_'+insmode $
				 +'_'+station_1 $
				 +'-'+station_2 $
				 +'-'+station_3 $
				 +'-'+station_4,/remove_all)
endfor
;
return,configurations
;
end
;-------------------------------------------------------------------------------
pro selectmatissefiles,all=all
;
; Displays the gorgonzola gui to select files. Stores the results in
; common block. If all are requested, return only those corresponding
; to the currently selected configuration.
;
; Note: fbtFitsFileList.pro in oyster/source/fbt was modified:
; Line 364 was added as HAWAII DIT ends up as double, while Aquarius as float.
; This causes an error in the conctenation of the two structures.
;
common MatisseGuiWids,matisse_wid,files_wid,configids_wid,pipeline_wid
common MatissePipeline,rootdatadir,productsdir,searchstrng,configid,recipes
common MatisseObservation,obht,objindex
common MatisseOptions,matisse_options
;
keys=['OBS START','OBS NAME','DET CHIP NAME','DET SEQ1 DIT', $
      'INS FIL NAME','INS DIL ID','INS FIN NAME','INS DIN ID', $
      'ISS CONF STATION1','ISS CONF STATION2', $
      'ISS CONF STATION3','ISS CONF STATION4']
print,'Please wait, reading files...'
searchstrings=nameparse(searchstrng)
if keyword_set(all) then begin
	first=1
	for i=0,n_elements(searchstrings)-1 do begin
		ob_i=obj_new('fitsfilelist',dir=productsdir, $
			searchString=searchstrings(i))
		if heap_refcount(ob_i) eq 0 then continue
		obht_i=ob_i->getHeaderKeys(keys)
		nobht_i=n_elements(obht_i)
;		ob->setHeader,obht,keys
		obsfiles=ob_i->files()
		obht_i.filename=obsfiles
		if first then begin
			ob=ob_i
			obht=obht_i
			nobht=nobht_i
			first=0
		endif else begin
			ob=[ob,ob_i]
			obht=[obht,obht_i]
			nobht=nobht+nobht_i
		endelse
		obj_destroy,ob_i
	endfor
endif else begin
	ob=obj_new('fitsfilelist',dir=productsdir, $
		guikey=keys,guisearch="'*.fits'")
; 	The header table initially contains all files
	obht=ob->headertable()
	nobht=n_elements(obht)
	obsfiles=ob->files()
	files=obht.filename
	n=n_elements(obsfiles)
	if strlen(obsfiles(0)) eq 0 then n=0
	if n gt 0 then begin
		index=intarr(n)
		for i=0,n-1 do begin
			words=nameparse(obsfiles(i))
			index(i)=where(files eq words(0))
		endfor
		obht=obht(index)
		obht.filename=obsfiles
	endif
	obj_destroy,ob
endelse
;
index=where(strpos(obht.detchipname,'HAWAII') eq 0,count)
if count gt 0 then obht(index).detchipname='HAWAII-2'
; obht.detreadcurname=strmid(obht.detreadcurname,0,8)
;
; For option all, we do however select according to the current configid
if keyword_set(all) then begin
	configurations=matisseconfigids(obht)
	index=where(configurations eq configid,count)
	if count gt 0 then obht=obht(index)
	if matisse_options.pipeoptions eq 'Cal' then begin
		index=where(strpos(obht.filename,'CAL_') ge 0,count)
		index=where(strpos(obht.filename,'_CAL_INT_') ge 0,count)
		if count gt 0 then obht=obht(index)
	endif else begin
		index=where(strpos(obht.filename,'CAL_') lt 0,count)
		index=where(strpos(obht.filename,'_CAL_INT_') lt 0,count)
		if count gt 0 then obht=obht(index)
	endelse
endif
;
; The obsindex shall refer exactly to those files needed for a visibility
objindex=indgen(n_elements(obht))	; all files for MATISSE
;
; Sort with increasing time
si=sort(obht.obsstart)
obht=obht(si)
;
; Display selected files in list widget
if n_elements(files_wid) eq 0 then files_wid=0l
if widget_info(files_wid,/valid) then begin
	if count gt 0 then begin
		widget_control,files_wid,set_value=obht.filename, $
					 set_uvalue=obht.filename
	endif else begin
		widget_control,files_wid,set_value='',set_uvalue=''
	endelse
endif
;
end
;-------------------------------------------------------------------------------
function matissestarids,scans
;
; Adopt a single starid (the first) for stars of the same name.
; This is important for the OBJ designation in case it was derived
; from apparent coordinates.
;
stars=unique(scans.star)
for i=0,n_elements(stars)-1 do begin
	index=where(scans.star eq stars(i))
	scans(index).starid=scans(index(0)).starid
endfor
;
return,scans
;
end
;-------------------------------------------------------------------------------
function merge_intphot,scans,scantable
;
; Merge interferometry with fluxes
;
if total(scans(0).photometryerr > 0) eq 0 $
	and total(scans(1).photometryerr > 0) gt 0 $
	and total(scans(0).vissqerr > 0) gt 0 $
	and total(scans(1).vissqerr > 0) eq 0  $
then begin
	if (scans(1).time-scans(0).time)/60 gt 20 then $
		print,'Warning: possible problem w/observation!'
	scans(0).photometry=scans(1).photometry
	scans(0).photometryerr=scans(1).photometryerr
	scans=scans(0)
	scantable=scantable(0)
	return,scans
endif
;
; The order can change (is it mat_mergeAllOiFits.py?)
if total(scans(1).photometryerr > 0) eq 0 $
	and total(scans(0).photometryerr > 0) gt 0 $
	and total(scans(1).vissqerr > 0) gt 0 $
	and total(scans(0).vissqerr > 0) eq 0 $
then begin
	if (scans(1).time-scans(0).time)/60 gt 2 then $
		print,'Warning: possible problem w/observation!'
	scans(1).photometry=scans(0).photometry
	scans(1).photometryerr=scans(0).photometryerr
	scans=scans(1)
	scantable=scantable(1)
	return,scans
endif
;
; If number of exposure cycles > 1 do nothing
return,scans
;
end
;************************************************************************Block 4
function matissegui_event,event
;
; This is the call back for buttons on the main MyMatisseGui.
;
common MatisseGuiWids,matisse_wid,files_wid,configids_wid,pipeline_wid
common MatissePipeline,rootdatadir,productsdir,searchstrng,configid,recipes
common MatisseObservation,obht,objindex
common MatisseOptions,matisse_options
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
case event.value of
;
'Show/select product files':begin
;		Clean desktop a little
		wdall
		set_screen
		!x.range=0
		!y.range=0
		!p.psym=0
;		Store current OYSTER observation
		if n_elements(geninfo) ne 0 then begin
;			storenight,11
;			save,recipes,filename='Recipes_'+genconfig.configid+'.xdr'
		endif
;
		selectmatissefiles
;
;		If a complete observation was selected, get additional info
		if objindex(0) ge 1 then begin 
		fitsfiles=obht(objindex(0)).filename
		fitsfile=obj_new('fitsfile',fitsfiles(0))
		prihead=fitsfile->prihead()
		airmass=strtrim(prihead->getpar('ISS AIRM END'))
		seeing=strtrim(prihead->getpar('ISS AMBI FWHM'))
		dispersion=strtrim(prihead->getpar('INS MODE'))
		obj_destroy,prihead
		obj_destroy,fitsfile
		print,'Target: ',obht(objindex(0)).ftrobjname
		print,'Airmass:',airmass,', seeing:',seeing
;		Load corresponding OYSTER scans structure
		if n_elements(geninfo) ne 0 then begin
			currentconfig=unique(matisseconfigids(obht(objindex)))
			index=where(geninfo.configid eq currentconfig,count)
;			This is currently not strictly needed
			if count eq 1 then begin
				datum=geninfo(index).date
				loadnight,datum,currentconfig
				restore,'Recipes_'+currentconfig+'.xdr'
			endif
		endif
		endif
		end
'MATISSE':	begin
		spawn,'dfits '+strjoin(obht.filename,' ')+' | fitsort '+ $
			'OBS.NAME '+ $
			'DET2.SEQ1.DIT '+ $
			'DET2.NDIT ',r
;			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=dfits_matisse(r),width=142
		end
'FT':	begin
		spawn,'dfits '+strjoin(obht.filename,' ')+' | fitsort '+ $
			'OBS.NAME '+ $
			'OBS.NAME '+ $
			'QC.PHASE_FT12.RMS  '+ $
			'QC.PHASE_FT13.RMS  '+ $
			'QC.PHASE_FT14.RMS  '+ $
			'QC.PHASE_FT23.RMS  '+ $
			'QC.PHASE_FT24.RMS  '+ $
			'QC.PHASE_FT34.RMS  ',r
;			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=dfits_matisse(r),width=142
		end
'COUDE':	begin
		spawn,'dfits '+strjoin(obht.filename,' ')+' | fitsort '+ $
			'OBS.NAME '+ $
			'FT.ROBJ.NAME '+ $
			'COU.AO2.T0_MEAN COU.AO2.FWHM_MEAN ',r
;			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=dfits_matisse(r),width=142
		end
'AMBIENT':	begin
		selectmatissefiles,/all
		spawn,'dfits '+strjoin(obht.filename,' ')+' | fitsort '+ $
			'OBS.NAME '+ $
			'ISS.AMBI.FWHM.START ISS.AMBI.FWHM.END '+ $
			'ISS.AMBI.TAU0.START ISS.AMBI.TAU0.END '+ $
			'ISS.AMBI.WINDSP',r
;			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=dfits_matisse(r),width=142
		end
'OYSTER': 	begin
		mymatissepipe
		oyster
		end
'Restore':	begin
		xdr_file=configid+'.cha.xdr'
		if file_test(xdr_file) then begin
			print,'Restoring '+xdr_file
			get_xdr,xdr_file
		endif else print,xdr_file+' does not exist!'
		end
'Pipeline':	begin
		!p.multi=[0,2,2]
		mymatissepipe
		end
'Recipes': 	begin
		matisserecipesgui
		end
'Delete':	begin
		for i=0,n_elements(obht.filename)-1 do $
			print,obht(i).filename
		answer=' '
		read,prompt='Delete these files? (yes/no): ',answer
		if answer eq 'yes' then spawn,'rm -f '+strjoin(obht.filename,' ')
		if answer ne 'yes' then print,'Files not removed.'
		end
'Rename':	begin
		for i=0,n_elements(obht.filename)-1 do begin
			fitsfile=obj_new('fitsfile',obht(i).filename)
			prihead=fitsfile->prihead()
			dateobs=strmid(strtrim(prihead->getpar('DATE-OBS')),0,23)
			obj_destroy,fitsfile
			outfile='MATISSE.'+dateobs+'.fits'
			if obht(i).filename ne outfile then begin
				command='mv '+obht(i).filename+' '+outfile
				print,command
				spawn,command
			endif
		endfor
		end
;
endcase
;
end
;-------------------------------------------------------------------------------
function matissegui_productsdirs,event
;
; Function called when starting the matisse GUI to return lists of directories
; with MATISSE data. It is first called with the current (rootdatadir) as event.
; The directories searched include reflex_end_products and sv, each with a 
; particular structure and file naming convention, with part. search strings.
;
; The search string depends on the products, and is used by selectmatissefiles.
;
; The configurations are listed by analyzematisseconfigids.
;
common MatissePipeline,rootdatadir,productsdir,searchstrng,configid,recipes
common MatisseGuiWids,matisse_wid,files_wid,configids_wid,pipeline_wid
;
; Set products directory
r=size(event)
; Check for structure event (indicates call via menu selection
if r(n_elements(r)-2) eq 8 then begin
	widget_control,event.id,get_uvalue=productsdirs
	productsdir=productsdirs(event.index)
endif else productsdir=event
;
; Reflex search strings
r_string="'*_TARGET_???_INT_*.fits' '*_CALIB_???_INT_*.fits'"
; Gasgano search strings
g_string="'*single???vis_????.fits'"
; SV search strings
s_string="'*_RAW_INT_*.fits'"
s_string="'*_RAW_INT_*.fits' '/reflex_end_products/*/*_CAL_INT_*.fits'"
d_string="????-??-??"
;
case productsdir of
'reflex_end_products':	begin
		productsdir=rootdatadir+'/reflex_end_products'
		searchstrng=r_string
		end
'gasgano':	begin
		productsdir=rootdatadir+'/gasgano'
		searchstrng=g_string
		end
'sv':		begin
		productsdir=rootdatadir+'/sv/';+words(1)
		searchstrng=s_string
		end
else:		begin
;		Come here the first time to see which data are available
		IF r(n_elements(r)-2) ne 8 THEN BEGIN 
		productsdirs=''
;
;		Check working directory for FITS files
		productsdir=rootdatadir
		if strlen(file_search(productsdir,/test_dir)) ne 0 then begin
		searchstrng='*.fits'
		searchstrng='\*.fits'
;		spawn,'find -L . -iname '+searchstrng,f
;		spawn,'ls '+searchstrng0,f
		spawn,'ls *.fits',f
		if strlen(f(0)) ne 0 then productsdirs=[productsdirs,productsdir]
		endif
;
;		Check for Reflex end products
		productsdir='reflex_end_products'
		if strlen(file_search(productsdir,/test_dir)) ne 0 then begin
		searchstrng=r_string
;		spawn,'find -L '+productsdir+' -iname '+searchstrng,f
		f=find_files(productsdir,searchstrng)
		index=where(strlen(f) ne 0,count)
		if count ge 1 then productsdirs=[productsdirs,productsdir]
		endif
;
;		Check for gasgano products
		productsdir='gasgano'
		if strlen(file_search(productsdir,/test_dir)) ne 0 then begin
		searchstrng=g_string
		spawn,'find -L '+productsdir+' -iname '+searchstrng,f
		if strlen(f(0)) ne 0 then productsdirs=[productsdirs,productsdir]
		endif
;
;		Check for SV products (folder sv, subfolders by date)
		productsdir='sv'
		if strlen(file_search(productsdir,/test_dir)) ne 0 then begin
		cd,productsdir,current=old_dir
;		searchstrng=s_string
;		spawn,'find -L '+productsdir+' -iname '+searchstrng,f
		spawn,'ls -d '+d_string,f
		f='sv/'+f
		if strlen(f(0)) ne 0 then productsdirs=[productsdirs,f]
		cd,old_dir
		endif
;
;		Prepare return
		n=n_elements(productsdirs)
		if n gt 1 then productsdirs=productsdirs(1:n-1)
		return,productsdirs
		ENDIF ELSE BEGIN
		searchstrng='*.fits'
		searchstrng='\*.fits'
		ENDELSE
		end
endcase
;
if widget_info(configids_wid,/valid) then begin
	configids=analyzematisseconfigids()
	widget_control,configids_wid,set_value=configids,set_uvalue=configids
	configid=configids(0)
	widget_control,configids_wid,/sensitive
endif
;
return,productsdir
;
end
;-------------------------------------------------------------------------------
function matissegui_configid,event
;
; Call back for the selection of a particular configuration.
;
common MatissePipeline,rootdatadir,productsdir,searchstrng,configid,recipes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common MatisseGuiWids,matisse_wid,files_wid,configids_wid,pipeline_wid
;
; Set configid
r=size(event)
if r(n_elements(r)-2) eq 8 then begin
	widget_control,event.id,get_uvalue=configids
	configid=configids(event.index)
endif else configid=event
selectmatissefiles,/all
;
end
;-------------------------------------------------------------------------------
function files_event,event
;
; When clicking on a file, start fv on it.
;
widget_control,event.id,get_uvalue=files
if n_elements(files) ne 0 then begin
print,'Starting fv on ',files(event.index)
spawn,'/usr/local/src/fv5.4/fv -cmap 2 '+files(event.index)+' &'
endif
;
end
;-------------------------------------------------------------------------------
function matissegui_options,event
;
; Callback for setting various options.
;
common MatisseOptions,matisse_options
;
r=size(event)
;
; Check for structure event (indicates call via menu selection)
if r(n_elements(r)-2) ne 8 then begin
	matisse_options.pipeoptions=event
endif else begin
	widget_control,event.id,get_uvalue=command,get_value=values
	matisse_options.pipeoptions=values(event.index)
;
endelse
;
selectmatissefiles,/all
;
end
;-------------------------------------------------------------------------------
pro create_all_cal_xdr
;
common MatisseObservation,obht,objindex
common MatisseOptions,matisse_options
common MatissePipeline,rootdatadir,productsdir,searchstrng,configid,recipes
;
matisse_options=alloc_matisse_options()
;
rootdatadir=pwd()
productsdir=matissegui_productsdirs(rootdatadir)
configids=analyzematisseconfigids()
configid=configids(0)
;
cal_files=file_search('CAL_*.fits')
if strlen(cal_files(0)) ne 0 then r=matissegui_options('Cal') else return
;
for i=0,n_elements(configids)-1 do begin
	configid=configids(i)
	mymatissepipe
endfor
;
end
;-------------------------------------------------------------------------------
pro mymatissegui
;
; Creates the MyMatisseGui main widget.
;
common MatisseGuiWids,matisse_wid,files_wid,configids_wid,pipeline_wid
common MatisseOptions,matisse_options
common MatisseObservation,obht,objindex
common MatissePipeline,rootdatadir,productsdir,searchstrng,configid,recipes
;
; Initialization and cleanup
if n_elements(matisse_wid) eq 0 then matisse_wid=0L
if n_elements(configids_wid) eq 0 then configids_wid=0L
if widget_info(matisse_wid,/valid) then widget_control,matisse_wid,/destroy
;
; Rootdatadir shall be a directory which may contain the following folders:
; reflex_end_products, gasgano, sv. 
spawn,'pwd',rootdatadir & rootdatadir=rootdatadir(0) 
;
; Allocate options
matisse_options=alloc_matisse_options()
;
; Create top GUI
matisse_wid=widget_base(/column,title=rootdatadir)
row_wid=widget_base(matisse_wid,/row,/frame)
;
templates=['Show/select product files']
button_wid=cw_bgroup(row_wid,/row,templates, $
	event_func='matissegui_event',/return_name)
;
; Get list of directories with data products available
productsdirs=matissegui_productsdirs(rootdatadir)
if strlen(productsdirs(0)) eq 0 then return
productsdir=productsdirs(0)
opmenup_wid=widget_droplist(row_wid,event_func='matissegui_productsdirs', $
	uvalue=productsdirs,value=productsdirs)
productsdir=matissegui_productsdirs(productsdir)
;
templates=['MATISSE','FT','COUDE','AMBIENT']
templates=['MATISSE','FT','AMBIENT']
button_wid=cw_bgroup(matisse_wid,/row,templates, $
	event_func='matissegui_event',/return_name)
;
files_wid=widget_list(matisse_wid,value=strarr(10)+' ', $
	event_func='files_event', $
	xsize=15,ysize=10)
;
templates=['Reset','SmartMove','Move','Delete','Rename']
templates=['Delete']
button_wid=cw_bgroup(matisse_wid,/row,templates, $
	event_func='matissegui_event',/return_name)
;
col_wid=widget_base(matisse_wid,/row,/frame)
;
configids=analyzematisseconfigids()
configids_wid=widget_droplist(col_wid,event_func='matissegui_configid', $
	uvalue=configids,value=configids)
configid=configids(0)
widget_control,configids_wid,sensitive=0
;
pipeoptions=['Unc','Cal']
opmenue_wid=widget_droplist(col_wid,event_func='matissegui_options', $
	uvalue='pipeoptions',value=pipeoptions)
; Test for calibrated files in the current directory
cal_files=file_search('SCI_*-CAL_*.fits')
if strlen(cal_files(0)) ne 0 then begin
	r=matissegui_options('Cal')
	widget_control,opmenue_wid,set_value=pipeoptions(1)
endif
;
row_wid=widget_base(matisse_wid,/row)
;
templates=['OYSTER','Restore']
pipeline_wid=cw_bgroup(row_wid,/row,templates, $
	event_func='matissegui_event',/return_name)
;
widget_control,matisse_wid,/realize
xmanager,'mymatissegui',matisse_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro mymatissepipe,skipselect=skipselect
;
; Processes all MATISSE files returned by selectmatissefiles and stores results 
; in OYSTER scans structure. Data should be of same configuration.
;
; Mymatissepipe uses the instructions contained in the recipes structure for 
; the selection of the reduction method. The default recipes are created
; and stored when initializing OYSTER.
;
common MatisseObservation,obht,objindex
common MatisseGuiWids,matisse_wid,files_wid,configids_wid,pipeline_wid
common MatisseOptions,matisse_options
common MatissePipeline,rootdatadir,productsdir,searchstrng,configid,recipes
;
common Hds,path,hds_file_stub
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ModelFit,parameters,ds_options
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
common AmbientData,AmbientTable
;
RAD=180/!pi
;
if not keyword_set(skipselect) then selectmatissefiles,/all
;
; Determine how many observations there are
count_obs=n_elements(objindex)
num_scans=intarr(count_obs)
;
; Load first file
print,'Reading first file: ',obht(0).filename+'...'
get_oifits,obht(0).filename
;
; Merge interferometry with fluxes
if n_elements(scans) eq 2 then scans=merge_intphot(scans,scantable)
;
ns=n_elements(scans)
num_scans(0)=ns
scantable.starid=scans.starid
;
; Read header for more information
d=mrdfits(obht(0).filename,0,header)
scans.r0=(float(fitshparse(header,'FWHMSTART')) $
	 +float(fitshparse(header,'FWHMEND')))/2
scans.t0=(float(fitshparse(header,'TAU0START')) $
	 +float(fitshparse(header,'TAU0END')))/2
scans.pres=float(fitshparse(header,'AMBIPRES'))
scans.rhum=float(fitshparse(header,'AMBIRHUM'))
scans.temp=float(fitshparse(header,'AMBITEMP'))
scans.int_time=float(fitshparse(header,'DETSEQ1DIT')) $
	      *float(fitshparse(header,'DETNDIT'))
;
scans_all=scans
scantable_all=scantable
;
scans_cal_all=intarr(ns)
if strpos(strupcase(obht(0).filename),'CAL') ge 0 then $
scans_cal_all(*)=1
;
FOR iobs=1,count_obs-1 DO BEGIN
	print,'Reading file: ',obht(iobs).filename+'...'
	get_oifits,obht(iobs).filename
;
; 	Merge interferometry with fluxes, if in separate scans
	if n_elements(scans) eq 2 then scans=merge_intphot(scans,scantable)
;
	ns=n_elements(scans)
	num_scans(iobs)=ns
;
	d=mrdfits(obht(iobs).filename,0,header,/silent)
	scans.r0=(float(fitshparse(header,'FWHMSTART')) $
		 +float(fitshparse(header,'FWHMEND')))/2
	scans.t0=(float(fitshparse(header,'TAU0START')) $
		 +float(fitshparse(header,'TAU0END')))/2
	scans.pres=fitshparse(header,'AMBIPRES')
	scans.rhum=fitshparse(header,'AMBIRHUM')
	scans.temp=fitshparse(header,'AMBITEMP')
	scans.int_time=float(fitshparse(header,'DETSEQ1DIT')) $
		      *float(fitshparse(header,'DETNDIT'))
	
	scans_all=[scans_all,scans]
	scantable_all=[scantable_all,scantable]
;
	scans_cal=intarr(ns)
	if strpos(strupcase(obht(iobs).filename),'CAL') ge 0 then $
	scans_cal(*)=1
	scans_cal_all=[scans_cal_all,scans_cal]
ENDFOR
stations=strarr(4)
for i=0,3 do begin
	stations(i) $
		=fitshparse(header,'ISSCONFSTATION'+string(i+1,format='(i1)'))
	j=where(genconfig.stationid eq 'A'+stations(i))
	if j(0) eq -1 then $
	j=where(genconfig.stationid eq 'U'+stations(i))
	genconfig.siderostatid(j)=i+1
endfor
;
scans=scans_all
scans_cal=scans_cal_all
scantable=scantable_all
si=sort(scans.time)
scans=scans(si)
scans_cal=scans_cal(si)
scantable=scantable(si)
;
; Fix a problem which occurs once in a while
index=where(scans.star eq 'Name',count)
if count gt 0 then scans(index).star=scans(index).starid
;
; Store names for stars in StarTable
list_stars,starids,names
get_startable,starids,names=names
;
; Identify calibrators in StarTable
for i=0,n_elements(startable)-1 do begin
	j=where(scans.star eq startable(i).name,count)
	if count gt 0 then begin
	scans(j).starid=startable(i).starid
	if scans_cal(j(0)) eq 1 then startable(i).bflag='C' $
				else startable(i).bflag='.'
	endif
endfor
scantable.starid=scans.starid
;
words1=nameparse(genconfig.configid,'_')
words2=nameparse(instrument_id(systemid),'-')
genconfig.configid=instrument_id(systemid)+'-'+words1(2)
geninfo.configid=genconfig.configid
hds_file_stub=genconfig.date+'_'+genconfig.configid
hds_file_stub=configid	; Alternate name
put_xdr,hds_file_stub
;
; Also save in OIFITS
; put_oifits,hds_file_stub+'.fits'
;
end
;-------------------------------------------------------------------------------
pro cp_cal_ln
;
; Run in /iot/matisse/sv/sv_raw_estimates. Copies L-band data files
; to sv_raw_estimates_L and N-band files to sv_raw_estimates_N.
;
; ls CAL_*.fits | wc -l should return 576 in sv_raw_estimates
;
sv_path='/iot/matisse/sv'
;
calibs_l=['epssco','epsSco','gam2Nor','HD115418','IRAS12563','HD123139', $
	  'IRAS08534','IRAS10153']
calibs_n=['epssco','epsSco','IRAS12563','gsco','HD123139','IRAS10153']
;
calibs_l_path=sv_path+'/sv_raw_estimates_L'
calibs_n_path=sv_path+'/sv_raw_estimates_N'
;
f=file_search('CAL_*.fits')
;
for i=0,n_elements(f)-1 do begin
;
words=nameparse(f(i),'_')
calib=words(1)
;
d=mrdfits(f(i),0,h,/silent)
detector=fitshparse(h,'HIERARCH ESO DET CHIP NAME')
if strpos(detector,'HAWAII') ge 0 then band='L'
if strpos(detector,'AQUARIUS') ge 0 then band='N'
;
if band eq 'L' then begin
	for j=0,n_elements(calibs_l)-1 do $
	if strpos(f(i),calibs_l(j)) ge 0 then file_copy,f(i),calibs_l_path
endif
;
if band eq 'N' then begin
	for j=0,n_elements(calibs_n)-1 do $
	if strpos(f(i),calibs_n(j)) ge 0 then file_copy,f(i),calibs_n_path
endif
;
endfor
;
return
;
for i=0,n_elements(calibs_l)-1 do begin
	index=where(strpos(f,calibs_l(i)) ge 0,count)
	for j=0,count-1 do file_copy,f(index(j)),calibs_l_path
endfor
;
for i=0,n_elements(calibs_n)-1 do begin
	index=where(strpos(f,calibs_n(i)) ge 0,count)
	for j=0,count-1 do file_copy,f(index(j)),calibs_n_path
endfor
;
end
;-------------------------------------------------------------------------------
pro mymatisseplots,band=band,config=config,target=target,file=file, $
	v2=v2,cp=cp,fl=fl,yrange=yrange,adapt=adapt,help=help
;
; Band must be either 'L' or 'N'
; config: 'UTs','small','medium','large','extended'
;
; Fiiles must be of type TARGET_RAW_INT or CALIB_RAW_INT
;
; From MATISSE science verification run:
; mymatisseplots,target='AFGL4176',band='LM',config='small'
; mymatisseplots,target='AFGL4176',band='N'
; mymatisseplots,target='AHSco',band='LM'
; mymatisseplots,target='AHSco',band='N'
; mymatisseplots,target='alfsco',band='LM'
; mymatisseplots,target='alfsco',band='N'
; mymatisseplots,target='IRAS08544',band='LM'
; mymatisseplots,target='IRAS08544',band='N'
; mymatisseplots,target='HD143006',band='LM'
; mymatisseplots,target='HD143006',band='N' (too faint in N!)
;
; Call for 2024 MATISSE problem (L-band should be OK?):
; mymatisseplots,band='N',/v2,/adapt
; mymatisseplots,band='L',/v2,/adapt
; ...or use yrange keyword for all plots
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
; dfits *.fits | fitsort DET.NAME INS.DIL.ID INS.DIN.ID
; LM-LR LM-MR LM-HR N-LR
;
if keyword_set(help) then begin
	print,'Usage: mymatisseplots,band=band,target=target,/cp'
	return
endif
;
if not keyword_set(v2) and not keyword_set(cp) and not keyword_set(fl) then $
begin
	v2=1
	cp=1
	fl=1
endif
if keyword_set(v2) then v2_string='_v2' else v2_string=''
if keyword_set(cp) then cp_string='_cp' else cp_string=''
if keyword_set(fl) then fl_string='_fl' else fl_string=''
v2cpfl_string=v2_string+cp_string+fl_string
;
if n_elements(band) eq 0 then band=''
;
if n_elements(config) eq 0 then config=''
;
if keyword_set(file) then begin
	target=strmid(file,4,strlen(file)-4-30)
	j=strpos(targets(i),'_TARGET')
	k=strpos(targets(i),'_CALIB')
	if j gt 0 then targets(i)=strmid(targets(i),0,j)
	if k gt 0 then targets(i)=strmid(targets(i),0,k)
	files=file_search(file)
	count=1
endif else begin
	if n_elements(target) eq 0 then begin
		files=file_search('*.fits')
		targets=strarr(n_elements(files))
		for i=0,n_elements(files)-1 do begin
			targets(i)=strmid(files(i),4,strlen(files(i))-4-30)
			j=strpos(targets(i),'_TARGET')
			k=strpos(targets(i),'_CALIB')
			if j gt 0 then targets(i)=strmid(targets(i),0,j)
			if k gt 0 then targets(i)=strmid(targets(i),0,k)
		endfor
		targets=unique(targets)
		if n_elements(targets) gt 1 then begin
			print,targets
			target=''
			read,prompt='Please enter target ("all" for all): ', $
				target
			if strlen(target) eq 0 then return
		endif else begin
			target=targets(0)
		endelse
	endif
	files=file_search('*.fits')
	count=n_elements(files)
	if target ne 'all' then begin
		files=files(where(strpos(files,target) ge 0,count))
		if count eq 0 then return
	endif
endelse
;
dates=strarr(count)
for i=0,count-1 do begin
	pos1=strlen(files(i))-29
	dates(i)=strmid(files(i),pos1,10)
endfor
si=sort(dates)
files=files(si)
dates=dates(si)
if strlen(band) eq 0 then band_string='' else band_string='_'+band
if strlen(config) eq 0 then config_string='' else config_string='_'+config
;
plot_file=target+'_MATISSE'+band_string+config_string+v2cpfl_string+'.eps'
set_ps,plot_file
;
!xsize=800
!ysize=1024
;set_ps,page_size=din_size(landscape)
!p.charsize=1.5
;
; Begin reading files
;
FOR ifile=0,n_elements(files)-1 DO BEGIN
;
d=mrdfits(files(ifile),0,header,/silent)
get_oifits,files(ifile)
all_scans=scans
;
array_config=''
case strjoin(genconfig.stationid,'-') of
	'UT1-UT2-UT3-UT4':array_config='UTs'
	'AA0-AB2-AC1-AD0':array_config='small'
	'AA0-AB2-AD0-AC1':array_config='small'
	'AD0-AG2-AJ3-AK0':array_config='medium'
	'AD0-AG1-AJ3-AK0':array_config='medium'
	'AA0-AB5-AD0-AG1':array_config='medium'
	'AB2-AD0-AJ3-AK0':array_config='medium'
	'AK0-AG2-AD0-AJ3':array_config='medium'
	'AA0-AG1-AJ2-AJ3':array_config='large'
	'AA0-AG2-AJ2-AJ3':array_config='large'	; not offered
	'AA0-AD0-AG1-AJ3':array_config='large'
	'AA0-AG1-AJ2-AK0':array_config='large'
	'AA0-AB2-AD0-AJ3':array_config='large'
	'AA0-AB5-AD0-AJ3':array_config='large'
	'AA0-AB5-AJ2-AJ6':array_config='extended'
endcase
if strlen(array_config) eq 0 then begin
	print,strjoin(g,'-')+' is not classified!'
	return
endif
;
if ifile eq 0 then begin
	if init_plot('scan','ch') ne 0 then return
	ps_options=alloc_ps_options()
	parms=init_class(class)
	if init_indexsel('x') ne 0 then return
	if init_indexsel('y') ne 0 then return
	if init_indexsel('z') ne 0 then return
endif
ps_options.e=0
ps_options.l=1
;
if strlen(config) ne 0 then $
	if strpos(array_config,config) lt 0 then continue
;
if strlen(band) ne 0 then $
if strpos(instrument_id(systemid),band) lt 0 then continue
;
d=mrdfits(files(ifile),0,h)
bcd1=fitshparse(h,'INS BCD1 NAME')
bcd2=fitshparse(h,'INS BCD2 NAME')
bcd=bcd1+'-'+bcd2
;
calib=0
if strmid(files(ifile),0,3) eq 'CAL' then begin
	calib=1
	get_mdfc
endif
;
; Estimate target fluxes
restore,!oyster_dir+'source/matisse/zerocounts.xdr'
if strpos(fitshparse(h,'DET.NAME'),'LM') ge 0 then begin
	band1='L'
	ins_mode=fitshparse(h,'INS.DIL.ID')
endif
if strpos(fitshparse(h,'DET.NAME'),'N') ge 0 then begin
	band1='N'
	ins_mode=fitshparse(h,'INS.DIN.ID')
endif
if strpos(genconfig.stationid(0),'A') ge 0 then array='AT'
if strpos(genconfig.stationid(0),'U') ge 0 then array='UT'
det_mode=fitshparse(h,'DET.READ.CURNAME')
det_dit=strmid( $
	strtrim(string(double(fitshparse(h,'DET.SEQ1.DIT'))),1),0,10)
tpl_nexp=strtrim(string(fix(fitshparse(h,'TPL.NEXP'))),1)
index=where(zerocounts.band eq band1 and $
	    zerocounts.array eq array and $
	    zerocounts.ins_mode eq ins_mode and $
	    zerocounts.det_mode eq det_mode and $
	    zerocounts.det_dit eq det_dit and $
	    zerocounts.tpl_nexp eq tpl_nexp,count)
coeff_factor=1.
if count eq 0 then begin
	if tpl_nexp eq '40' then begin
	index=where(zerocounts.band eq band1 and $
	    zerocounts.array eq array and $
	    zerocounts.ins_mode eq ins_mode and $
	    zerocounts.det_mode eq det_mode and $
	    zerocounts.det_dit eq det_dit and $
	    zerocounts.tpl_nexp eq '52',count)
    	if count gt 0 and band1 eq 'L' then coeff_factor=1./2
    	if count gt 0 and band1 eq 'N' then coeff_factor=2
	endif
	if tpl_nexp eq '52' then begin
	index=where(zerocounts.band eq band1 and $
	    zerocounts.array eq array and $
;	    zerocounts.ins_mode eq ins_mode and $
;	    zerocounts.det_mode eq det_mode and $
;	    zerocounts.det_dit eq det_dit and $
	    zerocounts.tpl_nexp eq '40',count)
    	if count gt 0 and band1 eq 'L' then coeff_factor=2
    	if count gt 0 and band1 eq 'N' then coeff_factor=1./2
	endif
endif
; Compute median coefficients for each telescope (siderostat)
flp_est=0
f12_est=0
if count ne 0 then begin
; 	Found zerocount coefficients
	zcs=fltarr(genconfig.numsid)
	if count eq 1 then zcs=zerocounts(index).zcs*coeff_factor $
	else for i=0,genconfig.numsid-1 do $
		zcs(i)=median(zerocounts(index).zcs(i)*coeff_factor)
; 	Compute estimated L or N band fluxes in Jy for each telescope
;
	case band1 of
		'L': flp_est=mean(matisse_zerocounts(zcs=zcs))
		'N': f12_est=mean(matisse_zerocounts(zcs=zcs))
	endcase
	if fitshistory(h,'mat_mergeAllOiFits') then begin
		flp_est=flp_est*4
		f12_est=f12_est*4
	endif
endif
;
FOR j=0,n_elements(scans)-1 DO BEGIN
;
scans=all_scans(j)
;
index=where(scans.vissqcerr gt 0,count)
if count eq 0 then continue
;
ds_stars=startable.starid
;
x_dir.ch='All'
x_dir.bl='All'
x_dir.pt='All'
;
y_dir.ch='All'
y_dir.bl='All'
y_dir.pt='All'
;
; Plot versus wavelength
ds_x.item=45
set_plotsel
;
!x.range=0
if band eq 'L' and strpos(target,'alfsco') ge 0 then !x.range=[3.962,4.00]
if band eq 'L' and strpos(target,'AHSco') ge 0 then !x.range=[3.87,4.03]
if band eq 'L' and strpos(target,'gam2Nor') ge 0 then !x.range=[3.87,4.03]
;
IF keyword_set(v2) THEN BEGIN
;
; Squared visibilities
ds_y.item=25
set_plotsel
!y.range=[0,1.2]
;
ps_options.e=0
ps_options.l=1
;
case array_config of
	'UTs':	begin
		!y.range=0
		end
	'small':begin
		!y.range=[0,1.0]
		if strpos(target,'alfsco') ge 0 then !y.range=[0,0.1]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[0,0.7]
		end
	'medium':begin
		!y.range=[0,0.3]
		if strpos(target,'alfsco') ge 0 then !y.range=[0,0.01]
		if strpos(target,'AHSco') ge 0 then !y.range=[0,0.7]
		if strpos(target,'AFGL4176') ge 0 then !y.range=[0,0.1]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[0,0.3]
		end
	'large':begin
		!y.range=[0,0.05]
		if strpos(target,'alfsco') ge 0 then !y.range=[0,0.002]
		if strpos(target,'AHSco') ge 0 then !y.range=[0,0.5]
		if strpos(target,'HD143006') ge 0 then !y.range=[0,0.5]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[0,0.1]
		end
	'extended':begin
		!y.range=[0,0.05]
		if strpos(target,'alfsco') ge 0 then !y.range=[0,0.002]
		if strpos(target,'AHSco') ge 0 then !y.range=[0,0.5]
		if strpos(target,'HD143006') ge 0 then !y.range=[0,0.5]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[0,0.1]
		end
endcase
if strpos(target,'HD143006') ge 0 then !y.range=[0,1.2]
;
; Adjustments for N band
if strpos(genconfig.configid,'-N_') ge 0 then begin
	if strpos(target,'AHSco') ge 0 then !y.range=[0,0.3]
	if strpos(target,'AFGL4176') ge 0 then !y.range=[0,0.3]
case array_config of
	'UTs':	begin
		!y.range=0
		end
	'small':begin
		!y.range=[0,1.0]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[0,1.0]
		end
	'medium':begin
		!y.range=[0,0.3]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[0,0.3]
		if strpos(target,'AHSco') ge 0 then !y.range=[0,0.2]
		end
	'large':begin
		!y.range=[0,0.05]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[0,0.1]
		end
	'extended':begin
		!y.range=[0,0.05]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[0,0.1]
		end
endcase
endif
;
if calib then !y.range=[0,1.2]
if keyword_set(adapt) then !y.range=[0,max(scans.vissqc)]
;
plotdata
;
; Add label with atmospheric parameters
xyouts,0.2,.96,'Seeing [a.sec]: '+string(scans.r0,format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.2,.94,'Coh. time [ms]: '+string(scans.t0(0)*1000,format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.96,'Array size: '+array_config, $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.94,'Time (UT): '+hms(scans.time/(24*60),/scan), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.92,'PWV (mm): '+string(scans.pwv(0),format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
;
ENDIF
;
IF keyword_set(cp) THEN BEGIN
;
scans.triplephasec=-scans.triplephasec
scans.triplephasecerr=scans.triplephasecerr/2
;
; Closure phases
if genconfig.numtriple gt 0 then begin
ds_y.item=33
set_plotsel
!y.range=[-180,180]
;
ps_options.e=0
ps_options.l=1
;
case array_config of
	'small':begin
		!y.range=[-90,90]
		if strpos(target,'AHSco') ge 0 then !y.range=[-20,20]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[-30,30]
		end
	'medium':begin
		!y.range=[-140,140]
		if strpos(target,'AHSco') ge 0 then !y.range=[-30,30]
		if strpos(target,'IRAS08544') ge 0 then !y.range=[-150,150]
		if strpos(target,'AFGL4176') ge 0 then !y.range=[-180,180]
		end
	'large':begin
		if strpos(target,'AHSco') ge 0 then !y.range=[-40,40]
		!y.range=[-180,180]
		end
	'extended':begin
		if strpos(target,'AHSco') ge 0 then !y.range=[-40,40]
		!y.range=[-180,180]
		end
endcase
if strpos(target,'HD143006') ge 0 then !y.range=[-50,50]
if strpos(target,'alfsco') ge 0 then !y.range=0
if strpos(target,'gam2Nor') ge 0 then !y.range=[-10,10]
;
; Adjustments for N band
if strpos(genconfig.configid,'-N_') ge 0 then begin
       !y.range=[-180,180]
endif
;
ds_y.tr=0
!p.multi=[0,2,2,0,1]
plotdata,/keep_window
;
if 0 then begin
xyouts,0.2,.96,'Time (UT): '+strmid(dates(ifile),11,9), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.96,'Array size: '+array_config, $
	/normal,color=tci(0),charsize=1.0,charthick=1
endif
;
; Add label with atmospheric parameters
xyouts,0.2,.96,'Seeing [a.sec]: '+string(scans.r0,format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.2,.94,'Coh. time [ms]: '+string(scans.t0(0)*1000,format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.2,.92,bcd, $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.96,'Array size: '+array_config, $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.94,'Time (UT): '+strmid(dates(ifile),11,9), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.92,'PWV (mm): '+string(scans.pwv(0),format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
;
for k=1,3 do begin
	ds_y.tr=k
	plotdata,/keep_window
endfor
;
endif
;
ENDIF
;
IF keyword_set(fl) AND total(scans.photometryc) gt 0 THEN BEGIN
;
; Fluxes
ds_y.item=95
set_plotsel
!y.range=[min(scans.photometryc),max(scans.photometryc)]
!y.range=[0,max([100,max(scans.photometryc)*1.2])]
;
; Here, use telecope IDs
g=genconfig
genconfig.stationid='AT'+string(genconfig.siderostatid,format='(i1)')
;
ps_options.e=0
ps_options.l=1
;
if ds_y.item eq 94 then begin
if strpos(target,'AHSco') ge 0 then !y.range=[-1e3,10e3]
if strpos(target,'IRAS08544') ge 0 then !y.range=[-1e3,25e3]
if strpos(target,'AFGL4176') ge 0 then !y.range=[-1e3,15e3]
if strpos(target,'HD123139') ge 0 then !y.range=[0,14e4]
if strpos(target,'alfsco') ge 0 then !y.range=[-1e3,100e3]
if strpos(target,'epssco') ge 0 then !y.range=[0,800]
if strpos(target,'gam2Nor') ge 0 then !y.range=[0,1200]
if strpos(target,'gsco') ge 0 then !y.range=[0,500]
;
; Adjustments for N band
if strpos(genconfig.configid,'-N_') ge 0 then begin
	if strpos(target,'AHSco') ge 0 then !y.range=[-1e3,10e3]
	if strpos(target,'IRAS08544') ge 0 then !y.range=[-500,2e3]
	if strpos(target,'AFGL4176') ge 0 then !y.range=[-500,2e3]
	if strpos(target,'HD123139') ge 0 then !y.range=[0,800]
	if strpos(target,'alfsco') ge 0 then !y.range=[-1e3,100e3]
	if strpos(target,'epssco') ge 0 then !y.range=[0,1000]
	if strpos(target,'gam2Nor') ge 0 then !y.range=[-20,200]
endif
endif
;
ds_y.ib=0
!p.multi=[0,2,2,0,1]
plotdata,/keep_window
oplot,[3,15],[0,0]
;
; Add label with atmospheric parameters
xyouts,0.2,.96,'Seeing [a.sec]: '+string(scans.r0,format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.2,.94,'Coh. time [ms]: '+string(scans.t0(0)*1000,format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.96,'Array size: '+array_config, $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.94,'Time (UT): '+strmid(dates(ifile),11,9), $
	/normal,color=tci(0),charsize=1.0,charthick=1
xyouts,0.7,.90,'PWV (mm): '+string(scans.pwv(0),format='(f5.2)'), $
	/normal,color=tci(0),charsize=1.0,charthick=1
;
; Add label if CAL with fluxes
if calib then begin
	help,genconfig.configid
	if strpos(genconfig.configid,'-N_') ge 0 then begin
		if startable.f12 ne 0 then $
		xyouts,0.2,0.92,'N (MDFC) [Jy] = '+string(startable.f12), $
			/normal,color=tci(0),charsize=1.0,charthick=1
		xyouts,0.7,0.92,'N (est.) [Jy] = '+string(f12_est), $
			/normal,color=tci(0),charsize=1.0,charthick=1
	endif
	if strpos(genconfig.configid,'-L_') ge 0 then begin
		if startable.flp ne 0 then $
		xyouts,0.2,0.92,'L (MDFC) [Jy] = '+string(startable.flp), $
			/normal,color=tci(0),charsize=1.0,charthick=1
		xyouts,0.7,0.92,'L (est.) [Jy] = '+string(flp_est), $
			/normal,color=tci(0),charsize=1.0,charthick=1
	endif
endif else begin
	if strpos(genconfig.configid,'_N_') ge 0 then $
		xyouts,0.2,0.92,'N (es-.) [Jy] = '+string(f12_est), $
			/normal,color=tci(0),charsize=1.0,charthick=1
	if strpos(genconfig.configid,'-L_') ge 0 then $
		xyouts,0.2,0.92,'L (est.) [Jy] = '+string(flp_est), $
			/normal,color=tci(0),charsize=1.0,charthick=1
endelse
;
for k=1,3 do begin
	ds_y.ib=k
	plotdata,/keep_window
	oplot,[3,15],[0,0]
endfor
;
genconfig=g
;
ENDIF
;
ENDFOR	; scans
;
ENDFOR	; files
;
device,/close
print,'Plot file: '+plot_file
;
end
;-------------------------------------------------------------------------------
pro allcalplots_fl_LM
;
mymatisseplots,target='gam2Nor',band='LM',/fl
mymatisseplots,target='epsSco',band='LM',/fl
mymatisseplots,target='epssco',band='LM',/fl
mymatisseplots,target='gsco',band='LM',/fl
mymatisseplots,target='IRAS12563',band='LM',/fl
mymatisseplots,target='IRAS10153',band='LM',/fl
mymatisseplots,target='IRAS08534',band='LM',/fl
mymatisseplots,target='HD115418',band='LM',/fl
mymatisseplots,target='HD123139',band='LM',/fl
;
end
;-------------------------------------------------------------------------------
pro allcalplots_fl_N
;
mymatisseplots,target='gam2Nor',band='N',/fl
mymatisseplots,target='epsSco',band='N',/fl
mymatisseplots,target='epssco',band='N',/fl
mymatisseplots,target='gsco',band='N',/fl
mymatisseplots,target='IRAS12563',band='N',/fl
mymatisseplots,target='IRAS10153',band='N',/fl
mymatisseplots,target='IRAS08534',band='N',/fl
mymatisseplots,target='HD115418',band='N',/fl
mymatisseplots,target='HD123139',band='N',/fl
;
end
;-------------------------------------------------------------------------------
pro allsciplots_v2_LM
;
mymatisseplots,target='AHSco',band='LM',/v2
mymatisseplots,target='alfsco',band='LM',/v2
mymatisseplots,target='AFGL4176',band='LM',/v2
mymatisseplots,target='IRAS08544',band='LM',/v2
;
end
;-------------------------------------------------------------------------------
pro allsciplots_v2_N
;
mymatisseplots,target='AHSco',band='N',/v2
mymatisseplots,target='alfsco',band='N',/v2
mymatisseplots,target='AFGL4176',band='N',/v2
mymatisseplots,target='IRAS08544',band='N',/v2
;
end
;-------------------------------------------------------------------------------
pro allsciplots_cp_LM
;
mymatisseplots,target='AHSco',band='LM',/cp
mymatisseplots,target='alfsco',band='LM',/cp
mymatisseplots,target='AFGL4176',band='LM',/cp
mymatisseplots,target='IRAS08544',band='LM',/cp
;
end
;-------------------------------------------------------------------------------
pro allsciplots_cp_N
;
mymatisseplots,target='AHSco',band='N',/cp
mymatisseplots,target='alfsco',band='N',/cp
mymatisseplots,target='AFGL4176',band='N',/cp
mymatisseplots,target='IRAS08544',band='N',/cp
;
end
;-------------------------------------------------------------------------------
pro allsciplots_fl_LM
;
mymatisseplots,target='AHSco',band='LM',/fl
mymatisseplots,target='alfsco',band='LM',/fl
mymatisseplots,target='AFGL4176',band='LM',/fl
mymatisseplots,target='IRAS08544',band='LM',/fl
;
end
;-------------------------------------------------------------------------------
pro allsciplots_fl_N
;
mymatisseplots,target='AHSco',band='N',/fl
mymatisseplots,target='alfsco',band='N',/fl
mymatisseplots,target='AFGL4176',band='N',/fl
mymatisseplots,target='IRAS08544',band='N',/fl
;
end
;-------------------------------------------------------------------------------
pro allsvplots_fl
;
mymatisseplots,target='AHSco',band='N',/fl
mymatisseplots,target='alfsco',band='N',/fl
mymatisseplots,target='AFGL4176',band='N',/fl
mymatisseplots,target='IRAS08544',band='N',/fl
mymatisseplots,target='gam2Nor',band='N',/fl
mymatisseplots,target='epsSco',band='N',/fl
mymatisseplots,target='epssco',band='N',/fl
mymatisseplots,target='gsco',band='N',/fl
mymatisseplots,target='IRAS12563',band='N',/fl
mymatisseplots,target='IRAS10153',band='N',/fl
mymatisseplots,target='IRAS08534',band='N',/fl
mymatisseplots,target='HD115418',band='N',/fl
mymatisseplots,target='HD123139',band='N',/fl
;
mymatisseplots,target='AHSco',band='LM',/fl
mymatisseplots,target='alfsco',band='LM',/fl
mymatisseplots,target='AFGL4176',band='LM',/fl
mymatisseplots,target='IRAS08544',band='LM',/fl
mymatisseplots,target='gam2Nor',band='LM',/fl
mymatisseplots,target='epsSco',band='LM',/fl
mymatisseplots,target='epssco',band='LM',/fl
mymatisseplots,target='gsco',band='LM',/fl
mymatisseplots,target='IRAS12563',band='LM',/fl
mymatisseplots,target='IRAS10153',band='LM',/fl
mymatisseplots,target='IRAS08534',band='LM',/fl
mymatisseplots,target='HD115418',band='LM',/fl
mymatisseplots,target='HD123139',band='LM',/fl
;
end
;-------------------------------------------------------------------------------
pro allsvplots_small
;
mymatisseplots,config='small',target='AHSco',band='N',/fl
mymatisseplots,config='small',target='alfsco',band='N',/fl
mymatisseplots,config='small',target='AFGL4176',band='N',/fl
mymatisseplots,config='small',target='IRAS08544',band='N',/fl
mymatisseplots,config='small',target='gam2Nor',band='N',/fl
mymatisseplots,config='small',target='epsSco',band='N',/fl
mymatisseplots,config='small',target='epssco',band='N',/fl
mymatisseplots,config='small',target='gsco',band='N',/fl
mymatisseplots,config='small',target='IRAS12563',band='N',/fl
mymatisseplots,config='small',target='IRAS10153',band='N',/fl
mymatisseplots,config='small',target='IRAS08534',band='N',/fl
mymatisseplots,config='small',target='HD115418',band='N',/fl
mymatisseplots,config='small',target='HD123139',band='N',/fl
;
mymatisseplots,config='small',target='AHSco',band='LM',/fl
mymatisseplots,config='small',target='alfsco',band='LM',/fl
mymatisseplots,config='small',target='AFGL4176',band='LM',/fl
mymatisseplots,config='small',target='IRAS08544',band='LM',/fl
mymatisseplots,config='small',target='gam2Nor',band='LM',/fl
mymatisseplots,config='small',target='epsSco',band='LM',/fl
mymatisseplots,config='small',target='epssco',band='LM',/fl
mymatisseplots,config='small',target='gsco',band='LM',/fl
mymatisseplots,config='small',target='IRAS12563',band='LM',/fl
mymatisseplots,config='small',target='IRAS10153',band='LM',/fl
mymatisseplots,config='small',target='IRAS08534',band='LM',/fl
mymatisseplots,config='small',target='HD115418',band='LM',/fl
mymatisseplots,config='small',target='HD123139',band='LM',/fl
;
mymatisseplots,config='small',target='AHSco',band='N',/v2
mymatisseplots,config='small',target='alfsco',band='N',/v2
mymatisseplots,config='small',target='AFGL4176',band='N',/v2
mymatisseplots,config='small',target='IRAS08544',band='N',/v2
mymatisseplots,config='small',target='gam2Nor',band='N',/v2
mymatisseplots,config='small',target='epsSco',band='N',/v2
mymatisseplots,config='small',target='epssco',band='N',/v2
mymatisseplots,config='small',target='gsco',band='N',/v2
mymatisseplots,config='small',target='IRAS12563',band='N',/v2
mymatisseplots,config='small',target='IRAS10153',band='N',/v2
mymatisseplots,config='small',target='IRAS08534',band='N',/v2
mymatisseplots,config='small',target='HD115418',band='N',/v2
mymatisseplots,config='small',target='HD123139',band='N',/v2
;
mymatisseplots,config='small',target='AHSco',band='LM',/v2
mymatisseplots,config='small',target='alfsco',band='LM',/v2
mymatisseplots,config='small',target='AFGL4176',band='LM',/v2
mymatisseplots,config='small',target='IRAS08544',band='LM',/v2
mymatisseplots,config='small',target='gam2Nor',band='LM',/v2
mymatisseplots,config='small',target='epsSco',band='LM',/v2
mymatisseplots,config='small',target='epssco',band='LM',/v2
mymatisseplots,config='small',target='gsco',band='LM',/v2
mymatisseplots,config='small',target='IRAS12563',band='LM',/v2
mymatisseplots,config='small',target='IRAS10153',band='LM',/v2
mymatisseplots,config='small',target='IRAS08534',band='LM',/v2
mymatisseplots,config='small',target='HD115418',band='LM',/v2
mymatisseplots,config='small',target='HD123139',band='LM',/v2
;
mymatisseplots,config='small',target='AHSco',band='N',/cp
mymatisseplots,config='small',target='alfsco',band='N',/cp
mymatisseplots,config='small',target='AFGL4176',band='N',/cp
mymatisseplots,config='small',target='IRAS08544',band='N',/cp
mymatisseplots,config='small',target='gam2Nor',band='N',/cp
mymatisseplots,config='small',target='epsSco',band='N',/cp
mymatisseplots,config='small',target='epssco',band='N',/cp
mymatisseplots,config='small',target='gsco',band='N',/cp
mymatisseplots,config='small',target='IRAS12563',band='N',/cp
mymatisseplots,config='small',target='IRAS10153',band='N',/cp
mymatisseplots,config='small',target='IRAS08534',band='N',/cp
mymatisseplots,config='small',target='HD115418',band='N',/cp
mymatisseplots,config='small',target='HD123139',band='N',/cp
;
mymatisseplots,config='small',target='AHSco',band='LM',/cp
mymatisseplots,config='small',target='alfsco',band='LM',/cp
mymatisseplots,config='small',target='AFGL4176',band='LM',/cp
mymatisseplots,config='small',target='IRAS08544',band='LM',/cp
mymatisseplots,config='small',target='gam2Nor',band='LM',/cp
mymatisseplots,config='small',target='epsSco',band='LM',/cp
mymatisseplots,config='small',target='epssco',band='LM',/cp
mymatisseplots,config='small',target='gsco',band='LM',/cp
mymatisseplots,config='small',target='IRAS12563',band='LM',/cp
mymatisseplots,config='small',target='IRAS10153',band='LM',/cp
mymatisseplots,config='small',target='IRAS08534',band='LM',/cp
mymatisseplots,config='small',target='HD115418',band='LM',/cp
mymatisseplots,config='small',target='HD123139',band='LM',/cp
;
end
;-------------------------------------------------------------------------------
pro allsvplots_medium
;
mymatisseplots,config='medium',target='AHSco',band='N',/fl
mymatisseplots,config='medium',target='alfsco',band='N',/fl
mymatisseplots,config='medium',target='AFGL4176',band='N',/fl
mymatisseplots,config='medium',target='IRAS08544',band='N',/fl
mymatisseplots,config='medium',target='gam2Nor',band='N',/fl
mymatisseplots,config='medium',target='epsSco',band='N',/fl
mymatisseplots,config='medium',target='epssco',band='N',/fl
mymatisseplots,config='medium',target='gsco',band='N',/fl
mymatisseplots,config='medium',target='IRAS12563',band='N',/fl
mymatisseplots,config='medium',target='IRAS10153',band='N',/fl
mymatisseplots,config='medium',target='IRAS08534',band='N',/fl
mymatisseplots,config='medium',target='HD115418',band='N',/fl
mymatisseplots,config='medium',target='HD123139',band='N',/fl
;
mymatisseplots,config='medium',target='AHSco',band='LM',/fl
mymatisseplots,config='medium',target='alfsco',band='LM',/fl
mymatisseplots,config='medium',target='AFGL4176',band='LM',/fl
mymatisseplots,config='medium',target='IRAS08544',band='LM',/fl
mymatisseplots,config='medium',target='gam2Nor',band='LM',/fl
mymatisseplots,config='medium',target='epsSco',band='LM',/fl
mymatisseplots,config='medium',target='epssco',band='LM',/fl
mymatisseplots,config='medium',target='gsco',band='LM',/fl
mymatisseplots,config='medium',target='IRAS12563',band='LM',/fl
mymatisseplots,config='medium',target='IRAS10153',band='LM',/fl
mymatisseplots,config='medium',target='IRAS08534',band='LM',/fl
mymatisseplots,config='medium',target='HD115418',band='LM',/fl
mymatisseplots,config='medium',target='HD123139',band='LM',/fl
;
mymatisseplots,config='medium',target='AHSco',band='N',/v2
mymatisseplots,config='medium',target='alfsco',band='N',/v2
mymatisseplots,config='medium',target='AFGL4176',band='N',/v2
mymatisseplots,config='medium',target='IRAS08544',band='N',/v2
mymatisseplots,config='medium',target='gam2Nor',band='N',/v2
mymatisseplots,config='medium',target='epsSco',band='N',/v2
mymatisseplots,config='medium',target='epssco',band='N',/v2
mymatisseplots,config='medium',target='gsco',band='N',/v2
mymatisseplots,config='medium',target='IRAS12563',band='N',/v2
mymatisseplots,config='medium',target='IRAS10153',band='N',/v2
mymatisseplots,config='medium',target='IRAS08534',band='N',/v2
mymatisseplots,config='medium',target='HD115418',band='N',/v2
mymatisseplots,config='medium',target='HD123139',band='N',/v2
;
mymatisseplots,config='medium',target='AHSco',band='LM',/v2
mymatisseplots,config='medium',target='alfsco',band='LM',/v2
mymatisseplots,config='medium',target='AFGL4176',band='LM',/v2
mymatisseplots,config='medium',target='IRAS08544',band='LM',/v2
mymatisseplots,config='medium',target='gam2Nor',band='LM',/v2
mymatisseplots,config='medium',target='epsSco',band='LM',/v2
mymatisseplots,config='medium',target='epssco',band='LM',/v2
mymatisseplots,config='medium',target='gsco',band='LM',/v2
mymatisseplots,config='medium',target='IRAS12563',band='LM',/v2
mymatisseplots,config='medium',target='IRAS10153',band='LM',/v2
mymatisseplots,config='medium',target='IRAS08534',band='LM',/v2
mymatisseplots,config='medium',target='HD115418',band='LM',/v2
mymatisseplots,config='medium',target='HD123139',band='LM',/v2
;
mymatisseplots,config='medium',target='AHSco',band='N',/cp
mymatisseplots,config='medium',target='alfsco',band='N',/cp
mymatisseplots,config='medium',target='AFGL4176',band='N',/cp
mymatisseplots,config='medium',target='IRAS08544',band='N',/cp
mymatisseplots,config='medium',target='gam2Nor',band='N',/cp
mymatisseplots,config='medium',target='epsSco',band='N',/cp
mymatisseplots,config='medium',target='epssco',band='N',/cp
mymatisseplots,config='medium',target='gsco',band='N',/cp
mymatisseplots,config='medium',target='IRAS12563',band='N',/cp
mymatisseplots,config='medium',target='IRAS10153',band='N',/cp
mymatisseplots,config='medium',target='IRAS08534',band='N',/cp
mymatisseplots,config='medium',target='HD115418',band='N',/cp
mymatisseplots,config='medium',target='HD123139',band='N',/cp
;
mymatisseplots,config='medium',target='AHSco',band='LM',/cp
mymatisseplots,config='medium',target='alfsco',band='LM',/cp
mymatisseplots,config='medium',target='AFGL4176',band='LM',/cp
mymatisseplots,config='medium',target='IRAS08544',band='LM',/cp
mymatisseplots,config='medium',target='gam2Nor',band='LM',/cp
mymatisseplots,config='medium',target='epsSco',band='LM',/cp
mymatisseplots,config='medium',target='epssco',band='LM',/cp
mymatisseplots,config='medium',target='gsco',band='LM',/cp
mymatisseplots,config='medium',target='IRAS12563',band='LM',/cp
mymatisseplots,config='medium',target='IRAS10153',band='LM',/cp
mymatisseplots,config='medium',target='IRAS08534',band='LM',/cp
mymatisseplots,config='medium',target='HD115418',band='LM',/cp
mymatisseplots,config='medium',target='HD123139',band='LM',/cp
;
end
;-------------------------------------------------------------------------------
pro allsvplots_large
;
mymatisseplots,config='large',target='AHSco',band='N',/fl
mymatisseplots,config='large',target='alfsco',band='N',/fl
mymatisseplots,config='large',target='AFGL4176',band='N',/fl
mymatisseplots,config='large',target='IRAS08544',band='N',/fl
mymatisseplots,config='large',target='gam2Nor',band='N',/fl
mymatisseplots,config='large',target='epsSco',band='N',/fl
mymatisseplots,config='large',target='epssco',band='N',/fl
mymatisseplots,config='large',target='gsco',band='N',/fl
mymatisseplots,config='large',target='IRAS12563',band='N',/fl
mymatisseplots,config='large',target='IRAS10153',band='N',/fl
mymatisseplots,config='large',target='IRAS08534',band='N',/fl
mymatisseplots,config='large',target='HD115418',band='N',/fl
mymatisseplots,config='large',target='HD123139',band='N',/fl
;
mymatisseplots,config='large',target='AHSco',band='LM',/fl
mymatisseplots,config='large',target='alfsco',band='LM',/fl
mymatisseplots,config='large',target='AFGL4176',band='LM',/fl
mymatisseplots,config='large',target='IRAS08544',band='LM',/fl
mymatisseplots,config='large',target='gam2Nor',band='LM',/fl
mymatisseplots,config='large',target='epsSco',band='LM',/fl
mymatisseplots,config='large',target='epssco',band='LM',/fl
mymatisseplots,config='large',target='gsco',band='LM',/fl
mymatisseplots,config='large',target='IRAS12563',band='LM',/fl
mymatisseplots,config='large',target='IRAS10153',band='LM',/fl
mymatisseplots,config='large',target='IRAS08534',band='LM',/fl
mymatisseplots,config='large',target='HD115418',band='LM',/fl
mymatisseplots,config='large',target='HD123139',band='LM',/fl
;
mymatisseplots,config='large',target='AHSco',band='N',/v2
mymatisseplots,config='large',target='alfsco',band='N',/v2
mymatisseplots,config='large',target='AFGL4176',band='N',/v2
mymatisseplots,config='large',target='IRAS08544',band='N',/v2
mymatisseplots,config='large',target='gam2Nor',band='N',/v2
mymatisseplots,config='large',target='epsSco',band='N',/v2
mymatisseplots,config='large',target='epssco',band='N',/v2
mymatisseplots,config='large',target='gsco',band='N',/v2
mymatisseplots,config='large',target='IRAS12563',band='N',/v2
mymatisseplots,config='large',target='IRAS10153',band='N',/v2
mymatisseplots,config='large',target='IRAS08534',band='N',/v2
mymatisseplots,config='large',target='HD115418',band='N',/v2
mymatisseplots,config='large',target='HD123139',band='N',/v2
;
mymatisseplots,config='large',target='AHSco',band='LM',/v2
mymatisseplots,config='large',target='alfsco',band='LM',/v2
mymatisseplots,config='large',target='AFGL4176',band='LM',/v2
mymatisseplots,config='large',target='IRAS08544',band='LM',/v2
mymatisseplots,config='large',target='gam2Nor',band='LM',/v2
mymatisseplots,config='large',target='epsSco',band='LM',/v2
mymatisseplots,config='large',target='epssco',band='LM',/v2
mymatisseplots,config='large',target='gsco',band='LM',/v2
mymatisseplots,config='large',target='IRAS12563',band='LM',/v2
mymatisseplots,config='large',target='IRAS10153',band='LM',/v2
mymatisseplots,config='large',target='IRAS08534',band='LM',/v2
mymatisseplots,config='large',target='HD115418',band='LM',/v2
mymatisseplots,config='large',target='HD123139',band='LM',/v2
;
mymatisseplots,config='large',target='AHSco',band='N',/cp
mymatisseplots,config='large',target='alfsco',band='N',/cp
mymatisseplots,config='large',target='AFGL4176',band='N',/cp
mymatisseplots,config='large',target='IRAS08544',band='N',/cp
mymatisseplots,config='large',target='gam2Nor',band='N',/cp
mymatisseplots,config='large',target='epsSco',band='N',/cp
mymatisseplots,config='large',target='epssco',band='N',/cp
mymatisseplots,config='large',target='gsco',band='N',/cp
mymatisseplots,config='large',target='IRAS12563',band='N',/cp
mymatisseplots,config='large',target='IRAS10153',band='N',/cp
mymatisseplots,config='large',target='IRAS08534',band='N',/cp
mymatisseplots,config='large',target='HD115418',band='N',/cp
mymatisseplots,config='large',target='HD123139',band='N',/cp
;
mymatisseplots,config='large',target='AHSco',band='LM',/cp
mymatisseplots,config='large',target='alfsco',band='LM',/cp
mymatisseplots,config='large',target='AFGL4176',band='LM',/cp
mymatisseplots,config='large',target='IRAS08544',band='LM',/cp
mymatisseplots,config='large',target='gam2Nor',band='LM',/cp
mymatisseplots,config='large',target='epsSco',band='LM',/cp
mymatisseplots,config='large',target='epssco',band='LM',/cp
mymatisseplots,config='large',target='gsco',band='LM',/cp
mymatisseplots,config='large',target='IRAS12563',band='LM',/cp
mymatisseplots,config='large',target='IRAS10153',band='LM',/cp
mymatisseplots,config='large',target='IRAS08534',band='LM',/cp
mymatisseplots,config='large',target='HD115418',band='LM',/cp
mymatisseplots,config='large',target='HD123139',band='LM',/cp
;
end
;-------------------------------------------------------------------------------
pro allsvsciplots_small
;
mymatisseplots,config='small',target='AHSco',band='N',/fl
mymatisseplots,config='small',target='alfsco',band='N',/fl
mymatisseplots,config='small',target='AFGL4176',band='N',/fl
mymatisseplots,config='small',target='IRAS08544',band='N',/fl
;
mymatisseplots,config='small',target='AHSco',band='LM',/fl
mymatisseplots,config='small',target='alfsco',band='LM',/fl
mymatisseplots,config='small',target='AFGL4176',band='LM',/fl
mymatisseplots,config='small',target='IRAS08544',band='LM',/fl
;
return
mymatisseplots,config='small',target='AHSco',band='N',/v2
mymatisseplots,config='small',target='alfsco',band='N',/v2
mymatisseplots,config='small',target='AFGL4176',band='N',/v2
mymatisseplots,config='small',target='IRAS08544',band='N',/v2
;
mymatisseplots,config='small',target='AHSco',band='LM',/v2
mymatisseplots,config='small',target='alfsco',band='LM',/v2
mymatisseplots,config='small',target='AFGL4176',band='LM',/v2
mymatisseplots,config='small',target='IRAS08544',band='LM',/v2
;
mymatisseplots,config='small',target='AHSco',band='N',/cp
mymatisseplots,config='small',target='alfsco',band='N',/cp
mymatisseplots,config='small',target='AFGL4176',band='N',/cp
mymatisseplots,config='small',target='IRAS08544',band='N',/cp
;
mymatisseplots,config='small',target='AHSco',band='LM',/cp
mymatisseplots,config='small',target='alfsco',band='LM',/cp
mymatisseplots,config='small',target='AFGL4176',band='LM',/cp
mymatisseplots,config='small',target='IRAS08544',band='LM',/cp
;
end
;-------------------------------------------------------------------------------
pro allsvsciplots_medium
;
mymatisseplots,config='medium',target='AHSco',band='N',/fl
mymatisseplots,config='medium',target='alfsco',band='N',/fl
mymatisseplots,config='medium',target='AFGL4176',band='N',/fl
mymatisseplots,config='medium',target='IRAS08544',band='N',/fl
;
mymatisseplots,config='medium',target='AHSco',band='LM',/fl
mymatisseplots,config='medium',target='alfsco',band='LM',/fl
mymatisseplots,config='medium',target='AFGL4176',band='LM',/fl
mymatisseplots,config='medium',target='IRAS08544',band='LM',/fl
;
return
mymatisseplots,config='medium',target='AHSco',band='N',/v2
mymatisseplots,config='medium',target='alfsco',band='N',/v2
mymatisseplots,config='medium',target='AFGL4176',band='N',/v2
mymatisseplots,config='medium',target='IRAS08544',band='N',/v2
;
mymatisseplots,config='medium',target='AHSco',band='LM',/v2
mymatisseplots,config='medium',target='alfsco',band='LM',/v2
mymatisseplots,config='medium',target='AFGL4176',band='LM',/v2
mymatisseplots,config='medium',target='IRAS08544',band='LM',/v2
;
mymatisseplots,config='medium',target='AHSco',band='N',/cp
mymatisseplots,config='medium',target='alfsco',band='N',/cp
mymatisseplots,config='medium',target='AFGL4176',band='N',/cp
mymatisseplots,config='medium',target='IRAS08544',band='N',/cp
;
mymatisseplots,config='medium',target='AHSco',band='LM',/cp
mymatisseplots,config='medium',target='alfsco',band='LM',/cp
mymatisseplots,config='medium',target='AFGL4176',band='LM',/cp
mymatisseplots,config='medium',target='IRAS08544',band='LM',/cp
;
end
;-------------------------------------------------------------------------------
pro allsvsciplots_large
;
mymatisseplots,config='large',target='AHSco',band='N',/fl
mymatisseplots,config='large',target='alfsco',band='N',/fl
mymatisseplots,config='large',target='AFGL4176',band='N',/fl
mymatisseplots,config='large',target='IRAS08544',band='N',/fl
;
mymatisseplots,config='large',target='AHSco',band='LM',/fl
mymatisseplots,config='large',target='alfsco',band='LM',/fl
mymatisseplots,config='large',target='AFGL4176',band='LM',/fl
mymatisseplots,config='large',target='IRAS08544',band='LM',/fl
;
return
mymatisseplots,config='large',target='AHSco',band='N',/v2
mymatisseplots,config='large',target='alfsco',band='N',/v2
mymatisseplots,config='large',target='AFGL4176',band='N',/v2
mymatisseplots,config='large',target='IRAS08544',band='N',/v2
;
mymatisseplots,config='large',target='AHSco',band='LM',/v2
mymatisseplots,config='large',target='alfsco',band='LM',/v2
mymatisseplots,config='large',target='AFGL4176',band='LM',/v2
mymatisseplots,config='large',target='IRAS08544',band='LM',/v2
;
mymatisseplots,config='large',target='AHSco',band='N',/cp
mymatisseplots,config='large',target='alfsco',band='N',/cp
mymatisseplots,config='large',target='AFGL4176',band='N',/cp
mymatisseplots,config='large',target='IRAS08544',band='N',/cp
;
mymatisseplots,config='large',target='AHSco',band='LM',/cp
mymatisseplots,config='large',target='alfsco',band='LM',/cp
mymatisseplots,config='large',target='AFGL4176',band='LM',/cp
mymatisseplots,config='large',target='IRAS08544',band='LM',/cp
;
end
;************************************************************************Block 5
pro linknightstoendproducts
;
; Obsolete!
; In the current folder (e.g., /iot/matisse/sv), create date-named folders with 
; links to TPL folders containing SV endproducts produced by Garching pipeline. 
;
; Remove links with "rm -rf ????-??-??'.
; 
dirname='/iot/matisse/sv'
tpl='MATIS.????-??-??T??:??:??.???_tpl'
tpl_dirs=file_search(dirname+'/reflex_end_products',tpl)
tpl_fils=file_basename(tpl_dirs)
tpl_dates=strmid(tpl_fils,6,10)
tpl_times=strmid(tpl_fils,17,8)
for i=0,n_elements(tpl_dirs)-1 do if hms2h(tpl_times(i)) lt 12 $
	then tpl_dates(i)=previousdate(tpl_dates(i))
dates=unique(tpl_dates)
for i=0,n_elements(dates)-1 do begin
	first=1
	index=where(tpl_dates eq dates(i),count)
	for j=0,count-1 do begin
		r=file_search(tpl_dirs(index(j)),'*.fits')
		if strlen(r(0)) ne 0 then begin
			if first then begin
				file_mkdir,dates(i)
				cd,dates(i),current=old_dir
				first=0
			endif
			spawn,'ln -s '+tpl_dirs(index(j))+' '+tpl_fils(j)
		endif
	endfor
	if not first then cd,old_dir
endfor
;
end
;-------------------------------------------------------------------------------
pro renamesciopsendproducts
;
; Run in /iot/matisse/sv. Procedure will create folder reflex_end_products and
; in it a folder with today's date. All files in directories YYY-MM-DD
; will be renamed and copied to the regular reflex_end_products tree. Rename
; r.MATIS*_tpl*.fits to SCI/CAL_<target>_RAW_INT_YYY-MM-DDTHH:MM:SS.SSS.fits.
; and move the products of the SciOps pipeline into a folders like MATIS.*_tpl.
;
; Note: There are no files for 2019-05-06 (bad weather)
;
; List files ordered with time: ls | sort -t _ +5 -6
;
if strpos(pwd(),'/iot/matisse/SVDataPipe') then begin
	print,'You must be in /iot/matisse/SV... to run this procedure!'
	return
endif       
rep='reflex_end_products'
if not file_test(rep,/directory) $
	then file_mkdir,'reflex_end_products'
t=systime()
words=nameparse(t)
time_stamp=words(3)
parseidldate,t,y,m,d
date=constrictordate(y,m,d)+'T'+time_stamp
cd,'reflex_end_products',current=old_dir
file_mkdir,date
cd,old_dir
rep=rep+'/'+date
;
spawn,'/scisoft/bin/dfits */*.fits | fitsort DATE-OBS PRO.CATG OBS.TARG.NAME',f
f=f(1:n_elements(f)-1)
nf=n_elements(f)
;
jd=dblarr(nf)
tpl=strarr(nf)
;
for i=0,nf-1 do begin
	words=nameparse(f(i))
	keywords=words(1:n_elements(words)-1)
	words=nameparse(words(0),'/')
	tpl(i)=strmid(words(1),2,33)
	parsedate,strmid(keywords(0),0,10),y,m,d
	dt=nameparse(keywords(0),'T')
	df=hms2h(dt(1))/24
	jd(i)=julian(y,m,double(d)+df)
endfor
;
ljd=long(jd)
ujd=unique(ljd,ui)
;
for i=0,n_elements(ui)-1 do begin
	dest_dir=rep+'/'+tpl(ui(i))
	if strpos(dest_dir,'MATIS.2019-05-13') then continue
	file_mkdir,dest_dir
	index=where(ljd eq ujd(i),n)
	for j=0,n-1 do begin
		keywords=nameparse(f(index(j)))
		nk=n_elements(keywords)
		if nk gt 4 then keywords=[keywords(0:2), $
					strjoin(keywords(3:nk-1),'-')]
		if strpos(keywords(2),'RAW_INT') ge 0 then begin
		words=nameparse(keywords(2),'_')
		case words(0) of
		'CALIB': tag='CAL_'+keywords(3)+'_CALIB_RAW_INT'
		'TARGET':tag='SCI_'+keywords(3)+'_TARGET_RAW_INT'
		endcase
		tag=tag+'_'+keywords(1)
		f0=tag+'.fits'
		spawn,'cp '+keywords(0)+' '+dest_dir+'/'+f0
		endif
	endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro renamewkfendproducts
;
; Procedure will rename files to M.* master calibrations for the demo data set.
; cd,reflex_end_products/YYYY-MM-DDTHH:MM:SS/MATIS.YYYY-MM-DDTHH:MM:SS.???_tpl.
; Run in each folder with the FITS files. Then:
; cd reflex_input/reflex_end_products/2020-05-01T14:04:32
; mv */*.fits /iot/matisse/data_wkf/reflex_input/matisse/matisse_viscal/Calibrators_processed/LM
; cd reflex_input/reflex_end_products/2020-05-01T15:07:50
; mv */*.fits /iot/matisse/data_wkf/reflex_input/matisse/matisse_viscal/Calibrators_processed/N
;
; Get date stamp of folder under reflex_end_products
path=pwd()
k=strpos(path,'reflex_end_products')
today=strmid(path,k+20,10)
k=strpos(path,'_tpl')
tpl=strmid(path,k-3,3)
;
f=file_search('*.fits')
;
for i=0,n_elements(f)-1 do begin
	d=mrdfits(f(i),0,h,/silent)
	obs_date=fitshparse(h,'DATE-OBS')
	obs_time=strmid(obs_date,11,9+3);+tpl
	spawn,'mv '+f(i)+' M.MATISSE.'+today+'T'+obs_time+'.fits'
endfor
;
end
;-------------------------------------------------------------------------------
pro checkuvcoords
;
; In the current directory, read all OIFITS files and write
; the results of the uv coordinate check into a text file.
;
f=file_search('*.fits')
if strlen(f(0)) ne 0 then begin
;
for i=0,n_elements(f)-1 do get_oifits,f(i),/uvcheck
spawn,'cat *.txt | grep Median | sort'
;
endif
end
;-------------------------------------------------------------------------------
