;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;midiUtilities
;#description#
;general, mostly undocumented midi utilities
;#end_class#
;*******************************************************

PRO tvsclm,a,b,c,d
if (N_PARAMS() LT 2) then b=1
s=size(a)
if (N_PARAMS() LT 3) then tvscl,rebin(a,b*s[1], b*s[2]) $
else tvscl,rebin(a,b*s[1],b*s[2]),c,d
RETURN
END

PRO tvm,a,b,c,d
if (N_PARAMS() LT 2) then b=1
s=size(a)
if (N_PARAMS() LT 3) then tv,rebin(a,b*s[1], b*s[2]) $
else tv,rebin(a,b*s[1],b*s[2]),c,d
RETURN
END

FUNCTION gf,last,silent=silent,dir=dir
if (NOT KEYWORD_SET(dir)) then dir=''
z=findfile(dir+'M*.fits')
if (NOT KEYWORD_SET(silent)) then print,N_ELEMENTS(z),' files'
if (N_ELEMENTS(last) GE 0) then last=z[N_ELEMENTS(z)-1]
RETURN,z
END

FUNCTION gfn,last,silent=silent,dir=dir
if (NOT KEYWORD_SET(dir)) then dir=''
z=findfile(dir+'M*.fits')
nz = N_ELEMENTS(z)
ok = bytarr(nz)
for i=0,nz-1 do begin
   a = obj_new('fitsfile',z[i])
   ok[i] = N_ELEMENTS(a->listextensions()) GT 2
   a->close
   obj_destroy,a
endfor
if (total(ok) GT 0) then zz = z(where(ok)) else zz=''
if (NOT KEYWORD_SET(silent)) then print,N_ELEMENTS(zz),' files'
if (N_ELEMENTS(last) GE 0) then last=zz[N_ELEMENTS(zz)-1]
RETURN,zz
END


FUNCTION gfu,last,silent=silent,dir=dir
if (NOT KEYWORD_SET(dir)) then dir='.'
spawn,'find '+dir+' -name "M*.fits" -size +1000 -print >fitslist'
openr,unit,'fitslist',/get_lun
z=''
ll=''
readf,unit,z
while(not eof(unit)) do begin readf,unit,ll&z=[z,ll]&endwhile
close,1
if (NOT KEYWORD_SET(silent)) then print,N_ELEMENTS(z),' files'
if (N_ELEMENTS(last) GE 0) then last=z[N_ELEMENTS(z)-1]
RETURN,z
END

FUNCTION getfits, firstfile, nfile
f = gf(/silent)
t = size(firstfile,/type)
if (t eq 7) then begin
   first = where(strpos(f, firstfile) GE 0)
   if (max(first) eq -1) then begin
      print, 'No files found containing '+firstfile
      RETURN,''
   endif else if (N_ELEMENTS(first) GT 1) then begin
      print, 'More than one files contain '+firstfile
      print, f(first)
      RETURN,''
   endif
   first = first[0]
endif else first = firstfile
RETURN, f(first:first+nfile-1)
END

pro readchop,f1,np, a1, a2, a1sum, a2sum
f=gf()
a1=fltarr(62,69,np)
a2=a1
j=0
for i=f1,f1+np-1 do begin
   a=oirgetmeanrms(f[i])
   a1(*,*,j)=a[0].data1
   a2(*,*,j)=a[0].data2
   j=j+1
endfor
for j=0, np-1, 2 do begin
   a1(*,*,j)=-a1(*,*,j)
   a2(*,*,j)=-a2(*,*,j)
endfor

a1sum = total(a1,3)/np
a2sum = total(a2,3)/np

RETURN
END

FUNCTION unstripe,image,dy
s=size(image)
z=total(image(*,0:dy-1),2)+total(image(*,(s[2]-dy):*),2)
z=.5*z/dy
for i=0,s[2]-1 do image(*,i)=image(*,i)-z
RETURN,image
END

pro getcube,fs,nc
f=gf()
RETURN
END

pro makenewMask, oldmask, newmask, x1, y1, x2, y2, dx
maskdata = oirgetdata(oldmask)
det = obj_new('imagedetector',oldmask)&ndata = det->getpar('NREGION')
dettable=det->readrows()
corner=dettable.corner
det->close
obj_destroy,det
maskdata.data1[*] = 0.
maskdata.data2[*] = 0.
a = fix([x1,y1]+.5) - corner(*,0)
print,a
maskdata.data1[a[0]-dx:a[0]+dx, a[1]-dx:a[1]+dx] = 1.
a = fix([x2,y2]+.5) - corner(*,1)
print,a
maskdata.data2[a[0]-dx:a[0]+dx, a[1]-dx:a[1]+dx] = 1.
oirNewData, oldmask, newmask, maskdata
RETURN
END

FUNCTION midirms,a
z=mean(a)
RETURN,sqrt(mean((a-z)^2))
END

FUNCTION midiairy, x
   ax = ABS(x)
   out = 2. * BESELJ(ax,1)
   wx = WHERE(ax EQ 0.)
   IF (MAX(wx) LT 0) then RETURN, out/ax
   ax[wx] = 1.
   out = out/ax
   out[wx] = 1.
   RETURN,out
END

FUNCTION dcombine, data, dx, dy, rel2
if (N_PARAMS() LT 4) then rel2 = 1.
r1 = float(data.data1)+32768.
r2 = float(data.data2)+32768.
nd = n_elements(data)
t1 = total(r1,3)/nd
t2 = total(r2,3)/nd
for i=0, nd-1 do r1(*,*,i)=r1(*,*,i)-t1
print,'finished r1'
for i=0, nd-1 do r2(*,*,i)=r2(*,*,i)-t2
print,'finished r2'
for i=0, nd-1 do r1(*,*,i)=r1(*,*,i)-rel2*shift(r2(*,*,i), dx, dy)
RETURN,r1
END

FUNCTION cubesmooth, data, dx
s=size(data)
output=data
for i=0,s[3]-1 do begin
   z = reform(data(*,*,i))
   zz = ffsmooth(z, dx)
   output(*,*,i)= zz
ENDfor
RETURN,output
END

FUNCTION cubesharp, data, dx
s=size(data)
output=data
for i=0,s[1]-1 do for j=0,s[2]-1 do output(i,j,*)= $
   data(i,j,*)-ffsmooth1(reform(data(i,j,*)),dx)
RETURN,output
END

FUNCTION filterpower, opd, fringe, scansize, fftobj, power
nscan = n_elements(opd)/scansize
lopd = reform(opd, scansize, nscan)
wt = replicate(1., scansize)
lfringe=reform(fringe,scansize,nscan)
fopd = fftobj->coord()
output=fltarr(n_elements(fopd), nscan)
power = output
for i=0, nscan-1 do begin
   output(*,i)=fftobj->filter(lopd(*,i), lfringe(*,i),  wt, fftpower)
   power(*,i) = abs(fftpower)^2
endfor
if (total(finite(output) eq 0) gt 0)then power(where(finite(power) eq 0)) = 0
RETURN,output
END

FUNCTION cuberms,data
s=size(data)
mdata = total(data,3)/s[3]
rdata=mdata
for i=0,s[3]-1 do rdata=rdata+(data(*,*,i)-mdata)^2
RETURN, sqrt(rdata/s[3])
END

FUNCTION makefftobject, minopd, maxopd, npt, filtername
   ff = obj_new('fft1d', [minopd, (maxopd-minopd)/npt, npt])
   kk=ff->kgrid()
   filter=obj_new('filterdata', filtername)
   fdata=filter->interpolate(kk)
   fdata=fdata+reverse(fdata)
   ff->set,filter=fdata
   obj_destroy,filter
   RETURN,ff
END

FUNCTION makefftobject2, minopd, maxopd, npt, filtername
   ff = obj_new('fft1d', [minopd, (maxopd-minopd)/npt, npt])
   kk=ff->kgrid()
   filter=obj_new('filterdata', filtername)
   fdata=filter->interpolate(kk)
   fdata=fdata
   ff->set,filter=fdata
   obj_destroy,filter
RETURN,ff
END

FUNCTION crosscube, data
   r1 = float(data.data1)
   r2 = float(data.data2)
   nd = n_elements(data)
   t1 = total(r1,3)/nd
   t2 = total(r2,3)/nd
   for i=0, nd-1 do r1(*,*,i)=r1(*,*,i)-t1
   print,'finished r1'
   for i=0, nd-1 do r2(*,*,i)=r2(*,*,i)-t2
   print,'finished r2'
   cc=fltarr(7,7)
   for i=-3,3 do for j=-3,3 do cc(i+3,j+3)=total(r1*shift(r2,i,j,0))
   cc=cc/max(cc)
   print,0,-3+indgen(7),format='(i2,2x,7f7.3)'
   for j=-3,3 do print,j,cc(*,j+3),format='(i2,2x,7f7.3)'
RETURN,cc
END

FUNCTION midiExtractFiles, multiFile
;   breakup a concatenated list of input files
   RETURN,STRSPLIT(multiFile,' ',/extract)
END

FUNCTION midiFirstFile, multiFile
;   RETURN first file from a concatened list
   f = midiExtractFiles(multiFile)
RETURN,f[0]
END

FUNCTION midiConcatFiles, fileList
;   concatenate a bunch of files, separated by a single space
   out = STRTRIM(fileList[0])
   for i=1,N_ELEMENTS(fileList)-1 do out=out+' '+STRTRIM(fileList[i],2)
RETURN,out
END

FUNCTION midiGui, fileObj, dir=dir, search=search, save=save, restore=restore,$
   recursive=recursive,fdir=fdir,newkeys=newkeys
;

   IF(NOT KEYWORD_SET(search)) then search = 'MIDI*.fits'
   if (NOT KEYWORD_SET(dir)) then dir=''
   if (NOT KEYWORD_SET(restore)) then begin
      if (dir EQ '' ) then begin
         cd,current=c
         print,'Reading files in current directory '+c
      endif else print,'Reading files in directory '+dir
      guikey=['OBS TARG','NRTS MODE','INS GRIS','INS OPT1','INS FILT NAME',$
         'INS SHUT']
      if (KEYWORD_SET(newkeys)) then guikey=[guikey,newkeys]
      f = OBJ_NEW('fitsfilelist', dir=dir, $
         guikey=guikey,guisearch=search,/midiCompress, $
         recursive=recursive)
   endif else begin
      saveFile = FINDFILE(restore)
      if (saveFile NE '') then begin
         f = OBJ_NEW('fitsfilelist',restore=saveFile)
         f->guiSelect
      endif else begin
         print,' saveFile '+restore+' not Found'
         f = OBJ_NEW()
      endelse
   endelse
   if (OBJ_VALID(f)) then list =f->files() else begin
      print,'Failed to create legitimate file list from input data '
      RETURN, ''
   endelse
   if (KEYWORD_SET(save)) then begin
      f->save, save,/select
   endif
   OBJ_DESTROY,f
RETURN,list
END

FUNCTION midiGuiS, dir=dir, recursive=recursive, clear=clear
;
   search='MIDI*.fits'
   IF (NOT KEYWORD_SET(dir)) then dir='' 
   if (dir EQ '' ) then begin
      cd,current=c
      print,'Reading files in current directory '+c
      saveFileName = 'midiGui.SAV'
   endif else begin
      print,'Reading files in directory '+dir
      saveFileName = dir+'/midiGui.SAV'
   endelse
   if (KEYWORD_SET(clear)) then begin
      OPENR,lun, saveFileName,/DELETE,/GET_LUN, ERROR=ierr
      if (ierr NE 0) then print, !ERROR_STATE.MSG
      if(ierr EQ 0) then  FREE_lun, lun
   endif
   saveFile = FINDFILE(saveFileName)
   if (saveFile NE '') then begin
      f = OBJ_NEW('fitsfilelist',restore=saveFile)
      f->guiSelect
   endif else begin
      f = obj_new('fitsfilelist',dir=dir,$
         guikey=['OBS TARG','NRTS MODE','INS GRIS','INS OPT1','INS FILT NAME',$
         'INS SHUT','CHOP FREQ'],guisearch=search,/midiCompress,$
         recursive=recursive)
   endelse
   if (OBJ_VALID(f)) then list =f->files() else begin
      print,'Failed to create legitimate file list from input data '
      list = ''
      RETURN,list
   endelse
   if (saveFile EQ '') then f->save, saveFileName
   OBJ_DESTROY,f
RETURN,list
END

FUNCTION midiFiles, last, dir=dir
;
   IF (NOT KEYWORD_SET(dir)) then dir='.'
   f = obj_new('fitsfilelist',dir=dir,$
      search='MIDI*.fits',/midiCompress)
   list =f->files()
   last = list[N_ELEMENTS(list) -1]
   obj_destroy,f
RETURN,list
END

pro olines,lines,labels,ylab,th
; on top of existing plot make vertical stripes at
; the positions of given spectral lines
; shift each wavelength before plotting by z
; default line thickness
   if (n_params() eq 4) then t=th else t = 1
;  loop over list of lines
   if (n_params() eq 2) then ylab = .8*!y.crange(1)
   xr=!x.crange
   if (!x.type eq 1) then xr=10.^xr
   n=n_elements(lines)-1
   for i=0,n do begin
      zl = lines[i]
;  only over plot lines that fall within wavelength range of plot
;  the x-coordinate could be logarithmic
      if (zl gt xr(0)) and (zl lt xr(1)) then begin
         if (t gt 0) then oplot,[zl,zl],!y.crange,thick=t
;  if a list of line names have been given, write these on plot
      if n_params() gt 2 then xyouts,zl,ylab,labels(i),orient=90
      endif
   endfor
RETURN
end

FUNCTION pex,d,x
;  if d is list of polynomial coefficients, expand
;  polynomial at each value of vector x and RETURN vector of values
p = n_elements(d)
w = d(p-1)
for i=p-2,0,-1 do w=w*x+d(i)
RETURN,w
end

FUNCTION pex2,d,x,y
p = size(d)
cox=fltarr(p(1))
for i=0,p(1) -1 do cox(i) = pex(d(i,*),y)
RETURN, pex(cox, x)
end

FUNCTION medcont, data
; RETURN median of a vertical stack of images
ss=size(data)
out=fltarr(ss(1),ss(2))
for i=0,ss(1)-1 do for j=0,ss(2)-1 do out(i,j)=median(data(i,j,*))
RETURN,out
end

FUNCTION ps1,a,dx
;shift an 1-d array a by dx (fractional) pixels
pi2 = 2.*!pi
f = fft(a,1)
ss = size(a)
nx = ss(1)
sx = exp(complex (0, pi2 * dx * (indgen(nx) - nx/2)/nx))
sx = shift (sx, -nx/2)
f=f*sx
RETURN, float(fft(f,-1))
END

FUNCTION bblam,temp,wave
;  RETURN normalized bb curve in energy/area/sec/wavelength unit
;  given wavelengths in microns, temp in K and n is normalizing point
z = wave^(-5)/(exp(14350./(temp*wave))-1.)
n=fix(.5*n_elements(wave))
RETURN, z/z(n)
end

FUNCTION bbJy,temp,wave
;  RETURN normalized bb curve in energy/area/sec/Hz unit
;  given wavelengths in microns, temp in K and n is normalizing point
z = wave^(-3)/(exp(14350./(temp*wave))-1.)
n=fix(.5*n_elements(wave))
RETURN, z/z(n)
end

FUNCTION pk,x,y,z
; using quadratic approximation of y(x) RETURN
; position (x value) of peak of y to subpixel accuracy
; optionally RETURN interpolated y value in z
   q=max(y,j)
   if (j eq 0) then j = 1
   if (j eq n_elements(x)-1) then j=n_elements(x)-2
   dj=(y(j-1)-y(j+1))*.5/(y(j+1)+y(j-1)-2.*y(j))
   z = y(j)+.25*dj*(y(j+1)-y(j-1))
RETURN,x(j)+.5*dj*(x(j+1)-x(j-1))
end

FUNCTION pshift,a,dx,dy
;two dimensional partial pixel shift
pi2 = 2.*!pi
f = fft(a,1)
ss = size(a)
nx = ss(1)
ny = ss(2)
sx = exp(complex (0, pi2 * dx * (indgen(nx) - nx/2)/nx))
sx = shift (sx, -nx/2)
for i = 0, ny - 1 do f(*,i) = f(*,i) * sx
sy = exp(complex (0, pi2 * dy * (indgen(ny)-ny/2)/ny))
sy = shift (sy, -ny/2)
for i = 0, nx - 1 do f(i,*) = f(i,*) * sy
RETURN, float(fft(f,-1))
END
FUNCTION cross,a,b
RETURN,float(fft(fft(a,1)*fft(b,-1),-1))
end
FUNCTION makex,nx,ny
mask = intarr(nx, ny)
x = indgen(nx)
for i=0,ny-1 do mask(*, i) = x
RETURN,mask
end
FUNCTION makey,nx,ny
mask = intarr(nx, ny)
x = indgen(ny)
for i=0,nx-1 do mask(i, *) = x
RETURN,mask
end


FUNCTION gau,nx,ny,sx,sy
g = (shift((makex(nx,ny)-.5*nx)/sx,nx/2,0))^2
g = .5*(g + (shift((makey(nx,ny)-.5*ny)/sy,0,ny/2))^2)
if (total(g gt 30.) gt 0) then g(where(g gt 30.)) = 30.
RETURN ,exp(-g)
end

FUNCTION gau1,nx,sx
if (sx lt .001) then sx = .001
g = .5*(shift((indgen(nx)-.5*nx)/sx,nx/2))^2
if (total(g gt 30.) gt 0) then g(where(g gt 30.)) = 30.
RETURN ,exp(-g)
end

FUNCTION fgau,nx,ny,sx,sy
f=fft(gau(nx,ny,sx,sy),1)
RETURN,f/f(0,0)
end
FUNCTION fgau1,nx,sx
f=fft(gau1(nx,sx),1)
RETURN,f/f(0)
end

FUNCTION fsmooth,image,s
; two dimensional fft smoothing of image
ss=size(image)
g=fgau(ss(1),ss(2),s,s)
RETURN,float(fft(g*fft(image,1),-1))
END

FUNCTION fsmooth2,image,sx, sy
; two dimensional fft smoothing of image with different constants in x and y
ss=size(image)
g=fgau(ss(1),ss(2),sx,sy)
RETURN,float(fft(g*fft(image,1),-1))
END

FUNCTION fsmooth1,image,s
ss=size(image)
g=fgau1(ss(1),s)
RETURN,float(fft(g*fft(image,1),-1))
END

FUNCTION csmooth1,image,s
ss=size(image)
g=fgau1(ss(1),s)
RETURN,fft(g*fft(image,1),-1)
end

FUNCTION csmooth2,image,s
ss=size(image)
g=fgau1(ss(1),s)
output = image
for i=0,ss[2]-1 do output(*,i) = fft(g*fft(image(*,i),1),-1)
RETURN, output
END


FUNCTION fsmooth2,image,s1,s2
ss=size(image)
g=fgau(ss(1),ss(2),s1,s2)
RETURN,float(fft(g*fft(image,1),-1))
end


PRO midiGetINS, file, dit=dit, grism=grism, slit=slit, opt1=opt1, $
   camera=camera, filter=filter, nrts=nrts, object=object, shutter=shutter, $
   ndit = ndit, ierr = ierr
;
   f = file[0]
   f = STRSPLIT(f, ' ',/EXTRACT)
   f = f[0]
   ierr = 0
   fits = OBJ_NEW('fitsfile', f, ierr=ierr)
   if (ierr NE 0) then begin
      midiPrintErrMsg
      RETURN
   endif
   ph   = fits->head()
   dit  = ph->getpar('DET DIT')
   ndit  = ph->getpar('DET NDIT')
   grism = ph->getpar('INS GRIS NAME')
   slit = ph->getpar('INS SLIT NAME')
   opt1 = ph->getpar('INS OPT1 NAME')
   camera = ph->getpar('INS CAM NAME')
   filter = ph->getpar('INS FILT NAME')
   shutter = ph->getpar('INS SHUT NAME')
   nrts = ph->getpar('DET NRTS MODE')
   ob   = ph->getpar('TARG NAME',nob)
   if (nob GT 0) then object=ob else object=''
   OBJ_DESTROY, ph
   OBJ_DESTROY, fits
RETURN
END

FUNCTION midiPhase, x
   RETURN,ATAN(IMAGINARY(x), FLOAT(x))
END
FUNCTION midiPhased,x
   RETURN,!radeg*midiPhase(x)
END

FUNCTION midiCvis, vis
;    convert vis data in a standard oirVis structure into
;    a complex vector
RETURN, COMPLEX(vis.visamp * EXP(COMPLEX(0.,1.) * vis.visphi/!radeg))
END

PRO midiDelayPlot, base, title,twopass=twopass
   delay = oirgetdelay(base+'.groupdelay.fits')
   delaytime=REFORM(delay.time)&delaytime=86400*(delaytime-delaytime[0])
   opd = oirgetopd(base+'.fringes.fits',opdtime)
   delay = reform(delay.delay)
   delay = INTERPOL(delay, delaytime, opdtime)
   if (TOTAL(opd EQ 0.) GT 0 ) then $
      delay(where(opd EQ 0.0)) = median(reform(delay)); bad points
   if (KEYWORD_SET(twopass)) then begin
      delay1 = oirgetdelay(base+'.groupdelay1.fits')
      delay1 = REFORM(delay1.delay)
      delay1 = INTERPOL(delay1, delaytime, opdtime)
      delay2 = oirgetdelay(base+'.powerdelay.fits')
      delay2 = REFORM(delay2.delay)
      if (TOTAL(opd EQ 0.) GT 0 ) then begin
         delay1(where(opd EQ 0.0)) = median(delay1); bad points
         delay2(where(opd EQ 0.0)) = median(delay2); bad points
      endif
   endif
   if (TOTAL(opd EQ 0.) GT 0 ) then $
      opd(where(opd EQ 0.0)) = median(opd)  ; bad points
   ndelay=N_ELEMENTS(opd)
   if (ndelay gt 4000) then $
   plot,opdtime, opd,yrange=median(opd[ndelay-3000:*])+$
      6*midirms(opd[ndelay-3000:*])*[-1,1],title=base+':'+title, $
      xtitle='time (sec)',ytitle='Delay(microns)' $
   else plot,opdtime, opd,yrange=[min([opd,delay]),max([opd,delay])],title=base+':'+title, $
      xtitle='time (sec)',ytitle='Delay(microns)'
   if (KEYWORD_SET(twopass)) then begin
      oplot,opdtime,delay1, color = 255
      oplot,opdtime,delay2, color = 255l*256l
   endif
   oplot,opdtime,delay,thick=2
RETURN
END


PRO midiDelayAmpPlot, base, title, gsmooth=gsmooth, time=time
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth=3
   delay = oirgetdelay(base+'.groupdelay.fits')
   if (NOT KEYWORD_SET(time)) then $
      amp   = MEDIAN(REFORM(delay.amplitude), gsmooth) $
   else amp = REFORM(delay.amplitude)
   delay = REFORM(delay.delay)
   aSort = sort(amp[10:*])
   nSort = N_ELEMENTS(aSort)
   aMed = amp[10+aSort[.5*nSort]]
   aMax = amp[10+aSort[nSort-1]]
   aSig = amp[10+aSort[.8413*nSort]]
   if (NOT KEYWORD_SET(time)) then begin
      opd = oirgetdata(base+'.fringes.fits',col=['opd','localopd'])
      opd=1.e6*reform((opd.opd[1]+opd.localopd[1]-opd.opd[0]-opd.localopd[0]))
      if (TOTAL(opd EQ 0.) GT 0 ) then begin
         delay(where(opd EQ 0.0)) = median(delay); bad points
      endif
   endif
   nD = N_ELEMENTS(delay)
   if (KEYWORD_SET(time)) then n0 = 80 else n0 = 300
   plot,delay[n0:nD-2], abs(amp[n0:nD-2]), title=base+':'+title, $
      xtitle='delay (micron)',ytitle='Amplitude' ,/ylog, psym=7,charsize=2,ystyle=1
   oplot,1.e9*[-1,1],aMed*[1,1],thick=2
   for i=1,4 do oplot,1.e9*[-1,1],aMed+(aSig-aMed)*i*[1,1],line=2
   xyouts,delay[10+aSort[nSort-1]], .95*aMax, $
      STRING((aMax-aMed)/(aSig-aMed),format='(f4.1)')
RETURN
END


PRO midiDelayAmpPlot1, base, title, gsmooth=gsmooth, time=time
   if (NOT KEYWORD_SET(gsmooth)) then $
      if (KEYWORD_SET(time)) then gsmooth = 3 $
      else gsmooth = 30

   delay = oirgetdelay(base+'.groupdelay.fits')
   dTime = REFORM(delay.time)
   delay = REFORM(delay.delay)
;        in time version delays are not calculated
;        for each frame, so interpolate them
   if (KEYWORD_SET(time)) then begin
      gTime = oirgetdata(base+'.groupdelay.fits',col='time')
      gTime = REFORM(gtime.time)
;         
      delay = INTERPOL(delay, dTime, gTime,/spline)
   endif
   nData = N_ELEMENTS(delay)
   n0    = 300
   dn    = 500

   nd    = floor(nData-n0)/dn
   nd    = dn*nd
   delay = delay[n0:n0+nd-1]
   delay = median(delay,2*gsmooth)
   gdata = oirgetdata(base+'.groupdelay.fits')
   gdata = pseudoComplex(gdata.data1)
   gdata = gdata[n0:n0+nd-1,*]
   gdata = csmooth2(gdata,gsmooth)
   gamp = fltarr(nd)
   for i=0, nd-1 do gamp[i] = MAX(ABS(gdata[i,*]))
   erase
   !p.multi = [0,1,3,0,0]
   oldx = !x.style
   !x.style=1
   plot,delay ,charsize=2
   plot,gamp,charsize=2
   midiDelayAmpPlot, base,' ', gsmooth=gsmooth,time=time
   !x.style=oldx
   !p.multi = 0
   pkDelay = pk(delay[50:nd-50], gamp[50:nd-50], gpk)
   ramp = midirms(gamp)
   rmean= mean(gamp)
   rmed = median(gamp)
;  print, long(gpk-rmean),long(ramp),(gpk-rmean)/ramp,long(rmean),pkDelay,$
   print, long(gpk-rmed),long(ramp),(gpk-rmed)/ramp,long(rmed),pkDelay,$
   format='("Max amplitude of ",i7,"+/-",i6,"(",f3.1," sig) above",i8," at delay ",f8.1)'
RETURN
END

PRO midiCalJy, inFile, database, outFile
;  create a fdata output containing flux extracted from database
;  this all assumes that the source in inFile can be found in
;  database
;   inputs:
;      inFile   midi data file used to extract header, source name & wavelengths
;      database database file containing spectrophotometry
;      outFile  name of output file in oir imagedata format
;               if not specified try to construct a name from input file
;
;   construct outfile if not given
   myFile = midiFirstFile(inFile)
   sourceData=database->sourceFromFile(myFile, specData=specData, $
      ierr=ierr)
   if (ierr NE 0) then begin
      print,'    spectrumFromFile source not found in file '+fitsFile
      RETURN
   endif
   specData = *(specData[0])
   inwave = 1.e6*specData.wavelength
   influx   = specData.flux
   if (N_PARAMS() LT 3) then begin
      in   = STRSPLIT(myFile,'.',/extract)
      nin  = N_ELEMENTS(in)
      outFile = in[nin-3]+'.calJy.fits'
   endif
   print,'   using '+outFile+' as outFile '
   detector = OBJ_NEW('IMAGEDETECTOR', myFile)
   detector->close
   detTable = detector->table()
   detHead  = detector->priHead()
;     set naxis2 to 1
   (*detTable).naxis[1] = 1
   nRegion = N_ELEMENTS(*detTable)
   data = detector->dataTemplate()
   tempTags = STRTRIM(TAG_NAMES(data),2)
   tempData = WHERE(STRPOS(tempTags, 'DATA') GE 0)
   for i=0,nRegion-1 do data.(tempData[i])=INTERPOL(influx, $
      inwave, oirGetWavelength(myFile,1+i),/spline)
   image = OBJ_NEW('imagedata',detector,/float)
   detector->newFile, outFile, priHead=detHead,ierr=ierr
   image->appendToFile,detector,ierr=ierr
   image->writerows,data,ierr=ierr
   image->close
   OBJ_DESTROY,image
   OBJ_DESTROY,detector
   OBJ_DESTROY,detHead
RETURN
END

PRO midiInsVisPlot, base,title
   vis = oirGetVis(base+'.redcal.fits', wave=wave)
   plot,wave,vis.visamp,xr=[7.5,13.5],xstyle=1,title=base+':'+title, $
      yr=[0>MIN(vis.visamp), 1.5<MAX(vis.visamp[WHERE(wave gt 8.1 and (wave lt 13.5))])], $
      xtitle='Wavelength(microns)', ytitle='Visibility'
;  errplot,wave,vis.visamp-vis.visamperr,vis.visamp+vis.visamperr
RETURN
END

PRO midiPlotOverlap, base, title, mask=mask, oldmask=oldmask, fringe=fringe
   if (N_PARAMS() LT 2) then title='OVERLAP'
   pa = oirGetData(base+'.Aphotometry.fits')
   pb = oirGetData(base+'.Bphotometry.fits')
   if (KEYWORD_SET(fringe)) then begin
      pi = oirGetData(base+'.fringeimageave.fits')
      pi = TRANSPOSE(pseudocomplex(pi.data1))
      si = total(pi[50:100,*],1)
      s = abs(si)
   endif
   midiGetIns,base+'.Aphotometry.fits', opt1=opt1
   if (STRTRIM(opt1,2) eq "HIGH_SENS") then begin 
      sa = total(pa[0].data1[50:100,*],1)
      sb = total(pb[0].data1[50:100,*],1)
   endif else if (STRTRIM(opt1,2) eq "SCI_PHOT") then begin
      sa = total(pa[0].data2[50:100,*],1)
      sb = total(pb[0].data2[50:100,*],1)
   endif else print,'OPT1 of '+opt1+' not valid for plotoverlap'
   ns = N_ELEMENTS(sa)
   sa=sa/max(sa[5:ns-5])
   sb=sb/max(sb[5:ns-5])
   plot,sa,title=title,yr=[-.2,1.2],ystyle=1
   oplot,sb,line=2
   if (KEYWORD_SET(fringe)) then oplot,s/max(s[5:ns-5]),thick=2
   if (KEYWORD_SET(mask)) then begin
      m = oirGetData(mask)
      mm=total(m[0].data1[50:100,*],1)
      mm = mm/max(mm)
      oplot,mm,thick=3
   endif
   if (KEYWORD_SET(oldmask)) then begin
      m = oirGetData(oldmask)
      mm=total(m[0].data1[50:100,*],1)
      mm = mm/max(mm)
      oplot,mm,thick=3, line=2
   endif
RETURN
END
;          Tristram code; suitably Jaffeized

;+
; NAME:
;       MOD_STRUCT
;
; PURPOSE:
;       Modify an existing IDL structure.
;
; CATEGORY:
;       MIDI data tools.
;
; CALLING SEQUENCE:
;       newstr = mod_struct(oldstr, tagnam, tagval [, positi=0], [, /delete])
;
; REQUIRED INPUTS:
;       oldstr      the old structure to be modified
;       tagnam      name of the tag to be modified or added
;
; OPTIONAL INPUTS:
;       tagval      value of the tag to be modified or added
;
; KEYWORDS:
;       positi      position of the new or modified tag in the new struct
;       delete      delete the tag instead of adding or modifying it
;
; OUTPUTS:
;       newstr      a new structure with modified tags
;
; DESCRIPTION AND EXAMPLE:
;       This function modifies the IDL structure provided in OLDSTR by adding 
;       the new structure tag with the name TAGNAM and the value TAGVAL at 
;       the position specified by POSITI. If the tag already exists, then 
;       the old tag is overwritten, i.e. this function can be used to modify 
;       the data type of a tag or to move it within the structure. If no 
;       position is specified, then the tag is added at the end of the 
;       structure. For POSITI < 0 the tag is removed from the struct if 
;       previously present. The same result is obtained when the keyword 
;       DELETE is set. In this case TAGVAL does not have to be specified.
;
; CALLED BY:
;       midiMskfit
;
; CALLING:
;       none
;
; MODIFICATION HISTORY:
;       2010-11-02 Written by Konrad R. Tristram
;
FUNCTION MOD_STRUCT, OLDSTR, TAGNAM, TAGVAL, POSITI=POSITI, DELETE=DELETE

if (size(oldstr, /type) ne 8) or (size(tagnam, /type) ne 7) then begin
	message, 'Error in the parameters provided.', /continue
	message, 'Usage: newstr = mod_struct(oldstr, tagnam, tagval ' + $
	         '[, positi=0], [, /delete])', /continue
	message, '    oldstr - structure to be modified', /continue
	message, '    tagnam - name of the tag to be added', /continue
	message, '    help, tagval - value of the tag to be added', /continue
	message, '    positi - position of the new tag', /continue
	message, '    delete - delete the tag instead of adding it', /continue
	RETURN, -1
endif

; INITIALISE THE COMMAND STRING AND THE SEPARATOR STRING
;-------------------------------------------------------------------------------
comman = 'newstr = create_struct('
sepstr = ''

; GET THE OLD TAG NAMES AND THE LOCATION OF THE SPECIFIED TAG
;-------------------------------------------------------------------------------
oldtag = tag_names(oldstr)
oldpos = where(oldtag eq strupcase(tagnam))

; IF THE TAG IS NEW THEN ADD IT AT THE END (UNLESS POSITI IS DEFINED)
;-------------------------------------------------------------------------------
if oldpos eq -1 then oldpos = n_tags(oldstr)

; SET THE NEW POSITION OF THE SPECIFIED TAG
;-------------------------------------------------------------------------------
if n_elements(positi) eq 0 then newpos = oldpos else newpos = positi
if keyword_set(delete) then newpos = -1

; LOOP OVER THE TAGS IN THE STRUCTURE
;-------------------------------------------------------------------------------
for i=0,n_tags(oldstr)-1 do begin
	; IF AT THE CORRECT POSITION, THEN INSERT THE NEW TAG
	;-----------------------------------------------------------------------
	if i eq newpos then begin
		comman += sepstr+'tagnam, tagval'
		sepstr = ', '
	endif
	; ADD ALL OTHER TAGS TO THE COMMAND
	;-----------------------------------------------------------------------
	if i ne oldpos then begin
		comman += sepstr+'oldtag[' +strtrim(i, 2)+'], ' + $
	                         'oldstr.('+strtrim(i, 2)+')'
		sepstr = ', '
	endif
endfor

; IF THE NEW TAG IS SUPPOSED TO BE AT THE END OF THE STRUCT THEN ADD IT NOW
;-------------------------------------------------------------------------------
if newpos ge i then comman += sepstr+'tagnam, tagval'

; FINISH THE COMMAND STRING AND EXECUTE THE COMMAND
;-------------------------------------------------------------------------------
comman += ')'
tmpvar = execute(comman)

; RETURN THE RESULT
;-------------------------------------------------------------------------------
RETURN, newstr

END
;+
; NAME:
;       MIDIGAUDIS
;
; PURPOSE:
;       Fit a Gaussian to every channel in a dispersed MIDI window.
;
; CATEGORY:
;       MIDI data tools.
;
; CALLING SEQUENCE:
;       result =  midiGaudis(detwin [, rrange=rrange] [, smtval=smtval]
;
; REQUIRED INPUTS:
;       detwin      2D array corresponding to a MIDI detector window
;
; OPTIONAL INPUTS:
;       None.
;
; KEYWORDS:
;       rrange      the range of rows used for the fitting
;       smtval      two element array with the smoothing values in y and y
;
; OUTPUTS:
;       result      struct containg the resulting fit parameters and the chi^2
;
; DESCRIPTION AND EXAMPLE:
;       This function fits a 1D Gaussian distribution to every column in a
;       dispersed MIDI window (e.g. to photometries for PRISM or for GRISM
;       data) using the function GAUSSFIT implemented in IDL. For the fit
;       only the rows specified by the parameter RRANGE are used. If the
;       keyword /OUTPUT is set then a comparison between the data and the
;       fits will be also plotted. The resulting parameters as a function of
;       column (the scaling factor for the Gaussian, the position of the peak
;       and the FWHM) and their errors as well as the chi^2 are RETURNed in
;       an array of struct. The function is called by:
;       result =  midiGaudis(detwin)
;
; CALLED BY:
;       midiMskfit
;
; CALLING:
;       none
;
; MODIFICATION HISTORY:
;       2008-10-13 Written by Konrad R. Tristram
;       2008-11-19 K. Tristram: Added the possibility to smooth DETWIN.
;       2009-04-26 K. Tristram: Guess the row where the spectrum is located.
;       2010-11-02 K. Tristram: Changed array of struct to struct off arrays.
;

FUNCTION MIDIGAUDIS, DETWIN, RRANGE=RRANGE, SMTVAL=SMTVAL

; INITIALISE A FEW VARIABLES
;-------------------------------------------------------------------------------
   if n_elements(rrange) ne 2 then rrange = [7, 26]
   y_vals = findgen(rrange[1]-rrange[0]+1) + rrange[0]
   winsiz = size(detwin)
   nx     = winsiz[1]

;-------------------------------------------------------------------------------
; CREATE THE STRUCT CONTAINING THE FIT PARAMETERS
;-------------------------------------------------------------------------------
   tmpvar = fltarr(nx)
;  params = {factor:tmpvar, factorerr:tmpvar, positi:tmpvar, positierr:tmpvar, $
;         fwhmax:tmpvar, fwhmaxerr:tmpvar, chisqa:tmpvar, $
;         errflg:bytarr(nx)}
   params = {factor:0.0, factorerr:0.0, positi:0.0, positierr:0.0, $
          fwhmax:0.0, fwhmaxerr:0.0, chisqa:0.0,  errflg:0B}
   params = REPLICATE(params, nx)

; OPTIONALLY SMOOTH THE DATA
;-------------------------------------------------------------------------------
   if n_elements(smtval) eq 2 then detwin = smooth(detwin, smtval, /edge)

; GUESS THE ROW WHERE THE SPECTRUM IS LOCATED
;-------------------------------------------------------------------------------
; This might go wrong if the spectrum is very curved, e.g. for GRISM!!
   xx     = findgen(winsiz[2])
   dd     = total(detwin,1)
   maxpos = pk(xx[rrange[0]:rrange[1]], dd[rrange[0]:rrange[1]])
;  tmpvar = max(total(detwin[*,rrange[0]:rrange[1]], 1), maxpos)

; LOOP OVER ALL COLUMNS I.E. WAVELENGTH BINS
;-------------------------------------------------------------------------------
   for i=0,nx-1 do begin
; FIND THE MAXIMUM VALUE OF THE COLUMN (HOPING IT IS THE SPECTRUM)
;-----------------------------------------------------------------------
      maxcol = max(detwin[i,rrange[0]:rrange[1]])
; FIT A GAUSSIAN TO THE COLUMN
;-----------------------------------------------------------------------
      tmpvar = gaussfit(y_vals, detwin[i,rrange[0]:rrange[1]], partmp, $
;        estimates = [maxcol, maxpos+rrange[0], 2, 0, 0], $
         estimates = [maxcol, maxpos, 2, 0, 0], $
         chisq=chisqa, nterms=5, sigma=errtmp)
;-----------------------------------------------------------------------
; COPY THE PARAMETERS INTO THE RESULT STRUCT
;-----------------------------------------------------------------------
      params[i].factor    = partmp[0]
      params[i].factorerr = errtmp[0]
      params[i].positi    = partmp[1]
      params[i].positierr = errtmp[1]
      params[i].fwhmax    = abs(partmp[2]) * 2 * sqrt(2*alog(2))
      params[i].fwhmaxerr = errtmp[2] * 2 * sqrt(2*alog(2))
      params[i].chisqa    = chisqa
; TRY TO FIND BAD FITS AND FLAG THE DATA ACCORDINGLY
;-----------------------------------------------------------------------
      badfit = ((partmp[0] lt 0.0) or (params[i].fwhmax lt 1.0) or $
         (partmp[1] lt rrange[0]+2) or (partmp[1] gt rrange[1]-2))
      if badfit then params[i].errflg = 1B
   endfor

; RETURN THE RESULT STRUCT
;-------------------------------------------------------------------------------
RETURN, params
END


;+
; NAME:
;       MIDIMSKFIT
;
; PURPOSE:
;       Create a MIDI mask from image detector files.
;
; CATEGORY:
;       MIDI data tools.
;
; CALLING SEQUENCE:
;       mskdat =  midiMskfit(infile [, msknam] [, factor=factor]
;                             [, additi=additi] [, smtval=smtval] 
;                             [, psfile=psfile] [, skymsk=skymsk]
;                             [, skydis=skydis] [, skywid=skywid])
;
; REQUIRED INPUTS:
;       infile      input FITS file with image detector data 
;
; OPTIONAL INPUTS:
;       msknam      name of the mask file for saving the resulting mask
;
; KEYWORDS:
;       factor      a factor for enlarging the FWHM
;       additi      a constant for enlarging the FWHM
;       smtval      two element array with the smoothing values in x and y
;       skymsk      struct containing the sky mask
;       skydis      distance of the sky band from the trace
;       skywid      width of the sky band in pixels
;       linpar      parameter for linear weighting in wavelength direction
;
; OUTPUTS:
;       mskdat      struct containing the mask of the spectrum & its parameters
;
; DESCRIPTION AND EXAMPLE:
;	This function takes either a name of an imaging detector FITS file
;	or an IDL structure with imaging detector data as an input for the
;	creation of a MIDI mask.  If the data is complex (e.g. a fringeimage from
;	oirAverageVisImage) then only the real part is considered, The input can
;	have an arbitrary number of detector windows (e.g. for 2 for HIGH_SENSE
;	or 4 for SCI_PHOT data). Multiple frames in the input are interpreted
;	as different beams (e.g. the two beams for A and B photometry). Then
;	the trace (i.e. the peak position as a function of column/wavelength)
;	and the FWHM of the spectrum is determined for each of the two windows
;	by fitting a 1D Gaussian distribution to each column of the detector
;	window. This is done for every beam seperately as well as for the average
;	of all beams. The trace and the FWHM are then parametrised as a function
;	of column/wavelength by a polynomial. The final trace of the mask is
;	defined as the average of the traces of the individual beams. The FWHM of
;	the mask is defined as the maximum of the FWHMaxima of the different beams
;	multiplied by FACTOR, plus half of the distance between the traces plus
;	ADDITI. The mask is then calculated as a modified Gaussian distribution
;	z=exp( -0.5 * abs((y - trace) / FWHM*sqrt(2*alog(2)))^EXPONE ) in order
;	to obtain a more bell shaped profile. The result RETURNed by this function
;	is a struct containg the mask for the two detector windows as well as the
;	parameters used for the creation of the mask. Additionally a sky mask is
;	calculated. The two sky bands are located symmetrically below and above
;	the trace at a distance of SKYDIS * FWHM and the width of the sky bands
;	is governed by the paramter SKYWID. If the keyword OUTPUT is set then a
;	plot with the traces and the FWHM as well as comparisons between the final
;	masks and the data are created. The masks can be written to FITS files by
;	specifying a base for the filenames in MSKNAM. The function is called by:
;       result = midiMskfit(phtfil)
;
; CALLED BY:
;       none
;
; CALLING:
;       midiGaudis
;
; MODIFICATION HISTORY:
;       2008-10-13 Written by Konrad R. Tristram
;       2008-11-19 K. Tristram: Added the possibility to smooth the windows.
;       2009-04-15 K. Tristram: Corrected error in handling of parameter RRANGE.
;       2009-04-26 K. Tristram: Added keyword ADDITI with default value of 1
;                               pixel and changed determination of FWHM.
;       2009-12-21 K. Tristram: Added creation of a sky mask.
;       2010-07-16 K. Tristram: Added linear weighting in wavelength direction.
;       2010-11-04 K. Tristram: Major rewrite, accepts only final images.
;

FUNCTION MIDIMSKFIT, INFILE, MSKNAM, FACTOR=FACTOR, ADDITI=ADDITI, $
                      SMTVAL=SMTVAL, SKYMSK=SKYMSK, SKYDIS=SKYDIS, $
                      SKYWID=SKYWID, LINPAR=LINPAR

; INITIALISE A FEW VARIABLES
;-------------------------------------------------------------------------------
   expone = 3 ; this gives a nice bell shape, similar to the standard EWS mask
   rrange = [6, 24] ; range of rows (i.e. y-values) used for the fitting
   rrange = [6, 16] ; range of rows for the most recent UT data in 2009
   rrange = [6, 26]
   if n_elements(factor) ne 1 then factor = 1.25
; This should be better 1.5!!!
   if n_elements(additi) ne 1 then additi = 2.00
   if n_elements(smtval) ne 2 then smtval = [0., 0.]
   if n_elements(skydis) ne 1 then skydis = 1.0
   if n_elements(skywid) ne 1 then skywid = 4.0
   if n_elements(linpar) lt 1 then linpar = 0.0

; DEFINE A VECTOR CONTAINING THE ATMOSPHERIC TRANSPARANCY
;-------------------------------------------------------------------------------
transp = [0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000, $
          0.000,  0.000,  0.000,  0.006,  0.015,  0.021,  0.030,  0.050, $
          0.080,  0.101,  0.105,  0.097,  0.088,  0.090,  0.099,  0.106, $
          0.114,  0.135,  0.180,  0.250,  0.353,  0.484,  0.601,  0.689, $
          0.754,  0.804,  0.842,  0.870,  0.888,  0.898,  0.907,  0.913, $
          0.914,  0.915,  0.922,  0.926,  0.928,  0.929,  0.917,  0.901, $
          0.895,  0.904,  0.925,  0.944,  0.955,  0.963,  0.969,  0.975, $
          0.980,  0.983,  0.982,  0.983,  0.987,  0.989,  0.989,  0.989, $
          0.988,  0.985,  0.979,  0.979,  0.987,  0.993,  0.995,  0.994, $
          0.994,  0.995,  0.995,  0.994,  0.993,  0.995,  0.999,  1.000, $
          0.999,  0.998,  0.999,  0.999,  0.998,  0.999,  1.000,  0.999, $
          0.996,  0.992,  0.991,  0.994,  0.996,  0.994,  0.988,  0.982, $
          0.975,  0.958,  0.924,  0.869,  0.793,  0.704,  0.614,  0.541, $
          0.488,  0.453,  0.447,  0.444,  0.429,  0.503,  0.685,  0.851, $
          0.930,  0.954,  0.962,  0.963,  0.960,  0.955,  0.949,  0.944, $
          0.944,  0.944,  0.941,  0.934,  0.934,  0.942,  0.943,  0.925, $
          0.901,  0.861,  0.771,  0.627,  0.469,  0.353,  0.285,  0.251, $
          0.247,  0.249,  0.226,  0.170,  0.103,  0.055,  0.024,  0.007, $
          0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000, $
          0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.009,  0.025, $
          0.033,  0.023,  0.005,  0.000,  0.000,  0.000,  0.000,  0.000, $
          0.000,  0.000,  0.000]

; READ THE DATA OR ASSUME IT IS ALREADY A STRUCT OF THE CORRECT TYPE
;-------------------------------------------------------------------------------
   indata = oirgetdata(infile)
   refile = infile
   numele = n_elements(indata)

; FIND HOW MANY AND WHERE THE DATA WINDOWS ARE
;-------------------------------------------------------------------------------
   tagnam = tag_names(indata)
   tagnum = n_tags(indata)
   winidx = where(strmatch(tagnam, 'DATA*'))
   winnum = n_elements(winidx)

; IF DATA IS PSEUDOCOMPLEX (E.G. FROM FRINGEIMAGE) THEN MAKE IT FLOAT
;-------------------------------------------------------------------------------
   winsiz = size(indata[0].data1)
   nx     = winsiz[1]
   if (nx GT 320) then begin
; CREATE A NEW STRUCTURE AND MODIFY THE SIZE OF THE DATA FIELDS
;-----------------------------------------------------------------------
      tmpvar = indata[0]
      for i=0,winnum-1 do tmpvar = mod_struct(tmpvar, tagnam[winidx[i]], $
         fltarr(nx/2, winsiz[2]))
	 tmpvar = replicate(tmpvar, numele)
; CALCULATE THE REAL PART OF THE DATA AND COPY IT INTO THE NEW STRUCTURE
;-----------------------------------------------------------------------
         for i=0,winnum-1 do tmpvar.(winidx[i]) = $
	    transpose(abs(pseudocomplex(indata.(winidx[i]))))
; CREATE A NEW STRUCTURE AND MODIFY THE SIZE OF THE DATA FIELDS
;-----------------------------------------------------------------------
         indata = temporary(tmpvar)
         nx     = nx/2
   endif

; FIND OUT THE DATA TYPE (GRISM OR PRISM)
;-------------------------------------------------------------------------------
   disper = STRTRIM(STRLOWCASE(midigetkeyword('GRIS NAME', infile)),2)
   case disper of
      'prism': begin
         colmin =  30
         colmax = 140
         colexa = [ 50, 120]
         yrange = [  8,  22]
      end
      'grism': begin
         colmin =  20
         colmax = 250
         colexa = [ 30, 200]
         yrange = [  5,  25]
      end
      else: begin
         print, 'Unknown dispersion type or window type! Exiting.'
         RETURN, -1
      end
   endcase

; GET THE WAVELENGTH SCALING IF NOT ALREADY DONE
;-------------------------------------------------------------------------------
   if n_elements(refile) lt 1 then refile = getenv(disper+'hmask')
   wavele = oirgetwavelength(getenv(disper+'hmask'))

; CALCULATE A FEW ARRAYS DEPENDING ON THE DATA TYPE
;-------------------------------------------------------------------------------
   coln_1 = findgen(colmax-colmin+1)+colmin
   coln_2 = findgen(nx)
   rownum = findgen(winsiz[2])

; CREATE THE STRUCT HOLDING THE RESULTING MASKS
;-------------------------------------------------------------------------------
   newmsk = indata[0]
   for i=1,winnum do newmsk = create_struct(newmsk, $
      'TRACE'+strtrim(i, 2), fltarr(3), 'FWHM'+strtrim(i, 2), fltarr(3))

; CHECK IF MASK IS INTEGER BECAUSE OF INPUT, THEN CHANGE TO FLOAT
;-------------------------------------------------------------------------------
   for i=0,winnum-1 do if size(newmsk.(winidx[i]), /type) eq 2 then $
      newmsk = mod_struct(newmsk, tagnam[winidx[i]], 1.0*newmsk.(winidx[i]))

; NOW ALSO CREATE THE SKY MASK, WHICH IS IDENTICAL TO THE MASK IN STRUCTURE
;-------------------------------------------------------------------------------
   skymsk = newmsk

; CALCULATE WEIGHTS DEPENDING LINEARLY ON WAVELENGTH
;-------------------------------------------------------------------------------
   if linpar[0] ge 0. then slopef = 1. - linpar[0]*(max(wavele)-wavele) < 1. > 0. $
      else slopef = 1. - linpar[0]*(min(wavele)-wavele) < 1. > 0.

; LOOP OVER THE DETECTOR WINDOWS
;-------------------------------------------------------------------------------
   for i=0,winnum-1 do begin
; GET THE TAG INDEX FOR THE DATA OF THE CURRENT WINDOW
;-----------------------------------------------------------------------
      curidx = winidx[i]
; FIT GAUSSIANS COLUMNWISE TO THE COMBINATION OF THE BEAMS
;-----------------------------------------------------------------------
      if numele gt 1 then tmpvar = total(indata.(curidx), 3) $
         else tmpvar = indata.(curidx)
      params = midiGaudis(tmpvar, smtval=smtval, rrange=rrange)
; NOW FIT GAUSSIANS COLUMNWISE TO ALL BEAMS INDIVIDUALLY
;-----------------------------------------------------------------------
      tmpvar = params
      params = replicate(tmpvar[0],  nx, numele+1)
      params[*,numele] = tmpvar
      for j=0,numele-1 do params[*,j] =  midiGaudis(indata[j].(curidx), $
         smtval=smtval, rrange=rrange)
; SET THE ERRORS FOR BAD FITS TO INFINITY
;-----------------------------------------------------------------------
      traces = replicate({p:fltarr(3), v:fltarr(nx)}, numele+1)
      fwhmax = replicate({p:fltarr(3), v:fltarr(nx)}, numele+2)
      for j=0,numele do begin
; SET THE ERRORS FOR BAD FITS TO INFINITY
;---------------------------------------------------------------
         badfit = where(params[*,j].errflg eq 1B)
         if (MAX(badfit) GE 0) then begin
            params[badfit,j].positierr = !values.f_infinity
            params[badfit,j].fwhmaxerr = !values.f_infinity
         endif
; FIT THE TRACES BY POLYNOMIALS
;---------------------------------------------------------------
         goodfit = where(params[*,j].errflg eq 0B)
         traces[j].p = poly_fit(coln_2[goodfit], params[goodfit,j].positi,$
            2, measure=params[goodfit,j].positierr)
         traces[j].v = pex(traces[j].p, coln_2)
; FIT THE FWHMS BY POLYNOMIALS
;---------------------------------------------------------------
         fwhmax[j].p = poly_fit(coln_2[goodfit], params[goodfit,j].fwhmax,$
            2, measure=params[goodfit,j].fwhmaxerr)
         fwhmax[j].v = pex(traces[j].p, coln_2)
      endfor
; FIT THE SEPARATION OF THE TRACES BY A POLYNOMIAL
;-----------------------------------------------------------------------
;	The following fits the separation of the traces:
      tmpvar = 0.5* (max(traces.v, dim=2) - min(traces.v, dim=2))
;	The following directly fits the separation of the data:
;	tmpvar = 0.5* (max(params.positi, maxidx, dim=2) - $
;                      min(params.positi, minidx, dim=2))
      separa = poly_fit(coln_1, tmpvar[colmin:colmax], 2)
; COPY THE AVERAGED TRACE PARAMETERS INTO THE RESULTING MASK STRUCT
;-----------------------------------------------------------------------
      newmsk.(tagnum+2*i+0) = total(reform(traces[0:numele-1].p, $
         3, numele), 2) / numele

; FIT THE ENVELOPE OF THE FWHM BY A POLYNOMIAL
;-----------------------------------------------------------------------
;     tmpvar = max(params.fwhmax, maxidx, dim=2)
;           multi dimensional max_subscript does not seem to work
      ferr = fltarr(nx)
      for k=0,nx-1 do begin
         if (TOTAL(params[k,*].errflg) EQ 0) then begin
            tmpvar[k] = max(params[k,*].fwhmax,z)
            ferr[k] = params[k,z].fwhmaxerr
         endif
      endfor
      fwhmax[numele+1].p = poly_fit(coln_2[goodfit], tmpvar[goodfit], 1, measure=ferr[goodfit])
      fwhmax[numele+1].v = pex(fwhmax[numele+1].p, coln_2)

; COPY THE TRACE INTO THE RESULTING MASK STRUCT
;-----------------------------------------------------------------------
   newmsk.(tagnum+2*i+1) = factor * fwhmax[numele+1].p + 2.0 * separa + $
     [additi, 0.0, 0.0]
; COPY THE MASK PARAMETERS FOR THIS WINDOW ALSO INTO THE SKY MASK
;-----------------------------------------------------------------------
   for j=0,1 do skymsk.(tagnum+2*i+j) = newmsk.(tagnum+2*i+j)

; CALCULATE THE MASK FOR THE SPECTRUM AND THE SKY
;-----------------------------------------------------------------------
; an easy way to limit the mask in wavelength direction is to only
; calculate it for a certain range of columns
   for j=0,nx-1 do begin
; FIRST CALCULATE THE MASK FOR THE SPECTRUM
;---------------------------------------------------------------
      tmpvar = (abs((rownum - poly(j, newmsk.(tagnum+2*i+0))) / $
         poly(j, newmsk.(tagnum+2*i+1)) * $
         2*sqrt(2*alog(2))))^expone
      newmsk.(curidx)[j,*] = exp(-0.5*tmpvar) * slopef[j]; * transp[j]
; EITHER USE VARIABLE OR FIXED SKY DISTANCE
;---------------------------------------------------------------
      if 1 then skysep = skydis * poly(j, newmsk.(tagnum+2*i+1)) $
      else skysep = skydis
; NOW CALCULATE THE MASK FOR THE SKY
;---------------------------------------------------------------
      tmpvar  = exp(-0.5 * (2.0 / skywid * $
         abs((rownum-poly(j, newmsk.(tagnum+2*i)) - skysep)))^10.)
      tmpvar += exp(-0.5 * (2.0 / skywid * $
         abs((rownum-poly(j, newmsk.(tagnum+2*i)) + skysep)))^10.)
      skymsk.(curidx)[j,*] = tmpvar
   endfor
   newmsk.(curidx) = (1.1 * newmsk.(curidx) - 0.05) < 1.0 > 0.0
   skymsk.(curidx) = (2.0 * skymsk.(curidx) - 1.00) < 1.0 > 0.0

endfor

; SAVE THE MASK AS A FITS FILE IF A NAME HAS BEEN GIVEN
;-------------------------------------------------------------------------------
   refile = (strsplit(refile[0], /extract))[0]
   tmpvar = OBJ_NEW('fitsfile', refile)
   header = tmpvar->head()
   tmpvar->close
   header->addpar, 'OBJECT', 'Object mask', com=' Original target'
   header->addpar, 'ORIGIN', 'MIDIMSKFIT', $
      commen=' MIDIMSKFIT mask fitting routine v2010-11-04'
   comstr = ' MIDIMSKFIT: '
   header->addpar, 'M_FACTOR', factor,    com=comstr+'broadening factor'
   header->addpar, 'M_ADDITI', additi,    com=comstr+'additional width'
   header->addpar, 'M_SMOOTX', smtval[0], com=comstr+'smoothing width in x'
   header->addpar, 'M_SMOOTY', smtval[1], com=comstr+'smoothing width in y'
   header->addpar, 'M_SKYDIS', skydis,    com=comstr+'sky distance'
   header->addpar, 'M_SKYWID', skywid,    com=comstr+'sky band width'
   keywrd = ['(constant)', '(linear)', '(quadratic)']
   for i=0,winnum-1 do begin
      tmpstr = ' window '+strtrim(i+1,2)+' '
      for j=0B,2B do header->addpar, 'M_T'+strtrim(i+1,2)+$
         'COE'+strtrim(j+1,2), (newmsk.(tagnum+2*i+0))[j], $
         comment=comstr+'Trace'+tmpstr+'coeff '+strtrim(j+1,2)+' '+keywrd[j]
      for j=0B,2B do header->addpar, 'M_W'+strtrim(i+1,2)+$
         'COE'+strtrim(j+1,2), (newmsk.(tagnum+2*i+1))[j], $
         comment=comstr+'Width'+tmpstr+'coeff '+strtrim(j+1,2)+' '+keywrd[j]
   endfor
   skyHead = header->clone()
;         prepare output detector object; adjust nx if necessary
;         we can't use oirNewData because we changed data formats
   refDet = OBJ_NEW('IMAGEDETECTOR', refile)
   refDet->close
   refTab = refDet->table()
   for i=0,winnum-1 do (*refTab)[i].naxis[0] = nx

   image = OBJ_NEW('FITSTABLE', newmsk, extna='IMAGING_DATA')
   imHead = image->head()
   imHead->addpar,'NREGION', winnum
   imHead->addpar,'MAXTEL', 2
   imHead->addpar,'MAXINS', 2
   imHead->addpar,'MAXSTEP', 40

  
   refDet->newFile, msknam+'.srcmask.fits',priHead=header
   image->appendToFile, refDet
   image->writeRows,newmsk
   image->close
   refDet->close

   skyHead->addpar, 'OBJECT', 'Sky mask', com=' Original target'
   refDet->newFile, msknam+'.skymask.fits',priHead=skyHead
   image->appendToFile, refDet
   image->writeRows,skymsk
   image->close
   refDet->close
   OBJ_DESTROY,image
   OBJ_DESTROY,refDet

; RETURN THE RESULT MASK
;-------------------------------------------------------------------------------
RETURN, newmsk

END

;          End tristram code
PRO midiVisPipe, base, file, mask=mask, smooth=smooth, gsmooth=gsmooth,$
    dAve=dAve, minopd=minopd, maxopd=maxopd, fringeimage=fringeimage
;    set defaults  for fringe smooth and group delay smooth
   if (NOT KEYWORD_SET(smooth))  then smooth  = 50.
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 4.
   if (NOT KEYWORD_SET(maxopd))  then maxopd  = 100.
   mxopd = ' '+strtrim(maxopd,2)+' '  ; string version

   if (KEYWORD_SET(dAve)) then begin
      doAve = " -removeAverage" 
      if (KEYWORD_SET(minopd)) then mnopd = ' '+strtrim(minopd,2)+' ' $
         else mnopd = " 20. "
   endif else begin
      doAve=" -noremoveAverage"
      if (KEYWORD_SET(minopd)) then mnopd = ' '+strtrim(minopd,2)+' '$
         else mnopd = " 0. " 
   endelse
;    where is executable?
   cbin = getenv('vltiCbin')+'/'

;    parse input files to see if both visibility and photometry present
   midiGetIns,file,shutter=shutter, grism=grism, object=object, opt1=opt1,$
      ierr=ierr
   if (ierr NE 0) then RETURN
   shutter = STRTRIM(shutter,2)
   grism   = STRTRIM(grism,2)
   if (shutter NE 'ABOPEN') then begin
      print,'I need ABOPEN interferometry data'
      RETURN
   endif
   if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
      print,'I only know how to deal with prism or grism data'
      RETURN
   endif

   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
   print,' input vis files have ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(file)))),' input frames'

   disp='dispVis "' ; cshell script name
   command = cbin+disp+file+'" '+mask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth) + doAve + mnopd + mxopd
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispVis failed'
      print,command
      RETURN
   endif
   erase
   !p.multi=[0,1,2,0,0]
   midiDelayPlot,base,object+':'+file
   v=oirgetvis(base+'.corr.fits',wave=wave)
;  plot,wave,v.visamp
   plot,wave,midiCvis(v)

   !p.multi=0
   if (KEYWORD_SET(fringeImage)) then begin
      print,'Computing fringe image '
      command = cbin+'dispCenter "'+file+'" '+base
      spawn,command
   endif
RETURN
END

PRO midiPhotoPipe, base, files, mask=mask, dSky=dSky, curve=curve
   if (NOT KEYWORD_SET(dSky)) then dSky = 0
   cbin = getenv('vltiCbin')+'/'
;              check input files
   d=files
   midiGetIns,files[0],shutter=shutter, grism=grism, opt1=opt1,ierr=ierr
   if (ierr NE 0) then RETURN
   grism = STRTRIM(grism,2)
   opt1 = STRTRIM(opt1,2)
   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
   fileList = REPLICATE(-1,2)
   for i=0,1 do begin
      midiGetIns,d[i],shutter=shutter, grism=grism
      grism=STRTRIM(grism,2)
      shutter = STRTRIM(shutter,2)
      if (shutter EQ 'AOPEN') then fileList[0]=i $
      else if (shutter EQ 'BOPEN') then fileList[1] = i
      if ((grism NE 'PRISM') AND (grism NE 'GRISM')) then begin
         print,'I only know how to deal with prism or grism data'
         RETURN
      endif
   endfor

   if (total(filelist EQ (-1)) NE 0) then begin
       print,'I need PA+PB to continue'
       RETURN
   endif
;   print,'dSky is ',dsky
   if (KEYWORD_SET(curve)) then ref = ' -ref '+curve else ref = ''
   command = cbin+'dispPhot "'+d[filelist[0]]+'" "'+d[filelist[1]]+'" '+mask+' '+base+' '+STRTRIM(dSky,2)+ref
   spawn,command
   p = oirgetdata(base+'.photometry.fits')
   plot,p[5].data1
RETURN
END

PRO midiPipe, base, inFiles, files=files, mask=mask, smooth=smooth, $
   gsmooth=gsmooth,fast=fast, dSky=dSky, filter=filter, fudge=fudge, $
      dAve=dAve, minopd=minopd, maxopd=maxopd, $
      fringeImage=fringeImage, shiftMask=shiftMask
;    set defaults  for fringe smooth and group delay smooth
   if (NOT KEYWORD_SET(smooth))  then smooth  = 50.
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 4.
   if (NOT KEYWORD_SET(gsmooth) AND (KEYWORD_SET(fast))) then gsmooth = .1
   if (NOT KEYWORD_SET(dSky))    then dSky    = 0
   if (NOT KEYWORD_SET(maxopd))  then maxopd  = 100.
   mx = ' '+strtrim(maxopd,2)+' ' 
   if (KEYWORD_SET(dAve)) then begin
      doAve = " -removeAverage" 
      if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' ' $
         else mo = " 20. "
   endif else begin
         doAve=" -noremoveAverage"
         if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' '$
            else mo = " 0. " 
   endelse
   chopPhotoDone = 0
;    consider temporary filter variant
   if (KEYWORD_SET(filter)) then disp='dispVis20 "' else disp='dispVis "'
   if (KEYWORD_SET(fudge))  then disp='dispVisF "'
;    where is executable?
   cbin = getenv('vltiCbin')+'/'
;    if files not specified, bring up gui
   if (NOT KEYWORD_SET(files)) then begin
      if(N_PARAMS() eq 2) then d = inFiles else d = midiGuis() 
   endif else d=files
;    parse input files to see if both visibility and photometry present
   if (N_ELEMENTS(d) NE 3) then begin
      print,'You need to specify two visibility file(s) and one' +$
         ' interferometry file.'
      RETURN
   endif
   fileList = REPLICATE(-1,3)
   for i=0,2 do begin
      midiGetIns,d[i],shutter=shutter, grism=grism, object=object, opt1=opt1,$
         ierr=ierr
      if (ierr NE 0) then RETURN
      shutter = STRTRIM(shutter,2)
      grism = STRTRIM(grism,2)
      if (shutter EQ 'AOPEN') then fileList[0]=i $
      else if (shutter EQ 'BOPEN') then fileList[1] = i $
      else if (shutter EQ 'ABOPEN') then fileList[2] = i
      if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
         print,'I only know how to deal with prism or grism data'
         RETURN
      endif
   endfor
   if (TOTAL(filelist eq (-1)) NE 0) then begin
      print,'You need to specify two visibility file(s) and one' +$
         ' interferometry file.'
      RETURN
   endif
;         look for default masks if not specified
   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
;  begin CAH
   if (grism EQ 'PRISM') then curvefile=getenv('prismscurve')
   if (grism EQ 'GRISM') then curvefile=getenv('grismscurve')
;  end CAH
;                consider shifting mask based on photometry
   finalMask = mask
   if (KEYWORD_SET(shiftMask)) then begin
      print,' '
      print,'   COMPUTING NEW MASK'
      command = cbin+'oirChopPhotoImages -in "'+d[fileList[0]]+'" -out '+ $
         base+'.Aphotometry.fits -dSky '+STRTRIM(dSky,2)
;     begin CAH
      command = command + ' -ref '+ curvefile + ' -order 1'
;     end CAH
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'oirChopPhotoImages failed for A channel'
         print,command
         RETURN
      endif
      command = cbin+'oirChopPhotoImages -in "'+d[fileList[1]]+'" -out '+ $
         base+'.Bphotometry.fits -dSky '+STRTRIM(dSky,2)
;     begin CAH
      command = command + ' -ref '+ curvefile + ' -order 1'
;     end CAH
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'oirChopPhotoImages failed for B channel'
         print,command
         RETURN
      endif
      command = cbin+'oirShiftMask -A '+base+'.Aphotometry.fits -B '+ $
         base+'.Bphotometry.fits -out '+base+'.newmask.fits -refMask '+$
         mask
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'oirShiftMask failed'
         print,command
         RETURN
      endif
      finalMask = base+'.newmask.fits'
      midiPlotOverlap, base, object+':'+d[filelist[0]],mask=finalMask,$
         oldmask=mask
      print,'   FINISHED COMPUTING NEW MASK '+finalMask
      chopPhotoDone = 1
   endif
;                process visibility measurement
   if (NOT KEYWORD_SET(fast)) then $
      command = cbin+disp+d[filelist[2]]+'" '+finalMask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth) + doAve +mo +mx $
   else command = cbin+'dispVisT "'+d[filelist[2]]+'" '+finalMask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth)+doAve+mo+mx
   print,' input vis files have ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(d[filelist[2]])))),' input frames'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispVis failed'
      print,command
      RETURN
   endif
   erase
   !p.multi=[0,1,2,0,0]
   midiDelayPlot,base,object+':'+d[filelist[2]]
;                process photometry
   if (chopPhotoDone eq 0) then $
      command = cbin+'dispPhot "'+d[filelist[0]]+'" "'+d[filelist[1]]+'" '+$
      finalMask+' '+ base+' '+STRTRIM(dSky,2) $
   else command = cbin+'dispPhotShort '+base+' '+finalMask
   print,' input photA file has ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(d[filelist[0]])))),' input frames'
   print,' input photB file has ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(d[filelist[1]])))),' input frames'
;  begin CAH
   command = command + ' ' + curvefile + ' 1'
;  end CAH
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispPhot failed'
      print,command
      RETURN
   endif
   spawn,cbin+'oirRedCal '+base, EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirRedCal failed'
      print,command
      RETURN
   endif
   midiInsVisPlot,base,object+':'+d[filelist[0]]
   !p.multi = 0
   if (KEYWORD_SET(fringeImage)) then begin
      print,'Computing fringe image '
      command = cbin+'dispCenter "'+d[filelist[2]]+'" '+base
      spawn,command
      midiPlotOverlap, base, object+':'+d[filelist[0]],mask=finalMask,/fringe
   endif
RETURN
END

PRO midiShiftMask, tag, oldMask, newMaskFile
   aFile = tag+'.Aphotometry.fits'
   bFile = tag+'.Bphotometry.fits'
   spawn,getenv('vltiCbin')+'/oirShiftMask -A '+afile+' -B '+bfile+ $
      ' -out '+ newMaskFile+' -refMask '+oldMask
RETURN
END

PRO midiCurveCal, files, outFile
   cbin = getenv('vltiCbin')+'/'
RETURN
END

PRO midiCrossCoeff, base, files, curveFile=curveFile, sky=sky
;    check and set defaults
   cbin = getenv('vltiCbin')+'/'
   if (KEYWORD_SET(sky)) then doSky = ' -sky ' else doSky = ' '
   fileList = INTARR(2)
   for ifile =0,1 do begin 
      midiGetIns,files[ifile],shutter=shutter, grism=grism, object=object, $
         opt1=opt1,ierr=ierr
      if (ierr NE 0) then RETURN
      shutter = STRTRIM(shutter,2)
      grism   = STRTRIM(grism,2)
      opt1    = STRTRIM(opt1,2)
      if (opt1 NE 'SCI_PHOT') then begin
         print,'I can only calculate cross coupling coefficients for SCI_PHOT data'
         RETURN
      end
      if (shutter EQ 'AOPEN') then fileList[0]=ifile $
      else if (shutter EQ 'BOPEN') then fileList[1] = ifile 
      if ((grism NE 'PRISM') AND (grism NE 'GRISM')) then begin
         print,'I only know how to deal with prism or grism data'
         RETURN
      endif
   endfor
   if ((TOTAL(fileList eq 1) NE 1) OR (TOTAL(fileList eq 0) ne 1)) then begin
      print,'I need one AOPEN file(set) and one BOPEN file(set)'
      RETURN
   endif
   if (NOT KEYWORD_SET(curveFile)) then begin
      if (grism EQ 'PRISM') then curveFile = getenv('prismscurve')
      if (grism EQ 'GRISM') then curveFile = getenv('grismscurve')
   endif
   command = cbin+'dispCrossCoeff "'+files[fileList[0]]+'" "'+files[fileList[1]]+ $
      '" '+curveFile+' '+base+doSky
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispCrossCoeff failed'
      print,command
      RETURN
   endif
RETURN
END

PRO midiSPipe, base, files, mask=mask, smooth=smooth, gsmooth=gsmooth, $
   dSky=dSky, dAve=dAve, cross=cross, curve=curve, photoFile=photoFile, $
   minopd=minopd
;    set defaults  for fringe smooth and group delay smooth
   if (NOT KEYWORD_SET(smooth)) then smooth = 50.
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 4.
   if (NOT KEYWORD_SET(gsmooth) AND (KEYWORD_SET(fast))) then gsmooth = .1
   if (NOT KEYWORD_SET(dSky)) then dSky = 0
   if (KEYWORD_SET(dAve)) then begin
      doAve = " -removeAverage" 
      if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' ' $
         else mo = " 20. "
   endif else begin
      doAve=" -noremoveAverage"
         if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' '$
            else mo = " 0. " 
   endelse
;    consider temporary filter variant
;    where is executable?
   cbin = getenv('vltiCbin')+'/'
;
;           sort out options for photometry processing:
;           (1) only one input file, no photofile.  This should be an AB chopped file
;               and we get the photometry from the photometry channels
;           (2) same as above but with photofile present and assumed to be a reduced
;               set of photometry images.  After getting photometry from AB as
;               above, rescale photofile and use it as photometry
;           (3) 3 input files, presumed to be an AB file+A+B files.  Compute a photofile 
;               A- and B- files, and process as in (2).  Ignore anything in the
;               photofile arguement
;
   nfiles = N_ELEMENTS(files)
   if (nfiles EQ 1) then inFiles = files else begin
      if (nfiles NE 3) then begin
         print,'I need either 1 or 3 input files '
         RETURN
      endif
      inFiles = strarr(3)
      for i=0,2 do begin
         midiGetIns,files[i],shutter=shutter
         shutter = STRTRIM(shutter,2)
         if (shutter eq 'ABOPEN') then inFiles[0] = files[i] $
         else if (shutter eq 'AOPEN') then inFiles[1] = files[i] $
         else if (shutter eq 'BOPEN') then inFiles[2] = files[i]
      endfor
      if (total(inFiles eq '') NE 0) then begin
         print,'If you give me three files they must be ABOPEN+AOPEN+BOPEN'
         RETURN
      endif
   endelse
   midiGetIns,inFiles[0],shutter=shutter, grism=grism, object=object, opt1=opt1
   shutter = STRTRIM(shutter,2)
   grism   = STRTRIM(grism,2)
   opt1    = STRTRIM(opt1,2)
   if (opt1 NE 'SCI_PHOT') then begin
      print,'midiSPipe only works on SCI_PHOT data, not '+opt1+' data'
      RETURN
   endif
   if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
      print,'I only know how to deal with prism or grism data'
      RETURN
   endif

   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then mask = getenv('prismsmask')
      if (grism EQ 'GRISM') then mask = getenv('grismsmask')
   endif
   if (NOT KEYWORD_SET(curve)) then begin
      if (grism EQ 'PRISM') then curve= getenv('prismscurve')
      if (grism EQ 'GRISM') then curve= getenv('grismscurve')
   endif
   if (NOT KEYWORD_SET(cross)) then begin
      print,'I need a specified cross coupling coefficient file'
      RETURN
   endif else begin
;      midiGetIns,cross,grism=crossgrism, opt1=opt1
;      if (opt1 NE 'SCI_PHOT') then begin
;         print,'cross file ',cross,' does not use SCI_PHOT mode'
;         RETURN
;      endif 
;      if (grism NE crossgrism) then begin
;         print,'Data is in '+grism+' mode but cross file is in '+crossgrism+' mode.'
;         RETURN
;      endif
   endelse
;
;       now consider the reduced photometry file
;       if there are 3 input files, create it from #2 and #3 and store it in localPhoto
;       if there is 1 input file and photofile is specified, copy it to localPhoto
;       if not set localPhoto = ""
   localPhoto = ""
   if (nFiles eq 3) then begin
      print,'   Computing A- and B- photometry from 2nd and 3rd input files '
      print,'      and writing into '+base+'.P.photometry.fits'
      command = cbin+'dispPhot "'+files[1]+'" "'+files[2]+'" '+mask+' '+base+'.P '+STRTRIM(dSky,2)
;     begin CAH
      command = command + ' ' + curve + ' 1'
;     end CAH
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'dispPhot local failed'
         print,command
         RETURN
      endif
      localPhoto = base+'.P.photometry.fits'
   endif else if (KEYWORD_SET(photofile)) then localPhoto = photofile
   disp = 'dispVis "'
   command = cbin+disp+files+'" '+mask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth) + doAve +mo
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispVis failed'
      print,command
      RETURN
   endif
   erase
   !p.multi=[0,1,2,0,0]
   midiDelayPlot,base,object+':'+files[0]
   if (localPhoto eq "") then begin
      print,'   calculating photometry from PA+PB SCI_PHOT channels'
      command = cbin+'dispSciPhot "'+files[0]+'" '+mask+' '+curve+' '+$
         cross+' '+base+' '+STRTRIM(dSky,2) 
   endif else begin
      print,'   calculating photometry scale from PA+PB channels, and'
      print,'     applying it to I1+I2 photometric data in '+localphoto
      command = cbin+'dispSciPhotPhot "'+files[0]+'" '+mask+' '+curve+' '+$
         cross+' '+localPhoto+' '+base+' '+STRTRIM(dSky,2) 
   endelse
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispSciPhot failed'
      print,command
      RETURN
   endif
   spawn,cbin+'oirRedCal '+base, EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirRedCal failed'
      print,command
      RETURN
   endif
   midiInsVisPlot,base,object+':'+files[0]
   !p.multi = 0
RETURN
END

FUNCTION reduceDay, dir=dir, infiles=infiles, outdir=outdir, base=base, mask=mask
oldx = !x
oldy = !y
!x.style=1
!y.style=1

;
   if (NOT KEYWORD_SET(outdir)) then odir='' else odir=outdir+'/'
   if (KEYWORD_SET(infiles)) then files = infiles else $
      if (KEYWORD_SET(dir)) then files=midiGuis(dir=dir) $
         else files=midiGuis()
   nFiles = N_ELEMENTS(files)
   print,'Got ',nFiles, ' Files. Proceed?'
   ll = ''
   read,ll
   if ((ll EQ 'N') OR (ll EQ 'n')) then RETURN,files

   if (0 NE (nFiles mod 3)) then begin
      print,'Number of files not divisible by 3'
      RETURN,files
   endif

   n3 = nFiles/3
   base = STRARR(n3)
   for i=0, n3-1 do begin
      midiGetIns,files[3*i+1],object=object
      base[i]=STRTRIM(object,2)
   endfor
   print,'raw base names ',base
; rename duplicates
   for i=0, n3-1 do begin
      w = WHERE(base[i] EQ base)
      nW = N_ELEMENTS(w)
      if (nW GT 1) then $
         for j=1,nW-1 do base[w[j]]=STRTRIM(base[w[j]])+'-'+STRTRIM(j,2)
   endfor
   print,'new base names ',base

   for iFile=0, n3-1 do begin
      fileList = REPLICATE(-1,3)
      d = files[3*iFile+indgen(3)]
      for i=0,2 do begin
         s = STRSPLIT(d[i],' ',/EXTRACT)
         d[i] = s[0]
         midiGetIns,d[i],shutter=shutter, grism=grism
         shutter = STRTRIM(shutter,2)
         grism = STRTRIM(grism,2)
         if (shutter EQ 'AOPEN') then fileList[0]=i
         if (shutter EQ 'BOPEN') then fileList[1]=i
         if (shutter EQ 'ABOPEN') then  fileList[2]=i
         if ((grism NE 'PRISM') AND (grism NE 'GRISM')) then begin
            print,'I only know how to deal with prism or grism data'
            RETURN,files
         endif
      endfor
      if (TOTAL(fileList EQ -1) NE 0) then begin
         print,'For base ',base[iFile],' I dont have Int+2*Phot'
         RETURN,files
      endif
   endfor
   for iFile=0, n3-1 do begin
      if (STRMID(base[iFile],0,4) EQ 'MIDI') then begin
          print,'I wont process "MIDI" files'
          RETURN,files
      endif
      base[iFile] = STRTRIM(odir+base[iFile])
      if(base[iFile] EQ '') then RETURN,files
      print,iFile,' cleaning old files named '+base[iFile]+'.*.fits'
      spawn,' rm '+base[iFile]+'*.fits'
      print,iFile,' processing ',base[iFile]
      midipipe,base[iFile],files=files[3*iFile+indgen(3)],mask=mask
   endfor
!x=oldx
!y=oldy
RETURN,files
END

PRO midiCalibrate, tag, calTag, photoTag=photoTag, $
   calDatabase=calDatabase, $
   calFlux10=calFlux10, diam=diam, nophot=nophot, print=print
   
   if (N_PARAMS() EQ 0) then begin
      print,'midiCalibrate, tag, calTag, photoTag=photoTag, '
      print,'   calDatabase=calDatabase,calFlux10=calFlux10, diam=diam, nophot=nophot, print=print'
      RETURN
   endif
;         save externally defined plotting preferences
   oldXY    = [!x.style, !y.style]
   oldxr    = !x.range
   oldyr    = !y.range
   oldMulti = !p.multi

   doDB = KEYWORD_SET(calDatabase)
   doprint = KEYWORD_SET(print)
   doPhot  = (NOT KEYWORD_SET(nophot))

   if (NOT KEYWORD_SET(calFlux10)) then calFlux10 = 1.
   if (NOT KEYWORD_SET(diam))      then diam = 0.

   if (doDB) then begin
      calFile = calTag+'.fringes.fits'
      calSource = calDatabase->sourceFromFile(calFile, diamData=diamData, $
         ierr=ierr)
      if (ierr NE 0) then begin
         print,'   could not find calibrator in database '
         RETURN
      endif
      diam = 1.e+3 * diamData[0].diameter ; millarcsec
      calJyFile = calTag+'.calJy.fits'
      midiCalJy, calFile, calDatabase, calJyFile
   endif
   print,'   using diameter =',diam,' millarcsec ',format='(a,f8.2,a)'
   cbin = getenv('vltiCbin')+'/'
   cmd = cbin+'oirCalibrateVis '+tag+' '+calTag + ' '
   if (KEYWORD_SET(photoTag)) then cmd = cmd + "-sp "+photoTag+" "
   if (doDB) then cmd = cmd+" -calspec "+calJyFile+" " $
   else if (KEYWORD_SET(calFlux10)) then cmd = cmd + "-calflux "+ $
      STRTRIM(calFlux10,2)+" "
   if (KEYWORD_SET(diam)) then cmd = cmd + "-cald "+STRTRIM(diam,2)+" "
   if (KEYWORD_SET(nophot)) then cmd = cmd + " -nophot "
   spawn,cmd
   file = OBJ_NEW('fitsfile',caltag+'.corr.fits')
   head = file->head()
   caldate = head->getpar('DATE-OBS')
   obj_destroy,head
   file->close
   obj_destroy,file
   file = OBJ_NEW('fitsfile',tag+'.corr.fits')
   head = file->head()
   tardate = head->getpar('DATE-OBS')
   file->close
   obj_destroy,head
   obj_destroy,file
   plotname=!d.name
   set_plot,'ps'
   device,file=tag+'calPlots.ps',xs=15,ys=25,yo=1
   !p.multi=[0,1,4,0,0]
   title=calTag+':'+caldate
;                   instrumental visibilities
;                   modify visamp by supplied cal flux.
   calCorr  = oirGetVis(calTag+'.corr.fits',wave=wave)
   targCorr = oirGetVis(tag+'.corr.fits')
   calFlux = calFlux10*100./wave^2
   !x.style=1
   !y.style=1
   !x.range=[7.5,13.5]
   xw = where(wave gt 7.6 and (wave lt 13.5))
   nw = N_ELEMENTS(xw)/8
   xer = xw[8*indgen(nw)]
   xt = 'Wavelength, (mu)'

   nWave = N_ELEMENTS(wave) -1
   nxWave = N_ELEMENTS(xw) - 1
   if (doPhot) then begin ; I have decent photometry
      calVis = oirGetVis(calTag+'.redcal.fits',wave=wave)

; Calibrator; plot total A+B/2 raw flux, total mask sqrt(AB) flux raw correlated flux

      calPhot = oirgetData(calTag+'.photometry.fits')
;
      data = .5*(calPhot[0].data1+calPhot[1].data1)
      err  = calPhot[8].data1
      plot,wave,data,yr=[0,1.2*max(data[xw])],$
      xtitle=xt,title=title+' Raw Total Photometry'
      errplot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.RawTotalPhotometry.dat',/GET_LUN
         for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
         FREE_LUN,unit
      endif
;
      data = calPhot[5].data1
      err  = calPhot[11].data1
      plot,wave,data,yr=[0,1.2*max(data[xw])],$
         xtitle=xt,title=title+' Raw Masked Photometry'
      errplot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.RawMaskedPhotometry.dat',/GET_LUN
         for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
         FREE_LUN,unit
      endif
   endif   ; photometry desired doPhot
;
   data = calCorr[0].visamp
   err  = calCorr[0].visamperr
   plot,wave,data,yr=[0,1.2*max(data[xw])],$
      xtitle=xt,title=title+' Raw Correlated Amplitude'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]

   if (doprint) then begin
      openw,unit,title+'.RawCorrelatedAmplitude.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif
;
   data = calCorr[0].visphi
   plot,wave,data, xtitle=xt,title=title+' Raw Correlated Phase'
      errPlot,wave[xer],data[xer]-calCorr[0].visphierr[xer], $
         data[xer]+calCorr[0].visphierr[xer]
   if (doprint) then begin
      openw,unit,title+'.RawCorrelatedPhase.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif

;calibrator reduced values

   if (doPhot) then begin
      data = calVis[0].visamp
      err  = calVis[0].visamperr
      plot,wave,data, yr=[0,1.2*max(data[xw])], $
         xtitle=xt,title=title+' Instrumental Visibility'
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.InstrumentalVisibility.dat',/GET_LUN
         for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
         FREE_LUN,unit
      endif
;
      data = calVis[0].visphi
      err  = calVis[0].visphierr
      plot,wave,data, xtitle=xt,title=title+' Instrumental Phase'
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.InstrumentalPhase.dat',/GET_LUN
         for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
         FREE_LUN,unit
      endif
;
      data = .5*(calPhot[0].data1 + calPhot[1].data1)/calFlux
      err = calPhot[8].data1/calFlux
      plot,wave,data, xtitle=xt,title=title+' ADU/s/Jy ', $
         yr=[0,1.2*max(data[xw])]
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
;
;target raw values

      targPhot = oirGetData(tag+'.photometry.fits')
      title=tag+':'+tardate
;
      data = 0.5*(targPhot[0].data1+targPhot[1].data1)
      err  = targPhot[8].data1
      plot,wave,data, yr=1.2*[0,max(data[xw])], xtitle=xt,$
         title=title+'Raw Total A+B photometry (ADU/s/chan) '
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.RawTotalPhotometry.dat',/GET_LUN
         for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
         FREE_LUN,unit
      endif
;
      data = targPhot[5].data1
      err  = targPhot[11].data1
      plot,wave,data, yr=1.2*[0,max(data[xw])], xtitle=xt,$
         title=title+'Raw Masked sqrt(AB) photometry (ADU/s/chan) '
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.RawMaskedPhotometry.dat',/GET_LUN
         for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
         FREE_LUN,unit
      endif
   endif   ; doPhot needed
;
   yr = [0,1.2*max(targCorr.visamp[xw])]
   for i=0,n_elements(targCorr)-1 do begin
      data = targCorr[i].visamp
      err  = targCorr[i].visamperr
      plot,wave,data,yr=yr, xtitle=xt,$
         title=title+'Raw Correlated Flux (ADU/s/channel)'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.RawCorrelatedFlux.dat',/GET_LUN
         for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
         FREE_LUN,unit
      endif
   endfor

;instrumental visibility
   if (doPhot) then begin
      targInsVis = oirGetVis(tag+'.redcal.fits')
      yr = [0,1.2*max(targInsVis.visamp[xw])]
      yrd = [min(targInsVis.visphi[xw]), max(targInsVis.visphi[xw])]
      for i=0,n_elements(targInsVis)-1 do begin
         data = targInsVis[i].visamp
         err  = targInsVis[i].visamperr
         plot,wave,data,yr=yr, $
            xtitle=xt,title=title+' Instrumental Visibility '
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
         if (doprint) then begin
            openw,unit,title+'.InstrumentalVisibility.dat',/GET_LUN
            for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
            FREE_LUN,unit
         endif

         data = targInsVis[i].visphi
         err  = targInsVis[i].visphierr
         plot,wave,data , yr=yrd,xtitle=xt,title=title+' Instrumental Phase'
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
         if (doprint) then begin
            openw,unit,title+'.InstrumentalPhase.dat',/GET_LUN
            for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
            FREE_LUN,unit
         endif
      endfor

;calibrated target values
      targVis  = oirGetVis(tag+'.calvis.fits')
      targPhot = oirGetData(tag+'.calphot.fits')

;calibrated visamp and phase
      yr = [0,1.2*max(targVis.visamp[xw])<1.5]
      yrd = [min(targVis.visphi[xw]), max(targVis.visphi[xw])]
      for i=0,n_elements(targVis)-1 do begin
         data = targVis[i].visamp
         err  = targVis[i].visamperr
         plot,wave,data,yr=yr, $
            xtitle=xt,title=title+' Calibrated Visibility '
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
         if (doprint) then begin
            openw,unit,title+'.CalibratedVisibility.dat',/GET_LUN
            for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
            FREE_LUN,unit
         endif

         data = targVis[i].visphi
         err  = targVis[i].visphierr
         plot,wave,data , yr=yrd,xtitle=xt,title=title+' Calibrated Phase'
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
         if (doprint) then begin
            openw,unit,title+'.CalibratedPhase.dat',/GET_LUN
            for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
            FREE_LUN,unit
         endif
      endfor

;calibrated photometry, total and masked
      data = targPhot[0].data1   ; total
      err  = targPhot[2].data1
      yr = 1.2*max(data[xw])*[0,1]
      plot,wave,data, yr=yr, xtitle=xt, $
         title=title+' Calibrated Total Flux(Jy)'
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.CalibratedTotalPhotometry.dat',/GET_LUN
         for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
         FREE_LUN,unit
      endif

;masked flux (Jy)

      data = targPhot[1].data1   ; masked
      err  = targPhot[3].data1
      plot,wave,data, yr=yr,xtitle=xt, $
         title=title+' Calibrated Flux under Mask (Jy)'
         errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.CalibratedMaskedPhotometry.dat',/GET_LUN
         for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
         FREE_LUN,unit
      endif
   endif   ;  photometry available
;correlated flux (Jy) = masked flux*visibility

;            default is correlated flux = visibility * JY(calib) * 
;                 total_flux(source)/total_flux(calibrator)
;
;            if photometry is lousy you can use alternatively
;                 correlated flux = visibility * JY(calib) * 
;                 masked_flux(source)/masked_flux(calibrator)
;            since the visibility = corr_flux(source)/masked_flux(source) *masked_flux(cal)/corr_flux(cal)
;            the masked_fluxes cancel out.
   if (doPhot) then nVis = N_ELEMENTS(targVis) else nVis = N_ELEMENTS(targCorr)
   for i=0, nVis -1 do begin
      if (NOT doPhot) then begin
         cc = calCorr.visamp > (.003 * max(calCorr.visamp)) ; truncated version of calcorr
         va = targCorr[i].visamp
         data = calFlux * va / cc
         err  = calFlux*SQRT ((targCorr[i].visamperr/cc)^2 + $
            (va * calCorr.visamperr/(cc*cc))^2)
      endif else begin
         data = targVis[i].visamp * targPhot[0].data1
         err = targVis[i].visamperr * targPhot[0].data1
      endelse
      uvw = midifitsuvw(tag+'.fringes.fits', alt=alt)
      plot,wave,data,yr=max(data[xw])*[0,1.2], $
         xtitle=xt,title=title+' Correlated Flux(Jy) UV ='+string(uvw[0:1],format='(2f8.2)')
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.CalibratedCorrelatedFlux.dat',/GET_LUN
         for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
         FREE_LUN,unit

         openw,unit,title+'UVW.dat',/GET_LUN
         uvw = midifitsuvw(tag+'.fringes.fits', alt=alt)
         midiFieldAngle, tag+'.fringes.fits',para, phiN
         printf,unit,'UVW(meters) ',uvw, format='(a,2x,3f10.2)'
         printf,unit,'AirMass ',1./sin(alt/!radeg),format='(a, f10.2)'
         printf,unit,'North is ',180-phiN,' degrees counterclockwise from UP',$
         format='(a,2x,f10.2,2x,a)'
      FREE_LUN,unit
      endif
   endfor

device,/close
set_plot,plotname
!p.multi = oldMulti
!x.range = oldxr
!y.range = oldyr
!x.style = oldXY[0]
!y.style = oldXY[1]
RETURN
END

PRO pwd
cd,current=c&print,c
RETURN
END

PRO midiMovie,file,datacol,nmax=nmax,dt=dt,mag=mag,highpass=highpass,doall=doall
;
   if (KEYWORD_SET(doall)) then f = file $
      else begin
         f = STRSPLIT(file,' ',/extract)
         f = f[0]
      endelse
   print,'Processing ',f
   if (N_PARAMS() LT 2) then datacol = 1
   col='DATA'+STRTRIM(STRING(datacol),2)
   print,'Retrieving '+col
   data = oirgetdata(file,col=col)
   nData = N_ELEMENTS(data)
   if (NOT KEYWORD_SET(nmax)) then nmax=ndata
   print,nmax,' Frames '
   data = float(data.(0))
   if (KEYWORD_SET(highpass)) then mdata = TOTAL(data,3)/nData $
      else mdata = 0.*REFORM(data(*,*,0))
   If (NOT KEYWORD_SET(dt)) then dt = .05
   if (NOT KEYWORD_SET(mag)) then mag = 2
   for i=0, nmax-1 do begin tvsclm,data(*,*,i)-mdata,mag
      wait,dt
   endfor
RETURN
END

FUNCTION midiGetCorr,tag,wave=wave
   vis=oirGetVis(tag+'.corr.fits',wave=wave)
   RETURN,vis.visamp
END
FUNCTION midiGetPhase,tag,wave=wave
   vis=oirGetVis(tag+'.corr.fits',wave=wave)
   RETURN,vis.visphi
END
FUNCTION midiGetPhot,tag
   RETURN,oirGetData(tag+'.photometry.fits')
END

FUNCTION midiGetComplex,tag,qual
   data = oirGetData(tag+'.'+qual+'.fits', col='data1')
   data = pseudoComplex(data.data1)
   RETURN,data
END

FUNCTION midiGetDelay,tag, time
   delay = oirGetDelay(tag+'.groupdelay.fits')
   time = REFORM(delay.time)
   time = 86400.*(time - time[0])
   RETURN,delay
END

PRO midiPutDelay, delay, file, ierr=ierr
;    put delay structure into new output file
   dObj = OBJ_NEW('delay')
   dObj->newFile,file, ierr=ierr
   if (ierr NE 0) then RETURN
   outDelay  = REPLICATE(dObj->template(), N_ELEMENTS(delay))
   outDelay.telescope = REFORM(delay.telescope)
   outDelay.time = REFORM(delay.time)
   outDelay.delay=REFORM(delay.delay)/2.9979258e14 ; seconds, not microns
   outDelay.amplitude=REFORM(delay.amplitude)
   dObj->writerows,outDelay
   dObj->close
   OBJ_DESTROY,dObj
RETURN
END
FUNCTION midiGetOpd,tag
   RETURN,oirGetOpd(tag+'.fringes.fits')
END

FUNCTION midiGetGroupData,tag
   z = oirGetData(tag+'.groupdelay.fits')
   RETURN,pseudocomplex(z.data1)
END

FUNCTION midiGetDispersion,tag
   z = OBJ_NEW('fitstable',tag+'.ungroupdelay.fits',extname='DISPERSION')
   zd = z->readrows()
   OBJ_DESTROY,z
RETURN,zd
END

PRO midiPutVis, outFile, wave, visamp, visphi, visamperr, visphierr, flag, $
   uv=uv, target=target, ra=ra, dec=dec, refFile=refFile
;
;   outFile = output file name
;   wave    = wavelength array in microns
;   visamp  = visibility amplitude
;   visphi  = visibility phase (degrees)
;   visamperr = rms visibility error (optional, default=0)
;   visphierr = rms phase error (degrees) (optional, default=0)
;   flag = (1 = good, 0 = bad) (optional, default=1)
;   refFile; get uv,ra,dec,target from another file
;      following override refFile if given
;   uv   = uv coordinates (meters).  If not given defaults to 0!
;   target   = target name; default = blank
;   ra       = target ra;  default = 0.
;   dec      = target dec; default = 0.
;
    vistable = OBJ_NEW('oi_vis', N_ELEMENTS(wave), insname='MIDI', arrname='VLTI')
    template        = vistable->dataTemplate()
    template.visamp = visamp
    if (N_PARAMS() GT 3) then template.visphi    = visphi
    if (N_PARAMS() GT 4) then template.visamperr = visamperr
    if (N_PARAMS() GT 5) then template.visphierr = visphierr
    if (N_PARAMS() GT 6) then template.flag = flag else template.flag[*] = 0
    ra0        = 0.d0
    dec0       = 0.d0
    target0    = "UNKNOWN"
    uv0        = dblarr(2)
    if (KEYWORD_SET(refFile)) then begin
       ra0     = midiGetKeyword('RA', refFile)
       dec0    = midiGetKeyword('Dec', refFile)
       target0 = midiGetKeyword('TARG NAME', refFile)
       uvw     = midifitsuvw(refFile)
       uv0     = uvw[0:1]
    endif
    if (NOT KEYWORD_SET(uv)) then  uv  = uv0
    if (NOT KEYWORD_SET(ra)) then  ra  = ra0
    if (NOT KEYWORD_SET(dec)) then dec = dec0
    if (NOT KEYWORD_SET(target)) then target = target0
    template.ucoord    = uv[0]
    template.vcoord    = uv[1]
    template.target_id = 1

;
   dwave          = abs(wave-shift(wave,1))
   nwave          = N_ELEMENTS(dwave)
   dwave[0]       = dwave[1]
   dwave[nwave-1] = dwave[nwave-2]

   vistable->newfile, outFile
   visFile = vistable->file()
   vistable->writeRows, template
   wavetable = OBJ_NEW('oi_wavelength', wave=wave*1.e-6,band='N')
   wavetable->appendtofile, visFile
   template  = REPLICATE(wavetable->template(), nwave)
   template.eff_wave = wave*1.e-6 ; meters
   template.eff_band = dwave*1.e-6 ; meters
   wavetable->writerows,template
   targtable = OBJ_NEW('oi_target')
   template           = targtable->template()
   template.raep0     = ra
   template.decep0    = dec
   template.target_id = 1
   tt = template.target
   STRPUT,tt,target,0
   template.target    = tt
   targtable->appendtofile, visFile
   targtable->writeRows, template
   targtable->close
   OBJ_DESTROY,vistable
   OBJ_DESTROY,wavetable
   OBJ_DESTROY,targtable
   OBJ_DESTROY,visFile
RETURN
END

PRO midiPutVis2, outFile, wave, vis2data, vis2err, flag, uv=uv
;
;   outFile = output file name
;   wave    = wavelength array in microns
;   vis2data    = squared visibility
;   vis2err = visibility^2 error (optional, default=0)
;   flag = (1 = good, 0 = bad) (optional, default=1)
;   uv   = uv coordinates (meters).  If not given defaults to 0!
;
    vistable = OBJ_NEW('oi_vis2', N_ELEMENTS(wave), insname='MIDI', arrname='VLTI')
    template = vistable->dataTemplate()
    template.vis2data = vis2data
    if (N_PARAMS() GT 2) then template.vis2err = vis2err else $
       template.vis2err[*] = 0.
    if (N_PARAMS() GT 3) then template.flag = flag else template.flag[*] = 1.
    if (NOT KEYWORD_SET(uv)) then uv = [0., 0.]
    template.ucoord = uv[0]
    template.vcoord = uv[1]
;
   vistable->newfile, outFile
   visFile = vistable->file()
   vistable->writeRows, template
   wavetable = OBJ_NEW('oi_wavelength', wave=wave*1.e-6,band='N')
   wavetable->appendtofile, visFile
   wavetable->close
   OBJ_DESTROY,vistable
   OBJ_DESTROY,wavetable
   OBJ_DESTROY,visFile
RETURN
END

PRO midiCleanTag, tag, all=all
   defClean=['compressed','fringes','groupdelay','ungroupdelay']
   for i=0,N_ELEMENTS(defClean)-1 do spawn,'rm '+tag+'.'+defClean[i]+'.fits'
RETURN
END

FUNCTION midiFitsImage, infile, extnu, ierr=ierr
;
   if (N_PARAMS() LT 2) then extnu=1
   im = OBJ_NEW('fitsimageextension', infile, extnu, ierr=ierr)
   if (ierr NE 0) then begin
      midiprinterrmsg
      RETURN,0
   endif
   image = im->readImage(ierr=ierr)
   if (ierr NE 0) then midiprinterrmsg
   OBJ_DESTROY,im
RETURN, image
END



;********************************************************
;#class#
;faintUtilities
;#description#
;general, mostly undocumented midi utilities
;#end_class#
;*******************************************************


PRO faintVisPipe, base, file, mask=mask, smooth=smooth, gsmooth=gsmooth, $
   noAve=noAve, minopd=minopd, maxopd=maxopd, $
   twopass=twopass,msmooth=msmooth,noc=noc,ngrad=ngrad
;    set defaults  and create character string versions
   if (n_elements(ngrad) LE 0) then  ngrad  = 2
   ng = ' -ngrad '+string(ngrad)+' '
   if (NOT KEYWORD_SET(smooth)) then smooth = 50.
   sm = '-smooth '+string(smooth)
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 0.1 ; seconds
   gsm     = '-smooth '+string(gsmooth)+' '
   psmooth = gsmooth
   if (NOT KEYWORD_SET(maxopd)) then maxopd = 100.
   mxopd = '-maxopd '+strtrim(maxopd,2)+' ' 
   if (KEYWORD_SET(msmooth)) then msm = '-medsmooth '+string(msmooth) $
      else msm = ' -medsmooth '+string(gsmooth)

   if (NOT KEYWORD_SET(noAve)) then begin
      doAve = " -removeAverage" 
      if (KEYWORD_SET(minopd)) then mo = ' -minopd '+strtrim(minopd,2)+' ' $
         else mo = "-minopd 20. "
   endif else begin
      doAve=" "
      if (KEYWORD_SET(minopd)) then mo = '-minopd '+strtrim(minopd,2)+' '$
         else mo = " " 
   endelse
;    where is executable?
   cbin = getenv('vltiCbin')+'/'

;    parse input files to see if both interferometry 
   midiGetIns,file,shutter=shutter, grism=grism, object=object, opt1=opt1,$
      ierr=ierr
   if (ierr NE 0) then return
   shutter = STRTRIM(shutter,2)
   grism   = STRTRIM(grism,2)
   if (shutter NE 'ABOPEN') then begin
      print,'I need ABOPEN interferometry data'
      RETURN
   endif
   if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
      print,'I only know how to deal with prism or grism data'
      RETURN
   endif

;     prepare to convert from times to frames vvs.

   dit = MAX(midiGetKeyword('DET DIT',file))
   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
   print,' input vis files have ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(file)))),' input frames'
if (NOT KEYWORD_SET(noc)) then begin      
   command = cbin+'oir1dCompressData "'+file+'" '+mask+' '+base+'.compressed.fits'
   print,'   compressing raw data '
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oir1dCompressData failed'
      print,command
      RETURN
   endif
   print,'   compression complete'
endif
   command = cbin+'oirFormFringes '+base+'.compressed.fits '+base+'.fringes.fits '+sm+$
      ' '+doAve
   print,'   forming fringes '
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirFormFringes failed'
      print,command
      RETURN
   endif
   print,'   fringing complete'

   command = cbin+'oirRotateInsOpd '+base+'.fringes.fits '+base+'.insopd.fits'
   print,'   removing piezo OPD'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirInsOpd failed'
      print,command
      RETURN
   endif
   print,'   insOpd complete'


   if (KEYWORD_SET(twopass)) then outfile=base+'.groupdelay1.fits ' $
      else outfile=base+'.groupdelay.fits '
   command = cbin+'oirFGroupDelay '+base+'.insopd.fits '+outfile $
      +gsm+mxopd+msm+ng
   print,'   first pass GroupDelay'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirFGroupDelay failed'
      print,command
      RETURN
   endif
   print,'   first pass GroupDelay complete'

   if (KEYWORD_SET(twopass)) then begin
;     gsf = ' -smooth '+string(gsmooth/dit)+' '
      gsf = ' -smooth '+string(gsmooth)+' '
      command = cbin+'oirPowerDelay '+base+'.groupdelay1.fits '+base+'.powerdelay.fits ' $
         +gsf+' -ampsmooth 8. '
      print,'   look for total power fringe'
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'oirPowerDelay failed'
         print,command
         RETURN
      endif
      print,'   power search complete'

      command = cbin+'oirFGroupDelay '+base+'.insopd.fits '+base+'.groupdelay.fits ' $
         +ng+gsm+' -maxopd 40 '+' -first '+base+'.powerdelay.fits '+msm
      print,'   second pass GroupDelay'
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'oirFGroupDelay failed'
         print,command
         RETURN
      endif
      print,'   second pass GroupDelay complete'

   endif ; two passes
   gsf = ' -smooth '+string(psmooth)+' '
   command = cbin+'oirRotateGroupDelay '+base+'.fringes.fits '$
      +base+'.groupdelay.fits ' +base+'.ungroupdelay.fits '$
      +gsf+'  -ngrad 3 '
   print,'   Rotate GroupDelay'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirRotateGroupDelay failed'
      print,command
      RETURN
   endif
   print,'   Rotate GroupDelay complete'

   command = cbin+'oirAutoFlag '+base+'.fringes.fits '+base+'.groupdelay.fits ' $
      +base+'.flag.fits -maxopd 100 -minopd 10 '
   print,'   Auto Flagging Data'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirAutoFlag failed '
      print,command
      RETURN
   endif
   print,'   Auto Flagging  complete'

   command = cbin+'oirAverageVis '+base+'.ungroupdelay.fits '+base+'.flag.fits '+base+'.corr.fits'
   print,'   Computes average correlation'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirAverageVis failed '
      print,command
      RETURN
   endif
   print,'   Visibility Averaging  complete'

   erase
   !p.multi=[0,1,2,0,0]
   midiDelayPlot,base,object+':'+file,twopass=twopass
   v=oirgetvis(base+'.corr.fits',wave=wave)
   plot,wave,midiCvis(v)

   !p.multi=0
RETURN
END

PRO faintPhotoPipe, base, files, mask=mask, dSky=dSky, curve=curve
   if (NOT KEYWORD_SET(dSky)) then dSky = 0
   ds = " -dSky "+ STRTRIM(dSky,2)
   cbin = getenv('vltiCbin')+'/'

   print,' input photA file has ',$
      FIX(total(midigetkeyword('DET NDIT',midiextractfiles(files[0])))),$
      ' frames '

   print,' input photB file has ',$
      FIX(total(midigetkeyword('DET NDIT',midiextractfiles(files[1])))),$
      ' frames '

   command = cbin+'oirChopPhotoImages -in "' + files[0]+'" -out ' $
      +base+'.Aphotometry.fits '+ds
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'ChopPhotoImages A failed'
      print,command
      RETURN
   endif
   command = cbin+'oirChopPhotoImages -in "' + files[1]+'" -out ' $
      +base+'.Bphotometry.fits '+ds
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'ChopPhotoImages B failed'
      print,command
      RETURN
   endif
;               get default masks if not specified  
   midiGetIns,files[0],shutter=shutter, grism=grism, opt1=opt1,ierr=ierr
   if (ierr NE 0) then RETURN
   grism = STRTRIM(grism,2)
   opt1  = STRTRIM(opt1,2)

   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif

   command = cbin+'oirMakePhotoSpectra -A ' + base+'.Aphotometry.fits -B '$
      +base+'.Bphotometry.fits -out '+base+'.photometry.fits -mask '+mask+' -shift 0'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirMakePhotoSpectra  failed'
      print,command
      RETURN
   endif
   p    = oirgetdata(base+'.photometry.fits')
   wave = oirgetwavelength(base+'.photometry.fits')
   plot,wave,p[5].data1
RETURN
END

PRO faintPipe, base, inFiles, mask=mask, smooth=smooth, $
   gsmooth=gsmooth,dSky=dSky, noAve=noAve, minopd=minopd, maxopd=maxopd, $
   twopass=twopass, msmooth=msmooth,ngrad=ngrad

;    parse input files to see if both visibility and photometry present

   d = inFiles
   if (N_ELEMENTS(d) NE 3) then begin
      print,'You need to specify two visibility file(s) and one' +$
         ' interferometry file.'
      RETURN
   endif

   fileList = REPLICATE(-1,3)
   for i=0,2 do begin
      midiGetIns,d[i],shutter=shutter, grism=grism, object=object, opt1=opt1,$
         ierr=ierr
      if (ierr NE 0) then RETURN
      shutter = STRTRIM(shutter,2)
      grism = STRTRIM(grism,2)
      if (shutter EQ 'AOPEN') then       fileList[0] = i $
      else if (shutter EQ 'BOPEN') then  fileList[1] = i $
      else if (shutter EQ 'ABOPEN') then fileList[2] = i
      if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
         print,'I only know how to deal with prism or grism data'
         RETURN
      endif
   endfor

   if (TOTAL(filelist eq (-1)) NE 0) then begin
      print,'You need to specify two visibility file(s) and one' +$
         ' interferometry file.'
      RETURN
   endif
;         look for default masks if not specified
   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif

   faintVisPipe, base, inFiles[fileList[2]], mask=mask, $
      smooth=smooth, gsmooth=gsmooth, $
      noAve=noAve, minopd=minopd, maxopd=maxopd, $
      twopass=twopass,msmooth=msmooth,ngrad=ngrad


   faintPhotoPipe, base, inFiles[fileList[0:1]], mask=mask, dSky=dSky

   cbin = getenv('vltiCbin')+'/'
   spawn,cbin+'oirRedCal '+base, EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirRedCal failed'
      print,command
      RETURN
   endif
   midiInsVisPlot,base,object+':'+d[filelist[0]]
   !p.multi = 0
RETURN
END

PRO faintSearch, base, file, mask=mask, smooth=smooth, gsmooth=gsmooth, $
   noAve=noAve, minopd=minopd, maxopd=maxopd, ngrad=ngrad,$
   msmooth=msmooth
;    set defaults  for fringe smooth and group delay smooth
   if (n_elements(ngrad) LE 0) then  ngrad  = 2
   ng = ' -ngrad '+string(ngrad)+' '
   if (NOT KEYWORD_SET(smooth)) then smooth = 50.
   sm = '-smooth '+string(smooth)
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 0.1 ; seconds
   gsm     = '-smooth '+string(gsmooth)+' '
   psmooth = gsmooth
   if (NOT KEYWORD_SET(maxopd)) then maxopd = 100.
   mxopd = '-maxopd '+strtrim(maxopd,2)+' ' 
   if (KEYWORD_SET(msmooth)) then msm = '-medsmooth '+string(msmooth) $
      else msm = ' -medsmooth '+string(gsmooth)

   if (NOT KEYWORD_SET(noAve)) then begin
      doAve = " -removeAverage" 
      if (KEYWORD_SET(minopd)) then mnopd = ' -minopd '+strtrim(minopd,2)+' ' $
         else mnopd = "-minopd 20. "
   endif else begin
      doAve=" "
      if (KEYWORD_SET(minopd)) then mnopd = '-minopd '+strtrim(minopd,2)+' '$
         else mnopd = " " 
   endelse
;    where is executable?
   cbin = getenv('vltiCbin')+'/'

;    parse input files to see if both interferometry 
   midiGetIns,file,shutter=shutter, grism=grism, object=object, opt1=opt1,$
      ierr=ierr
   if (ierr NE 0) then return
   shutter = STRTRIM(shutter,2)
   grism   = STRTRIM(grism,2)
   if (shutter NE 'ABOPEN') then begin
      print,'I need ABOPEN interferometry data'
      RETURN
   endif
   if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
      print,'I only know how to deal with prism or grism data'
      RETURN
   endif

;     prepare to convert from times to frames vvs.

   dit = MAX(midiGetKeyword('DET DIT',file))
   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
   print,' input vis files have ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(file)))),' input frames'
   command = cbin+'oir1dCompressData "'+file+'" '+mask+' '+base+'.compressed.fits'
   print,'   compressing raw data '
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oir1dCompressData failed'
      print,command
      RETURN
   endif
   print,'   compression complete'
   command = cbin+'oirFormFringes '+base+'.compressed.fits '+base+'.fringes.fits '+sm+$
      ' '+doAve
   print,'   forming fringes '
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirFormFringes failed'
      print,command
      RETURN
   endif
   print,'   fringing complete'

   command = cbin+'oirRotateInsOpd '+base+'.fringes.fits '+base+'.insopd.fits'
   print,'   removing piezo OPD'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirInsOpd failed'
      print,command
      RETURN
   endif
   print,'   insOpd complete'


   command = cbin+'oirFGroupDelay '+base+'.insopd.fits '+base+'.groupdelay.fits ' $
      +gsm+mxopd+msm+ng+' -search '
   print,'   first pass GroupDelay'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirFGroupDelay failed'
      print,command
      RETURN
   endif
   print,'   first pass GroupDelay complete'
   erase
   midiDelayAmpPlot1,base,object+':'+midiFirstFile(file), gsmooth=gsmooth/dit,/time
RETURN
END

PRO midiSearch, base, files, mask=mask, smooth=smooth, gsmooth=gsmooth, $
  dAve=dAve, maxopd=maxopd,minopd=minopd
;    set defaults  for fringe smooth and group delay smooth
   if (NOT KEYWORD_SET(smooth))  then smooth = 4.
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 4.
   if (KEYWORD_SET(dAve))        then doAve = " -removeAverage" else doAve=""
   if (NOT KEYWORD_SET(maxopd))  then maxopd = 400.
   if (NOT KEYWORD_SET(minopd))  then $
      if (KEYWORD_SET(dave)) then minopd = 0. else minopd=20.
   mxopd = ' '+strtrim(maxopd,2)+' ' 
   mnopd = ' '+strtrim(minopd,2)+' ' 
;    where is executable?
   cbin = getenv('vltiCbin')+'/'
;    if files not specified, bring up gui
   if (N_PARAMS() LT 2) then d = midiGuis() else d=files
;    parse input files to see if both visibility and photometry present
   midiGetIns,d[0],shutter=shutter, grism=grism, object=object, opt1=opt1
   shutter = STRTRIM(shutter,2)
   grism = STRTRIM(grism,2)
   if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
      print,'I only know how to deal with prism or grism data'
      RETURN
   endif

   if (N_ELEMENTS(mask) LE 0) then begin
;      mask='$midiutil/prismnewmask.fits'
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
   command = cbin+'dispSearch "'+d[0]+'" '+mask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth) + doAve + mxopd +mnopd 
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispSearch failed'
      print,command
      RETURN
   endif
   erase
   midiDelayAmpPlot1,base,object+':'+d[0], gsmooth=gsmooth
RETURN
END


PRO midiFringeImage, base, file, nodelete=nodelete, smooth=smooth, noAve=noAve

   if (NOT KEYWORD_SET(smooth)) then smooth = 50.
   sm = '-smooth '+string(smooth)

   if (NOT KEYWORD_SET(noAve)) then doAve = " -removeAverage" $
   else   doAve=" "
;    where is executable?
   cbin = getenv('vltiCbin')+'/'

   command = cbin+'oirFormFringes "'+file+'" '+base+'.imagefringes.fits '$
      +sm+ ' '+doAve
   print,'   forming fringes '
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirFormFringes failed'
      print,command
      RETURN
   endif
   print,'   fringing complete'

   command = cbin+'oirRotateGroupDelay '+base+'.imagefringes.fits '+$
      base+'.groupdelay.fits '+' '+base+'.imageungroupdelay.fits -phase '+$
      base+'.ungroupdelay.fits'
   print,'   rotate fringes '
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirRotateGroupDelay failed'
      print,command
      RETURN
   endif
   print,'   rotation complete'

   command = cbin+'oirAverageVisImage '+base+'.imageungroupdelay.fits '+$
       base+'.flag.fits '+base+'.fringeimages.fits'
   print,'   averaging data '
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirAverageVisImage failed'
      print,command
      RETURN
   endif
   print,'   averaging complete'

   if (NOT KEYWORD_SET(nodelete)) then spawn,'rm '+base+'.imageungroupdelay.fits'
   if (NOT KEYWORD_SET(nodelete)) then spawn,'rm '+base+'.imagefringes.fits'
RETURN
END

PRO midiMakeMask, base, rawFiles, initialMask=initialMask,$
   shiftOnly=shiftOnly, smooth=smooth, gsmooth=gsmooth, $
   noAve=noAve, factor=factor, noDelete=noDelete
;       set defaults
   if (NOT KEYWORD_SET(factor)) then factor = 1.0
   if (NOT KEYWORD_SET(shiftOnly)) then shift = 0 else shift = 1
;        first pass to get atmospheric group delay
   print,'   First Pass with initial Mask to get atmospheric OPD '
   faintVisPipe, base, rawFiles, mask=initialMask, smooth=smooth, $
      gsmooth=gsmooth, noAve=noAve

   print,'   Now calculate Fringe Image '
   midiFringeImage, base, rawFiles, smooth=smooth
;            get wavelength information for plots, crosscorrelation  etc
   wave       = oirGetWavelength(initialMask)
   wrange = WHERE(wave GT 8.5 AND (wave LT 12.5))
   wmin   = MIN(wrange)
   wmax   = MAX(wrange)
;            for weak sources it may be best to only shift mask
   if (shift eq 0) then begin
      print,'   Creating and storing mask ',base+'.srcmask.fits'
      maskimage = midiMskFit(base+'.fringeimages.fits', base, factor=factor)
   endif else begin
      print,'   Measuring shift from input mask to fringe image'
;            get data from disk and find which images we want from struct
      oldMask    = oirGetData(initialMask)
      fringeMask = oirGetData(base+'.fringeimages.fits')
      tags       = TAG_NAMES(oldMask)
      ftags      = TAG_NAMES(fringeMask)
      dTags      = WHERE(STRPOS(tags,'DATA') GE 0)
      nTags      = N_ELEMENTS(dTags)
      dPos       = intarr(2)
      fPos       = intarr(2)
      if (nTags EQ 2) then begin
         dPos[0] = WHERE(tags  EQ 'DATA1')
         dPos[1] = WHERE(tags  EQ 'DATA2')
         fPos[0] = WHERE(ftags EQ 'DATA1')
         fPos[1] = WHERE(ftags EQ 'DATA2')
      endif else if (nTags EQ 4) then begin
         dPos[0] = WHERE(tags  EQ 'DATA2')
         dPos[1] = WHERE(tags  EQ 'DATA3')
         fPos[0] = WHERE(ftags EQ 'DATA2')
         fPos[1] = WHERE(ftags EQ 'DATA3')
      endif else begin
         print,'huh?? number of data channels = ',nTags
         RETURN
      endelse
;            find maximum of cross correlation with images
;            shifted in y-direction
      yshift = FLTARR(2)
      nshift = 4
      xx     = indgen(2*nshift+1)-nshift
      yy     = FLTARR(2*nshift+1)
      for i=0,1 do begin
         dold = oldMask.(dPos[i])[wmin:wmax,*]
         dff  = FLOAT(pseudoComplex(fringeMask.(fPos[i]),/no))
         dff  = dff[wmin:wmax,*]
         if (i eq 1) then dff = -dff ; phase of I2 is negative
         for j=0,2*nshift do yy[j] = $
            TOTAL(dff*shift(dold,0,xx[j]))
         yshift[i] = pk(xx, yy)
         oldMask.(dPos[i]) = pshift(oldMask.(dPos[i]), 0, yshift[i])
      endfor
      print,'   Shifted masks by ',yshift,' pixels'
;              write shifted mask to disk
      oirNewData, initialMask,base+'.srcmask.fits', oldMask,/copy
   endelse
;       display new/old masks (.data2) and fringe image
   oldMask = oirGetData(initialMask, col='data2')
   oldMask = oldMask.data2[100,*]

   newMask = oirGetData(base+'.srcmask.fits',col='data2')
   newMask = newMask.data2[100,*]

   fringeImage = oirGetData(base+'.fringeimages.fits')
;  fringeImage = pseudoComplex(fringeImage.data1,/no) $
;     - pseudoComplex(fringeImage.data2,/no)
   fringeImage = - pseudoComplex(fringeImage.data2,/no)
   fringeImage = TOTAL(FLOAT(fringeImage[wmin:wmax,*]),1)

   plot,oldMask,thick=2,yrange=[-.2,1.2]
   oplot,newMask
   oplot,fringeImage/max(fringeImage), col=255
;         unless forbidden, delete the intermediate files
   if (NOT KEYWORD_SET(noDelete)) then begin
      print,'    Deleting intermediate files '
      spawn, 'rm '+base+'.compressed.fits'
      spawn, 'rm '+base+'.fringes.fits'
      spawn, 'rm '+base+'.insopd.fits'
      spawn, 'rm '+base+'.groupdelay.fits'
      spawn, 'rm '+base+'.ungroupdelay.fits'
      spawn, 'rm '+base+'.flag.fits'
      spawn, 'rm '+base+'.corr.fits'
      spawn, 'rm '+base+'.fringeimages.fits'
   endif
RETURN
END
