;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;fft1d
;#description#
;1-d FFT routines including gridding and filtering
;#end_class#
;*******************************************************

PRO fft1D__DEFINE
;********************************************************
;#structure#
; fft1d
;#inheritance#
; NONE
;#description#
;data for 1d non-uniform spaced fft class
;#structure_text#
p={fft1d, $ ;
convFn:'gauss',  $  ; convolving function name
interpFn:OBJ_NEW(),  $ ; convolving function as interpolation object
convParm:fltarr(2),  $ ; convolving function specification parameters
gridData:fltarr(3),  $ ; grid spacing parameters, x0, dx, nx
grid:PTR_NEW(),      $ ; gridded data in kspace 
grid2:PTR_NEW(),     $ ; gridded 2nd data in kspace
kGrid:PTR_NEW(),     $ ; values of k corresponding to grid parm
filter:PTR_NEW(),    $ ; filter in k-space
gridCorr:PTR_NEW()   $ ; function to correct for gridding convolution
}
;#end_structure#
;********************************************************
RETURN
END

PRO fft1D::cleanup
;destructor for class
   PTR_FREE, self.grid
   PTR_FREE, self.grid2
   PTR_FREE, self.gridCorr
   PTR_FREE, self.kGrid
   PTR_FREE, self.filter
RETURN
END

FUNCTION makeHermit, z
; make z hermitian by adding it to its flipped conjugate.
; assume zero point is in the middle
RETURN, 0.5*(z + shift(conj(REVERSE(z)),1))
END

PRO dbleRealFft, real1, real2, fft1, fft2, direction
; perform two real ffts of the same length inside one complex fft by sticking one input
; in the imaginary part of the input and then using hermitian
; properties to fish them out of the result
; foward output has zero point in middle, i.e. NOT IDL standard
;
   if (direction GE 0) then begin
; stick them in and take fft
      tFft = fft(real1 + complex(0,1.)*real2,1)
;shift zero point
      tFft = shift (tFft, N_ELEMENTS (tFft) / 2) 
; fish out two parts by insisting on hermitian property
      fft1 = makeHermit(tFft)
      fft2 = makeHermit(complex(0,-1.)*tFft)
   endif else begin
      tFft = makeHermit(fft1) + complex(0,1.)*makeHermit(fft2)
      tfft = shift(tFft, -N_ELEMENTS(tFft)/2)
      tFft = fft(tFft, -1)
      real1 = float(tFft)
      real2 = imaginary(tFft)
   endelse
RETURN
END

FUNCTION fft1d::init, gridData, cFunction=cFunction, fnParm=fnParm, filter=filter
;constructure for fft1d class
   if (KEYWORD_SET(cFunction)) then begin
      self.convFn = 'interp' 
      self.interpObj = cFunction
   endif else  self.convFn = 'gauss'
   self.gridData = gridData
   self.grid  = PTR_NEW(fltarr(gridData[2]))
   self.grid2 = PTR_NEW(fltarr(gridData[2]))
   self.kGrid = PTR_NEW((findgen(gridData[2])-gridData[2]/2.)*2.*!pi/(gridData[2] * gridData[1]))
;if not specified set default convParm
   if (KEYWORD_SET(fnParm)) then self.convParm = fnParm else self.convParm = [.85, 3.]
   self.convParm[0] = self.convParm[0] > .5
   self.convParm[1] = self.convParm[1] > 2.
;correction for convolution
   nX = self.gridData[2]
   self.gridCorr = PTR_NEW(exp(.5*(self.convParm[0] * shift(findgen(nX)-nX/2,-nX/2) *2*!pi/nX)^2 < 5))
   if (KEYWORD_SET(filter)) then self.filter = PTR_NEW(filter)
RETURN,1
END

FUNCTION fft1d::grid & RETURN,*self.grid & END
FUNCTION fft1d::grid2 & RETURN,*self.grid2 & END
FUNCTION fft1d::coord & RETURN,self.gridData[0] + self.gridData[1]*findgen(self.gridData[2]) & END
FUNCTION fft1d::gridCorr & RETURN,*self.gridCorr & END
FUNCTION fft1d::kGrid & RETURN,*self.kGrid & END
FUNCTION fft1d::filterData & if (PTR_VALID(self.filter)) then RETURN, *self.filter else RETURN,0 & END
PRO fft1d::set, filter=filter
; store a k-space filter internally
   if (KEYWORD_SET(filter)) then begin
      PTR_FREE, self.filter
      self.filter = PTR_NEW(filter)
   endif
RETURN
END

PRO fft1d::makeXfilter, xcoord, filterData
;filter is specified in x-space instead of kspace
   PTR_FREE, self.filter
   self.filter = PTR_NEW(abs(self->gridFft(xCoord, filterData)))
RETURN
END
;
PRO fft1d::gridData, xPos, data, data2
;grid the data onto the stored grid
   do2 = N_ELEMENTS(data2) GT 0
   grid = 0*(*self.grid)
   if (do2) then grid2 = grid
   gMax = N_ELEMENTS(*self.grid) - 1
   nData = N_ELEMENTS(data)
;calculate sigma of exponential (pixels)
   sig = 0.5 / (self.convParm[0])^2
   dx = self.convParm[1]
   xCoord = (xPos - self.gridData[0])/self.gridData[1]
   xMin = ceil(xCoord - dx) > 0
   xMax = floor(xCoord + dx) < gMax
   for iD = 0, nData -1 do begin
      if ((xCoord[iD] LT 0.) OR (xCoord[iD] GT gMax)) then GOTO, skipPoint
      xGrid = xMin[iD] + indgen (xMax[iD] - xMin[iD]+1)
      xFn = exp(-sig*(xGrid-xCoord[iD])^2 )
      grid[xGrid] = grid[xGrid] + data[iD] * xFn/total(xFn)
      if (do2) then grid2[xGrid] = grid2[xGrid] + data2[iD] * xFn/total(xFn)
skipPoint:
   endfor
   (*self.grid) = grid
   if (do2) then *self.grid2 = grid2
RETURN
END

FUNCTION fft1D::multiGrid, xCoord, data
;grid a large number of sequences on the same grid
;
   sData = SIZE(data, /dimensions)
   nOut = 1
   nData = sData[0]
   for i=1, N_ELEMENTS(sData)-1 do nOut = nOut * sData[i]
   data = REFORM(data, nData, nOut,/overwrite)
   nGrid = self.griddata[2]
   outData = fltarr(nGrid, nOut)
   grid = 0*(*self.grid)
   sig = 0.5 / (self.convParm[0])^2
   dx = self.convParm[1]
   xCoord = (xPos - self.gridData[0])/self.gridData[1]
   xMin = ceil(xCoord - dx) > 0
   xMax = floor(xCoord + dx) < gMax
   for iD = 0, nData -1 do begin
      if ((xCoord[iD] LT 0.) OR (xCoord[iD] GT gMax)) then GOTO, skipPoint
      xGrid = xMin[iD] + indgen (xMax[iD] - xMin[iD]+1)
      xFn = exp(-sig*(xGrid-xCoord[iD])^2 )
      for iOut = 0, nOut-1 do outData[xGrid,iOut] = outData[xGrid,iOut] $
         + data[iD, iOut] * xFn/total(xFn)
skipPoint:
   endfor
   snData = N_ELEMENTS(sData) 
   CASE snData of
      1: 
      2: BEGIN
         data = REFORM(data, nData, sData[1], /overwrite)
         outData = REFORM(outData, nGrid, sData[1], /overwrite)
      END
      3: BEGIN
         data = REFORM(data, nData, sData[1], sData[2],/overwrite)
         outData = REFORM(outData, nGrid, sData[1],sData[2], /overwrite)
      END
      4: BEGIN
         data = REFORM(data, nData, sData[1], sData[2], sData[3], /overwrite)
         outData = REFORM(outData, nGrid, sData[1],sData[2], sData[3], /overwrite)
      END
   ENDCASE
RETURN, outData
END

FUNCTION fft1D::gridFfT, xCoord, data, data2, fft2
;grid real data and if desired also 2nd data array on same grid, and FFT them
;when finished, 0 freq pt of fft is in middle of array(s); NOT IDL standard
;should I do 2 real ffts in one complex array?
   do2 = N_ELEMENTS(data2) GT 0
   nx = self.gridData[2]
   self->gridData, xCoord, data, data2
;shift to be applied after fft to compensate for fact that 0pt in x-space
;is not the 1st element of array, but rather given by gridData
   zerophase = exp(complex(0,1.)*2*!pi*self.gridData[0]*(findgen(nx)-nx/2)/$
      (nx*self.griddata[1]))
;save time doing 2 reals ffts at once if required
   if(do2) then begin
      dbleRealFft, *self.grid, *self.grid2, fft1, fft2, 1
      fft2 = shift((*self.gridCorr < 15), nx/2)*fft2 * zerophase
      RETURN, shift((*self.gridCorr < 15), nx/2)*fft1 * zerophase
   endif
RETURN,shift(fft(*self.grid,1)* (*self.gridCorr < 15), nx/2) * zerophase
END

FUNCTION fft1d::cFft, xCoord, data
;do grid fft assuming data is complex
rfft = self->gridFft(xCoord, float(data), imaginary(data), cfft)
RETURN,rfft + COMPLEX(0,1) * cfft
END


FUNCTION fft1d::multiGridFft, xCoord, data
;do lots of FFTs on the same grid
   sData = SIZE(data, /dimensions)
   nOut = 1
   nData = sData[0]
   nGrid = self.griddata[2]
   for i=1, N_ELEMENTS(sData)-1 do nOut = nOut * sData[i]
   gridData = self->multiGrid(xCoord, data)
   gridData = REFORM(gridData, nGrid, nOut,/overwrite)
   outData = complexArray(nGrid, nOut)
   zerophase = exp(complex(0,1.)*2*!pi*self.gridData[0]*(findgen(nx)-nx/2)/$
      (nx*self.griddata[1]))
   corr = shift((*self.gridCorr < 15), nGrid/2) * zerophase
   for iOut = 0, nOut-1, 2 do begin
      dbleRealFft, gridData[*, iOut], gridData[*, iOut+1], fft1, fft2, 1
      outData(*, iOut)   = corr * fft1
      outData(*, iOut+1) = corr * fft2
   endfor
   if ((nOut mod 2) EQ 1) then begin
      outData[*, nOut-1] = shift(fft(gridData[*,nOut-1],1)* (*self.gridCorr < 15), nx/2) * zerophase
;...reform...
   endif
RETURN, outData
END

FUNCTION fft1d::DFFT, xCoord, data
   kGrid = *self.kGrid
   output = complexarr(self.gridData[2])
   ii = complex(0,1.)
   for iP = 0, N_ELEMENTS(xCoord)-1 do output = output + data[iP]*exp(ii * xCoord[iP] * kGrid)
RETURN, output
END

FUNCTION fft1d::xShift, xCoord, data, dx, direct=direct, filter=filter
;shift input data by dx using fourier transform and phase shift
  if (KEYWORD_SET(direct)) then fftData = self->dfft(xcoord, data) $
      else fftData = self->gridFft(xcoord, data)
  if ((KEYWORD_SET(filter)) AND (PTR_VALID(self.filter))) then fftData =fftData*(*self.filter)
  fData = fftData * exp(complex(0.,1.) * (*self.kGrid) * dx)
RETURN, shift(float(fft(shift(fData, -self.gridData[2]/2),-1)),self.gridData[2]/2)
END

FUNCTION fft1d::kShift, fData, dx
;same as xShift but assume that 1st fft has already been done and filtered
   fftData = fData * exp(complex(0.,1.) * (*self.kGrid) * (dx))
RETURN, shift(float(fft(shift(fftData, -self.gridData[2]/2),-1)),self.gridData[2]/2)
END

FUNCTION fft1d::filter, xCoord, data, weight, fftData, fftWeight, direct=direct
;take FT of xCoord and Data, multiply by internally stored filter function
;and then FFT back.  Do the same to a function that is 1 at each of the
;measured points.  The back FFT of this fn is weight.  fftData and fftWeight
;are the transforms of the input and weighting functions after multiplication
;by the filter.
;  if direct is set, do a direct Ft, otherwise a gridded FFT
   wtData = REPLICATE(1, N_ELEMENTS(data))
   nx = self.griddata[2]
   if (KEYWORD_SET(direct)) then begin
      fftData = self->dfft(xcoord, data) 
      fftWeight = self->dfft(xcoord, wtData)
   endif else fftData = self->gridFft(xcoord, data, wtData, fftWeight)
   if (PTR_VALID(self.filter)) then begin
      fftD= fftData * (*self.filter)
      fftW= fftWeight * (*self.filter)
   endif else begin
      fftD = fftData
      fftW = fftWeight
   endelse
   zerophase = exp(complex(0,-1.)*2*!pi*self.gridData[0]*(findgen(nx)-nx/2)/$
      (nx*self.griddata[1]))
   dbleRealFft, outData, weight, fftD*zerophase, fftW*zerophase, -1
RETURN, outData
END

FUNCTION fft1d::Cfilter, xCoord, data, weight, fftData, fftWeight, direct=direct
;take FT of xCoord and Data, multiply by internally stored filter function
;and then FFT back.  Do the same to a function that is 1 at each of the
;measured points.  The back FFT of this fn is weight.  fftData and fftWeight
;are the transforms of the input and weighting functions after multiplication
;by the filter.  Different from ::filter because full complex return ffts are kept
;  if direct is set, do a direct Ft, otherwise a gridded FFT
   wtData = REPLICATE(1, N_ELEMENTS(data))
   nx = self.griddata[2]
   if (KEYWORD_SET(direct)) then begin
      fftData = self->dfft(xcoord, data) 
      fftWeight = self->dfft(xcoord, wtData)
   endif else fftData = self->gridFft(xcoord, data, wtData, fftWeight)
   if (PTR_VALID(self.filter)) then begin
      fftD= fftData * (*self.filter)
      fftW= fftWeight * (*self.filter)
   endif else begin
      fftD= fftData 
      fftW= fftWeight
   endelse
  zerophase = exp(complex(0,-1.)*2*!pi*self.gridData[0]*(findgen(nx)-nx/2)/$
     (nx*self.griddata[1]))
;   outData   = SHIFT(fft(shift(fftD,-nx/2),-1),nx/2)
;  weight    = SHIFT(fft(shift(fftW,-nx/2),-1),nx/2)
   weight    = fft(shift(fftW*zerophase,-nx/2),-1)
   outData   = fft(shift(fftD*zerophase,-nx/2),-1)
RETURN, outData
END
