/*
 * Copyright 2005 University of Leiden.
 *
 * This file is part of MIA+EWS.
 *
 * MIA+EWS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MIA+EWS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MIA; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include "oirSupport.h"
#include "fitsio.h"
#include "imaging_data.h"
#include "imaging_fdata.h"
#define NBIN 5

/**********************************************************
 * Usage:
 *    oirChopPhotoImages "AFiles" "BFiles" outFile -nDrop nDrop -dSky dSky
 *
 *    Output is a IMAGING_DATA table in a fits file with
 *    2*NBINS+2 rows, each with 2 DATA arrays for detector windows 1 and 2
 *    The rows alternate being A-side and B-side data 
 *
 *    The first 2 rows are  the grand average of all data.  The
 *    next rows are NBINS subaverages, useful in estimate RMS errors.
 *    All images are in instrumental units: ADUs/s/pixel
 *
 *    nDrop is the number of frames to drop after a shift from Sky to Target
 *    dSky represents the shift of the sky subtraction zones from their nominal position
 ************************************************************* */
/***************************************************************
 * read through all raw files associated with one shutter setting
 * separate target and sky frames and accumulate them in
 * given arrays; divide by number of frames
 * INPUTS:
 *    char **string     array of file names
 *    int    nFiles     number of file names
 *    int    chan1      0-based number of 1st interferometric channel
 *    int    chan2      0-based number of 2nd interferometric channel
 *    int    nDrop      drop this many points after change of chop phase
 *    int    npix1      no. pixels in array in 1st channel
 *    int    npix2      no. pixels in array in 2nd channel
 *    float* target1    array to receive target1 data
 *    float* target2    array to receive target2 data
 *    float* sky1       array to receive target1 data
 *    float* sky2       array to receive target2 data
 *    int  * nTarget    number of target frames
 *    int  * nSky       number of sky frames
 **************************************************************** */

void unChopData(char **fileList, int nFiles, int chan1, int chan2, 
   int nDrop, long npix1, long npix2, long nx1, long ny1, long nx2, long ny2, 
   float *profiles1, float *profiles2, long ntotalrows,
   float *target1, float *target2, 
   float *sky1, float *sky2, int *nTarget, int *nSky,
   int *nCycle, int *ierr) {

   fitsfile *inFile;
   imaging_data_table_def *idata;
   imaging_data_row *inData;
   int iFile;

   long iRow, nRow, iTotalRow=0;
   int i;
   int TN, SN;
   long iPix;

   char Sky = 'S';
   char Target = 'T';
   char targ, oldTarg;
   int sinceChange;
   float mjd;
   int old;

   *ierr    = 0;
   *nCycle = 0;
   TN = 0;
   SN = 0;
   oldTarg = ' ';

/*
      loop over files, read rows, read target types, add data
      into appropriate accumulators 
*/

   for (iFile =0; iFile <nFiles; iFile++) {
/*
      open file, open table, create table row
*/
      i = fits_open_file(&inFile, fileList[iFile], READONLY, ierr);
      if (*ierr !=0) {
         printf("error %i opening file %s\n", *ierr, fileList[iFile]);
         printf("%s \n",strerror(errno));
         return;
      }
/*
      get observing data to check chopping flag coding
*/
      i = fits_read_key(inFile, TFLOAT, "MJD-OBS", &mjd,0,ierr);
      old = mjd < 52960.;
      if (*ierr !=0) {
         printf("error %i reading mjd file %s\n", 
            *ierr, fileList[iFile]);
         return;
      }
      idata = open_imaging_data_table(inFile, ierr);
      if (*ierr !=0) {
         printf("error %i opening imaging data in file %s\n", 
            *ierr, fileList[iFile]);
         return;
      }
      inData = make_imaging_data_row(idata);
      i = fits_get_num_rows (inFile,  &nRow, ierr);
      /* suppress autoscaling to avoid overflow */
      for (i=0; i<idata->nregion; i++)
         fits_set_tscale(inFile, 1+idata->data_col[i], 1.0, 0.0, ierr);

/*
         read lines, check target type and decide whether to accumulate
*/
      for (iRow =1; iRow<=nRow; iRow++) {
         i = read_imaging_data_table(inFile, idata, inData, iRow, ierr);
         if (*ierr !=0) {
            printf("error %i reading imaging data line %i in file %s\n", 
               *ierr, iRow, fileList[iFile]);
            return;
         }
         if (old) targ = inData->targtype[0]; 
         else targ = inData->targtype[1]; 
         if (targ != oldTarg ) {
            sinceChange = 0;
            oldTarg     = targ;
            if ((targ == Target) || (targ == Sky)) (*nCycle)++;
         } else {
            ++sinceChange;
         }
/*
            if sinceChange > nDrop decide what type and accumulate data
*/
         if (sinceChange > nDrop && (targ == Target)) {
            for (iPix=0;iPix<npix1;iPix++) {
		    target1[iPix] =
               target1[iPix] + 32768. + (short)inData->data[chan1][iPix];
	    }
            for (iPix=0;iPix<npix2;iPix++) {
		    target2[iPix] =
               target2[iPix] + 32768. + (short)inData->data[chan2][iPix];
	    }
            ++TN;
         } else if (sinceChange > nDrop && (targ == Sky)) {
            for (iPix=0;iPix<npix1;iPix++) {
		    sky1[iPix] =
               sky1[iPix] + 32768. + (short)inData->data[chan1][iPix];
	    }
            for (iPix=0;iPix<npix2;iPix++) {
		    sky2[iPix] =
               sky2[iPix] + 32768. + (short)inData->data[chan2][iPix];
	    }
            ++SN;
         }
         for (iPix=0;iPix<npix1;iPix++) {
		profiles1[(iPix / nx1) + iTotalRow*ny1] 
			+= ((32768. + (short)inData->data[chan1][iPix]));
	 }
	 for (iPix=0;iPix<ny1;iPix++)profiles1[iPix+iTotalRow*ny1]/=nx1;
         for (iPix=0;iPix<npix2;iPix++) {
		profiles2[(iPix / nx2) + iTotalRow*ny2] 
			+= ((32768. + (short)inData->data[chan2][iPix]));
	 }
	 for (iPix=0;iPix<ny2;iPix++)profiles2[iPix+iTotalRow*ny2]/=nx2;
	 iTotalRow += 1;
      } /* read rows in one file */
      
/*
         close file and table, destroy row
*/
      i= close_imaging_data_table(inFile, idata, ierr);
      kill_imaging_data_row(inData);
   } /* read all files in list */
   *nTarget = TN;
   *nSky    = SN;
   (*nCycle) = (*nCycle)/2;
   return;
} /* end of unChopData */

void binChopData(char **fileList, int nFiles, int chan1, int chan2, int nDrop, 
   float *offsets1, float *offsets2, long ntotalrows,
   long npix1, long npix2, int nCycle,
   float *a1[NBIN], float *a2[NBIN], int *ierr) {

   fitsfile *inFile;
   imaging_data_table_def *idata;
   imaging_data_row *inData;
   int iFile;

   long iTotalRow=0,nstdv=4;
   long iRow, nRow, minFrames=10;
   int i,j;
   int TN, SN;
   long iPix;

   char Sky = 'S';
   char Target = 'T';
   char targ, oldTarg;
   int sinceChange;
   float mjd;
   int old;

   float *aTarg1, *aTarg2, *aSky1, *aSky2; /* chop accumulators */
   int iBin, nBin, nBin1=0, nBin2=0;
   int iCycle = 0;
   float *off1mean, *off2mean, *off1stdv, *off2stdv;
   float offset1min,offset2min;
   float offbinsize=0.01;
   long iOff,noff,*noff1,*noff2,*tnoff1,*snoff1,*tnoff2,*snoff2,nrowsperbin;

   *ierr = 0;
   TN = 0;
   SN = 0;
   oldTarg = ' ';

/*
      how many cycles per bin?
*/
   nBin = nCycle/NBIN;

/*    
 *    Determine for each bin in NBIN mean and std dev of offsets
 */
   nrowsperbin=ntotalrows/NBIN;
   iBin=0;
   noff=0;
   off1mean=(float*)malloc(NBIN*sizeof(float));
   memset(off1mean,0,NBIN*sizeof(float));
   for (i=0;i<ntotalrows;i++){
	   off1mean[iBin]+=offsets1[i];
	   noff+=1;
	   if (((i+1) % nrowsperbin) == 0){
		   off1mean[iBin]/=noff;
		   noff=0;
		   iBin+=1;
	   }
   }
   iBin=0;
   noff=0;
   off2mean=(float*)malloc(NBIN*sizeof(float));
   memset(off2mean,0,NBIN*sizeof(float));
   for (i=0;i<ntotalrows;i++){
	   off2mean[iBin]+=offsets2[i];
	   noff+=1;
	   if (((i+1) % nrowsperbin) == 0){
		   off2mean[iBin]/=noff;
		   noff=0;
		   iBin+=1;
	   }
   }
   iBin=0;
   noff=0;
   off1stdv=(float*)malloc(NBIN*sizeof(float));
   memset(off1stdv,0,NBIN*sizeof(float));
   for (i=0;i<ntotalrows;i++){
	   off1stdv[iBin]+=pow(offsets1[i]-off1mean[iBin],2);
	   noff+=1;
	   if (((i+1) % nrowsperbin) == 0){
		   off1stdv[iBin]=sqrt(off1stdv[iBin]/noff);
		   noff=0;
		   iBin+=1;
	   }
   }
   iBin=0;
   noff=0;
   off2stdv=(float*)malloc(NBIN*sizeof(float));
   memset(off2stdv,0,NBIN*sizeof(float));
   for (i=0;i<ntotalrows;i++){
	   off2stdv[iBin]+=pow(offsets2[i]-off2mean[iBin],2);
	   noff+=1;
	   if (((i+1) % nrowsperbin) == 0){
		   off2stdv[iBin]=sqrt(off2stdv[iBin]/noff);
		   noff=0;
		   iBin+=1;
	   }
   }
   noff1=(long*)malloc(NBIN*sizeof(long));
   noff2=(long*)malloc(NBIN*sizeof(long));
   for (i=0;i<NBIN;i++){
	   noff1[i]=nstdv*2*off1stdv[i]/offbinsize; if (noff1[i] == 0) noff1[i]=1;
	   noff2[i]=nstdv*2*off2stdv[i]/offbinsize; if (noff2[i] == 0) noff2[i]=1;
   }
/*
      loop over files, read rows, read target types, add data
      into appropriate accumulators 
*/
   for (iFile =0; iFile <nFiles; iFile++) {
/*
      open file, open table, create table row
*/
      i = fits_open_file(&inFile, fileList[iFile], READONLY, ierr);
      if (*ierr !=0) {
         printf("error %i opening file %s\n", *ierr, fileList[iFile]);
         printf("%s \n",strerror(errno));
         return;
      }
      i = fits_read_key(inFile, TFLOAT, "MJD-OBS", &mjd,0,ierr);
      old = mjd < 52960.;
      if (*ierr !=0) {
         printf("error %i reading mjd file %s\n", 
            *ierr, fileList[iFile]);
         return;
      }
      idata = open_imaging_data_table(inFile, ierr);
      if (*ierr !=0) {
         printf("error %i opening imaging data in file %s\n", 
            *ierr, fileList[iFile]);
         return;
      }
      inData = make_imaging_data_row(idata);
      i = fits_get_num_rows (inFile,  &nRow, ierr);
      /* suppress autoscaling to avoid overflow */
      for (i=0; i<idata->nregion; i++)
         fits_set_tscale(inFile, 1+idata->data_col[i], 1.0, 0.0, ierr);

/*
         read lines, check target type and decide whether to accumulate
*/
      for (iRow =1; iRow<=nRow; iRow++) {
         i = read_imaging_data_table(inFile, idata, inData, iRow, ierr);
         if (*ierr !=0) {
            printf("error %i reading imaging data line %i in file %s\n", 
               *ierr, iRow, fileList[iFile]);
            return;
         }
         if (old) targ = inData->targtype[0]; 
         else targ = inData->targtype[1]; 
         if (targ != oldTarg ) {
            sinceChange = 0;
            oldTarg     = targ;
         } else {
            ++sinceChange;
         }

	 iBin = (int) iTotalRow/nrowsperbin;

	 if ((iTotalRow % nrowsperbin) == 0){
/*
		 printf("New bin %d, noff1=%d, noff2=%d\n",iBin,noff1[iBin],noff2[iBin]);
		 printf("Mean offset1=%f +/- %f, offset2=%f +/- %f\n",off1mean[iBin],off1stdv[iBin],off2mean[iBin],off2stdv[iBin]);
*/
   		aTarg1 = (float *)malloc(noff1[iBin]*npix1 * sizeof(float));
		memset(aTarg1,0,noff1[iBin]*npix1 * sizeof(float));
   		aTarg2 = (float *)malloc(noff2[iBin]*npix2 * sizeof(float));
		memset(aTarg2,0,noff2[iBin]*npix2 * sizeof(float));
   		aSky1  = (float *)malloc(noff1[iBin]*npix1 * sizeof(float));
		memset(aSky1,0,noff1[iBin]*npix1 * sizeof(float));
   		aSky2  = (float *)malloc(noff2[iBin]*npix2 * sizeof(float));
		memset(aSky2,0,noff2[iBin]*npix2 * sizeof(float));
   		tnoff1=(long*)malloc(noff1[iBin]*sizeof(long));
		memset(tnoff1,0,noff1[iBin]*sizeof(long));
   		snoff1=(long*)malloc(noff1[iBin]*sizeof(long));
		memset(snoff1,0,noff1[iBin]*sizeof(long));
   		tnoff2=(long*)malloc(noff2[iBin]*sizeof(long));
		memset(tnoff2,0,noff2[iBin]*sizeof(long));
   		snoff2=(long*)malloc(noff2[iBin]*sizeof(long));
		memset(snoff2,0,noff2[iBin]*sizeof(long));
	 }
/*
         if sinceChange > nDrop decide what type and accumulate data
*/
	 offset1min=off1mean[iBin]-nstdv*off1stdv[iBin];
	 offset2min=off2mean[iBin]-nstdv*off2stdv[iBin];
         if (sinceChange > nDrop && (targ == Target)) {
	    iOff=(long)((offsets1[iTotalRow]-offset1min)/offbinsize);
	    if (iOff >= 0 && iOff < noff1[iBin]){
            for (iPix=0;iPix<npix1;iPix++) aTarg1[iPix+iOff*npix1] =
               aTarg1[iPix+iOff*npix1] + 32768. + (short)inData->data[chan1][iPix];
	    tnoff1[iOff]+=1;
	    }
	    iOff=(long)((offsets2[iTotalRow]-offset2min)/offbinsize);
	    if (iOff >= 0 && iOff < noff2[iBin]){
            for (iPix=0;iPix<npix2;iPix++) aTarg2[iPix+iOff*npix2] =
               aTarg2[iPix+iOff*npix2] + 32768. + (short)inData->data[chan2][iPix];
	    tnoff2[iOff]+=1;
	    }
         } else if (sinceChange > nDrop && (targ == Sky)) {
	    iOff=(long)((offsets1[iTotalRow]-offset1min)/offbinsize);
	    if (iOff >= 0 && iOff < noff1[iBin]){
            for (iPix=0;iPix<npix1;iPix++) aSky1[iPix+iOff*npix1] =
               aSky1[iPix+iOff*npix1] + 32768. + (short)inData->data[chan1][iPix];
	    snoff1[iOff]+=1;
	    }
	    iOff=(long)((offsets2[iTotalRow]-offset2min)/offbinsize);
	    if (iOff >= 0 && iOff < noff2[iBin]){
            for (iPix=0;iPix<npix2;iPix++) aSky2[iPix+iOff*npix2] =
               aSky2[iPix+iOff*npix2] + 32768. + (short)inData->data[chan2][iPix];
	    snoff2[iOff]+=1;
	    }
         }

/*
       normalize accumulated target and sky
*/
	 if (((iTotalRow+1) % nrowsperbin) == 0){
	 for (j=0;j<noff1[iBin];j++){
               if (tnoff1[j] > 0) {
                  for (iPix=0;iPix<npix1;iPix++) aTarg1[iPix+j*npix1] =
                     aTarg1[iPix+j*npix1]/tnoff1[j];
               }
               if(snoff1[j] > 0) {
                  for (iPix=0;iPix<npix1;iPix++) aSky1[iPix+j*npix1] =
                     aSky1[iPix+j*npix1]/snoff1[j];
	       }
	 }
	 for (j=0;j<noff2[iBin];j++){
               if (tnoff2[j] > 0) {
                  for (iPix=0;iPix<npix2;iPix++) aTarg2[iPix+j*npix2] =
                     aTarg2[iPix+j*npix2]/tnoff2[j];
               }
               if(snoff2[j] > 0) {
                  for (iPix=0;iPix<npix2;iPix++) aSky2[iPix+j*npix2] =
                     aSky2[iPix+j*npix2]/snoff2[j];
	       }
	 }
/*
       accumulate difference into bin
*/
	 for (j=0;j<noff1[iBin];j++){
	     if (tnoff1[j] > minFrames && snoff1[j] > minFrames){
/*
		 printf("Channel 1: iOff=%d tnoff1=%d snoff1=%d\n",j,tnoff1[j],snoff1[j]);
*/
                     for (iPix=0;iPix<npix1;iPix++) a1[iBin][iPix] =
                        a1[iBin][iPix] + aTarg1[iPix+j*npix1] - aSky1[iPix+j*npix1];
		     nBin1+=1;
	     }
	 }
	 for (j=0;j<noff2[iBin];j++){
	     if (tnoff2[j] > minFrames && snoff2[j] > minFrames){
/*
		 printf("Channel 2: iOff=%d tnoff1=%d snoff1=%d\n",j,tnoff2[j],snoff2[j]);
*/
                     for (iPix=0;iPix<npix2;iPix++) a2[iBin][iPix] =
                        a2[iBin][iPix] + aTarg2[iPix+j*npix2] - aSky2[iPix+j*npix2];
		     nBin2+=1;
	     }
	 }
	 free(aTarg1);
	 free(aTarg2);
	 free(aSky1);
	 free(aSky2);
	 free(tnoff1);
	 free(tnoff2);
	 free(snoff1);
	 free(snoff2);
	 }

      iTotalRow+=1;
      } /* read rows in one file */
      
/*
      close file and table, destroy row
*/
      i= close_imaging_data_table(inFile, idata, ierr);
      kill_imaging_data_row(inData);
   } /* read all files in list */
/*
        normalize bins
*/
   for (iBin =0; iBin<NBIN; iBin++) {
      for (iPix=0;iPix<npix1;iPix++) a1[iBin][iPix] =
         a1[iBin][iPix]/nBin1;
      for (iPix=0;iPix<npix2;iPix++) a2[iBin][iPix] =
         a2[iBin][iPix]/nBin2;
   }
   free(off1mean);
   free(off2mean);
   free(off1stdv);
   free(off2stdv);
   free(noff1);
   free(noff2);
   return;
} /* end of binChopData */

void jitterChopData(long ny1, long ny2, 
   float *profiles1, float *profiles2, long ntotalrows,
   float *offset1, float *offset2, int yBottom, int yTop) {

   long nlag;
   long i,j,k;

   float *mprofile1, *mprofile2;
   float *lags,*diff1,*diff2,mindiff1,mindiff2;
   double x1,x2,x3,y1,y2,y3,y4,y5,a1,a2,a3,tay1,tay2,tay3;
   float mrat1,mrat2;
   FILE *fp;

   mprofile1 = (float*)malloc(ny1*sizeof(float));
   memset(mprofile1,0,ny1*sizeof(float));
   mprofile2 = (float*)malloc(ny2*sizeof(float));
   memset(mprofile2,0,ny2*sizeof(float));

/* 
 * Determine mean profile
 */
   for (j=0;j<ny1;j++) {
	   for (i=0;i<ntotalrows;i++) {
		   mprofile1[j] += profiles1[j+i*ny1];
	   }
	   mprofile1[j] /= ntotalrows;
   }
   for (j=0;j<ny2;j++) {
	   for (i=0;i<ntotalrows;i++) {
		   mprofile2[j] += profiles2[j+i*ny1];
	   }
	   mprofile2[j] /= ntotalrows;
   }
/*
	 printf("Writing mean profile to disk...\n");
	 fp=fopen("mprofile.dat","w");
   	 for (j=0;j<ny1;j++) {
		 fprintf(fp,"%f\n",mprofile1[j]);
	 }
	 fclose(fp);
*/
/*
 * Correlate each profile with the mean to determine the offset
 */
   nlag=5;
   lags=(float*)malloc(nlag*sizeof(float));
   diff1=(float*)malloc(nlag*sizeof(float));
   diff2=(float*)malloc(nlag*sizeof(float));

   for (j=0;j<nlag;j++) lags[j]=j-nlag/2;

   for (i=0;i<ntotalrows;i++){
	   mrat1=0;
	   for (j=yBottom;j<=yTop;j++) mrat1+=(profiles1[j+i*ny1]/mprofile1[j]);
	   mrat1=1.0;
	   if (mrat1 > 0) {
		mrat1/=(yTop-yBottom+1);
		mrat1=1.0;
	   	for (j=0;j<nlag;j++){
			diff1[j]=0;
			for (k=yBottom;k<=yTop;k++){
				diff1[j] += pow(profiles1[k+i*ny1]
						-mrat1*mprofile1[k+(j-nlag/2)],2);	
			}
	   	}
	   	mindiff1=diff1[0];
	   	k=0;
	   	for (j=1;j<nlag;j++){
		   	if (diff1[j]<mindiff1){
			   	mindiff1=diff1[j];
			   	k=j;
		   	}
	   	}
/*
	   if (i > 2000){
	 fp=fopen("debug.dat","w");
	 printf("ntotalrows=%d, ny1=%d, k=%d, %d %d\n",ntotalrows,ny1,k,yBottom,yTop);
	 fp=fopen("debug.dat","w");
         for (j=0;j<ny1;j++)fprintf(fp,"%f %f\n",mrat1*mprofile1[j],profiles1[j+i*ny1]);
         for (j=0;j<nlag;j++)fprintf(fp,"%f\n",diff1[j]);
	 fclose(fp);
	 return;
	   	}
         for (j=0;j<nlag;j++)fprintf(fp,"%f ",diff1[j]);
	 fprintf(fp,"\n");
	 */

	   	if (k>0 && k<nlag){

        		y1 = diff1[k-1];
	        	y2 = diff1[k];
			y3 = diff1[k+1];
			x1 = lags[k-1];
			x2 = lags[k];
			x3 = lags[k+1];
			a3 = ((y3-y2)*(x2-x1)-(y2-y1)*(x3-x2))
			     / ((x3-x2)*(x2-x1)*(x3-x1));
			a2 = ((y3-y2)*(x2*x2-x1*x1)-(y2-y1)*(x3*x3-x2*x2))
			     / ((x3-x2)*(x2-x1)*(x1-x3));
			a1 = y1 - (a3*x1*x1+a2*x1);

			y1=diff1[0];
			y2=diff1[1];
			y3=diff1[2];
			y4=diff1[3];
			y5=diff1[4];
			tay1=y1+y2+y3+y4+y5;
			tay2=-2*y1-y2+y4+2*y5;
			tay3=4*y1+y2+y4+4*y5;

			a1=(1.0/2.0588235)*tay1-(1.0/7.0)*tay3;
			a2=0.1*tay2;
			a3=-(1.0/7.0)*tay1+(1.0/14.0)*tay3;

			offset1[i]=(-a2/(2*a3));
		}
	   }
	   mrat2=0;
	   for (j=yBottom;j<=yTop;j++) mrat2+=(profiles2[j+i*ny2]/mprofile2[j]);
	   mrat2=1.0;
	   if (mrat2 > 0) {
		mrat2/=(yTop-yBottom+1);
		mrat2=1.0;
	   	for (j=0;j<nlag;j++){
			diff2[j]=0;
			for (k=yBottom;k<=yTop;k++){
				diff2[j] += pow(profiles2[k+i*ny2]
						-mrat2*mprofile2[k+(j-nlag/2)],2);	
			}
	   	}
	   	mindiff2=diff2[0];
	   	k=0;
	   	for (j=1;j<nlag;j++){
		   	if (diff2[j]<mindiff2){
			   	mindiff2=diff2[j];
			   	k=j;
		   	}
	   	}
	   	if (k>0 && k<nlag){

        		y1 = diff2[k-1];
	        	y2 = diff2[k];
			y3 = diff2[k+1];
			x1 = lags[k-1];
			x2 = lags[k];
			x3 = lags[k+1];
			a3 = ((y3-y2)*(x2-x1)-(y2-y1)*(x3-x2))
			     / ((x3-x2)*(x2-x1)*(x3-x1));
			a2 = ((y3-y2)*(x2*x2-x1*x1)-(y2-y1)*(x3*x3-x2*x2))
			     / ((x3-x2)*(x2-x1)*(x1-x3));
			a1 = y1 - (a3*x1*x1+a2*x1);

			y1=diff2[k-2];
			y2=diff2[k-1];
			y3=diff2[k];
			y4=diff2[k+1];
			y5=diff2[k+2];
			tay1=y1+y2+y3+y4+y5;
			tay2=-2*y1-y2+y4+2*y5;
			tay3=4*y1+y2+y4+4*y5;

			a1=(1.0/2.0588235)*tay1-(1.0/7.0)*tay3;
			a2=0.1*tay2;
			a3=-(1.0/7.0)*tay1+(1.0/14.0)*tay3;

			offset2[i]=(-a2/(2*a3));
		   }
	   }

   }
   free(mprofile1);
   free(mprofile2);
   free(lags);
   free(diff1);
   free(diff2);
   
} /* end of jitterChopData */

void normalizeExposure (float *data, long nPix, int nFrames, float dit) {
   float norm;
   long iPix;

   if (nFrames <= 0) return;
   norm = 1./(nFrames * dit);
   for (iPix =0; iPix<nPix; iPix++) data[iPix] = data[iPix]*norm;
   return;
}

void deSky (float *data, int nx, int ny, int y1, int y2) {
   int ix, iy;
   float dy, x1, x2;
   float *topSky, *bottomSky;
   float *yLine;
   int dnsky = 2;
/*
      arrays to hold average of a few rows at top and bottom
*/
   topSky = (float *)malloc(nx * sizeof(float));
   memset(topSky, 0, nx*sizeof(float));
   bottomSky = (float *)malloc(nx * sizeof(float));
   memset(bottomSky, 0, nx*sizeof(float));

/*
      average a few rows
*/
   for (iy =-dnsky; iy<=dnsky; iy++) for (ix=0;ix<nx;ix++) {
      topSky[ix]    = topSky[ix] + data[ix + (iy+y2)*(long)nx];
      bottomSky[ix] = bottomSky[ix] + data[ix + (iy+y1)*(long)nx];
   }
   for (ix=0;ix<nx;ix++) {
      topSky[ix]    = topSky[ix]/(2*dnsky+1);
      bottomSky[ix] = bottomSky[ix]/(2*dnsky+1);
   }
   dy = y2 - y1;

/*
      linear fit and subtraction
*/
   for (iy=0; iy<ny; iy++) {
      yLine = data+iy*(long)nx;
      x1 = ((float)iy - y1)/dy;
      x2 = ((float)iy - y2)/dy;
      for (ix=0; ix<nx;ix++) yLine[ix] = yLine[ix] - topSky[ix]*x1 + bottomSky[ix]*x2;
   }

/*
      free arrays
*/
   free(topSky);
   free(bottomSky);
}

int main (int argc, char** argv) {

   char opt1[12];
   char gris[12];
   char shutter[12];
   float dit;

   char **fileList;
   int nFiles;
   int iShutter;
   int i,j,ierr;
   int nx, ny;
   long nx1, ny1, nx2, ny2; /* CAH */
   long nRow,nTotalRows;
   long npix;
   long iPix;
   int  ix, iy, iBin;
   float xx, mm;
   long  ii;
   int iKey;
   int nCycle;

   int yTop = 34;
   int yBottom = 0;
   int chan1, chan2;
   int iarg;

   char outFileName[130];
   char *bang="!";

   fitsfile *inFile;
   fitsfile *outFile = NULL;

   fitsfile *debugFile = NULL;
   imaging_fdata_table_def *debugTable;
   imaging_fdata_row       *debugRow;

   imaging_fdata_table_def *odata;
   imaging_detector_table_def *DetectorDef;
   imaging_fdata_row *outData;
   float *maskImages[12];
   int nxList[12], nyList[12];
   int nRegion;

   char *Ashutter = "AOPEN";
   char *Bshutter = "BOPEN";
   int  nShutter;
   char *highSens = "HIGH_SENS";

   float *ATarget1, *ATarget2; 
   float *ASky1, *ASky2;
   float *BTarget1, *BTarget2; 
   float *BSky1, *BSky2;

   float *Aprofiles1, *Aprofiles2; /* CAH */
   float *Bprofiles1, *Bprofiles2; /* CAH */
   float *Aoffsets1,*Aoffsets2,*Boffsets1,*Boffsets2;
   FILE *fp; /* CAH */

   float *ABin1[NBIN], *ABin2[NBIN], *BBin1[NBIN], *BBin2[NBIN];
   int    ATargetN, ASkyN, BTargetN, BSkyN;

   int nDrop;
   int dSky;

   ierr = 0;
/*
      see if nDrop, dSky are specified externally
*/
   nDrop = 2;
   dSky  = 0;
   iarg  = 0;
   if (argc > 5) while(iarg < argc ) {
      if(!strcmp("-nDrop", argv[iarg])) sscanf(argv[++iarg], "%i", &nDrop);
      if(!strcmp("-dSky", argv[iarg])) sscanf(argv[++iarg], "%i", &dSky);
      ++iarg;
   }
   printf("nDrop = %i\n", nDrop);
   printf("dSky  = %i\n", dSky);

/*        
       break up file list and print what you find 
       then read and unchop files
*/
   for (iShutter = 1;iShutter<=2; iShutter++) {
      fileList = oirStrSplit(argv[iShutter], &nFiles);

/*
      open first file and get a lot of parameters for iShutter=1
*/
      i = fits_open_file(&inFile, fileList[0], READONLY, &ierr) ;
      if (ierr != 0) {
         printf("cfitsio error %i opening file %s\n", ierr, fileList[0]);
         printf("%s \n",strerror(errno));
         return 1;
      }
      ierr = 0;
/*
      prepare output file
*/
      strcpy(outFileName, bang);
      strcat(outFileName, argv[3]);

      fits_create_file (&outFile, outFileName, &ierr);
      if (ierr !=0) {
         fprintf (stderr, "error opening fits output %s %i\n",argv[4], ierr);
         printf("%s \n",strerror(errno));
         return 1;
      }
/*
              copy header
*/
      i = fits_copy_header(inFile, outFile, &ierr); 
      i = fits_read_key(inFile, TSTRING, "HIERARCH ESO INS SHUT NAME", 
         shutter, 0, &ierr);
      printf("shutter = %s\n", shutter);
      nShutter = 0;
      if (strncmp(shutter, Ashutter, 5)==0) nShutter=1;
      if (strncmp(shutter, Bshutter, 5)==0) nShutter=2; 
      if (iShutter == 1) {
         i = fits_read_key(inFile, TFLOAT, "HIERARCH ESO DET DIT", &dit, 
            0, &ierr);
         printf("dit = %f\n", dit);
         ierr = 0;
         i = fits_read_key(inFile, TSTRING, "HIERARCH ESO INS OPT1 NAME", opt1,0, &ierr);
         printf("beam combiner = %s\n", opt1);

         if (strncmp(opt1, highSens, 9) == 0) { chan1 = 0; chan2 = 1;}
            else {chan1 = 1; chan2 = 2;}
         i = fits_read_key(inFile, TSTRING, "HIERARCH ESO INS GRIS NAME", gris,0,&ierr);
         printf("disperser is = %s\n", gris);
/*
      move to detector section
*/
         DetectorDef = read_imaging_detector_table(inFile, &ierr);
         if (ierr !=0) {
            printf("error %i opening detector table in %s\n", 
               ierr, fileList[0]);
            return 1;
         }
         nRegion = DetectorDef->nregion;
         for (i=0;i<nRegion;i++) {
            nxList[i] = DetectorDef->naxis[i][0];
            nyList[i] = DetectorDef->naxis[i][1];
         }
         nx = DetectorDef->naxis[chan1][0];
         ny = DetectorDef->naxis[chan1][1];

/*
      allocate accumulator arrays
*/
  
         npix = ny * (long)nx;
         ATarget1 = (float *)malloc(npix * sizeof(float));
         memset (ATarget1, 0, npix*sizeof(float));
         BTarget1 = (float *)malloc(npix * sizeof(float));
         memset (BTarget1, 0, npix*sizeof(float));
         ASky1    = (float *)malloc(npix * sizeof(float));
         memset (ASky1, 0, npix*sizeof(float));
         BSky1    = (float *)malloc(npix * sizeof(float));
         memset (BSky1, 0, npix*sizeof(float));
         ATarget2 = (float *)malloc(npix * sizeof(float));
         memset (ATarget2, 0, npix*sizeof(float));
         BTarget2 = (float *)malloc(npix * sizeof(float));
         memset (BTarget2, 0, npix*sizeof(float));
         ASky2    = (float *)malloc(npix * sizeof(float));
         memset (ASky2, 0, npix*sizeof(float));
         BSky2    = (float *)malloc(npix * sizeof(float));
         memset (BSky2, 0, npix*sizeof(float));

         for (i=0; i<NBIN; i++) {
            ABin1[i] = (float *)malloc(npix * sizeof(float));
            memset (ABin1[i], 0, npix*sizeof(float));
            ABin2[i] = (float *)malloc(npix * sizeof(float));
            memset (ABin2[i], 0, npix*sizeof(float));
            BBin1[i] = (float *)malloc(npix * sizeof(float));
            memset (BBin1[i], 0, npix*sizeof(float));
            BBin2[i] = (float *)malloc(npix * sizeof(float));
            memset (BBin2[i], 0, npix*sizeof(float));
         }

      }  /* iShutter = 1 */
      i = fits_close_file(inFile, &ierr);
/*
          now process the real data
*/
      if (nShutter == 1) {
         printf("Found %i A-photometry Files\n", nFiles);
         for (i=0; i<nFiles; i++) printf("%s \n", fileList[i]);

/*
 	 Determine total number of rows for this shutter (CAH)
 */
      	 nTotalRows = 0;
      	 for (i=0; i<nFiles; i++) {
      		 j = fits_open_file(&inFile, fileList[i], READONLY, &ierr);
		 j = fits_movnam_hdu(inFile, BINARY_TBL, "IMAGING_DATA", 0, &ierr);
      		 if (ierr != 0) {
         		 printf("cfitsio error %i opening file %s\n", ierr, fileList[i]);
         		 printf("%s \n",strerror(errno));
         		 return 1;
      		 }
      		 j = fits_get_num_rows (inFile,  &nRow, &ierr);
      		 j = fits_close_file(inFile, &ierr);
		 nTotalRows += nRow;
      	 }
	 printf("Total number of rows shutter A: %i\n",nTotalRows);
      	 Aprofiles1 = (float *)malloc(nTotalRows * ny * sizeof(float));
      	 memset (Aprofiles1, 0, nTotalRows * ny * sizeof(float));
      	 Aprofiles2 = (float *)malloc(nTotalRows * ny * sizeof(float));
      	 memset (Aprofiles2, 0, nTotalRows * ny * sizeof(float));
	 nx1=nx;
	 ny1=ny;
	 nx2=nx;
	 ny2=ny;

         unChopData(fileList, nFiles, chan1, chan2, nDrop,
            npix, npix, nx1, ny1, nx2, ny2,
	    Aprofiles1, Aprofiles2, nTotalRows,
	    ATarget1, ATarget2, ASky1, ASky2, &ATargetN, &ASkyN,
            &nCycle, &ierr);
         if (ierr != 0) {
            printf("Failed to process A-Data: error: %i\n", ierr);
            return 0;
         }
/*     
 *     Determine offsets and bins
*/
   	 if (gris[0] == 'G') yBottom = 6; else yBottom = 9;
   	 if (gris[0] == 'G') yTop = 29; else yTop = 25;
	 Aoffsets1=(float*)malloc(nTotalRows*sizeof(float));
	 memset(Aoffsets1,0,nTotalRows*sizeof(float));
	 Aoffsets2=(float*)malloc(nTotalRows*sizeof(float));
	 memset(Aoffsets2,0,nTotalRows*sizeof(float));
	 jitterChopData(ny1,ny2,Aprofiles1, Aprofiles2, nTotalRows, 
			 Aoffsets1, Aoffsets2, yBottom, yTop);
/*
	 printf("Writing offsets to disk...\n");
	 fp=fopen("offsets.dat","w");
         for (i=0;i<nTotalRows;i++) {
		 fprintf(fp,"%f %f\n",Aoffsets1[i],Aoffsets2[i]);
	 }
	 fclose(fp);
*/
/*
       normalize data by dividing by #frames
*/
         normalizeExposure(ATarget1, npix, ATargetN, dit);
         normalizeExposure(ATarget2, npix, ATargetN, dit);
         normalizeExposure(ASky1, npix, ASkyN, dit);
         normalizeExposure(ASky2, npix, ASkyN, dit);
/*
       subtract sky
*/
         for (iPix = 0; iPix< npix; iPix++) {
            ATarget1[iPix] = ATarget1[iPix] - ASky1[iPix];
            ATarget2[iPix] = ATarget2[iPix] - ASky2[iPix];
         }
   /* go back through data and accumlate data in bins */
         printf("Processing sub-A-exposures\n");
         binChopData(fileList, nFiles, chan1, chan2, nDrop, 
	    Aoffsets1, Aoffsets2, nTotalRows,
            npix, npix, nCycle, ABin1, ABin2, &ierr);
         if (ierr != 0) {
            printf("Failed to process ARMS-Data: error: %i\n", ierr);
            return 0;
         }
/*
              normalize 
*/
         for (i=0; i<NBIN; i++) {
            normalizeExposure(ABin1[i], npix, 1., dit);
            normalizeExposure(ABin2[i], npix, 1., dit);
         }
      }
      if (nShutter == 2) {
         printf("Found %i B-photometry Files\n", nFiles);
         for (i=0; i<nFiles; i++) printf("%s \n", fileList[i]);

      	 nTotalRows = 0;
      	 for (i=0; i<nFiles; i++) {
      		 j = fits_open_file(&inFile, fileList[i], READONLY, &ierr) ;
		 j = fits_movnam_hdu(inFile, BINARY_TBL, "IMAGING_DATA", 0, &ierr);
      		 if (ierr != 0) {
         		 printf("cfitsio error %i opening file %s\n", ierr, fileList[i]);
         		 printf("%s \n",strerror(errno));
         		 return 1;
      		 }
      		 j = fits_get_num_rows (inFile,  &nRow, &ierr);
      		 j = fits_close_file(inFile, &ierr);
		 nTotalRows += nRow;
      	 }
      	 Bprofiles1 = (float *)malloc(nTotalRows * ny * sizeof(float));
      	 memset (Bprofiles1, 0, nTotalRows * ny * sizeof(float));
      	 Bprofiles2 = (float *)malloc(nTotalRows * ny * sizeof(float));
      	 memset (Bprofiles2, 0, nTotalRows * ny * sizeof(float));
	 printf("Total number of rows shutter B: %i\n",nTotalRows);

         unChopData(fileList, nFiles, chan1, chan2, nDrop,
            npix, npix, nx, ny, nx, ny,
	    Bprofiles1, Bprofiles2, nTotalRows,
	    BTarget1, BTarget2, BSky1, BSky2, &BTargetN, &BSkyN,
            &nCycle, &ierr);
         if (ierr != 0) {
            printf("Failed to process B-Data: error: %i\n", ierr);
            return 0;
         }
	 /*
	 printf("Writing profiles to disk...\n");
	 fp=fopen("profiles.dat","w");
         for (i=0;i<nTotalRows;i++) {
		 for (iPix=0;iPix<ny;iPix++)
		 fprintf(fp,"%f ",Bprofiles1[iPix+i*ny]);
	 	 fprintf(fp,"\n");
	 }
	 fclose(fp);
	 */
   	 if (gris[0] == 'G') yBottom = 6; else yBottom = 10;
   	 if (gris[0] == 'G') yTop = 29; else yTop = 25;
	 Boffsets1=(float*)malloc(nTotalRows*sizeof(float));
	 memset(Boffsets1,0,nTotalRows*sizeof(float));
	 Boffsets2=(float*)malloc(nTotalRows*sizeof(float));
	 memset(Boffsets2,0,nTotalRows*sizeof(float));
	 jitterChopData(ny1,ny2,Bprofiles1, Bprofiles2, nTotalRows, 
			 Boffsets1, Boffsets2, yBottom, yTop);
	 /*
	 fclose(fp);
         for (i=0;i<nTotalRows;i++) {
		 for (iPix=0;iPix<ny;iPix++)
		 fprintf(fp,"%f ",Aprofiles2[iPix+i*ny]);
	 	 fprintf(fp,"\n");
	 }
	 */

         normalizeExposure(BTarget1, npix, BTargetN, dit);
         normalizeExposure(BTarget2, npix, BTargetN, dit);
         normalizeExposure(BSky1, npix, BSkyN, dit);
         normalizeExposure(BSky2, npix, BSkyN, dit);

         for (iPix = 0; iPix< npix; iPix++) {
            BTarget1[iPix] = BTarget1[iPix] - BSky1[iPix];
            BTarget2[iPix] = BTarget2[iPix] - BSky2[iPix];
         }
   /* go back and bin data */
         printf("Processing sub-B-exposures\n");
         binChopData(fileList, nFiles, chan1, chan2, nDrop, 
	    Boffsets1, Boffsets2, nTotalRows,
            npix, npix, nCycle, BBin1, BBin2, &ierr);
         if (ierr != 0) {
            printf("Failed to process BRMS-Data: error: %i\n", ierr);
            return 0;
         }
         for (i=0; i<NBIN; i++) {
            normalizeExposure(BBin1[i], npix, 1., dit);
            normalizeExposure(BBin2[i], npix, 1., dit);
         }
      }
      oirKillFileList(fileList, nFiles);
   }   /* loop over shutters */

/*
      deSky
      y coordinates of sky are cludge for prism; need better algorithm to find background
      set yBottom = 12 for prism, 5 for grism
*/
   /*
   if (gris[0] == 'G') yBottom = 6-dSky; else yBottom = 9-dSky;
   if (gris[0] == 'G') yTop = 29+dSky; else yTop = 25+dSky;
   if ((yBottom > 2) && (yTop < 38)) {
      printf("Sky Subtract from %i to %i \n", yBottom, yTop);
      deSky(ATarget1, nx, ny, yBottom, yTop); 
      deSky(ATarget2, nx, ny, yBottom, yTop);
      deSky(BTarget1, nx, ny, yBottom, yTop);
      deSky(BTarget2, nx, ny, yBottom, yTop);
      for (i=0; i<NBIN; i++) {
         deSky(ABin1[i], nx, ny, yBottom, yTop);
         deSky(ABin2[i], nx, ny, yBottom, yTop);
         deSky(BBin1[i], nx, ny, yBottom, yTop);
         deSky(BBin2[i], nx, ny, yBottom, yTop);
      }
   }
   else {
      printf("No Sky Subtraction along slit\n");
      yBottom = 2;
      yTop    = 38;
   }
   */

   odata = make_imaging_fdata_table_def (outFile, DetectorDef, 1, &ierr);
   if (ierr !=0) {
      fprintf (stderr, "error creating data definition\n");
      return 1;
   }
   outData = make_imaging_fdata_row(odata);

   ii = 1;
   for (ix=0;ix<npix;ix++) outData->data[0][ix] = ATarget1[ix];
   for (ix=0;ix<npix;ix++) outData->data[1][ix] = ATarget2[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, ii++, &ierr);

   for (ix=0;ix<npix;ix++) outData->data[0][ix] = BTarget1[ix];
   for (ix=0;ix<npix;ix++) outData->data[1][ix] = BTarget2[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, ii++, &ierr);

   for (iBin=0;iBin<NBIN;iBin++) {
      for (ix=0;ix<npix;ix++) outData->data[0][ix] = ABin1[iBin][ix];
      for (ix=0;ix<npix;ix++) outData->data[1][ix] = ABin2[iBin][ix];
      i = write_imaging_fdata_table(outFile, odata, outData, ii++, &ierr);

      for (ix=0;ix<npix;ix++) outData->data[0][ix] = BBin1[iBin][ix];
      for (ix=0;ix<npix;ix++) outData->data[1][ix] = BBin2[iBin][ix];
      i = write_imaging_fdata_table(outFile, odata, outData, ii++, &ierr);
   }
   i = close_imaging_fdata_table(outFile, odata, &ierr);
   kill_imaging_fdata_row(outData);

   free(ATarget1);
   free(ATarget2);
   free(BTarget1);
   free(BTarget2);
   free(ASky1);
   free(ASky2);
   free(BSky1);
   free(BSky2);
   for (i=0; i<NBIN; i++) {
      free(ABin1[i]); 
      free(ABin2[i]); 
      free(BBin1[i]); 
      free(BBin2[i]); 
   }
   free(Aprofiles1);
   free(Aprofiles2);
   free(Bprofiles1);
   free(Bprofiles2);
   return 0;
}
