/*
 * Copyright 2005 University of Leiden.
 *
 * This file is part of MIA+EWS.
 *
 * MIA+EWS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MIA+EWS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MIA; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include "oirSupport.h"
#include "fitsio.h"
#include "imaging_fdata.h"
#define NBIN 5
/**********************************************************
 * Usage:
 *    oirMakePhotoSpectra inFile maskFile outFile -jitter jitter
 *
 *    inFile is the output of oirChopPhotoImages
 *
 *    maskFile defines the pixel weighting before collapsing the spectra in Y
 *
 *    Output is a IMAGING_DATA table in a fits file with
 *    12-rows, each with only one DATA array (sum of channels 1 and 2)
 *
 *    1:  Total Flux A-side (ADUs/s/channel)
 *    2:  Total Flux B-side 
 *    3:  Total Flux Geometric Mean AB
 *    4:  Masked Flux A-side 
 *    5:  Masked Flux B-side 
 *    6:  Masked Flux Geometric Mean AB
 *    7->12 are estimated RMS errors in rows 1->6 resp.
 *
 ************************************************************* */

int main (int argc, char** argv) {

   int i,ierr;
   int nx, ny;
   long npix;
   long iPix;
   int  ix, iy;
   float xx, mm;
   long  ii;
   int iKey;
   long nRow;
   int iarg, iBin;

   char outFileName[130];

   fitsfile *inFile;
   fitsfile *outFile = NULL;

   imaging_fdata_table_def *odata, *idata;
   imaging_detector_table_def *DetectorDef;
   imaging_fdata_row *outData, *inData;
   float *maskImages[12];
   int nxList[12], nyList[12];
   int nRegion;


   float *ATarget1, *ATarget2; 
   float *ASky1, *ASky2;
   float *BTarget1, *BTarget2; 
   float *BSky1, *BSky2;
   float *ARms1, *ARms2, *BRms1, *BRms2;

   float *ABin1[NBIN], *ABin2[NBIN], *BBin1[NBIN], *BBin2[NBIN];
   float Ascratch, Bscratch, ABscratch;

   float *AMask, *BMask, *ABMask;
   float *ATotal, *BTotal, *ABTotal;

   float *ARMask, *BRMask, *ABRMask;
   float *ARTotal, *BRTotal, *ABRTotal;

   char *highSens = "HIGH_SENS";
   char *bang="!";
   int yTop    = 25;
   int yBottom = 5;
   int one = 1;
   int jitter = 0; /* CAH */
   ierr = 0;

/*
      see if jitter is specified externally
*/
      jitter=0;
      iarg=0;
      if (argc > 4) while (iarg < argc) {
	if(!strcmp("-jitter", argv[iarg])) sscanf(argv[++iarg], "%i", &jitter);
	++iarg;
      }

/*
      open input file and get a lot of parameters for iShutter=1
*/
   i = fits_open_file(&inFile, argv[1], READONLY, &ierr) ;
   if (ierr != 0) {
      printf("cfitsio error %i opening file %s\n", ierr, argv[1]);
      printf("%s \n",strerror(errno));
      return 1;
   }
   ierr = 0;
/*
      prepare output file
*/
   strcpy(outFileName, bang);
   strcat(outFileName, argv[3]);

   fits_create_file (&outFile, outFileName, &ierr);
   if (ierr !=0) {
      fprintf (stderr, "error opening fits output %s %i\n",argv[3], ierr);
      printf("%s \n",strerror(errno));
      return 1;
   }
  i = fits_copy_header(inFile, outFile, &ierr);
/*
      move to detector section
*/
   DetectorDef = read_imaging_detector_table(inFile, &ierr);
   if (ierr !=0) {
      printf("error %i opening detector table in %s\n", 
         ierr, argv[1]);
      return 1;
   }
   nRegion = DetectorDef->nregion;
   for (i=0;i<nRegion;i++) {
      nxList[i] = DetectorDef->naxis[i][0];
      nyList[i] = DetectorDef->naxis[i][1];
   }
   nx = DetectorDef->naxis[0][0];
   ny = DetectorDef->naxis[0][1];

/*
      allocate accumulator arrays
*/

   npix = ny * (long)nx;
   ATarget1 = (float *)malloc(npix * sizeof(float));
   BTarget1 = (float *)malloc(npix * sizeof(float));
   ATarget2 = (float *)malloc(npix * sizeof(float));
   BTarget2 = (float *)malloc(npix * sizeof(float));
   for (i=0; i<NBIN; i++) {
      ABin1[i] = (float *)malloc(npix * sizeof(float));
      ABin2[i] = (float *)malloc(npix * sizeof(float));
      BBin1[i] = (float *)malloc(npix * sizeof(float));
      BBin2[i] = (float *)malloc(npix * sizeof(float));
   }

   AMask  = (float *)malloc(nx * sizeof(float));
   BMask  = (float *)malloc(nx * sizeof(float));
   ABMask = (float *)malloc(nx * sizeof(float));
   ATotal  = (float *)malloc(nx * sizeof(float));
   BTotal  = (float *)malloc(nx * sizeof(float));
   ABTotal = (float *)malloc(nx * sizeof(float));

   ARMask  = (float *)malloc(nx * sizeof(float));
   BRMask  = (float *)malloc(nx * sizeof(float));
   ABRMask = (float *)malloc(nx * sizeof(float));
   ARTotal  = (float *)malloc(nx * sizeof(float));
   BRTotal  = (float *)malloc(nx * sizeof(float));
   ABRTotal = (float *)malloc(nx * sizeof(float));

/*
          now read in the real data
*/
   
   idata = open_imaging_fdata_table(inFile, &ierr);
   if (ierr !=0) {
      printf("error %i opening imaging data in file %s\n", 
         ierr, argv[1]);
      return;
   }
   inData = make_imaging_fdata_row(idata);
   i = fits_get_num_rows (inFile,  &nRow, &ierr);
   ii = 1;
   i = read_imaging_fdata_table(inFile, idata, inData, ii++, &ierr);
   for (ix=0;ix<npix;ix++) ATarget1[ix] = inData->data[0][ix];
   for (ix=0;ix<npix;ix++) ATarget2[ix] = inData->data[1][ix];
   i = read_imaging_fdata_table(inFile, idata, inData, ii++, &ierr);
   for (ix=0;ix<npix;ix++) BTarget1[ix] = inData->data[0][ix];
   for (ix=0;ix<npix;ix++) BTarget2[ix] = inData->data[1][ix];

   if (jitter == 1) {
	memset(ATarget1, 0, npix*sizeof(float));
	memset(ATarget2, 0, npix*sizeof(float));
	memset(BTarget1, 0, npix*sizeof(float));
	memset(BTarget2, 0, npix*sizeof(float));
   }
   for (iBin = 0; iBin<NBIN;iBin++) {
      i = read_imaging_fdata_table(inFile, idata, inData, ii++, &ierr);
      for (ix=0;ix<npix;ix++) ABin1[iBin][ix] = inData->data[0][ix];
      for (ix=0;ix<npix;ix++) ABin2[iBin][ix] = inData->data[1][ix];
      if (jitter == 1) {
	for (ix=0;ix<npix;ix++) ATarget1[ix] += inData->data[0][ix];
	for (ix=0;ix<npix;ix++) ATarget2[ix] += inData->data[1][ix];
      }
      i = read_imaging_fdata_table(inFile, idata, inData, ii++, &ierr);
      for (ix=0;ix<npix;ix++) BBin1[iBin][ix] = inData->data[0][ix];
      for (ix=0;ix<npix;ix++) BBin2[iBin][ix] = inData->data[1][ix];
      if (jitter == 1) {
	for (ix=0;ix<npix;ix++) BTarget1[ix] += inData->data[0][ix];
	for (ix=0;ix<npix;ix++) BTarget2[ix] += inData->data[1][ix];
      }
   }
   if (jitter == 2) {
	for (ix=0;ix<npix;ix++) {
		ATarget1[ix] = ATarget1[ix] / NBIN;
		ATarget2[ix] = ATarget2[ix] / NBIN;
		BTarget1[ix] = BTarget1[ix] / NBIN;
		BTarget2[ix] = BTarget2[ix] / NBIN;
        }
   }
   i = close_imaging_fdata_table(inFile, idata, &ierr);
   kill_imaging_fdata_row(inData);
   if (ierr == 0) printf("Successfully read in input data\n");
/*
      get mask array
*/
   oirGetMaskImages(argv[2], maskImages, DetectorDef->nregion, nxList, nyList, &ierr);
   if (ierr != 0) {
      printf("error %i reading mask data from file %s\n", ierr, argv[2]);
      return 1;
   }
/*
      calculate total fluxes 
*/
   for (ix = 0; ix <nx; ix++) {
      for (iy=yBottom; iy <=yTop; iy++) {
         ii = ix + nx*(long)iy;
         ATotal[ix] = ATotal[ix] + ATarget1[ii] + ATarget2[ii];
         BTotal[ix] = BTotal[ix] + BTarget1[ii] + BTarget2[ii];
         xx = ATarget1[ii] * BTarget1[ii];
         if (xx > 0) ABTotal[ix] = ABTotal[ix] + sqrt(xx);
         xx = ATarget2[ii] * BTarget2[ii];
         if (xx > 0) ABTotal[ix] = ABTotal[ix] + sqrt(xx);
      } /* iy */
/*
      calculate rmss from binned totals
*/
      for (i=0; i<NBIN; i++) {
         Ascratch  = 0.;
         Bscratch  = 0.;
         ABscratch = 0.;
         for (iy=yBottom; iy <=yTop; iy++) {
            ii = ix + nx*(long)iy;
            Ascratch = Ascratch + ABin1[i][ii] + ABin2[i][ii];
            Bscratch = Bscratch + BBin1[i][ii] + BBin2[i][ii];
            xx = ABin1[i][ii] * BBin1[i][ii];
            if (xx > 0) ABscratch = ABscratch + sqrt(xx);
            xx = ABin2[i][ii] * BBin2[i][ii];
            if (xx > 0) ABscratch = ABscratch + sqrt(xx);
         } /* iy */
         Ascratch  = Ascratch  - ATotal[ix];
         Bscratch  = Bscratch  - BTotal[ix];
         ABscratch = ABscratch - ABTotal[ix];
         ARTotal[ix]  = ARTotal[ix]  + Ascratch*Ascratch;
         BRTotal[ix]  = BRTotal[ix]  + Bscratch*Bscratch;
         ABRTotal[ix] = ABRTotal[ix] + ABscratch*ABscratch;
      } /* iBin */
      if (ARTotal[ix] > 0)  ARTotal[ix]  = sqrt(ARTotal[ix]/(NBIN*(NBIN-1)));
      if (BRTotal[ix] > 0)  BRTotal[ix]  = sqrt(BRTotal[ix]/(NBIN*(NBIN-1)));
      if (ABRTotal[ix] > 0) ABRTotal[ix] = sqrt(ABRTotal[ix]/(NBIN*(NBIN-1)));
   } /* ix */

/*
      calculate masked fluxes
*/
   for (ix = 0; ix <nx; ix++) {
      for (iy=yBottom; iy <=yTop; iy++) { 
         ii = ix + nx*(long)iy;
         AMask[ix] = AMask[ix] + ATarget1[ii] * maskImages[0][ii] +
            ATarget2[ii] * maskImages[1][ii];
         BMask[ix] = BMask[ix] + BTarget1[ii] * maskImages[0][ii] +
            BTarget2[ii] * maskImages[1][ii];
         mm = maskImages[0][ii];
         xx = ATarget1[ii]*BTarget1[ii] * mm * mm;
         if (xx > 0) ABMask[ix] = ABMask[ix] + sqrt(xx);
         mm = maskImages[1][ii];
         xx = ATarget2[ii]*BTarget2[ii] * mm * mm;
         if (xx > 0) ABMask[ix] = ABMask[ix] + sqrt(xx);
      } /* iy */
/*
      calculate rmss from binned totals
*/
      for (i=0; i<NBIN; i++) {
         Ascratch  = 0.;
         Bscratch  = 0.;
         ABscratch = 0.;
         for (iy=yBottom; iy <=yTop; iy++) {
            ii = ix + nx*(long)iy;
            Ascratch = Ascratch + ABin1[i][ii] * maskImages[0][ii] +
               + ABin2[i][ii] * maskImages[1][ii];
            Bscratch = Bscratch + BBin1[i][ii] * maskImages[0][ii] + 
               BBin2[i][ii] * maskImages[1][ii];
            mm = maskImages[0][ii];
            xx = ABin1[i][ii]*BBin1[i][ii] * mm * mm;
            if (xx > 0) ABscratch = ABscratch + sqrt(xx);
            mm = maskImages[1][ii];
            xx = ABin2[i][ii]*BBin2[i][ii] * mm * mm;
            if (xx > 0) ABscratch = ABscratch + sqrt(xx);
         } /* iy */
         Ascratch  = Ascratch  - AMask[ix];
         Bscratch  = Bscratch  - BMask[ix];
         ABscratch = ABscratch - ABMask[ix];
         ARMask[ix]  = ARMask[ix]  + Ascratch*Ascratch;
         BRMask[ix]  = BRMask[ix]  + Bscratch*Bscratch;
         ABRMask[ix] = ABRMask[ix] + ABscratch*ABscratch;
      } /* iBin */
      if (ARMask[ix] > 0) ARMask[ix] = sqrt(ARMask[ix]/(NBIN*(NBIN-1)));
      if (BRMask[ix] > 0) BRMask[ix] = sqrt(BRMask[ix]/(NBIN*(NBIN-1)));
   } /* ix */
   DetectorDef->nregion = 1;
   DetectorDef->naxis[0][1] = 1;
   iKey = table_find_key(DetectorDef->def, "NREGION");
   table_put_keyword(DetectorDef->def, iKey, (void *)&one, &ierr);

   odata = make_imaging_fdata_table_def (outFile, DetectorDef, 1, &ierr);
   if (ierr !=0) {
      fprintf (stderr, "error creating data definition\n");
      return 1;
   }
   outData = make_imaging_fdata_row(odata);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = ATotal[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 1, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = BTotal[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 2, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = ABTotal[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 3, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = AMask[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 4, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = BMask[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 5, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = ABMask[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 6, &ierr);
   for (ix=0;ix<nx;ix++) outData->data[0][ix] = ARTotal[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 7, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = BRTotal[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 8, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = ABRTotal[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 9, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = ARMask[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 10, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = BRMask[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 11, &ierr);

   for (ix=0;ix<nx;ix++) outData->data[0][ix] = BRMask[ix];
   i = write_imaging_fdata_table(outFile, odata, outData, 12, &ierr);
   if (ierr == 0) printf("Successfully wrote output data\n");
   i = close_imaging_fdata_table(outFile, odata, &ierr);
   kill_imaging_fdata_row(outData);

   free(ATarget1);
   free(ATarget2);
   free(BTarget1);
   free(BTarget2);
   /* CAH
   free(ASky1);
   free(ASky2);
   free(BSky1);
   free(BSky2);
   */
   free(AMask);
   free(BMask);
   free(ABMask);
   free(ATotal);
   free(BTotal);
   free(ABTotal);
   free(ARMask);
   free(BRMask);
   free(ABRMask);
   free(ARTotal);
   free(BRTotal);
   free(ABRTotal);
   for (i=0;i<nRegion;i++) free(maskImages[i]);
   for (i=0; i<NBIN; i++) {
      free(ABin1[i]); 
      free(ABin2[i]); 
      free(BBin1[i]); 
      free(BBin2[i]); 
   }
   return 0;
}
