/*
 * Copyright 2005, 2006 University of Leiden.
 *
 * This file is part of MIA+EWS.
 *
 * MIA+EWS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MIA+EWS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MIA; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include "fitsio.h"
#include "imaging_fdata.h"
#include "delay.h"
#include "minrtsFftUtil.h"
#define clight  2.9979250e8 /*m per sec */
#define twopi   6.2831853072
#define NFFTGRISM 1024
#define NFFTPRISM 512
#define WAVENOMARGIN 4.e5
#define SMOOTHSIGMA  4.0
#define ENDDROP 0.1  /* fraction pts at ends of delayogram to drop */
/*#define DEBUG 1*/

/*
    Usage::
    oirGroupDelay  DataFile DelayFile -smooth smooth -weight weightfile
*/
void Usage()
{
  printf ("Usage: oirGroupDelay DataFile DelayFile ");
  printf ("[-smooth smooth] [-weight weightfile] [-search] \n\n");
  exit(1);
}
/* ************************************************************************
 * Find the index and value of the maximum and mininum of a float array
 * Inputs:
 *         a    array of input values 
 *         n    number of input values
 * Outputs:
 *         *min value of mininum of array
 *         *max value of maximum of array
 *         *imin position (0-rel) of minimum value in array
 *         *imax position (0-rel) of maximum value in array
 *   
 *********************************************************************** */
void mmax(float *a, int n, float *min, float *max, int *imin, int *imax) {
   int i,j,k;
   float in,ax;
   j=0;
   k=0;
   in = a[0];
   ax = in;
   for (i=0;i<n;i++){
      if (a[i] > ax) {
         ax = a[i];
         j  = i;
      } 
      if (a[i] < in) {
         in = a[i];
         k  = i;
      } 
   }
   *imin = k;
   *imax = j;
   *min  = in;
   *max  = ax;
   return;
}

/* **************************************************************************
 *  Sum the rows of a buffer with relative weights exp(-0.5*((iRow-oRow)/sigma))^2)
 *  INPUTS:
 *         buffer[2*nY+1][nX]   array of input rows
 *         inRows[2*nY+1]       array of input row positions, one for each row
 *         oRow                 "central" input row position
 *         nX                   pixels per row
 *         nY                   number of rows is 2*nY + 1
 *         sigma                standard deviation of weighting function in rows
 *         smooth[nX]           output row
 ************************************************************************** */
void smoothDelayData (float *buffer, int *inRows,  int oRow, int nX, int nY, float sigma, float *smooth) {
   int i,j;
   int nx2;
   float weight, totalWeight;
   float x;

   nx2 = 2*nY + 1;
   totalWeight = 0.;  /* total of weights */
   for (j=0;j<nX;j++) smooth[j] = 0.;/* zero accumulator */

   for (i=0; i<nx2; i++) if (inRows[i] >= 0){
      x = (inRows[i]-oRow)/sigma;
      weight = exp(-0.5* x*x);
      totalWeight = totalWeight + weight;
      for (j=0;j<nX;j++) smooth[j] = smooth[j] + weight * (*(buffer+j+i*nX));
   }
   if (totalWeight > 0.) for (j=0;j<nX;j++) smooth[j] = smooth[j]/totalWeight;
   return;
}
/* **************************************************************************
 *    compute absolute value of delay function, 
 *    find peak, do parabolic interpolation to subpixel accuracy
 *    around peak, and return position and value of peak
 *  INPUTS:
 *      fft        fft object containing coordinate and pixel information
 *      delayData  complex delay function data
 *  OUTPUTS:
 *      mx         value of interpolated peak of absolute value of delayData
 *  RETURNS:
 *                 position of interpolated peak of absolute value, in units specified by fft
 *      
 ************************************************************************** */
double getGroupOpd(minrtsFft1d *fft, float *delayData, float *mx){
   float nX;
   float mn;
   int ix, in;
   int i;
   int nEnd;
   float ix2;

   nX = fft->nX;
   nEnd = (int)(ENDDROP * nX);
/* 
      compute square of abs value and move to 1st half of array 
*/
   for (i=0;i<nX;i++) delayData[i] = sqrt(delayData[2*i]*delayData[2*i]
      +delayData[2*i+1]*delayData[2*i+1]);
   mmax (delayData+nEnd, nX-2*nEnd, &mn, mx, &in, &ix); /* min max and position */
   ix = ix + nEnd;
   if (ix < 3 || (ix > (nX-4))) ix2 = (double)fft->kGrid[ix]; 
    else ix2 = minrtsQPk(7, fft->kGrid+ix-3, delayData+ix-3, mx, 0);
    return (double)ix2;

} /* end of get Group Opd */

int main (int argc, char** argv) {

   int i, j, ierr;
   float x;
   int nfft, nfft2;
   long nRow, iRow, oRow, pRow ;
   int nx, nx2 ;
   float *frequency, *waveNo;
   float minWave, maxWave;

   char outFileName[130];
   char *bang="!";
   char* infile_name     = NULL;
   char* outfile_name    = NULL;
   char* weightfile_name = NULL;
   fitsfile *inFile;
   fitsfile *outFile = NULL;

   char gris[12];
   char history[80];

   int iKey, maxstep;
   double opd, opd0;
   float  opd1, opd2, opdScan;
   float opdMin, opdMax;
   float *rotOpd;
   imaging_fdata_table_def *idata;
   imaging_fdata_table_def *odata;
   imaging_detector_table_def *DetectorDef;
   imaging_fdata_row *inData;
   imaging_fdata_row *outData;
   imaging_fdata_row *weightData;

   delay_table_row *delayData;
   delay_table_def *dTable;
   double *delayTimes;
   double *groupDelays;
   float  *delayAmp;

   minrtsFft1d *groupFft;
   float *work1, *work2, *work3, *work4;
   float gridParms[3];


   float   *smBuffer, *smBufferOut;
   int     smBufferWidth, smBW2;
   int     *smBufferI;
   long    *smFrame;
   int     smI;
   float   opt_smooth;

   int doSearch; /* search mode rather than track mode */

   int iarg;  /* for parsing input */

#ifdef DEBUG
   FILE *debugFile;
#endif
   float  rms, mean;
   float  amplitudeRMS = 0.;
   float  amplitudeMean = 0.;
   int    nRMS = 0;

   ierr       = 0;
   doSearch   = 0;
   weightData = NULL;


   /**
    * Parse the command line options 
    */
   if (argc < 3) {
      Usage();
      return 1;
   }
   infile_name  = argv[1];
   outfile_name = argv[2];

   opt_smooth = SMOOTHSIGMA; /* default */
   iarg = 3; 
   while (iarg < argc ) {
      if (!strcmp("-smooth", argv[iarg])) sscanf(argv[++iarg],"%f",&opt_smooth); 
      if (!strcmp("-weight", argv[iarg])) weightfile_name=argv[++iarg];
      if (!strcmp("-search",argv[iarg])) doSearch = 1;
      ++iarg;
   }
   printf("      ******   oirGroupDelay   ******\n");
   printf ("   Smoothing over %f bins\n",opt_smooth);
   if (doSearch == 1) printf("   Operating in SEARCH mode\n"); 
      else printf("   Operating in TRACK mode\n");
/*
      is there a weight file?  If so read it in.
*/
   if (weightfile_name != NULL) {
      printf("Using weightfile %s\n",weightfile_name);
      fits_open_file (&inFile, weightfile_name, READONLY, &ierr);
      if (ierr != 0) {
         printf("ERROR:fits open weight file of %s gave error %i\n",weightfile_name,
            ierr);
      printf("%s \n",strerror(errno));
         return 1;
       }
      idata          = open_imaging_fdata_table(inFile, &ierr);
      if (ierr != 0) {
         printf("ERROR:fits open fdata weight file of %s gave error %i\n",weightfile_name,
            ierr);
         return 1;
       }
      weightData = make_imaging_fdata_row(idata);
      i = read_imaging_fdata_table(inFile, idata, weightData, 1, &ierr);
      if (ierr != 0) {
         printf("ERROR:fits read weight file of %s gave error %i\n",weightfile_name,
            ierr);
         return 1;
       }
      i = close_imaging_fdata_table(inFile, idata, &ierr);
/*      fits_close_file(inFile, &ierr);  previous statement does this*/
   }
/*
      open input file and get detector info
*/
#ifdef DEBUG
   debugFile = fopen("/tmp/debug.dat","w");
#endif
   fits_open_file (&inFile, infile_name, READONLY, &ierr);
   if (ierr != 0) {
      printf("ERROR:fits openfile of %s gave error %i\n",infile_name,
         ierr);
      printf("%s \n",strerror(errno));
      return 1;
   }
   i = fits_read_key(inFile, TSTRING, "HIERARCH ESO INS GRIS NAME", gris,0,&ierr);
   printf("   disperser is = %s\n", gris);
/*
      create output data table 
*/
   strcpy(outFileName, bang);
   strcat(outFileName, outfile_name);

   fits_create_file (&outFile, outFileName, &ierr);
   if (ierr !=0) {
      printf ("ERROR: opening fits output %s %i\n",outfile_name, ierr);
      printf("%s \n",strerror(errno));
      return 1;
      }
/*
      copy header and add some new keywords
*/
   i = fits_copy_header(inFile, outFile, &ierr);
   sprintf(history,"oirGroupDelay INFILE %s", infile_name);
   i = fits_write_history(outFile, history, &ierr);
   i = fits_write_key(outFile, TFLOAT, "OSMOOTH", (void*)&opt_smooth, 
      "Smoothing halfwidth in oirGroupDelay", &ierr);
   DetectorDef = read_imaging_detector_table(inFile, &ierr);
   nx  = DetectorDef->naxis[0][0];
   nx2 = nx/2;
/*       
      get frequency information from detector table
      screw around with naxis; this is necessary
      because data in .insopd file is "pseudocomplex"
      so nx is twice what it is supposed to be.
*/
   DetectorDef->naxis[0][0] = nx2;
   frequency = imaging_detector_wcs(DetectorDef, 1);
   waveNo = (float *)malloc(nx2 * sizeof(float));
   for (i=0; i<nx2; i++) waveNo[i] = twopi * frequency[i]/clight;
   rotOpd = (float *)malloc(nx * sizeof(float)); /* phase rotation in search mode */
/*
      configure data table and get MAXSTEP
*/
   idata          = open_imaging_fdata_table(inFile, &ierr);
   iKey = table_find_key(idata->def, "MAXSTEP");
   if (iKey>0)  table_get_keyword (idata->def, iKey, (void*)&maxstep,
      &ierr);

/* 
      pick nfft and
      configure smoothing buffer
*/
   if (gris[0] == 'G') nfft = NFFTGRISM; else nfft = NFFTPRISM;
   nfft2         = 2* nfft;
   smBW2         = (int)(.5+opt_smooth);
   smBufferWidth = 1 + 2* smBW2;
   smBuffer      = (float *)malloc(smBufferWidth * nfft2 *sizeof(float));
   smBufferOut   = (float *)malloc(nfft2 * sizeof(float));
   smBufferI     = (int *)malloc(smBufferWidth *sizeof(int));
   for (i=0;i<smBufferWidth;i++) smBufferI[i] = -32000;
   
/*     
   row size(viewed as complex); alter it for gridded output
*/
   DetectorDef->naxis[0][0] = nfft2;

   odata = make_imaging_fdata_table_def (outFile, 
      DetectorDef, maxstep, &ierr); 
   if (ierr !=0) {
      printf ("ERROR: creating data definition\n");
      return 1;
      }
   inData = make_imaging_fdata_row(idata);
   outData = make_imaging_fdata_row(odata);
   i = fits_get_num_rows (inFile,  &nRow, &ierr); 
   printf("   Total Number of Data rows = %li\n", nRow);
/*
      set up FFT objects and work arrays
*/
   mmax(waveNo, nx2, &minWave, &maxWave, &i, &j); 
   gridParms[0]  = minWave - WAVENOMARGIN;
   gridParms[1]  = (maxWave - minWave + 2.*WAVENOMARGIN)/nfft;
   gridParms[2]  = nfft;
   groupFft = minrtsMakeFft1d(gridParms, 0);

   opd1 = -groupFft->kGrid[0];
   opd2 = groupFft->kGrid[0] - groupFft->kGrid[1];
   work1 = (float *)malloc(nx * sizeof(float));
   work2 = (float *)malloc(nfft2 * sizeof(float));
   work3 = (float *)malloc(nx * sizeof(float));
   work4 = (float *)malloc(nfft2 * sizeof(float));
   smFrame = (long *)malloc(nRow * sizeof(long));
/*       
         get OPD offset value from header
*/
   iKey = table_find_key(idata->def, "OPD0");
   if (iKey>0)  table_get_keyword (idata->def, iKey, (void*)&opd0, &ierr);
   opdScan = opd0; /* initialize to harmless values */
   opdMin  = opd0 + 0.5 * groupFft->kGrid[0];
   opdMax  = opd0 - 0.5 * groupFft->kGrid[0];

/*
         set up tables to calculate and output group delays
*/
   delayTimes  = (double *)malloc(nRow * sizeof(double));
   groupDelays = (double *)malloc(nRow * sizeof(double));
   delayAmp    = (float *)malloc(nRow * sizeof(float));
/*
         now loop over input data, copy some info, take ffts etc.
*/ 
   smI = 0;
   oRow = 0;
   printf("   done with row:  ");
   for (iRow=1; iRow<=nRow; iRow++) {
      i = read_imaging_fdata_table(inFile, idata, inData, iRow, &ierr);

/*
            copy nonData info
*/
      outData->time      = inData->time;
      delayTimes[iRow-1] = inData->time;
      outData->opd[0]    = inData->opd[0];
      outData->opd[1]    = inData->opd[1];
      outData->local_opd[1] = inData->local_opd[1];
      outData->local_opd[0] = inData->local_opd[0];
      outData->exp       = inData->exp;
      outData->offset[0] = inData->offset[0];
      outData->offset[1] = inData->offset[1];
      outData->rotation  = inData->rotation;
      outData->frame     = inData->frame;
      smFrame[iRow-1]    = inData->frame;
      outData->step      = inData->step;
      outData->reference = inData->reference;
      outData->targtype[0] = inData->targtype[0];  
/*
        if search mode, put back opd0 - opdScan in phase
        change center of FFT opd range to current opd if it falls
        outside the current search range.

        If you do this, zap the current smoother buffer by setting row
        values to large negative numbers
*/
      if (doSearch) {
         opd = inData->opd[1]+inData->local_opd[1] - inData->opd[0] - inData->local_opd[0];
         if ((opd > opdMax) || (opd < opdMin) || (iRow == 1)) {
            opdScan = opd0 - opd;
            for (j=0; j<nx2; j++) {
               rotOpd[2*j]   = cos(opdScan * waveNo[j]);
               rotOpd[2*j+1] = sin(opdScan * waveNo[j]);
            }
            opdMin  = opd + 0.5 * groupFft->kGrid[0];
            opdMax  = opd - 0.5 * groupFft->kGrid[0];
/*            for (i=0; i< smBufferWidth; i++) smBufferI[i] = -32000; */
         }
         for (j=0; j<nx2; j++) {
            x = inData->data[0][2*j]*rotOpd[2*j] - 
               inData->data[0][2*j+1]*rotOpd[2*j+1];
            inData->data[0][2*j+1] = inData->data[0][2*j]*rotOpd[2*j+1] +
               inData->data[0][2*j+1]*rotOpd[2*j];
            inData->data[0][2*j] = x;
         }
      }
/*
        weight in frequency domain if specified
*/
      if (weightData != NULL) {
         for (j=0; j<nx2; j++) {
            inData->data[0][2*j] = weightData->data[0][j]*inData->data[0][2*j];
            inData->data[0][2*j+1] = weightData->data[0][j]*inData->data[0][2*j+1];
         }
      }
/*
        do complex ffts and output
*/
      
      minrtsFftGridFftC(groupFft, nx2, waveNo, inData->data[0], 0, outData->data[0]);
 
      i = write_imaging_fdata_table(outFile, odata, outData, iRow, &ierr);
      if (ierr != 0) {
         printf("ERROR:output error at line %li %i\n", oRow, ierr);
         return 1;
      }
/*
         copy into smoothing buffer
*/
      memcpy (smBuffer+smI*nfft2, outData->data[0], nfft2*sizeof(float));
      smBufferI[smI] = smFrame[iRow-1]; 
      if (++smI == smBufferWidth) smI = 0;
/*
         which row should we output (if any)
*/
/* CAH: this is (probably) a bug
      if (iRow < smBufferWidth) pRow = (int)(.5*(iRow+1)); else pRow = iRow - smBW2;
      if (pRow > oRow) {
         oRow = pRow;
*/
      if (iRow >= smBufferWidth) {
         oRow++;
/*
         sum with gaussian weights into averaging buffer
*/
         smoothDelayData (smBuffer, smBufferI,  smFrame[oRow-1], nfft2, smBW2,  
            opt_smooth, smBufferOut);
#ifdef DEBUG
   for (i=0;i<nfft2;i++) fprintf(debugFile,"%f\n", smBufferOut[i]); 
#endif
/*            
         compute some statistics on the data
*/
         minrtsMoments(nfft2, smBufferOut, &mean, &rms);
         amplitudeMean = amplitudeMean + mean;
         amplitudeRMS  = amplitudeRMS  + rms;
         ++nRMS;

         groupDelays[oRow-1] = getGroupOpd(groupFft, smBufferOut, delayAmp+oRow-1);
         if (doSearch) groupDelays[oRow-1] = groupDelays[oRow-1]+opdScan;
      }  /* end of new output necessary */
      j = iRow/100;
      if ((iRow-j*100)==0) fprintf(stderr,"Frame:%li\r ",iRow);
   }  /*  loop over input rows */
/*
         process last output rows
*/
   for (pRow = oRow+1; pRow < nRow; pRow++) {
/* CAH: if-else added to speed up full length smoothing */
	 if ((int)opt_smooth < nRow || pRow == (oRow+1)) {
         smoothDelayData (smBuffer, smBufferI,  smFrame[pRow-1], nfft2, smBW2, opt_smooth, smBufferOut);
#ifdef DEBUG
   for (i=0;i<nfft2;i++) fprintf(debugFile,"%f\n", smBufferOut[i]);
#endif
         groupDelays[pRow-1] = getGroupOpd(groupFft, smBufferOut, delayAmp+pRow-1);
         if (doSearch) groupDelays[pRow-1] = groupDelays[pRow-1]+opdScan;
	 } else groupDelays[pRow-1] = groupDelays[pRow-2];
   }
   printf("%li\n", pRow-1);
   i = fits_write_key(outFile, TDOUBLE, "OPD0", (void*)(&opd0), "OFFSET TO OPD", &ierr);
   i = fits_write_key(outFile, TFLOAT, "OPD1", (void*)(&opd1), " ", &ierr);
   i = fits_write_key(outFile, TFLOAT, "OPD2", (void*)(&opd2), " ", &ierr);
   i = fits_write_key(outFile, TFLOAT, "GRIDPAR1", (void*)(gridParms), "WaveNo Grid Zero Point", &ierr);
   i = fits_write_key(outFile, TFLOAT, "GRIDPAR2", (void*)(1+gridParms), "WaveNo Grid Delta", &ierr);
   kill_imaging_fdata_table_def(idata);
   kill_imaging_fdata_table_def(odata);
   kill_imaging_fdata_row(inData);
   kill_imaging_fdata_row(outData);
#ifdef DEBUG
   fclose(debugFile);
#endif
   if (weightData != NULL) kill_imaging_fdata_row(weightData);
/*
       compute average statistics
*/
   if (nRMS > 0) {
      amplitudeRMS = sqrt(2.)*amplitudeRMS/nRMS;
      amplitudeMean = amplitudeMean/nRMS;
   }
   printf("   RMS delay amplitude %f \n", amplitudeRMS);
/*
       calculate and output estimates of group delay
*/
   dTable = make_delay_table_def(outFile, "1776-07-04", &ierr);
   i = fits_write_key(outFile, TFLOAT, "DNOISE", (void*)(&amplitudeRMS), 
      "Delay Amplitude Noise", &ierr);
   delayData = make_delay_table_row(dTable);
   for (iRow=1;iRow<=nRow;iRow++) {
      delayData->time      = delayTimes[iRow-1];
      delayData->telescope = 2;
      delayData->amplitude = delayAmp[iRow-1];
      delayData->delay     = (opd0-groupDelays[iRow-1])/clight; /* in seconds */
      i = write_delay_table(outFile, dTable, delayData, iRow, &ierr);
   }
   kill_delay_table_def(dTable);
   kill_delay_table_row(delayData);

   fits_close_file(inFile, &ierr);
   fits_close_file(outFile, &ierr);
   free(frequency);
   free(waveNo);
   free(work1);
   free(work2);
   free(work3);
   free(work4);
   free(delayTimes);
   free(delayAmp);
   free(groupDelays);
   free(smBuffer);
   free(smBufferI);
   free(smBufferOut);
   free(smFrame);
   free(rotOpd);
   minrtsKillFft1d(groupFft);
   return 0;
}
