;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
; $Id: chop_nod_phot.pro,v 1.11 2005/05/23 11:48:32 koehler Exp $
;
; Last Change: Sun May 22 00:28:10 2005
;
; Main routine is "chop_nod_phot"
;
; INPUTS:
;	fname	      MIDI chopping or nodding data
;	fnamesky      MIDI nodding data of sky
;	threshold     convergence factor to determine best aperture radius
;	fwhm          limit of FWHM allowed for single frame photometry
;	/CHOPDIR      determine the chopping direction with 'U' states
;       /FRAMES       check photometry for each 'T' state
;	/CONV         results multiplied with conversion factor 145
;	/SILENT       tells routine to be non interactive
;	/REVERSE      flags for object and sky are reversed (Feb.2003 run)
;	/WRITE        write acquisition images to disk (fits format)
; OUTPUTS:
;	an ascii file in subdirectory PHOTOMETRY
;
; TODO: 
;	is Beam A always in data2 and Beam B in data1?
;	is the orientation correct?
;
;-----------------------------------------------------------------------
;-----------------------------------------------------------------------

;draw a circle (by Kevin Ivory)
function circle, xcenter, ycenter, radius
points = (2*!PI/99.0) * FINDGEN(100)
x = xcenter + radius *cos(points)
y = ycenter + radius *sin(points)
return,transpose([[x],[y]])
end

;-----------------------------------------------------------------------

function trunc,nb
if nb-fix(nb) gt 0.5 then result=fix(nb)+1 else result=fix(nb)
return, result
end

;-----------------------------------------------------------------------
;-----------------------------------------------------------------------
;---------------	MAIN PROG	--------------------------------
;-----------------------------------------------------------------------

pro chop_nod_phot, fname, skyname, FWHM=fwhm, THRESHOLD=threshold, CHOPDIR=chopdir, $
                   FRAMES=frames, CONV=conv, SILENT=silent, REVERSE=reverse, WRITE=write

  chopping = 1
  print,'tartyp used to determine chopping-direction: ', chopping

  use = 2
  print,'tartyp used for photometric frames: ', use

  factor = 5
  photo_test = 10

  convfwhm=(2*SQRT(2)*SQRT(ALOG(2)))

  if not keyword_set(fwhm) then fwhm = 4/convfwhm else fwhm = fwhm/convfwhm
  if not keyword_set(threshold) then threshold = 0.1

  ; make once and for all sure that fname is a scalar
  fname = fname[0]
  hdrname= (oirFileList(fname))[0]

  if not file_test("PHOTOMETRY") then spawn,'mkdir "PHOTOMETRY"'

  if N_Params() gt 1  then begin
      if not keyword_set(silent) then $
        print, 'skyname given, this must be nodding data'

      skyname = skyname[0]

      ;im_on= oirgetmeanrms(fname)
      spawn, 'rm PHOTOMETRY/nod_ftemp.fits PHOTOMETRY/nod_skytemp.fits'
      spawn, getenv("vltiCbin") + '/oirMeanRMS "' + fname + '" PHOTOMETRY/nod_ftemp.fits'
      im_on= oirgetdata('PHOTOMETRY/nod_ftemp.fits')

      ;im_sky = oirgetmeanrms(skyname)
      spawn, getenv("vltiCbin") + '/oirMeanRMS "' + skyname + '" PHOTOMETRY/nod_skytemp.fits'
      im_sky= oirgetdata('PHOTOMETRY/nod_skytemp.fits')

      win1 = im_on[0].data1-im_sky[0].data1
      win2 = im_on[0].data2-im_sky[0].data2
      win1_stdev= im_on[1].data1-im_sky[1].data1
      win2_stdev= im_on[1].data2-im_sky[1].data2

  endif else begin
      if not keyword_set(silent) then $
        print, 'no skyname given, this must be chopping data'

      ;-------------- Walter routine to extract chooped images --------------
      ;------- before and after keywors are valid only for Feb. Chopping-----
      ;------- It was a bad tuning --------------------------------

      chop_image= MIDIChopImageC(fname)
      if keyword_set(reverse) then begin
          on_source_win1 = chop_image[1].data1
          on_source_win2 = chop_image[1].data2
          off_source_win1= chop_image[0].data1
          off_source_win2= chop_image[0].data2
      endif else begin
          on_source_win1 = chop_image[0].data1
          on_source_win2 = chop_image[0].data2
          off_source_win1= chop_image[1].data1
          off_source_win2= chop_image[1].data2
      endelse
      win1 = on_source_win1-off_source_win1
      win2 = on_source_win2-off_source_win2
  endelse

  win1(where(win1 lt 1))=1
  win2(where(win2 lt 1))=1

  n=size(win1)
  x_size=n[1]
  y_size=n[2]

  ;FOV_x1 = 15
  ;FOV_x2 = 45
  ;FOV_y1 = 15
  ;FOV_y2 = 45

  FOV_x1 = 0
  FOV_x2 = x_size-1
  FOV_y1 = 0
  FOV_y2 = y_size-1

  im=obj_new('fitsfile',hdrname)
  he = im->prihead()
  OBJECT    = STRCOMPRESS(he->getpar('OBJECT'),/remove_all)
  DIT       = he->getpar('DET DIT')
  obj_destroy,im

  first_char    = STRPOS(fname,'/MIDI.20')+6
  last_char	  = STRPOS(fname,'.fits')
  short_file_name=STRMID(fname,first_char,last_char-first_char)

  ; conversion factor
  if keyword_set(conv) then conv = 145.   
  if not keyword_set(conv) then conv = 1.  

;---------------- WRITE ACQUISITION  --------------

if keyword_set(write) then begin
  writefits,'PHOTOMETRY/'+short_file_name+'_1.fits',on_source_win1
  writefits,'PHOTOMETRY/'+short_file_name+'_2.fits',off_source_win1
  writefits,'PHOTOMETRY/'+short_file_name+'_3.fits',win1
  writefits,'PHOTOMETRY/'+short_file_name+'_4.fits',on_source_win2
  writefits,'PHOTOMETRY/'+short_file_name+'_5.fits',off_source_win2
  writefits,'PHOTOMETRY/'+short_file_name+'_6.fits',win2
endif  

;---------------- CHOPPING DIRECTION --------------
  
if keyword_set(chopdir) then begin

  ima=oirgetdata(fname)
  image_win1=FLOAT(ima.data1)+32768.
  image_win2=FLOAT(ima.data2)+32768.
  cleanname= (oirFileList(fname))[0]

  m=SIZE(ima.data1)

  if chopping eq 1 then begin 
      chopping = ima.TARTYP1 
  endif else begin
      chopping = ima.TARTYP2 
    endelse
  
  if factor gt 3 then fct = factor - 2

  window, 0, xsize=fct*m[1]*4, ysize=fct*m[2]*2
  tvscl, rebin(sqrt(win1),m[1]*fct,m[2]*fct),4
  tvscl, rebin(sqrt(win2),m[1]*fct,m[2]*fct),0

  unknown1 = win1 - win1
  unknown2 = win2 - win2
  rms1 = fltarr(m[1],m[2],trunc(m[3]))
  rms2 = fltarr(m[1],m[2],trunc(m[3]))
  std1 = fltarr(m[1],m[2])
  std2 = fltarr(m[1],m[2])
  nounc = 0

  for frame_index=0,m[3]-1 DO BEGIN
      if (chopping[frame_index] EQ 'U ') THEN BEGIN
	 tmp_im1 = FLOAT(image_win1[*,*,frame_index])
	 tmp_im2 = FLOAT(image_win2[*,*,frame_index])
	 unknown1 = unknown1 + tmp_im1
	 unknown2 = unknown2 + tmp_im2
	 rms1[*,*,nounc] = tmp_im1 - off_source_win1
	 rms2[*,*,nounc] = tmp_im2 - off_source_win2
	 tvscl, rebin(tmp_im1 - off_source_win1,m[1]*fct,m[2]*fct),5
	 tvscl, rebin(tmp_im2 - off_source_win2,m[1]*fct,m[2]*fct),1
	 nounc = nounc+1
      endif
  endfor


  ; try to identify the chopping direction

  unknown1=rebin(unknown1/nounc-off_source_win1,m[1]*fct,m[2]*fct)
  unknown2=rebin(unknown2/nounc-off_source_win2,m[1]*fct,m[2]*fct)
  tvscl,unknown1,6
  tvscl,unknown2,2


  for i=0,m[1]-1 DO BEGIN
  
    for j=0,m[2]-1 DO BEGIN
  
      std1[i,j]=stddev(rms1[i,j,0:nounc-1])
      std2[i,j]=stddev(rms2[i,j,0:nounc-1])
  
    endfor
  
  endfor

  std1=rebin(std1-mean(std1),m[1]*fct,m[2]*fct)
  std2=rebin(std2-mean(std2),m[1]*fct,m[2]*fct)
  tvscl,std1,7
  tvscl,std2,3


  ; display number of used U states

  print,'Number of U states : ',nounc

endif



;---------------- PHOTOMETRY OF EACH FRAME --------------
  
if keyword_set(frames) then begin

  chop_frames= MIDIChopFrames(fname, before=0, after=0)
  size = size(chop_frames)
  NDIT = size[3]
  
  flag1 = chop_frames.tartyp1
  flag2 = chop_frames.tartyp2

  notar = 0
  ntar1 = 0
  ntar2 = 0
  temp1 = 0
  temp2 = 0
  file  = 0
  
  x1=findgen(NDIT)-findgen(NDIT)
  y1=findgen(NDIT)-findgen(NDIT)
  x2=findgen(NDIT)-findgen(NDIT)
  y2=findgen(NDIT)-findgen(NDIT)
  fx1=findgen(NDIT)-findgen(NDIT)
  fy1=findgen(NDIT)-findgen(NDIT)
  fx2=findgen(NDIT)-findgen(NDIT)
  fy2=findgen(NDIT)-findgen(NDIT)
  m1=findgen(NDIT)-findgen(NDIT)
  m2=findgen(NDIT)-findgen(NDIT)

  window,1,xsize=2*n[1]*factor,ysize=n[2]*factor
 
  for l = 0, NDIT-1 do begin
   
    on1 = chop_frames[l].data1
    on2 = chop_frames[l].data2

    on1(where(on1 lt 1)) = 1
    on2(where(on2 lt 1)) = 1

    tvscl, rebin(sqrt(on1),n[1]*factor,n[2]*factor),1
    tvscl, rebin(sqrt(on2),n[1]*factor,n[2]*factor),0
    
    XYOUTS,0        *factor,(n[2]-5)*factor,l         ,/DEVICE
    XYOUTS,1        *factor,5              ,'Tartyp2:',/DEVICE
    XYOUTS,10       *factor,5              ,flag2[l]  ,/DEVICE
    XYOUTS,(1+n[1]) *factor,5              ,'Tartyp1:',/DEVICE
    XYOUTS,(10+n[1])*factor,5              ,flag1[l]  ,/DEVICE

    if use eq 2 then begin 
      XYOUTS,13*factor,5,'(in use)',/DEVICE
      flag = flag2
    endif else begin
      XYOUTS,(13+n[1])*factor,5,'(in use)',/DEVICE
      flag = flag1
    endelse

    if (flag[l] eq 'T ' ) then begin

      notar = notar + 1
 
      print,''
      fit2D_A= gauss2dfit(on2(FOV_x1:FOV_x2,FOV_y1:FOV_y2),a)
      print, 'Star gauss. A:', a[4]+FOV_x1, a[5]+FOV_y1
      print, 'FWHM x, y',  a[2]*convfwhm, a[3]*convfwhm
      print, 'Max image', max(on2, status), ' Pos X = ', status mod x_size, ' Pos y = ', trunc(status/x_size)
      xm1 = status mod x_size
      ym1 = trunc(status/x_size)
      print,''
      fit2D_B= gauss2dfit(on1(FOV_x1:FOV_x2,FOV_y1:FOV_y2),b)
      print, 'Star gauss. B:', b[4]+FOV_x1, b[5]+FOV_y1
      print, 'FWHM x, y',  b[2]*convfwhm, b[3]*convfwhm
      print, 'Max image', max(on1, status), ' Pos X = ', status mod x_size, ' Pos y = ', trunc(status/x_size)
      xm2 = status mod x_size
      ym2 = trunc(status/x_size)

      PLOTS,[xm1*factor-3,xm1*factor+3],[ym1*factor+3,ym1*factor-3],/DEVICE,Color=1
      PLOTS,[xm1*factor+3,xm1*factor-3],[ym1*factor+3,ym1*factor-3],/DEVICE,Color=1
      PLOTS,[(xm2+n[1])*factor-3,(xm2+n[1])*factor+3],[ym2*factor+3,ym2*factor-3],/DEVICE,Color=1
      PLOTS,[(xm2+n[1])*factor+3,(xm2+n[1])*factor-3],[ym2*factor+3,ym2*factor-3],/DEVICE,Color=1

      PLOTS,[a[4]*factor-5,a[4]*factor+5],[a[5]*factor,a[5]*factor],/DEVICE,Color=1
      PLOTS,[a[4]*factor,a[4]*factor],[a[5]*factor-5,a[5]*factor+5],/DEVICE,Color=1
      PLOTS,[(b[4]+n[1])*factor-5,(b[4]+n[1])*factor+5],[b[5]*factor,b[5]*factor],/DEVICE,Color=1
      PLOTS,[(b[4]+n[1])*factor,(b[4]+n[1])*factor],[b[5]*factor-5,b[5]*factor+5],/DEVICE,Color=1

      x2[l]=a[4]
      y2[l]=a[5]
      fx2[l]=a[2]
      fy2[l]=a[3]

      x1[l]=b[4]
      y1[l]=b[5]
      fx1[l]=b[2]
      fy1[l]=b[3]

      photo_A      = dblarr(photo_test)
      photo_B      = dblarr(photo_test)
      conv_photo_A = dblarr(photo_test-2)
      conv_photo_B = dblarr(photo_test-2)
      best_radius_found_A = 0
      best_radius_found_B = 0
  
      FOR i=1, photo_test DO BEGIN
      
        midiaper, win2, a[4]+FOV_x1,a[5]+FOV_y1,magsA,errapA,skyA,skyerrA,conv,[i],[i+6,i+8],[-99999,99999],/FLUX, /SILENT
        midiaper, win1, b[4]+FOV_x1,b[5]+FOV_y1,magsB,errapB,skyB,skyerrB,conv,[i],[i+6,i+8],[-99999,99999],/FLUX, /SILENT
        photo_A(i-1)=magsA(0)
        photo_B(i-1)=magsB(0)
       
        IF ((i gt 1) and (i lt photo_test)) THEN BEGIN
          conv_photo_A(i-2) =(photo_A(i-1)-photo_A(i-2))/photo_A(i-1)
          conv_photo_B(i-2) =(photo_B(i-1)-photo_B(i-2))/photo_B(i-1)

          if ((conv_photo_A(i-2) le threshold) and (best_radius_found_A eq 0)) then begin
            best_radius_A=i
            best_radius_found_A = 1
          endif

          if ((conv_photo_B(i-2) le threshold) and (best_radius_found_B eq 0)) then begin
            best_radius_B=i
            best_radius_found_B = 1
          endif
      
        ENDIF
      
      ENDFOR

      if best_radius_found_A eq 0 then begin
        print, '<!> NO PHOTOMETRIC RADIUS FOUND (Beam A)'
        best_radius_A = 0
	m2[l] = 0
        rA = 0
      endif

      if best_radius_found_B eq 0 then begin
        print, '<!> NO PHOTOMETRIC RADIUS FOUND (Beam B)'
	best_radius_B = 0
	m1[l] = 0
	rB = 0
      endif
        
      if best_radius_found_A ne 0 then begin
        rA = best_radius_A 
        midiaper, win2, a[4]+FOV_x1,a[5]+FOV_y1,magsA,errapA,skyA,skyerrA,conv,[rA,rA+1],[rA+6,rA+8],[-99999,99999], maskA, /FLUX, /SILENT
        m2[l]=magsA[0]
        if (x2[l] gt 25 and x2[l] lt 35 and $
            y2[l] gt 25 and y2[l] lt 35 and $
            fx2[l] lt fwhm and fy2[l] lt fwhm) then begin
            temp2 = temp2 + m2[l]
            ntar2 = ntar2 + 1
        endif
        wset,1
        PLOTS,CIRCLE(a[4]*factor,a[5]*factor,rA*factor),/DEVICE
        PLOTS,CIRCLE(a[4]*factor,a[5]*factor,(rA+6)*factor),/DEVICE
        PLOTS,CIRCLE(a[4]*factor,a[5]*factor,(rA+8)*factor),/DEVICE
      endif

      if best_radius_found_B ne 0 then begin
        rB = best_radius_B
        midiaper, win1, b[4]+FOV_x1,b[5]+FOV_y1,magsB,errapB,skyB,skyerrB,conv,[rB,rB+1],[rB+6,rB+8],[-99999,99999], maskB, /FLUX, /SILENT
        m1[l]=magsB[0]
        if (x1[l] gt 25 and x1[l] lt 35 and $
            y1[l] gt 25 and y1[l] lt 35 and $
            fx1[l] lt fwhm and fy1[l] lt fwhm) then begin
            temp1 = temp1 + m1[l]
            ntar1 = ntar1 + 1
        endif
	wset,1
        PLOTS,CIRCLE((b[4]+n[1])*factor,b[5]*factor,rB*factor),/DEVICE
        PLOTS,CIRCLE((b[4]+n[1])*factor,b[5]*factor,(rB+6)*factor),/DEVICE
        PLOTS,CIRCLE((b[4]+n[1])*factor,b[5]*factor,(rB+8)*factor),/DEVICE   
      endif
      
      print,''
      print, 'Apertures   ','Beam A: ',best_radius_A,' Beam B: ',best_radius_B
      print, 'Fluxes   ','Beam A: ',m2[l],' Beam B: ',m1[l]
      print,''
      
      ;draw the acquisition images and the apertures

    endif
  
  endfor

  if not keyword_set(silent) then begin
    
    window, 1, xsize=1200, ysize=800

    psfile:
    
    !p.multi=[0, 0, 5, 0, 0]
    !p.charsize=1.5
    
    !X.TITLE='Frame no.'
    !Y.TITLE='x-positions [pix]'
    !Y.RANGE=[20,40]
    !P.LINESTYLE=0
    plot,x2
    !P.LINESTYLE=1
    oplot,x1
    !P.LINESTYLE=2
    ;plots, 0, 25
    ;plots, NDIT, 25, /CONTINUE
    ;plots, 0, 35
    ;plots, NDIT, 35, /CONTINUE

    !Y.TITLE='x-FWHM [pix]'
    !Y.RANGE=[1,5]
    !P.LINESTYLE=0
    plot,fx2*convfwhm
    !P.LINESTYLE=1
    oplot,fx1*convfwhm
    !P.LINESTYLE=2
    ;plots, 0, fwhm*convfwhm
    ;plots, NDIT, fwhm*convfwhm, /CONTINUE

    !Y.TITLE='y-positions [pix]'
    !Y.RANGE=[20,40]
    !P.LINESTYLE=0
    plot,y2
    !P.LINESTYLE=1
    oplot,y1
    !P.LINESTYLE=2
    ;plots, 0, 25
    ;plots, NDIT, 25, /CONTINUE
    ;plots, 0, 35
    ;plots, NDIT, 35, /CONTINUE

    !Y.TITLE='y-FWHM [pix]'
    !Y.RANGE=[1,5]
    !P.LINESTYLE=0
    plot,fy2*convfwhm
    !P.LINESTYLE=1
    oplot,fy1*convfwhm
    !P.LINESTYLE=2
    ;plots, 0, fwhm*convfwhm
    ;plots, NDIT, fwhm*convfwhm, /CONTINUE

    mean2 = temp2 / ntar2
    mean1 = temp1 / ntar1
    
    if mean2 gt mean1 then begin
      max = mean2
    endif else begin
      max = mean1
    endelse
        
    !Y.RANGE=[0,max+max/10]    

    !Y.TITLE='Flux (Counts)'
    !P.LINESTYLE=0
    plot,m2
    !P.LINESTYLE=1
    oplot, m1
    !P.LINESTYLE=0
    ;plots, 0, mean2
    ;plots, NDIT, mean2, /CONTINUE
    ;plots, 0, mean1
    ;plots, NDIT, mean1, /CONTINUE
    
    !Y.RANGE=0
    !p.charsize=1.0
    !p.multi=[0, 0, 1, 0, 0]
     
   endif    


   ; replot into file

   if file ne 1 then begin
     SET_PLOT,'PS'
     DEVICE,FILE='PHOTOMETRY/SFP_'+short_file_name+'.ps',/LANDSCAPE
     file = 1
     goto,psfile
   endif
   
   DEVICE,/CLOSE
   SET_PLOT,'X'
   

   print,''
   print,'-------------  MEAN VALUES OF THE FLUXES ---------------'
   print,''
   print,notar,' single target frames have been measured:'
   print,ntar2,' single target frames have been used for calculating the mean (Beam A)'
   print,ntar1,' single target frames have been used for calculating the mean (Beam B)'
   print,threshold,' (threshold)'
   print,fwhm*convfwhm,' (limiting FWHM)'
   print,mean2,' counts'
   print,mean1,' counts'
   print,DIT,' seconds'
   print,''

endif



;---------------- PHOTOMETRY ROUTINE --------------

print,''
print,'-------------  FIRST FIND THE STAR ---------------'
print,''
fit2D_A= gauss2dfit(win2(FOV_x1:FOV_x2,FOV_y1:FOV_y2),a)
print, 'Star gauss. A:', a[4]+FOV_x1, a[5]+FOV_y1
print, 'FWHM x, y',  a[2]*convfwhm, a[3]*convfwhm
print, 'Max image', max(win2, status), ' Pos X = ', status mod x_size, ' Pos y = ', trunc(status/x_size)
xm1 = status mod x_size
ym1 = trunc(status/x_size)
print,''
fit2D_B= gauss2dfit(win1(FOV_x1:FOV_x2,FOV_y1:FOV_y2),b)
print, 'Star gauss. B:', b[4]+FOV_x1, b[5]+FOV_y1
print, 'FWHM x, y',  b[2]*convfwhm, b[3]*convfwhm
print, 'Max image', max(win1, status), ' Pos X = ', status mod x_size, ' Pos y = ', trunc(status/x_size)
xm2 = status mod x_size
ym2 = trunc(status/x_size)
print,''

;draw the acquisition images, mark the peak of the fitted gaussians & maximae 

if not keyword_set(silent) then begin ; CAH
window,3,xsize=2*n[1]*factor,ysize=n[2]*factor
;win1(where(win1 lt 1))=1
;win2(where(win2 lt 1))=1

tvscl, rebin(sqrt(win1),n[1]*factor,n[2]*factor),1
tvscl, rebin(sqrt(win2),n[1]*factor,n[2]*factor),0

XYOUTS,5*factor,60*factor,'Beam A',/DEVICE
XYOUTS,(5+n[1])*factor,60*factor,'Beam B',/DEVICE
endif

midifieldangle,fname,para,phin,phie,rot
print,''
print,'-------------  ORIENTATION OF THE FIELD ---------------'
print,''
print,''
print,'North counted clockwise with respect to >>down<< (-V), East = North - 90 deg:'
print,''
print,'Phi North: ',phin
print,'Phi East: ',phie
phin = 180-phin
phie = 180-phie
print,''

sn=sin(2*!PI*phin/360)
se=sin(2*!PI*phie/360)
cn=cos(2*!PI*phin/360)
ce=cos(2*!PI*phie/360)

if not keyword_set(silent) then begin ; CAH
PLOTS,[62*factor,(62-5*sn)*factor],[15*factor,(15+5*cn)*factor],/DEVICE
XYOUTS,(62-10*sn)*factor,(15+10*cn)*factor,'N',/DEVICE
PLOTS,[62*factor,(62-5*se)*factor],[15*factor,(15+5*ce)*factor],/DEVICE
XYOUTS,(62-10*se)*factor,(15+10*ce)*factor,'E',/DEVICE

PLOTS,[xm1*factor-3,xm1*factor+3],[ym1*factor+3,ym1*factor-3],/DEVICE,Color=1
PLOTS,[xm1*factor+3,xm1*factor-3],[ym1*factor+3,ym1*factor-3],/DEVICE,Color=1
PLOTS,[(xm2+n[1])*factor-3,(xm2+n[1])*factor+3],[ym2*factor+3,ym2*factor-3],/DEVICE,Color=1
PLOTS,[(xm2+n[1])*factor+3,(xm2+n[1])*factor-3],[ym2*factor+3,ym2*factor-3],/DEVICE,Color=1

plot:
PLOTS,[a[4]*factor-5,a[4]*factor+5],[a[5]*factor,a[5]*factor],/DEVICE,Color=1
PLOTS,[a[4]*factor,a[4]*factor],[a[5]*factor-5,a[5]*factor+5],/DEVICE,Color=1
PLOTS,[(b[4]+n[1])*factor-5,(b[4]+n[1])*factor+5],[b[5]*factor,b[5]*factor],/DEVICE,Color=1
PLOTS,[(b[4]+n[1])*factor,(b[4]+n[1])*factor],[b[5]*factor-5,b[5]*factor+5],/DEVICE,Color=1
endif

if not keyword_set(silent) then begin
    !p.multi=[0, 0, 4, 0, 0]
    window, 4, xsize=600, ysize=600
    plot, win2(FOV_x1:FOV_x2,trunc(a[5])+FOV_y1), xtitle='x', title="Beam A, fit of x-position"
    oplot, fit2D_A(*,trunc(a[5])), linestyle=1
    plot, win2(trunc(a[4])+FOV_x1,FOV_y1:FOV_y2), xtitle='y', title="Beam A, fit of y-position"
    oplot, fit2D_A(trunc(a[4]),*), linestyle=1
    plot, win1(FOV_x1:FOV_x2,trunc(b[5])+FOV_y1), xtitle='x', title="Beam B, fit of x-position"
    oplot, fit2D_B(*,trunc(b[5])), linestyle=1
    plot, win1(trunc(b[4])+FOV_x1,FOV_y1:FOV_y2), xtitle='y', title="Beam B, fit of y-position"
    oplot, fit2D_B(trunc(b[4]),*), linestyle=1
    !p.multi=[0, 0, 1, 0, 0]

    testOk1= "string"
    read, '<?> Are the gaussian fits (+) okay? (y/n) ? ', testOK1
    if (strupcase(testOK1) eq "N") then begin
        testOk2= "string"
        read, '<?> Enter coordinates with mouse? (n/y) ? ', testOK2
        if (strupcase(testOK2) eq "Y") then begin
        wset, 3

        a[4] = 100
        a[5] = 100
        b[4] = 100
        b[5] = 100
        
        while (a[4] lt 0 or a[4] gt 62 or a[5] lt 0 or a[5] gt 69) do begin
        print,'<!>Click in the left image to select your source'
        cursor, ax, ay, /DEVICE,/DOWN
        a[4] = ax / factor
        a[5] = ay / factor
        endwhile
        
        while (b[4] lt 0 or b[4] gt 62 or b[5] lt 0 or b[5] gt 69) do begin
        print,'<!>Click in the right image to select your source'
        cursor, bx, by, /DEVICE,/DOWN
        b[4] = bx / factor - n[1]
        b[5] = by / factor
        endwhile

        print,a[4],a[5],b[4],b[5]

        goto, plot
        endif else begin
        print, '<!> OK, try another file'
        goto, ende
        endelse
    endif
    
endif

photo_A      = dblarr(photo_test)
photo_B      = dblarr(photo_test)
conv_photo_A = dblarr(photo_test-2)
conv_photo_B = dblarr(photo_test-2)
best_radius_found_A = 0
best_radius_found_B = 0
  
FOR i=1, photo_test DO BEGIN
    midiaper, win2, a[4]+FOV_x1,a[5]+FOV_y1,magsA,errapA,skyA,skyerrA,conv,[i],[i+6,i+8],[-99999,99999],/FLUX, /SILENT
    midiaper, win1, b[4]+FOV_x1,b[5]+FOV_y1,magsB,errapB,skyB,skyerrB,conv,[i],[i+6,i+8],[-99999,99999],/FLUX, /SILENT
    photo_A(i-1)=magsA(0)
    photo_B(i-1)=magsB(0)
    if not keyword_set(silent) then print,"aperture radius",i,magsA[0],magsB[0]

    IF ((i gt 1) and (i lt photo_test)) THEN BEGIN
        conv_photo_A(i-2) =(photo_A(i-1)-photo_A(i-2))/photo_A(i-1)
        conv_photo_B(i-2) =(photo_B(i-1)-photo_B(i-2))/photo_B(i-1)

        if ((conv_photo_A(i-2) le threshold) and (best_radius_found_A eq 0)) then begin
            best_radius_A=i
            best_radius_found_A = 1
        endif

        if ((conv_photo_B(i-2) le threshold) and (best_radius_found_B eq 0)) then begin
            best_radius_B=i
            best_radius_found_B = 1
        endif
    ENDIF
ENDFOR

;open Text-File
OPENW,lun,'PHOTOMETRY/Phot_'+short_file_name+'.txt',/get_lun

printf,lun,OBJECT
printf,lun,short_file_name
if N_Params() gt 1  then begin
    first_char    = STRPOS(skyname,'/MIDI.20')+6
    last_char	  = STRPOS(skyname,'.fits')
    printf,lun, STRMID(skyname,first_char,last_char-first_char), $
      '    name of sky'
endif
printf,lun, DIT,'  Time exposure in s'

if best_radius_found_A eq 0 then begin
  print, '<!> NO PHOTOMETRIC RADIUS FOUND (Beam A)'
endif

if best_radius_found_B eq 0 then begin
  print, '<!> NO PHOTOMETRIC RADIUS FOUND (Beam B)'
endif
  
if best_radius_found_A ne 0 then begin
  rA = best_radius_A 
  midiaper, win2, a[4]+FOV_x1,a[5]+FOV_y1, magsA,errapA,skyA,skyerrA,conv,[rA,rA+1],[rA+6,rA+8],[-99999,99999], maskA, /FLUX, /SILENT
  print,''
  print,'------------- Aperture Photometry (Beam A)---------------'
  print,''
  print, '           	Flux                 error     		Sky	    error'
  print,''
  print,'rA=', rA
  print, 'MIDI Beam A ', magsA[0]*conv/DIT,errapA[0]*conv/DIT,skyA[0]*conv/DIT,skyerrA[0]*conv/DIT
  print,'rA= ', rA+1
  print, 'MIDI Beam A ', magsA[1]*conv/DIT,errapA[1]*conv/DIT,skyA[0]*conv/DIT,skyerrA[0]*conv/DIT
  print,''

  print,''
  printf,lun, a[4]+FOV_x1,a[5]+FOV_y1, '     Star position in Beam A'
  printf,lun,rA,'    Aperture for Beam A in pixels'
  printf,lun,magsA[0]  *conv/DIT,'   Flux A'
  printf,lun,errapA[0] *conv/DIT,'   error Flux A'
  printf,lun,skyA[0]   *conv/DIT,'   sky A'
  printf,lun,skyerrA[0]*conv/DIT,'   error sky A'
  print,''
  printf,lun,rA+1,'  Aperture for Beam A in pixels'
  printf,lun,magsA[1] *conv/DIT,'    Flux A'
  printf,lun,errapA[1]*conv/DIT,'    error Flux A'

  ;draw the acquisition image and the apertures
if not keyword_set(silent) then begin ; CAH
  wset,3
  PLOTS,CIRCLE(a[4]*factor,a[5]*factor,rA*factor),/DEVICE
  PLOTS,CIRCLE(a[4]*factor,a[5]*factor,(rA+6)*factor),/DEVICE
  PLOTS,CIRCLE(a[4]*factor,a[5]*factor,(rA+8)*factor),/DEVICE
endif
endif

if best_radius_found_B ne 0 then begin
  rB = best_radius_B
  midiaper, win1, b[4]+FOV_x1,b[5]+FOV_y1, magsB,errapB,skyB,skyerrB,conv,[rB,rB+1],[rB+6,rB+8],[-99999,99999], maskB, /FLUX, /SILENT
  print,''
  print,'------------- Aperture Photometry (Beam B)---------------'
  print,''
  print, '           	Flux                 error     		Sky	    error'
  print,''
  print,'rB=', rB
  print, 'MIDI Beam B ', magsB[0]*conv/DIT,errapB[0]*conv/DIT,skyB[0]*conv/DIT,skyerrB[0]*conv/DIT
  print,'rB= ', rB+1
  print, 'MIDI Beam B ', magsB[1]*conv/DIT,errapB[1]*conv/DIT,skyB[0]*conv/DIT,skyerrB[0]*conv/DIT
  print,''

  print,''
  printf,lun, b[4]+FOV_x1,b[5]+FOV_y1, '     Star position in Beam B'
  printf,lun,rB,'    Aperture for Beam B in pixels'
  printf,lun,magsB[0]  *conv/DIT,'   Flux B'
  printf,lun,errapB[0] *conv/DIT,'   error Flux B'
  printf,lun,skyB[0]   *conv/DIT,'   sky B'
  printf,lun,skyerrB[0]*conv/DIT,'   error sky B'
  print,''
  printf,lun,rB+1,'  Aperture for Beam B in pixels'
  printf,lun,magsB[1] *conv/DIT,'    Flux B'
  printf,lun,errapB[1]*conv/DIT,'    error Flux B'

  ;draw the acquisition image and the apertures
if not keyword_set(silent) then begin ; CAH
  wset,3
  PLOTS,CIRCLE((b[4]+n[1])*factor,b[5]*factor,rB*factor),/DEVICE
  PLOTS,CIRCLE((b[4]+n[1])*factor,b[5]*factor,(rB+6)*factor),/DEVICE
  PLOTS,CIRCLE((b[4]+n[1])*factor,b[5]*factor,(rB+8)*factor),/DEVICE
endif
endif

if not keyword_set(silent) then begin
  wset,4
  absc = dindgen(photo_test-2)+2
  !p.multi=[0, 0, 2, 0, 0]
  if best_radius_found_A ne 0 then begin
    plot, absc, conv_photo_A, xtitle='pixels', yrange=[0,0.8], ytitle='convergence', title='Beam A'
    oplot,[rA,rA],[0, threshold]
  endif
  if best_radius_found_B ne 0 then begin
    plot, absc, conv_photo_B, xtitle='pixels', yrange=[0,0.8], ytitle='convergence', title='Beam B'
    oplot,[rB,rB],[0, threshold]
  endif
endif

;close Text-File
CLOSE,lun & free_lun,lun

if best_radius_found_A ne 0 and best_radius_found_B ne 0 then begin 
  ;help,maskA,maskB
  oirNewData, hdrname, 'PHOTOMETRY/Mask_'+short_file_name+'.fits', $
    { data1: maskB[*,*,0], data2: maskA[*,*,0] }
  ; the data for the mask is a struct with entries data1 and data2
  ;channel B is from win1, which is from data1
endif

ende:

end
