;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
; $Id: xmidispvisi.pro,v 1.43 2006/09/18 12:27:26 koehler Exp $
;
; xmidivisi.pro
; Created:     Thu Jun 12 01:13:57 2003 by Koehler@rommie
; Last change: Mon Sep 18 14:21:03 2006
;
; PURPOSE:
;	save the world (or a small part of it)
;
; CALLING SEQUENCE:
;	x->obj_new('xmidivisi', fringefile[,photfileA, photfileB])
;       without photfiles SCI PHOT channels are used for photometry
;	*** (there are more options, use the source!) ***
;
; SHORTCUT:
;	x = xmdv(files)
;	same options as in the oo case
;
; INPUT:
;	fringefile:	MIDI file with fringe search or track data
;	photfileA/B:	MIDI files with chopping data, shutter A/B
;			open, respectively.;
;	/RUNNING_HIPASS apply hi-pass filter by subtracting
;			boxcar-smoothed fringes.  Default is to
;			subtract mean of each scan from itself.
;
; OUTPUT:
;	an object that can do wonderful things
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
PRO xmidispvisi__Define
dings = { xmidispvisi, $
          ;; parameters set by the caller
          fringefile: '', photfiles: ['',''], maskfile: '', $
          dispersedbins: 0, ewsphot: 1, dSky: 0, tartypskip: 2, $
          ;; parameters from data files
          objectname: '', $
          PhFilt: '', FrFilt: '', Grism: '', Beam: '', DIT: 0., $
          scanLength: 0, totopd: 0., $
          longwl: 0., shortwl: 0., nLambda: 320, lambda: fltarr(320), $
          maskobj: obj_new(), $
          trace: replicate({midisptrace},2,4),$	 ;;; 1.index=file, 2.index=window
          traceFr: replicate({midisptrace},4),$	 ;;; 4 windows for SCI_PHOT
          specA1: fltarr(320), specA2: fltarr(320), specAph: fltarr(320),$
          specB1: fltarr(320), specB2: fltarr(320), specBph: fltarr(320),$
          specAB: fltarr(320), $
          opd: ptr_new(), realopd: ptr_new(), tartyp: ptr_new(),$
          flux1: ptr_new(), flux2: ptr_new(), fluxA: ptr_new(), fluxB: ptr_new(), $
          fringe: ptr_new(), foufri: ptr_new(), amplfou: ptr_new(),$
          ;; parameters controlled via the GUI
          goodthresh: -1., oldgthres: -1., goodfringes: ptr_new(), Ngood: -1, $
          badthresh:  -1., oldbthres: -1.,  badfringes: ptr_new(), Nbad: -1, $
          noiseopd: 1e-3, noisefreq: [0,0,0,0], $
          la_start: 0, la_end: 128, la_step: 4, la_width: 4, $
          ;; parameters to control plotting
          frame_rng: [0,12000], opd_rng: [0.,0.], $
          flux_rng: [0.,0.], fdiff_rng: [0.,0.], ampl_rng: [0.,0.], $
          menu: -1, $
          topdraw: -1, topdrawID: -1, $
          lftdraw: -1, lftdrawID: -1, $
          rgtdraw: -1, rgtdrawID: -1, $
          goodthl: -1, badthl: -1, $
          sigamp: -1, noiamp: -1, corramp: -1, corrflx: -1, visibi: -1, $
          help: -1, quit: -1, $
          color: lonarr(6), charsize: 1., multi: intarr(5), button: 0 $
}
          ;frimov: -1, foumov: -1, print:  -1, wrtlog: -1,
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function xmidispvisi::get_objectname
  return, self.objectname
end

Function xmidispvisi::get_filename, PHOT=phot, MASK=mask

  if keyword_set(phot) then return, self.photfiles
  if keyword_set(mask) then return, self.maskfile
  return, self.fringefile
end

Function xmidispvisi::baseline, base_pa

  filelist = oirFileList(self.fringefile)
  im = obj_new('imagedata', filelist[0])
  he = im->prihead()
  stat1  = StrCompress(he->getpar('HIERARCH ESO ISS CONF STATION1'),/remove_all)
  stat2  = StrCompress(he->getpar('HIERARCH ESO ISS CONF STATION2'),/remove_all)
  date   = StrCompress(he->getpar('DATE-OBS'),/remove_all)
  lst    = StrCompress(he->getpar('LST'),/remove_all)
  RA     = StrCompress(he->getpar('RA') ,/remove_all)
  Dec    = StrCompress(he->getpar('DEC'),/remove_all)
  obj_destroy,im

  if (stat1 gt stat2) then begin
      tmp = stat1 & stat1 = stat2 & stat2 = tmp
  endif

  print, "object:              ", self.objectname
  print, "Baseline:            ", stat1, " - ", stat2
  print, "Date of observation: ", date
  print, "Local Sid Time:      ", LST
  print, "Right Ascension:     ", RA
  print, "Declination:         ", Dec

  project_baseline, (LST/3600. - RA/15.), Dec, stat1+"-"+stat2, base_len, base_pa

  print, "Projected Baseline"
  print, Format='("             length: ", F6.2)', base_len
  print, Format='("     position angle: ", F6.2)', base_pa

  return, base_len
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function xmidispvisi::tartyp
  return, *self.tartyp
end

Function xmidispvisi::fringes
  sz = size(*self.opd)
  result = fltarr(2,sz[1],sz[2])
  result[0,*,*] = *self.opd
  result[1,*,*] = total(*self.fringe,1)
  return, result
end

Function xmidispvisi::dispfringes
  return, *self.fringe
end

Function xmidispvisi::whitefringeampl
  return, *self.amplfou
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi::save_visibility, filename
  if N_elements(filename) lt 1 then begin
    filename = Dialog_Pickfile(Title="Select filename for visibility",$
                               Default_EXtension="fits", Filter="*.fits", $
                               /Overwrite_Prompt, /Write)
    if filename eq "" then return
    print,"Writing visibility to ",filename
  end

  visi = self->visibility()
  midiPutVis, filename, reform(visi[0,*]), reform(visi[2,*])
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function xmidispvisi::max_fringe_packet_width
  nlabins= fix( (self.la_end-self.la_start)/self.la_step + 1)
  lolams = self.la_start + self.la_step * indgen(nlabins)
  hilams = lolams + self.la_width - 1

  mean_lam = (self.lambda[lolams] + self.lambda[hilams]) / 2.
  delta_lam=  self.lambda[lolams] - self.lambda[hilams]

  return, max(mean_lam * mean_lam / delta_lam) * 1e-6
  ; result is in meters
end

Function xmidispvisi::get_thresholds
  return, [ self.goodthresh, self.badthresh, self.noiseopd ]
end

Pro xmidispvisi::set_thresholds, good, bad, noiseopd
  if N_params() gt 0 then self.goodthresh = good
  if N_params() gt 1 then self.badthresh  = bad
  if N_params() gt 2 then self.noiseopd = noiseopd

  if (bad gt 0.) then begin
      print,"  +----------------------------------------------------------------------+"
      print,"  | Expert advice:                                                       |"
      print,"  | Setting the noise threshold to anything except 0 is not a good idea! |"
      print,"  +----------------------------------------------------------------------+"
  endif
  max_packet_width = self->max_fringe_packet_width()
  if noiseopd lt max_packet_width  then begin
      print,"  +----------------------------------------------------------------------+"
      print,"  | Expert advice:                                                       |"
      print,"  | You still have signal in your noise scans!                           |"
      print,"  | With your current lambda binning,                                    |"
      print,"  | noiseopd should be at least",max_packet_width," meters"
      print,"  +----------------------------------------------------------------------+"
  endif

  if Widget_info( long(self.topdraw),/Valid_id) then begin
      self->update_gui
  endif else self->update_GoodBad
end

Function xmidispvisi::get_percent_goodbad
  nScans = N_elements(*self.amplfou)
  return, [ 100.*self.Ngood/nScans, 100.*self.Nbad/nScans ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function xmidispvisi::get_trace, iFile, WIN=win

  if (iFile ge 0) then begin
      ;print,"Trace of photometric file",iFile
      trace = self.maskobj->get_tracePh()
      trace = reform(trace[iFile,*])
  endif else begin
      ;print,"Trace of Fringe"
      trace = self.maskobj->get_traceFr()
  endelse
  if keyword_set(win) then return,trace[win-1]
  return, trace
end


Pro xmidispvisi::show_traces, fname
  self.maskobj->show_traces, fname
end

Pro xmidispvisi::maskgui
  print,"NOTE: Changes to the mask parameters will have *NO* effect."
  print,"      Rerun xmdv() if you want to change the mask."
  self.maskobj->gui
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
PRO xmidispvisi::do_photometry, photfile, maskdata, PhFilt, spec1,spec2,specPh,$
        	PHOTFILEB=photfileB, SCI_MODE=sci_mode, $
		BEFORE=before, AFTER=after, ERRORLOG=errorlog

  if not keyword_set(before)  then before  = 1
  if not keyword_set(after)   then after   = 1
  if not keyword_set(sci_mode) then sci_mode= 0

  im = obj_new('fitsfile',(oirFileList(photfile))[0])
  he = im->prihead()
  shutter= strcompress(he->getpar('INS SHUT NAME'),/remove_all)
  PhFilt = strcompress(he->getpar('INS FILT NAME'),/remove_all)
  Grism  = strcompress(he->getpar('INS GRIS NAME'),/remove_all)
  DIT    = strcompress(he->getpar('DET DIT'),/remove_all)
  obj_destroy,im

  if Grism eq 'OPEN' then $
    print,"This is undispersed data.  I'm not sure I can handle that..."

  if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"

  print, 'Reading Chopping data...'
  print, photfile
  chop_image= MIDISkySubImage(photfile, before=before, after=after, ERRORLOG=errorlog)

  if sci_mode then begin   ;;; by FP
          win1 = chop_image.data1
          win2 = chop_image.data4
  endif else begin
      win1 = chop_image.data1
      win2 = chop_image.data2
      nWin = 2
      if (where(Tag_Names(chop_image) eq "DATA4") ne -1) then begin
          nWin = 4
          win3 = chop_image.data3
          win4 = chop_image.data4
      endif
  endelse

  if keyword_set(photfileB) then begin
      print, photfileB
      chop_image= MIDISkySubImage(photfileB, before=before, after=after, ERRORLOG=errorlog)
      win1 = sqrt(win1 * chop_image.data1 > 0.)
      win2 = sqrt(win2 * chop_image.data2 > 0.)
      if nWin ge 4 then begin
          ;; NOTE: win1 and win4 are not useful in this case!
          win3 = sqrt(win3 * chop_image.data3 > 0.)
          win4 = sqrt(win4 * chop_image.data4 > 0.)
      endif
  endif

  if sci_mode then begin   ;;; by FP
      spec1 = total(win1 * maskdata.data1, 2) / DIT
      spec2 = total(win2 * maskdata.data4, 2) / DIT
  endif else begin
      if nWin eq 2 then begin
          spec1 = total(win1 * maskdata.data1, 2) / DIT
          spec2 = total(win2 * maskdata.data2, 2) / DIT
      endif else begin
          spec1 = total(win2 * maskdata.data2, 2) / DIT
          spec2 = total(win3 * maskdata.data3, 2) / DIT
          if (shutter eq 'AOPEN') then begin
              specPh= total(win4 * maskdata.data4, 2) / DIT
          endif else begin
              specPh= total(win1 * maskdata.data1, 2) / DIT
          endelse
      endelse
  endelse
  print,"total flux 1:",total(spec1),",  total flux 2:",total(spec2)
  if nWin eq 4 then print,"sciphot flux:",total(specPh)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi::ews_photometry, ERRORLOG=errorlog

  print, ""
  print, "========== Doing photometry (EWS-style) ==========="

  if self.dSky then begin
      chopimgfile0= oirChopPhotoImages(self.photfiles[0],DSKY=self.dsky)
      chopimgfile1= oirChopPhotoImages(self.photfiles[1],DSKY=self.dsky)
  endif else begin
      chopimgfile0= oirChopPhotoImages(self.photfiles[0])
      chopimgfile1= oirChopPhotoImages(self.photfiles[1])
  endelse

  middle_name = MIDIname_datime(self.photfiles[0]) + "+" + $
		MIDIname_time(self.photfiles[1]) + "_dsky" + strtrim(self.dSky,2)
  chopspecfile= "PHOTOMETRY/ChSp_" + middle_name + ".fits"

  cmd = getenv("drsRoot") + '/c/bin/oirMakePhotoSpectra'
  if file_test(cmd,/EXECUTABLE) then begin
      if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"
      cmd = cmd + ' -A ' + chopimgfile0 + ' -B ' + chopimgfile1 + $
        ' -mask ' + self.maskfile + ' -out ' + chopspecfile
      print,cmd
      spawn,cmd,EXIT_STATUS=status
      if (status ne 0) then begin
          print,'oirMakePhotoSpectra failed'
          Log_error, chopimgfile0, 'oirMakePhotoSpectra failed'
          return
      endif
  endif else begin
      print,'The program'
      print,cmd
      print,'does not exist or is not executable.'
      print,"Please make sure MIA+EWS was installed properly."
      return
  endelse
  chop_spectra = oirgetdata(chopspecfile)

  im = obj_new('fitsfile',chopspecfile)
  he = im->prihead()
  ;PhFilt= strcompress(he->getpar('INS FILT NAME'),/remove_all)
  Grism = strcompress(he->getpar('INS GRIS NAME'),/remove_all)
  ;DIT   = strcompress(he->getpar('DET DIT'),/remove_all)
  beam  = he->getpar('INS OPT1 ID')
  obj_destroy,im

  if Grism eq 'OPEN' then $
    print,"This is undispersed data.  I'm not sure I can handle that..."

  if (beam ne 'HIGH_SENS') then print,"Warning: data sens not high enough"

  self.specA1 = chop_spectra[3].data1 & self.specA2 = replicate(0.,320)
  self.specB1 = chop_spectra[4].data1 & self.specB2 = replicate(0.,320)
  self.specAB = chop_spectra[5].data1
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Result:
;; [0,*] = lambda at center of wavelength bins
;; [1,*] = width of wavelength bins
;; [2,*] = correlated flux

Function xmidispvisi::raw_correlflux, lolams,hilams, SSN=ssn,PLOT=plot,VERBOSE=verbose

  ; make sure we round up, the last bin might extend beyond la_end
  nlabins= fix( (self.la_end-self.la_start)/self.la_step + 1)
  corrflx= fltarr(3,nlabins)

  if Keyword_set(plot) then begin
      if !D.name eq "X" then window,2,ysize=990
      !P.multi= [0,2,nlabins]
  endif

  print,self.Ngood," good and",self.nbad," bad fringes"

  lolams = self.la_start + self.la_step * indgen(nlabins)
  hilams = lolams + self.la_width - 1

  if Keyword_set(ssn) then print,"Estimating noise from same scans as signal"

  for l=0,nlabins-1 do begin
      lolam = lolams[l] 	; lowest lambda-pix
      hilam = hilams[l] 	; highest lambda-pix

      if lolam gt hilam then begin
          tmp=lolam & lolam=hilam & hilam=tmp
      endif

      ;; we have to extrapolate half a pixel at the edges of the valid
      ;; lambda-range (0...nLmbda-1)

      if (lolam gt 0) then $
	lowave= (self.lambda[lolam-1] + self.lambda[lolam]) / 2. $
      else lowave= self.lambda[0] - (self.lambda[1]-self.lambda[0])/2.

      if (hilam+1 lt self.nLambda) then $
	hiwave= (self.lambda[hilam] + self.lambda[hilam+1]) / 2. $
      else hiwave= self.lambda[hilam] + (self.lambda[hilam]-self.lambda[hilam-1])/2.

      corrflx[0,l] = (hiwave + lowave) / 2.
      corrflx[1,l] = abs(hiwave - lowave)

      mfreq = self.totopd / corrflx[0,l]
      lofreq= floor(mfreq)-1
      hifreq= lofreq + 3

      if Keyword_set(verbose) then $
        print, l, mfreq, hifreq+1-lofreq, $
	  Format='("lambda-bin",I3,", mean freq ",F6.3,", bins:",I3)'

      freqs = [lofreq,lofreq,hifreq,hifreq]

      ; select pixels within lambda-range for this bin
      ; and integrate over lambda (or what's left of it)'
      lamfr= total((*self.fringe)[ lolam:hilam,*,*],1)

      if keyword_set(ssn) then begin
          noisefr= indgen(N_elements(lamfr[*,0]))
          noisefr= where( noisefr gt 0  and  noisefr lt lofreq-1  or $
                          noisefr gt hifreq+1 )
          print,format='("noise freq: ", 40(I3))',noisefr
          print,"=> ",N_Elements(noisefr)," points"
      endif

      if (self.Ngood gt 0) then begin
          ;; select good fringes and FFT along scans
          goodfou = abs(FFTdim( lamfr[*,*self.goodfringes], Dimension=1))^2
          if Keyword_set(plot) then begin
              ymax = max(goodfou)
              ;; NOTE: we could plot histograms with PSYM=10, but that looks ugly
              plot, goodfou[*,0], yrange=[0,ymax], $
                Xticks=1,Xtickname=[" "," "],Yticks=1,Ytickname=[" "," "],Ymargin=[0,0]

              for f=1,self.Ngood-1 do oplot, goodfou[*,f]
              if (self.Ngood gt 1) then $
                oplot, total(goodfou,2) / self.Ngood, color=self.color[2]
              ymax = ymax*2
              oplot, freqs, [0,ymax,ymax,0], color=self.color[3]
              oplot, self.scanlength-freqs, [0,ymax,ymax,0], color=self.color[3]
          endif
          if Keyword_set(ssn) then begin
              goodamp= total(goodfou[ lofreq:hifreq,*])
              noisamp= total(goodfou[ noisefr,*]) * (hifreq+1.-lofreq)/N_elements(noisefr)
              print," good", goodamp,", noise",noisamp,", n2s-factor ",(hifreq+1.-lofreq)/N_elements(noisefr)
              goodamp= (goodamp - noisamp) / self.Ngood
          endif else begin
              goodamp= total(goodfou[ lofreq:hifreq,*])/self.Ngood
          endelse
      endif else begin
          if Keyword_set(plot) then $
            plot, [0,1], xrange=[0,self.scanlength], /nodata, $
                Xticks=1,Xtickname=[" "," "],Yticks=1,Ytickname=[" "," "],Ymargin=[0,0]
          goodamp= 0.
      endelse

      if (self.Nbad gt 0) then begin
                                ; select bad fringes and FFT along scans
          badfou = abs(FFTdim( lamfr[*,*self.badfringes], Dimension=1))^2
          if Keyword_set(plot) then begin
              if (self.Ngood lt 1 ) then ymax = max(badfou)
              plot, badfou[*,0], yrange=[0,ymax], $
                Xticks=1,Xtickname=[" "," "],Yticks=1,Ytickname=[" "," "],Ymargin=[0,0]

              for f=1,self.Nbad-1 do oplot, badfou[*,f]
              if (self.Nbad gt 1) then $
                oplot, total(badfou,2) / self.Nbad, color=self.color[2]
              oplot, freqs, [0,ymax,ymax,0], color=self.color[4]
              oplot, self.scanlength-freqs, [0,ymax,ymax,0], color=self.color[4]
          endif
          badamp= total( badfou[ lofreq:hifreq,*])/self.Nbad
      endif else begin
          if Keyword_set(plot) then $
            plot, [0,1], xrange=[0,self.scanlength], /nodata, $
                Xticks=1,Xtickname=[" "," "],Yticks=1,Ytickname=[" "," "],Ymargin=[0,0]
          ymax = 1.
          badamp= 0.
      endelse
      if keyword_set(plot) then $
        XYOuts, 0.,0.4*ymax, string(corrflx[0,l],format='(F4.1,"m")'),alignment=1.

      corrflx[2,l] = sqrt( (goodamp-badamp) > 0.) / self.DIT
      if Keyword_set(verbose) then $
        print, "lambda = ",lolam,"...",hilam," pix, corrflux =",corrflx[2,l]
  endfor
  if Keyword_set(plot) then !P.multi= 0
  return, corrflx
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function xmidispvisi::photometry, QUIET=quiet, PLOT=plot, $
                    ATOTAL=atot, A1=a1, A2=a2, APHOT=aphot,$
                    BTOTAL=btot, B1=b1, B2=b2, BPHOT=bphot, AB=ab, UNMASKED=unmasked

  fname_a= file_basename(self.photfiles[0])
  fname_b= file_basename(self.photfiles[1])
  fname  = fname_a + " + " + fname_b
  if keyword_set(unmasked) then begin
      ;; total photometry is in the ews-file only...

      middle_name = MIDIname_datime(self.photfiles[0]) + "+" + $
                    MIDIname_time(self.photfiles[1]) + "_dsky" + strtrim(self.dSky,2)
      chop_spectra= oirgetdata("PHOTOMETRY/ChSp_" + middle_name + ".fits")

      ;; return square-rooted photometry, unless instructed otherwise
      phot = chop_spectra[2].data1
      if keyword_set(atot) then begin & phot= chop_spectra[0].data1 & fname= fname_a & end
      if keyword_set(btot) then begin & phot= chop_spectra[1].data1 & fname= fname_b & end
      if keyword_set(aphot) or keyword_set(bphot) then begin
          print,"Total sci_phot photometry is not available"
          return,-1
      endif
      if keyword_set(a1) or keyword_set(a2) or keyword_set(b1) or keyword_set(b2) then $
        phot = 0

  endif else begin
      ;; return square-rooted photometry, unless instructed otherwise
      phot   = self.specAB
      if keyword_set(a1) then begin & phot= self.specA1 & fname= fname_a & end
      if keyword_set(a2) then begin & phot= self.specA2 & fname= fname_a & end
      if keyword_set(b1) then begin & phot= self.specB1 & fname= fname_b & end
      if keyword_set(b2) then begin & phot= self.specB2 & fname= fname_b & end
      if keyword_set(aphot) then begin return,self.specAph & fname= fname_a & end
      if keyword_set(bphot) then begin return,self.specBph & fname= fname_b & end
      if keyword_set(atot) then begin phot= self.specA1 + self.specA2 & fname= fname_a & end
      if keyword_set(btot) then begin phot= self.specB1 + self.specB2 & fname= fname_b & end
  endelse

  if n_elements(phot) eq 1 and phot[0] le 0. then begin	; CAH: added first test
      print,"It seems you were using EWS-photometry."
      print,"Please note that EWS does not provide seperate measurements for"
      print,"the two windows, but only the combined flux."
      return,-1
  endif

  nlabins= fix( (self.la_end-self.la_start)/self.la_step + 1)
  photom = fltarr(3,nlabins)

  for l=0,nlabins-1 do begin
      lolam = self.la_start + self.la_step*l
      hilam = lolam + self.la_width - 1
      photom[0,l]= (self.lambda[lolam] + self.lambda[hilam]) / 2.
      photom[1,l]= abs(self.lambda[lolam] - self.lambda[hilam])
      photom[2,l]= total( phot[ lolam:hilam ])/self.la_width
  endfor
  if not keyword_set(quiet) and (self.la_width gt 1) then $
    print,"This flux has been divided by the bin-size (",strtrim(self.la_width,2)," pixels)"

  if keyword_set(plot) then begin
      window,0
      !P.multi= 0
      plot, photom[0,*], photom[2,*], ystyle=2, xtitle="Wavelength [micron]",$
        title= self.objectname + ", " + fname, ytitle="Photometric Flux"
  endif
  return,photom
end

Function xmidispvisi::get_photometry, LAMBDA=lambda,PLOT=plot, $
                    ATOTAL=atot,A1=a1,A2=a2,BTOTAL=btot,B1=b1,B2=b2,AB=ab

  phot= self->photometry(ATOTAL=atot,A1=a1,A2=a2,BTOTAL=btot,B1=b1,B2=b2,AB=ab,PLOT=plot)
  lambda= phot[0,*]
  return, phot[2,*]
end

Function xmidispvisi::sci_photometry, A=a, B=b
  if keyword_set(b) then return, self.fluxB
  return, self.fluxA
end

Function xmidispvisi::correlflux, PLOT=plot, CFPLOT=cfplot, QUIET=quiet

  cf = self->raw_correlflux(PLOT=plot)
  if not keyword_set(quiet) then begin
      print,"This correlated flux is defined so that visibility = corrflux/photflux"
      print,"This is different from the definition in Version 1.0 by a factor of 2."
  endif
  if (self.la_width gt 1) then begin
    cf[2,*] = cf[2,*]/self.la_width
    if not keyword_set(quiet) then $
      print,"Furthermore, it has been divided by the bin-size (",$
	strtrim(self.la_width,2)," pixels)"
  endif
  if keyword_set(cfplot) then begin
      window,0
      !P.multi= 0
      fname = (oirfilelist(self.fringefile))[0]
      first_char= Strpos(fname,'MIDI.20')
      fname = Strmid(fname,first_char)
      plot, cf[0,*], cf[2,*], ystyle=2, xtitle="Wavelength [micron]",$
        title= self.objectname + ", " + fname, ytitle="Correlated Flux"
  endif
  return, cf
end

Function xmidispvisi::visibility, PLOT=plot, VISPLOT=visplot

  if keyword_set(plot) and self.nbad gt 0 then begin
      max_packet_width = self->max_fringe_packet_width()
      if self.noiseopd lt max_packet_width  then begin
          msg= dialog_message(["You still have signal in your noise scans!",$
                               "With your current lambda binning,",$
                               "noiseopd should be at least "+$
	                          strtrim(string(max_packet_width*1000),2)+" mm",$
                               "Shall I set noiseopd to this value?"],$
                              Title="Expert advice",/cancel)
          if msg eq "OK" then begin
              self.noiseopd = max_packet_width
              self->update_gui
          endif
      endif
  endif
  visibi = self->raw_correlflux(lolams,hilams,PLOT=plot)
  nlabins= (size(visibi))[2]

  v = fltarr(4,nlabins)
  v[0:2,*] = visibi
  visibi = v

  for l=0,nlabins-1 do begin
      lolam = lolams[l]
      hilam = hilams[l]
      phot = total( self.specAB[ lolam:hilam ])
      visibi[2,l] = visibi[2,l] / phot
      visibi[3,l] = phot
  endfor

  if Keyword_set(visplot) then begin
      if !D.name eq "X" then window,0
      !P.multi= 0
      fname = (oirfilelist(self.fringefile))[0]
      first_char= Strpos(fname,'MIDI.20')
      fname = Strmid(fname,first_char)
      plot, visibi[0,*],visibi[2,*], ystyle=2, xtitle="Wavelength [micron]",$
        title= self.objectname + ", " + fname, ytitle="Visibiltity"
  endif
  ;print,"REMEMBER THAT THIS IS AN EXPERIMENTAL VERSION."
  return, visibi
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; instrumental visibility,
;; i.e. measured visibility divided by expected visibility.
;; currently, only simple disk model is implemented.
;; the parameter is the diameter in mas
;; if diameter is not given, we look it up in the calibrator-database
;; The name under which to look it up can be given with the NAME parameter
;;
Function xmidispvisi::instruvisi, diameter, NAME=name, PLOT=plot, VISPLOT=visplot

  visarr = self->visibility(PLOT=plot)
  if N_elements(diameter) le 0 then begin
      print,"Looking up ",self.objectname," in calibrator database"
      if not keyword_set(name) then name= self.objectname
      diameter = (calibrator(name)).diam
      print,"Found: ",diameter," mas"
      if diameter < 0. then return,0
  endif
  visexp = calibrate_disk(self.fringefile, diameter, visarr[0,*])
  visarr[2,*] = visarr[2,*] / visexp
  if keyword_set(visplot) then plot_visibility, visarr
  return, visarr
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; get visibility calibrated by a calibrator observation
;; this uses the instruvisi-method of the calibrator,
;; so we need the diameter of the calibrator
;;
Function xmidispvisi::calibratedvisi, cals,diameter,NAME=name,PLOT=plot,VISPLOT=visplot

  for i=0,N_elements(cals)-1 do begin
    cal = cals[i]
    print,"Calibrator",i," (",cal->get_objectname(),")"

    ; tell the calibrator to use our lambda-binning
    cal->set_lambda_bins, self.la_start,self.la_end,self.la_step,self.la_width

    ; get the instrumental visibility
    insvis = cal->instruvisi(diameter,NAME=name,PLOT=plot)

    if (i eq 0) then suminsvis = insvis[2,*] $
    else suminsvis = suminsvis + insvis[2,*]
  endfor
  insvis = suminsvis / N_elements(cals)

  ; get our own visibility
  visarr = self->visibility(PLOT=plot)

  visarr[2,*] = visarr[2,*] / insvis
  if keyword_set(visplot) then $
    plot_visibility, visarr, ytitle="Calibrated visibility", title=self.objectname
  return, visarr
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi::compute_intvisi, correl_flux, visibility

  cf = self->correlflux(/quiet)
  correl_flux= total(cf[2,*])

  vis= self->visibility()
  visibility = mean(vis[2,*])
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro xmidispvisi::update_GoodBad
  meanopd= total(*self.opd,1)/self.scanlength

  ;*self.goodfringes= where(abs(meanopd-median(meanopd)) lt self.totopd*1e-6/2. $
  *self.goodfringes= where(*self.tartyp eq 'T'  and $
                           *self.amplfou gt self.goodthresh, Ngood)
   self.Ngood = Ngood
  *self.badfringes = where(*self.amplfou lt self.badthresh  or $
                           abs(meanopd-median(meanopd)) gt self.noiseopd, Nbad)
   self.Nbad = Nbad
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi::print_results, fname, CALIBRATOR=cal, NOCOLOR=nocolor, DONTCLOSE=dontclose
;; so far, there's nothing that uses color, so NOCOLOR is pretty
;; useless :-)

  fringenames = oirFileList(self.fringefile)
  if N_params() lt 1 then $
    fname= "./MIAplots" + MIDIname_datime(fringenames[0]) + ".ps"
  print,"Printing to file ",fname

  colors_saved = self.color

  if keyword_set(cal) then begin
      cal->print_results,fname,/dontclose,NOCOLOR=nocolor
  endif else begin
      set_plot,"PS"
      device,filename=fname,landscape=0, YOffset=1,YSIZE=10,/inches
      if not keyword_set(nocolor) then begin
          device,/color
          tek_color,0,8
          self.color= [ 2, 3, 4, 5, 6, 7 ]
      endif
      !P.multi= [ 4,1,4,0,0]
  endelse

  title = self.objectname + " - " + fringenames[0]

  photA= self->photometry(/Atotal,/Quiet)
  photB= self->photometry(/Btotal,/Quiet)
  plot, photA[0,*], (photA[2,*]+photB[2,*])/2., xtitle="Wavelenth [micron]",$
    title= title + " - Raw photometry (arithmetic average of A and B) [ADU/s]"

  phot= self->photometry(/AB)
  plot, phot[0,*], phot[2,*], xtitle="Wavelenth [micron]",$
    title= title + " - Geometric average of photometry [ADU/s]"

  cf = self->correlflux(/QUIET)
  plot, cf[0,*], cf[2,*], xtitle="Wavelenth [micron]",$
    title= title + " - Raw Correlated Amplitude [ADU/s]"

  vis= self->visibility()
  plot, vis[0,*], vis[2,*], xtitle="Wavelenth [micron]",$
    title= title + " - Raw Visibility", Yrange=[0,max(vis[2,*]*1.1)]

  cf = self->raw_correlflux(/Plot)

  if keyword_set(cal) then begin
      vis = cal->instruvisi()
      plot, vis[0,*], vis[2,*], xtitle="Wavelenth [micron]",$
        title= "Instrumental Visibility", Yrange=[0,max(vis[2,*]*1.1)]

      vis = self->calibratedvisi(cal)
      plot, vis[0,*], vis[2,*], xtitle="Wavelenth [micron]",$
        title= "Calibrated Visibility", Yrange=[0,max(vis[2,*]*1.1)]
  endif

  if not keyword_set(dontclose) then begin
      device,/close
      set_plot,"X"
      self.color= colors_saved
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

FUNCTION xmidispvisi::init, fringefile_in, photfileA, photfileB, $
                    NOISEFILE=noisefile, RUNNING_HIPASS=running_hipass, $
                    MASKFILE=maskfile_in, MASKWIDTH=maskwd, FITMASK=fitmask, $
                    NOCOLOR=nocolor, TRUECOLOR=truecolor, REUSE=reuse, DWIM=dwim,$
                    DISPERSEDTRACK=dispersedbins, ERRORLOG=errorlog, $
                    NOEWSPHOT=noewsphot, dSky=dSky, TARTYPSKIP=tartypskip,$
                    CROSS=cross

  sci_mode = 0	;; that means "sci_mode by FP (under development)

  if (N_params() lt 3) then begin    ;;; modified for sci_mode (=no photfiles)
      self.fringefile = fringefile_in[0]
      if not keyword_set(cross) then begin
          print,'No photometry files given, entering SCI MODE by FP (under development)'
          sci_mode = 1
          self.photfiles[0] = self.fringefile
          self.photfiles[1] = self.fringefile
          photfileA = self.photfiles[0]
      endif
      print,"only fringefile given"
  endif else begin
      self.fringefile= fringefile_in[0]
      photfileA = photfileA[0]
      photfileB = photfileB[0]
      if (N_Params() gt 2) then $
        self.photfiles = [ photfileA, photfileB ] $
      else self.photfiles = photfileA
  endelse

  if N_elements(errorlog) gt 0 then $
    print,"xmidispvisi: errorlog given, not stopping for anything"

  if keyword_set(noisefile) then $
    self.fringefile = noisefile + " " + self.fringefile

  !X.style = 1
  !Y.style = 1
  if keyword_set(dispersedbins) then self.dispersedbins= dispersedbins
  if keyword_set(noewsphot) then  self.ewsphot=0  else  self.ewsphot=1
  if keyword_set(dSky)      then  self.dSky=dSky  else  self.dSky=0
  if not keyword_set(tartypskip) then tartypskip=2
  self.tartypskip = tartypskip
  self.color= mia_std_colortable(truecolor)

  ; we have to use the original filenames because oircompress
  ; removes all the interesting things from the header
  filelist = oirFileList(self.fringefile)

  im = obj_new('imagedata', filelist[0])
  he = im->prihead()
  self.objectname= StrCompress(he->getpar('OBS TARG NAME'),/remove_all)
  self.beam   = he->getpar('INS OPT1')
  self.DIT    = StrCompress(he->getpar('DET DIT'),	/remove_all)
  self.FrFilt = StrCompress(he->getpar('INS FILT NAME'),/remove_all)
  self.Grism  = strcompress(he->getpar('INS GRIS NAME'),/remove_all)
  detwin1strx = he->getpar('DET WIN1 STRX')
  ;detwin1stry= he->getpar('DET WIN1 STRY')
  ;detwin2strx= he->getpar('DET WIN2 STRX')
  ;detwin2stry= he->getpar('DET WIN2 STRY')

  self.scanlength = (im->head())->getpar('maxstep')
  obj_destroy,im

  short_fringefile= MIDIname_datime(self.fringefile)

  ;;;;;;;;;;; MAKE SCI PHOTOMETRY HIGH ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"

  if (self.beam EQ 'SCI_PHOT' and keyword_set(cross)) then begin
      print,"+----------------------------------------------------------------------+"
      print,"|                        SCI_PHOT mode enabled                         |"
      print,"+----------------------------------------------------------------------+"
      Log_error, self.fringefile, "SCI_PHOT mode enabled"

      if (self.Grism EQ 'PRISM') then begin
          curveFile= getenv('prismscurve')
          ews_mask = getenv('prismsmask')
      endif else if (self.Grism EQ 'GRISM') then begin
          curveFile= getenv('grismscurve')
          ews_mask = getenv('grismsmask')
      endif else begin
          err = "Grism keyword is "+self.Grism+", I cannot deal with this"
          Log_error, self.fringefile, err
          if N_Elements(errorlog) gt 0 then begin
              errorlog = errorlog + self.fringefile + ": " + err + string(10B)
          endif else begin
              print,err
              return,0
          endelse
      endelse
      base= "PHOTOMETRY/MIDI." + MIDIname_datime(self.fringefile)

      cmd = getenv("drsRoot") + '/c/bin/dispSciPhot "' + self.fringefile + '" ' + $
        ews_mask + ' ' + curveFile + ' ' + cross + " " + base + " " + strtrim(self.dSky,2)
      print,"calling: ",cmd
      spawn,cmd,EXIT_STATUS=status
      if (status ne 0) then begin
          print,'dispSciPhot failed'
          Log_error, self.fringefile, 'dispSciPhot failed'
          RETURN,0
      endif else print,"+----- dispSciPhot successful -----------------------------------------+"

      if self.photfiles[0] eq "" then begin
          print, "Faking photometry :-)"
          ;; this is an ugly hack, but miamask expects both the de-chopped
          ;; and the original (photometry) file.  Symlinking doesn't work...
          ext= "_dsky"+strtrim(self.dSky,2)+".fits"

          file_copy, base+".Aphotometry.fits", "PHOTOMETRY/ChIm_"+short_fringefile+".Aphotometry"+ext, /Overwrite
          file_copy, base+".Bphotometry.fits", "PHOTOMETRY/ChIm_"+short_fringefile+".Bphotometry"+ext, /Overwrite
          self.photfiles= [ base+".Aphotometry.fits", $
                            base+".Bphotometry.fits" ]
      endif
  endif else begin

      if (self.beam EQ 'SCI_PHOT') then begin
          print,"SCI_PHOT data, but no cross-coupling coefficients given"
          Log_error, self.fringefile,"SCI_PHOT data, but no cross-coupling coefficients given"
      endif
      if (keyword_set(cross)) then $
        print,"HIGH_SENS data, ignoring useless cross-coupling coefficients"

      print,"+----------------------------------------------------------------------+"
      print,"|                        HIGH_SENS mode enabled                        |"
      print,"+----------------------------------------------------------------------+"
  endelse

  ;;;;;;;;;;; MASK ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  if keyword_set(maskfile_in) then begin
      self.maskobj = obj_new('miamask',self.fringefile,self.photfiles,WIDTH=maskwd,$
                             MASKFILE=maskfile_in, ERRORLOG=errorlog )
  endif else begin
      self.maskobj = obj_new('miamask',self.fringefile,self.photfiles,WIDTH=maskwd,$
                             ERRORLOG=errorlog,FITMASK=fitmask,NOEWSPHOT=noewsphot,dSky=dSky)
  endelse
  if not keyword_set(dwim) then self.maskobj->gui,TRUECOLOR=truecolor
  self.maskobj->write_mask
  self.maskfile = self.maskobj->get_mask_name()
  maskdata = self.maskobj->get_mask_data()

  print,"Mask created -----------------------------------------------------------"

  ;;;;;;;;;;; SCI PHOTOMETRY ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  if (self.beam EQ 'SCI_PHOT' and keyword_set(cross)) then begin

      if N_params() lt 3 then begin
          print,"No photometric files -> use sci_photometry directly"

          ;; do it again with the right maskfile ;-)
          cmd = getenv("drsRoot") + '/c/bin/oirMakePhotoSpectra -A ' + $
            base+".Aphotometry.fits -B " + base+".Bphotometry.fits -out " + $
            base+".photometry.fits -mask " + self.maskfile
          print,"calling: ",cmd
          spawn,cmd,EXIT_STATUS=status
          if (status ne 0) then begin
              print,'oirMakePhotoSpectra failed'
              Log_error, self.fringefile, 'oirMakePhotoSpectra failed'
              RETURN,0
          endif
      endif else begin
          print,"Photometric files given -> use sci_photometry to rescale real photometry"

          if keyword_set(dsky) then begin
              chopimgfile0= oirChopPhotoImages(self.photfiles[0],DSKY=dsky)
              chopimgfile1= oirChopPhotoImages(self.photfiles[1],DSKY=dsky)
          endif else begin
              chopimgfile0= oirChopPhotoImages(self.photfiles[0])
              chopimgfile1= oirChopPhotoImages(self.photfiles[1])
              ;dsky=0
          endelse

          cmd = getenv("drsRoot") + '/c/bin/oirMakePhotoSpectra -A ' + $
            chopimgfile0 + " -B " + chopimgfile1 + " -out " + $
            base+".photphotometry.fits -mask " + self.maskfile
          print,"calling: ",cmd
          spawn,cmd,EXIT_STATUS=status
          if (status ne 0) then begin
              print,'oirMakePhotoSpectra failed'
              Log_error, self.fringefile, 'oirMakePhotoSpectra failed'
              RETURN,0
          endif

          cmd = getenv("drsRoot") + '/c/bin/dispSciPhotPhot "' + self.fringefile + $
            '" ' + self.maskfile + ' ' + curveFile + ' ' + cross + ' ' + $
            base+'.photphotometry.fits ' + base + " " + strtrim(self.dsky,2)
          print,"calling: ",cmd
          spawn,cmd,EXIT_STATUS=status
          if (status ne 0) then begin
              print,'dispSciPhotPhot failed'
              Log_error, self.fringefile, 'dispSciPhotPhot failed'
              RETURN,0
          endif
      endelse

      chop_spectra= oirgetdata(base+".photometry.fits")
      self.specA1 = chop_spectra[3].data1
      self.specB1 = chop_spectra[4].data1
      self.specAB = chop_spectra[5].data1

      print,"+----- SCI_PHOT mode photometry successful (I think) ------------------+"
  endif else begin

  ;;;;;;;;;;; HIGH PHOTOMETRY ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

      if keyword_set(noewsphot) then begin
          print,"Photometry A..."
          self->do_photometry, photfileA,maskdata,PhFiltA,spec1,spec2,specPh,ERRORLOG=errorlog
          self.PhFilt = PhFiltA
          self.specA1 = spec1
          self.specA2 = spec2
          self.specAph= specPh

          print,"Photometry B..."
          self->do_photometry, photfileB,maskdata,PhFiltB,spec1,spec2,specPh,ERRORLOG=errorlog
          self.specB1 = spec1
          self.specB2 = spec2
          self.specBph= specPh

          if PhFiltA ne PhFiltB then begin
              eniki= "any key"
              print, "Photometric filter A is ",PhFiltA, ", but"
              print, "Photometric filter B is ",PhFiltB,"."
              err = photfileA + " and " + photfileB + ": different Filters"
              Log_error, photfileA, err
              if N_Elements(errorlog) gt 0 then begin
                  errorlog = errorlog + err + string(10B)
              endif else begin
                  read,  "Do you know what you are doing? ",eniki
                  if eniki ne "y" then return,0
              endelse
          endif

          print,"sqrt(A*B)..."
          self->do_photometry, photfileA, maskdata, PHOTFILEB=photfileB,$
            PhFiltB, spec1,spec2, ERRORLOG=errorlog
          self.specAB = spec1 + spec2

      endif else begin	;; EwsChopImg
          self->ews_photometry, ERRORLOG=errorlog
      endelse
  endelse

  if sci_mode then begin   ;;; by FP
      print,'Photometry from SCI PHOT channels'
      self->do_photometry, photfileA, maskdata,$
          PhFilt, spec2,spec1, sci_mode=1, ERRORLOG=errorlog

      ;;; shift ist necessary since photometric beams are not exactely
      ;;; in line with interferometric beams
      ;;; Note that also a rebinning and interpolation would be necessary,
      ;;; but Frank ignored that.
      ;;; And by the way, the coefficients are not constant.

      ;;; conversion from sci_phot flux to flux in interferometric channels (by FP)

      coef_pa_i1 = [     0.883152,  -0.00202189,  0.000115228, -7.40366e-07]
      coef_pa_i2 = [     0.673113,   -0.0118838,  0.000485562, -3.02225e-06]
      coef_pb_i1 = [      1.29049,   -0.0372456,  0.000758532, -3.86339e-06]
      coef_pb_i2 = [      1.15930,   -0.0124029,  0.000224616, -1.06422e-06]

	;;; coeffs calculated from photometric data sets:
	;;; MIDI.2003-12-12T02:38:44.750.fits , MIDI.2003-12-12T02:40:00.750.fits  (HD18884)

      xcoef = findgen(n_elements(spec1))

      self.specA1 = shift(spec1,1) / poly(xcoef,coef_pa_i1)
      self.specA2 = shift(spec1,2) / poly(xcoef,coef_pa_i2)
      self.specB1 = shift(spec2,0) / poly(xcoef,coef_pb_i1)
      self.specB2 = shift(spec2,1) / poly(xcoef,coef_pb_i2)
      self.specAB = (self.specA1+self.specA2+self.specB1+self.specB2)/2.
  endif

  ;;;;;;;;;;; GET DATA ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  data = getCompressSpec(self.fringefile, self.maskfile, $
                         'PHOTOMETRY/CmprSp_'+short_fringefile+'_'+File_Basename(self.maskfile),$
                         REUSE=reuse )

  if (self.beam EQ 'HIGH_SENS') then begin
      flux1spec = data.data1
      flux2spec = data.data2
      tartyp = data.tartyp1
  endif else if (self.beam EQ 'SCI_PHOT') then begin
      sciBspec  = data.data1
      flux1spec = data.data2
      flux2spec = data.data3
      sciAspec  = data.data4
      tartyp = data.tartyp2
  endif else begin
      print,"INS OPT1 is",self.beam
      print,"Neither HIGH_SENS nor SCI_PHOT, don't know where the fringes are!"
      return,0
  endelse
  fringe = flux1spec - flux2spec
  ;
  ; fringe is now 2D: 1.dim is wavelength, 2.dim is time
  ;
  ;;;;;;;;;;; WAVELENGTH ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  sz = size(fringe)
  self.nLambda= sz[1]
  self.lambda = oirGetWaveLength(filelist[0])

  ;old: self.lambda = -0.0519 * (dindgen(nLambda)+detwin1strx) + 21.1844
  ;P = dindgen(self.nLambda) + detwin1strx
  ;if self.Grism eq "PRISM" then begin
  ;    print,"using wavelength calibration for PRISM"
  ;    self.lambda = -0.0001539 * P^2 + 0.009491 * P + 15.451905
  ;endif else begin
  ;    print,"+------------------------------------------------------+"
  ;    print,"| WARNING! WARNING! WARNING! WARNING! WARNING! WARNING |"
  ;    print,"| Using preliminary wavelength calibration for grism.  |"
  ;    print,"| This is valid for the first interferometric channel, |"
  ;    print,"| but not for the second.  This means that all the     |"
  ;    print,"| results you will get are preliminary and useless!    |"
  ;    print,"+------------------------------------------------------+"
  ;    Log_error, self.fringefile, "Using preliminary wavelength calibration for GRISM"
  ;    if N_elements(errorlog) gt 0 then begin
  ;        errorlog = errorlog + "Using preliminary wavelength calibration for GRISM"+string(10B)
  ;    endif else begin
  ;        ok="" & read,"Do you want to continue? ",ok
  ;        if ok ne "" then print,"I continue anyway"
  ;    endelse
  ;    self.lambda = -1.21122e-6 * P^2 + 0.0232192 * P + 6.61739
  ;endelse
  la_range = where(self.lambda ge 8. and self.lambda le 13.)
  self->set_lambda_bins, min(la_range), (max(la_range) < self.nLambda-4), 3, 3

  ;;;;;;;;;;; RUNNING HIPASS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  if keyword_set(running_hipass) then begin
      print, "Applying running Hi-pass filter..."

      ; the following was (partly) stolen from midiRawFringe
      dtime = 86400. * (data[1].time[0] - data[0].time[0])
      tZero = 1.
      nZero = (tZero/dtime) > 10
      ;print,"dtime ",dtime,", nzero ",nzero

      for i=0,self.nLambda-1 do begin
          ; 1st two points are garbage
          fringe[i, 0:1] = MEDIAN(fringe[i, 2:200])
                                ; remove low freq trend
          fringe[i,*] = fringe[i,*] - SMOOTH(fringe[i,*], nZero,/EDGE_TRUNCATE)
      end
  endif

  ;;;;;;;;;;; SORT SCANS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  nData = N_ELEMENTS(data)

  opd = data.localOpd[1]-data.localOpd[0] + data.opd[1]-data.opd[0]
  ; last two opd points are garbage
  ;opd[nData-2:nData-1] = median(opd[nData-self.scanlength:nData-2])

  while (self.scanlength le 0) do begin
    print,""
    print,"Impossible number of frames per scan: ",strtrim(self.scanlength,2)

    ;; end of scan is where the opd suddenly goes down
    jumps = where( opd[0:nData-2] gt opd[1:nData-1] )
    scanlen= fix(median( jumps[1:N_elements(jumps)-1] - jumps[0:N_elements(jumps)-2] ))
    jumps = "free ram!"
    err   = "impossible number of frames per scan, assuming "+strtrim(scanlen,2)
    Log_error, self.fringefile, err
    if N_Elements(errorlog) gt 0 then begin
        errorlog = errorlog + filelist[0] + ": " + err + string(10B)
        self.scanlength = scanlen
    endif else begin
        line= ""
        read, Prompt="Please enter the correct number (my guess is "+strtrim(scanlen,2)+"): ",line
        if line eq "" then begin
            self.scanlength = scanlen
        endif else begin
            self.scanlength = line
        endelse
    endelse
  endwhile

  nScan = nData/self.scanlength
  print, nData," frames =>", nScan, " scans"

  ; reform into scans, droping last fractional scan if any
  nFringe= LONG(self.scanlength) * LONG(nScan)
  fringe = reform( fringe[*,0:nFringe-1],self.nLambda,self.scanlength,nScan,/overwrite)
  opd    = reform(    opd[  0:nFringe-1],	      self.scanlength,nScan,/overwrite)
  tartyp = reform(strtrim(tartyp[0:nFringe-1],2),     self.scanlength,nScan,/overwrite)
  if (self.beam EQ 'SCI_PHOT') then begin
      sciAspec= reform(sciAspec[*,0:nFringe-1],self.nLambda,self.scanlength,nScan,/overwrite)
      sciBspec= reform(sciBspec[*,0:nFringe-1],self.nLambda,self.scanlength,nScan,/overwrite)
      self.fluxA= ptr_new( sciAspec, /NO_COPY)
      self.fluxB= ptr_new( sciBspec, /NO_COPY)
      scifA = total(*self.fluxA,1)
      scifB = total(*self.fluxB,1)
  endif
  self.fringe= ptr_new( fringe, /NO_COPY)
  self.opd   = ptr_new(   opd , /NO_COPY)
  self.tartyp= ptr_new( strarr(nScan), /NO_COPY)
  self.goodfringes= ptr_new(fltarr(nScan,/NOzero),/NO_COPY)
  self.badfringes = ptr_new(fltarr(nScan,/NOzero),/NO_COPY)

  ;;;;;;;;;;; TARTYP PER SCAN and OPD-LEN PER SCAN ;;;;;;;;;;;;;;;;;;;;;;;;;;;

  junk= where(tartyp ne 'T', count)
  ;if (count gt 0) then begin
  ;    print,"-------------------------------------------------------------------------"
  ;    print,"WARNING: tartyp is not always 'T'!  Was chopping on during fringe track??"
  ;    print,"-------------------------------------------------------------------------"
  ;endif

  print,"Skipping tartyp of",tartypskip," frames at beginning and end"
  opdlen = dblarr(nScan)
  for sc=0,nScan-1 do begin
      opdlen[sc]= max((*self.opd)[*,sc]) - min((*self.opd)[*,sc])

      if (count gt 0) then begin
          tt = tartyp[ tartypskip:self.scanlength-1-tartypskip, sc]
          ;;print, Format='("Scan",I5,": ",999A1, $)', sc, tartyp[*, sc]
          junk= where(tt eq 'T', cnt)
          if cnt eq N_Elements(tt) then begin
              (*self.tartyp)[sc]= 'T'
          endif else begin
              junk= where(tt eq 'S', cnt)
              if cnt eq N_Elements(tt) then (*self.tartyp)[sc]= 'S' $
              else begin
                  print, Format='("Scan",I5,": ",999A1, $)', sc, tartyp[*, sc]
                  (*self.tartyp)[sc]= 'U'
                  print, " => ",(*self.tartyp)[sc]
              endelse
          endelse
          ;;print, " => ",(*self.tartyp)[sc]
      endif else (*self.tartyp)[sc]= 'T'
  endfor
  self.totopd = median(opdlen) * 1e6
  print," total opd length per scan:",self.totopd," micron"

  junk= where(*self.tartyp eq 'T',tcnt)
  junk= where(*self.tartyp eq 'S',scnt)
  junk= where(*self.tartyp eq 'U',ucnt)
  print, tcnt,scnt,ucnt, $
    format='(I5," scans on target,",I5," scans on sky,",I5," scans unusable")'

  ;;;;;;;;;;; STANDING HIPASS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  if not keyword_set(running_hipass) then begin
      print, "Subtracting mean amplitude of scans..."
      for la=0,self.nLambda-1 do begin
	  ; 1st two points are garbage
          (*self.fringe)[la,0:1,0]= MEDIAN( (*self.fringe)[la,2:self.scanlength-1,0])
          for sc=0,nScan-1 do begin
              (*self.fringe)[la,*,sc] = $
                (*self.fringe)[la,*,sc] - mean( (*self.fringe)[la,*,sc])
          endfor
      endfor
  endif

  ;;;;;;;;;;; WHITE FRINGE AMPL ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  case self.FrFilt of
	; "OPEN" added by FP, wavelength range just estimated
	; limited to 13.4, because that is the edge of the window, RK 01-aug03
  'OPEN':  begin print,"filter: open"  & self.longwl=13.4 & self.shortwl= 7.  & end
  'Nband': begin print,"N-band filter" & self.longwl=13.  & self.shortwl= 7.7 & end
  'N8.7':  begin print,"N 8.7 filter"  & self.longwl=10.3 & self.shortwl= 7.2 & end
  'SiC':   begin print,"SiC filter"    & self.longwl=13.0 & self.shortwl=10.5 & end
  else: begin print,"Unknown filter: ", self.FrFilt & return,0 & end
  endcase

  wherelam = where(self.lambda ge self.shortwl and self.lambda le self.longwl)
  lo = min(wherelam) > 0
  hi = max(wherelam) < self.nLambda-1

  ;; we do this now because we did not know the lo:hi range before
  self.flux1 = ptr_new( total( flux1spec[lo:hi,*], 1), /NO_COPY)
  self.flux2 = ptr_new( total( flux2spec[lo:hi,*], 1), /NO_COPY)

  self->make_white_fringe, BINS=dispersedbins

  ;;;;;;;;;;; PLOT RANGES ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  self.frame_rng= [0, nData]
  self.opd_rng  = [ min(*self.opd), max(*self.opd)] * 1e3

  if (self.beam EQ 'SCI_PHOT') then begin
      minflux= min([ min(*self.flux1), min(*self.flux2), min(scifA), min(scifB) ])
      maxflux= max([ max(*self.flux1), max(*self.flux2), max(scifA), max(scifB) ])
  endif else begin
      minflux= (min(*self.flux1) < min(*self.flux2))
      maxflux= (max(*self.flux1) > max(*self.flux2))
  endelse
  self.flux_rng = [ minflux, maxflux ] / 1d3

  frng = total(*self.fringe,1)/1d3
  self.fdiff_rng= [ min(frng), max(frng)]
  frng = 0
  self.ampl_rng = [ 0., 100.]

  ;;;;;;;;;;; DEFAULT THRESHOLDS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  self.goodthresh= 33.
  self.badthresh = 0.
  self.noiseopd  = 4e-3	; this is in meters
  self.charsize  = 1.7;
  self->update_GoodBad

  return,1	;; we made it
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro xmidispvisi::make_white_fringe, BINS=dispersedbins

  nScan = N_Elements(*self.tartyp)

  wherelam = where(self.lambda ge self.shortwl and self.lambda le self.longwl)
  lo = min(wherelam) > 0
  hi = max(wherelam) < self.nLambda-1

  if keyword_set(dispersedbins) then begin
    if (dispersedbins le 1  $
        or dispersedbins gt hi-lo) then dispersedbins= 10
    if (dispersedbins MOD 2 ne 0) then dispersedbins= dispersedbins-1

    print,"Dispersed tracking, using",dispersedbins," bins in wavelength"
    mid = fix((lo+hi)/2.)
    dpx = fix((hi-lo)/dispersedbins)	; No. of pix per lambda-bin
    lo = mid - dispersedbins/2 * dpx
    hi = mid + dispersedbins/2 * dpx - 1
    ;; sum over lambda-bins, then do fft along opd-scan
    foufriSpec = FFTdim( rebin((*self.fringe)[lo:hi,*,*],$
                               dispersedbins,self.scanlength,nScan),Dimension=2)
  endif else begin
    print,"Using code for Undispersed tracking"
    dispersedbins = 0
    ;; sum over lambda, then do fft along opd-scan
    foufriSpec = FFTdim( total((*self.fringe)[lo:hi,*,*], 1), Dimension=1)
  endelse
  print, self.shortwl, self.longwl, lo, hi, $
    Format='("integrating over lambda = ",F4.1,"...",F4.1,", x =",I3,"...",I3)'

  self.foufri = ptr_new( abs(foufriSpec)^2, /NO_COPY)
  self.amplfou= ptr_new( fltarr(nScan), /NO_COPY)

  ;; compute the final white-light fringe amplitude and put it in self.amplfou

  if (dispersedbins gt 0) then begin
      lastep = (self.longwl-self.shortwl)/float(dispersedbins)
      print,"lastep =",lastep
      for la=0,dispersedbins-1 do begin
          lowfreq = floor( self.totopd / (self.shortwl+lastep*(la+1)) +0.5)
          highfreq= floor( self.totopd / (self.shortwl+lastep* la   ) +0.5)
          if (highfreq gt self.scanlength/2) then highfreq= self.scanlength/2

          print, "dispersed bin",la,": ",lowfreq,"...",highfreq,", wl =",self.shortwl+lastep*la

          *self.amplfou = (*self.amplfou) + sqrt(total((*self.foufri)[la,lowfreq:highfreq,*],2))
      endfor
      if (dispersedbins gt 1) then *self.foufri = total(*self.foufri,1)
      ;; we need the *integrated* power-spectra for the good/bad spectra plots
  endif else begin
      dk = self.totopd / 2./(self.longwl-self.shortwl)	; half bandwidth in k
      mk = self.totopd * 2./(self.longwl+self.shortwl)	; mean k
      lowfreq = floor( mk - dk +0.5)
      highfreq= floor( mk + dk +0.5)
      if (highfreq gt self.scanlength/2) then highfreq= self.scanlength/2

      *self.amplfou = sqrt(total( (*self.foufri)[lowfreq:highfreq,*],1))
  endelse

  *self.amplfou = *self.amplfou * (100./max(*self.amplfou))

  self->redraw_topdraw
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi::cleanup, KEEP_COMPR=keep_compr
  print,"xmidispvisi cleaning up...Aaaarghhhh..."

  if not keyword_set(keep_compr) then begin
      ComprFile = $
        'PHOTOMETRY/CmprSp_'+MIDIname_datime(self.fringefile)+'_'+File_Basename(self.maskfile)
      if file_test(ComprFile) then begin
          print,"Removing ",ComprFile
          file_delete, ComprFile
      endif
  endif
  obj_destroy, self.maskobj
  PTR_FREE, self.opd
  PTR_FREE, self.tartyp
  PTR_FREE, self.flux1
  PTR_FREE, self.flux2
  PTR_FREE, self.fringe
  PTR_FREE, self.foufri
  PTR_FREE, self.amplfou
  PTR_FREE, self.goodfringes
  PTR_FREE, self.badfringes
end


PRO xmidispvisi::stop
  help,*self,/struct
  stop
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; wrapper for lazy programmers
;;
Function xmdv, files_in, PREFIX=prefix,$
               NOISEFILE=noisefile, RUNNING_HIPASS=running_hipass, $
               MASKFILE=maskfile_in, MASKWIDTH=maskwd, FITMASK=fitmask, $
               NOCOLOR=nocolor, TRUECOLOR=truecolor, REUSE=reuse, DWIM=dwim,$
               DISPERSEDTRACK=dispersedbins, ERRORLOG=errorlog, $
               NOEWSPHOT=noewsphot, dSky=dSky, TARTYPSKIP=tartypskip,$
               CROSS=cross

  ; _EXTRA=extra would be much nicer, but then the check if keywords are
  ; allowed is turned off...

  ;; if errorlog is given, make sure it is defined, so our subroutines
  ;; can check for it
  if ARG_Present(errorlog) and (N_elements(errorlog) eq 0) then errorlog=""

  if keyword_set(prefix) then begin
      for i=0,N_elements(files_in)-1 do begin
          flist = oirFileList(files_in[i])
          files_in[i] = prefix + "/" + flist[0]
          for j=1,N_elements(flist)-1 do $
            files_in[i]= files_in[i] + " " + prefix + "/" + flist[j]
      endfor
  endif
  if File_Test(files_in[0],/Directory) then files_in= midiguis(dir=files_in)
  ; user gave a dir, so ask for files in it

  if N_Elements(files_in) eq 1 then begin
    ftrfile= files_in
    file0  = (oirFileList(ftrfile))[0]
    im = obj_new('imagedata', file0)
    he = im->prihead()
    objectname= STRCOMPRESS(he->getpar('OBS TARG NAME'),/remove_all)
    nrts_mode = strcompress(he->getpar('NRTS MODE'),/remove_all)
    obj_destroy,im
    if not strmatch(nrts_mode,'OBS_FRINGE_*',/Fold_case) then begin
		print,"you must specify a fringe-track file!" & return,0
    endif
    ;print,ftrfile," has object ",objectname," in NRTS_mode ",nrts_mode

    last_char = strpos(file0,'MIDI.20')
    dirname = strmid(file0,0,last_char)
    print,"dir is '",dirname,"'"
    restore, dirname+"midiGui.SAV"
    arg = 1
    for f=0,N_Elements(FBTFilelistFilelist)-1 do begin
      ff0 = (oirFileList(FBTFilelistFilelist[f]))[0]
      if (ff0 gt file0) then begin
        im = obj_new('imagedata', ff0)
        he = im->prihead()
        objname= strcompress(he->getpar('OBS TARG NAME'),/remove_all)
        nrts_m = strcompress(he->getpar('NRTS MODE'),/remove_all)
	shutter= strcompress(he->getpar('INS SHUT NAME'),/remove_all)
        obj_destroy,im
	if (objectname ne objname) then break
        if (nrts_m eq "DEFAULT_CHOP") then begin
          ;print,ff0," -> ",objname,", ",nrts_m,", ",shutter
	  if (shutter eq 'AOPEN') then phAfile = FBTFilelistFilelist[f]
	  if (shutter eq 'BOPEN') then phBfile = FBTFilelistFilelist[f]
        endif
      endif
    endfor
    if (N_elements(phAfile) le 0 or N_Elements(phBfile) le 0) then begin
	print,"no photometry with same object after fringe"
        print,"I will try sci_mode by FP (under development)"
        print,"You may want to panic now."
        print,""
        return, obj_new('xmidispvisi', ftrfile, $
                  NOISEFILE=noisefile, MASKFILE=maskfile_in, MASKWIDTH=maskwd,$
                  RUNNING_HIPASS=running_hipass, $
                  NOCOLOR=nocolor, TRUECOLOR=truecolor, REUSE=reuse, DWIM=dwim,$
                  DISPERSEDTRACK=dispersedbins, ERRORLOG=errorlog,$
                  NOEWSPHOT=noewsphot, dSky=dSky, TARTYPSKIP=tartypskip	)
    endif
  endif else begin
    ; more than one file given
    for i=0,N_elements(files_in)-1 do begin
      print, "checking ", files_in[i]
      file0  = (oirFileList(files_in[i]))[0]
      if not file_test(file0,/READ) then begin
	print,"file ",file0," does not exist or is not readable"
        return,0
      endif
      im = obj_new('imagedata', file0)
      he = im->prihead()
      nrts_mode= strcompress(he->getpar('NRTS MODE'),/remove_all)
      shutter  = strcompress(he->getpar('INS SHUT NAME'),/remove_all)
      obj_destroy,im
      if (strmatch(nrts_mode,'OBS_FRINGE_*',/Fold_case)) then begin
          print,"interferometric file"
          ftrfile = files_in[i]
      endif else begin
          if (strmatch(nrts_mode,'*_CHOP',/Fold_case)) then begin
              ;;print,ff0," -> ",objname,", ",nrts_m,", ",shutter
              if (shutter eq 'AOPEN') then begin
                  print,"photometry A" & phAfile = files_in[i]
              endif else if (shutter eq 'BOPEN') then begin
                  print,"photometry B" & phBfile = files_in[i]
              endif
          endif
      endelse
    endfor
  endelse

  print,""
  print,"Fringes:      ",ftrfile
  print,"Photometry A: ",phAfile
  print,"Photometry B: ",phBfile
  print,"------------------------------------------------------------------------"
  return, obj_new('xmidispvisi', ftrfile, phAfile, phBfile, $
                  NOISEFILE=noisefile, RUNNING_HIPASS=running_hipass, $
                  MASKFILE=maskfile_in, MASKWIDTH=maskwd, FITMASK=fitmask, $
                  NOCOLOR=nocolor, TRUECOLOR=truecolor, REUSE=reuse, DWIM=dwim,$
                  DISPERSEDTRACK=dispersedbins, ERRORLOG=errorlog,$
                  NOEWSPHOT=noewsphot, dSky=dSky, TARTYPSKIP=tartypskip, CROSS=cross)
end
