;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
; $Id: xmidispvisi_gui.pro,v 1.12 2006/09/29 11:23:30 koehler Exp $
;
; xmidispvisi_gui.pro
; Created:     Fri Sep 30 16:57:57 2005 by Koehler@kaag
; Last change: Tue Sep 26 15:30:50 2006
;
; PURPOSE:
;	GUI routines for the xmidispvisi-object
;
; CALLING SEQUENCE:
;	x->gui
;
; INPUT:
;	an xmidispvisi-object
;
; OUTPUT:
;	one or more windows on your computer screen
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi::white_fringe_gui, BASE=papa

  desc= ['0, LABEL, Method to compute white fringe amplitude, LEFT',$
         '0, LABEL, 0 bins means integrate over N-band, LEFT',$
         '0, LABEL, Numbers > 0 mean divide N-band into so many bins, LEFT',$
         '0, LABEL, (Read the manual if you do not know what this means), LEFT',$
         '0, INTEGER, 0, LABEL_LEFT=Number of bins:, WIDTH=10, TAG=BINS',$
         '0, BUTTON, Set White Fringe Method, QUIT, TAG=Quit' ]

  if keyword_set(papa) then begin
      res = cw_form(desc,Title='White Fringe Amplitude',/Column,GROUP_LEADER=papa)
  endif else begin
      res = cw_form(desc,Title='White Fringe Amplitude',/Column)
  endelse

  self->make_white_fringe, BINS=res.bins
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi::set_frame_range, rng
  if (self.frame_rng[0] ne rng[0]) or (self.frame_rng[1] ne rng[1]) then begin
      self.frame_rng = rng
      self->redraw_topdraw
  endif
end

PRO xmidispvisi::set_opd_range, rng
  if (self.opd_rng[0] ne rng[0]) or (self.opd_rng[1] ne rng[1]) then begin
      self.opd_rng = rng
      self->redraw_topdraw
  endif
end

PRO xmidispvisi::set_flux_range, rng
  if (self.flux_rng[0] ne rng[0]) or (self.flux_rng[1] ne rng[1]) then begin
      self.flux_rng = rng
      self->redraw_topdraw
  endif
end

PRO xmidispvisi::set_fluxdiff_range, rng
  if (self.fdiff_rng[0] ne rng[0]) or (self.fdiff_rng[1] ne rng[1]) then begin
      self.fdiff_rng = rng
      self->redraw_topdraw
  endif
end

PRO xmidispvisi::set_fringe_ampl_range, rng
  if (self.ampl_rng[0] ne rng[0]) or (self.ampl_rng[1] ne rng[1]) then begin
      self.ampl_rng = rng
      self->redraw_topdraw
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi::plot_opd, NOCOLOR=nocolor

  plot, *self.opd * 1e3, xtickname=replicate(" ",30), ymargin=[0,2],$
	ytitle= 'OPD [mm]', xrange= self.frame_rng, yrange= self.opd_rng

  if not keyword_set(nocolor) then begin
      nFrames= N_elements(*self.opd)
      meanopd= total(*self.opd,1)/self.scanlength
      ;;oplot, rebin( meanopd, nFrames), color=self.color[1]
      med = median(meanopd)
      ;;print,nFrames," Frames, median OPD =",med
      ;oplot, [0,nFrames], [med,med]*1e3, linestyle=2
      oplot, self.frame_rng, [med,med]*1e3, linestyle=2
      good = where( abs(meanopd - med) lt self.totopd*1e-6/2.,Ngood)
      if Ngood gt 0 then begin
          for iScan=0,Ngood-1 do begin
              goodi= good[iScan]*self.scanlength + indgen(self.scanlength)
              oplot, goodi, (*self.opd)[goodi]*1e3,color=self.color[1]
          endfor
      endif

      bad = where( abs(meanopd - med) gt self.noiseopd, Nbad)
      if Nbad gt 0 then begin
          for iScan=0,Nbad-1 do begin
              badi= bad[iScan]*self.scanlength + indgen(self.scanlength)
              oplot, badi, (*self.opd)[badi]*1e3,color=self.color[0]
          endfor
      endif
  endif
end
;;
;; make sure !P.multi is set up properly before calling plot_static
;;
PRO xmidispvisi::plot_static, NOCOLOR=nocolor

  print,"frmrange",self.frame_rng
  tspace = 10^fix(alog10((self.frame_rng[1] - self.frame_rng[0])/16)+0.8)
  print,"tspcae ",tspace
  tick0 = float(self.frame_rng[0])/tspace
  if (fix(tick0) lt tick0) then tick0= fix(tick0)+1
  tick0 = tick0 * tspace
  print,"tick0 ",tick0

  !X.ticks= fix((self.frame_rng[1] - tick0)/tspace)
  !X.tickv= tick0 + indgen(!X.ticks+1)*tspace

  ;; 1.plot: OPD
  self->plot_opd, NOCOLOR=nocolor

  ;; 2.plot: flux1 and 2
  plot, *self.flux1/1d3, xrange=self.frame_rng, yrange=self.flux_rng, ymargin=[2,0],$
	ytickformat='(I6)', xtickname=replicate(" ",30), ytitle= "Flux [Kcnts]"
  oplot,*self.flux2/1d3, linestyle=1
  if Ptr_Valid(self.fluxA) then oplot,total(*self.fluxA,1)/1d3, color=self.color[3]
  if Ptr_Valid(self.fluxB) then oplot,total(*self.fluxB,1)/1d3, color=self.color[4]

  ;; 3.plot: fringe signal
  frng = total(*self.fringe,1)/1d3
  plot, frng, xrange=self.frame_rng, yrange=self.fdiff_rng,ymargin=[4,-2],$
	xtickformat='(I5)', xtitle= "Frame no.", ytitle="Flux Diff. [Kcnts]"

  sky = where(*self.tartyp eq 'S')	; scan-nos on sky
  if sky[0] ne -1 then $
    for s=0,N_elements(sky)-1 do $
      oplot, sky[s]*self.scanlength + indgen(self.scanlength), frng[*,sky[s]], color=self.color[2]

  und = where(*self.tartyp eq 'U')	; scan-nos undefined tartyp
  if und[0] ne -1 then $
    for s=0,N_elements(und)-1 do $
      oplot, und[s]*self.scanlength + indgen(self.scanlength), frng[*,und[s]], color=self.color[0]

  self.multi = !P.multi
  !X.ticks = 0
  !X.tickv = [0,0]

  ;; 4.plot: fringe amplitude per scan
  self->plot_fringeampl

  !P.multi = self.multi
end

PRO xmidispvisi::plot_fringeampl, CLEAR=clear

  range = float(self.frame_rng) / self.scanlength
  tspace= fix((range[1] - range[0])/16+0.5) > 1
  tick0 = fix( range[0]) & if (tick0 lt range[0]) then tick0++
  ticks = fix((range[1]-tick0)/tspace) > 0
  tickv = tick0 + indgen(ticks+1)*tspace

  plot, *self.amplfou, xrange=range, yrange=self.ampl_rng, xticks=ticks,xtickv=tickv,$
    ymargin=[4,0], xtitle= "Scan no.", ytitle= "Fringe Amplitude"

  if (self.Ngood gt 1) then begin
      if keyword_set(clear) then begin
          oplot,*self.goodfringes, (*self.amplfou)[*self.goodfringes], color=0
          ;;oplot,(*self.amplfou) > self.oldgthres, color=0
          oplot,(*self.amplfou) < self.oldbthres, color=0
          oplot, range, [self.oldgthres,self.oldgthres],linestyle=1,color=0
          oplot, range, [self.oldbthres,self.oldbthres],linestyle=1,color=0
      endif else begin
          oplot,*self.goodfringes, (*self.amplfou)[*self.goodfringes], color=self.color[1]
          ;;oplot,(*self.amplfou) > self.goodthresh,color=self.color[1]
          oplot,(*self.amplfou) < self.badthresh, color=self.color[0]
          oplot, range, [self.goodthresh,self.goodthresh],linestyle=1
          oplot, range, [self.badthresh, self.badthresh ],linestyle=1
          self.oldgthres= self.goodthresh
          self.oldbthres= self.badthresh
      endelse
  endif
end

PRO xmidispvisi::plot_goodbad

  ;!p.charsize= 1.
  ymax = max(*self.foufri)
  if (self.Ngood gt 0) then begin
      goodfou = (*self.foufri)[*,*self.goodfringes]
      plot, goodfou[*,0], yrange=[0,ymax], $
        title="Signal Scans", xtitle= "Frequency", ytitle="Spectral Power"
      if (self.Ngood gt 1) then begin
          for f=1,self.Ngood-1 do oplot, goodfou[*,f]
          oplot, total(goodfou,2)/self.Ngood, color=self.color[2]
      endif

      lowfreq = floor( self.totopd/self.longwl+0.5)
      highfreq= floor( self.totopd/self.shortwl+0.5)
      if (highfreq gt self.scanlength/2) then highfreq= self.scanlength/2
      oplot, [ lowfreq, lowfreq], [0,ymax], linestyle=2
      oplot, [highfreq,highfreq], [0,ymax], linestyle=2

  endif

  if (self.Nbad gt 0) then begin
      badfou = (*self.foufri)[*,*self.badfringes]
      plot, badfou[*,0], yrange=[0,ymax], $
        title="Noise Scans", xtitle= "Frequency", ytitle="Spectral Power"
      if (self.Nbad gt 1) then begin
          for f=1,self.Nbad-1 do oplot, badfou[*,f]
          oplot, total(badfou,2)/self.Nbad, color=self.color[2]
      endif
  endif
end

PRO xmidispvisi::plot_fringe_histogram, NOCOLOR=nocolor

  hist = histogram(*self.amplfou,NBINS=40,OMIN=hmin,OMAX=hmax)
  xhist= findgen(40) * (hmax-hmin)/39. + hmin
  mx   = max(hist) * 1.2

  plot, xhist, hist, title="Histogram of Fourier-Amplitudes",ystyle=2, psym=10

  if( self.Ngood gt 0) then begin
      hist = histogram( (*self.amplfou)[*self.goodfringes],$
                        NBINS=40, MIN=hmin, MAX=hmax)
      if keyword_set(nocolor) then begin
          oplot, xhist,hist, linestyle=1, psym=10
      endif else begin
          oplot, xhist,hist, color=self.color[1], psym=10
      endelse
  endif
  if( self.Nbad gt 0) then begin
      hist = histogram( (*self.amplfou)[*self.badfringes],$
                        NBINS=40, MIN=hmin, MAX=hmax)
      if keyword_set(nocolor) then begin
          oplot, xhist,hist, linestyle=2, psym=10
      endif else begin
          oplot, xhist,hist, color=self.color[0], psym=10
      endelse
  endif
  oplot, [self.goodthresh,self.goodthresh], [0,mx]
  oplot, [ self.badthresh, self.badthresh], [0,mx], linestyle=1
end

PRO xmidispvisi::plot_opd_histogram, NOCOLOR=nocolor

  meanopd= total(*self.opd,1)/self.scanlength * 1e3
  hist = histogram(meanopd, NBINS=40,OMIN=hmin,OMAX=hmax)
  xhist= findgen(40) * (hmax-hmin)/39. + hmin
  mx   = max(hist) * 1.2

  plot, xhist, hist, title="Histogram of Mean Optical Path Differences",ystyle=2, psym=10
  if( self.Ngood gt 0) then begin
      hist = histogram( meanopd[*self.goodfringes], NBINS=40, MIN=hmin, MAX=hmax)
      if keyword_set(nocolor) then begin
          oplot, xhist,hist, linestyle=1, psym=10
      endif else begin
          oplot, xhist,hist, color=self.color[1], psym=10
      endelse
  endif
  if( self.Nbad gt 0) then begin
      hist = histogram( meanopd[*self.badfringes], NBINS=40, MIN=hmin, MAX=hmax)
      if keyword_set(nocolor) then begin
          oplot, xhist,hist, linestyle=2, psym=10
      endif else begin
          oplot, xhist,hist, color=self.color[0], psym=10
      endelse
  endif
  thresh= median(meanopd) - self.noiseopd*1e3  &  oplot, [thresh,thresh], [0,mx]
  thresh= median(meanopd) + self.noiseopd*1e3  &  oplot, [thresh,thresh], [0,mx]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro xmidispvisi::redraw_topdraw
  if self.topdrawID gt 0 then begin
      wset, self.topdrawID
      !p.multi= [0, 0, 4, 0, 0]
      save_charsize = !p.charsize
      !p.charsize = self.charsize
      self->plot_static
      !p.charsize= save_charsize
      self->update_gui
  endif ;else print,"GUI closed, not redrawing"
end

Pro xmidispvisi::update_gui, CLEAR=clear

  ;;;;; plot fourier fringe amplitude and thresholds

  wset, self.topdrawID
  save_charsize = !p.charsize
  !p.charsize = self.charsize
  if keyword_set(clear) then !p.multi= [0, 0, 4, 0, 0] $
    else !p.multi= [4, 0, 4, 0, 0]
  self->plot_opd

  !p.multi= self.multi
  self->plot_fringeampl,/clear
  self->update_GoodBad
  !p.multi= self.multi
  self->plot_fringeampl
  !p.charsize = save_charsize

  self->compute_intvisi, correl_flux,visibility

  Widget_Control, self.goodthl,SET_Value="Good threshold: "+strtrim(self.goodthresh,2)
  Widget_Control, self.badthl ,SET_Value="Noise OPD:  "+strtrim(self.noiseopd*1000.,2)+" mm"
  Widget_Control, self.corrflx,SET_VALUE="Correlated flux "+$
    strtrim(string(correl_flux,   format='(G10.3)'),2)
  Widget_Control, self.visibi, SET_VALUE="VISIBILITY = "+$
    strtrim(string(visibility,    format='(F6.4)'),2)

  ;;;;; plot the good, the bad, and the ugly

  wset, self.lftdrawID
  !P.multi = [0,0,2,0,0]
  self->plot_goodbad

  ;;;;; plot histogram

  wset,self.rgtdrawID
  !P.multi = [0,0,2,0,0]
  self->plot_fringe_histogram
  self->plot_opd_histogram
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; this is pretty much just a copy of the GUI, so it should live in
;; this source file.
;;
PRO xmidispvisi::Hardcopy, fname,NOCOLOR=nocolor,TRACES=traces,DONTCLOSE=dontclose

  if N_params() lt 1 then $
    fname= "./Midivisi" + MIDIname_datime(self.fringefile) + ".ps"
  print,"Printing to file ",fname

  xmargin_saved= !X.Margin
  colors_saved = self.color
  set_plot,"PS"
  device,filename=fname,/Landscape
  if not keyword_set(nocolor) then begin
	device,/color
	tek_color,0,8
	self.color= [ 2,3,4 ]
  endif
  !P.multi= [ 8, 1, 8, 0, 0 ]
  self->plot_static,NOCOLOR=nocolor

  !P.multi= [ 6, 2, 4, 0, 1 ]
  !X.Margin[1] = 30
  self->plot_goodbad

  !P.multi= [ 2, 2, 4, 0, 1 ]
  !X.Margin[0] = 30 & !X.Margin[1] = xmargin_saved[1]
  self->plot_fringe_histogram,NOCOLOR=nocolor

  !P.multi= [ 1, 2, 4, 0, 1 ]
  self->plot_opd_histogram,NOCOLOR=nocolor

  !X.Margin= xmargin_saved
  !P.multi = [ 1, 1, 1, 0, 1 ]
  device,font_size=10
  nScan = N_elements(*self.amplfou)
  XYOuts, 0.5, 0.45, "Object: "+self.objectname, Alignment=0.5, /NORMAL
  XYOuts, 0.5, 0.42, "Photom.Filter: "+self.PhFilt, Alignment=0.5,/Normal
  XYOuts, 0.5, 0.39, "Fringe Filter: "+self.FrFilt, Alignment=0.5,/Normal
  XYOuts, 0.5, 0.36, "Scanlength "+strtrim(self.scanlength,2)+$
    " => "+strtrim(nScan,2)+" scans", Alignment=0.5,/Normal
  XYOuts, 0.5, 0.33, "Beam is "+self.beam, Alignment=0.5,/Normal
  fluxstr= string( total(self.specA1), total(self.specA2), $
                   Format='("A: ",G0.3,", ",G0.3," ph/sec")')
  XYOuts, 0.5, 0.30, fluxstr, Alignment=0.5,/Normal
  fluxstr= string( total(self.specB1), total(self.specB2), $
                   Format='("B: ",G0.3,", ",G0.3," ph/sec")')
  XYOuts, 0.5, 0.27, fluxstr, Alignment=0.5,/Normal

  XYOuts, 0.5, 0.23, "Good threshold: "+strtrim(self.goodthresh,2), Alignment=0.5,/Normal
  XYOuts, 0.5, 0.20, "Noise OPD: "+strtrim(self.noiseopd*1000.,2)+" mm",Alignment=0.5,/Normal

  self->compute_intvisi, correl_flux,visibility

  XYOuts, 0.5, 0.07, "Correlated Flux "+$
    strtrim(string(correl_flux,  format='(G10.3)'),2), Alignment=0.5,/Normal
  XYOuts, 0.5, 0.03, "VISIBILITY = "+$
    strtrim(string(visibility,    format='(F6.4)'),2), Alignment=0.5,/Normal

  if keyword_set(TRACES) then begin
      !X.Margin = xmargin_saved
      ;!P.multi= [ 0, 2, 1, 0, 1 ]
      ;if not keyword_set(nocolor) then begin
      ;    tek_color
      ;    colors= [ 2,3,4 ]
      ;endif
      ;plot_traces, [self.traceFr[0],self.trace[0,0],self.trace[1,0]], title="Window 1", COLORS=colors
      ;plot_traces, [self.traceFr[1],self.trace[0,1],self.trace[1,1]], title="Window 2", COLORS=colors
      self.maskobj->show_traces, /NOWIN, Colors=self.color
  endif

  if not keyword_set(dontclose) then begin
      device,/close
      set_plot,"X"
  endif
  !X.Margin = xmargin_saved
  self.color= colors_saved
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO topdrawparams_event, ev
  WIDGET_CONTROL, ev.TOP, GET_UVALUE=me

  if ev.id eq me.form_id then begin
      case (ev.tag) of
          "APPLY": begin
              widget_control, me.form_id, Get_Value=val
              ;;help, val, /str
              ;;help, me, /str
              ;;help, me.obj, /str
              me.obj->set_frame_range,	    [val.frmin,  val.frmax  ]
              me.obj->set_opd_range,	    [val.opdmin, val.opdmax ]
              me.obj->set_flux_range,	    [val.fluxmin,val.fluxmax]
              me.obj->set_fluxdiff_range,   [val.diffmin,val.diffmax]
              me.obj->set_fringe_ampl_range,[val.ampmin, val.ampmax ]
          end
          "QUIT": WIDGET_CONTROL, ev.TOP, /DESTROY
          ELSE: ;; relax
      endcase
  endif
end

PRO xmidispvisi::topdrawparams, papa

  screensz = get_screen_size()
  top= Widget_Base(GROUP_LEADER=papa, TITLE="Parameters for long plots", YOffset=screensz[1]/1.9)
  fid= cw_form(top,$
               ['1, BASE, ,ROW, FRAME',$
                 '0, LABEL, Frame Nos..', $
                 '0, INTEGER,'+string(self.frame_rng[0])+',LABEL_LEFT= , WIDTH=10, TAG=frmin',$
                 '2, INTEGER,'+string(self.frame_rng[1])+',LABEL_LEFT=-, WIDTH=10, TAG=frmax',$
                '1, BASE, ,ROW, FRAME',$
                 '0, LABEL, OPD........', $
                 '0, FLOAT, '+string(self.opd_rng[0])+',  LABEL_LEFT= , WIDTH=10, TAG=opdmin',$
                 '2, FLOAT, '+string(self.opd_rng[1])+',  LABEL_LEFT=-, WIDTH=10, TAG=opdmax',$
                '1, BASE, ,ROW, FRAME',$
                 '0, LABEL, Flux.......', $
                 '0, FLOAT, '+string(self.flux_rng[0])+', LABEL_LEFT= , WIDTH=10, TAG=fluxmin',$
                 '2, FLOAT, '+string(self.flux_rng[1])+', LABEL_LEFT=-, WIDTH=10, TAG=fluxmax',$
                '1, BASE, ,ROW, FRAME',$
                 '0, LABEL, Flux diff..', $
                 '0, FLOAT, '+string(self.fdiff_rng[0])+',LABEL_LEFT= , WIDTH=10, TAG=diffmin',$
                 '2, FLOAT, '+string(self.fdiff_rng[1])+',LABEL_LEFT=-, WIDTH=10, TAG=diffmax',$
                '1, BASE, ,ROW, FRAME',$
                 '0, LABEL, Fringe Amp.', $
                 '0, FLOAT, '+string(self.ampl_rng[0])+', LABEL_LEFT= , WIDTH=10, TAG=ampmin',$
                 '2, FLOAT, '+string(self.ampl_rng[1])+', LABEL_LEFT=-, WIDTH=10, TAG=ampmax',$
                '1, BASE, ,ROW',$
                '0, BUTTON, |        Apply       |, TAG=APPLY',$
                '0, BUTTON, |       Dismiss      |, QUIT, TAG=QUIT' ], $
                /Column)

  WIDGET_CONTROL, top, /REALIZE
  WIDGET_CONTROL, top, SET_UVALUE= { obj: self, form_id: fid }

  XMANAGER, 'topdrawparams',top,/NO_BLOCK
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO xmidispvisi_event, ev
  ;print,"event1!"
  WIDGET_CONTROL, ev.TOP, GET_UVALUE=st
  st->event, ev
end

PRO xmidispvisi::event, ev

  ;print,"event2!"
  update = 0
  ;obsolete: draw when our initial event comes
  ;if (TAG_NAMES(ev, /STRUCTURE_NAME) EQ 'WIDGET_TIMER') then update=1
  case (ev.ID) OF
      self.menu: begin
          case (ev.value) of
              "Help.About": begin
                  msg= dialog_message(["MIA+EWS Version 1.5",$
                                       "by Rainer Koehler, Walter Jaffe,",$
                                       "Bill Cotton, Olivier Chesneau and others"],$
                                      /INFORMATION, DIALOG_PARENT=ev.top)
              end
              "Help.Manual":	self->help
              "Plot.Dispersed Power":	v= self->correlflux(/plot)
              "Plot.Visibility":	v= self->visibility(/visplot)
              "Plot.Correlated Flux":	v= self->correlflux(/cfplot)
              "Plot.Photmetric Flux":	v= self->photometry(/ab,/plot)
              "Plot.Photometry A1":	v= self->photometry(/a1,/plot)
              "Plot.Photometry A2":	v= self->photometry(/a2,/plot)
              "Plot.Photometry A":	v= self->photometry(/at,/plot)
              "Plot.Photometry A1+2":	v= self->photometry(/at,/plot)
              "Plot.Photometry B1":	v= self->photometry(/b1,/plot)
              "Plot.Photometry B2":	v= self->photometry(/b2,/plot)
              "Plot.Photometry B":	v= self->photometry(/bt,/plot)
              "Plot.Photometry B1+2":	v= self->photometry(/bt,/plot)
              "Plot.Almost Everything":	begin
                  window,0
                  !P.multi = [0,0,2]
                  v= self->visibility()
                  plot, v[0,*],v[2,*], ystyle=2, xtitle="Wavelength [micron]",$
                    title= self.objectname, ytitle="Visibiltity"
                  cf= self->correlflux()
                  ph= self->photometry(/ab)
                  mix=min(ph[0,*])
                  sx= max(ph[0,*])-mix
                  my= max([cf[2,*],ph[2,*]])
                  plot, ph[0,*], ph[2,*], ystyle=2, yrange=[0,my],$
                    title= self.objectname, xtitle="Wavelength [micron]", ytitle="Flux"
                  oplot,cf[0,*], cf[2,*], linestyl=1
                  oplot, mix+[0.75,0.8]*sx, [0.99,0.99]*my
                  oplot, mix+[0.75,0.8]*sx, [0.91,0.91]*my,linesty=1
                  XYouts,mix+ 0.81*sx, 0.98*my,"photometric flux"
                  XYouts,mix+ 0.81*sx, 0.90*my,"correlated flux"
                  !P.multi = [0,0,0]
              end
              "Movies.Fringes": 	xmidimovie, *self.opd, total(*self.fringe,1)
              "Movies.Power Spectra":	xmidimovie, findgen(self.scanlength), *self.foufri
              "Print.Results": begin
                  fnames= oirFileList(self.fringefile)
                  fname = Dialog_Pickfile(Title="Please give name for Postscript-file",$
                                          File="MIAplots"+MIDIname_datime(fnames[0])+".ps")
                  if (fname ne "") then self->print_results, fname
              end
              "Print.Visibility": begin
                  fname = Dialog_Pickfile(Title="Please enter name for Postscript-file")
                  if fname ne "" then begin
                      set_plot,"PS"
                      device,file=fname,/landscape
                      v= self->visibility(/visplot)
                      device,/close
                      set_plot,"X"
                  endif
              end
              "File.Lambda GUI": self->lambda_gui
              "File.Mask GUI": self->maskgui
              "File.White Fringe Method": self->white_fringe_gui, BASE=ev.top
              "File.Save Visibility":	self->save_visibility
              "File.Hardcopy":	 update = 1
              "File.Write to Log": update = 1
              "File.Save xmdv-object": begin
                  fname= Dialog_Pickfile(Title="Select filename for xmdv-object",$
                                         Default_EXtension="sav", Filter="*.sav", $
                                         /Overwrite_Prompt, /Write)
                  if fname ne "" then begin
                      print,"Writing xmdv-object to ",fname
                      xmdv= self
                      save,xmdv,filename=fname
                  endif
              end
              "File.Quit": begin
                  WIDGET_CONTROL, ev.TOP, /DESTROY
                  self.topdrawID = 0
                  return
              end
              ELSE: print,"unknown menu event: ",ev.value
          end
      end
      ;self.frimov: xmidimovie, *self.opd, total(*self.fringe,1)
      ;self.foumov: xmidimovie, findgen(self.scanlength), *self.foufri
      ;self.print:  update = 1	; force update to recompute visibility
      ;self.wrtlog: update = 1	; force update to recompute visibility
      self.topdraw: $
          if (ev.type eq 0) then self->topdrawparams, ev.top
      self.lftdraw: $
          if (ev.type eq 0) then v=self->visibility(/plot,/visplot)
      self.rgtdraw: begin
          if (ev.type eq 0) then self.button= ev.press
          if (ev.type eq 1) then self.button= 0
          if (ev.type eq 0 or (ev.type eq 2 and self.button ne 0)) then begin
              wset,self.rgtdrawID
              ymid = (get_screen_size())[1]/4-40
              if (ev.y gt ymid) then begin
                  !P.multi = [2,0,2,0,0] & self->plot_fringe_histogram
              endif else begin
                  !P.multi = [1,0,2,0,0] & self->plot_opd_histogram
              endelse
              pos = convert_coord( ev.x, ev.y, /DEVICE, /TO_DATA)
                                ;print, "data pos: ",pos
              ;case self.button of
              ;    1:  self.goodthresh = pos[0]
              ;    else: self.badthresh= pos[0]
              ;endcase
              if (ev.y gt ymid) then self.goodthresh = pos[0] $
                else begin
                  med = median(total(*self.opd,1))/self.scanlength * 1e3
                  self.noiseopd = abs(med-pos[0])/1e3
              endelse
              update = 1
          endif
      end
      self.help: self->help
      self.quit: begin
          WIDGET_CONTROL, ev.TOP, /DESTROY
          self.topdrawID = 0
          return
      end
      else: ;relax
  endcase

  if (update eq 0) then return

  self->update_gui

  if (ev.ID eq self.menu) then begin
      if (ev.value eq "File.Hardcopy") then self->hardcopy

      if (ev.value eq "File.Write to Log") then begin
          self->compute_intvisi, correl_flux,visibility

          if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"
          openw, lun, "PHOTOMETRY/visibilities",/get_lun,/append
          printf,lun, "Object: ",self.objectname
          printf,lun, "Fringefile(s):" & printf,lun,self.fringefile
          printf,lun, "Photfile(s):  " & printf,lun,self.photfiles
          printf,lun, "Maskfile:   ",self.maskfile
          printf,lun, "Noise threshold:",self.badthresh,", Signal threshold:",self.goodthresh
          printf,lun, "Correlated flux "+strtrim(string(correl_flux,format='(G10.3)'),2)
          printf,lun, "VISIBILITY = " +  strtrim(string(visibility,format='(F8.6)'),2)
          printf,lun, "----------------------------------------------------------------------"
          close, lun & free_lun, lun
      endif
  endif
end

pro xmidispvisi::help
  browser = getenv("BROWSER")
  if (browser eq "") then browser = "netscape"
  print,"starting ",browser
  spawn,browser+" file:"+getenv('drsRoot')+"/Manual/mia_gui.html &"
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro xmidispvisi::gui, BLOCK=block, TRUECOLOR=truecolor, CHARSIZE=charsize, base ; CAH

  if keyword_set(charsize) then self.charsize= charsize

  self.color= mia_std_colortable(truecolor)
  screensz = get_screen_size()

  ; Create a base widget containing a draw widget and a sub-base
  base  = Widget_Base(Title="Midivisi "+self.objectname, $
                      Space=0, MBAR=mbar, /COLUMN, /BASE_ALIGN_LEFT)

  self.menu = cw_pdmenu(mbar,$
                        ['1\File', '0\Lambda GUI', '0\Mask GUI', '0\White Fringe Method',$
                         	   '0\Save Visibility', '0\Hardcopy', '0\Write to Log',$
				   '0\Save xmdv-object', '2\Quit', $
                         '1\Plot', '0\Dispersed Power', '0\Visibility', $
                    		   '0\Correlated Flux', '0\Photmetric Flux',$
		               	   '0\Photometry A', '0\Photometry B', $
                         	   '2\Almost Everything', $
                         '1\Movies','0\Fringes', '2\Power Spectra', $
                         '1\Print','0\Results', '2\Visibility', $
                         '1\Help', '0\Manual', '2\About' ],$
                        /COLUMN,/MBAR,/HELP,/RETURN_FULL_NAME)

  ;; not available with EWS-phot:
  ;;                       	   '0\Photometry A1', '0\Photometry A2', '0\Photometry A1+2',$
  ;;                       	   '0\Photometry B1', '0\Photometry B2', '2\Photometry B1+2',$

  filename = oirFileList(self.fringefile)
  fringelbl = filename[0]
  for i=1,N_elements(filename)-1 do $
    fringelbl = fringelbl + " + " + MIDIname_time(filename[i]) + ".fits"

  fringe_l= Widget_Label(base, Value= "Fringefile: "+fringelbl)
  if N_elements(self.photfiles) le 1 then begin
      phot_l= Widget_Label(base,Value="  Photfile: "+self.photfiles[0])
  endif else begin
      phot_l= Widget_Label(base,Value="  Photfile: "+self.photfiles[0] + $
                           " + " + MIDIname_time(self.photfiles[1]) + ".fits")
  endelse
  mask_l  = Widget_Label(base, Value= "  Maskfile: "+self.maskfile)

  self.topdraw= Widget_Draw(base, XSIZE=screensz[0]-16, YSIZE=screensz[1]/2-60,$
                            /BUTTON_EVENT)
  base2 = Widget_Base(base, Space=0, /ROW)
  self.lftdraw= Widget_Draw(base2, XSIZE=(screensz[0]-185)/2, YSIZE=screensz[1]/2-80,$
                            /BUTTON_EVENT)
  bbase = Widget_Base(base2, Space=2, /Column, /Align_Top, /BASE_align_center)
  self.rgtdraw= Widget_Draw(base2, XSIZE=(screensz[0]-185)/2, YSIZE=screensz[1]/2-80,$
                            /BUTTON_EVENT, /MOTION_EVENT)

  objectl= Widget_Label(bbase, Value="Object name: "+self.objectname)
  phfiltl= Widget_Label(bbase, Value="Photometric Filter: "+self.PhFilt)
  frfiltl= Widget_Label(bbase, Value="Fringe Filter: "+self.FrFilt)
  scanlnl= Widget_Label(bbase, Value="Scanlength: "+strtrim(self.scanlength,2))
  scans_l= Widget_Label(bbase, Value=" => "+strtrim(N_elements(*self.amplfou),2)+" scans")
  beaml  = Widget_Label(bbase, Value="Beam is "+self.beam)

  if (total(self.specA2) gt 0.) then begin
      fluxl= Widget_Label(bbase, Value=$
                          string( total(self.specA1), Format='("A1: ",G8.3," ph/sec")'))
      fluxl= Widget_Label(bbase, Value=$
                          string( total(self.specA2), Format='("A2: ",G8.3," ph/sec")'))
      fluxl= Widget_Label(bbase, Value=$
                          string( total(self.specB1), Format='("B1: ",G8.3," ph/sec")'))
      fluxl= Widget_Label(bbase, Value=$
                          string( total(self.specB2), Format='("B2: ",G8.3," ph/sec")'))
  endif else begin
      fluxl= Widget_Label(bbase, Value=$
                          string( total(self.specA1), Format='("A: ",G8.3," ph/sec")'))
      fluxl= Widget_Label(bbase, Value=$
                          string( total(self.specA1), Format='("B: ",G8.3," ph/sec")'))
  endelse
  self.goodthl= Widget_Label(bbase, Value="Good threshold: "+strtrim(self.goodthresh,2))
  self.badthl = Widget_Label(bbase, Value="Noise OPD:  "+strtrim(self.noiseopd*1000.,2)+" mm")

  ;self.sigamp = Widget_Label(bbase, Value="Mean signal ampl. ????????")
  ;self.noiamp = Widget_Label(bbase, Value="Mean noise  ampl. ????????")
  ;self.corramp= Widget_Label(bbase, Value="Corrected amplit. ????????")
  self.corrflx= Widget_Label(bbase, Value="Correlated Flux ??????????")
  self.visibi = Widget_Label(bbase, Value="Visibility = 0.12345",/Align_Center)

  ;self.frimov = Widget_Button(bbase, Value="Fringe Movie", XSize=200-24, Ysize=36)
  ;self.foumov = Widget_Button(bbase, Value="Fourier Movie",XSize=200-24, Ysize=36)
  ;self.print  = Widget_Button(bbase, Value="Hardcopy", XSize=150, Ysize=36)
  ;self.wrtlog = Widget_Button(bbase, Value="Write to log", XSize=150, Ysize=36)
  self.help   = Widget_Button(bbase, Value="Help", XSize=150, Ysize=36)
  self.quit   = Widget_Button(bbase, Value="Quit", XSize=150, Ysize=36)

  WIDGET_CONTROL, base, /REALIZE
  WIDGET_CONTROL, self.topdraw, GET_VALUE=ID & self.topdrawID= ID
  WIDGET_CONTROL, self.lftdraw, GET_VALUE=ID & self.lftdrawID= ID
  WIDGET_CONTROL, self.rgtdraw, GET_VALUE=ID & self.rgtdrawID= ID

  ;; now draw the static stuff

  wset, self.topdrawID
  !p.multi= [0, 0, 4, 0, 0]
  save_charsize = !p.charsize
  !p.charsize = self.charsize
  self->plot_static
  !p.charsize= save_charsize

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  WIDGET_CONTROL, base, SET_UVALUE=self

  ;obsolete: trigger some event to draw dynamic stuff
  ;WIDGET_CONTROL, self.topdraw, TIMER=0.0
  ;device,decomposed=0
  ;done by mia_std_colortable: if (self.color[0] < 100) then tek_color

  ;print,"updating gui"
  self->update_gui

  ;print,"starting xmanager"
  if keyword_set(block) then begin
      XMANAGER, 'xmidispvisi', base
  endif else begin
      XMANAGER, 'xmidispvisi', base, /NO_BLOCK
  endelse
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
