;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;midiUtilities
;#description#
;general, mostly undocumented midi utilities
;#end_class#
;*******************************************************

PRO tvsclm,a,b,c,d
if (N_PARAMS() LT 2) then b=1
s=size(a)
if (N_PARAMS() LT 3) then tvscl,rebin(a,b*s[1], b*s[2]) $
else tvscl,rebin(a,b*s[1],b*s[2]),c,d
return
end

PRO tvm,a,b,c,d
if (N_PARAMS() LT 2) then b=1
s=size(a)
if (N_PARAMS() LT 3) then tv,rebin(a,b*s[1], b*s[2]) $
else tv,rebin(a,b*s[1],b*s[2]),c,d
return
end

FUNCTION gf,last,silent=silent,dir=dir
if (NOT KEYWORD_SET(dir)) then dir=''
z=findfile(dir+'M*.fits')
if (NOT KEYWORD_SET(silent)) then print,N_ELEMENTS(z),' files'
if (N_ELEMENTS(last) GE 0) then last=z[N_ELEMENTS(z)-1]
return,z
end

FUNCTION gfn,last,silent=silent,dir=dir
if (NOT KEYWORD_SET(dir)) then dir=''
z=findfile(dir+'M*.fits')
nz = N_ELEMENTS(z)
ok = bytarr(nz)
for i=0,nz-1 do begin
   a = obj_new('fitsfile',z[i])
   ok[i] = N_ELEMENTS(a->listextensions()) GT 2
   a->close
   obj_destroy,a
endfor
if (total(ok) GT 0) then zz = z(where(ok)) else zz=''
if (NOT KEYWORD_SET(silent)) then print,N_ELEMENTS(zz),' files'
if (N_ELEMENTS(last) GE 0) then last=zz[N_ELEMENTS(zz)-1]
RETURN,zz
END


FUNCTION gfu,last,silent=silent,dir=dir
if (NOT KEYWORD_SET(dir)) then dir='.'
spawn,'find '+dir+' -name "M*.fits" -size +1000 -print >fitslist'
openr,unit,'fitslist',/get_lun
z=''
ll=''
readf,unit,z
while(not eof(unit)) do begin readf,unit,ll&z=[z,ll]&endwhile
close,1
if (NOT KEYWORD_SET(silent)) then print,N_ELEMENTS(z),' files'
if (N_ELEMENTS(last) GE 0) then last=z[N_ELEMENTS(z)-1]
RETURN,z
END

FUNCTION getfits, firstfile, nfile
f = gf(/silent)
t = size(firstfile,/type)
if (t eq 7) then begin
   first = where(strpos(f, firstfile) GE 0)
   if (max(first) eq -1) then begin
      print, 'No files found containing '+firstfile
      return,''
   endif else if (N_ELEMENTS(first) GT 1) then begin
      print, 'More than one files contain '+firstfile
      print, f(first)
      return,''
   endif
   first = first[0]
endif else first = firstfile
return, f(first:first+nfile-1)
end

PRO meancube, firstfile, nfile, d1, d2, d3, d4, d5, d6, silent=silent
f = getfits(firstfile, nfile)
if (f[0] eq '') then return
nfiles = N_ELEMENTS(f)
idata = obj_new('imagedata', f[0])
data = idata->readrows()
dcol = idata->datacolumns()-1
ndata = n_elements(dcol)
nrows = n_elements(data)
s = size(data.(dcol[0])) & d1 = fltarr(s[1], s[2], nfiles)
if (ndata gt 1) then s = size(data.(dcol[1])) & d2 = fltarr(s[1], s[2], nfiles)
if (ndata gt 2) then s = size(data.(dcol[2])) & d3 = fltarr(s[1], s[2], nfiles)
if (ndata gt 3) then s = size(data.(dcol[3])) & d4 = fltarr(s[1], s[2], nfiles)
if (ndata gt 4) then s = size(data.(dcol[4])) & d5 = fltarr(s[1], s[2], nfiles)
if (ndata gt 5) then s = size(data.(dcol[5])) & d6 = fltarr(s[1], s[2], nfiles)
idata->close
obj_destroy,idata
for ifile =0, nfiles-1 do begin
   if(not KEYWORD_SET(silent)) then print,ifile,' ',f[ifile]
   data = oirgetdata(f[ifile])
   d1(*,*,ifile) = total(float(data.data1),3)/n_elements(data)
   if (ndata gt 1) then d2(*,*,ifile) = total(float(data.data2),3)/n_elements(data)
   if (ndata gt 2) then d3(*,*,ifile) = total(float(data.data3),3)/n_elements(data)
   if (ndata gt 3) then d4(*,*,ifile) = total(float(data.data4),3)/n_elements(data)
   if (ndata gt 4) then d5(*,*,ifile) = total(float(data.data5),3)/n_elements(data)
   if (ndata gt 5) then d6(*,*,ifile) = total(float(data.data6),3)/n_elements(data)
endfor
return
end

pro readchop,f1,np, a1, a2, a1sum, a2sum
f=gf()
a1=fltarr(62,69,np)
a2=a1
j=0
for i=f1,f1+np-1 do begin
   a=oirgetmeanrms(f[i])
   a1(*,*,j)=a[0].data1
   a2(*,*,j)=a[0].data2
   j=j+1
endfor
for j=0, np-1, 2 do begin
   a1(*,*,j)=-a1(*,*,j)
   a2(*,*,j)=-a2(*,*,j)
endfor

a1sum = total(a1,3)/np
a2sum = total(a2,3)/np

return
end

FUNCTION unstripe,image,dy
s=size(image)
z=total(image(*,0:dy-1),2)+total(image(*,(s[2]-dy):*),2)
z=.5*z/dy
for i=0,s[2]-1 do image(*,i)=image(*,i)-z
return,image
end

pro getcube,fs,nc
f=gf()
return
end

pro makenewMask, oldmask, newmask, x1, y1, x2, y2, dx
maskdata = oirgetdata(oldmask)
det = obj_new('imagedetector',oldmask)&ndata = det->getpar('NREGION')
dettable=det->readrows()
corner=dettable.corner
det->close
obj_destroy,det
maskdata.data1[*] = 0.
maskdata.data2[*] = 0.
a = fix([x1,y1]+.5) - corner(*,0)
print,a
maskdata.data1[a[0]-dx:a[0]+dx, a[1]-dx:a[1]+dx] = 1.
a = fix([x2,y2]+.5) - corner(*,1)
print,a
maskdata.data2[a[0]-dx:a[0]+dx, a[1]-dx:a[1]+dx] = 1.
oirNewData, oldmask, newmask, maskdata
return
end

FUNCTION rms,a
z=mean(a)
return,sqrt(mean((a-z)^2))
end

FUNCTION midiairy, x
   ax = ABS(x)
   out = 2. * BESELJ(ax,1)
   wx = WHERE(ax EQ 0.)
   IF (MAX(wx) LT 0) then RETURN, out/ax
   ax[wx] = 1.
   out = out/ax
   out[wx] = 1.
   RETURN,out
END

FUNCTION dcombine, data, dx, dy, rel2
if (N_PARAMS() LT 4) then rel2 = 1.
r1 = float(data.data1)+32768.
r2 = float(data.data2)+32768.
nd = n_elements(data)
t1 = total(r1,3)/nd
t2 = total(r2,3)/nd
for i=0, nd-1 do r1(*,*,i)=r1(*,*,i)-t1
print,'finished r1'
for i=0, nd-1 do r2(*,*,i)=r2(*,*,i)-t2
print,'finished r2'
for i=0, nd-1 do r1(*,*,i)=r1(*,*,i)-rel2*shift(r2(*,*,i), dx, dy)
return,r1
end

FUNCTION cubesmooth, data, dx
s=size(data)
output=data
for i=0,s[3]-1 do begin
   z = reform(data(*,*,i))
   zz = ffsmooth(z, dx)
   output(*,*,i)= zz
endfor
return,output
end

FUNCTION cubesharp, data, dx
s=size(data)
output=data
for i=0,s[1]-1 do for j=0,s[2]-1 do output(i,j,*)= $
   data(i,j,*)-ffsmooth1(reform(data(i,j,*)),dx)
return,output
end

FUNCTION filterscans, opd, fringe, scansize, fftobj, srms, trms
nscan = n_elements(opd)/scansize
lopd = reform(opd, scansize, nscan)
lfringe=reform(fringe,scansize,nscan)
fopd = fftobj->coord()
output=fltarr(n_elements(fopd), nscan)
for i=0, nscan-1 do output(*,i)=fftobj->filter(lopd(*,i), lfringe(*,i))
if (total(finite(output) eq 0) gt 0) then output(where(finite(output) eq 0)) = 0
srms = fltarr(nscan)
for i=0,nscan-2 do srms(i)=max(abs(output(*,i)))
srms(nscan-2) = median(srms)
trms = mean(srms)
srms=srms/trms
print,'max of scans over mean ',max(srms)
return,output
end

FUNCTION filterscans2, opd, fringe, scansize, fftobj, srms, trms
nscan = n_elements(opd)/scansize
lopd = reform(opd, scansize, nscan)
lfringe=reform(fringe,scansize,nscan)
fopd = fftobj->coord()
output=fltarr(n_elements(fopd), nscan)
for i=0, nscan-1 do output(*,i)=abs(fftobj->cfilter(lopd(*,i), lfringe(*,i)))
srms = fltarr(nscan)
if (total(finite(output) eq 0) gt 0) then output(where(finite(output) eq 0)) = 0
for i=0,nscan-2 do srms(i)=max(abs(output(*,i)))
srms(nscan-2) = median(srms)
trms = mean(srms)
srms=srms/trms
print,'max of scans over mean ',max(srms)
return,output
end

FUNCTION filterpower, opd, fringe, scansize, fftobj, power
nscan = n_elements(opd)/scansize
lopd = reform(opd, scansize, nscan)
wt = replicate(1., scansize)
lfringe=reform(fringe,scansize,nscan)
fopd = fftobj->coord()
output=fltarr(n_elements(fopd), nscan)
power = output
for i=0, nscan-1 do begin
   output(*,i)=fftobj->filter(lopd(*,i), lfringe(*,i),  wt, fftpower)
   power(*,i) = abs(fftpower)^2
endfor
if (total(finite(output) eq 0) gt 0)then power(where(finite(power) eq 0)) = 0
return,output
end

FUNCTION cuberms,data
s=size(data)
mdata = total(data,3)/s[3]
rdata=mdata
for i=0,s[3]-1 do rdata=rdata+(data(*,*,i)-mdata)^2
return, sqrt(rdata/s[3])
end

FUNCTION makefftobject, minopd, maxopd, npt, filtername
ff = obj_new('fft1d', [minopd, (maxopd-minopd)/npt, npt])
kk=ff->kgrid()
filter=obj_new('filterdata', filtername)
fdata=filter->interpolate(kk)
fdata=fdata+reverse(fdata)
ff->set,filter=fdata
obj_destroy,filter
return,ff
end

FUNCTION makefftobject2, minopd, maxopd, npt, filtername
ff = obj_new('fft1d', [minopd, (maxopd-minopd)/npt, npt])
kk=ff->kgrid()
filter=obj_new('filterdata', filtername)
fdata=filter->interpolate(kk)
fdata=fdata
ff->set,filter=fdata
obj_destroy,filter
return,ff
end

pro makemovie, fftobject, scans
s = size(scans)
fopd = 1.e6*fftobject->coord()
yr=[min(scans), max(scans)]
for i=0,s[2]-1 do begin plot,fopd, scans(*,i),yr=yr&wait,.02&endfor
return
end

FUNCTION crosscube, data
   r1 = float(data.data1)
   r2 = float(data.data2)
   nd = n_elements(data)
   t1 = total(r1,3)/nd
   t2 = total(r2,3)/nd
   for i=0, nd-1 do r1(*,*,i)=r1(*,*,i)-t1
   print,'finished r1'
   for i=0, nd-1 do r2(*,*,i)=r2(*,*,i)-t2
   print,'finished r2'
   cc=fltarr(7,7)
   for i=-3,3 do for j=-3,3 do cc(i+3,j+3)=total(r1*shift(r2,i,j,0))
   cc=cc/max(cc)
   print,0,-3+indgen(7),format='(i2,2x,7f7.3)'
   for j=-3,3 do print,j,cc(*,j+3),format='(i2,2x,7f7.3)'
return,cc
end

FUNCTION midiExtractFiles, multiFile
;   breakup a concatenated list of input files
RETURN,STRSPLIT(multiFile,' ',/extract)
END

FUNCTION midiFirstFile, multiFile
;   return first file from a concatened list
   f = midiExtractFiles(multiFile)
RETURN,f[0]
END

FUNCTION midiConcatFiles, fileList
;   concatenate a bunch of files, separated by a single space
   out = STRTRIM(fileList[0])
   for i=1,N_ELEMENTS(fileList)-1 do out=out+' '+STRTRIM(fileList[i],2)
RETURN,out
END

FUNCTION midiGui, fileObj, dir=dir, search=search, save=save, restore=restore,$
   nonrecursive=nonrecursive,fdir=fdir,newkeys=newkeys
;

   IF(NOT KEYWORD_SET(search)) then search = 'MIDI*.fits'
   if (NOT KEYWORD_SET(dir)) then $
      if (KEYWORD_SET(nonrecursive)) then dir = '' $
      else dir = '.' 
   if (NOT KEYWORD_SET(restore)) then begin
      if (dir EQ '' OR (dir EQ '.')) then begin
         cd,current=c
         print,'Reading files in current directory '+c
      endif else print,'Reading files in directory '+dir
;     f = obj_new('fitsfilelist',dir=dir,$
      guikey=['OBS TARG','NRTS MODE','INS GRIS','INS OPT1','INS FILT NAME',$
         'INS SHUT']
      if (KEYWORD_SET(newkeys)) then guikey=[guikey,newkeys]
      f = OBJ_NEW('fitsfilelist', dir=dir, $
      guikey=guikey,guisearch=search,/midiCompress)
;     guikey=['OBS TARG','NRTS MODE','INS GRIS','INS OPT1','INS FILT NAME',$
;        'INS SHUT'],guisearch=search,/midiCompress)
   endif else begin
      saveFile = FINDFILE(restore)
      if (saveFile NE '') then begin
         f = OBJ_NEW('fitsfilelist',restore=saveFile)
         f->guiSelect
      endif else begin
         print,' saveFile '+restore+' not Found'
         f = OBJ_NEW()
      endelse
   endelse
   if (OBJ_VALID(f)) then list =f->files() else begin
      print,'Failed to create legitimate file list from input data '
      RETURN, ''
   endelse
   if (KEYWORD_SET(save)) then begin
      f->save, save,/select
   endif
   OBJ_DESTROY,f
RETURN,list
END

FUNCTION midiGuiS, dir=dir, nonrecursive=nonrecursive, clear=clear
;
   search='MIDI*.fits'
;   IF (NOT KEYWORD_SET(nonrecursive)) then search='*/'+search
;  IF (KEYWORD_SET(dir)) then search = dir+'/'+search $
;     else search = './'+search
   IF (NOT KEYWORD_SET(dir)) then  $
      if (KEYWORD_SET(nonrecursive)) then dir = ' '$
      else dir='.'
   if (dir EQ ' ' OR (dir EQ '.')) then begin
      cd,current=c
      print,'Reading files in current directory '+c
      saveFileName = 'midiGui.SAV'
   endif else begin
      print,'Reading files in directory '+dir
      saveFileName = dir+'/midiGui.SAV'
   endelse
   if (KEYWORD_SET(clear)) then begin
      OPENR,lun, saveFileName,/DELETE,/GET_LUN, ERROR=ierr
      if (ierr NE 0) then print, !ERROR_STATE.MSG
      if(ierr EQ 0) then  FREE_lun, lun
   endif
   saveFile = FINDFILE(saveFileName)
   if (saveFile NE '') then begin
      f = OBJ_NEW('fitsfilelist',restore=saveFile)
      f->guiSelect
   endif else begin
      f = obj_new('fitsfilelist',dir=dir,$
;     f = obj_new('fitsfilelist',$
      guikey=['OBS TARG','NRTS MODE','INS GRIS','INS OPT1','INS FILT NAME',$
         'INS SHUT','CHOP FREQ'],guisearch=search,/midiCompress)
   endelse
   if (OBJ_VALID(f)) then list =f->files() else begin
      print,'Failed to create legitimate file list from input data '
      list = ''
      RETURN,list
   endelse
   if (saveFile EQ '') then f->save, saveFileName
   OBJ_DESTROY,f
RETURN,list
END

FUNCTION midiFiles, last, dir=dir
;
   IF (NOT KEYWORD_SET(dir)) then dir='.'
   f = obj_new('fitsfilelist',dir=dir,$
      search='MIDI*.fits',/midiCompress)
   list =f->files()
   last = list[N_ELEMENTS(list) -1]
   obj_destroy,f
RETURN,list
END

PRO doscan,infile,maskfile,filterfile,opd,raw,fopd,filtered
print,'read data from ',infile
data = oircompressdata(infile,maskfile,opd=opd)
raw=data.data1-data.data2
raw=raw-fsmooth1(raw,50)
raw=reform(raw,48,100)
opd=reform(opd,48,100)
opd(47,*)=opd(45,*)
opd(46,*)=opd(45,*)
nfreq=1024.
print,'create filter object from ',filterfile
filter=obj_new('filterdata',filterfile)
fftobj=obj_new('fft1d',[-2.e-3,4.e-3/nfreq,nfreq])
fftobj->set,filter=filter->interpolate(fftobj->kgrid())
fopd=fftobj->coord()
filtered=fltarr(nfreq,100)
print,'filter data'
for i=0,99 do filtered(*,i)=abs(fftobj->filter(opd(*,i),raw(*,i)))
obj_destroy,filter
obj_destroy,fftobj
return
end

PRO ph,a,b
if (N_PARAMS() LT 2) then b = 100
if (a NE 0) AND (0 EQ (a MOD b)) then PRINT,A,format='(i,$)'
RETURN
END

pro olines,lines,labels,ylab,th
; on top of existing plot make vertical stripes at
; the positions of given spectral lines
; shift each wavelength before plotting by z
; default line thickness
   if (n_params() eq 4) then t=th else t = 1
;  loop over list of lines
   if (n_params() eq 2) then ylab = .8*!y.crange(1)
   xr=!x.crange
   if (!x.type eq 1) then xr=10.^xr
   n=n_elements(lines)-1
   for i=0,n do begin
      zl = lines[i]
;  only over plot lines that fall within wavelength range of plot
;  the x-coordinate could be logarithmic
      if (zl gt xr(0)) and (zl lt xr(1)) then begin
         if (t gt 0) then oplot,[zl,zl],!y.crange,thick=t
;  if a list of line names have been given, write these on plot
      if n_params() gt 2 then xyouts,zl,ylab,labels(i),orient=90
      endif
   endfor
return
end

FUNCTION pex,d,x
;  if d is list of polynomial coefficients, expand
;  polynomial at each value of vector x and return vector of values
p = n_elements(d)
w = d(p-1)
for i=p-2,0,-1 do w=w*x+d(i)
return,w
end

FUNCTION pex2,d,x,y
p = size(d)
cox=fltarr(p(1))
for i=0,p(1) -1 do cox(i) = pex(d(i,*),y)
return, pex(cox, x)
end

FUNCTION medcont, data
; return median of a vertical stack of images
ss=size(data)
out=fltarr(ss(1),ss(2))
for i=0,ss(1)-1 do for j=0,ss(2)-1 do out(i,j)=median(data(i,j,*))
return,out
end

FUNCTION ps1,a,dx
;shift an 1-d array a by dx (fractional) pixels
pi2 = 2.*!pi
f = fft(a,1)
ss = size(a)
nx = ss(1)
sx = exp(complex (0, pi2 * dx * (indgen(nx) - nx/2)/nx))
sx = shift (sx, -nx/2)
f=f*sx
return, float(fft(f,-1))
end


FUNCTION bblam,temp,wave
;  return normalized bb curve in energy/area/sec/wavelength unit
;  given wavelengths in microns, temp in K and n is normalizing point
z = wave^(-5)/(exp(14350./(temp*wave))-1.)
n=fix(.5*n_elements(wave))
return, z/z(n)
end

FUNCTION bbJy,temp,wave
;  return normalized bb curve in energy/area/sec/Hz unit
;  given wavelengths in microns, temp in K and n is normalizing point
z = wave^(-3)/(exp(14350./(temp*wave))-1.)
n=fix(.5*n_elements(wave))
return, z/z(n)
end

FUNCTION pk,x,y,z
; using quadratic approximation of y(x) return
; position (x value) of peak of y to subpixel accuracy
; optionally return interpolated y value in z
   q=max(y,j)
   if (j eq 0) then j = 1
   if (j eq n_elements(x)-1) then j=n_elements(x)-2
   dj=(y(j-1)-y(j+1))*.5/(y(j+1)+y(j-1)-2.*y(j))
   z = y(j)+.25*dj*(y(j+1)-y(j-1))
return,x(j)+.5*dj*(x(j+1)-x(j-1))
end

FUNCTION pshift,a,dx,dy
;two dimensional partial pixel shift
pi2 = 2.*!pi
f = fft(a,1)
ss = size(a)
nx = ss(1)
ny = ss(2)
sx = exp(complex (0, pi2 * dx * (indgen(nx) - nx/2)/nx))
sx = shift (sx, -nx/2)
for i = 0, ny - 1 do f(*,i) = f(*,i) * sx
sy = exp(complex (0, pi2 * dy * (indgen(ny)-ny/2)/ny))
sy = shift (sy, -ny/2)
for i = 0, nx - 1 do f(i,*) = f(i,*) * sy
return, float(fft(f,-1))
end
FUNCTION ps1,a,dx
pi2 = 2.*!pi
f = fft(a,1)
ss = size(a)
nx = ss(1)
sx = exp(complex (0, pi2 * dx * (indgen(nx) - nx/2)/nx))
sx = shift (sx, -nx/2)
f=f*sx
return, float(fft(f,-1))
end
FUNCTION cross,a,b
return,float(fft(fft(a,1)*fft(b,-1),-1))
end
FUNCTION makex,nx,ny
mask = intarr(nx, ny)
x = indgen(nx)
for i=0,ny-1 do mask(*, i) = x
return,mask
end
FUNCTION makey,nx,ny
mask = intarr(nx, ny)
x = indgen(ny)
for i=0,nx-1 do mask(i, *) = x
return,mask
end


FUNCTION gau,nx,ny,sx,sy
g = (shift((makex(nx,ny)-.5*nx)/sx,nx/2,0))^2
g = .5*(g + (shift((makey(nx,ny)-.5*ny)/sy,0,ny/2))^2)
if (total(g gt 30.) gt 0) then g(where(g gt 30.)) = 30.
return ,exp(-g)
end

FUNCTION gau1,nx,sx
if (sx lt .001) then sx = .001
g = .5*(shift((indgen(nx)-.5*nx)/sx,nx/2))^2
if (total(g gt 30.) gt 0) then g(where(g gt 30.)) = 30.
return ,exp(-g)
end

FUNCTION fgau,nx,ny,sx,sy
f=fft(gau(nx,ny,sx,sy),1)
return,f/f(0,0)
end
FUNCTION fgau1,nx,sx
f=fft(gau1(nx,sx),1)
return,f/f(0)
end

FUNCTION fsmooth,image,s
; two dimensional fft smoothing of image
ss=size(image)
g=fgau(ss(1),ss(2),s,s)
return,float(fft(g*fft(image,1),-1))
end

FUNCTION fsmooth1,image,s
ss=size(image)
g=fgau1(ss(1),s)
return,float(fft(g*fft(image,1),-1))
end

FUNCTION csmooth1,image,s
ss=size(image)
g=fgau1(ss(1),s)
return,fft(g*fft(image,1),-1)
end

FUNCTION csmooth2,image,s
ss=size(image)
g=fgau1(ss(1),s)
output = image
for i=0,ss[2]-1 do output(*,i) = fft(g*fft(image(*,i),1),-1)
return, output
end


FUNCTION fsmooth2,image,s1,s2
ss=size(image)
g=fgau(ss(1),ss(2),s1,s2)
return,float(fft(g*fft(image,1),-1))
end


PRO sortmask,oldfile,newfile
olddet = obj_new('imagedetector', oldfile)
olddetdata=olddet->readrows()
olddet->close
yorder = sort(olddetdata.corner[1])
print,olddetdata.corner[1]
print,olddetdata[yorder].corner[1]
newdetdata=olddetdata[yorder]
for i=0,n_elements(yorder)-1 do newdetdata[i].region=i+1
oldtable=olddet->table()
*oldtable=newdetdata
olddata = obj_new('imagedata',oldfile)
olddatacols = olddata->datacolumns()-1
oldmask = olddata->readrows()
obj_destroy,olddata
newtemp=olddet->datatemplate(/float)

olddet->newfile, newfile
newdata = obj_new('imagedata', olddet,/float)
newdatacols=newdata->datacolumns()-1
for i=0,n_elements(yorder)-1 do newtemp.(newdatacols[i])= $
   oldmask.(olddatacols[yorder[i]])
newdata->appendtofile, olddet
newdata->writerows,newtemp
newdata->close
obj_destroy,newdata
RETURN
END


PRO sortwindow,oldfile,newfile
olddet = obj_new('imagedetector', oldfile)
olddetdata=olddet->readrows()
olddet->close
yorder = sort(olddetdata.corner[1])
print,olddetdata.corner[1]
print,olddetdata[yorder].corner[1]
newdetdata=olddetdata[yorder]
for i=0,n_elements(yorder)-1 do newdetdata[i].region=i+1
oldtable=olddet->table()
*oldtable=newdetdata
olddata = obj_new('imagedata',oldfile)
olddatacols = olddata->datacolumns()-1
obj_destroy,olddata
olddet->newfile, newfile
newdata = obj_new('imagedata', olddet,/int)
newdata->appendtofile, olddet
newdata->close
obj_destroy,newdata
RETURN
END

PRO listmask,filelist,outfile
   nFile = N_ELEMENTS(filelist)
   openw,unit,outfile,/get_lun
   for iFile = 1, nFile do begin
      name=filelist[iFile-1]
      det=obj_new('imagedetector', name)
      detdata=det->readrows()
      nRows = N_ELEMENTS(detdata)
      printf,unit,name+':'
      for iRow=0, nRows-1 do begin
         region=strtrim(detdata[iRow].region,2)
         d=detdata[iRow]
         outs='DET.WIN'+region+'.STARTX '
         outs=outs+strtrim(d.corner[0],2)+' DET.WIN'+region+'.STARTY '
         outs=outs+strtrim(d.corner[1],2)+' DET.WIN'+region+'.NX '
         outs=outs+strtrim(d.naxis[0],2)+' DET.WIN'+region+'.NY '
         outs=outs+strtrim(d.naxis[1],2)+' (Corr: '+strtrim(d.correlation,2)+')'
         printf,unit,outs
      endfor
      printf,unit,' '
   endfor
   FREE_LUN,unit
RETURN
END

PRO fixwindow,oldfile
   newfile = oldfile+'.tmp'
   olddet = obj_new('imagedetector', oldfile)
   olddetdata=olddet->readrows()
   olddet->close
   nWindows = N_ELEMENTS(olddetdata)
   newdetdata=olddetdata
   xx = indgen(4)
   for i=0,nWindows-1 do newdetdata[i].region=i+1
   print,' found ',nWindows,' windows'
SETVALUES:
   for i=0, nWindows-1 do begin
      print,i+1,'X0','Y0','NX','NY',format='(i2,2x,4(a,4x))'
      print,olddetdata[i].corner, olddetdata[i].naxis, $
         format='(4i6)'
      read,xx
      newdetdata[i].corner= xx[0:1]
      newdetdata[i].naxis = xx[2:3]
      if (newdetdata[i].naxis[0] NE olddetdata[i].naxis[0]) $
         then print,' you changed NX, this may invalidate WCS data'
   endfor
   print,'NEW VALUES'
   for i=0, nWindows-1 do begin
      print,i+1,'X0','Y0','NX','NY',format='(i2,2x,4(a,4x))'
      print,newdetdata[i].corner, newdetdata[i].naxis, $
         format='(4i6)'
   endfor
   print,'TO REPEAT TYPE "r": TO ACCEPT TYPE "y" else I quit'
   ll = ''
   read,ll
   if (STRUPCASE(ll) EQ 'R') then GOTO,SETVALUES
   if (STRUPCASE(ll) EQ 'Y') then begin
      oldtable=olddet->table()
      *oldtable=newdetdata
      olddet->newfile, newfile
      newdata = obj_new('imagedata', olddet,/int)
      newdata->appendtofile, olddet
      newdata->close
      obj_destroy,newdata
      spawn,'mv '+oldfile+' '+oldfile+'~'
      spawn,'mv '+newfile+' '+oldfile
      print,'renamed ',oldfile,' ',oldfile+'~'
   endif
   obj_destroy,olddet
RETURN
END

PRO midiGetINS, file, dit=dit, grism=grism, slit=slit, opt1=opt1, $
   camera=camera, filter=filter, nrts=nrts, object=object, shutter=shutter, $
   ndit = ndit, ierr = ierr
;
   f = file[0]
   f = STRSPLIT(f, ' ',/EXTRACT)
   f = f[0]
   ierr = 0
   fits = OBJ_NEW('fitsfile', f, ierr=ierr)
   if (ierr NE 0) then begin
      midiPrintErrMsg
      RETURN
   endif
   ph   = fits->head()
   dit  = ph->getpar('DET DIT')
   ndit  = ph->getpar('DET NDIT')
   grism = ph->getpar('INS GRIS NAME')
   slit = ph->getpar('INS SLIT NAME')
   opt1 = ph->getpar('INS OPT1 NAME')
   camera = ph->getpar('INS CAM NAME')
   filter = ph->getpar('INS FILT NAME')
   shutter = ph->getpar('INS SHUT NAME')
   nrts = ph->getpar('DET NRTS MODE')
   ob   = ph->getpar('TARG NAME',nob)
   if (nob GT 0) then object=ob else object=''
   OBJ_DESTROY, ph
   OBJ_DESTROY, fits
RETURN
END

FUNCTION midiPhase, x
   RETURN,ATAN(IMAGINARY(x), FLOAT(x))
END
FUNCTION midiPhased,x
   RETURN,!radeg*midiPhase(x)
END

PRO midiDelayPlot, base, title
   delay = oirgetdelay(base+'.groupdelay.fits')
   delaytime=REFORM(delay.time)&delaytime=86400*(delaytime-delaytime[0])
   opd = oirgetopd(base+'.fringes.fits',opdtime)
   delay = reform(delay.delay)
   if (TOTAL(opd EQ 0.) GT 0 ) then begin
      delay(where(opd EQ 0.0)) = median(reform(delay)); bad points
      opd(where(opd EQ 0.0)) = median(opd)  ; bad points
   endif
   ndelay=N_ELEMENTS(opd)
   if (ndelay gt 4000) then $
   plot,opdtime, opd,yrange=median(opd[ndelay-3000:*])+$
      6*rms(opd[ndelay-3000:*])*[-1,1],title=base+':'+title, $
      xtitle='time (sec)',ytitle='Delay(microns)' $
   else plot,opdtime, opd,yrange=[min([opd,delay]),max([opd,delay])],title=base+':'+title, $
      xtitle='time (sec)',ytitle='Delay(microns)'
   oplot,delaytime,delay
RETURN
END


PRO midiDelayAmpPlot, base, title, gsmooth=gsmooth
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth=3
   delay = oirgetdelay(base+'.groupdelay.fits')
   amp   = MEDIAN(REFORM(delay.amplitude), gsmooth)
   delay = REFORM(delay.delay)
   aSort = sort(amp[10:*])
   nSort = N_ELEMENTS(aSort)
   aMed = amp[10+aSort[.5*nSort]]
   aMax = amp[10+aSort[nSort-1]]
   aSig = amp[10+aSort[.8413*nSort]]
   opd = oirgetdata(base+'.fringes.fits',col=['opd','localopd'])
   opd=1.e6*reform((opd.opd[1]+opd.localopd[1]-opd.opd[0]-opd.localopd[0]))
   if (TOTAL(opd EQ 0.) GT 0 ) then begin
      delay(where(opd EQ 0.0)) = median(delay); bad points
   endif
   nD = N_ELEMENTS(delay)
;plot,delay[300:*], abs(amp[300:*]), title=base+':'+title, $
   plot,delay[300:nD-2], abs(amp[300:nD-2]), title=base+':'+title, $
      xtitle='delay (micron)',ytitle='Amplitude' ,/ylog, psym=7,charsize=2,ystyle=1
   oplot,1.e9*[-1,1],aMed*[1,1],thick=2
   for i=1,4 do oplot,1.e9*[-1,1],aMed+(aSig-aMed)*i*[1,1],line=2
;xyouts,delay[300+aSort[nSort-1]], .95*aMax, $
   xyouts,delay[10+aSort[nSort-1]], .95*aMax, $
      STRING((aMax-aMed)/(aSig-aMed),format='(f4.1)')
RETURN
END


PRO midiDelayAmpPlot1, base, title, gsmooth=gsmooth
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 30
   delay = oirgetdelay(base+'.groupdelay.fits')
   nData = N_ELEMENTS(delay)
   nd = floor(nData-300)/500
   nd = 500*nd
   delay = REFORM(delay.delay)
   delay = delay[300:300+nd-1]
   delay = median(delay,2*gsmooth)
   gdata = oirgetdata(base+'.groupdelay.fits')
   gdata = pseudoComplex(gdata.data1)
   gdata = gdata[300:300+nd-1,*]
   gdata = csmooth2(gdata,gsmooth)
   gamp = fltarr(nd)
   for i=0, nd-1 do gamp[i] = MAX(ABS(gdata[i,*]))
   erase
   !p.multi = [0,1,3,0,0]
   oldx = !x.style
   !x.style=1
   plot,delay ,charsize=2
   plot,gamp,charsize=2
   midiDelayAmpPlot, base,' ', gsmooth=gsmooth
   !x.style=oldx
   !p.multi = 0
   pkDelay = pk(delay[50:nd-50], gamp[50:nd-50], gpk)
   ramp = rms(gamp)
   rmean= mean(gamp)
   print, gpk-rmean,ramp,(gpk-rmean)/ramp,rmean,pkDelay,$
   format='("Max amplitude of",f7.0,"+/-",f6.0,"(",f3.1," sig) above",f8.0," at delay ",f8.2)'
RETURN
END

PRO midiInsVisPlot, base,title
   vis = oirGetVis(base+'.redcal.fits', wave=wave)
   plot,wave,vis.visamp,xr=[7.5,13.5],xstyle=1,title=base+':'+title, $
      yr=[0>MIN(vis.visamp), 1.5<MAX(vis.visamp[WHERE(wave gt 8.1 and (wave lt 13.5))])], $
      xtitle='Wavelength(microns)', ytitle='Visibility'
;  errplot,wave,vis.visamp-vis.visamperr,vis.visamp+vis.visamperr
RETURN
END

PRO midiPlotOverlap, base, title, mask=mask, oldmask=oldmask, fringe=fringe
   if (N_PARAMS() LT 2) then title='OVERLAP'
   pa = oirGetData(base+'.Aphotometry.fits')
   pb = oirGetData(base+'.Bphotometry.fits')
   if (KEYWORD_SET(fringe)) then begin
      pi = oirGetData(base+'.fringeimageave.fits')
      pi = TRANSPOSE(pseudocomplex(pi.data1))
      si = total(pi[50:100,*],1)
      s = abs(si)
   endif
   midiGetIns,base+'.Aphotometry.fits', opt1=opt1
   if (STRTRIM(opt1,2) eq "HIGH_SENS") then begin 
      sa = total(pa[0].data1[50:100,*],1)
      sb = total(pb[0].data1[50:100,*],1)
   endif else if (STRTRIM(opt1,2) eq "SCI_PHOT") then begin
      sa = total(pa[0].data2[50:100,*],1)
      sb = total(pb[0].data2[50:100,*],1)
   endif else print,'OPT1 of '+opt1+' not valid for plotoverlap'
   ns = N_ELEMENTS(sa)
   sa=sa/max(sa[5:ns-5])
   sb=sb/max(sb[5:ns-5])
   plot,sa,title=title,yr=[-.2,1.2],ystyle=1
   oplot,sb,line=2
   if (KEYWORD_SET(fringe)) then oplot,s/max(s[5:ns-5]),thick=2
   if (KEYWORD_SET(mask)) then begin
      m = oirGetData(mask)
      mm=total(m[0].data1[50:100,*],1)
      mm = mm/max(mm)
      oplot,mm,thick=3
   endif
   if (KEYWORD_SET(oldmask)) then begin
      m = oirGetData(oldmask)
      mm=total(m[0].data1[50:100,*],1)
      mm = mm/max(mm)
      oplot,mm,thick=3, line=2
   endif
RETURN
END

PRO midiVisPipe, base, file, mask=mask, smooth=smooth, gsmooth=gsmooth,$
   fast=fast, filter=filter, fudge=fudge, dAve=dAve, minopd=minopd, $
   fringeimage=fringeimage
;    set defaults  for fringe smooth and group delay smooth
   if (NOT KEYWORD_SET(smooth)) then smooth = 50.
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 4.
   if (NOT KEYWORD_SET(gsmooth) AND (KEYWORD_SET(fast))) then gsmooth = .1
   if (KEYWORD_SET(dAve)) then begin
      doAve = " -removeAverage" 
      if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' ' $
         else mo = " 20. "
      endif else begin
         doAve=" -noremoveAverage"
         if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' '$
            else mo = " 0. " 
      endelse
;    where is executable?
   cbin = getenv('vltiCbin')+'/'
;    consider temporary filter variant
   if (KEYWORD_SET(filter)) then disp='dispVis20 "' else disp='dispVis "'
   if (KEYWORD_SET(fast)) then disp='dispVisT "'
   if (KEYWORD_SET(fudge)) then disp='dispVisF "'
;    if files not specified, bring up gui
   if (N_PARAMS() < 2) then d = midiGuis() else d=file
;    parse input files to see if both visibility and photometry present
   midiGetIns,d[0],shutter=shutter, grism=grism, object=object, opt1=opt1,$
      ierr=ierr
   if (ierr NE 0) then return
   shutter = STRTRIM(shutter,2)
   grism = STRTRIM(grism,2)
   if (shutter NE 'ABOPEN') then begin
      print,'I need ABOPEN interferometry data'
      RETURN
   endif
   if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
      print,'I only know how to deal with prism or grism data'
      RETURN
   endif

   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
   print,' input vis files have ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(d)))),' input frames'
;   if (NOT KEYWORD_SET(fast)) then $
;  command = cbin+'dispVis "'+d+'" '+mask+' '+base+' '+$
   command = cbin+disp+d+'" '+mask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth) + doAve + mo
;   else command = cbin+'dispVisT "'+d+'" '+mask+' '+base+' '+$
;     STRING(smooth)+' '+STRING(gsmooth)
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispVis failed'
      print,command
      RETURN
   endif
   erase
   !p.multi=[0,1,2,0,0]
   midiDelayPlot,base,object+':'+d[0]
   v=oirgetvis(base+'.corr.fits',wave=wave)
   plot,wave,v.visamp
   !p.multi=0
RETURN
END

PRO midiPhotoPipe, base, files, mask=mask, dSky=dSky, curve=curve
   if (NOT KEYWORD_SET(dSky)) then dSky = 0
   cbin = getenv('vltiCbin')+'/'
   d=files
   midiGetIns,files[0],shutter=shutter, grism=grism, opt1=opt1,ierr=ierr
   if (ierr NE 0) then RETURN
   grism = STRTRIM(grism,2)
   opt1 = STRTRIM(opt1,2)
   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
   fileList = REPLICATE(-1,2)
   for i=0,1 do begin
      midiGetIns,d[i],shutter=shutter, grism=grism
      grism=STRTRIM(grism,2)
      shutter = STRTRIM(shutter,2)
      if (shutter EQ 'AOPEN') then fileList[0]=i $
      else if (shutter EQ 'BOPEN') then fileList[1] = i
      if ((grism NE 'PRISM') AND (grism NE 'GRISM')) then begin
         print,'I only know how to deal with prism or grism data'
         RETURN
      endif
   endfor

   if (total(filelist EQ (-1)) NE 0) then begin
       print,'I need PA+PB to continue'
       RETURN
   endif
;   print,'dSky is ',dsky
   if (KEYWORD_SET(curve)) then ref = ' -ref '+curve else ref = ''
   command = cbin+'dispPhot "'+d[filelist[0]]+'" "'+d[filelist[1]]+'" '+mask+' '+base+' '+STRTRIM(dSky,2)+ref
   spawn,command
   p = oirgetdata(base+'.photometry.fits')
   plot,p[5].data1
RETURN
END

PRO midiPipe, base, inFiles, files=files, mask=mask, smooth=smooth, $
   gsmooth=gsmooth,fast=fast, dSky=dSky, filter=filter, fudge=fudge, $
      dAve=dAve, minopd=minopd, fringeImage=fringeImage, shiftMask=shiftMask
;    set defaults  for fringe smooth and group delay smooth
   if (NOT KEYWORD_SET(smooth)) then smooth = 50.
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 4.
   if (NOT KEYWORD_SET(gsmooth) AND (KEYWORD_SET(fast))) then gsmooth = .1
   if (NOT KEYWORD_SET(dSky)) then dSky = 0
   chopPhotoDone = 0
   if (KEYWORD_SET(dAve)) then begin
      doAve = " -removeAverage" 
      if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' ' $
         else mo = " 20. "
   endif else begin
         doAve=" -noremoveAverage"
         if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' '$
            else mo = " 0. " 
   endelse
;    consider temporary filter variant
   if (KEYWORD_SET(filter)) then disp='dispVis20 "' else disp='dispVis "'
   if (KEYWORD_SET(fudge)) then disp='dispVisF "'
;    where is executable?
   cbin = getenv('vltiCbin')+'/'
;    if files not specified, bring up gui
   if (NOT KEYWORD_SET(files)) then begin
      if(N_PARAMS() eq 2) then d = inFiles else d = midiGuis() 
   endif else d=files
;    parse input files to see if both visibility and photometry present
   if (N_ELEMENTS(d) NE 3) then begin
      print,'You need to specify two visibility file(s) and one' +$
         ' interferometry file.'
      RETURN
   endif
   fileList = REPLICATE(-1,3)
   for i=0,2 do begin
      midiGetIns,d[i],shutter=shutter, grism=grism, object=object, opt1=opt1,$
         ierr=ierr
      if (ierr NE 0) then RETURN
      shutter = STRTRIM(shutter,2)
      grism = STRTRIM(grism,2)
      if (shutter EQ 'AOPEN') then fileList[0]=i $
      else if (shutter EQ 'BOPEN') then fileList[1] = i $
      else if (shutter EQ 'ABOPEN') then fileList[2] = i
      if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
         print,'I only know how to deal with prism or grism data'
         RETURN
      endif
   endfor
   if (TOTAL(filelist eq (-1)) NE 0) then begin
      print,'You need to specify two visibility file(s) and one' +$
         ' interferometry file.'
      RETURN
   endif
;         look for default masks if not specified
   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
;  begin CAH
   if (grism EQ 'PRISM') then curvefile=getenv('prismscurve')
   if (grism EQ 'GRISM') then curvefile=getenv('grismscurve')
;  and CAH
;                consider shifting mask based on photometry
   finalMask = mask
   if (KEYWORD_SET(shiftMask)) then begin
      print,' '
      print,'   COMPUTING NEW MASK'
      command = cbin+'oirChopPhotoImages -in "'+d[fileList[0]]+'" -out '+ $
         base+'.Aphotometry.fits -dSky '+STRTRIM(dSky,2)
;     begin CAH
      command = command + ' -ref '+ curvefile + ' -order 1'
;     end CAH
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'oirChopPhotoImages failed for A channel'
         print,command
         RETURN
      endif
      command = cbin+'oirChopPhotoImages -in "'+d[fileList[1]]+'" -out '+ $
         base+'.Bphotometry.fits -dSky '+STRTRIM(dSky,2)
;     begin CAH
      command = command + ' -ref '+ curvefile + ' -order 1'
;     end CAH
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'oirChopPhotoImages failed for B channel'
         print,command
         RETURN
      endif
      command = cbin+'oirShiftMask -A '+base+'.Aphotometry.fits -B '+ $
         base+'.Bphotometry.fits -out '+base+'.newmask.fits -refMask '+$
         mask
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'oirShiftMask failed'
         print,command
         RETURN
      endif
      finalMask = base+'.newmask.fits'
      midiPlotOverlap, base, object+':'+d[filelist[0]],mask=finalMask,$
         oldmask=mask
      print,'   FINISHED COMPUTING NEW MASK '+finalMask
      chopPhotoDone = 1
   endif
;                process visibility measurement
   if (NOT KEYWORD_SET(fast)) then $
      command = cbin+disp+d[filelist[2]]+'" '+finalMask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth) + doAve +mo $
   else command = cbin+'dispVisT "'+d[filelist[2]]+'" '+finalMask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth)
   print,' input vis files have ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(d[filelist[2]])))),' input frames'
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispVis failed'
      print,command
      RETURN
   endif
   erase
   !p.multi=[0,1,2,0,0]
   midiDelayPlot,base,object+':'+d[filelist[2]]
;                process photometry
   if (chopPhotoDone eq 0) then $
      command = cbin+'dispPhot "'+d[filelist[0]]+'" "'+d[filelist[1]]+'" '+$
      finalMask+' '+ base+' '+STRTRIM(dSky,2) $
   else command = cbin+'dispPhotShort '+base+' '+finalMask
   print,' input photA file has ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(d[filelist[0]])))),' input frames'
   print,' input photB file has ',FIX(TOTAL(midigetkeyword('DET NDIT', $
      midiextractfiles(d[filelist[1]])))),' input frames'
;  begin CAH
   command = command + ' ' + curvefile + ' 1'
;  end CAH
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispPhot failed'
      print,command
      RETURN
   endif
   spawn,cbin+'oirRedCal '+base, EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirRedCal failed'
      print,command
      RETURN
   endif
   midiInsVisPlot,base,object+':'+d[filelist[0]]
   !p.multi = 0
   if (KEYWORD_SET(fringeImage)) then begin
      print,'Computing fringe image '
      command = cbin+'dispCenter "'+d[filelist[2]]+'" '+base
      spawn,command
      midiPlotOverlap, base, object+':'+d[filelist[0]],mask=finalMask,/fringe
   endif
RETURN
END

PRO midiShiftMask, tag, oldMask, newMaskFile
   aFile = tag+'.Aphotometry.fits'
   bFile = tag+'.Bphotometry.fits'
   spawn,getenv('vltiCbin')+'/oirShiftMask -A '+afile+' -B '+bfile+ $
      ' -out '+ newMaskFile+' -refMask '+oldMask
RETURN
END

PRO midiCurveCal, files, outFile
   cbin = getenv('vltiCbin')+'/'
RETURN
END

PRO midiCrossCoeff, base, files, curveFile=curveFile, sky=sky
;    check and set defaults
   cbin = getenv('vltiCbin')+'/'
   if (KEYWORD_SET(sky)) then doSky = ' -sky ' else doSky = ' '
   fileList = INTARR(2)
   for ifile =0,1 do begin 
      midiGetIns,files[ifile],shutter=shutter, grism=grism, object=object, $
         opt1=opt1,ierr=ierr
      if (ierr NE 0) then RETURN
      shutter = STRTRIM(shutter,2)
      grism   = STRTRIM(grism,2)
      opt1    = STRTRIM(opt1,2)
      if (opt1 NE 'SCI_PHOT') then begin
         print,'I can only calculate cross coupling coefficients for SCI_PHOT data'
         RETURN
      end
      if (shutter EQ 'AOPEN') then fileList[0]=ifile $
      else if (shutter EQ 'BOPEN') then fileList[1] = ifile 
      if ((grism NE 'PRISM') AND (grism NE 'GRISM')) then begin
         print,'I only know how to deal with prism or grism data'
         RETURN
      endif
   endfor
   if ((TOTAL(fileList eq 1) NE 1) OR (TOTAL(fileList eq 0) ne 1)) then begin
      print,'I need one AOPEN file(set) and one BOPEN file(set)'
      RETURN
   endif
   if (NOT KEYWORD_SET(curveFile)) then begin
      if (grism EQ 'PRISM') then curveFile = getenv('prismscurve')
      if (grism EQ 'GRISM') then curveFile = getenv('grismscurve')
   endif
   command = cbin+'dispCrossCoeff "'+files[fileList[0]]+'" "'+files[fileList[1]]+ $
      '" '+curveFile+' '+base+doSky
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispCrossCoeff failed'
      print,command
      RETURN
   endif
RETURN
END

PRO midiSPipe, base, files, mask=mask, smooth=smooth, gsmooth=gsmooth, $
   dSky=dSky, dAve=dAve, cross=cross, curve=curve, photoFile=photoFile, $
   minopd=minopd
;    set defaults  for fringe smooth and group delay smooth
   if (NOT KEYWORD_SET(smooth)) then smooth = 50.
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 4.
   if (NOT KEYWORD_SET(gsmooth) AND (KEYWORD_SET(fast))) then gsmooth = .1
   if (NOT KEYWORD_SET(dSky)) then dSky = 0
   if (KEYWORD_SET(dAve)) then begin
      doAve = " -removeAverage" 
      if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' ' $
         else mo = " 20. "
   endif else begin
      doAve=" -noremoveAverage"
         if (KEYWORD_SET(minopd)) then mo = ' '+strtrim(minopd,2)+' '$
            else mo = " 0. " 
   endelse
;    consider temporary filter variant
;    where is executable?
   cbin = getenv('vltiCbin')+'/'
;
;           sort out options for photometry processing:
;           (1) only one input file, no photofile.  This should be an AB chopped file
;               and we get the photometry from the photometry channels
;           (2) same as above but with photofile present and assumed to be a reduced
;               set of photometry images.  After getting photometry from AB as
;               above, rescale photofile and use it as photometry
;           (3) 3 input files, presumed to be an AB file+A+B files.  Compute a photofile 
;               A- and B- files, and process as in (2).  Ignore anything in the
;               photofile arguement
;
   nfiles = N_ELEMENTS(files)
   if (nfiles EQ 1) then inFiles = files else begin
      if (nfiles NE 3) then begin
         print,'I need either 1 or 3 input files '
         RETURN
      endif
      inFiles = strarr(3)
      for i=0,2 do begin
         midiGetIns,files[i],shutter=shutter
         shutter = STRTRIM(shutter,2)
         if (shutter eq 'ABOPEN') then inFiles[0] = files[i] $
         else if (shutter eq 'AOPEN') then inFiles[1] = files[i] $
         else if (shutter eq 'BOPEN') then inFiles[2] = files[i]
      endfor
      if (total(inFiles eq '') NE 0) then begin
         print,'If you give me three files they must be ABOPEN+AOPEN+BOPEN'
         RETURN
      endif
   endelse
   midiGetIns,inFiles[0],shutter=shutter, grism=grism, object=object, opt1=opt1
   shutter = STRTRIM(shutter,2)
   grism   = STRTRIM(grism,2)
   opt1    = STRTRIM(opt1,2)
   if (opt1 NE 'SCI_PHOT') then begin
      print,'midiSPipe only works on SCI_PHOT data, not '+opt1+' data'
      RETURN
   endif
   if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
      print,'I only know how to deal with prism or grism data'
      RETURN
   endif

   if (N_ELEMENTS(mask) LE 0) then begin
      if (grism EQ 'PRISM') then mask = getenv('prismsmask')
      if (grism EQ 'GRISM') then mask = getenv('grismsmask')
   endif
   if (NOT KEYWORD_SET(curve)) then begin
      if (grism EQ 'PRISM') then curve= getenv('prismscurve')
      if (grism EQ 'GRISM') then curve= getenv('grismscurve')
   endif
   if (NOT KEYWORD_SET(cross)) then begin
      print,'I need a specified cross coupling coefficient file'
      RETURN
   endif else begin
;      midiGetIns,cross,grism=crossgrism, opt1=opt1
;      if (opt1 NE 'SCI_PHOT') then begin
;         print,'cross file ',cross,' does not use SCI_PHOT mode'
;         RETURN
;      endif 
;      if (grism NE crossgrism) then begin
;         print,'Data is in '+grism+' mode but cross file is in '+crossgrism+' mode.'
;         RETURN
;      endif
   endelse
;
;       now consider the reduced photometry file
;       if there are 3 input files, create it from #2 and #3 and store it in localPhoto
;       if there is 1 input file and photofile is specified, copy it to localPhoto
;       if not set localPhoto = ""
   localPhoto = ""
   if (nFiles eq 3) then begin
      print,'   Computing A- and B- photometry from 2nd and 3rd input files '
      print,'      and writing into '+base+'.P.photometry.fits'
      command = cbin+'dispPhot "'+files[1]+'" "'+files[2]+'" '+mask+' '+base+'.P '+STRTRIM(dSky,2)
;     begin CAH
      command = command + ' ' + curve + ' 1'
;     end CAH
      spawn,command,EXIT_STATUS=status
      if (status ne 0) then begin
         print,'dispPhot local failed'
         print,command
         RETURN
      endif
      localPhoto = base+'.P.photometry.fits'
   endif else if (KEYWORD_SET(photofile)) then localPhoto = photofile
   disp = 'dispVis "'
   command = cbin+disp+files+'" '+mask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth) + doAve +mo
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispVis failed'
      print,command
      RETURN
   endif
   erase
   !p.multi=[0,1,2,0,0]
   midiDelayPlot,base,object+':'+files[0]
   if (localPhoto eq "") then begin
      print,'   calculating photometry from PA+PB SCI_PHOT channels'
      command = cbin+'dispSciPhot "'+files[0]+'" '+mask+' '+curve+' '+$
         cross+' '+base+' '+STRTRIM(dSky,2) 
   endif else begin
      print,'   calculating photometry scale from PA+PB channels, and'
      print,'     applying it to I1+I2 photometric data in '+localphoto
      command = cbin+'dispSciPhotPhot "'+files[0]+'" '+mask+' '+curve+' '+$
         cross+' '+localPhoto+' '+base+' '+STRTRIM(dSky,2) 
   endelse
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispSciPhot failed'
      print,command
      RETURN
   endif
   spawn,cbin+'oirRedCal '+base, EXIT_STATUS=status
   if (status ne 0) then begin
      print,'oirRedCal failed'
      print,command
      RETURN
   endif
   midiInsVisPlot,base,object+':'+files[0]
   !p.multi = 0
RETURN
END
PRO midiSearch, base, files, mask=mask, smooth=smooth, gsmooth=gsmooth, $
  fast=fast, dAve=dAve
;    set defaults  for fringe smooth and group delay smooth
   if (NOT KEYWORD_SET(smooth)) then smooth = 4.
   if (NOT KEYWORD_SET(gsmooth)) then gsmooth = 4.
   if (KEYWORD_SET(dAve)) then doAve = " -removeAverage" else doAve=""
   if ((KEYWORD_SET(fast)) AND (NOT KEYWORD_SET(gsmooth))) then gsmooth = 0.2
;    where is executable?
   cbin = getenv('vltiCbin')+'/'
;    if files not specified, bring up gui
   if (N_PARAMS() LT 2) then d = midiGuis() else d=files
;    parse input files to see if both visibility and photometry present
   midiGetIns,d[0],shutter=shutter, grism=grism, object=object, opt1=opt1
   shutter = STRTRIM(shutter,2)
   grism = STRTRIM(grism,2)
   if ((grism NE 'PRISM') AND (grism NE 'GRISM'))then begin
      print,'I only know how to deal with prism or grism data'
      RETURN
   endif

   if (N_ELEMENTS(mask) LE 0) then begin
;      mask='$midiutil/prismnewmask.fits'
      if (grism EQ 'PRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('prismhmask') $
            else mask = getenv('prismsmask')
      endif
      if (grism EQ 'GRISM') then begin
         if (opt1 EQ 'HIGH_SENS') then mask = getenv('grismhmask') $
            else mask = getenv('grismsmask')
      endif
   endif
   if (NOT KEYWORD_SET(fast)) then $
      command = cbin+'dispSearch "'+d[0]+'" '+mask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth) + doAve $
   else command = cbin+'dispSearchT "'+d[0]+'" '+mask+' '+base+' '+$
     STRING(smooth)+' '+STRING(gsmooth)
   spawn,command,EXIT_STATUS=status
   if (status ne 0) then begin
      print,'dispSearch failed'
      print,command
      RETURN
   endif
   erase
   midiDelayAmpPlot1,base,object+':'+d[0], gsmooth=gsmooth
RETURN
END

FUNCTION reduceDay, dir=dir, infiles=infiles, outdir=outdir, base=base, mask=mask
oldx = !x
oldy = !y
!x.style=1
!y.style=1

;
   if (NOT KEYWORD_SET(outdir)) then odir='' else odir=outdir+'/'
   if (KEYWORD_SET(infiles)) then files = infiles else $
      if (KEYWORD_SET(dir)) then files=midiGuis(dir=dir) $
         else files=midiGuis()
   nFiles = N_ELEMENTS(files)
   print,'Got ',nFiles, ' Files. Proceed?'
   ll = ''
   read,ll
   if ((ll EQ 'N') OR (ll EQ 'n')) then RETURN,files

   if (0 NE (nFiles mod 3)) then begin
      print,'Number of files not divisible by 3'
      RETURN,files
   endif

   n3 = nFiles/3
   base = STRARR(n3)
   for i=0, n3-1 do begin
      midiGetIns,files[3*i+1],object=object
      base[i]=STRTRIM(object,2)
   endfor
   print,'raw base names ',base
; rename duplicates
   for i=0, n3-1 do begin
      w = WHERE(base[i] EQ base)
      nW = N_ELEMENTS(w)
      if (nW GT 1) then $
         for j=1,nW-1 do base[w[j]]=STRTRIM(base[w[j]])+'-'+STRTRIM(j,2)
   endfor
   print,'new base names ',base

   for iFile=0, n3-1 do begin
      fileList = REPLICATE(-1,3)
      d = files[3*iFile+indgen(3)]
      for i=0,2 do begin
         s = STRSPLIT(d[i],' ',/EXTRACT)
         d[i] = s[0]
         midiGetIns,d[i],shutter=shutter, grism=grism
         shutter = STRTRIM(shutter,2)
         grism = STRTRIM(grism,2)
         if (shutter EQ 'AOPEN') then fileList[0]=i
         if (shutter EQ 'BOPEN') then fileList[1]=i
         if (shutter EQ 'ABOPEN') then  fileList[2]=i
         if ((grism NE 'PRISM') AND (grism NE 'GRISM')) then begin
            print,'I only know how to deal with prism or grism data'
            RETURN,files
         endif
      endfor
      if (TOTAL(fileList EQ -1) NE 0) then begin
         print,'For base ',base[iFile],' I dont have Int+2*Phot'
         RETURN,files
      endif
   endfor
   for iFile=0, n3-1 do begin
      if (STRMID(base[iFile],0,4) EQ 'MIDI') then begin
          print,'I wont process "MIDI" files'
          RETURN,files
      endif
      base[iFile] = STRTRIM(odir+base[iFile])
      if(base[iFile] EQ '') then RETURN,files
      print,iFile,' cleaning old files named '+base[iFile]+'.*.fits'
      spawn,' rm '+base[iFile]+'*.fits'
      print,iFile,' processing ',base[iFile]
      midipipe,base[iFile],files=files[3*iFile+indgen(3)],mask=mask
   endfor
!x=oldx
!y=oldy
RETURN,files
END

PRO midiCalibrate, tag, calTag, photoTag=photoTag, $
   calFlux10=calFlux10, diam=diam, nophot=nophot, print=print

   if (not KEYWORD_SET(calFlux10)) then calFlux10 = 1.
   doprint = KEYWORD_SET(print)

   if (NOT KEYWORD_SET(diam)) then diam = 0.
   cbin = getenv('vltiCbin')+'/'
   cmd = cbin+'oirCalibrateVis '+tag+' '+calTag + ' '
   if (KEYWORD_SET(photoTag)) then cmd = cmd + "-sp "+photoTag+" "
   if (KEYWORD_SET(calFlux10)) then cmd = cmd + "-calflux "+ $
      STRTRIM(calFlux10,2)+" "
   if (KEYWORD_SET(diam)) then cmd = cmd + "-cald "+STRTRIM(diam,2)+" "
   if (KEYWORD_SET(nophot)) then cmd = cmd + " -nophot "

   spawn,cmd
   file = OBJ_NEW('fitsfile',caltag+'.redcal.fits')
   head = file->head()
   caldate = head->getpar('DATE-OBS')
   obj_destroy,head
   file->close
   obj_destroy,file
   file = OBJ_NEW('fitsfile',tag+'.redcal.fits')
   head = file->head()
   tardate = head->getpar('DATE-OBS')
   file->close
   obj_destroy,head
   obj_destroy,file
   plotname=!d.name
   set_plot,'ps'
   device,file=tag+'calPlots.ps',xs=15,ys=25,yo=1
   oldMulti = !p.multi
   !p.multi=[0,1,4,0,0]
   title=calTag+':'+caldate
;                   instrumental visibilities
;                   modify visamp by supplied cal flux.
   calVis = oirGetVis(calTag+'.redcal.fits',wave=wave)
   calFlux = calFlux10*100./wave^2

   !x.style=1
   !y.style=1
   !x.range=[7.5,13.5]
   xw = where(wave gt 7.6 and (wave lt 13.5))
   nw = N_ELEMENTS(xw)/8
   xer = xw[8*indgen(nw)]
   xt = 'Wavelength, (mu)'

   nWave = N_ELEMENTS(wave) -1
   nxWave = N_ELEMENTS(xw) - 1
; Calibrator; plot total A+B/2 raw flux, total mask sqrt(AB) flux raw correlated flux

   calPhot = oirgetData(calTag+'.photometry.fits')
   calCorr = oirGetVis(calTag+'.corr.fits')
;
   data = .5*(calPhot[0].data1+calPhot[1].data1)
   err  = calPhot[8].data1
   plot,wave,data,yr=[0,1.2*max(data[xw])],$
   xtitle=xt,title=title+' Raw Total Photometry'
      errplot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.RawTotalPhotometry.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif
;
   data = calPhot[5].data1
   err  = calPhot[11].data1
   plot,wave,data,yr=[0,1.2*max(data[xw])],$
      xtitle=xt,title=title+' Raw Masked Photometry'
      errplot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.RawMaskedPhotometry.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif
;
   data = calCorr[0].visamp
   err  = calCorr[0].visamperr
   plot,wave,data,yr=[0,1.2*max(data[xw])],$
      xtitle=xt,title=title+' Raw Correlated Amplitude'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]

   if (doprint) then begin
      openw,unit,title+'.RawCorrelatedAmplitude.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif
;
   data = calCorr[0].visphi
   plot,wave,data, xtitle=xt,title=title+' Raw Correlated Phase'
      errPlot,wave[xer],data[xer]-calCorr[0].visphierr[xer], $
         data[xer]+calCorr[0].visphierr[xer]
   if (doprint) then begin
      openw,unit,title+'.RawCorrelatedPhase.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif

;calibrator reduced values

   data = calVis[0].visamp
   err  = calVis[0].visamperr
   plot,wave,data, yr=[0,1.2*max(data[xw])], $
      xtitle=xt,title=title+' Instrumental Visibility'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.InstrumentalVisibility.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif
;
   data = calVis[0].visphi
   err  = calVis[0].visphierr
   plot,wave,data, xtitle=xt,title=title+' Instrumental Phase'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.InstrumentalPhase.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif
;
   data = .5*(calPhot[0].data1 + calPhot[1].data1)/calFlux
   err = calPhot[8].data1/calFlux
   plot,wave,data, xtitle=xt,title=title+' ADU/s/Jy ', $
      yr=[0,1.2*max(data[xw])]
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
;
;target raw values

   targCorr = oirGetVis(tag+'.corr.fits')
   targPhot = oirGetData(tag+'.photometry.fits')
   title=tag+':'+tardate
;
   data = 0.5*(targPhot[0].data1+targPhot[1].data1)
   err  = targPhot[8].data1
   plot,wave,data, yr=1.2*[0,max(data[xw])], xtitle=xt,$
      title=title+'Raw Total A+B photometry (ADU/s/chan) '
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.RawTotalPhotometry.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif
;
   data = targPhot[5].data1
   err  = targPhot[11].data1
   plot,wave,data, yr=1.2*[0,max(data[xw])], xtitle=xt,$
      title=title+'Raw Masked sqrt(AB) photometry (ADU/s/chan) '
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.RawMaskedPhotometry.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif
;
   yr = [0,1.2*max(targCorr.visamp[xw])]
   for i=0,n_elements(targCorr)-1 do begin
      data = targCorr[i].visamp
      err  = targCorr[i].visamperr
      plot,wave,data,yr=yr, xtitle=xt,$
         title=title+'Raw Correlated Flux (ADU/s/channel)'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.RawCorrelatedFlux.dat',/GET_LUN
         for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
         FREE_LUN,unit
      endif
   endfor

;instrumental visibility
   targInsVis = oirGetVis(tag+'.redcal.fits')
   yr = [0,1.2*max(targInsVis.visamp[xw])]
   yrd = [min(targInsVis.visphi[xw]), max(targInsVis.visphi[xw])]
   for i=0,n_elements(targInsVis)-1 do begin
      data = targInsVis[i].visamp
      err  = targInsVis[i].visamperr
      plot,wave,data,yr=yr, $
         xtitle=xt,title=title+' Instrumental Visibility '
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.InstrumentalVisibility.dat',/GET_LUN
         for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
         FREE_LUN,unit
      endif

      data = targInsVis[i].visphi
      err  = targInsVis[i].visphierr
      plot,wave,data , yr=yrd,xtitle=xt,title=title+' Instrumental Phase'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.InstrumentalPhase.dat',/GET_LUN
         for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
         FREE_LUN,unit
      endif
   endfor

;calibrated target values
   targVis  = oirGetVis(tag+'.calvis.fits')
   targPhot = oirGetData(tag+'.calphot.fits')

;calibrated visamp and phase
   yr = [0,1.2*max(targVis.visamp[xw])<1.5]
   yrd = [min(targVis.visphi[xw]), max(targVis.visphi[xw])]
   for i=0,n_elements(targVis)-1 do begin
      data = targVis[i].visamp
      err  = targVis[i].visamperr
      plot,wave,data,yr=yr, $
         xtitle=xt,title=title+' Calibrated Visibility '
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.CalibratedVisibility.dat',/GET_LUN
      for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
      FREE_LUN,unit
   endif

      data = targVis[i].visphi
      err  = targVis[i].visphierr
      plot,wave,data , yr=yrd,xtitle=xt,title=title+' Calibrated Phase'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.CalibratedPhase.dat',/GET_LUN
      for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
      FREE_LUN,unit
   endif
   endfor

;calibrated photometry, total and masked
   data = targPhot[0].data1   ; total
   err  = targPhot[2].data1
   yr = 1.2*max(data[xw])*[0,1]
   plot,wave,data, yr=yr, xtitle=xt, $
      title=title+' Calibrated Total Flux(Jy)'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.CalibratedTotalPhotometry.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif

;masked flux (Jy)

   data = targPhot[1].data1   ; masked
   err  = targPhot[3].data1
   plot,wave,data, yr=yr,xtitle=xt, $
      title=title+' Calibrated Flux under Mask (Jy)'
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
   if (doprint) then begin
      openw,unit,title+'.CalibratedMaskedPhotometry.dat',/GET_LUN
      for i=0, nxWave do printf,unit,wave[xw[i]], data[xw[i]], err[xw[i]]
      FREE_LUN,unit
   endif

;correlated flux (Jy) = masked flux*visibility

;            default is correlated flux = visibility * JY(calib) * 
;                 total_flux(source)/total_flux(calibrator)
;
;            if photometry is lousy you can use alternatively
;                 correlated flux = visibility * JY(calib) * 
;                 masked_flux(source)/masked_flux(calibrator)
;            since the visibility = corr_flux(source)/masked_flux(source) *masked_flux(cal)/corr_flux(cal)
;            the masked_fluxes cancel out.
   for i=0,n_elements(targVis)-1 do begin
      if (KEYWORD_SET(nophot)) then begin
         cc = calCorr.visamp > (.003 * max(calCorr.visamp)) ; truncated version of calcorr
         va = targCorr[i].visamp
         data = calFlux * va / cc
         err  = calFlux*SQRT ((targCorr[i].visamperr/cc)^2 + $
            (va * calCorr.visamperr/(cc*cc))^2)
      endif else begin
         data = targVis[i].visamp * targPhot[0].data1
         err = targVis[i].visamperr * targPhot[0].data1
      endelse
      uvw = midifitsuvw(tag+'.fringes.fits', alt=alt)
      plot,wave,data,yr=max(data[xw])*[0,1.2], $
         xtitle=xt,title=title+' Correlated Flux(Jy) UV ='+string(uvw[0:1],format='(2f8.2)')
      errPlot,wave[xer],data[xer]-err[xer],data[xer]+err[xer]
      if (doprint) then begin
         openw,unit,title+'.CalibratedCorrelatedFlux.dat',/GET_LUN
         for j=0, nxWave do printf,unit,wave[xw[j]], data[xw[j]], err[xw[j]]
         FREE_LUN,unit

         openw,unit,title+'UVW.dat',/GET_LUN
         uvw = midifitsuvw(tag+'.fringes.fits', alt=alt)
         midiFieldAngle, tag+'.fringes.fits',para, phiN
         printf,unit,'UVW(meters) ',uvw, format='(a,2x,3f10.2)'
         printf,unit,'AirMass ',1./sin(alt/!radeg),format='(a, f10.2)'
         printf,unit,'North is ',180-phiN,' degrees counterclockwise from UP',$
         format='(a,2x,f10.2,2x,a)'
      FREE_LUN,unit
      endif
   endfor

device,/close
set_plot,plotname
!p.multi = oldMulti
RETURN
END

PRO pwd
cd,current=c&print,c
RETURN
END

PRO midiMovie,file,datacol,nmax=nmax,dt=dt,mag=mag,highpass=highpass,doall=doall
;
   if (KEYWORD_SET(doall)) then f = file $
      else begin
         f = STRSPLIT(file,' ',/extract)
         f = f[0]
      endelse
   print,'Processing ',f
   if (N_PARAMS() LT 2) then datacol = 1
   col='DATA'+STRTRIM(STRING(datacol),2)
   print,'Retrieving '+col
   data = oirgetdata(file,col=col)
   nData = N_ELEMENTS(data)
   if (NOT KEYWORD_SET(nmax)) then nmax=ndata
   print,nmax,' Frames '
   data = float(data.(0))
   if (KEYWORD_SET(highpass)) then mdata = TOTAL(data,3)/nData $
      else mdata = 0.*REFORM(data(*,*,0))
   If (NOT KEYWORD_SET(dt)) then dt = .05
   if (NOT KEYWORD_SET(mag)) then mag = 2
   for i=0, nmax-1 do begin tvsclm,data(*,*,i)-mdata,mag
      wait,dt
   endfor
RETURN
END

FUNCTION midiGetCorr,tag,wave=wave
   vis=oirGetVis(tag+'.corr.fits',wave=wave)
   return,vis.visamp
END
FUNCTION midiGetPhase,tag,wave=wave
   vis=oirGetVis(tag+'.corr.fits',wave=wave)
   return,vis.visphi
END
FUNCTION midiGetPhot,tag
   RETURN,oirGetData(tag+'.photometry.fits')
END

FUNCTION midiGetComplex,tag,qual
   data = oirGetData(tag+'.'+qual+'.fits', col='data1')
   data = pseudoComplex(data.data1)
   RETURN,data
END

FUNCTION midiGetDelay,tag, time
   delay = oirGetDelay(tag+'.groupdelay.fits')
   time = REFORM(delay.time)
   time = 86400.*(time - time[0])
   RETURN,delay
END

PRO midiPutDelay, delay, file, ierr=ierr
;    put delay structure into new output file
   dObj = OBJ_NEW('delay')
   dObj->newFile,file, ierr=ierr
   if (ierr NE 0) then RETURN
   outDelay  = REPLICATE(dObj->template(), N_ELEMENTS(delay))
   outDelay.telescope = REFORM(delay.telescope)
   outDelay.time = REFORM(delay.time)
   outDelay.delay=REFORM(delay.delay)/2.9979258e14 ; seconds, not microns
   outDelay.amplitude=REFORM(delay.amplitude)
   dObj->writerows,outDelay
   dObj->close
   OBJ_DESTROY,dObj
RETURN
END
FUNCTION midiGetOpd,tag
   RETURN,oirGetOpd(tag+'.fringes.fits')
END

FUNCTION midiGetGroupData,tag
   z = oirGetData(tag+'.groupdelay.fits')
   RETURN,pseudocomplex(z.data1)
END

FUNCTION midiGetDispersion,tag
   z = OBJ_NEW('fitstable',tag+'.ungroupdelay.fits',extname='DISPERSION')
   zd = z->readrows()
   OBJ_DESTROY,z
RETURN,zd
END
PRO midiPutVis, outFile, wave, visamp, visphi, visamperr, visphierr, flag, uv=uv
;
;   outFile = output file name
;   wave    = wavelength array in microns
;   visamp  = visibility amplitude
;   visphi  = visibility phase (degrees)
;   visamperr = rms visibility error (optional, default=0)
;   visphierr = rms phase error (degrees) (optional, default=0)
;   flag = (1 = good, 0 = bad) (optional, default=1)
;   uv   = uv coordinates (meters).  If not given defaults to 0!
;
    vistable = OBJ_NEW('oi_vis', N_ELEMENTS(wave), insname='MIDI', arrname='VLTI')
    template = vistable->dataTemplate()
    template.visamp = visamp
    if (N_PARAMS() GT 3) then template.visphi = visphi
    if (N_PARAMS() GT 4) then template.visamperr = visamperr
    if (N_PARAMS() GT 5) then template.visamperr = visphierr
    if (N_PARAMS() GT 6) then template.flag = flag else template.flag[*] = 1.
    if (NOT KEYWORD_SET(uv)) then uv = [0., 0.]
    template.ucoord = uv[0]
    template.vcoord = uv[1]
;
   vistable->newfile, outFile
   vistable->writeRows, template
   wavetable = OBJ_NEW('oi_wavelength', wave=wave*1.e-6,band='N')
   wavetable->appendtofile, vistable
;  wavetable->writeRows, *(wavetable->table())
   wavetable->close
   OBJ_DESTROY,vistable
   OBJ_DESTROY,wavetable
RETURN
END

PRO midiPutVis2, outFile, wave, vis2data, vis2err, flag, uv=uv
;
;   outFile = output file name
;   wave    = wavelength array in microns
;   vis2data    = squared visibility
;   vis2err = visibility^2 error (optional, default=0)
;   flag = (1 = good, 0 = bad) (optional, default=1)
;   uv   = uv coordinates (meters).  If not given defaults to 0!
;
    vistable = OBJ_NEW('oi_vis2', N_ELEMENTS(wave), insname='MIDI', arrname='VLTI')
    template = vistable->dataTemplate()
    template.vis2data = vis2data
    if (N_PARAMS() GT 2) then template.vis2err = vis2err else $
       template.vis2err[*] = 0.
    if (N_PARAMS() GT 3) then template.flag = flag else template.flag[*] = 1.
    if (NOT KEYWORD_SET(uv)) then uv = [0., 0.]
    template.ucoord = uv[0]
    template.vcoord = uv[1]
;
   vistable->newfile, outFile
   vistable->writeRows, template
   wavetable = OBJ_NEW('oi_wavelength', wave=wave*1.e-6,band='N')
   wavetable->appendtofile, vistable
;  wavetable->writeRows, *(wavetable->table())
   wavetable->close
   OBJ_DESTROY,vistable
   OBJ_DESTROY,wavetable
RETURN
END

PRO midiCleanTag, tag, all=all
   defClean=['compressed','fringes','groupdelay','ungroupdelay']
   for i=0,N_ELEMENTS(defClean)-1 do spawn,'rm '+tag+'.'+defClean[i]+'.fits'
RETURN
END

