;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
; $Id: utilities.pro,v 1.24 2007/01/02 14:15:42 koehler Exp $
;
; utilities.pro
; Created:     Fri Aug  8 11:34:40 2003 by Koehler@sun47
; Last change: Tue Jan  2 14:57:26 2007
;
; PURPOSE:
;	some utility routines related to xmidispvisi et al.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function midifile_search, dir_spec, PATTERN=pattern, OBJECT=obj_patt, $
                          FRINGE_SEARCH=fringe_search, FSEARCH=fsearch,$
                          FRINGE_TRACK=fringe_track, FTRACK=ftrack, $
                          PHOTOMETRY=photometry, PRISM=prism, GRISM=grism,$
                          HIGH_SENSE=high_sens, SCI_PHOT=sci_phot, $
                          FIRST=first, LAST=last,$
                          KEYWORD=keyword, KEYVALUE=keyvalue, VERBOSE=verbose,$
                          OBJ_NAMES=obj_names, NRTS_MODES=nrts_modes, $
                          SHUTTERS=shutters, DISPERSERS=dispersers, $
                          BEAMSPLITS=beamsplits

  if (keyword_set(fringe_search)) then fsearch=1
  if (keyword_set(fringe_track))  then ftrack=1

  if keyword_set(pattern) then begin
      if N_params() lt 1 then dir_spec= "."
      files_in= file_search(dir_spec,pattern,COUNT=nFiles,/Test_Read,/Test_Regular)
  endif else begin
      if N_params() lt 1 then dir_spec= "MIDI.*.fits"
      files_in= file_search(dir_spec,COUNT=nFiles,/Test_Read,/Test_Regular)
  endelse
  print,nFiles," files total"
  if nFiles le 0 then return,""

  files_out= strarr(nFiles)
  if arg_present(obj_names)  then obj_names = strarr(nFiles)
  if arg_present(nrts_modes) then nrts_modes= strarr(nFiles)
  if arg_present(shutters)   then shutters  = strarr(nFiles)
  if arg_present(dispersers) then dispersers= strarr(nFiles)
  if arg_present(beamsplits) then beamsplits= strarr(nFiles)

  iIn = 0
  iOut= -1
  TplStart= ''
  FileNo  = 9999
  key = ''

  while (iIn lt nFiles) do begin
      fname = files_in[iIn]
      if keyword_set(verbose) then print,iIn," = ",fname
      iIn++	;; in case we want to use 'continue'

      if keyword_set(first) or keyword_set(last) then begin
          bname = file_basename(fname)

          if keyword_set(first) then  if bname lt first then continue
          if keyword_set(last) then   if bname gt last then continue
      endif

      newFits = OBJ_NEW('fitsfile',fname)
      if not obj_valid(newFits) then begin
          if keyword_set(verbose) then print,"Cannot read file"
          continue
      endif
      newHead = newFits->head()
      if not obj_valid(newHead) then begin
          if keyword_set(verbose) then print,"Cannot read file header"
          obj_destroy,newFits
          continue
      endif
      newExtns= newFits->listextensions()
      newTplStart= newHead->getPar('TPL START', ntpl)
      newFileNo  = newHead->getPar('EXPO FILENO')

      obj_name = strcompress(newHead->getpar('OBS TARG NAME'),/remove_all)
      nrts_mode= strcompress(newHead->getpar('NRTS MODE'),/remove_all)
      shutter  = strcompress(newHead->getpar('INS SHUT NAME'),/remove_all)
      disperser= strcompress(newHead->getpar('INS GRIS NAME'),/remove_all)
      beam     = strcompress(newHead->getpar('INS OPT1 ID'),/remove_all)

      if keyword_set(keyword) then $
        key= strcompress(newHead->getpar(keyword),/remove_all)

      obj_destroy,newFits
      obj_destroy,newHead

      if (max(strpos(newExtns, 'IMAGING_DETECTOR')) lt 0) then begin
          if keyword_set(verbose) then print,"No MIDI data"
          continue
      endif
      if keyword_set(verbose) then $
        print, "obj '",obj_name, "' mode ",nrts_mode,$
        " shut ",shutter, " grism ",disperser, " ukey ",key
      ;;
      ;; check whether we want it
      ;;
      if keyword_set(obj_patt) then $
        if not strmatch(obj_name,obj_patt,/Fold_Case) then continue
      ;; do not use "and", obj_patt may be undefined

      if keyword_set(fsearch) or keyword_set(ftrack) or keyword_set(photometry) then begin
          sel = 0
          if (keyword_set(fsearch) $
              and strmatch(nrts_mode,'OBS_FRINGE_SEARCH_*',/Fold_case)) then sel=1
          if (keyword_set(ftrack) $
              and strmatch(nrts_mode,'OBS_FRINGE_TRACK_*',/Fold_case)) then sel=1
          if keyword_set(photometry) and $
            (strcmp(shutter,'AOPEN',/Fold_case) or strcmp(shutter,'BOPEN',/Fold_case))$
            then sel=1
          if sel eq 0 then continue
      endif

      if keyword_set(prism) and not strcmp(disperser,"PRISM",/Fold_Case) then continue
      if keyword_set(grism) and not strcmp(disperser,"GRISM",/Fold_Case) then continue

      if keyword_set(high_sens) and not strcmp(beam,"HIGH_SENS",/Fold_Case) then continue
      if keyword_set(sci_phot)  and not strcmp(beam,"SCI_PHOT", /Fold_Case) then continue

      if keyword_set(keyvalue) then $
        if not strmatch(key,keyvalue, /Fold_Case) then continue

      if ((ntpl gt 0) and (TplStart eq newTplStart) $
          and (newFileNo gt FileNo)) then begin
          ;; this is a midi continuation file
          files_out[iOut] = files_out[iOut] + ' ' + fname
      endif else begin
          ;; this is a new file, make a new entry in output list
          iOut++
          files_out[iOut] = fname
          if arg_present(obj_names)  then  obj_names[iOut]= obj_name
          if arg_present(nrts_modes) then nrts_modes[iOut]= nrts_mode
          if arg_present(shutters)   then   shutters[iOut]= shutter
          if arg_present(dispersers) then dispersers[iOut]= disperser
          if arg_present(beamsplits) then beamsplits[iOut]= beam
      endelse
      TplStart= newTplStart
      FileNo  = newFileNo
   endwhile
   if iOut lt 0 then begin
       print,"no datasets matching criteria"
       return,""
   endif
   print,iOut+1L," datasets match criteria"

   if arg_present(obj_names)  then  obj_names=  obj_names[0:iOut]
   if arg_present(nrts_modes) then nrts_modes= nrts_modes[0:iOut]
   if arg_present(shutters)   then   shutters=   shutters[0:iOut]
   if arg_present(dispersers) then dispersers= dispersers[0:iOut]
   if arg_present(beamsplits) then beamsplits= beamsplits[0:iOut]
   return, files_out[0:iOut]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; try to fix paths in midiGui.SAV-files...
;;
Function miaguis, dir
  if N_params() gt 0 then begin
      dir = dir + "/"
      dsets = midiguis(dir=dir)
  endif else begin
      dir = ""
      dsets = midiguis()
  endelse
  for i=0,N_elements(dsets)-1 do begin
      files= dir + file_basename( oirFileList(dsets[i]))
      dsets[i]= strjoin(files," ")
  endfor
  return, dsets
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro baseline, filename, LENGTH=base_len, ANGLE=base_pa, QUIET=quiet

  filelist = oirFileList(filename[0])
  im = obj_new('imagedata', filelist[0])
  he = im->prihead()
  objectname= STRCOMPRESS(he->getpar('OBS TARG NAME'),/remove_all)
  esolen = StrCompress(he->getpar('HIERARCH ESO ISS PBL12 START'),/remove_all)
  stat1  = StrCompress(he->getpar('HIERARCH ESO ISS CONF STATION1'),/remove_all)
  stat2  = StrCompress(he->getpar('HIERARCH ESO ISS CONF STATION2'),/remove_all)
  date   = StrCompress(he->getpar('DATE-OBS'),/remove_all)
  lst    = StrCompress(he->getpar('LST'),/remove_all)
  RA     = StrCompress(he->getpar('RA') ,/remove_all)
  Dec    = StrCompress(he->getpar('DEC'),/remove_all)
  obj_destroy,im

  if (stat1 gt stat2) then begin
      tmp = stat1 & stat1 = stat2 & stat2 = tmp
  endif

  project_baseline, (LST/3600. - RA/15.), Dec, stat1+"-"+stat2, base_len, base_pa

  if not keyword_set(quiet) then begin
      print,''
      print, "object:              ", objectname
      print, "Baseline:            ", stat1, " - ", stat2
      print, "ESO's baseline len.: ", esolen
      print, "Date of observation: ", date
      print, "Local Sid Time:      ", LST
      print, "Right Ascension:     ", RA
      print, "Declination:         ", Dec
      print,''
      print, "Projected Baseline"
      print, Format='("             length: ", F6.2)', base_len
      print, Format='("     position angle: ", F6.2)', base_pa
      print,''
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; CALLING SEQUENCE:
;	plot_visibility, vis [, vis2...]
;
; INPUT:
;	up to 6 visibility-arrays as returned from xmidispvisi
;
; OUTPUT:
;	plot of visibility vs. lambda in current plot window
;
PRO plot_visibility, v0, v1, v2, v3, v4, v5,$
                     XTITLE=xtitle, YTITLE=ytitle, YRANGE=yrange, _EXTRA=extra

  if not keyword_set(xtitle) then xtitle= "Wavelength [micron]"
  if not keyword_set(ytitle) then ytitle= "Visibility"
  if not keyword_set(yrange) then yrange= [0,1]

  plot, v0[0,*], v0[2,*], YRANGE=yrange, XTITLE=xtitle, YTITLE=ytitle, _EXTRA=extra

  if N_params() gt 1 then oplot, v1[0,*], v1[2,*], linestyle=1
  if N_params() gt 2 then oplot, v2[0,*], v2[2,*], linestyle=2
  if N_params() gt 3 then oplot, v3[0,*], v3[2,*], linestyle=3
  if N_params() gt 4 then oplot, v4[0,*], v4[2,*], linestyle=4
  if N_params() gt 5 then oplot, v5[0,*], v5[2,*], linestyle=5

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; CALLING SEQUENCE:
;	mtv, data [, atv-flags]
;
; INPUT:
;	data - one slice of MIDI data (a struct with elements data1
;              and data2)
;	atv-flags - all flags accepted by atv
;
; OUTPUT:
;	calls atv to display the frame data
;
PRO mtv, datarr, _EXTRA = extra
  data = datarr[0]	;just in case, but don't mess with caller's data

  sz1 = size(data.data1)
  sz2 = size(data.data2)
  sz_x= sz1[1] + 1 + sz2[1]

  tags = strtrim(tag_names(data),2)
  if where(tags eq "DATA3") ge 0 then begin
      ; there is a data3 field, so this is probably sci-phot data
      sz3 = size(data.data3)
      sz4 = size(data.data4)

      sz_x= max( [sz_x, sz3[1] + 1 + sz4[1]] )
      sz_y= max( [sz1[2]+1+sz3[2], sz2[2]+1+sz4[2]] )

      img = fltarr(sz_x,sz_y)
      img[     0  : sz1[1]-1, sz3[2]+1 : sz_y-1 ] = data.data1
      img[ sz1[1]+1 : sz_x-1, sz4[2]+1 : sz_y-1 ] = data.data2
      img[     0  : sz3[1]-1,     0  : sz3[2]-1 ] = data.data3
      img[ sz3[1]+1 : sz_x-1,     0  : sz4[2]-1 ] = data.data4

  endif else begin
      sz_y= max( [sz1[2],sz2[2]] )

      img = fltarr(sz_x,sz_y)
      img[   0 : sz1[1]-1 ,* ] = data.data1
      img[ sz1[1]+1:sz_x-1,* ] = data.data2
  endelse
  ;print,sz_x," x ",sz_y

  atv, img, _EXTRA=extra
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; MIDI-filename processing:
;
; MIDIname_date(filename)   returns date-part of filename
; MIDIname_time(filename)   returns time-part of filename
; MIDIname_datime(filename) returns date- and time-part
;
Function MIDIname_date, fname
  first_char= Strpos(fname,'/MIDI.20')+6
  fname	    = Strmid(fname,first_char)
  last_char = Strpos(fname,'T')
  return, Strmid(fname,0,last_char)
end

Function MIDIname_time, fname
  first_char= Strpos(fname,'/MIDI.20')+6
  first_char= first_char + Strpos(Strmid(fname,first_char),'T')+1
  last_char = Strpos(fname,'.fits')
  return, Strmid(fname,first_char,last_char-first_char)
end

Function MIDIname_datime, fname
  first_char= Strpos(fname,'/MIDI.20')+6
  last_char = Strpos(fname,'.fits')
  return, Strmid(fname,first_char,last_char-first_char)
end

function make_maskname, fname
  first_char      = STRPOS(fname,'/MIDI.20')+6
  last_char       = STRPOS(fname,'.fits')
  short_file_name = STRMID(fname,first_char,last_char-first_char)
  return, 'PHOTOMETRY/Mask_'+short_file_name+'.fits'
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Wrappers for C-programs
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function Cached_MeanRMS, filename
    first_char= STRPOS(filename,'/MIDI.20')+6
    last_char = STRPOS(filename,'.fits')
    rmsname = "PHOTOMETRY/Mean_" + $
      STRMID(filename,first_char,last_char-first_char) + ".fits"
    ; this gets rid of the other filenames in the list, too

    if !MIAdebug ne 0 then print,'Mean+RMS file is "', rmsname,'"'

    if file_test(rmsname) then begin
        if !MIAdebug ne 0 then print,"Reading cached file..."
	rmsdata = oirgetdata(rmsname,ierr=ierr)
	if ierr then begin
            print,"Cached file damaged, initiating repair"
            file_move,rmsname,rmsname+"-junk",/Verbose,/Overwrite
        endif
    endif else ierr=42

    if ierr ne 0 then begin
        if !MIAdebug ne 0 then print,"Computing mean&rms..."
        cmd = getenv("drsRoot") + '/c/bin/oirMeanRMS'
        if file_test(cmd,/EXECUTABLE) then begin
            if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"
            cmd = cmd + ' "' + filename + '" ' + rmsname
            print,cmd
            spawn,cmd,EXIT_STATUS=status
            if (status ne 0) then begin
                print,'oirMeanRMS failed'
                Log_error, filename, 'oirMeanRMS failed'
                return,0
            endif
            rmsdata = oirgetdata(rmsname)
        endif else begin
            print, cmd + " not found, falling back on IDL code"
            rmsdata = oirgetmeanrms(filename)
        endelse
    endif
    return, rmsdata
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function MIDIChopImageC, filename, BEFORE=before, AFTER=after

  ; default values used in MIDIChopImage (at least on 24-Jul-2003)

  if not keyword_set(before) then before= 0
  if not keyword_set(after)  then after = 1

  first_char= STRPOS(filename,'/MIDI.20')+6
  last_char = STRPOS(filename,'.fits')
  short_name= STRMID(filename,first_char,last_char-first_char)

  cachefile = string(Format='(%"PHOTOMETRY/Chop_%s_b%d_a%d.fits")',$
                     short_name, before, after)

  if not file_test(cachefile) then begin
      print, "Cached chop-frame does not exist, let's create it!"

      cmd = getenv("drsRoot") + '/c/bin/midiChopimg'
      if file_test(cmd,/EXECUTABLE) then begin
          if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"
          cmd = cmd +	' -b ' + strtrim(before,2) + $
			' -a ' + strtrim(after,2) + $
			' "' + filename + '" ' + cachefile
          print,cmd
          spawn,cmd,EXIT_STATUS=status
          if (status ne 0) then begin
              print,'midiChopimg failed'
              Log_error, filename, 'midiChopimg failed'
              return,0
          endif
      endif else begin
          print, cmd + " not found,"
          print, "falling back on IDL-code"
          chop_image= MIDIChopImage(filename, before=before, after=after)
          oirNewData, (oirFileList(filename))[0], cachefile, chop_image
          return, chop_image
      endelse
  endif
  chop_image= oirgetdata(cachefile)
  return, chop_image
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; maybe we can add nodding to the same function?

Function MIDISkySubImage, filename, BEFORE=before, AFTER=after, ERRORLOG=errorlog

  chop_image= MIDIChopImageC(filename, BEFORE=before, AFTER=after)

  im = obj_new('imagedata',(oirFileList(filename))[0])
  he = im->prihead()

  if he->getpar('HIERARCH ESO ISS CHOP THROW') eq 0. then begin
      ;print,"MIDISkySubImg: errorlog is:_"+errorlog+"_ = "+string(N_elements(errorlog))+" elements"

      Log_error, filename, "Chopping throw is 0.0"
      if N_elements(errorlog) gt 0 then begin
          errorlog = errorlog + filename + ": Chopping throw is 0.0" + string(10B)
      endif else begin
          ans = ""
          print,"  Chopping throw is 0.0, your photometric data is probably useless!"
          read, "  [Hit return] ",ans
      endelse
  end
  beam= he->getpar('INS OPT1')
  obj_destroy,im
  if (beam EQ 'HIGH_SENS') then begin
      print,"MIDISkySubImage: High_sens"
      data1 = chop_image[0].data1 - chop_image[1].data1
      data2 = chop_image[0].data2 - chop_image[1].data2
      return, { data1: data1, data2: data2 }

  endif else if (beam EQ 'SCI_PHOT') then begin
      print,"MIDISkySubImage: sci_phot"
      data1 = chop_image[0].data1 - chop_image[1].data1
      data2 = chop_image[0].data2 - chop_image[1].data2
      data3 = chop_image[0].data3 - chop_image[1].data3
      data4 = chop_image[0].data4 - chop_image[1].data4
      return, { data1: data1, data2: data2, data3: data3, data4: data4 }

  endif else begin
      print,"INS OPT1 is",beam
      print,"Neither HIGH_SENS nor SCI_PHOT, don't know where the data is!"
      return, -1
  endelse
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; this returns the name of the cached file, not the data!

Function oirChopPhotoImages, filename, DSKY=dsky

  if not keyword_set(dsky) then begin
      dsky=0 & dskyOpt= ""
  endif else dskyOpt= " -dSky " + strtrim(dSky,2)

  chopimgfile= "PHOTOMETRY/ChIm_" +MIDIname_datime(filename)+ "_dsky" +strtrim(dSky,2)+ ".fits"

  if not file_test(chopimgfile) then begin
      print, "Cached chop-frame does not exist, let's create it!"

      cmd = getenv("drsRoot") + '/c/bin/oirChopPhotoImages'
      if file_test(cmd,/EXECUTABLE) then begin
          if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"
          cmd= cmd + ' -in "' + filename + '" -out ' + chopimgfile + dskyOpt
;         begin CAH
          files = strsplit(filename,/extract)
          fitsfile = obj_new('fitsfile',files(0))
          prihead=fitsfile->prihead()
          insgris = prihead->getpar('INS GRIS')
          obj_destroy,fitsfile
          if strpos(insgris,'PRISM') ge 0 then curvefile=getenv('prismscurve')
          if strpos(insgris,'GRISM') ge 0 then curvefile=getenv('grismscurve')
          cmd = cmd + ' -ref '+ curvefile + ' -order 1'
;         end CAH
          if !MIAdebug ne 0 then print,cmd
          spawn,cmd,EXIT_STATUS=status
          if (status ne 0) then begin
              print,'oirChopPhotoImages failed'
              Log_error, filename, 'oirChopPhotoImages failed'
              return,'oirChopPhotoImages failed'
          endif
      endif else begin
          print,'The program'
          print,cmd
          print,'does not exist or is not executable.'
          print,"Please make sure MIA+EWS was installed properly."
          chopimgfile= "BAD INSTALL"
      endelse
  endif
  return, chopimgfile
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function getCompressSpec, fringefile, maskfile, comprfile, REUSE=reuse

  if keyword_set(reuse) and file_test(comprfile) then begin
      print, "Using existing ", comprfile
  endif else begin
      if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"
      print, ''
      cmd = getenv("drsRoot") + '/c/bin/oir1dCompressData'
      if file_test(cmd,/EXECUTABLE) then begin
          if file_test(comprfile) then file_delete, comprfile
          cmd = cmd + ' "' + fringefile +'" '+ maskfile +' '+ comprfile
          if !MIAdebug ne 0 then print,cmd
          spawn,cmd,EXIT_STATUS=status
          if (status ne 0) then begin
              print,'oir1dCompressData failed'
              Log_error, fringefile, 'oir1dCompressData failed'
              return,0
          endif
      endif else begin
          print, cmd + " not found, falling back on IDL code"
          return, oirCompressData(fringefile, maskfile, nDim=1)
      endelse
  endelse
  print, ''
  if !MIAdebug ne 0 then print, 'getting data...'
  return, oirgetdata(comprfile)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Helper-routine to provide dimension-keyword with old idl-versions
;
Function FFTdim,input,Dimension=dim

 if !Version.release ge 5.5 then return, FFT(input,Dimension=dim)

 ;print,"Using FFT-workaround for ancient IDL-versions"
 sz = size(input)
 output= complex(input)

 if sz[0] eq 2 then begin
     if dim eq 1 then begin
         for i=0,sz[2]-1 do  output[*,i]= FFT(input[*,i])
     endif else if dim eq 2 then begin
         for i=0,sz[1]-1 do  output[i,*]= FFT(input[i,*])
     endif else print,"No such dimension:",dim
 endif else if sz[0] eq 3 then begin
     if dim eq 1 then begin
         print,"dim 1 of 3"
         for i=0,sz[2]-1 do $
           for j=0,sz[3]-1 do  output[*,i,j]= FFT(input[*,i,j])
     endif else if dim eq 2 then begin
         print,"dim 2 of 3"
         for i=0,sz[1]-1 do $
           for j=0,sz[3]-1 do  output[i,*,j]= FFT(input[i,*,j])
     endif else if dim eq 3 then begin
         print,"dim 3 of 3"
         for i=0,sz[1]-1 do $
           for j=0,sz[2]-1 do  output[i,j,*]= FFT(input[i,j,*])
     endif else print,"No such dimension:",dim
 endif
 return,output
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; magic to get a usable colortable even on truecolor-displays
;
Function mia_std_colortable, truecolor

  if N_elements(truecolor) gt 1 then begin
      tvlct,indgen(256),indgen(256),indgen(256)
      device,decomposed=1
      return, truecolor
  endif else begin
      if N_elements(truecolor) eq 0 then begin
          ; truecolor-keyword not given, figure it out yourself
          ;print,"No of colors =",!D.N_colors
          if !D.N_colors gt 256 then truecolor=1
      endif
      if keyword_set(truecolor) then begin
          if !MIAdebug ne 0 then $
            print,"Using truecolor mode (",strtrim(!D.N_colors,2)," colors available)"
          tvlct,indgen(256),indgen(256),indgen(256)
          device,decomposed=1
          return, [ '0000ff'x, '00ff00'x, 'ff0000'x, 'f0f000'x, 'f000f0'x, '00f0f0'x ]
      endif else begin
          if !MIAdebug ne 0 then $
            print,"Using pseudocolor mode (",strtrim(!D.N_colors,2)," colors available)"
          tek_color, 0, 8
          return, [ 2, 3, 4, 5, 6, 7 ]
      endelse
  endelse
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Log errors somewhere safe
;
Pro Log_Error, fname, msg
  if not file_test("ERRORS") then file_mkdir,"ERRORS"
  fname = (oirFileList(fname))[0]
  first_char= Strpos(fname,'/',/Reverse_Search)
  last_char = Strpos(fname,'.fits')
  if first_char lt 0 then first_char=0
  fname= "ERRORS/"+STRMID(fname,first_char,last_char-first_char)+".log"

  openw, lun, fname, /get_lun,/append
  printf,lun,msg
  close, lun
  free_lun,lun
end
