;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
; $Id: mask_utilities.pro,v 1.30 2009/08/21 15:17:54 koehler Exp $
;
; mask_utilities.pro
; Created:     Tue Apr 19 16:12:49 2005 by Koehler@rommie
; Last change: Fri Aug 21 16:52:35 2009
;
; PURPOSE:
;	make the world a better place by masking the ugly parts
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO midisptrace__Define
foo = { midisptrace, $
         xsize: -1, ysize: -1, $
         trace: fltarr(320), FWHM: fltarr(320), fluxmax: fltarr(320),$
         range: fltarr(320), nRange: 0, $
         trace_coeff: fltarr(3), sigma_coeff: fltarr(2) $
}
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; the official way to compute the trace_fit.
; guaranteed free of Chesneau-code!
; NOTE: it does not do the actual fitting,
;	it only creates the fitted trace!
;
; CALLING SEQUENCE:
;	trace_fit = make_trace(x_size,trace_coeff,sigma_coeff)
;
; INPUT:
;	x_size		size in x-direction (length of trace)
;	trace_coeff	coefficients of trace-fit
;	sigma_coeff	coefficients of sigma-fit
;
; OUTPUT:
;	2D-array:
;	trace_fit[0,*]	trace
;	trace_fit[1,*]	trace-FWHM
;	trace_fit[2,*]	trace+FWHM
;
Function make_trace, x_size, trace_coeff, sigma_coeff

  tf = fltarr(3,x_size)
  tf[0,*]= poly(findgen(x_size),trace_coeff)
  FWHM   = poly(findgen(x_size),sigma_coeff) * sqrt(2.*alog(2))
  tf[1,*]= tf[0,*] - FWHM
  tf[2,*]= tf[0,*] + FWHM
  return, tf
end
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
PRO plot_trace, iFile, x_size, mtrace, COLORS=colors

	;    Fringes PhotA PhotB
  symbols= [ 2,	     1,    7	]

  rng = mtrace.range[0:mtrace.nrange]
  tr = make_trace(x_size,mtrace.trace_coeff,mtrace.sigma_coeff)

  if keyword_set(colors) then begin
      defcol = !P.color
      !P.color= colors[iFile]
      oploterr, rng, mtrace.trace[rng], mtrace.FWHM[rng], symbols[iFile]
      oplot, tr[0,*]
      oplot, tr[1,*]
      oplot, tr[2,*]
      !P.color= defcol
  endif else begin
      oplot, rng, mtrace.trace[rng], psym=symbols[iFile]
      oplot, tr[0,*], linestyle=iFile
      oplot, tr[1,*], linestyle=iFile
      oplot, tr[2,*], linestyle=iFile
  endelse
end
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
PRO plot_traces, mtraces, TITLE=title, COLORS=colors, POSITION=position, YRANGE=yrange

  nTraces= N_elements(mtraces)
  x_size = mtraces[0].xsize

  if not keyword_set(yrange) then begin
      mint = 999
      maxt = 0
      for i=0,nTraces-1 do begin
          if mtraces[i].trace_coeff[0] ne 0 then begin
              tr= make_trace(x_size, mtraces[i].trace_coeff, mtraces[i].sigma_coeff)
              mint = min([ mint, reform(tr[1,*]) ])
              maxt = max([ maxt, reform(tr[2,*]) ])
          endif
      endfor
      yrange = [mint,maxt]
  endif

  if keyword_set(position) then begin
      ;print,"Position is ",position
      plot, [0,x_size], yrange, TITLE=title, /nodata,$
        POSITION=position, /device, /noerase
  endif else begin
      plot, [0,x_size], yrange, TITLE=title, /nodata
  endelse

  XYOuts, x_size*0.03,yrange[0]+(yrange[1]-yrange[0])*0.93, $
    "Center = " + strtrim(string(mtraces[0].trace_coeff[0]),2) + $
      " + x * " + strtrim(string(mtraces[0].trace_coeff[1]),2) + $
    " + x^2 * " + strtrim(string(mtraces[0].trace_coeff[2]),2)

  XYOuts, x_size*0.03,yrange[0]+(yrange[1]-yrange[0])*0.85, $
    " FWHM = " + strtrim(string(mtraces[0].sigma_coeff[0]),2) + $
      " + x * " + strtrim(string(mtraces[0].sigma_coeff[1]),2)

  for i=0,nTraces-1 do $
    if mtraces[i].trace_coeff[0] ne 0 then begin
      ;if keyword_set(colors) then print,"trace",i,", color=",colors[i]
      plot_trace, i, x_size, mtraces[i], COLORS=colors
    endif
end
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; CAH: added from_trace keyword
;
Function make_mask_from_trace, trace, WIDTH=width, from_trace=from_trace

  if not keyword_set(width) then width=1.

  mask= fltarr( trace.xsize, trace.ysize)
  if keyword_set(from_trace) then begin
  trace_fit=trace.trace(0:trace.xsize-1)
  endif else begin
  trace_fit =  poly( findgen(trace.xsize), trace.trace_coeff)
  endelse
  sigm2_fit = (poly( findgen(trace.xsize), trace.sigma_coeff) * width)^2 * 2.
  y = findgen(trace.ysize)
  FOR i= 0, trace.xsize-1 DO BEGIN
      expos =  -(y-trace_fit[i])^2. / sigm2_fit[i]
      j = where(abs(expos) lt 10)
      mask[i,j]= exp(expos(j))	; CAH: to avoid underflow
  endfor
  return, mask
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function default_mask, winin, SCIPHOT=sciphot, GRISM=grism

  win= winin	;; do not mess with the parameter passed down to us!

  if keyword_set(grism) and (grism eq "GRISM") then begin
      ;;print,"Grism detected"
      if keyword_set(sciphot) then begin
          case win of
              0: return, { trace_coeff: [  8.75,  0.178, -0.00046 ],$
                           sigma_coeff: [  2.30, -0.00332 ] }
              1: return, { trace_coeff: [ 10.36,  0.077, -0.00017 ],$
                           sigma_coeff: [  1.70, -0.00046 ] }
              2: return, { trace_coeff: [ 17.05, -0.0177, 0.00012 ],$
                           sigma_coeff: [  1.45,  0.00165 ] }
              3: return, { trace_coeff: [ 21.49, -0.1148,  0.00043 ],$
                           sigma_coeff: [  1.84, -0.00042 ] }
              Else: return, "mask undefined for this window"
          endcase
      endif else begin
          case win of
              0: return, { trace_coeff: [ 11.74,  0.074, -0.00016 ],$
                           sigma_coeff: [  1.78, -0.0015 ] }
              1: return, { trace_coeff: [ 18.03, -0.013,  0.00010 ],$
                           sigma_coeff: [  1.24,  0.0013 ] }
              ;; EWS-mask:
              ;;1: return, { trace_coeff: [ 11.1330,  0.0844535, -0.000184444 ],$
              ;;             sigma_coeff: [ 2.52734, -0.00167005 ] }
              ;;2: return, { trace_coeff: [ 19.6048, -0.0145668,  0.000102877 ],$
              ;;             sigma_coeff: [ 2.24500, -0.00021343 ] }
              Else: return, "mask undefined for this window"
          endcase
      endelse
  endif else begin
      ;;print,"Not Grism"
      if keyword_set(sciphot) then begin
          case win of
              0: return, { trace_coeff: [ 13.52, 0.055,-0.00033 ],$
                           sigma_coeff: [  2.52,-0.0080 ] }
              1: return, { trace_coeff: [ 14.24, 0.015, -7e-5 ],$
                           sigma_coeff: [  2.41,-0.0080 ] }
              2: return, { trace_coeff: [ 15.98,-0.0032, 7e-5 ],$
                           sigma_coeff: [  2.35,-0.0076 ] }
              3: return, { trace_coeff: [ 16.54,-0.034, 0.00028 ],$
                           sigma_coeff: [  2.14,-0.0064 ] }
              Else: return, "mask undefined for this window"
          endcase
      endif else begin
          case win of
              0: return, { trace_coeff: [ 16.10, 0.016,-9e-5 ],$
                           sigma_coeff: [  2.42,-0.0065 ] }
              1: return, { trace_coeff: [ 17.30, 0.000, 3e-5 ],$
                           sigma_coeff: [  2.42,-0.0065 ] }
              ;; EWS-mask:
              ;;1: return, { trace_coeff: [ 17.2924, 0.0188047, -0.000111631 ],$
              ;;             sigma_coeff: [ 3.21756, 6.09195e-05 ] }
              ;;2: return, { trace_coeff: [ 19.2461,-0.00798039, 7.58052e-05 ],$
              ;;             sigma_coeff: [ 3.20798,-1.30944e-05 ] }
              Else: return, "mask undefined for this window"
          endcase
      endelse
  endelse
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; CAH: added t1 field

PRO miamask__Define
xx = { miamask, $
       fringefile: '', photfiles: ['',''], maskfile: '', $
       trace: replicate({midisptrace},2,4),$	 ;;; 1.index=file, 2.index=window
       traceFr: replicate({midisptrace},4),$	 ;;; 4 windows for SCI_PHOT
       traceFr0: replicate({midisptrace},4),$	 ;;; initial values
       fringemean: ptr_new(), photAmean: ptr_new(), photBmean: ptr_new(),$
       factor: 1., color: lonarr(6), formID: -1, drawID: -1, $
       BgImg: 2, Window: 1, Lines: [1,1,1], $
       maskfrom: 0, TraceFix: [0,0,0], FWHMFix: [0,0,0], $
       grism: '', beam: '', t1: '', width: 1., dSky: 0, ErrorLog: '' $
}
end

;; maskfrom: 0=fixed, 1=y0 only, 2=full fit, 3=file
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
Function miamask::init, fringefile, photfiles, WIDTH=width, dSky=dSky, $
                	MASKFILE=maskfile, ERRORLOG=errorlog, NOEWSPHOT=noewsphot, $
                	FITMASK=fitmask

  if keyword_set(width) then begin
      print,"Maskwidth is",width
      self.width = width
  endif else self.width= 1.
  if keyword_set(errorlog) then self.errorlog= errorlog
  if keyword_set(dsky) then self.dSky=dsky else self.dsky=0

  if N_elements(fitmask) lt 1 then fitmask=1
  ;; default is to fit the mask, unless caller says something

  ;print,"miamask: errorlog is:_"+errorlog+"_ = "+string(N_elements(errorlog))

  nWin = 0	; 1.file tells us if we have HAI-Sense or SY-Phot

  ;print,"Fringes: ",fringefile
  self.fringefile= fringefile
  self.fringemean= ptr_new(Cached_MeanRMS(self.fringefile), /no_copy)

  ;;; SCI MODE by FP (under development):
  if N_params() lt 2 then begin    ;;; modified for sci_mode (=no photfiles)
      print,'No photometry files given, entering SCI MODE by FP (under development)'
      self.photfiles[0] = self.fringefile
      self.photfiles[1] = self.fringefile
      nWin = 4                  ; puke if files are not SCI_photic enough
  endif else begin
      self.photfiles = photfiles
  endelse

  if N_elements(self.photfiles) lt 2 then begin
      Log_error, self.photfiles[0], "miamask needs a second photometry file"
      if N_Elements(errorlog) gt 0 then begin
          errorlog = errorlog + "miamask needs two photometry files, but got only one"
      endif else begin
          print,"miamask needs two photometry files, but got only one!"+string(7B)
      endelse
      return,0
  endif
  if N_elements(self.photfiles) gt 2 then begin
      Log_error, self.photfiles[2], "miamask uses only two photometry files"
      print,"miamask uses only two photometry files"
  endif

  if keyword_set(maskfile) then begin
      self.maskfrom= 3  &  self.maskfile = maskfile
  endif else begin
      self.maskfrom= (keyword_set(fitmask)) ? fitmask : 0
      self.maskfile= 'PHOTOMETRY/Mask' + $
        "_" + MIDIname_datime(self.photfiles[0]) + $
        "+" + MIDIname_time(self.photfiles[1]) + ".fits"
  endelse

  datatag = intarr(4)

  for iFile= 0, 1 do begin
      if !MIAdebug ne 0 then begin
          print, ""
          print, "========== Reading photometry",iFile," =========="
          print, self.photfiles[iFile]
      endif
      if keyword_set(noewsphot) then begin
          chop_image= MIDISkySubImage(self.photfiles[iFile], before=1, after=1, ERRORLOG=errorlog)
      endif else begin
          chop_image= (oirgetdata(oirChopPhotoImages(self.photfiles[iFile],DSKY=dsky)))[0]
      endelse

      tagnms= Tag_Names(chop_image)
      datatag[0] = where(tagnms eq "DATA1")
      datatag[1] = where(tagnms eq "DATA2")
      datatag[2] = where(tagnms eq "DATA3")
      datatag[3] = where(tagnms eq "DATA4")

      if nWin eq 0 then begin
          if datatag[3] eq -1 then nWin=2 else nWin=4
      endif
      if !MIAdebug gt 1 then print,nWin," Windows at tags",datatag

      if (datatag[0] lt 0) or (datatag[1] lt 0) or (nWin ge 4 and datatag[2] lt 0) then begin
          print,"Can't find enough windows, I'm confused!"
          return,0
      endif

      ;; this destroys chop_image and must therefore be at the end
      if iFile eq 0 then $
        self.photAmean= ptr_new(chop_image,/no_copy) $
      else if iFile eq 1 then $
        self.photBmean= ptr_new(chop_image,/no_copy)
;     CAH
      if self.photfiles[1] eq self.photfiles[0] then begin
        self.photBmean= ptr_new(chop_image,/no_copy)
	break
      endif
  endfor	;; iFile

; CAH: this block followed the if-else below
  im = obj_new('imagedata',(oirFileList(self.photfiles[0]))[0])
  he = im->prihead()
  self.grism= strcompress(he->getpar('INS GRIS NAME'),/remove_all)
  self.beam = strcompress(he->getpar('INS OPT1 ID'),/remove_all)
  self.t1 = strmid(strcompress(he->getpar('ISS CONF T1NAME'),/remove_all),0,2)
  obj_destroy,im
  obj_destroy,he

  if keyword_set(fitmask) then begin
      self->fit_spectra, Y0_ONLY=(fitmask eq 1)
  endif else begin
      for iWin= 0,nWin-1 do self.trace[0,iWin].trace_coeff[0]= 0.
  endelse

  print,""
  if !MIAdebug ne 0 then $
      print,"========== traces and widths in fringe data ========="

  iFile= N_elements(self.photfiles)
  sz = size( (*self.photAmean).data1)	;; sizes of windows should better be identical
  for iWin= 0,nWin-1 do begin
      if !MIAdebug ne 0 then $
        print,"----- window",iWin," = ",tagnms[datatag(iWin)],"-----"

      if not keyword_set(fitmask) then begin
          co= 0
          if nWin le 2 then begin
              co= default_mask(iWin,GRISM=self.grism)
          endif else $
            co= default_mask(iWin,GRISM=self.grism,/SCIPHOT)

          if size(co,/type) eq 8 then begin
              self.traceFr[iWin].trace_coeff = co.trace_coeff
              self.traceFr[iWin].sigma_coeff = co.sigma_coeff * self.width
          endif
          if !MIAdebug ne 0 then begin
              print,"trace coeff: ",self.traceFr[iWin].trace_coeff
              print,"sigma coeff: ",self.traceFr[iWin].sigma_coeff
          endif
      endif

      szi= size( (*self.photAmean).(datatag[iWin]))
      ;;print,"size of ",tagnms[iWin]," =",szi
      if (sz[1] ne szi[1]) or (sz[2] ne szi[2]) then begin
          print,"Sizes of windows not equal, I can't handle this!"
          return,0
      endif
      self.traceFr[iWin].xsize= sz[1]
      self.traceFr[iWin].ysize= sz[2]
      self.traceFr[iWin].trace= poly(findgen(sz[1]),self.traceFr[iWin].trace_coeff)
      self.traceFr[iWin].FWHM = poly(findgen(sz[1]),self.traceFr[iWin].sigma_coeff)*sqrt(2.*alog(2.))
      self.traceFr[iWin].nRange= sz[1]
      self.traceFr[iWin].range = findgen(sz[1])
  endfor
  self.traceFr0= self.traceFr
  return,1
end

function miamask, files, WIDTH=width, MASKFILE=maskfile, DSKY=dsky
      return, obj_new('miamask',files[0],files[1:2],WIDTH=width,MASKFILE=maskfile, DSKY=dsky)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask::gui, TRUECOLOR=truecolor, NO_BLOCK=no_block

  self.color = mia_std_colortable(truecolor)

  screensz= get_screen_size()
  datasz  = size( (*self.fringemean).data1)
  self.factor= floor( (screensz[0]-150)/datasz[1] < (screensz[1]-210)/(datasz[2]*2) < 7)
  ; controls are >300pix on rommie
  if !MIAdebug ne 0 then $
    print,FORMAT='("screen size: ", I4," x ",I4,", data size: ",I3," x ",I2," => enlarging by factor",I2)',$
    screensz, datasz[[1,2]], self.factor

  sz= size( (*self.fringemean)[1].data1) * self.factor
  sz[2] = sz[2]*2

  !X.style=1	; make sure the plots are exactly the size we want!
  !Y.style=1

  self.BgImg = 0	; set defaults to same values as in cw_form!
  self.Window= 0
  self.Lines = [1,1,1]
  self.TraceFix= [0,0,0]
  self.FWHMFix = [0,0,0]

  base = Widget_Base(Title="Maskedit",/COLUMN)
  desc = [ $
; CAH    '1, BASE, ,COLUMN, FRAME', $
; CAH      '0, FLOAT, 0.0, LABEL_TOP=Scale Min,Width=6,TAG=Scale_min',$
; CAH      '2, FLOAT, 0.0, LABEL_TOP=Scale Max,Width=6,TAG=Scale_max',$
; CAH: add ROW base because we changed the general layout to COLUMN
         '1, BASE, ,ROW', $
         '0, BUTTON, Fringes|Photometry A|Photometry B, Column, Frame,'+$
	        'LABEL_TOP=Background Img, SET_VALUE=0,EXCLUSIVE,TAG=BgImg',$
           ;; make sure the following line gets altered for SciPhot-mode
         '0, BUTTON, Window 1|Window 2, Column, Frame,'+$
		'LABEL_Top=Detector Win,SET_VALUE=0,EXCLUSIVE,TAG=Window',$
         '0, BUTTON, Fringes|Photometry A|Photometry B, Column, Frame,'+$
		'LABEL_TOP=Lines plotted, SET_VALUE=[1\,1\,1], TAG=Lines',$
         '0, BUTTON, Fixed|Fit y0|Full Fit|File, Column, Frame,'+$
                'LABEL_TOP=Mask from:, SET_VALUE='+string(self.maskfrom)+',EXCLUSIVE,TAG=MaskFrom',$
        '1, BASE, ,COLUMN',$
         '1, BASE, ,ROW, Frame',$
          '1, BASE, ,COLUMN',$
           '0, LABEL, Trace Coefficients', $
            $;'1, BASE, ,COLUMN, FRAME',$
            '1, BASE, ,ROW', $
             '0, FLOAT,'+string(self.traceFr[0].trace_coeff[0])+',LABEL_LEFT=Trace=,Width= 4,TAG=Trace0', $
             '0, FLOAT,'+string(self.traceFr[0].trace_coeff[1])+',LABEL_LEFT=+ x*,  Width= 6,TAG=Trace1', $
             '2, FLOAT,'+string(self.traceFr[0].trace_coeff[2])+',LABEL_LEFT=+ x*,Width=8,TAG=Trace2',$
            $;'2, BUTTON, Fixed        |Fixed            |Fixed, ROW, RIGHT, LABEL_LEFT= Fit :,TAG=TraceFix',$
           '2,BASE',$
          '1, BASE, ,COLUMN',$
           '0, LABEL, Width Coefficients',$
            $;'1, BASE, ,COLUMN, FRAME',$
            '1, BASE, ,ROW', $
             '0, FLOAT,'+string(self.traceFr[0].sigma_coeff[0])+',LABEL_LEFT=FWHM=,Width= 4,TAG=FWHM0',$
             '2, FLOAT,'+string(self.traceFr[0].sigma_coeff[1])+',LABEL_LEFT=+ x*, Width= 6,TAG=FWHM1',$
            $;'2, BUTTON, Fixed         |Fixed, ROW, RIGHT, LABEL_LEFT= Fit:,TAG=FWHMFix',$
           '2,BASE',$
          $;'1, BASE, ,COLUMN',$
           $;'0, BUTTON, Fit Parameters, TAG=FIT',$
           '2, BUTTON, Reset, TAG=RESET',$
	  '1, BASE, ,ROW, Frame',$
           '0, TEXT, '+self.maskfile+', Width=58, LABEL_LEFT=Filename:, TAG=Filename',$
           '0, BUTTON,  Browse , TAG=BROWSE',$
           '2, BUTTON,  EWS-Mask , TAG=EWS',$
          '2, BASE',$
	  '2, BASE']
;         '2, BASE',$   ; CAH: add another BASE termination because of additional ROW base
;         '0, BUTTON, Dismiss GUI, TAG=CREATE' ]

  if where( Tag_Names(*self.photAmean) eq "DATA4") ge 0 then begin
      ;; absolute indices are a bad hack, of course
;     CAH: index was "4", changed to 2
      desc[2] = '0, BUTTON, Window 1|Window 2|Window 3|Window 4, Column, Frame,'+$
		  'LABEL_Top=Detector Window,SET_VALUE=0,EXCLUSIVE,TAG=Window'
      ;; this would require changing all the trace_coeffs, too:
      ;;self.Window= 1
  endif

  ;left of draw: form = cw_form(base,desc,/column)
; CAH: add row wid for Dismiss button in new location
  row_wid=widget_base(base,/row)
  draw = Widget_Draw(row_wid, XSIZE=sz[1]+150, YSIZE=sz[2]+50, $
;	CAH: add call back for interactive scaling
	/BUTTON_EVENTS,/MOTION_EVENTS,event_pro = 'mia_draw_motion_event')
  if self.grism eq 'GRISM' then $
  dismiss = widget_button(row_wid,value='Dismiss'+string(10b)+'GUI',uvalue='Dismiss',event_pro='miamask_event')
  row_wid=widget_base(base,/row)
  self.formID= cw_form(row_wid,desc);,/column)
  if self.grism eq 'PRISM' then $
  dismiss = widget_button(row_wid,value='Dismiss'+string(10b)+'GUI',uvalue='Dismiss',event_pro='miamask_event')

  WIDGET_CONTROL, base, /REALIZE
  WIDGET_CONTROL, base, SET_UVALUE=self
  WIDGET_CONTROL, draw, GET_VALUE=drawID & self.drawID= drawID

  self->draw, /SET_MINMAX

  xmanager, 'miamask', base, NO_BLOCK=no_block
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; New functionality add by CAH
pro mia_draw_motion_event,event
;
common LocalMiaDrawColorEvent,cstretch
;
if n_elements(cstretch) eq 0 then cstretch=0
;
case event.type of
    0: begin           ; button press
        if (event.press EQ 1) then begin
            cstretch = 1
            mia_stretchct, event
        endif else return
    end
    1: begin
        cstretch = 0  ; button release
        return
    end
    2: begin                ; motion event
       if (cstretch EQ 1) then begin
            mia_stretchct, event
       endif else return
    end
endcase
;
WIDGET_CONTROL, event.TOP, GET_UVALUE=m
m->draw
;
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; New functionality add by CAH
pro mia_stretchct,event
;
brightness=float(event.x)/!d.x_size
contrast=float(event.y)/!d.y_size
;
r=bindgen(256)
g=bindgen(256)
b=bindgen(256)
;
x = brightness*(256-1)
y = contrast*(256-1) > 2   ; Minor change by AJB
high = x+y & low = x-y
diff = (high-low) > 1
;
slope = float(256-1)/diff ;Scale to range of 0 : nc-1
intercept = -slope*low
p = long(findgen(256)*slope+intercept) ;subscripts to select
tvlct, r[p], g[p], b[p]
;
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; New function added by CAH
function wcscurve,insgris,iwin
;
if strpos(insgris,'PRISM') ge 0 then curvefile=getenv('prismscurve')
if strpos(insgris,'GRISM') ge 0 then curvefile=getenv('grismscurve')
;
det=OBJ_NEW('imageDetector',curvefile)
dt=det->table()
region = 1
ports  = 1
correlation  = 1
naxis = (*dt)[1].naxis
naxis[1,0] = 1
corner = (*dt)[1].corner
crVal  = (*dt)[1].crVal
crPix  = (*dt)[1].crPix
cType  = (*dt)[1].cType
cD     = (*dt)[1].cD
dmp     = (*dt)[1].dmp
dmc     = (*dt)[1].dmc
;
cod=OBJ_NEW('CoordDistortion',det,region=iwin+1)
cod->coordgrid,xc,yc
r=size(xc)
nx=r(1)
ny=r(2)
;
det->close
OBJ_DESTROY,det
OBJ_DESTROY,cod
;
return,crpix(1)-yc(*,crpix(1))
;
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


PRO miamask_event, ev
  ;print,"event1!"
  WIDGET_CONTROL, ev.TOP, GET_UVALUE=m
  m->event, ev
end

PRO miamask::event, ev

  if ev.id eq self.formID then begin
      ;print,"form event, tag: ",ev.tag,", value: ",ev.value

      case (ev.tag) of
          "SCALE_MIN": self->draw
          "SCALE_MAX": self->draw
          "BGIMG": if self.BgImg ne ev.value then begin
              self.BgImg= ev.value
              self->draw, /SET_MINMAX
          endif
          "WINDOW": if self.Window ne ev.value then begin
              self.Window= ev.value
              ; they changed the window, reload all the coeffs
              Widget_Control, ev.ID, Get_value=v
              v.Trace0= self.traceFr[self.window].trace_coeff[0]
              v.Trace1= self.traceFr[self.window].trace_coeff[1]
              v.Trace2= self.traceFr[self.window].trace_coeff[2]
              v.FWHM0 = self.traceFr[self.window].sigma_coeff[0]
              v.FWHM1 = self.traceFr[self.window].sigma_coeff[1]
              Widget_Control, ev.ID, Set_value=v
              self->draw, /SET_MINMAX
          endif
          "LINES": $
            if (self.Lines[0] ne ev.value[0]) $
            or (self.Lines[1] ne ev.value[1]) $
            or (self.Lines[2] ne ev.value[2]) then begin
              self.Lines= ev.value
              self->draw
          endif
          "MASKFROM": if self.maskfrom ne ev.value then begin
              print,"Maskfrom: ",ev.value
              if ev.value eq 3 then begin
                  if file_test(self.maskfile,/read,/regular) then begin
                      self.maskfrom= ev.value
                      self->draw, /SET_MINMAX
                  endif else begin
                      self.maskfrom= 0
                      Widget_Control, ev.ID, Get_value=v
                      v.Maskfrom = 0
                      Widget_Control, ev.ID, Set_value=v
                      msg= dialog_message(["A file named",$
                                           self.maskfile,$
                                           "does not exist, is not a regular file,",$
                                           "or you don't have permission to read it.",$
                                           "You can't use that file as mask."],$
                                          /ERROR, Dialog_Parent=ev.top)
                  endelse
              endif else begin
                  self.maskfile= 'PHOTOMETRY/Mask' + $
                    "_" + MIDIname_datime(self.photfiles[0]) + $
                    "+" + MIDIname_time(self.photfiles[1]) + ".fits"
                  if ev.value ge 1 then begin
                      self.maskfrom = ev.value
                      self->fit_spectra, Y0_ONLY=(ev.value eq 1)
                  endif else begin
                      self.maskfrom = 0
                      if where( Tag_Names(*self.photAmean) eq "DATA4") ge 0 then begin
                          for iWin= 0,3 do begin
                              co= default_mask(iWin,GRISM=self.grism,/SCIPHOT)
                              if size(co,/type) eq 8 then begin
                                  self.traceFr[iWin].trace_coeff = co.trace_coeff
                                  self.traceFr[iWin].sigma_coeff = co.sigma_coeff
                              endif
                          endfor
                      endif else begin
                          for iWin= 0,1 do begin
                              co= default_mask(iWin,GRISM=self.grism)
                              if size(co,/type) eq 8 then begin
                                  self.traceFr[iWin].trace_coeff = co.trace_coeff
                                  self.traceFr[iWin].sigma_coeff = co.sigma_coeff
                              endif
                          endfor
                      endelse
                  endelse
              endelse
              Widget_Control, ev.ID, Get_value=v
              v.Filename= self.maskfile
              v.Trace0= self.traceFr[self.window].trace_coeff[0]
              v.Trace1= self.traceFr[self.window].trace_coeff[1]
              v.Trace2= self.traceFr[self.window].trace_coeff[2]
              v.FWHM0 = self.traceFr[self.window].sigma_coeff[0]
              v.FWHM1 = self.traceFr[self.window].sigma_coeff[1]
              Widget_Control, ev.ID, Set_value=v
              self->draw, /SET_MINMAX
          end
          ;;;;;;;;;;;;;;;;;;;; Coefficients ;;;;;;;;;;;;;;;;;;;;
          "TRACE0": begin
              ;; perverted logic: if the user types or deletes a
              ;; digit, the value changes, but if he hits return, the
              ;; value does NOT change.  So we draw on non-changes...
              ;;if self.traceFr[self.window].trace_coeff[0] eq ev.value then self->draw
              self.traceFr[self.window].trace_coeff[0]= ev.value
              if self.lines[0] eq 0 then $
                print,"Warning: Plot is off. You won't see what you changed!",string(7B)
              self->draw
          end
          "TRACE1": begin
              ;;if self.traceFr[self.window].trace_coeff[1] eq ev.value then self->draw
              self.traceFr[self.window].trace_coeff[1]= ev.value
              if self.lines[0] eq 0 then $
                print,"Warning: Plot is off. You won't see what you changed!",string(7B)
              self->draw
          end
          "TRACE2": begin
              ;;if self.traceFr[self.window].trace_coeff[2] eq ev.value then self->draw
              self.traceFr[self.window].trace_coeff[2]= ev.value
              if self.lines[0] eq 0 then $
                print,"Warning: Plot is off. You won't see what you changed!",string(7B)
              self->draw
          end
          "TRACEFIX":
          "FWHM0": begin
              ;;if self.traceFr[self.window].sigma_coeff[0] eq ev.value then self->draw
              self.traceFr[self.window].sigma_coeff[0]= ev.value
              if self.lines[0] eq 0 then $
                print,"Warning: Plot is off. You won't see what you changed!",string(7B)
              self->draw
          end
          "FWHM1": begin
              ;;if self.traceFr[self.window].sigma_coeff[1] eq ev.value then self->draw
              self.traceFr[self.window].sigma_coeff[1]= ev.value
              if self.lines[0] eq 0 then $
                print,"Warning: Plot is off. You won't see what you changed!",string(7B)
              self->draw
          end
          "FWHMFIX":
          "RESET": begin
              self.traceFr= self.traceFr0
              Widget_Control, ev.ID, Get_value=v
              v.Trace0= self.traceFr[self.window].trace_coeff[0]
              v.Trace1= self.traceFr[self.window].trace_coeff[1]
              v.Trace2= self.traceFr[self.window].trace_coeff[2]
              v.FWHM0 = self.traceFr[self.window].sigma_coeff[0]
              v.FWHM1 = self.traceFr[self.window].sigma_coeff[1]
              Widget_Control, ev.ID, Set_value=v
              self->draw
          end
          ;;;;;;;;;;;;;;;;;;;;;; file ;;;;;;;;;;;;;;;;;;;;;;
          "FILENAME": self.maskfile= ev.value
          "BROWSE": begin
              maskfile = dialog_pickfile(Dialog_Parent=ev.top, /Must_Exist,$
                                         Default_Extension= ".fits",$
                                         Filter = "*Mask_*.fits", $
                                         File = file_basename(self.maskfile),$
                                         Path = file_dirname(self.maskfile),$
                                         Title= "Select a Fits-file containing a MIDI-mask")
              if maskfile ne "" then begin
                  self.maskfile= maskfile
                  self.maskfrom= 3
                  Widget_Control, ev.ID, Get_value=v
                  v.Filename= self.maskfile
                  v.MaskFrom= self.maskfrom
                  Widget_Control, ev.ID, Set_value=v
                  self->draw
              endif
          end
          "EWS": begin
              if self.grism eq "PRISM" then begin
                  envar= (self.beam eq 'HIGH_SENS') ? 'prismhmask' : 'prismsmask'
              endif else begin
                  envar= (self.beam eq 'HIGH_SENS') ? 'grismhmask' : 'grismsmask'
              endelse
              self.maskfile= getenv(envar)
              self.maskfrom= 3
              Widget_Control, ev.ID, Get_value=v
              v.Filename= self.maskfile
              v.MaskFrom= self.maskfrom
              Widget_Control, ev.ID, Set_value=v
              self->draw
          end
          "CREATE": Widget_control, ev.top, /Destroy	; self->destruct
      endcase
  endif
  widget_control, ev.id, get_uvalue=uvalue
  if n_elements(uvalue) eq 0 then uvalue=''
  if uvalue eq 'Dismiss' then $
  Widget_control, ev.top, /Destroy	; self->destruct
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask::draw, SET_MINMAX=set_minmax

  ;;print,"BgImg",self.BgImg,", Window",self.window,", Lines",self.lines
  case self.BgImg of
      0: img= (*self.fringemean)[1]
      1: img= (*self.photAmean)[0]
      2: img= (*self.photBmean)[0]
  endcase
  if self.maskfrom eq 3 and file_test(self.maskfile,/read,/regular) then begin
      mask = oirgetdata(self.maskfile)
      ;;help,mask,/str
      img.data1 = img.data1 * mask.data1
      img.data2 = img.data2 * mask.data2
  endif
  case self.Window of
      0: img = img.data1
      1: img = img.data2
      2: img = img.data3
      3: img = img.data4
  endcase
  if self.BgImg eq 0 then img= sqrt(img)

; CAH: commented following lines
; Widget_Control, long(self.formID), Get_value=v
; if keyword_set(set_minmax) then begin
;     scale_min= min(img, Max=scale_max)
;     v.scale_min = scale_min
;     v.scale_max = scale_max
;     Widget_Control, long(self.formID), Set_value=v
; endif
; CAH: add this instead
  v={scale_min:min(img),scale_max:max(img)}
  sz= size( img) * self.factor
  sz[2] = sz[2]*2

  img = (!D.TABLE_SIZE-1) * (img-v.scale_min) / (v.scale_max-v.scale_min)
  w = where(img lt 0)               & if w[0] ne -1 then img[w]= 0
  w = where(img gt !D.TABLE_SIZE-1) & if w[0] ne -1 then img[w]= !D.TABLE_SIZE-1

; CAH begin
  tvlct,r,g,b,/get
  img=r(img)
; CAH end

  wset,self.drawID
  tv, rebin(byte(img), sz[1], sz[2], /sample), 30,30
  position= [30, 30, 30+sz[1], 30+sz[2]]
  yrange = [-0.5, sz[2]/self.factor/2-0.5]

  ;; plot frame of coordinate axes
  plot, [0,sz[1]/self.factor], yrange, POSITION=position,/nodata,/device,/noerase

  ;; colorbar
  tv, bytarr(100,sz[2]), 50+sz[1],30	;; clean the blackboard first
  axis, 50+sz[1]+24,30, /device,/yaxis,yrange=[v.scale_min,v.scale_max],yticklen=-0.01
  tv, r(byte(replicate(1,24)) # indgen(sz[2])*256/sz[2]), 50+sz[1],30	; CAH: embed in r vector

  if self.maskfrom le 2 then begin
     traces= [ self.traceFr[self.window], self.trace[0,self.window], self.trace[1,self.window] ]
     index = where(self.Lines ne 0 and traces.xsize gt 0,count)	; CAH: add xsize test
     if count gt 0 then begin
        plot_traces, traces[index], POSITION=position, COLORS=self.color[index], YRANGE=yrange
      endif else begin  ; CAH
        plot, [0,sz[1]/self.factor], yrange, POSITION=position,/nodata,/device,/noerase
      endelse
  endif else begin ; CAH
      plot, [0,sz[1]/self.factor], yrange, POSITION=position,/nodata,/device,/noerase
  endelse

  if self.dsky lt yrange[1]/2 then begin
;     CAH: fixed sky windows are obsolete!
      if 0 then begin
      if self.grism eq 'PRISM' then begin
          top= 25 & bottom= 9
      endif else begin
          top= 29 & bottom= 6
      endelse
      top += self.dsky
      if top gt yrange[1]-2. then top= yrange[1]-2.
      oplot,[0,sz[1]/self.factor], [top-2.5, top-2.5], color=self.color[3]
      oplot,[0,sz[1]/self.factor], [top+2.5, top+2.5], color=self.color[3]

      bottom -= self.dsky
      if bottom lt 2. then bottom= 2.
      oplot,[0,sz[1]/self.factor], [bottom-2.5,bottom-2.5], color=self.color[3]
      oplot,[0,sz[1]/self.factor], [bottom+2.5,bottom+2.5], color=self.color[3]
      endif
;
;     CAH: new code
      if self.BgImg ge 1 then begin	; Only plot sky windows with photometric backgrounds
        top=wcscurve(self.grism,self.window)
        mytrace=self.trace(self.BgImg-1,self.window)
        if mytrace.xsize gt 0 then begin
          mytrace.trace(0:mytrace.xsize-1)=top
          mymask=make_mask_from_trace(mytrace,/from_trace)
	  maxshift=4
          v=fltarr(maxshift*2+1)
          for i=-maxshift,maxshift do $
		v(i+maxshift)=total(img(*,maxshift:mytrace.ysize-maxshift) $
		 *shift(mymask(*,maxshift:mytrace.ysize-maxshift),1,-i))
          i=where(v eq max(v)) & i=i(0)-maxshift
	  if i gt mytrace.ysize/2 then i=i-mytrace.ysize
;         oplot,findgen(n_elements(top)), top-i, color=self.color[3],psym=0,thick=1
;	  The following variables DSKY.. and SIGMA are found in oirChopPhotoImages
	  if self.t1 eq 'UT' then DKYDEF=10 else DSKYDEF=6
	  if self.t1 eq 'UT' then SIGMA=2.5 else SIGMA=1.0
	  spix=sqrt(alog(0.5)/(-0.5))*SIGMA
          oplot,findgen(n_elements(top)), top-i+DSKYDEF+self.dsky+spix, color=self.color[5],psym=0,thick=1
          oplot,findgen(n_elements(top)), top-i+DSKYDEF+self.dsky-spix, color=self.color[5],psym=0,thick=1
          oplot,findgen(n_elements(top)), top-i-DSKYDEF-self.dsky+spix, color=self.color[5],psym=0,thick=1
          oplot,findgen(n_elements(top)), top-i-DSKYDEF-self.dsky-spix, color=self.color[5],psym=0,thick=1
	endif
      endif

  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro miamask::show_traces, fname, NOWIN=nowin, COLORS=colors

  if (N_Params() gt 0) then begin
      set_plot,"ps"
      device,file=fname,/landscape,/color
      nowin= 1
  endif
  if not keyword_set(nowin) then window,3,title="Traces"
  if not keyword_set(colors) then colors= self.color

  nWin= N_Tags(*self.photAmean)
  sz  = size((*self.photAmean).data1)
  !P.multi=[0,1,nWin]

  for i= 0,nWin-1 do begin
    plot_traces, [self.traceFr[i],self.trace[0,i],self.trace[1,i]],$
	title="Window "+strtrim(string(i+1),2), COLORS=colors, yrange=[-0.5,sz[2]-0.5]
    rng = self.traceFr[i].range[0:self.traceFr[i].nrange]
    oplot, rng, self.traceFr[i].trace[rng] + self.traceFr[i].FWHM[rng], psym=3
    oplot, rng, self.traceFr[i].trace[rng] - self.traceFr[i].FWHM[rng], psym=3
  endfor

  if (N_params() gt 0) then begin
      device,/close
      set_plot,"X"
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro miamask::fit_window, win, t_order, w_order, mtrace,$
           DTRACE=dtrace, XMIN=xmin,XMAX=xmax, BGNOISE=bgnoise

common MidiOptions,midi_options,maskfile_mymidi,kappafile,photofile,la_bins ; CAH

  if n_elements(midi_options) eq 0 then midi_options={m:0} ; CAH

  sz = size(win)
  x_size= sz[1]
  y_size= sz[2]
  if !MIAdebug ne 0 then print,"window is ",sz[1:sz[0]]
  if not keyword_set(bgnoise) then begin
      bgnoise = mean(win(*,sz[2]*0.8:sz[2]-1))
      if (bgnoise lt 0.0) then bgnoise= 0.0
  endif
  mtrace = {midisptrace}
  mtrace.xsize= x_size
  mtrace.ysize= y_size
  mtrace.trace_coeff = [0,0,0]	;; make sure there is something
  mtrace.sigma_coeff = [0,0]

  sigma  = fltarr(x_size)
  column = dindgen(y_size)

  if not keyword_set(xmin) then xmin= 0
  if not keyword_set(xmax) then xmax= x_size-1

; We now find the spectrum in a similar way to how Walter does it
  top=wcscurve(self.grism,self.window)
  mtrace.trace(0:mtrace.xsize-1)=top
  ctrace= default_mask(self.window,GRISM=self.grism,SCIPHOT=(self.beam eq 'SCI_PHOT'))
  mtrace.trace_coeff=ctrace.trace_coeff
  mtrace.sigma_coeff=ctrace.sigma_coeff
  mymask=make_mask_from_trace(mtrace,/from_trace)
  maxshift=4	; Same as in oirChopPhotoImages
  v=fltarr(maxshift*2+1)
  for i=-maxshift,maxshift do $
	v(i+maxshift)=total(win(*,maxshift:mtrace.ysize-maxshift) $
		  *shift(mymask(*,maxshift:mtrace.ysize-maxshift),1,-i))
  i0=where(v eq max(v)) & i0=i0(0)-maxshift
  bgmask=shift(mymask,1,-i0)
  bgwin=win*0
  for i=0,x_size-1 do begin
	win(i,0:top(i)-i0-maxshift)=0
	win(i,top(i)-i0+maxshift:sz(2)-1)=0
	win(i,top(i)-i0-maxshift+1:top(i)-i0+maxshift-1)= $
	win(i,top(i)-i0-maxshift+1:top(i)-i0+maxshift-1)- $
	min(win(i,top(i)-i0-maxshift+1:top(i)-i0+maxshift-1))
	if max(win(i,*)) gt 0 then $
	bgwin(i,*)=win(i,*)-bgmask(i,*)*max(win(i,*))
  endfor
  bgnoise=stddev(bgwin(where(bgwin ne 0)))
  ;
  ; first find pos and width for each column with sufficient signal
  ;
  if midi_options.m then begin  ; CAH
        fitmidimask,win,mtrace
        sigma=mtrace.FWHM(0:x_size-1)/sqrt(2.*alog(2))
  endif else begin
  FOR i= xmin, xmax DO BEGIN
;     IF (total(win(i,*)) gt bgnoise * y_size) THEN BEGIN
      IF (max(win(i,*)) gt 5*bgnoise) THEN BEGIN
          ; do a little median smoothing on the column to remove badpix
          fit1D= gaussfit(column, median(reform(win[i,*]),3), b, NTERMS=3)

          if (b[1] gt 0 and b[1] lt y_size) then begin
              mtrace.trace[i]  = b[1]
              mtrace.FWHM[i]   = b[2] * sqrt(2.*alog(2))
              mtrace.fluxmax[i]= b[0]
              sigma[i] = b[2]
          endif
      endif
  endfor
  endelse       ; CAH

  ;; use only points with flux to compute the median
  range = where(mtrace.fluxmax gt 0, count)
  if count le 0 then begin
      print,'ERROR: No peak with positive flux found!'
      return
  endif

  bad_range = where(mtrace.fluxmax le 0, count)
  if count gt 0 then mtrace.trace[bad_range] = median(mtrace.trace[range])

  ;; forget points that are too weird
; CAH: In the following line, we use 5-pt median otherwise this didn't work with GRISM
 bad_trace= where( abs(mtrace.trace-median(mtrace.trace,11)) ge 4.2 $
	or mtrace.fluxmax le 0, bad_count)
 if bad_count gt 0  then begin
     mtrace.fluxmax[bad_trace] = 0.
     range = where( mtrace.fluxmax gt 0)
 endif

  if keyword_set(dtrace) then begin
      ;; caller gave higher-order coeffs, find the rest-position
      ;;
      mtrace.trace_coeff = dtrace.trace_coeff
      mtrace.sigma_coeff = dtrace.sigma_coeff

      mtrace.trace_coeff[0] = 0.
      curve= mtrace.trace - poly(findgen(x_size),mtrace.trace_coeff)
      mtrace.trace_coeff[0] = mean(curve[range])
  endif else begin
      ;; fit a full polynomial to trace and sigma
      ;;
      mtrace.trace_coeff= poly_fit(range, mtrace.trace[range], t_order,$
                                   MEASURE_ERRORS = 1./mtrace.fluxmax[range])
      mtrace.sigma_coeff= poly_fit(range, sigma[range], w_order,$
                                   MEASURE_ERRORS = 1./mtrace.fluxmax[range])
  endelse
  if !MIAdebug ne 0 then begin
      print,"trace coeff: ",mtrace.trace_coeff
      print,"sigma coeff: ",mtrace.sigma_coeff
  endif
  mtrace.range = range  ;plot only good points
  mtrace.Nrange= N_Elements(range)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro miamask::fit_spectra, Y0_ONLY=y0_only

common MidiOptions,midi_options,maskfile_mymidi,kappafile,photofile,la_bins ; CAH

  if n_elements(midi_options) eq 0 then midi_options={m:0} ; CAH

  if keyword_set(y0_only) then begin
      print,"Fitting y-position of spectrum only"
  endif else print,"Fitting polynomial trace and sigma"

  mtrace= {midisptrace}
  mean_tcoeff= fltarr(4,3)	;; 1.idx = window, 2.idx = polynom-coeff
  mean_scoeff= fltarr(4,2)
  nFiles = intarr(4)		;; counts the files per window we get coeffs from
  datatag= intarr(4)

  for iFile= 0, 1 do begin
      print, ""
      print, "========== Fitting ",iFile," =========="
      print, self.photfiles[iFile]

      if iFile eq 0 then chop_image= *self.photAmean $
        else chop_image= *self.photBmean

      tagnms= Tag_Names(chop_image)
      datatag[0] = where(tagnms eq "DATA1")
      datatag[1] = where(tagnms eq "DATA2")
      datatag[2] = where(tagnms eq "DATA3")
      datatag[3] = where(tagnms eq "DATA4")

      if datatag[3] eq -1 then nWin=2 else nWin=4
      ;;print,nWin," Windows at tags",datatag
      if !MIAdebug ne 0 then print,""

      im = obj_new('imagedata',(oirFileList(self.photfiles[iFile]))[0])
      he = im->prihead()
      shutter= strcompress(he->getpar('INS SHUT NAME'),/remove_all)
      obj_destroy,im
      obj_destroy,he

      for iWin= 0,nWin-1 do begin
          if !MIAdebug ne 0 then $
            print,Format='("----- doing window",I2," = ",A,"-----")',iWin,tagnms[datatag[iWin]]
          if (nWin ge 4) and $
            (iWin eq 0  and  shutter eq 'AOPEN') or $
            (iWin eq 3  and  shutter eq 'BOPEN') then begin
              print,"No light in SCI_PHOT mode"
              continue
          endif

          if midi_options.m then $
          fitmidimask,/init,iFile=iFile,iWin=iWin   ; CAH

          answer= ''
          dtrace= 0
          if keyword_set(y0_only) then $
            dtrace= default_mask(iWin,GRISM=self.grism,SCIPHOT=(nWin gt 2))

	  self.window=iWin	; CAH
          self->fit_window, chop_image.(datatag[iWin]),2,1,mtrace,DTRACE=dtrace

          dtrace= default_mask(iWin,GRISM=self.grism,SCIPHOT=(nWin gt 2))  ;; prepare for trouble
          if mtrace.trace_coeff[0] eq 0. then begin
              err = "Window " + strtrim(string(iWin),2) + ": not enough flux in spectrum"
              Log_error, self.photfiles[iFile], err
              if N_Elements(self.errorlog) gt 0 then begin
                  self.errorlog = self.errorlog + self.photfiles[iFile]+ ", "+err+string(10B)
              endif else begin
                  read,"TROUBLE: not enough flux in spectrum! [Hit Return]"+string(7B),answer
              endelse
              mtrace.trace_coeff = dtrace.trace_coeff
              mtrace.sigma_coeff = dtrace.sigma_coeff
          endif
          self.trace[iFile,iWin] = mtrace

          trace= poly(findgen(mtrace.xsize),mtrace.trace_coeff)
          if (min(trace) lt 0) or (max(trace) ge mtrace.ysize) then begin
              err = "Window " + strtrim(string(iWin),2) + ": spectrum outside of window"
              Log_error, self.photfiles[iFile], err
              if N_Elements(self.errorlog) gt 0 then begin
                  self.errorlog = self.errorlog + self.photfiles[iFile]+", "+err+string(10B)
              endif else begin
                  read,"TROUBLE: spectrum outside of window! [Hit Return]"+string(7B),answer
              endelse
              mtrace.trace_coeff = dtrace.trace_coeff
              mtrace.sigma_coeff = dtrace.sigma_coeff
          endif
          if (mtrace.sigma_coeff[0] le 0.) or $
            (mtrace.sigma_coeff[0] + mtrace.sigma_coeff[1]*mtrace.xsize le 0.) then begin
              err = "Window " + strtrim(string(iWin),2) + ": negative FWHM of Spectrum"
              Log_error, self.photfiles[iFile], err
              if N_Elements(self.errorlog) gt 0 then begin
                  self.errorlog = self.errorlog + self.photfiles[iFile]+", "+err+string(10B)
              endif else begin
                  read,"TROUBLE: negative FWHM of spectrum! [Hit Return]"+string(7B),answer
              endelse
              mtrace.sigma_coeff = dtrace.sigma_coeff
          endif
          mean_tcoeff[iWin,*] = mean_tcoeff[iWin,*] + mtrace.trace_coeff
          mean_scoeff[iWin,*] = mean_scoeff[iWin,*] + mtrace.sigma_coeff
          nFiles[iWin] = nFiles[iWin]+1
      	  if self.photfiles[1] eq self.photfiles[0] then self.trace[1,iWin] = self.trace[0,iWin]
      endfor	;; iWin
;     CAH
      if self.photfiles[1] eq self.photfiles[0] then break
  endfor	;; iFile

  print,""
  print,"========== averaging traces and widths ========="
  iFile= N_elements(self.photfiles)
  sz = size( (*self.photAmean).data1)	;; sizes of windows should better be identical
  for iWin= 0,nWin-1 do begin
      self.traceFr[iWin].trace_coeff = mean_tcoeff[iWin,*] / nFiles[iWin]
      self.traceFr[iWin].sigma_coeff = mean_scoeff[iWin,*] / nFiles[iWin] * self.width

      if !MIAdebug ne 0 then begin
          print,"----- window",iWin," = ",tagnms[datatag(iWin)],"-----"
          print,"trace coeff: ",self.traceFr[iWin].trace_coeff
          print,"sigma coeff: ",self.traceFr[iWin].sigma_coeff
      endif
      szi= size( (*self.photAmean).(datatag[iWin]))
      self.traceFr[iWin].trace= poly(findgen(sz[1]),self.traceFr[iWin].trace_coeff)
      self.traceFr[iWin].FWHM = poly(findgen(sz[1]),self.traceFr[iWin].sigma_coeff)*sqrt(2.*alog(2.))
  endfor
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function miamask::get_tracePh
  return, self.trace
end

Function miamask::get_traceFr
  return, self.traceFr
end

Function miamask::get_mask_name
  return, self.maskfile
end

Function miamask::get_mask_data

  if self.maskfrom eq 3 then begin
      if !MIAdebug ne 0 then print,"Reading mask from file ",self.maskfile
;     return, oirgetdata(self.maskfile)
; CAH Added (4) following lines to allow reading of ESI pipeline masks
      mask=mrdfits(self.maskfile,1)
      index=where(strpos(tag_names(mask),'DATA1') ge 0,count)
      if count eq 0 then mask=mrdfits(maskfile,2)
      return,mask
  endif else begin
      if !MIAdebug ne 0 then print,"Creating mask from fitted spectra"
      if (where(Tag_Names(*self.photAmean) eq "DATA4") lt 0) then nWin=2 else nWin=4

      mask= fltarr( nWin, self.traceFr[0].xsize, self.traceFr[0].ysize)
      for iWin= 0,nWin-1 do begin
          if !MIAdebug gt 1 then $
            print,"trace_coeffs No.",iWin,": ",self.traceFr[iWin].trace_coeff
          mask[iWin,*,*]= make_mask_from_trace(self.traceFr[iWin])
      endfor

      if (nWin ge 4) then begin
          if !MIAdebug ne 0 then print,nWin," windows => Sci-Phot"
          return, { data1: reform(mask[0,*,*]), $
                    data2: reform(mask[1,*,*]), $
                    data3: reform(mask[2,*,*]), $
                    data4: reform(mask[3,*,*]) }
      endif else begin
          if !MIAdebug ne 0 then print,nWin," windows => Hi-Sense"
          return, { data1: reform(mask[0,*,*]), data2: reform(mask[1,*,*]) }
      endelse
  endelse
end

Pro miamask::write_mask
  if self.maskfrom eq 3 then begin
      print,"Mask loaded from file, not overwriting it"
  endif else begin
      if !MIAdebug ne 0 then print,"Writing mask ",self.maskfile
      dirname = file_dirname(self.maskfile)
      if not file_test(dirname) then file_mkdir,dirname
      oirNewData, (oirFilelist(self.photfiles[0]))[0], self.maskfile, self->get_mask_data()
      print,"Collecting garbage"
      heap_gc
  endelse
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask::cleanup
  if !MIAdebug ne 0 then print,"miamask cleaning up: ", self.maskfile
  ptr_free, self.fringemean
  ptr_free, self.photAmean
  ptr_free, self.photBmean
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; end of mask_utilities
