/*
   jpleph.c: Interface to JPL Ephemeris.

   Version 1.1

   1-94 Modify to read JPL 92 ephemeris.
   Version 1.2 - modified by DFB 1994-07-16 to move includes of
	         system include files from .h file to .c file
*/

#pragma segment JPLEPH

/*
   Standard Libradies
*/

#include <math.h>
#include <stdio.h>
#include "jpleph.h"

/*
   Define pointer to ephemeris file.
*/

FILE *EPHFILE = NULL;

static char TTL[252], CNAM[2400];

/*
   'IPV' is the position/velocity switch.
      = 1, only positions will be computed.
      = 2, both positions and velocities will be computed.
      Default value is 2.

   KM...flag defining physical units of the output states. 
      = 1, km and km/sec
      = 0, AU and AU/day
      Default value is 0 (KM determines time unit for nutations.
                          Angle unit is always radians.)
   BARY...flag defining output center. Only the 9 planets are affected.
      = 1, center is solar-system barycenter.
      = 0, center is Sun
      Default value is 0.
*/

static long DENUM, NCON, IPT[3][12], KM = 0, BARY=0, IPV = 2, LIST[11],
     L[2], TC[2], NEMB, NRL, NP, NV;

/*
   PVSUN...6-element double array containing the barycentric position
           and velocity of the Sun.
*/

static double SS[3], JPLAU, VE[2], JED[2], PV[6][13], EMBF[2], FAC, 
       PVSUN[3][2], AUFAC, PC[18], VC[18], TWOT, CVAL[400], EMRAT;

/********ephopn */

short int ephopn (char fname[51],

                  double *jdbeg, double *jdend)
/*
------------------------------------------------------------------------

   PURPOSE:    
      This subroutine opens a JPL planetary ephemeris file and
      sets initial values.  This subroutine must be called
      prior to calls to the other JPL ephemeris routines.

   REFERENCES: 
      Standish, E.M. and Newhall, X X (1988). "The JPL Export
         Planetary Ephemeris"; JPL document dated 17 June 1988.

   INPUT
   ARGUMENTS:
      fname (char[51])
         Name of the direct-access ephemeris file.

   OUTPUT
   ARGUMENTS:
      *jdbeg (double)
         Beginning Julian date of the ephemeris file.
      *jdend (double)
         Ending Julian date of the ephemeris file.

   RETURNED
   VALUE:
      (short int)
         0 ... file exists
         1 ... file does not exist

   GLOBALS
   USED:
      TTL,CNAM,DENUM,NCON,IPT,LIST,L,TC,NEMB,NRL,NP,NV,SS,JPLAU,VE,JED,
      PV,EMBF,FAC,AUFAC,PC,VC,TWOT,CVAL,EMRAT

   FUNCTIONS
   CALLED:
      None.

   VER./DATE/
   PROGRAMMER:
      V1.0/03-93/WTH (USNO/AA): Convert FORTRAN to C.
      V1.1/07-93/WTH (USNO/AA): Update to C standards.

   NOTES:
      None.

------------------------------------------------------------------------
*/
{
   short int i, j;

/*
   Close any file currently open file pointed to by EPHFILE
*/

   if (EPHFILE)
      fclose(EPHFILE);

/*
   Open file fname.
*/

   if (!(EPHFILE = fopen(fname, "rb")))
      return 1;
    else
   {

/*
   File found. Set initializations and default values.
*/

      KM = 0;
      BARY = 0;
      IPV = 2;
      EMBF[0] = -1.0;
      EMBF[1] =  1.0;
      FAC = 0.0;
      NEMB = 1;

      NRL = 0;
      AUFAC = 1.0;

      NP = 2;
      NV = 3;
      TWOT = 0.0;
      PC[0] = 1.0;
      PC[1] = 0.0;
      VC[1] = 1.0;

      for (i = 0; i < 2; i++)
      {
         JED[i] = 0.0;
         L[i] = 0;
         TC[i] = 0;
      }

      for (i = 0; i < 6; i++)
         for (j = 0; j < 13; j++)
            PV[i][j] = 0.0;

      for (i = 0; i < 11; i++)
         LIST[i] = 0;

/*
   Read record 1 and set other variables to initial values.
*/

      if (fread (TTL, sizeof TTL, 1, EPHFILE) != 1)
         return 2;

      if (fread (CNAM, sizeof CNAM, 1, EPHFILE) != 1)
         return 3;
      if (fread (SS, sizeof SS, 1, EPHFILE) != 1)
         return 4;
      if (fread (&NCON, sizeof NCON, 1, EPHFILE) != 1)
         return 5;
      if (fread (&JPLAU, sizeof JPLAU, 1, EPHFILE) != 1)
         return 7;
      if (fread (&EMRAT, sizeof EMRAT, 1, EPHFILE) != 1)
         return 8;

      for (i = 0; i < 12; i++)
         for (j = 0; j < 3; j++)
            if (fread (&IPT[j][i],sizeof(long), 1, EPHFILE) != 1)
               return 9;

      if (fread (&DENUM,sizeof DENUM, 1, EPHFILE) != 1)
         return 10;
      fseek(EPHFILE, 6608, SEEK_SET);
      if (fread (CVAL, sizeof CVAL, 1, EPHFILE) != 1)
         return 6;
      *jdbeg = SS[0];
      *jdend = SS[1];
      VE[0] = 1.0 / (1.0 + EMRAT);
      VE[1] = EMRAT * VE[0];
   }
   return 0;
}

/********pleph */

short int pleph (double jd, short int targ, short int cent, 

                 double rrd[6])
/*
------------------------------------------------------------------------

   PURPOSE:    
      This subroutine accesses the JPL planetary ephemeris to give the
      position and velocity of the point 'targ' with respect to 'cent'.

   REFERENCES: 
      Standish, E.M. and Newhall, X X (1988). "The JPL Export
         Planetary Ephemeris"; JPL document dated 17 June 1988.

   INPUT
   ARGUMENTS:
      jd (double)
         Julian date (TDB) at which interpolation is desired.
      targ (short int)
         Number of 'target' point.
      cent (short int)
         Number of 'center' (origin) point.
         The numbering convention for 'targ' and'cent' is:
            0  =  Mercury           7 = Neptune
            1  =  Venus             8 = Pluto
            2  =  Earth             9 = Moon
            3  =  Mars             10 = Sun
            4  =  Jupiter          11 = Solar system bary.
            5  =  Saturn           12 = Earth-Moon bary.
            6  =  Uranus           13 = Nutations (long. and obliq.)
            (If nutations are desired, set 'targ' = 14;
             'cent' will be ignored on that call.)

   OUTPUT
   ARGUMENTS:
      rrd (double)
         6-element array containing position and velocity of point
         'targ' relative to 'cent'. The units are AU and AU/day. 
         In the case of nutations, the first four words of 'rrd'
         will be set to nutations and rates, having units of radians
         and radians/day.

   RETURNED
   VALUE:
      (short int)
         0...No Errors
         1...Error occured in 'state'

   GLOBALS
   USED:
      KM,BARY,IPV,LIST,L,TC,VE,PV,NEMB,EMBF,FAC,PVSUN

   FUNCTIONS
   CALLED:
      state

   VER./DATE/
   PROGRAMMER:
      V1.0/03-93/WTH (USNO/AA): Convert FORTRAN to C.
      V1.1/07-93/WTH (USNO/AA): Update to C standards.

   NOTES:
      1. In many cases the user will need only position values for
         ephemerides, or nutations. For position-only output, the 
         global variable 'IPV'  should be set = 1 before the next call
         to pleph. (Its default value is 2, which returns both positions
         and rates.)

------------------------------------------------------------------------
*/
{
   long int llst[13] = {0,1,9,3,4,5,6,7,8,9,10,10,2}, ncmp, lme, bsave;
   short int i, error;

/*
   Initialize 'JED' for 'state' and set up component count.
*/

   JED[0] = jd;
   ncmp = 3 * IPV;
   
/*
   Check for nutation request.
*/

   if (targ >= 14)
   {
      LIST[10] = IPV;
      if (state (JED,LIST, rrd))
         return 1;
      LIST[10] = 0;
      return 0;
   }

/*
   Check for target point = center point.
*/

   if (targ == cent)
   {
      for (i = 0;i < ncmp; i++)
         rrd[i] = 0.0;
      return 0;
   }

/*
   Force barycentric output by 'state'.
*/

   bsave = BARY;
   BARY = 1;

/*
   Set up proper entries in 'list' array for 'state' call.
*/

   TC[0] = targ;
   TC[1] = cent;
   lme = 0;

   for (i = 0; i < 2; i++)
   {
      L[i] = llst[TC[i]];
      if (L[i] < 10)
         LIST[L[i]] = IPV;
      if (TC[i] == 2)
      {
         lme = 2;
         FAC = -VE[0];
      }
       else if (TC[i] == 9)
      {
         lme = 9;
         FAC = VE[1];
      }
       else if (TC[i] == 12)
         NEMB = i;
   }

   if ((LIST[9] == IPV) && (L[0] != L[1]))
      LIST[2] = IPV - LIST[2];

/*
   Make call to 'state'.
*/

   if ((error = state (JED,LIST, rrd)))
      return 1;

/*
   Case: Earth-to-Moon.
*/

   if ((targ == 9) && (cent == 2))
      for (i = 0; i < ncmp; i++)
         rrd[i] = PV[i][9];

/*
   Case: Moon-to-Earth.
*/

    else if ((targ == 2) && (cent == 9))
      for (i = 0; i < ncmp; i++)
         rrd[i] = -PV[i][9];

/*
   Case: E-M barycenter to Moon or Earth.
*/

    else if (((targ == 12) || (cent == 12)) && (LIST[9] == IPV))
      for (i = 0; i < ncmp; i++)
         rrd[i] = PV[i][9] * FAC * EMBF[NEMB];

/*
   Otherwise, get Earth or Moon vector and then get output vector.
*/

    else
      for (i = 0; i < ncmp; i++)
      {
         if (i < 3)
            PV[i][10] = PVSUN[i][0];
          else
            PV[i][10] = PVSUN[i-3][1];
         PV[i][12] = PV[i][2];
         if (lme > 0)
            PV[i][lme] = PV[i][2] + FAC * PV[i][9];
         rrd[i] = PV[i][targ] - PV[i][cent];
      }

/*
   Clear 'state' body array and restore barycenter flag.
*/

   LIST[2] = 0;
   LIST[L[0]] = 0;
   LIST[L[1]] = 0;
   BARY = bsave;

   return 0;

}

/********state */

short int state (double jed[2], long int llist[11], 

                 double p[6])
/*
------------------------------------------------------------------------

   PURPOSE:    
      This subroutine reads and interpolates the JPL planetary 
      ephemeris file.

   REFERENCES: 
      Standish, E.M. and Newhall, X X (1988). "The JPL Export
         Planetary Ephemeris"; JPL document dated 17 June 1988.

   INPUT
   ARGUMENTS:
      jed (double)
         2-element Julian date (TDB) at which interpolation is wanted.
         Any combination of jed[0]+jed[1] which falls within the time 
         span on the file is a permissible epoch.  See Note 1 below.
      llist (long int)
         11-element array specifying what interpolation is desired
         for each of the bodies on the file.
         llist[i] = 0, no interpolation for body i
                  = 1, position only
                  = 2, position and velocity
         The designation of the astronomical bodies by i is:
               i = 0: Mercury,               1: Venus, 
                 = 2: Earth-Moon barycenter, 3: Mars, 
                 = 4: Jupiter,               5: Saturn, 
                 = 6: Uranus,                7: Neptune, 
                 = 8: Pluto,                 9: geocentric Moon, 
                 =10: nutations in longitude and obliquity

   OUTPUT
   ARGUMENTS:
      p[6] (double)
         Array that will contain requested interpolated quantities.
         The body specified by llist will have its state in the array
         starting at p.  (On any given call, only those words in 'PV' 
         which are affected by the first 10 'list' entries are set.
         The rest of the 'PV' array is untouched). The order of 
         components starting in p[i] is: x, y, z, dx, dy, dz.
         All output vectors are referenced to the Earth mean equator and
         equinox of epoch. The Moon state is always geocentric; the
         other nine states are either heliocentric or solar-system 
         barycentric, depending on the setting of global flags 
         (see Note 2 below).
              or
         first four elements of the array will contain nutations and 
          rates, depending on the setting of llist[10].  The order of 
          quantities in 'p' is:
                  d psi...nutation in longitude
                  d epsilon...nutation in obliquity
                  d psi dot
                  d epsilon dot

   RETURNED
   VALUE:
      (short int)
         0...everythin OK.
         1...error reading ephemeris file.
         2...epoch out of range.

   GLOBALS
   USED:
      EPHFILE,IPT,KM,BARY,SS,JPLAU,PV,AUFAC

   FUNCTIONS
   CALLED:
      interp

   VER./DATE/
   PROGRAMMER:
      V1.0/03-93/WTH (USNO/AA): Convert FORTRAN to C.
      V1.1/07-93/WTH (USNO/AA): Update to C standards.

   NOTES:
      1. For ease in programming, the user may put the entire epoch in
         jed[0] and set jed[1] = 0. For maximum interpolation accuracy, 
         set jed[0] = the most recent midnight at or before
         interpolation epoch, and set jed[1] = fractional part of a day
         elapsed between jed[0] and epoch. As an alternative, it may
         prove convenient to set jed[0] = some fixed epoch, such as
         start of the integration and jed[1] = elapsed interval between
         then and epoch.

------------------------------------------------------------------------
*/
{
   short int i, j;
   long nr, rec, ncm = 3, fl = 2;
   double t[2], jd[4], s, nut[4];
   static double buf[900];

/*
   Set units based on value of the 'KM' flag.
*/

   if (KM)
      t[1] = SS[2] * 86400.0;
    else
   {
      t[1] = SS[2];
      AUFAC = 1.0 / JPLAU;
   }

/*
   Check epoch.
*/

   s = jed[0] - 0.5;

/*
   jd[1] = modf (s, &s2);
   jd[0] = s2;
   jd[3] = modf (jed[1], &s2);
   jd[2] = s2;
   jd[0] = jd[0] + jd[2] + 0.5;
   jd[1] = jd[1] + jd[3];

   jd[3] = modf (jd[1], &s2);
   jd[2] = s2;
   jd[0] = jd[0] + jd[2];
*/

   split(s, &jd[0]);
   split(jed[1], &jd[2]);
   jd[0] = jd[0] + jd[2] + 0.5;
   jd[1] = jd[1] + jd[3];
   split (jd[1], &jd[2]);
   jd[0] = jd[0] + jd[2];

/*
   Return 2 for epoch out of range.
*/

   if ((jd[0] < SS[0]) || ((jd[0] + jd[3]) > SS[1]))
      return 2;

/*
   Calculate record number and relative time interval.
*/

   nr = (short int) ((jd[0] - SS[0]) / SS[2]) + 2;
   if (jd[0] == SS[1])
      nr -= 1;

   t[0] = ((jd[0] - ((double) (nr-2) * SS[2] + SS[0])) + jd[3]) / SS[2];

/*
   Read correct record if it is not already in memory.
*/

   if (nr != NRL)
   {
      NRL = nr;
      rec = (nr - 1) * 6608;
      fseek (EPHFILE, rec, SEEK_SET);
      if (!fread (buf, sizeof buf, 1, EPHFILE))
         return 1;
   }

/*
   Interpolate barycentric position and velocity of the Sun.
*/
   interp (buf,t,IPT[0][10]-1,IPT[1][10],ncm,IPT[2][10],fl, p);
   for (j = 0;j < 3; j++)
   {
      PVSUN[j][0] = p[j];
      PVSUN[j][1] = p[j+3];
   }

/*
   Check and interpolate whichever bodies are requested.
*/

   for (i = 0; i < 3; i++)
      for (j = 0; j < 2; j++)
         PVSUN[i][j] *= AUFAC;

   for (i = 0; i < 10; i++)
      if (llist[i] > 0)
      {
         interp (buf,t,IPT[0][i]-1,IPT[1][i],ncm,IPT[2][i],llist[i], p);
         for (j = 0; j < 6; j++)
            PV[j][i] = p[j];
         for (j = 0; j < llist[i]*3; j++)
         {
            if ((i <= 8) && (!BARY))
               if (j<3)
                  PV[j][i] = PV[j][i] * AUFAC - PVSUN[j][0];
                else
                  PV[j][i] = PV[j][i] * AUFAC - PVSUN[j][1];
             else
               PV[j][i] = PV[j][i] * AUFAC;
         }
      }

/*
   Do nutations if requested (and if on the ephemeris file).
*/

   if (llist[10] > 0 && IPT[1][11] > 0)
   {
      interp (buf,t,IPT[1][11]-1,IPT[1][11],fl,IPT[2][11],llist[10], p);
      for (j = 0; j < 4; j++)
         nut[j] = p[j];
   }

   return 0;
}

/********split */

void split (double tt, 

            double fr[2])
/*
------------------------------------------------------------------------

   PURPOSE:    
      This subroutine breaks up a double number into a double integer
      part and a fractional part.

   REFERENCES: 
      Standish, E.M. and Newhall, X X (1988). "The JPL Export
         Planetary Ephemeris"; JPL document dated 17 June 1988.

   INPUT
   ARGUMENTS:
      tt (double)
         Input number.

   OUTPUT
   ARGUMENTS:
      fr[2] (double)
         2-element output array; fr[0] contains integer part, 
         fr[1] contains fractional part. For negative input numbers, 
         fr[0] contains the next more negative integer; 
         fr[1] contains a positive fraction.

   RETURNED
   VALUE:
      None.

   GLOBALS
   USED:
      None.

   FUNCTIONS
   CALLED:
      None.

   VER./DATE/
   PROGRAMMER:
      V1.0/06-90/JAB (USNO/NA): CA coding standards
      V1.1/03-93/WTH (USNO/AA): Convert to C.
      V1.2/07-93/WTH (USNO/AA): Update to C standards.

   NOTES:
      None.

------------------------------------------------------------------------
*/
{

/*
   Get integer and fractional parts.
*/

   fr[0] = (double)((long int) tt);
   fr[1] = tt - fr[0];

/*
   Make adjustments for negative input number.
*/

   if ((tt >= 0.0) || (fr[1] == 0.0))
      return;
    else
   {
      fr[0] = fr[0] - 1.0;
      fr[1] = fr[1] + 1.0;
   }

   return;
}

/********interp */

void interp (double buf[900], double t[2], long n, long ncf, long ncm, 
             long na, long fl, 

             double p[6])
/*
------------------------------------------------------------------------

   PURPOSE:    
      This subroutine differentiates and interpolates a set of
      Chebyshev coefficients to give position and velocity.

   REFERENCES: 
      Standish, E.M. and Newhall, X X (1988). "The JPL Export
         Planetary Ephemeris"; JPL document dated 17 June 1988.

   INPUT
   ARGUMENTS:
      buf[900] (double)
         Array of Chebyshev coefficients of position.
      t (double)
         t[0] is fractional time interval covered by coefficients at
         which interpolation is desired (0 <= t(1) <= 1).
         t[1] is length of whole interval in input time units.
      n (long int)
         Starting point in array buf.
      ncf (long int)
         Number of coefficients per component.
      ncm (long int)
         Number of components per set of coefficients.
      na (long int)
         Number of sets of coefficients in full array
         (i.e., number of sub-intervals in full interval).
      fl (long int)
         = 1 ... position only
         = 2 ... position and velocity

   OUTPUT
   ARGUMENTS:
      p[6] (double)
         Interpolated quantities requested.

   RETURNED
   VALUE:
      None.

   GLOBALS
   USED:
      NP,NV,PC,VC,TWOT

   FUNCTIONS
   CALLED:
      None.

   VER./DATE/
   PROGRAMMER:
      V1.0/03-93/WTH (USNO/AA): Convert FORTRAN to C.
      V1.1/07-93/WTH (USNO/AA): Update to C standards.

   NOTES:
      None.

------------------------------------------------------------------------
*/
{
   short int l, i, j, k;

   double dna, dt1, temp, tc, vfac;

/*
   Get correct sub-interval number for this set of coefficients and
   then get normalized Chebyshev time within that subinterval.
*/

   dna = (double) na;
   dt1 = (double) ((long int) t[0]);
   temp = dna * t[0];
   l = (short int) (temp - dt1) + 1;

/*
   'tc' is the normalized Chebyshev time (-1 <= tc <= 1).
*/

   tc = 2.0 * (fmod (temp, 1.0) + dt1) - 1.0;

/*
   Check to see whether Chebyshev time has changed, and compute new
   polynomial values if it has.  (The element pc(2) is the value of
   t1[tc] and hence contains the value of 'tc' on the previous call.)
*/

   if (tc != PC[1])
   {
      NP = 2;
      NV = 3;
      PC[1] = tc;
      TWOT = tc + tc;
   }

/*
   Be sure that at least 'ncf' polynomials have been evaluated and
   are stored in the array 'pc'.
*/

   if (NP < ncf)
   {
      for (i = NP; i < ncf; i++)
         PC[i] = TWOT * PC[i-1] - PC[i-2];
      NP = ncf;
   }

/*
   Interpolate to get position for each component.
*/

   for (i = 0; i < ncm; i++)
   {
      p[i] = 0.0;
      for (j = ncf - 1; j >= 0; j--)
      {
         k = n + j + i * (ncf) + (l - 1) * (ncf * ncm);
         p[i] = p[i] + PC[j] * buf[k];
      }
   }

   if (fl <= 1)
      return;

/*
   If velocity interpolation is desired, be sure enough derivative
   polynomials have been generated and stored.
*/

   vfac = (dna + dna) / t[1];
   VC[2] = TWOT + TWOT;
   if (NV < ncf)
   {
      for (i = NV; i < ncf; i++)
         VC[i] = TWOT * VC[i-1] + PC[i-1] + PC[i-1] - VC[i-2];

      NV = ncf;
   }

/*
   Interpolate to get velocity for each component.
*/

   for (i = 3;i < 2 * ncm; i++)
   {
      p[i] = 0.0;
      for (j = ncf - 1; j > 0; j--)
      {
         k = n + j + (i - 3) * (ncf) + (l - 1) * (ncf * ncm);
         p[i] = p[i] + VC[j] * buf[k];
      }
      p[i] *= vfac;
   }

   return;
}

