/*
   NOVAS-C Version 3.0
   Nutation Models

   Naval Observatory Vector Astrometry Software
   C Version

   U. S. Naval Observatory
   Astronomical Applications Dept.
   3450 Massachusetts Ave., NW
   Washington, DC  20392-5420
*/

#ifndef _NOVAS_
   #include "novas.h"
#endif

/********iau2000a */

void iau2000a (double jd_high, double jd_low,

               double *dpsi, double *deps)
/*
------------------------------------------------------------------------

   PURPOSE:
      To compute the forced nutation of the non-rigid Earth based on
      the IAU 2000A nutation model.

   REFERENCES:
      IERS Conventions (2003), Chapter 5.
      Simon et al. (1994) Astronomy and Astrophysics 282, 663-683,
         esp. Sections 3.4-3.5.

   INPUT
   ARGUMENTS:
      jd_high (double)
         High-order part of TT Julian date.
      jd_low (double)
         Low-order part of TT Julian date.

   OUTPUT
   ARGUMENTS:
      *dpsi (double)
         Nutation (luni-solar + planetary) in longitude, in radians.
      *deps (double)
         Nutation (luni-solar + planetary) in obliquity, in radians.

   RETURNED
   VALUE:
      None.

   GLOBALS
   USED:
      T0, ASEC2RAD, TWOPI

   FUNCTIONS
   CALLED:
      fund_args    novas.c
      fmod         math.h
      sin          math.h
      cos          math.h

   VER./DATE/
   PROGRAMMER:
      V1.0/03-04/JAB (USNO/AA)

   NOTES:
     1. The IAU 2000A nutation model is MHB_2000 without the free core
     nutation and without the corrections to Lieske precession.
     2. This function is the "C" version of NOVAS Fortran routine
     'nu2000a'.

------------------------------------------------------------------------
*/
{
   short int i;

   double t, a[5], dp, de, arg, sarg, carg, factor, dpsils, depsls,
      al, alsu, af, ad, aom, alme, alve, alea, alma, alju, alsa, alur,
      alne, apa, dpsipl, depspl;

/*
   Luni-Solar argument multipliers:
       L     L'    F     D     Om
*/

   const short int nals_t[678][5] =
      {0,    0,    0,    0,    1,
       0,    0,    2,   -2,    2,
       0,    0,    2,    0,    2,
       0,    0,    0,    0,    2,
       0,    1,    0,    0,    0,
       0,    1,    2,   -2,    2,
       1,    0,    0,    0,    0,
       0,    0,    2,    0,    1,
       1,    0,    2,    0,    2,
       0,   -1,    2,   -2,    2,
       0,    0,    2,   -2,    1,
      -1,    0,    2,    0,    2,
      -1,    0,    0,    2,    0,
       1,    0,    0,    0,    1,
      -1,    0,    0,    0,    1,
      -1,    0,    2,    2,    2,
       1,    0,    2,    0,    1,
      -2,    0,    2,    0,    1,
       0,    0,    0,    2,    0,
       0,    0,    2,    2,    2,
       0,   -2,    2,   -2,    2,
      -2,    0,    0,    2,    0,
       2,    0,    2,    0,    2,
       1,    0,    2,   -2,    2,
      -1,    0,    2,    0,    1,
       2,    0,    0,    0,    0,
       0,    0,    2,    0,    0,
       0,    1,    0,    0,    1,
      -1,    0,    0,    2,    1,
       0,    2,    2,   -2,    2,
       0,    0,   -2,    2,    0,
       1,    0,    0,   -2,    1,
       0,   -1,    0,    0,    1,
      -1,    0,    2,    2,    1,
       0,    2,    0,    0,    0,
       1,    0,    2,    2,    2,
      -2,    0,    2,    0,    0,
       0,    1,    2,    0,    2,
       0,    0,    2,    2,    1,
       0,   -1,    2,    0,    2,
       0,    0,    0,    2,    1,
       1,    0,    2,   -2,    1,
       2,    0,    2,   -2,    2,
      -2,    0,    0,    2,    1,
       2,    0,    2,    0,    1,
       0,   -1,    2,   -2,    1,
       0,    0,    0,   -2,    1,
      -1,   -1,    0,    2,    0,
       2,    0,    0,   -2,    1,
       1,    0,    0,    2,    0,
       0,    1,    2,   -2,    1,
       1,   -1,    0,    0,    0,
      -2,    0,    2,    0,    2,
       3,    0,    2,    0,    2,
       0,   -1,    0,    2,    0,
       1,   -1,    2,    0,    2,
       0,    0,    0,    1,    0,
      -1,   -1,    2,    2,    2,
      -1,    0,    2,    0,    0,
       0,   -1,    2,    2,    2,
      -2,    0,    0,    0,    1,
       1,    1,    2,    0,    2,
       2,    0,    0,    0,    1,
      -1,    1,    0,    1,    0,
       1,    1,    0,    0,    0,
       1,    0,    2,    0,    0,
      -1,    0,    2,   -2,    1,
       1,    0,    0,    0,    2,
      -1,    0,    0,    1,    0,
       0,    0,    2,    1,    2,
      -1,    0,    2,    4,    2,
      -1,    1,    0,    1,    1,
       0,   -2,    2,   -2,    1,
       1,    0,    2,    2,    1,
      -2,    0,    2,    2,    2,
      -1,    0,    0,    0,    2,
       1,    1,    2,   -2,    2,
      -2,    0,    2,    4,    2,
      -1,    0,    4,    0,    2,
       2,    0,    2,   -2,    1,
       2,    0,    2,    2,    2,
       1,    0,    0,    2,    1,
       3,    0,    0,    0,    0,
       3,    0,    2,   -2,    2,
       0,    0,    4,   -2,    2,
       0,    1,    2,    0,    1,
       0,    0,   -2,    2,    1,
       0,    0,    2,   -2,    3,
      -1,    0,    0,    4,    0,
       2,    0,   -2,    0,    1,
      -2,    0,    0,    4,    0,
      -1,   -1,    0,    2,    1,
      -1,    0,    0,    1,    1,
       0,    1,    0,    0,    2,
       0,    0,   -2,    0,    1,
       0,   -1,    2,    0,    1,
       0,    0,    2,   -1,    2,
       0,    0,    2,    4,    2,
      -2,   -1,    0,    2,    0,
       1,    1,    0,   -2,    1,
      -1,    1,    0,    2,    0,
      -1,    1,    0,    1,    2,
       1,   -1,    0,    0,    1,
       1,   -1,    2,    2,    2,
      -1,    1,    2,    2,    2,
       3,    0,    2,    0,    1,
       0,    1,   -2,    2,    0,
      -1,    0,    0,   -2,    1,
       0,    1,    2,    2,    2,
      -1,   -1,    2,    2,    1,
       0,   -1,    0,    0,    2,
       1,    0,    2,   -4,    1,
      -1,    0,   -2,    2,    0,
       0,   -1,    2,    2,    1,
       2,   -1,    2,    0,    2,
       0,    0,    0,    2,    2,
       1,   -1,    2,    0,    1,
      -1,    1,    2,    0,    2,
       0,    1,    0,    2,    0,
       0,   -1,   -2,    2,    0,
       0,    3,    2,   -2,    2,
       0,    0,    0,    1,    1,
      -1,    0,    2,    2,    0,
       2,    1,    2,    0,    2,
       1,    1,    0,    0,    1,
       1,    1,    2,    0,    1,
       2,    0,    0,    2,    0,
       1,    0,   -2,    2,    0,
      -1,    0,    0,    2,    2,
       0,    1,    0,    1,    0,
       0,    1,    0,   -2,    1,
      -1,    0,    2,   -2,    2,
       0,    0,    0,   -1,    1,
      -1,    1,    0,    0,    1,
       1,    0,    2,   -1,    2,
       1,   -1,    0,    2,    0,
       0,    0,    0,    4,    0,
       1,    0,    2,    1,    2,
       0,    0,    2,    1,    1,
       1,    0,    0,   -2,    2,
      -1,    0,    2,    4,    1,
       1,    0,   -2,    0,    1,
       1,    1,    2,   -2,    1,
       0,    0,    2,    2,    0,
      -1,    0,    2,   -1,    1,
      -2,    0,    2,    2,    1,
       4,    0,    2,    0,    2,
       2,   -1,    0,    0,    0,
       2,    1,    2,   -2,    2,
       0,    1,    2,    1,    2,
       1,    0,    4,   -2,    2,
      -1,   -1,    0,    0,    1,
       0,    1,    0,    2,    1,
      -2,    0,    2,    4,    1,
       2,    0,    2,    0,    0,
       1,    0,    0,    1,    0,
      -1,    0,    0,    4,    1,
      -1,    0,    4,    0,    1,
       2,    0,    2,    2,    1,
       0,    0,    2,   -3,    2,
      -1,   -2,    0,    2,    0,
       2,    1,    0,    0,    0,
       0,    0,    4,    0,    2,
       0,    0,    0,    0,    3,
       0,    3,    0,    0,    0,
       0,    0,    2,   -4,    1,
       0,   -1,    0,    2,    1,
       0,    0,    0,    4,    1,
      -1,   -1,    2,    4,    2,
       1,    0,    2,    4,    2,
      -2,    2,    0,    2,    0,
      -2,   -1,    2,    0,    1,
      -2,    0,    0,    2,    2,
      -1,   -1,    2,    0,    2,
       0,    0,    4,   -2,    1,
       3,    0,    2,   -2,    1,
      -2,   -1,    0,    2,    1,
       1,    0,    0,   -1,    1,
       0,   -2,    0,    2,    0,
      -2,    0,    0,    4,    1,
      -3,    0,    0,    0,    1,
       1,    1,    2,    2,    2,
       0,    0,    2,    4,    1,
       3,    0,    2,    2,    2,
      -1,    1,    2,   -2,    1,
       2,    0,    0,   -4,    1,
       0,    0,    0,   -2,    2,
       2,    0,    2,   -4,    1,
      -1,    1,    0,    2,    1,
       0,    0,    2,   -1,    1,
       0,   -2,    2,    2,    2,
       2,    0,    0,    2,    1,
       4,    0,    2,   -2,    2,
       2,    0,    0,   -2,    2,
       0,    2,    0,    0,    1,
       1,    0,    0,   -4,    1,
       0,    2,    2,   -2,    1,
      -3,    0,    0,    4,    0,
      -1,    1,    2,    0,    1,
      -1,   -1,    0,    4,    0,
      -1,   -2,    2,    2,    2,
      -2,   -1,    2,    4,    2,
       1,   -1,    2,    2,    1,
      -2,    1,    0,    2,    0,
      -2,    1,    2,    0,    1,
       2,    1,    0,   -2,    1,
      -3,    0,    2,    0,    1,
      -2,    0,    2,   -2,    1,
      -1,    1,    0,    2,    2,
       0,   -1,    2,   -1,    2,
      -1,    0,    4,   -2,    2,
       0,   -2,    2,    0,    2,
      -1,    0,    2,    1,    2,
       2,    0,    0,    0,    2,
       0,    0,    2,    0,    3,
      -2,    0,    4,    0,    2,
      -1,    0,   -2,    0,    1,
      -1,    1,    2,    2,    1,
       3,    0,    0,    0,    1,
      -1,    0,    2,    3,    2,
       2,   -1,    2,    0,    1,
       0,    1,    2,    2,    1,
       0,   -1,    2,    4,    2,
       2,   -1,    2,    2,    2,
       0,    2,   -2,    2,    0,
      -1,   -1,    2,   -1,    1,
       0,   -2,    0,    0,    1,
       1,    0,    2,   -4,    2,
       1,   -1,    0,   -2,    1,
      -1,   -1,    2,    0,    1,
       1,   -1,    2,   -2,    2,
      -2,   -1,    0,    4,    0,
      -1,    0,    0,    3,    0,
      -2,   -1,    2,    2,    2,
       0,    2,    2,    0,    2,
       1,    1,    0,    2,    0,
       2,    0,    2,   -1,    2,
       1,    0,    2,    1,    1,
       4,    0,    0,    0,    0,
       2,    1,    2,    0,    1,
       3,   -1,    2,    0,    2,
      -2,    2,    0,    2,    1,
       1,    0,    2,   -3,    1,
       1,    1,    2,   -4,    1,
      -1,   -1,    2,   -2,    1,
       0,   -1,    0,   -1,    1,
       0,   -1,    0,   -2,    1,
      -2,    0,    0,    0,    2,
      -2,    0,   -2,    2,    0,
      -1,    0,   -2,    4,    0,
       1,   -2,    0,    0,    0,
       0,    1,    0,    1,    1,
      -1,    2,    0,    2,    0,
       1,   -1,    2,   -2,    1,
       1,    2,    2,   -2,    2,
       2,   -1,    2,   -2,    2,
       1,    0,    2,   -1,    1,
       2,    1,    2,   -2,    1,
      -2,    0,    0,   -2,    1,
       1,   -2,    2,    0,    2,
       0,    1,    2,    1,    1,
       1,    0,    4,   -2,    1,
      -2,    0,    4,    2,    2,
       1,    1,    2,    1,    2,
       1,    0,    0,    4,    0,
       1,    0,    2,    2,    0,
       2,    0,    2,    1,    2,
       3,    1,    2,    0,    2,
       4,    0,    2,    0,    1,
      -2,   -1,    2,    0,    0,
       0,    1,   -2,    2,    1,
       1,    0,   -2,    1,    0,
       0,   -1,   -2,    2,    1,
       2,   -1,    0,   -2,    1,
      -1,    0,    2,   -1,    2,
       1,    0,    2,   -3,    2,
       0,    1,    2,   -2,    3,
       0,    0,    2,   -3,    1,
      -1,    0,   -2,    2,    1,
       0,    0,    2,   -4,    2,
      -2,    1,    0,    0,    1,
      -1,    0,    0,   -1,    1,
       2,    0,    2,   -4,    2,
       0,    0,    4,   -4,    4,
       0,    0,    4,   -4,    2,
      -1,   -2,    0,    2,    1,
      -2,    0,    0,    3,    0,
       1,    0,   -2,    2,    1,
      -3,    0,    2,    2,    2,
      -3,    0,    2,    2,    1,
      -2,    0,    2,    2,    0,
       2,   -1,    0,    0,    1,
      -2,    1,    2,    2,    2,
       1,    1,    0,    1,    0,
       0,    1,    4,   -2,    2,
      -1,    1,    0,   -2,    1,
       0,    0,    0,   -4,    1,
       1,   -1,    0,    2,    1,
       1,    1,    0,    2,    1,
      -1,    2,    2,    2,    2,
       3,    1,    2,   -2,    2,
       0,   -1,    0,    4,    0,
       2,   -1,    0,    2,    0,
       0,    0,    4,    0,    1,
       2,    0,    4,   -2,    2,
      -1,   -1,    2,    4,    1,
       1,    0,    0,    4,    1,
       1,   -2,    2,    2,    2,
       0,    0,    2,    3,    2,
      -1,    1,    2,    4,    2,
       3,    0,    0,    2,    0,
      -1,    0,    4,    2,    2,
       1,    1,    2,    2,    1,
      -2,    0,    2,    6,    2,
       2,    1,    2,    2,    2,
      -1,    0,    2,    6,    2,
       1,    0,    2,    4,    1,
       2,    0,    2,    4,    2,
       1,    1,   -2,    1,    0,
      -3,    1,    2,    1,    2,
       2,    0,   -2,    0,    2,
      -1,    0,    0,    1,    2,
      -4,    0,    2,    2,    1,
      -1,   -1,    0,    1,    0,
       0,    0,   -2,    2,    2,
       1,    0,    0,   -1,    2,
       0,   -1,    2,   -2,    3,
      -2,    1,    2,    0,    0,
       0,    0,    2,   -2,    4,
      -2,   -2,    0,    2,    0,
      -2,    0,   -2,    4,    0,
       0,   -2,   -2,    2,    0,
       1,    2,    0,   -2,    1,
       3,    0,    0,   -4,    1,
      -1,    1,    2,   -2,    2,
       1,   -1,    2,   -4,    1,
       1,    1,    0,   -2,    2,
      -3,    0,    2,    0,    0,
      -3,    0,    2,    0,    2,
      -2,    0,    0,    1,    0,
       0,    0,   -2,    1,    0,
      -3,    0,    0,    2,    1,
      -1,   -1,   -2,    2,    0,
       0,    1,    2,   -4,    1,
       2,    1,    0,   -4,    1,
       0,    2,    0,   -2,    1,
       1,    0,    0,   -3,    1,
      -2,    0,    2,   -2,    2,
      -2,   -1,    0,    0,    1,
      -4,    0,    0,    2,    0,
       1,    1,    0,   -4,    1,
      -1,    0,    2,   -4,    1,
       0,    0,    4,   -4,    1,
       0,    3,    2,   -2,    2,
      -3,   -1,    0,    4,    0,
      -3,    0,    0,    4,    1,
       1,   -1,   -2,    2,    0,
      -1,   -1,    0,    2,    2,
       1,   -2,    0,    0,    1,
       1,   -1,    0,    0,    2,
       0,    0,    0,    1,    2,
      -1,   -1,    2,    0,    0,
       1,   -2,    2,   -2,    2,
       0,   -1,    2,   -1,    1,
      -1,    0,    2,    0,    3,
       1,    1,    0,    0,    2,
      -1,    1,    2,    0,    0,
       1,    2,    0,    0,    0,
      -1,    2,    2,    0,    2,
      -1,    0,    4,   -2,    1,
       3,    0,    2,   -4,    2,
       1,    2,    2,   -2,    1,
       1,    0,    4,   -4,    2,
      -2,   -1,    0,    4,    1,
       0,   -1,    0,    2,    2,
      -2,    1,    0,    4,    0,
      -2,   -1,    2,    2,    1,
       2,    0,   -2,    2,    0,
       1,    0,    0,    1,    1,
       0,    1,    0,    2,    2,
       1,   -1,    2,   -1,    2,
      -2,    0,    4,    0,    1,
       2,    1,    0,    0,    1,
       0,    1,    2,    0,    0,
       0,   -1,    4,   -2,    2,
       0,    0,    4,   -2,    4,
       0,    2,    2,    0,    1,
      -3,    0,    0,    6,    0,
      -1,   -1,    0,    4,    1,
       1,   -2,    0,    2,    0,
      -1,    0,    0,    4,    2,
      -1,   -2,    2,    2,    1,
      -1,    0,    0,   -2,    2,
       1,    0,   -2,   -2,    1,
       0,    0,   -2,   -2,    1,
      -2,    0,   -2,    0,    1,
       0,    0,    0,    3,    1,
       0,    0,    0,    3,    0,
      -1,    1,    0,    4,    0,
      -1,   -1,    2,    2,    0,
      -2,    0,    2,    3,    2,
       1,    0,    0,    2,    2,
       0,   -1,    2,    1,    2,
       3,   -1,    0,    0,    0,
       2,    0,    0,    1,    0,
       1,   -1,    2,    0,    0,
       0,    0,    2,    1,    0,
       1,    0,    2,    0,    3,
       3,    1,    0,    0,    0,
       3,   -1,    2,   -2,    2,
       2,    0,    2,   -1,    1,
       1,    1,    2,    0,    0,
       0,    0,    4,   -1,    2,
       1,    2,    2,    0,    2,
      -2,    0,    0,    6,    0,
       0,   -1,    0,    4,    1,
      -2,   -1,    2,    4,    1,
       0,   -2,    2,    2,    1,
       0,   -1,    2,    2,    0,
      -1,    0,    2,    3,    1,
      -2,    1,    2,    4,    2,
       2,    0,    0,    2,    2,
       2,   -2,    2,    0,    2,
      -1,    1,    2,    3,    2,
       3,    0,    2,   -1,    2,
       4,    0,    2,   -2,    1,
      -1,    0,    0,    6,    0,
      -1,   -2,    2,    4,    2,
      -3,    0,    2,    6,    2,
      -1,    0,    2,    4,    0,
       3,    0,    0,    2,    1,
       3,   -1,    2,    0,    1,
       3,    0,    2,    0,    0,
       1,    0,    4,    0,    2,
       5,    0,    2,   -2,    2,
       0,   -1,    2,    4,    1,
       2,   -1,    2,    2,    1,
       0,    1,    2,    4,    2,
       1,   -1,    2,    4,    2,
       3,   -1,    2,    2,    2,
       3,    0,    2,    2,    1,
       5,    0,    2,    0,    2,
       0,    0,    2,    6,    2,
       4,    0,    2,    2,    2,
       0,   -1,    1,   -1,    1,
      -1,    0,    1,    0,    3,
       0,   -2,    2,   -2,    3,
       1,    0,   -1,    0,    1,
       2,   -2,    0,   -2,    1,
      -1,    0,    1,    0,    2,
      -1,    0,    1,    0,    1,
      -1,   -1,    2,   -1,    2,
      -2,    2,    0,    2,    2,
      -1,    0,    1,    0,    0,
      -4,    1,    2,    2,    2,
      -3,    0,    2,    1,    1,
      -2,   -1,    2,    0,    2,
       1,    0,   -2,    1,    1,
       2,   -1,   -2,    0,    1,
      -4,    0,    2,    2,    0,
      -3,    1,    0,    3,    0,
      -1,    0,   -1,    2,    0,
       0,   -2,    0,    0,    2,
       0,   -2,    0,    0,    2,
      -3,    0,    0,    3,    0,
      -2,   -1,    0,    2,    2,
      -1,    0,   -2,    3,    0,
      -4,    0,    0,    4,    0,
       2,    1,   -2,    0,    1,
       2,   -1,    0,   -2,    2,
       0,    0,    1,   -1,    0,
      -1,    2,    0,    1,    0,
      -2,    1,    2,    0,    2,
       1,    1,    0,   -1,    1,
       1,    0,    1,   -2,    1,
       0,    2,    0,    0,    2,
       1,   -1,    2,   -3,    1,
      -1,    1,    2,   -1,    1,
      -2,    0,    4,   -2,    2,
      -2,    0,    4,   -2,    1,
      -2,   -2,    0,    2,    1,
      -2,    0,   -2,    4,    0,
       1,    2,    2,   -4,    1,
       1,    1,    2,   -4,    2,
      -1,    2,    2,   -2,    1,
       2,    0,    0,   -3,    1,
      -1,    2,    0,    0,    1,
       0,    0,    0,   -2,    0,
      -1,   -1,    2,   -2,    2,
      -1,    1,    0,    0,    2,
       0,    0,    0,   -1,    2,
      -2,    1,    0,    1,    0,
       1,   -2,    0,   -2,    1,
       1,    0,   -2,    0,    2,
      -3,    1,    0,    2,    0,
      -1,    1,   -2,    2,    0,
      -1,   -1,    0,    0,    2,
      -3,    0,    0,    2,    0,
      -3,   -1,    0,    2,    0,
       2,    0,    2,   -6,    1,
       0,    1,    2,   -4,    2,
       2,    0,    0,   -4,    2,
      -2,    1,    2,   -2,    1,
       0,   -1,    2,   -4,    1,
       0,    1,    0,   -2,    2,
      -1,    0,    0,   -2,    0,
       2,    0,   -2,   -2,    1,
      -4,    0,    2,    0,    1,
      -1,   -1,    0,   -1,    1,
       0,    0,   -2,    0,    2,
      -3,    0,    0,    1,    0,
      -1,    0,   -2,    1,    0,
      -2,    0,   -2,    2,    1,
       0,    0,   -4,    2,    0,
      -2,   -1,   -2,    2,    0,
       1,    0,    2,   -6,    1,
      -1,    0,    2,   -4,    2,
       1,    0,    0,   -4,    2,
       2,    1,    2,   -4,    2,
       2,    1,    2,   -4,    1,
       0,    1,    4,   -4,    4,
       0,    1,    4,   -4,    2,
      -1,   -1,   -2,    4,    0,
      -1,   -3,    0,    2,    0,
      -1,    0,   -2,    4,    1,
      -2,   -1,    0,    3,    0,
       0,    0,   -2,    3,    0,
      -2,    0,    0,    3,    1,
       0,   -1,    0,    1,    0,
      -3,    0,    2,    2,    0,
       1,    1,   -2,    2,    0,
      -1,    1,    0,    2,    2,
       1,   -2,    2,   -2,    1,
       0,    0,    1,    0,    2,
       0,    0,    1,    0,    1,
       0,    0,    1,    0,    0,
      -1,    2,    0,    2,    1,
       0,    0,    2,    0,    2,
      -2,    0,    2,    0,    2,
       2,    0,    0,   -1,    1,
       3,    0,    0,   -2,    1,
       1,    0,    2,   -2,    3,
       1,    2,    0,    0,    1,
       2,    0,    2,   -3,    2,
      -1,    1,    4,   -2,    2,
      -2,   -2,    0,    4,    0,
       0,   -3,    0,    2,    0,
       0,    0,   -2,    4,    0,
      -1,   -1,    0,    3,    0,
      -2,    0,    0,    4,    2,
      -1,    0,    0,    3,    1,
       2,   -2,    0,    0,    0,
       1,   -1,    0,    1,    0,
      -1,    0,    0,    2,    0,
       0,   -2,    2,    0,    1,
      -1,    0,    1,    2,    1,
      -1,    1,    0,    3,    0,
      -1,   -1,    2,    1,    2,
       0,   -1,    2,    0,    0,
      -2,    1,    2,    2,    1,
       2,   -2,    2,   -2,    2,
       1,    1,    0,    1,    1,
       1,    0,    1,    0,    1,
       1,    0,    1,    0,    0,
       0,    2,    0,    2,    0,
       2,   -1,    2,   -2,    1,
       0,   -1,    4,   -2,    1,
       0,    0,    4,   -2,    3,
       0,    1,    4,   -2,    1,
       4,    0,    2,   -4,    2,
       2,    2,    2,   -2,    2,
       2,    0,    4,   -4,    2,
      -1,   -2,    0,    4,    0,
      -1,   -3,    2,    2,    2,
      -3,    0,    2,    4,    2,
      -3,    0,    2,   -2,    1,
      -1,   -1,    0,   -2,    1,
      -3,    0,    0,    0,    2,
      -3,    0,   -2,    2,    0,
       0,    1,    0,   -4,    1,
      -2,    1,    0,   -2,    1,
      -4,    0,    0,    0,    1,
      -1,    0,    0,   -4,    1,
      -3,    0,    0,   -2,    1,
       0,    0,    0,    3,    2,
      -1,    1,    0,    4,    1,
       1,   -2,    2,    0,    1,
       0,    1,    0,    3,    0,
      -1,    0,    2,    2,    3,
       0,    0,    2,    2,    2,
      -2,    0,    2,    2,    2,
      -1,    1,    2,    2,    0,
       3,    0,    0,    0,    2,
       2,    1,    0,    1,    0,
       2,   -1,    2,   -1,    2,
       0,    0,    2,    0,    1,
       0,    0,    3,    0,    3,
       0,    0,    3,    0,    2,
      -1,    2,    2,    2,    1,
      -1,    0,    4,    0,    0,
       1,    2,    2,    0,    1,
       3,    1,    2,   -2,    1,
       1,    1,    4,   -2,    2,
      -2,   -1,    0,    6,    0,
       0,   -2,    0,    4,    0,
      -2,    0,    0,    6,    1,
      -2,   -2,    2,    4,    2,
       0,   -3,    2,    2,    2,
       0,    0,    0,    4,    2,
      -1,   -1,    2,    3,    2,
      -2,    0,    2,    4,    0,
       2,   -1,    0,    2,    1,
       1,    0,    0,    3,    0,
       0,    1,    0,    4,    1,
       0,    1,    0,    4,    0,
       1,   -1,    2,    1,    2,
       0,    0,    2,    2,    3,
       1,    0,    2,    2,    2,
      -1,    0,    2,    2,    2,
      -2,    0,    4,    2,    1,
       2,    1,    0,    2,    1,
       2,    1,    0,    2,    0,
       2,   -1,    2,    0,    0,
       1,    0,    2,    1,    0,
       0,    1,    2,    2,    0,
       2,    0,    2,    0,    3,
       3,    0,    2,    0,    2,
       1,    0,    2,    0,    2,
       1,    0,    3,    0,    3,
       1,    1,    2,    1,    1,
       0,    2,    2,    2,    2,
       2,    1,    2,    0,    0,
       2,    0,    4,   -2,    1,
       4,    1,    2,   -2,    2,
      -1,   -1,    0,    6,    0,
      -3,   -1,    2,    6,    2,
      -1,    0,    0,    6,    1,
      -3,    0,    2,    6,    1,
       1,   -1,    0,    4,    1,
       1,   -1,    0,    4,    0,
      -2,    0,    2,    5,    2,
       1,   -2,    2,    2,    1,
       3,   -1,    0,    2,    0,
       1,   -1,    2,    2,    0,
       0,    0,    2,    3,    1,
      -1,    1,    2,    4,    1,
       0,    1,    2,    3,    2,
      -1,    0,    4,    2,    1,
       2,    0,    2,    1,    1,
       5,    0,    0,    0,    0,
       2,    1,    2,    1,    2,
       1,    0,    4,    0,    1,
       3,    1,    2,    0,    1,
       3,    0,    4,   -2,    2,
      -2,   -1,    2,    6,    2,
       0,    0,    0,    6,    0,
       0,   -2,    2,    4,    2,
      -2,    0,    2,    6,    1,
       2,    0,    0,    4,    1,
       2,    0,    0,    4,    0,
       2,   -2,    2,    2,    2,
       0,    0,    2,    4,    0,
       1,    0,    2,    3,    2,
       4,    0,    0,    2,    0,
       2,    0,    2,    2,    0,
       0,    0,    4,    2,    2,
       4,   -1,    2,    0,    2,
       3,    0,    2,    1,    2,
       2,    1,    2,    2,    1,
       4,    1,    2,    0,    2,
      -1,   -1,    2,    6,    2,
      -1,    0,    2,    6,    1,
       1,   -1,    2,    4,    1,
       1,    1,    2,    4,    2,
       3,    1,    2,    2,    2,
       5,    0,    2,    0,    1,
       2,   -1,    2,    4,    2,
       2,    0,    2,    4,    1};

/*
   Luni-Solar nutation coefficients, unit 1e-7 arcsec:
   longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)

   Each row of coefficients in 'cls_t' belongs with the corresponding
   row of fundamental-argument multipliers in 'nals_t'.
*/

   const double cls_t[678][6] =
      {-172064161.0, -174666.0,  33386.0, 92052331.0,  9086.0, 15377.0,
        -13170906.0,   -1675.0, -13696.0,  5730336.0, -3015.0, -4587.0,
         -2276413.0,    -234.0,   2796.0,   978459.0,  -485.0,  1374.0,
          2074554.0,     207.0,   -698.0,  -897492.0,   470.0,  -291.0,
          1475877.0,   -3633.0,  11817.0,    73871.0,  -184.0, -1924.0,
          -516821.0,    1226.0,   -524.0,   224386.0,  -677.0,  -174.0,
           711159.0,      73.0,   -872.0,    -6750.0,     0.0,   358.0,
          -387298.0,    -367.0,    380.0,   200728.0,    18.0,   318.0,
          -301461.0,     -36.0,    816.0,   129025.0,   -63.0,   367.0,
           215829.0,    -494.0,    111.0,   -95929.0,   299.0,   132.0,
           128227.0,     137.0,    181.0,   -68982.0,    -9.0,    39.0,
           123457.0,      11.0,     19.0,   -53311.0,    32.0,    -4.0,
           156994.0,      10.0,   -168.0,    -1235.0,     0.0,    82.0,
            63110.0,      63.0,     27.0,   -33228.0,     0.0,    -9.0,
           -57976.0,     -63.0,   -189.0,    31429.0,     0.0,   -75.0,
           -59641.0,     -11.0,    149.0,    25543.0,   -11.0,    66.0,
           -51613.0,     -42.0,    129.0,    26366.0,     0.0,    78.0,
            45893.0,      50.0,     31.0,   -24236.0,   -10.0,    20.0,
            63384.0,      11.0,   -150.0,    -1220.0,     0.0,    29.0,
           -38571.0,      -1.0,    158.0,    16452.0,   -11.0,    68.0,
            32481.0,       0.0,      0.0,   -13870.0,     0.0,     0.0,
           -47722.0,       0.0,    -18.0,      477.0,     0.0,   -25.0,
           -31046.0,      -1.0,    131.0,    13238.0,   -11.0,    59.0,
            28593.0,       0.0,     -1.0,   -12338.0,    10.0,    -3.0,
            20441.0,      21.0,     10.0,   -10758.0,     0.0,    -3.0,
            29243.0,       0.0,    -74.0,     -609.0,     0.0,    13.0,
            25887.0,       0.0,    -66.0,     -550.0,     0.0,    11.0,
           -14053.0,     -25.0,     79.0,     8551.0,    -2.0,   -45.0,
            15164.0,      10.0,     11.0,    -8001.0,     0.0,    -1.0,
           -15794.0,      72.0,    -16.0,     6850.0,   -42.0,    -5.0,
            21783.0,       0.0,     13.0,     -167.0,     0.0,    13.0,
           -12873.0,     -10.0,    -37.0,     6953.0,     0.0,   -14.0,
           -12654.0,      11.0,     63.0,     6415.0,     0.0,    26.0,
           -10204.0,       0.0,     25.0,     5222.0,     0.0,    15.0,
            16707.0,     -85.0,    -10.0,      168.0,    -1.0,    10.0,
            -7691.0,       0.0,     44.0,     3268.0,     0.0,    19.0,
           -11024.0,       0.0,    -14.0,      104.0,     0.0,     2.0,
             7566.0,     -21.0,    -11.0,    -3250.0,     0.0,    -5.0,
            -6637.0,     -11.0,     25.0,     3353.0,     0.0,    14.0,
            -7141.0,      21.0,      8.0,     3070.0,     0.0,     4.0,
            -6302.0,     -11.0,      2.0,     3272.0,     0.0,     4.0,
             5800.0,      10.0,      2.0,    -3045.0,     0.0,    -1.0,
             6443.0,       0.0,     -7.0,    -2768.0,     0.0,    -4.0,
            -5774.0,     -11.0,    -15.0,     3041.0,     0.0,    -5.0,
            -5350.0,       0.0,     21.0,     2695.0,     0.0,    12.0,
            -4752.0,     -11.0,     -3.0,     2719.0,     0.0,    -3.0,
            -4940.0,     -11.0,    -21.0,     2720.0,     0.0,    -9.0,
             7350.0,       0.0,     -8.0,      -51.0,     0.0,     4.0,
             4065.0,       0.0,      6.0,    -2206.0,     0.0,     1.0,
             6579.0,       0.0,    -24.0,     -199.0,     0.0,     2.0,
             3579.0,       0.0,      5.0,    -1900.0,     0.0,     1.0,
             4725.0,       0.0,     -6.0,      -41.0,     0.0,     3.0,
            -3075.0,       0.0,     -2.0,     1313.0,     0.0,    -1.0,
            -2904.0,       0.0,     15.0,     1233.0,     0.0,     7.0,
             4348.0,       0.0,    -10.0,      -81.0,     0.0,     2.0,
            -2878.0,       0.0,      8.0,     1232.0,     0.0,     4.0,
            -4230.0,       0.0,      5.0,      -20.0,     0.0,    -2.0,
            -2819.0,       0.0,      7.0,     1207.0,     0.0,     3.0,
            -4056.0,       0.0,      5.0,       40.0,     0.0,    -2.0,
            -2647.0,       0.0,     11.0,     1129.0,     0.0,     5.0,
            -2294.0,       0.0,    -10.0,     1266.0,     0.0,    -4.0,
             2481.0,       0.0,     -7.0,    -1062.0,     0.0,    -3.0,
             2179.0,       0.0,     -2.0,    -1129.0,     0.0,    -2.0,
             3276.0,       0.0,      1.0,       -9.0,     0.0,     0.0,
            -3389.0,       0.0,      5.0,       35.0,     0.0,    -2.0,
             3339.0,       0.0,    -13.0,     -107.0,     0.0,     1.0,
            -1987.0,       0.0,     -6.0,     1073.0,     0.0,    -2.0,
            -1981.0,       0.0,      0.0,      854.0,     0.0,     0.0,
             4026.0,       0.0,   -353.0,     -553.0,     0.0,  -139.0,
             1660.0,       0.0,     -5.0,     -710.0,     0.0,    -2.0,
            -1521.0,       0.0,      9.0,      647.0,     0.0,     4.0,
             1314.0,       0.0,      0.0,     -700.0,     0.0,     0.0,
            -1283.0,       0.0,      0.0,      672.0,     0.0,     0.0,
            -1331.0,       0.0,      8.0,      663.0,     0.0,     4.0,
             1383.0,       0.0,     -2.0,     -594.0,     0.0,    -2.0,
             1405.0,       0.0,      4.0,     -610.0,     0.0,     2.0,
             1290.0,       0.0,      0.0,     -556.0,     0.0,     0.0,
            -1214.0,       0.0,      5.0,      518.0,     0.0,     2.0,
             1146.0,       0.0,     -3.0,     -490.0,     0.0,    -1.0,
             1019.0,       0.0,     -1.0,     -527.0,     0.0,    -1.0,
            -1100.0,       0.0,      9.0,      465.0,     0.0,     4.0,
             -970.0,       0.0,      2.0,      496.0,     0.0,     1.0,
             1575.0,       0.0,     -6.0,      -50.0,     0.0,     0.0,
              934.0,       0.0,     -3.0,     -399.0,     0.0,    -1.0,
              922.0,       0.0,     -1.0,     -395.0,     0.0,    -1.0,
              815.0,       0.0,     -1.0,     -422.0,     0.0,    -1.0,
              834.0,       0.0,      2.0,     -440.0,     0.0,     1.0,
             1248.0,       0.0,      0.0,     -170.0,     0.0,     1.0,
             1338.0,       0.0,     -5.0,      -39.0,     0.0,     0.0,
              716.0,       0.0,     -2.0,     -389.0,     0.0,    -1.0,
             1282.0,       0.0,     -3.0,      -23.0,     0.0,     1.0,
              742.0,       0.0,      1.0,     -391.0,     0.0,     0.0,
             1020.0,       0.0,    -25.0,     -495.0,     0.0,   -10.0,
              715.0,       0.0,     -4.0,     -326.0,     0.0,     2.0,
             -666.0,       0.0,     -3.0,      369.0,     0.0,    -1.0,
             -667.0,       0.0,      1.0,      346.0,     0.0,     1.0,
             -704.0,       0.0,      0.0,      304.0,     0.0,     0.0,
             -694.0,       0.0,      5.0,      294.0,     0.0,     2.0,
            -1014.0,       0.0,     -1.0,        4.0,     0.0,    -1.0,
             -585.0,       0.0,     -2.0,      316.0,     0.0,    -1.0,
             -949.0,       0.0,      1.0,        8.0,     0.0,    -1.0,
             -595.0,       0.0,      0.0,      258.0,     0.0,     0.0,
              528.0,       0.0,      0.0,     -279.0,     0.0,     0.0,
             -590.0,       0.0,      4.0,      252.0,     0.0,     2.0,
              570.0,       0.0,     -2.0,     -244.0,     0.0,    -1.0,
             -502.0,       0.0,      3.0,      250.0,     0.0,     2.0,
             -875.0,       0.0,      1.0,       29.0,     0.0,     0.0,
             -492.0,       0.0,     -3.0,      275.0,     0.0,    -1.0,
              535.0,       0.0,     -2.0,     -228.0,     0.0,    -1.0,
             -467.0,       0.0,      1.0,      240.0,     0.0,     1.0,
              591.0,       0.0,      0.0,     -253.0,     0.0,     0.0,
             -453.0,       0.0,     -1.0,      244.0,     0.0,    -1.0,
              766.0,       0.0,      1.0,        9.0,     0.0,     0.0,
             -446.0,       0.0,      2.0,      225.0,     0.0,     1.0,
             -488.0,       0.0,      2.0,      207.0,     0.0,     1.0,
             -468.0,       0.0,      0.0,      201.0,     0.0,     0.0,
             -421.0,       0.0,      1.0,      216.0,     0.0,     1.0,
              463.0,       0.0,      0.0,     -200.0,     0.0,     0.0,
             -673.0,       0.0,      2.0,       14.0,     0.0,     0.0,
              658.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
             -438.0,       0.0,      0.0,      188.0,     0.0,     0.0,
             -390.0,       0.0,      0.0,      205.0,     0.0,     0.0,
              639.0,     -11.0,     -2.0,      -19.0,     0.0,     0.0,
              412.0,       0.0,     -2.0,     -176.0,     0.0,    -1.0,
             -361.0,       0.0,      0.0,      189.0,     0.0,     0.0,
              360.0,       0.0,     -1.0,     -185.0,     0.0,    -1.0,
              588.0,       0.0,     -3.0,      -24.0,     0.0,     0.0,
             -578.0,       0.0,      1.0,        5.0,     0.0,     0.0,
             -396.0,       0.0,      0.0,      171.0,     0.0,     0.0,
              565.0,       0.0,     -1.0,       -6.0,     0.0,     0.0,
             -335.0,       0.0,     -1.0,      184.0,     0.0,    -1.0,
              357.0,       0.0,      1.0,     -154.0,     0.0,     0.0,
              321.0,       0.0,      1.0,     -174.0,     0.0,     0.0,
             -301.0,       0.0,     -1.0,      162.0,     0.0,     0.0,
             -334.0,       0.0,      0.0,      144.0,     0.0,     0.0,
              493.0,       0.0,     -2.0,      -15.0,     0.0,     0.0,
              494.0,       0.0,     -2.0,      -19.0,     0.0,     0.0,
              337.0,       0.0,     -1.0,     -143.0,     0.0,    -1.0,
              280.0,       0.0,     -1.0,     -144.0,     0.0,     0.0,
              309.0,       0.0,      1.0,     -134.0,     0.0,     0.0,
             -263.0,       0.0,      2.0,      131.0,     0.0,     1.0,
              253.0,       0.0,      1.0,     -138.0,     0.0,     0.0,
              245.0,       0.0,      0.0,     -128.0,     0.0,     0.0,
              416.0,       0.0,     -2.0,      -17.0,     0.0,     0.0,
             -229.0,       0.0,      0.0,      128.0,     0.0,     0.0,
              231.0,       0.0,      0.0,     -120.0,     0.0,     0.0,
             -259.0,       0.0,      2.0,      109.0,     0.0,     1.0,
              375.0,       0.0,     -1.0,       -8.0,     0.0,     0.0,
              252.0,       0.0,      0.0,     -108.0,     0.0,     0.0,
             -245.0,       0.0,      1.0,      104.0,     0.0,     0.0,
              243.0,       0.0,     -1.0,     -104.0,     0.0,     0.0,
              208.0,       0.0,      1.0,     -112.0,     0.0,     0.0,
              199.0,       0.0,      0.0,     -102.0,     0.0,     0.0,
             -208.0,       0.0,      1.0,      105.0,     0.0,     0.0,
              335.0,       0.0,     -2.0,      -14.0,     0.0,     0.0,
             -325.0,       0.0,      1.0,        7.0,     0.0,     0.0,
             -187.0,       0.0,      0.0,       96.0,     0.0,     0.0,
              197.0,       0.0,     -1.0,     -100.0,     0.0,     0.0,
             -192.0,       0.0,      2.0,       94.0,     0.0,     1.0,
             -188.0,       0.0,      0.0,       83.0,     0.0,     0.0,
              276.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
             -286.0,       0.0,      1.0,        6.0,     0.0,     0.0,
              186.0,       0.0,     -1.0,      -79.0,     0.0,     0.0,
             -219.0,       0.0,      0.0,       43.0,     0.0,     0.0,
              276.0,       0.0,      0.0,        2.0,     0.0,     0.0,
             -153.0,       0.0,     -1.0,       84.0,     0.0,     0.0,
             -156.0,       0.0,      0.0,       81.0,     0.0,     0.0,
             -154.0,       0.0,      1.0,       78.0,     0.0,     0.0,
             -174.0,       0.0,      1.0,       75.0,     0.0,     0.0,
             -163.0,       0.0,      2.0,       69.0,     0.0,     1.0,
             -228.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               91.0,       0.0,     -4.0,      -54.0,     0.0,    -2.0,
              175.0,       0.0,      0.0,      -75.0,     0.0,     0.0,
             -159.0,       0.0,      0.0,       69.0,     0.0,     0.0,
              141.0,       0.0,      0.0,      -72.0,     0.0,     0.0,
              147.0,       0.0,      0.0,      -75.0,     0.0,     0.0,
             -132.0,       0.0,      0.0,       69.0,     0.0,     0.0,
              159.0,       0.0,    -28.0,      -54.0,     0.0,    11.0,
              213.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
              123.0,       0.0,      0.0,      -64.0,     0.0,     0.0,
             -118.0,       0.0,     -1.0,       66.0,     0.0,     0.0,
              144.0,       0.0,     -1.0,      -61.0,     0.0,     0.0,
             -121.0,       0.0,      1.0,       60.0,     0.0,     0.0,
             -134.0,       0.0,      1.0,       56.0,     0.0,     1.0,
             -105.0,       0.0,      0.0,       57.0,     0.0,     0.0,
             -102.0,       0.0,      0.0,       56.0,     0.0,     0.0,
              120.0,       0.0,      0.0,      -52.0,     0.0,     0.0,
              101.0,       0.0,      0.0,      -54.0,     0.0,     0.0,
             -113.0,       0.0,      0.0,       59.0,     0.0,     0.0,
             -106.0,       0.0,      0.0,       61.0,     0.0,     0.0,
             -129.0,       0.0,      1.0,       55.0,     0.0,     0.0,
             -114.0,       0.0,      0.0,       57.0,     0.0,     0.0,
              113.0,       0.0,     -1.0,      -49.0,     0.0,     0.0,
             -102.0,       0.0,      0.0,       44.0,     0.0,     0.0,
              -94.0,       0.0,      0.0,       51.0,     0.0,     0.0,
             -100.0,       0.0,     -1.0,       56.0,     0.0,     0.0,
               87.0,       0.0,      0.0,      -47.0,     0.0,     0.0,
              161.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               96.0,       0.0,      0.0,      -50.0,     0.0,     0.0,
              151.0,       0.0,     -1.0,       -5.0,     0.0,     0.0,
             -104.0,       0.0,      0.0,       44.0,     0.0,     0.0,
             -110.0,       0.0,      0.0,       48.0,     0.0,     0.0,
             -100.0,       0.0,      1.0,       50.0,     0.0,     0.0,
               92.0,       0.0,     -5.0,       12.0,     0.0,    -2.0,
               82.0,       0.0,      0.0,      -45.0,     0.0,     0.0,
               82.0,       0.0,      0.0,      -45.0,     0.0,     0.0,
              -78.0,       0.0,      0.0,       41.0,     0.0,     0.0,
              -77.0,       0.0,      0.0,       43.0,     0.0,     0.0,
                2.0,       0.0,      0.0,       54.0,     0.0,     0.0,
               94.0,       0.0,      0.0,      -40.0,     0.0,     0.0,
              -93.0,       0.0,      0.0,       40.0,     0.0,     0.0,
              -83.0,       0.0,     10.0,       40.0,     0.0,    -2.0,
               83.0,       0.0,      0.0,      -36.0,     0.0,     0.0,
              -91.0,       0.0,      0.0,       39.0,     0.0,     0.0,
              128.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
              -79.0,       0.0,      0.0,       34.0,     0.0,     0.0,
              -83.0,       0.0,      0.0,       47.0,     0.0,     0.0,
               84.0,       0.0,      0.0,      -44.0,     0.0,     0.0,
               83.0,       0.0,      0.0,      -43.0,     0.0,     0.0,
               91.0,       0.0,      0.0,      -39.0,     0.0,     0.0,
              -77.0,       0.0,      0.0,       39.0,     0.0,     0.0,
               84.0,       0.0,      0.0,      -43.0,     0.0,     0.0,
              -92.0,       0.0,      1.0,       39.0,     0.0,     0.0,
              -92.0,       0.0,      1.0,       39.0,     0.0,     0.0,
              -94.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               68.0,       0.0,      0.0,      -36.0,     0.0,     0.0,
              -61.0,       0.0,      0.0,       32.0,     0.0,     0.0,
               71.0,       0.0,      0.0,      -31.0,     0.0,     0.0,
               62.0,       0.0,      0.0,      -34.0,     0.0,     0.0,
              -63.0,       0.0,      0.0,       33.0,     0.0,     0.0,
              -73.0,       0.0,      0.0,       32.0,     0.0,     0.0,
              115.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
             -103.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               63.0,       0.0,      0.0,      -28.0,     0.0,     0.0,
               74.0,       0.0,      0.0,      -32.0,     0.0,     0.0,
             -103.0,       0.0,     -3.0,        3.0,     0.0,    -1.0,
              -69.0,       0.0,      0.0,       30.0,     0.0,     0.0,
               57.0,       0.0,      0.0,      -29.0,     0.0,     0.0,
               94.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
               64.0,       0.0,      0.0,      -33.0,     0.0,     0.0,
              -63.0,       0.0,      0.0,       26.0,     0.0,     0.0,
              -38.0,       0.0,      0.0,       20.0,     0.0,     0.0,
              -43.0,       0.0,      0.0,       24.0,     0.0,     0.0,
              -45.0,       0.0,      0.0,       23.0,     0.0,     0.0,
               47.0,       0.0,      0.0,      -24.0,     0.0,     0.0,
              -48.0,       0.0,      0.0,       25.0,     0.0,     0.0,
               45.0,       0.0,      0.0,      -26.0,     0.0,     0.0,
               56.0,       0.0,      0.0,      -25.0,     0.0,     0.0,
               88.0,       0.0,      0.0,        2.0,     0.0,     0.0,
              -75.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               85.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               49.0,       0.0,      0.0,      -26.0,     0.0,     0.0,
              -74.0,       0.0,     -3.0,       -1.0,     0.0,    -1.0,
              -39.0,       0.0,      0.0,       21.0,     0.0,     0.0,
               45.0,       0.0,      0.0,      -20.0,     0.0,     0.0,
               51.0,       0.0,      0.0,      -22.0,     0.0,     0.0,
              -40.0,       0.0,      0.0,       21.0,     0.0,     0.0,
               41.0,       0.0,      0.0,      -21.0,     0.0,     0.0,
              -42.0,       0.0,      0.0,       24.0,     0.0,     0.0,
              -51.0,       0.0,      0.0,       22.0,     0.0,     0.0,
              -42.0,       0.0,      0.0,       22.0,     0.0,     0.0,
               39.0,       0.0,      0.0,      -21.0,     0.0,     0.0,
               46.0,       0.0,      0.0,      -18.0,     0.0,     0.0,
              -53.0,       0.0,      0.0,       22.0,     0.0,     0.0,
               82.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
               81.0,       0.0,     -1.0,       -4.0,     0.0,     0.0,
               47.0,       0.0,      0.0,      -19.0,     0.0,     0.0,
               53.0,       0.0,      0.0,      -23.0,     0.0,     0.0,
              -45.0,       0.0,      0.0,       22.0,     0.0,     0.0,
              -44.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
              -33.0,       0.0,      0.0,       16.0,     0.0,     0.0,
              -61.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               28.0,       0.0,      0.0,      -15.0,     0.0,     0.0,
              -38.0,       0.0,      0.0,       19.0,     0.0,     0.0,
              -33.0,       0.0,      0.0,       21.0,     0.0,     0.0,
              -60.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               48.0,       0.0,      0.0,      -10.0,     0.0,     0.0,
               27.0,       0.0,      0.0,      -14.0,     0.0,     0.0,
               38.0,       0.0,      0.0,      -20.0,     0.0,     0.0,
               31.0,       0.0,      0.0,      -13.0,     0.0,     0.0,
              -29.0,       0.0,      0.0,       15.0,     0.0,     0.0,
               28.0,       0.0,      0.0,      -15.0,     0.0,     0.0,
              -32.0,       0.0,      0.0,       15.0,     0.0,     0.0,
               45.0,       0.0,      0.0,       -8.0,     0.0,     0.0,
              -44.0,       0.0,      0.0,       19.0,     0.0,     0.0,
               28.0,       0.0,      0.0,      -15.0,     0.0,     0.0,
              -51.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -36.0,       0.0,      0.0,       20.0,     0.0,     0.0,
               44.0,       0.0,      0.0,      -19.0,     0.0,     0.0,
               26.0,       0.0,      0.0,      -14.0,     0.0,     0.0,
              -60.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               35.0,       0.0,      0.0,      -18.0,     0.0,     0.0,
              -27.0,       0.0,      0.0,       11.0,     0.0,     0.0,
               47.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               36.0,       0.0,      0.0,      -15.0,     0.0,     0.0,
              -36.0,       0.0,      0.0,       20.0,     0.0,     0.0,
              -35.0,       0.0,      0.0,       19.0,     0.0,     0.0,
              -37.0,       0.0,      0.0,       19.0,     0.0,     0.0,
               32.0,       0.0,      0.0,      -16.0,     0.0,     0.0,
               35.0,       0.0,      0.0,      -14.0,     0.0,     0.0,
               32.0,       0.0,      0.0,      -13.0,     0.0,     0.0,
               65.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               47.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               32.0,       0.0,      0.0,      -16.0,     0.0,     0.0,
               37.0,       0.0,      0.0,      -16.0,     0.0,     0.0,
              -30.0,       0.0,      0.0,       15.0,     0.0,     0.0,
              -32.0,       0.0,      0.0,       16.0,     0.0,     0.0,
              -31.0,       0.0,      0.0,       13.0,     0.0,     0.0,
               37.0,       0.0,      0.0,      -16.0,     0.0,     0.0,
               31.0,       0.0,      0.0,      -13.0,     0.0,     0.0,
               49.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               32.0,       0.0,      0.0,      -13.0,     0.0,     0.0,
               23.0,       0.0,      0.0,      -12.0,     0.0,     0.0,
              -43.0,       0.0,      0.0,       18.0,     0.0,     0.0,
               26.0,       0.0,      0.0,      -11.0,     0.0,     0.0,
              -32.0,       0.0,      0.0,       14.0,     0.0,     0.0,
              -29.0,       0.0,      0.0,       14.0,     0.0,     0.0,
              -27.0,       0.0,      0.0,       12.0,     0.0,     0.0,
               30.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -11.0,       0.0,      0.0,        5.0,     0.0,     0.0,
              -21.0,       0.0,      0.0,       10.0,     0.0,     0.0,
              -34.0,       0.0,      0.0,       15.0,     0.0,     0.0,
              -10.0,       0.0,      0.0,        6.0,     0.0,     0.0,
              -36.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -9.0,       0.0,      0.0,        4.0,     0.0,     0.0,
              -12.0,       0.0,      0.0,        5.0,     0.0,     0.0,
              -21.0,       0.0,      0.0,        5.0,     0.0,     0.0,
              -29.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
              -15.0,       0.0,      0.0,        3.0,     0.0,     0.0,
              -20.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               28.0,       0.0,      0.0,        0.0,     0.0,    -2.0,
               17.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -22.0,       0.0,      0.0,       12.0,     0.0,     0.0,
              -14.0,       0.0,      0.0,        7.0,     0.0,     0.0,
               24.0,       0.0,      0.0,      -11.0,     0.0,     0.0,
               11.0,       0.0,      0.0,       -6.0,     0.0,     0.0,
               14.0,       0.0,      0.0,       -6.0,     0.0,     0.0,
               24.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               18.0,       0.0,      0.0,       -8.0,     0.0,     0.0,
              -38.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -31.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -16.0,       0.0,      0.0,        8.0,     0.0,     0.0,
               29.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -18.0,       0.0,      0.0,       10.0,     0.0,     0.0,
              -10.0,       0.0,      0.0,        5.0,     0.0,     0.0,
              -17.0,       0.0,      0.0,       10.0,     0.0,     0.0,
                9.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
               16.0,       0.0,      0.0,       -6.0,     0.0,     0.0,
               22.0,       0.0,      0.0,      -12.0,     0.0,     0.0,
               20.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -13.0,       0.0,      0.0,        6.0,     0.0,     0.0,
              -17.0,       0.0,      0.0,        9.0,     0.0,     0.0,
              -14.0,       0.0,      0.0,        8.0,     0.0,     0.0,
                0.0,       0.0,      0.0,       -7.0,     0.0,     0.0,
               14.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               19.0,       0.0,      0.0,      -10.0,     0.0,     0.0,
              -34.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -20.0,       0.0,      0.0,        8.0,     0.0,     0.0,
                9.0,       0.0,      0.0,       -5.0,     0.0,     0.0,
              -18.0,       0.0,      0.0,        7.0,     0.0,     0.0,
               13.0,       0.0,      0.0,       -6.0,     0.0,     0.0,
               17.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -12.0,       0.0,      0.0,        5.0,     0.0,     0.0,
               15.0,       0.0,      0.0,       -8.0,     0.0,     0.0,
              -11.0,       0.0,      0.0,        3.0,     0.0,     0.0,
               13.0,       0.0,      0.0,       -5.0,     0.0,     0.0,
              -18.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -35.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                9.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
              -19.0,       0.0,      0.0,       10.0,     0.0,     0.0,
              -26.0,       0.0,      0.0,       11.0,     0.0,     0.0,
                8.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
              -10.0,       0.0,      0.0,        4.0,     0.0,     0.0,
               10.0,       0.0,      0.0,       -6.0,     0.0,     0.0,
              -21.0,       0.0,      0.0,        9.0,     0.0,     0.0,
              -15.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                9.0,       0.0,      0.0,       -5.0,     0.0,     0.0,
              -29.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -19.0,       0.0,      0.0,       10.0,     0.0,     0.0,
               12.0,       0.0,      0.0,       -5.0,     0.0,     0.0,
               22.0,       0.0,      0.0,       -9.0,     0.0,     0.0,
              -10.0,       0.0,      0.0,        5.0,     0.0,     0.0,
              -20.0,       0.0,      0.0,       11.0,     0.0,     0.0,
              -20.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -17.0,       0.0,      0.0,        7.0,     0.0,     0.0,
               15.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                8.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
               14.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -12.0,       0.0,      0.0,        6.0,     0.0,     0.0,
               25.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -13.0,       0.0,      0.0,        6.0,     0.0,     0.0,
              -14.0,       0.0,      0.0,        8.0,     0.0,     0.0,
               13.0,       0.0,      0.0,       -5.0,     0.0,     0.0,
              -17.0,       0.0,      0.0,        9.0,     0.0,     0.0,
              -12.0,       0.0,      0.0,        6.0,     0.0,     0.0,
              -10.0,       0.0,      0.0,        5.0,     0.0,     0.0,
               10.0,       0.0,      0.0,       -6.0,     0.0,     0.0,
              -15.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -22.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               28.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               15.0,       0.0,      0.0,       -7.0,     0.0,     0.0,
               23.0,       0.0,      0.0,      -10.0,     0.0,     0.0,
               12.0,       0.0,      0.0,       -5.0,     0.0,     0.0,
               29.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
              -25.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               22.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -18.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               15.0,       0.0,      0.0,        3.0,     0.0,     0.0,
              -23.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               12.0,       0.0,      0.0,       -5.0,     0.0,     0.0,
               -8.0,       0.0,      0.0,        4.0,     0.0,     0.0,
              -19.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -10.0,       0.0,      0.0,        4.0,     0.0,     0.0,
               21.0,       0.0,      0.0,       -9.0,     0.0,     0.0,
               23.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
              -16.0,       0.0,      0.0,        8.0,     0.0,     0.0,
              -19.0,       0.0,      0.0,        9.0,     0.0,     0.0,
              -22.0,       0.0,      0.0,       10.0,     0.0,     0.0,
               27.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               16.0,       0.0,      0.0,       -8.0,     0.0,     0.0,
               19.0,       0.0,      0.0,       -8.0,     0.0,     0.0,
                9.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
               -9.0,       0.0,      0.0,        4.0,     0.0,     0.0,
               -9.0,       0.0,      0.0,        4.0,     0.0,     0.0,
               -8.0,       0.0,      0.0,        4.0,     0.0,     0.0,
               18.0,       0.0,      0.0,       -9.0,     0.0,     0.0,
               16.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
              -10.0,       0.0,      0.0,        4.0,     0.0,     0.0,
              -23.0,       0.0,      0.0,        9.0,     0.0,     0.0,
               16.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
              -12.0,       0.0,      0.0,        6.0,     0.0,     0.0,
               -8.0,       0.0,      0.0,        4.0,     0.0,     0.0,
               30.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               24.0,       0.0,      0.0,      -10.0,     0.0,     0.0,
               10.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
              -16.0,       0.0,      0.0,        7.0,     0.0,     0.0,
              -16.0,       0.0,      0.0,        7.0,     0.0,     0.0,
               17.0,       0.0,      0.0,       -7.0,     0.0,     0.0,
              -24.0,       0.0,      0.0,       10.0,     0.0,     0.0,
              -12.0,       0.0,      0.0,        5.0,     0.0,     0.0,
              -24.0,       0.0,      0.0,       11.0,     0.0,     0.0,
              -23.0,       0.0,      0.0,        9.0,     0.0,     0.0,
              -13.0,       0.0,      0.0,        5.0,     0.0,     0.0,
              -15.0,       0.0,      0.0,        7.0,     0.0,     0.0,
                0.0,       0.0,  -1988.0,        0.0,     0.0, -1679.0,
                0.0,       0.0,    -63.0,        0.0,     0.0,   -27.0,
               -4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                0.0,       0.0,      5.0,        0.0,     0.0,     4.0,
                5.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                0.0,       0.0,    364.0,        0.0,     0.0,   176.0,
                0.0,       0.0,  -1044.0,        0.0,     0.0,  -891.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                0.0,       0.0,    330.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                0.0,       0.0,      5.0,        0.0,     0.0,     0.0,
                0.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                6.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        0.0,     0.0,     0.0,
              -12.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        3.0,     0.0,     0.0,
                7.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
                0.0,       0.0,    -12.0,        0.0,     0.0,   -10.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        3.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                0.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                7.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        3.0,     0.0,     0.0,
                5.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -8.0,       0.0,      0.0,        3.0,     0.0,     0.0,
                9.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                6.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                5.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                9.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                9.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                8.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                6.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                9.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        3.0,     0.0,     0.0,
              -13.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               10.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               10.0,       0.0,     13.0,        6.0,     0.0,    -5.0,
                0.0,       0.0,     30.0,        0.0,     0.0,    14.0,
                0.0,       0.0,   -162.0,        0.0,     0.0,  -138.0,
                0.0,       0.0,     75.0,        0.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        4.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                6.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                9.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                7.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -6.0,       0.0,     -3.0,        3.0,     0.0,     1.0,
                0.0,       0.0,     -3.0,        0.0,     0.0,    -2.0,
               11.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               11.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -1.0,       0.0,      3.0,        3.0,     0.0,    -1.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                0.0,       0.0,    -13.0,        0.0,     0.0,   -11.0,
                3.0,       0.0,      6.0,        0.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        3.0,     0.0,     0.0,
                8.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               11.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                8.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               11.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -6.0,       0.0,      0.0,        3.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -8.0,       0.0,      0.0,        4.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        3.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
                6.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -6.0,       0.0,      0.0,        3.0,     0.0,     0.0,
                6.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                6.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                6.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                0.0,       0.0,    -26.0,        0.0,     0.0,   -11.0,
                0.0,       0.0,    -10.0,        0.0,     0.0,    -5.0,
                5.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
              -13.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                7.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -6.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -7.0,       0.0,      0.0,        3.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               13.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
              -11.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                6.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
              -12.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
                0.0,       0.0,     -5.0,        0.0,     0.0,    -2.0,
               -7.0,       0.0,      0.0,        4.0,     0.0,     0.0,
                6.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
                3.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        3.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               12.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                6.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -6.0,       0.0,      0.0,        3.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                6.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                6.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -6.0,       0.0,      0.0,        3.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                7.0,       0.0,      0.0,       -4.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -5.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                5.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -6.0,       0.0,      0.0,        3.0,     0.0,     0.0,
               -6.0,       0.0,      0.0,        3.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               10.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                7.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                7.0,       0.0,      0.0,       -3.0,     0.0,     0.0,
                4.0,       0.0,      0.0,        0.0,     0.0,     0.0,
               11.0,       0.0,      0.0,        0.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -6.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                5.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -4.0,       0.0,      0.0,        2.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0,
                4.0,       0.0,      0.0,       -2.0,     0.0,     0.0,
                3.0,       0.0,      0.0,       -1.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        1.0,     0.0,     0.0,
               -3.0,       0.0,      0.0,        2.0,     0.0,     0.0};

/*
   Planetary argument multipliers:
       L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
*/
   const short int napl_t[687][14] =
      {0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -8, 16, -4, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  2,  2,
       0,  0,  0,  0,  0,  0,  0, -4,  8, -1, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0, 10, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  6, -3,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -4,  8, -3,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  6,  4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  2,
       2,  0, -1, -1,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0,
       1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  2, -4,  0, -3,  0,  0,  0,  0,
       1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0, -4, 10,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  0,  2,  0,  0, -5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -7,  4,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0, 18,-16,  0,  0,  0,  0,  0,  0,
      -2,  0,  1,  1,  2,  0,  0,  1,  0, -2,  0,  0,  0,  0,
      -1,  0,  1, -1,  1,  0, 18,-17,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  1,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  2,
       0,  0,  2, -2,  2,  0, -8, 11,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  8,-14,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  1,
      -2,  0,  0,  2,  1,  0,  0,  2,  0, -4,  5,  0,  0,  0,
      -2,  0,  0,  2,  2,  0,  3, -3,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  1,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  3, -5,  0,  2,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -4,  3,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,
      -1,  0,  1,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -2, -2,  0,  0,  0,
      -2,  0,  2,  0,  2,  0,  0, -5,  9,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0, -1,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  2,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2,
      -1,  0,  0,  1,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  2,  0,  0,  0,
       0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  2,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -9, 17,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  2,  0, -3,  5,  0,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  2,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,
       1,  0,  0, -2,  0,  0, 17,-16,  0, -2,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  1, -3,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  0,  5, -6,  0,  0,  0,  0,  0,
       0,  0, -2,  2,  0,  0,  0,  9,-13,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  1,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,
       0,  0, -2,  2,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  1,  0,  5, -7,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0,
       2,  0,  1, -3,  1,  0, -6,  7,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0,
       0,  0, -1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  2,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -9, 15,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0,
       1,  0, -1, -1,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0,
       2,  0,  0, -2,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -5,  5,  0,  0,  0,
       2,  0,  0, -2,  1,  0,  0, -6,  8,  0,  0,  0,  0,  0,
       2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
      -2,  0,  1,  1,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0,
      -2,  0,  1,  1,  1,  0,  0,  1,  0, -3,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -1, -5,  0,  0,  0,
      -1,  0,  0,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
      -1,  0,  1,  1,  1,  0,-20, 20,  0,  0,  0,  0,  0,  0,
       1,  0,  0, -2,  0,  0, 20,-21,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  8,-15,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0,-10, 15,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  4,  0,  0,  0,
       2,  0,  0, -2,  1,  0, -6,  8,  0,  0,  0,  0,  0,  0,
       0,  0, -2,  2,  1,  0,  5, -6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,
       0,  0,  2, -2,  1,  0,  0, -9, 13,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  7,-13,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -9, 17,  0,  0,  0,  0,  2,
       1,  0,  0, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0,
       1,  0,  0, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  2,  0,  0, -1,  2,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,
       0,  0, -2,  2,  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -5,  0,  2,  0,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  1,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  3, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  8,-12,  0,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0, -8, 11,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  1,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,
      -1,  0,  0,  0,  1,  0, 18,-16,  0,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  1,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  3, -7,  4,  0,  0,  0,  0,  0,
      -2,  0,  1,  1,  1,  0,  0, -3,  7,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0,  0, -1,  0, -2,  5,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0,
       1,  0,  0,  0,  1,  0,-10,  3,  0,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  0,  1,  0, 10, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0,
       2,  0, -1, -1,  1,  0,  0,  3, -7,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0,  0, -5,  0,  0,  0,
       0,  0,  0,  0,  1,  0, -3,  7, -4,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0,
      -2,  0,  1,  1,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0, -8, 12,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  1,
      -1,  0,  0,  1,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  1,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -2,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2,
       0,  0,  1, -1,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0, -3,  4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -2,  4,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  5, -8,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  6, -8,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -8, 15,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  0,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  0,  6, -8,  0,  0,  0,  0,  0,
       1,  0,  0, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2,
       0,  0,  1, -1,  2,  0,  0, -1,  0,  0, -1,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -7, 13,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0,  0,
       2,  0,  0, -2,  1,  0,  0, -5,  6,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -8, 11,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  3,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2,
      -2,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0,  0, -1,  0,  2,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0,  0,  0, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  2,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  3, -6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2,
       0,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0,  1, -4,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2,
       0,  0,  2, -2,  2,  0, -5,  6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0,  0, -1,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0, -1,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -2,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -6, 11,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0,  0,
       2,  0,  0, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -7,  9,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2,
       0,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2,
       0,  0,  0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  2, -4,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -4,  4,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  2,  0, -5,  7,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -4,  6,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  2,
       0,  0, -1,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
      -2,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       0,  0, -2,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -4,  8,  0,  0,  0,  0,  2,
       0,  0, -2,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -5, 10,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  1, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -7, 11,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -7, 11,  0,  0,  0,  0,  0,  1,
       0,  0, -2,  2,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0, -4,  4,  0,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -4,  6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  5,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -7, 12,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -3,  0,  3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  2,
      -2,  0,  0,  2,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -3,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,
       0,  0,  0,  0,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -8, 14,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -3,  8, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  5,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 12,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 12,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -2,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,  2,
       0,  0,  2, -2,  1,  0, -5,  5,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -2,  6,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -7, 10,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -7, 10,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -4,  8,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -4,  5,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -4,  5,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -2,  0,  5,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -9, 13,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -1,  5,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -2,  0,  4,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -2,  7,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -3,  9,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5, 10,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -3,  3,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -3,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -5, 13,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -1,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -6, 15,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -3,  9, -4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  2, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -2,  8, -1, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -6, 16, -4, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6, -8,  1,  5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  3, -5,  4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  3, -3,  0,  2,  0,  0,  0,  2,
       0,  0,  2, -2,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       0,  0,  1, -1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -1,  6,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -4,  4,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -3,  0,  5,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -9, 12,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -4,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -2,  6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -6,  7,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -2,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -1,  0,  0,  2,
       0,  0,  2, -2,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -8, 16,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0,  2, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  7, -8,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -5, 16, -4, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -1,  8, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -3,  8,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  5,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6, -5,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  4,  0, -4,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -6,  6,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4,  0,  0, -2,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5, -2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  8, -9,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -7,  7,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  5,  0, -4,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5,  0, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5,  0, -2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8,  8,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,
       1,  0,  0, -2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       1,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
       1,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       1,  0,  0, -2,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
      -1,  0,  0,  2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       1,  0,  0, -2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
       1,  0, -1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0,
      -2,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0,
       1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0,
      -1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,
       1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0,
      -1,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
      -1,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
      -1,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0,
       1,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
       1,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0,
       1,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
       1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
       1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0, -2,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0, -2,  3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       0,  0,  1,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
      -1,  0,  2,  0,  2,  0, 10, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
      -1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
       2,  0,  2, -2,  2,  0,  0, -2,  0,  3,  0,  0,  0,  0,
       1,  0,  2,  0,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
       0,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,
      -1,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
      -2,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0,  2, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0,  2, -2,  0,  0,  0,  0,  0,  0,
      -1,  0,  2,  2,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       1,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0,
      -1,  0,  2,  2,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0,
       2,  0,  2,  0,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0,
       1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
       1,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       0,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
       2,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
      -1,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0,
      -1,  0,  2,  2,  2,  0,  3, -3,  0,  0,  0,  0,  0,  0,
       1,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0,
       0,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0};

/*
   Planetary nutation coefficients, unit 1e-7 arcsec:
   longitude (sin, cos), obliquity (sin, cos)

   Each row of coefficients in 'cpl_t' belongs with the corresponding
   row of fundamental-argument multipliers in 'napl_t'.
*/

   const double cpl_t[687][4] =
      {1440.0,          0.0,          0.0,          0.0,
         56.0,       -117.0,        -42.0,        -40.0,
        125.0,        -43.0,          0.0,        -54.0,
          0.0,          5.0,          0.0,          0.0,
          3.0,         -7.0,         -3.0,          0.0,
          3.0,          0.0,          0.0,         -2.0,
       -114.0,          0.0,          0.0,         61.0,
       -219.0,         89.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          0.0,
       -462.0,       1604.0,          0.0,          0.0,
         99.0,          0.0,          0.0,        -53.0,
         -3.0,          0.0,          0.0,          2.0,
          0.0,          6.0,          2.0,          0.0,
          3.0,          0.0,          0.0,          0.0,
        -12.0,          0.0,          0.0,          0.0,
         14.0,       -218.0,        117.0,          8.0,
         31.0,       -481.0,       -257.0,        -17.0,
       -491.0,        128.0,          0.0,          0.0,
      -3084.0,       5123.0,       2735.0,       1647.0,
      -1444.0,       2409.0,      -1286.0,       -771.0,
         11.0,        -24.0,        -11.0,         -9.0,
         26.0,         -9.0,          0.0,          0.0,
        103.0,        -60.0,          0.0,          0.0,
          0.0,        -13.0,         -7.0,          0.0,
        -26.0,        -29.0,        -16.0,         14.0,
          9.0,        -27.0,        -14.0,         -5.0,
         12.0,          0.0,          0.0,         -6.0,
         -7.0,          0.0,          0.0,          0.0,
          0.0,         24.0,          0.0,          0.0,
        284.0,          0.0,          0.0,       -151.0,
        226.0,        101.0,          0.0,          0.0,
          0.0,         -8.0,         -2.0,          0.0,
          0.0,         -6.0,         -3.0,          0.0,
          5.0,          0.0,          0.0,         -3.0,
        -41.0,        175.0,         76.0,         17.0,
          0.0,         15.0,          6.0,          0.0,
        425.0,        212.0,       -133.0,        269.0,
       1200.0,        598.0,        319.0,       -641.0,
        235.0,        334.0,          0.0,          0.0,
         11.0,        -12.0,         -7.0,         -6.0,
          5.0,         -6.0,          3.0,          3.0,
         -5.0,          0.0,          0.0,          3.0,
          6.0,          0.0,          0.0,         -3.0,
         15.0,          0.0,          0.0,          0.0,
         13.0,          0.0,          0.0,         -7.0,
         -6.0,         -9.0,          0.0,          0.0,
        266.0,        -78.0,          0.0,          0.0,
       -460.0,       -435.0,       -232.0,        246.0,
          0.0,         15.0,          7.0,          0.0,
         -3.0,          0.0,          0.0,          2.0,
          0.0,        131.0,          0.0,          0.0,
          4.0,          0.0,          0.0,          0.0,
          0.0,          3.0,          0.0,          0.0,
          0.0,          4.0,          2.0,          0.0,
          0.0,          3.0,          0.0,          0.0,
        -17.0,        -19.0,        -10.0,          9.0,
         -9.0,        -11.0,          6.0,         -5.0,
         -6.0,          0.0,          0.0,          3.0,
        -16.0,          8.0,          0.0,          0.0,
          0.0,          3.0,          0.0,          0.0,
         11.0,         24.0,         11.0,         -5.0,
         -3.0,         -4.0,         -2.0,          1.0,
          3.0,          0.0,          0.0,         -1.0,
          0.0,         -8.0,         -4.0,          0.0,
          0.0,          3.0,          0.0,          0.0,
          0.0,          5.0,          0.0,          0.0,
          0.0,          3.0,          2.0,          0.0,
         -6.0,          4.0,          2.0,          3.0,
         -3.0,         -5.0,          0.0,          0.0,
         -5.0,          0.0,          0.0,          2.0,
          4.0,         24.0,         13.0,         -2.0,
        -42.0,         20.0,          0.0,          0.0,
        -10.0,        233.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          1.0,
         78.0,        -18.0,          0.0,          0.0,
          0.0,          3.0,          1.0,          0.0,
          0.0,         -3.0,         -1.0,          0.0,
          0.0,         -4.0,         -2.0,          1.0,
          0.0,         -8.0,         -4.0,         -1.0,
          0.0,         -5.0,          3.0,          0.0,
         -7.0,          0.0,          0.0,          3.0,
        -14.0,          8.0,          3.0,          6.0,
          0.0,          8.0,         -4.0,          0.0,
          0.0,         19.0,         10.0,          0.0,
         45.0,        -22.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          0.0,
          0.0,         -3.0,          0.0,          0.0,
          0.0,          3.0,          0.0,          0.0,
          3.0,          5.0,          3.0,         -2.0,
         89.0,        -16.0,         -9.0,        -48.0,
          0.0,          3.0,          0.0,          0.0,
         -3.0,          7.0,          4.0,          2.0,
       -349.0,        -62.0,          0.0,          0.0,
        -15.0,         22.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          0.0,
        -53.0,          0.0,          0.0,          0.0,
          5.0,          0.0,          0.0,         -3.0,
          0.0,         -8.0,          0.0,          0.0,
         15.0,         -7.0,         -4.0,         -8.0,
         -3.0,          0.0,          0.0,          1.0,
        -21.0,        -78.0,          0.0,          0.0,
         20.0,        -70.0,        -37.0,        -11.0,
          0.0,          6.0,          3.0,          0.0,
          5.0,          3.0,          2.0,         -2.0,
        -17.0,         -4.0,         -2.0,          9.0,
          0.0,          6.0,          3.0,          0.0,
         32.0,         15.0,         -8.0,         17.0,
        174.0,         84.0,         45.0,        -93.0,
         11.0,         56.0,          0.0,          0.0,
        -66.0,        -12.0,         -6.0,         35.0,
         47.0,          8.0,          4.0,        -25.0,
          0.0,          8.0,          4.0,          0.0,
         10.0,        -22.0,        -12.0,         -5.0,
         -3.0,          0.0,          0.0,          2.0,
        -24.0,         12.0,          0.0,          0.0,
          5.0,         -6.0,          0.0,          0.0,
          3.0,          0.0,          0.0,         -2.0,
          4.0,          3.0,          1.0,         -2.0,
          0.0,         29.0,         15.0,          0.0,
         -5.0,         -4.0,         -2.0,          2.0,
          8.0,         -3.0,         -1.0,         -5.0,
          0.0,         -3.0,          0.0,          0.0,
         10.0,          0.0,          0.0,          0.0,
          3.0,          0.0,          0.0,         -2.0,
         -5.0,          0.0,          0.0,          3.0,
         46.0,         66.0,         35.0,        -25.0,
        -14.0,          7.0,          0.0,          0.0,
          0.0,          3.0,          2.0,          0.0,
         -5.0,          0.0,          0.0,          0.0,
        -68.0,        -34.0,        -18.0,         36.0,
          0.0,         14.0,          7.0,          0.0,
         10.0,         -6.0,         -3.0,         -5.0,
         -5.0,         -4.0,         -2.0,          3.0,
         -3.0,          5.0,          2.0,          1.0,
         76.0,         17.0,          9.0,        -41.0,
         84.0,        298.0,        159.0,        -45.0,
          3.0,          0.0,          0.0,         -1.0,
         -3.0,          0.0,          0.0,          2.0,
         -3.0,          0.0,          0.0,          1.0,
        -82.0,        292.0,        156.0,         44.0,
        -73.0,         17.0,          9.0,         39.0,
         -9.0,        -16.0,          0.0,          0.0,
          3.0,          0.0,         -1.0,         -2.0,
         -3.0,          0.0,          0.0,          0.0,
         -9.0,         -5.0,         -3.0,          5.0,
       -439.0,          0.0,          0.0,          0.0,
         57.0,        -28.0,        -15.0,        -30.0,
          0.0,         -6.0,         -3.0,          0.0,
         -4.0,          0.0,          0.0,          2.0,
        -40.0,         57.0,         30.0,         21.0,
         23.0,          7.0,          3.0,        -13.0,
        273.0,         80.0,         43.0,       -146.0,
       -449.0,        430.0,          0.0,          0.0,
         -8.0,        -47.0,        -25.0,          4.0,
          6.0,         47.0,         25.0,         -3.0,
          0.0,         23.0,         13.0,          0.0,
         -3.0,          0.0,          0.0,          2.0,
          3.0,         -4.0,         -2.0,         -2.0,
        -48.0,       -110.0,        -59.0,         26.0,
         51.0,        114.0,         61.0,        -27.0,
       -133.0,          0.0,          0.0,         57.0,
          0.0,          4.0,          0.0,          0.0,
        -21.0,         -6.0,         -3.0,         11.0,
          0.0,         -3.0,         -1.0,          0.0,
        -11.0,        -21.0,        -11.0,          6.0,
        -18.0,       -436.0,       -233.0,          9.0,
         35.0,         -7.0,          0.0,          0.0,
          0.0,          5.0,          3.0,          0.0,
         11.0,         -3.0,         -1.0,         -6.0,
         -5.0,         -3.0,         -1.0,          3.0,
        -53.0,         -9.0,         -5.0,         28.0,
          0.0,          3.0,          2.0,          1.0,
          4.0,          0.0,          0.0,         -2.0,
          0.0,         -4.0,          0.0,          0.0,
        -50.0,        194.0,        103.0,         27.0,
        -13.0,         52.0,         28.0,          7.0,
        -91.0,        248.0,          0.0,          0.0,
          6.0,         49.0,         26.0,         -3.0,
         -6.0,        -47.0,        -25.0,          3.0,
          0.0,          5.0,          3.0,          0.0,
         52.0,         23.0,         10.0,        -23.0,
         -3.0,          0.0,          0.0,          1.0,
          0.0,          5.0,          3.0,          0.0,
         -4.0,          0.0,          0.0,          0.0,
         -4.0,          8.0,          3.0,          2.0,
         10.0,          0.0,          0.0,          0.0,
          3.0,          0.0,          0.0,         -2.0,
          0.0,          8.0,          4.0,          0.0,
          0.0,          8.0,          4.0,          1.0,
         -4.0,          0.0,          0.0,          0.0,
         -4.0,          0.0,          0.0,          0.0,
         -8.0,          4.0,          2.0,          4.0,
          8.0,         -4.0,         -2.0,         -4.0,
          0.0,         15.0,          7.0,          0.0,
       -138.0,          0.0,          0.0,          0.0,
          0.0,         -7.0,         -3.0,          0.0,
          0.0,         -7.0,         -3.0,          0.0,
         54.0,          0.0,          0.0,        -29.0,
          0.0,         10.0,          4.0,          0.0,
         -7.0,          0.0,          0.0,          3.0,
        -37.0,         35.0,         19.0,         20.0,
          0.0,          4.0,          0.0,          0.0,
         -4.0,          9.0,          0.0,          0.0,
          8.0,          0.0,          0.0,         -4.0,
         -9.0,        -14.0,         -8.0,          5.0,
         -3.0,         -9.0,         -5.0,          3.0,
       -145.0,         47.0,          0.0,          0.0,
        -10.0,         40.0,         21.0,          5.0,
         11.0,        -49.0,        -26.0,         -7.0,
      -2150.0,          0.0,          0.0,        932.0,
        -12.0,          0.0,          0.0,          5.0,
         85.0,          0.0,          0.0,        -37.0,
          4.0,          0.0,          0.0,         -2.0,
          3.0,          0.0,          0.0,         -2.0,
        -86.0,        153.0,          0.0,          0.0,
         -6.0,          9.0,          5.0,          3.0,
          9.0,        -13.0,         -7.0,         -5.0,
         -8.0,         12.0,          6.0,          4.0,
        -51.0,          0.0,          0.0,         22.0,
        -11.0,       -268.0,       -116.0,          5.0,
          0.0,         12.0,          5.0,          0.0,
          0.0,          7.0,          3.0,          0.0,
         31.0,          6.0,          3.0,        -17.0,
        140.0,         27.0,         14.0,        -75.0,
         57.0,         11.0,          6.0,        -30.0,
        -14.0,        -39.0,          0.0,          0.0,
          0.0,         -6.0,         -2.0,          0.0,
          4.0,         15.0,          8.0,         -2.0,
          0.0,          4.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          1.0,
          0.0,         11.0,          5.0,          0.0,
          9.0,          6.0,          0.0,          0.0,
         -4.0,         10.0,          4.0,          2.0,
          5.0,          3.0,          0.0,          0.0,
         16.0,          0.0,          0.0,         -9.0,
         -3.0,          0.0,          0.0,          0.0,
          0.0,          3.0,          2.0,         -1.0,
          7.0,          0.0,          0.0,         -3.0,
        -25.0,         22.0,          0.0,          0.0,
         42.0,        223.0,        119.0,        -22.0,
        -27.0,       -143.0,        -77.0,         14.0,
          9.0,         49.0,         26.0,         -5.0,
      -1166.0,          0.0,          0.0,        505.0,
         -5.0,          0.0,          0.0,          2.0,
         -6.0,          0.0,          0.0,          3.0,
         -8.0,          0.0,          1.0,          4.0,
          0.0,         -4.0,          0.0,          0.0,
        117.0,          0.0,          0.0,        -63.0,
         -4.0,          8.0,          4.0,          2.0,
          3.0,          0.0,          0.0,         -2.0,
         -5.0,          0.0,          0.0,          2.0,
          0.0,         31.0,          0.0,          0.0,
         -5.0,          0.0,          1.0,          3.0,
          4.0,          0.0,          0.0,         -2.0,
         -4.0,          0.0,          0.0,          2.0,
        -24.0,        -13.0,         -6.0,         10.0,
          3.0,          0.0,          0.0,          0.0,
          0.0,        -32.0,        -17.0,          0.0,
          8.0,         12.0,          5.0,         -3.0,
          3.0,          0.0,          0.0,         -1.0,
          7.0,         13.0,          0.0,          0.0,
         -3.0,         16.0,          0.0,          0.0,
         50.0,          0.0,          0.0,        -27.0,
          0.0,         -5.0,         -3.0,          0.0,
         13.0,          0.0,          0.0,          0.0,
          0.0,          5.0,          3.0,          1.0,
         24.0,          5.0,          2.0,        -11.0,
          5.0,        -11.0,         -5.0,         -2.0,
         30.0,         -3.0,         -2.0,        -16.0,
         18.0,          0.0,          0.0,         -9.0,
          8.0,        614.0,          0.0,          0.0,
          3.0,         -3.0,         -1.0,         -2.0,
          6.0,         17.0,          9.0,         -3.0,
         -3.0,         -9.0,         -5.0,          2.0,
          0.0,          6.0,          3.0,         -1.0,
       -127.0,         21.0,          9.0,         55.0,
          3.0,          5.0,          0.0,          0.0,
         -6.0,        -10.0,         -4.0,          3.0,
          5.0,          0.0,          0.0,          0.0,
         16.0,          9.0,          4.0,         -7.0,
          3.0,          0.0,          0.0,         -2.0,
          0.0,         22.0,          0.0,          0.0,
          0.0,         19.0,         10.0,          0.0,
          7.0,          0.0,          0.0,         -4.0,
          0.0,         -5.0,         -2.0,          0.0,
          0.0,          3.0,          1.0,          0.0,
         -9.0,          3.0,          1.0,          4.0,
         17.0,          0.0,          0.0,         -7.0,
          0.0,         -3.0,         -2.0,         -1.0,
        -20.0,         34.0,          0.0,          0.0,
        -10.0,          0.0,          1.0,          5.0,
         -4.0,          0.0,          0.0,          2.0,
         22.0,        -87.0,          0.0,          0.0,
         -4.0,          0.0,          0.0,          2.0,
         -3.0,         -6.0,         -2.0,          1.0,
        -16.0,         -3.0,         -1.0,          7.0,
          0.0,         -3.0,         -2.0,          0.0,
          4.0,          0.0,          0.0,          0.0,
        -68.0,         39.0,          0.0,          0.0,
         27.0,          0.0,          0.0,        -14.0,
          0.0,         -4.0,          0.0,          0.0,
        -25.0,          0.0,          0.0,          0.0,
        -12.0,         -3.0,         -2.0,          6.0,
          3.0,          0.0,          0.0,         -1.0,
          3.0,         66.0,         29.0,         -1.0,
        490.0,          0.0,          0.0,       -213.0,
        -22.0,         93.0,         49.0,         12.0,
         -7.0,         28.0,         15.0,          4.0,
         -3.0,         13.0,          7.0,          2.0,
        -46.0,         14.0,          0.0,          0.0,
         -5.0,          0.0,          0.0,          0.0,
          2.0,          1.0,          0.0,          0.0,
          0.0,         -3.0,          0.0,          0.0,
        -28.0,          0.0,          0.0,         15.0,
          5.0,          0.0,          0.0,         -2.0,
          0.0,          3.0,          0.0,          0.0,
        -11.0,          0.0,          0.0,          5.0,
          0.0,          3.0,          1.0,          0.0,
         -3.0,          0.0,          0.0,          1.0,
         25.0,        106.0,         57.0,        -13.0,
          5.0,         21.0,         11.0,         -3.0,
       1485.0,          0.0,          0.0,          0.0,
         -7.0,        -32.0,        -17.0,          4.0,
          0.0,          5.0,          3.0,          0.0,
         -6.0,         -3.0,         -2.0,          3.0,
         30.0,         -6.0,         -2.0,        -13.0,
         -4.0,          4.0,          0.0,          0.0,
        -19.0,          0.0,          0.0,         10.0,
          0.0,          4.0,          2.0,         -1.0,
          0.0,          3.0,          0.0,          0.0,
          4.0,          0.0,          0.0,         -2.0,
          0.0,         -3.0,         -1.0,          0.0,
         -3.0,          0.0,          0.0,          0.0,
          5.0,          3.0,          1.0,         -2.0,
          0.0,         11.0,          0.0,          0.0,
        118.0,          0.0,          0.0,        -52.0,
          0.0,         -5.0,         -3.0,          0.0,
        -28.0,         36.0,          0.0,          0.0,
          5.0,         -5.0,          0.0,          0.0,
         14.0,        -59.0,        -31.0,         -8.0,
          0.0,          9.0,          5.0,          1.0,
       -458.0,          0.0,          0.0,        198.0,
          0.0,        -45.0,        -20.0,          0.0,
          9.0,          0.0,          0.0,         -5.0,
          0.0,         -3.0,          0.0,          0.0,
          0.0,         -4.0,         -2.0,         -1.0,
         11.0,          0.0,          0.0,         -6.0,
          6.0,          0.0,          0.0,         -2.0,
        -16.0,         23.0,          0.0,          0.0,
          0.0,         -4.0,         -2.0,          0.0,
         -5.0,          0.0,          0.0,          2.0,
       -166.0,        269.0,          0.0,          0.0,
         15.0,          0.0,          0.0,         -8.0,
         10.0,          0.0,          0.0,         -4.0,
        -78.0,         45.0,          0.0,          0.0,
          0.0,         -5.0,         -2.0,          0.0,
          7.0,          0.0,          0.0,         -4.0,
         -5.0,        328.0,          0.0,          0.0,
          3.0,          0.0,          0.0,         -2.0,
          5.0,          0.0,          0.0,         -2.0,
          0.0,          3.0,          1.0,          0.0,
         -3.0,          0.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          0.0,
          0.0,         -4.0,         -2.0,          0.0,
      -1223.0,        -26.0,          0.0,          0.0,
          0.0,          7.0,          3.0,          0.0,
          3.0,          0.0,          0.0,          0.0,
          0.0,          3.0,          2.0,          0.0,
         -6.0,         20.0,          0.0,          0.0,
       -368.0,          0.0,          0.0,          0.0,
        -75.0,          0.0,          0.0,          0.0,
         11.0,          0.0,          0.0,         -6.0,
          3.0,          0.0,          0.0,         -2.0,
         -3.0,          0.0,          0.0,          1.0,
        -13.0,        -30.0,          0.0,          0.0,
         21.0,          3.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          1.0,
         -4.0,          0.0,          0.0,          2.0,
          8.0,        -27.0,          0.0,          0.0,
        -19.0,        -11.0,          0.0,          0.0,
         -4.0,          0.0,          0.0,          2.0,
          0.0,          5.0,          2.0,          0.0,
         -6.0,          0.0,          0.0,          2.0,
         -8.0,          0.0,          0.0,          0.0,
         -1.0,          0.0,          0.0,          0.0,
        -14.0,          0.0,          0.0,          6.0,
          6.0,          0.0,          0.0,          0.0,
        -74.0,          0.0,          0.0,         32.0,
          0.0,         -3.0,         -1.0,          0.0,
          4.0,          0.0,          0.0,         -2.0,
          8.0,         11.0,          0.0,          0.0,
          0.0,          3.0,          2.0,          0.0,
       -262.0,          0.0,          0.0,        114.0,
          0.0,         -4.0,          0.0,          0.0,
         -7.0,          0.0,          0.0,          4.0,
          0.0,        -27.0,        -12.0,          0.0,
        -19.0,         -8.0,         -4.0,          8.0,
        202.0,          0.0,          0.0,        -87.0,
         -8.0,         35.0,         19.0,          5.0,
          0.0,          4.0,          2.0,          0.0,
         16.0,         -5.0,          0.0,          0.0,
          5.0,          0.0,          0.0,         -3.0,
          0.0,         -3.0,          0.0,          0.0,
          1.0,          0.0,          0.0,          0.0,
        -35.0,        -48.0,        -21.0,         15.0,
         -3.0,         -5.0,         -2.0,          1.0,
          6.0,          0.0,          0.0,         -3.0,
          3.0,          0.0,          0.0,         -1.0,
          0.0,         -5.0,          0.0,          0.0,
         12.0,         55.0,         29.0,         -6.0,
          0.0,          5.0,          3.0,          0.0,
       -598.0,          0.0,          0.0,          0.0,
         -3.0,        -13.0,         -7.0,          1.0,
         -5.0,         -7.0,         -3.0,          2.0,
          3.0,          0.0,          0.0,         -1.0,
          5.0,         -7.0,          0.0,          0.0,
          4.0,          0.0,          0.0,         -2.0,
         16.0,         -6.0,          0.0,          0.0,
          8.0,         -3.0,          0.0,          0.0,
          8.0,        -31.0,        -16.0,         -4.0,
          0.0,          3.0,          1.0,          0.0,
        113.0,          0.0,          0.0,        -49.0,
          0.0,        -24.0,        -10.0,          0.0,
          4.0,          0.0,          0.0,         -2.0,
         27.0,          0.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          1.0,
          0.0,         -4.0,         -2.0,          0.0,
          5.0,          0.0,          0.0,         -2.0,
          0.0,         -3.0,          0.0,          0.0,
        -13.0,          0.0,          0.0,          6.0,
          5.0,          0.0,          0.0,         -2.0,
        -18.0,        -10.0,         -4.0,          8.0,
         -4.0,        -28.0,          0.0,          0.0,
         -5.0,          6.0,          3.0,          2.0,
         -3.0,          0.0,          0.0,          1.0,
         -5.0,         -9.0,         -4.0,          2.0,
         17.0,          0.0,          0.0,         -7.0,
         11.0,          4.0,          0.0,          0.0,
          0.0,         -6.0,         -2.0,          0.0,
         83.0,         15.0,          0.0,          0.0,
         -4.0,          0.0,          0.0,          2.0,
          0.0,       -114.0,        -49.0,          0.0,
        117.0,          0.0,          0.0,        -51.0,
         -5.0,         19.0,         10.0,          2.0,
         -3.0,          0.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          2.0,
          0.0,         -3.0,         -1.0,          0.0,
          3.0,          0.0,          0.0,          0.0,
          0.0,         -6.0,         -2.0,          0.0,
        393.0,          3.0,          0.0,          0.0,
         -4.0,         21.0,         11.0,          2.0,
         -6.0,          0.0,         -1.0,          3.0,
         -3.0,          8.0,          4.0,          1.0,
          8.0,          0.0,          0.0,          0.0,
         18.0,        -29.0,        -13.0,         -8.0,
          8.0,         34.0,         18.0,         -4.0,
         89.0,          0.0,          0.0,          0.0,
          3.0,         12.0,          6.0,         -1.0,
         54.0,        -15.0,         -7.0,        -24.0,
          0.0,          3.0,          0.0,          0.0,
          3.0,          0.0,          0.0,         -1.0,
          0.0,         35.0,          0.0,          0.0,
       -154.0,        -30.0,        -13.0,         67.0,
         15.0,          0.0,          0.0,          0.0,
          0.0,          4.0,          2.0,          0.0,
          0.0,          9.0,          0.0,          0.0,
         80.0,        -71.0,        -31.0,        -35.0,
          0.0,        -20.0,         -9.0,          0.0,
         11.0,          5.0,          2.0,         -5.0,
         61.0,        -96.0,        -42.0,        -27.0,
         14.0,          9.0,          4.0,         -6.0,
        -11.0,         -6.0,         -3.0,          5.0,
          0.0,         -3.0,         -1.0,          0.0,
        123.0,       -415.0,       -180.0,        -53.0,
          0.0,          0.0,          0.0,        -35.0,
         -5.0,          0.0,          0.0,          0.0,
          7.0,        -32.0,        -17.0,         -4.0,
          0.0,         -9.0,         -5.0,          0.0,
          0.0,         -4.0,          2.0,          0.0,
        -89.0,          0.0,          0.0,         38.0,
          0.0,        -86.0,        -19.0,         -6.0,
          0.0,          0.0,        -19.0,          6.0,
       -123.0,       -416.0,       -180.0,         53.0,
          0.0,         -3.0,         -1.0,          0.0,
         12.0,         -6.0,         -3.0,         -5.0,
        -13.0,          9.0,          4.0,          6.0,
          0.0,        -15.0,         -7.0,          0.0,
          3.0,          0.0,          0.0,         -1.0,
        -62.0,        -97.0,        -42.0,         27.0,
        -11.0,          5.0,          2.0,          5.0,
          0.0,        -19.0,         -8.0,          0.0,
         -3.0,          0.0,          0.0,          1.0,
          0.0,          4.0,          2.0,          0.0,
          0.0,          3.0,          0.0,          0.0,
          0.0,          4.0,          2.0,          0.0,
        -85.0,        -70.0,        -31.0,         37.0,
        163.0,        -12.0,         -5.0,        -72.0,
        -63.0,        -16.0,         -7.0,         28.0,
        -21.0,        -32.0,        -14.0,          9.0,
          0.0,         -3.0,         -1.0,          0.0,
          3.0,          0.0,          0.0,         -2.0,
          0.0,          8.0,          0.0,          0.0,
          3.0,         10.0,          4.0,         -1.0,
          3.0,          0.0,          0.0,         -1.0,
          0.0,         -7.0,         -3.0,          0.0,
          0.0,         -4.0,         -2.0,          0.0,
          6.0,         19.0,          0.0,          0.0,
          5.0,       -173.0,        -75.0,         -2.0,
          0.0,         -7.0,         -3.0,          0.0,
          7.0,        -12.0,         -5.0,         -3.0,
         -3.0,          0.0,          0.0,          2.0,
          3.0,         -4.0,         -2.0,         -1.0,
         74.0,          0.0,          0.0,        -32.0,
         -3.0,         12.0,          6.0,          2.0,
         26.0,        -14.0,         -6.0,        -11.0,
         19.0,          0.0,          0.0,         -8.0,
          6.0,         24.0,         13.0,         -3.0,
         83.0,          0.0,          0.0,          0.0,
          0.0,        -10.0,         -5.0,          0.0,
         11.0,         -3.0,         -1.0,         -5.0,
          3.0,          0.0,          1.0,         -1.0,
          3.0,          0.0,          0.0,         -1.0,
         -4.0,          0.0,          0.0,          0.0,
          5.0,        -23.0,        -12.0,         -3.0,
       -339.0,          0.0,          0.0,        147.0,
          0.0,        -10.0,         -5.0,          0.0,
          5.0,          0.0,          0.0,          0.0,
          3.0,          0.0,          0.0,         -1.0,
          0.0,         -4.0,         -2.0,          0.0,
         18.0,         -3.0,          0.0,          0.0,
          9.0,        -11.0,         -5.0,         -4.0,
         -8.0,          0.0,          0.0,          4.0,
          3.0,          0.0,          0.0,         -1.0,
          0.0,          9.0,          0.0,          0.0,
          6.0,         -9.0,         -4.0,         -2.0,
         -4.0,        -12.0,          0.0,          0.0,
         67.0,        -91.0,        -39.0,        -29.0,
         30.0,        -18.0,         -8.0,        -13.0,
          0.0,          0.0,          0.0,          0.0,
          0.0,       -114.0,        -50.0,          0.0,
          0.0,          0.0,          0.0,         23.0,
        517.0,         16.0,          7.0,       -224.0,
          0.0,         -7.0,         -3.0,          0.0,
        143.0,         -3.0,         -1.0,        -62.0,
         29.0,          0.0,          0.0,        -13.0,
         -4.0,          0.0,          0.0,          2.0,
         -6.0,          0.0,          0.0,          3.0,
          5.0,         12.0,          5.0,         -2.0,
        -25.0,          0.0,          0.0,         11.0,
         -3.0,          0.0,          0.0,          1.0,
          0.0,          4.0,          2.0,          0.0,
        -22.0,         12.0,          5.0,         10.0,
         50.0,          0.0,          0.0,        -22.0,
          0.0,          7.0,          4.0,          0.0,
          0.0,          3.0,          1.0,          0.0,
         -4.0,          4.0,          2.0,          2.0,
         -5.0,        -11.0,         -5.0,          2.0,
          0.0,          4.0,          2.0,          0.0,
          4.0,         17.0,          9.0,         -2.0,
         59.0,          0.0,          0.0,          0.0,
          0.0,         -4.0,         -2.0,          0.0,
         -8.0,          0.0,          0.0,          4.0,
         -3.0,          0.0,          0.0,          0.0,
          4.0,        -15.0,         -8.0,         -2.0,
        370.0,         -8.0,          0.0,       -160.0,
          0.0,          0.0,         -3.0,          0.0,
          0.0,          3.0,          1.0,          0.0,
         -6.0,          3.0,          1.0,          3.0,
          0.0,          6.0,          0.0,          0.0,
        -10.0,          0.0,          0.0,          4.0,
          0.0,          9.0,          4.0,          0.0,
          4.0,         17.0,          7.0,         -2.0,
         34.0,          0.0,          0.0,        -15.0,
          0.0,          5.0,          3.0,          0.0,
         -5.0,          0.0,          0.0,          2.0,
        -37.0,         -7.0,         -3.0,         16.0,
          3.0,         13.0,          7.0,         -2.0,
         40.0,          0.0,          0.0,          0.0,
          0.0,         -3.0,         -2.0,          0.0,
       -184.0,         -3.0,         -1.0,         80.0,
         -3.0,          0.0,          0.0,          1.0,
         -3.0,          0.0,          0.0,          0.0,
          0.0,        -10.0,         -6.0,         -1.0,
         31.0,         -6.0,          0.0,        -13.0,
         -3.0,        -32.0,        -14.0,          1.0,
         -7.0,          0.0,          0.0,          3.0,
          0.0,         -8.0,         -4.0,          0.0,
          3.0,         -4.0,          0.0,          0.0,
          0.0,          4.0,          0.0,          0.0,
          0.0,          3.0,          1.0,          0.0,
         19.0,        -23.0,        -10.0,          2.0,
          0.0,          0.0,          0.0,        -10.0,
          0.0,          3.0,          2.0,          0.0,
          0.0,          9.0,          5.0,         -1.0,
         28.0,          0.0,          0.0,          0.0,
          0.0,         -7.0,         -4.0,          0.0,
          8.0,         -4.0,          0.0,         -4.0,
          0.0,          0.0,         -2.0,          0.0,
          0.0,          3.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          1.0,
         -9.0,          0.0,          1.0,          4.0,
          3.0,         12.0,          5.0,         -1.0,
         17.0,         -3.0,         -1.0,          0.0,
          0.0,          7.0,          4.0,          0.0,
         19.0,          0.0,          0.0,          0.0,
          0.0,         -5.0,         -3.0,          0.0,
         14.0,         -3.0,          0.0,         -1.0,
          0.0,          0.0,         -1.0,          0.0,
          0.0,          0.0,          0.0,         -5.0,
          0.0,          5.0,          3.0,          0.0,
         13.0,          0.0,          0.0,          0.0,
          0.0,         -3.0,         -2.0,          0.0,
          2.0,          9.0,          4.0,          3.0,
          0.0,          0.0,          0.0,         -4.0,
          8.0,          0.0,          0.0,          0.0,
          0.0,          4.0,          2.0,          0.0,
          6.0,          0.0,          0.0,         -3.0,
          6.0,          0.0,          0.0,          0.0,
          0.0,          3.0,          1.0,          0.0,
          5.0,          0.0,          0.0,         -2.0,
          3.0,          0.0,          0.0,         -1.0,
         -3.0,          0.0,          0.0,          0.0,
          6.0,          0.0,          0.0,          0.0,
          7.0,          0.0,          0.0,          0.0,
         -4.0,          0.0,          0.0,          0.0,
          4.0,          0.0,          0.0,          0.0,
          6.0,          0.0,          0.0,          0.0,
          0.0,         -4.0,          0.0,          0.0,
          0.0,         -4.0,          0.0,          0.0,
          5.0,          0.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          0.0,
          4.0,          0.0,          0.0,          0.0,
         -5.0,          0.0,          0.0,          0.0,
          4.0,          0.0,          0.0,          0.0,
          0.0,          3.0,          0.0,          0.0,
         13.0,          0.0,          0.0,          0.0,
         21.0,         11.0,          0.0,          0.0,
          0.0,         -5.0,          0.0,          0.0,
          0.0,         -5.0,         -2.0,          0.0,
          0.0,          5.0,          3.0,          0.0,
          0.0,         -5.0,          0.0,          0.0,
         -3.0,          0.0,          0.0,          2.0,
         20.0,         10.0,          0.0,          0.0,
        -34.0,          0.0,          0.0,          0.0,
        -19.0,          0.0,          0.0,          0.0,
          3.0,          0.0,          0.0,         -2.0,
         -3.0,          0.0,          0.0,          1.0,
         -6.0,          0.0,          0.0,          3.0,
         -4.0,          0.0,          0.0,          0.0,
          3.0,          0.0,          0.0,          0.0,
          3.0,          0.0,          0.0,          0.0,
          4.0,          0.0,          0.0,          0.0,
          3.0,          0.0,          0.0,         -1.0,
          6.0,          0.0,          0.0,         -3.0,
         -8.0,          0.0,          0.0,          3.0,
          0.0,          3.0,          1.0,          0.0,
         -3.0,          0.0,          0.0,          0.0,
          0.0,         -3.0,         -2.0,          0.0,
        126.0,        -63.0,        -27.0,        -55.0,
         -5.0,          0.0,          1.0,          2.0,
         -3.0,         28.0,         15.0,          2.0,
          5.0,          0.0,          1.0,         -2.0,
          0.0,          9.0,          4.0,          1.0,
          0.0,          9.0,          4.0,         -1.0,
       -126.0,        -63.0,        -27.0,         55.0,
          3.0,          0.0,          0.0,         -1.0,
         21.0,        -11.0,         -6.0,        -11.0,
          0.0,         -4.0,          0.0,          0.0,
        -21.0,        -11.0,         -6.0,         11.0,
         -3.0,          0.0,          0.0,          1.0,
          0.0,          3.0,          1.0,          0.0,
          8.0,          0.0,          0.0,         -4.0,
         -6.0,          0.0,          0.0,          3.0,
         -3.0,          0.0,          0.0,          1.0,
          3.0,          0.0,          0.0,         -1.0,
         -3.0,          0.0,          0.0,          1.0,
         -5.0,          0.0,          0.0,          2.0,
         24.0,        -12.0,         -5.0,        -11.0,
          0.0,          3.0,          1.0,          0.0,
          0.0,          3.0,          1.0,          0.0,
          0.0,          3.0,          2.0,          0.0,
        -24.0,        -12.0,         -5.0,         10.0,
          4.0,          0.0,         -1.0,         -2.0,
         13.0,          0.0,          0.0,         -6.0,
          7.0,          0.0,          0.0,         -3.0,
          3.0,          0.0,          0.0,         -1.0,
          3.0,          0.0,          0.0,         -1.0};

/*
   Interval between fundamental epoch J2000.0 and given date.
*/

   t = ((jd_high - T0) + jd_low) / 36525.0;

/*
   Compute fundamental arguments from Simon et al. (1994), in radians.
*/

   fund_args (t, a);

/*
   ** Luni-solar nutation. **
*/

/*
   Initialize the nutation values.
*/

   dp = 0.0;
   de = 0.0;

/*
   Summation of luni-solar nutation series (in reverse order).
*/

   for (i = 677; i >= 0; i--)
   {

/*
   Argument and functions.
*/

      arg = fmod ((double) nals_t[i][0] * a[0]  +
                  (double) nals_t[i][1] * a[1]  +
                  (double) nals_t[i][2] * a[2]  +
                  (double) nals_t[i][3] * a[3]  +
                  (double) nals_t[i][4] * a[4], TWOPI);

      sarg = sin (arg);
      carg = cos (arg);

/*
   Term.
*/

      dp += (cls_t[i][0] + cls_t[i][1] * t) * sarg
              +   cls_t[i][2] * carg;
      de += (cls_t[i][3] + cls_t[i][4] * t) * carg
              +   cls_t[i][5] * sarg;
   }

/*
   Convert from 0.1 microarcsec units to radians.
*/

   factor = 1.0e-7 * ASEC2RAD;
   dpsils = dp * factor;
   depsls = de * factor;

/*
   ** Planetary nutation. **
*/

/*
   Mean anomaly of the Moon.
*/

   al = fmod (2.35555598 + 8328.6914269554 * t, TWOPI);

/*
   Mean anomaly of the Sun.
*/

   alsu = fmod (6.24006013 + 628.301955 * t, TWOPI);

/*
   Mean argument of the latitude of the Moon.
*/

   af = fmod (1.627905234 + 8433.466158131 * t, TWOPI);

/*
   Mean elongation of the Moon from the Sun.
*/

   ad = fmod (5.198466741 + 7771.3771468121 * t, TWOPI);

/*
   Mean longitude of the ascending node of the Moon.
*/

   aom = fmod (2.18243920 - 33.757045 * t, TWOPI);

/*
   General accumulated precession in longitude.
*/

   apa = (0.02438175 + 0.00000538691 * t) * t;
/*
   Planetary longitudes, Mercury through Neptune (Souchay et al. 1999).
*/

   alme = fmod (4.402608842 + 2608.7903141574 * t, TWOPI);
   alve = fmod (3.176146697 + 1021.3285546211 * t, TWOPI);
   alea = fmod (1.753470314 +  628.3075849991 * t, TWOPI);
   alma = fmod (6.203480913 +  334.0612426700 * t, TWOPI);
   alju = fmod (0.599546497 +   52.9690962641 * t, TWOPI);
   alsa = fmod (0.874016757 +   21.3299104960 * t, TWOPI);
   alur = fmod (5.481293871 +    7.4781598567 * t, TWOPI);
   alne = fmod (5.321159000 +    3.8127774000 * t, TWOPI);

/*
   Initialize the nutation values.
*/

   dp = 0.0;
   de = 0.0;

/*
   Summation of planetary nutation series (in reverse order).
*/

   for (i = 686; i >= 0; i--)
   {

/*
   Argument and functions.
*/

      arg = fmod ((double) napl_t[i][ 0] * al    +
                  (double) napl_t[i][ 1] * alsu  +
                  (double) napl_t[i][ 2] * af    +
                  (double) napl_t[i][ 3] * ad    +
                  (double) napl_t[i][ 4] * aom   +
                  (double) napl_t[i][ 5] * alme  +
                  (double) napl_t[i][ 6] * alve  +
                  (double) napl_t[i][ 7] * alea  +
                  (double) napl_t[i][ 8] * alma  +
                  (double) napl_t[i][ 9] * alju  +
                  (double) napl_t[i][10] * alsa  +
                  (double) napl_t[i][11] * alur  +
                  (double) napl_t[i][12] * alne  +
                  (double) napl_t[i][13] * apa, TWOPI);

      sarg = sin (arg);
      carg = cos (arg);

/*
   Term.
*/

      dp += cpl_t[i][0] * sarg + cpl_t[i][1] * carg;
      de += cpl_t[i][2] * sarg + cpl_t[i][3] * carg;
   }

   dpsipl = dp * factor;
   depspl = de * factor;

/*
   Total: Add planetary and luni-solar components.
*/

   *dpsi = dpsipl + dpsils;
   *deps = depspl + depsls;

   return;
}

/********iau2000b */

void iau2000b (double jd_high, double jd_low,

               double *dpsi, double *deps)
/*
------------------------------------------------------------------------

   PURPOSE:
      To compute the forced nutation of the non-rigid Earth based on
      the IAU 2000B precession/nutation model.

   REFERENCES:
      McCarthy, D. and Luzum, B. (2003). "An Abridged Model of the
         Precession & Nutation of the Celestial Pole," Celestial
         Mechanics and Dynamical Astronomy, Volume 85, Issue 1,
         Jan. 2003, p. 37. (IAU 2000B)
      IERS Conventions (2003), Chapter 5.

   INPUT
   ARGUMENTS:
      jd_high (double)
         High-order part of TT Julian date.
      jd_low (double)
         Low-order part of TT Julian date.

   OUTPUT
   ARGUMENTS:
      *dpsi (double)
         Nutation (luni-solar + planetary) in longitude, in radians.
      *deps (double)
         Nutation (luni-solar + planetary) in obliquity, in radians.

   RETURNED
   VALUE:
      None.

   GLOBALS
   USED:
      T0, ASEC2RAD, TWOPI

   FUNCTIONS
   CALLED:
      fmod      math.h
      sin       math.h
      cos       math.h

   VER./DATE/
   PROGRAMMER:
      V1.0/09-03/JAB (USNO/AA)

   NOTES:
      1. IAU 2000B reproduces the IAU 2000A model to a precision of
      1 milliarcsecond in the interval 1995-2020.

------------------------------------------------------------------------
*/
{
   short int i;

/*
   Planetary nutation (arcsec).  These fixed terms account for the
   omission of the long-period planetary terms in the truncated model.
*/

   double dpplan = -0.000135;
   double deplan =  0.000388;

   double t, el, elp, f, d, om, arg, dp, de, sarg, carg, factor, dpsils,
      depsls, dpsipl, depspl;

/*
   Luni-Solar argument multipliers:
       L     L'    F     D     Om
*/

   const short int nals_t[77][5] =
      {0,    0,    0,    0,    1,
       0,    0,    2,   -2,    2,
       0,    0,    2,    0,    2,
       0,    0,    0,    0,    2,
       0,    1,    0,    0,    0,
       0,    1,    2,   -2,    2,
       1,    0,    0,    0,    0,
       0,    0,    2,    0,    1,
       1,    0,    2,    0,    2,
       0,   -1,    2,   -2,    2,
       0,    0,    2,   -2,    1,
      -1,    0,    2,    0,    2,
      -1,    0,    0,    2,    0,
       1,    0,    0,    0,    1,
      -1,    0,    0,    0,    1,
      -1,    0,    2,    2,    2,
       1,    0,    2,    0,    1,
      -2,    0,    2,    0,    1,
       0,    0,    0,    2,    0,
       0,    0,    2,    2,    2,
       0,   -2,    2,   -2,    2,
      -2,    0,    0,    2,    0,
       2,    0,    2,    0,    2,
       1,    0,    2,   -2,    2,
      -1,    0,    2,    0,    1,
       2,    0,    0,    0,    0,
       0,    0,    2,    0,    0,
       0,    1,    0,    0,    1,
      -1,    0,    0,    2,    1,
       0,    2,    2,   -2,    2,
       0,    0,   -2,    2,    0,
       1,    0,    0,   -2,    1,
       0,   -1,    0,    0,    1,
      -1,    0,    2,    2,    1,
       0,    2,    0,    0,    0,
       1,    0,    2,    2,    2,
      -2,    0,    2,    0,    0,
       0,    1,    2,    0,    2,
       0,    0,    2,    2,    1,
       0,   -1,    2,    0,    2,
       0,    0,    0,    2,    1,
       1,    0,    2,   -2,    1,
       2,    0,    2,   -2,    2,
      -2,    0,    0,    2,    1,
       2,    0,    2,    0,    1,
       0,   -1,    2,   -2,    1,
       0,    0,    0,   -2,    1,
      -1,   -1,    0,    2,    0,
       2,    0,    0,   -2,    1,
       1,    0,    0,    2,    0,
       0,    1,    2,   -2,    1,
       1,   -1,    0,    0,    0,
      -2,    0,    2,    0,    2,
       3,    0,    2,    0,    2,
       0,   -1,    0,    2,    0,
       1,   -1,    2,    0,    2,
       0,    0,    0,    1,    0,
      -1,   -1,    2,    2,    2,
      -1,    0,    2,    0,    0,
       0,   -1,    2,    2,    2,
      -2,    0,    0,    0,    1,
       1,    1,    2,    0,    2,
       2,    0,    0,    0,    1,
      -1,    1,    0,    1,    0,
       1,    1,    0,    0,    0,
       1,    0,    2,    0,    0,
      -1,    0,    2,   -2,    1,
       1,    0,    0,    0,    2,
      -1,    0,    0,    1,    0,
       0,    0,    2,    1,    2,
      -1,    0,    2,    4,    2,
      -1,    1,    0,    1,    1,
       0,   -2,    2,   -2,    1,
       1,    0,    2,    2,    1,
      -2,    0,    2,    2,    2,
      -1,    0,    0,    0,    2,
       1,    1,    2,   -2,    2};

/*
   Luni-Solar nutation coefficients, unit 1e-7 arcsec:
   longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)

   Each row of coefficients in 'cls_t' belongs with the corresponding
   row of fundamental-argument multipliers in 'nals_t'.
*/

   const double cls_t[77][6] =
      {-172064161.0, -174666.0,  33386.0, 92052331.0,  9086.0, 15377.0,
        -13170906.0,   -1675.0, -13696.0,  5730336.0, -3015.0, -4587.0,
         -2276413.0,    -234.0,   2796.0,   978459.0,  -485.0,  1374.0,
          2074554.0,     207.0,   -698.0,  -897492.0,   470.0,  -291.0,
          1475877.0,   -3633.0,  11817.0,    73871.0,  -184.0, -1924.0,
          -516821.0,    1226.0,   -524.0,   224386.0,  -677.0,  -174.0,
           711159.0,      73.0,   -872.0,    -6750.0,     0.0,   358.0,
          -387298.0,    -367.0,    380.0,   200728.0,    18.0,   318.0,
          -301461.0,     -36.0,    816.0,   129025.0,   -63.0,   367.0,
           215829.0,    -494.0,    111.0,   -95929.0,   299.0,   132.0,
           128227.0,     137.0,    181.0,   -68982.0,    -9.0,    39.0,
           123457.0,      11.0,     19.0,   -53311.0,    32.0,    -4.0,
           156994.0,      10.0,   -168.0,    -1235.0,     0.0,    82.0,
            63110.0,      63.0,     27.0,   -33228.0,     0.0,    -9.0,
           -57976.0,     -63.0,   -189.0,    31429.0,     0.0,   -75.0,
           -59641.0,     -11.0,    149.0,    25543.0,   -11.0,    66.0,
           -51613.0,     -42.0,    129.0,    26366.0,     0.0,    78.0,
            45893.0,      50.0,     31.0,   -24236.0,   -10.0,    20.0,
            63384.0,      11.0,   -150.0,    -1220.0,     0.0,    29.0,
           -38571.0,      -1.0,    158.0,    16452.0,   -11.0,    68.0,
            32481.0,       0.0,      0.0,   -13870.0,     0.0,     0.0,
           -47722.0,       0.0,    -18.0,      477.0,     0.0,   -25.0,
           -31046.0,      -1.0,    131.0,    13238.0,   -11.0,    59.0,
            28593.0,       0.0,     -1.0,   -12338.0,    10.0,    -3.0,
            20441.0,      21.0,     10.0,   -10758.0,     0.0,    -3.0,
            29243.0,       0.0,    -74.0,     -609.0,     0.0,    13.0,
            25887.0,       0.0,    -66.0,     -550.0,     0.0,    11.0,
           -14053.0,     -25.0,     79.0,     8551.0,    -2.0,   -45.0,
            15164.0,      10.0,     11.0,    -8001.0,     0.0,    -1.0,
           -15794.0,      72.0,    -16.0,     6850.0,   -42.0,    -5.0,
            21783.0,       0.0,     13.0,     -167.0,     0.0,    13.0,
           -12873.0,     -10.0,    -37.0,     6953.0,     0.0,   -14.0,
           -12654.0,      11.0,     63.0,     6415.0,     0.0,    26.0,
           -10204.0,       0.0,     25.0,     5222.0,     0.0,    15.0,
            16707.0,     -85.0,    -10.0,      168.0,    -1.0,    10.0,
            -7691.0,       0.0,     44.0,     3268.0,     0.0,    19.0,
           -11024.0,       0.0,    -14.0,      104.0,     0.0,     2.0,
             7566.0,     -21.0,    -11.0,    -3250.0,     0.0,    -5.0,
            -6637.0,     -11.0,     25.0,     3353.0,     0.0,    14.0,
            -7141.0,      21.0,      8.0,     3070.0,     0.0,     4.0,
            -6302.0,     -11.0,      2.0,     3272.0,     0.0,     4.0,
             5800.0,      10.0,      2.0,    -3045.0,     0.0,    -1.0,
             6443.0,       0.0,     -7.0,    -2768.0,     0.0,    -4.0,
            -5774.0,     -11.0,    -15.0,     3041.0,     0.0,    -5.0,
            -5350.0,       0.0,     21.0,     2695.0,     0.0,    12.0,
            -4752.0,     -11.0,     -3.0,     2719.0,     0.0,    -3.0,
            -4940.0,     -11.0,    -21.0,     2720.0,     0.0,    -9.0,
             7350.0,       0.0,     -8.0,      -51.0,     0.0,     4.0,
             4065.0,       0.0,      6.0,    -2206.0,     0.0,     1.0,
             6579.0,       0.0,    -24.0,     -199.0,     0.0,     2.0,
             3579.0,       0.0,      5.0,    -1900.0,     0.0,     1.0,
             4725.0,       0.0,     -6.0,      -41.0,     0.0,     3.0,
            -3075.0,       0.0,     -2.0,     1313.0,     0.0,    -1.0,
            -2904.0,       0.0,     15.0,     1233.0,     0.0,     7.0,
             4348.0,       0.0,    -10.0,      -81.0,     0.0,     2.0,
            -2878.0,       0.0,      8.0,     1232.0,     0.0,     4.0,
            -4230.0,       0.0,      5.0,      -20.0,     0.0,    -2.0,
            -2819.0,       0.0,      7.0,     1207.0,     0.0,     3.0,
            -4056.0,       0.0,      5.0,       40.0,     0.0,    -2.0,
            -2647.0,       0.0,     11.0,     1129.0,     0.0,     5.0,
            -2294.0,       0.0,    -10.0,     1266.0,     0.0,    -4.0,
             2481.0,       0.0,     -7.0,    -1062.0,     0.0,    -3.0,
             2179.0,       0.0,     -2.0,    -1129.0,     0.0,    -2.0,
             3276.0,       0.0,      1.0,       -9.0,     0.0,     0.0,
            -3389.0,       0.0,      5.0,       35.0,     0.0,    -2.0,
             3339.0,       0.0,    -13.0,     -107.0,     0.0,     1.0,
            -1987.0,       0.0,     -6.0,     1073.0,     0.0,    -2.0,
            -1981.0,       0.0,      0.0,      854.0,     0.0,     0.0,
             4026.0,       0.0,   -353.0,     -553.0,     0.0,  -139.0,
             1660.0,       0.0,     -5.0,     -710.0,     0.0,    -2.0,
            -1521.0,       0.0,      9.0,      647.0,     0.0,     4.0,
             1314.0,       0.0,      0.0,     -700.0,     0.0,     0.0,
            -1283.0,       0.0,      0.0,      672.0,     0.0,     0.0,
            -1331.0,       0.0,      8.0,      663.0,     0.0,     4.0,
             1383.0,       0.0,     -2.0,     -594.0,     0.0,    -2.0,
             1405.0,       0.0,      4.0,     -610.0,     0.0,     2.0,
             1290.0,       0.0,      0.0,     -556.0,     0.0,     0.0};

/*
   Interval between fundamental epoch J2000.0 and given date.
*/

   t = ((jd_high - T0) + jd_low) / 36525.0;

/*
   ** Luni-solar nutation. **

   Fundamental (Delaunay) arguments from Simon et al. (1994),
   in radians.
*/

/*
   Mean anomaly of the Moon.
*/

   el  = fmod (485868.249036 +
          t * 1717915923.2178, ASEC360) * ASEC2RAD;

/*
   Mean anomaly of the Sun.
*/

   elp = fmod (1287104.79305 +
             t * 129596581.0481, ASEC360) * ASEC2RAD;

/*
   Mean argument of the latitude of the Moon.
*/

   f   = fmod (335779.526232 +
             t * 1739527262.8478, ASEC360) * ASEC2RAD;

/*
   Mean elongation of the Moon from the Sun.
*/

   d   = fmod (1072260.70369 +
             t * 1602961601.2090, ASEC360) * ASEC2RAD;

/*
   Mean longitude of the ascending node of the Moon.
*/

   om  = fmod (450160.398036 -
             t * 6962890.5431, ASEC360) * ASEC2RAD;

/*
  Initialize the nutation values.
*/

   dp = 0.0;
   de = 0.0;

/*
  Summation of luni-solar nutation series (in reverse order).
*/

   for (i = 76; i >= 0; i--)
   {

/*
   Argument and functions.
*/

      arg = fmod ((double) nals_t[i][0] * el  +
                  (double) nals_t[i][1] * elp +
                  (double) nals_t[i][2] * f   +
                  (double) nals_t[i][3] * d   +
                  (double) nals_t[i][4] * om,   TWOPI);

      sarg = sin (arg);
      carg = cos (arg);

/*
   Term.
*/

      dp += (cls_t[i][0] + cls_t[i][1] * t) * sarg
              +   cls_t[i][2] * carg;
      de += (cls_t[i][3] + cls_t[i][4] * t) * carg
              +   cls_t[i][5] * sarg;
   }

/*
  Convert from 0.1 microarcsec units to radians.
*/

   factor = 1.0e-7 * ASEC2RAD;
   dpsils = dp * factor;
   depsls = de * factor;

/*
  ** Planetary nutation. **

  Fixed terms to allow for long-period nutation, in radians.
*/

   dpsipl = dpplan * ASEC2RAD;
   depspl = deplan * ASEC2RAD;

/*
   Total: Add planetary and luni-solar components.
*/

   *dpsi = dpsipl + dpsils;
   *deps = depspl + depsls;

   return;
}

/********nu2000k */

void nu2000k (double jd_high, double jd_low,

              double *dpsi, double *deps)
/*
------------------------------------------------------------------------

   PURPOSE:
      To compute the forced nutation of the non-rigid Earth:
      Model NU2000K.  This model is a modified version of IAU 2000A,
      which has been truncated for speed of execution, and uses Simon
      et al. (1994) fundamental arguments throughout.  NU2000K agrees
      with IAU 2000A at the 0.1 milliarcsecond level from 1700 to
      2300.

   REFERENCES:
      IERS Conventions (2003), Chapter 5.
      Simon et al. (1994) Astronomy and Astrophysics 282, 663-683,
         esp. Sections 3.4-3.5.

   INPUT
   ARGUMENTS:
      jd_high (double)
         High-order part of TT Julian date.
      jd_low (double)
         Low-order part of TT Julian date.

   OUTPUT
   ARGUMENTS:
      *dpsi (double)
         Nutation (luni-solar + planetary) in longitude, in radians.
      *deps (double)
         Nutation (luni-solar + planetary) in obliquity, in radians.

   RETURNED
   VALUE:
      None.

   GLOBALS
   USED:
      T0, ASEC2RAD, TWOPI

   FUNCTIONS
   CALLED:
      fund_args    novas.c
      fmod         math.h
      sin          math.h
      cos          math.h

   VER./DATE/
   PROGRAMMER:
      V1.0/03-04/JAB (USNO/AA)

   NOTES:
      1. NU2000K was compared to IAU 2000A over six centuries (1700-
      2300).  The average error in dpsi is 20 microarcseconds, with 98%
      of the errors < 60 microarcseconds;  the average error in deleps
      is 8 microarcseconds, with 100% of the errors < 60
      microarcseconds.
     2. NU2000K was developed by G. Kaplan (USNO) in March 2004.
     3. This function is the "C" version of NOVAS Fortran routine
      'nu2000k'.

------------------------------------------------------------------------
*/
{
   short int i;

   double t, a[5], dp, de, arg, sarg, carg, factor, dpsils,
      depsls, alme, alve, alea, alma, alju, alsa, alur, alne, apa,
      dpsipl, depspl;

/*
   Luni-Solar argument multipliers:
       L     L'    F     D     Om
*/

   const short int nals_t[323][5] =
      {0,    0,    0,    0,    1,
       0,    0,    2,   -2,    2,
       0,    0,    2,    0,    2,
       0,    0,    0,    0,    2,
       0,    1,    0,    0,    0,
       0,    1,    2,   -2,    2,
       1,    0,    0,    0,    0,
       0,    0,    2,    0,    1,
       1,    0,    2,    0,    2,
       0,   -1,    2,   -2,    2,
       0,    0,    2,   -2,    1,
      -1,    0,    2,    0,    2,
      -1,    0,    0,    2,    0,
       1,    0,    0,    0,    1,
      -1,    0,    0,    0,    1,
      -1,    0,    2,    2,    2,
       1,    0,    2,    0,    1,
      -2,    0,    2,    0,    1,
       0,    0,    0,    2,    0,
       0,    0,    2,    2,    2,
       0,   -2,    2,   -2,    2,
      -2,    0,    0,    2,    0,
       2,    0,    2,    0,    2,
       1,    0,    2,   -2,    2,
      -1,    0,    2,    0,    1,
       2,    0,    0,    0,    0,
       0,    0,    2,    0,    0,
       0,    1,    0,    0,    1,
      -1,    0,    0,    2,    1,
       0,    2,    2,   -2,    2,
       0,    0,   -2,    2,    0,
       1,    0,    0,   -2,    1,
       0,   -1,    0,    0,    1,
      -1,    0,    2,    2,    1,
       0,    2,    0,    0,    0,
       1,    0,    2,    2,    2,
      -2,    0,    2,    0,    0,
       0,    1,    2,    0,    2,
       0,    0,    2,    2,    1,
       0,   -1,    2,    0,    2,
       0,    0,    0,    2,    1,
       1,    0,    2,   -2,    1,
       2,    0,    2,   -2,    2,
      -2,    0,    0,    2,    1,
       2,    0,    2,    0,    1,
       0,   -1,    2,   -2,    1,
       0,    0,    0,   -2,    1,
      -1,   -1,    0,    2,    0,
       2,    0,    0,   -2,    1,
       1,    0,    0,    2,    0,
       0,    1,    2,   -2,    1,
       1,   -1,    0,    0,    0,
      -2,    0,    2,    0,    2,
       3,    0,    2,    0,    2,
       0,   -1,    0,    2,    0,
       1,   -1,    2,    0,    2,
       0,    0,    0,    1,    0,
      -1,   -1,    2,    2,    2,
      -1,    0,    2,    0,    0,
       0,   -1,    2,    2,    2,
      -2,    0,    0,    0,    1,
       1,    1,    2,    0,    2,
       2,    0,    0,    0,    1,
      -1,    1,    0,    1,    0,
       1,    1,    0,    0,    0,
       1,    0,    2,    0,    0,
      -1,    0,    2,   -2,    1,
       1,    0,    0,    0,    2,
      -1,    0,    0,    1,    0,
       0,    0,    2,    1,    2,
      -1,    0,    2,    4,    2,
      -1,    1,    0,    1,    1,
       0,   -2,    2,   -2,    1,
       1,    0,    2,    2,    1,
      -2,    0,    2,    2,    2,
      -1,    0,    0,    0,    2,
       1,    1,    2,   -2,    2,
      -2,    0,    2,    4,    2,
      -1,    0,    4,    0,    2,
       2,    0,    2,   -2,    1,
       2,    0,    2,    2,    2,
       1,    0,    0,    2,    1,
       3,    0,    0,    0,    0,
       3,    0,    2,   -2,    2,
       0,    0,    4,   -2,    2,
       0,    1,    2,    0,    1,
       0,    0,   -2,    2,    1,
       0,    0,    2,   -2,    3,
      -1,    0,    0,    4,    0,
       2,    0,   -2,    0,    1,
      -2,    0,    0,    4,    0,
      -1,   -1,    0,    2,    1,
      -1,    0,    0,    1,    1,
       0,    1,    0,    0,    2,
       0,    0,   -2,    0,    1,
       0,   -1,    2,    0,    1,
       0,    0,    2,   -1,    2,
       0,    0,    2,    4,    2,
      -2,   -1,    0,    2,    0,
       1,    1,    0,   -2,    1,
      -1,    1,    0,    2,    0,
      -1,    1,    0,    1,    2,
       1,   -1,    0,    0,    1,
       1,   -1,    2,    2,    2,
      -1,    1,    2,    2,    2,
       3,    0,    2,    0,    1,
       0,    1,   -2,    2,    0,
      -1,    0,    0,   -2,    1,
       0,    1,    2,    2,    2,
      -1,   -1,    2,    2,    1,
       0,   -1,    0,    0,    2,
       1,    0,    2,   -4,    1,
      -1,    0,   -2,    2,    0,
       0,   -1,    2,    2,    1,
       2,   -1,    2,    0,    2,
       0,    0,    0,    2,    2,
       1,   -1,    2,    0,    1,
      -1,    1,    2,    0,    2,
       0,    1,    0,    2,    0,
       0,   -1,   -2,    2,    0,
       0,    3,    2,   -2,    2,
       0,    0,    0,    1,    1,
      -1,    0,    2,    2,    0,
       2,    1,    2,    0,    2,
       1,    1,    0,    0,    1,
       1,    1,    2,    0,    1,
       2,    0,    0,    2,    0,
       1,    0,   -2,    2,    0,
      -1,    0,    0,    2,    2,
       0,    1,    0,    1,    0,
       0,    1,    0,   -2,    1,
      -1,    0,    2,   -2,    2,
       0,    0,    0,   -1,    1,
      -1,    1,    0,    0,    1,
       1,    0,    2,   -1,    2,
       1,   -1,    0,    2,    0,
       0,    0,    0,    4,    0,
       1,    0,    2,    1,    2,
       0,    0,    2,    1,    1,
       1,    0,    0,   -2,    2,
      -1,    0,    2,    4,    1,
       1,    0,   -2,    0,    1,
       1,    1,    2,   -2,    1,
       0,    0,    2,    2,    0,
      -1,    0,    2,   -1,    1,
      -2,    0,    2,    2,    1,
       4,    0,    2,    0,    2,
       2,   -1,    0,    0,    0,
       2,    1,    2,   -2,    2,
       0,    1,    2,    1,    2,
       1,    0,    4,   -2,    2,
      -1,   -1,    0,    0,    1,
       0,    1,    0,    2,    1,
      -2,    0,    2,    4,    1,
       2,    0,    2,    0,    0,
       1,    0,    0,    1,    0,
      -1,    0,    0,    4,    1,
      -1,    0,    4,    0,    1,
       2,    0,    2,    2,    1,
       0,    0,    2,   -3,    2,
      -1,   -2,    0,    2,    0,
       2,    1,    0,    0,    0,
       0,    0,    4,    0,    2,
       0,    0,    0,    0,    3,
       0,    3,    0,    0,    0,
       0,    0,    2,   -4,    1,
       0,   -1,    0,    2,    1,
       0,    0,    0,    4,    1,
      -1,   -1,    2,    4,    2,
       1,    0,    2,    4,    2,
      -2,    2,    0,    2,    0,
      -2,   -1,    2,    0,    1,
      -2,    0,    0,    2,    2,
      -1,   -1,    2,    0,    2,
       0,    0,    4,   -2,    1,
       3,    0,    2,   -2,    1,
      -2,   -1,    0,    2,    1,
       1,    0,    0,   -1,    1,
       0,   -2,    0,    2,    0,
      -2,    0,    0,    4,    1,
      -3,    0,    0,    0,    1,
       1,    1,    2,    2,    2,
       0,    0,    2,    4,    1,
       3,    0,    2,    2,    2,
      -1,    1,    2,   -2,    1,
       2,    0,    0,   -4,    1,
       0,    0,    0,   -2,    2,
       2,    0,    2,   -4,    1,
      -1,    1,    0,    2,    1,
       0,    0,    2,   -1,    1,
       0,   -2,    2,    2,    2,
       2,    0,    0,    2,    1,
       4,    0,    2,   -2,    2,
       2,    0,    0,   -2,    2,
       0,    2,    0,    0,    1,
       1,    0,    0,   -4,    1,
       0,    2,    2,   -2,    1,
      -3,    0,    0,    4,    0,
      -1,    1,    2,    0,    1,
      -1,   -1,    0,    4,    0,
      -1,   -2,    2,    2,    2,
      -2,   -1,    2,    4,    2,
       1,   -1,    2,    2,    1,
      -2,    1,    0,    2,    0,
      -2,    1,    2,    0,    1,
       2,    1,    0,   -2,    1,
      -3,    0,    2,    0,    1,
      -2,    0,    2,   -2,    1,
      -1,    1,    0,    2,    2,
       0,   -1,    2,   -1,    2,
      -1,    0,    4,   -2,    2,
       0,   -2,    2,    0,    2,
      -1,    0,    2,    1,    2,
       2,    0,    0,    0,    2,
       0,    0,    2,    0,    3,
      -2,    0,    4,    0,    2,
      -1,    0,   -2,    0,    1,
      -1,    1,    2,    2,    1,
       3,    0,    0,    0,    1,
      -1,    0,    2,    3,    2,
       2,   -1,    2,    0,    1,
       0,    1,    2,    2,    1,
       0,   -1,    2,    4,    2,
       2,   -1,    2,    2,    2,
       0,    2,   -2,    2,    0,
      -1,   -1,    2,   -1,    1,
       0,   -2,    0,    0,    1,
       1,    0,    2,   -4,    2,
       1,   -1,    0,   -2,    1,
      -1,   -1,    2,    0,    1,
       1,   -1,    2,   -2,    2,
      -2,   -1,    0,    4,    0,
      -1,    0,    0,    3,    0,
      -2,   -1,    2,    2,    2,
       0,    2,    2,    0,    2,
       1,    1,    0,    2,    0,
       2,    0,    2,   -1,    2,
       1,    0,    2,    1,    1,
       4,    0,    0,    0,    0,
       2,    1,    2,    0,    1,
       3,   -1,    2,    0,    2,
      -2,    2,    0,    2,    1,
       1,    0,    2,   -3,    1,
       1,    1,    2,   -4,    1,
      -1,   -1,    2,   -2,    1,
       0,   -1,    0,   -1,    1,
       0,   -1,    0,   -2,    1,
      -2,    0,    0,    0,    2,
      -2,    0,   -2,    2,    0,
      -1,    0,   -2,    4,    0,
       1,   -2,    0,    0,    0,
       0,    1,    0,    1,    1,
      -1,    2,    0,    2,    0,
       1,   -1,    2,   -2,    1,
       1,    2,    2,   -2,    2,
       2,   -1,    2,   -2,    2,
       1,    0,    2,   -1,    1,
       2,    1,    2,   -2,    1,
      -2,    0,    0,   -2,    1,
       1,   -2,    2,    0,    2,
       0,    1,    2,    1,    1,
       1,    0,    4,   -2,    1,
      -2,    0,    4,    2,    2,
       1,    1,    2,    1,    2,
       1,    0,    0,    4,    0,
       1,    0,    2,    2,    0,
       2,    0,    2,    1,    2,
       3,    1,    2,    0,    2,
       4,    0,    2,    0,    1,
      -2,   -1,    2,    0,    0,
       0,    1,   -2,    2,    1,
       1,    0,   -2,    1,    0,
       2,   -1,    0,   -2,    1,
      -1,    0,    2,   -1,    2,
       1,    0,    2,   -3,    2,
       0,    1,    2,   -2,    3,
      -1,    0,   -2,    2,    1,
       0,    0,    2,   -4,    2,
       2,    0,    2,   -4,    2,
       0,    0,    4,   -4,    4,
       0,    0,    4,   -4,    2,
      -2,    0,    0,    3,    0,
       1,    0,   -2,    2,    1,
      -3,    0,    2,    2,    2,
      -2,    0,    2,    2,    0,
       2,   -1,    0,    0,    1,
       1,    1,    0,    1,    0,
       0,    1,    4,   -2,    2,
      -1,    1,    0,   -2,    1,
       0,    0,    0,   -4,    1,
       1,   -1,    0,    2,    1,
       1,    1,    0,    2,    1,
      -1,    2,    2,    2,    2,
       3,    1,    2,   -2,    2,
       0,   -1,    0,    4,    0,
       2,   -1,    0,    2,    0,
       0,    0,    4,    0,    1,
       2,    0,    4,   -2,    2,
      -1,   -1,    2,    4,    1,
       1,    0,    0,    4,    1,
       1,   -2,    2,    2,    2,
       0,    0,    2,    3,    2,
      -1,    1,    2,    4,    2,
       3,    0,    0,    2,    0,
      -1,    0,    4,    2,    2,
      -2,    0,    2,    6,    2,
      -1,    0,    2,    6,    2,
       1,    1,   -2,    1,    0,
      -1,    0,    0,    1,    2,
      -1,   -1,    0,    1,    0,
      -2,    0,    0,    1,    0,
       0,    0,   -2,    1,    0,
       1,   -1,   -2,    2,    0,
       1,    2,    0,    0,    0,
       3,    0,    2,    0,    0,
       0,   -1,    1,   -1,    1,
      -1,    0,    1,    0,    3,
      -1,    0,    1,    0,    2,
      -1,    0,    1,    0,    1,
      -1,    0,    1,    0,    0,
       0,    0,    1,    0,    2,
       0,    0,    1,    0,    1,
       0,    0,    1,    0,    0};

/*
   Luni-Solar nutation coefficients, unit 1e-7 arcsec:
   longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)

   Each row of coefficients in 'cls_t' belongs with the corresponding
   row of fundamental-argument multipliers in 'nals_t'.
*/

   const double cls_t[323][6] =
      {-172064161.0,-174666.0, 33386.0, 92052331.0, 9086.0,15377.0,
        -13170906.0,  -1675.0,-13696.0,  5730336.0,-3015.0,-4587.0,
         -2276413.0,   -234.0,  2796.0,   978459.0, -485.0, 1374.0,
          2074554.0,    207.0,  -698.0,  -897492.0,  470.0, -291.0,
          1475877.0,  -3633.0, 11817.0,    73871.0, -184.0,-1924.0,
          -516821.0,   1226.0,  -524.0,   224386.0, -677.0, -174.0,
           711159.0,     73.0,  -872.0,    -6750.0,    0.0,  358.0,
          -387298.0,   -367.0,   380.0,   200728.0,   18.0,  318.0,
          -301461.0,    -36.0,   816.0,   129025.0,  -63.0,  367.0,
           215829.0,   -494.0,   111.0,   -95929.0,  299.0,  132.0,
           128227.0,    137.0,   181.0,   -68982.0,   -9.0,   39.0,
           123457.0,     11.0,    19.0,   -53311.0,   32.0,   -4.0,
           156994.0,     10.0,  -168.0,    -1235.0,    0.0,   82.0,
            63110.0,     63.0,    27.0,   -33228.0,    0.0,   -9.0,
           -57976.0,    -63.0,  -189.0,    31429.0,    0.0,  -75.0,
           -59641.0,    -11.0,   149.0,    25543.0,  -11.0,   66.0,
           -51613.0,    -42.0,   129.0,    26366.0,    0.0,   78.0,
            45893.0,     50.0,    31.0,   -24236.0,  -10.0,   20.0,
            63384.0,     11.0,  -150.0,    -1220.0,    0.0,   29.0,
           -38571.0,     -1.0,   158.0,    16452.0,  -11.0,   68.0,
            32481.0,      0.0,     0.0,   -13870.0,    0.0,    0.0,
           -47722.0,      0.0,   -18.0,      477.0,    0.0,  -25.0,
           -31046.0,     -1.0,   131.0,    13238.0,  -11.0,   59.0,
            28593.0,      0.0,    -1.0,   -12338.0,   10.0,   -3.0,
            20441.0,     21.0,    10.0,   -10758.0,    0.0,   -3.0,
            29243.0,      0.0,   -74.0,     -609.0,    0.0,   13.0,
            25887.0,      0.0,   -66.0,     -550.0,    0.0,   11.0,
           -14053.0,    -25.0,    79.0,     8551.0,   -2.0,  -45.0,
            15164.0,     10.0,    11.0,    -8001.0,    0.0,   -1.0,
           -15794.0,     72.0,   -16.0,     6850.0,  -42.0,   -5.0,
            21783.0,      0.0,    13.0,     -167.0,    0.0,   13.0,
           -12873.0,    -10.0,   -37.0,     6953.0,    0.0,  -14.0,
           -12654.0,     11.0,    63.0,     6415.0,    0.0,   26.0,
           -10204.0,      0.0,    25.0,     5222.0,    0.0,   15.0,
            16707.0,    -85.0,   -10.0,      168.0,   -1.0,   10.0,
            -7691.0,      0.0,    44.0,     3268.0,    0.0,   19.0,
           -11024.0,      0.0,   -14.0,      104.0,    0.0,    2.0,
             7566.0,    -21.0,   -11.0,    -3250.0,    0.0,   -5.0,
            -6637.0,    -11.0,    25.0,     3353.0,    0.0,   14.0,
            -7141.0,     21.0,     8.0,     3070.0,    0.0,    4.0,
            -6302.0,    -11.0,     2.0,     3272.0,    0.0,    4.0,
             5800.0,     10.0,     2.0,    -3045.0,    0.0,   -1.0,
             6443.0,      0.0,    -7.0,    -2768.0,    0.0,   -4.0,
            -5774.0,    -11.0,   -15.0,     3041.0,    0.0,   -5.0,
            -5350.0,      0.0,    21.0,     2695.0,    0.0,   12.0,
            -4752.0,    -11.0,    -3.0,     2719.0,    0.0,   -3.0,
            -4940.0,    -11.0,   -21.0,     2720.0,    0.0,   -9.0,
             7350.0,      0.0,    -8.0,      -51.0,    0.0,    4.0,
             4065.0,      0.0,     6.0,    -2206.0,    0.0,    1.0,
             6579.0,      0.0,   -24.0,     -199.0,    0.0,    2.0,
             3579.0,      0.0,     5.0,    -1900.0,    0.0,    1.0,
             4725.0,      0.0,    -6.0,      -41.0,    0.0,    3.0,
            -3075.0,      0.0,    -2.0,     1313.0,    0.0,   -1.0,
            -2904.0,      0.0,    15.0,     1233.0,    0.0,    7.0,
             4348.0,      0.0,   -10.0,      -81.0,    0.0,    2.0,
            -2878.0,      0.0,     8.0,     1232.0,    0.0,    4.0,
            -4230.0,      0.0,     5.0,      -20.0,    0.0,   -2.0,
            -2819.0,      0.0,     7.0,     1207.0,    0.0,    3.0,
            -4056.0,      0.0,     5.0,       40.0,    0.0,   -2.0,
            -2647.0,      0.0,    11.0,     1129.0,    0.0,    5.0,
            -2294.0,      0.0,   -10.0,     1266.0,    0.0,   -4.0,
             2481.0,      0.0,    -7.0,    -1062.0,    0.0,   -3.0,
             2179.0,      0.0,    -2.0,    -1129.0,    0.0,   -2.0,
             3276.0,      0.0,     1.0,       -9.0,    0.0,    0.0,
            -3389.0,      0.0,     5.0,       35.0,    0.0,   -2.0,
             3339.0,      0.0,   -13.0,     -107.0,    0.0,    1.0,
            -1987.0,      0.0,    -6.0,     1073.0,    0.0,   -2.0,
            -1981.0,      0.0,     0.0,      854.0,    0.0,    0.0,
             4026.0,      0.0,  -353.0,     -553.0,    0.0, -139.0,
             1660.0,      0.0,    -5.0,     -710.0,    0.0,   -2.0,
            -1521.0,      0.0,     9.0,      647.0,    0.0,    4.0,
             1314.0,      0.0,     0.0,     -700.0,    0.0,    0.0,
            -1283.0,      0.0,     0.0,      672.0,    0.0,    0.0,
            -1331.0,      0.0,     8.0,      663.0,    0.0,    4.0,
             1383.0,      0.0,    -2.0,     -594.0,    0.0,   -2.0,
             1405.0,      0.0,     4.0,     -610.0,    0.0,    2.0,
             1290.0,      0.0,     0.0,     -556.0,    0.0,    0.0,
            -1214.0,      0.0,     5.0,      518.0,    0.0,    2.0,
             1146.0,      0.0,    -3.0,     -490.0,    0.0,   -1.0,
             1019.0,      0.0,    -1.0,     -527.0,    0.0,   -1.0,
            -1100.0,      0.0,     9.0,      465.0,    0.0,    4.0,
             -970.0,      0.0,     2.0,      496.0,    0.0,    1.0,
             1575.0,      0.0,    -6.0,      -50.0,    0.0,    0.0,
              934.0,      0.0,    -3.0,     -399.0,    0.0,   -1.0,
              922.0,      0.0,    -1.0,     -395.0,    0.0,   -1.0,
              815.0,      0.0,    -1.0,     -422.0,    0.0,   -1.0,
              834.0,      0.0,     2.0,     -440.0,    0.0,    1.0,
             1248.0,      0.0,     0.0,     -170.0,    0.0,    1.0,
             1338.0,      0.0,    -5.0,      -39.0,    0.0,    0.0,
              716.0,      0.0,    -2.0,     -389.0,    0.0,   -1.0,
             1282.0,      0.0,    -3.0,      -23.0,    0.0,    1.0,
              742.0,      0.0,     1.0,     -391.0,    0.0,    0.0,
             1020.0,      0.0,   -25.0,     -495.0,    0.0,  -10.0,
              715.0,      0.0,    -4.0,     -326.0,    0.0,    2.0,
             -666.0,      0.0,    -3.0,      369.0,    0.0,   -1.0,
             -667.0,      0.0,     1.0,      346.0,    0.0,    1.0,
             -704.0,      0.0,     0.0,      304.0,    0.0,    0.0,
             -694.0,      0.0,     5.0,      294.0,    0.0,    2.0,
            -1014.0,      0.0,    -1.0,        4.0,    0.0,   -1.0,
             -585.0,      0.0,    -2.0,      316.0,    0.0,   -1.0,
             -949.0,      0.0,     1.0,        8.0,    0.0,   -1.0,
             -595.0,      0.0,     0.0,      258.0,    0.0,    0.0,
              528.0,      0.0,     0.0,     -279.0,    0.0,    0.0,
             -590.0,      0.0,     4.0,      252.0,    0.0,    2.0,
              570.0,      0.0,    -2.0,     -244.0,    0.0,   -1.0,
             -502.0,      0.0,     3.0,      250.0,    0.0,    2.0,
             -875.0,      0.0,     1.0,       29.0,    0.0,    0.0,
             -492.0,      0.0,    -3.0,      275.0,    0.0,   -1.0,
              535.0,      0.0,    -2.0,     -228.0,    0.0,   -1.0,
             -467.0,      0.0,     1.0,      240.0,    0.0,    1.0,
              591.0,      0.0,     0.0,     -253.0,    0.0,    0.0,
             -453.0,      0.0,    -1.0,      244.0,    0.0,   -1.0,
              766.0,      0.0,     1.0,        9.0,    0.0,    0.0,
             -446.0,      0.0,     2.0,      225.0,    0.0,    1.0,
             -488.0,      0.0,     2.0,      207.0,    0.0,    1.0,
             -468.0,      0.0,     0.0,      201.0,    0.0,    0.0,
             -421.0,      0.0,     1.0,      216.0,    0.0,    1.0,
              463.0,      0.0,     0.0,     -200.0,    0.0,    0.0,
             -673.0,      0.0,     2.0,       14.0,    0.0,    0.0,
              658.0,      0.0,     0.0,       -2.0,    0.0,    0.0,
             -438.0,      0.0,     0.0,      188.0,    0.0,    0.0,
             -390.0,      0.0,     0.0,      205.0,    0.0,    0.0,
              639.0,    -11.0,    -2.0,      -19.0,    0.0,    0.0,
              412.0,      0.0,    -2.0,     -176.0,    0.0,   -1.0,
             -361.0,      0.0,     0.0,      189.0,    0.0,    0.0,
              360.0,      0.0,    -1.0,     -185.0,    0.0,   -1.0,
              588.0,      0.0,    -3.0,      -24.0,    0.0,    0.0,
             -578.0,      0.0,     1.0,        5.0,    0.0,    0.0,
             -396.0,      0.0,     0.0,      171.0,    0.0,    0.0,
              565.0,      0.0,    -1.0,       -6.0,    0.0,    0.0,
             -335.0,      0.0,    -1.0,      184.0,    0.0,   -1.0,
              357.0,      0.0,     1.0,     -154.0,    0.0,    0.0,
              321.0,      0.0,     1.0,     -174.0,    0.0,    0.0,
             -301.0,      0.0,    -1.0,      162.0,    0.0,    0.0,
             -334.0,      0.0,     0.0,      144.0,    0.0,    0.0,
              493.0,      0.0,    -2.0,      -15.0,    0.0,    0.0,
              494.0,      0.0,    -2.0,      -19.0,    0.0,    0.0,
              337.0,      0.0,    -1.0,     -143.0,    0.0,   -1.0,
              280.0,      0.0,    -1.0,     -144.0,    0.0,    0.0,
              309.0,      0.0,     1.0,     -134.0,    0.0,    0.0,
             -263.0,      0.0,     2.0,      131.0,    0.0,    1.0,
              253.0,      0.0,     1.0,     -138.0,    0.0,    0.0,
              245.0,      0.0,     0.0,     -128.0,    0.0,    0.0,
              416.0,      0.0,    -2.0,      -17.0,    0.0,    0.0,
             -229.0,      0.0,     0.0,      128.0,    0.0,    0.0,
              231.0,      0.0,     0.0,     -120.0,    0.0,    0.0,
             -259.0,      0.0,     2.0,      109.0,    0.0,    1.0,
              375.0,      0.0,    -1.0,       -8.0,    0.0,    0.0,
              252.0,      0.0,     0.0,     -108.0,    0.0,    0.0,
             -245.0,      0.0,     1.0,      104.0,    0.0,    0.0,
              243.0,      0.0,    -1.0,     -104.0,    0.0,    0.0,
              208.0,      0.0,     1.0,     -112.0,    0.0,    0.0,
              199.0,      0.0,     0.0,     -102.0,    0.0,    0.0,
             -208.0,      0.0,     1.0,      105.0,    0.0,    0.0,
              335.0,      0.0,    -2.0,      -14.0,    0.0,    0.0,
             -325.0,      0.0,     1.0,        7.0,    0.0,    0.0,
             -187.0,      0.0,     0.0,       96.0,    0.0,    0.0,
              197.0,      0.0,    -1.0,     -100.0,    0.0,    0.0,
             -192.0,      0.0,     2.0,       94.0,    0.0,    1.0,
             -188.0,      0.0,     0.0,       83.0,    0.0,    0.0,
              276.0,      0.0,     0.0,       -2.0,    0.0,    0.0,
             -286.0,      0.0,     1.0,        6.0,    0.0,    0.0,
              186.0,      0.0,    -1.0,      -79.0,    0.0,    0.0,
             -219.0,      0.0,     0.0,       43.0,    0.0,    0.0,
              276.0,      0.0,     0.0,        2.0,    0.0,    0.0,
             -153.0,      0.0,    -1.0,       84.0,    0.0,    0.0,
             -156.0,      0.0,     0.0,       81.0,    0.0,    0.0,
             -154.0,      0.0,     1.0,       78.0,    0.0,    0.0,
             -174.0,      0.0,     1.0,       75.0,    0.0,    0.0,
             -163.0,      0.0,     2.0,       69.0,    0.0,    1.0,
             -228.0,      0.0,     0.0,        1.0,    0.0,    0.0,
               91.0,      0.0,    -4.0,      -54.0,    0.0,   -2.0,
              175.0,      0.0,     0.0,      -75.0,    0.0,    0.0,
             -159.0,      0.0,     0.0,       69.0,    0.0,    0.0,
              141.0,      0.0,     0.0,      -72.0,    0.0,    0.0,
              147.0,      0.0,     0.0,      -75.0,    0.0,    0.0,
             -132.0,      0.0,     0.0,       69.0,    0.0,    0.0,
              159.0,      0.0,   -28.0,      -54.0,    0.0,   11.0,
              213.0,      0.0,     0.0,       -4.0,    0.0,    0.0,
              123.0,      0.0,     0.0,      -64.0,    0.0,    0.0,
             -118.0,      0.0,    -1.0,       66.0,    0.0,    0.0,
              144.0,      0.0,    -1.0,      -61.0,    0.0,    0.0,
             -121.0,      0.0,     1.0,       60.0,    0.0,    0.0,
             -134.0,      0.0,     1.0,       56.0,    0.0,    1.0,
             -105.0,      0.0,     0.0,       57.0,    0.0,    0.0,
             -102.0,      0.0,     0.0,       56.0,    0.0,    0.0,
              120.0,      0.0,     0.0,      -52.0,    0.0,    0.0,
              101.0,      0.0,     0.0,      -54.0,    0.0,    0.0,
             -113.0,      0.0,     0.0,       59.0,    0.0,    0.0,
             -106.0,      0.0,     0.0,       61.0,    0.0,    0.0,
             -129.0,      0.0,     1.0,       55.0,    0.0,    0.0,
             -114.0,      0.0,     0.0,       57.0,    0.0,    0.0,
              113.0,      0.0,    -1.0,      -49.0,    0.0,    0.0,
             -102.0,      0.0,     0.0,       44.0,    0.0,    0.0,
              -94.0,      0.0,     0.0,       51.0,    0.0,    0.0,
             -100.0,      0.0,    -1.0,       56.0,    0.0,    0.0,
               87.0,      0.0,     0.0,      -47.0,    0.0,    0.0,
              161.0,      0.0,     0.0,       -1.0,    0.0,    0.0,
               96.0,      0.0,     0.0,      -50.0,    0.0,    0.0,
              151.0,      0.0,    -1.0,       -5.0,    0.0,    0.0,
             -104.0,      0.0,     0.0,       44.0,    0.0,    0.0,
             -110.0,      0.0,     0.0,       48.0,    0.0,    0.0,
             -100.0,      0.0,     1.0,       50.0,    0.0,    0.0,
               92.0,      0.0,    -5.0,       12.0,    0.0,   -2.0,
               82.0,      0.0,     0.0,      -45.0,    0.0,    0.0,
               82.0,      0.0,     0.0,      -45.0,    0.0,    0.0,
              -78.0,      0.0,     0.0,       41.0,    0.0,    0.0,
              -77.0,      0.0,     0.0,       43.0,    0.0,    0.0,
                2.0,      0.0,     0.0,       54.0,    0.0,    0.0,
               94.0,      0.0,     0.0,      -40.0,    0.0,    0.0,
              -93.0,      0.0,     0.0,       40.0,    0.0,    0.0,
              -83.0,      0.0,    10.0,       40.0,    0.0,   -2.0,
               83.0,      0.0,     0.0,      -36.0,    0.0,    0.0,
              -91.0,      0.0,     0.0,       39.0,    0.0,    0.0,
              128.0,      0.0,     0.0,       -1.0,    0.0,    0.0,
              -79.0,      0.0,     0.0,       34.0,    0.0,    0.0,
              -83.0,      0.0,     0.0,       47.0,    0.0,    0.0,
               84.0,      0.0,     0.0,      -44.0,    0.0,    0.0,
               83.0,      0.0,     0.0,      -43.0,    0.0,    0.0,
               91.0,      0.0,     0.0,      -39.0,    0.0,    0.0,
              -77.0,      0.0,     0.0,       39.0,    0.0,    0.0,
               84.0,      0.0,     0.0,      -43.0,    0.0,    0.0,
              -92.0,      0.0,     1.0,       39.0,    0.0,    0.0,
              -92.0,      0.0,     1.0,       39.0,    0.0,    0.0,
              -94.0,      0.0,     0.0,        0.0,    0.0,    0.0,
               68.0,      0.0,     0.0,      -36.0,    0.0,    0.0,
              -61.0,      0.0,     0.0,       32.0,    0.0,    0.0,
               71.0,      0.0,     0.0,      -31.0,    0.0,    0.0,
               62.0,      0.0,     0.0,      -34.0,    0.0,    0.0,
              -63.0,      0.0,     0.0,       33.0,    0.0,    0.0,
              -73.0,      0.0,     0.0,       32.0,    0.0,    0.0,
              115.0,      0.0,     0.0,       -2.0,    0.0,    0.0,
             -103.0,      0.0,     0.0,        2.0,    0.0,    0.0,
               63.0,      0.0,     0.0,      -28.0,    0.0,    0.0,
               74.0,      0.0,     0.0,      -32.0,    0.0,    0.0,
             -103.0,      0.0,    -3.0,        3.0,    0.0,   -1.0,
              -69.0,      0.0,     0.0,       30.0,    0.0,    0.0,
               57.0,      0.0,     0.0,      -29.0,    0.0,    0.0,
               94.0,      0.0,     0.0,       -4.0,    0.0,    0.0,
               64.0,      0.0,     0.0,      -33.0,    0.0,    0.0,
              -63.0,      0.0,     0.0,       26.0,    0.0,    0.0,
              -38.0,      0.0,     0.0,       20.0,    0.0,    0.0,
              -43.0,      0.0,     0.0,       24.0,    0.0,    0.0,
              -45.0,      0.0,     0.0,       23.0,    0.0,    0.0,
               47.0,      0.0,     0.0,      -24.0,    0.0,    0.0,
              -48.0,      0.0,     0.0,       25.0,    0.0,    0.0,
               45.0,      0.0,     0.0,      -26.0,    0.0,    0.0,
               56.0,      0.0,     0.0,      -25.0,    0.0,    0.0,
               88.0,      0.0,     0.0,        2.0,    0.0,    0.0,
              -75.0,      0.0,     0.0,        0.0,    0.0,    0.0,
               85.0,      0.0,     0.0,        0.0,    0.0,    0.0,
               49.0,      0.0,     0.0,      -26.0,    0.0,    0.0,
              -74.0,      0.0,    -3.0,       -1.0,    0.0,   -1.0,
              -39.0,      0.0,     0.0,       21.0,    0.0,    0.0,
               45.0,      0.0,     0.0,      -20.0,    0.0,    0.0,
               51.0,      0.0,     0.0,      -22.0,    0.0,    0.0,
              -40.0,      0.0,     0.0,       21.0,    0.0,    0.0,
               41.0,      0.0,     0.0,      -21.0,    0.0,    0.0,
              -42.0,      0.0,     0.0,       24.0,    0.0,    0.0,
              -51.0,      0.0,     0.0,       22.0,    0.0,    0.0,
              -42.0,      0.0,     0.0,       22.0,    0.0,    0.0,
               39.0,      0.0,     0.0,      -21.0,    0.0,    0.0,
               46.0,      0.0,     0.0,      -18.0,    0.0,    0.0,
              -53.0,      0.0,     0.0,       22.0,    0.0,    0.0,
               82.0,      0.0,     0.0,       -4.0,    0.0,    0.0,
               81.0,      0.0,    -1.0,       -4.0,    0.0,    0.0,
               47.0,      0.0,     0.0,      -19.0,    0.0,    0.0,
               53.0,      0.0,     0.0,      -23.0,    0.0,    0.0,
              -45.0,      0.0,     0.0,       22.0,    0.0,    0.0,
              -44.0,      0.0,     0.0,       -2.0,    0.0,    0.0,
              -33.0,      0.0,     0.0,       16.0,    0.0,    0.0,
              -61.0,      0.0,     0.0,        1.0,    0.0,    0.0,
              -38.0,      0.0,     0.0,       19.0,    0.0,    0.0,
              -33.0,      0.0,     0.0,       21.0,    0.0,    0.0,
              -60.0,      0.0,     0.0,        0.0,    0.0,    0.0,
               48.0,      0.0,     0.0,      -10.0,    0.0,    0.0,
               38.0,      0.0,     0.0,      -20.0,    0.0,    0.0,
               31.0,      0.0,     0.0,      -13.0,    0.0,    0.0,
              -32.0,      0.0,     0.0,       15.0,    0.0,    0.0,
               45.0,      0.0,     0.0,       -8.0,    0.0,    0.0,
              -44.0,      0.0,     0.0,       19.0,    0.0,    0.0,
              -51.0,      0.0,     0.0,        0.0,    0.0,    0.0,
              -36.0,      0.0,     0.0,       20.0,    0.0,    0.0,
               44.0,      0.0,     0.0,      -19.0,    0.0,    0.0,
              -60.0,      0.0,     0.0,        2.0,    0.0,    0.0,
               35.0,      0.0,     0.0,      -18.0,    0.0,    0.0,
               47.0,      0.0,     0.0,       -1.0,    0.0,    0.0,
               36.0,      0.0,     0.0,      -15.0,    0.0,    0.0,
              -36.0,      0.0,     0.0,       20.0,    0.0,    0.0,
              -35.0,      0.0,     0.0,       19.0,    0.0,    0.0,
              -37.0,      0.0,     0.0,       19.0,    0.0,    0.0,
               32.0,      0.0,     0.0,      -16.0,    0.0,    0.0,
               35.0,      0.0,     0.0,      -14.0,    0.0,    0.0,
               32.0,      0.0,     0.0,      -13.0,    0.0,    0.0,
               65.0,      0.0,     0.0,       -2.0,    0.0,    0.0,
               47.0,      0.0,     0.0,       -1.0,    0.0,    0.0,
               32.0,      0.0,     0.0,      -16.0,    0.0,    0.0,
               37.0,      0.0,     0.0,      -16.0,    0.0,    0.0,
              -30.0,      0.0,     0.0,       15.0,    0.0,    0.0,
              -32.0,      0.0,     0.0,       16.0,    0.0,    0.0,
              -31.0,      0.0,     0.0,       13.0,    0.0,    0.0,
               37.0,      0.0,     0.0,      -16.0,    0.0,    0.0,
               31.0,      0.0,     0.0,      -13.0,    0.0,    0.0,
               49.0,      0.0,     0.0,       -2.0,    0.0,    0.0,
               32.0,      0.0,     0.0,      -13.0,    0.0,    0.0,
              -43.0,      0.0,     0.0,       18.0,    0.0,    0.0,
              -32.0,      0.0,     0.0,       14.0,    0.0,    0.0,
               30.0,      0.0,     0.0,        0.0,    0.0,    0.0,
              -34.0,      0.0,     0.0,       15.0,    0.0,    0.0,
              -36.0,      0.0,     0.0,        0.0,    0.0,    0.0,
              -38.0,      0.0,     0.0,        0.0,    0.0,    0.0,
              -31.0,      0.0,     0.0,        0.0,    0.0,    0.0,
              -34.0,      0.0,     0.0,        0.0,    0.0,    0.0,
              -35.0,      0.0,     0.0,        0.0,    0.0,    0.0,
               30.0,      0.0,     0.0,       -2.0,    0.0,    0.0,
                0.0,      0.0, -1988.0,        0.0,    0.0,-1679.0,
                0.0,      0.0,   -63.0,        0.0,    0.0,  -27.0,
                0.0,      0.0,   364.0,        0.0,    0.0,  176.0,
                0.0,      0.0, -1044.0,        0.0,    0.0, -891.0,
                0.0,      0.0,   330.0,        0.0,    0.0,    0.0,
                0.0,      0.0,    30.0,        0.0,    0.0,   14.0,
                0.0,      0.0,  -162.0,        0.0,    0.0, -138.0,
                0.0,      0.0,    75.0,        0.0,    0.0,    0.0};

/*
   Planetary argument multipliers:
       L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
*/
   const short int napl_t[165][14] =
      {0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -8, 16, -4, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2,
       0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0, 10, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  1,
       1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0,
       1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0,
      -1,  0,  0,  0,  0,  0, 18,-16,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,
      -1,  0,  0,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,
       0,  0, -2,  2,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0,
       2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0,
      -1,  0,  0,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,
       0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0,
      -1,  0,  0,  0,  1,  0, 18,-16,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0,
      -2,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2,
       0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
       0,  0, -1,  1,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0,
      -2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2,
      -2,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0,
       2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0,
       0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,
       0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2,
       0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -3,  3,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -6, 16, -4, -5,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2,
       0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2,
      -1,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
       1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
      -1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0};

/*
   Planetary nutation coefficients, unit 1e-7 arcsec:
   longitude (sin, cos), obliquity (sin, cos)

   Each row of coefficients in 'cpl_t' belongs with the corresponding
   row of fundamental-argument multipliers in 'napl_t'.
*/

   const double cpl_t[165][4] =
      {1440.0,       0.0,       0.0,       0.0,
         56.0,    -117.0,     -42.0,     -40.0,
        125.0,     -43.0,       0.0,     -54.0,
       -114.0,       0.0,       0.0,      61.0,
       -219.0,      89.0,       0.0,       0.0,
       -462.0,    1604.0,       0.0,       0.0,
         99.0,       0.0,       0.0,     -53.0,
         14.0,    -218.0,     117.0,       8.0,
         31.0,    -481.0,    -257.0,     -17.0,
       -491.0,     128.0,       0.0,       0.0,
      -3084.0,    5123.0,    2735.0,    1647.0,
      -1444.0,    2409.0,   -1286.0,    -771.0,
        103.0,     -60.0,       0.0,       0.0,
        -26.0,     -29.0,     -16.0,      14.0,
        284.0,       0.0,       0.0,    -151.0,
        226.0,     101.0,       0.0,       0.0,
        -41.0,     175.0,      76.0,      17.0,
        425.0,     212.0,    -133.0,     269.0,
       1200.0,     598.0,     319.0,    -641.0,
        235.0,     334.0,       0.0,       0.0,
        266.0,     -78.0,       0.0,       0.0,
       -460.0,    -435.0,    -232.0,     246.0,
          0.0,     131.0,       0.0,       0.0,
        -42.0,      20.0,       0.0,       0.0,
        -10.0,     233.0,       0.0,       0.0,
         78.0,     -18.0,       0.0,       0.0,
         45.0,     -22.0,       0.0,       0.0,
         89.0,     -16.0,      -9.0,     -48.0,
       -349.0,     -62.0,       0.0,       0.0,
        -53.0,       0.0,       0.0,       0.0,
        -21.0,     -78.0,       0.0,       0.0,
         20.0,     -70.0,     -37.0,     -11.0,
         32.0,      15.0,      -8.0,      17.0,
        174.0,      84.0,      45.0,     -93.0,
         11.0,      56.0,       0.0,       0.0,
        -66.0,     -12.0,      -6.0,      35.0,
         47.0,       8.0,       4.0,     -25.0,
         46.0,      66.0,      35.0,     -25.0,
        -68.0,     -34.0,     -18.0,      36.0,
         76.0,      17.0,       9.0,     -41.0,
         84.0,     298.0,     159.0,     -45.0,
        -82.0,     292.0,     156.0,      44.0,
        -73.0,      17.0,       9.0,      39.0,
       -439.0,       0.0,       0.0,       0.0,
         57.0,     -28.0,     -15.0,     -30.0,
        -40.0,      57.0,      30.0,      21.0,
        273.0,      80.0,      43.0,    -146.0,
       -449.0,     430.0,       0.0,       0.0,
         -8.0,     -47.0,     -25.0,       4.0,
          6.0,      47.0,      25.0,      -3.0,
        -48.0,    -110.0,     -59.0,      26.0,
         51.0,     114.0,      61.0,     -27.0,
       -133.0,       0.0,       0.0,      57.0,
        -18.0,    -436.0,    -233.0,       9.0,
         35.0,      -7.0,       0.0,       0.0,
        -53.0,      -9.0,      -5.0,      28.0,
        -50.0,     194.0,     103.0,      27.0,
        -13.0,      52.0,      28.0,       7.0,
        -91.0,     248.0,       0.0,       0.0,
          6.0,      49.0,      26.0,      -3.0,
         -6.0,     -47.0,     -25.0,       3.0,
         52.0,      23.0,      10.0,     -23.0,
       -138.0,       0.0,       0.0,       0.0,
         54.0,       0.0,       0.0,     -29.0,
        -37.0,      35.0,      19.0,      20.0,
       -145.0,      47.0,       0.0,       0.0,
        -10.0,      40.0,      21.0,       5.0,
         11.0,     -49.0,     -26.0,      -7.0,
      -2150.0,       0.0,       0.0,     932.0,
         85.0,       0.0,       0.0,     -37.0,
        -86.0,     153.0,       0.0,       0.0,
        -51.0,       0.0,       0.0,      22.0,
        -11.0,    -268.0,    -116.0,       5.0,
         31.0,       6.0,       3.0,     -17.0,
        140.0,      27.0,      14.0,     -75.0,
         57.0,      11.0,       6.0,     -30.0,
        -14.0,     -39.0,       0.0,       0.0,
        -25.0,      22.0,       0.0,       0.0,
         42.0,     223.0,     119.0,     -22.0,
        -27.0,    -143.0,     -77.0,      14.0,
          9.0,      49.0,      26.0,      -5.0,
      -1166.0,       0.0,       0.0,     505.0,
        117.0,       0.0,       0.0,     -63.0,
          0.0,      31.0,       0.0,       0.0,
          0.0,     -32.0,     -17.0,       0.0,
         50.0,       0.0,       0.0,     -27.0,
         30.0,      -3.0,      -2.0,     -16.0,
          8.0,     614.0,       0.0,       0.0,
       -127.0,      21.0,       9.0,      55.0,
        -20.0,      34.0,       0.0,       0.0,
         22.0,     -87.0,       0.0,       0.0,
        -68.0,      39.0,       0.0,       0.0,
          3.0,      66.0,      29.0,      -1.0,
        490.0,       0.0,       0.0,    -213.0,
        -22.0,      93.0,      49.0,      12.0,
        -46.0,      14.0,       0.0,       0.0,
         25.0,     106.0,      57.0,     -13.0,
       1485.0,       0.0,       0.0,       0.0,
         -7.0,     -32.0,     -17.0,       4.0,
         30.0,      -6.0,      -2.0,     -13.0,
        118.0,       0.0,       0.0,     -52.0,
        -28.0,      36.0,       0.0,       0.0,
         14.0,     -59.0,     -31.0,      -8.0,
       -458.0,       0.0,       0.0,     198.0,
          0.0,     -45.0,     -20.0,       0.0,
       -166.0,     269.0,       0.0,       0.0,
        -78.0,      45.0,       0.0,       0.0,
         -5.0,     328.0,       0.0,       0.0,
      -1223.0,     -26.0,       0.0,       0.0,
       -368.0,       0.0,       0.0,       0.0,
        -75.0,       0.0,       0.0,       0.0,
        -13.0,     -30.0,       0.0,       0.0,
        -74.0,       0.0,       0.0,      32.0,
       -262.0,       0.0,       0.0,     114.0,
        202.0,       0.0,       0.0,     -87.0,
         -8.0,      35.0,      19.0,       5.0,
        -35.0,     -48.0,     -21.0,      15.0,
         12.0,      55.0,      29.0,      -6.0,
       -598.0,       0.0,       0.0,       0.0,
          8.0,     -31.0,     -16.0,      -4.0,
        113.0,       0.0,       0.0,     -49.0,
         83.0,      15.0,       0.0,       0.0,
          0.0,    -114.0,     -49.0,       0.0,
        117.0,       0.0,       0.0,     -51.0,
        393.0,       3.0,       0.0,       0.0,
         18.0,     -29.0,     -13.0,      -8.0,
          8.0,      34.0,      18.0,      -4.0,
         89.0,       0.0,       0.0,       0.0,
         54.0,     -15.0,      -7.0,     -24.0,
          0.0,      35.0,       0.0,       0.0,
       -154.0,     -30.0,     -13.0,      67.0,
         80.0,     -71.0,     -31.0,     -35.0,
         61.0,     -96.0,     -42.0,     -27.0,
        123.0,    -415.0,    -180.0,     -53.0,
          0.0,       0.0,       0.0,     -35.0,
          7.0,     -32.0,     -17.0,      -4.0,
        -89.0,       0.0,       0.0,      38.0,
          0.0,     -86.0,     -19.0,      -6.0,
       -123.0,    -416.0,    -180.0,      53.0,
        -62.0,     -97.0,     -42.0,      27.0,
        -85.0,     -70.0,     -31.0,      37.0,
        163.0,     -12.0,      -5.0,     -72.0,
        -63.0,     -16.0,      -7.0,      28.0,
        -21.0,     -32.0,     -14.0,       9.0,
          5.0,    -173.0,     -75.0,      -2.0,
         74.0,       0.0,       0.0,     -32.0,
         83.0,       0.0,       0.0,       0.0,
       -339.0,       0.0,       0.0,     147.0,
         67.0,     -91.0,     -39.0,     -29.0,
         30.0,     -18.0,      -8.0,     -13.0,
          0.0,    -114.0,     -50.0,       0.0,
        517.0,      16.0,       7.0,    -224.0,
        143.0,      -3.0,      -1.0,     -62.0,
         50.0,       0.0,       0.0,     -22.0,
         59.0,       0.0,       0.0,       0.0,
        370.0,      -8.0,       0.0,    -160.0,
         34.0,       0.0,       0.0,     -15.0,
        -37.0,      -7.0,      -3.0,      16.0,
         40.0,       0.0,       0.0,       0.0,
       -184.0,      -3.0,      -1.0,      80.0,
         31.0,      -6.0,       0.0,     -13.0,
         -3.0,     -32.0,     -14.0,       1.0,
        -34.0,       0.0,       0.0,       0.0,
        126.0,     -63.0,     -27.0,     -55.0,
       -126.0,     -63.0,     -27.0,      55.0};

/*
   Interval between fundamental epoch J2000.0 and given date.
*/

   t = ((jd_high - T0) + jd_low) / 36525.0;

/*
   Compute fundamental arguments from Simon et al. (1994),
   in radians.
*/

   fund_args (t, a);

/*
   ** Luni-solar nutation. **
*/

/*
   Initialize the nutation values.
*/

   dp = 0.0;
   de = 0.0;

/*
   Summation of luni-solar nutation series (in reverse order).
*/

   for (i = 322; i >= 0; i--)
   {

/*
   Argument and functions.
*/

      arg = fmod ((double) nals_t[i][0] * a[0]  +
                  (double) nals_t[i][1] * a[1]  +
                  (double) nals_t[i][2] * a[2]  +
                  (double) nals_t[i][3] * a[3]  +
                  (double) nals_t[i][4] * a[4], TWOPI);

      sarg = sin (arg);
      carg = cos (arg);

/*
   Term.
*/

      dp += (cls_t[i][0] + cls_t[i][1] * t) * sarg
              +   cls_t[i][2] * carg;
      de += (cls_t[i][3] + cls_t[i][4] * t) * carg
              +   cls_t[i][5] * sarg;
   }

/*
   Convert from 0.1 microarcsec units to radians.
*/

   factor = 1.0e-7 * ASEC2RAD;
   dpsils = dp * factor;
   depsls = de * factor;

/*
   ** Planetary nutation. **
*/

/*
   Planetary longitudes, Mercury through Neptune, wrt mean dynamical
   ecliptic and equinox of J2000, with high order terms omitted
   (Simon et al. 1994, 5.8.1-5.8.8).
*/

   alme = fmod (4.402608842461 + 2608.790314157421 * t, TWOPI);
   alve = fmod (3.176146696956 + 1021.328554621099 * t, TWOPI);
   alea = fmod (1.753470459496 +  628.307584999142 * t, TWOPI);
   alma = fmod (6.203476112911 +  334.061242669982 * t, TWOPI);
   alju = fmod (0.599547105074 +   52.969096264064 * t, TWOPI);
   alsa = fmod (0.874016284019 +   21.329910496032 * t, TWOPI);
   alur = fmod (5.481293871537 +    7.478159856729 * t, TWOPI);
   alne = fmod (5.311886286677 +    3.813303563778 * t, TWOPI);

/*
   General precession in longitude (Simon et al. 1994), equivalent
   to 5028.8200 arcsec/cy at J2000.
*/

   apa = (0.024380407358 + 0.000005391235 * t) * t;

/*
   Initialize the nutation values.
*/

   dp = 0.0;
   de = 0.0;

/*
   Summation of planetary nutation series (in reverse order).
*/

   for (i = 164; i >= 0; i--)
   {

/*
   Argument and functions.
*/

      arg = fmod ((double) napl_t[i][ 0] * a[0]  +
                  (double) napl_t[i][ 1] * a[1]  +
                  (double) napl_t[i][ 2] * a[2]  +
                  (double) napl_t[i][ 3] * a[3]  +
                  (double) napl_t[i][ 4] * a[4]  +
                  (double) napl_t[i][ 5] * alme  +
                  (double) napl_t[i][ 6] * alve  +
                  (double) napl_t[i][ 7] * alea  +
                  (double) napl_t[i][ 8] * alma  +
                  (double) napl_t[i][ 9] * alju  +
                  (double) napl_t[i][10] * alsa  +
                  (double) napl_t[i][11] * alur  +
                  (double) napl_t[i][12] * alne  +
                  (double) napl_t[i][13] * apa, TWOPI);

      sarg = sin (arg);
      carg = cos (arg);

/*
   Term.
*/

      dp += cpl_t[i][0] * sarg + cpl_t[i][1] * carg;
      de += cpl_t[i][2] * sarg + cpl_t[i][3] * carg;
   }

   dpsipl = dp * factor;
   depspl = de * factor;

/*
   Total: Add planetary and luni-solar components.
*/

   *dpsi = dpsipl + dpsils;
   *deps = depspl + depsls;

   return;
}

