;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;oirArrayGeometry
;#description#
;deal with Array Geometry Tables
;#end_class#
;*******************************************************


PRO arrayGeometry__DEFINE
;********************************************************
;#structure#
; arrayGeometry
;#inheritance#
; fitsTable
;#description#
; Internal variables for ArrayGeometry Object
;#structure_text#
oir7={arrayGeometry, $; #class members#
INHERITS FITSTABLE, $;  #class inheritance#
revision:1,    $ int & revision number
arrNam:STRING(REPLICATE(32B,16)), $ ;str & telescope name e.g. 'VLTI'
frame:STRING(REPLICATE(32B,16)), $ ; str &coordinate frame e.g. 'GEOCENTRIC'
arrayX:0.0D, $ ;  dbl &array center x-coordinate in meters
arrayY:0.0D, $ ;  dbl &array center x-coordinate in meters
arrayZ:0.0D  $ ;  dbl &array center x-coordinate in meters
}
;#end_structure#
;********************************************************

;********************************************************
;#structure#
; arrayGeometry
;#inheritance#
; NONE
;#description#
;variables per row of ArrayGeometry Table
;#structure_text#
oir8={arrayGeometryStruct, $ ; #FITS TABLE structure#
tel_Name:STRING(REPLICATE(32B,16)), $ ; station (telescope) name
sta_Name:STRING(REPLICATE(32B,16)), $ ; station (telescope) name
index:1,   $      ; station number (can be used as index)
diameter:0.0,$    ; element diameter (meters)
staXYZ:dblarr(3),$; station coordinates relative to center (meters)
mntSta:0,    $    ; station mount type
mntParam:0.0D $   ; as yet unspecified mount geometry parameters
}
;#end_structure#
;********************************************************
END

FUNCTION arrayGeometry::init, input, iErr=iErr, arrNam=arrNam, $
   frame=frame, arrayX=arrayX, arrayY=arrayY, arrayZ=arrayZ
;construct a array geometry table object
;
;   input  poly
;   (1) if a number assume that this is number of stations
;       and create a default, mostly empty source table.
;
;   (2) a string, assumed to be a file name.  Open the
;       file and get the source table there
;   (3) a FITSFILE object;  read down the file and
;       get source table
;   (4) a (subclass of) a FITSEXTENSION object
;       grab the fitsfile object and treat as (3)
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
;determine input type
   if(N_PARAMS() LE 0) then $
      midiSetError, 'arrayGeometry constructor needs input'
   sInput = size(input)
   sInput = sInput[sInput[0]+1]
;integer, thus input is the size of the ImageDet table
   if (sInput ge 1 and (sInput le 3)) then begin
      nStation = input
      template = {ArrayGeometryStruct}
      self.table = PTR_NEW(replicate(template, nStation))
      if (0 eq self->FITSTABLE::init(template, $
         extname='ARRAY_GEOMETRY', iErr=iErr)) then $
         midiSetError,'Array Geometry Table Creation from Disk Failed',$
         /notInitial
      if (KEYWORD_SET(arrNam)) then begin
         self.arrNam = arrNam
         (self.head)->addPar,'ARRNAM', arrNam 
      endif
      if (KEYWORD_SET(frame)) then begin
         self.frame = frame
         (self.head)->addPar,'FRAME', frame 
      endif
      if (KEYWORD_SET(arrayx)) then begin
         self.arrayx = arrayx
         (self.head)->addPar,'ARRAYX', arrayx 
      endif
      if (KEYWORD_SET(arrayy)) then begin
         self.arrayy = arrayy
         (self.head)->addPar,'ARRAYY', arrayy 
      endif
      if (KEYWORD_SET(arrayz)) then begin
         self.arrayz = arrayz
         (self.head)->addPar,'ARRAYZ', arrayz 
      endif
;end of integer input, try string or object input
   endif else if (sInput eq 7 or (sInput eq 11)) then begin 
;if string input save in temporary variable
      if (sInput eq 7) then tInput = input else begin
;if object input, test class and extract inFile if necessary
         inClass = OBJ_CLASS(input)
	 inClass = strTrim(inClass[0],2)
	 if (inClass EQ 'FITSFILE') then tInput = input else begin
;find highest order superclass
            sc = inClass
            while (sc NE '') do begin
               superClass = sc
               sc = OBJ_CLASS(sc,/super)
               sc = sc[0]
            endwhile
            superClass = superClass[0]
	    if (inClass EQ 'FITSEXTENSION' OR (superClass EQ 'FITSEXTENSION'))$
	       then tInput = input->file() else $
	       midiSetError, 'Source input object of unknown type'
         endelse        ; not a FITSFILE object
      endelse           ; not a string input
;call FITSTABLE constructor with string or FITSFILE input
      if (0 eq self->FITSTABLE::init(tInput, extName='ARRAY_GEOMETRY',$
         /table, iErr=iErr))  then $
         midiSetError, 'Source Table Creation from Disk Failed',/notInitial
      self.arrNam = (self.head)->getPar('ARRNAM', matches)
      self.frame  = (self.head)->getPar('FRAME', matches)
      self.arrayX = (self.head)->getPar('ARRAYX', matches)
      self.arrayY = (self.head)->getPar('ARRAYY', matches)
      self.arrayZ = (self.head)->getPar('ARRAYZ', matches)
   endif  else $           ; input from disk
      midiSetError, 'Source construction input of unknown type'
RETURN,1
END

PRO ArrayGeometry::CLEANUP
PTR_FREE,self.table
RETURN
END
