;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
PRO dataFilter__DEFINE
;define structure for dataFilter Class
oir12={dataFilter, $  ; #class members#
TransferSize:1024l*1024l,       $ long & buffer size
dataCol:PTR_NEW(),       $      ; ptr->int array&list of column numbers of data structs 
tZero:PTR_NEW(),  $             ; ptr->dbl array&store input zero offsets
tScale:PTR_NEW(), $             ; ptr->dbl array&store input zero offsets
inData:OBJ_NEW(),  $            ; object &input data object
outData:OBJ_NEW(), $            ; object &output data object
inputDetector:OBJ_NEW(),$       ; object &image detector object for input data
outputDetector:OBJ_NEW(),$      ; object &image detector object for output data
outputTemplate:PTR_NEW(), $     ; ptr->struct&what output data looks like
dataType:4,        $            ; int &output data type
autofloat:0        $            ; do automatic conversion of data->float
}
END

FUNCTION dataFilter::init, buffSize=buffSize, $
   float=float, byte=byte, long=long, double=double, int=int, $
   complex=complex, autofloat=autofloat
;
; A virtual class with a few default methods
;
; The idea of the class is a prototype of
;   classes that take an input ImageData set
;   with appropriate ImageDetector object
;   and produce an output ImageData set
;
; INPUTS:
;   buffSize   long   Max size of internal transfers
;                     Default = 1 Mb, if not specified
;   /float ...        Data type of output data
;                     Default = type of input data, if not specified
;   autofloat  bool   if set and output type is float, do conversion
;                     on the fly on input i.e. use readrowsf instead
;                     of readrows in FilterData method
;
   if (KEYWORD_SET(buffSize)) then self.transferSize = buffSize $
                              else self.transferSize = 1024L * 1024L
   if (KEYWORD_SET(float))          then self.dataType = 4 $
     else if (KEYWORD_SET(double))  then self.dataType = 5 $
     else if (KEYWORD_SET(complex)) then self.dataType = 6 $
     else if (KEYWORD_SET(long))    then self.dataType = 3 $
     else if (KEYWORD_SET(int))     then self.dataType = 2 $
     else if (KEYWORD_SET(byte))    then self.dataType = 1 
   if (KEYWORD_SET(autofloat))      then self.autoFloat = 1
RETURN,1
END

PRO dataFilter::cleanup
;
; Release internal storage
;
  PTR_FREE, self.dataCol
  PTR_FREE, self.tZero
  PTR_FREE, self.tScale
  PTR_FREE, self.outputTemplate
END

FUNCTION dataFilter::detector, detector, iErr=iErr
;
; Given the ImageDetector object from the input data set,
;   generate the output detector object.
;
; This version does a literal copy, and should be replaced
;   in any subclass by a real detector generator.
;
; This is the appropriate place to add new keywords to the
;   output detector header.
;
; INPUTS
;   detector : the detector object from the input data
; OUTPUTS
;   a new imageDetector object, appropriate for output
;
RETURN, detector->clone(iErr=iErr)
END

PRO dataFilter::fixDataHeader, Head
;
; Virtual function to add, if necessary, new records to the
;   ImageData header of the output dataset.
;
; This virtual version is a no-op
;
; INPUTS
;   Head : Header object of output dataset
;
RETURN
END

FUNCTION dataFilter::filterRows, inputData, iErr=iErr
;
; Virtual function to process a series of input ImageData Table
;   rows into output ImageData Table rows
;
; This virtual version is a straight copy
;
; INPUTS
;   inputData : a number of rows from an ImageData table
; OUTPUTS
;             : a number of processed rows
;
   iErr = 0
RETURN,inputData
END

PRO dataFilter::filterData, inData, outData, iErr=iErr
;
; A real, non-virtual procedure.
;
; Loop over an input ImageData object, processing blocks of
;   data into output dataset
;
; INPUTS
;   inData  : an ImageData object
;   outData : an ImageData object
;
;---------------;
; Error handler ;
;---------------;
  cErr = 0
  catch, cErr
  if (cErr NE 0) then begin ; suppress further handling at this level
    catch, /cancel
    midiCatchError
    iErr = 1
    RETURN
  endif ; handle actual error
;----------------------;
; End of Error handler ;
;----------------------;
;
; Check input types
;
   iErr = 0
   inType  = size(inData,/type)
   outType = size(outData,/type)
   inputOK = (inType EQ 11) AND (outType EQ 11)  ; both must be objects
   if (NOT inputOK) then midiSetError, $
     'input and/or output data set not an object'

   if (inputOK) then inputOK = (OBJ_CLASS(inData)  EQ 'IMAGEDATA')  AND $
                               (OBJ_CLASS(outData) EQ 'IMAGEDATA')
   if (NOT inputOK) then midiSetError, $
     'input and/or output data set not an ImageData object'
;
; >>> Should we make sure than tZero and tScale are right??? <<<
;
;
; How much data should we read in one shot
;
   dataSize = inData->nAxis()
;rows left to read; start with total
   nRows = dataSize[1]
   cRows = 1
;rows per buffer
   rRows = self.transferSize/dataSize[0]
;autoconversion to float?
   auto = (self.autoFloat NE 0) AND (self.dataType EQ 4)
;read loop
   while (nRows GT 0) do begin
     Rows = rRows < nRows
     if (auto) then inRows = inData->readRowsf(cRows+indgen(rows), iErr=iErr)$
        else inRows = inData->readRows(cRows+indgen(rows), iErr=iErr)
     if (iErr NE 0) then midiSetError,'',/notInitial
     outData->writeRows, self->filterRows(inRows), iErr=iErr
     if (iErr NE 0) then midiSetError,'',/notInitial
     nRows = nRows - Rows
     cRows = cRows + Rows
   endwhile
RETURN
END



FUNCTION dataFilter::filter, inData, outData, extNumber=extNumber, $
   close=close, iErr=iErr
;
; Process an entire data set.
; This procedure is also non-virtual
;
; INPUTS
;   inData  poly : Specifies input data set.  This can be:
;                    (1) a string, assumed to be a file name
;                    (2) a FITSFILE object or FITSEXTENSION object
;                        from which an (open) file can be extracted
;                    (3) an ImageData object
;   outData poly : Specifies output data set.  This can be
;                    (1) a string, assumed to be a file name
;                    (2) a FITSFILE or FITSEXTENSION object pointing
;                        to an open file
;   extNumber    : If there are multiple ImageData and ImageDetector
;                    tables on input file, this specifies which
;                    one is desired.  Default = 1
;   close        : on finish, close the output object
; RETURNS
;   the output ImageData object
;
;---------------;
; Error handler ;
;---------------;
  cErr = 0
  catch, cErr
  if (cErr NE 0) then begin  ; suppress further handling at this level
    catch, /cancel
    midiCatchError
    iErr = 1
    RETURN, OBJ_NEW()
  endif ; handle actual error
;----------------------;
; End of Error handler ;
;----------------------;
  inType = size(InData,/type)
  input = inData                ; <<< why this copy? <<<
  inputOK = (inType EQ 7)                           ; string
  if ((NOT inputOK) AND (inType EQ 11)) then begin  ; object
    inClass = topClass(input)
    inputOK = (inClass EQ 'FITSFILE')
    if (NOT inputOK) then begin
      if (inClass EQ 'FITSEXTENSION') then begin
        input = inData->file(iErr=iErr)
        inputOK = (iErr EQ 0)
      endif
    endif
  endif
  if (NOT inputOK) then midiSetError, 'couldn''t open input data'
;
; Work on the output dataset
;
  outType = size(outData,/type)
  outputOK = (outType EQ 7)                            ; string
  if ((NOT outputOK) and (outType EQ 11)) then begin   ; object
    outClass = topClass(outData)
    outputOK = (outClass EQ 'FITSFILE') OR (outClass EQ 'FITSEXTENSION')
  endif
  if (NOT outputOK) then midiSetError, $
    'couldn''t deal with output data set'
;
; Open input data set and get old detector
; >>> What about extnumber???  <<<
;
;
; >>> modification OK? <<<
;
  if (InType EQ 11) then inDataSet = indata $
  else begin
    inDataSet = obj_new('ImageData', input, iErr=iErr)
    if (iErr NE 0) then midiSetError,'',/notInitial
  endelse
;
; Store input data set, it might be useful
;
   self.inData = inDataSet
   self.inputDetector = inDataSet->detector()
   if (NOT OBJ_VALID(self.inputDetector)) then midiSetError, $
     'couldn''t open input detector',/notInitial
;
; Get tScale and tZero and data Cols for input data set
;
   self.dataCol = PTR_NEW(inDataSet->dataColumns())
   self.tZero = PTR_NEW(inDataSet->tZero())
   self.tScale = PTR_NEW(inDataSet->tScale())
;
; Get new detector
;
   outputDetector = self->detector(self.inputDetector, iErr=iErr)
   if (iErr NE 0) then midiSetError, $
     'couldn''t generate output detector object',/notInitial
   self.outputDetector = outputDetector
;
; Put outputDetector into/onto file
;
   if (outType EQ 7) then begin  ; string, so new file
     self.outputDetector->newFile, outData, /table, iErr=iErr
     if (iErr NE 0) then midiSetError, $
       'error opening output data file',/notInitial
   endif else begin   ; existing file
     self.outputDetector->appendToFile, outData, /table, iErr=iErr
     if (iErr NE 0) then midiSetError, $
       'error appending detector to output data file',/notInitial
   endelse
;
; Check if output data type is unknown; then use input type
;
   if (self.dataType LE 0) then begin
     temp = inDataSet->template()
     self.dataType = size(temp.DATA1,/type)
   endif
;
; Create output data set
;
   CASE self.dataType OF
     4: outDataSet = obj_new('imageData', self.outputDetector, $
        /float,  iErr=iErr)
     5: outDataSet = obj_new('imageData', self.outputDetector, $
        /double,  iErr=iErr)
     6: outDataSet = obj_new('imageData', self.outputDetector, $
        /complex,  iErr=iErr)
     3: outDataSet = obj_new('imageData', self.outputDetector, $
        /long,  iErr=iErr)
     2: outDataSet = obj_new('imageData', self.outputDetector, $
        /int,  iErr=iErr)
     1: outDataSet = obj_new('imageData', self.outputDetector, $
        /byte,  iErr=iErr)
   ENDCASE
   if (iErr NE 0) then midiSetError, $
      'error creating output data object',/notInitial
;
; Store output data set, it might be useful
;
   self.outData = outDataSet
;
; Fixup output header if necessary
;
   self->fixDataHeader, outDataSet->head()
;
; Add output object to output file
;
   outDataSet->appendToFile, self.outputDetector, iErr=iErr
   if (iErr NE 0) then midiSetError, $
      'error appending output data object to output file',/notInitial
;
; Do some real work for a change, the actual data filtering
;
   self->filterData, inDataSet, outDataSet, iErr=iErr
   if (iErr NE 0) then midiSetError, $
      'data filtering failed',/notInitial

   if (KEYWORD_SET(close)) then outDataSet->close

RETURN, outDataSet
END

