;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;oirUtilities
;#description#
;some utilities that dont belong in specific "classes" within OIR
;#end_class#
;*******************************************************

FUNCTION oirFileList, filename
nFile = N_ELEMENTS(filename)
outFiles = ''
for iFile = 0, nFile-1 do outFiles=[outFiles, STRSPLIT(filename[iFile],' ',/EXTRACT)]
RETURN, outFiles[1:*]
END

FUNCTION oirGetData, filename, rows, col=col, ierr=ierr
;********************************************************
;#function#
;getData
;#description#
;combine several oir calls to get data from disk
;#call#
;mydata = oirGetData(filename, rows, col=col, ierr=ierr )
;#inputs#
;\anArg{filename}{string }{filename containing oir image data}
;\anArg{}{}{may also be an array of (to be concatenated) filenames}
;\anArg{rows}{int array }{optional list of row numbers; if omitted all rows}
;\anArg{}{}{ignored if multiple input files}
;\anArg{col=col}{int array or string arr}{optional list of column numbers or names}
;#return#
;\anArg{-}{array of structures}{image data for specified rows and columns}
;#end_function#
;*******************************************************
 ; check if single input or multiple
   fileList = oirFileList(fileName)
   fileSize = SIZE(fileList)
   if (fileSize[0] eq 0) then nFiles = 1 $
      else nFiles=N_ELEMENTS(fileList)
   fileRows = lonarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', fileList[iFile], ierr=ierr)
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      naxis = imageObj->naxis()
      fileRows[iFile] = naxis[1]
      OBJ_DESTROY, imageObj
   endfor
   isRow = N_PARAMS() GT 1
   if (isRow) then begin
      if (nFiles gt 1) then begin
         print,'oirGetData cant handle rows with multiple files'
         return,0
      endif else fileRows[0] = N_ELEMENTS(rows)
   endif
   isCol = N_ELEMENTS(col) GT 0
; get typical row
   imageObj = OBJ_NEW('imagedata', fileList[0], ierr=ierr)
   if (ierr NE 0) then begin
      ewsPrintErrMsg
      return,0
   endif
   if (isCol) then typicalRow = imageObj->readRows(1, col=col) $
      else typicalRow = imageObj->readrows(1)
   OBJ_DESTROY, imageObj
; create output array
   data = REPLICATE(typicalRow, TOTAL(fileRows))
   iRows = 0
; read input files
   for iFile = 0, nFiles-1 do begin
      imageObj = obj_new('imagedata', fileList[iFile], ierr=ierr)
      if (ierr NE 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      if (isRow) then begin
         if (isCol) then begin
            dtemp =imageObj->readRows(rows, col=col, ierr=ierr) 
            if (ierr EQ 0) then data[*] = dtemp else begin
               ewsPrintErrMsg
               OBJ_DESTROY, imageObj
               return,0
            endelse
         endif else begin
            dtemp = imageObj->readRows(rows, ierr=ierr)
            if (ierr EQ 0) then data[*] = dtemp else begin
               ewsPrintErrMsg
               OBJ_DESTROY, imageObj
               return,0
            endelse
         endelse
      endif else begin
         if (isCol) then data[iRows:iRows+fileRows[iFile]-1] = $
            imageObj->readRows(col=col, ierr=ierr) $
           else data[iRows:iRows+fileRows[iFile]-1] = $
              imageObj->readRows(ierr=ierr)
      endelse
      iRows = iRows + fileRows[iFile]
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      OBJ_DESTROY, imageObj
   endfor
RETURN,data
END

FUNCTION oirGetDataF, filename, rows, col=col, ierr=ierr
;********************************************************
;#function#
;getData
;#description#
;combine several oir calls to get data from disk in float format
;#call#
;mydata = oirGetData(filename, rows, col=col, ierr=ierr )
;#inputs#
;\anArg{filename}{string }{filename containing oir image data}
;\anArg{}{}{may also be an array of (to be concatenated) filenames}
;\anArg{rows}{int array }{optional list of row numbers; if omitted all rows}
;\anArg{}{}{ignored if multiple input files}
;\anArg{col=col}{int array or string arr}{optional list of column numbers or names}
;#return#
;\anArg{-}{array of structures}{image data for specified rows and columns}
;#end_function#
;*******************************************************
 ; check if single input or multiple
   fileList = oirFileList(fileName)
   fileSize = SIZE(fileList)
   if (fileSize[0] eq 0) then nFiles = 1 $
      else nFiles=N_ELEMENTS(fileList)
   fileRows = lonarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', fileList[iFile], ierr=ierr)
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      naxis = imageObj->naxis()
      fileRows[iFile] = naxis[1]
      OBJ_DESTROY, imageObj
   endfor
   isRow = N_PARAMS() GT 1
   if (isRow) then begin
      totalRows = INTARR(nFiles)
      lastRow = INTARR(nFiles)
      totalRows[0] = fileRows[0]
      for iFile=1,nFiles-1 do totalRows[iFile]=totalRows[iFile-1]+fileRows[iFile]
      sRow = SORT(rows)
      sortRows = rows(sRow)
      for iFile = 0,nFiles-1 do lastRow[iFile] = MAX(WHERE(sortRows LE totalRows[iFile]))
      lr1 = 0
   endif ; isRow

   isCol = N_ELEMENTS(col) GT 0
; get typical row
   imageObj = OBJ_NEW('imagedata', fileList[0], ierr=ierr)
   if (isCol) then typicalRow = imageObj->readRowsf(1, col=col) $
      else typicalRow = imageObj->readrowsf(1)
   OBJ_DESTROY, imageObj
; create output array
   if (isRow) then data = REPLICATE(typicalRow, N_ELEMENTS(rows)) $
      else data = REPLICATE(typicalRow, TOTAL(fileRows))
   iRows = 0
; read input files
   for iFile = 0, nFiles-1 do begin
      imageObj = obj_new('imagedata', fileList[iFile], ierr=ierr)
      if (isRow) then begin
            lr2 = lastRow[iFile]
         if (lr2 gt lr1) then begin
            if (iFile EQ 0) then cRows = sortRows[lr1:lr2] else cRows = sortRows[lr1:lr2]-totalRows[iFile-1]
            if (isCol) then data[sRow[lr1:lr2]]=imageObj->readRowsf(cRows, col=col, ierr=ierr) $
               else data[sRow[lr1:lr2]] = imageObj->readRowsf(cRows, ierr=ierr)
         endif
        lr1 = lr2 + 1
      endif else begin
         if (isCol) then data[iRows:iRows+fileRows[iFile]-1] = $
            imageObj->readRowsF(col=col, ierr=ierr) $
           else data[iRows:iRows+fileRows[iFile]-1] = $
              imageObj->readRowsF(ierr=ierr)
      endelse
      iRows = iRows + fileRows[iFile]
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      OBJ_DESTROY, imageObj
   endfor
RETURN,data
END

FUNCTION oirGetVis, filename, version, wave=wave, ierr=ierr
;********************************************************
;#function#
;getData
;#description#
;combine several oir calls to get visibility data from disk
;#call#
;mydata = oirGetVis(filename, version, wave=wave, ierr=ierr )
;#inputs#
;\anArg{filename}{string }{filename containing oir image data}
;\anArg{version}{int }{optional if multiple vis tables in file, if omitted=1}
;#return#
;\anArg{-}{array of structures}{vis data from table}
;#end_function#
;*******************************************************
   if (N_PARAMS() EQ 1) then version = 1
   waveObj = OBJ_NEW('oi_wavelength', filename, ierr=ierr)
   if (ierr NE 0) then begin
      print,'couldnt open wave object for file ',filename
      RETURN,0
   endif
   wave = waveObj->readrows()
   wave = 1.e6*REFORM(wave.eff_wave) ; microns
   waveObj->close
   OBJ_DESTROY, waveObj
   visObj = OBJ_NEW('oi_vis', filename, version = version, ierr=ierr)
   if (ierr NE 0) then begin
      print,'couldnt open oi_vis table object for file ',filename
      RETURN,0
   endif
   data = visObj->readRows()
   visObj->close
   OBJ_DESTROY, visObj
RETURN,data
END

FUNCTION oirGetVisAmp, filename, wave=wave, ierr=ierr
   vis = oirGetVis(filename, wave=wave, ierr=ierr)
   if (ierr EQ 0) then RETURN, vis.visamp
RETURN,0
END

FUNCTION oirGetVisPhi, filename, wave=wave, ierr=ierr
   vis = oirGetVis(filename, wave=wave, ierr=ierr)
   if (ierr EQ 0) then RETURN, vis.visphi
RETURN,0
END

FUNCTION oirGetVisReal, filename, wave=wave, ierr=ierr
   vis = oirGetVis(filename, wave=wave, ierr=ierr)
   if (ierr EQ 0) then RETURN, vis.visamp*COS(vis.visphi/!radeg)
RETURN,0
END
FUNCTION oirGetVisComplex, filename, wave=wave, ierr=ierr
   vis = oirGetVis(filename, wave=wave, ierr=ierr)
   if (ierr EQ 0) then RETURN, vis.visamp* $
         EXP(COMPLEX(0,1.)*vis.visphi/!radeg)
RETURN,0
END

FUNCTION oirGetDelay, filename
   delayObj = OBJ_NEW('fitstable', filename, extname='DELAY', ierr=ierr)
   if (ierr NE 0) then begin
      print,'couldnt open delay object for file ',filename
      RETURN,0
   endif
   delay = delayObj->readrows()
   delay.delay = 2.9979258e14*delay.delay ; microns not seconds
   OBJ_DESTROY, delayObj
RETURN,delay
END

FUNCTION oirGetOpd, filename, time

 ; check if single input or multiple
   fileList = oirFileList(fileName)
   fileSize = SIZE(fileList)
   if (fileSize[0] eq 0) then nFiles = 1 $
      else nFiles=N_ELEMENTS(fileList)
   fileRows = lonarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', fileList[iFile], ierr=ierr)
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      naxis = imageObj->naxis()
      fileRows[iFile] = naxis[1]
      OBJ_DESTROY, imageObj
   endfor
   totalRows = TOTAL(fileRows)
   if (totalRows LE 1) then begin
      print,'only found ',totalRows,' output rows'
      RETURN,0
   endif
; create output array
   opd  = fltarr(totalRows)
   time = fltarr(totalRows)
   iRows = 0L
; read input files
   for iFile = 0, nFiles-1 do begin
      imageObj = obj_new('imagedata', fileList[iFile], ierr=ierr)
      for i=1L, fileRows[iFile] do begin
         d = imageObj->readrows(i,col=['time','localopd','opd'])
         opd[iRows] = 1.e6*(d.opd[1]+d.localopd[1]-d.opd[0]-d.localopd[0])
         if (i eq 1) then startTime = MIN(d.time)
         time[iRows] = 86400.*(d.time - startTime)
         iRows = 1+iRows
      endfor
      OBJ_DESTROY, imageObj
   endfor
RETURN,opd
END

FUNCTION oirGetFlag, fitsfile, ierr = ierr
;
   flag = OBJ_NEW('fitstable',fitsfile,extname='FLAG', ierr = ierr)
   if (ierr NE 0) then begin
      midiprinterrmsg
      RETURN,0
   endif
   flagdata = flag->readrows()
   OBJ_DESTROY,flag
RETURN, flagdata
END

FUNCTION oirFlagTimes, fitsfile, ierr = ierr
;  get flag times, and concatenate overlapping times
   flagData = oirGetFlag(fitsfile, ierr=ierr)
   if (ierr NE 0) then begin
      midiprinterrmsg
      RETURN,0
   endif
   nFlag = N_ELEMENTS(flagData)
   tSort = SORT(REFORM(flagData.timerang[0]))
   flagTimes = flagData[tSort].timerang
   i=0
   j=1
   while (j LT nFlag) do begin
      while (flagTimes[0,j] LE flagTimes[1,i]) do begin
         flagTimes[1,i] = (flagTimes[1,i] > flagTimes[1,j]) 
         j = j+1 
      endwhile
      i = i+1 
      flagTimes[*,i] = flagTimes[*,j]
      j = j+1
   endwhile
   flagTimes = flagTimes[*,0:i-1]
RETURN,flagTimes
END

FUNCTION oirExpandFlag, flagTable, dataTable
;
   nData = N_ELEMENTS(dataTable)
   nFlag = N_ELEMENTS(flagTable)
   flagTime = flagTable.timerang
   eflag = BYTARR(nData)
   for i= 0, nData-1 do begin
      t = dataTable[i].time
      for j=0,nFlag-1 do if ((flagTime[0,j] LE t) AND (flagTime[1,j] GE t)) $
         then eFlag[i] = 1
   endfor
RETURN, eFlag
END

FUNCTION oirGetMeanRMS, inputFiles, flagFile=flagFile
;********************************************************
;#function#
;getMeanRms
;#description#
;create mean and rms images for all data regions in file
;#call#
;meanRms = oirGetMeanRms(inputFiles, flagfile=flagfile, ierr=ierr)
;#inputs#
;\anArg{inputFiles}{string }{filename or array of files containing oir image data}
;\anArg{flagfile}{string }{filename specifying oirfits flag file}
;#return#
;\anArg{-}{array of structures}{1st row contains mean images (float);2nd rms}
;#end_function#
;*******************************************************
 ; check if single input or multiple
   filename = oirFileList(inputFiles)
   fileSize = SIZE(filename)
   if (fileSize[0] eq 0) then nFiles = 1 $
      else nFiles=N_ELEMENTS(filename)
   fileRows = intarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', filename[iFile], ierr=ierr)
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      naxis = imageObj->naxis()
      fileRows[iFile] = naxis[1]
      OBJ_DESTROY, imageObj
   endfor
; establish output structure
   imageObj = obj_new('imagedata', filename[0], ierr=ierr)
   dataColumns = imageObj->dataColumns()
   data = imageObj->readRows(1)
   timeCol = WHERE(STRTRIM(TAG_NAMES(data),2) EQ 'TIME')
   readCols = [1+timeCol, dataColumns]
   data = imageObj->readRowsf(1, col=dataColumns, ierr=ierr)
   meanArray = REPLICATE(data,2)
   OBJ_DESTROY,imageObj
   tags = TAG_NAMES(data)
   nTags = N_ELEMENTS(tags)
   for i=0,nTags-1 do meanArray.(i) = 0.
;  if required, get flag data
   doFlag = 0
   if (KEYWORD_SET(flagFile)) then begin
      doFlag    = 1
      flagTimes = oirFlagTimes(flagFile, ierr=ierr)
      iFlag     = 0
      nFlag = N_ELEMENTS(flagTimes[0,*])
      if (ierr NE 0) then begin
         midiprinterrmsg
         RETURN,0
      endif 
   endif
;compute mean
   nRows = 0
   for iFile = 0, nFiles-1 do begin
      imageObj = obj_new('imagedata', filename[iFile], ierr=ierr)
      for iRow = 1, fileRows[iFile] do begin
         noFlag = 1
         data = imageObj->readRowsf(iRow, col=readCols, ierr=ierr)
         if (doFlag) then begin
            time = data[0].time
            while(time GT flagTimes[1,iFlag] AND (iFlag LT (nFlag-1)))$
               do iFlag=iFlag+1
            if (time GE flagTimes[0,iFlag]) then noFlag = 0
            if (iRow EQ 1 AND (iFile EQ 0)) then noFlag = 0; always zap first point
         endif
         if (noFlag) then begin
            for iTag = 1, nTags do $
               meanArray[0].(iTag-1)=meanArray[0].(iTag-1) + data.(iTag)
            nRows = nRows + 1
         endif
      endfor
      OBJ_DESTROY,imageObj
   endfor
   if (nRows GT 0) then for iTag = 0,nTags-1 do $
      meanArray[0].(iTag)=meanArray[0].(iTag)/nRows
;compute mean^2
   for iFile    = 0, nFiles-1 do begin
      imageObj  = obj_new('imagedata', filename[iFile], ierr=ierr)
      for iRow  = 1, fileRows[iFile] do begin
         noFlag = 1
         iFlag  = 0
         data = imageObj->readRowsf(iRow, col=readCols, ierr=ierr)
         if (doFlag) then begin
            time = data[0].time
            while(time GT flagTimes[1,iFlag] AND (iFlag LT (nFlag-1))) $
               do iFlag=iFlag+1
            if (time GE flagTimes[0,iFlag]) then noFlag = 0
            if (iRow EQ 1 AND (iFile EQ 0)) then noFlag = 0
         endif
         if (noFlag) then $
            for iTag = 0, nTags-1 do $
               meanArray[1].(iTag)=meanArray[1].(iTag) +  $
                  (data.(iTag) - meanArray[0].(iTag))^2
      endfor
      OBJ_DESTROY,imageObj
   endfor
   if (nRows GT 0) then for iTag = 0,nTags-1 do $
      meanArray[1].(iTag) = SQRT(meanArray[1].(iTag)/nRows)
RETURN,meanArray
END


FUNCTION oirCorrelateImage, inputFiles, signal, flag
;********************************************************
;#function#
;CorrelateImage
;#description#
;correlate each pixel in detector windows with a given fringe signal
;#call#
;correlatedPixels = oirCorrelateImage(inputFiles, signal, flag )
;#inputs#
;\anArg{inputFiles}{string }{filename or array of files containing oir image data}
;\anArg{signal}{array of float }{Fringe signal (1value per frame to correlate w/ each pixel)}
;\anArg{flag}{array of Bool }{Optional list of rows to include}
;#return#
;\anArg{-}{structure}{at each pixel in DATA1... correlation coefficient w/ signal}
;#end_function#
;*******************************************************
 ; check if single input or multiple
   filename = oirFileList(inputFiles)
   fileSize = SIZE(filename)
   if (fileSize[0] eq 0) then nFiles = 1 $
      else nFiles=N_ELEMENTS(filename)
   fileRows = intarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', filename[iFile], ierr=ierr)
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      naxis = imageObj->naxis()
      fileRows[iFile] = naxis[1]
      OBJ_DESTROY, imageObj
   endfor
; establish output structure
   nSig = N_ELEMENTS(signal)
   if (TOTAL(fileRows) NE nSig) then begin
      print,'SIGNAL SIZE ',nSig,' NE #rows ',TOTAL(fileRows)
      RETURN,0
   endif
   IF (N_PARAMS() LT 3 OR (N_ELEMENTS(flag) LT nSig)) then $
      fg = REPLICATE(1., nSig) else fg = flag

   imageObj = obj_new('imagedata', filename[0], ierr=ierr)
   dataColumns = imageObj->dataColumns()
   data = imageObj->readRowsf(1, col=dataColumns, ierr=ierr)
   meanArray = data
   OBJ_DESTROY,imageObj
   tags = TAG_NAMES(data)
   nTags = N_ELEMENTS(tags)
   for i=0,nTags-1 do meanArray.(i) = 0.
   s = signal*fg - mean(signal*fg)
   weight = 0.
;compute mean
   nRows = 0
   for iFile = 0, nFiles-1 do begin
      imageObj = obj_new('imagedata', filename[iFile], ierr=ierr)
      for iRow = 1, fileRows[iFile] do begin
         data = imageObj->readRowsf(iRow, col=dataColumns, ierr=ierr)
         for iTag = 0, nTags-1 do $
            meanArray[0].(iTag)=meanArray[0].(iTag) + $
               data.(iTag)*s[nRows]
         weight = weight + fg[nRows]
         nRows = nRows + 1
      endfor
      OBJ_DESTROY,imageObj
   endfor
   if (weight GT 0) then for iTag = 0,nTags-1 do $
      meanArray[0].(iTag)=meanArray[0].(iTag)/weight
RETURN,meanArray
END


FUNCTION oirCorrelateSpectra, inputFiles, OPD, flag
;********************************************************
;#function#
;CorrelateSpectra
;#description#
;correlate each pixel in windows with that expected from given OPD
;#call#
;correlatedPixels = oirCorrelateSPectra(inputFiles, Opd, flag)
;#inputs#
;\anArg{inputFiles}{string }{filename or array of files containing oir image data}
;\anArg{signal}{array of float }{Fringe signal (1value per frame to correlate w/ each pixel)}
;\anArg{flag}{array of Bool }{Optional list of rows to include}
;#return#
;\anArg{-}{structure}{at each pixel in DATA1... correlation coefficient w/ signal}
;#end_function#
;*******************************************************
 ; check if single input or multiple
   filename = oirFileList(inputFiles)
   fileSize = SIZE(filename)
   if (fileSize[0] eq 0) then nFiles = 1 $
      else nFiles=N_ELEMENTS(filename)
   fileRows = intarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', filename[iFile], ierr=ierr)
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      naxis = imageObj->naxis()
      fileRows[iFile] = naxis[1]
      OBJ_DESTROY, imageObj
   endfor
   waveNo = oirGetWaveno(filename[0])
; establish output structure
   nOpd = N_ELEMENTS(opd)
   if (TOTAL(fileRows) NE nOpd) then begin
      print,'OPD SIZE ',nOpd,' NE #rows ',TOTAL(fileRows)
      RETURN,0
   endif
   IF (N_PARAMS() LT 3 OR (N_ELEMENTS(flag) LT nOpd)) then $
      fg = REPLICATE(1., nOpd) else fg = flag

   imageObj = obj_new('imagedata', filename[0], ierr=ierr)
   dataColumns = imageObj->dataColumns()
   data = imageObj->readRowsf(1, col=dataColumns, ierr=ierr)
   outArray = data ; output array
   nData = N_ELEMENTS(dataColumns)
   sData = SIZE(data.(0),/dim)
   comCorr = COMPLEXARR(sData[0], sData[1], nData) ;complex correlation
   nWave = N_ELEMENTS(waveNo)
   OBJ_DESTROY,imageObj
   weight = 0.
   iWave = COMPLEXARR(sData[0], sData[1])
   for i=0, sData[1]-1 do iWave(*,i) = -COMPLEX(0.,1.)*waveNo
;compute mean
   meanData = oirGetMeanRms(inputFiles)
   meanData = meanData[0]
   nRows = 0
   for iFile = 0, nFiles-1 do begin
      imageObj = obj_new('imagedata', filename[iFile], ierr=ierr)
      for iRow = 1, fileRows[iFile] do begin
         data = imageObj->readRowsf(iRow, col=dataColumns, ierr=ierr)
         cData = EXP(iWave * opd[nRows])
         for iData = 0, nData-1 do $
            comCorr[*,*,iData] = comCorr[*,*,iData] + $
               (data.(iData)-meanData.(iData))*fg[nRows]*cData
         weight = weight + fg[nRows]
         nRows = nRows + 1
      endfor
      OBJ_DESTROY,imageObj
   endfor
   if (weight GT 0) then for iData = 0,nData-1 do $
      outArray[0].(iData)=ABS(comCorr(*,*,iData))/weight
RETURN,outArray
END

FUNCTION oirCompressData, inputFiles, maskFile, nDim = nDim, opd = opd, $
   time=time, flipOpd=flipOpd, iErr=iErr
;********************************************************
;#function#
;oirCompressData
;#description#
;reduce dimensionality of data by summing in one or two dimensions
;#call#
;cData = oirCompressData(inputFiles)
;cData = oirCompressData(inputFiles, maskFile)
;cData = oirCompressData(inputFiles, maskFile, nDim = 1, opd=opd, time=time)
;#inputs#
;\anArg{inputFiles}{string }{filename(s) containing oir image data}
;\anArg{maskFile}{string }{optional file containing weighting array)
;\anArg{nDim}{int }{compress to 0-dimensional data (default) or 1-d}
;\anArg{opd}{array of flt }{the opd settings corresponding to the image data}
;\anArg{time}{array of flt }{the time settings corresponding to the image data}
;#return#
;\anArg{-}{array of structures}{compressed image data }
;#end_function#
;*******************************************************
 ; check if single input or multiple
   filename = oirFileList(inputFiles)
   fileSize = SIZE(filename)
   if (fileSize[0] eq 0) then nFiles = 1 $
      else nFiles=N_ELEMENTS(filename)
   fileRows = intarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', filename[iFile], ierr=ierr)
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      naxis = imageObj->naxis()
      fileRows[iFile] = naxis[1]
      OBJ_DESTROY, imageObj
   endfor
;create output structure
   imageObj = OBJ_NEW('imagedata', filename[0], ierr=ierr)
   if (N_ELEMENTS(nDim) NE 1) then nDim = 0 $
      else if (nDim NE 1) then nDim = 0
   isMask = (N_PARAMS() GT 1)
   if (isMask) then maskData = oirGetData(maskFile, iErr=iErr) $
   else begin
      maskData = inData
      for iTag = 1, nTags-1 do $ 
         if (STRMID(tags[iTag], 0, 4) EQ 'DATA') then $
            maskData.(iTag) = 1.0
   endelse
   if (iErr NE 0) then RETURN,0
   nMask = N_ELEMENTS(maskData)
   iErr = 0
   inData = imageObj->readRowsf(1)
   OBJ_DESTROY, imageObj
   tags = TAG_NAMES(inData)
   nTags = N_ELEMENTS(tags)
   outData = CREATE_STRUCT(tags[0], inData.(0))
   for iTag = 1, nTags-1 do begin
      if (STRMID(tags[iTag], 0, 4) NE 'DATA') then $
         outData = CREATE_STRUCT(outData, tags[iTag], inData.(iTag))$
      else begin
         if (nDim EQ 0) then $
            if (nMask EQ 1) then $
               outData = CREATE_STRUCT(outData, tags[iTag], 0.)$
               else outData = CREATE_STRUCT(outData, tags[iTag], fltarr(nMask))$
         else begin 
            S = SIZE(inData[0].(iTag))
            if (nMask EQ 1) then $
               outData = CREATE_STRUCT(outData, tags[iTag], fltarr(S[1])) $
            else outData = CREATE_STRUCT(outData, tags[iTag], fltarr(S[1],nMask))
         endelse
      endelse
   endfor
   cData = REPLICATE(outData, TOTAL(fileRows))
   time  = dblarr(TOTAL(fileRows))
   opd   = fltarr(TOTAL(fileRows))
   row = 0
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', filename[iFile], ierr=ierr)
      for iRow = 1, fileRows[iFile] do begin
         inData = imageObj->readRowsf(iRow)
         for iTag = 0, nTags-1 do begin
            if (STRMID(tags[iTag], 0, 4) NE 'DATA') then $
               cData[row].(iTag) = inData.(iTag) $
            else if (nDim EQ 0) then for i=0,nMask-1 do $
               cData[row].(iTag)[i] =  $
                  TOTAL(inData.(iTag) * maskData[i].(iTag)) $
               else for i=0,nMask-1 do cData[row].(iTag)[*,i] =  $
                  TOTAL(inData.(iTag)*maskData[i].(iTag),2)
         endfor
         time[row] = inData.time
         if (KEYWORD_SET(flipOpd)) then $
            opd[row] = inData.localOpd[1]-inData.localOpd[0] - $
               inData.opd[1] + inData.opd[0] $
         else opd[row] = inData.localOpd[1]-inData.localOpd[0] + $
               inData.opd[1] - inData.opd[0] 
         row = row + 1
      endfor
      OBJ_DESTROY,imageObj
   endfor
   time = 86400.*(time - time[0])
   opd = 1.e6*opd
RETURN, cData
END

FUNCTION oirCompressDataC, inputFiles, maskFile, outFile,$
   nDim = nDim, opd = opd, time=time, flipOpd=flipOpd, iErr=iErr
;
   iErr = 0
   if (N_ELEMENTS(nDim) NE 1) then nDim = 0 $
      else if (nDim NE 1) then nDim = 0
   iFiles = ' "'+inputFiles+'" '
   prog = getenv('vltiCbin')
   if (nDim EQ 0) then prog = prog+'/oirCompress ' $
      else prog=prog+'/oirCompressSpec '
   if (N_PARAMS() LT 3) then begin
      tmpDir = GETENV('midiTmpDir')
      if (tmpDir EQ '') then tmpDir = '/tmp'
      outFile = tmpDir+'/oirCompress.tmp'
   endif
   if (FILE_TEST(outFile)) then spawn,'rm '+outFile
   spawn,prog+iFiles+maskFile+' '+outFile
   inData = oirGetData(outFile)
   if (N_PARAMS() LT 3) then spawn,'rm '+outFile
   opdM = inData.localopd[1]-inData.localopd[0]
   opdV = inData.opd[1] - inData.opd[0]
   if (KEYWORD_SET(flipOpd)) then opd = $
      inData.localOpd[1] + inData.opd[1] - inData.localOpd[0]-inData.opd[0]$
      else opd = $
      inData.localOpd[1] - inData.opd[1] - inData.localOpd[0]+inData.opd[0]
   time = 86400.*REFORM(inData.time)
RETURN, inData
END

PRO oirNewData, detectorFile, outputFile, data, copyHead=copyHead, $
   newHead=newHead, iErr = iErr
;********************************************************
;#procedure#
;oirNewData
;#description#
;write new data file after copying detector information from old file
;#call#
;oirNewData detector, outputFileName, dataArray
;#inputs#
;\anArg{detector}{string }{filename image_detector table}
;\anArg{outputFile}{string }{file for output data)
;\anArg{data}{array of struct }{data in format compatable with detector table}
;\anArg{copyHead}{bool}{copy primary header from detector File?}
;\anArg{newHead}{header object}{new primary header; overrides copyHead }
;#end_procedure#
;********************************************************
;useage
   if (N_PARAMS() EQ 0) then begin
      print,' oirNewData detectorTemplateFile outputFile data'
      RETURN
   endif
; get detector
   detector = OBJ_NEW('imagedetector', detectorFile, iErr = iErr)
   if (iErr ne 0) then begin
      ewsPrintErrMsg
      return
   endif
   if (KEYWORD_SET(newHead)) then detHead= newHead->clone() $
   else if (KEYWORD_SET(copyHead)) then $
      detHead = (detector->priHead())->clone()
   detector->close
;  prepare new output file
; figure out what data type is in data 
; and create data template and output data object
   dataType = SIZE(data.DATA1, /type)
   if (dataType EQ 2) then begin
      temp = detector->dataTemplate(/int) 
      output = OBJ_NEW('imagedata', detector, /int)
   endif else if (dataType EQ 3) then begin
      temp = detector->dataTemplate(/long) 
      output = OBJ_NEW('imagedata', detector, /long)
   endif else if (dataType EQ 4) then begin
      temp = detector->dataTemplate(/float) 
      output = OBJ_NEW('imagedata', detector, /float)
   endif else begin
      iErr = 1; illegal data type
      ewsAddErrMsg,' Illegal data type '+STRING(dataType)
      ewsPrintErrMsg
      RETURN
   endelse
; check if template and data really are same format
   tempTags = STRTRIM(TAG_NAMES(temp),2)
   tempData = WHERE(STRPOS(tempTags, 'DATA') GE 0)
   tempTarg = WHERE(STRPOS(tempTags, 'TARTYP') GE 0)
   nData    = N_ELEMENTS(tempData)
   dataTags = STRTRIM(TAG_NAMES(data),2)
   dataData = WHERE(STRPOS(dataTags, 'DATA') GE 0)
   dataTarg = WHERE(STRPOS(dataTags, 'TARTYP') GE 0)
   iErr = 2
   if (nData NE N_ELEMENTS(dataDATA)) then begin
      ewsAddErrMsg,' data structure does not agree with detector definition'
      ewsPrintErrMsg
      OBJ_DESTROY, detector
      OBJ_DESTROY, output
      RETURN
   endif
; check array sizes
   for iData = 0, nData -1 do begin
      if(TOTAL(SIZE(temp.(tempData[iData]),/dimensions) $
         NE SIZE(data.(dataData[iData]),/dimensions)) NE 0) then begin
         ewsAddErrMsg,' data structure does not agree with detector definition'
         ewsPrintErrMsg
         OBJ_DESTROY, detector
         OBJ_DESTROY, output
         RETURN
      endif
   endfor
; try to set MAXSTEP correctly
   stepTag = MIN(WHERE(STRPOS(dataTags,'STEPPING_PHASE') GE 0))
   if (stepTag GE 0) then step = MAX(data.stepping_phase) $
   else step = 40 ; default
   maxins = 2
; append data to output
   if ((KEYWORD_SET(copyHead)) OR (KEYWORD_SET(newHead))) then begin
      detector->newFile, outputFile, priHead=detHead,iErr=iErr 
      OBJ_DESTROY,detHead
   endif else  detector->newFile, outputFile, iErr=iErr 
   if (iErr ne 0) then begin
      ewsPrintErrMsg
      return
   endif
   (output->head())->addPar,'MAXSTEP', step
   (output->head())->addPar,'MAXINS', maxins
   output->appendToFile, detector, iErr=iErr
   if (iErr ne 0) then begin
      ewsPrintErrMsg
      return
   endif
; write data
   for iData = 1, N_ELEMENTS(data) do begin
      for iTemp = 0, nData -1 do begin 
         temp.(tempData[iTemp]) = data[iData-1].(dataData[iTemp])
         if (iTemp lt N_elements(dataTarg)) then $
           if (dataTarg[iTemp] ge 0) then $
             temp.(tempTarg[iTemp]) = data[iData-1].(dataTarg[iTemp])
      endfor
      if (TOTAL(dataTags eq 'TIME') GT 0)     then temp.TIME = data[iData-1].TIME
      if (TOTAL(dataTags eq 'EXPTIME') GT 0)  then temp.EXPTIME = data[iData-1].EXPTIME
      if (TOTAL(dataTags eq 'OPD') GT 0)      then temp.OPD = data[iData-1].OPD
      if (TOTAL(dataTags eq 'LOCALOPD') GT 0) then temp.LOCALOPD = data[iData-1].LOCALOPD
      if (TOTAL(dataTags eq 'STEPPING_PHASE') GT 0) then temp.STEPPING_PHASE = data[iData-1].STEPPING_PHASE
      if (TOTAL(dataTags eq 'FRAME') GT 0)     then temp.FRAME = data[iData-1].FRAME
      output->writeRows, temp
   endfor
   output->close
   OBJ_DESTROY, output
   RETURN
   END

PRO oirMakeMask, detectorFile, outputFile, maskdata, iErr = iErr
; get detector
   detector = OBJ_NEW('imagedetector', detectorFile, iErr = iErr)
   if (iErr ne 0) then begin
      ewsPrintErrMsg
      return
   endif
   detector->close
;  prepare new output file
; and create data template and output data object
   dTemp = detector->datatemplate(/float)
   detector->newFile, outputFile, iErr=iErr
   output = OBJ_NEW('imagedata', detector, /float)
   output->appendtofile,detector
   dataColumns = output->dataColumns() - 1
   maskTags = TAG_NAMES(maskdata)
   dataTags = WHERE(STRPOS(maskTags, 'DATA') GE 0)
   for i=0, N_ELEMENTS(dataColumns)-1 do dTemp.(dataColumns[i]) = maskdata.(dataTags[i])
   output->writerows, dTemp
   output->close
   OBJ_DESTROY, detector
   OBJ_DESTROY, output
RETURN
END

FUNCTION oirDetectorImage, filename, row
;********************************************************
;#function#
;oirDetectorImage
;#description#
;place windowed table data into a 320x240 pseudo-detector array
;#call#
;DetectorDataArray = oirDetectorImage(tableFileName)
;#inputs#
;\anArg{tableFileName}{string }{file containing table format data}
;\anArg{row}{int }{which row in table, defaults to first}
;#return#
;\anArg{-}{data cube}{array of reconstructed detector images}
;#end_function#
;********************************************************
;  handle default row
   if (N_PARAMS() LT 2) then row = 1
;  get detector with window coordinates
  det=obj_new('imagedetector', filename, ierr=ierr)
  if (ierr NE 0) then begin
     ewsprinterrmsg
     RETURN,0
  endif
; read detector info
  detable=det->readrows()
  image = fltarr(320,240)
  det->close
; now get table format image data
  dat = obj_new('imagedata', filename)
  dcol = dat->datacolumns() -1;which columns have data
  ndata = N_ELEMENTS(dcol)
  naxis = dat->naxis()  ; how many rows in table
  isdata = naxis[1] gt 0
  if (isdata) then data=dat->readrows(row) else begin 
     data= det->datatemplate(/float)
     for i=0,ndata-1 do data.(dcol[i])=1.
  endelse
; clean up data objects
  dat->close
  obj_destroy,det
  obj_destroy,dat
; write windows back into 320x240 arrays
  for i=0, ndata-1 do begin
     x0 = detable[i].corner[0]-1
     y0 = detable[i].corner[1]-1
     x1 = x0+detable[i].naxis[0]-1
     y1 = y0+detable[i].naxis[1]-1
     image(x0:x1, y0:y1) = data.(dcol[i])
   endfor
RETURN,image
END

FUNCTION oirGetDetector, inputFiles, iErr=iErr
;********************************************************
;#function#
;oirGetDetector
;#description#
;get information in an imaging detector table
;#call#
;imaging_detector_array = oirGetDetector(inputFiles)
;#inputs#
;\anArg{tableFileName}{string }{file containing imaging_detector data}
;#return#
;\anArg{-}{array of struct }{table containing imaging_detector info}
;#end_function#
;********************************************************
   filename = oirFileList(inputFiles)
   detector = OBJ_NEW('imagedetector', filename[0], iErr=iErr)
   if (iErr ne 0) then begin
      ewsPrintErrMsg
      return,0
   endif
   table = *(detector->table())
   OBJ_DESTROY, detector
RETURN, table
END

FUNCTION oirGetWaveNo, detectorFile, region, noFudge=noFudge
;********************************************************
;#function#
;oirGetWaveNo
;#description#
;get wavenumber (2pi*freq/c) information from a imaging detector table
;#call#
;waveNumberArray = oirGetWaveNo(detectorFile)
;#inputs#
;\anArg{detectorFile}{string }{file containing imaging_detector data}
;\anArg{region}{int }{wavenumbers for which window (1-relative)}
;#return#
;\anArg{-}{array of flt }{table containing wavenumber info for each x-pixel}
;#end_function#
;********************************************************
;  get wavelength information (actually spatial frequency)
;
;  if multiple files take the first one
;
;       correct for known error in internal wavelength scale
;
   if (KEYWORD_SET(noFudge)) then fudge = 1. else fudge = 0.990574
   
   dF = STRSPLIT(detectorFile[0],' ',/extract)
   dF = dF[0]
   if (N_PARAMS() LT 2) then region = 1
   detector = OBJ_NEW('imagedetector', dF)
   coord = OBJ_NEW('coordDistortion', detector, region=region)
   coord->coordGrid, xCoord, yCoord
   sCoord = size(xCoord)
   if (sCoord[0] EQ 1) then freq = xCoord $
      else freq = TOTAL(xCoord, 2)/sCoord[2] 
   wave = 2.e-6*!pi * freq/(2.9979258e8 * fudge)
   OBJ_DESTROY, detector
   OBJ_DESTROY, coord
RETURN, wave
END

FUNCTION oirGetWindowWCS, detectorFile
;********************************************************
;#function#
;oirGetWindowCoords
;#description#
;get first two world coordinates for each window described by input file
;#call#
;coordData = oirGetWindowWCS(detectorFile)
;#inputs#
;\anArg{detectorFile}{string }{file containing imaging_detector data}
;#return#
;\anArg{-}{imagingData table[2] }{in first row of table each data point
; is replaced by x-WCS value; in 2nd row by y-WCS value}
;#end_function#
;********************************************************
;  get coord info
;
;  if multiple files take the first one
;
   dF = STRSPLIT(detectorFile[0],' ',/extract)
   dF = dF[0]
;     get detector object, create equivalent data array
   detector = OBJ_NEW('imagedetector', dF)
   nRegion = detector->naxis()
   nRegion = nRegion[1]
   temp = REPLICATE(detector->dataTemplate(),2)
   data = WHERE(STRPOS(TAG_NAMES(temp),'DATA') GE 0)
   for iRegion=0,nRegion-1 do begin
      coord = OBJ_NEW('coordDistortion', detector, region=iRegion+1)
      coord->coordGrid, xCoord, yCoord
      temp[0].(data[iRegion]) = xCoord
      temp[1].(data[iRegion]) = yCoord
      OBJ_DESTROY, coord
   endfor
   OBJ_DESTROY, detector
RETURN, temp
END

FUNCTION oirGetWaveLength, detectorFile, region, noFudge=noFudge
;********************************************************
;#function#
;oirGetWaveLength
;#description#
;get wavelength (microns) information from a imaging detector table
;#call#
;waveLengthArray = oirGetWaveLength(detectorFile)
;#inputs#
;\anArg{detectorFile}{string }{file containing imaging_detector data}
;\anArg{region}{int }{wavelengths(microns) for which window (1-relative)}
;#return#
;\anArg{-}{array of flt }{table containing wavenumber info for each x-pixel}
;#end_function#
;********************************************************
;  get wavelength information 
;
;  if multiple files take the first one
;
;       correct for known error in internal wavelength scale
;
   if (KEYWORD_SET(noFudge)) then fudge = 1.0 else fudge = 0.990574
   
   dF = STRSPLIT(detectorFile[0],' ',/extract)
   dF = dF[0]
   if (N_PARAMS() LT 2) then region = 1
   detector = OBJ_NEW('imagedetector', dF)
   coord = OBJ_NEW('coordDistortion', detector, region=region)
   coord->coordGrid, xCoord, yCoord
   sCoord = size(xCoord)
   if (sCoord[0] EQ 1) then freq = xCoord $
      else freq = TOTAL(xCoord, 2)/sCoord[2] 
   wave = fudge*2.9979258e14/freq
   OBJ_DESTROY, detector
   OBJ_DESTROY, coord
RETURN, wave
END

FUNCTION oirSubSetData, inputFiles, corner1, corner2, naxis1, naxis2
;********************************************************
;*******************************************************
 ; check if single input or multiple
   filename = oirFileList(inputFiles)
   fileSize = SIZE(filename)
   if (fileSize[0] eq 0) then nFiles = 1 $
      else nFiles=N_ELEMENTS(filename)
   fileRows = intarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      imageObj = OBJ_NEW('imagedata', filename[iFile], ierr=ierr)
      if (ierr ne 0) then begin
         ewsPrintErrMsg
         return,0
      endif
      naxis = imageObj->naxis()
      fileRows[iFile] = naxis[1]
      OBJ_DESTROY, imageObj
   endfor
;create output structure
   imageObj = OBJ_NEW('imagedata', filename[0], ierr=ierr)
   iErr = 0
   inData = imageObj->readRowsf(1)
   OBJ_DESTROY, imageObj
   tags = TAG_NAMES(inData)
   nTags = N_ELEMENTS(tags)
   outData = CREATE_STRUCT(tags[0], inData.(0))
   iData = 0
   for iTag = 1, nTags-1 do begin
      if (STRMID(tags[iTag], 0, 4) NE 'DATA') then $
         outData = CREATE_STRUCT(outData, tags[iTag], inData.(iTag))$
      else begin
         outData = CREATE_STRUCT(outData, tags[iTag], $
            fltarr(naxis1[iData], naxis2[iData]))
         iData = iData + 1
      endelse
   endfor
   cData = REPLICATE(outData, TOTAL(fileRows))
   row = 0
   c1 = corner1 - 1
   c2 = corner2 - 1
   n1 = naxis1 - 1
   n2 = naxis2 - 1
   for iFile = 0, nFiles-1 do begin
      print,'subsetting ',filename[iFile]
      imageObj = OBJ_NEW('imagedata', filename[iFile], ierr=ierr)
      for iRow = 1, fileRows[iFile] do begin
         inData = imageObj->readRowsf(iRow)
         iData = 0
         for iTag = 0, nTags-1 do begin
            if (STRMID(tags[iTag], 0, 4) NE 'DATA') then $
               cData[row].(iTag) = inData.(iTag) $
            else  begin 
               cData[row].(iTag) = inData.(iTag)[c1[iData]:c1[iData]+n1[iData],$
                  c2[iData]:c2[iData]+n2[iData]]
            endelse
         endfor
         row = row + 1
      endfor
      OBJ_DESTROY,imageObj
   endfor
RETURN, cData
END

FUNCTION oirGetKey,fitsfile, keyword, nmatch, ierr=ierr
;********************************************************
;#function#
;oirGetKey
;#description#
;get a single keyword out of primary header of fits file
;#call#
;keyvalue = oirGetKey(fitsfile, keyword, nmatch)
;#inputs#
;\anArg{fitsfile}{string }{file primary header}
;\anArg{keyword}{string }{desired keyword; ESO style is OK too}
;#outputs#
;#return#
;\anArg{nmatch}{int }{number of keys matching specified; 0 if none}
;\anArg{-}{?}{value of keyword}
;#end_function#
;********************************************************
   fileList = oirFileList(fitsfile)
   nfiles = N_ELEMENTS(fileList)
   f = OBJ_NEW('fitsfile',fileList[0])
   h = f->head()
   value=h->getpar(keyword, nmatch)
   OBJ_DESTROY,f
   OBJ_DESTROY,h
   if (nFiles GT 1) then begin
      value = REPLICATE(value,nFiles)
      for i=1,nFiles-1 do begin
         f = OBJ_NEW('fitsfile',fileList[0])
         h = f->head()
         value[i]=h->getpar(keyword, nmatch)
         OBJ_DESTROY,f
         OBJ_DESTROY,h
      endfor
   endif
RETURN,value
END

FUNCTION oirGetPriHead, fitsfile, ierr=ierr
;
   fileList = oirFileList(fitsfile)
   fits = OBJ_NEW('fitsfile', fileList[0], ierr=ierr)
   if (ierr NE 0) then RETURN,0
   head = fits->sthead()
   OBJ_DESTROY,fits
RETURN,head
END

FUNCTION oirGetTime, fitsfile
;
   time = oirGetData(fitsfile, col='TIME')
   time = time.time
RETURN, REFORM(86400.*(time - min(time)))
END

PRO oirNewVis, oldVisFile, outputFile, visData=visData, $
   waveData=waveData, targData=targData, iErr = iErr
;********************************************************
;#procedure#
;oirNewVis
;#description#
;copy vis file with optional updating of table contents
;#call#
;oirNewVis oldVisFile, newVisFile, visData=visData, $
;          targData=targData, waveData=waveData
;#inputs#
;\anArg{oldVisFile}{string }{filename existing Vis file}
;\anArg{outputFile}{string }{file for output data)
;\anArg{Visdata}{struct } oi_vis data to be inserted in output table
;\anArg{waveData}{struct } oi_wavelength data to be inserted in output table
;\anArg{targData}{struct } oi_target data to be inserted in output table
;#end_procedure#
;********************************************************
;useage
   if (N_PARAMS() EQ 0) then begin
      print,' oirNewVis oldVisFile outputFile [visData=visData] $'
      print,'    [waveData=waveData], [targData=targData] '
      RETURN
   endif
; get oldVis table
   visTable = OBJ_NEW('oi_vis', oldVisFile, iErr = iErr)
   if (iErr ne 0) then begin
      ewsPrintErrMsg
      return
   endif
   priHead    = visTable->priHead()
   oldVisData = visTable->readRows()
   visTable->close
;      get target and wavelength tables
   targTable    = OBJ_NEW('oi_target', oldVisFile, iErr = iErr)
   oldTargData  = targTable->readrows()
   targTable->close
   waveTable    = OBJ_NEW('oi_Wavelength', oldVisFile, iErr = iErr)
   oldWaveData  = waveTable->readrows()
   waveTable->close
;  prepare new output file
; and create data template and output data object
;
   visTable->newFile, outputFile, priHead=priHead, iErr=iErr 
   if (iErr ne 0) then begin
      ewsPrintErrMsg
      return
   endif
;    is new data specified?  if not just write  old data
   if (NOT KEYWORD_SET(visData)) then visTable->writeRows, oldVisData $
      else begin
; copy data into template and write to disk
      oldVisData = oldVisData[0] ; only need one row at a time
      for iData = 0, N_ELEMENTS(visdata)-1 do begin
         oldVisData.visamp     = visdata[iData].visamp 
         oldVisData.visamperr  = visdata[iData].visamperr
         oldVisData.visphi     = visdata[iData].visphi 
         oldVisData.visphierr  = visdata[iData].visphierr
         oldVisData.flag       = visdata[iData].flag
         oldVisData.target_id  = visdata[iData].target_id
         oldVisData.time       = visdata[iData].time
         oldVisData.mjd        = visdata[iData].mjd
         oldVisData.int_time   = visdata[iData].int_time
         oldVisData.ucoord     = visdata[iData].ucoord
         oldVisData.vcoord     = visdata[iData].vcoord
         oldVisData.sta_index  = visdata[iData].sta_index
         visTable->writerows,oldVisData
      endfor
   endelse ; used new data
   waveTable->appendToFile, visTable ; work on wavelength table
   if (NOT KEYWORD_SET(waveData)) then waveTable->writeRows, oldWaveData $
      else begin
      oldWaveData = oldWaveData[0]
      for iData = 0, N_ELEMENTS(waveData)-1 do begin
         oldWaveData.eff_wave = waveData.eff_wave
         oldWaveData.eff_band = waveData.eff_band
         waveTable->writeRows, oldWaveData
      endfor
   endelse
   targTable->appendToFile, waveTable
   if (NOT KEYWORD_SET(targData)) then targTable->writeRows, oldTargData $
      else begin
      oldTargData = oldTargData[0]
      for iData = 0, N_ELEMENTS(targData)-1 do begin
         oldTargData.target_id = targData.target_id
         zz = oldTargData.target
         STRPUT,zz,targData.target,0 ; preserve string length
         oldTargData.target = zz
         oldTargData.raep0     = targData.raep0
         oldTargData.decep0    = targData.decep0
         oldTargData.equinox   = targData.equinox
         oldTargData.ra_err    = targData.ra_err
         oldTargData.dec_err   = targData.dec_err
         oldTargData.sysvel    = targData.sysvel
         zz = oldTargData.veltyp
         STRPUT,zz,targData.veltyp,0 ; preserve string length
         oldTargData.veltyp = zz
         zz = oldTargData.veldef
         STRPUT,zz,targData.veldef,0 ; preserve string length
         oldTargData.veldef = zz
         oldTargData.pmra      = targData.pmra
         oldTargData.pmdec     = targData.pmdec
         oldTargData.pmra_err  = targData.pmra_err
         oldTargData.pmdec_err = targData.pmdec_err
         oldTargData.parallax  = targData.parallax
         oldTargData.para_err  = targData.para_err
         zz = oldTargData.spectyp
         STRPUT,zz,targData.spectyp,0 ; preserve string length
         oldTargData.spectyp = zz
         targTable->writeRows, oldTargData
      endfor
   endelse
   targTable->close
   OBJ_DESTROY, visTable
   OBJ_DESTROY, waveTable
   OBJ_DESTROY, targTable
   RETURN
   END


FUNCTION oirGetFitsImage, filename, extnu=extnu, ierr=ierr
;
;       get an image stored in a fits image extension table
    if (NOT KEYWORD_SET(extnu)) then extnu = 1
    fi = OBJ_NEW('fitsImageExtension', filename, extnu, ierr=ierr)
    if (ierr NE 0) then RETURN,0

    image = fi->readImage()
    OBJ_DESTROY, fi
RETURN, image
END
