;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;**********************************************************
;#class#
;fitsFileList
; none
;#description#
; objects dealing with lists of fits files
;#end_class#
;*********************************************************

PRO fitsFileList__DEFINE
;*********************************************************
;#structure#
;fitsFileList
;#inheritance#
;none
;#description#
;structure holding information on a list of fits files
;#structure_text#
a={fitsFileList,  $ ;#class members#
fileList:PTR_NEW(), $ ;ptr to strarr & stored list of files
headFileList:PTR_NEW(), $ ;ptr to strarr & all files from corr to headKey
sList:PTR_NEW(),     $ ;ptr to lonarr & sorted indices of above
dirList:PTR_NEW(),   $ ;ptr to strarr &directories I might look at.
keyList:PTR_NEW(),   $ ;ptr to strarr & names of keywords in files
headKey:PTR_NEW(),   $ ;ptr to structarr & values of keywords in files
currentFile:-1L       $ ;long & current file of interest
}
;#end_structure#
;*********************************************************
RETURN
END

PRO fitsFileList::cleanup
;********************************************************************
;#procedure#
; cleanup
;#call#
; fitsFileList->cleanup
;#description#
; Class destructor
;#end_procedure#
;********************************************************************
   PTR_FREE, self.fileList
   PTR_FREE, self.headFileList
   PTR_FREE, self.sList
   PTR_FREE, self.dirList
   PTR_FREE, self.keyList
   PTR_FREE, self.headKey
RETURN
END

FUNCTION fitsFileList::init, list, textFile=textFile, $
   directory=directory, searchString=searchString, $
   selectKeys=selectKeys,  keyValues=keyValues, guiKeys=guiKeys, $
   restore=restoreFile, guiSearchString = guiSearchString, $
   midiCompress=midiCompress, iErr=iErr
;********************************************************************
;#function#
; init
;#description#
; fitsFileList constructor
; with polymorphic input.  It assembles a list of files
; from a variety of sources and in some cases allows a
; filtering of the list on the basis of keyword values in the headers
;To compile the initial list you can specify :
; 1.  A string array of file names.
; 2.  A file name containing a list of file names
; 3.  A top directory and UNIX style wildcard search string
; 4.  A top directory and no search string: a directory GUI will be brought up
;To further filter the list you can specify:
; 1. A list of keyWords (selectKeys) and keyValues (an IDL structure).  Only
;    those files whose keywords match the specified values are included for
;    further processing
; 2. AND/OR A list of gui Keywords.  The headers of all files passing the
;    above processes will be searched to assemble these keywords,
;    and you can select among them with a GUI.
;#call#
; fitsFileListObj = OBJ_NEW('fitsFileList',fileList)
; fitsFileListObj = OBJ_NEW('fitsFileList',textFile=textFile)
; fitsFileListObj = OBJ_NEW('fitsFileList',directory=topdir, search='VINCI*.fits')
; fitsFileListObj = OBJ_NEW('fitsFileList',dir=topdir, selectKey=keyList, keyValue=keyValues)
; fitsFileListObj = OBJ_NEW('fitsFileList',dir=topdir, guiKeys=guiKeyList)
; fitsFileListObj = OBJ_NEW('fitsFileList',restore=restoreFile)
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{list }  { strarr }{ A list of file names}
;\anArg{textFile } { string }{ Name of file containing file names}
;\anArg{directory}{string}{A top directory to search for files }
;\anArg{searchString}{string}{a wild card type specification;e.g. 'VINCI*.fits'}
;\anArg{}{}{if not given a GUI will be called to pick directories}
;\anArg{selectKeys}{strarr}{set of primary header keywords to show in gui search}
;\anArg{keyValues}{idl Struct}{specified values corresponding to selectKeys}
;\anArg{}{}{Only files whose header keywords equal the specified }
;\anArg{}{}{values are included. If not specified a GUI is brought up}
;\anArg{guiKeys}{strarr}{set of primary header keywords to use in gui search}
;\anArg{restoreFile}{strarr}{file name where fileList->save stored earlier version}
;#return#
;\anArg{-}{fitsFileList object}{}
;#end_function#
;********************************************************************
;restore from save file
   if (KEYWORD_SET(restoreFile)) then begin
      restore, restoreFile
       self.fileList = PTR_NEW(fbtfilelistfilelist)
       self.headFileList = PTR_NEW(fbtfilelistfilelist);
      if (7 EQ SIZE(fbtfilelistkeylist,/type)) then self.keyList = PTR_NEW(fbtfilelistkeylist)
      if (8 EQ SIZE(fbtfilelistheadkey,/type)) then self.headKey = PTR_NEW(fbtfilelistheadKey)
      RETURN,1
   endif
;initialize object with given list
   if (KEYWORD_SET(directory)) then begin
      d = FILE_SEARCH(directory)
      if (d EQ '') then begin
         print,'Directory '+directory+' does not exist'
         RETURN,0
      endif
      dir=directory 
   endif else dir=''
   if (N_ELEMENTS(list) GT 0) then self.fileList=PTR_NEW(list) $
;choose files from a textfile
   else if (KEYWORD_SET(textfile)) then begin
      list = ''
      file = ''
      if (dir NE '') then dir = dir+'/'
      OPENR,unit,dir+textfile,/GET_LUN
      READF, unit, list
      while(NOT eof(unit)) do begin
         READF, unit, file
         list=[list, file]
      endwhile
      FREE_LUN, unit
      self.fileList=PTR_NEW(list)
;neither of above, go looking for files myself
;two choices for initial list: a search string
;or use the gui to select directories
   endif else if (KEYWORD_SET(searchString)) then begin
      print,'Searching for files '+dir+'/'+searchString
      a=systime(1)
;     if (dir NE '') then self.fileList=PTR_NEW(FINDFILE(dir+'/'+searchString)) $
;        else self.fileList=PTR_NEW(FINDFILE(searchString))
;CHU  if (dir NE '') then self.fileList=PTR_NEW(file_search(dir,searchString)) $
;CHU     else self.fileList=PTR_NEW(file_search(searchString))
;CHU
      if (dir NE '') then begin
	spawn,'find -L '+dir+' -iname '+searchString,f
	spawn,'find -L '+dir+' -iname '+searchString,f
	self.fileList=PTR_NEW(f)
      endif else begin
        self.fileList=PTR_NEW(file_search(searchString))
      endelse
      if (KEYWORD_SET(midiCompress)) then self->midiCompress
      print,'Found ',N_ELEMENTS(*self.fileList), 'Files'
      print,'That took ',a-systime(1), 'seconds'
   endif else begin
      if (KEYWORD_SET(guiSearchString)) then gss = guiSearchString $
         else gss = '*.fits'
;CHU  fileList =FILE_SEARCH(dir,gss)
      spawn,'find -L '+dir+' -iname '+gss,fileList
      PTR_FREE,self.fileList
      self.fileList = PTR_NEW(fileList) 
      if (KEYWORD_SET(midiCompress)) then self->midiCompress
   endelse               ; select inital list
;after initial list I can reduce list on basis of keywords
;I can do this either with a gui or by specifying keyword values
;or both or neither  -> no keyword selection
;
;check if anything turned up
   if ('' EQ (*self.fileList)[0]) then begin
      print,' no appropriate files found in selected directories'
      RETURN,0
   endif
   if (KEYWORD_SET(keyValues)) then self->selectOnKeys, selectKeys, keyValues
   if (KEYWORD_SET(guiKeys)) then self->guiSelect, guiKeys
   RETURN,1
END

FUNCTION fitsFileList::nFiles
;********************************************************************
;#function#
; nFiles
;#description#
;get number of file names currently stored
;#call#
;numberOfFiles = fileListObject->nFiles()
;#return#
;\anArg{-}{long}{number of stored file names}
;#end_function#
;********************************************************************
   if(NOT PTR_VALID(self.fileList)) then RETURN,0 $
   else RETURN,N_ELEMENTS(*self.fileList)
END

FUNCTION fitsFileList::files
;********************************************************************
;#function#
; files
;#description#
;get all stored file names
;#call#
;Files = fileListObject->Files()
;#return#
;\anArg{-}{strArr}{stored file names}
;#end_function#
;********************************************************************
   if(NOT PTR_VALID(self.fileList)) then RETURN,'' $
   else RETURN,*self.fileList
END

FUNCTION fitsFileList::dirList
;********************************************************************
;#function#
; dirList
;#description#
;get all stored directory names
;#call#
;directories = fileListObject->dirList()
;#return#
;\anArg{-}{strArr}{stored directory names}
;#end_function#
;********************************************************************
   if(NOT PTR_VALID(self.dirList)) then RETURN,'' $
   else RETURN,*self.dirList
END

FUNCTION fitsFileList::headerTable, keyList, iErr=iErr
;*******************************************************************
;#function#
;headerTable
;#description#
;return a table containing the header key words
;of all internally stored files
;#call#
;HeaderListStructure = fileListObj->headerTable(keyList,iErr=iErr)
;#return#
;\anArg{-}{idl sructArr}{table with one row per file and one}
;\anArg{}{}{column per keyword containing values}
;#outputs#
;\anArg{keyList}{strArr}{list of keywords used to assemble file}
;\anArg{iErr}{int}{0=OK, 1=no headkey table present; use 'getHeaderKeys'}}
;#end_function#
;*******************************************************************
   if (NOT PTR_VALID(self.headKey)) then begin
      print,'headers have not yet been retrieved'
      print,'use fitsFileList->getHeaderKeys'
      iErr = 1
      keyList = ''
      RETURN,0
   endif
   if (PTR_VALID(self.keyList)) then keyList = *self.keyList else $
      keyList = ''
   iErr = 0
RETURN, *self.headKey
END

PRO fitsFileList::setHeader, headTable, keyList
;*******************************************************************
;#procedure#
;setHeader
;#description#
;insert a header keyvalue table.
;this is a private function intended only for internal  use
;head Table and keylist must be consistent with each other
;and with the files
;#call#
;fileListObj->setHeader,headTable, keyList
;#inputs#
;\anArg{HeadTable}{structArr}{value of keywords in file}
;\anArg{keyList}{strArr}{list of keywords used to assemble file}
;#end_procedure#
;*******************************************************************
   PTR_FREE, self.headKey
   PTR_FREE, self.keyList
   self.headKey = PTR_NEW(headTable)
   self.keyList = PTR_NEW(keyList)
RETURN
END

FUNCTION fitsFileList::getHeaderKeys, keyList, extName=extName, $
   extNumber=extNumber, iErr=iErr
;********************************************************************
;#function#
; getHeaderKeys
;#description#
;retrieve specified keys from all stored files
;into an idl structure/table: append filename as an extra column
;#call#
;table = fileListObject->getHeaderKeys(keyList,fileName=fileName,extName=extName,
;   extNumber=extNumber, iErr=iErr)
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ keyList }  { strarr }{ A list of keyWords to extract }
;\anArg{}{}{These may also be ESO HIERARCH keywords}
;\anArg{}{}{To indicate this specify two or more keywords separated by blanks}
;\anArg{}{}{Do not include the "ESO HIERARCH"}
;\anArg{extName }{ string }{ which extension to examine;default=primary}
;\anArg{extNumber }{ int }{ which extension to examine;default=primary}
;#return#
;\anArg{-}{IDL structure}{requested keywords+filenames }
;#end_function#
;********************************************************************
   primary = (NOT KEYWORD_SET(extName)) AND (NOT KEYWORD_SET(extNumber))
   iErr = 1
   if (NOT PTR_VALID(self.filelist)) then RETURN,0
;go through files to find all requested keywords.  Use the information
;you find to construct the output table with the correct variable types
;fill in dummies if you cant find them
;initial output structure has only filename
   rowStruct = CREATE_STRUCT('filename','')
   nKeys = N_ELEMENTS(keyList)
;mapping from input keylist to output keylist
; -1 means this key hasn't been found yet
   keyPos = REPLICATE(-1, nKeys)
   lastPos = 1
   nFiles = N_ELEMENTS(*self.filelist)
   DONE = (1 EQ 0)
   iFile = 0
;get a  file and header to analyze structure
   while ((NOT DONE) and (iFile LT nFiles)) do begin
      fileName = (*self.fileList)[iFile]
;if multiple space separated files, get first one
      fileName = STRSPLIT(fileName,' ',/EXTRACT)
      fileName = fileName[0]
;get appropriate header
      if (primary) then begin
         fileObject = OBJ_NEW('fitsFile', fileName, iErr=iErr)
         if (iErr NE 0) then begin
         print,'gethead couldnt open file ',(*self.filelist)[iFile]
         GOTO,SKIP1
         endif
         head = fileObject->priHead()
      endif else begin
         fileObject = OBJ_NEW('fitsExtension', fileName, $
         extName=extName, extNumber=extNumber, iErr=iErr)
         if (iErr NE 0) then begin
         print,'gethead couldnt open file ',(*self.filelist)[iFile]
         GOTO,SKIP1
         endif
         head = fileObject->head()
      endelse
;loop through keys that you havent found yet
      for iKey = 0, nKeys-1 do if (keyPos[iKey] EQ -1) then begin
         key = keyList[iKey]
         value = head->getPar(key, match)
         if (match GT 0) then begin
;generate column name
            keyPos[iKey] = lastPos
            lastPos = lastPos + 1
            split = STRSPLIT(key,' -',/EXTRACT)
            colName = split[0]
            for iN = 1, N_ELEMENTS(split)-1 do colName=colName+split(iN)
         rowStruct=CREATE_STRUCT(rowStruct, colName, value[0])
         endif  ; found this key
      endif          ; loop over keys not found
      DONE = (TOTAL(keyPos EQ (-1)) LE 0)
SKIP1:
      iFile = iFile + 1
      OBJ_DESTROY, fileObject
   endwhile
;if some columns not found, warn user and append dummy elements
;at the end of the structure
   if (NOT DONE) then begin
      for iKey = 0, nKeys-1 do if (keyPos(iKey) LT 0) then begin
         key = keyList[iKey]
         print,'couldnt find key ', key, ' in any file '
         keyPos[iKey] = lastPos
         lastPos = lastPos + 1
         split = STRSPLIT(key,' -',/EXTRACT)
         colName = split[0]
         for iN = 1, N_ELEMENTS(split)-1 do colName=colName+split(iN)
         rowStruct = CREATE_STRUCT(rowStruct, colName, 0)
      endif
   endif
;create output array
   outArray = REPLICATE(rowStruct, nFiles)
   for iFile = 0, nFiles-1 do begin
      fileName = STRSPLIT((*self.fileList)[iFile],' ',/EXTRACT)
      fileName = fileName[0]
      if (primary) then begin
         fileObject = OBJ_NEW('fitsFile', fileName, iErr=iErr)
         if (iErr NE 0) then begin
         print,'gethead couldnt open file ',(*self.filelist)[iFile]
         GOTO,SKIP
         endif
         head = fileObject->priHead()
      endif else begin
         fileObject = OBJ_NEW('fitsExtension', fileName, $
         extName=extName, extNumber=extNumber, iErr=iErr)
         if (iErr NE 0) then begin
         print,'gethead couldnt open file ',(*self.filelist)[iFile]
         GOTO,SKIP
         endif
         head = fileObject->head()
      endelse
;parse it a bit
      split = STRSPLIT(fileName,'/',/EXTRACT)
      fileName = split(N_ELEMENTS(split)-1)
      for iKey = 0, nKeys-1 do begin
         key = keyList[iKey]
         value = head->getPar(key, match)
         if(match GE 1) then outArray[iFile].(keyPos[iKey]) = value[0] $
;replace with null value
         else begin
            type = SIZE(outArray[0].(keyPos[iKey]),/type)
;numbers to zero
            if (type LE 6) then outArray[iFile].(keyPos[iKey]) = 0 $
;strings to empty string
            else if (type EQ 7) then outArray[iFile].(keyPos[iKey]) = ''
         endelse
      endfor          ; loop over keys
      outArray[iFile].(0) = fileName
      fileObject->close
      OBJ_DESTROY, fileObject
      OBJ_DESTROY, head
SKIP:
   endfor  ; main file loop
RETURN,outArray
ERR_RET:
   print,'error in fitsFileList::getHeaderKeys'
   RETURN,0
END

PRO fitsFileList::guiSelect, keyList, extName=extName, $
   extNumber=extNumber, doSelect=doSelect, iErr=iErr
;********************************************************************
;#procedure#
; guiSelect
;#description#
;use GUI to allow user to select files from a directory
;#call#
;fitsObj->guiSelect,keyList, extName=extName, extNumber=extNumber
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ keyList }  { strarr }{ A list of keyWords to examine }
;\anArg{ }  {}{if not specified and a previous search has been stored}
;\anArg{ }  {}{use the old values}
;\anArg{}{}{These may also be ESO HIERARCH keywords}
;\anArg{ extName }{ string }{ which extension to examine;default=primary}
;\anArg{ extNumber }{ int }{ which extension to examine;default=primary}
;#end_procedure#
;********************************************************************
;check if keywords specified
   if(N_PARAMS() LT 1) then begin
      if (NOT PTR_VALID(self.keyList)) then begin
         print,'no keywords specified or stored'
         iErr = 1
         RETURN
      endif else keyList = *self.keyList
   endif
;check if we already have the table
   gotKeys = PTR_VALID(self.keyList) AND PTR_VALID(self.headKey)
   if (gotKeys) then gotKeys = $
      N_ELEMENTS(keyList) EQ N_ELEMENTS(*self.keyList)
   if (gotKeys) then gotKeys = $
      (TOTAL(STRTRIM(keyList,2) NE STRTRIM(*self.keyList,2)) EQ 0)
;  if we have everything already, deselect file list
   if (gotKeys) then begin
      PTR_FREE,self.fileList
      self.fileList = PTR_NEW(*self.headFileList)
;if they dont agree get table
   endif else begin
      PTR_FREE, self.keyList
      PTR_FREE, self.headKey
      self.keyList = PTR_NEW(keyList)
      a=systime(1)
      print,'creating header keyword table'
      self.headKey = PTR_NEW(self->getHeaderKeys(keyList, iErr = iErr))
      a=systime(1)-a
      print,'that took ',a,' seconds'
      if ((iErr NE 0) OR (SIZE(*self.headKey,/type) NE 8)) then begin
         print, 'couldnt create table'
         RETURN
      endif
   endelse    ; didnt already have table
   print,'current list has ', self->nFiles(), ' files'
   if(N_ELEMENTS(*self.headKey) LT 1) then begin
      iErr = 1
      print,'No files left in table'
      RETURN
   endif
   select = guiTableSelect(*self.headKey, doSelect, /initialOff)
   if (NOT doSelect) then select[*] = 0
   if ((doSelect EQ 1)) then  self->reduce,select
RETURN
END
 
PRO fitsFileList::guiSelectDirs, baseDir, iErr = iErr
;********************************************************************
;#function#
; guiSelectDirs
;#description#
;use the table selection gui to pick a set of subdirectories
;for use in the fileList creation routines
;#call#
;fileListObject->guiSelectDirs(baseDir)
;#inputs#
;\anArg{baseDir}{strArr}{directory containing all directories of interest}
;#end_function#
;********************************************************************
   iErr = 1
;check input
   if (N_ELEMENTS(baseDir) LE 0) then begin
      print,'base Directory not specified'
      RETURN
   endif else if (7 NE SIZE(baseDir,/type)) then begin
      print,'base Directory must be string'
      RETURN
   endif
;get the contents of the base directory
   dirList = FINDFILE(baseDir)
   if (N_ELEMENTS(dirList) LT 1) then begin
      print,'No files in base directory'
      RETURN
   endif
;create a rather simple table with directory names
   dirTable = REPLICATE({Dir:''}, N_ELEMENTS(dirList))
   dirTable.(0) = dirList
;bring up table selection gui
   select = guiTableSelect(dirTable, doSelect,/initialOff)
;check if anything selected
   if (total(select NE 0) LE 0) then begin
      print,'No directories selected'
      RETURN
   endif
;store results
   if (doSelect) then begin
      PTR_FREE, self.dirList
      self.dirList = PTR_NEW(dirList(where(select NE 0)))
   endif
   iErr = 0
   RETURN
END

FUNCTION fitsFileList::getTableValues, columns, rows, extName=extName, $
   extNumber=extNumber, iErr=iErr
;********************************************************************
;#function#
; getTableValues
;#description#
;retrieve into a structure the values of the given columns
; and rows in the given extension of all my files
;#call#
;keyWordTable = fileList->getTableValues(['DATA1'], indgen(20),[1]
;#inputs#
;#return#
;\anArg{-}{idl structure}{Effective row and structures}
;#end_function#
;********************************************************************
;retrieve into a structure the values of the given columns
;in the given extension of all my files
   iErr = 1
   if (NOT PTR_VALID(self.filelist)) then RETURN,0
   nFiles = N_ELEMENTS(*self.filelist)
   nRows = N_ELEMENTS(rows)
   tRows = nFiles * nRows
   nCols = N_ELEMENTS(columns)
;cant extract tables from primary
   primary = (NOT KEYWORD_SET(extName)) AND (NOT KEYWORD_SET(extNumber))
   if (primary) then GOTO, ERR_RET
;get first file and header to set up first row of structure
   table = OBJ_NEW('fitsTable', (*self.fileList)[0], $
      extName=extName, extNumber=extNumber, iErr=iErr)
   if (iErr NE 0) then GOTO, ERR_RET
   colTemp = table->readRows(rows[0], col=columns)
   outArray = REPLICATE(colTemp, tRows)
   table->close
   OBJ_DESTROY, table
   for iFile = 0, nFiles-1 do begin
      fileName = STRSPLIT((*self.fileList)[iFile], ' ',/EXTRACt)
      fileName = fileName[0]
      table = OBJ_NEW('fitsTable', fileName, $
         extName=extName, extNumber=extNumber, iErr=iErr)
      if (iErr NE 0) then GOTO, ERR_RET
      array = table->readRows(rows, col=columns, iErr=iErr)
      if (iErr NE 0) then GOTO, ERR_RET
      for iCol = 0, nCols-1 do outArray[iFile*nRows+lindgen(nRows)].(iCol) =$
      array.(iCol)
      table->close
      OBJ_DESTROY, table
   endfor
RETURN,outArray
ERR_RET:
   print,'error in fitsFileList::getTableValues'
   RETURN,0
END

PRO fitsFileList::concatTables, outFile, columns, rows,  $
   extName=extName, extNumber=extNumber, iErr=iErr
;********************************************************************
;#procedure#
; concatTables
;#description#
;make a huge table with specified rows and columns
;from an extension of all my files.  Write the
;result out as a fits file.
;#call#
;fileList->concatTables, outFileName, columnNames, desiredRows,
; extName=extName, extNu=extNumber
;#inputs#
;\anArg{outFile}{string}{Name of file to contain output table}
;\anArg{columns}{strarr or intArr}{column numbers of names to copy into table}
;\anArg{rows}{intArr}{row numbers from each input file to copy into table}
;\anArg{extName}{String}{one way to specify which table to look at}
;\anArg{extName}{extNumber}{another way to specify which table to look at}
;#end_procedure#
;********************************************************************
;retrieve into a structure the values of the given columns
;in the given extension of all my files
   iErr = 1
   if (NOT PTR_VALID(self.filelist)) then RETURN
   nFiles = N_ELEMENTS(*self.filelist)
;cant extract tables from primary
   primary = (NOT KEYWORD_SET(extName)) AND (NOT KEYWORD_SET(extNumber))
   if (primary) then GOTO, ERR_RET
;get first file and header to set up first row of structure
   fileName = STRSPLIT((*self.fileList)[0], ' ',/EXTRACT)
   fileName = fileName[0]
   table = OBJ_NEW('fitsTable', fileName, $
      extName=extName, extNumber=extNumber, iErr=iErr)
   if (iErr NE 0) then GOTO, ERR_RET
   outExt = table->extName()
   colTemp = table->readRows(1, col=columns)
   table->close
   OBJ_DESTROY, table
;create output data file
   outTable = OBJ_NEW('fitsTable', colTemp, extName=outExt, iErr=iErr)
   if (iErr NE 0) then GOTO, ERR_RET
   outTable->newFile, outFile, iErr=iErr
   if (iErr NE 0) then GOTO, ERR_RET
   for iFile = 0, nFiles-1 do begin
      fileName = STRSPLIT((*self.fileList)[0], ' ',/EXTRACT)
      fileName = fileName[0]
      table = OBJ_NEW('fitsTable', fileName, $
         extName=extName, extNumber=extNumber, iErr=iErr)
      if (iErr NE 0) then GOTO, ERR_RET
      if (N_ELEMENTS(rows) GT 0) then $
         outTable->writeRows, table->readRows(rows, col=columns, iErr=iErr), iErr=iErr $
      else outTable->writeRows, table->readRows(col=columns, iErr=iErr), iErr=iErr
      if (iErr NE 0) then GOTO, ERR_RET
      table->close
      OBJ_DESTROY, table
   endfor
   outTable->close
RETURN
ERR_RET:
   print,'error in fitsFileList::getTableValues'
RETURN
END

PRO fitsFileList::reduce, subList
;********************************************************************
;#procedure#
; reduce
;#description#
;discard all but a subset of files
;#call#
;fileListObject->reduce,subList
;#inputs#
;\anArg{subList}{boolean array}{Files where subList[i] is "TRUE" are retained}
;#end_procedure#
;********************************************************************
;choose a subset of current file list
   if (NOT PTR_VALID(self.fileList)) then RETURN
; copy unselected list into save area
   PTR_FREE,self.headFileList
   w = where(subList)
   if (MAX(w) GE 0) then begin ; found something
      self.headFileList = PTR_NEW(*self.fileList)
      newList = (*self.fileList)[where(subList)]
      PTR_FREE, self.fileList
      self.fileList = PTR_NEW(newList)
   endif else PTR_FREE,self.fileList
   PTR_FREE, self.sList
RETURN
END

PRO fitsFileList::selectOnKeys, keyList, keyValues, $
   noTagCheck=noTagCheck, iErr=iErr
;********************************************************************
;#procedure#
; selectOnKeys
;#description#
;select files by keyword values
;Retain only those files whose header keywords have
;specified values.  All these selections are ANDed.
;That is, only files for which all specified keyword
;value agree are taken.
;#call#
;fileListObject->selectOnKeys, keyWordList, keyValues
;#inputs#
;\anArg{keyList}{strArr}{a list of keyword names; can be ESO keywords}
;\anArg{keyValues}{struct}{a matching list of values specified as strings}
;\anArg{noTagCheck}{bool}{if set: dont check that tag names of keyValues}
;\anArg{}{}{correspond with keyList}
;#end_procedure#
;********************************************************************
   iErr = 0
;check if we already have the table
   gotKeys = PTR_VALID(self.keyList) AND PTR_VALID(self.headKey)
   if (gotKeys) then gotKeys = $
      N_ELEMENTS(keyList) EQ N_ELEMENTS(*self.keyList)
   if (gotKeys) then gotKeys = $
      (TOTAL(STRTRIM(keyList,2) NE STRTRIM(*self.keyList,2)) EQ 0)
;if they dont agree get table
   if (NOT gotKeys) then begin
      PTR_FREE, self.keyList
      PTR_FREE, self.headKey
      self.keyList = PTR_NEW(keyList)
      self.headKey = PTR_NEW(self->getHeaderKeys( keyList, iErr = iErr))
      if (iErr NE 0) then begin
         print, ' failed to creat selection table'
         RETURN
      ENDIF
   endif
   nKeys = N_ELEMENTS(keyList)
;check that tags agree
   tagTable = TAG_NAMES((*self.headKey))
   tagKeys  = TAG_NAMES(keyValues)
   if (N_ELEMENTS(tagTable) NE (1+N_ELEMENTS(tagKeys))) then iErr = 1 else $
      if (TOTAL(STRTRIM(tagTable,2) NE ['FILENAME',STRTRIM(tagKeys,2)]) GT 0) then $
      iErr = 1
   if (iErr NE 0) then begin
      print, 'specified keyNames and keyValues dont correspond'
      RETURN
   ENDIF
;start with everything selected
   select = REPLICATE(1B, N_ELEMENTS(*self.fileList))
   for iKey = 0, nKeys-1 do begin
      value = keyValues.(iKey)
      select = select AND (value EQ (*self.headKey).(iKey+1))
   endfor
   print,long(total(select)), ' files selected'
   if (total(select) EQ 0) then begin
      print,' no files selected'
      RETURN
   end
   self->reduce, select
RETURN
END

PRO fitsFileList::midiCompress
;********************************************************************
;#procedure#
; midiCompress
;#description#
;go through self.fileList and concatenate MIDI "split" filenames into a
;blank separated string
;#call#
;fileListObject->midiCompress
;#end_procedure#
;********************************************************************
   if (NOT PTR_VALID(self.fileList)) then RETURN
   nOld = N_ELEMENTS(*self.fileList)
   newList = STRARR(nOld)
   new = 0
   old = 0
   oldTplStart = ''
   oldFileNo   = 0
   WHILE (old LE nOld-1) DO BEGIN
      oldFileName = (*self.fileList)[old]
      old = 1 + old
      newFits = OBJ_NEW('fitsfile', oldFileName)
      newHead = newFits->head()
      newTplStart = newHead->getPar('TPL START', ntpl)
      newFileNo   = newHead->getPar('EXPO FILENO')
; this is midi continuation file
      if ((oldTplStart EQ newTplStart) AND (newFileNo GT oldFileNo)$
           AND (ntpl GT 0)) then $
         newList[new-1] = newList[new-1] + ' ' + oldFileName $
      else begin
; check if this a _99 file; if not update new list
         extn = newFits->listextensions()
         if (MAX(STRPOS(extn, 'IMAGING_DETECTOR')) GE 0) then begin
            newList[new] = oldFileName
            new = new + 1
         endif
      endelse
      OBJ_DESTROY,newFits
      OBJ_DESTROY,newHead
      oldTplStart = newTplStart
      oldFileNo = newFileNo
   ENDWHILE
   PTR_FREE,self.fileList
   self.fileList = PTR_NEW(newList[0:new-1])
   print,'Reduced list from ', nOld, ' to ',new,' files'
RETURN
END

FUNCTION fitsFileList::findFile, fileName
;********************************************************************
;#function#
; findFile
;#description#
;return the sequence number (0-rel) of given fileName
;or -1 if not found.  Binary search should be
;fairly efficient for doing this many times.
;#call#
;fileSeqNo = fileListObject->findFile(fileName)
;#return#
;\anArg{-}{long}{index in stored array; 0-rel}
;#end_function#
;********************************************************************
   if (NOT PTR_VALID(self.sList)) then self.sList=PTR_NEW(sort(*self.fileList))
   nFile = N_ELEMENTS(*self.fileList)
   xFile = nFile/2
   dx = 2^ceil(alog10(xFile)/alog10(2))/2
   yFile = (*self.fileList)[(*self.sList)[xFile]]
   while ((yFile NE fileName) AND (dx GT 0)) do begin
      xFile = xFile + dx*(1-2*(yFile GT fileName))
      dx = dx/2
      yFile = (*self.fileList)[(*self.sList)[xFile]]
   endwhile
   if (yFile EQ fileName) then RETURN,(*self.sList)[xFile] $
      else RETURN,-1
END

FUNCTION fitsFileList::clone
;********************************************************************
;#function#
; clone
;#description#
;return a copy of myself
;#call#
;newObject = fileListObject->Files()
;#return#
;\anArg{-}{strArr}{copy of me }
;#end_function#
;********************************************************************
;return a new object with the same files
   newList = OBJ_NEW('fitsFileList',*self.fileList)
   if (PTR_VALID(self.headKey)) then newList->setHeader, $
      *self.headKey, *self.keyList
RETURN, newList
END

PRO fitsFileList::merge, list2
;********************************************************************
;#procedure#
; merge
;#description#
;merge a second file list with this one
;remove duplicates
;#call#
;fileListObject->merge(fileList2)
;#end_procedure#
;********************************************************************
   totalList = [(*self.fileList), list2->files()]
   PTR_FREE, self.filelist
   PTR_FREE, self.sList
   self.sList = PTR_NEW(SORT(totalList))
;get list of indices to after removing duplicates
   uind = UNIQ(totalList, *self.sList)
;resort so that order is roughly the same as original order
   self.fileList = PTR_NEW(totalList(uInd(sort(uInd))))
RETURN
END

PRO fitsFileList::start
;********************************************************************
;#procedure#
; start
;#description#
;initialize internal file counter
;this is used if you want to use ->getFileNames or ->getFileObs
;#call#
;fileListObject->start
;#end_procedure#
;********************************************************************
   self.currentFile = -1
RETURN
END

FUNCTION fitsFileList::getFileNames, iFile, iErr=iErr
;********************************************************************
;#function#
; getFileNames
;#description#
;retrieve a file names, one by one
;if iFile is specified, get this one (0-relative)
;iErr is set to 4 if iFile is out of range
;if iFile is not specified get "next" file
;#call#
;fileName = fileListObject->getFileNames()
;fileName = fileListObject->getFileNames(7)
;#inputs#
;\anArg{iFile}{int}{sequence number of stored filename:optional}
;#return#
;\anArg{-}{string}{a single file name}
;#end_function#
;********************************************************************
;return the file names one by one
   iErr =1
;no iFile was specified.  If first is set return
   if(N_PARAMS() LT 1) then iFile = self.currentFile + 1
   nFiles = N_ELEMENTS(*self.fileList)
;check if we have gone too far
   if (iFile GE nFiles OR (iFile LT 0)) then begin
      iErr = 4
      RETURN,''
   endif
;looks OK
   self.currentFile = iFile
   iErr = 0
RETURN, (*self.fileList)[iFile]
END

FUNCTION fitsFileList::getFileObs,  iFile, extName=extName,iErr=iErr
;********************************************************************
;#function#
; getFileObs
;#description#
;open stored files names and return file Objects or Table Object
;one by one ;if iFile not specified get "next" object.
;initialize counter with call to ::start
;#call#
;fitsFileObj = fileListObject->()
;fitsFileObj = fileListObject->(3)
;fitsTable = fileListObject->(3, extName='BATCH')
;#inputs#
;\anArg{iFile}{long}{optional:0-based sequence number of internal list}
;\anArg{extName}{string}{optional:open this extention as a table;}
;\anArg{}{}{otherwise open whoe file as a fitsfile}
;#return#
;\anArg{-}{fitsFileObj or fitsTableObj}{requested object }
;#end_function#
;********************************************************************
   iErr = 1
;no iFile was specified.  If first is set return
;first file and set pointer, otherwise return next file
   if(N_PARAMS() LT 1) then  iFile = self.currentFile + 1
   nFiles = N_ELEMENTS(*self.fileList)
;check if we have gone too far
   if (iFile GE nFiles OR (iFile LT 0)) then begin
      iErr = 4
      RETURN, OBJ_NEW()
   endif
;looks OK
   self.currentFile = iFile
;no extension specified, return fitsfile object
   if (NOT KEYWORD_SET(extName)) then begin
      fileObject = OBJ_NEW('fitsfile', (*self.fileList)[iFile], iErr=iErr)
      if (iErr NE 0) then begin
      print,'failed to open fits file ',(*self.fileList[iFile])
      RETURN, OBJ_NEW()
      endif
   endif else begin
      fileObject = OBJ_NEW('fitsTable', (*self.fileList)[iFile], $
      extName=extName, iErr=iErr)
      if (iErr NE 0) then begin
      print,'failed to open fits table ',(*self.fileList[iFile])
      RETURN, OBJ_NEW()
      endif
   endelse
RETURN, fileObject
END

PRO fitsFileList::save, diskFileName, select=select
;*****************************************************************
;#procedure#
; save
;#description#
;save contents of this list to disk in IDL save format
;note that a regular "save" or writeu wont work
;because the object contains pointers
;#call#
;fileListObject->save, diskFileName
;#inputs#
;\anArg{diskFileName}{string}{an acceptable file name}
;#end_procedure#
;********************************************************************
;figure out if anything is valid
;store them in variable the user is not likely to use
   if (PTR_VALID(self.headFileList)) then $
       fbtfilelistfilelist = *self.headFilelist 
   if (PTR_VALID(self.fileList) AND  $
      ((KEYWORD_SET(select)) OR (NOT PTR_VALID(self.headFileList)) )) then $
       fbtfilelistfilelist = *self.filelist 
   if (N_ELEMENTS(fbtfilelistfilelist) LT 0) then RETURN
   if (PTR_VALID(self.headKey) AND (PTR_VALID(self.keyList))) then begin
       if (NOT KEYWORD_SET(select)) then fbtfilelistheadkey = *self.headKey $
          else begin
             fbtfilelistheadkey = REPLICATE((*self.headKey)[0], N_ELEMENTS(*self.filelist))
             for i=0,N_ELEMENTS(*self.filelist)-1 do $
                fbtfilelistheadkey[i] = (*self.headKey)[where((*self.headFileList) EQ $
                   (*self.fileList)[i])]
          endelse
       fbtfilelistkeylist = *self.keyList
   endif else begin
       fbtfilelistheadkey = 0
       fbtfilelistkeylist = 0
   endelse
       save,file=diskFileName, fbtfilelistfilelist, fbtfilelistheadkey, $
       fbtfilelistkeylist
RETURN
END


