pro cubegridm,datafile

COMMON SHAREIT, diam, sep, pa, theMin, theMax, min_flag, sym_Delta, $ 
                sym_ChiSq, starname, npoiid, riflag

COMPILE_OPT DEFINT32, STRICTARR, STRICTARRSUBS

;***********************************************************
;;DESCRIPTION
; Modification of cubegrid to handle files created with the 
; mdata.pro procedure. This will grab the diameters from the 
; saved structure instead of requiring them to be input 
; by hand. The diameters are obtained in doublesearch 
; procedure below, and d2 is set to 0.5*d1.
; The structure is also expanded to hold some additional star 
; information.
; Note that if riflag returns True than the diameter was not
; in diameter.bsc and the r-i color was computed as a call 
; within mdata.pro

;;RELATED PROGRAMS
; mdata.pro wrote out the visibility data used here
; doubleSearch procedure below does the grid search.
 
;; OLD cubegrid.pro DESCRIPTION
; Procedure to grid search visibility data for presence of a binary.
; x and y axes of cube are ra and dec on the sky, z axis would be
; the delta-mag. The program repetitively calls doublesearch with 
; differing delta-mags, starting at zero and going to 4.0 in steps of
; 0.5 magnitudes.
;************************************************************

;;BEGIN CODE
; Initialize datafile, number of deltamag iterations, output strings
; and whether to make plots

if n_elements(datafile) eq 0 then begin
	print,'You must call with data file name!'
	return
endif
;datafile='bper-2012-03-16.dat'
;datafile='betaur-2012-12-16.dat'
;datafile='omi_Leo-2004-03-18.dat'
;datafile='xi_Gem-2004-03-10.dat'
;datafile='phi_Her-2004-04-30.dat'
; Now set the dmag steps. Note that as the increment 
; is 0.1 as indicated in the next FOR loop the 
; largest delta-mag we examine is num_dmag*0.1
num_dmag=40
outprefix = STRMID(datafile,0,STRLEN(datafile)-4)
makeplots = 'yes'
PRINT,'Will cubegridm make plots? ___',makeplots,'___'

; Define structure to contain deltamag, sep, pa, min and max ChiSq 
; and minimum flag for each deltamag call to doubleSearch
; Note that the first 4 entries are liek a header and are single
; elements, not arrays.

gridout = {grid, starname:' ', npoiid:' ', diam:0.0, riflag:' ', $ 
           deltamag:FLTARR(num_dmag+1), sep:FLTARR(num_dmag+1), $
           pa:FLTARR(num_dmag+1), theMin:FLTARR(num_dmag+1), theMax:FLTARR(num_dmag+1), $
           min_flag:LONARR(num_dmag+1)}

; Initialize start time and loop calls to doubleSearch
; Note we are looping over 0.1 mag steps.

start_time = SYSTIME(1)
FOR i=0,num_dmag DO BEGIN
    deltamag=0.1*i
    doubleSearch,datafile,deltamag
    gridout.deltamag[i]=deltamag
    gridout.sep[i]=sep
    gridout.pa[i]=pa
    gridout.theMin[i]=theMin
    gridout.theMax[i]=theMax
    gridout.min_flag[i]=min_flag
ENDFOR
stepps = STRCOMPRESS(STRING(num_dmag+1))
totaltime = STRCOMPRESS(STRING(SYSTIME(1)-start_time))
print,''
print,'****GRID SEARCH COMPLETED****************'
PRINT, 'Total time to run through '+stepps+' delta-mag steps= '+totaltime+' s'

; Find minimum and notify user

globalMin=MIN(gridout.theMin,minLoc)

print,''
print,'For star ',starname
print,'Minimum reduced chi-squared of ',globalMin,' found at deltaMag of ',gridout.deltamag[minLoc]
print,' at a seperation of ',gridout.sep[minLoc],' and a position angle of ',gridout.pa[minLoc]  

; Only make plots if we really want them
; If we do make a single page with three plots on it

IF makeplots EQ 'yes' THEN BEGIN
    USERSYM, COS(FINDGEN(17)*(!PI*2/16.0)), SIN(FINDGEN(17)*(!PI*2/16.0))
    set_plot,'PS'
    !P.MULTI=[0,1,3]
    DEVICE,filename=outprefix+'.ps',xsize=7.25,ysize=10.0,xoffset=0.5,yoffset=0.5,/inches,FONT_SIZE=20
    PLOT,gridout.deltamag,gridout.theMin,XTITLE=sym_Delta+'!6mag',YTITLE=sym_ChiSq, $
      TITLE='!6'+datafile,CHARTHICK=2.0,THICK=3.0,XTHICK=3.0,YTHICK=3.0, $
      PSYM=-8, SYMSIZE=0.5, XRANGE=[-0.1,4.2], XSTYLE=1, $ 
      SUBTITLE='Global minimum at '+sym_Delta+'mag ='+STRING(gridout.deltamag[minLoc],FORMAT='(F6.2)')
    OPLOT,[gridout.deltamag[minLoc],gridout.deltamag[minLoc]],[0,max(gridout.theMax)],THICK=3.0, $
          LINESTYLE=2
    USERSYM, COS(FINDGEN(17)*(!PI*2/16.0)), SIN(FINDGEN(17)*(!PI*2/16.0)),/FILL
    PLOT,gridout.deltamag,gridout.sep,XTITLE=sym_Delta+'mag',YTITLE='Separation (mas)', $
      CHARTHICK=2.0,THICK=3.0,XTHICK=3.0,YTHICK=3.0, $
      PSYM=8, SYMSIZE=0.5, XRANGE=[-0.1,4.2], XSTYLE=1, $ 
      SUBTITLE='Global minimum at sep (mas) ='+STRING(gridout.sep[minLoc],FORMAT='(F6.2)')
    OPLOT,[gridout.deltamag[minLoc],gridout.deltamag[minLoc]],[0,max(gridout.sep)+0.2*max(gridout.sep)], $ 
          THICK=3.0, LINESTYLE=2
    PLOT,gridout.deltamag,gridout.pa,XTITLE=sym_Delta+'mag',YTITLE='PA (deg)', $
      CHARTHICK=2.0,THICK=3.0,XTHICK=3.0,YTHICK=3.0, $
      PSYM=8, SYMSIZE=0.5, XRANGE=[-0.1,4.2], XSTYLE=1, $ 
      SUBTITLE='Global minimum at PA (deg) ='+STRING(gridout.pa[minLoc],FORMAT='(F6.2)')
    OPLOT,[gridout.deltamag[minLoc],gridout.deltamag[minLoc]],[0,200],THICK=3.0, $
          LINESTYLE=2
    DEVICE,/close_file
    set_plot,'X'    
ENDIF

;stop

; Set the returned header-like information
gridout.starname = starname
gridout.npoiid = npoiid
gridout.diam = diam
gridout.riflag = riflag

save,gridout,filename=outprefix+'.gridout.dat'

print,'**********************************'
print,'Grid search for a binary completed.'
print,'Data saved to an idl file called '+outprefix+'.gridout.dat'
print,'**********************************'

end

;********************************************************************

PRO doubleSearch,datafile,deltamag

COMPILE_OPT DEFINT32, STRICTARR, STRICTARRSUBS

COMMON SHAREIT

; Modification of Chris Tycner's binarySearch.pro so that the 
; channel widths are properly included in the binary star 
; visibility model equation.

; RELATED PROGRAM: mdata.pro, cubegridm (above)
; mdata procedure reads in the data from OYSTER CHA files and 
; puts it into a struture appropriate for this and other tasks. 
; This exists as a stand-alone procedure and is a modification 
; of sdata

; cubegridm (above) is the controlling program for the grid 
; search

; PERFORMANCE
; Runtime searching over 251001 rho, theta pairs (501*501mas grid)
; takes 68.47 seconds on alex. This is 0.0003 seconds per rho,theta 
; pair or grid element.
; Chris' original version, neglecting channel widths, takes 
; 39.93 seconds for an identical rho, theta grid

; CHANNEL WIDTHS problem: Some cha files from the TPF survey have 
; a zero channel width. This creates NaN entries when we calculate
; the sinc_factor. I will make a test for zero channel widths 
; and impose a model channel width based on a fit the 
; the channel widths. A 2nd order polynomial fit gives:
; chanwidths = 3.33149e-08-2.35728e-09*(channel_number)+6.24985e-11*(channel_number)^2

; PLOTTING VARIABLES/FLAGS
; psOutput controls whether postscript or X window plots of 
; individual delta-mag results are made.
;
; makedMagPlots controls whether we make those plots at all.
; Skipping them saves about 2 seconds per star for a small 
; [45150 element] grid

pi = !DPI
; make sure the multi flag from cubegridm doesn't impact us here.
!P.MULTI=0

!EXCEPT=0
; Report to user the line number of any math errors. You might get one
; at line containing sincfactor if you pass through the origin. 
; We allow this error and don't fix it to aviod IF loops and tests 
; which will slow the code. Also, if we eventually vectorize entire
; code for speed, we won't be able to use any IF statements or other tests. 

; Define plot symbol Delta,ChiSq

sym_Delta='!7D!X!N' 
sym_ChiSq='!7v!S!E!72!R!B!I!7m!N!X'

LOADCT, 3, /SILENT
; psOutput = 0 makes plots to screen, 1 makes ps plots
psOutput = 0
; Now set a flag for whether to make plots or not
makedMagPlots='no'

file=datafile
RESTORE, FILENAME=file

pick_ch = 30 ; select channel to exclude
index = WHERE(sdata.channel NE pick_ch)
PRINT, 'Excluding ' + STRING(N_ELEMENTS(sdata.v2cd)-N_ELEMENTS(index)) + ' data points.'
PRINT, 'Keeping ' + STRING(N_ELEMENTS(index)) + ' data points.'

num_datapts=N_ELEMENTS(index)

print,'num_datapts is: ',num_datapts

; put the u, v coordinates, calibrated v^2 values into vectors.
; make a vector of the uv radii, channel widths and 
; channel effective wavelengths
u = sdata.u[index]
v = sdata.v[index]
v2 = sdata.v2cd[index]
v2_err = sdata.v2cd_err[index]
uvrad = SQRT(u*u+v*v)
chanw = sdata.chan_width
lam_eff = sdata.uv_wave

; Check for any zero channel widths 
zeroChanInd = WHERE(chanw EQ 0, zerocount)

; Replace zero channel widths with the parabolic fit estimate
IF zerocount NE 0 THEN BEGIN
    chanw[zeroChanInd] = 3.33149e-08 $ 
                         - 2.35728e-9*sdata.channel[zeroChanInd] $ 
                         + 6.24985e-11*sdata.channel[zeroChanInd]^2
ENDIF

; calculate the PA of each u-v point (measured E from N)
basePA = 90.0d - 180.0d/pi*ATAN(v,u) ; in deg

; select the range in mas of x and y space to search
x_min = +0.01d
x_max = +20.01d
y_min = -20.01d
y_max = +20.01d
stepSize = 0.11d
;
; Use this for Xi Tau
x_min = -12.0d
x_max = -06.0d
y_min = +12.0d
y_max = +18.0d
stepSize = 0.1d
;
; Use this for Sig Ori
x_min = +240.0d
x_max = +260.0d
y_min = +00.0d
y_max = +20.0d
stepSize = 0.1d

n_x = FLOOR((x_max - x_min)/stepSize) + 1
n_y = FLOOR((y_max - y_min)/stepSize) + 1
n_total = n_x * n_y
stringy = STRMID(STRCOMPRESS(STRING(stepSize)),0,4)
PRINT, 'Requested region is ' + STRING(n_x) + ' by ' + STRING(n_y)
PRINT, 'I will increment by '+stringy+' mas.'
PRINT, 'with total number of elements = ' + STRING(n_total)

; set up the x and y vectors
x = DBLARR(n_total)
y = DBLARR(n_total)
counter = 0L
FOR i = 0, n_x-1 DO BEGIN
    FOR j = 0, n_y-1 DO BEGIN
        x[counter] = x_min + DOUBLE(i)*stepSize
        y[counter] = y_min + DOUBLE(j)*stepSize
        counter += 1
    END
END

; calculate the PA of x-y point
; Note: x is E and y is N
xyPA = 90.0d - 180.0d/pi*ATAN(y,x)   ; in deg

; calculate separations for each x-y point in radians
xySep = SQRT(x^2 + y^2)/1000d/206265d

; chose binary parameters
; ratio = brightness ratio, d1 and d2 are diameters of
; stars 1 and 2 in mas. mas_to_rad is conversion factor
; mas -> radians
ratio = 10^(-1*deltamag/2.5)
mas_to_rad = 1/(206265*1000.)
diam = sdata.diam
starname = sdata.name
riflag = sdata.riflag
npoiid = sdata.npoiid 
d1=sdata.diam
d2=d1*0.5
d1rad = d1*mas_to_rad
d2rad = d2*mas_to_rad
; Calculate V^2 vector for the two stars
arg_1 = pi*uvrad*d1rad
arg_2 = pi*uvrad*d2rad
vstar1 = 2*beselj(arg_1,1)/arg_1
vstar2 = 2*beselj(arg_2,1)/arg_2

; Start gridding

PRINT, 'Gridding chi^2 space...'

; calculate ch^2 for each x-y point
chi2 = DBLARR(n_total)
s = uvrad
; before loop convert xyPA and basePA to degrees
basePA = pi*basePA/180d
xyPA = pi*xyPA/180d
FOR k = 0, n_total-1 DO BEGIN
    IF k MOD 400000 EQ 0 THEN PRINT,'k = ',k
    proj_sep = xySep[k]*COS(xyPA[k]-basePA)
    sinc_arg = pi*chanw*uvrad*proj_sep/lam_eff
    sinc_factor = sin(sinc_arg)/sinc_arg
    v2binary_model = $
      (vstar1^2 + (sinc_factor*ratio*vstar2)^2 + $ 
                   2d*sinc_factor*ratio*vstar1*vstar2*COS(2d*pi*s*proj_sep))/((1d)+ratio)^2
    chi2[k] = TOTAL((((v2 - v2binary_model)^2)/(v2_err^2))/num_datapts, /DOUBLE)
END
index_min = WHERE(chi2 EQ MIN(chi2))
PRINT,'*************************************'
PRINT,'*****Binary grid search for '+starname
PRINT,'*****For a delta-mag of '+STRCOMPRESS(STRING(deltamag))
PRINT,'*************************************'
atx = STRTRIM(STRING(x[index_min]),1)
aty = STRTRIM(STRING(y[index_min]),1)
printMin = STRTRIM(STRING(MIN(chi2)),1)
PRINT, 'Min reduced chi^2 of ' + printMin +' found at x=' + atx +' y=' + aty
PRINT, 'This corresponds to Sep =' + STRING(SQRT(x[index_min]^2 + y[index_min]^2)) + $
  ' and PA of ' + STRING(90.0d - 180.0d/pi*ATAN(y[index_min],x[index_min]))

; set-up variables for determining sep and pa and for plotting
plot_x = (DINDGEN(n_x)/DOUBLE(n_x-1))*(x_max-x_min)+x_min
plot_y = (DINDGEN(n_y)/DOUBLE(n_y-1))*(y_max-y_min)+y_min
plot_chi2 = DBLARR(n_x, n_y)
counter = 0L
FOR i = 0, n_x-1 DO BEGIN
    FOR j = 0, n_y-1 DO BEGIN
        plot_chi2[i,j] = chi2[counter]
        counter += 1
    END
END

; Set some variables used for printing strings, and via the 
; COMMON statement returning variable values to cubegrid procedure.

subname=STRING(deltamag,FORMAT='(F3.1)')
theMin=MIN(chi2)
StrtheMin=STRCOMPRESS(theMin)
theMax=MAX(chi2)
StrtheMax=STRCOMPRESS(theMax)
rho_0=SQRT(x[index_min[0]]^2 + y[index_min[0]]^2)
pa_0=90.0d - 180.0d/pi*ATAN(y[index_min[0]],x[index_min[0]])
sep=rho_0
pa=pa_0
; Use a flag to indicate whether two minima in ChiSq are found
    min_flag=n_elements(index_min) - 1
IF makedMagPlots EQ 'yes' THEN BEGIN

    IF ~psOutput THEN WINDOW, 1, XSIZE=700, YSIZE=700, TITLE='Chi^2 Space'
    IF psOutput THEN SET_PLOT, 'ps'
    IF psOutput THEN DEVICE, /COLOR, BITS_PER_PIXEL=8
    IF psOutput THEN DEVICE, FILENAME=datafile+'.'+subname+'.ps', ENCAPSULATED=0
    IF psOutput THEN DEVICE, XSIZE=7, YSIZE=7, XOFFSET=0.5, YOFFSET=3, /INCHES
    IF psOutput THEN pick_thick=4 ELSE pick_thick=1
;Test for two equal minima, if there are two plot both positions
    IF min_flag EQ 1 THEN BEGIN
        rho_1=SQRT(x[index_min[1]]^2 + y[index_min[1]]^2)
        pa_1=90.0d - 180.0d/pi*ATAN(y[index_min[1]],x[index_min[1]])
        CONTOUR, plot_chi2, plot_x, plot_y,  XRANGE=[MAX(plot_x),MIN(plot_x)], NLEVELS=15, $
          TITLE=file, SUBTITLE='Min reduced '+sym_ChiSq+' of '+StrtheMin+' found at sep of '+STRCOMPRESS(rho_0) + $
          ' and PA of ' + STRCOMPRESS(pa_0) + $
          '!C and at sep of '+STRCOMPRESS(rho_1) + $
          ' and PA of ' + STRCOMPRESS(pa_1) + $
          '!C Maximum '+sym_ChiSq+ ' = '+StrtheMax + $
          '!C '+sym_Delta+'mag ='+STRING(deltamag,FORMAT='(F6.2)')+', D1 ='+STRING(d1,FORMAT='(F6.2)') + $
          ' mas, D2 ='+STRING(d2,FORMAT='(F6.2)')+' mas.', $  
          XSTYLE=1, YSTYLE=1,C_ANNOTATION=''
        OPLOT, x[index_min], y[index_min], PSYM=1, THICK=pick_thick
        IF psOutput THEN DEVICE, /CLOSE_FILE
ENDIF

; If only one minima found make this plot
;IF min_flag EQ 0 THEN BEGIN
; CONTOUR, plot_chi2, plot_x, plot_y, XRANGE=[MAX(plot_x),MIN(plot_x)], NLEVELS=10, $
;  TITLE=file, SUBTITLE='Min '+sym_ChiSq+' of '+StrtheMin+' found at sep of '+STRCOMPRESS(rho_0) + $
;  ' and PA of ' + STRCOMPRESS(pa_0) + $
;  '!C Maximum '+sym_ChiSq+ ' = '+StrtheMax + $
;  '!C '+sym_Delta+'mag ='+STRING(deltamag,FORMAT='(F6.2)')+', D1 ='+STRING(d1,FORMAT='(F6.2)') + $
;  ' mas, D2 ='+STRING(d2,FORMAT='(F6.2)')+' mas.', $  
;  XSTYLE=1, YSTYLE=1, C_ANNOTATION=''
; OPLOT, x[index_min], y[index_min], PSYM=1, THICK=pick_thick
; IF psOutput THEN DEVICE, /CLOSE_FILE
;ENDIF

SET_PLOT, 'X'

ENDIF

;IF deltamag GT 9.1 THEN save,chi2,filename='chi2.9.5.dat' 
;IF deltamag GT 4.6 THEN stop

END

