PRO extract_cha, file

;
; NOTE: before the data can be read by this program an oyster .cha
; file should be read in by oyster (get_data, '2004-12-03.cha') and
; then the relevant data should be extracted to a seperate IDL file
; via (SAVE, FILENAME='2004-12-03-all-scans.dat', /ALL).
;
  
; On 2012-08-22 Bob Z. and Jason S. modified the original
; extract_cha.pro file on how UT_sec is handled to avoid numbers that
; are greater than what INT type can handle.
; This modification resolves 'arithmetic error: floating
; illegal operand' due to FIX requiring a type code = 3
; for a longword integer.

;
; Note: it is assumed that the file is <file>-all-scans.dat
;
common StarBase,startable,notes
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
file = STRCOMPRESS(STRING(file), /REMOVE_ALL)
READS, STRMID(file, 5, 2), month, FORMAT='(I02)'
READS, STRMID(file, 8, 2), day, FORMAT='(I02)'
;
; Change the working directory
; 
CD, '/Users/tycne1c/Work/Besa'
CD, file
;
; CHU: execute IDL internal function mean before RESTORE
init_mean=mean([1,2,3,4])
;
; RESTORE, FILENAME=file+'-all-scans.dat'
RESTORE, FILENAME=file+'-all-scans.dat'
;
; CHU: reset environment variables to my OYSTER directories
; !oyster_dir='/home/chummel/oyster/'
; !atmospheres_dir='/home/chummel/oyster/atmospheres/'
; !catalogs_dir='/home/chummel/oyster/catalogs/'
; !evolution_dir='/home/chummel/oyster/evolution/'
; !external_lib='/home/chummel/oyster/linux/oyster.so'
;
; Determine the number of elements in the structured array
;
no_of_scans=N_ELEMENTS(scans)
;
; Setup the array that will hold the star and scan names
;
star_list=STRARR(no_of_scans)
scan_list=STRARR(no_of_scans)

;
; Determine how many input beams (i.e. elements) there are
; Note: the dimensions of scantable[i].station cannot be used, nor
; 	the number of 1's in the array can be trusted
;
counter=0
;
; Count the station IDs
;
FOR i=0, (N_ELEMENTS(GenInfo.StationID)-1) DO BEGIN
	IF (GenInfo.StationID[i] NE '') THEN counter=counter+1
END
n_way=counter

;;
;; We now assume that for n-way data only the first n elements in
;; scantable[i].station are relevant
;;
IF ((n_way NE 2) AND (n_way NE 3) AND (n_way NE 4) AND (n_way NE 5)) THEN BEGIN
	PRINT, 'The log file is setup to handle 2, 3, 4, or 5 elements only!'
	STOP
ENDIF

;;
;; Open new log file - if file already exists it is deleted
;;
OPENW, 1, file+'-all-scans.log'
PRINTF, 1, 'Log file created on : ', SYSTIME()


FOR i=0, (no_of_scans-1) DO BEGIN
    star_list[i]=scans[i].starid
    ;; Calculate the H:M:S for the UT time (that is in s)
    UT_hr = FIX(scans[i].time/3600.0)
    UT_min = FIX(scans[i].time/60.0)-60*UT_hr
    ;; keep seconds in LONG INT format until needed, then convert to
    ;; simple INT for consistency
    UT_sec = FIX(FIX(scans[i].time, TYPE=3)-LONG(3600)*UT_hr-LONG(60)*UT_min)

    PRINTF, 1, '----------------------------------------------'
    PRINTF, 1, STRING( $
       FORMAT='("Scan: ", I3.3, "   UT Time = ", I2.2, ":", I2.2, ":", I2.2, "   Star: ", A)', $
          scans[i].iscan, UT_hr, UT_min, UT_sec, scans[i].starid)

    CASE n_way OF
       2: BEGIN
          PRINTF, 1, STRING(FORMAT='("Code: ", I1, TR25, "2-Way: ", 2(I1))', $
                            scantable[i].code, scantable[i].station[0:n_way-1])
       END
       3: BEGIN
          PRINTF, 1, STRING(FORMAT='("Code: ", I1, TR25, "3-Way: ", 3(I1))', $
                            scantable[i].code, scantable[i].station[0:n_way-1])
       END
       4: BEGIN
          PRINTF, 1, STRING(FORMAT='("Code: ", I1, TR25, "4-Way: ", 4(I1))', $
                            scantable[i].code, scantable[i].station[0:n_way-1])
       END
       5: BEGIN
          PRINTF, 1, STRING(FORMAT='("Code: ", I1, TR25, "5-Way: ", 5(I1))', $
                            scantable[i].code, scantable[i].station[0:n_way-1])
       END
    ENDCASE

    PRINTF, 1, 'HA  : ', scans[i].ha, '  ZA  : ', scans[i].za
    PRINTF, 1, 'RA  : ', scans[i].ra, '  DEC : ', scans[i].dec
    PRINTF, 1, 'AZ  : ', scans[i].az, '  MA  : ', scans[i].ma

    rawuvw=scans[i].uvw
    rawv2=scans[i].vissq
    rawv2e=scans[i].vissqerr
    calv2=scans[i].vissqc
    calv2e=scans[i].vissqcerr
    raw3a=scans[i].tripleamp
    raw3ae=scans[i].tripleamperr
    cal3a=scans[i].tripleampc
    cal3ae=scans[i].tripleampcerr
    raw3p=scans[i].triplephase
    raw3pe=scans[i].triplephaseerr
    cal3p=scans[i].triplephasec
    cal3pe=scans[i].triplephasecerr

    photon_rate = scans[i].photonrate
    photon_rate_err = scans[i].photonrateerr
    bg_rate = scans[i].backgndrate
    bg_rate_err = scans[i].backgnderr
    nat_counts = scans[i].natcounts
    nat_counts_err = scans[i].natcountserr

    station_flags=scantable[i].station[0:n_way-1]
    scan_code=scantable[i].code

    valid_base_per_sp = GenInfo.numbaseline
    n_outputbeams = GenInfo.numoutbeam
    geninfo_baselineid = GenInfo.BaselineID
    genconfig_stationid = GenConfig.StationID
    genconfig_delaylineid = GenConfig.DelayLineID
    stationcoord = GenConfig.stationcoord
    numtriple = GenConfig.numtriple
    triplebase = GenConfig.triplebase
    triplebeam = GenConfig.triplebeam

    ;; get the list of spectrometers used
    spectrometers_text = GenConfig.spectrometerid
    ;; convert the text to integers
    spectrometers_numbers = INTARR(N_ELEMENTS(spectrometers_text))
    temp_spec = 0

    FOR j=0, N_ELEMENTS(spectrometers_text)-1 DO BEGIN
       ;; note: some older data (from 2000) might not contain any
       ;; information about the spectrometers.  In that case the
       ;; GenConfig.spectrometerid might contain strings 'DummyID'.
       ;; Test for presence of these strings before attempting to
       ;; convert the text number into an integer.
       IF (spectrometers_text[j] EQ 'DummyID') THEN BEGIN
          ;; in the case of old data assume that the spectrometers are
          ;; listed in ascending order, 1, 2, 3.  Older data was in
          ;; 3-way and used ALL 3 output beams.
          spectrometers_numbers[j] = j+1
       ENDIF ELSE BEGIN
          READS, spectrometers_text[j], temp_spec, FORMAT='(I1)'
          spectrometers_numbers[j] = temp_spec
       ENDELSE
    ENDFOR    

    star_name=scans[i].starid
    star_ha=scans[i].ha
    star_ra=scans[i].ra
    star_dec=scans[i].dec
    star_za=scans[i].za
    
    ;
    ; Check how many spectral channels there are
    ;
    IF (GenConfig.numspecchan[0] EQ 0 ) THEN BEGIN
	PRINT, 'ERROR: There are no valid spectral channels!'
	PRINT, 'Please check your data.'
	STOP
    ENDIF
    wavelengths = GenConfig.wavelength[0:GenConfig.numspecchan[0]-1]*1e9
    wavelengths_err = GenConfig.wavelengtherr[0:GenConfig.numspecchan[0]-1]*1e9

    ;; export the wavelengths for the 2nd output for comparison
    ;; reasons as well
    wavelengths2 = GenConfig.wavelength[*, 1]*1e9
    wavelengths2_err = GenConfig.wavelength[*, 1]*1e9

    file_name=STRING(FORMAT='(A, "-", A, "-scan", I3.3, ".idl")', scans[i].starid, file ,i+1)
    scan_list[i]=file_name

    SAVE, FILENAME=file_name, $
          rawuvw,rawv2,rawv2e,calv2,calv2e, $
          raw3a,raw3ae,cal3a,cal3ae,raw3p,raw3pe,cal3p,cal3pe, $
          station_flags, scan_code, $
          valid_base_per_sp, n_outputbeams, $
          geninfo_baselineid, genconfig_stationid, genconfig_delaylineid, $
          stationcoord, numtriple, triplebase, triplebeam, $
          wavelengths, wavelengths_err, $
          wavelengths2, wavelengths2_err, $
          star_name, star_ha, star_ra, star_dec, star_za, $
          UT_hr, UT_min, UT_sec, $
          photon_rate, photon_rate_err, bg_rate, bg_rate_err, $
          spectrometers_numbers

    ;; Create a dir for each star if it was not created yet
    IF (FILE_TEST(scans[i].starid, /DIRECTORY) NE 1) THEN FILE_MKDIR, scans[i].starid

    ;; Make a copy of the scan file and place it in /ALL directory
;   all_dir = '/Users/tycne1c/Work/Besa/ALL/' + scans[i].starid
    all_dir = scans[i].starid
    ;; create a star dir in ALL if necessary
    IF (FILE_TEST(all_dir, /DIRECTORY) NE 1) THEN FILE_MKDIR, all_dir
    FILE_COPY, file_name, all_dir, /OVERWRITE, /REQUIRE_DIRECTORY

    ;; Move the file to its appropriate directory
    FILE_MOVE, file_name, scans[i].starid, /OVERWRITE, /REQUIRE_DIRECTORY


END
CLOSE, 1

;
; Create READ_FILES.txt files in each star directory
;
PRINT, 'Scans for the following stars were found:'
FOR i=0, (no_of_scans-1) DO BEGIN
   IF( star_list[i] NE 'DONE') THEN BEGIN
	PRINT, star_list[i]
	index=WHERE(star_list EQ star_list[i])

	OPENW, 2, 'READ_FILES.txt'
	PRINTF, 2, TRANSPOSE(scan_list[index])
	CLOSE, 2

	FILE_MOVE, 'READ_FILES.txt', star_list[i], /OVERWRITE, /REQUIRE_DIRECTORY
	; 
	; Flag which stars were done
	;
	star_list[index]='DONE'
   ENDIF
END

PRINT, 'Extraction of data for ', file, ' completed.'

END
