;*******************************************************************************
; File: pearl.pro
;
; Description:
; ------------
; Container of IDL/PV-WAVE CL scripts for imaging interferometric data.
; PEARL implements an algorithm for broadband aperture synthesis by
; assigning effective temperatures to CLEAN components. This works well in
; the case of composite spectrum binaries. It should work in all circumstances
; where blackbody emission regions can be defined across an image.
;
; Ideas on how to implement the T fitting tool:
; For a given pixel, adopt a T. Compare predicted brightness with
; actual brightness (lambda) and derive calibration factor C.
; This could be done for each channel, but in the initial step, since
; all pixels have the same T, the new T would not depend on channel.
; If T was correct, then all other wavelengths, using the same C,
; would reproduce the SED of this pixel.
; Given a C, compute for pixels of a different brightness the T (might
; be ambiguous!). 
; Instead of using the pixels, it is best to convolve the CM with a beam
; at least the size of the CB.
; Then iterate once more. Now the Teff computation is done in each channel.
; If T was correct, and we have a good map, then the pixel T should be 
; same in each channel.
;
; Problem: pixels also have a "diameter"
; Procedure: 
; Fit photometry and derive Teff, assign to brightest pixel, the "reference".
; Assign diameter=1 to all pixels above threshold 5% of maxmimum.
; Compute teff for all pixels based on flux relative to reference pixel.
; Rescale flux of each pixel given teff 
; Compute chisq
; Cycle through all diameter combinations and find the best chisq
;
; Block directory:
; ----------------
; Block 1: beampatch,dirtybeam,cleanbeam,displaybeam,contourbeam
; Block 2: set_imagedata,pearlimage,dft,mft,cleankernel,cleanwindow,clean,
;          residualmap,selfcal,tvdb,tvrm,tvfm,tvcc,tvrg,tkfm,
;          clean_event,pearl_event,pearl
; Block 3: pearson
; Block 4: imagecube,teffmap
;
;************************************************************************Block 1
function beampatch,beam,level=level,blank=radius
;
; Return a map with the beam patch, i.e. all pixel values greater than 1/5
; of the maximum set to 1, all others set to 0.
;
if n_elements(level) eq 0 then level=0.2
if n_elements(radius) eq 0 then radius=0
;
imsize=n_elements(beam(*,0))
jmsize=n_elements(beam(0,*))
;
if radius ne 0 then begin
	b=shift(dist(imsize,jmsize),imsize/2,jmsize/2)
	b=-b+max(b)
	index=where(b lt sqrt(2)*((imsize+jmsize)/4-radius))
	beam(index)=0
endif
;
b=fltarr(imsize+2,jmsize+2)
patch=b
b(1:imsize,1:jmsize)=beam
bmax=max(b)
blim=bmax*level
;
i0=imsize/2+1
j0=jmsize/2+1
;
; Don't assume peak is at center
index=where(beam eq max(beam))
i0=index(0) mod imsize
j0=index(0) / jmsize
;
i=0l
repeat begin
	j=0l
	while b(i0+i,j0-j) gt blim do begin
		patch(i0+i,j0-j)=1
		j=j+1
	endwhile
	j1=j
	j=1l
	while b(i0+i,j0+j) gt blim do begin
		patch(i0+i,j0+j)=1
		j=j+1
	endwhile
	j2=j
	j0=j0+(j2-j1)/2
	i=i+1
endrep until b(i0+i,j0) le blim
;
i=1l
j0=jmsize/2+1
repeat begin
	j=0l
	while b(i0-i,j0-j) gt blim do begin
		patch(i0-i,j0-j)=1
		j=j+1
	endwhile
	j1=j
	j=0l
	while b(i0-i,j0+j) gt blim do begin
		patch(i0-i,j0+j)=1
		j=j+1
	endwhile
	j2=j
	j0=j0+(j2-j1)/2
	i=i+1
endrep until b(i0-i,j0) le blim
;
return,patch(1:imsize,1:jmsize)
;
end
;-------------------------------------------------------------------------------
function dirtybeam,star
;
; Return the synthesized beam for a given star and observation.
;
common BeamInfo,reduce
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(star) eq 0 then begin
	print,'***Error(DIRTYBEAM): no star specified!'
	return,-1
endif
index=where(scans.starid eq star,count)
if count eq 0 then begin
	print,'***Error(DIRTYBEAM): no data for this star!'
	return,-1
endif
;
imsize=512
reduce=2e6*max(abs(scans(index).uvw))/70e6
map=fltarr(imsize,imsize)
;
; This is the editing for the beam used in mapping
; set_complexvis
; e=reform(scans(index).complexweight,n_elements(scans(index).complexweight))
;
e=reform(scans(index).vissqcerr,n_elements(scans(index).vissqerr))
v=reform(scans(index).uvw(*,*,*,1),n_elements(scans(index).uvw(*,*,*,1)))
u=reform(scans(index).uvw(*,*,*,0),n_elements(scans(index).uvw(*,*,*,0)))
; If you want to compare the beam to the one in AIPS, flip the sign of u
u=-u
;
; Edit the uv-coverage
index=where(e gt 0,count)
if count eq 0 then begin
	print,'***Error(DIRTYBEAM): no data for this star!'
	return,-1
endif
u=u(index) & v=v(index)
;
; Place the data on a regular grid and add coordinates symmetric to origin
index=where(u lt 0,count) & if count gt 0 then u(index)=u(index)-reduce/2
index=where(u gt 0,count) & if count gt 0 then u(index)=u(index)+reduce/2
index=where(v lt 0,count) & if count gt 0 then v(index)=v(index)-reduce/2
index=where(v gt 0,count) & if count gt 0 then v(index)=v(index)+reduce/2
u=fix(u/reduce) & u=[u,-u]
v=fix(v/reduce) & v=[v,-v]
;
; Flip negative spatial frequencies
index=where(u lt 0,count) & if count gt 0 then u(index)=imsize+u(index)
index=where(v lt 0,count) & if count gt 0 then v(index)=imsize+v(index)
;
; Copy the grid into the map
map(u,v)=1
;
; Compute and return the beam
return,shift(float(fft(map,1)),imsize/2,imsize/2)
;
end
;-------------------------------------------------------------------------------
function cleanbeam,beam
;
; Return major and minor axis, and position angle [deg] of FWHM clean beam
; ellipse.
;
common BeamInfo,reduce
common FitEllipse,ex,ey
;
RAD=180/!pi
;
m=max(beam)
index=where(beam gt m/2-m/20 and beam lt m/2+m/20 and beampatch(beam) eq 1, $
		ndata)
;
if ndata lt 10 then begin
	print,'***Error(CLEANBEAM): too few half-maximum points!'
	return,-1
endif
;
imsize=n_elements(beam(0,*))
;
; Set data.
x=(index mod imsize)*(180/!pi)/(imsize*reduce)*3600e3 & x=x-avg(x)
y=(index/imsize)*(180/!pi)/(imsize*reduce)*3600e3 & y=y-avg(y)
;
r=sqrt(x^2+y^2)
p=!pi-atan(x,y)
;
; Set parameters to fit
ma=5
ia=lonarr(ma)+1 & ia(0)=0 & ia(1)=0
index=where(ia ne 0,mfit)
e_parms=fltarr(ma)
e_parms(0)=0
e_parms(1)=0
e_parms(2)=max(r)
e_parms(3)=min(r)
i=where(r eq max(r)) & i=i(0)
e_parms(4)=p(i)
if e_parms(4) gt !pi then e_parms(4)=e_parms(4)-!pi
a=e_parms
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
;
; Call marquardt
sig=fltarr(ndata)+0.1
pa=p	; ellfuncs changes p!
marquardt,'ellfuncs',pa,r,sig,ndata,a,ia,ma,chisq,covar
;
; Plot the half-power points and fit
; plot,-x,y,psym=1
; r=ellipse(a(2),a(3),p-a(4))
; oplot,r*sin(p),r*cos(p),psym=3
;
; Set new parameters
e_parms=a
e_parms(2)=a(2)*2
e_parms(3)=a(3)*2
e_parms(4)=a(4)*RAD
;
return,e_parms(2:4)
;
; Print inital estimates to screen
print,'__________________________________'
print,'Center x = ',e_parms(0)
print,'Center y = ',e_parms(1)
print,'Semi-major axis = ',e_parms(2)
print,'Semi-minor axis = ',e_parms(3)
print,'Position  angle = ',e_parms(4)*RAD
print,'_______________***________________'
;
end
;-------------------------------------------------------------------------------
pro displaybeam,star
;
; Display TV image of dirty beam.
;
beam=dirtybeam(star)
if beam(0) eq -1 then return
imsize=n_elements(beam(0,*))
window,/free,xsize=imsize,ysize=imsize
; loadct,5
loadct,0
tvscl,beam
;
end
;-------------------------------------------------------------------------------
pro contourbeam,star
;
; Plot contours of dirty beam.
;
common BeamInfo,reduce
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
beam=dirtybeam(star)
if beam(0) eq -1 then return
;
imsize=n_elements(beam(0,*))
x=(findgen(imsize)-imsize/2)*(180/!pi)/(imsize*reduce)*3600e3
y=x
plevels=[5,10,20,50,80]
nlevels=[-80,-50,-20,-10,-5]
levels=[nlevels,plevels]
levels_label=''
for i=0,n_elements(levels)-1 do $
	levels_label=levels_label+string(levels(i),format='(i4)')
contour,beam,x,y,levels=plevels*max(beam)/100,xstyle=1,ystyle=1, $
	xtitle='Contours: '+levels_label+' % of peak',ytitle='[mas]', $
	title='Dirty beam for '+star+' on '+date
contour,beam,x,y,levels=nlevels*max(beam)/100,c_linestyle=1,/overplot
oplot,[0,0],[y(0),y(imsize-1)],psym=0,linestyle=1
oplot,[x(0),x(imsize-1)],[0,0],psym=0,linestyle=1
;
end
;************************************************************************Block 2
pro set_imagedata,star,status
;
; Initialize imaging data.
;
; ov: observed (complex) visibility, never changed
; ow: visibility weight
; uc: u-coordinate
; vc: v-coordinate
; cv: corrected visibility, initially copy of ov, then phase calibrated
; mv: model visibility
; si: scan index
; bi: baseline index, e.g. 12, 56, no more than 9 stations
; ci: channel index
; wl: wavelength [nm]
; bw: bandwidth [nm]
; fl: total flux = SED, read from scans.photometry
; db: dirty beam, cube
; cb: clean beam, not a cube
; dm: dirty map, not a cube
; rm: residual map, not a cube
; cm: clean map, not a cube, accumulates all cc
; cc: clean components corresponding to one run of CLEAN, not a cube
; rt: temperature map
; rg: log(g) map
; rf: stellar atmosphere flux normalized by observed SED
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(star) eq 0 then begin
        print,'***Error(SET_IMAGEDATA): no star specified!'
	status=-1
        return
endif
;
for night=0,n_elements(bufferinfo)-1 > 0 do begin
if n_elements(bufferinfo) gt 1 then $
loadnight,GenInfo(night(0)).date, $
	  GeoInfo(night(0)).systemid, $
	  GenInfo(night(0)).configid
if night eq 0 then stationids=genconfig.stationid(0:genconfig.numsid-1) $
	      else stationids=[stationids, $
			      genconfig.stationid(0:genconfig.numsid-1)]
endfor
stationids=unique(stationids)
;
sio=0
nci=1
;
FOR night=0,n_elements(bufferinfo)-1 > 0 DO BEGIN
;
if n_elements(bufferinfo) gt 1 then $
loadnight,GenInfo(night(0)).date, $
	  GeoInfo(night(0)).systemid, $
	  GenInfo(night(0)).configid
;
index=where(scans.starid eq star,count)
if count eq 0 then begin
        print,'***Error(SET_IMAGEDATA): no scans on this star for this date!'
	status=-1
        return
endif
;
; These are reformed into flat arrays
ov=reform(scans(index).complexvis,n_elements(scans(index).complexvis))
ow=reform(scans(index).complexweight,n_elements(scans(index).complexweight))
uc=reform(scans(index).uvw(*,*,*,0),n_elements(scans(index).uvw(*,*,*,0)))
vc=reform(scans(index).uvw(*,*,*,1),n_elements(scans(index).uvw(*,*,*,1)))
;
; Create scan index, has same dimensions as ov, i.e. si(ob,ch,bl,sc)
si=fix(scans(index).complexvis)
for i=0,count-1 do si(*,*,*,i)=i+1+sio
si=reform(si,n_elements(si))
sio=sio+count
;
; Create baseline index, has same dimensions as ov, i.e. bi(ob,ch,bl,sc)
baselineid=intarr(genconfig.numoutbeam,max(genconfig.numbaseline))
for i=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline(i)-1 do begin
;       m=where(genconfig.stationid eq strmid(genconfig.baselineid(l,i),0,3))+1
;       n=where(genconfig.stationid eq strmid(genconfig.baselineid(l,i),4,3))+1
        m=where(stationids eq strmid(genconfig.baselineid(l,i),0,3))+1
        n=where(stationids eq strmid(genconfig.baselineid(l,i),4,3))+1
        baselineid(i,l)=10*m+n
endfor
endfor
bi=fix(scans(index).complexvis)
for j=0,max(genconfig.numspecchan)-1 do begin
	for i=0,count-1 do bi(*,j,*,i)=baselineid
endfor
bi=reform(bi,n_elements(bi))
;
; Create channel index, has same dimensions as ov, i.e. ci(ob,ch,bl,sc)
; Assume for now that channel layout is same in every OB!
ci=fix(scans(index).complexvis)
spectrometers=strarr(genconfig.numoutbeam)
for i=0,genconfig.numoutbeam-1 do $
	spectrometers(i)=strjoin(string( $
		genconfig.wavelength(0:genconfig.numspecchan(i)-1,i)*1e6))
uspms=unique(spectrometers)
for i=0L,n_elements(uspms)-1 do begin
	k=where(spectrometers eq uspms(i))
	wl0=genconfig.wavelength(0:genconfig.numspecchan(i)-1,k(0))*1e9
	bw0=genconfig.chanwidth(0:genconfig.numspecchan(i)-1,k(0))*1e9
	fl0=scans(index(0)).photometry(0,k(0),0:genconfig.numspecchan(i)-1)
	if night eq 0 and i eq 0 then wl=wl0 else wl=[wl,wl0]
	if night eq 0 and i eq 0 then bw=bw0 else bw=[bw,bw0]
	if night eq 0 and i eq 0 then fl=fl0 else fl=[fl,fl0]
	for j=0L,genconfig.numspecchan(i)-1 do begin
		ci(k,j,*,*)=j+nci
	endfor
	nci=nci+genconfig.numspecchan(i)
endfor
ci=reform(ci,n_elements(ci))
; Old code
if 0 then begin
ci=fix(scans(index).complexvis)
for j=0,max(genconfig.numspecchan)-1 do ci(*,j,*,*)=j+1
ci=reform(ci,n_elements(ci))
wl=1e9*total(genconfig.wavelength,2)/genconfig.numoutbeam
bw=1e9*total(genconfig.chanwidth,2)/genconfig.numoutbeam
endif
;
; Edit the uv-coverage
index=where(ow gt 0,count)
if count eq 0 then begin
        print,'***Error(SET_IMAGEDATA): no data for this star!'
	status=-1
        return
endif
uc=float(uc(index))
vc=float(vc(index))
ov=ov(index)
ow=ow(index)
bi=bi(index)
ci=ci(index)
si=si(index)
;
if night eq 0 then begin
	uc0=uc
	vc0=vc
	ov0=ov
	ow0=ow
	bi0=bi
	ci0=ci
	si0=si
endif else begin
	uc0=[uc0,uc]
	vc0=[vc0,vc]
	ov0=[ov0,ov]
	ow0=[ow0,ow]
	bi0=[bi0,bi]
	ci0=[ci0,ci]
	si0=[si0,si]
endelse
;
ENDFOR
;
uc=uc0
vc=vc0
ov=ov0
ow=ow0
bi=bi0
ci=ci0
si=si0
;
cv=ov
mv=ov*0
;
status=0
;
end
;-------------------------------------------------------------------------------
function pearlimage,imsze,cellsze,nc,relax=relax
;
; Return an image of dimension imsize(0)*imsize(1)*nc, i.e. column by row,
; width by height, and cellsize in mas. The image coordinates x and y
; are measured in radians, and x increases to the left (east) just as
; the right ascension does.
;
; Both dimensions must be odd numbers, so that there will be a pixel on the
; center of the image. The bottom left corner is at pixel (0,0), the top right
; corner is (imsize(0)-1,imsize(1)-1)
;
; This is the special PEARL image format including effective temperatures.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
MAS2RAD=pi_circle/(180l*3600000l)
;
if n_elements(relax) eq 0 then relax=0
if n_elements(imsze) eq 1 then imsze=[imsze,imsze]
;
if not relax then begin
if imsze(0) mod 2 ne 1 then begin
        print,'***Error(SET_MAP): imsize(0) must be odd number!'
        return,-1
endif
if imsze(1) mod 2 ne 1 then begin
        print,'***Error(SET_MAP): imsize(1) must be odd number!'
        return,-1
endif
endif
if n_elements(nc) eq 0 then nc=1
;
imsize=long(imsze)
cellsize=float(cellsze*MAS2RAD)
;
map=fltarr(imsize(0),imsize(1),nc)	; data cube
teff=fltarr(imsize(0),imsize(1))
logg=fltarr(imsize(0),imsize(1))
x=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
y=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
x=-x*cellsize	; RA increases to the left (East)
y=+y*cellsize
;
return,{map:map,teff:teff,logg:logg,x:x,y:y,scale:1.0,zero:0.0}
;
end
;-------------------------------------------------------------------------------
function pearlflux,teff,clambda,dlambda
;
; Average blackbody flux over rectangular band pass
; defined through center wavelength clambda [nm] and full width dlambda [nm].
;
n=n_elements(clambda)
flux=fltarr(n)+1
for i=0,n-1 do begin
        l=findgen(fix(dlambda(i))+1)+clambda-dlambda/2
        flux(i)=avg(blackbody(teff,l))
endfor
;
return,flux
;
end
;-------------------------------------------------------------------------------
function set_uvc,imsze,cellsze,nc,relax=relax
;
; Return a uv-coverage of dimension imsize(0)*imsize(1), i.e. column by row,
; width by height, and cellsize in Mlambda.
;
; Both dimensions must be odd numbers, so that there will be a pixel on the
; center of the image. The bottom left corner is at pixel (0,0), the top right
; corner is (imsize(0)-1,imsize(1)-1)
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
MAS2RAD=pi_circle/(180l*3600000l)
;
if n_elements(relax) eq 0 then relax=0
;
if not relax then begin
if imsze(0) mod 2 ne 1 then begin
        print,'***Error(SET_UVC): imsize(0) must be odd number!'
        return,-1
endif
if imsze(1) mod 2 ne 1 then begin
        print,'***Error(SET_UVC): imsize(1) must be odd number!'
        return,-1
endif
endif
if n_elements(nc) eq 0 then nc=1
;
imsize=long(imsze)
cellsize=float(cellsze)
;
map=complexarr(imsize(0),imsize(1),nc)
u=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
v=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
u=-u*cellsize
v=+v*cellsize
;
return,{map:map,u:u,v:v}
;
end
;-------------------------------------------------------------------------------
function set_scale,image,index
;
if n_elements(index) eq 0 then index=32
;
r=size(image)
if r(n_elements(r)-2) eq 8 then map=image.map else map=image
maxm=max(map)
minm=min(map)
scale=(255-index)/(maxm-minm)
zero=index-minm*scale
;
if r(n_elements(r)-2) eq 8 then begin
	image.zero=zero
	image.scale=scale
	return,0
endif
;
return,[scale,zero]
;
end
;-------------------------------------------------------------------------------
function dft,map,c,u,v,j,db=db
;
; Given a map and complex visibilities at coordinates (u,v), add to the map
; the direct Fourier transform of the visibilities evaluated at the map 
; coordinates. Zero the map before if needed.
;
; Index j specifies which plane of the map to compute.
; 
if n_elements(j) eq 0 then j=0
if n_elements(db) eq 0 then db=0
;
m=map
;
ur=-2*!pi*u
vr=-2*!pi*v
;
cr=float(c)/n_elements(c)
ci=imaginary(c)/n_elements(c)
;
if db then begin
;
	dbxsize=n_elements(m.map(*,0,0))
	dbysize=n_elements(m.map(0,*,0))
;
	k=dbysize/2+1
	l=k*dbxsize-1
;
	m2=m.map(*,0:k-1,j)
	x=m.x(0:l)
	y=m.y(0:l)
	for i=0L,n_elements(c)-1 do begin
        	p=ur(i)*x+vr(i)*y
        	m2=m2+cr(i)*cos(p)
	endfor
;
	m.map(*,0:k-1,j)=m2
	m.map(*,k:dbysize-1,j)=reverse(reverse(m.map(*,0:k-2,j),2),1)
;
endif else begin
;
	m2=m.map(*,*,j)
	for i=0L,n_elements(c)-1 do begin
        	p=ur(i)*m.x+vr(i)*m.y
        	m2=m2+(cr(i)*cos(p)-ci(i)*sin(p))
	endfor
	m.map(*,*,j)=m2
;
endelse
;
return,m
;
end
;-------------------------------------------------------------------------------
function mft,map,c,u,v,rt,rf,ci,normalize=normalize
;
; Given a map and coordinates (u,v), add to the complex visibilities
; the direct Fourier transform of the map evaluated at (u,v) and return
; the result. (u,v) can be vectors.
;
if n_elements(normalize) eq 0 then normalize=0
;
index=where(map.map ne 0,count)
if count eq 0 then begin
	print,'No clean components!'
	return,0
endif
cl=map.map(index)
et=map.teff(index)
x=map.x(index)
y=map.y(index)
ir=where(rt ne 0,nr)
uci=unique(ci)
nc=n_elements(uci)
;
mv=c
tf=abs(c)*0
for i=0L,n_elements(cl)-1 do begin
	p=2*!pi*(u*x(i)+v*y(i))
	m=where(rt(ir) eq et(i)) & m=m(0)
	for j=0,nc-1 do begin
		n=where(ci eq uci(j))
		f=(rf(j,m)/total(rf(j,ir)))*cl(i)
		tf(n)=tf(n)+f
		mv(n)=mv(n)+f*complex(cos(p(n)),sin(p(n)))
	endfor
endfor
;
if normalize then return,mv/tf $
	     else return,mv
;
end
;-------------------------------------------------------------------------------
function cleankernel,beam
;
; Old version!
;
common FitEllipse,ex,ey
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
MAS2RAD=pi_circle/(180l*3600000l)
RAD=180/!pi
;
r=size(beam.map)
if r(0) eq 3 then map=total(beam.map,3) else map=beam.map
m=max(map)
index=where(map gt m/2-m/20 and map lt m/2+m/20 $
	and beampatch(map) eq 1,ndata)
;
if ndata lt 10 then begin
        print,'***Error(CLEANKERNEL): too few half-maximum points!'
	stop
endif
;
imsize=n_elements(map(0,*))
;
; Set data
x=beam.x(index)/MAS2RAD & x=x-avg(x)
y=beam.y(index)/MAS2RAD & y=y-avg(y)
;
r=sqrt(x^2+y^2)
p=!pi-atan(x,y)
;
; Set parameters to fit
ma=5
ia=lonarr(ma)+1 & ia(0)=0 & ia(1)=0
index=where(ia ne 0,mfit)
e_parms=fltarr(ma)
e_parms(0)=0
e_parms(1)=0
e_parms(2)=max(r)
e_parms(3)=min(r)
i=where(r eq max(r)) & i=i(0)
e_parms(4)=p(i)
if e_parms(4) gt !pi then e_parms(4)=e_parms(4)-!pi
a=e_parms
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
;
; Call marquardt
sig=fltarr(ndata)+0.1
pa=p    ; ellfuncs changes p!
marquardt,'ellfuncs',pa,r,sig,ndata,a,ia,ma,chisq,covar
; Set new parameters
e_parms=a
e_parms(2)=a(2)*2*mas2rad
e_parms(3)=a(3)*2*mas2rad
e_parms(4)=a(4)
print,'Synthesized beam ',a(2)*2,' by ',a(3)*2,' mas, pa = ',a(4)*rad,' degrees.'
el=sqrt(e_parms(2)^2-e_parms(3)^2)
en=el/e_parms(2)
;
csize=abs(beam.x(0)-beam.x(1))
msize=fix(2*e_parms(2)/csize)
msize=2*(msize/2)+1
cbm=pearlimage([msize,msize],csize/mas2rad)
r=sqrt(cbm.x^2+cbm.y^2)
a=sqrt((e_parms(3)/2)^2/(1-en^2*cos(-atan(cbm.y,cbm.x)+e_parms(4)+!pi/2)^2))
cbm.map(*)=exp(-(r/a)^2)
;
return,cbm
;
end
;-------------------------------------------------------------------------------
function cleankernel,beam,a
;
; Return the CLEAN beam, which is the central peak in the dirty beam.
; The latter is converted to a white light beam if the input is a cube.
;
common FitEllipse,ex,ey
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
MAS2RAD=pi_circle/(180l*3600000l)
RAD=180/!pi
;
IF n_elements(a) eq 0 THEN BEGIN
;
r=size(beam.map)
if r(0) eq 3 then map=total(beam.map,3) else map=beam.map
index=where(map eq max(map))
;
imsize=n_elements(map(*,0))
jmsize=n_elements(map(0,*))
;
i=index(0) mod imsize
j=index(0)  /  imsize
index=[i-2,i-1,i,i+1,i+2]
jndex=[j-2,j-1,j,j+1,j+2]
;
cellsize=beam.x(0)-beam.x(1)
r=findgen(5)-2
h=[0.5,0.0,0.0]
x=fltarr(8)
y=x
; Fit quadratic polynomial, get FWHM by lowering 0.5 (max=1),
; and then solving the quadratic equation for r=0. This can
; fail if there are too few pixels across the maximum.
x(0:1)=quasol(poly_fit(r,map(index,j),2)-h)
y(2:3)=quasol(poly_fit(r,map(i,jndex),2)-h)
x(4:5)=quasol(poly_fit(r,map(index,jndex),2)-h)
y(4:5)=x(4:5)
x(6:7)=quasol(poly_fit(r,map(index,reverse(jndex)),2)-h)
y(6:7)=x([7,6])
;
; Set data
x=-x*cellsize/MAS2RAD
y=+y*cellsize/MAS2RAD
ndata=8
;
r=sqrt(x^2+y^2)
p=!pi-atan(x,y)
;
; Set parameters to fit
ma=5
ia=lonarr(ma)+1 & ia(0)=0 & ia(1)=0
index=where(ia ne 0,mfit)
e_parms=fltarr(ma)
e_parms(0)=0
e_parms(1)=0
e_parms(2)=max(r)
e_parms(3)=min(r)
i=where(r eq max(r)) & i=i(0)
e_parms(4)=p(i)
if e_parms(4) gt !pi then e_parms(4)=e_parms(4)-!pi
a=e_parms
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
;
; Call marquardt
sig=fltarr(ndata)+0.1
pa=p    ; ellfuncs changes p!
marquardt,'ellfuncs',pa,r,sig,ndata,a,ia,ma,chisq,covar
; Set new parameters
e_parms=a
e_parms(2)=a(2)*2*mas2rad
e_parms(3)=a(3)*2*mas2rad
e_parms(4)=a(4)
print,'Synthesized beam ',a(2)*2,' by ',a(3)*2,' mas, pa = ',360-a(4)*rad,' degrees.'
ENDIF ELSE BEGIN
;
e_parms=fltarr(5)
e_parms(2)=a(0)*mas2rad
e_parms(3)=a(1)*mas2rad
e_parms(4)=a(2)/rad
;
ENDELSE
;
el=sqrt(e_parms(2)^2-e_parms(3)^2)
en=el/e_parms(2)
;
csize=beam.x(0)-beam.x(1)
msize=fix(2*e_parms(2)/csize)
msize=2*(msize/2)+1
cbm=pearlimage([msize,msize],csize/mas2rad)
r=sqrt(cbm.x^2+cbm.y^2)
a=sqrt((e_parms(3)/2)^2/(1-en^2*cos(-atan(cbm.y,cbm.x)+e_parms(4)+!pi/2)^2))
cbm.map(*)=exp(-(r/a)^2)
;
return,cbm
;
end
;-------------------------------------------------------------------------------
function cleanwindow,map
;
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
;
wmap=map.map*0
if ncr eq 0 then wmap(*)=1 else $
for i=0,ncr-1 do wmap(crbox_x(0,i):crbox_x(1,i),crbox_y(3,i):crbox_y(4,i))=1
;
return,wmap*map.map
;
wmap=map.map
;
for i=0,ncl-1 do begin
	index=where((map.x lt clbox_x(0,i)) or (map.x gt clbox_x(1,i)) or $
		    (map.y gt clbox_y(1,i)) or (map.y lt clbox_y(2,i)),count)
	if count gt 0 then wmap(index)=0
endfor
return,wmap
;
end
;-------------------------------------------------------------------------------
pro clean,dm,db,rm,cc,niter,rt,rf
;
; Deconvolve the dirty map (dm) from the dirty beam (db) and return a map
; of the subtracted clean components (cc) and the residual map (rm).
; Do exactly niter iterations, and store these locally.
;
; Note: this is a multiwavelength CLEAN.
;
common LocalClean,rmmaps,ccmaps
;
if niter le 101 then begin
	rmmaps=replicate({map:rm.map},niter)
	ccmaps=replicate({map:cc.map},niter)
endif
;
dbxsize=n_elements(db.map(*,0,0))
dbysize=n_elements(db.map(0,*,0))
dmxsize=n_elements(dm.map(*,0))
dmysize=n_elements(dm.map(0,*))
;
nc=n_elements(rf(*,0))
nr=where(rt ne 0)
gain=0.1
;
; maxdb should by 1, always
r=size(db.map)
if r(0) eq 3 then maxdb=max(total(db.map,3)) else maxdb=max(db.map)
;
for k=0,niter-1 do begin
	maxrm=max(cleanwindow(rm))
	scale=maxrm*gain/maxdb
;	Find maximum in current residual map
	index=where(cleanwindow(rm) eq maxrm) & index=index(0)
	i=(index mod dmxsize)-dmxsize/2
	j=(index/dmxsize)-dmysize/2
;	Cut-out patch of dirty beam centered on peak and size of RM
	dbm=db.map(dbxsize/2-i-dmxsize/2:dbxsize/2-i+dmxsize/2, $
		   dbysize/2-j-dmysize/2:dbysize/2-j+dmysize/2,*)*scale
	m=where(rt(nr) eq cc.teff(i+dmxsize/2,j+dmysize/2)) & m=m(0)
;	Old code, rf not normalized
	flux=total(rf,2)
;	New code using SED, rf normalized when defining region
	flux(*)=1
	flux_scale=total(rf(*,m)/flux)/nc
;	Subtract dirty beam in each channel from RM
	for l=0,nc-1 do rm.map=rm.map-dbm(*,*,l)*(rf(l,m)/flux(l))/flux_scale
	cc.map(i+dmxsize/2,j+dmysize/2)=cc.map(i+dmxsize/2,j+dmysize/2)+scale
	if niter le 101 then begin
		rmmaps(k).map=rm.map
		ccmaps(k).map=cc.map
	endif
endfor
;
print,'CLEAN complete.'
;
end
;-------------------------------------------------------------------------------
pro residualmap
;
; Update the clean map (cm) with the current clean component list (cc),
; do a phase self-calibration and compute a new dirty/residual map.
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if total(abs(cc.map)) ne 0 then begin
	mv=mft(cc,mv,uc,vc,rt,rf,ci)
	cm.map=cm.map+cc.map
	cc.map=0
endif
;
selfcal
;
dm.map=0
dm=dft(dm,cv-mv,uc,vc)
status=set_scale(dm)
rm=dm
;
print,'Residual map rms: ',stdev(dm.map)
;
end
;-------------------------------------------------------------------------------
pro selfcal
;
; Perform a phase self-calibration of complex visibilities (c) using model
; visibilities (m). This algorithm uses a weighting "mask" and SVD to remove 
; phase wraps first by computing an exact solution, then considers more
; baselines to improve the solution up to all available data.
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Determine number of IFs
uci=unique(ci)
numchan=n_elements(uci)
;
; Determine number of stations
numsid=max(unique(bi) mod 10)
;
; Determine number of scans
numscan=n_elements(unique(si))
;
; Illumination factor
weight=100.0
;
; SVD eigenvalue tolerance
tolerance=1e-5
;
for i=0,numscan-1 do begin
for j=0,numchan-1 do begin
	index=where(ci eq uci(j) and si eq i+1,count)
	if count gt 0 then begin
	dmatrix=fltarr(count+1,numsid)
	for l=0,count-1 do begin
		dmatrix(l,(bi(index(l)) mod 10)-1)=+1
		dmatrix(l,(bi(index(l))  /  10)-1)=-1
	endfor
	dmatrix(count,genconfig.refstation-1)=weight
	dmatriz=dmatrix
	yweight=fltarr(count+1)+1
	for k=0,numsid-2 do begin
		y=[cphase(cv(index))-cphase(mv(index)),0]
		jndex=where(y gt !pi,jc)
		if jc gt 0 then y(jndex)=y(jndex)-2*!pi
		jndex=where(y lt -!pi,jc)
		if jc gt 0 then y(jndex)=y(jndex)+2*!pi
		bindex=where((bi(index)  /  10) eq k+1 $
			  or (bi(index) mod 10) eq k+1,count)
		if count gt 0 then begin
			dmatriz(bindex,*)=dmatriz(bindex,*)*weight
			yweight(bindex)=yweight(bindex)*weight
			y=y*yweight
			tmatriz=transpose(dmatriz)
			nmatriz=tmatriz#dmatriz
			r=tmatriz#y
			svdc,nmatriz,nw,nu,nv
			windex=where(nw/max(nw) lt tolerance,wcount)
			if wcount gt 0 then nw(windex)=0
			p=dmatrix#svsol(nu,nw,nv,r)
			cv(index)=cv(index)*conj(complex(cos(p),sin(p)))
		endif
	endfor
	endif
endfor
endfor
;
print,'Solution computed.'
;
end
;-------------------------------------------------------------------------------
pro tvdb
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlWids,i_wid,d_wid
;
widget_control,d_wid,get_value=win_num & wset,win_num
;
;tvlct,indgen(255),indgen(255),indgen(255)
r=size(db.map)
if r(0) eq 3 then map=total(db.map,3) else map=db.map
r=set_scale(map)
tv,map*r(0)+r(1)
;tvscl,total(db.map,3)
;
end
;-------------------------------------------------------------------------------
pro tvrm
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlWids,i_wid,d_wid
;
widget_control,d_wid,get_value=win_num & wset,win_num
;
xpos=n_elements(db.map(*,0,0))
;
;tvlct,indgen(255),indgen(255),indgen(255)
;tv,(rm.map-dm.zero)*dm.scale,xpos,0
tv,rm.map*dm.scale+dm.zero,xpos,0
;
end
;-------------------------------------------------------------------------------
pro tvfm
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlWids,i_wid,d_wid
;
widget_control,d_wid,get_value=win_num & wset,win_num
;
xpos=n_elements(db.map(*,0,0))
ypos=n_elements(dm.map(0,*))
;
map=convol(cm.map+cc.map,cb.map,/edge_wrap,/center)+rm.map
r=set_scale(map,1)
tv,map*r(0)+r(1),xpos,ypos
; 
tvscl,sqrt(map*r(0)+r(1)),xpos,ypos
;
end
;-------------------------------------------------------------------------------
pro tvcc
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common PearlWids,i_wid,d_wid
;
widget_control,d_wid,get_value=win_num & wset,win_num
;
cellsize=db.x(0)-db.x(1)
dbxsize=n_elements(db.map(*,0,0))
dbysize=n_elements(db.map(0,*,0))
dmxsize=n_elements(dm.map(*,0))
dmysize=n_elements(dm.map(0,*))
;
;tvlct,intarr(255),indgen(255),intarr(255)
index=where(cc.map ne 0,count)
if count gt 0 then plots,(index mod dmxsize)+dbxsize,index/dmxsize, $
	/device,color=tci(3),psym=3
;
;tvlct,intarr(255),intarr(255),indgen(255)
for i=0,ncr-1 do begin
	plots,crbox_x(*,i)+dbxsize,crbox_y(*,i), $
	/device,psym=0,color=tci(3)
endfor
;
end
;-------------------------------------------------------------------------------
pro tvrg
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common PearlWids,i_wid,d_wid
;
widget_control,d_wid,get_value=win_num & wset,win_num
;
cellsize=db.x(0)-db.x(1)
dbxsize=n_elements(db.map(*,0,0))
dbysize=n_elements(db.map(0,*,0))
dmxsize=n_elements(dm.map(*,0))
dmysize=n_elements(dm.map(0,*))
;
;tvlct,intarr(255),intarr(255),indgen(255)
for i=0,ncr-1 do begin
	plots,crbox_x(*,i)+dbxsize,crbox_y(*,i)+dmysize, $
	/device,psym=0,color=tci(3)
endfor
;
end
;-------------------------------------------------------------------------------
pro tkfm,map
;
; Plots contours of map.
;
rad2mas=180l*3600l*1000l/!pi
;
plevels=[0.1,0.2,0.5,1,2,5,10,20,50,80]
index=where(plevels gt abs(min(map.map)/max(map.map))*100)
; index=[2,3,4,5,6,7,8,9]
if index(0) gt 0 then index=[index(0)-1,index]
if index(0) eq -1 then index=[9]
plevels=plevels(index)
nlevels=[-plevels(0)]
;
levels=[nlevels,plevels]
levels_label=''
x=-unique(map.x)*rad2mas
y=+unique(map.y)*rad2mas
;
for i=0,n_elements(levels)-1 do begin
	if abs(levels(i)) lt 1 then $
		if levels(i) lt 0 then format='(f5.1)' else format='(f4.1)'
	if abs(levels(i)) ge 1 then $
		if levels(i) lt 0 then format='(i3)' else format='(i2)'
	if abs(levels(i)) ge 10 then $
		if levels(i) lt 0 then format='(i4)' else format='(i3)'
	levels_label=levels_label+string(levels(i),format=format)
endfor
!p.multi=0
contour,map.map,x,y,levels=plevels*max(map.map)/100,xstyle=1,ystyle=1, $
	xtitle='Contours: '+levels_label+' % of peak',ytitle='[mas]', $
	title='contour map',xrange=[max(x),min(x)]
contour,map.map,x,y,levels=nlevels*max(map.map)/100,c_linestyle=1,/overplot
;
end
;-------------------------------------------------------------------------------
pro clean_event,event
;
; CLEAN is run whenever the cc map is empty. Since the "Take" option allows
; to clean more than 100 components (the default range for CLEAN), the 
; cumulative clean component list is cc+ccc. ccc is maintained in the 
; callback for pearl.
;
common LocalPearlEvent,ccc
common PearlWids,i_wid,d_wid
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common LocalCleanEvent,dm_frames,rm_frames,cc_frames
common LocalClean,rmmaps,ccmaps
;
widget_control,d_wid,get_value=win_num & wset,win_num
;
device,cursor_standard=150
if total(abs(cc.map)) eq 0 then begin
	print,'Running CLEAN...please wait...',format='(a,$)'
	clean,dm,db,rm,cc,101,rt,rf
	print,'Finished CLEAN.'
endif
;
if n_elements(ccc) eq 0 then ccc=cc.map*0
;
widget_control,event.id,get_value=value
niter=fix(value)
rm.map=rmmaps(niter).map
cc.map=ccmaps(niter).map+ccc	; give all CLEAN comps. to tvcc and tvfm
;
tvrm
tvcc
tvfm
;
cc.map=cc.map-ccc	; restore cc to result from last CLEAN
device,/cursor_original
;
end
;-------------------------------------------------------------------------------
function pearl_event,event
;
common LocalPearlEvent,ccc
common PearlWids,i_wid,d_wid
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
;
cellsize=db.x(0)-db.x(1)
dbxsize=n_elements(db.map(*,0,0))
dbysize=n_elements(db.map(0,*,0))
dmxsize=n_elements(dm.map(*,0))
dmysize=n_elements(dm.map(0,*))
;
command=event.value
case command of
;	Call-backs for CLEAN slider
	'Take'  :begin
		 dm.map=rm.map
		 ccc=ccc+cc.map
		 cc.map=ccc
		 tvrm
		 tvcc
		 tvfm
		 cc.map=0
	       	 widget_control,i_wid,set_value=0
		 end
	'Update':begin
		 cc.map=cc.map+ccc
		 ir=where(rt ne 0,nr)
		 if nr gt 0 then residualmap
		 tvrm
		 tvfm
		 tvcc
	       	 widget_control,i_wid,set_value=0
		 ccc=cc.map*0
	       	 end
;	Call-backs for main menu
	'Selfcal':begin
		 selfcal
		 print,'Updating display...',format='(a,$)'
		 dm.map=0
		 dm=dft(dm,cv-mv,uc,vc)
		 status=set_scale(dm)
                 rm=dm
                 tvrm
		 tvcc
                 tvfm
		 print,'Display updated.'
		 end
	'Shift': begin
;		 Execute "Take" command
		 status=pearl_event({value:'Take'})
;		 Now get shift vector
		 read,x,y,prompt='Enter shift [mas] (right, up): '
		 mas2rad=!pi/(180l*3600000l)
		 i=fix(x*mas2rad/cellsize)
		 j=fix(y*mas2rad/cellsize)
		 cm.map=shift(cm.map,i,j)
		 rm.map=shift(rm.map,i,j)
		 dm.map=shift(dm.map,i,j)
		 ccc=shift(ccc,i,j)
		 cc.map=ccc
		 tvrm
		 tvcc
		 tvfm
		 cc.map=0
		 x=i*cellsize
		 y=j*cellsize
		 p=2*!pi*(-uc*x+vc*y)
		 cv(*)=cv*complex(cos(p),sin(p))
		 mv(*)=mv*complex(cos(p),sin(p))
		 end
	'Clear' :begin
		 mv=ov*0
		 cc.map=0
		 cm.map=0
		 dm.map=0
		 dm=dft(dm,cv,uc,vc)
		 status=set_scale(dm)
                 rm=dm
		 tvrm
		 tvfm
		 tvcc
		 print,'Clean map cleared.'
		 ccc=cc.map*0
		 end
	'Reset' :begin
		 cv=ov
		 mv=ov*0
		 cc.map=0
		 cm.map=0
		 dm.map=0
		 dm=dft(dm,cv,uc,vc)
		 status=set_scale(dm)
		 rm=dm
		 tvrm
		 tvcc
		 tvfm
		 print,'Reset maps and data.'
	       	 widget_control,i_wid,set_value=0
		 ccc=cc.map*0
		 end
	'Region':begin
		 print,'Place one window in final map; right click to reset!'
		 widget_control,d_wid,get_value=win_num & wset,win_num
		 !x.s=[0,1] & !y.s=[0,1] ; Normalized coordinates: (0,0)=BLC
		 icom=set_boxes(x0,y0)
		 if icom eq 4 then begin
			  rt(*)=0 & rg(*)=0 & rf(*)=0
			  cm.teff=0 & cm.logg=0
			  cc.teff=0 & cc.logg=0
			  ncr=0
			  icom=set_boxes(x0,y0,/clear)
			  widget_control,i_wid,sensitive=0
;			  tvfm
;			  tvrm
			  print,'Regions removed.'
			  return,0
		 endif
		 xsize=dbxsize+dmxsize
		 ysize=dmysize*2
;		 (x,y) in map pixels
		 x=(x0*xsize-dbxsize) > 0
		 y=(y0*ysize-dmysize) > 0
		 if total(x) eq 0 or total(y) eq 0 then begin
			print,'***Error: window placed outside FM!'
			return,-1
		 endif
		 r=size(cc.map)
;		 Make sure the box does not extend over the borders
		 x=x>0
		 x=x<r(1)
		 y=y>0
		 y=y<r(2)
		 icom=cw_form('0,LIST,Enter Teff\, log(g)|Default|Cancel,set_value=0,QUIT')
		 case icom.tag0 of
		 	0:begin
			  read,teff,logg,prompt='Enter Teff and log(g): '
			  crbox_x(*,ncr)=x(*,0)
			  crbox_y(*,ncr)=y(*,0)
			  cm.teff(x(0,0):x(1,0),y(3,0):y(4,0))=teff
			  cm.logg(x(0,0):x(1,0),y(3,0):y(4,0))=logg
			  cc.teff=cm.teff
			  cc.logg=cm.logg
			  rt(ncr)=teff
			  rg(ncr)=logg
			  index=unique(ci)-1
			  rf(*,ncr)=pearlflux(teff,wl(index),bw(index))
			  rf(*,ncr)=fluxband(teff,logg,wl(index),bw(index))
;			  Normalize rf with total flux, alternatively mult. with diameter^2
			  rf(*,ncr)=rf(*,ncr)/fl(index)
			  ncr=ncr+1
			  widget_control,i_wid,/sensitive
			  end
			1:begin
;			  Try to fit Teff to photometry
			  teff=5000
			  logg=4
			  a=[teff,1.0]
			  index=unique(ci)-1
			  xc=wl(index)*1d-9
			  yc=fl(index)
			  r=curvefit(xc,yc,yc*0+1,a,function_name='blackbody_pro',/noderiv)
			  teff=a(0)
			  print,'Fitted Teff = ',teff
			  crbox_x(*,ncr)=x(*,0)
			  crbox_y(*,ncr)=y(*,0)
			  cm.teff(x(0,0):x(1,0),y(3,0):y(4,0))=teff
			  cm.logg(x(0,0):x(1,0),y(3,0):y(4,0))=logg
			  cc.teff=cm.teff
			  cc.logg=cm.logg
			  rt(ncr)=teff
			  rg(ncr)=logg
			  rf(*,ncr)=pearlflux(teff,wl(index),bw(index))
			  rf(*,ncr)=fluxband(teff,logg,wl(index),bw(index))
;			  Normalize rf with total flux, alternatively mult. with diameter^2
			  rf(*,ncr)=rf(*,ncr)/fl(index)
			  ncr=ncr+1
			  widget_control,i_wid,/sensitive
			  end
			2:begin
			  icom=set_boxes(x0,y0,/clear)
			  end
		 endcase
		 print,'Number of active regions: ',ncr
		 end
;	CLEAN window (currently not used, functionality included in Region)
	'Window':begin
		 widget_control,d_wid,get_value=win_num & wset,win_num
		 !x.s=[0,1] & !y.s=[0,1] ; Normalized coordinates, (0,0)=BLC
		 icom=set_boxes(x0,y0,n)
		 xsize=dbxsize+dmxsize
		 ysize=dmysize*2
;		 (x,y) in map pixels
		 x=(x0*xsize-dbxsize)
		 y=(y0*ysize)
		 icom=cw_form('0,LIST,Keep new windows|Keep old windows|Remove windows,set_value=0,QUIT')
		 case icom.tag0 of
		 	0:begin
			  clbox_x=x
			  clbox_y=y
			  ncl=n
			  end
			1:
			2:begin
			  ncl=0
			  icom=set_boxes(x0,y0,n,/clear)
			  end
		 endcase
		 print,'Number of active windows: ',ncl
		 end
	'Contour':begin
		 if !d.name ne 'PS' then window,/free
		 fm=cm
		 fm.map=convol(cm.map+cc.map+ccc,cb.map,/edge_wrap,/center) $
		       +rm.map
		 tkfm,fm
		 if !d.name eq 'PS' then begin
			device,/close
			set_plot,!display
		 endif
		 widget_control,d_wid,get_value=id
		 wset,id
		 end
	'Convolve':begin
		 cb0=cb
		 read,a,b,p,prompt='Enter a,b[mas], pa[deg]: '
		 cb=cleankernel(db,[a,b,p])
		 tvfm
		 cb=cb0
		 end
	'Save':  begin
		 fitsfile='pearl.fits'
		 mas2rad=180/!pi*3600000.d0
		 naxis1=dmxsize
		 naxis2=dmysize
		 parseidldate,systime(),y,m,d
		 datum=constrictordate(y,m,d)
		 cdelt1=cellsize
		 cdelt2=cellsize
		 header=["SIMPLE  =                    T / Written by IDL", $
		         "BITPIX  =                  -32 / Number of bits per data pixel", $
		         "NAXIS   =                    2 / Number of data axes", $
		         "NAXIS1  =   "+string(naxis1)+" /", $
        		 "NAXIS2  =   "+string(naxis2)+" /", $
        		 "EXTEND  =                    T / FITS data may contain extensions", $
        		 "DATE    = '"+datum+"'         / Creation date of FITS header", $
        		 "BSCALE  =    1.00000000000E+00 /REAL = TAPE * BSCALE + BZERO", $
        		 "BZERO   =    0.00000000000E+00 /", $
        		 "CTYPE1  = 'RA---TAN'           /", $
        		 "CRVAL1  =    0.00000000000E+00 /", $
        		 "CDELT1  =   "+string(cdelt1)+" / radians", $
        		 "CRPIX1  =      1.000000000E+00 /", $
        		 "CROTA1  =      0.000000000E+00 /", $
        		 "CTYPE2  = 'DEC--TAN'           /", $
        		 "CRVAL2  =    0.00000000000E+00 /", $
        		 "CDELT2  =   "+string(cdelt2)+" / radians", $
        		 "CRPIX2  =      1.000000000E+00 /", $
        		 "CROTA2  =      0.000000000E+00 /", $
        		 "END     "]
		 writefits,fitsfile,cm.map+cc.map+ccc,header
	         ext_header=["XTENSION= 'IMAGE' / IMAGE extension", $
	                     "BITPIX  = -32 /  Number of bits per data pixel", $
	                     "NAXIS   = 2 / Number of data axes", $
	                     "NAXIS1  = "+string(naxis1)+" /", $
	                     "NAXIS2  = "+string(naxis2)+" /", $
	                     "PCOUNT  = 0 / No Group Parameters", $
	                     "GCOUNT  = 1 / One Data Group", $
	                     "EXTNAME = 'Teff' /", $
	                     "END     "]
	         writefits,fitsfile,cm.teff,ext_header,/append
	         ext_header(where(strpos(ext_header,'EXTNAME') ge 0))="EXTNAME = 'log(g)' /"
	         writefits,fitsfile,cm.logg,ext_header,/append
		 print,'Saved to file '+fitsfile
		 end
	'HELP':	 begin
		 print,'Selfcal: run phase self-calibration'
		 print,'Shift: shift map'
		 print,'Clear: remove all clean components'
		 print,'Reset: start over, but keep the regions'
		 print,'Region: define Teff/log(g) regions'
		 print,'Contour: plot a contour map'
		 print,'Convolve: convolve with Gaussian'
		 print,'Save: write clean map to disk'
		 end
endcase
;
end
;-------------------------------------------------------------------------------
pro pearl,star,cellsize=cellsize,imsize=imsize
;
; Setup and start an imaging session.
; Cellsize [mas], imsize=[nx,ny] or imsize=nxy (square map)
;
common PearlWids,i_wid,d_wid
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common LocalPearlEvent,ccc
;
; Initialize imaging data
if n_elements(star) eq 0 then begin
	print,'***Error(PEARL): you have to specify a star!'
	return
endif
set_imagedata,strupcase(star),status
if status ne 0 then return
;
; Initialize map parameters
uvr=sqrt(uc^2+vc^2)
fringespacing_min=(180/!pi)*3600000/max(uvr)
if n_elements(cellsize) eq 0 then begin
	cellsize=fringespacing_min/3
	print,'Cellsize [mas] set to: ',cellsize
endif else begin
	if cellsize gt fringespacing_min/3 then begin
	print,'Cellsize too large!'
	return
	endif
endelse
fringespacing_max=(180/!pi)*3600000/min(uvr)
if n_elements(imsize) eq 0 then begin
	imsize=(4*fringespacing_max)/cellsize
	imsize=fix(imsize/2)*2+1
endif
if n_elements(imsize) eq 1 then imsize=[imsize,imsize]
if (imsize(0) mod 2) eq 0 then begin
	imsize(0)=imsize(0)+1
	print,'Warning(PEARL): imsize(0) must be odd, set to: ',imsize(0)
endif
if (imsize(1) mod 2) eq 0 then begin
	imsize(1)=imsize(1)+1
	print,'Warning(PEARL): imsize(1) must be odd, set to: ',imsize(1)
endif
print,'Mapsize is ',imsize(0)*cellsize,' mas by ',imsize(1)*cellsize,' mas'
dmsize=imsize
dbsize=imsize*2-1
;
; Compute maps
uci=unique(ci)
nc=n_elements(uci)
db=pearlimage(dbsize,cellsize,nc)
;
print,'Computing dirty beam in '+string(nc,format='(i0)')+' channels...'
for j=0,nc-1 do begin
	index=where(ci eq uci(j))
	db=dft(db,cv(index)*0+1,uc(index),vc(index),j,/db)
endfor
db.map=db.map/nc
;
print,'Computing dirty map...'
dm=dft(pearlimage(dmsize,cellsize),cv,uc,vc)
status=set_scale(dm)
rm=dm
;
cm=pearlimage(dmsize,cellsize)	; clean map
cb=cleankernel(db)		; clean beam
cc=cm				; clean component map
ccc=cc.map*0
;
; Initialize regions and windows
nr=20
rt=fltarr(nr)
rg=fltarr(nr)
rf=fltarr(nc,nr)
ncr=0
ncl=0
crbox_x=fltarr(5,nr)
crbox_y=fltarr(5,nr)
;
base_wid=widget_base(/row,title='Pearl',resource_name='oyster')
column1_wid=widget_base(base_wid,/col)
column2_wid=widget_base(base_wid,/col)
;
d_wid=widget_draw(column1_wid,event_pro='clean_event', $
        scr_ysize=dbsize(1)+1,scr_xsize=dbsize(0)+dmsize(0))
row1_wid=widget_base(column1_wid,/row,/base_align_bottom)
i_wid=cw_fslider(row1_wid,minimum=0,maximum=100,value=0,/drag, $
        xsize=dbsize[0],format='(i3)')
b_wid=cw_bgroup(row1_wid,/row,['Take','Update'],event_func='pearl_event',/return_name)
;
buttons=['Selfcal','Shift','Clear','Reset','Region', $
	 'Contour','Convolve','Save','HELP']
b_wid=cw_bgroup(column2_wid,/col,buttons, $
	event_func='pearl_event',/return_name)
;
widget_control,base_wid,/realize
xmanager,'wwclean',base_wid,/no_block,event_handler='clean_event'
widget_control,i_wid,sensitive=0
;
tvdb
tvrm
tvfm
;
end
;************************************************************************Block 3
function pearson_funct,x,a
;
return,a(0)/(1.+((2.*x*sqrt(2.^(1./a(1))-1))/a(2))^2)^a(1)
;
end
;-------------------------------------------------------------------------------
pro pearson_pro,x,a,f
;
f=a(0)/(1.+((2.*x*sqrt(2.^(1./a(1))-1))/a(2))^2)^a(1)
;
end
;-------------------------------------------------------------------------------
pro pearson
;
nuv=61L
uvc=set_uvc([nuv,nuv],1.0)
uvc.map=sqrt(exp(-0.40*sqrt(uvc.u^2+uvc.v^2)))
c=float(uvc.map)
nim=127L
map=pearlimage([nim,nim],2.0d0)
uvc.u=uvc.u*1e6
uvc.v=uvc.v*1e6
img=dft(map,reform(uvc.map,nuv*nuv),uvc.u,uvc.v)
m=img.map
x=findgen(nim)-nim/2
y=m(*,nim/2)
ym=max(y)
index=where(y gt ym/2,fwhm)
a=[max(y),1.50,40.0]
a=[ym,1.5,fwhm]
xc=x
yc=y
r=curvefit(xc,yc,yc*0+1,a,function_name='pearson_pro',/noderiv)
!p.multi=[0,1,2]
plot,findgen(nuv)-nuv/2,c(*,nuv/2),psym=0
plot,x,y
oplot,x,pearson_funct(x,a),psym=0
print,a
;
; Check the back transformation
imsize=[nim,nim]
x=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
y=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
img.map=pearson_funct(sqrt(float(x)^2+float(y)^2),a)
img.teff=1
t=mft(img,reform(uvc.map*0,nuv*nuv),uvc.u,uvc.v,1,1,lonarr(nuv*nuv)+1)
t=reform(t,nuv,nuv)
window,/free
!p.multi=0
y1=float(uvc.map(*,nuv/2))
plot,y1,psym=0
y2=float(t(*,nuv/2))
f=median(y1/y2)
oplot,y2*f,psym=1
;
end
;-------------------------------------------------------------------------------
pro blackbody_pro,x,a,f
;
f=(1.191d-22/x^5)/(exp(1.439d-2/(x*a(0)))-1)*a(1)
;
end
;-------------------------------------------------------------------------------
