;-------------------------------------------------------------------------------
pro don,dcal=dcal,dsci=dsci,ecal=ecal,bl=bl,wl=wl,sigv2=sigv2
;
; Given a CAL diameter (dcal) and its error (ecal), compute the error of the 
; SCI diameter given its value (dsci), optionally with a random error for the 
; SCI and CAL visibilities (sigv2).
;
; Input parameters: all diameters in mas, wavelength in microns, baseline in m.
; Diameter error in %, visibility errors absolute
;
; All visibilities are squared visibilities.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
print,'Call with these keywords, e.g. dcal=3:'
print,'dcal [mas], dsci [mas], ecal [%], bl [m], wl [micron], sigv2 [absolute]'
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
if n_elements(bl) eq 0 then bl=40	; m
if n_elements(wl) eq 0 then wl=0.8	; microns
uvdist=float(bl)/(wl/1e6)
print,'Baseline length [m]: ',bl
print,'Wavelength [micron]: ',wl
;
if n_elements(dcal) eq 0 then dcal=1.
if n_elements(dsci) eq 0 then dsci=3.
print,'Science    diameter [mas]: ',dsci
print,'Calibrator diameter [mas]: ',dcal
;
; Check that visibility amplitude is larger than 0.1 for the science target
arg=factor*uvdist*dsci
vissqr_sci=(2*beselj(arg,1)/arg)^2
if vissqr_sci lt 0.1 then begin
	print,'Error: SCI diameter too large for this baseline!'
	return
endif else print,'SCI (squared) visibility: ',vissqr_sci
;
; Compute calibrator visibility and check if larger than 0.1
arg=factor*uvdist*dcal
vissqr_cal=(2*beselj(arg,1)/arg)^2
if vissqr_cal lt 0.1 then begin
	print,'Error: CAL diameter too large for this baseline!'
	return
endif else print,'CAL (squared) visibility: ',vissqr_cal
;
; Compute high and low values of calibrator visibility
if n_elements(ecal) eq 0 then ecal=10.
print,'Systematic error [%] in CAL diameter used: ',ecal
arg=factor*uvdist*dcal*(1+ecal/100.)
vissqr_lo=(2*beselj(arg,1)/arg)^2
arg=factor*uvdist*dcal*(1-ecal/100.)
vissqr_hi=(2*beselj(arg,1)/arg)^2
;
; (Re)calibrate SCI visibility and compute resulting diameter error
dsci_lo=hb(uvdist/1e6,vissqr_sci*vissqr_cal/vissqr_lo)
dsci_hi=hb(uvdist/1e6,vissqr_sci*vissqr_cal/vissqr_hi)
dsci_error_sys=(dsci_hi-dsci_lo)/dsci
print,'Systematic error [%] for SCI diameter resulting: ', $
	dsci_error_sys*100
;
; Add random noise to SCI and CAL visibilities
if n_elements(sigv2) eq 0 then sigv2=0.05	; absolute sigma!
if sigv2 ge 1 then begin
	print,'SIG_V2 too large, must be less than 1 (absolute sigma)!'
	return
endif else $
print,'Adding Gaussian noise to SCI & CAL visibilities with sigma: ',sigv2
v2e_sci=randomn(seed,100)*sigv2
v2e_cal=randomn(seed,100)*sigv2
;
vissqr_sci_noisy=vissqr_sci+v2e_sci
vissqr_cal_noisy=vissqr_cal+v2e_cal
dsci_noisy=hb(uvdist/1e6,vissqr_sci_noisy*vissqr_cal/vissqr_cal_noisy)
dsci_error_stat=stddev(dsci_noisy)/dsci
print,'Resulting statistical error [%] of SCI diameter: ', $
	dsci_error_stat*100
print,'Total systematic and statistical error [%] (added in quadrature) :', $
	sqrt(dsci_error_sys^2+dsci_error_stat^2)*100
;
end
;-------------------------------------------------------------------------------
pro don1
;
; Computes the minimum SCI diameter for an error of 1% as a function of the
; CAL diameter,given its error (10%).
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
bl=90		; m
wl=0.8e-6	; m
uvdist=bl/wl
n=400
dsci=(findgen(n)+1)/n*9.0	; Diameter of SCI in range of [3,5]
;
; Check that even for the largest science target diameter, NULL does not occur
arg=factor*uvdist*dsci
vissqr=(2*beselj(arg,1)/arg)^2
index=where(vissqr gt 0.05)
dsci=dsci[index]
n=n_elements(dsci)
;
dcal=(findgen(n)+1)/n*1.0	; Diameter of CAL in range of [0,1] mas
;
sig_sci=[0.05,0.02,0.01]
; Determine error on v2 related to 10% error on CAL diameter
arg=factor*uvdist*dcal*1.1
vissqr_lo=(2*beselj(arg,1)/arg)^2
arg=factor*uvdist*dcal*0.9
vissqr_hi=(2*beselj(arg,1)/arg)^2
viserr=(vissqr_hi-vissqr_lo)/(vissqr_hi+vissqr_lo)
dlim=dsci & dlim[*]=max(dsci)
;
for i=n-1,0,-1 do begin
;	Determine v2 of SCI
	arg=factor*uvdist*dsci[i]
	vissqr=(2*beselj(arg,1)/arg)^2
	vissqr_hi=vissqr*(1+viserr)
	vissqr_lo=vissqr*(1-viserr)
	dsci_lo=hb(uvdist/1e6,vissqr_hi)
	dsci_hi=hb(uvdist/1e6,vissqr_lo)
	derr=((dsci_hi-dsci_lo)/(dsci_hi+dsci_lo))*100
	index=where(derr lt 1.0,count)
	if count ge 1 then dlim[index]=dsci[i]
endfor
;
!p.charsize=1.0
!y.range=[0,3]
plot,dcal,dlim, $
	xtitle='Calibrator diameter [mas]', $
	ytitle='Minimum SCI diameter for 1% error', $
	title='CAL diameter error 10%, baseline ' $
		+strtrim(string(bl,format=i3),2)+' m'
;
end
;-------------------------------------------------------------------------------
