;*******************************************************************************
; File: misc.pro
;
; Description:
; ------------
;
; Miscellaneous procedures for every day's work with OYSTER.
; Note: all plotting here using tek_color; do not use tci.
;
; Block directory:
; ----------------
; Block 1: bias98,biascalc
; Block 2: printbias,plotrms,plotsm,printphase,plotdisp,
;	   printdisp,ratecal,redlist,starcal,vismap,
;	   reduceast,stellarrate,bbc,hanbury,brown,
;	   ewbaseline,cebaseline,cwbaseline,dana,
;	   binplot
;
; Block 2: dave,nick,don,ken,charmaine,olivier,klara
;
;************************************************************************Block 1
pro bias98
;
; For 1998-07-13, divide point data up into four scans.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
Rec0=[0,3998,7983,11970]
RecN=[3997,7982,11969,16008]
Iscan=[1,2,3,4]
StarId=['FKV0000','FKV0000','FKV0000','FKV0000']
;
StartTime=PointTime[Rec0]
StopTime=PointTime[RecN]
ScanId=Iscan
NumPoint=RecN-Rec0+1
NumCoh=lonarr(4)+1
NumIncoh=lonarr(4)+500
;
scantable=build_scantable(Iscan,ScanId,StarId,StartTime,StopTime,NumPoint, $
                NumCoh,NumIncoh)
;
end
;-------------------------------------------------------------------------------
pro biascalc,coeffs
;
; Compute the bias correction for each channel and spectrometer if the data
; was taken away from the fringe packet.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
coeffs=fltarr(GenConfig.NumOutBeam,max(GenConfig.NumSpecChan))
;
for i=0,GenConfig.NumOutBeam-1 do begin
for j=0,GenConfig.NumSpecChan[i]-1 do begin
	index=where(scans[*].vissqerr(i,j,0) gt 0,count)
	if count gt 1 then $
	coeffs[i,j]=avg(scans[index].vissq(i,j,0)*scans[index].photonrate(i,j))
endfor
endfor
;
end
;************************************************************************Block 2
pro printbias,avgbias
;
; Print the fluctuations of a variable (here: photonrate) over expected 
; fluctuations. Use scan data.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if checkdata([1,6,7,8,11]) ne 0 then return
;
NS=n_elements(Iscan)
bias=fltarr(GenConfig.NumOutBeam,max(GenConfig.NumSpecChan),NS)
;
for i=0,GenConfig.NumOutBeam-1 do begin 
for j=0,GenConfig.NumSpecChan[i]-1 do begin
for k=0,n_elements(Iscan)-1 do begin
;
  kk=Iscan[k]-1	; subtract 1 since arrays start with 0!
;
; Average photonrate and the other quantities
  PR=PhotonRate[i,j,Rec0[k]:RecN[k]]
  PRE=PhotonRateErr[i,j,Rec0[k]:RecN[k]]
  index=where(PRE gt 0,NumPoint)
  IF NumPoint gt 2 THEN BEGIN
    bias[i,j,k]=stdev(PR[index])/sqrt(avg(PR[index]) $
	       /(scantable[kk].numcoh*scantable[kk].numincoh))
  ENDIF
;
endfor
endfor
endfor
;
avgbias=fltarr(GenConfig.NumOutBeam,max(GenConfig.NumSpecChan))
;
openw,unit,'bias.dat',/get_lun
print,'Opened bias.dat...'
printf,unit,'            ',indgen(NS)+1,format='(a,100(i5))'
for i=0,GenConfig.NumOutBeam-1 do begin
for j=0,GenConfig.NumSpecChan[i]-1 do begin
printf,unit,'OB=',i+1,' Ch=',j+1,' bias=',bias[i,j,*], $
	format='(a,i1,a,i2,a,100(f4.1,1x))'
index=where(bias[i,j,*] ne 0,count)
if count gt 0 then avgbias[i,j]=avg(bias[i,j,index])
endfor
endfor
;
free_lun,unit
;
for j=0,max(GenConfig.NumSpecChan)-1 do print,avgbias[*,j]
print,'Average over all data: ',avg(avgbias[where(avgbias ne 0)])
;
end
;-------------------------------------------------------------------------------
pro plotrms,star
;
; Plot rms and rms/mean scan-to-scan fluctuations vs channel for all outputbeams
; for visibilities, photonrates, triple amplitudes and phases, all scan data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if checkdata([8,9]) ne 0 then return
;
if n_elements(star) eq 0 then begin
	star=''
	read,'Enter StarId: ',star
endif
;
index=where(scans.starid eq star,count)
if count eq 0 then return
;
xsize=512
ysize=800
!p.multi=[0,2,4,1,0]
;
if (!d.name eq 'PS' or !d.name eq 'ps') then begin
        width=17.78
        factor=width/640
        device,xsize=factor*xsize,ysize=factor*ysize, $
                xoffset=1,yoffset=1
endif else window,0,xsize=800,ysize=1000
;
for i=0,GenConfig.NumOutBeam-1 do begin
x=findgen(GenConfig.NumSpecChan[i])+1
y=sigma(scans[index].VisSq(i,*,0),0,2)
a=avg(scans[index].VisSq(i,*,0),2)
jndex=where(y gt 0,count)
plot,x[jndex],y[jndex]/a[jndex],psym=1,ytype=1,charsize=2.0, $
	title='VisSq of '+star, $
	xtitle='Channel',ytitle='RMS/Mean, OB='+string(i+1,format='(i2)')
xyouts,0.98,!y.window[1],Date,/normal,orientation=-90
endfor
;
for i=0,GenConfig.NumOutBeam-1 do begin
x=findgen(GenConfig.NumSpecChan[i])+1
y=sigma(scans[index].PhotonRate(i,*),0,2)
a=avg(scans[index].PhotonRate(i,*),2)
jndex=where(y gt 0,count)
plot,x[jndex],y[jndex]/a[jndex],psym=1,ytype=1,charsize=2.0, $
	title='Photonrate of '+star, $
	xtitle='Channel',ytitle='RMS/Mean, OB='+string(i+1,format='(i2)')
xyouts,0.98,!y.window[1],Date,/normal,orientation=-90
endfor
;
for i=0,GenConfig.NumTriple-1 do begin
x=findgen(GenConfig.NumSpecChan[i])+1
y=sigma(scans[index].TripleAmp(i,*),0,2)
a=avg(scans[index].TripleAmp(i,*),2)
jndex=where(y gt 0,count)
plot,x[jndex],y[jndex]/a[jndex],psym=1,ytype=1,charsize=2.0, $
	title='TrplAmp of '+star, $
	xtitle='Channel',ytitle='RMS/Mean, Tr='+string(i+1,format='(i2)')
xyouts,0.98,!y.window[1],Date,/normal,orientation=-90
endfor
;
RAD=180/pi_circle
for i=0,GenConfig.NumTriple-1 do begin
x=findgen(GenConfig.NumSpecChan[i])+1
y=sigma(scans[index].TriplePhase(i,*),0,2)*RAD
jndex=where(y gt 0,count)
plot,x[jndex],y[jndex],psym=1,ytype=1,charsize=2.0, $
	title='TrplPhase of '+star, $
	xtitle='Channel',ytitle='RMS, Tr='+string(i+1,format='(i2)')
xyouts,0.98,!y.window[1],Date,/normal,orientation=-90
endfor
;
end
;-------------------------------------------------------------------------------
pro plotsm,stars,color
;
; Plot the system magnitude vs channel from scan data.
; color = 1 for colored plot, 0 otherwise.
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
numstars=n_elements(stars)
if numstars eq 0 then begin
	print,'Warning(PLOTSM): no stars defined!'
	list_stars,stars
endif
;
if n_elements(color) eq 0 then color=0
if color ge 1 then color=1
if color lt 1 then color=0
;
!xsize=640
!ysize=512
frame_color=1
if (!d.name eq 'PS') then begin
	width=17.78
	factor=width/640
	device,xsize=factor*!xsize,ysize=factor*!ysize, $
		xoffset=1,yoffset=1,color=color
	if factor*!xsize gt width then device,/landscape
	!p.charsize=1.0
	if color then frame_color=0 else frame_color=1
endif
;
!p.color=frame_color
tek_color
plot,findgen(2),findgen(2),/nodata,xrange=[0,32],yrange=[0,12], $
	xtitle='Channel',ytitle='System magnitude',title=Date
;
psymbol=1
;
; Initialize data arrays
x=fltarr(numstars)
y=fltarr(numstars)
e=fltarr(numstars)
t=fltarr(numstars)
;
; Set the data
for ob=0,GenConfig.NumOutBeam-1 do begin
;
sm=fltarr(GenConfig.NumSpecChan[ob])
ss=fltarr(GenConfig.NumSpecChan[ob])
;
for ch=0,GenConfig.NumSpecChan[ob]-1 do begin
for st=0,numstars-1 do begin
	index=where(scans.starid eq stars[st],count)
	if count ne 0 then begin
		j=where(startable.starid eq stars[st],count)
		if count ne 0 then begin
			x[st]=startable[j].bv
			tt=scans[index].time
			pr=scans[index].photonrate(ob,ch)
			pre=scans[index].photonrateerr(ob,ch)
			index=where(pre gt 0,count)
			if count gt 1 then sigma=stdev(pr[index],mean) $
			else if count eq 1 then begin
				sigma=pre[index]
				mean=pr[index]
			endif
			if count gt 0 then begin
				y[st]=2.5*alog10(mean)+startable[j].mv
				e[st]=2.5*sigma/mean/alog(10)
				t[st]=avg(tt[index])
			endif
		endif
	endif
endfor
;
; Do a linear fit
index=where(e gt 0,count)
if count gt 1 then begin
	r=poly_fit(x[index],y[index],1,yft,ybd,sig,mat)
	sm[ch]=r[0]
	ss[ch]=ch+1
endif
endfor
;
!p.color=ob+2
oplot,ss,sm,psym=psymbol
psymbol=psymbol+1
if psymbol eq 3 then psymbol=4
endfor
;
!p.color=1
;
if (!d.name eq 'PS') then begin
	device,/close_file
	set_plot,!display
endif
;
end
;-------------------------------------------------------------------------------
pro printphase,star
;
; Print raw scan triple phases to file for a given star.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(star) eq 0 then begin
	star=''
	read,'Please enter star name: ',star
endif
;
filename='phase.dat'
openw,unit,filename,/get_lun
for i=0,n_elements(scans)-1 do begin
if scans[i].starid eq star then begin
printf,unit,scans[i].time,scans[i].triplephase(0,0:19),format='(f7.1,2x,32(f5.2,1x))'
printf,unit,scans[i].time,scans[i].triplephaseerr(0,0:19),format='(f7.1,2x,32(f5.2,1x))'
endif
endfor
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro plotdisp,quad,outputbeam,baseline,scan_no
;
; Plot the quadruplephase vs residual delayline position for the pointdata
; in one scan. scan_no can be an array. quad should be 1 or 2.
; CONSTRICTOR file should be open.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
qd=quad-1
ob=outputbeam-1
bl=baseline-1
;
sid1=strmid(GenConfig.BaselineId[bl,ob],0,3)
i1=where(GenConfig.StationId eq sid1)
sid2=strmid(GenConfig.BaselineId[bl,ob],4,3)
i2=where(GenConfig.StationId eq sid2)
;
for i=0,n_elements(scan_no)-1 do begin
	k=scan_no[i]-1
	get_quadruplevis,qv,qve,scan_no[i]
	get_points,scan_no[i]
	d1=FDLPos[i1,Rec0[0]:RecN[0]]
	d2=FDLPos[i2,Rec0[0]:RecN[0]]
	delay=(d1-d2)*1d6 & delay=reform(delay,n_elements(delay))
;	delay=softdelay(ob,bl,*)
	r=poly_fit(dindgen(n_elements(delay)),delay,2,dft)
	residual=float(dft-delay)
	phase=atan(qv[ob,1,qd,bl,*],qv[ob,0,qd,bl,*])*RAD
	plot,residual,phase, $
		xtitle='Residual delay [mu]',ytitle='Quadruple phase [deg]', $
		title=date+', '+genconfig.baselineid[bl,ob]+ $
		     ', Scan '+string(scan_no[i],format='(i3)')+ $
		     ' ('+scantable[scan_no[i]-1].starid+')', $
		psym=1,charsize=1.0
	r=poly_fit(residual,phase,1)
	x=findgen(100)-50 & y=poly(x,r) & oplot,x,y,psym=0
	print,r
endfor
;
end
;-------------------------------------------------------------------------------
pro printdisp,outputbeam,channels
;
; Print the estimated slope deg/mu of the quadruplephase vs residual delay 
; for a given channel combination with equidistant wavenumbers.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
ob=outputbeam-1
ch=channels-1
;
l1=genconfig.wavelength[ch[0],ob]*1d6
n1=edlen(l1)-1
d1=n1*1000
l2=genconfig.wavelength[ch[1],ob]*1d6
n2=edlen(l2)-1
d2=n2*1000
l3=genconfig.wavelength[ch[2],ob]*1d6
n3=edlen(l3)-1
d3=n3*1000
delay1=d2
phase1=d1/l1-2*d2/l2+d3/l3
l1=genconfig.wavelength[ch[0],ob]*1d6
n1=edlen(l1)-1
d1=n1*10000
l2=genconfig.wavelength[ch[1],ob]*1d6
n2=edlen(l2)-1
d2=n2*10000
l3=genconfig.wavelength[ch[2],ob]*1d6
n3=edlen(l3)-1
d3=n3*10000
delay2=d2
phase2=d1/l1-2*d2/l2+d3/l3
slope=(phase2-phase1)/(delay2-delay1)
;
print,slope*360
;
end
;-------------------------------------------------------------------------------
pro ratecal,filespec,bv,ratio
;
; Load specified .cha files and measure the flux ratio as a function of (B-V).
;
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,startable,notes
;
files=findfile(filespec)
;
bv=fltarr(10000)
ratio=fltarr(10000)
n=0
;
for i=0,n_elements(files)-1 do begin
	hds_open,files[i]
	get_scandata
	list_stars,stars
	for j=0,n_elements(stars)-1 do begin
		index=where(scans.starid eq stars[j])
		p_green=scans[index].photonrate(2,17)
		p_green_err=scans[index].photonrateerr(2,17)
		p_red=scans[index].photonrate(2,1)
		p_red_err=scans[index].photonrateerr(2,1)
		index=where(p_green_err gt 0 and p_red_err gt 0,count)
		if count gt 0 then begin
			ratio[n]=avg(p_red[index])/avg(p_green[index])
			index=where(startable.starid eq stars[j])
			bv[n]=startable[index].bv
			n=n+1
		endif
	endfor
	hds_close
endfor
;
bv=bv[0:n-1]
ratio=ratio[0:n-1]
index=where(finite(ratio) ne 0)
bv=bv[index]
ratio=ratio[index]
;
end
;-------------------------------------------------------------------------------
pro redlist,bv,ratio
;
; Compile a list of calibrator candidates using the data found in
; diameter.bsc and the requirement that the star have enough flux 
; in the red to be fringe tracked.
;
common StarBase,startable,notes
;
r=[2.01,0.82,2.53]
if n_elements(bv) gt 0 and n_elements(ratio) gt 0 then r=poly_fit(bv,ratio,2)
;
get_bsc
index=where(startable.mv ne 100 $
        and startable.bv le 2.0 $
	and startable.dec gt -30)
startable=startable[index]
;
rate_green=10^((4.4-startable.mv)/2.5)*2.0
rate_red=poly(startable.bv,r)*rate_green
index=where(rate_red gt 6)
print,'Number of stars with red count rate > 6:',n_elements(index)
startable=startable[index]
;
get_diameter
index=where(startable.diameter lt 1.0 $
	and startable.diameter ne 0 $
	and startable.bflag ne 'B')
startable=startable[index]
;
end
;-------------------------------------------------------------------------------
pro starcal,star,filespec,rmsa,rmsc
;
; Load specified files and get RMS quality of visibility and phase calibration.
;
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,startable,notes
;
files=findfile(filespec)
num_files=n_elements(files)
if num_files eq 0 then return
num_outbeam=3
num_specchan=20
;
rmsa=fltarr(num_files,num_outbeam,num_specchan)
rmsc=fltarr(num_files,num_specchan)
;
for i=0,num_files-1 do begin
	hds_open,files[i]
	get_scandata
	index=where(scans.starid eq star,count)
	if count gt 0 then begin
		for k=0,num_specchan-1 do begin
			t=scans[index].time/3600
			v=scans[index].triplephase(0,k)
			ve=scans[index].triplephasecerr(0,k)
			jndex=where(ve gt 0,count)
			if count gt 2 then begin
				r=poly_fit(t[jndex],v[jndex],1)
				rmsc[i,k]=stdev(v[jndex])
				rmsc[i,k]=r[1]
			endif
		for j=0,num_outbeam-1 do begin
			v=scans[index].vissqec(j,k,0)
			ve=scans[index].vissqecerr(j,k,0)
			jndex=where(ve gt 0,count)
			if count gt 2 then rmsa[i,j,k]=stdev(v[jndex])
		endfor
		endfor
	endif
	hds_close
endfor
;
end
;-------------------------------------------------------------------------------
pro vismap,filespec
;
; Load specified files and print out data enabling the computation of a 
; "visibility map".
;
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,startable,notes
;
files=findfile(filespec)
num_files=n_elements(files)
if num_files eq 0 then return
;
openw,unit,'vismap.dat',/get_lun
;
for i=0,num_files-1 do begin
	hds_open,files[i]
	get_scandata
	for j=0,n_elements(scans)-1 do begin
		if scans[j].time gt 0 then begin
		index=where(startable.starid eq scans[j].starid)
		if startable[index].bflag eq 'C' then begin
		if min(scans[j].vissq(*,4,0)) gt 0 then begin
		printf,unit,scans[j].ha,scans[j].za,scans[j].dec, $
			scans[j].vissq(0,4,0), $
			scans[j].vissq(1,4,0), $
			scans[j].vissq(2,4,0),format='(6(f8.4,2x))'
		endif
		endif
		endif
	endfor
	hds_close
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro reduce
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if init_plot('point','pt') ne 0 then return
create_pointflagtable
for ob=0,genconfig.numoutbeam-1 do begin
	ds_y.ob=ob
	ds_y.item=10
	s=set_datasel['x']
	s=set_datasel['y']
	plotdata,1,'AUTO'
	ds_y.item=8
	ds_y.ch=indgen(32)
	s=set_datasel['x']
	s=set_datasel['y']
	plotdata,1,'AUTO'
	ds_y.item=3
	s=set_datasel['x']
	s=set_datasel['y']
	plotdata,1,'AUTO'
endfor
average
create_scanflagtable
flagscandata,item=24,ch=stringparse('11,21-32')
;
end
;-------------------------------------------------------------------------------
pro reduceast,night
;
; Script for automated astrometry reduction.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
hds_open,night,'READ',status
if status ne 0 then return
get_scantable
get_startable
get_SysConfig
fixearlydays
get_stationtable,checkdate()+'.stn'
defaultbg
list_stars,stars
for i=0,n_elements(stars)-1 do begin
	get_star,stars[i]
	fdlpos_bck=fdlpos
	calcgeo
	fdlpos=fdlpos-geodelay
	dispflag
	unwrapphase
        dispcorr,/dw,/pt
	if init_plot('point','pt') ne 0 then return
	ds_y.item=16
	ds_y.ib=1
	s=set_datasel['x']
	s=set_datasel['y']
	plotdata,1,'AUTO'
	ds_y.ib=2
	s=set_datasel['y']
	plotdata,1,'AUTO'
	fdlpos=fdlpos_bck
;	metrocorr
	metroinit
	average
endfor
return
;
end
;-------------------------------------------------------------------------------
function calcwhite
;
; Compute white light corrections for point data.
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([8,9,11]) ne 0 then return,-1
;
; Must have white light data for all siderostats
flag=0
for i=0,GenConfig.NumSid-1 do begin
	index=where(scans.starid eq 'FKV0000' $
		and scans.fdlposerr[i] gt 0,count)
	if count eq 0 then begin
	   print,'***Error(CALCWHITE): no white light data for beam',i+1,'!'
	   flag=1
	endif
endfor
if flag eq 1 then begin
	print,'No white light corrections applied!'
	return,-1
endif
;
corr=fltarr(GenConfig.NumSid,n_elements(PointTime))
for i=0,GenConfig.NumSid-1 do begin
	index=where(scans.starid eq 'FKV0000' $
		and scans.fdlposerr[i] gt 0)
	d=scans[index].fdlpos(i)*(i ne GenConfig.RefStation-1) & d=d-avg(d)
	t=scans[index].time
	index=where(abs(PointTime) gt 0)
	corr[i,index]=spline(t,d,abs(PointTime[index]))
endfor
;
return,corr
;
end
;-------------------------------------------------------------------------------
pro hr
;
common StarBase,startable,notes
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
tek_color
!p.psym=0
!p.charsize=1.5
!x.tickname=['O5','B0','B5','A0','A5','F0','F5','G0','G5','K0','K5','M0','M5']
!x.ticks=12
!x.tickv=findgen(13)*5+5
!x.title='Spectral type'
!y.title='Absolute visual magnitude'
; 
; PostScript output
if !d.name eq 'PS0' then begin
        !p.color=0
        width=17.78
        factor=width/640
        device,xsize=factor*640,ysize=factor*800,xoffset=1,yoffset=1,color=1
        device,xsize=factor*640,ysize=factor*800,xoffset=1,yoffset=1
endif
;
t=findgen(60)+5
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.V',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
plot,t,m,xrange=[0,70],yrange=[12,-10]
xyouts,60,8,'V'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.III',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,60,-1,'III'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.II',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,60,-3,'II'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Ib',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-4.5,'Ib'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Iab',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-5.5,'Iab'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Ia',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-6.5,'Ia'
;
!x.tickname=''
!x.tickv=0
!x.ticks=0
;
end
;-------------------------------------------------------------------------------
pro hr2
;
common StarBase,startable,notes
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
tek_color
!p.psym=0
!p.color=0
!p.charsize=1.5
!x.title='Effective temperature'
!y.title='Luminosity'
!x.type=1
!y.type=1
!x.style=1
!y.style=1
; 
; PostScript output
if !d.name eq 'PS0' then begin
        !p.color=0
        set_plot,'ps'
        width=17.78
        factor=width/640
        device,xsize=factor*640,ysize=factor*800,xoffset=1,yoffset=1,color=1
        device,xsize=factor*640,ysize=factor*800,xoffset=1,yoffset=1
endif
;
t=reverse(teff_dwarf(findgen(60)+5))
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.V',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
si=sort(type) & si=reverse(si) & type=type[si] & mv=mv[si]
teff=teff_dwarf(type)
llum=(mv+bcf(teff)-4.74)/(-2.5)
l=spline(alog10(teff),llum,alog10(t))
plot,t,10^l,xrange=[20000,1000],yrange=[0.001,10000],xtype=!x.type,ytype=!y.type
xyouts,60,8,'V'
;
return
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.III',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,60,-1,'III'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.II',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,60,-3,'II'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Ib',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-4.5,'Ib'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Iab',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-5.5,'Iab'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Ia',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-6.5,'Ia'
;
if !d.name eq 'PS0' then device,/close
;
end
;-------------------------------------------------------------------------------
pro bbc,dk12,dxy
;
; Procedure to give an idea about the accuracy of masses across the HR diagram
; when spectroscopy and astrometry with the input uncertainties are performed
; on the BBC catalog.
;
common StarBase,startable,notes
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
if n_params() eq 0 then begin
	dk12=500	; Uncertainty of K in m/s
	dxy=70		; Uncertainty of position measurement in arc seconds
endif
;
t=findgen(60)+5
;
tek_color
!p.psym=0
!p.color=1
!p.charsize=1.5
!x.tickname=['O5','B0','B5','A0','A5','F0','F5','G0','G5','K0','K5','M0','M5']
!x.ticks=12
!x.tickv=findgen(13)*5+5
!x.title='Spectral type'
!y.title='Absolute visual magnitude'
!p.title='!4r[!8K]!3 = '+string(dk12,format='(i4)')+' m/s' $
        +', !4r[!8r]!3 = '+string(dxy,format='(i3)')+' !4l!3as'
;
; PostScript output
if !d.name eq 'PS' then begin
	!p.color=0
	set_plot,'ps'
	width=17.78
	factor=width/640
	device,xsize=factor*640,ysize=factor*800,xoffset=1,yoffset=1,color=1
	device,xsize=factor*640,ysize=factor*800,xoffset=1,yoffset=1
endif
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.V',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
plot,t,m,xrange=[0,70],yrange=[12,-10]
xyouts,60,8,'V'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.III',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,60,-1,'III'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.II',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,60,-3,'II'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Ib',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-4.5,'Ib'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Iab',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-5.5,'Iab'
;
sp=''
type=0.0
mv=0.0
s=dc_read_free(!oyster_dir+'starbase/spec_MV.Ia',sp,type,mv, $
	resize=[1,2,3],/col,ignore=['!'])
m=spline(type,mv,t)
oplot,t,m
xyouts,65,-6.5,'Ia'
;
; Define a nice sized dot for the plot symbol
l=1.0
n=45
p=findgen(n+1)*2*!pi/n
x=l*sin(p)
y=l*cos(p)
usersym,x,y,/fill
;
if n_elements(startable) eq 0 then restore,!catalogs_dir+'bbc/bbc.xdr'
index=where(startable.px le 0,count)
if count gt 0 then startable[index].px=0.001
dmod=-5+5*alog10(1/startable.px)
;
RAD=180/!pi
; Determine K. Assume components are identical
a12sini=startable.a/startable.px*a_unit/1000
k12=a12sini/13751/startable.p
;
; Determine mass
m12sin3i=1.0385e-7*4*k12^3*startable.p
m12=m12sin3i/sin(startable.i/rad)^3
;
; Determine mass uncertainty
i_rad=startable.i/RAD
index=where(startable.i eq 90,count)
if count gt 0 then i_rad[index]=89.9/RAD
dk12=dk12/1e3
dx=dxy/1e6
di=atan(dx/startable.a)
dmom=sqrt((3*dk12/k12)^2+(3*di/tan(i_rad))^2)*100
;
; Plot the starlist
!p.psym=8
tp=startable.type
mv=startable.mv-dmod
;
index=where(dmom gt 10,count)
!p.color=2
if count gt 0 then oplot,tp[index],mv[index]
xyouts,5,10,'10% < !4r!3[!13M!3]'
xyouts,35,10,'('+string(count,format='(i3)')+'!3+!3)'
;
index=where(dmom gt 5 and dmom le 10,count)
!p.color=8
if count gt 0 then oplot,tp[index],mv[index]
xyouts,5,9,'5% < !4r!3[!13M!3] < 10%'
xyouts,35,9,'('+string(count,format='(i3)')+'!9X!3)'
;
index=where(dmom gt 1 and dmom le 5,count)
!p.color=5
if count gt 0 then oplot,tp[index],mv[index]
xyouts,5,8,'1% < !4r!3[!13M!3] < 5%'
xyouts,35,8,'('+string(count,format='(i3)')+'!9V!3)'
;
index=where(dmom le 1,count)
!p.color=3
if count gt 0 then oplot,tp[index],mv[index]
xyouts,5,7,'!4r!3[!13M!3] < 1%'
xyouts,35,7,'('+string(count,format='(i3)')+'!20B!3)'
;
if !d.name eq 'PS' then device,/close
;
end
;-------------------------------------------------------------------------------
pro ncal,files,coefficients,jd
;
; For a list of files, return ncal (i.e. channels sensitivity) information.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
files=findfile(files)
for i=0,n_elements(files)-1 do begin
	print,'Now processing: ',files[i],'...'
        hds_open,files[i]
	get_scandata
	hds_close
	s=init_plot('ncal','pt')
	y_dir.ch='All'
	for j=0,GenConfig.NumOutBeam-1 do begin
		ds_y.ob=j
		s=set_datasel['x']
		s=set_datasel['y']
		s=set_datasel['z']
		plotncal,coeffs
		if i eq 0 then begin
			c={coeff_struct,coeffs:coeffs,jd:0L}
			coefficients=replicate(c,n_elements(files))
		endif
		coefficients[i].coeffs=coeffs
		parsedate,date,year,month,day
		coefficients[i].jd=julian(year,month,day)
	endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro ncalplot,c,i
;
; Plot results from ncal.
;
jy=jd2jy(c.jd)
;
!x.title='Year'
!x.ticks=3
; !y.title='V of * with (B-V)=0 for 1 count/2 ms'
!y.title='System magnitude'
;
!p.multi=[0,2,2,1,1]
!p.charsize=1.0
;
!y.range=[3,8]
;
if 0 then begin
for j=0,31,8 do begin
index=where(abs(c[*].coeffs(1,j,i)-median(c[*].coeffs(1,j,i))) lt 0.5,count)
if count gt 0 then $
plot,jy[index],c[index].coeffs(0,j,i), $
	title='Sp.'+string(i+1,format='(i2)')+ $
               ', ch. '+string(j+1,format='(i3)')
endfor
endif
;
!y.range=[-1.5,1.5]
!y.range=0
!y.title='Slope'
;
for j=0,31,8 do begin
index=where(abs(c[*].coeffs(1,j,i)-median(c[*].coeffs(1,j,i))) lt 0.5,count)
if count gt 0 then $
plot,jy[index],c[index].coeffs(1,j,i), $
	title='Sp.'+string(i+1,format='(i2)')+ $
               ', ch. '+string(j+1,format='(i3)')
endfor
;
end
;-------------------------------------------------------------------------------
pro pcal,files
;
; For a list of files, return pcal information.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
openw,unit,'pcal.dat',/get_lun
printf,unit,'(B-V)   ratio 26/3    HA    DEC'
;
files=findfile(files)
for i=0,n_elements(files)-1 do begin
	print,'Now processing: ',files[i],'...'
        hds_open,files[i]
	get_scandata
	hds_close
	for j=0,n_elements(scans)-1 do begin
		if scans[j].starid ne 'FKV0000' then begin
		index=where(startable.starid eq scans[j].starid)
		for k=0,GenConfig.NumOutBeam-1 do begin
			if scans[j].photonrateerr(k,25) gt 0 $
		       and scans[j].photonrateerr(k,2) gt 0 $
		       and scans[j].photonrate(k,2) gt 1 $
		       and scans[j].photonrate(k,25) gt 1 then begin
				ratio=scans[j].photonrate(k,25) $
				     /scans[j].photonrate(k,2)
				printf,unit,startable[index].bv,ratio, $
					k+1,scans[j].ha,scans[j].dec
			endif
		endfor
		endif
	endfor
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro bcal,files,avbg,sibg,dates
;
; For a list of .cha files, read the back ground rates, average them,
; and return the data for plotting.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; goto,plot
;
files=findfile(files)
numfiles=n_elements(files)
if numfiles eq 0 then return
;
avbg=fltarr(numfiles,3,32)
sibg=avbg
dates=fltarr(numfiles)
;
for i=0,n_elements(files)-1 do begin
	print,'Now processing: ',files[i],'...'
        hds_open,files[i]
	get_sysconfig
	get_scantable
	get_scans
	hds_close
	parsedate,date,y,m,d
	dates[i]=jd2jy(julian(y,m,d))-1900
	for j=0,GenConfig.NumOutBeam-1 do begin
	for k=0,GenConfig.NumSpecChan[j]-1 do begin
		index=where(scans.backgndrate[j,k] gt 0 $
			and scans.starid ne 'FKV0000',count)
		if count gt 3 then begin
			avbg[i,j,k]=avg(scans[index].backgndrate(j,k))
			sibg[i,j,k]=stddev(scans[index].backgndrate(j,k))
			if sibg[i,j,k] gt 1 then begin
				avbg[i,j,k]=0
				sibg[i,j,k]=0
			endif
		endif
	endfor
	endfor
endfor
;
plot:
!p.multi=[0,4,5]
!p.psym=3
!p.charsize=1.5
!y.range=[0,6]
if !d.name eq 'PS' then begin
	!ysize=900
	!xsize=700
        width=17.78
        factor=width/640
        device,xsize=factor*!xsize,ysize=factor*!ysize, $
                xoffset=1,yoffset=1
endif
for j=0,2 do begin
for k=0,19 do begin
	title=strcompress(('OB'+string(j+1)+', CH'+string(k+1)),/remove_all)
	plot,dates,avbg[*,j,k],ytitle=title,xmargin=[4,1],ymargin=[1,1]
	oploterr,dates,avbg[*,j,k],sibg[*,j,k],3
endfor
endfor
;
if !d.name eq 'PS' then device,/close
;
end
;-------------------------------------------------------------------------------
pro vcal,files,avvs,dates
;
; For a list of .cha files, read the visibility data, average them,
; and return the data for plotting.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; goto,plot
;
files=findfile(files)
numfiles=n_elements(files)
if numfiles eq 0 then return
;
avvs=fltarr(numfiles,3,32)
dates=fltarr(numfiles)
;
for i=0,n_elements(files)-1 do begin
	print,'Now processing: ',files[i],'...'
        get_scandata,files[i]
	hds_close
	numscans=n_elements(scans)
	calindex=intarr(numscans)
	list_stars,stars
	for l=0,n_elements(stars)-1 do begin
		if startable[l].bflag eq 'C' then $
			calindex[where(scans.starid eq stars[l])]=1
	endfor
	index=where(calindex eq 1,count)
	if count gt 0 then begin
	scans=scans[index]
	parsedate,date,y,m,d
	dates[i]=jd2jy(julian(y,m,d))-1900
	for j=0,GenConfig.NumOutBeam-1 do begin
	for k=0,GenConfig.NumSpecChan[j]-1 do begin
		index=where(scans.vissqerr[j,k] gt 0,count)
		if count gt 20 then begin
			avvs[i,j,k]=avg(scans[index].vissq(j,k))
		endif
	endfor
	endfor
	endif
endfor
;
plot:
!p.multi=[0,4,5]
!p.psym=1
!p.charsize=1.5
!x.range=[96,100]
!y.range=[0,0.8]
if !d.name eq 'PS' then begin
	!ysize=900
	!xsize=700
        width=17.78
        factor=width/640
        device,xsize=factor*!xsize,ysize=factor*!ysize, $
                xoffset=1,yoffset=1
endif
for j=0,2 do begin
for k=0,19 do begin
	title=strcompress(('OB'+string(j+1)+', CH'+string(k+1)),/remove_all)
	plot,dates,avvs[*,j,k],ytitle=title,xmargin=[6,1],ymargin=[1,1]
endfor
endfor
;
if !d.name eq 'PS' then device,/close
;
end
;-------------------------------------------------------------------------------
pro jcal,files,avjs,dates
;
; For a list of .cha files, read the visibility data, average them,
; and return the data for plotting.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; goto,plot
;
files=findfile(files)
numfiles=n_elements(files)
if numfiles eq 0 then return
;
avjs=fltarr(numfiles,3)
dates=fltarr(numfiles)
;
for i=0,n_elements(files)-1 do begin
	print,'Now processing: ',files[i],'...'
	get_scandata,files[i]
;	get_sysconfig
;	get_scantable
;	list_stars,stars
;	get_startable,stars
;	get_scans
	hds_close
	numscans=n_elements(scans)
	calindex=intarr(numscans)
	list_stars,stars
	for l=0,n_elements(stars)-1 do begin
		if startable[l].bflag eq 'C' then $
			calindex[where(scans.starid eq stars[l])]=1
	endfor
	index=where(calindex eq 1,count)
	if count gt 0 then begin
	scans=scans[index]
	parsedate,date,y,m,d
	dates[i]=jd2jy(julian(y,m,d))-1900
	for j=0,GenConfig.NumOutBeam-1 do begin
		index=where(scans.delayjittererr[j] gt 0,count)
		if count gt 20 then begin
			avjs[i,j]=median(scans[index].delayjitter(j))*1e6
		endif
	endfor
	endif
endfor
;
plot:
!p.multi=[0,1,3]
!p.psym=1
!p.charsize=2.0
!x.range=[96,100]
!y.range=[0,8]
if !d.name eq 'PS' then begin
	!ysize=900
	!xsize=700
        width=17.78
        factor=width/640
        device,xsize=factor*!xsize,ysize=factor*!ysize, $
                xoffset=1,yoffset=1
endif
for j=0,2 do begin
	title=strcompress(('OB'+string(j+1)),/remove_all)
	plot,dates,avjs[*,j],ytitle=title,xmargin=[6,1],ymargin=[1,1]
endfor
;
if !d.name eq 'PS' then device,/close
;
end
;-------------------------------------------------------------------------------
pro longcal
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
bl=200	; m
wl=1.6e-6	; m
uvdist=bl/wl
n=400
UD=(findgen(n)+1)/n*1.0
sc=[0.05,0.02,0.01]
m=20
sd=(findgen(m)+1)/100
dmax=fltarr(3,m)
for i=0,2 do begin
for k=0,m-1 do begin
for j=0,n-1 do begin
	arg=factor*uvdist*UD[j]*(1-sd[k])
	visamp=2*beselj(arg,1)/arg
	vissql=visamp^2
	arg=factor*uvdist*UD[j]*(1+sd[k])
	visamp=2*beselj(arg,1)/arg
	vissqh=visamp^2
	if abs(vissqh-vissql)/((vissqh+vissql)/2) lt sc[i] then dmax[i,k]=UD[j]
endfor
endfor
endfor
;
!p.charsize=1.5
plot,sd*100,dmax[0,*], $
	xtitle='Uncertainty of calibrator diameter [%]', $
	ytitle='Maximum allowable calibrator diameter [mas]', $
	title='Calibration of V!E2!N on a 200 m baseline in J-band'
oplot,sd*100,dmax[1,*]
oplot,sd*100,dmax[2,*]
;
xyouts,5,0.6,'to a precision of 5%'
xyouts,5,0.35,'2%'
xyouts,5,0.15,'1%'
;
end
;-------------------------------------------------------------------------------
pro doncal
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
bl=90		; m
wl=0.8e-6	; m
uvdist=bl/wl
n=400
dsci=(findgen(n)+1)/n*9.0	; Diameter of SCI in range of [3,5]
;
; Check that even for the largest science target diameter, NULL does not occur
arg=factor*uvdist*dsci
vissqr=(2*beselj(arg,1)/arg)^2
index=where(vissqr gt 0.05)
dsci=dsci[index]
n=n_elements(dsci)
;
dcal=(findgen(n)+1)/n*1.0	; Diameter of CAL in range of [0,1] mas
;
sig_sci=[0.05,0.02,0.01]
; Determine error on v2 related to 10% error on CAL diameter
arg=factor*uvdist*dcal*1.1
vissqr_lo=(2*beselj(arg,1)/arg)^2
arg=factor*uvdist*dcal*0.9
vissqr_hi=(2*beselj(arg,1)/arg)^2
viserr=(vissqr_hi-vissqr_lo)/(vissqr_hi+vissqr_lo)
dlim=dsci & dlim[*]=max(dsci)
;
for i=n-1,0,-1 do begin
;	Determine v2 of SCI
	arg=factor*uvdist*dsci[i]
	vissqr=(2*beselj(arg,1)/arg)^2
	vissqr_hi=vissqr*(1+viserr)
	vissqr_lo=vissqr*(1-viserr)
	dsci_lo=hb(uvdist/1e6,vissqr_hi)
	dsci_hi=hb(uvdist/1e6,vissqr_lo)
	derr=((dsci_hi-dsci_lo)/(dsci_hi+dsci_lo))*100
	index=where(derr lt 1.0,count)
	if count ge 1 then dlim[index]=dsci[i]
endfor
;
!p.charsize=1.0
!y.range=[0,3]
plot,dcal,dlim, $
	xtitle='Calibrator diameter [mas]', $
	ytitle='Minimum SCI diameter for 1% error', $
	title='CAL diameter error 10%, baseline ' $
		+strtrim(string(bl,format=i3),2)+' m'
;
end
;-------------------------------------------------------------------------------
pro hanbury,ld_coeff
;
; Plot stellar disk visibilities.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
UD=5.0
uvdist=(findgen(100)+1)*1e6
arg=factor*uvdist*UD
visamp=2*beselj(arg,1)/arg
vissq=visamp^2
;plot,uvdist,alog10(vissq)
!p.charsize=1.5
plot,uvdist,vissq,ytype=1,xtitle='uv radius [wavelengths]',ytitle='V^2'
;
if n_elements(ld_coeff) eq 0 then return
UD1=sqrt((1-ld_coeff/3)/(1-7*ld_coeff/15))*UD
alpha=1-ld_coeff
beeta=ld_coeff
arg=factor*uvdist*UD1
visamp=(alpha*beselj(arg,1)/arg+beeta*sqrt(pi_circle/2)* $
       sqrt(2/(pi_circle*arg))*(sin(arg)/arg-cos(arg))/ $
       sqrt(arg*arg*arg))/(alpha/2+beeta/3)
vissq=visamp^2
;oplot,uvdist,alog10(vissq),linestyle=2 
oplot,uvdist,vissq,linestyle=2 
;
end
;-------------------------------------------------------------------------------
pro brown
;
; Prepare grid of uv spacings and visibilities and corresponding diameters
; for use in function hb.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
uvdist=(findgen(100)+1)*1e6
UD=findgen(100)/10+0.1
r0=fltarr(n_elements(uvdist))
r1=fltarr(n_elements(uvdist))
r2=fltarr(n_elements(uvdist))
r3=fltarr(n_elements(uvdist))
for i=0,n_elements(uvdist)-1 do begin
	arg=factor*uvdist[i]*UD
	visamp=2*beselj(arg,1)/arg
	vissq=visamp^2
;	plot,vissq,UD,psym=1
	index=where(vissq gt 0.1 and vissq lt 0.9,count)
	if count gt 3 then begin
		r=poly_fit(vissq[index],ud[index],3,yfit)
		r0[i]=r[0]
		r1[i]=-r[1]
		r2[i]=r[2]
		r3[i]=-r[3]
;		oplot,vissq(index),yfit,psym=0,color=3
	endif
endfor
;
!p.multi=[0,1,4]
index=where(r0 ne 0 and r1 ne 0 and r2 ne 0 and r3 ne 0)
uvdist=uvdist[index]/1e6
r0=alog(r0[index])
r1=-alog(r1[index])
r2=alog(r2[index])
r3=alog(r3[index])
;
plot,uvdist,r0,psym=1
index=where(uvdist le 18)
print,poly_fit(uvdist[index],r0[index],3,yfit)
oplot,uvdist[index],yfit,psym=0,color=2
index=where(uvdist ge 18)
print,poly_fit(uvdist[index],r0[index],3,yfit)
oplot,uvdist[index],yfit,psym=0,color=3
;
plot,uvdist,r1,psym=1
index=where(uvdist le 18)
print,poly_fit(uvdist[index],r1[index],3,yfit)
oplot,uvdist[index],yfit,psym=0,color=2
index=where(uvdist ge 18)
print,poly_fit(uvdist[index],r1[index],3,yfit)
oplot,uvdist[index],yfit,psym=0,color=3
;
plot,uvdist,r2,psym=1
index=where(uvdist le 18)
print,poly_fit(uvdist[index],r2[index],3,yfit)
oplot,uvdist[index],yfit,psym=0,color=2
index=where(uvdist ge 18)
print,poly_fit(uvdist[index],r2[index],3,yfit)
oplot,uvdist[index],yfit,psym=0,color=3
;
plot,uvdist,r3,psym=1
index=where(uvdist le 18)
print,poly_fit(uvdist[index],r3[index],3,yfit)
oplot,uvdist[index],yfit,psym=0,color=2
index=where(uvdist ge 18)
print,poly_fit(uvdist[index],r3[index],3,yfit)
oplot,uvdist[index],yfit,psym=0,color=3
;
end
;-------------------------------------------------------------------------------
function udvis,spacfreq,ud,dvis
;
; Return modulo of visibility amplitude (not squared!) for input
; spacial frequencies (in 1/arcsec) and uniform disk diameter (in mas).
; Also compute the derivatives with respect to the diameter.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=pi_circle/1000
;
arg=factor*spacfreq*ud
visamp=abs(2*beselj(arg,1)/arg)
;
if n_params() eq 3 then begin
	f_incr=1.01
	arg=factor*spacfreq*(ud*f_incr)
	visamph=abs(2*beselj(arg,1)/arg)
	arg=factor*spacfreq*(ud/f_incr)
	visampl=abs(2*beselj(arg,1)/arg)
	dvis=(visamph-visampl)/(ud*f_incr-ud/f_incr)
endif
;
return,visamp
;
end
;-------------------------------------------------------------------------------
pro udfit,spacfreq,visamp,visampe,ud
;
; Fit uniform disk diameter (in mas) to visibility amplitude.
; Spacial frequencies are in 1/arcsec.
;
nvis=n_elements(visamp)
;
if n_elements(ud) eq 0 then begin
	ud=0.
	f=(180l*3600l/!pi)/1e6
	for i=0,nvis-1 do ud=ud+hb(spacfreq[i]*f,visamp[i]^2)
	ud=ud/nvis
endif
;
tol=0.0001
;
visampf=udvis(spacfreq,ud,d)
tchisq=total(((visamp-visampf)/visampe)^2)
print,'Begin with ud, total chisq=',ud,tchisq
;
repeat begin
	tchisqo=tchisq
	dm=d/visampe
	rh=(visamp-visampf)/visampe
	n=transpose(dm)#dm
	r=transpose(dm)#rh
	ud=ud+r[0]/n[0]
	visampf=udvis(spacfreq,ud,d)
	tchisq=total(((visamp-visampf)/visampe)^2)
endrep until 1-tchisq/tchisqo lt tol
ude=sqrt(1/n[0])
rchisq=tchisq/(nvis-1)
help,ud,ude,tchisq,rchisq
;
end
;-------------------------------------------------------------------------------
pro ewbaseline
;
; For 1-baseline observations, using EW, re-format configuration and data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if n_elements(pointtime) ne 0 then begin
	referencestation,'AE0','point'
	genconfig.stationid[0:1]=genconfig.stationid[1:2]
	genconfig.stationid[2]=''
	genconfig.stationcoord[*,0:1]=genconfig.stationcoord[*,1:2]
	genconfig.numsid=2
	genconfig.numoutbeam=1
	genconfig.numtriple=0
	genconfig.refstation=1
	fdlpos[0,*]=fdlpos[1,*]
	fdlpos[1,*]=fdlpos[2,*]
	fdlpos=fdlpos[0:1,*]
	fdlposerr[0,*]=fdlposerr[1,*]
	fdlposerr[1,*]=fdlposerr[2,*]
	fdlposerr=fdlposerr[0:1,*]
;	metropos(0,*)=metropos(1,*)
;	metropos(1,*)=metropos(2,*)
;	metroposerr(0,*)=metroposerr(1,*)
;	metroposerr(1,*)=metroposerr(2,*)
	natjitter[0,*]=natjitter[1,*]
	natjitter[1,*]=natjitter[2,*]
	natjittererr[0,*]=natjittererr[1,*]
	natjittererr[1,*]=natjittererr[2,*]
endif
;
end
;-------------------------------------------------------------------------------
pro cebaseline
;
; For 1-baseline observations, using CE, re-format configuration and data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if n_elements(pointtime) ne 0 then begin
	genconfig.numsid=2
	genconfig.numoutbeam=1
	genconfig.numtriple=0
	genconfig.refstation=1
	genconfig.baselineid[0]='AC0-AE0'
	vissq[0,*,0,*]=vissq[2,*,0,*]
	vissqerr[0,*,0,*]=vissqerr[2,*,0,*]
	visphase[0,*,0,*]=visphase[2,*,0,*]
	visphaseerr[0,*,0,*]=visphaseerr[2,*,0,*]
	visamp[0,*,0,*]=visamp[2,*,0,*]
	visamperr[0,*,0,*]=visamperr[2,*,0,*]
	photonrate[0,*,*]=photonrate[2,*,*]
	photonrateerr[0,*,*]=photonrateerr[2,*,*]
endif
;
end
;-------------------------------------------------------------------------------
pro cwbaseline
;
; For 1-baseline observations, using CW, re-format configuration and data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if n_elements(pointtime) ne 0 then begin
	genconfig.numsid=2
	genconfig.numoutbeam=1
	genconfig.numtriple=0
	genconfig.refstation=1
	genconfig.stationid[1]='AW0'
	genconfig.baselineid[0]='AC0-AW0'
	vissq[0,*,0,*]=vissq[1,*,0,*]
	vissqerr[0,*,0,*]=vissqerr[1,*,0,*]
	visphase[0,*,0,*]=visphase[1,*,0,*]
	visphaseerr[0,*,0,*]=visphaseerr[1,*,0,*]
	visamp[0,*,0,*]=visamp[1,*,0,*]
	visamperr[0,*,0,*]=visamperr[1,*,0,*]
	photonrate[0,*,*]=photonrate[1,*,*]
	photonrateerr[0,*,*]=photonrateerr[1,*,*]
	fdlpos[1,*]=fdlpos[2,*]
	fdlposerr[1,*]=fdlposerr[2,*]
	metropos[1,*]=metropos[2,*]
	metroposerr[1,*]=metroposerr[2,*]
	natjitter[1,*]=natjitter[2,*]
	natjittererr[1,*]=natjittererr[2,*]
endif
;
end
;-------------------------------------------------------------------------------
pro dana
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
!p.psym=10
;
index502=where(scans.starid eq 'FKV0502')
index509=where(scans.starid eq 'FKV0509')
;
r502=scans[index502].fdlpos(1)-scans[index502].geodelay(1)
r509=scans[index509].fdlpos(1)-scans[index509].geodelay(1)
r502=r502*1e6
r509=r509*1e6
d2=r502-r509
x2=findgen(15)*2-14
h2=histogram(d2,binsize=2,min=-15,max=15)
e2=sqrt(h2)
plot,x2,h2,xtitle='Residual delay difference [microns]',ytitle='N'
oploterr,x2,h2,e2,3
;
r502=scans[index502].fdlpos(2)-scans[index502].geodelay(2)
r509=scans[index509].fdlpos(2)-scans[index509].geodelay(2)
r502=r502*1e6
r509=r509*1e6
d3=r502-r509
x3=findgen(15)*2-14
h3=histogram(d3,binsize=2,min=-15,max=15)
e3=sqrt(h3)
plot,x3,h3,xtitle='Residual delay difference [microns]',ytitle='N'
oploterr,x3,h3,e3,3
;
end
;-------------------------------------------------------------------------------
pro binplot
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
!p.multi=[0,1,5]
!p.charsize=2.0
!p.psym=0
;
lambda_grid=(findgen(250)*2+401)*1e-9  ; [m]
num_grid=n_elements(lambda_grid)
fluxes_primary=stellarfluxes(star_model[0],lambda_grid,ld_primary)
fluxes_secondary=stellarfluxes(star_model[1],lambda_grid,ld_secondary)
;
window,/free,title='Binary model photometry',xsize=500,ysize=800
;
plot,lambda_grid*1e9,fluxes_primary,psym=0, $
	title=star_model[0].component,ytitle='Flux'
plot,lambda_grid*1e9,fluxes_secondary,psym=0, $
	title=star_model[1].component,ytitle='Flux'
dm=-2.5*alog10(fluxes_secondary/fluxes_primary)
plot,lambda_grid*1e9,dm,psym=0, $
	ytitle='Magnitude difference'
for j=0,GenConfig.NumSpecChan[0]-1 do begin
	x=GenConfig.Wavelength[j,0]*1e9
	oplot,[x,x],[-3,3],psym=0,linestyle=1
endfor
plot,lambda_grid*1e9,ld_primary, $
	title=star_model[0].component,ytitle='LD coeff.'
plot,lambda_grid*1e9,ld_secondary, $
	title=star_model[1].component,ytitle='LD coeff.'
;
end
;************************************************************************Block 2
pro deane
;
; Print raw scan triple visibilities to file.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
rad=180/!pi
;
filename='deane.dat'
openw,unit,filename,/get_lun
;
index=where(scans.starid eq 'FKV0699')
scans_bck=scans
scans=scans[index]
for i=0,n_elements(scans)-1 do begin
for j=0,31 do begin
	if scans[i].tripleampcerr(0,j) gt 0 then begin
	printf,unit,genconfig.wavelength[j,0]*1e9, $
		scans[i].ha, $
		scans[i].vissqc(0,j,0),scans[i].vissqcerr(0,j,0), $
		scans[i].uvw(0,j,0,0,0)/1e6,scans[i].uvw(0,j,0,1,0)/1e6, $
		scans[i].vissqc(1,j,0),scans[i].vissqcerr(1,j,0), $
		scans[i].uvw(1,j,0,0,0)/1e6,scans[i].uvw(1,j,0,1,0)/1e6, $
		scans[i].vissqc(2,j,0),scans[i].vissqcerr(2,j,0), $
		scans[i].uvw(2,j,0,0,0)/1e6,scans[i].uvw(2,j,0,1,0)/1e6, $
		scans[i].tripleampc(0,j),scans[i].tripleampcerr(0,j), $
		scans[i].triplephasec(0,j)*rad,scans[i].triplephasecerr(0,j)*rad, $
	format='(f5.1,1x,f5.2,1x,3(2(f7.4,1x),2(f7.3,1x)),2(f6.4,1x),2(f6.1,1x))'
	endif
endfor
endfor
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro dave,chafile
;
; Print raw scan triple visibilities to file.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
get_scandata,chafile
hds_close
;
filename='phase.dat'
openw,unit,filename,/get_lun
;
for i=0,n_elements(scans)-1 do begin
	printf,unit,scans[i].time/3600,scans[i].starid, $
		scans[i].compltriple(0,*,0:19), $
		scans[i].compltripleerr(0,*,0:19), $
		format='(f7.4,1x,a9,2x,80(f8.4,1x))'
endfor
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro nick
;
; For 1999-03-02, divide point data up several scans.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
Rec0=[0,196,242,293,338,382,394,426,492,543,591,650]
RecN=[195,241,292,337,381,393,425,491,542,590,649,869]
Iscan=indgen(n_elements(Rec0))+1
StarId=strarr(n_elements(Rec0))
StarId[*]='FKV0000'
;
StartTime=PointTime[Rec0]
StopTime=PointTime[RecN]
ScanId=Iscan
NumPoint=RecN-Rec0+1
NumCoh=lonarr(n_elements(Rec0))+1
NumIncoh=lonarr(n_elements(Rec0))+500
;
scantable=build_scantable(Iscan,ScanId,StarId,StartTime,StopTime,NumPoint, $
                NumCoh,NumIncoh)
;
end
;-------------------------------------------------------------------------------
pro nick2
;
; For 1999-03-05, divide point data up several scans.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
Rec0=[ $
           0   ,      87   ,     176   ,     $
	532,612,678,742,840   ,     958   ,    1378, $
        1550   ,    1727   ,    1882   ,    1999   ,    2165   ,    2204, $
        2363   ,    2514   ,    2686   ,    2845   ,    2910   ,    2998, $
        3011   ,    3154   ,    3334   ,    3363   ,    3523   ,    3679, $
        3837   ,    3983   ,    4130   ,    4277   ,    4439   ,    4605, $
        4774   ,    4951   ,    5124                                      ]     
RecN=[ $
          86   ,     175   ,     531   ,     $
	611,677,741,839,957   ,    1377   ,    1549, $
        1726   ,    1881   ,    1998   ,    2164   ,    2203   ,    2362, $
        2513   ,    2685   ,    2844   ,    2909   ,    2997   ,    3010, $
        3153   ,    3333   ,    3362   ,    3522   ,    3678   ,    3836, $
        3982   ,    4129   ,    4276   ,    4438   ,    4604   ,    4773, $
        4950   ,    5123   ,    5289                                      ]     

Iscan=indgen(n_elements(Rec0))+1
starid=[starid[0:2],['FKV0120','FKV0120','FKV0120','FKV0120','FKV0120'], $
	starid[4:32]]
;
StartTime=PointTime[Rec0]
StopTime=PointTime[RecN]
ScanId=Iscan
NumPoint=RecN-Rec0+1
NumCoh=lonarr(n_elements(Rec0))+1
NumIncoh=lonarr(n_elements(Rec0))+500
;
scantable=build_scantable(Iscan,ScanId,StarId,StartTime,StopTime,NumPoint, $
                NumCoh,NumIncoh)
;
end
;-------------------------------------------------------------------------------
pro don3
;
; Prepare plots for Don of metrology data.
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common MetroData,fbairtemp,fbsolidtmp
;
siderostat=[', Center',', East',', West']
;
;if checkdata([8,11]) ne 0 then return
;
tmin=min(pointtime[where(pointtime ge 0)]/3600)
tmax=max(pointtime[where(pointtime ge 0)]/3600)
if total(abs(!x.range)) eq 0 then !x.range=[tmin,tmax]
if total(abs(!y.range)) eq 0 then !y.range=[-10,30]
xpos=tmin-1.0*(tmax-tmin)
;
; The delay and temperature data have different time stamps, so interpolate them
num_t=20000
tgrid=findgen(num_t)/num_t*(tmax-tmin)+tmin
air_d=fltarr(genconfig.numsid,num_t)
air_m=intarr(genconfig.numsid)
air_n=intarr(genconfig.numsid)
sol_d=fltarr(genconfig.numsid,9,num_t)
sol_n=intarr(genconfig.numsid)
;
if !d.name eq 'PS' then begin
        width=17.78
        factor=width/640
        device,xsize=factor*640,ysize=factor*900, $
                xoffset=1,yoffset=1,color=1
	xpos=xpos-1.0*(tmax-tmin)
	frame_color=0
endif else begin
	frame_color=1
endelse
tek_color
!p.charsize=1.0
!x.margin=[53,1]
!x.title='UT [h]'
!x.ticks=0
!y.ticklen=0
;
; Prepare plots for siderostats
!y.title='Temperature [C]'
!p.multi=[0,1,1]
ymin=100
ymax=100
;
for i=0,genconfig.numsid-1 do begin
	!p.title=date+siderostat[i]
	if !d.name eq 'PS' then begin
		device,filename='sid'+string(i+1,format='(i1)')+'.ps'
	endif else window,i
;
	d=fbairtemp.data[*,0,i]
	t=fbairtemp.timestamp[*,i]
	index=where(tgrid lt max(t),count)
	air_n[i]=count
	index=where(tgrid lt min(t),count)
	air_m[i]=count
	si=sort(t) & t=t[si] & d=d[si]
	air_d[i,*]=interpol(d,t,tgrid)
;
	plot,t,d,/nodata,color=frame_color,xstyle=0,ystyle=1,psym=3
	oplot,t,d,color=2,psym=3
	k=where(t ge tmin)
	xyouts,xpos,d[k[0]],metroconfig.fbairtemp.label[i],color=2
;
	if ymin eq 100 then begin
		ymin=min(d)
		ymax=max(d)
	endif else begin
		ymin=min([ymin,min(d)])
		ymax=max([ymax,max(d)])
	endelse
;
	d=fbsolidtmp.data[*,*,i]
	t=fbsolidtmp.timestamp[*,i]
	index=where(tgrid lt max(t),count)
	sol_n[i]=count
	si=sort(t) & t=t[si] & d=d[si,*]
	for j=0,n_elements(d[0,*])-1 do begin
		d[*,j]=median(d[*,j],11)
		sol_d[i,j,*]=interpol(d[*,j],t,tgrid)
		offset=0
		if j eq 5 then offset=11
		oplot,t,d[*,j],color=2+j+offset,psym=3
		ymin=min([ymin,min(d[*,j])])
		ymax=max([ymax,max(d[*,j])])
		k=where(t ge tmin)
		xyouts,xpos,d[k[0],j],metroconfig.fbsolidtmp.label[j,i],color=2+j+offset
	endfor
	if !d.name eq 'PS' then device,/close
;
endfor
;
!y.range=[ymin,ymax]
;
; Return if there is no white light data
index=where(starid eq 'FKV0000',count)
if count eq 0 then return
;
; Prepare plots for baselines
!y.title='Temperature difference [C]'
!p.multi=[0,1,2]
;
ibr=genconfig.refstation-1
;
for i=0,genconfig.numoutbeam-1 do begin
	if !d.name eq 'PS' then begin
		device,filename='bas'+string(i+1,format='(i1)')+'.ps'
	endif else window,i+3
	!p.title=date+', '+genconfig.baselineid[i]
;
;	Design matrix for least squares solution
	m=dblarr(n_elements(tgrid),11)
	m[*,10]=1
	clabels=strarr(11) & clabels[10]='Constant'
	clabels[0]=metroconfig.fbairtemp.label[i]
;
;	Plot temperature differences
	j1=where(genconfig.stationid eq strmid(genconfig.baselineid[i],0,3))
	j2=where(genconfig.stationid eq strmid(genconfig.baselineid[i],4,3))
	j1=j1[0]
	j2=j2[0]
	y=air_d[j2,*]-air_d[j1,*]
	m[*,0]=y
	if i eq 0 then begin
	  miny=min(sol_d[j2,[0,1,2,3,5,6,7,9],0:min(sol_n)-1] $
		  -sol_d[j1,[0,1,2,3,5,6,7,9],0:min(sol_n)-1])
	  maxy=max(sol_d[j2,[0,1,2,3,5,6,7,9],0:min(sol_n)-1] $
		  -sol_d[j1,[0,1,2,3,5,6,7,9],0:min(sol_n)-1])
	endif else begin
	  minys=min(sol_d[j2,[0,1,2,3,5,6,7,9],0:min(sol_n)-1] $
		   -sol_d[j1,[0,1,2,3,5,6,7,9],0:min(sol_n)-1])
	  maxys=max(sol_d[j2,[0,1,2,3,5,6,7,9],0:min(sol_n)-1] $
		   -sol_d[j1,[0,1,2,3,5,6,7,9],0:min(sol_n)-1])
	  minya=min(y[air_m[i]+1:air_n[i]-1])
	  maxya=max(y[air_m[i]+1:air_n[i]-1])
	  miny=min([minys,minya])
	  maxy=max([maxys,maxya])
	endelse
	plot,tgrid,y,/nodata,yrange=[miny,maxy],color=frame_color, $
		xstyle=0,ystyle=1
	oplot,tgrid[air_m[i]+1:air_n[i]-1],y[air_m[i]+1:air_n[i]-1],color=2,psym=3
	xyouts,xpos,y[1],metroconfig.fbairtemp.label[i],color=2
;
	for j=0,n_elements(sol_d[i,*,0])-1 do begin
		clabels[j+1]=metroconfig.fbsolidtmp.label[j,2]
		if i eq 0 or (j ne 4 and j ne 8) then begin
			offset=0
			if j eq 5 then offset=11
			y=sol_d[j2,j,*]-sol_d[j1,j,*]
			m[*,j+1]=y
			oplot,tgrid[0:min(sol_n)-1],y[0:min(sol_n)-1],color=2+j+offset,psym=3
			xyouts,xpos,y[1],metroconfig.fbsolidtmp.label[j,2],color=2+j+offset
		endif
	endfor
;
	windex=intarr(n_elements(pointtime))
	for j=0,n_elements(scantable)-1 do $
		if starid[j] eq 'FKV0000' then windex[rec0[j]:recn[j]]=1
	windex=where(windex eq 1)
	time=pointtime[windex]/3600
	delay=(fdlpos[j2,windex]*(j2 ne ibr))-(fdlpos[j1,windex]*(j1 ne ibr)) 
	index=where(signof(fdlposerr[j2,windex],fdlposerr[j1,windex]) gt 0)
	delay=delay[index]
	delay=(delay-avg(delay))*1e6
	time=time[index]
	d=interpol(delay,time,tgrid)
;
;	For 2000-11-30 read pressure data from /home/cah/mnr1130.xdr'
	if date eq '2000-11-32' then begin
	restore,'/home/cah/mnr1130.xdr'
	p=interpol(pressure,ptime,tgrid)
	m[*,0]=p		; use column for air temps.
	irow=0
	endif else irow=1	; i.e. do not use air temps.
;
;	Select data for least squares solution
	t=double(tgrid)
	m=m[*,irow:10]
	clabels=clabels[irow:10]
	index=where(total(m,1) ne 0)
	index=[0,3,5,9]
	index=[0,1,2,3,5,6,7,9]
	countr=n_elements(index)
	m=m[*,index]
	clabels=clabels[index]
	index=where(time lt min(tgrid[sol_n]))
	mm=dblarr(n_elements(time[index]),n_elements(m[0,*]))
	for j=0,n_elements(mm[0,*])-1 do mm[*,j]=interpol(m[*,j],t,time[index])
	tm=transpose(mm)
	n=tm#mm
	r=tm#delay[index]
	s=invert(n)#r
;	svdc,n,w,u,v,/double
;	s=svsol(u,w,v,r,/double)
;
;	Now setup solution with data sampled every 30 minutes
	if n_elements(unique(scantable.starid)) eq 1 then begin
	ut=unique(fix(time))
	alloc=0
	for k=0,n_elements(ut)-1 do begin
		index=where(abs(time-ut[k]) lt 0.02,count)
		if count gt 0 then begin
			if alloc eq 0 then begin
				tindex=index
				alloc=1
			endif else tindex=[tindex,index]
		endif
		index=where(abs(time-(ut[k]+0.5)) lt 0.02,count)
		if count gt 0 then begin
			if alloc eq 0 then begin
				tindex=index
				alloc=1
			endif else tindex=[tindex,index]
		endif
	endfor
	time_e=time[tindex]
	delay_e=delay[tindex]
	mm=dblarr(n_elements(time_e),n_elements(m[0,*]))
	for j=0,n_elements(mm[0,*])-1 do mm[*,j]=interpol(m[*,j],t,time_e)
	tm_e=transpose(mm)
	n_e=tm_e#mm
	r_e=tm_e#delay_e
	s_e=invert(n_e)#r_e
;	svdc,n_e,w,u,v,/double
;	s_e=svsol(u,w,v,r_e,/double)
	endif
;
	for k=0,countr-1 do begin
		clabels[k]=string(s[k],format='(f7.2)')+' ('+clabels[k]+')'
;		print,'Baseline ',i,'  ',clabels(k)
	endfor
;
;	Plot the delay data with the model fits
	plot,t,m#s,/nodata,psym=3,yrange=[min(delay),max(delay)],xstyle=0, $
		ytitle='Residual delay [microns]',title='',color=frame_color
	oplot,t[0:min(sol_n)-1],smooth(m[0:min(sol_n)-1,*]#s,101),psym=3,color=13
	if n_elements(unique(scantable.starid)) eq 1 then $
	oplot,t,smooth(m#s-m#s_e,101)+avg(m#s),psym=3,color=15
	oplot,time,delay,psym=3,color=frame_color
;
	xyouts,xpos,!y.crange[1]+0.05*(!y.crange[1]-!y.crange[0]), $
		strjoin(string(s,format='(10f8.2)'))
	xyouts,xpos,!y.crange[1]-0.1*(!y.crange[1]-!y.crange[0]), $
		'RMS(fit)='+string(stdev(m#s-d))
	if n_elements(unique(scantable.starid)) eq 1 then $
	xyouts,xpos,!y.crange[1]-0.2*(!y.crange[1]-!y.crange[0]), $
		'RMS(mod)='+string(stdev(m#s-m#s_e))
;
;	!y.range=0
;	plot,t,m#s,/nodata,psym=3,yrange=[min(delay),max(delay)],xstyle=1, $
;		ytitle='Residual delay [microns]',title='',color=frame_color
;	oplot,t,m#s_e,psym=3,color=13
;	oplot,time,delay,psym=3,color=frame_color
;
	if !d.name eq 'PS' then device,/close
;
endfor
;
end
;-------------------------------------------------------------------------------
pro don4
;
; Prepare plots for Don of metrology data.
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common MetroData,fbairtemp,fbsolidtmp
;
siderostat=[', Center',', East',', West',', North']
;
;if checkdata([8,11]) ne 0 then return
;
tmin=min(pointtime[where(pointtime ge 0)]/3600)
tmax=max(pointtime[where(pointtime ge 0)]/3600)
if total(abs(!x.range)) eq 0 then !x.range=[tmin,tmax]
if total(abs(!y.range)) eq 0 then !y.range=[-10,30]
xpos=tmin-1.5*(tmax-tmin)
;
; The delay and temperature data have different time stamps, so interpolate them
num_t=20000
tgrid=findgen(num_t)/num_t*(tmax-tmin)+tmin
air_d=fltarr(genconfig.numsid,num_t)
air_m=intarr(genconfig.numsid)
air_n=intarr(genconfig.numsid)
sol_d=fltarr(genconfig.numsid,9,num_t)
sol_n=intarr(genconfig.numsid)
;
if !d.name eq 'PS' then begin
        width=17.78
        factor=width/640
        device,xsize=factor*640,ysize=factor*900, $
                xoffset=1,yoffset=1,color=1
	xpos=xpos-1.0*(tmax-tmin)
	frame_color=0
endif else begin
	frame_color=1
endelse
tek_color
!p.charsize=1.0
!x.margin=[53,1]
!x.title='UT [h]'
!x.ticks=0
!y.ticklen=0
;
; Prepare plots for siderostats
!y.title='Temperature [C]'
!p.multi=[0,1,1]
ymin=100
ymax=100
;
for i=0,genconfig.numsid-1 do begin
	!p.title=date+siderostat[i]
	if !d.name eq 'PS' then begin
		device,filename='sid'+string(i+1,format='(i1)')+'.ps'
	endif else window,i
;
	d=fbairtemp.data[*,0,i] & d=median(d,7)
	t=fbairtemp.timestamp[*,i]
	index=where(tgrid lt max(t),count)
	air_n[i]=count
	index=where(tgrid lt min(t),count)
	air_m[i]=count
	si=sort(t) & t=t[si] & d=d[si]
	air_d[i,*]=interpol(d,t,tgrid)
;
	plot,t,d,/nodata,color=frame_color,xstyle=0,ystyle=1,psym=3
	oplot,t,d,color=2,psym=3
	k=where(t ge tmin)
	xyouts,xpos,d[k[0]],metroconfig.fbairtemp.label[i],color=2
;
	if ymin eq 100 then begin
		ymin=min(d)
		ymax=max(d)
	endif else begin
		ymin=min([ymin,min(d)])
		ymax=max([ymax,max(d)])
	endelse
;
	d=fbsolidtmp.data[*,*,i]
	t=fbsolidtmp.timestamp[*,i]
	index=where(tgrid lt max(t),count)
	sol_n[i]=count
	si=sort(t) & t=t[si] & d=d[si,*]
	for j=0,n_elements(d[0,*])-1 do begin
		d[*,j]=median(d[*,j],11)
		sol_d[i,j,*]=interpol(d[*,j],t,tgrid)
		offset=0
		if j eq 5 then offset=11
		oplot,t,d[*,j],color=2+j+offset,psym=3
		ymin=min([ymin,min(d[*,j])])
		ymax=max([ymax,max(d[*,j])])
		k=where(t ge tmin)
		xyouts,xpos,d[k[0],j],metroconfig.fbsolidtmp.label[j,i],color=2+j+offset
	endfor
	if !d.name eq 'PS' then device,/close
;
endfor
;
!y.range=[ymin,ymax]
;
; Return if there is no white light data
index=where(starid eq 'FKV0000',count)
if count eq 0 then return
;
; Prepare plots for baselines
!y.title='Temperature difference [C]'
!p.multi=[0,1,2]
;
ibr=genconfig.refstation-1
;
for i=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline[i]-1 do begin
;
;	Design matrix for least squares solution
	m=dblarr(n_elements(tgrid),11)
	m[*,10]=1
	clabels=strarr(11) & clabels[10]='Constant'
	clabels[0]=metroconfig.fbairtemp.label[0]
;
;	Determine station indices
	j1=where(genconfig.stationid eq strmid(genconfig.baselineid[l,i],0,3))
	j2=where(genconfig.stationid eq strmid(genconfig.baselineid[l,i],4,3))
	j1=j1[0]
	j2=j2[0]
;
; 	See if we have the delay data
	windex=intarr(n_elements(pointtime))
	for j=0,n_elements(scantable)-1 do $
		if starid[j] eq 'FKV0000' then windex[rec0[j]:recn[j]]=1
	windex=where(windex eq 1)
	time=pointtime[windex]/3600
	delay=(fdlpos[j2,windex]*(j2 ne ibr))-(fdlpos[j1,windex]*(j1 ne ibr)) 
	index=where(signof(fdlposerr[j2,windex],fdlposerr[j1,windex]) gt 0,count)
	if count eq 0 then goto,SKIPPLOT
	delay=delay[index]
	delay=(delay-avg(delay))*1e6
	time=time[index]
	d=interpol(delay,time,tgrid)
;
;	Open plot window
	if !d.name eq 'PS' then begin
		device,filename='bas'+string((i+1)*10+l+1,format='(i2.2)')+'.ps'
	endif else window,/free
	!p.title=date+', '+genconfig.baselineid[l,i]
;
;	Plot temperature differences
	y=air_d[j2,*]-air_d[j1,*]
	m[*,0]=y
	if i eq 0 then begin
	  miny=min(sol_d[j2,[0,1,2,3,5,6,7,9],0:min(sol_n)-1] $
		  -sol_d[j1,[0,1,2,3,5,6,7,9],0:min(sol_n)-1])
	  maxy=max(sol_d[j2,[0,1,2,3,5,6,7,9],0:min(sol_n)-1] $
		  -sol_d[j1,[0,1,2,3,5,6,7,9],0:min(sol_n)-1])
	endif else begin
	  minys=min(sol_d[j2,[0,1,2,3,5,6,7,9],0:min(sol_n)-1] $
		   -sol_d[j1,[0,1,2,3,5,6,7,9],0:min(sol_n)-1])
	  maxys=max(sol_d[j2,[0,1,2,3,5,6,7,9],0:min(sol_n)-1] $
		   -sol_d[j1,[0,1,2,3,5,6,7,9],0:min(sol_n)-1])
	  minya=min(y[air_m[i]+1:air_n[i]-1])
	  maxya=max(y[air_m[i]+1:air_n[i]-1])
	  miny=min([minys,minya])
	  maxy=max([maxys,maxya])
	endelse
	plot,tgrid,y,/nodata,yrange=[miny,maxy],color=frame_color, $
		xstyle=0,ystyle=1
	oplot,tgrid[air_m[i]+1:air_n[i]-1],y[air_m[i]+1:air_n[i]-1],color=2,psym=3
	xyouts,xpos,y[1],metroconfig.fbairtemp.label[i],color=2
;
	for j=0,n_elements(sol_d[i,*,0])-1 do begin
		clabels[j+1]=metroconfig.fbsolidtmp.label[j,2]
		if i eq 0 or (j ne 4 and j ne 8) then begin
			offset=0
			if j eq 5 then offset=11
			y=sol_d[j2,j,*]-sol_d[j1,j,*]
			m[*,j+1]=y
			oplot,tgrid[0:min(sol_n)-1],y[0:min(sol_n)-1],color=2+j+offset,psym=3
			xyouts,xpos,y[1],metroconfig.fbsolidtmp.label[j,2],color=2+j+offset
		endif
	endfor
;
;	For 2000-11-30 read pressure data from /home/cah/mnr1130.xdr'
	if date eq '2000-11-32' then begin
	restore,'/home/cah/mnr1130.xdr'
	p=interpol(pressure,ptime,tgrid)
	m[*,0]=p		; use column for air temps.
	irow=0
	endif else irow=1	; i.e. do not use air temps.
;
;	Select data for least squares solution
	t=double(tgrid)
	m=m[*,irow:10]
	clabels=clabels[irow:10]
	index=where(total(m,1) ne 0)
	index=[0,3,5,9]
	index=[0,1,2,3,5,6,7,9]
	countr=n_elements(index)
	m=m[*,index]
	clabels=clabels[index]
	index=where(time lt min(tgrid[sol_n]))
	mm=dblarr(n_elements(time[index]),n_elements(m[0,*]))
	for j=0,n_elements(mm[0,*])-1 do mm[*,j]=interpol(m[*,j],t,time[index])
	tm=transpose(mm)
	n=tm#mm
	r=tm#delay[index]
	s=invert(n)#r
;	svdc,n,w,u,v,/double
;	s=svsol(u,w,v,r,/double)
;
;	Now setup solution with data sampled every 30 minutes
	if n_elements(unique(scantable.starid)) eq 1 then begin
	ut=unique(fix(time))
	alloc=0
	for k=0,n_elements(ut)-1 do begin
		index=where(abs(time-ut[k]) lt 0.02,count)
		if count gt 0 then begin
			if alloc eq 0 then begin
				tindex=index
				alloc=1
			endif else tindex=[tindex,index]
		endif
		index=where(abs(time-(ut[k]+0.5)) lt 0.02,count)
		if count gt 0 then begin
			if alloc eq 0 then begin
				tindex=index
				alloc=1
			endif else tindex=[tindex,index]
		endif
	endfor
	time_e=time[tindex]
	delay_e=delay[tindex]
	mm=dblarr(n_elements(time_e),n_elements(m[0,*]))
	for j=0,n_elements(mm[0,*])-1 do mm[*,j]=interpol(m[*,j],t,time_e)
	tm_e=transpose(mm)
	n_e=tm_e#mm
	r_e=tm_e#delay_e
	s_e=invert(n_e)#r_e
;	svdc,n_e,w,u,v,/double
;	s_e=svsol(u,w,v,r_e,/double)
	endif
;
	for k=0,countr-1 do begin
		clabels[k]=string(s[k],format='(f7.2)')+' ('+clabels[k]+')'
;		print,'Spectrometer ',i,', Baseline ',j,'  ',clabels(k)
	endfor
	if i eq 0 and l eq 0 then print,clabels,format='(a)'
;
;	Plot the delay data with the model fits
	plot,t,m#s,/nodata,psym=3,yrange=[min(delay),max(delay)],xstyle=0, $
		ytitle='Residual delay [microns]',title='',color=frame_color
	oplot,t[0:min(sol_n)-1],smooth(m[0:min(sol_n)-1,*]#s,101),psym=3,color=13
	if n_elements(unique(scantable.starid)) eq 1 then $
	oplot,t,smooth(m#s-m#s_e,101)+avg(m#s),psym=3,color=15
	oplot,time,delay,psym=3,color=frame_color
;
	xyouts,xpos,!y.crange[1]+0.05*(!y.crange[1]-!y.crange[0]), $
		strjoin(string(s,format='(10f8.2)'))
	xyouts,xpos,!y.crange[1]-0.1*(!y.crange[1]-!y.crange[0]), $
		'RMS(fit)='+string(stdev(mm#s-delay))
	if n_elements(unique(scantable.starid)) eq 1 then $
	xyouts,xpos,!y.crange[1]-0.2*(!y.crange[1]-!y.crange[0]), $
		'RMS(mod)='+string(stdev(m#s-m#s_e))
;
;	!y.range=0
;	plot,t,m#s,/nodata,psym=3,yrange=[min(delay),max(delay)],xstyle=1, $
;		ytitle='Residual delay [microns]',title='',color=frame_color
;	oplot,t,m#s_e,psym=3,color=13
;	oplot,time,delay,psym=3,color=frame_color
;
	if !d.name eq 'PS' then device,/close
;
	SKIPPLOT:
endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro joint,star1,star2
;
; For the reduction of the experiments on 1999-04-10 and 1999-04-11.
; Will do a star position fit, but user must have selected these options
; using the widget beforehand! Stars should be selected in the widget,
; and not stationcoords should be selected.
;
common Tables,ScanTable,BGTable,StationTable
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; star1='FKV0502'
; star2='FKV0509'
;
openw,unit,'joint.dat',/get_lun
printf,unit,'Star       RA[h]         DEC[deg]     dRA[h]     dDec[deg]'
;
index1=where(startable.starid eq star1)
index2=where(startable.starid eq star2)
;
parsedate,date,y,m,d
jd=julian(y,m,d)
jd2000=julian(2000,1,1)
depoch=jd2jy(jd2000)-jd2jy(jd)
ra_cal=startable[index2].ra-depoch/100*startable[index2].pmra/3600*0
dec_cal=startable[index2].dec-depoch/100*startable[index2].pmdec/3600*0
ra_prog=startable[index1].ra-depoch/100*startable[index1].pmra/3600
dec_prog=startable[index1].dec-depoch/100*startable[index1].pmdec/3600
;
hdra=startable[index2].ra-startable[index1].ra
hddec=startable[index2].dec-startable[index1].dec
;
printf,unit,'Hipparcos positions:'
printf,unit,star1,startable[index1].ra,startable[index1].dec, $
	format='(a,2x,f11.8,2x,f10.7)'
printf,unit,star2,startable[index2].ra,startable[index2].dec,hdra,hddec, $
	format='(a,2x,f11.8,2x,f10.7,3x,f11.8,2x,f11.7)'
;
read_catalogs
cdra=startable[index2].ra-startable[index1].ra
cddec=startable[index2].dec-startable[index1].dec
;
printf,unit,'Catalog positions:'
printf,unit,star1,startable[index1].ra,startable[index1].dec, $
	format='(a,2x,f11.8,2x,f10.7)'
printf,unit,star2,startable[index2].ra,startable[index2].dec,cdra,cddec, $
	format='(a,2x,f11.8,2x,f10.7,3x,f11.8,2x,f10.7)'
;
calcastrom
solveastrom
sdra=startable[index2].ra-startable[index1].ra
sddec=startable[index2].dec-startable[index1].dec
;
ra_prog=ra_prog+hdra-sdra
dec_prog=dec_prog+hddec-sddec
;
printf,unit,'Solved positions:'
printf,unit,star1,startable[index1].ra,startable[index1].dec, $
	format='(a,2x,f11.8,2x,f10.7)'
printf,unit,star2,startable[index2].ra,startable[index2].dec,sdra,sddec, $
	format='(a,2x,f11.8,2x,f10.7,3x,f11.8,2x,f10.7)'
;
sra=sqrt(startable[index1].rae^2+startable[index2].rae^2)*15L*3600000L
sdec=sqrt(startable[index1].dece^2+startable[index2].dece^2)*3600000L
;
printf,unit,'Catalog-Hipparcos [mas]', $
	(cdra-hdra)*15L*3600000L,(cddec-hddec)*3600000L
printf,unit,'Solved -Hipparcos [mas]', $
	(sdra-hdra)*15L*3600000L,(sddec-hddec)*3600000L
printf,unit,'Uncertainties     [mas]', $
	sra,sdec
printf,unit,''
printf,unit,star1+' (J2000): RA = '+hms(ra_prog)+', Dec = '+dms(dec_prog)
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro ken,star1,star2,scans_int
;
; Work with data from 1999-04-10/11.
; Produce histogram of delay differences between FKV0502 (BSC5110) and
; its Hipparcos calibrator FKV0509. Also do a spline interpolation to
; the program star epochs and repeat histogram computation. Return
; interpolated delays.
; Use GrpDelay!
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
scans_int=scans
;
index502=where(scans.starid eq star1 and scans.time gt 0)
index509=where(scans.starid eq star2 and scans.time gt 0)
if n_elements(index502) ne n_elements(index509) then begin
	print,'***Error(KEN): '+star1+' and '+star2+ $
	      ' must have same number of scans!'
	return
endif
;
t_502=scans[index502].time
t_509=scans[index509].time
d1_502=(scans[index502].drydelay(1)-scans[index502].geodelay(1))*1e6
d1_509=(scans[index509].drydelay(1)-scans[index509].geodelay(1))*1e6
d2_502=(scans[index502].drydelay(2)-scans[index502].geodelay(2))*1e6
d2_509=(scans[index509].drydelay(2)-scans[index509].geodelay(2))*1e6
;
;d1_509_int=spline(t_509,d1_509,t_502,10)
;d2_509_int=spline(t_509,d2_509,t_502,10)
d1_509_int=interpol(d1_509,t_509,t_502)
d2_509_int=interpol(d2_509,t_509,t_502)
;
scans_bck=scans
scans[index509].time=t_502
calcastrom
scans_int[index509].drydelay(1)=scans[index509].geodelay(1)+d1_509_int/1e6
scans_int[index509].drydelay(2)=scans[index509].geodelay(2)+d2_509_int/1e6
scans_int[index509].geodelay(1)=scans[index509].geodelay(1)
scans_int[index509].geodelay(2)=scans[index509].geodelay(2)
scans_int[index509].time=t_502
scans=scans_bck
;
dd1=d1_502-d1_509
dd2=d2_502-d2_509
;
dd1_int=d1_502-d1_509_int
dd2_int=d2_502-d2_509_int
;
!p.multi=[0,1,2]
;
title=date
;
maxv=30
;
fr=histogram(dd1_int,min=-maxv,max=maxv,bin=3)
x=findgen(n_elements(fr))*3-maxv
plot,x,fr,psym=10,xtitle='E-C (int)',title=title
yfit=gaussfit(x,fr,a,nterms=3,estimates=[max(fr),0,10])
oplot,x,yfit,psym=0,linestyle=2
x_int=findgen(80)/2-20
yfit_int=spline(x,yfit,x_int)
index=where(yfit_int gt max(yfit_int)/2)
fwhm=max(x_int[index])-min(x_int[index])
xyouts,!x.crange[1]/3,max(yfit),'FWHM='+string(fwhm,format='(f4.1)')
;
fr=histogram(dd2_int,min=-maxv,max=maxv,bin=3)
x=findgen(n_elements(fr))*3-maxv
plot,x,fr,psym=10,xtitle='W-C (int)',title=title
yfit=gaussfit(x,fr,a,nterms=3,estimates=[max(fr),0,10])
oplot,x,yfit,psym=0,linestyle=2
x_int=findgen(80)/2-20
yfit_int=spline(x,yfit,x_int)
index=where(yfit_int gt max(yfit_int)/2)
fwhm=max(x_int[index])-min(x_int[index])
xyouts,!x.crange[1]/3,max(yfit),'FWHM='+string(fwhm,format='(f4.1)')
;
return
;
fr=histogram(dd1,min=-maxv,max=maxv,bin=3)
x=findgen(n_elements(fr))*3-maxv
plot,x,fr,psym=10,xtitle='E-C',title=title
yfit=gaussfit(x,fr,a,nterms=3,estimates=[max(fr),0,10])
oplot,x,yfit,psym=0,linestyle=2
x_int=findgen(80)/2-20
yfit_int=spline(x,yfit,x_int)
index=where(yfit_int gt max(yfit_int)/2)
fwhm=max(x_int[index])-min(x_int[index])
xyouts,!x.crange[1]/3,max(yfit),'FWHM='+string(fwhm,format='(f4.1)')
;
fr=histogram(dd2,min=-maxv,max=maxv,bin=3)
x=findgen(n_elements(fr))*3-maxv
plot,x,fr,psym=10,xtitle='W-C',title=title
yfit=gaussfit(x,fr,a,nterms=3,estimates=[max(fr),0,10])
oplot,x,yfit,psym=0,linestyle=2
x_int=findgen(80)/2-20
yfit_int=spline(x,yfit,x_int)
index=where(yfit_int gt max(yfit_int)/2)
fwhm=max(x_int[index])-min(x_int[index])
xyouts,!x.crange[1]/3,max(yfit),'FWHM='+string(fwhm,format='(f4.1)')
;
end
;-------------------------------------------------------------------------------
pro reducejoint
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
hds_open,'1999-04-11.inch'
get_metroconfig
hds_close
;
get_scandata,'1999-04-11.cha'
scans.grpdelay=scans.fdlpos
scans.grpdelay[0]=0
scans.grpdelayerr=scans.fdlposerr
;
end
;-------------------------------------------------------------------------------
pro scansim,UD,v2sigma,dm
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
uvdist=45e6
;
; Make lookup table
diameter=findgen(3000)/1000+0.1
arg=factor*uvdist*diameter
visamp=2*beselj(arg,1)/arg
vissq=visamp^2
;
; Now fake squared visibilities
arg=factor*uvdist*UD
vsq0=(2*beselj(arg,1)/arg)^2
vsq=vsq0+randomn(seed,10000)*v2sigma
;
; Get diameters for faked visibilities
dm=fltarr(n_elements(vsq))
for i=0,n_elements(vsq)-1 do begin
	r=min(abs(vsq[i]-vissq),index)
	dm[i]=diameter[index]
endfor
;
end
;-------------------------------------------------------------------------------
pro delcep,files,diameters847,errors847,diameters844,errors844,phases,bins
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,StarTable,Notes
;
; Delta Cephei ephemeris
jd0=2443674.144d0
per=5.366316d0
;
n=n_elements(files)
diameters847=fltarr(n)
errors847=fltarr(n)
diameters844=fltarr(n)
errors844=fltarr(n)
phases=fltarr(n)
bins=fltarr(n)
seed=-4
;
for i=0,n_elements(files)-1 do begin
	get_scandata,files[i]
	print,'Data loaded for --------> ',date
;
	startable.a0=0
	startable.a1=0
	startable.a2=0
	index=where(startable.starid eq 'FKV0847')
	startable[index].diameter=1.5
	index=where(startable.starid eq 'FKV0844')
	startable[index].diameter=1.5
	scans.vissq=randomn(seed,3,32,1,n_elements(scans))*0.05+0.75
	scans.vissqerr=0.05
	index=where(scans.starid eq 'FKV0848')
	scans[index].vissq=scans[index].vissq+0.25
;	scans.vissq=scans.vissq/2
;	scans.vissqerr=scans.vissqerr/2
	calcviscal
	scans[index].vissqe=scans[index].vissq
;
	hds_close
	index=where(startable.starid eq 'FKV0847')
	startable[index].diameter=1.5
	startable[index].a0= 2.079
	startable[index].a1=-3.947e-3
	startable[index].a2= 2.289e-6
	index=where(scans.starid eq 'FKV0847' and scans.time gt 0)
	parsedate,date,y,m,d
	phases[i]=(julian(y,m,d)+avg(scans[index].time)/86400l-jd0)/per mod 1
	bins[i]=(max(scans[index].time)-min(scans[index].time))/(2*86400l*per)
	d=1.5
	getdiam,'FKV0844','FKV0848',d,e,chisq
	diameters844[i]=d
	errors844[i]=e
	d=1.5
	getdiam,'FKV0847','FKV0848',d,e,chisq
	diameters847[i]=d
	errors847[i]=e
endfor
;
end
;-------------------------------------------------------------------------------
pro delcepp,files,dm847a,dm847b,phases,bins
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,StarTable,Notes
;
; Delta Cephei ephemeris
jd0=2443674.144d0
per=5.366316d0
;
n=n_elements(files)
dm847a=fltarr(n)
dm847b=fltarr(n)
phases=fltarr(n)
bins=fltarr(n)
;
for i=0,n_elements(files)-1 do begin
	get_scandata,files[i]
	print,'Data loaded for --------> ',date
	hds_close
	index=where(startable.starid eq 'FKV0847')
	index=where(scans.starid eq 'FKV0847' and scans.time gt 0)
	parsedate,date,y,m,d
	phases[i]=(julian(y,m,d)+avg(scans[index].time)/86400l-jd0)/per mod 1
	bins[i]=(max(scans[index].time)-min(scans[index].time))/(2*86400l*per)
;
!p.multi=[0,2,3]
	index847=where(scans.starid eq 'FKV0847' $
	        and scans.photonrateerr[0,16] gt 0 $
		and scans.photonrateerr[0,17] gt 0)
	r847_17=poly_fit(scans[index847].time/3600,scans[index847].photonrate(0,16),2,yft)
plot,scans[index847].time/3600,scans[index847].photonrate(0,16),psym=1
oplot,scans[index847].time/3600,yft,psym=0
	r847_18=poly_fit(scans[index847].time/3600,scans[index847].photonrate(0,17),2,yft)
plot,scans[index847].time/3600,scans[index847].photonrate(0,17),psym=1
oplot,scans[index847].time/3600,yft,psym=0
	index844=where(scans.starid eq 'FKV0844' $
	        and scans.photonrateerr[0,16] gt 0 $
		and scans.photonrateerr[0,17] gt 0)
	r844_17=poly_fit(scans[index844].time/3600,scans[index844].photonrate(0,16),2,yft)
plot,scans[index844].time/3600,scans[index844].photonrate(0,16),psym=1
oplot,scans[index844].time/3600,yft,psym=0
	r844_18=poly_fit(scans[index844].time/3600,scans[index844].photonrate(0,17),2,yft)
plot,scans[index844].time/3600,scans[index844].photonrate(0,17),psym=1
oplot,scans[index844].time/3600,yft,psym=0
	index848=where(scans.starid eq 'FKV0848' $
	        and scans.photonrateerr[0,16] gt 0 $
		and scans.photonrateerr[0,17] gt 0)
	r848_17=poly_fit(scans[index848].time/3600,scans[index848].photonrate(0,16),2,yft)
plot,scans[index848].time/3600,scans[index848].photonrate(0,16),psym=1
oplot,scans[index848].time/3600,yft,psym=0
	r848_18=poly_fit(scans[index848].time/3600,scans[index848].photonrate(0,17),2,yft)
plot,scans[index848].time/3600,scans[index848].photonrate(0,17),psym=1
oplot,scans[index848].time/3600,yft,psym=0
	t0=avg(scans[index847].time/3600)
	dm847a[i]=2.5*alog10((poly(t0,r847_17)/poly(t0,r848_17)+poly(t0,r847_18)/poly(t0,r848_18))/2)
	dm847b[i]=2.5*alog10((poly(t0,r847_17)/poly(t0,r844_17)+poly(t0,r847_18)/poly(t0,r844_18))/2)
endfor
;
end
;-------------------------------------------------------------------------------
pro dutycycle,files,cycle,mcycle,nscans,totime
;
common StarBase,StarTable,Notes
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
id1=[39,153,269,542,580,664,838,1122,1165,1220,2004,2540,2763,2845,3975,4033, $
     4368,4554,5107,5435,5511,6175,6396,6410,6789,7178,7235,7236,7420,7528, $
     7750,7852,8028,8142,8454,8518,8585,8634]
id2=[21,163,168,265,294,351,424,437,442,464,489,617,824,951,1017,1256,1865, $
     2473,2973,3249,3950,3980,4247,4432,4518,4932,5200,5681,6220,6418,7176, $
     7314,7328,7525,7570,7602,7796,7957,8079,8252,8414,8538,8571,8632,8667, $
     8684,8923,8961,8974,9045]
;
ids=[id1,id2]
bsc='BSC'+string(ids,format='(i4.4)')
fkv=cri(ids,'bsc-fkv')
index=where(fkv gt 0)
fkv='FKV'+string(fkv,format='(i4.4)')
stars=bsc
stars[index]=fkv[index]
;
cycle=0
for i=0,n_elements(files)-1 do begin
	print,files[i]
	get_scandata,files[i]
	hds_close
	for j=0,n_elements(startable)-1 do begin
		flag=0
		for k=0,n_elements(stars)-1 do begin
			if startable[j].starid eq stars[k] then flag=1
		endfor
		if flag then begin
		index=where(scans.starid eq startable[j].starid $
			and scans.time gt 0,count)
		if count gt 2 and count lt 20 then begin
			dt=scans[index[1:count-1]].time-scans[index[0:count-2]].time
			if n_elements(cycle) eq 1 then begin
				cycle=dt 
				mcycle=median(dt)
				nscans=count
				totime=scans[index[count-1]].time-scans[index[0]].time
			endif else begin
				cycle=[cycle,dt]
				mcycle=[mcycle,median(dt)]
				nscans=[nscans,count]
				totime=[totime,scans[index[count-1]].time-scans[index[0]].time]
			endelse
		endif
		endif
	endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro obslimits,file
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
files=findfile(file,count=fcount)
if fcount eq 0 then return
;
ha=dblarr(fcount*200)
dec=dblarr(fcount*200)
;
k=0L
;
for i=0,n_elements(files)-1 do begin
	print,'Processing ',files[i]
	get_data,files[i]
	hds_close
	index=where(scans.starid ne 'FKV0000',count)
	if count gt 0 then begin
		n=n_elements(scans[index])
		ha[k:k+n-1]=scans[index].ha
		dec[k:k+n-1]=scans[index].dec
		k=k+n
	endif
endfor
;
ha=ha[0:k-1]
dec=dec[0:k-1]
;
az=ha
el=az
azC=az & azE=az & azW=az
elC=el & elE=el & elW=el
;
modelC=double([85.322,18.227,256.787,90.314,0.,102.187,0.,49.242])
modelE=double([84.861,18.971,269.189,90.223,0.,89.242,0.,49.996])
modelW=double([85.493,18.761,87.168,89.793,0.,270.862,0.,50.173])
;
yC=sidpointing(modelC,[[ha],[dec]],-1) & azC=yC[*,0] & elC=yC[*,1]
yE=sidpointing(modelE,[[ha],[dec]],-1) & azE=yE[*,0] & elE=yE[*,1]
yW=sidpointing(modelW,[[ha],[dec]],-1) & azW=yW[*,0] & elW=yW[*,1]
;
print,max(azC),max(azE),max(azW)
print,max(elC),max(elE),max(elW)
print,min(azC),min(azE),min(azW)
print,min(elC),min(elE),min(elW)
;
end
;-------------------------------------------------------------------------------
pro charmaine
;
hds_open,'/gemini/npoi/raw/2000-10-11.001'
;
get_object,'scandata.rawdata(1).time',timestamp
;
get_object,'scandata.rawdata(1).inputbeam(1).quad',natc
get_object,'scandata.rawdata(1).inputbeam(2).quad',nate
get_object,'scandata.rawdata(1).inputbeam(3).quad',natw
;
get_object,'scandata.rawdata(1).inputbeam(1).fdldelay',fdlc
get_object,'scandata.rawdata(1).inputbeam(2).fdldelay',fdle
get_object,'scandata.rawdata(1).inputbeam(3).fdldelay',fdlw
;
hds_close
;
openw,unit,'charmaine_c.dat',/get_lun
for i=0l,n_elements(timestamp)-1 do $
	printf,unit,timestamp[i],natc[*,i],fdlc[i], $
	format='(f11.1,1x,4(f5.1,1x),e13.5)'
free_lun,unit
;
openw,unit,'charmaine_e.dat',/get_lun
for i=0l,n_elements(timestamp)-1 do $
	printf,unit,timestamp[i],nate[*,i],fdle[i], $
	format='(f11.1,1x,4(f5.1,1x),e13.5)'
free_lun,unit
;
openw,unit,'charmaine_w.dat',/get_lun
for i=0l,n_elements(timestamp)-1 do $
	printf,unit,timestamp[i],natw[*,i],fdlw[i], $
	format='(f11.1,1x,4(f5.1,1x),e13.5)'
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro olivier
;
; Print raw scan triple visibilities to file.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
filename='olivier.dat'
openw,unit,filename,/get_lun
;
for i=0,n_elements(geninfo)-1 do begin
;
loadnight,geninfo[i].date,geoinfo[i].systemid,geninfo[i].configid
scans.uvw=scans.uvw/1e6
;
for j=0,n_elements(scans)-1 do begin
	if scans[j].vissqcerr(0,0,0) gt 0 then $
	printf,unit,scans[j].time/3600, $
		scans[j].uvw(0,0,0,0),scans[j].uvw(0,0,0,1), $
		scans[j].vissqc(0,0,0),scans[j].vissqcerr(0,0,0)
	if scans[j].triplephasecerr(0,0) gt 0 then $
	printf,unit,scans[j].time/3600, $
		scans[j].uvw(0,0,0,0),scans[j].uvw(0,0,0,1), $
		scans[j].uvw(0,0,1,0),scans[j].uvw(0,0,1,1), $
		scans[j].triplephasec(0,0)*180/!pi,scans[j].triplephasecerr(0,0)*180/!pi
;		format='(f7.4,1x,a9,2x,80(f8.4,1x))'
endfor
endfor
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro klara
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
starid='HDN017014'
get_startable,starid
date='2011-01-01'
systemid='VLTI/MIDI-N-Band'
get_geoparms
;
uts=[3.d0,4.d0,5.d0,6.d0,7.d0]*3600+15.d0*60
uts[4]=uts[4]-15.d0*60
;
for i=0,n_elements(uts)-1 do begin
;
utc=uts[i]
print,'UTC [h] = ',utc/3600
print,'LST [deg]=',ut12gst(utc,/local)*15
;
topostar,utc,starid,StarTable,ra_app,dec_app
gst=ut12gst(utc)
;
ha=hourangle(gst,ra_app)
print,'HA [deg] = ',ha*15
;
; Convert to azimuth angle (0=N, 90=W)
hadec=[ha,dec_app]
azel=horizon2azel(equatorial2horizon(hadec2equatorial(hadec)))
az=reform(azel[0,*])
print,'Altitude [deg] = ',azel[1]
;
; Change AZ to VLTI "standard" convention West - 0, N - 90, E - 180, S - 270
azs=90-az+360
print,'AZ Standard [deg] = ',azs
;
azshadow=az+180
print,'AZ Shadow [deg] = ',azshadow
;
endfor
;
end
;-------------------------------------------------------------------------------
pro globalsim
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
systemid='NPOI'
date='2000-01-01'
;
get_bsc
index=where(startable.mv lt 5 and startable.dec gt -20)
startable=startable[index]
get_diameter
index=where(startable.diameter lt 3 and startable.bflag ne 'B')
startable=startable[index]
rename_starids,'bsc-hip'
index=where(startable.hic ne 0)
startable=startable[index]
read_catalogs
rename_starids,'hip-bsc'
;
; Create grid star list observed in every night
startable_bck=startable
indexg=where(startable.mv lt 3)
index0=where(startable.mv ge 3)
startable=startable[index0]
;
; Create four sub-lists of the startable mv >= 3
index1=where(fix(startable.ra) mod 2 eq 0 and fix(startable.dec) mod 2 eq 0)
index2=where(fix(startable.ra) mod 2 ne 0 and fix(startable.dec) mod 2 eq 0)
index3=where(fix(startable.ra) mod 2 eq 0 and fix(startable.dec) mod 2 ne 0)
index4=where(fix(startable.ra) mod 2 ne 0 and fix(startable.dec) mod 2 ne 0)
;
stations=['AC0','AE0','AW0','AN0']
;
num_days=120
num_skip=365/num_days
;
for i=0,num_days-1 do begin
	case i mod 4 of
		0:startable=merge_startable(startable_bck(indexg),startable_bck(index0(index1)))
		1:startable=merge_startable(startable_bck(indexg),startable_bck(index0(index2)))
		2:startable=merge_startable(startable_bck(indexg),startable_bck(index0(index3)))
		3:startable=merge_startable(startable_bck(indexg),startable_bck(index0(index4)))
	endcase
	fakescans,stations
	put_scandata,date+'.cha'
	for j=1,num_skip do date=nextdate(date)
endfor
;
end
;-------------------------------------------------------------------------------
pro globalsim
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
systemid='NPOI'
date='1999-01-01'
;
status=dc_read_free(!catalogs_dir+'npoi/obslist.grid',fkv,/col)
get_startable,'FKV'+string(fkv,format='(i4.4)')
rename_starids,'fkv-hip'
index=where(startable.hic ne 0)
startable=startable[index]
read_catalogs
rename_starids,'hip-fkv'
table_g=startable
;
status=dc_read_free(!catalogs_dir+'npoi/obslist.don',fkv,/col)
get_startable,'FKV'+string(fkv,format='(i4.4)')
rename_starids,'fkv-hip'
index=where(startable.hic ne 0 and startable.mv lt 6)
startable=startable[index]
read_catalogs
rename_starids,'hip-fkv'
table_d=startable
;
; Create four sub-lists of the startable mv >= 3
index1=where(fix(table_d.ra) mod 2 eq 0 and fix(table_d.dec) mod 2 eq 0)
index2=where(fix(table_d.ra) mod 2 ne 0 and fix(table_d.dec) mod 2 eq 0)
index3=where(fix(table_d.ra) mod 2 eq 0 and fix(table_d.dec) mod 2 ne 0)
index4=where(fix(table_d.ra) mod 2 ne 0 and fix(table_d.dec) mod 2 ne 0)
;
stations=['AC0','AE0','AW0']
;
num_days=120
num_skip=365/num_days
;
for i=0,num_days-1 do begin
	case i mod 4 of
		0:startable=merge_startable(table_g,table_d[index1])
		1:startable=merge_startable(table_g,table_d[index2])
		2:startable=merge_startable(table_g,table_d[index3])
		3:startable=merge_startable(table_g,table_d[index4])
	endcase
	fakescans,stations,dutycycle=2.0
	put_scandata,date+'.cha'
	for j=1,num_skip do date=nextdate(date)
endfor
;
end
;-------------------------------------------------------------------------------
pro jimc,filespec,scan_seeing,scan_length,scan_points,scan_v_mags
;
common Tables,ScanTable,BGTable,StationTable
common StarBase,StarTable,Notes
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
files=findfile(filespec)
;
n=n_elements(files)
m=n*200
scan_seeing=fltarr(m)
scan_length=fltarr(m)
scan_points=fltarr(m)
scan_v_mags=fltarr(m)
;
k=0
for i=0,n-1 do begin
	print,'Now processing ',files[i]
	hds_open,files[i]
	get_numscan,n
	if n gt 0 then begin
	get_sysconfig
	get_scantable
	fixearlydays
	if total(strlen(scantable.starid)) gt 0 then begin
	get_points
	list_stars,stars
	get_startable,stars
	hds_close
	index=where(scantable.starid ne 'FKV0000',count)
	if count gt 0 then begin
	scantable=scantable[index]
	rec0=rec0[index]
	recn=recn[index]
	num=n_elements(scantable)
	scan_length[k:k+num-1]=scantable.stoptime-scantable.starttime+1
	scan_points[k:k+num-1]=scantable.numpoint
	for j=0,num-1 do begin
		scan_seeing[k+j]=avg(delayjitter[*,0,rec0[j]:recn[j]])
		scan_v_mags[k+j]=startable[where(startable.starid eq scantable[j].starid)].mv
	endfor
	k=k+num
	endif
	endif else hds_close
	endif else hds_close
endfor
;
scan_seeing=scan_seeing[0:k-1]*1e6
scan_length=scan_length[0:k-1]
scan_points=scan_points[0:k-1]
scan_v_mags=scan_v_mags[0:k-1]
;
end
;-------------------------------------------------------------------------------
pro jimp,scan_seeing,scan_length,scan_points,scan_v_mags
;
!p.multi=[0,2,2]
ip=where(scan_points gt 30 and scan_points le 90)
rmin=1
rmax=2
rbin=0.05
x=findgen((rmax-rmin)/rbin)*rbin
r=scan_length/scan_points
!p.title='scan length/integration'
histograph,r[ip],min=rmin,max=rmax,bin=rbin
!p.title=''
!y.title='scan length/integration'
!x.title='Seeing index'
plot,scan_seeing[ip],r[ip],xrange=[0,5],yrange=[1,2],psym=3
!x.title='magnitude'
plot,scan_v_mags[ip],r[ip],xrange=[0,5],yrange=[1,2],psym=3
;
end
;-------------------------------------------------------------------------------
pro jim,t5,az,el
;
common InchData,Sol0,SolN,Jscan,MetroTime,PivotMotion, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
common MetroData,fbairtemp,fbsolidtmp
;
if !d.name eq 'PS' then begin
width=17.78
factor=width/640
device,xsize=factor*!xsize,ysize=factor*!ysize,xoffset=1,yoffset=1
endif
for it=0,8 do begin
;
if n_elements(t5) lt 0 then begin
for j=0,2 do begin
;!x.range=[7,14]
!x.title=metroconfig.fbsolidtmp.label[it,j]+' [C]'
index=where(az[*,j] ne 0 and el[*,j] ne 0)
y0=median(az[index,j])
plot,t5[index],az[index,j],psym=1,/ynozero,title=genconfig.stationid[j], $
	ytitle='AZ (FKV0000)',yrange=[y0-0.05,y0+0.05],ystyle=0
y0=median(el[index,j])
plot,t5[index],el[index,j],psym=1,/ynozero,title=genconfig.stationid[j], $
	ytitle='EL (FKV0000)',yrange=[y0-0.07,y0+0.07],ystyle=0
endfor
return
endif
;
dates=['2001-03-16','2001-03-26','2001-04-01','2001-04-02','2001-04-15','2001-04-16','2001-04-25','2001-04-26']
;
num=n_elements(dates)
;
az=fltarr(2000,3)
el=fltarr(2000,3)
t5=fltarr(2000,3)
m=0l
;
for k=0,num-1 do begin
;
hds_open,'/gemini/npoi/con/'+dates[k]+'.con'
get_sysconfig
get_scantable
get_metrodata
hds_close
hds_open,'/gemini/npoi/inch/'+dates[k]+'.inch'
get_avegroup,1
hds_close
;
ndex=where(scantable.starid eq 'FKV0000',n)
;
!p.multi=[0,2,3]
for j=0,2 do begin
for i=0,n-1 do begin
	az[m+i,j]=avg(motorangle[sol0[ndex[i]]:soln[ndex[i]],0,j])
	el[m+i,j]=avg(motorangle[sol0[ndex[i]]:soln[ndex[i]],1,j])
	t=fbsolidtmp.timestamp[*,j]*3600.d0
	index=where(abs(t-avg(metrotime[sol0[ndex[i]]:soln[ndex[i]]])) lt 40,count)
	if count gt 0 then t5[m+i,j]=avg(fbsolidtmp.data[index,it,j])
endfor
endfor
m=m+n
endfor
for j=0,2 do begin
;!x.range=[7,14]
!x.title=metroconfig.fbsolidtmp.label[it,j]+' [C]'
index=where(az[*,j] ne 0 and el[*,j] ne 0)
y0=median(az[index,j])
plot,t5[index],az[index,j],psym=1,/ynozero,title=genconfig.stationid[j], $
	ytitle='AZ (FKV0000)',yrange=[y0-0.05,y0+0.05],ystyle=0
y0=median(el[index,j])
plot,t5[index],el[index,j],psym=1,/ynozero,title=genconfig.stationid[j], $
	ytitle='EL (FKV0000)',yrange=[y0-0.07,y0+0.07],ystyle=0
endfor
endfor
if !d.name eq 'PS' then device,/close
;
end
;-------------------------------------------------------------------------------
pro keck,hip
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
status=dc_read_free('keck.H.list',hipnos,a,dm,/col,ignore=['!'])
get_startable,'HIP'+string(hipnos,format='(i6.6)')
startable.mv=startable.mv-0.75
index=where(startable.class1 eq 0,count)
if count gt 0 then startable[index].class1=5
diameter_bv
startable.mv=startable.mv+0.75
save,startable
;
i=where(startable.hic eq hip,count)
if count eq 0 then return
;
doy=(startable[i].ra-6.5)/24*365
jd2date,julian(2002,1,1)+doy,y,m,d
date=constrictordate(y,m,d)
;
readmodel,'gatewood.model'
gen_model[0].starid=startable[i].starid
gen_model[0].ra=startable[i].ra
gen_model[0].dec=startable[i].dec
star_model[*].diameter=startable[i].diameter
star_model[1].magnitudes=dm[i]
binary_model[0].rho=a[i]
;
systemid='Keck'
stations=['K01','K02']
startable=startable[i]
fakescans,stations,dutycycle=0.5,status=status
if status ne 0 then return
calcmodel
fakescans
;
if init_plot('scan','pt') ne 0 then return
ds_y.item=25
s=set_datasel['x']
s=set_datasel['y']
ps_options.l=1
ps_options.e=1
ps_options.p=1
!y.range=[0,1]
if !d.name eq 'PS' then device,filename=startable[0].starid+'.45.ps'
plotdata
if !d.name eq 'PS' then begin
	device,/close
	spawn,'ps2epsi '+startable[0].starid+'.45.ps '+startable[0].starid+'.45.eps'
;	spawn,'ps2gif '+startable(0).starid+'.45.ps'
endif
;
binary_model[0].theta=binary_model[0].theta+45
calcmodel
fakescans
if !d.name eq 'PS' then device,filename=startable[0].starid+'.90.ps'
plotdata
if !d.name eq 'PS' then begin
	device,/close
	spawn,'ps2epsi '+startable[0].starid+'.90.ps '+startable[0].starid+'.90.eps'
;	spawn,'ps2gif '+startable(0).starid+'.90.ps'
endif
;
if init_plot('uv','pt') ne 0 then return
ds_x.item=50
ds_y.item=51
s=set_datasel['x']
s=set_datasel['y']
!y.range=0
if !d.name eq 'PS' then device,filename=startable[0].starid+'.uv.ps'
plotuv
if !d.name eq 'PS' then begin
	device,/close
	spawn,'ps2epsi '+startable[0].starid+'.uv.ps '+startable[0].starid+'.uv.eps'
;	spawn,'ps2gif '+startable(0).starid+'.uv.ps'
endif
;
end
;-------------------------------------------------------------------------------
pro corrpar,dpx,dpy,dpz,dmx,dmy,dmz,v
;
common InchData,Sol0,SolN,Jscan,MetroTime,PivotMotion, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
index=where(scantable.starid ne 'FKV0000' and sol0 ne -1,n)
x=fltarr(n)
y=fltarr(n)
z=fltarr(n)
m=fltarr(n,2)
;
l=0
i=2
for k=0,n-1 do begin
	j=where(parxerr[i,sol0[index[k]]:soln[index[k]]] gt 0,count)
		if count gt 0 then begin
			x[l]=avg(parx[i,j+sol0[index[k]]])
			y[l]=avg(pary[i,j+sol0[index[k]]])
			z[l]=avg(parz[i,j+sol0[index[k]]])
			m[l,*]=avg(motorangle[j+sol0[index[k]],*,i],0)
			l=l+1
		endif
endfor
dpx=x[1:l-1]-x[0:l-2]
dpy=y[1:l-1]-y[0:l-2]
dpz=z[1:l-1]-z[0:l-2]
model=sidmodel(i+1)
hadec=sidpointing(model,m,1)
mp=mirrorpivot(model,hadec[*,0],hadec[*,1],v)
dmx=mp[1:l-1,0]-mp[0:l-2,0]
dmy=mp[1:l-1,1]-mp[0:l-2,1]
dmz=mp[1:l-1,2]-mp[0:l-2,2]
;
end
;-------------------------------------------------------------------------------
pro motorplot,sid
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common LocalLMMotor,ma
;
y=lmmotor(x,a,/init,sid=sid,times=times)
num=n_elements(y)
;
case sid of
	1:maxv=100
	2:maxv=100
	3:maxv=100
endcase
set_screen
window,/free,xsize=!xsize,ysize=!xsize*(0.27/0.25)
tek_color
!p.title='Sid '+string(sid)+', '+date
!x.title='delta RA ["]'
!y.title='delta Dec ["]'
!p.charsize=1.5
plot,findgen(10),findgen(10),/nodata,yrange=[-maxv,maxv],xrange=[-maxv,maxv]
;
for i=0,n_elements(scantable)-1 do begin
	index=where(times gt metrotime[sol0[i]] $
	        and times lt metrotime[soln[i]],count)
	if count gt 0 then begin
		ha=fltarr(count)
		dec=fltarr(count)
		for j=0,count-1 do begin
			b=lmmotor(x[index[j]],a)
			ha[j]=b[0]
			b=lmmotor(x[index[j]+num/2],a)
			dec[j]=b[0]
		endfor
		k=where(startable.starid eq scantable[i].starid) & k=k[0]
		dha=y[index]-ha
		ddec=y[index+num/2]-dec
		if i eq 39 then stop
		oplot,dha*3600,ddec*3600,color=getcolor(k),psym=0
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro periodicplot,sid,scans,order=order
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common LocalLMMotor,ma
;
if n_elements(order) eq 0 then order=1
if n_elements(scans) eq 0 then scans=indgen(n_elements(scantable))+1
;
y=lmmotor(x,a,/init,sid=sid,times=times)
num=n_elements(y)
;
set_screen
window,/free,xsize=1200,ysize=400
tek_color
!p.title='Sid '+string(sid)+', '+date
!y.title='["]'
!p.charsize=1.5
!p.multi=[0,1,1]
!p.psym=0
;
j=sid-1
if n_elements(scans) gt 1 then begin
	!x.range=[-20,30]
	!y.range=[-5,5]
	plot,!x.range,!y.range,/nodata
endif
for k=0,n_elements(scans)-1 do begin
i=scans[k]-1
index=where(times gt metrotime[sol0[i]] $
        and times lt metrotime[soln[i]],count)
if count gt 2 then begin
	azel=sidpointing(sidmodel(sid),natural(y[index]/15,y[index+num/2]),-1) 
;	!x.title='Azimuth angle'
;	oplot,azel(0,*),polyres(times(index),ma(0,index)-azel(0,*),order)*3600 
	!x.title='Elevation angle'
	oplot,azel[1,*],polyres(times[index],ma[1,index]-azel[1,*],order)*3600 
endif
endfor
;
end
;-------------------------------------------------------------------------------
pro periodicfit,sid,x2g,y2g
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common LocalLMMotor,ma
;
y=lmmotor(x,a,/init,sid=sid,times=times)
num=n_elements(y)
;
set_screen
;window,/free
window_slide,xsize=2000
tek_color
!p.title='Sid '+string(sid)+', '+date
!y.title='["]'
!p.charsize=1.5
!p.psym=1
;
j=sid-1
i1=39
i2=42
;
num12=2500
x12=dindgen(num12)/num12*50-20
;
!x.title='Elevation angle'
index=where(times gt metrotime[sol0[i1]] $
        and times lt metrotime[soln[i1]],num1)
azel=sidpointing(sidmodel(sid),natural(y[index]/15,y[index+num/2]),-1) 
x1=reform(azel[1,*])
y1=polyres(times[index],ma[1,index]-azel[1,*],1)*3600 
si=sort(x1)
x1=x1[si]
y1=y1[si]
index=where(x12 ge x1[0] and x12 le x1[num1-1],num1g)
x1g=x12[index]
y1g=interpol(y1,x1,x1g)
;
index=where(times gt metrotime[sol0[i2]] $
        and times lt metrotime[soln[i2]],num2)
azel=sidpointing(sidmodel(sid),natural(y[index]/15,y[index+num/2]),-1) 
x2=reform(azel[1,*])
y2=polyres(times[index],ma[1,index]-azel[1,*],1)*3600 
si=sort(x2)
x2=x2[si]
y2=y2[si]
y2=y2[0:230]
x2=x2[0:230]
num2=231
index=where(x12 ge x2[0] and x12 le x2[num2-1],num2g)
x2g=x12[index]
y2g=interpol(y2,x2,x2g)
;
plot,[x1g,x2g],[y1g,y2g]
;oplot,x2g,ts_smooth(y2g,41),psym=0,color=2
numf=300
i=where(x12 eq x2g[num2g-1])
x2f=x12[i[0]+1:i[0]+numf]
y2f=ts_fcast(y2g,n_elements(y2g)-1,numf)
oplot,x2f,y2f,color=1,psym=0,linestyle=2
;
index=where(x12 ge x2[0])
x2gf=x12[index]
p=fft(y2g,-1)
p[10:285]=0
yp=[y2g,y2g,y2g,y2g]*0
for i=0,9 do begin
	yp1=yp
	for j=0,n_elements(yp)-1 do yp1(j)=2*(float(p[i])*cos(2*!pi*i*j/295.) $
			                 -imaginary(p[i])*sin(2*!pi*i*j/295.))
	yp=yp+yp1
oplot,x2gf,yp1,psym=0,color=2+i
endfor
oplot,x2gf,yp,psym=0,color=1
;
end
;-------------------------------------------------------------------------------
function lmperiodic,x,a
;
; y=a0 + a1 * sin( 2*pi /a2 *x + a3)
;
y=a[0]+a[1]*sin((2*!pi*(x-a[3]))/a[2])
dy=a
da=[1,1,0.1,1]
;
for i=0,3 do begin
	b=a
	b[i]=b[i]+da[i]
	yh=b[0]+b[1]*sin((2*!pi*(x-b[3]))/b[2])
	b[i]=b[i]-2*da[i]
	yl=b[0]+b[1]*sin((2*!pi*(x-b[3]))/b[2])
	dy[i]=(yh-yl)/2
endfor
;
return,[y,dy]
;
end
;-------------------------------------------------------------------------------
pro periodicfit,sid,doaz=doaz,doel=doel,order=order,wave=wave,usep=usep, $
	nlow=nlow,nhigh=nhigh
;
; Using 2001-06-21, scans 40 and 43, determine tables of periodic drive 
; error.
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common LocalLMMotor,ma
;
if n_elements(doel) eq 0 then doel=1
if n_elements(doaz) eq 0 then doaz=1
if n_elements(order) eq 0 then order=1
if n_elements(usep) eq 0 then usep=1
if n_elements(nlow) eq 0 then nlow=0
if n_elements(nhigh) eq 0 then nhigh=0
;
; Get predicted AZEL data
y=lmmotor(x,a,/init,sid=sid,times=times)
num=n_elements(y)
;
; Initialize plot
set_screen
window,/free,ysize=300,xsize=1100
;window_slide,xsize=2000
tek_color
!p.title='Sid '+string(sid)+', '+date
!y.title='Periodic error ["]'
!p.charsize=1.5
!p.psym=1
;!p.multi=[0,1,2]
;
j=sid-1
i1=39
i2=42
;
IF doel THEN BEGIN
; Do elevation first for requested siderostat
!x.title='Elevation angle'
;
; We want periodic error table for this range of elevation angles
num12=3000
xrange=50
x12=dindgen(num12)/num12*xrange-20
sampling_int=float(xrange)/num12
;
; This is the first scan
index=where(times gt metrotime[sol0[i1]] $
        and times lt metrotime[soln[i1]],num1)
azel=sidpointing(sidmodel(sid),natural(y[index]/15,y[index+num/2]),-1) 
x1=reform(azel[1,*])
y1=polyres(times[index],ma[1,index]-azel[1,*],order)*3600 
si=sort(x1)
x1=x1[si]
y1=y1[si]
index=where(x12 ge x1[0] and x12 le x1[num1-1],num1g)
x1g=x12[index]
y1g=interpol(y1,x1,x1g)
;
; This is the second scan
index=where(times gt metrotime[sol0[i2]] $
        and times lt metrotime[soln[i2]],num2)
azel=sidpointing(sidmodel(sid),natural(y[index]/15,y[index+num/2]),-1) 
x2=reform(azel[1,*])
y2=polyres(times[index],ma[1,index]-azel[1,*],order)*3600 
si=sort(x2)
x2=x2[si]
y2=y2[si]
; Plot observed data
plot,[x1,x2],[y1,y2],xrange=[min(x12),max(x12)],psym=3,color=1
x2=x2[nlow:n_elements(si)-1-nhigh]
y2=y2[nlow:n_elements(si)-1-nhigh]
num2=n_elements(x2)
index=where(x12 ge x2[0] and x12 le x2[num2-1],num2g)
x2g=x12[index]
y2g=interpol(y2,x2,x2g)
;
; Plot available grid interpolated data
;oplot,[x1g,x2g],[y1g,y2g],psym=3,color=6
oplot,x1g,y1g,psym=0,color=1,thick=1
oplot,x2g,y2g,psym=0,color=1,thick=2
;
; Compute Fourier fit for second scan, using only 20 frequencies
p=fft(y2g,-1)
n=n_elements(p)
yp=y2g*0
for i=0,19 do begin
	for j=0,n_elements(yp)-1 do $
		yp[j]=yp[j]+2*(float(p[i])*cos(2*!pi*i*j/n) $
			  -imaginary(p[i])*sin(2*!pi*i*j/n))
endfor
;oplot,x2g,yp,psym=0,color=5
;
; Determine fundamental wavelength
i=where(abs(p[0:n/2]) eq max(abs(p[0:n/2]))) & i=i[0]
wl=n*sampling_int/i
print,'Fundamental wavelength: ',wl
; Cross check with auto correlation
nl=10
l=findgen(2*nl+1)-nl+fix(wl/sampling_int)
a=a_correlate(y2g,l)
i=where(a eq max(a)) & i=i[0]
wl=l[i]*sampling_int
print,'Fundamental lag: ',wl
if n_elements(wave) ne 0 then wl=wave
;
; Use linear prediction to extrapolate Fourier fit to full wavelengths
numw=long(wl/sampling_int)
numf=numw-(num2g mod numw)
nums=(num2g/numw)*numw
i=where(x12 eq x2g[num2g-1])
x2f=x12[i[0]+1:i[0]+numf]
y2f=ts_fcast(y2g,n_elements(y2g)-1,numf)
if usep then oplot,x2f,y2f,color=2,psym=0,linestyle=0
;
; Now do another Fourier fit to the full wavelengths set
x2f=[x2g,x2f]
y2f=[yp,y2f] 
if not usep then begin
	x2f=x2f[0:nums-1]
	y2f=y2f[0:nums-1]
endif
p=fft(y2f,-1)
n=n_elements(p)
wle=n*sampling_int
nwle=nint((x2g[0]-min(x12))/wle)
xp=x12-x12[0]+x2g[0]-nwle*wle
yp=xp*0
for i=0,19 do begin
	for j=0,n_elements(yp)-1 do $
		yp[j]=yp[j]+2*(float(p[i])*cos(2*!pi*i*j/n) $
			  -imaginary(p[i])*sin(2*!pi*i*j/n))
endfor
oplot,xp,yp,psym=0,color=3
ENDIF
;.................................................................
IF doaz THEN BEGIN
; Do azimuth now for requested siderostat
!x.title='Azimuth angle'
;
; We want periodic error table for this range of elevation angles
num12=2500
xrange=80
x12=dindgen(num12)/num12*xrange-40
sampling_int=float(xrange)/num12
;
; This is the second scan
index=where(times gt metrotime[sol0[i2]] $
        and times lt metrotime[soln[i2]],num1)
azel=sidpointing(sidmodel(sid),natural(y[index]/15,y[index+num/2]),-1) 
x1=reform(azel[0,*])
y1=polyres(times[index],ma[0,index]-azel[0,*],order)*3600 
si=sort(x1)
x1=x1[si]
y1=y1[si]
index=where(x12 ge x1[0] and x12 le x1[num1-1],num1g)
x1g=x12[index]
y1g=interpol(y1,x1,x1g)
;
; This is the first scan
index=where(times gt metrotime[sol0[i1]] $
        and times lt metrotime[soln[i1]],num2)
azel=sidpointing(sidmodel(sid),natural(y[index]/15,y[index+num/2]),-1) 
x2=reform(azel[0,*])
y2=polyres(times[index],ma[0,index]-azel[0,*],order)*3600 
si=sort(x2)
x2=x2[si]
y2=y2[si]
; Plot observed data
plot,[x1,x2],[y1,y2],xrange=[min(x12),max(x12)],psym=3,color=1
x2=x2[nlow:n_elements(si)-1-nhigh]
y2=y2[nlow:n_elements(si)-1-nhigh]
num2=n_elements(x2)
index=where(x12 ge x2[0] and x12 le x2[num2-1],num2g)
x2g=x12[index]
y2g=interpol(y2,x2,x2g)
;
; Plot available grid interpolated data
;oplot,[x1g,x2g],[y1g,y2g],psym=0,color=1,thick=2
oplot,x1g,y1g,psym=0,color=1,thick=1
oplot,x2g,y2g,psym=0,color=1,thick=2
;
; Compute Fourier fit for second scan, using only 10 frequencies
p=fft(y2g,-1)
n=n_elements(p)
yp=y2g*0
for i=0,19 do begin
	for j=0,n_elements(yp)-1 do $
		yp[j]=yp[j]+2*(float(p[i])*cos(2*!pi*i*j/n) $
			  -imaginary(p[i])*sin(2*!pi*i*j/n))
endfor
oplot,x2g,yp,psym=0,color=1
;
; Determine fundamental wavelength
i=where(abs(p[0:n/2]) eq max(abs(p[0:n/2]))) & i=i[0]
wl=n*sampling_int/i
print,'Fundamental wavelength: ',wl
; Cross check with auto correlation
nl=10
l=findgen(2*nl+1)-nl+fix(wl/sampling_int)
a=a_correlate(y2g,l)
i=where(a eq max(a)) & i=i[0]
wl=l[i]*sampling_int
print,'Fundamental lag: ',wl
if n_elements(wave) ne 0 then wl=wave
;
; Use linear prediction to extrapolate Fourier fit to full wavelengths
numw=long(wl/sampling_int)
numf=numw-(num2g mod numw)
nums=(num2g/numw)*numw
i=where(x12 eq x2g[num2g-1])
x2f=x12[i[0]+1:i[0]+numf]
y2f=ts_fcast(y2g,n_elements(y2g)-1,numf)
if usep then oplot,x2f,y2f,color=2,psym=0,linestyle=0
;
; Now do another Fourier fit to the full wavelengths set
x2f=[x2g,x2f] 
y2f=[yp,y2f] 
if not usep then begin
	x2f=x2f[0:nums-1]
	y2f=y2f[0:nums-1]
endif
p=fft(y2f,-1)
n=n_elements(p)
wle=n*sampling_int
nwle=nint((x2g[0]-min(x12))/wle)
xp=x12-x12[0]+x2g[0]-nwle*wle
yp=xp*0
for i=0,19 do begin
	for j=0,n_elements(yp)-1 do $
		yp[j]=yp[j]+2*(float(p[i])*cos(2*!pi*i*j/n) $
			  -imaginary(p[i])*sin(2*!pi*i*j/n))
endfor
oplot,xp,yp,psym=0,color=3
ENDIF
;
;
end
;-------------------------------------------------------------------------------
pro periodicerror
;
periodicfit,1,/doel,doaz=0,usep=1,wave=2.25
periodicfit,2,/doel,doaz=0,wave=2.26,usep=1,nhigh=7
periodicfit,3,/doel,doaz=0,order=1,usep=0,wave=2.25
;
periodicfit,1,/doaz,doel=0,usep=1,nlow=15,wave=2.22
periodicfit,2,/doaz,doel=0,usep=1,wave=2.24,nlow=5
periodicfit,3,/doaz,doel=0,usep=1,wave=2.21
;
end
;-------------------------------------------------------------------------------
function piston,fa,ma
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
RAD=180/!pi
model=sidmodel(1)
;
l=53.0
a=32.0
b=ma-a
c=b+fa/2
o0=l*sin(a/rad)/sin((90-ma)/rad)
ov=[cos(b/rad),-sin(b/rad)]*o0
dv=[-sin((90-ma-b)/rad),cos((90-ma-b)/rad)]
;
c0=l*sin((180-a-(90-ma))/rad)/sin((90-ma)/rad)
b0=c0*sin(fa/rad)/sin((180-fa-(90-ma))/rad)
a0=c0*sin((90-ma)/rad)/sin((180-fa-(90-ma))/rad)
;
a1=o0-b0
b1=a1*sin((fa/2)/rad)/sin((90-ma+fa-fa/2)/rad)
c1=a1*sin((180-fa/2-(90-ma+fa-fa/2))/rad)/sin((90-ma+fa-fa/2)/rad)
;
a2=a0+b1
;
cv=[cos(c/rad),-sin(c/rad)]*c1
x=(total(cv*dv)-total(ov*dv))/total(dv^2)
r=sqrt(total((cv-(ov+x*dv))^2))
;return,a2+r-c0
;
b2=a2*sin(fa/rad)/sin((180-fa-(2*ma-fa))/rad)
c2=a2*sin((2*ma-fa)/rad)/sin((180-fa-(2*ma-fa))/rad)
c3=c0-c2
b3=sin((2*ma-90)/rad)*c3
;
;return,b3+b2+b1+a0-c0
;
c0=l*sin((180-a-(90-ma))/rad)/sin((90-ma)/rad)
a2=l*sin((180-a-(90-ma)+fa/2)/rad)/sin((180-(180-a-(90-ma)+fa/2)-(a-fa))/rad)
x=a2*cos((a-fa)/rad)-l
y=a2*sin((a-fa)/rad)
o0=l*sin(a/rad)/sin((90-ma)/rad)
d=o0*sin(ma/rad)
alpha=180-90-ma+180-(180-a-(90-ma))
;help,c0,a2,x,y,o0,d,alpha
;
return,a2-(x*cos(alpha/rad)+y*sin(alpha/rad)-d)-c0
end
;-------------------------------------------------------------------------------
pro natstation
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
for i=0,n_elements(scantable)-1 do begin
for k=0,GenConfig.NumSid-1 do begin
	index=where(natcountserr[k,rec0[i]:recn[i]] gt 0,count)
	if count eq 0 then scantable[i].station(k)=0
endfor
endfor
;
for k=0,genconfig.numsid-1 do begin
	print,'station',k
	index=where(scantable.station[k] eq 0,count)
	if count gt 0 then print,long((scantable[index].starttime+scantable[index].stoptime)/2),format='(6i6)'
endfor
end
;-------------------------------------------------------------------------------
pro pro_sine,t,a,f
;
; a=[half amplitude,period,offset]
;
f=a[0]*(1+sin(2*!pi*(t-a[2])/a[1]))
;
end
;-------------------------------------------------------------------------------
pro pro_step,t,a,f
;
print,a
f=t*0
;
tmod=(t-a[2]) mod a[1] 
tmod=tmod/a[1]
index=where(tmod ge 7./8. or tmod le 1./8.,n)
if n gt 0 then f[index]=a[0]
index=where(tmod ge 1./8. and tmod le 3./8.,n)
if n gt 0 then f[index]=2*a[0]
index=where(tmod ge 3./8. and tmod le 5./8.,n)
if n gt 0 then f[index]=a[0]
index=where(tmod ge 5./8. and tmod le 7./8.,n)
if n gt 0 then f[index]=0
;
end
;-------------------------------------------------------------------------------
function funct_step,t,a,mask
;
f=t
mask=long(t)*0
;
tmod=(t-a[2]) mod a[1] 
tmod=tmod/a[1]
index=where(tmod gt 7./8. or tmod lt 1./8.)
f[index]=a[0]
mask[index]=1
index=where(tmod gt 1./8. and tmod lt 3./8.)
f[index]=2*a[0]
mask[index]=2
index=where(tmod gt 3./8. and tmod lt 5./8.)
f[index]=a[0]
mask[index]=3
index=where(tmod gt 5./8. and tmod lt 7./8.)
f[index]=0
mask[index]=4
;
return,f
;
end
;-------------------------------------------------------------------------------
pro pro_decay,t,a,f
;
f=a[0]+a[1]*exp(-a[2]*t)
;
end
;-------------------------------------------------------------------------------
pro reduce1025,a_1,a_2,a_3,b_1,b_2,b_3,c_1,c_2,c_3,mflux,ratio,error,chisq,bic, $
	fid=fid,rid=rid,sid=sid,bid=bid,cid=cid, $
	do_slide=do_slide
;
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
;  f    f   f   f     N
; 0.3x4 1x4 3x4 10x4 (1000) 
; 0.3x4 1x4 3x4 10x4 (500) 
; 0.3x4 1x4 3x4 10x4 (100) 
; 0.3x4 1x4 3x4 10x4 (50)
;
; frequencies rates scans beams channels
nc=16
if n_elements(mflux) eq 0 then mflux=fltarr(4,4,4,2,nc)
if n_elements(ratio) eq 0 then ratio=fltarr(4,4,4,2,nc)
if n_elements(error) eq 0 then error=fltarr(4,4,4,2,nc)
if n_elements(chisq) eq 0 then chisq=fltarr(4,4,4,2,nc)
if n_elements(bic) eq 0 then bic=fltarr(4,4,4,4,2,nc)
;
if n_elements(a_1) eq 0 then a_1=fltarr(4,4,2,nc)
if n_elements(a_2) eq 0 then a_2=fltarr(4,4,2,nc)
if n_elements(a_3) eq 0 then a_3=fltarr(4,4,2,nc)
if n_elements(b_1) eq 0 then b_1=fltarr(4,4,2,nc)
if n_elements(b_2) eq 0 then b_2=fltarr(4,4,2,nc)
if n_elements(b_3) eq 0 then b_3=fltarr(4,4,2,nc)
if n_elements(c_1) eq 0 then c_1=fltarr(4,4,2,nc)
if n_elements(c_2) eq 0 then c_2=fltarr(4,4,2,nc)
if n_elements(c_3) eq 0 then c_3=fltarr(4,4,2,nc)
;
if n_elements(fid) eq 0 then begin
	fids=[1,2,3,4]
endif else fids=fid
if n_elements(rid) eq 0 then begin
	rids=[1,2,3,4]
endif else rids=rid
if n_elements(sid) eq 0 then begin
	sids=[1,2,3,4]
endif else sids=sid
if n_elements(bid) eq 0 then begin
	bids=[1,2]
endif else bids=bid
if n_elements(cid) eq 0 then begin
	cids=indgen(nc)+1
endif else cids=cid
;
if n_elements(do_slide) eq 0 then do_slide=0
;
freqs=1/[0.3,1,3,10]
scan_index=(indgen(16) mod 4)
rate_index=(indgen(16)/4)
;
!x.margin=[8,2]
!y.margin=[4,2]
!p.charsize=1.5
;
FOR riter=0,n_elements(rids)-1 DO BEGIN
rid=rids[riter]
FOR biter=0,n_elements(bids)-1 DO BEGIN
bid=bids[biter]
FOR siter=0,n_elements(sids)-1 DO BEGIN
sid=sids[siter]
FOR citer=0,n_elements(cids)-1 DO BEGIN
cid=cids[citer]
if !d.name eq 'PS' then begin
	label=string(rid)+'.'+string(sid)+'.'+string(bid)+'.'+string(cid)+'.ps'
	label=strcompress(label,/remove_all)
	device,filename='rsbc.'+label
	width=17.78
        factor=width/640
        device,xsize=factor*700,ysize=factor*900, $
		xoffset=1,yoffset=1,color=1
	!p.charsize=1.5
endif
!p.multi=[0,2,2]
FOR fiter=0,n_elements(fids)-1 DO BEGIN
fid=fids[fiter]
;
get_rawdata,'2002-10-25.raw.'+string(sid+(fid-1)*4+(rid-1)*16,format='(i3.3)')
get_bincounts,bid
hds_close
;
r=total(bincounts,1)
t=fdlstamp()/1000.d0 & t=t-t[0]
x=fdlstamp()
;
index=where(t lt 30)
r=r[*,index]
t=t[index]
x=x[index]
;
;
;	Determine period and phase first with total of all channels
	y=total(r,1)
	series=powerpatch(x,y,limit=100000)
	sampling_int=0.002
	series=poweroftwo(series) & series=series-avg(series)
	if max(series) gt 0 then begin
	tp=power[series,sampling_int,f,p,f_avg,p_avg]
	if do_slide then begin
		window_slide,xsize=10000,xv=800,wid=wid
		!p.psym=3
		!y.range=0
		!x.range=0
	endif
	i=where(p_avg eq max(p_avg))
;	a=[tp*sqrt(2),1/f_avg(i),0]
	a=[stdev(y)*sqrt(2),1/f_avg[i],0]
	index=where(y lt 16)
	a[2]=median(t[index[0:6]])-(a[1]/4)*3
	n=n_elements(t)/2
	yfit=curvefit(t[0:n],y[0:n],y[0:n]*0+1,a,function_name='pro_sine',/noderivat)
	yfit=curvefit(t,y,y*0+1,a,function_name='pro_sine',/noderivat)
;
;	Now use only data of requested channel
	y=reform(r[cid-1,*])
	a[0]=stdev(y)*sqrt(2)
	yfit=curvefit(t,y,y*0+1,a,function_name='pro_sine',/noderivat)
	for iter=1,5 do begin
		a2a=(indgen(21)-10)*(a[1]/40)+a[2]
		gof=a2a
		for k=0,n_elements(a2a)-1 do begin
			yfit=funct_step(t,[a[0],a[1],a2a[k]],m)
			gof[k]=total((y-yfit)^2)
		endfor
		k0=where(gof eq min(gof),n0)
		if n0 eq 1 then a[2]=a2a[k0] mod a[1]
		a1a=(indgen(21)-10)*(a[1]/200)+a[1]
		gof=a1a
		for k=0,n_elements(a1a)-1 do begin
			yfit=funct_step(t,[a[0],a1a[k],a[2]],m)
			gof[k]=total((y-yfit)^2)
		endfor
		k0=where(gof eq min(gof),n0)
		if n0 eq 1 then a[1]=a1a[k0]
	endfor
	yfit=funct_step(t,a,m)
	cycle=fix((t-a[2]+a[1]/8)/a[1])
;
t0=t
y0=y
;
;	Plot signals
	if do_slide then begin
		plot,t,y
		oplot,t,yfit,psym=0
	endif else begin
		a2=(a[2]+10*a[1]) mod a[1] - a[1]/8
		tma=(t-a2+a[1]) mod a[1]
		si=sort(tma)
		tma=tma[si]
		msi=m[si]
		by=y[si]
;		by=box(by,50)
		by=median(by,11)
		!x.title='Time [s]'
		!y.title='OB='+string(bid,format='(i1)') $
			+', CH='+string(cid,format='(i2)') $
			+', SC='+string(sid,format='(i1)')
if fiter eq 0 then begin
	!y.range=[0,max(by)*1.3]
endif
;		if siter eq 0 and !d.name eq 'PS' then !p.color=0
;		if siter eq 0 then plot,tma,by,psym=3,xrange=[0,max(tma)],xstyle=1 $
;			      else oplot,tma,by,psym=3
		if !d.name eq 'PS' then !p.color=0 else !p.color=1
	    	!p.title='Fid='+string(fid,format='(i1)')
		index_a=where(msi eq 1,na)
		index_b=where(msi eq 2,nb)
		index_c=where(msi eq 3,nc)
		index_d=where(msi eq 4,nd)
		bin_a=median(by[index_a])
		bin_b=median(by[index_b])
		bin_c=median(by[index_c])
		bin_d=median(by[index_d])
		sig_a=stdev(by[index_a])/sqrt(na)
		sig_b=stdev(by[index_b])/sqrt(nb)
		sig_c=stdev(by[index_c])/sqrt(nc)
		sig_d=stdev(by[index_d])/sqrt(nd)
		q=((bin_a-bin_d)+(bin_c-bin_d))/(bin_b-bin_d)
		e=sqrt(sig_a^2+sig_c^2 $
			   +(q*sig_b)^2 $
			   +((q+2)*sig_d)^2)/(bin_b-bin_d)
		ratio[fid-1,rid-1,sid-1,bid-1,cid-1]=q
		error[fid-1,rid-1,sid-1,bid-1,cid-1]=e
		bic[*,fid-1,rid-1,sid-1,bid-1,cid-1]=[bin_a,bin_b,bin_c,bin_d]
;
		plot,tma,by,psym=3,xrange=[0,max(tma)],xstyle=1
		xyouts,!x.crange[0]+(!x.crange[1]-!x.crange[0])*0.04, $
		       !y.crange[1]-(siter+3)*(!y.crange[1]-!y.crange[0])/20,/data, $
			'Q = '+string(q,format='(f5.3)')+' +/- '+string(e,format='(f5.3)') , $
			charsize=1
;
;		Fit exponential decay to fid=4 data (the lowest frequency)
		if fid eq 4 then begin
;
		ncut=na/20
		t=tma[index_a]
		y=by[index_a]
index=where(abs(y-median(y)) lt 100)
ymed=medianve(y,ymede)
index=where(abs(y-ymed) lt ymede*4)
		te=t[ncut:na-ncut] 
te=t[index]
		te0=min(te)
		ye=y[ncut:na-ncut]
ye=y[index]
oplot,te,ye,color=7,psym=0
		a_decay_a=[median(y),median(y)/4,3.]
		yfit=curvefit(te-te0,ye,ye*0+1,a_decay_a,function_name='pro_decay',/noderivat) 
		oplot,te,yfit,color=2,psym=0
		label='C='+string(a_decay_a[0],format='(f6.1)') $
			+', a='+string(a_decay_a[1],format='(f6.1)') $
			+', t0='+string(1/a_decay_a[2],format='(f4.2)')
		xyouts,0.35,0.54,/normal,label,color=2
		a_1[rid-1,sid-1,bid-1,cid-1]=a_decay_a[0]
		a_2[rid-1,sid-1,bid-1,cid-1]=a_decay_a[1]
		a_3[rid-1,sid-1,bid-1,cid-1]=1/a_decay_a[2]
;
		ncut=nb/20
		t=tma[index_b]
		y=by[index_b]
index=where(abs(y-median(y)) lt 100)
ymed=medianve(y,ymede)
index=where(abs(y-ymed) lt ymede*4)
		te=t[ncut:nb-ncut]
te=t[index]
		ye=y[ncut:nb-ncut]
ye=y[index]
oplot,te,ye,color=7,psym=0
		a_decay_b=[median(ye),median(ye)/6,3.]
		yfit=curvefit(te-a[1]/4,ye,ye*0+1,a_decay_b,function_name='pro_decay',/noderivat) 
		oplot,te,yfit,color=2,psym=0
		label='C='+string(a_decay_b[0],format='(f6.1)') $
			+', a='+string(a_decay_b[1],format='(f6.1)') $
			+', t0='+string(1/a_decay_b[2],format='(f4.2)')
		xyouts,0.35,0.50,/normal,label,color=2
		b_1[rid-1,sid-1,bid-1,cid-1]=a_decay_b[0]
		b_2[rid-1,sid-1,bid-1,cid-1]=a_decay_b[1]
		b_3[rid-1,sid-1,bid-1,cid-1]=1/a_decay_b[2]
;
		ncut=nb/20
		t=tma[index_c]
		y=by[index_c]
index=where(abs(y-median(y)) lt 100)
ymed=medianve(y,ymede)
index=where(abs(y-ymed) lt ymede*4)
		te=t[ncut:nc-ncut]
te=t[index]
		ye=y[ncut:nc-ncut]
ye=y[index]
oplot,te,ye,color=7,psym=0
		a_decay_c=[median(ye),-median(ye)/4,3.]
		yfit=curvefit(te-a[1]/2,ye,ye*0+1,a_decay_c,function_name='pro_decay',/noderivat) 
		oplot,te,yfit,color=2,psym=0
		label='C='+string(a_decay_c[0],format='(f6.1)') $
			+', a='+string(a_decay_c[1],format='(f6.1)') $
			+', t0='+string(1/a_decay_c[2],format='(f4.2)')
		xyouts,0.35,0.46,/normal,label,color=2
		c_1[rid-1,sid-1,bid-1,cid-1]=a_decay_c[0]
		c_2[rid-1,sid-1,bid-1,cid-1]=a_decay_c[1]
		c_3[rid-1,sid-1,bid-1,cid-1]=1/a_decay_c[2]
;
		endif else begin
;
		t=tma[index_a]
		pro_decay,abs(t),a_decay_a,f
		oplot,t,f,color=2,psym=0
;
		t=tma[index_b]
		pro_decay,abs(t-a[1]/4),a_decay_b,f
		oplot,t,f,color=2,psym=0
;
		t=tma[index_c]
		pro_decay,abs(t-a[1]/2),a_decay_c,f
		oplot,t,f,color=2,psym=0
;
		endelse
	endelse
;
	t=t0
	y=y0
;
	kmed=nint(median(cycle))
	index=where(m eq 2 and cycle eq kmed,ni)
	ncut=ni/4
	kmin=min(cycle)
	kmax=max(cycle)
	index=where(m eq 1 and cycle eq kmin,ni)
	if ni lt ncut*2 then kmin=kmin+1
	index=where(m eq 4 and cycle eq kmax,ni)
	if ni lt ncut*2 then kmax=kmax-1
	q=fltarr(kmax-kmin+1)
	e=q
	p=q
	kk=0
	for k=kmin,kmax do begin
;
		index=where(m eq 1 and cycle eq k,ni)
		if do_slide then oplot,t[index],y[index],psym=3,color=0
		index=index[ncut:ni-ncut]
		bin_a=avg(y[index])
		if n_elements(index) lt 2 then sig_a=sqrt(avg(y[index])) $
					  else sig_a=stdev(y[index])/sqrt(n_elements(index)-1)
		if do_slide then oplot,t[index],y[index],psym=1,color=2
;
		index=where(m eq 2 and cycle eq k,ni)
		if do_slide then oplot,t[index],y[index],psym=3,color=0
		index=index[ncut:ni-ncut]
		bin_b=avg(y[index])
		if n_elements(index) lt 2 then sig_b=sqrt(avg(y[index])) $
					  else sig_b=stdev(y[index])/sqrt(n_elements(index)-1)
		if do_slide then oplot,t[index],y[index],psym=1,color=3
;
		index=where(m eq 3 and cycle eq k,ni)
		if do_slide then oplot,t[index],y[index],psym=3,color=0
		index=index[ncut:ni-ncut]
		bin_c=avg(y[index])
		if n_elements(index) lt 2 then sig_c=sqrt(avg(y[index])) $
					  else sig_c=stdev(y[index])/sqrt(n_elements(index)-1)
		if do_slide then oplot,t[index],y[index],psym=1,color=5
;
		index=where(m eq 4 and cycle eq k,ni)
		if do_slide then oplot,t[index],y[index],psym=3,color=0
		index=index[ncut:ni-ncut]
		bin_d=avg(y[index])
		if n_elements(index) lt 2 then sig_d=sqrt(avg(y[index])) $
					  else sig_d=stdev(y[index])/sqrt(n_elements(index)-1)
		if do_slide then oplot,t[index],y[index],psym=1,color=6
;
		p[kk]=bin_b
		q[kk]=((bin_a-bin_d)+(bin_c-bin_d))/(bin_b-bin_d)
		e[kk]=sqrt(sig_a^2+sig_c^2 $
			   +(q[kk]*sig_b)^2 $
			   +((q[kk]+2)*sig_d)^2)/(bin_b-bin_d)
		kk=kk+1
	endfor
; frequencies rates scans beams, channels
	mflux[fid-1,rid-1,sid-1,bid-1,cid-1]=avg(p)
;	ratio(fid-1,rid-1,sid-1,bid-1,cid-1)=avg(q)
;	error(fid-1,rid-1,sid-1,bid-1,cid-1)=stdev(q)/sqrt(n_elements(q)-1)
	chisq[fid-1,rid-1,sid-1,bid-1,cid-1]=total(((q-avg(q))/e)^2)/(n_elements(q)-1)
;
;	Plot individual Q factors
if 0 then begin
	!p.psym=0
	!y.range=[0.9,1.3]
	!p.title='F='+string(freqs[fid-1],format='(f3.1)')+' Hz'+ $
		', S='+string(scan_index[l]+1,format='(i1)')
	!x.title='B='+string(beam,format='(i1)')+ $
		', C='+string(cid+1,format='(i2)')+ $
		', N='+string(avg(p),format='(i4)')
	!y.title='Ratio'+', RChiSq='+string(chisq[fid-1,rid-1,sid-1,bid-1,cid], $
						format='(f4.1)')
	!p.charsize=2.0
	if !d.name ne 'PS' then wset,0
	!p.multi=[16-cid,4,4]
	if !d.name eq 'PS' then !p.multi=[16-cid,3,6]
	ploterr,findgen(n_elements(q))+1,q,e,psym=3
endif
;
	if do_slide then return
	endif
;
ENDFOR
ENDFOR
if !d.name eq 'PS' then device,/close
ENDFOR
ENDFOR
ENDFOR
;
end
;-------------------------------------------------------------------------------
pro print1025,a_1,a_2,a_3,b_1,b_2,b_3,c_1,c_2,c_3
;
a1=avg(a_1,1)
a2=avg(a_2,1)
a3=avg(a_3,1)
b1=avg(b_1,1)
b2=avg(b_2,1)
b3=avg(b_3,1)
c1=avg(c_1,1)
c2=avg(c_2,1)
c3=avg(c_3,1)
;
openw,unit,'exponential.dat',/get_lun
for pr=0,3 do begin
for ob=0,1 do begin
for ch=0,15 do begin
a1=median(a_1[pr,*,ob,ch])
a2=median(a_2[pr,*,ob,ch])
a3=median(a_3[pr,*,ob,ch])
b1=median(b_1[pr,*,ob,ch])
b2=median(b_2[pr,*,ob,ch])
b3=median(b_3[pr,*,ob,ch])
c1=median(c_1[pr,*,ob,ch])
c2=median(c_2[pr,*,ob,ch])
c3=median(c_3[pr,*,ob,ch])
printf,unit,ob+1,pr+1,ch+1, $
	a1,a2,a3, $
	b1,b2,b3, $
	c1,c2,c3, $
	format='(i3,i3,i3,2x,3(4x,f6.1,1x,f5.1,1x,f5.2))'
endfor
endfor
endfor
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro plot1025,mflux,ratio,error,chisq,bic
;
openw,unit,'linear.dat',/get_lun
for fr=0,3 do begin
for pr=0,3 do begin
for sc=0,3 do begin
for ob=0,1 do begin
for ch=0,15 do begin
printf,unit,ob+1,fr+1,pr+1,sc+1,ch+1, $
	bic[0,fr,pr,sc,ob,ch],bic[1,fr,pr,sc,ob,ch],bic[2,fr,pr,sc,ob,ch],bic[3,fr,pr,sc,ob,ch], $
	mflux[fr,pr,sc,ob,ch],ratio[fr,pr,sc,ob,ch],error[fr,pr,sc,ob,ch],chisq[fr,pr,sc,ob,ch], $
	format='(i3,i3,i3,i3,i3,1x,4i5,1x,f6.1,1x,f6.3,1x,f7.4,1x,f5.2)'
endfor
endfor
endfor
endfor
endfor
free_lun,unit
;
return
;
freqs=1/[0.3,1,3,10]
!y.title='((A-D)+(C-D)) / (B-D)'
!y.title=''
!x.title='B'
!x.title=''
!y.range=[0.95,1.35]
!x.range=[0,1100]
!y.style=1
!x.style=1
!x.margin=0
!y.margin=0
;
for ob=0,1 do begin
;
if !d.name eq 'PS' then begin
	device,filename=strcompress('plot.'+string(ob+1)+'.ps',/remove)
	width=17.78
        factor=width/640
        device,xsize=factor*700,ysize=factor*900, $
		xoffset=1,yoffset=1,color=1
endif
!p.multi=[0,4,4]
;
for ch=0,15 do begin
for fr=3,0,-1 do begin
;
if fr eq 3 then plot,mflux[fr,*,*,ob,ch],ratio[fr,*,*,ob,ch],psym=1,color=0 else $
		oplot,mflux[fr,*,*,ob,ch],ratio[fr,*,*,ob,ch],psym=1,color=4-fr
r=poly_fit(mflux[fr,*,*,ob,ch],ratio[fr,*,*,ob,ch],1)
x=findgen(1100)
if fr eq 3 then !p.color=0 else !p.color=4-fr
oplot,poly(x,r),psym=0
;
endfor
endfor
;
if !d.name eq 'PS' then device,/close
;
endfor
;
return
;
end
;-------------------------------------------------------------------------------
pro reduce517,mflux,ratio,error,chisq,fid=fid,beam=beam
;
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
hds_open,'2002-05-17.raw.001.'+string(fid,format='(i2.2)')
get_rawdata
get_bincounts,beam
hds_close
r=total(bincounts,1)
t=fdlstamp()/1000.d0 & t=t-t[0]
case fid of 
1:	begin
	index0=where(t ge 1284 and t le 1319)
	a=[599.01668,1.0782847,-0.25236902]
	end
2:	begin
	index0=where(t ge 971 and t le 1005)
	a=[186.34796,1.0748374,-0.50253726]
	end
3:	begin
	index0=where(t ge 1010 and t le 1040)
	a=[66.675226,1.0735351,-0.26965308]
	end
4:	begin
	index0=where(t ge 910 and t le 943)
	a=[22.538566,1.0729600,-0.10563721]
	end
5:	begin
	index0=where(t ge 920 and t le 952)
	a=[6.7488444,1.0734975,+0.29639835]
	end
endcase
x=t[index0]
x=x-x[0]
;
nc=16
;mflux=fltarr(nc)
;ratio=fltarr(nc)
;error=fltarr(nc)
;chisq=fltarr(nc)
;
for j=0,nc-1 do begin
	y=reform(r[j,index0])
	if max(y) gt 0 then begin
	window_slide,xsize=10000,xv=800,wid=wid
	!p.psym=3
	!y.range=0
	!x.range=0
	yfit=curvefit(x,y,y*0+1,a,function_name='pro_sine',/noderivat)
	plot,x,y
	yfit=funct_step(x,a,m)
	oplot,x,yfit,psym=0
	n=fix(max(x)/a[1])-1
	q=fltarr(n)
	e=q
	p=q
	ncut=10
	for k=1,n do begin
		cycle=fix((x-a[2]+a[1]/8)/a[1])
;
		index=where(m eq 1 and cycle eq k,ni)
		oplot,x[index],y[index],psym=3,color=0
		index=index[ncut:ni-ncut]
		bin_a=avg(y[index])
		sig_a=stdev(y[index])/sqrt(n_elements(index)-1)
		oplot,x[index],y[index],psym=1,color=2
;
		index=where(m eq 2 and cycle eq k,ni)
		oplot,x[index],y[index],psym=3,color=0
		index=index[ncut:ni-ncut]
		bin_b=avg(y[index])
		sig_b=stdev(y[index])/sqrt(n_elements(index)-1)
		oplot,x[index],y[index],psym=1,color=3
;
		index=where(m eq 3 and cycle eq k,ni)
		oplot,x[index],y[index],psym=3,color=0
		index=index[ncut:ni-ncut]
		bin_c=avg(y[index])
		sig_c=stdev(y[index])/sqrt(n_elements(index)-1)
		oplot,x[index],y[index],psym=1,color=5
;
		index=where(m eq 4 and cycle eq k,ni)
		oplot,x[index],y[index],psym=3,color=0
		index=index[ncut:ni-ncut]
		bin_d=avg(y[index])
		sig_d=stdev(y[index])/sqrt(n_elements(index)-1)
		oplot,x[index],y[index],psym=1,color=6
;
		p[k-1]=bin_b
		q[k-1]=((bin_a-bin_d)+(bin_c-bin_d))/(bin_b-bin_d)
		e[k-1]=sqrt(sig_a^2+sig_c^2 $
			   +(q[k-1]*sig_b)^2 $
			   +((q[k-1]+2)*sig_d)^2)/(bin_b-bin_d)
	endfor
	window,/free
	set_screen
	!p.psym=0
	!y.range=[0.9,1.2]
	!p.title='Channel='+string(j+1,format='(i2)') $
	        +', beam='+string(beam,format='(i1)')
	!p.charsize=2.0
	ploterr,q,e,psym=1
;
	mflux[fid-1,j]=avg(p)
	ratio[fid-1,j]=avg(q)
	error[fid-1,j]=stdev(q)/sqrt(n_elements(q)-1)
	chisq[fid-1,j]=total(((q-ratio[j])/e)^2)/(n_elements(q)-1)
	wait,1
	wdall
	widget_control,wid,/destroy
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro fidcheck,filename,d1,d2,d3
;
s=dc_read_free(filename,t,d1,d2,d3,/col)
t=fix(t)+(t mod 1)/0.6
i1=indgen(11)
i2=i1+11
;
!p.multi=[0,2,2]
plot,t,d1,yrange=[0,60],xtitle='UT',ytitle='C delay [microns]'
x=findgen(15)
r=poly_fit(t[i1],d1[i1],2)
oplot,x,poly(x,r),psym=0,color=2
d1[i2]=d1[i2]+poly(t[i2[1]],r)-d1[i2[1]]
oplot,t[i2],d1[i2]
r=poly_fit(t,d1,2,y)
oplot,t,y,psym=0,color=3
;
plot,t,d2,yrange=[0,90],xtitle='UT',ytitle='E delay [microns]'
x=findgen(15)
r=poly_fit(t[i1],d2[i1],2)
oplot,x,poly(x,r),psym=0,color=2
d2[i2]=d2[i2]+poly(t[i2[1]],r)-d2[i2[1]]
oplot,t[i2],d2[i2]
r=poly_fit(t,d2,2,y)
oplot,t,y,psym=0,color=3
;
plot,t,d3,yrange=[0,90],xtitle='UT',ytitle='W delay [microns]'
x=findgen(15)
r=poly_fit(t[i1],d3[i1],2)
oplot,x,poly(x,r),psym=0,color=2
d3[i2]=d3[i2]+poly(t[i2[1]],r)-d3[i2[1]]
oplot,t[i2],d3[i2]
r=poly_fit(t,d3,2,y)
oplot,t,y,psym=0,color=3
;
end
;-------------------------------------------------------------------------------
pro derail,xy,dxy,xyerr,dxyerr
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
files=findfile('2002-10-17.raw.*')
nf=n_elements(files)
;
xy=fltarr(2,nf,3)
dxy=fltarr(2,nf,2)
xyerr=fltarr(2,nf,3)
dxyerr=fltarr(2,nf,2)
;
for i=0,nf-1 do begin
	hds_open,files[i]
	get_rawdata
	hds_close
	xyc=natoffset(quadcounts[1,*,*])
	xye=natoffset(quadcounts[2,*,*])
	xyw=natoffset(quadcounts[3,*,*])
	xy[0,i,0]=avg(xyc[*,0])
	xy[1,i,0]=avg(xyc[*,1])
	xy[0,i,1]=avg(xye[*,0])
	xy[1,i,1]=avg(xye[*,1])
	xy[0,i,2]=avg(xyw[*,0])
	xy[1,i,2]=avg(xyw[*,1])
	xyerr[0,i,0]=stdev(xyc[*,0])
	xyerr[1,i,0]=stdev(xyc[*,1])
	xyerr[0,i,1]=stdev(xye[*,0])
	xyerr[1,i,1]=stdev(xye[*,1])
	xyerr[0,i,2]=stdev(xyw[*,0])
	xyerr[1,i,2]=stdev(xyw[*,1])
	dxy[0,i,0]=avg(xyc[*,0]-xyw[*,0])
	dxy[1,i,0]=avg(xyc[*,1]-xyw[*,1])
	dxy[0,i,1]=avg(xye[*,0]-xyw[*,0])
	dxy[1,i,1]=avg(xye[*,1]-xyw[*,1])
	dxyerr[0,i,0]=stdev(xyc[*,0]-xyw[*,0])
	dxyerr[1,i,0]=stdev(xyc[*,1]-xyw[*,1])
	dxyerr[0,i,1]=stdev(xye[*,0]-xyw[*,0])
	dxyerr[1,i,1]=stdev(xye[*,1]-xyw[*,1])
endfor
;
set_screen
!y.range=0
!x.range=0
!x.title='UT [h]'
;
!p.multi=[0,2,2]
;
window,/free
!p.charsize=1.0
!y.title='X'
!p.title='C-W'
ploterr,scans.time/3600,dxy[0,*,0],dxyerr[0,*,0],psym=3
!y.title='Y'
ploterr,scans.time/3600,dxy[1,*,0],dxyerr[1,*,0],psym=3
;
!y.title='X'
!p.title='E-W'
ploterr,scans.time/3600,dxy[0,*,1],dxyerr[0,*,1],psym=3
!y.title='Y'
ploterr,scans.time/3600,dxy[1,*,1],dxyerr[1,*,1],psym=3
;
!p.multi=[0,2,3]
!p.charsize=2.0
;
window,/free
!p.title='C'
!y.title='X'
ploterr,scans.time/3600,xy[0,*,0],xyerr[0,*,0],psym=3
!y.title='Y'
ploterr,scans.time/3600,xy[1,*,0],xyerr[1,*,0],psym=3
;
!p.title='E'
!y.title='X'
ploterr,scans.time/3600,xy[0,*,1],xyerr[0,*,1],psym=3
!y.title='Y'
ploterr,scans.time/3600,xy[1,*,1],xyerr[1,*,1],psym=3
;
!p.title='W'
!y.title='X'
ploterr,scans.time/3600,xy[0,*,2],xyerr[0,*,2],psym=3
!y.title='Y'
ploterr,scans.time/3600,xy[1,*,2],xyerr[1,*,2],psym=3
end
;-------------------------------------------------------------------------------
pro rflinear1
;
s=dc_read_free('/home/cah/homepage/research/cobra/6way/2002-10-25/linear1.dat', $
	b,f,r,s,c,rate,ratio,error,chisq,/col)
q=fltarr(2,16,5)+1
n=fltarr(2,16,5)
;
for i=0,1 do begin
for j=0,15 do begin
;
	index=where(b eq i+1 and c eq j+1 and f eq 4)
	n[i,j,1:4]=reverse(add(rate[index],4)/4)
	q[i,j,1:4]=reverse(add(ratio[index],4)/4)
;
endfor
endfor
;
save,n,q,filename='/home/cah/oyster/npoi/linear.xdr'
;
end
;-------------------------------------------------------------------------------
pro funct_mismatch,x,a,f
;
f=a[0]*evalfunction('P_0',x) $
 +a[1]*evalfunction('P_1',x) $
 +a[2]*evalfunction('P_2',x)
;
end
;-------------------------------------------------------------------------------
function glassmismatch,l,p,y
;
a=[0.0,0.0,0.0]
y=curvefit(l,p,p*0+1,a,function_name='funct_mismatch',/noderivat)
;
return,a
;
end
;-------------------------------------------------------------------------------
pro movie
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
get_data,'../2002-02-15.cha'
hds_close
; readmodel,'movie.model'
readmodel,'../etavir.model'
star_model.diameter=star_model.diameter*10
;
for i=0,500 do begin
;
Date=nextdate(Date)
Date=nextdate(Date)
;
xyimage
restore
;
filestub='epoch'+string(i,format='(i3.3)')
write_ppm,filestub+'.ppm', $
	255-bytscl(reverse(image(256-100:256+50,256-170:256+30),2))
spawn,'ppmtogif '+filestub+'.ppm > '+filestub+'.gif'
;
endfor
;
spawn,'gifsicle --delay=10 --loop *.gif > anim.gif'
;
end
;-------------------------------------------------------------------------------
pro tomovie,images
;
spawn,'rm -f image*.gif image*.ppm'
;
for i=0,n_elements(images[0,0,*])-1 do begin
;
filestub='image'+string(i,format='(i3.3)')
write_ppm,filestub+'.ppm',bytscl(images[*,*,i])
spawn,'ppmtogif '+filestub+'.ppm > '+filestub+'.gif'
;
endfor
;
spawn,'gifsicle --delay=2 --loop *.gif > images.gif'
;
end
;-------------------------------------------------------------------------------
pro calaqc
;
common StarBase,StarTable,Notes
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
janskys=startable.f12
fluxes=fltarr(2,n_elements(janskys))
;
ya=0.
yb=0.
;
for i=0,n_elements(startable)-1 do begin
	for n=0,n_elements(geninfo)-1 do begin
		loadnight,geninfo[n].date,geninfo[n].configid
		index=where(scans.starid eq startable[i].starid and scans.acqfilter eq 'N8.7',count)
		if count gt 0 then begin
			ya=[ya,scans[index].acqflux(0)]
			yb=[yb,scans[index].acqflux(1)]
		endif
	endfor
	n=n_elements(ya)
	if n gt 1 then begin
	ya=ya[1:n-1]
	yb=yb[1:n-1]
	n=n-1
	if  n eq 1 then begin
		fluxes[0,i]=ya
		fluxes[1,i]=yb
	endif else if n eq 2 then begin
		fluxes[0,i]=avg(ya)
		fluxes[1,i]=avg(yb)
	endif else if n gt 2 then begin
		fluxes[0,i]=median(ya)
		fluxes[1,i]=median(yb)
	endif
	endif
;
	ya=0.
	yb=0.
endfor
;
index=where(fluxes[0,*] gt 0)
fluxes=fluxes[*,index]
janskys=janskys[index]
;	
x=findgen(100)
!p.multi=[0,1,2]
plot,janskys,fluxes[0,*],psym=1
r=poly_fit(janskys,fluxes[0,*],1)
oplot,x,poly(x,r),psym=0
plot,janskys,fluxes[1,*],psym=1
r=poly_fit(janskys,fluxes[1,*],1)
oplot,x,poly(x,r),psym=0
;
y=-2.5*alog10(fluxes[0,*])-jy2n(janskys)
print,mean(y),stddev(y)
;
y=-2.5*alog10(fluxes[1,*])-jy2n(janskys)
print,mean(y),stddev(y)
end
;-------------------------------------------------------------------------------
pro calpho
;
common StarBase,StarTable,Notes
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
n=n_elements(scans)
janskys=fltarr(n)
fluxes=fltarr(n)
za=scans.za
;
for i=0,n-1 do begin
	index=where(startable.starid eq scans[i].starid,count)
	if count eq 1 then janskys[i]=startable[index].f12
	index=where(genconfig.wavelength[*,0] gt 11e-6 $
		and genconfig.wavelength[*,0] lt 13e-6)
	fluxes[i]=avg(scans[i].photonrate(0,index))
endfor
;
index=where(janskys gt 0)
fluxes=fluxes[index]
janskys=janskys[index]
mr=mean(fluxes/janskys)
sd=stddev(fluxes/janskys)
za=za[index]
;	
!p.multi=[0,1,2]
!p.title=date+' (UT)'
plot,janskys,fluxes/janskys,psym=1,xtitle='12 micron flux [Jy]',ytitle='Counts/flux'
xyouts,100,20,/data,'Mean = '+string(mr,format='(f4.1)')+' +/- '+string(sd,format='(f3.1)')
plot,za,fluxes/janskys,psym=1,xtitle='ZA [deg]',ytitle='Counts/flux'
end
;-------------------------------------------------------------------------------
pro calprate,calstar
;
common StarBase,StarTable,Notes
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
get_f12
;
f=cohen(calstar,l)
nl=n_elements(l)
index=where(l[1:nl-1]-l[0:nl-2] gt 0)
index=[0,index+1]
l=l[index]
f=f[index]
lambda=genconfig.wavelength[0:genconfig.numspecchan[0]-1,0]
fluxes=spline(l,f,reverse(lambda))
fluxes=reverse(fluxes)
index=where(genconfig.wavelength[0:genconfig.numspecchan[0]-1,0] gt 11e-6 $
	and genconfig.wavelength[0:genconfig.numspecchan[0]-1,0] lt 13e-6)
i=where(startable.starid eq calstar)
fluxes=fluxes/avg(fluxes[index])*startable[i].f12
;
; Airmass correction
for i=0,n_elements(scans)-1 do $
	scans[i].photonrate=scans[i].photonrate/cos(scans[i].za/(180/!pi))
;
index=where(scans.starid eq calstar,count)
index=index[[0,1,4,5,6]]
rates=scans[index].photonrate
rates=total(reform(rates[0,*,*]),2)/count
;
coeffs=fluxes/rates[0:genconfig.numspecchan[0]-1];*lambda^2/3e-16
;
for i=0,n_elements(scans)-1 do begin
	scans[i].photonrate(0,0:genconfig.numspecchan[0]-1)= $
	scans[i].photonrate(0,0:genconfig.numspecchan[0]-1)*coeffs
endfor
;
end
;-------------------------------------------------------------------------------
pro calscisep,angle,angle_s
;
common StarBase,StarTable,Notes
;
get_stecklum,/init
;
files=findfile('SCI_*.obx')
;
n=n_elements(files)
angle=fltarr(n)
angle_s=fltarr(n)
;
for i=0,n-1 do begin
;
lines=''
status=dc_read_fixed(files[i],lines,/col,resize=[1],format='(a80)')
m=n_elements(lines)
;
	j=where(strpos(lines,'Baseline') eq 0) & j=j[0]
	if strpos(lines[j],'UT') lt 0 then begin
;
	j=where(strpos(lines,'ra') eq 0) & j=j[0]
	words=nameparse(lines[j],'"')
	ra=hms2h(strtrim(words[1],2))
;
	j=where(strpos(lines,'dec') eq 0) & j=j[0]
	words=nameparse(lines[j],'"')
	dec=dms2d(strtrim(words[1],2))
;
	j=where(strpos(lines,'InstrumentComments') eq 0) & j=j[0]
	words=nameparse(lines[j],'"')
	calob=findfile(words[1]+'.obx')
	if strlen(calob[0]) ne 0 then begin
	lines=''
	status=dc_read_fixed(calob,lines,/col,resize=[1],format='(a80)')
;
	j=where(strpos(lines,'ra') eq 0) & j=j[0]
	words=nameparse(lines[j],'"')
	rac=hms2h(strtrim(words[1],2))
;
	j=where(strpos(lines,'dec') eq 0) & j=j[0]
	words=nameparse(lines[j],'"')
	decc=dms2d(strtrim(words[1],2))
;
	angle[i]=winkel(ra,dec,rac,decc)
;
	j=where(strpos(lines,'INS.GRIS.NAME') eq 0) & j=j[0]
	words=nameparse(lines[j],'"')
	insgris=words[1]
	j=where(strpos(lines,'INS.OPT1.NAME') eq 0) & j=j[0]
	words=nameparse(lines[j],'"')
	insopt1=words[1]
	case insgris of
	'PRISM':	begin
			case insopt1 of
			'HIGH_SENS': limit=20.0
			'SCI_PHOT':  limit=40.0
			endcase
			end
	'GRISM':	begin
			case insopt1 of
			'HIGH_SENS': limit=30.0
			'SCI_PHOT':  limit=60.0
			endcase
			end
	endcase
	index=where(startable.f12 gt limit,ns)
	ras=startable[index].ra
	decs=startable[index].dec
	angles=fltarr(ns)
	for j=0,ns-1 do angles[j]=winkel(ra,dec,ras[j],decs[j])
	angle_s[i]=min(angles)
	endif
	endif
endfor
;
index=where(angle ne 0)
angle=angle[index]
angle_s=angle_s[index]
index=uniq(angle)
angle=angle[index]
angle_s=angle_s[index]
;
end
;-------------------------------------------------------------------------------
pro scisim,navg,cohint,v2m3,ibm3,factor=factor,finito=finito
;
if n_elements(factor) eq 0 then factor=10
;
numbin=4L
;
sintable_r=fltarr(numbin)
sintable_i=fltarr(numbin)
;
nbin=20
;
for ibin=0,numbin-1 do begin
	theta=(findgen(nbin)/(nbin-1))*(2*!pi/numbin)+ibin*(2*!pi/numbin)
	sintable_r[ibin]=total(cos(theta))/nbin
	sintable_i[ibin]=total(sin(theta))/nbin
endfor
;
numframe=1024L
vis=fltarr(numframe*numbin)
phs=fltarr(numframe*numbin)
ib1=fltarr(numframe*numbin)+1
ib2=fltarr(numframe*numbin)+1
;
vis[*]=1.0
phs[*]=0
;
if keyword_set(finito) then begin
	restore,'idlsave.dat'
	n_int=fix((cohint/numbin)/t_int)
	help,i1,i2,n_int
	ib1s=add(i1,n_int)/n_int & ib1s=ib1s[0:numframe*numbin-1]
	ib2s=add(i2,n_int)/n_int & ib2s=ib2s[0:numframe*numbin-1]
	ib1s=ib1s/median(ib1s)
	ib2s=ib2s/median(ib2s)
endif else begin
; Compute scintillation
	fmax=100.0
	fknee=0.1
	pmax=1e-9
	pknee=1e-1
;
	fmax=50.0
	fknee=1.0
	pmax=1e-15
	pknee=1e-5
;
	fmax=100.0
	fknee=1.0
	pmax=1e-9
	pknee=1e-5
;
	f=findgen(numframe*numbin)/(numframe*numbin-1)*100
	p=f
	p[where(f lt fknee)]=pknee
	index=where(f gt fknee)
	p[index]=10.0^(alog10(pknee)+(alog10(pmax)-alog10(pknee)) $
				    /(alog10(fmax)-alog10(fknee)) $
				    *(alog10(f[index])-alog10(fknee)))
	s1=siggen(p,fmax) & s1=s1[0:numframe*numbin-1]
	s2=siggen(p,fmax) & s2=s2[0:numframe*numbin-1]
;
	ib1s=(ib1+s1*factor) > 0
	ib2s=(ib2+s2*factor) > 0
endelse
;
series=ib1s
sampling_int=(float(cohint)/1000)/numbin
rms=powerspectrum(series,sampling_int,f,p,f_avg,p_avg)
set_screen
plot,f_avg,alog10(p_avg),xrange=[f[1],1/(2*sampling_int)],xstyle=1,/xlog, $
     xtitle='Frequency [Hz]',psym=0, $
     title=!p.title, $
     xgridstyle=1,xticklen=1,ygridstyle=1,yticklen=1

;
window_slide,xsize=10000
!p.charsize=2.0
!p.multi=[0,1,3]
plot,ib1s,yrange=[0,2]
plot,ib2s,yrange=[0,2]
plot,2*sqrt(ib1s*ib2s)/(ib1s+ib2s),yrange=[0,1]
!p.multi=0
;
vis[*]=vis[*]*2*sqrt(ib1s*ib2s)/(ib1s+ib2s)
;
nph=50.0	; remember bincounts is bytarr with 255 max!
index=lindgen(numframe*numbin) mod numbin
binlevels=(vis*cos(phs)*sintable_r[index]-vis*sin(phs)*sintable_i[index]+1) $
	 *(ib1s+ib2s)*nph/numbin
print,'Maximum binlevel: ',max(binlevels)
bincounts=bytarr(n_elements(binlevels))
s=linknload(!external_lib,'randpoisson',bincounts,binlevels,n_elements(binlevels))
;
bincounts=float(bincounts)
index=lindgen(numframe)*numbin
x=bincounts[index+0]-bincounts[index+2]
y=bincounts[index+1]-bincounts[index+3]
n=bincounts[index]+bincounts[index+1]+bincounts[index+2]+bincounts[index+3]
;
v2m3=!pi^2/2*(add(x^2+y^2-n,navg)/navg)/(add(n,navg)/navg)^2
index=where(n gt 0)
v2am=!pi^2/2*(x[index]^2+y[index]^2-n[index])/n[index]^2
;
ib1m3=add(ib1s,navg*numbin)/(navg*numbin)
ib2m3=add(ib2s,navg*numbin)/(navg*numbin)
ibm3=2*sqrt(ib1m3*ib2m3)/(ib1m3+ib2m3)
;
ib1am=add(ib1s,numbin)/numbin
ib2am=add(ib2s,numbin)/numbin
ibam=2*sqrt(ib1am*ib2am)/(ib1am+ib2am) > 0.01
;
print,'Average using Shao formula: (mean,stdev,sigma)'
print,mean(v2m3),stddev(v2m3)
print,mean(v2m3/ibm3^2),stddev(v2m3/ibm3^2),stddev(v2m3/ibm3^2)/sqrt(numframe/navg)
print,''
print,'AMBER average: (mean,stdev,sigma)'
print,mean(v2am/ibam^2),stddev(v2am/ibam^2),stddev(v2am/ibam^2)/sqrt(numframe)
;
end
;-------------------------------------------------------------------------------
pro finito
;
lines=''
s=dc_read_fixed('ADU_Sky_3e.dat',lines,format='(a300)',/col,resize=[1])
;
n=n_elements(lines)
t=fltarr(n)
p0=fltarr(n)
p2=fltarr(n)
;
t=hms2h(strmid(lines,11,15))
p0=float(strmid(lines,128,6))
p2=float(strmid(lines,144,6))
;
i1=p0
i2=p2
;
t_int=median(t[1:n-1]-t[0:n-2])*3600.0*1000.0	; ms
save,i1,i2,t_int
;
end
;-------------------------------------------------------------------------------
pro svdtest
;
; In OYSTER, in a set of linear equations, DM x = R, the design matrix
; has dimensions (nrow,npar), where npar is the number of unknowns,
; and nrow is the number of "measurements"
;
A = [[1.0, 2.0, -1.0, 2.5], $
     [1.5, 3.3, -0.5, 2.0], $
     [3.1, 0.7,  2.2, 0.0], $
     [0.0, 0.3, -2.0, 5.3], $
     [2.1, 1.0,  4.3, 2.2], $
     [0.0, 5.5,  3.8, 0.2]]
B = [0.0, 1.0, 5.3, -2.0, 6.3, 3.8]
;
nrow=n_elements(b)
;
; Using normal matrices because we have more rows than unknowns
dm=transpose(a)
tm=transpose(dm)
nm=tm#dm
rm=tm#b
;
svd8,nm,w,u,v
svb8,u,w,v,rm,s
print,s
print,dm#s
;
svdc,nm,w,u,v
s=svsol(u,w,v,rm)
print,s
print,dm#s
;
; As the example given in the IDL manual
SVDC, A, W, U, V
s=SVSOL(U, W, V, B)
print,s
print,transpose(a)#s
;
; Solving an equation with fewer rows than unknowns
dm=a
b=b[0:3]
;
svd8,dm,w,u,v
w[where(w/max(w) lt 1e-5)]=0
svb8,u,w,v,b,s
print,b
print,s
print,dm#s
;
svdc,transpose(dm),w,u,v
w[where(w/max(w) lt 1e-5)]=0
s=svsol(u,w,v,b)
print,b
print,s
print,dm#s
;
end
;-------------------------------------------------------------------------------
pro joerguwe
;
common StarBase,StarTable,Notes
;
get_wbs
i=where(startable.ra gt 8 and startable.ra lt 13 and startable.dec lt 10)
startable=startable[i]
i=where(startable.a gt 0 and startable.a le 0.2 and startable.dmv lt 3)
startable=startable[i]
spec_parse
startable.class=5
mk_dwarfs
;
j=where(startable.mk lt 4)
stop
;
end
;-------------------------------------------------------------------------------
function simbad_references,target0
;
words=nameparse(target0)
if strpos(target0,'HDN') ge 0 then words=['HD',strmid(target0,3,6)]
if strpos(target0,'HIP') ge 0 then words=['HIP',strmid(target0,3,6)]
target=strjoin(words,'+')
; Simbad4 call (using scripts)
simbad='http://simweb.u-strasbg.fr/simbad/'
script='sim-script?submit=submit+script&script=query+id+'
browser='lynx -dump '
spawn,browser+'"'+simbad+script+target+'"',r
j=where(strpos(r,'liste bibcodes') ge 0,count) & j=j[0]
words=nameparse(r[j],['(',')'])
;
return,fix(words[1])
;
end
;-------------------------------------------------------------------------------
pro famous,ra_in,dec_in,observed
;
status=dc_read_fixed('ra.txt',lra,/col,format='(a80)')
status=dc_read_fixed('dec.txt',ldec,/col,format='(a80)')
n=n_elements(lra)
;
ra=fltarr(n)
dec=fltarr(n)
;
for i=0,n-1 do begin
	w=nameparse(lra[i],'"')
	ra[i]=hms2h(strtrim(w[1],2))
	w=nameparse(ldec[i],'"')
	dec[i]=dms2d(strtrim(w[1],2))
endfor
;
m=n_elements(ra_in)
observed=intarr(m)
;
for i=0,m-1 do begin
	d=sqrt(((ra-ra_in[i])*15)^2+(dec-dec_in[i])^2)
	if min(d) lt 2./3600 then observed[i]=1
endfor
;
end
;-------------------------------------------------------------------------------
pro compile_proposal,ra_min,ra_max,n_in=n_in
;
common StarBase,StarTable,Notes
;
if n_elements(ra_min) eq 0 then ra_min=0
if n_elements(ra_max) eq 0 then ra_max=12
;
; get_bsc
; get_n12
; rename_starids,'bsc-hdn'
; get_f12
;
get_sky
index=where(startable.ra gt ra_min $
	and startable.ra lt ra_max $
	and startable.dec lt 20)
startable=startable[index]
get_i12
;
index=where(startable.f12 gt 30,count)
startable=startable[index]
;
if n_elements(n_in) eq 0 then begin
	n_in=intarr(count)
	for j=0,count-1 do n_in[j]=simbad_references(startable[j].starid)
	save,n_in
endif
n=n_in
;
ra=startable.ra
dec=startable.dec
starid=startable.starid
famous,ra,dec,observed
;
jndex=where(observed eq 0 and ra gt ra_min and ra lt ra_max,count)
;
starid=starid[jndex]
n=n[jndex]
si=sort(n)
starid=starid[si]
n=n[si]
;
for i=0,count-1 do print,starid[i],n[i]
;
end
;-------------------------------------------------------------------------------
function threecolor,p,t,f
;
; General order: blue, green, red
l=double([2.02,2.22,2.44])
l=double([0.50,0.55,0.80])
s=1/l
a=(2372.434d0+684255.24d0/(130.d0-s^2)+4549.4d0/(38.9d0-s^2))
b=a-(6487.31d0+58.058d0*s^2-0.7115d0*s^4+0.08851d0*s^6)
x=[a[2]*b[1]-a[1]*b[2],a[0]*b[2]-a[2]*b[0],a[1]*b[0]-a[0]*b[1]]
d=owens(l,p,t,f)*100 ; m
d=d+randomn(seed,3)*1e-6
;
d3c=total(d*x)/total(x)
return,d3c-100
;
end
;-------------------------------------------------------------------------------
