;*******************************************************************************
; File: astrom.pro
;
; Description: 
; ------------
; This file contains functions (Block 1) and procedures (other blocks)
; to do astrometric calculations.
;
; Block directory:
; ----------------
; Block 1: polarmotion,utc2ut1,ut12gst,ra2date,
;	   hourangle,zenithangle,mirrorangle,
;	   catseyedelay,
;          horizon2equatorial,equatorial2horizon,azel2horizon,horizon2azel,
;	   equatorial2hadec,hadec2equatorial,
;	   baselinecoord,stationdelay,calcuv,calcpro,
; Block 2: topostar,apparentstar,topoplanet,apparentplanet,
;	   calcgeo,calcastrom,solveastrom,referencestation,
;	   initvolvox,calcvolvox,solvevolvox,
;	   getcorr,whitecorr,pivotcorr,volvoxcorr
; Block 3: sidmodel,sidpointing,sidlimits,fdllimits,refraction,riseset,roseset
;
;************************************************************************Block 1
function polarmotion,utc,latitude
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
Longitude=GeoParms.Longitude
Latitude=GeoParms.Latitude
Altitude=GeoParms.Altitude
EarthRadius=GeoParms.EarthRadius
J2=GeoParms.J2
;
; Apply polarmotion corrections to Longitude and Latitude
; Convert to earth-fixed, rotating, rectangular, right-handed geocentric system
C=1/sqrt((cos(Latitude/RAD))^2+(1-J2)^2*(sin(Latitude/RAD))^2)
S=(1-J2)^2*C
n_utc=n_elements(utc)
pos1=dblarr(3,n_utc,/nozero)
pos1(0,*)=(EarthRadius*C+Altitude)*cos(Latitude/RAD)*cos(Longitude/RAD)
pos1(1,*)=(EarthRadius*C+Altitude)*cos(Latitude/RAD)*sin(Longitude/RAD)
pos1(2,*)=(EarthRadius*S+Altitude)*sin(Latitude/RAD)
pos2=pos1
;
utc_days=utc/86400
pole_x=poly(utc_days,GeoParms.pole_x_coeffs)
pole_y=poly(utc_days,GeoParms.pole_y_coeffs)
status=linknload(!external_lib,'polarmotion',n_utc,pole_x,pole_y,pos1,pos2)
;
; Convert back to longitude, latitude system
Longitude=atan(pos2(1,*),pos2(0,*))*RAD
Latitude=asin(pos2(2,*)/(EarthRadius*S+Altitude))*RAD
;
if n_utc eq 1 then begin
	Longitude=Longitude(0)
	Latitude=Latitude(0)
endif
;
return,longitude
;
end
;-------------------------------------------------------------------------------
function utc2ut1,utc
;
; Returns UT1 in seconds. UTC must be in seconds. Uses polynomial fit
; to data contained in IERS Bulletin A (Mark3 geodetic VLBI observations).
; Fit is made in get_GeoParms(access.lib).
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
utc_days=double(utc)/86400
ut1=utc+poly(utc_days,GeoParms.ut1utc_coeffs)
if n_elements(ut1) eq 1 then return,ut1(0) else return,ut1
;
end
;-------------------------------------------------------------------------------
function ut12gst,utc,ut1,mean=mean,local=local
;
; Returns Greenwich apparent sidereal time (in hours). 
; Calls calcsiderealtime in NOVAS C-library of astrometry subroutines 
; (Kaplan, G.H. et al. 1989, AJ, 97, 1197). 
; Input arrays, utc, and ut1 in seconds, should be type double.
; If no ut1 specified, default to utc. If mean=1, return mean time.
; If local=1, return local sidereal time.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(ut1) eq 0 then ut1=utc
if n_elements(mean) eq 0 then mean=0L else mean=long(mean)
if n_elements(local) eq 0 then local=0
;
parsedate,Date,y,m,d
num_times=n_elements(UT1)
julianhi=dblarr(num_times)+long(julian(y,m,d))
julianlo=(UTC+GeoParms.TAI_UTC+GeoParms.TDT_TAI)/86400+0.5
TDT_UT1=utc-ut1+GeoParms.TAI_UTC+GeoParms.TDT_TAI
;
gst=dblarr(num_times)
status=linknload(!external_lib,'ut12gst',num_times,julianhi,julianlo,TDT_UT1, $
	gst,mean)
if status ne 0 then print,'***Error(UT12GST): NOVAS return error code',status
if local then gst=gst+geoparms.longitude/15
if n_elements(gst) eq 1 then return,gst(0) else return,gst
;
end
;-------------------------------------------------------------------------------
function ra2date,ra
;
; Return date on which the star of given RA passes the meridian
; at midnight local time for the location loaded in geoparms.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
lst0=ut12gst(-geoparms.longitude*240,/local)
parsedate,date,y,m,d
jd2date,julian(y,m,d)+(ra-lst0)*(365.25/24),y,m,d
return,constrictordate(y,m,d)
;
end
;-------------------------------------------------------------------------------
function hourangle,gst,ra
;
; Return hour angle in hours. Input GST is in hours, as is RA.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
ha=gst-ra+polarmotion(-geoparms.longitude/15)/15
j=where(ha gt +12,count)
if count gt 0 then ha(j)=ha(j)-24
j=where(ha lt -12,count)
if count gt 0 then ha(j)=ha(j)+24
if n_elements(ha) eq 1 then return,ha(0) else return,ha
;
end
;-------------------------------------------------------------------------------
function zenithangle,hourangle,declination
;
; Returns zenith angle in degrees. Input HA is in hours, DEC in degrees.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
rad=180/pi_circle
dec=declination/rad
ha=hourangle*(15/rad)
longitude=polarmotion(-geoparms.longitude/15,latitude)
za=(rad*acos(sin(dec)*sin(Latitude/rad) $
  	        +cos(dec)*cos(Latitude/rad)*cos(ha)))
if n_elements(za) eq 1 then return,za(0) else return,za
;
end
;-------------------------------------------------------------------------------
function mirrornormal,model,ha,dec
;
; Mirror normal in XYZ-coordinates (right-handed, X east, Y north, Z up):
; Input HA is in hours, DEC in degrees.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
rad=180/pi_circle
ha_r=ha*(15/rad)
dec_r=dec/rad
longitude=polarmotion(-geoparms.longitude/15,latitude)
;
; Feed vector, originating at siderostat mirror
feed_az=model(0)/rad; about 88.5 deg, i.e. pointing towards West
feed_el=model(1)/rad; about 18.5 deg, i.e. beam rises towards NAT mirror
feed_x=-sin(feed_az)*cos(feed_el)
feed_y=+cos(feed_az)*cos(feed_el)
feed_z=+sin(feed_el)
;
; Star vector
x=-cos(dec_r)*sin(ha_r)
y=+sin(dec_r)*cos(Latitude/rad) $
  -cos(dec_r)*sin(Latitude/rad)*cos(ha_r)
z=+cos(dec_r)*cos(Latitude/rad)*cos(ha_r) $
  +sin(dec_r)*sin(Latitude/rad)
;
; Mirror normal
m1=(feed_x+x)
m2=(feed_y+y)
m3=(feed_z+z)
length=sqrt(m1^2+m2^2+m3^2)
m1=m1/length
m2=m2/length
m3=m3/length
;
return,[[m1],[m2],[m3]]
;
end
;-------------------------------------------------------------------------------
function mirrorangle,ha,dec
;
; Return mirror angle in degrees. Input HA is in hours, DEC in degrees.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
rad=180/pi_circle
ha_r=ha*(15/rad)
dec_r=dec/rad
case system_id(SystemId) of
	'NPOI'    :begin
		   feed_angle=18.5/rad
		   feed_x=-cos(feed_angle)
		   feed_y= 0
		   feed_z=+sin(feed_angle)
		   x=-cos(dec_r)*sin(ha_r)
		   z=+cos(dec_r)*cos(GeoParms.Latitude/rad)*cos(ha_r) $
		     +sin(dec_r)*sin(GeoParms.Latitude/rad)
		   ma=rad*acos(feed_x*x+feed_z*z)
		   if n_elements(ma) eq 1 then return,ma(0) else return,ma
		   end
	      else:begin
		   ma=rad*acos(-sin(ha_r)*cos(dec_r))
		   if n_elements(ma) eq 1 then return,ma(0) else return,ma
		   end
endcase
;
end
;-------------------------------------------------------------------------------
function mirrorpivot,model,ha,dec,v
;
; Return coordinates, in X(east)Y(north)Z(up) system, of catseye center given a
; specific siderostat pointing. The offset of the catseye center from the 
; pivot is given by vector v, in a coordinate system of the mirror which is
; defined to be identical to the XYZ system if the mirror is pointing exactly
; west. In this orientation, the D coord. (axis offset) also increases east.
; The coordinates of the unit vectors of the mirror system in the XYZ
; system are derived from the mirror normal (identical to the X axis at the 
; reference orientation), and the vectors orthogonal to it and the azimuth and
; elevation axes, respectively. Then the coordinates are given by the vector
; sum of the unit vectors times the coordinates of v.
;
mn=-mirrornormal(model,reform([ha]),reform([dec]))
mp=mn
az=[0,0,1.0]
for i=0,n_elements(ha)-1 do begin
;	Contribution from vector mirror-catseye
	el=kreuzprodukt(mn(i,*),az,/n)
	mp(i,*)=mn(i,*)*v(0) $
	       +kreuzprodukt(az,mn(i,*),/n)*v(1) $
	       +kreuzprodukt(el,mn(i,*),/n)*v(2)
;	Contribution from vector az-axis el-axis offset
	mp(i,*)=mp(i,*)+kreuzprodukt(az,el,/n)*v(3)
endfor
;
return,mp
;
; This is another method, which applies the rotation to the 
; unit vectors in the opposite sense, and then determines
; the coordinates of the fixed catseye offset vector using
; its dot product with the rotated unit vectors. Rotations
; by angle alpha around the Y axis, then beta around the 
; Z (i.e. azimuth) axis will result in coordinates
; [cos(a)*cos(b),cos(a)*sin(b),-sin(a)] for the CC. Applying
; the same rotations in the opposite sense first by -alpha
; and then -beta around the rotated Z axis will result in
; coordinates [cos(a)*cos(b),-sin(b),sin(a)cos(b)].
;
mn=-mirrornormal(model,ha,dec)
mp=mn
az=[0,0,1.0]
for i=0,n_elements(ha)-1 do begin
	x=mn(i,*)
	x(1)=-(mn(i,1)/sqrt(1.-mn(i,2)^2))
	x(2)=-mn(i,2)*sqrt(1.-x(1)^2)
	z=kreuzprodukt(x,[0,1.,0],/n)
	y=kreuzprodukt(z,x,/n)
	mp(i,*)=[total(v*x),total(v*y),total(v*z)]
endfor
;
return,mp
;
end
;-------------------------------------------------------------------------------
function catseyedelay,feed_angle,cx,cy,cz,ha,dec,offset,cxe,cye,cze,e
;
; Return change in optical path due to change in catseye position xyz.
; Right-handed coordinate system, X east, Y north, Z up.
; Input HA is in hours, DEC in degrees.
;
; The formula works differently for corrections of stellar and white light
; observations. For stellar light, the component of the pivot motion in the
; direction of the mirror normal is computed, then multiplied with the 
; projection of the mirror normal onto the feed (or star) direction. The
; optical path change, which stellar light suffers as a consequence of 
; the motion of the pivot and which was compensated with the delay lines,
; is twice that value. 
;
; White light measurements are used to monitor the optical from the siderostats
; through the feed system to the beam combiner. The autocollimation setup is
; that of a Michelson interferometer. If the siderostat mirror moves by a
; distance x, then the delay line moves by x/2 to keep the path length the same.
; The optical path change in the delay line measured by the metrology is twice
; the physical motion of the delay cart and is equal to x. In other words, the
; metrology delay changes are equal to the physical motion of the siderostat,
; or path length changes in the feed system. Therefore, the optical path change
; due to pivot motion as measured by the siderostat metrology and derived in
; the previous paragraph has to be divided by 2 before applying it to the 
; white light delay measurements. The corrected white light delays, as
; measures of the optical path changes solely due to feed system motions,
; can then be directly applied to the stellar delays.
;
; Note that the calling routine has to divide the optical path by 2
; for white light measurements in order to convert them to single-pass
; optical paths.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
rad=180/pi_circle
ha_r=ha*(15/rad)
dec_r=dec/rad
;
; Feed vector, originating at siderostat mirror
feed_az=feed_angle(0)/rad; about 88.5 deg, i.e. pointing towards West
feed_el=feed_angle(1)/rad; about 18.5 deg, i.e. beam rises towards NAT mirror
feed_x=-sin(feed_az)*cos(feed_el)
feed_y=+cos(feed_az)*cos(feed_el)
feed_z=+sin(feed_el)
;
; Star vector
x=-cos(dec_r)*sin(ha_r)
y=+sin(dec_r)*cos(GeoParms.Latitude/rad) $
  -cos(dec_r)*sin(GeoParms.Latitude/rad)*cos(ha_r)
z=+cos(dec_r)*cos(GeoParms.Latitude/rad)*cos(ha_r) $
  +sin(dec_r)*sin(GeoParms.Latitude/rad)
;
; Mirror normal
m1=(feed_x+x)
m2=(feed_y+y)
m3=(feed_z+z)
length=sqrt(m1^2+m2^2+m3^2)
m1=m1/length
m2=m2/length
m3=m3/length
;
if n_elements(cze) ne 0 then $
	e=2*(m1*x+m2*y+m3*z)*sqrt((cxe*m1)^2+(cye*m2)^2+(cze*m3)^2)
index=where(cxe le 0 or cye le 0 or cze le 0,count)
if count gt 0 then e(index)=-e(index)
;
; A note on the sign: if additional optical path is introduced through
; motion of the pivot away from the feed, the returned delay is negative,
; indicating that the corresponding delay line moved towards the front,
; i.e. removing optical path.
;
return,2*(cx*m1+cy*m2+cz*m3)*(m1*x+m2*y+m3*z)+2*offset*(m1*x+m2*y+m3*z)
;
end
;-------------------------------------------------------------------------------
function horizon2equatorial,h_coord,utc
;
; Convert station coordinates from horizon system to equatorial. Note that the 
; X-axis is along local meridian, NOT the Greenwich meridian as used in VLBI. 
; This corresponds to the X,Y,Z system used by Thompson, Moran and Swenson
; "Interferometry and Synthesis in Radio Astronomy", page 86 (1st Edition).
; That is, Z points towards the north pole along the earth's rotational axis,
; X along the projection of the local meridian on the equator, increasing as
; you go from the earth's axis to the local point on the earth's surface and
; Y points east.
;
; As in all following coordinate transformation function, inputs can be
; natural order arrays, i.e. (2) or (2,*) and (3) or (3,*).
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if n_elements(utc) eq 0 then utc=0.d0
longitude=polarmotion(utc,latitude)
;
e_coord=h_coord	; to preserve type
;
e_coord(0,*)=cos(Latitude/RAD)*h_coord(2,*) $
   	    -sin(Latitude/RAD)*h_coord(1,*)
e_coord(1,*)=h_coord(0,*)
e_coord(2,*)=cos(Latitude/RAD)*h_coord(1,*) $
	    +sin(Latitude/RAD)*h_coord(2,*)
;
return,e_coord
;
end
;-------------------------------------------------------------------------------
function equatorial2horizon,e_coord,utc
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if n_elements(utc) eq 0 then utc=0.d0
longitude=polarmotion(utc,latitude)
;
h_coord=e_coord
;
h_coord(0,*)=e_coord(1,*)
h_coord(1,*)=cos(Latitude/RAD)*e_coord(2,*) $
	    -sin(Latitude/RAD)*e_coord(0,*)
h_coord(2,*)=cos(Latitude/RAD)*e_coord(0,*) $
	    +sin(Latitude/RAD)*e_coord(2,*)
;
return,h_coord
;
end
;-------------------------------------------------------------------------------
function azel2horizon,a_coord
;
; Convert azimuth angle (0=N, 90=W) and elevation angle to horizontal
; coordinates.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
b=fltarr(1,n_elements(a_coord(0,*)))
h_coord=[a_coord,b]
h_coord(0,*)=-sin(a_coord(0,*)/RAD)*cos(a_coord(1,*)/RAD)
h_coord(1,*)= cos(a_coord(0,*)/RAD)*cos(a_coord(1,*)/RAD)
h_coord(2,*)= 		            sin(a_coord(1,*)/RAD)
;
return,h_coord
;
end
;-------------------------------------------------------------------------------
function horizon2azel,h_coord
;
; Convert horizontal coordinates (xyz) to azimuth angle (0=N, 90=W) 
; and elevation. xyz is east,north,up.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
a_coord=h_coord(0:1,*)
a_coord(0,*)=(atan(h_coord(1,*),h_coord(0,*))*RAD+270) mod 360
a_coord(1,*)=(atan(h_coord(2,*),sqrt(h_coord(0,*)^2+h_coord(1,*)^2)))*RAD
;
return,a_coord
;
end
;-------------------------------------------------------------------------------
function hadec2equatorial,a_coord
;
; Convert hour angle and declination to equatorial XYZ coordinates. Note that
; the X-axis is along local meridian, NOT the Greenwich meridian as used in VLBI. 
; This corresponds to the X,Y,Z system used by Thompson, Moran and Swenson
; "Interferometry and Synthesis in Radio Astronomy", page 86 (1st Edition).
; That is, Z points towards the north pole along the earth's rotational axis,
; X along the projection of the local meridian on the equator, increasing as
; you go from the earth's axis to the local point on the earth's surface and
; Y points east.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
b=fltarr(1,n_elements(a_coord(0,*)))
e_coord=[a_coord,b]
e_coord(2,*)=+sin(a_coord(1,*)/RAD)
e_coord(1,*)=-sin(a_coord(0,*)*15/RAD)*cos(a_coord(1,*)/RAD)
e_coord(0,*)=+cos(a_coord(0,*)*15/RAD)*cos(a_coord(1,*)/RAD)
;
return,e_coord
;
end
;-------------------------------------------------------------------------------
function equatorial2hadec,e_coord
;
; Convert equatorial coordinates to hour angle (h) and declination (deg).
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
a_coord=e_coord(0:1,*)
a_coord(0,*)=atan(-e_coord(1,*),e_coord(0,*))*RAD/15
a_coord(1,*)=asin( e_coord(2,*))*RAD
;
return,a_coord
;
end
;-------------------------------------------------------------------------------
function hadec2azel,ha_coord,lat=lat
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
RAD=180/pi_circle
;
if n_elements(lat) eq 0 then lat=geoparms.latitude
lat0=lat/RAD
;
ha=ha_coord(0,*)*15/RAD
dec=ha_coord(1,*)/RAD
;
ho_coord=ha_coord*0
sin_a=sin(dec)*sin(lat0)+cos(dec)*cos(lat0)*cos(ha)
ho_coord(1,*)=asin(sin_a)
i=where(sin_a ne 1,count)
if count gt 0 then $
ho_coord(0,i)=acos((sin(dec(i))-sin(lat0)*sin_a(i)) $
		  /(cos(lat0)*sqrt(1-sin_a(i)^2)))
index=where(sin(ha) gt 0,count)
if count gt 0 then ho_coord(0,index)=2*pi_circle-ho_coord(0,index)
;
return,ho_coord*RAD
;
end
;-------------------------------------------------------------------------------
function azel2hadec,azel,lat=lat
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
RAD=180/pi_circle
;
if n_elements(lat) eq 0 then lat=geoparms.latitude
lat0=lat/RAD
;
az=azel(0,*)/RAD
el=azel(1,*)/RAD
;
hadec=azel*0
sin_d=sin(el)*sin(lat0)+cos(el)*cos(lat0)*cos(az)
hadec(1,*)=asin(sin_d)
i=where(sin_d ne 1,count)
if count gt 0 then begin
	arg=(sin(el(i))-sin(lat0)*sin_d(i))/(cos(lat0)*sqrt(1-sin_d(i)^2))
	hadec(0,i)=acos(arg < 1)
endif
index=where(sin(az) gt 0,count)
if count gt 0 then hadec(0,index)=2*pi_circle-hadec(0,index)
hadec(0,*)=hadec(0,*)/15
;
return,hadec*RAD
;
end
;-------------------------------------------------------------------------------
function baselinecoord,ob,bl,utc
;
; Computes baseline coordinates from station coordinates.
; Convention used is: station 2 - station 1. That way, a baseline vector
; B_12 is oriented from telescope 1 to telescope 2.
; Also applies polar motion corrections.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
i=where(GenConfig.StationId(0:GenConfig.NumSid-1) $
	eq strmid(GenConfig.BaselineId(bl,ob),0,3))
j=where(GenConfig.StationId(0:GenConfig.NumSid-1) $
	eq strmid(GenConfig.BaselineId(bl,ob),4,3))
e_coord= $
   horizon2equatorial(GenConfig.StationCoord(*,j)-GenConfig.StationCoord(*,i))
;
n_utc=n_elements(utc)
pos1=dblarr(3,n_utc,/nozero)
pos2=pos1
pos1(0,*)=e_coord(0)
pos1(1,*)=e_coord(1)
pos1(2,*)=e_coord(2)
utc_days=utc/86400
pole_x=poly(utc_days,GeoParms.pole_x_coeffs)
pole_y=poly(utc_days,GeoParms.pole_y_coeffs)
status=linknload(!external_lib,'polarmotion',n_utc,pole_x,pole_y,pos1,pos2)
;
; We still don't trust pos2 even though I think it was tested
return,pos1
;
end
;-------------------------------------------------------------------------------
function stationdelay,uvw
;
; This function computes station based delays from baseline delays. It should
; only be used for astrometric delays, since it does not do a least-squares
; reduction. The delay of the first station in the first baseline is set to 0.
; Note that the station offset (the path to the array center) difference 2-1
; is subtracted from station j, which is equivalent to adding it to
; station i. If the difference is positiv, station i has to put in extra
; delay. Note that a delay line position is positive if it puts in extra 
; delay; the position is zero if the cart is at the entrance window, and it
; is +35 m (for NPOI) if the cart is all the way back at the end of the delay 
; line tank.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
n=n_elements(uvw(0,0,0,0,*))
d=dblarr(GenConfig.NumSid,n)
s=strarr(n_elements(d(*,0)))
;
REPEAT begin
for ob=0,GenConfig.NumOutBeam-1 do begin
for bl=0,GenConfig.NumBaseline(ob)-1 do begin
	i=where(GenConfig.StationId(0:GenConfig.NumSid-1) $
                eq strmid(GenConfig.BaselineId(bl,ob),0,3))
	i=i(0)
	j=where(GenConfig.StationId(0:GenConfig.NumSid-1) $
                eq strmid(GenConfig.BaselineId(bl,ob),4,3))
	j=j(0)
	if s(i) ne '' and s(j) eq '' then begin
		s(j)=GenConfig.StationId(j)
		d(j,*)=d(i,*)+uvw(ob,0,bl,2,*)*GenConfig.Wavelength(0,ob)- $
	              (GenConfig.StationCoord(3,j)-GenConfig.StationCoord(3,i))
	endif else if s(i) eq '' and s(j) ne '' then begin
		s(i)=GenConfig.StationId(i)
		d(i,*)=d(j,*)-uvw(ob,0,bl,2,*)*GenConfig.Wavelength(0,ob)+ $
		      (GenConfig.StationCoord(3,j)-GenConfig.StationCoord(3,i))
	endif else if s(i) eq '' and s(j) eq '' then begin
		s(i)=GenConfig.StationId(i)
		s(j)=GenConfig.StationId(j)
		d(i,*)=0
		d(j,*)=uvw(ob,0,bl,2,*)*GenConfig.Wavelength(0,ob)- $
		      (GenConfig.StationCoord(3,j)-GenConfig.StationCoord(3,i))
	endif
endfor
endfor
icheck=0
for i=0,GenConfig.NumSid-1 do if s(i) eq '' then icheck=1
ENDREP until icheck eq 0
;
return,d
;
end
;-------------------------------------------------------------------------------
function calcuv,hourangle,declination,utc,uvw
;
; Compute and return uvw coordinates, a right-handed coordinate system
; with v increasing to the N and u increasing to the E. This corresponds
; to astronomical images with E to the left and N up.
;
; To compute the baseline position angle of (u,v), use atan(u,v)=atan(u/v).
; The baseline PA then is the angle on the ground of the projected baseline
; starting from N over E.
;
; X,Y,Z coordinate system (see also Interferometry and Synthesis in Radio
; Astronomy, Thompson, Moran, & Swenson, page 86):
; X in the meridian plane, Y towards the east, Z towards the north pole.
; utc (in seconds) will be used for polar motion correction of the baselines.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
rad=180/pi_circle
ha=hourangle*(15/rad)
dec=declination/rad
;
n=n_elements(ha)
su=dblarr(n,3) & sv=su & sw=su
;
su(*,0)=+sin(ha)
su(*,1)=+cos(ha)
su(*,2)=+0
sv(*,0)=-sin(dec)*cos(ha)
sv(*,1)=+sin(dec)*sin(ha)
sv(*,2)=+cos(dec)
sw(*,0)=+cos(dec)*cos(ha)
sw(*,1)=-cos(dec)*sin(ha)
sw(*,2)=+sin(dec)
;
if n_elements(utc) eq 0 then utc=fltarr(n)
;
if n_elements(uvw) eq 0 then $
	uvw=dblarr(GenConfig.NumOutBeam, $
		max(GenConfig.NumSpecChan),max(GenConfig.NumBaseline),3,n)
;
for ob=0,GenConfig.NumOutBeam-1 do begin
	for bl=0,GenConfig.NumBaseline(ob)-1 do begin
		b_xyz=baselinecoord(ob,bl,utc)
		um=su(*,0)*b_xyz(0,*)+su(*,1)*b_xyz(1,*)+su(*,2)*b_xyz(2,*)
		vm=sv(*,0)*b_xyz(0,*)+sv(*,1)*b_xyz(1,*)+sv(*,2)*b_xyz(2,*)
		wm=sw(*,0)*b_xyz(0,*)+sw(*,1)*b_xyz(1,*)+sw(*,2)*b_xyz(2,*)
		for ic=0,GenConfig.NumSpecChan(ob)-1 do begin
			uvw(ob,ic,bl,0,*)=um/GenConfig.Wavelength(ic,ob)
			uvw(ob,ic,bl,1,*)=vm/GenConfig.Wavelength(ic,ob)
			uvw(ob,ic,bl,2,*)=wm/GenConfig.Wavelength(ic,ob)
		endfor
	endfor
endfor
;
return,uvw
;
end
;-------------------------------------------------------------------------------
function calcpro,baseline,ut
;
; Compute projected baseline and return length [m] and pa [degrees, N over E].
; Input baseline, e.g. 'VE0-VG0', ut [hours]
;
; Startable and GeoParms must be loaded.
;
common StarBase,startable,notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
times=ut*3600
num_times=n_elements(times)
;
; Compute apparent star positions
stars=strarr(num_times)+startable(0).starid
topostar,times,stars,startable,ra_app,dec_app
;
; Compute u,v coordinates
;
rad=180/pi_circle
ha=hourangle(ut12gst(times,utc2ut1(times)),ra_app)*(15/rad)
dec=dec_app/rad
;
su=dblarr(num_times,3) & sv=su & sw=su
;
su(*,0)=+sin(ha)
su(*,1)=+cos(ha)
su(*,2)=+0
sv(*,0)=-sin(dec)*cos(ha)
sv(*,1)=+sin(dec)*sin(ha)
sv(*,2)=+cos(dec)
sw(*,0)=+cos(dec)*cos(ha)
sw(*,1)=-cos(dec)*sin(ha)
sw(*,2)=+sin(dec)
;
i=where(stationtable.StationId eq strmid(Baseline,0,3))
j=where(stationtable.StationId eq strmid(Baseline,4,3))
s1c=[stationtable(i).x,stationtable(i).y,stationtable(i).z]
s2c=[stationtable(j).x,stationtable(j).y,stationtable(j).z]
b_xyz=horizon2equatorial(s2c-s1c)
;
um=su(*,0)*b_xyz(0)+su(*,1)*b_xyz(1)+su(*,2)*b_xyz(2)
vm=sv(*,0)*b_xyz(0)+sv(*,1)*b_xyz(1)+sv(*,2)*b_xyz(2)
wm=sw(*,0)*b_xyz(0)+sw(*,1)*b_xyz(1)+sw(*,2)*b_xyz(2)
;
bl=sqrt(um^2+vm^2)
pa=atan(um,vm)*rad
;
return,[[bl],[pa]]
;
end
;************************************************************************Block 2
pro checknovas
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
get_startable,['BSC0424','BSC1851','BSC4199']
rename_bsc
stars=startable.starid
;
startable.ra=[2.5301955556d0,5.5334438889d0,10.7159355556d0]
startable.dec=[89.2640888889d0,-0.2991333333d0,-64.3944666667d0]
startable.pmra=[19.8770,0.0100,-0.3480]
startable.pmdec=[-1.520,-0.220,1.000]
startable.px=[0.0070,0.0140,0.0000]
startable.rv=[-17.0,16.0,24.0]
;
get_geoparms
geoparms.latitude=45.0
geoparms.longitude=-75.0
geoparms.altitude=0.0
;
dates=['1996-04-30','1996-04-30','1996-11-30','1996-08-05']
times=dblarr(n_elements(startable))-60.
geoparms.tai_utc=30
geoparms.tdt_tai=30
geoparms.ut1utc_coeffs=0
;
for i=0,n_elements(dates)-1 do begin
	date=dates(i)
	topostar,times,stars,startable,ra_app,dec_app
	print,startable.name
	print,ra_app,format='(4(f12.9,2x))'
	print,dec_app,format='(4(f12.8,2x))'
endfor
;
end
;-------------------------------------------------------------------------------
pro topostar,times,stars,startable,ra_app,dec_app
;
; Calculate topocentric apparent star positions. Procedure calls C function 
; tpstar for calls to topostar in the NOVAS C-library of astrometry subroutines 
; (Kaplan, G.H. et al. 1989, AJ, 97 1197).  
; Input variables can be arrays to speed up processing.
;
; Format of date (string): YYYY-MM-DD
; times (double): seconds since 0 UTC.
; stars (string): star names as in startable.
;
; These are the units (type double):
; ra=Right Ascension in hours plus fraction thereof
; dec=Declination in degrees plus fractions thereof
; pmra=Centennial change in Ra in seconds of time due to proper motion
; pmdec=Centennial change in Dec in seconds of arc due to proper motion
; px=Parallax in arcseconds
; rv=Radial velocity in km/s
; 
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Check input
num_times=n_elements(times)
num_stars=n_elements(stars)
if num_times ne num_stars then begin
	print,'***Error(TOPOSTAR): times and stars have different lengths!'
	return
endif
;
; Allocate star data
starname=strarr(num_times)
starnumber=lonarr(num_times)
ra=dblarr(num_times)
ra_app=dblarr(num_times)
dec=dblarr(num_times)
dec_app=dblarr(num_times)
pmra=dblarr(num_times)
pmdec=dblarr(num_times)
px=dblarr(num_times)
rv=dblarr(num_times)
;
; Homogenize the star list
if n_elements(stars) gt 1 then rename_starids,stars
;
for i=0,n_elements(startable)-1 do begin
	index=where(stars eq startable(i).starid,count)
	if count gt 0 then begin
		starname(index)=startable(i).starid
		starnumber(index)=1
		ra(index)=startable(i).ra
		dec(index)=startable(i).dec
		pmra(index)=startable(i).pmra
		pmdec(index)=startable(i).pmdec
		px(index)=startable(i).px
		rv(index)=startable(i).rv
	endif
endfor
;
if total(starnumber) ne num_times then begin
	print,'Warning(TOPOSTAR): '+ $
		'star information could not be obtained for all scans!'
endif
;
; UTC is in seconds. Remember, CONSTRICTOR point times are in milliseconds,
; averaged scan times are in seconds since 0 UT!
UTC=times
parsedate,Date,y,m,d
tjd=(UTC+GeoParms.TAI_UTC+GeoParms.TDT_TAI)/86400+julian(y,m,d)
;
TDT_UT1=UTC+GeoParms.TAI_UTC+GeoParms.TDT_TAI-utc2ut1(UTC)
;
loc=dblarr(5)
loc(0)=GeoParms.Longitude
loc(1)=GeoParms.Latitude
loc(2)=GeoParms.Altitude
loc(3)=10.0	; Temperature [C]
loc(4)=1010.0	; Pressure [mBar]
;
status=linknload(!external_lib,'tpstar',num_times,tjd,TDT_UT1, $
		loc,starname,starnumber, $
		ra,dec,pmra,pmdec,px,rv,ra_app,dec_app)
if status ne 0 then print,'***Error(TOPOSTAR): tpstar returns code ',status,'!'
;
if num_times eq 1 then begin
	ra_app=ra_app(0)
	dec_app=dec_app(0)
endif
;
end
;-------------------------------------------------------------------------------
pro apparentstar,times,stars,startable,ra_app,dec_app
;
; Calculate apparent star positions. Procedure calls C function apstar for calls
; to apparentstar in the NOVAS C-library of astrometry subroutines 
; (Kaplan, G.H. et al. 1989, AJ, 97 1197).  
; Input variables can be arrays to speed up processing.
;
; Format of date (string): YYYY-MM-DD
; times (double): seconds since 0 UTC.
; stars (string): star names as in startable.
;
; These are the units (type double):
; ra=Right Ascension in hours plus fraction thereof
; dec=Declination in degrees plus fractions thereof
; pmra=Centennial change in Ra in seconds of time due to proper motion
; pmdec=Centennial change in Dec in seconds of arc due to proper motion
; px=Parallax in arcseconds
; rv=Radial velocity in km/s
; mv=Visual magnitude
; 
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Check input
num_times=n_elements(times)
num_stars=n_elements(stars)
if num_times ne num_stars then begin
	print,'***Error(APPARENTSTAR): times and stars have different lengths!'
	return
endif
;
; Allocate star data
starname=strarr(num_times)
starnumber=lonarr(num_times)
ra=dblarr(num_times)
ra_app=dblarr(num_times)
dec=dblarr(num_times)
dec_app=dblarr(num_times)
pmra=dblarr(num_times)
pmdec=dblarr(num_times)
px=dblarr(num_times)
rv=dblarr(num_times)
;
for i=0,n_elements(startable)-1 do begin
	index=where(stars eq startable(i).starid,count)
	if count gt 0 then begin
		starname(index)=startable(i).starid
		starnumber(index)=1
		ra(index)=startable(i).ra
		dec(index)=startable(i).dec
		pmra(index)=startable(i).pmra
		pmdec(index)=startable(i).pmdec
		px(index)=startable(i).px
		rv(index)=startable(i).rv
	endif
endfor
;
if total(starnumber) ne num_times then begin
	print,'Warning(APPARENTSTAR): '+ $
		'star information could not be obtained for all scans!'
endif
;
; UTC is in seconds. Remember, CONSTRICTOR point times are in milliseconds,
; averaged scan times are in seconds since 0 UT!
UTC=times
parsedate,Date,y,m,d
tjd=(UTC+GeoParms.TAI_UTC+GeoParms.TDT_TAI)/86400+julian(y,m,d)
;
status=linknload(!external_lib,'apstar',num_times,tjd, $
		starname,starnumber, $
		ra,dec,pmra,pmdec,px,rv,ra_app,dec_app)
if status ne 0 then print,'***Error(APPARENTSTAR): apstar returns code ',status
;
if num_times eq 1 then begin
	ra_app=ra_app(0)
	dec_app=dec_app(0)
endif
;
end
;-------------------------------------------------------------------------------
pro topoplanet,times,planets,ra_app,dec_app,distance
;
; Calculate topocentric apparent planet positions. Procedure calls C function 
; tpplanet for calls to topoplanet in the NOVAS C-library of astrometry 
; subroutines (Kaplan, G.H. et al. 1989, AJ, 97 1197).  
; Input variables can be arrays to speed up processing.
;
; Format of date (string): YYYY-MM-DD
; times (double): seconds since 0 UTC.
; planets (long): star names as in startable.
; Mercury: 1,..., Pluto: 9, Sun: 10, Moon:11
;
; These are the units (type double):
; ra=Right Ascension in hours plus fraction thereof
; dec=Declination in degrees plus fractions thereof
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Check input
num_times=n_elements(times)
num_planets=n_elements(planets)
if num_times ne num_planets then begin
	print,'***Error(TOPOPLANET): times and planets have different lengths!'
	return
endif
;
ra_app=dblarr(num_times)
dec_app=dblarr(num_times)
distance=dblarr(num_times)
;
times=double(times)
planets=long(planets)
;
; UTC is in seconds. Remember, CONSTRICTOR point times are in milliseconds,
; averaged scan times are in seconds since 0 UT!
UTC=times
parsedate,Date,y,m,d
tjd=(UTC+GeoParms.TAI_UTC+GeoParms.TDT_TAI)/86400+julian(y,m,d)
;
TDT_UT1=UTC+GeoParms.TAI_UTC+GeoParms.TDT_TAI-utc2ut1(UTC)
;
loc=dblarr(3)
loc(0)=GeoParms.Longitude
loc(1)=GeoParms.Latitude
loc(2)=GeoParms.Altitude
;
status=linknload(!external_lib,'tpplanet',num_times,planets,tjd,TDT_UT1, $
		loc,ra_app,dec_app,distance)
if status ne 0 then $
	print,'***Error(TOPOPLANET): tpplanet returns code ',status,'!'
;
if num_times eq 1 then begin
	ra_app=ra_app(0)
	dec_app=dec_app(0)
	distance=distance(0)
endif
;
end
;-------------------------------------------------------------------------------
pro apparentplanet,times,planets,ra_app,dec_app,distance
;
; Calculate apparent planet positions. Procedure calls C function 
; applanet for calls to appplanet in the NOVAS C-library of astrometry 
; subroutines (Kaplan, G.H. et al. 1989, AJ, 97 1197).  
; Input variables can be arrays to speed up processing.
;
; Format of date (string): YYYY-MM-DD
; times (double): seconds since 0 UTC.
; planets (long): star names as in startable.
;
; These are the units (type double):
; ra=Right Ascension in hours plus fraction thereof
; dec=Declination in degrees plus fractions thereof
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Check input
num_times=n_elements(times)
num_planets=n_elements(planets)
if num_times ne num_planets then begin
	print,'***Error(APPPLANET): times and planets have different lengths!'
	return
endif
;
ra_app=dblarr(num_times)
dec_app=dblarr(num_times)
distance=dblarr(num_times)
;
; UTC is in seconds. Remember, CONSTRICTOR point times are in milliseconds,
; averaged scan times are in seconds since 0 UT!
UTC=times
parsedate,Date,y,m,d
tjd=(UTC+GeoParms.TAI_UTC+GeoParms.TDT_TAI)/86400+julian(y,m,d)
;
status=linknload(!external_lib,'applanet',num_times,planets,tjd, $
		ra_app,dec_app,distance)
if status ne 0 then $
	print,'***Error(APPPLANET): applanet returns code ',status,'!'
;
if num_times eq 1 then begin
	ra_app=ra_app(0)
	dec_app=dec_app(0)
	distance=distance(0)
endif
;
end
;-------------------------------------------------------------------------------
pro calcgeo
;
; Compute GeoDelays for point data.
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if checkdata([3,11]) ne 0 then return
;
num_times=n_elements(PointTime)
;
; Compute apparent star positions
times=abs(PointTime)
stars=strarr(num_times)
for i=0,n_elements(StarId)-1 do stars(Rec0(i):RecN(i))=StarId(i)
topostar,times,stars,startable,ra_app,dec_app
;
; Compute uv-coverage
uvw=dblarr(n_elements(vissq(*,0,0,0)), $
	   n_elements(vissq(0,*,0,0)), $
	   n_elements(vissq(0,0,*,0)),3,n_elements(times))
uvw=calcuv(hourangle(ut12gst(times,utc2ut1(times)),ra_app),dec_app,times,uvw)
jndex=where(stars eq 'FKV0000',count)
if count gt 0 then uvw(*,*,*,*,jndex)=0
GeoDelay=stationdelay(uvw)
;
; Reference to reference station
ibr=GenConfig.RefStation-1
for i=0,GenConfig.NumSid-1 do begin
      if i ne ibr then $
      GeoDelay(i,*)= $
      GeoDelay(i,*)-GeoDelay(ibr,*)
endfor
GeoDelay(ibr,*)=0
;
end
;-------------------------------------------------------------------------------
pro calcastrom,compute=compute,skipuv=skipuv
;
; Computes astrometric quantities for all averaged scans.
;
common LocalCalcAstrom,compute_app
common Tables,scantable,bgtable,stationtable
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([3,9]) ne 0 then return
if n_elements(skipuv) eq 0 then skipuv=0
if total(genconfig.stationcoord) eq 0 then skipuv=1
;
; Compute the astrometry also for flagged data!
index=where(abs(scans.time) ge 0,count)
if count eq 0 then begin
	print,'***Error(CALCASTROM): no valid scan times found!'
	return
endif
;
; If not compute, use the apparent positions as stored in scans
if n_elements(compute_app) eq 0 then compute_app=1
if n_elements(compute) eq 0 then compute=compute_app
compute_app=compute
;
; Compute apparent star positions
times=abs(scans(index).time)
stars=scans(index).StarId
if compute then begin
	topostar,times,stars,StarTable,ra_app,dec_app
endif else begin
	ra_app=scans.ra
	dec_app=scans.dec
endelse
;
; Compute UT1
ut1=utc2ut1(times)
;
; Compute Greenwich apparent sidereal time
gst=ut12gst(times,ut1)
;
; Compute hour angle
scans(index).ha=hourangle(gst,ra_app)
;
; Store Declination
scans(index).dec=dec_app
;
; Compute and store azimuth of star
hadec=transpose([[scans(index).ha],[scans(index).dec]])
azel=horizon2azel(equatorial2horizon(hadec2equatorial(hadec)))
scans(index).az=reform(azel(0,*))
;
; Store Right Ascension
scans(index).ra=ra_app
;
; Compute zenith angle
scans(index).za=zenithangle(scans(index).ha,dec_app)
;
; Compute mirror angle
scans(index).ma=mirrorangle(scans(index).ha,dec_app)
;
if skipuv then return
;
; Compute uv-coverage
uvw=dblarr(n_elements(scans(0).vissq(*,0,0)), $
	   n_elements(scans(0).vissq(0,*,0)), $
	   n_elements(scans(0).vissq(0,0,*)),3,n_elements(times))
scans(index).uvw=calcuv(scans(index).ha,dec_app,times,uvw)
;
; Treat white light source separately
jndex=where(scans.StarId eq 'FKV0000',count)
if count gt 0 then begin
	scans(jndex).uvw=0
	scans(jndex).ha=5	; nominal value
	scans(jndex).dec=12	; nominal value
endif
;
; Compute astrometric delay. Note that the delay offset is included!
if system_id(SystemId) eq 'NPOI' or system_id(SystemId) eq 'VLTI' then begin
scans(index).GeoDelay(0:GenConfig.NumSid-1)=stationdelay(scans(index).UVW)
j=GenConfig.NumSid
for i=0,GenConfig.NumSid-1 do if total(scans(index).GeoDelay(i)) eq 0 then j=i
if j eq GenConfig.NumSid then begin
	print,'***Error(CALCASTROM): problem with geometric delays!'
	return
endif
endif
;
; Reference to reference station
ibr=GenConfig.RefStation-1
for i=0,GenConfig.NumSid-1 do begin
      if i ne ibr then $
      scans(index).GeoDelay(i)= $
      scans(index).GeoDelay(i)-scans(index).GeoDelay(ibr)
endfor
scans(index).GeoDelay(ibr)=0
;
print,'Finished astrometry computations.'
;
end
;-------------------------------------------------------------------------------
pro solveastrom
;
; Solve for new station positions and/or star positions. Input data are
; station based delays. Because of that, the solution for any station is
; independent from the others. The delays for the reference station are
; zero, so that the corrections for this station will also be zero. 
; For coefficients, check with Hummel et al. 1994, AJ 108, 326. Coefficient
; for the delay offset is negative.
;
; The coordinates in GenConfig are updated, as well as those in the
; stationtable. However, the constant term in the stationtable, corresponding
; to a path to a common reference in the lab, is incrementally updated
; instead of copying the value from GenConfig.
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([8,9]) ne 0 then return
;
; Select all scans with valid time stamps
index=where(scans.time gt 0 and scans.starid ne 'FKV0000',count)
index=where(scans.time gt 0,count)
if count eq 0 then begin
	print,'***Error(SOLVEASTROM): no valid stellar scans found!'
	return
endif
;
rad=180/pi_circle
ha=scans(index).ha*(15/rad)
dec=scans(index).dec/rad
starid=scans(index).starid
geodelay=scans(index).GeoDelay
;
; Select the observed delays
case fit_data of
	'FDL':begin
	      obsdelay=scans(index).FDLPos
	      obsdelay(GenConfig.RefStation-1,*)=0
	      obsdelayerr=scans(index).FDLPosErr
	      end
	'GRP':begin
	      obsdelay=scans(index).GrpDelay
	      obsdelayerr=scans(index).GrpDelayErr
	      end
	'DRY':begin
	      obsdelay=scans(index).DryDelay
	      obsdelayerr=scans(index).DryDelayErr
	      end
	'WET':begin
	      obsdelay=scans(index).WetDelay
	      obsdelayerr=scans(index).WetDelayErr
	      end
endcase
;
; Make sure data with a bad reference are not used. Change reference if needed.
for i=0,GenConfig.NumSid-1 do $
obsdelayerr(i,*)=abs(obsdelayerr(i,*)) $
	     *signof(obsdelayerr(i,*),obsdelayerr(GenConfig.RefStation-1,*))
;
; Check station and star selection
nstn=n_elements(fit_stations)
nstr=n_elements(fit_stars)
if nstn gt 0 then if fit_stations(0) eq '' then nstn=0
if nstr gt 0 then if fit_stars(0) eq '' then nstr=0
;
; Remove the white light source from list
if nstr gt 0 then begin
	index= $
	where(fit_stars ne 'FKV0000',nstr)
	if nstr gt 0 then fit_stars0=fit_stars(index)
endif
;
; Remove reference station from list, if included
if nstn gt 0 then begin
	index= $
	where(fit_stations ne GenConfig.StationId(GenConfig.RefStation-1),nstn)
	if nstn gt 0 then fit_stations0=fit_stations(index)
endif
if nstn+nstr eq 0 then begin
	print,'***Error(SOLVEASTROM): no fits selected!'
	return
endif
;
scan_index=indgen(n_elements(starid)) 
if nstr gt 0 then begin
	id=0
	for i=0,nstr-1 do begin
		index=where(starid eq fit_stars0(i),count)
		if count gt 0 then scan_index(id:id+count-1)=index
		id=id+count
	endfor
	if id gt 0 then scan_index=scan_index(0:id-1) else begin
		print,'***Error(SOLVEASTROM): no data for selected stars!'
		return
	endelse
endif
ch=cos(ha(scan_index))
sh=sin(ha(scan_index))
cd=cos(dec(scan_index))
sd=sin(dec(scan_index))
starid=starid(scan_index)
index=where(starid eq 'FKV0000',count)
if count gt 0 then begin
	ch(index)=0
	cd(index)=0
	sh(index)=0
	sd(index)=0
endif
geodelay=geodelay(*,scan_index)
obsdelay=obsdelay(*,scan_index)
obsdelayerr=obsdelayerr(*,scan_index)
;
; Determine maximum number of rows
index=where(obsdelayerr gt 0,nrow)
;
; Add one row for removal of RA degeneracy
if nstr gt 0 then nrow=nrow+1
;
; Determine number of columns
ncol=nstn*4+nstr*2
;
; Allocate arrays
m=make_array(nrow,ncol,/double)
r=make_array(nrow,/double)
s=make_array(nrow,/string)
;
; Set up design matrix
id=0
for i=0,GenConfig.NumSid-1 do begin
    index_e=where(obsdelayerr(i,*) gt 0,count_e)
    if count_e gt 0 and i ne GenConfig.RefStation-1 then begin
    wt=1/obsdelayerr(i,index_e)
    r(id:id+count_e-1)=(obsdelay(i,index_e)-geodelay(i,index_e))*wt
    s(id:id+count_e-1)=starid(index_e)
    count=0
    if nstn gt 0 then index=where(fit_stations0 eq GenConfig.StationId(i),count)
    if count ne 0 then begin
		m(id:id+count_e-1,index*4+0)=+cd(index_e)*ch(index_e)*wt; X
		m(id:id+count_e-1,index*4+1)=-cd(index_e)*sh(index_e)*wt; Y
		m(id:id+count_e-1,index*4+2)=+sd(index_e)*wt		; Z
		m(id:id+count_e-1,index*4+3)=-wt			; D
    endif
;
;   Transform horizon station coordinates to equatorial
    StationCoord= $
	horizon2equatorial(GenConfig.StationCoord(*,i) $
			  -GenConfig.StationCoord(*,GenConfig.RefStation-1))
    for j=0,nstr-1 do begin
	index=where(obsdelayerr(i,index_e) gt 0 $
		and starid(index_e) eq fit_stars0(j),count)
	if count gt 0 then begin
;		Right ascension
		m(id+index,nstn*4+0+j*2)=wt(index)*( $
		   +StationCoord(1)*cd(index_e(index))*ch(index_e(index)) $
		   +StationCoord(0)*cd(index_e(index))*sh(index_e(index)))
;		Declination
		m(id+index,nstn*4+1+j*2)=wt(index)*( $
		   -StationCoord(0)*sd(index_e(index))*ch(index_e(index)) $
		   +StationCoord(1)*sd(index_e(index))*sh(index_e(index)) $
		   +StationCoord(2)*cd(index_e(index)))
	endif
    endfor
    id=id+count_e
    endif
endfor
;
; Fix the right ascension degeneracy if both stations and stars are fitted
; The large weighting factor forces the solution.
index=where(obsdelayerr gt 0)
wscale=1/median(obsdelayerr(index))
if nstr gt 0 and nstn gt 0 then $
	m(nrow-1,nstn*4)=wscale $
		*max(abs(genconfig.stationcoord(0:3,0:genconfig.numsid-1)))
;
; Remove zero rows in the design matrix
index=where(avg(abs(m),1) ne 0,count)
if count gt 0 then begin
	m=m(index,*)
	r=r(index)
	s=s(index)
endif
;
; Remove data from white light source, or, if only this one, solve for C only
stars=unique(s)
if n_elements(stars) eq 1 and stars(0) eq 'FKV0000' then begin
	do_c_only=1
	nstr=0
	m=m(*,indgen(nstn)*4+3)
endif else begin
	do_c_only=0
;	index=where(s ne 'FKV0000')
;	Do not remove the data, it was conditioned to constrain the const.term
;	m=m(index,*)
;	r=r(index)
;	s=s(index)
endelse
;
; Enough data?
nrow=n_elements(m(*,0))
ncol=n_elements(m(0,*))
if nrow lt ncol then begin
	print,'***Error(SOLVEASTROM): not enough data!'
	return
endif
;
t=transpose(m)
n=t#m
y=t#r
;
if n_elements(n) eq 1 then begin
	s=y/n
endif else begin
	svd,n,w,u,v		; Singular value decomposition
; 	print,'Eigenvalues (normalized): ',w/max(w)
	small=where(w lt max(w)*1.0e-8,count)
	if count gt 0 then begin
;	print,'SVD: will edit',count,' singular values!'
		print,'***Error(SOLVEASTROM): singular matrix!'
		return
		w(small)=0
	endif
;	svbksb,u,w,v,y,s	; SVD solution
;
	in=invert(n,status)
	s=in#y			; Direct solution, is more precise
endelse
;
print,'Solution computed.'
;
; Update the station coordinates, also copy the ref. station values
if n_elements(stationtable) eq 0 then get_stationtable,update=0
for i=0,nstn do begin
	if i eq nstn then j=GenConfig.RefStation-1 $
		     else j=long(where(GenConfig.StationId $
				eq fit_stations0(i)),0)
	ct0=GenConfig.StationCoord(3,j)
	if do_c_only then begin
	GenConfig.StationCoord(3,j)=GenConfig.StationCoord(3,j) $
				   +s(i) 
	endif else if i lt nstn then begin
	GenConfig.StationCoord(*,j)=GenConfig.StationCoord(*,j) $
				   +equatorial2horizon(s(i*4:(i+1)*4-1))
	endif
	ct1=GenConfig.StationCoord(3,j)
;	Update stations in stationtable too
	if i eq nstn $
	then k=long(where(stationtable.stationid $
			eq genconfig.stationid(genconfig.refstation-1))) $
	else k=long(where(stationtable.stationid eq fit_stations0(i)),0)
	stationtable(k).x=genconfig.stationcoord(0,j)
	stationtable(k).y=genconfig.stationcoord(1,j)
	stationtable(k).z=genconfig.stationcoord(2,j)
	stationtable(k).d=stationtable(k).d+(ct1-ct0)
endfor
;
; For single configuration fit, store config info
for i=0,GenConfig.NumSid-1 do begin
	j=where(stationtable.stationid eq genconfig.stationid(i)) & j=j(0)
	stationtable(j).DL_ID=genconfig.delaylineid(i)
	stationtable(j).BC_IN=genconfig.bcinputid(i)
endfor
if nstn gt 0 then print,'GenConfig.StationCoord updated.'
;
; Update the star positions
for i=0,nstr-1 do begin
	j=long(where(StarTable.starid eq fit_stars0(i)),0)
	StarTable(j).ra  =StarTable(j).ra +s(nstn*4+0+i*2)*RAD/15
	StarTable(j).rae =sqrt(in(nstn*4+0+i*2,nstn*4+0+i*2))*RAD/15
	StarTable(j).dec =StarTable(j).dec+s(nstn*4+1+i*2)*RAD
	StarTable(j).dece=sqrt(in(nstn*4+1+i*2,nstn*4+1+i*2))*RAD
endfor
if nstr gt 0 then print,'StarTable updated.'
;
; Update the astrometry
calcastrom	; for scandata
calcgeo		; for pointdata
;
end
;-------------------------------------------------------------------------------
pro referencestation,station,class
;
; Change the reference station.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if checkdata([8]) ne 0 then return
;
if n_elements(class) eq 0 then class='all'
;
ibr=where(GenConfig.StationId eq station,count) & ibr=ibr(0)
if count eq 0 then begin
	print,'***Error(REFERENCESTATION): invalid station!'
	return
endif
obr=GenConfig.RefStation-1
GenConfig.RefStation=ibr+1
;
; Convert "pos" data to absolute
if (class eq 'scan' or class eq 'all') and n_elements(scans) ne 0 then begin
	for i=0,GenConfig.NumSid-1 do begin
		if i ne obr then $
		scans(*).FDLPos(i)=scans(*).FDLPos(i)+scans(*).FDLPos(obr)
	endfor
endif
if (class eq 'point' or class eq 'all') and n_elements(Rec0) ne 0 then begin
	for i=0,GenConfig.NumSid-1 do begin
		if i ne obr then begin
		FDLPos(i,*)=FDLPos(i,*)+FDLPos(obr,*)
		if n_elements(MetroPos) gt 0 then $
		MetroPos(i,*)=MetroPos(i,*)+MetroPos(obr,*)
		endif
	endfor
endif
if (class eq 'raw' or class eq 'all') and n_elements(Raw0) ne 0 then begin
	for i=0,GenConfig.NumSid-1 do begin
		if i ne obr then begin
		LaserPos(i,*)=LaserPos(i,*)+LaserPos(obr,*)
		GeoPos(i,*)=GeoPos(i,*)+GeoPos(obr,*)
		endif
	endfor
endif
;
; Reference to new reference station
if (class eq 'scan' or class eq 'all') and n_elements(scans) ne 0 then begin
	for i=0,GenConfig.NumSid-1 do begin
		if i ne ibr then begin
		scans(*).FDLPos(i)=scans(*).FDLPos(i)-scans(*).FDLPos(ibr)
		scans(*).GeoDelay(i)=scans(*).GeoDelay(i)-scans(*).GeoDelay(ibr)
		scans(*).GrpDelay(i)=scans(*).GrpDelay(i)-scans(*).GrpDelay(ibr)
		scans(*).DryDelay(i)=scans(*).DryDelay(i)-scans(*).DryDelay(ibr)
		scans(*).WetDelay(i)=scans(*).WetDelay(i)-scans(*).WetDelay(ibr)
		scans(*).MetroDelay(i)=scans(*).MetroDelay(i)-scans(*).MetroDelay(ibr)
		scans(*).WhiteDelay(i)=scans(*).WhiteDelay(i)-scans(*).WhiteDelay(ibr)
		endif
	endfor
	scans(*).GeoDelay(ibr)=0
	scans(*).GrpDelay(ibr)=0
	scans(*).DryDelay(ibr)=0
	scans(*).WetDelay(ibr)=0
	scans(*).MetroDelay(ibr)=0
	scans(*).WhiteDelay(ibr)=0
endif
if (class eq 'point' or class eq 'all') and n_elements(Rec0) ne 0 then begin
	for i=0,GenConfig.NumSid-1 do begin
		if i ne ibr then begin
		FDLPos(i,*)=FDLPos(i,*)-FDLPos(ibr,*)
		if n_elements(MetroPos) gt 0 then $
		MetroPos(i,*)=MetroPos(i,*)-MetroPos(ibr,*)
		if n_elements(GrpDelay) gt 0 then begin
		GrpDelay(i,*)=GrpDelay(i,*)-GrpDelay(ibr,*)
		DryDelay(i,*)=DryDelay(i,*)-DryDelay(ibr,*)
		WetDelay(i,*)=WetDelay(i,*)-WetDelay(ibr,*)
		endif
		endif
	endfor
	if n_elements(GrpDelay) gt 0 then begin
		GrpDelay(ibr,*)=0
		DryDelay(ibr,*)=0
		WetDelay(ibr,*)=0
	endif
endif
if (class eq 'raw' or class eq 'all') and n_elements(Raw0) ne 0 then begin
	for i=0,GenConfig.NumSid-1 do begin
		if i ne ibr then begin
			LaserPos(i,*)=LaserPos(i,*)-LaserPos(ibr,*)
			GeoPos(i,*)=GeoPos(i,*)-GeoPos(ibr,*)
		endif
	endfor
endif
;
print,'Reference station set.'
;
end
;-------------------------------------------------------------------------------
pro initvolvox,command
;
common StarBase,StarTable,Notes
;
table=startable
rename_starids,'fkv-bsc' & table.bsc=startable.bsc
rename_starids,'bsc-hip' & table.hic=startable.hic
read_catalogs
;
case command of
	'pm':	begin
		table.pmra=startable.pmra
		table.pmdec=startable.pmdec
		end
	'pos':	begin
		table.ra=startable.ra
		table.dec=startable.dec
		end
	'hip':	begin
		table.pmra=startable.pmra
		table.pmdec=startable.pmdec
		table.ra=startable.ra
		table.dec=startable.dec
		end
endcase
;
startable=table
;
end
;-------------------------------------------------------------------------------
pro calcvolvox,stationtable,skipuv=skipuv
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(skipuv) eq 0 then skipuv=0
;
for i=0,n_elements(geninfo.date)-1 do begin
        loadnight,geninfo(i).date,geoinfo(i).systemid,geninfo(i).configid
	if n_elements(stationtable) ne 0 then begin
		if strlen(stationtable) eq 0 then get_stationtable $
					     else get_stationtable,stationtable
	endif
	calcastrom,skipuv=skipuv
	storenight,11
endfor
;
end
;-------------------------------------------------------------------------------
pro solvevolvox
;
; Solve global astrometry using sparse matrix Cholesky decomposition.
; To read about sparse matrix technology, see book with same title by
; Sergio Pissanetsky, 1984, Academic Press Inc., London
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([12,13]) ne 0 then return
;
; Check station and star selection
nstn=n_elements(fit_stations)
nstr=n_elements(fit_stars)
if nstn gt 0 then if fit_stations(0) eq '' then nstn=0
if nstr gt 0 then if fit_stars(0) eq '' then nstr=0
;
; Sanity check
if nstr+nstn eq 0 then begin
	print,'***Error(SOLVEVOLVOX): no fits were selected!'
	return
endif
;
RAD=180/pi_circle
TOL=1e-8
;
; Set nominal dimensions and allocate sparse matrix arrays
nast=2L				; parms/star
nbnd=nstr*nast			; width of border band
nstp=4				; station coordinate parameters
narc=nstp*nstn			; arc parms/night
nnit=n_elements(fit_nights)	; number of nights
nrow=nnit*narc+nbnd		; Full dimension of square matrix
;
numr=nrow-nbnd
numv=nnit*((narc*(narc+1))/2+narc*nbnd)
numt=(nbnd*(nbnd+1))/2
vd=dblarr(numv+numt)		; row-wise sparse part of matrix
jv=uintarr(numv+numt)		; column numbers in vd, beginning with 1
iv=ulonarr(nrow+1)		; where row i begins in vd, beginning with 1
ni=uintarr(nnit)		; dimensions of sub-matrices
rd=dblarr(nrow)			; right hand side
;
; Number of observations per star
if nstr gt 0 then sc=intarr(nstr)
;
ns=0U				; sub-matrix counter
r0=0L				; rd counter
n0=0L				; vd counter
;
nsolved=intarr(nnit)		; was this night included? (1=Yes)
nzcount=intarr(nnit)		; number of non-zero border band columns
;
FOR night=0,nnit-1 DO BEGIN
;
print,'Now processing ',fit_nights(night),'...'
;
loadnight,fit_nights(night)
calcastrom
;
; Select all scans with valid time stamps
index=where(scans.time gt 0 and scans.starid ne 'FKV0000',count)
if count eq 0 then begin
	print,'Warning(SOLVEVOLVOX): no valid data on ',fit_nights(night),'!'
	goto,SKIP
endif
ha=scans(index).ha*(15/rad)
dec=scans(index).dec/rad
starid=scans(index).starid
geodelay=scans(index).GeoDelay
case fit_data of
	'FDL':begin
	      obsdelay=scans(index).FDLPos
	      obsdelay(GenConfig.RefStation-1,*)=0
	      obsdelayerr=scans(index).FDLPosErr
	      end
	'GRP':begin
	      obsdelay=scans(index).GrpDelay
	      obsdelayerr=scans(index).GrpDelayErr
	      end
	'DRY':begin
	      obsdelay=scans(index).DryDelay
	      obsdelayerr=scans(index).DryDelayErr
	      end
	'WET':begin
	      obsdelay=scans(index).WetDelay
	      obsdelayerr=scans(index).WetDelayErr
	      end
endcase
;
; Determine which fit stations are to be included
nstn0=0
if nstn gt 0 then begin
	stn_index=intarr(nstn)
	for i=0,GenConfig.NumSid-1 do begin
		if i ne GenConfig.RefStation-1 then begin
		   index=where(fit_stations eq GenConfig.StationId(i),count)
		   if count gt 0 then stn_index(index)=1
		endif
	endfor
	index=where(stn_index eq 1,nstn0)
	if nstn0 gt 0 then fit_stations0=fit_stations(index)
endif
;
; Extract data for stars if selected
scan_index=indgen(n_elements(starid)) 
if nstr gt 0 then begin
	id=0
	for i=0,nstr-1 do begin
		index=where(starid eq fit_stars(i),count)
		if count gt 0 then scan_index(id:id+count-1)=index
		id=id+count
	endfor
	if id gt 0 then scan_index=scan_index(0:id-1) else begin
		print,'Warning(SOLVEVOLVOX): no data for stars on ', $
				fit_nights(night),'!'
		goto,SKIP
	endelse
endif
ch=cos(ha(scan_index))
sh=sin(ha(scan_index))
cd=cos(dec(scan_index))
sd=sin(dec(scan_index))
starid=starid(scan_index)
geodelay=geodelay(*,scan_index)
obsdelay=obsdelay(*,scan_index)
obsdelayerr=obsdelayerr(*,scan_index)
;
; Determine maximum number of rows
index=where(obsdelayerr gt 0,nr)
;
; Determine number of columns
nc=nstn0*nstp+nstr*nast
;
; Allocate design matrix for this night
m=make_array(nr,nc,/double)
r=make_array(nr,/double)
;
; Set up design matrix
id=0
for i=0,GenConfig.NumSid-1 do begin
    index_e=where(obsdelayerr(i,*) gt 0,count_e)
    if count_e gt 0 and i ne GenConfig.RefStation-1 then begin
    wt=1/obsdelayerr(i,index_e)
    r(id:id+count_e-1)=(obsdelay(i,index_e)-geodelay(i,index_e))*wt
    count=0
    if nstn0 gt 0 then index=where(fit_stations0 eq GenConfig.StationId(i),count)
    if count ne 0 then begin
		m(id:id+count_e-1,index*4+0)=+cd(index_e)*ch(index_e)*wt; X
		m(id:id+count_e-1,index*4+1)=-cd(index_e)*sh(index_e)*wt; Y
		m(id:id+count_e-1,index*4+2)=+sd(index_e)*wt		; Z
		m(id:id+count_e-1,index*4+3)=-wt			; D
    endif
;
;   Transform horizon station coordinates to equatorial
    StationCoord= $
	horizon2equatorial(GenConfig.StationCoord(*,i) $
			  -GenConfig.StationCoord(*,GenConfig.RefStation-1))
    for j=0,nstr-1 do begin
	if j eq 0 then sc1=sc*0
	index=where(obsdelayerr(i,index_e) gt 0 $
		and starid(index_e) eq fit_stars(j),count)
	if count gt 0 then begin
		sc1(j)=count
;		Right ascension
		m(id+index,nstn0*nstp+0+j*nast)=wt(index)*( $
		   +StationCoord(1)*cd(index_e(index))*ch(index_e(index)) $
		   +StationCoord(0)*cd(index_e(index))*sh(index_e(index)))
;		Declination
		m(id+index,nstn0*nstp+1+j*nast)=wt(index)*( $
		   -StationCoord(0)*sd(index_e(index))*ch(index_e(index)) $
		   +StationCoord(1)*sd(index_e(index))*sh(index_e(index)) $
		   +StationCoord(2)*cd(index_e(index)))
	endif
    endfor
    id=id+count_e
    endif
endfor
;
; Remove zero rows in the design matrix
index=where(total(abs(m),2) ne 0)
m=m(index,*)
r=r(index)
;
; Remove zero columns in border band
if nstr gt 0 then begin
	nzindex=where(total(abs(m(*,nstn0*nstp:nc-1)),1) ne 0,nzc)
	index=nstn0*nstp+nzindex
	if nstn0 gt 0 then index=[indgen(nstn0*nstp),index]
	m=m(*,index)
	nc=nstn0*nstp+nzc
endif
;
; Normalized matrix
t=transpose(m)
n=t#m
y=t#r
;
; Check singularity of station coordinate solution
if nstn0 gt 0 then begin
	svd,n(0:nstn0*nstp-1,0:nstn0*nstp-1),w
	if min(w)/max(w) lt 1e-4 then goto,SKIP 
endif
;
; Check singularity of full solution
svd,n,w
w=w(sort(w))
if w(1)/w(n_elements(w)-1) lt 1e-6 then goto,SKIP
;
; This night is now part of the solution
nsolved(night)=1
ns=ns+1
ni(ns-1)=nstn0*nstp
;
; Fill in right hand side
if nstn0 gt 0 then begin
	rd(r0:r0+ni(ns-1)-1)=y(0:ni(ns-1)-1)
	r0=r0+ni(ns-1)
endif
if nstr gt 0 then begin
	nzcount(ns-1)=nzc
	sc=sc+sc1
	rd(numr+nzindex)=rd(numr+nzindex)+y(ni(ns-1):nc-1)
endif
;
; Shift border columns
if nstr gt 0 then begin
	n1=0
	for l=0,ns-2 do begin
		for k=0,ni(l)-1 do begin
			n2=iv(n1+k)+ni(l)-k-1
			jv(n2:n2+nzcount(l)-1)=jv(n2:n2+nzcount(l)-1)+ni(ns-1)
		endfor
		n1=n1+ni(l)
	endfor
endif
;
; Fill in sparse border
n1=ni(ns-1)+nzcount(ns-1)
if ns gt 1 then n3=fix(total(ni(0:ns-2))) else n3=0
nl=-1
for k=0,ni(ns-1)-1 do begin
	iv(n3+k)=n0+1
	vd(n0:n0+n1-1)=n(k,k:k+n1-1)
	jvc=indgen(ni(ns-1)-k)+k
	if nstr gt 0 then jvc=[jvc,ni(ns-1)+nzindex]
	jv(n0:n0+n1-1)=jvc+1+n3
	nl=n0+n1-1
	n0=n0+n1
	n1=n1-1
endfor
;
; Fill corner matrix
for k=0,nzcount(ns-1)-1 do begin
	index=nzindex(k:nzcount(ns-1)-1)-nzindex(k) $
	     +nzindex(k)*nbnd-(nzindex(k)*(nzindex(k)-1))/2
	vd(numv+index)=vd(numv+index)+n(ni(ns-1)+k:nc-1,ni(ns-1)+k)
endfor
;
SKIP:
;
ENDFOR
;
; Remove degeneracies due to RA zero point and missing stars
print,'Removing known degeneracies...'
if nstr gt 0 then begin
	minobs=nast
	index=where(sc lt minobs,icount)
	jndex=where(sc ge minobs,jcount)
	cd=dblarr(nbnd)
	if jcount gt 0 then cd(jndex(0)*nast)=1
	if icount gt 0 then begin
		cd(index*nast)=1
		cd(index*nast+1)=1
	endif
	wscale=max(abs(vd(where(vd ne 0))))
	k=lindgen(nbnd)
	vd(numv+k*nbnd-(k*(k-1))/2)=vd(numv+k*nbnd-(k*(k-1))/2)+cd*wscale
endif
;
; Finish sparse matrix setup
print,'Finishing sparse matrix...'
ni=ni(0:ns-1)
index=byte(rd)*0+1 & if numr gt r0 then index(r0:numr-1)=0
rd=rd(where(index eq 1))
nm=uint(n_elements(rd))
;
index=byte(vd)*0+1 & if numv gt nl+1 then index(nl+1:numv-1)=0
vd=vd(where(index eq 1))
;
n1=nbnd
n3=fix(total(ni))
n0=nl+1
for k=0,nbnd-1 do begin
	iv(n3+k)=n0+1
	jv(n0:n0+n1-1)=indgen(n1)+1+k+n3
	nl=n0+n1-1
	n0=n0+n1
	n1=n1-1
endfor
iv=[iv(0:nm-1),iv(nm-1)+1]
jv=jv(0:nl)
;
; Display TV image with sparse matrix
print,'Displaying sparse matrix...'
numpix=min([600,nm])
window,/free,xsize=numpix,ysize=numpix
image=bytarr(numpix,numpix)
for k=0,nm-1 do begin
        i1=iv(k)-1
        i2=iv(k+1)-2
	j1=nint(float(jv(i1:i2)-1)/nm*numpix)
	j2=nint((nm-1-(fltarr(i2-i1+1)+k))/nm*numpix)
        image(j1,j2)=1
endfor
if 0 then begin
for k=0,nbnd-1 do begin
	j1=nint((nm-findgen(nbnd-k)-1)/nm*numpix)
	j2=nint((nbnd-fltarr(nbnd-k)-1-k)/nm*numpix)
	image(j1,j2)=1
endfor
endif
tvscl,image
;
; Solve matrix using Hyper-Cholesky
if nstr eq 0 then ns=ns-1
if nstn eq 0 then ns=0
print,'Computing solution..., nm= ',nm,', ns= ',ns
vt=vd
hichol,vt,iv,jv,ni,ns,nm
sl=dblarr(nm)
rn=-rd
hisolve,vt,iv,jv,ni,ns,nm,rn,sl
;
; Improve solution by iteration
for iter=0,1 do begin
	hires,vd,iv,jv,rd,rn,sl,nm
	hisolve,vt,iv,jv,ni,ns,nm,rn,sl
endfor
;
; Compute smallest eigenvalue by inverse iteration
print,'Computing eigenvalues...'
rd(*)=1
s=dblarr(nm)
sum=0.d0
tiny=1d-3
i=0
niter=30
repeat begin
	ev=sum
	s(*)=0
	hisolve,vt,iv,jv,ni,ns,nm,rd,s
	sum=sqrt(total(s^2))
	rd(*)=s/sum
	i=i+1
endrep until abs(1-ev/sum) lt tiny or i eq niter
sev=1/sum
;
; Compute largest eigenvalue using the power method
u=dblarr(nm)+1
sum=0.d0
j=0
repeat begin
	ev=sum
	hires,vd,iv,jv,rd*0,rd,u,nm
	sum=sqrt(total(double(rd)^2))
	u(*)=rd/sum
	j=j+1
endrep until abs(1-ev/sum) lt tiny or j eq niter
lev=sum
;
; Check matrix condition
if i eq niter or j eq niter then begin
	print,'***Error(SOLVEVOLVOX): did not converge; return w/out app. sol.'
	return
endif
print,'Ratio largest/smallest eigenvalue= ',lev/sev
if sev/lev lt TOL then begin
	print,'***Error(SOLVEVOLVOX): matrix singular; return w/out app. sol.!'
	return
endif
;
; Calculate diagonal elements of inverse stellar parameter matrix
print,'Computing errors...'
if nstr gt 0 then begin
	se=fltarr(nbnd)
	for i=nm-nbnd,nm-1 do begin
		rd(*)=0 & rd(i)=1
		s(*)=0
		hisolve,vt,iv,jv,ni,ns,nm,rd,s
		se(i-nm+nbnd)=sqrt(rd(i))
	endfor
endif
;
; Apply solutions
;
; Update the star positions
for i=0,nstr-1 do begin
	j=long(where(StarTable.starid eq fit_stars(i)),0)
	StarTable(j).ra  =StarTable(j).ra +sl (nm-nstr*2+0+i*nast)*RAD/15
	StarTable(j).rae =se(i*nast+0)*RAD/15
	StarTable(j).dec =StarTable(j).dec+sl (nm-nstr*2+1+i*nast)*RAD
	StarTable(j).dece=se(i*nast+1)*RAD
endfor
if nstr gt 0 then print,'StarTable updated.'
;
ns=0
;
FOR night=0,nnit-1 DO BEGIN
;
loadnight,fit_nights(night)
;
if nsolved(night) then begin
;
; Update the station coordinates
if nstn gt 0 then begin
	stn_index=intarr(nstn)
	for i=0,GenConfig.NumSid-1 do begin
		if i ne GenConfig.RefStation-1 then begin
		index=where(fit_stations eq GenConfig.StationId(i),count)
		if count gt 0 then stn_index(index)=1
		endif
	endfor
	index=where(stn_index eq 1,nstn0)
	if nstn0 gt 0 then fit_stations0=fit_stations(index)
endif
for i=0,nstn0-1 do begin
	j=long(where(GenConfig.StationId eq fit_stations0(i)),0)
	GenConfig.StationCoord(*,j)=GenConfig.StationCoord(*,j) $
			   +equatorial2horizon(sl(ns+i*nstp:ns+(i+1)*nstp-1))
endfor
j=where(geninfo.date eq fit_nights(night))
; geninfo(j).stationcoord=genconfig.stationcoord
if nstn0 gt 0 then print,'GenConfig.StationCoord updated for ',fit_nights(night)
ns=ns+nstn0*nstp
;
; Update the astrometry
calcastrom
storenight,11
;
endif
;
ENDFOR
;
; Inform about unusable nights
if nstn gt 0 then begin
	index=where(nsolved eq 0,count)
	if count gt 0 then begin
		print,'These nights have not been used:'
		print,fit_nights(index)
	endif
endif
;
; Inform about stars not solved for
if nstr gt 0 then begin
	index=where(sc lt minobs,count)
	if count gt 0 then begin
		print,'These stars have not been solved for:'
		print,fit_stars(index)
	endif
endif
;
end
;-------------------------------------------------------------------------------
pro fitvolvox
;
; Used to fit a common set of station coordinates to several night's data.
; Star positions are not fitted.
;
; The coordinates in GenConfig are updated, as well as those in the
; stationtable. However, the constant term in the stationtable, corresponding
; to a path to a common reference in the lab, is incrementally updated
; instead of copying the value from GenConfig.
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([12,13]) ne 0 then return
;
; Check station and star selection
nstr=n_elements(fit_stars)
nstn=n_elements(fit_stations)
if nstn gt 0 then if fit_stations(0) eq '' then nstn=0
if nstr gt 0 then if fit_stars(0) eq '' then nstr=0
;
; For the common reference station, use the first one in stationtable
; Remove reference station from list, if included
if nstn gt 0 then begin
	index= $
	where(fit_stations ne stationtable(0).stationid,nstn)
	if nstn gt 0 then fit_stations0=fit_stations(index)
endif
;
; Sanity check
if nstn eq 0 then begin
	print,'***Error(FITVOLVOX): no fits were selected!'
	return
endif
;
RAD=180/pi_circle
TOL=1e-8
;
nnit=n_elements(fit_nights)	; number of nights
;
; Determine number of columns, we are not fitting star parameters
nc=nstn*4
;
; Allocate a hopefully sufficient number of rows
nr=1000
;
; Allocate design matrix for this night
m=make_array(nr,nc,/double)
r=make_array(nr,/double)
s=make_array(nc,/string)
;
id=0
;
FOR night=0,nnit-1 DO BEGIN
;
print,'Now processing ',fit_nights(night),'...'
;
words=nameparse(fit_nights(night))
loadnight,words(0),geoinfo(night).systemid,words(1)
calcastrom
index=where(genconfig.stationid eq stationtable(0).stationid,count)
if count gt 0 then referencestation,stationtable(0).stationid,'scan'
;
; Select all scans with valid time stamps
index=where(scans.time gt 0 and scans.starid ne 'FKV0000',count)
if count eq 0 then begin
	print,'Warning(FITVOLVOX): no valid data on ',fit_nights(night),'!'
	goto,SKIP
endif
ha=scans(index).ha*(15/rad)
dec=scans(index).dec/rad
starid=scans(index).starid
geodelay=scans(index).GeoDelay
case fit_data of
	'FDL':begin
	      obsdelay=scans(index).FDLPos
	      obsdelay(GenConfig.RefStation-1,*)=0
	      obsdelayerr=scans(index).FDLPosErr
	      end
	'GRP':begin
	      obsdelay=scans(index).GrpDelay
	      obsdelayerr=scans(index).GrpDelayErr
	      end
	'DRY':begin
	      obsdelay=scans(index).DryDelay
	      obsdelayerr=scans(index).DryDelayErr
	      end
	'WET':begin
	      obsdelay=scans(index).WetDelay
	      obsdelayerr=scans(index).WetDelayErr
	      end
endcase
;
; Extract data for stars if selected
scan_index=indgen(n_elements(starid)) 
if nstr gt 0 then begin
	id=0
	for i=0,nstr-1 do begin
		index=where(starid eq fit_stars(i),count)
		if count gt 0 then scan_index(id:id+count-1)=index
		id=id+count
	endfor
	if id gt 0 then scan_index=scan_index(0:id-1) else begin
		print,'Warning(FITVOLVOX): no data for stars on ', $
				fit_nights(night),'!'
		goto,SKIP
	endelse
endif
ch=cos(ha(scan_index))
sh=sin(ha(scan_index))
cd=cos(dec(scan_index))
sd=sin(dec(scan_index))
starid=starid(scan_index)
geodelay=geodelay(*,scan_index)
obsdelay=obsdelay(*,scan_index)
obsdelayerr=obsdelayerr(*,scan_index)
;
index=where(obsdelayerr gt 0,nr)
;
; Fill in the design matrix
for i=0,genconfig.numsid-1 do begin
    index_e=where(obsdelayerr(i,*) gt 0,count_e)
    if count_e gt 0 and i ne genconfig.refstation-1 then begin
    wt=1/obsdelayerr(i,index_e)
    r(id:id+count_e-1)=(obsdelay(i,index_e)-geodelay(i,index_e))*wt
    count=0
    index=where(fit_stations0 eq genconfig.stationId(i),count)
    if count ne 0 then begin
		m(id:id+count_e-1,index*4+0)=+cd(index_e)*ch(index_e)*wt; X
		m(id:id+count_e-1,index*4+1)=-cd(index_e)*sh(index_e)*wt; Y
		m(id:id+count_e-1,index*4+2)=+sd(index_e)*wt		; Z
		m(id:id+count_e-1,index*4+3)=-wt			; D
       		if genconfig.stationid(genconfig.refstation-1) $
		ne stationtable(0).stationid then begin
		index=where(fit_stations0 $
			 eq genconfig.stationid(genconfig.refstation-1))
		m(id:id+count_e-1,index*4+0)=-cd(index_e)*ch(index_e)*wt; X
		m(id:id+count_e-1,index*4+1)=+cd(index_e)*sh(index_e)*wt; Y
		m(id:id+count_e-1,index*4+2)=-sd(index_e)*wt		; Z
		m(id:id+count_e-1,index*4+3)=+wt			; D
		endif
    endif
;
    id=id+count_e
    endif
endfor
;
SKIP:
;
ENDFOR
;
; Remove zero rows in the design matrix
index=where(avg(abs(m),1) ne 0,count)
if count gt 0 then begin
	m=m(index,*)
	r=r(index)
	s=s(index)
endif
;
; If data is from white light source only, solve for Constant-Term only
stars=unique(s)
if n_elements(stars) eq 1 and stars(0) eq 'FKV0000' then begin
	do_c_only=1
	nstr=0
	m=m(*,indgen(nstn)*4+3)
endif else begin
	do_c_only=0
endelse
;
; Enough data?
nrow=n_elements(m(*,0))
ncol=n_elements(m(0,*))
if nrow lt ncol then begin
	print,'***Error(FITVOLVOX): not enough data!'
	return
endif
;
t=transpose(m)
n=t#m
y=t#r
;
if n_elements(n) eq 1 then begin
	s=y/n
endif else begin
	svd,n,w,u,v		; Singular value decomposition
; 	print,'Eigenvalues (normalized): ',w/max(w)
	small=where(w lt max(w)*1.0e-8,count)
	if count gt 0 then begin
;	print,'SVD: will edit',count,' singular values!'
		print,'***Error(FITVOLVOX): singular matrix!'
		return
		w(small)=0
	endif
; 	svbksb,u,w,v,y,s	; SVD solution
;
	in=invert(n,status)
	s=in#y			; Direct solution, is more precise
endelse
;
print,'Solution computed.'
;
FOR night=0,nnit-1 DO BEGIN
;
words=nameparse(fit_nights(night))
loadnight,words(0),geoinfo(night).systemid,words(1)
;
; Update the station coordinates
for i=0,nstn-1 do begin
	j=long(where(GenConfig.StationId eq fit_stations0(i)),0)
	if j ge 0 then begin
	ct0=GenConfig.StationCoord(3,j)
	if do_c_only then begin
	GenConfig.StationCoord(3,j)=GenConfig.StationCoord(3,j) $
				   +s(i)
	endif else begin
	GenConfig.StationCoord(*,j)=GenConfig.StationCoord(*,j) $
				   +equatorial2horizon(s(i*4:(i+1)*4-1))
	endelse
	ct1=GenConfig.StationCoord(3,j)
	k=long(where(stationtable.stationid eq fit_stations0(i)),0)
	stationtable(k).x=genconfig.stationcoord(0,j)
	stationtable(k).y=genconfig.stationcoord(1,j)
	stationtable(k).z=genconfig.stationcoord(2,j)
	stationtable(k).d=stationtable(k).d+(ct1-ct0)
	endif
endfor
;
if nstn gt 0 then print,'GenConfig.StationCoord updated for ',fit_nights(night)
;
; Update the astrometry
calcastrom
storenight,11
;
ENDFOR
;
end
;-------------------------------------------------------------------------------
pro getcorr,corr_xyz,corr_del,stationcoords,times=times,file=file
;
; stationcoords=[[genconfig.stationcoord],[genconfig.stationcoord],....]
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if Date eq '1999-04-10' then times=[28.4,31.5,34.8]*3600d0
if Date eq '1999-04-09' then times=[27.8,31.5,34.6]*3600d0
;
if n_elements(stationcoords) eq 0 or keyword_set(file) then begin
	x=0.d0
	y=0.d0
	z=0.d0
	c=0.d0
	status=dc_read_free(file,x,y,z,c,/col)
	stationcoords=dblarr(4,n_elements(x))
	stationcoords(0,*)=x
	stationcoords(1,*)=y
	stationcoords(2,*)=z
	stationcoords(3,*)=c
endif
;
num_epoch=n_elements(stationcoords(0,*))/3
num_sid=3
;
corr_xyz=reform(stationcoords(0:2,*),3,num_sid,num_epoch)
corr_del=reform(stationcoords(3,*),num_sid,num_epoch)
;
for i=0,num_epoch-1 do begin
	corr_xyz(*,*,i)=stationcoords(0:2,indgen(num_sid)+i*num_sid) $
		       -stationcoords(0:2,indgen(num_sid))
	corr_del(*,i)=stationcoords(3,indgen(num_sid)+i*num_sid) $
		     -stationcoords(3,indgen(num_sid))
endfor
;
corr_xyz=+corr_xyz*1d6
corr_del=-corr_del*1d6
;
end
;-------------------------------------------------------------------------------
pro readcterm,time,delay,file=file
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
status=dc_read_free(file,time,d1,d2,d3,d4,d5,d6,/col)
;
time=time*3600
delay=fltarr(genconfig.numsid,n_elements(time))
delay(0,*)=d1
if n_elements(d2) gt 0 then delay(1,*)=d2
delay=delay*2
;
end
;-------------------------------------------------------------------------------
pro whitecorr,base_time,base_del,remove=remove,apply=apply,compute=compute
;
; Subtract constant term corrections from calibrated delay data, i.e.
; the Grp-, Dry-, and Wet-delays. The corrections are derived from the 
; GRP white light position minus a linear fit through the differences
; between the GRP and DRY white light delays. Note that the FDL delays
; are not corrected.
; 
; The optional parameters are the fitted constant term values for the
; epochs given in the first parameter.
; Optional parameters: base_time (NumTime) [s]
;		       base_del(NumSid, NumTime) [microns]
;
; Every second call to this procedure reverses the action of the previous call!
; The remove and apply keywords force action, level 1 checks the status of the
; toggle switch (sign), level 2 forces application or removal of whitedelay.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common LocalWhiteCorr,sign
;
if checkdata([8,9]) ne 0 then return
;
if n_params() eq 1 then begin
	print,'***Error(WHITECORR): must have both time and corrections!'
	return
endif
;
; If compute=0 then take correction from whitedelay variable
if n_elements(compute) eq 0 then compute=1 else compute=compute gt 0
if n_elements(remove) eq 0 then remove=0
;
; Should have white light data for all siderostats
if n_elements(base_del) eq 0 and compute then begin
	w_exist=intarr(GenConfig.NumSid)
	for i=0,GenConfig.NumSid-1 do begin
		index=where(scans.starid eq 'FKV0000' $
			and scans.fdlposerr(i) gt 0 $
			and scans.drydelayerr(i) gt 0,count)
		if count eq 0 then begin
	   	if remove eq 0 then $
		print,'Warning(WHITECORR): no white light data for beam',i+1,'!'
		endif else w_exist(i)=1
	endfor
	if total(w_exist) eq 0 then begin
		if remove eq 0 then print,'No white light corrections applied!'
		return
	endif
endif
;
; This makes the procedure work like a toggle switch
if n_elements(sign) eq 0 then sign=1 else sign=-sign
;
if remove eq 1 and sign gt 0 then begin
	sign=-sign
	return
endif else if remove eq 2 then sign=-1
;
if n_elements(apply) eq 0 then apply=0
if apply eq 1 and sign lt 0 then begin
	sign=-sign
	return
endif else if apply eq 2 then sign=1
;
if sign gt 0 and compute then begin
;
	for i=0,GenConfig.NumSid-1 do begin
	  if n_elements(base_del) eq 0 then begin
	  if w_exist(i) then begin
	    index=where(scans.starid eq 'FKV0000' $
	   	    and scans.fdlposerr(i) gt 0)
	    d_fdl=scans(index).fdlpos(i)*(i ne GenConfig.RefStation-1) 
	    d_fdl=d_fdl-avg(d_fdl)
	    t_fdl=scans(index).time
	    index=where(scans.starid eq 'FKV0000' $
	    	    and scans.grpdelayerr(i) gt 0)
	    d_grp=scans(index).grpdelay(i) & d_grp=d_grp-avg(d_grp)
	    t_grp=scans(index).time
	    index=where(scans.starid eq 'FKV0000' $
	    	    and scans.drydelayerr(i) gt 0)
	    d_dry=scans(index).drydelay(i) & d_dry=d_dry-avg(d_dry)
	    t_dry=scans(index).time
;	    index=where(scans.starid eq 'FKV0000' $
;		    and scans.wetdelayerr(i) gt 0)
;	    d_wet=scans(index).wetdelay(i) & d_wet=d_wet-avg(d_wet)
;		t_wet=scans(index).time
	    index=where(scans.starid eq 'FKV0000' $
	    	and scans.grpdelayerr(i) gt 0 $
	    	and scans.drydelayerr(i) gt 0)
	    d_del=scans(index).grpdelay(i)-scans(index).drydelay(i)
	    d_del=d_del-avg(d_del)
	    t_del=scans(index).time
	    scans(*).whitedelay(i)=reform(spline(t_grp,d_grp, $
							abs(scans.time))) $
	    	      -poly(abs(scans.time),poly_fit(t_del,d_del,1))
	  endif
	  endif else begin
	  	d_fdl=reform(base_del(i,*))/1d6
	  	t_fdl=base_time
	  	scans(*).whitedelay(i)=reform(interpol(d_fdl,t_fdl, $
							abs(scans.time)))
	  endelse
	endfor
endif
;
scans.grpdelay=scans.grpdelay-scans.whitedelay*sign
scans.drydelay=scans.drydelay-scans.whitedelay*sign
;scans.wetdelay=scans.wetdelay-scans.whitedelay*sign
;
if sign gt 0 then print,'White light correction applied.' $
             else print,'White light correction removed.'
;
end
;-------------------------------------------------------------------------------
pro pivotcorr,base_time,base_xyz,remove=remove,apply=apply,compute=compute
;
; This procedure computes delay corrections from the pivot motion as
; measured with the baseline metrology system. It is mathematically identical
; to metrocorr, except it works with ScanData instead of PointData. 
; Note that the correction, as in whitecorr, is only applied to the
; calibrated delay data. The FDL data are not corrected.
;
; Use the averaged INCHWORM solution data to compute delay correction, or 
; derive the correction from the procedure parameters if supplied. The
; corrections are stored in the MetroDelay field of a scan. Note that they
; may be inconsistent with the solution data if they were derived from
; the procedure parameters.
;
; The optional parameters are the station coordinates for epochs given
; in the first parameter.
; Optional parameters: base_time(NumTime) [s]
;		       base_xyz(3,NumSid,NumTime) [microns]
;
; Every second call to this procedure reverses the action of the previous call!
;
; Every second call to this procedure reverses the action of the previous call!
; The remove and apply keywords force action, level 1 checks the status of the
; toggle switch (sign), level 2 forces application or removal of metrodelay.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common LocalPivotCorr,sign
;
if checkdata([3,8,9]) ne 0 then return
;
if n_params() eq 1 then begin
	print,'***Error(PIVOTCORR): must have both time and corrections!'
	return
endif
;
; If compute=0 then take correction from metrodelay variable
if n_elements(compute) eq 0 then compute=1 else compute=compute gt 0
if n_elements(remove) eq 0 then remove=0
;
if n_elements(base_xyz) eq 0 and compute $
	and total(abs(scans.parxerr) $
	         +abs(scans.paryerr) $
	         +abs(scans.parzerr)) eq 0 then begin
	if remove eq 0 then print,'***Error(PIVOTCORR): no metrology data!'
	return
endif
;
RAD=180/pi_circle
;
; This makes the procedure work like a toggle switch
if n_elements(sign) eq 0 then sign=1 else sign=-sign
;
if remove eq 1 and sign gt 0 then begin
	sign=-sign
	return
endif else if remove eq 2 then sign=-1
;
if n_elements(apply) eq 0 then apply=0
if apply eq 1 and sign lt 0 then begin
	sign=-sign
	return
endif else if apply eq 2 then sign=1
;
if sign gt 0 and compute then begin
;
;	MetroConfig has to be present before we continue
	if checkdata([21]) ne 0 then begin
		sign=-sign
		return
	endif
;
; 	Remove the sign from negative time stamps
	times=abs(scans.time)
;
; 	Compute apparent star positions
	topostar,times,scans.starid,StarTable,ra_app,dec_app
;
; 	Compute UT1
	ut1=utc2ut1(times)
;
; 	Compute Greenwich apparent sidereal time
	gst=ut12gst(times,ut1)
;
; 	Compute hour angle
	ha=hourangle(gst,ra_app)
;
	for j=0,GenConfig.NumSid-1 do begin
;
;       	Deal with FKV0000, i.e. the white light source, alias Alpha Lab
		index=where(scans.starid eq 'FKV0000',count_white)
		if count_white gt 0 then begin
			hadec=equatorial2hadec( $
			  horizon2equatorial( $
			  azel2horizon(MetroConfig.SidModel.FeedbeamAng(*,j))))
			ha(index)=hadec(0)
			dec_app(index)=hadec(1)
		endif
		if n_elements(base_xyz) ne 0 then begin
			base_x=spline(base_time,reform(base_xyz(0,j,*)),times)
			base_y=spline(base_time,reform(base_xyz(1,j,*)),times)
			base_z=spline(base_time,reform(base_xyz(2,j,*)),times)
			dec_r=dec_app/RAD & ha_r=ha*15/RAD
			x=-cos(dec_r)*sin(ha_r)
			y=+sin(dec_r)*cos(GeoParms.Latitude/rad) $
			  -cos(dec_r)*sin(GeoParms.Latitude/rad)*cos(ha_r)
			z=+cos(dec_r)*cos(GeoParms.Latitude/rad)*cos(ha_r) $
			  +sin(dec_r)*sin(GeoParms.Latitude/rad)
			d=base_x*x+base_y*y+base_z*z
			e=d*0+1
		endif else begin
			parx=scans.ParX(j,*)
			pary=scans.ParY(j,*)
			parz=scans.ParZ(j,*)
			d=catseyedelay(MetroConfig.SidModel.FeedbeamAng(*,j), $
		   	  parx,pary,parz, $
		   	  ha,dec_app,MetroConfig.SidModel.CatsEyeOff(j), $
		   	  scans.ParXErr(j,*), $
			  scans.ParYErr(j,*), $
			  scans.ParZErr(j,*),e)
		endelse
		if count_white gt 0 then begin
			index=where(scans.starid eq 'FKV0000')
			d(index)=d(index)/2
			e(index)=e(index)/2
		endif
		index=where(e le 0,count)
		if count gt 0 then begin
			print,'***Warning(PIVOTCORR): '+ $
                              'will interpolate/extrapolate MetroDelay for scan(s) '+ $
				retroparse(index+1)+', Sid ',j+1,'!', $
				format='(a,i2,a)'
			for k=0,n_elements(startable)-1 do begin
				kndex=where(scans.starid eq startable(k).starid)
				jndex=where(e(kndex) le 0,count)
				if count gt 0 then begin
				lndex=where(e(kndex) gt 0,count)
				if count eq 0 then begin
				print,'***Error(PIVOTVORR): '+ $
				'no valid scans for interpolation for star ', $
				startable(k).starid,'!'
				endif else begin
				d(kndex(jndex))= $
					spline(scans(kndex(lndex)).time, $
						 d(kndex(lndex)), $
						 scans(kndex(jndex)).time)
				print,'Repaired scans for star ', $
					startable(k).starid
				endelse
				endif
			endfor
		endif
		scans(*).metrodelay(j)=d
		scans(*).metrodelayerr(j)=e
	endfor
; 	Convert to "delay" format
	for j=0,GenConfig.NumSid-1 do begin
        	if j ne GenConfig.RefStation-1 then $
        	scans(*).metrodelay(j)=scans(*).metrodelay(j) $
			      	      -scans(*).metrodelay(GenConfig.RefStation-1)
	endfor
	scans(*).metrodelay(GenConfig.RefStation-1)=0
endif
;
scans.grpdelay=scans.grpdelay-scans.metrodelay*1d-6*sign
scans.drydelay=scans.drydelay-scans.metrodelay*1d-6*sign
scans.wetdelay=scans.wetdelay-scans.metrodelay*1d-6*sign
;
if sign gt 0 then print,'Pivot correction applied.' $
             else print,'Pivot correction removed.'
;
end
;-------------------------------------------------------------------------------
pro volvoxcorr,apply=apply,remove=remove,metro=metro
;
; For data loaded in AMOEBA, run through pivotcorr and whitecorr. It is
; important to run pivotcorr first since these corrections have to be applied
; to the white light data too. For removal, the order is not relevant.
;
; The apply and remove keywords are provided to force action. They are translated
; into level 2 "orders" which cause whitecorr and pivotcorr to remove the 
; corrections stored in whitedelay and metrodelay, respectively, no matter what
; state the toggle switch is in. This is because the toggle mode doesn't work 
; with multiple nights as stored by AMOEBA.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common LocalVolvoxCorr,sign
;
; This makes the procedure work like a toggle switch
if n_elements(sign) eq 0 then sign=1 else sign=-sign
;
; If remove=1 then force removal of correction
if n_elements(remove) eq 0 then remove=0 else remove=remove gt 0
if remove and sign gt 0 then begin
	sign=-sign
	return
endif
;
; If apply=1 force application of correction
if n_elements(apply) eq 0 then apply=0 else apply=apply gt 0
if apply and sign lt 0 then begin
	sign=-sign
	return
endif
;
; The default is to apply metrology corrections too
if n_elements(metro) eq 0 then metro=1 else metro=metro gt 0
;
case sign of
	+1:begin
	   apply=2
	   remove=0
	   end
	-1:begin
	   apply=0
	   remove=2
	   end
endcase
;
for i=0,n_elements(geninfo.date)-1 do begin
        if geoinfo(i).systemid eq 'NPOI' then begin
                loadnight,geninfo(i).date
		if metro then pivotcorr,apply=apply,remove=remove,compute=0
			      whitecorr,apply=apply,remove=remove
		storenight,11
        endif
endfor
;
end
;************************************************************************Block 3
function sidmodel,sid,model
;
; Return, in this order, the siderostat model parameters:
; Feed beam azimuth and elevation, siderostat azimuth and elevation,
; error in angle between axes (0=>orthogonal), zero point azimuth axis,
; tilt of mirror in its cell (0=>perpendicular) zero point elevation axis,
;
; If model is defined, store this in the metro configuration.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
i=sid-1
if n_elements(model) eq 0 then $
	return,[metroconfig.sidmodel.feedbeamang(0,i), $
		metroconfig.sidmodel.feedbeamang(1,i), $
		metroconfig.sidmodel.sidang(0,i), $
		metroconfig.sidmodel.sidang(1,i), $
		metroconfig.sidmodel.axisoffang(i), $
		metroconfig.sidmodel.zeroang(0,i), $
		metroconfig.sidmodel.mirroroffang(i), $
		metroconfig.sidmodel.zeroang(1,i)]
;
metroconfig.sidmodel.feedbeamang(0,i)=model(0)
metroconfig.sidmodel.feedbeamang(1,i)=model(1)
metroconfig.sidmodel.sidang(0,i)=model(2)
metroconfig.sidmodel.sidang(1,i)=model(3)
metroconfig.sidmodel.axisoffang(i)=model(4)
metroconfig.sidmodel.zeroang(0,i)=model(5)
metroconfig.sidmodel.mirroroffang(i)=model(6)
metroconfig.sidmodel.zeroang(1,i)=model(7)
;
return,0
;
end
;-------------------------------------------------------------------------------
function sidmodels,model,v_corr,v_id,time
;
; Create a series of models from model through application of corrective
; values to parameters specified in vid. 
;
common LocalSidModels,times
;
if n_elements(time) ne 0 then begin
	times=time
	models=dblarr(n_elements(model),n_elements(times))
	for i=0,n_elements(model)-1 do models(i,*)=model(i)
endif else models=model
;
for i=0,n_elements(v_id)-1 do begin
	j=where(v_corr(0,*) ge 0,count)
	if count gt 0 then begin
	models(v_id(i)-1,*)=models(v_id(i)-1,*) $
			   +interpol(v_corr(i+1,j),v_corr(0,j),times)
	endif
endfor
;
return,models
;
end
;-------------------------------------------------------------------------------
function sidpointing,m_in,x_in,dir
;
; Converts (az,el) into (ha,dec) for dir=+1,
;          (ha,dec) into (az,el) for dir=-1.
;
if n_elements(dir) eq 0 then return,-1
dir=long(dir)
;
x=double(reform(x_in))
;
model=double(m_in)
r=size(model)
if r(0) eq 1 then m=0L else m=r(1)
;
case dir of 
	1: begin
	   az=+x(0,*)
	   el=-x(1,*)
	   ha=az
	   dec=el
	   end
       -1: begin
	   ha=x(0,*)
	   dec=x(1,*)
	   az=ha
	   el=dec
	   end
     else: return,0
endcase
n=n_elements(az)
status=linknload(!external_lib,'siderostat',n,ha,dec,az,el,model,m,dir)
case dir of
	1: return,natural(ha,dec)
       -1: return,natural(az,-el)
endcase
;
end
;-------------------------------------------------------------------------------
function sidlimits,ha,dec
;
; Return an index array for whether (1) or not (0) a given array of (ha,dec)
; is within a specified siderostat's limits.
;
; The first set of limits was empirically determined from all data until 2000.
; The other sets were provided by B. O'Neill 2003-05-01. The index
; is the unique siderostat ID, not the StationID!.
;
; The model parameters:
; [feed az, feed el, sid az, sid el, axis off, zero az, mirror off, zero el]
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(ha) ne n_elements(dec) then begin
	print,'***Error(SIDLIMITS): ha and dec have different length!'
	return,-1
endif
num_elements=n_elements(ha)
if num_elements eq 0 then begin
	print,'***Error(SIDLIMITS): ha and dec undefined!'
	return,-1
endif
;
ret_index=intarr(num_elements)+1
;
for i=0,genconfig.numsid-1 do begin
case genconfig.siderostatid(i) of
	1:	begin
	  	model=[85.322,18.227,256.787,90.314,0.,102.187,0.,49.242]
;		Centered feed
		model=[86.578, 20.073, 80.888000, 89.736, 0.00, 278.095001, 0.00, 49.201]
;		Imaging feed
		model=[84.439, 20.096, 259.895, 90.258, 0.000,  99.084, 0.00, 49.185]
	  	limits=[-43.3,+37.4,-20.4,+34.6]
	  	limits=[-39.0,+41.5,-26.2,+34.8]
	  	limits=[-41.4,+40.7,-33.4,+27.0]
	  	end
	2:	begin
	  	model=[84.861,18.971,269.189,90.223,0.,89.242,0.,49.996]
;		Centered feed
		model=[86.230, 20.389, 84.887001, 89.785, 0.00, 273.606995, 0.00, 50.013]
;		Imaging feed
		model=[84.174, 20.271,  90.081, 89.768, 0.000, 268.411, 0.00, 50.002]
	  	limits=[-43.6,+37.5,-19.4,+35.8]
	  	limits=[-39.8,+42.8,-27.2,+36.1]
	  	limits=[-42.8,+39.9,-36.2,+27.2]
	  	end
	3:	begin
	  	model=[85.493,18.761,87.168,89.793,0.,270.862,0.,50.173]
;		Centered feed
		model=[86.790, 20.148, 88.589996, 89.790, 0.00, 269.424988, 0.00, 50.214]
;		Imaging feed
		model=[84.553, 20.287, 268.742, 90.208, 0.000, 449.274, 0.00, 50.224]
	  	limits=[-42.3,+38.5,-19.3,+35.7]
	  	limits=[-41.9,+40.9,-26.9,+36.1]
	  	limits=[-40.8,+42.0,-36.1,+27.0]
	  	end
	4:	begin
;		Centered feed
		model=[86.814, 20.230, 99.125999, 89.773, 0.00, 260.230988, 0.00, 49.981]
;		Imaging feed
		model=[84.553, 20.446,  94.720, 89.812, 0.000, 264.651, 0.00, 49.918]
		limits=[-43.5,+46.5,-33.3,+26.3]
		end
	5:	begin
;		Imaging feed
		model=[84.404, 22.114,  57.258, 89.727, 0.000, -59.838, 0.00, 70.016]
		limits=[-88.6,+85.6,-69.2,+18.4]
		end
	6:	begin
;		Imaging feed
		model=[84.592, 20.446,  58.695, 89.781, 0.000, -61.789, 0.00, 64.096]
		limits=[-91.2,+91.1,-61.6,+18.2]
		end
	 else:	begin
;		Imaging feed
	      	model=[85.322,18.227,256.787,90.314,0.,102.187,0.,49.242]
	      	limits=[-43.3,+37.4,-20.4,+34.6]
	      	limits=[-39.0,+41.5,-26.2,+34.8]
	      	end
endcase
model=double(model)
;
y=sidpointing(model,natural(ha,dec),-1)
az=y(0,*)
el=y(1,*)
;
index=where((az gt limits(0)) and (az lt limits(1)) $
	and (el gt limits(2)) and (el lt limits(3)),count)
;
sid_index=intarr(num_elements)
if count gt 0 then sid_index(index)=1
for j=0,count-2 do begin
	if index(j+1)-index(j) ne 1 then begin
		if j gt count/2 then sid_index(index(j+1):num_elements-1)=0 $
			        else sid_index(0:index(j))=0
	endif
endfor
;
ret_index=ret_index*sid_index
endfor
;
return,ret_index
;
end
;-------------------------------------------------------------------------------
function fdllimits,hourangle,declination
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(hourangle) ne n_elements(declination) then begin
	print,'***Error(FDLLIMITS): ha and dec have different length!'
	return,-1
endif
num_elements=n_elements(hourangle)
if num_elements eq 0 then begin
	print,'***Error(FDLLIMITS): ha and dec undefined!'
	return,-1
endif
;
rad=180/!pi
ha=hourangle*(15/rad)
dec=declination/rad
;
su=dblarr(num_elements,3) & sv=su & sw=su
;
su(*,0)=+sin(ha)
su(*,1)=+cos(ha)
su(*,2)=+0
sv(*,0)=-sin(dec)*cos(ha)
sv(*,1)=+sin(dec)*sin(ha)
sv(*,2)=+cos(dec)
sw(*,0)=+cos(dec)*cos(ha)
sw(*,1)=-cos(dec)*sin(ha)
sw(*,2)=+sin(dec)
;
d=fltarr(num_elements,genconfig.numsid)
;
for i=0,genconfig.numsid-1 do begin
	coord=horizon2equatorial(GenConfig.StationCoord(*,i))
	d(*,i)=sw(*,0)*coord(0)+sw(*,1)*coord(1)+sw(*,2)*coord(2)-coord(3)
endfor
;
fdl_limit=system_config(system_id(systemid),'FDLLIMIT')
;
ret_index=intarr(num_elements)
for i=0,num_elements-1 do begin
	opl=d(i,*)-min(d(i,*))
	j=where(opl eq max(opl)) & j=j(0)
	fdl_limit=system_config(system_id(systemid),'FDLLIMIT', $
		stations=genconfig.stationid(j))
	if max(opl) lt fdl_limit then ret_index(i)=1
endfor
;
return,ret_index
;
end
;-------------------------------------------------------------------------------
function refraction,hadec0,ft=ft,rh=rh,bp=bp,lambda=lambda
;
common LocalRefraction,ft0,fh0,bp0,lambda0
;
if n_elements(ft0) eq 0 then ft0=56.3
if n_elements(rh0) eq 0 then rh0=33.0
if n_elements(bp0) eq 0 then bp0=23.316
if n_elements(lambda0) eq 0 then lambda0=0.65
;
if n_elements(ft) eq 0 then ft=ft0
if n_elements(rh) eq 0 then rh=rh0
if n_elements(bp) eq 0 then bp=bp0
if n_elements(bp) eq 0 then bp=bp0
if n_elements(lambda) eq 0 then lambda=lambda0
;
RAD=180/!pi
;
azel=horizon2azel(equatorial2horizon(hadec2equatorial(hadec0)))
del=((oneill(lambda,ft,rh,bp)-1)/tan(azel(1,*)/RAD))*RAD
azel(1,*)=azel(1,*)+del
hadec=equatorial2hadec(horizon2equatorial(azel2horizon(azel)))
;
ft0=ft
rh0=rh
bp0=bp
;
return,hadec
;
end
;-------------------------------------------------------------------------------
pro riseset,uptime,nighttime,force=force,ldl=ldl
;
; Given a StarTable, compute the visibility for a given date and system.
; Return uptime(2,num_stars), the rise and set times for the interferometer
; horizon in seconds. Also return nighttime(2), beginning and end in seconds
; of a night.
;
; Note that only an average siderostat horizon (set by motor angle limits)
; is used. The delays, which depend on the specific configuration, are
; taken into account.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common LocalRiseSet,date_p,system_p,stars_p,stations_p, $
                times,gst,za_sun,za_moon,ha_moon,az_moon, $
		za_star_p,ha_star_p,az_star_p, $
		sun_ok,moon_ok,zas_ok,sid_ok_p,fdl_ok_p
;
; If force=1, always recompute circumstances
if n_elements(force) eq 0 then force=0
;
num_stars=n_elements(StarTable)
if num_stars eq 0 then begin
        print,'***Error(RISESET): no stars defined!'
        return
endif
;
if n_elements(Date) eq 0 then begin
	print,'Warning(RISESET): Date undefined!'
        parseidldate,systime(),y,m,d
        Date=nextdate(constrictordate(y,m,d))
	print,'Set date to: ',date
endif
if n_elements(SystemId) eq 0 then begin
	print,'Warning(RISESET): SystemId undefined!'
	SystemId='NPOI'
	print,'Set SystemId to: ',SystemId
endif
if system_config(system_id(systemid),'EXIST') ne 1 then begin
	print,'***Error(RISESET): SystemId unknown!'
	return
endif
;
if n_elements(date_p) eq 0 then date_p=''
if n_elements(system_p) eq 0 then system_p=''
if n_elements(stars_p) eq 0 then stars_p=''
if n_elements(stations_p) eq 0 then stations_p=''
;
; New computation of GST, Sun and Moon necessary if date or system changed
IF Date NE date_p OR SystemId NE system_p OR force THEN BEGIN
;
; 	Get GeoParms
	hds_close
	get_sysconfig
;
; 	Allocate array of UT hours
	num_ut=200
	times=dindgen(num_ut)/num_ut*86400-(geoparms.longitude/15+12)*3600
;
; 	Compute UT1
	ut1=utc2ut1(times)
;
; 	Compute Greenwich apparent sidereal time
	gst=ut12gst(times,ut1)
;
; 	Get Right Ascension and declination of the Sun
	topoplanet,times,intarr(num_ut)+10,ra_sun,dec_sun,distance
;
; 	Compute hour angle of the Sun
	ha_sun=hourangle(gst,ra_sun)
;
; 	Compute zenith angle of the Sun
	za_sun=zenithangle(ha_sun,dec_sun)
;
; 	Get Right Ascension and declination of the Moon (w/ephemeris access)
;	topoplanet,times,intarr(num_ut)+11,ra_moon,dec_moon,distance
;
; 	Get Right Ascension and declination of the Moon (w/out ephemeris access)
	parsedate,Date,y,m,d
	tjd=(times+GeoParms.TAI_UTC+GeoParms.TDT_TAI)/86400+julian(y,m,d)
	moonpos,tjd,ra_moon,dec_moon	; from idlastro library
	ra_moon=ra_moon/15
;
; 	Compute hour angle of the moon
	ha_moon=hourangle(gst,ra_moon)
;
; 	Compute zenith angle of the moon
	za_moon=zenithangle(ha_moon,dec_moon)
;
; 	Compute azimuth of the moon
	hadec=transpose([[ha_moon],[dec_moon]])
	azel=horizon2azel(equatorial2horizon(hadec2equatorial(hadec)))
	az_moon=reform(azel(0,*))
;
ENDIF
;
; Compute night time
za_sun_min=108.0
index=where(za_sun gt za_sun_min,count)
nighttime=[times(index(0)),times(index(count-1))]
;
; Allocate ZA, HA, and AZ
num_star=n_elements(StarTable)
num_ut=n_elements(times)
za_star=dblarr(num_ut,num_star)
ha_star=dblarr(num_ut,num_star)
az_star=dblarr(num_ut,num_star)
;
; These are binary 1/0 (T/F) indices for various conditions
sun_ok=intarr(num_ut)
moon_ok=intarr(num_ut,num_star)
zas_ok=intarr(num_ut,num_star)
sid_ok=intarr(num_ut,num_star)
fdl_ok=intarr(num_ut,num_star)+1
;
genconfig_checksum=strjoin(genconfig.stationid)
genconfig_checksum=''
for i=0,n_tags(genconfig)-1 do genconfig_checksum=genconfig_checksum $
				 +strjoin(string(genconfig.(i)),/single)
;
; Compute angles for the stars
stars=strarr(num_ut)
for i=0,num_star-1 do begin
	j=where(stars_p eq StarTable(i).starid,count) & j=j(0)
	if count eq 1 and Date eq date_p and not force $
		      and genconfig_checksum eq stations_p then begin
		ha_star(*,i)=ha_star_p(*,j)
		za_star(*,i)=za_star_p(*,j)
		az_star(*,i)=az_star_p(*,j)
		sid_ok(*,i)=sid_ok_p(*,j)
		fdl_ok(*,i)=fdl_ok_p(*,j)
	endif else begin
		stars(*)=StarTable(i).starid
		topostar,times,stars,StarTable,ra_app,dec_app
		ha_star(*,i)=hourangle(gst,ra_app)
		za_star(*,i)=zenithangle(ha_star(*,i),dec_app)
		hadec=transpose([[ha_star(*,i)],[dec_app]])
		azel=horizon2azel(equatorial2horizon(hadec2equatorial(hadec)))
		az_star(*,i)=reform(azel(0,*))
		sid_ok(*,i)=sidlimits(ha_star(*,i),dec_app)
		fdl_ok(*,i)=fdllimits(ha_star(*,i),dec_app)
		if keyword_set(ldl) then fdl_ok(*,i)=1
		if system_id(systemid) eq 'VLTI' then begin
			sid_ok(*,i)=1
			for j=0,genconfig.numsid-1 do begin
			case genconfig.stationid(j) of
;			Here AZ is 0 towards S, increasing Westwards
			'AA0':limits=[48,165,209]
			'AD0':limits=[48,112,156]
			'AE0':limits=[38,98,133]
	 		else:limits=[0,0,360]
			endcase
			for k=0,n_elements(times)-1 do begin
; 			Convert to 0 towards S, incr. Westwards
			az=(360+180-azel(0,k)) mod 360 
; 			Convert to 0 towards S, incr. Eastwards
;			az=(azel(0,k)+180) mod 360 
			el=azel(1,k)
			n_az=1l
			n_el=1l
			r=0.0
			t=0.0
			shadow=0.d0
			loc=vlti_station(genconfig.stationid(j),ts)
;			status=linknload(!external_lib,'shadow',n_az,n_el,az,el,shadow,loc,r,t,ts)
			if az gt limits(1) and az lt limits(2) $
					   and el lt limits(0) then shadow=1
			if shadow ne 0 then sid_ok(k,i)=0
			endfor
			endfor
		endif
	endelse
endfor
;
; if SystemId ne 'NPOI' then sid_ok(*,*)=1
; if SystemId ne 'NPOI' then fdl_ok(*,*)=1
;
uptime=fltarr(2,num_star)
index=where(za_sun gt za_sun_min,count)
if count ge 1 then sun_ok(index)=1
;
za_star_max=system_config(system_id(systemid),'ZALIMITMAX')
za_star_min=system_config(system_id(systemid),'ZALIMITMIN')
for i=0,num_star-1 do begin
	index=where(za_star(*,i) lt za_star_max $
		and za_star(*,i) gt za_star_min,count)
	if count ge 1 then zas_ok(index,i)=1
	index=where(zas_ok(*,i) and sid_ok(*,i) and fdl_ok(*,i),count)
	if StarTable(i).starid eq 'FKV0000' then begin
		index=indgen(num_ut)
		count=num_ut
	endif
;
	if count gt 1 then begin
		index=contiguous(index)
		count=n_elements(index)
		i0=index(0)
		i1=index(count-1)
		uptime(0,i)=times(i0)
		uptime(1,i)=times(i1)
	endif
endfor
;
if n_params() eq 0 then $
for i=0,num_star-1 do print,StarTable(i).starid,'  ',hms(uptime(*,i)/3600)
;
; Save the results for next call
date_p=Date
system_p=SystemId
stars_p=StarTable.starid
genconfig_checksum=strjoin(genconfig.stationid)
genconfig_checksum=''
for i=0,n_tags(genconfig)-1 do genconfig_checksum=genconfig_checksum $
				 +strjoin(string(genconfig.(i)),/single)
stations_p=genconfig_checksum
ha_star_p=ha_star
za_star_p=za_star
az_star_p=az_star
sid_ok_p=sid_ok
fdl_ok_p=fdl_ok
;
end
;-------------------------------------------------------------------------------
pro roseset,uptime
;
; In contrast to procedure riseset, this one gets the actual times from
; a the currently loaded data set.
;
common StarBase,StarTable,Notes
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
num_stars=n_elements(StarTable)
if num_stars eq 0 then begin
        print,'***Error(ROSESET): no stars defined!'
        return
endif
;
if n_elements(scans) eq 0 then begin
	print,'***Error(ROSESET): no ScanData!'
	return
endif
;
uptime=fltarr(2,num_stars)
for i=0,num_stars-1 do begin
        index=where(scans.starid eq StarTable(i).starid,count)
        if count gt 0 then begin
        	times=scans(index).time
        	index=where(times gt 0,count)
        	if count eq 0 then begin
                	uptime(*,i)=0
        	endif else begin
                	uptime(0,i)=min(times(index))
                	uptime(1,i)=max(times(index))
        	endelse
        endif else uptime(*,i)=0
endfor
;
end
;-------------------------------------------------------------------------------
