;*******************************************************************************
; File: chameleon.pro
;
; Description:
; ------------
; IDL scripts specific to CHAMELEON software. 
;
; CHAMELEON is used to read CONSTRICTOR output, display and edit data, average
; delays and visibilities, and to calibrate visibility amplitudes and phases.
; It also does delay dispersion corrections and applies metrology corrections.
;
; Block directory:
; ----------------
; Block 1: channelindex,pointindex,scanindex,
;	   defaultbg,expandbg,replacebg,processbg
;
; Block 2: tripleproduct,
;	   unwraptriple,unwrapphase,flagphase,flagstations,
;	   fringeid,dispinit,dispcorr,
;	   metrocorr,astrocorr,metroinit,inchav,
;	   solindex,lmpivot,pivotfit,lmmotor,motorfit,
;	   average,clearpointdata,
;	   rewraptriple,dewraptriple,
;	   fixearlydays,fixwavelengths,fixbg,
;	   deadcorr,moiree,medianob
;
; Block 3: create_calentries,set_calentries,
;	   calmatrix,calcoeffs,calibrate,uncal,resetcal,editcal,initcal,
;	   calchisq,caldiameter,calvis,bessel,calcviscal
;
; Block 4: list_stars,list_bgscans,list_scans,list_stations,list_flagreasons,
;	   list_calreasons,list_calentries,list_summary_chameleon,list_logs,
;	   list_scanlog,list_observerlog,list_constrictorlog,
;	   list_oysterlog,list_inchwormlog,
;	   print_scanforms,print_specforms
;
; Block 5: set_visampphase,set_tripleampphase,
;          set_complexvis,set_compltriple,
;	   set_viserrors,set_maxerrors,set_minerrors,
;	   npoipipe
;
; Block 6: startnpoiconfig,storenpoiconfig,npoiconfig,vlticonfig
; Block 7: apdrates,natrates,photom
;
;************************************************************************Block 1
function channelindex,specid,datum,bad=bad
;
; This function returns indices of good or bad channels based on
; a small data base of NPOI channel performances.
;
; Should also apply to H_alpha setups
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(specid) eq 0 then return,-1
if n_elements(datum) eq 0 then datum=date
if n_elements(bad) eq 0 then bad=0 else bad=bad gt 0
;
parsedate,date,y,m,d
jd=julian(y,m,d)
;
id=strmid(specid,strlen(specid)-1,1)
case id of
'1':	begin
;	Bad laser channel, bad blue response
	if jd lt julian(2000,1,1) then begin
		chg=stringparse('1-10,12-20')
		chb=stringparse('11,21-32')
;	Broken fiber
	endif else if jd lt julian(2001,9,28)then begin
		chg=stringparse('1-10,13-20')
		chb=stringparse('11,12,21-32')
;	6-way
	endif else if jd lt julian(2002,5,11)then begin
		chg=stringparse('1-10,13-16')
		chb=stringparse('11,12')
;	Spectrometer recabled to remove laser channel,
;	but broken fiber (jumper?) remains, is now channel 11
	endif else begin
		chg=stringparse('1-10,12-16')
		chb=11
	endelse
	end
'2':	begin
	if jd lt julian(2001,9,28)then begin
		chg=stringparse('1-10,12-20')
		chb=stringparse('11,21-32')
;	6-way
	endif else if jd lt julian(2002,5,11)then begin
		chg=stringparse('1-10,12-16')
		chb=11
	endif else if jd lt julian(2010,2,01)then begin
;		Channels all OK on 2002-05-20
		chg=stringparse('1-16')
		chb=0
	endif else begin
		chg=stringparse('1-10,12-16')
;		Still bad, looking at data from 2011-10 and 2012-03
		chb=11
	endelse
	end
'3':	begin
	if jd lt julian(2001,9,28)then begin
		chg=stringparse('1-10,12-20')
		chb=stringparse('11,21-32')
;	6-way
	endif else if jd lt julian(2002,5,11)then begin
		chg=stringparse('1-10,12-16')
		chb=11
;	Data from October 2003 have bad channel 11
	endif else if jd le julian(2003,10,18)then begin
		chg=stringparse('1-10,12-16')
		chb=11
	endif else if jd lt julian(2010,2,01)then begin
		chg=stringparse('1-16')
		chb=0
	endif else begin
;		Channel 4 has intermittend elevated background
;		chg=stringparse('1-3,5-10,12-16')
		chg=stringparse('1-3,5-16')
;		chb=stringparse('4,11')
		chb=stringparse('4')
	endelse
	end
endcase
;
if bad then return,chb else return,chg
;
end
;-------------------------------------------------------------------------------
function pointindex,star,coherent=coherent
;
; Return indices into pointdata items from an input selection.
;
common Tables,ScanTable,BGTable,StationTable
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if n_elements(coherent) ne 0 then jndex=where(scantable.code eq 1) $
			     else jndex=indgen(n_elements(scantable))
count=n_elements(StarId(jndex))
if n_elements(star) ne 0 then index=where(StarId(jndex) eq star,count) $
			 else index=indgen(count)
;
if count eq 0 then return,-1
return,stringparse(strcompress(strjoin(string(rec0(index(jndex)))+'-' $
				      +string(recn(index(jndex))),','),/remove))
end
;-------------------------------------------------------------------------------
function scanindex,times
;
; For a given list of times [s], return indices of matching scans.
; Ignore times outside of any scan boundaries.
;
common Tables,ScanTable,BGTable,StationTable
;
n=n_elements(times)
index=intarr(n)
k=0
for i=0,n-1 do begin
	j=where(scantable.starttime le (times(i)+1) $
	    and scantable.stoptime ge (times(i)-1),count)
	if count eq 1 then begin
		index(k)=j
		k=k+1
	endif
endfor
if k gt 0 then return,index(0:k-1) else return,-1
;
end
;-------------------------------------------------------------------------------
pro defaultbg
;
; Initializes background table and scans so that one can average pointdata 
; if no measured values are available. Default back ground rate is zero.
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([1,3,8]) ne 0 then return
numscan=n_elements(ScanTable)
;
Iscan=lindgen(numscan)+1
ScanId=lonarr(numscan) & ScanId(*)=ScanTable.ScanId
Time=dblarr(numscan) & Time(*)=ScanTable.StopTime
StarId=strarr(numscan) & StarId(*)=ScanTable.StarId
Ra=dblarr(numscan,/nozero)
Dec=dblarr(numscan,/nozero)
for i=0,n_elements(startable)-1 do begin
	index=where(StarId eq startable(i).starid,count)
	if count then begin
		Ra(index)=startable(i).Ra
		Dec(index)=startable(i).Dec
	endif
endfor
BGTable=build_bgtable(Iscan,ScanId,Time,Ra,Dec)
print,'Default Backgroundtable created.'
;
MS=max(GenConfig.NumSpecChan)
s=allocbgscan(GenConfig.NumOutBeam,MS)
bgscans=replicate(s,numscan)
bgscans.Time=BGTable.Time
bgscans.Rate=0
bgscans.RateErr=1
;
print,'Default Background data set.'
;
end
;-------------------------------------------------------------------------------
pro expandbg
;
; Go through ScanTable and check whether every scan has a background measure.
; If not, fill in background data from an average over other scans in the same
; outputbeam and channel.
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([1,2,3,8]) ne 0 then return
;
missing_bg_scans=intarr(n_elements(scantable))
;
for i=0,n_elements(ScanTable)-1 do begin
index=where(BGTable.ScanId eq ScanTable(i).ScanId,count)
if count gt 0 then begin
	if i eq 0 then begin
		ScanId=BGTable(index).ScanId
		Time=BGTable(index).Time
		Ra=BGTable(index).Ra
		Dec=BGTable(index).Dec
		s=bgscans(index)
	endif else begin
		ScanId=[ScanId,BGTable(index).ScanId]
		Time=[Time,BGTable(index).Time]
		Ra=[Ra,BGTable(index).Ra]
		Dec=[Dec,BGTable(index).Dec]
		s=[s,bgscans(index)]
	endelse
endif else begin
;	print,'No background scan for ScanNo:',ScanTable(i).Iscan
	missing_bg_scans(i)=ScanTable(i).Iscan
	j=where(startable.starid eq ScanTable(i).StarId,count)
	if count eq 0 then begin
		print,'***Error(EXPANDBG): star not found!'
		return
	endif
	if i eq 0 then begin
		ScanId=ScanTable(i).ScanId
		Time=ScanTable(i).StopTime
		Ra=startable(j).Ra
		Dec=startable(j).Dec
		s=bgscans(0)
		s.Time=ScanTable(i).StopTime
		s.RateErr=0
		replacebg,s
	endif else begin
		ScanId=[ScanId,ScanTable(i).ScanId]
		Time=[Time,ScanTable(i).StopTime]
		Ra=[Ra,startable(j).Ra]
		Dec=[Dec,startable(j).Dec]
		bgs=bgscans(0)
		bgs.Time=ScanTable(i).StopTime
		bgs.RateErr=0
		replacebg,bgs
		s=[s,bgs]
        endelse
endelse
endfor
;
Iscan=lindgen(n_elements(ScanId))+1
;
BGTable=build_bgtable(Iscan,ScanId,Time,Ra,Dec)
bgscans=s
index=where(missing_bg_scans ne 0,count)
if count gt 0 then begin
	print,'No background scans for ScanNo(s):'
	print,missing_bg_scans(index)
endif
print,'Background table/data expanded.'
;
end
;-------------------------------------------------------------------------------
pro replacebg,s
;
; Replace flagged background rates in a given background scan with 
; average of other scans in same channel. Beware of white light scans
; which are not recognized by this procedure!
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; If bg scan was created, warn only of missing channels
index=where(s(0).RateErr eq 0,count)
if count gt 0 then expand=1 else expand=0
;
for i=0,GenConfig.NumOutBeam-1 do begin
channels=0
dead_channel_index=total(reform(bgscans.rateerr(i,*) > 0),2)
;
for j=0,GenConfig.NumSpecChan(i)-1 do begin
;
if s(0).RateErr(i,j) le 0 then begin
	index=where(bgscans(*).RateErr(i,j) gt 0,count)
	if count gt 0 then begin
		s(0).Time=abs(s(0).Time)
		if count gt 1 then begin
			s(0).Rate(i,j)=avg(bgscans(index).Rate(i,j))
			s(0).RateErr(i,j)=stdev(bgscans(index).Rate(i,j))
;			The standard deviation is zero after running medianbg
			if s(0).RateErr(i,j) eq 0 then $
			s(0).RateErr(i,j)=avg(bgscans(index).RateErr(i,j))
		endif else begin
			s(0).Rate(i,j)=bgscans(index).Rate(i,j)
			s(0).RateErr(i,j)=sqrt(s(0).Rate(i,j))
		endelse
		channels=[channels,j+1]
	endif else begin
		if expand and dead_channel_index(j) ne 0 then $
		print,'Warning(REPLACEBG): no data found for beam ', $
			i+1,', channel ',j+1,'!',format='(a,i1,a,i2,a)'
	endelse
endif
;
endfor
if n_elements(channels) gt 1 and not expand then $
print,'Replaced value for BGScan ',hms(s(0).time/3600.0),', beam ',i+1, $
		', channels ',retroparse(channels(1:n_elements(channels)-1)), $
		format='(a,a,a,i1,a,a)'
endfor
;
end
;-------------------------------------------------------------------------------
pro processbg
;
; Replace flagged background data in all background scans.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([10]) ne 0 then return
;
bgs=bgscans
for k=0,n_elements(bgscans)-1 do begin
	s=bgs(k)
	replacebg,s	; Note that address of s is passed (bgs(k) by value)!
	bgs(k)=s
endfor
bgscans=bgs
;
print,'Background data processed.'
;
end
;-------------------------------------------------------------------------------
pro medianbg
;
; A more robust procedure to flag all bad channels using channelindex
; and to replace background values in each channel with the median of
; that channel.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
for i=0,GenConfig.NumOutBeam-1 do begin
	bad_channels=channelindex(genconfig.spectrometerid(i),/bad)-1
	if bad_channels(0) ge 0 then $
		bgscans(*).RateErr(i,bad_channels)= $
	   -abs(bgscans(*).RateErr(i,bad_channels))
for j=0,GenConfig.NumSpecChan(i)-1 do begin
	index=where(bgscans.RateErr(i,j) gt 0,count)
	if count gt 0 then begin
		v=medianve(bgscans(index).Rate(i,j),e)
		bgscans.Rate(i,j)=v
		bgscans.RateErr(i,j)=e
	endif
;
endfor
endfor
;
; Inform about bad channels
for i=0,genconfig.numoutbeam-1 do begin
dead_channel_index=total(reform(bgscans.rateerr(i,*) > 0),2)
index=where(dead_channel_index eq 0,count)
if count gt 1 then s='s' else s=''
if count ge 1 then $
print,'Warning(MEDIANBG): ' $
	+string(count,format='(i3)')+' bad channel'+s+' in beam ' $
	+string(i+1,format='(i1)')+': '+string(index+1,format='(i2)')
endfor
;
print,'BG data processed.'
;
end
;************************************************************************Block 2
pro tripleproduct
;
; Add to or remove a triple product from the pointdata. The information on the
; three baselines selected is contained in triple(TripleInfo). The information
; on which channels will be combined is contained in channels(TripleInfo)
; The triple product is the complex product of ComplexVis on three baselines.
;
common TripleInfo,triple,channels,action
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
; Check presence of GenConfig and PointData
if checkdata([8,11]) ne 0 then return
;
; Check validity of channel selection
index=where(channels ge 0,count)
if count mod 3 ne 0 then begin
	print,'***Error(TRIPLEPRODUCT): invalid channel selection!'
	return
endif
count_c=count/3
chans=channels(0:count_c-1,*)
;
; Check validity of triple and channel selection
index=where(triple lt 0,flag1)
index=where(chans lt 0,flag2)
flag=flag1+flag2
if flag eq 0 then begin
for i=0,2 do begin
	if triple(i,0) ge GenConfig.NumOutBeam then flag=1 else begin 
	if triple(i,1) ge GenConfig.NumBaseline(triple(i,0)) then flag=1
	if max(chans(*,i)) ge GenConfig.NumSpecChan(triple(i,0)) then flag=1
	endelse
endfor
endif
if flag ne 0 then begin
	print,'***Error(TRIPLEPRODUCT): invalid selection!'
	return
endif
;
MS=max(GenConfig.NumSpecChan)
MP=n_elements(PointTime)
;
IF action eq 'Add' then BEGIN
;
GenConfig.NumTriple=GenConfig.NumTriple+1
;
; Extend and allocate arrays
; (Note that a minimum of one triple is always allocated)
if GenConfig.NumTriple gt 1 then begin
        CT=fltarr(GenConfig.NumTriple,2,MS,MP,/nozero)
        CTE=fltarr(GenConfig.NumTriple,2,MS,MP,/nozero)
	CT(0:GenConfig.NumTriple-2,*,*,*)=ComplTriple
	CTE(0:GenConfig.NumTriple-2,*,*,*)=ComplTripleErr
	ComplTriple=CT
	ComplTripleErr=CTE
        TripleAmp=fltarr(GenConfig.NumTriple,MS,MP,/nozero)
        TripleAmpErr=fltarr(GenConfig.NumTriple,MS,MP,/nozero)
        TriplePhase=fltarr(GenConfig.NumTriple,MS,MP,/nozero)
        TriplePhaseErr=fltarr(GenConfig.NumTriple,MS,MP,/nozero)
endif
;
; Update GenConfig
GenConfig.TripleBeam(*,GenConfig.NumTriple-1)=triple(*,0)
GenConfig.TripleBase(*,GenConfig.NumTriple-1)=triple(*,1)
GenConfig.TripleChan(0:count_c-1,*,GenConfig.NumTriple-1)=chans
GenConfig.TripleNumChan(GenConfig.NumTriple-1)=count_c
;
; Compute complex triple product
ComplexVis1=complex(ComplexVis(triple(0,0),0,chans(*,0),triple(0,1),*), $
		    ComplexVis(triple(0,0),1,chans(*,0),triple(0,1),*))
ComplexVis2=complex(ComplexVis(triple(1,0),0,chans(*,1),triple(1,1),*), $
		    ComplexVis(triple(1,0),1,chans(*,1),triple(1,1),*))
ComplexVis3=complex(ComplexVis(triple(2,0),0,chans(*,2),triple(2,1),*), $
		    ComplexVis(triple(2,0),1,chans(*,2),triple(2,1),*))
ComplexTriple=ComplexVis1*ComplexVis2*ComplexVis3
ComplTriple(GenConfig.NumTriple-1,0,0:count_c-1,*)=float(ComplexTriple)
ComplTriple(GenConfig.NumTriple-1,1,0:count_c-1,*)=imaginary(ComplexTriple)
;
; Compute error of complex triple product
ComplexVisErr1= $
	complex(ComplexVisErr(triple(0,0),0,chans(*,0),triple(0,1),*), $
		ComplexVisErr(triple(0,0),1,chans(*,0),triple(0,1),*))
ComplexVisErr2= $
	complex(ComplexVisErr(triple(1,0),0,chans(*,1),triple(1,1),*), $
		ComplexVisErr(triple(1,0),1,chans(*,1),triple(1,1),*))
ComplexVisErr3= $
	complex(ComplexVisErr(triple(2,0),0,chans(*,2),triple(2,1),*), $
		ComplexVisErr(triple(2,0),1,chans(*,2),triple(2,1),*))
ComplexTripleErr=sqrt((ComplexVis2*ComplexVis3*ComplexVisErr1)^2+ $
		      (ComplexVis1*ComplexVis3*ComplexVisErr2)^2+ $
	  	      (ComplexVis1*ComplexVis2*ComplexVisErr3)^2)
ComplTripleErr(GenConfig.NumTriple-1,0,0:count_c-1,*)= $
						float(ComplexTripleErr)
ComplTripleErr(GenConfig.NumTriple-1,1,0:count_c-1,*)= $
						imaginary(ComplexTripleErr)
;
; Compute amplitude and phase of complex triple product
set_tripleampphase
print,'Triple added.'
;
ENDIF ELSE IF action eq 'Delete' then BEGIN
;
if GenConfig.NumTriple eq 0 then begin
	print,'Warning(TRIPLEPRODUCT): no triples found.'
	return
endif
;
removed=0
repeat begin
i=0
repeat begin
	has=0
	for j=0,2 do begin
		hit=0
		for k=0,2 do begin
			obs=[GenConfig.TripleBeam(k,i), $
		     	     GenConfig.TripleBase(k,i)]
			if total(abs(obs-triple(j,*))) eq 0 then hit=hit+1
		endfor
		if hit ne 0 then has=has+1
	endfor
	i=i+1
endrep until has eq 3 or i eq GenConfig.NumTriple
if has eq 3 then begin
	i=i-1
	GenConfig.NumTriple=GenConfig.NumTriple-1
	removed=removed+1
	if GenConfig.NumTriple gt 0 then begin
		GenConfig.TripleBeam(*,i:GenConfig.NumTriple-1)= $
			GenConfig.TripleBeam(*,i+1:GenConfig.NumTriple)
		GenConfig.TripleBase(*,i:GenConfig.NumTriple-1)= $
			GenConfig.TripleBase(*,i+1:GenConfig.NumTriple)
		GenConfig.TripleChan(*,*,i:GenConfig.NumTriple-1)= $
			GenConfig.TripleChan(*,*,i+1:GenConfig.NumTriple)
		GenConfig.TripleNumChan(i:GenConfig.NumTriple-1)= $
			GenConfig.TripleNumChan(i+1:GenConfig.NumTriple)
		CT=fltarr(GenConfig.NumTriple,2,MS,MP,/nozero)
		if i gt 0 then CT(0:i-1,*,*,*)=ComplTriple(0:i-1,*,*,*)
		CT(i:GenConfig.NumTriple-1,*,*,*)= $
			ComplTriple(i+1:GenConfig.NumTriple,*,*,*)
		ComplTriple=CT
		CTE=fltarr(GenConfig.NumTriple,2,MS,MP,/nozero)
		if i gt 0 then CTE(0:i-1,*,*,*)=ComplTripleErr(0:i-1,*,*,*)
		CTE(i:GenConfig.NumTriple-1,*,*,*)= $
			ComplTripleErr(i+1:GenConfig.NumTriple,*,*,*)
		ComplTripleErr=CTE
		TripleAmp=fltarr(GenConfig.NumTriple,MS,MP,/nozero)
		TripleAmpErr=fltarr(GenConfig.NumTriple,MS,MP,/nozero)
		TriplePhase=fltarr(GenConfig.NumTriple,MS,MP,/nozero)
		TriplePhaseErr=fltarr(GenConfig.NumTriple,MS,MP,/nozero)
		set_tripleampphase
	endif
	print,'Triple removed.'
endif
endrep until has lt 3 or GenConfig.NumTriple eq 0
if removed eq 0 then print,'Warning(TRIPLEPRODUCT): triple not found.'
;
ENDIF ELSE print,'***Error(TRIPLEPRODUCT): unknown action: ',action,'!'
;
end
;-------------------------------------------------------------------------------
pro unwraptriple
;
; Unwraps phase in TriplePhase as a function of wavelength. 
; Calls a C-program for fast execution.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if checkdata([8,11]) ne 0 then return
;
NTr=GenConfig.NumTriple
NCh=GenConfig.TripleNumChan
MTr=n_elements(TriplePhase(*,0,0))
MCh=max(GenConfig.NumSpecChan)
numpoint=n_elements(TriplePhase(0,0,*))
;
status=linknload(!external_lib,'unwraptriple',TriplePhase,TriplePhaseErr, $
	NTr,NCh,MTr,MCh,numpoint)
if status eq 0 then print,'Unwrapping complete.' $
	else print,'***Error(UNWRAPTRIPLE): UNWRAPTRIPLE bombs!'
;
end
;-------------------------------------------------------------------------------
pro unwrapphase
;
; Unwraps phase in VisPhase as a function of wavelength. 
; Calls a C-program for fast execution.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if checkdata([8,11]) ne 0 then return
;
; Actual array dimensions
NOB=n_elements(VisPhase(*,0,0,0))
NCH=n_elements(VisPhase(0,*,0,0))
NBL=n_elements(VisPhase(0,0,*,0))
numbaseline=GenConfig.NumBaseline
numchannel =GenConfig.NumSpecChan
numpoint=n_elements(VisPhase(0,0,0,*))
;
status=linknload(!external_lib,'unwrapphase',VisPhase,VisPhaseErr, $
	NOB,NBL,NCH,numbaseline,numchannel,numpoint)
if status eq 0 then print,'Unwrapping complete.' $
	else print,'***Error(UNWRAP): UNWRAP bombs!'
;
end
;-------------------------------------------------------------------------------
pro flagphase,beams,baselines,channels=channels
;
; Flag specified channels in visphase. Default if undefined.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if n_elements(beams) eq 0 then beems=indgen(GenConfig.NumOutBeam) $
			  else beems=beams-1
;
FOR i=0,n_elements(beems)-1 DO BEGIN
;
if beems(i) lt 0 or beems(i) ge GenConfig.NumOutBeam then begin
	print,'***Error(FLAGPHASE): invalid beam selection!'
	return
endif
;
; Get default channels first, then check for user input
chans=channelindex(genconfig.spectrometerid(beems(i)),/bad)-1
if n_elements(channels) gt 0 then begin
	s=size(channels)
	if s(0) le 1 then chans=channels(where(channels gt 0))-1 $
		     else chans=channels(beems(i),where(channels(beems(i),*) $
						  gt 0))-1
endif
if n_elements(baselines) eq 0 then bl=indgen(genconfig.numbaseline(beems(i))) $
			      else bl=baselines-1
if min(chans) ge 0 then begin
visphaseerr(beems(i),chans,bl,*)=-abs(visphaseerr(beems(i),chans,bl,*))
;
print,'VisPhase flagged for channels '+retroparse(chans+1)+ $
      ' and baselines '+retroparse(bl+1)+ $
      ' in beam '+string(beems(i)+1,format='(i1)')+'.'
endif
;
ENDFOR
;
end
;-------------------------------------------------------------------------------
pro flagstations,limit
;
; Check the natcounts and decide which station was likely off
; source. Enter the results in the scantable.station field and
; flag the relevant data. This procedure replaces the cumbersome
; manual data entry in fixearlydays.
;
; If NAT data were not processed, this procedure will not edit.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
if n_elements(limit) eq 0 then limit=10
;
; If all natcounts are zero, then the corresponding data were not processed
if n_elements(natcounts) eq 0 then return
if max(natcounts) eq 0 then return
;
if n_elements(pointflagtable) eq 0 then begin
	create_pointflagtable
	count_station=0
endif else begin
	index=where(pointflagtable.reason eq 'Station',count_station)
	if count_station gt 0 then begin
;		The flagtable already has the corresponding entries
		flagpointdata,pointflagtable(index),flag=1
	endif
endelse
if count_station eq 0 then begin
	index=where(pointflagtable.time ge 0,count)
	id0=count
	id=id0
	reason='Station'
endif
;
for i=0,n_elements(scantable)-1 do begin
for j=0,genconfig.numsid-1 do begin
	if median(natcounts(j,rec0(i):recn(i))) lt limit then begin
		scantable(i).station(j)=0
		n=recn(i)-rec0(i)+1
		k=lonarr(n)
		item=k+11	; Use NATJitter to avoid flagging photonrate
		inbeam=k+j
		if count_station eq 0 then begin
		set_pointflagtable,reason,item,inbeam, $
			k,k,k,k,abs(pointtime(rec0(i):recn(i)))
		id=id+n
		endif
	endif
endfor
endfor
;
if count_station eq 0 then begin
if id gt id0 then flagpointdata,pointflagtable(id0:id-1),flag=1 $
	     else print,'No data flagged.'
endif
;
end
;-------------------------------------------------------------------------------
function fringeid,solutions,r0,rn,pt=pt
;
; Function to return nearest integer values for given fringe ID solutions
; from dispersion fits. The ID solutions should have unity spacing, but
; may be offset from zero by non-integer values due to phase offsets.
; The offsets are either computed from all solutions or on a scan-by-scan
; basis (if r0 and rn are defined). 
;
; If keyword "pt" is true assume identical fringe ID solutions due to
; phase tracking. This is now the default for intra-scan solutions.
;
if n_elements(pt) eq 0 then pt=1 else pt=pt gt 0
r=size(solutions)
if r(0) eq 1 then s=reform(solutions,1,r(1)) else s=solutions
r=size(s)
new_solutions=s
;
if n_elements(r0) eq 0 or n_elements(rn) eq 0 then begin
	r0=[0]
	rn=[r(2)-1]
endif
;
num_steps=10
step=1.0/num_steps
;
FOR k=0,r(1)-1 DO BEGIN
FOR l=0,n_elements(r0)-1 DO BEGIN
;
count_max=0
i_max=0
for i=0,num_steps+1 do begin
	ymin=-0.5+i*step 
	ymax=ymin+step
	index=where(s(k,r0(l):rn(l)) gt ymin-1 $
		and s(k,r0(l):rn(l)) lt ymax-1,count1)
	index=where(s(k,r0(l):rn(l)) gt ymin   $
		and s(k,r0(l):rn(l)) lt ymax  ,count2)
	index=where(s(k,r0(l):rn(l)) gt ymin+1 $
		and s(k,r0(l):rn(l)) lt ymax+1,count3)
	index=where(s(k,r0(l):rn(l)) gt ymin+2 $
		and s(k,r0(l):rn(l)) lt ymax+2,count4)
	count=count1+count2+count3+count4
	if count gt count_max then begin
		count_max=count
		i_max=i
	endif
endfor
;
offset=-0.5+i_max*step+step/2
new_s=s(k,r0(l):rn(l))-offset
new_offset=avg(new_s-nint(new_s))+offset
if pt then $
   new_solutions(k,r0(l):rn(l))=medianve(new_solutions(k,r0(l):rn(l))) else $
   new_solutions(k,r0(l):rn(l))=nint(s(k,r0(l):rn(l))-new_offset)+new_offset
;
ENDFOR
ENDFOR
;
return,reform(new_solutions)
;
end
;-------------------------------------------------------------------------------
pro dispinit,stars,class=class
;
; Initialize Grp-, Dry-, and WetDelay for specified stars. Initialize
; pointdata if class='point', or scandata if class='scan'. Default is 'point'.
; Re-allocate delays for class 'point' if number of data points has changed.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if not keyword_set(class) then class='POINT' else class=strupcase(class)
;
if class eq 'SCAN' or class eq 'ALL' then begin
if n_elements(scans) ne 0 then begin
	scans.grpdelay=scans.fdlpos
	scans.grpdelay(genconfig.refstation-1)=0
	scans.grpdelayerr=scans.fdlposerr
	scans.drydelay=scans.fdlpos
	scans.drydelay(genconfig.refstation-1)=0
	scans.drydelayerr=scans.fdlposerr
	scans.wetdelay=scans.fdlpos
	scans.wetdelay(genconfig.refstation-1)=0
	scans.wetdelayerr=scans.fdlposerr
endif
endif
;
if class eq 'POINT' or class eq 'ALL' then begin
if n_elements(PointTime) eq 0 then begin
	GrpDelay=0
	GrpDelayErr=1
	DryDelay=0
	DryDelayErr=1
	WetDelay=0
	WetDelayErr=1
	print,'Dispersion corrected delays initialized.'
endif else begin 
	if n_elements(GrpDelay) ne n_elements(PointTime) then begin
		if checkdata([8]) ne 0 then return
		GrpDelay=fltarr(GenConfig.NumSid,n_elements(PointTime))
		DryDelay=fltarr(GenConfig.NumSid,n_elements(PointTime))
		WetDelay=fltarr(GenConfig.NumSid,n_elements(PointTime))
		GrpDelayErr=fltarr(GenConfig.NumSid,n_elements(PointTime))+1
		DryDelayErr=fltarr(GenConfig.NumSid,n_elements(PointTime))+1
		WetDelayErr=fltarr(GenConfig.NumSid,n_elements(PointTime))+1
		print,'Dispersion corrected delays initialized.'
	endif else begin
		if n_elements(stars) eq 0 then list_stars,stars
		for i=0,n_elements(stars)-1 do begin
			for j=0,n_elements(StarId)-1 do begin
				if StarId(j) eq stars(i) then begin
					GrpDelay(*,Rec0(j):RecN(j))=0
					DryDelay(*,Rec0(j):RecN(j))=0
					WetDelay(*,Rec0(j):RecN(j))=0
					GrpDelayErr(*,Rec0(j):RecN(j))=1
					DryDelayErr(*,Rec0(j):RecN(j))=1
					WetDelayErr(*,Rec0(j):RecN(j))=1
				endif
			endfor
		endfor
		print,'Dispersion corrected delays initialized for stars: ', $
			stars,'.'
	endelse
endelse
endif
;
end
;-------------------------------------------------------------------------------
pro dispcorr_old,solutions_out,dw=dw,pt=pt
;
; Calculate dispersion correction to station delays. Input data are unwrapped
; visibility phases from a coherent integration. All baselines are used to
; derive station based solutions of vacuum delay and atmospheric parameters.
; If solutions_in is defined, the phase wrap solution will be enforced in the
; new solutions.
;
; The phase is written as: phi/2pi = c + s*d0 + s*A(s)*d1 + s*B(s)*d2,
; where s=1/lambda, c is a baseline dependent phase offset to handle closure
; phase offsets, d0 is the vacuum delay, d1 is the dry air delay, and d2 is
; the wet air delay.
;
; The design matrix is set up in the following way:
; c1  c2  c3  d1  d2  d3  a1  a2  a3  b1  b2  b3
; ----------------------------------------------
; 1   0   0   s  -s   0   sa -sa  0   sb -sb  0
; ....for every channel on this baseline
; 0   1   0   s   0  -s   sa  0  -sa  sb  0  -sb
; ....for every channel on this baseline
; 0   0   1   0   s  -s   0   sa -sa  0   sb -sb
; ....for every channel on this baseline
;
; The right hand side is just phi/2pi. This is done for every sample.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if checkdata([8,11]) ne 0 then return
;
if n_elements(dw) eq 0 then dw=0 else dw=dw gt 0 & dp=1
if n_elements(pt) eq 0 then pt=0 else pt=pt gt 0
;
print,'DispCorr begins...'
;
; Set up the design matrix
nsd=GenConfig.NumSid
nph=long(total(GenConfig.NumBaseline(0:GenConfig.NumOutBeam-1)));
nrow=long(total(GenConfig.Numbaseline(0:GenConfig.NumOutBeam-1) $
               *GenConfig.NumSpecChan(0:GenConfig.NumOutBeam-1)))+3+nph
ncol=nsd*3+nph				; 3 solutions: grp, dry, wet
m=make_array(nrow,ncol,/double,/nozero)	; design matrix
r=make_array(nrow,/double,/nozero)	; right hand side
f=make_array(nrow,/double,/nozero)	; to reconstruct unweighted phases
;
; Initialize data
numpoint=n_elements(VisPhase(0,0,0,*))
GrpDelay=fltarr(nsd,numpoint)
DryDelay=fltarr(nsd,numpoint)
WetDelay=fltarr(nsd,numpoint)
GrpDelayErr=fltarr(nsd,numpoint)-1
DryDelayErr=fltarr(nsd,numpoint)-1
WetDelayErr=fltarr(nsd,numpoint)-1
;
GrpResidual=fltarr(2,numpoint)	; group-delay residuals, mean and STDEV
DryResidual=fltarr(2,numpoint)	; dry-air delay res., mean and STDEV
WetResidual=fltarr(2,numpoint)	; moist-air delay res., mean and STDEV
;
index=where(VisPhaseErr gt 0,n)
if n gt 0 then w_avg=avg(1.d0/(VisPhaseErr(index)/(2*pi_circle))) else return
;	       w_avg=10.d0; This could be more stable
;
solutions_out=dblarr(ncol,numpoint)
ill_condition=intarr(numpoint)
;
computation:
for k=0L,numpoint-1 do begin
	ph=0		; index for phase wrap solutions
	tcount=0
	m(*,*)=0.d0
	r(*)=0.d0
	f(*)=0.d0
	for i=0,GenConfig.NumOutBeam-1 do begin
	  nch=GenConfig.NumSpecChan(i)
	  nbl=GenConfig.NumBaseline(i)
	  s=1.d0/(GenConfig.Wavelength(0:nch-1,i)*1.d6)
	  a=a_disp(0)+a_disp(1)/(a_disp(2)-s^2)+a_disp(3)/(a_disp(4)-s^2)
	  a=a/10000.d0	; We want coefficients of similar magnitude
	  b=b_disp(0)+b_disp(1)*s^2
	  for j=0,nbl-1 do begin
	    index=where(VisPhaseErr(i,0:nch-1,j,k) gt 0,count)
	    if count gt 0 then begin
	      se=s(index)
	      ae=a(index)
	      be=b(index)
;	      p=VisPhase(i,index,j,k)+GeoDelay(i,j,k)*se*2*pi_circle
	      p=VisPhase(i,index,j,k)
;	      Note that here wt should NOT be 1/error^2
	      w=1.d0/(VisPhaseErr(i,index,j,k)/(2*pi_circle))
;	      w=1.d0; This could be more stable
	      m(tcount:tcount+count-1,ph)=w
	      sid1=strmid(GenConfig.BaselineId(j,i),0,3)
	      i1=where(GenConfig.StationId eq sid1)
	      m(tcount:tcount+count-1,nph+i1(0)+nsd*0)=w*se
	      m(tcount:tcount+count-1,nph+i1(0)+nsd*1)=w*se*ae
	      m(tcount:tcount+count-1,nph+i1(0)+nsd*2)=w*se*be
	      sid2=strmid(GenConfig.BaselineId(j,i),4,3)
	      i2=where(GenConfig.StationId eq sid2)
	      m(tcount:tcount+count-1,nph+i2(0)+nsd*0)=-w*se
	      m(tcount:tcount+count-1,nph+i2(0)+nsd*1)=-w*se*ae
	      m(tcount:tcount+count-1,nph+i2(0)+nsd*2)=-w*se*be
	      r(tcount:tcount+count-1)=p*w/(2*pi_circle)
	      f(tcount:tcount+count-1)=w/(2*pi_circle)
	      tcount=tcount+count
	    endif
	    ph=ph+1
	  endfor
	endfor
;
	IF tcount GT 0 THEN BEGIN
;
;	No correction; determine group delay only
	if dp eq 1 then begin
		me=m(0:tcount+0,0:nph+nsd*1-1)
		re=r(0:tcount+0)
	fe=f(0:tcount-1)
	for l=0,0 do me(tcount+l,nph+l*nsd)=w_avg	; remove degeneracy d1
	t=transpose(me)
	n=t#me						; normal matrix
	v=t#re						; right hand side
;	Test on singularity
	svd,n,w
	small=where(w lt max(w)*1.0e-8,count)
	if count eq 0 then begin
		in=invert(n,status)
		s=in#v					; direct solution
;		Reconstruct unweighted data and obtain residuals
		re=re(0:tcount-1)/fe
		pe=(me#s) & pe=pe(0:tcount-1)/fe
		std=stdev(sqrt((sin(re)-sin(pe))^2+(cos(re)-cos(pe))^2),mean)
		GrpResidual(0,k)=mean
		GrpResidual(1,k)=std
;		Copy the solution
		GrpDelay(*,k)=s(nph:nph+nsd-1)
		for l=0,nsd-1 do GrpDelayErr(l,k)=sqrt(in(nph+l,nph+l))
;			 	 GrpDelayErr(*,k)=std;	more appropriate?
	endif else ill_condition(k)=k+1
	endif
;
;	Correction with dry air dispersion only
	if dp eq 1 then begin
		me=m(0:tcount+1,0:nph+nsd*2-1)
		re=r(0:tcount+1)
	endif else begin
;		Extract nph more rows to set their values
		me=m(0:tcount+1+nph,0:nph+nsd*2-1)
		re=r(0:tcount+1+nph)
		for l=0,nph-1 do begin
			me(tcount+2+l,l)=w_avg
			re(tcount+2+l)=solutions_in(l,k)*w_avg
		endfor
	endelse
	fe=f(0:tcount-1)
	for l=0,1 do me(tcount+l,nph+l*nsd)=w_avg	; degeneracies d1,a1
	t=transpose(me)
	n=t#me
	v=t#re
	if ill_condition(k) eq 0 then begin
		in=invert(n,status)
		s=in#v
		solutions_out(0:nph+nsd*2-1,k)=s
;		Reconstruct unweighted data and obtain residuals
		re=re(0:tcount-1)/fe
		pe=(me#s) & pe=pe(0:tcount-1)/fe
		std=stdev(sqrt((sin(re)-sin(pe))^2+(cos(re)-cos(pe))^2),mean)
		DryResidual(0,k)=mean
		DryResidual(1,k)=std
;		Copy the solution
		DryDelay(*,k)=s(nph:nph+nsd-1)
		for l=0,nsd-1 do DryDelayErr(l,k)=sqrt(in(nph+l,nph+l))
;			 	 DryDelayErr(*,k)=std;	more appropriate?
	endif
;
;      	Correction including water vapor, very noisy!
	if dp eq 3 then begin
		me=m(0:tcount+2,*)
		re=r(0:tcount+2)
	fe=f(0:tcount-1)
	for l=0,2 do me(tcount+l,nph+l*nsd)=w_avg	; degeneracies d1,a1,b1
	t=transpose(me)
	n=t#me
	v=t#re
	svdc,n,w,u_array,v_array
	small=where(w lt max(w)*1.0e-8,count)
	if count gt 0 then begin
		w(small)=0
		s=svsol(u_array,w,v_array,v)
		in=n
		um=n*0 & um(indgen(ncol),indgen(ncol))=1
		for l=0,ncol-1 do in(*,l)=svsol(u_array,w,v_array,um(*,l))
	endif else begin
		in=invert(n,status)
		s=in#v
	endelse
;	Reconstruct unweighted data and obtain residuals
	re=re(0:tcount-1)/fe
	pe=(me#s) & pe=pe(0:tcount-1)/fe
	std=stdev(sqrt((sin(re)-sin(pe))^2+(cos(re)-cos(pe))^2),mean)
	WetResidual(0,k)=mean
	WetResidual(1,k)=std
;	Copy the solution
	WetDelay(*,k)=s(nph:nph+nsd-1)
	for l=0,nsd-1 do WetDelayErr(l,k)=sqrt(in(nph+l,nph+l))
;			 WetDelayErr(*,k)=std;	more appropriate?
	endif
;
	ENDIF
endfor
;
if dw and dp eq 1 then begin
	solutions_in=solutions_out
	solutions_in(0:nph-1,*)= $
		fringeid(solutions_in(0:nph-1,*),Rec0,RecN,pt=pt)
	dp=2
	goto,computation
endif
;
index=where(ill_condition ne 0,count)
if count gt 0 then $
print,'Warning(DISPCORR): matrix ill conditioned for points ', $
	retroparse(ill_condition(index))
;
print,'Finished dispersion correction.'
;
end
;-------------------------------------------------------------------------------
pro dispcorr,cl=cl,wt=wt
;
; New version!
;
; Calculate dispersion corrections to station delays. Input data are unwrapped
; visibility phases from a coherent integration. All baselines are used to
; derive station based solutions of vacuum delay and atmospheric parameters.
;
; The phase is written as: phi/2pi = c + s*d0 + s*A(s)*d1 + s*B(s)*d2,
; where s=1/lambda, c is a baseline dependent phase offset to handle unequal
; beam combiner paths, d0 is the vacuum delay, d1 is the dry air path, and d2
; is the wet air path.
;
; Solutions for c should be multi-modal, i.e. they should have a spacing
; of unity. This is enforced in a second iteration.
;
; This procedure enforces a zero closure group delay. This is done either
; by looking only at the reference baseline results (cl=0) or by adding
; mulitple paths with the proper closure phase offsets included.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if checkdata([8,11]) ne 0 then return
;
if n_elements(cl) eq 0 then cl=0 else cl=cl gt 0
if n_elements(wt) eq 0 then wt=0 else wt=wt gt 0
;
; Set up the design matrix
nsd=GenConfig.NumSid
;
; Initialize data
numpoint=n_elements(VisPhase(0,0,0,*))
GrpDelay=fltarr(nsd,numpoint)
DryDelay=fltarr(nsd,numpoint)
WetDelay=fltarr(nsd,numpoint)
GrpDelayErr=fltarr(nsd,numpoint)-1
DryDelayErr=fltarr(nsd,numpoint)-1
WetDelayErr=fltarr(nsd,numpoint)-1
;
; Baseline-based calibrated delays
grpdelays=fltarr(genconfig.numoutbeam, $
		 max(genconfig.numbaseline(0:genconfig.numoutbeam-1)), $
		 numpoint)
drydelays=fltarr(genconfig.numoutbeam, $
		 max(genconfig.numbaseline(0:genconfig.numoutbeam-1)), $
		 numpoint)
;
; Reference station
refstation=genconfig.stationid(genconfig.refstation-1)
;
; Integer fringe ID solutions
fidsol=fltarr(n_elements(StarID))
;
; Prepare plot window
if !d.name ne 'PS' then begin
	window,/free
	set_screen
endif
!p.multi=[0,genconfig.numoutbeam,max(genconfig.numbaseline)]
!p.charsize=2.0
;
data_check=0
;
for i=0,GenConfig.NumOutBeam-1 do begin
for l=0,GenConfig.NumBaseline(i)-1 do begin
;
;	Find out if baseline with reference
	baselineid=genconfig.baselineid(l,i)
	pos=strpos(baselineid,refstation)
	sid=-1
	if pos eq 0 then $
	sid=where(genconfig.stationid eq strmid(baselineid,4,3)) $
		    else if pos gt 0 then $
	sid=where(genconfig.stationid eq strmid(baselineid,0,3))
	sid=sid(0)
;
;	Find out which channels are flagged
	pe=VisPhaseErr(i,*,l,*)
	index=where(pe le 0,count)
	if count gt 0 then pe(index)=0
	chi=where(total(abs(pe),4) ne 0,nch)
;
if nch gt 0 then begin
	data_check=1
;	Initialize coefficients
	s=1.d0/(GenConfig.Wavelength(chi,i)*1.d6)
	a=a_disp(0)+a_disp(1)/(a_disp(2)-s^2)+a_disp(3)/(a_disp(4)-s^2)
	a=a/10000.d0	; We want coefficients of similar magnitude
	b=b_disp(0)+b_disp(1)*s^2
;
;	Group delays (do not use weights)
	y=reform(VisPhase(i,chi,l,*)/(2*!pi))
	m=dblarr(nch,2)
	m(*,0)=s
	m(*,1)=1
	tm=transpose(m)
	n=tm#m
	r=tm#y
	v=invert(n)#r
	grpdelays(i,l,*)=v(0,*)-SoftDelay(i,l,*)*1e6
	if sid ge 0 then grpdelayerr(sid,*)=1
;
;	Dry air solution, weighted for fringe ID solution
	fidsol=fltarr(n_elements(StarId))
	for k=0,n_elements(StarId)-1 do begin
;		Initialize data
		e=reform(VisPhaseErr(i,chi,l,rec0(k):recn(k)))/(2*!pi)
		w=e & for j=0,nch-1 do w(j,*)=1.0/median(e(j,*))
		if wt eq 0 then w(*)=1
		y=reform(VisPhase(i,chi,l,rec0(k):recn(k))/(2*!pi))*w
		w=reform(w(*,0))
		m=dblarr(nch,3)
		m(*,0)=s*w
		m(*,1)=s*a*w
		m(*,2)=w
		tm=transpose(m)
		n=tm#m
		r=tm#y
		v=invert(n)#r
		fidsol(k)=medianve(v(2,*))
	endfor
;
;	Plot fringe ID results
	indexs=where(starid ne 'FKV0000',counts)
	if counts gt 0 then $
	plot,indexs,fidsol(indexs),psym=1, $
		title='OB '+string(i+1)+', BL '+string(l+1), $
		xtitle='Scan ID',ytitle='Fringe ID'
	indexw=where(starid eq 'FKV0000',countw)
	if countw gt 0 and counts gt 0 then $
	oplot,indexw,fidsol(indexw),psym=1,color=tci(2)
        if counts eq 0 and countw gt 0 then $
        plot,indexw,fidsol(indexw),psym=1, $
                title='OB '+string(i+1)+', BL '+string(l+1)
;
;	Enforce integer fringe ID solution
	if counts eq 0 then indexs=lindgen(countw)
;	Set pt=0 since different scans can have different fringe IDs
	fidsol(indexs)=fringeid(fidsol(indexs),pt=0)
;
;	Compute final dry air delay solution
	for k=0,n_elements(StarId)-1 do begin
;		Initialize data
		e=reform(VisPhaseErr(i,chi,l,rec0(k):recn(k)))/(2*!pi)
		w=e & for j=0,nch-1 do w(j,*)=1.0/median(e(j,*))
		if wt eq 0 then w(*)=1
		y=[reform(VisPhase(i,chi,l,rec0(k):recn(k))/(2*!pi))*w, $
		   fltarr(1,recn(k)-rec0(k)+1)]
		y(nch,*)=fidsol(k)*avg(w)
		w=reform(w(*,0))
		m=dblarr(nch+1,3)
		m(0:nch-1,0)=s*w
		m(0:nch-1,1)=s*a*w
		m(0:nch-1,2)=w
		m(nch,2)=avg(w)
		tm=transpose(m)
		n=tm#m
		r=tm#y
		in=invert(n)
		v=in#r
		oplot,v(2,*)*0+k,v(2,*),psym=3,color=tci(3)
		drydelays(i,l,rec0(k):recn(k))=v(0,*)-SoftDelay(i,l,rec0(k):recn(k))*1e6
		if sid ge 0 then drydelayerr(sid,rec0(k):recn(k))=sqrt(in(0,0))
	endfor
endif
;
endfor
endfor
;
if not data_check then begin
	print,'Error(DISPCORR): no valid data found!'
	return
endif
;
grpdelayerr(GenConfig.RefStation-1,*)=1
drydelayerr(GenConfig.RefStation-1,*)=1
;
; Transfer baseline solutions to stations
for i=0,GenConfig.NumOutBeam-1 do begin
for l=0,GenConfig.NumBaseline(i)-1 do begin
;
baselineid=genconfig.baselineid(l,i)
pos=strpos(baselineid,refstation)
if pos ge 0 and total(signof(grpdelayerr(i,l,*))) gt -numpoint then begin
;
; Basic solution from baseline involving reference station
if pos eq 0 then begin
	k=where(genconfig.stationid eq strmid(baselineid,4,3))
	k=k(0)
	sign=-1 
endif else begin
	k=where(genconfig.stationid eq strmid(baselineid,0,3))
	k=k(0)
	sign=+1
endelse
grpdelay(k,*)=grpdelays(i,l,*)*sign
drydelay(k,*)=drydelays(i,l,*)*sign
;
; Exploit closure relation
if cl then begin
for j=0,genconfig.numtriple-1 do begin
	m0=-1
	for m=0,2 do if genconfig.triplebeam(m,j) eq i $
	    	    and genconfig.triplebase(m,j) eq l then m0=m
	if m0 ge 0 then begin
		pe1=VisPhaseErr(genconfig.triplebeam(0,j),*, $
			        genconfig.triplebase(0,j),*)
		pe2=VisPhaseErr(genconfig.triplebeam(1,j),*, $
			        genconfig.triplebase(1,j),*)
		pe3=VisPhaseErr(genconfig.triplebeam(2,j),*, $
			        genconfig.triplebase(2,j),*)
		index=where(pe1 le 0,count)
		if count gt 0 then pe1(index)=0
		index=where(pe2 le 0,count)
		if count gt 0 then pe2(index)=0
		index=where(pe3 le 0,count)
		if count gt 0 then pe3(index)=0
		chi=where(total(abs(pe1),4) ne 0 $
      		      and total(abs(pe2),4) ne 0 $
      		      and total(abs(pe3),4) ne 0,nch)
		cde=fltarr(n_elements(rec0))
		cdt=fltarr(n_elements(rec0))
		msk=fltarr(numpoint)+1
		for n=0,n_elements(rec0)-1 do begin
			p=fltarr(nch)
			for jj=0,nch-1 do $
			p(jj)=cphase(complex( $
				avg(compltriple(j,0,chi(jj),rec0(n):recn(n))), $
				avg(compltriple(j,1,chi(jj),rec0(n):recn(n)))))
			p=phaseunwrap(p)*180/!pi
			x=genconfig.wavelength(chi,i)*1e6
			dm=dblarr(nch,3)
			dm(*,0)=evalfunction('P_0',x)
			dm(*,1)=evalfunction('P_1',x)
			dm(*,2)=evalfunction('P_2',x)
			tm=transpose(dm)
			nm=tm#dm
			r=tm#p
			s=invert(nm)#r
			cde(n)=(s(1)+s(2))/360
			cdt(n)=avg(pointtime(rec0(n):recn(n)))/3600
			if StarId(n) eq 'FKV0000' then msk(rec0(n):recn(n))=0
		endfor
		index=where(StarId ne 'FKV0000')
		cdt=cdt(index)
		cde=cde(index)
		mcde=medianve(cde,mcdee)
		print,'Triple #',j+1,' median value CDE, error [microns]: ', $
			mcde,mcdee
		index=where(abs(cde-mcde) le 3*mcdee)
		cdt=cdt(index)
		cde=cde(index)
		case m0 of
			0:sign=-1
			1:sign=+1
			2:sign=-1
		endcase
		drydelay(k,*)=(drydelay(k,*)+msk*sign*( $
			     -drydelays(genconfig.triplebeam(0,j), $
					genconfig.triplebase(0,j),*)*(m0 ne 0) $
		   	     +drydelays(genconfig.triplebeam(1,j), $
					genconfig.triplebase(1,j),*)*(m0 ne 1) $
		   	     -drydelays(genconfig.triplebeam(2,j), $
					genconfig.triplebase(2,j),*)*(m0 ne 2) $
		   	     +poly(pointtime/3600,poly_fit(cdt,cde,2))))/2
	endif
endfor
endif
;
endif
;
endfor
endfor
;
print,'Finished dispersion correction.'
;
end
;-------------------------------------------------------------------------------
pro metrocorr
;
; Use the INCHWORM solution data in ParX,-Y,-Z to compute delay correction.
; Remember that the metrology data is in units of microns. Store in MetroPos.
;
common StarBase,startable,notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if checkdata([3,8,11,16,21]) ne 0 then return
;
print,'MetroCorr begins...'
;
RAD=180/pi_circle
;
; Do we want a plot?
do_plot=0
;
; Allocate data
MetroPos=fltarr(GenConfig.NumSid,n_elements(PointTime))
MetroPosErr=fltarr(GenConfig.NumSid,n_elements(PointTime))-1
;
metrotimes=abs(MetroTime)
pointtimes=abs(PointTime)
;
; Go through available point data one scan at a time
FOR i=0,n_elements(Iscan)-1 DO BEGIN
;
index=where(metrotimes ge pointtimes(Rec0(i)) $
	and metrotimes le pointtimes(RecN(i)),count)
;
if count eq 0 then begin
	print,'Warning(METROCORR): no data for scan ',Iscan(i),'!'
	goto,DONEXT
endif
;
times=MetroTime(index)
solx=ParX(*,index)
soly=ParY(*,index)
solz=ParZ(*,index)
solxerr=ParXErr(*,index)
solyerr=ParYErr(*,index)
solzerr=ParZErr(*,index)
;
index=where(times gt 0,count)
;
if count eq 0 then begin
	print,'Warning(METROCORR): no data for scan ',Iscan(i),'!'
	goto,DONEXT
endif
times=times(index)
solx=solx(*,index)
soly=soly(*,index)
solz=solz(*,index)
solxerr=solxerr(*,index)
solyerr=solyerr(*,index)
solzerr=solzerr(*,index)
;
stars=strarr(count) & stars(*)=ScanTable(Iscan(i)-1).starid
;
; Compute apparent star positions
topostar,times,stars,startable,ra_app,dec_app
;
; Compute UT1
ut1=utc2ut1(times)
;
; Compute Greenwich apparent sidereal time
gst=ut12gst(times,ut1)
;
; Compute hour angle
ha=hourangle(gst,ra_app)
;
for j=0,GenConfig.NumSid-1 do begin
;
; 	Deal with FKV0000, i.e. the white light source, alias Alpha Lab
	index=where(stars eq 'FKV0000',count_white)
	if count_white gt 0 then begin
		hadec=equatorial2hadec( $
		      horizon2equatorial( $
		      azel2horizon(MetroConfig.SidModel.FeedbeamAng(*,j))))
		ha(index)=hadec(0)	; Should be about 5.0
		dec_app(index)=hadec(1)	; Should be about 11.7
	endif
;
	index=where(solxerr(j,*) gt 0,count)
	if count gt 1 then begin
	t=times(index)
	d=catseyedelay(MetroConfig.SidModel.FeedbeamAng(*,j), $
		       solx(j,index),soly(j,index),solz(j,index), $
		       ha(index),dec_app(index), $
		       MetroConfig.SidModel.CatsEyeOff(j), $
		       solxerr(j,index),solyerr(j,index),solzerr(j,index),e)
	if count_white gt 0 then begin
		stars_e=stars(index)
		index=where(stars_e eq 'FKV0000',count)
		if count gt 0 then begin
			d(index)=d(index)/2
			e(index)=e(index)/2
		endif
	endif
	w=1/e^2
	r=poly_fit(t,d,1,measure_errors=e,yfit=yft,yband=ybd,sigma=sig,covar=mat)
;	MetroPos(j,Rec0(i):RecN(i))=poly(pointtimes(Rec0(i):RecN(i)),r)
	MetroPos(j,Rec0(i):RecN(i))=spline(t,d,pointtimes(Rec0(i):RecN(i)))
	MetroPosErr(j,Rec0(i):RecN(i))=sig
;
	if do_plot then begin
	plot,pointtimes(rec0(i):recn(i))/3600,MetroPos(j,rec0(i):recn(i)), $
		xtitle='UT [h]',ytitle='c [microns], Sid'+string(j+1), $
		title='Scan'+string(Iscan(i)),psym=3,charsize=1.5
	oplot,t/3600,d,psym=1
	oploterr,t/3600,d,e,3
	xyouts,0.01,0.01,'Sigma='+string(sig),/normal
	wait,1
	endif
;
	endif
endfor
;
DONEXT:
ENDFOR
;
; Convert to "pos" format
for j=0,GenConfig.NumSid-1 do begin
	if j ne GenConfig.RefStation-1 then $
	MetroPos(j,*)=MetroPos(j,*)-MetroPos(GenConfig.RefStation-1,*)
endfor
;
print,'Finished MetroCorr computation.'
;
end
;-------------------------------------------------------------------------------
pro astrocorr
;
; Use the INCHWORM delay solution data to compute corrections.
; Note: results store in MetroPos, a variable shared with metrocorr, since
; both routines should give the same answer.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if checkdata([3,8,11,16]) ne 0 then return
;
print,'AstroCorr begins...'
;
; Do we want a plot?
do_plot=0
;
; Allocate data
MetroPos=fltarr(GenConfig.NumSid,n_elements(PointTime))
MetroPosErr=fltarr(GenConfig.NumSid,n_elements(PointTime))-1
;
metrotimes=abs(MetroTime)
pointtimes=abs(PointTime)
;
; Go through available point data one scan at a time
FOR i=0,n_elements(Iscan)-1 DO BEGIN
;
index=where(metrotimes ge pointtimes(Rec0(i)) $
	and metrotimes le pointtimes(RecN(i)),count)
;
if count eq 0 then begin
	print,'Warning(ASTROCORR): no data for scan ',Iscan(i),'!'
	goto,DONEXT
endif
;
times=MetroTime(index)
delay=MetroPath(*,index)
delayerr=MetroPathErr(*,index)
;
index=where(times gt 0,count)
;
if count eq 0 then begin
	print,'Warning(ASTROCORR): no data for scan ',Iscan(i),'!'
	goto,DONEXT
endif
times=times(index)
delay=delay(*,index)
delayerr=delayerr(*,index)
;
for j=0,GenConfig.NumSid-1 do begin
	d=delay(j,*)
	e=delayerr(j,*)
	index=where(e gt 0,count)
	if count gt 1 then begin
	w=1/e(index)^2
	r=poly_fit(times(index),d(index),1,measure_errors=e,yfit=yft,yband=ybd,sigma=sig,covar=mat)
;	MetroPos(j,Rec0(i):RecN(i))=poly(pointtimes(Rec0(i):RecN(i)),r)
	MetroPos(j,Rec0(i):RecN(i))=spline(times(index),d(index),pointtimes(Rec0(i):RecN(i)))
	MetroPosErr(j,Rec0(i):RecN(i))=sig
;
	if do_plot then begin
	plot,pointtimes(rec0(i):recn(i))/3600,MetroPos(j,rec0(i):recn(i)), $
		xtitle='UT [h]',ytitle='c [microns], Sid'+string(j+1), $
		title='Scan'+string(Iscan(i)),psym=3,charsize=1.5
	oplot,times(index)/3600,d(index),psym=1
	oploterr,times(index)/3600,d(index),e,3
	xyouts,0.01,0.01,'Sigma='+string(sig),/normal
	wait,1
	endif
;
	endif
endfor
;
DONEXT:
ENDFOR
;
; Convert to "pos" format
for j=0,GenConfig.NumSid-1 do begin
	if j ne GenConfig.RefStation-1 then $
	MetroPos(j,*)=MetroPos(j,*)-MetroPos(GenConfig.RefStation-1,*)
endfor
;
print,'Finished AstroCorr computation.'
;
end
;-------------------------------------------------------------------------------
pro metroinit
;
; Initialize MetroPos(PointData).
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if n_elements(PointTime) eq 0 then begin
	MetroPos=0
	MetroPosErr=1
endif else begin 
	if checkdata([8]) ne 0 then return
	MetroPos=fltarr(GenConfig.NumSid,n_elements(PointTime))
	MetroPosErr=fltarr(GenConfig.NumSid,n_elements(PointTime))+1
endelse
;
print,'MetroPos initialized.'
;
end
;-------------------------------------------------------------------------------
pro inchav
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; Check presence of data
if checkdata([8,9,16,21]) ne 0 then return
;
for k=0,n_elements(scans)-1 do begin
;
kk=scans(k).Iscan-1
for i=0,GenConfig.NumSid-1 do begin
;
	if Sol0(kk) ge 0 and SolN(kk) ge 0 then begin
		x=ParX(i,Sol0(kk):SolN(kk))
		xerr=ParXErr(i,Sol0(kk):SolN(kk))
		y=ParY(i,Sol0(kk):SolN(kk))
		yerr=ParYErr(i,Sol0(kk):SolN(kk))
		z=ParZ(i,Sol0(kk):SolN(kk))
		zerr=ParZErr(i,Sol0(kk):SolN(kk))
		j=where(xerr gt 0 and yerr gt 0 and zerr gt 0,count)
		if count gt 0 then begin
			scans(kk).ParX(i)=avg(x(j))
			scans(kk).ParY(i)=avg(y(j))
			scans(kk).ParZ(i)=avg(z(j))
			if count gt 1 then begin
				scans(kk).ParXErr(i)=stdev(double(x(j)))/sqrt(count)
				scans(kk).ParYErr(i)=stdev(double(y(j)))/sqrt(count)
				scans(kk).ParZErr(i)=stdev(double(z(j)))/sqrt(count)
			endif else begin
				scans(kk).ParXErr(i)=ParXErr(j)
				scans(kk).ParYErr(i)=ParYErr(j)
				scans(kk).ParZErr(i)=ParZErr(j)
			endelse
		endif else begin
			scans(kk).ParX(i)=0
			scans(kk).ParXErr(i)=-1
			scans(kk).ParY(i)=0
			scans(kk).ParYErr(i)=-1
			scans(kk).ParZ(i)=0
			scans(kk).ParZErr(i)=-1
		endelse
	endif else begin
		scans(kk).ParX(i)=0
		scans(kk).ParXErr(i)=-1
		scans(kk).ParY(i)=0
		scans(kk).ParYErr(i)=-1
		scans(kk).ParZ(i)=0
		scans(kk).ParZErr(i)=-1
	endelse
;		
endfor
;
endfor
;
print,'Finished averaging.'
;
end
;-------------------------------------------------------------------------------
function solindex,t
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
;
r=t*0
for i=0,n_elements(sol0)-1 do begin
	index=where(t gt metrotime(sol0(i)) and t lt metrotime(soln(i)),count)
	if count gt 0 then r(index)=1
endfor
;
return,r
;
end
;-------------------------------------------------------------------------------
function lmpivot,x,v,init=init,sid=sid
;
; To be used with lmfit to fit pivot coordinates to the XYZ data.
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common LocalLMPivot,model,hadec
;
if n_elements(init) eq 0 then init=0
;
if init then begin
	v=[0.,0.,0.,0.]
	index=where(scantable.starid ne 'FKV0000' and sol0 ne -1,n)
	x=fltarr(n)
	y=fltarr(n)
	z=fltarr(n)
	m=fltarr(2,n)
;
	i=sid-1
	l=0
	for k=0,n-1 do begin
		j=where(parxerr(i,sol0(index(k)):soln(index(k))) gt 0,count)
		if count gt 0 then begin
			x(l)=avg(parx(i,j+sol0(index(k))))
			y(l)=avg(pary(i,j+sol0(index(k))))
			z(l)=avg(parz(i,j+sol0(index(k))))
			m(*,l)=avg(motorangle(*,i,j+sol0(index(k))),2)
			l=l+1
		endif
	endfor
	if l gt 0 then begin
		dpx=x(1:l-1)-x(0:l-2)
		dpy=y(1:l-1)-y(0:l-2)
		dpz=z(1:l-1)-z(0:l-2)
		model=sidmodel(i+1)
		hadec=sidpointing(model,m(*,0:l-1),1)
		x=findgen(n_elements(dpx)*3)
	endif else begin
		dpx=0
		dpy=0
		dpz=0
	endelse
	return,[dpx,dpy,dpz]
endif
;
n=n_elements(hadec(0,*))-1
j=nint(x) mod n
k=nint(x) / n
;
; Compute model value
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v)
dmp=mp(1,k)-mp(0,k)
;
; Compute partial derivatives
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v+[1,0,0,0])
dmpxh=mp(1,k)-mp(0,k)
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v-[1,0,0,0])
dmpxl=mp(1,k)-mp(0,k)
;
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v+[0,1,0,0])
dmpyh=mp(1,k)-mp(0,k)
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v-[0,1,0,0])
dmpyl=mp(1,k)-mp(0,k)
;
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v+[0,0,1,0])
dmpzh=mp(1,k)-mp(0,k)
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v-[0,0,1,0])
dmpzl=mp(1,k)-mp(0,k)
;
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v+[0,0,0,1])
dmpoh=mp(1,k)-mp(0,k)
mp=mirrorpivot(model,hadec(0,j:j+1),hadec(1,j:j+1),v-[0,0,0,1])
dmpol=mp(1,k)-mp(0,k)
;
return,[dmp,(dmpxh-dmpxl)/2,(dmpyh-dmpyl)/2,(dmpzh-dmpzl)/2,(dmpoh-dmpol)/2]
;
end
;-------------------------------------------------------------------------------
pro pivotfit,order=order
;
; Fit mean pivot and axis offset vectors to XYZ data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; Check presence of data
if checkdata([1,8,16,21]) ne 0 then return
;
if n_elements(order) eq 0 then order=intarr(GenConfig.NumSid)+3
if n_elements(order) eq 1 then order=replicate(order(0),GenConfig.NumSid)
;
PivotMotion=fltarr(n_elements(MetroTime),3,GenConfig.NumSid)
;
for i=0,GenConfig.NumSid-1 do begin
	y=lmpivot(x,a,/init,sid=i+1)
	if total(abs(y)) eq 0 then conv=0 else $
	r=lmfit(x,y,a,function_name='lmpivot',/double,conv=conv,sigma=e)
	if conv eq 1 then begin
		MetroConfig.Sidmodel.PivotOffset(*,i)=a
		MetroConfig.Sidmodel.PivotOffsetErr(*,i)=e
		print,'Siderostat ',i+1,' RMS [microns] = ',stdev(y-r)
;		Fit mean pivot motion
		index=where(parxerr(i,*) gt 0 and solindex(metrotime))
		hadec=sidpointing(sidmodel(i+1),motorangle(*,i,index),1)
		mp=mirrorpivot(sidmodel(i+1),hadec(0,*),hadec(1,*),a)
;		Polynomial
		PivotMotion(*,0,i)=polynom(MetroTime/60, $
			polyfit(MetroTime(index)/60,parx(i,index)-mp(*,0),order(i)))
		PivotMotion(*,1,i)=polynom(MetroTime/60, $
			polyfit(MetroTime(index)/60,pary(i,index)-mp(*,1),order(i)))
		PivotMotion(*,2,i)=polynom(MetroTime/60, $
			polyfit(MetroTime(index)/60,parz(i,index)-mp(*,2),order(i)))
	endif
endfor
;
print,'Pivot coordinates fit.'
;
end
;-------------------------------------------------------------------------------
function lmmotor,x,v,init=init,sid=sid,ft=ft,rh=rh,bp=bp,times=times
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common LocalLMMotor,ma,tm,vcorr
;
if n_elements(init) eq 0 then init=0
if n_elements(ft) eq 0 then ft=56.3
if n_elements(rh) eq 0 then rh=33.0
if n_elements(bp) eq 0 then bp=23.316
lambda=0.650
;
RAD=180/!pi
;
if init then begin
;
	stars=strarr(n_elements(metrotime))
	for i=0,n_elements(scantable)-1 do $
		if scantable(i).starid ne 'FKV0000' then $
			stars(sol0(i):soln(i))=scantable(i).starid
	index=where(strlen(stars) ne 0)
	stars=stars(index)
	times=abs(metrotime(index))
	ma=reform(motorangle(*,sid-1,index))
	mae=reform(motorangleerr(*,sid-1,index))
	index=where(mae(0,*) gt 0)
	stars=stars(index)
	times=times(index)
	ma=ma(*,index)
	mae=mae(*,index)
	topostar,times,stars,startable,ra_app,dec_app
	ut1=utc2ut1(times)
	gst=ut12gst(times,ut1)
	ha=hourangle(gst,ra_app)
	hadec=refraction(natural(ha,dec_app),ft=ft,rh=rh,bp=bp)
	hadec(0,*)=hadec(0,*)*15	; convert to degrees
;
	tm=times
	v=sidmodel(sid)
	vcorr=reform(feedvector(sid-1,*,*))
	y=reform(transpose(hadec),n_elements(hadec))
	x=dindgen(n_elements(y))
	return,y
;
endif
;
n=n_elements(ma(0,*))
k=nint(x) mod n
j=nint(x) / n
;
; Compute model value
hd=sidpointing(sidmodels(v,vcorr,[1,2],tm(k)),ma(*,k),1) 
hd(0)=hd(0)*15 &  hd=hd(j)
;
; These are the model parameters:
; Feed beam azimuth and elevation, siderostat azimuth and elevation,
; error in angle between axes (0=>orthogonal), zero point azimuth axis,
; tilt of mirror in its cell (0=>perpendicular) zero point elevation axis,
;
; Compute partial derivatives
pd=dblarr(8)
for i=0,7 do begin
	a=v
	a(i)=a(i)+1
	hdh=sidpointing(a,ma(*,k),1) & hdh(0)=hdh(0)*15 & hdh=hdh(j)
	a(i)=a(i)-2
	hdl=sidpointing(a,ma(*,k),1) & hdl(0)=hdl(0)*15 & hdl=hdl(j)
	pd(i)=(hdh-hdl)/2
endfor
;
return,[hd,pd]
;
end
;-------------------------------------------------------------------------------
pro motorfit,ft=ft,rh=rh,bp=bp
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Tables,ScanTable,BGTable,StationTable
;
; Check presence of data
if checkdata([1,8,16,21]) ne 0 then return
;
if n_elements(feedvector) eq 0 then begin
	feedvector=dblarr(genconfig.numsid,3,n_elements(scantable))-1
	for i=0,GenConfig.NumSid-1 do feedvector(i,0,*)=scantable.starttime
endif
;
; The are the meteorological parameters for the index of refraction
if n_elements(ft) eq 0 then ft=56.3	; Fahrenheit
if n_elements(rh) eq 0 then rh=33.0	; relative humidity [%]
if n_elements(bp) eq 0 then bp=23.316	; pressure in inch-Hg
;
for i=0,GenConfig.NumSid-1 do begin
	y=lmmotor(x,a,/init,sid=i+1,ft=ft,rh=rh,bp=bp)
	fita=[1,1,1,1,0,0,0,1]
	fita=[1,1,1,1,1,0,1,1]
	r=lmfit(x,y,a,function_name='lmmotor',/double,conv=conv,sigma=e,fita=fita)
	if conv eq 1 then begin
		print,'Siderostat ',i+1,' RMS ["] = ',stdev(y-r)*3600
		m=sidmodel(i+1,a)
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro feedmotion
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common LocalLMMotor,ma
;
RAD=180/!pi
;
feedvector=dblarr(genconfig.numsid,3,n_elements(scantable))-1
;
for k=0,genconfig.numsid-1 do begin
	for i=0,n_elements(scantable)-1 do begin
		if scantable(i).starid eq 'FKV0000' then begin
		index=where(motorangleerr(0,k,sol0(i):soln(i)) gt 0,count)
		if count gt 0 then begin
		hadec=sidpointing(sidmodel(k+1),motorangle(*,k,sol0(i)+index),1)
		feedvector(k,0,i)=avg(metrotime(sol0(i)+index))
		feedvector(k,1:2,i)=avg(horizon2azel( $
					equatorial2horizon( $
					hadec2equatorial(hadec))),1)
		endif
		endif
	endfor
	i=where(feedvector(k,0,*) ge 0,count) & i=i(0)
	if count gt 0 then begin
		feedvector(k,1,*)=feedvector(k,1,*)-feedvector(k,1,i)
		feedvector(k,2,*)=feedvector(k,2,*)-feedvector(k,2,i)
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro average
;
; Averages PointData. Applies background correction. Stores the data in scans.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ModelFit,parameters,ds_options
;
; Check presence of data
if checkdata([1,2,6,7,8,10,11]) ne 0 then return
;
; Minimum number of data points needed for a scan
MINPOINTS=10
;
NS=n_elements(ScanTable)
;
; Allocate and initialize arrays for ScanData if not done before or uptodate
flag=0
r=size(scans)
if r(n_elements(r)-2) ne 8 then flag=1
if not flag then if n_elements(scans) ne NS then flag=1
if not flag then if total(ScanTable.StarId eq scans.starid) ne NS then flag=1
if flag then begin
	scans=replicate(scan(),NS)
	s=size(scans(0))
	if s(n_elements(s)-2) ne 8 then begin
		print,'***Error(AVERAGE): GenConfig incomplete!'
		scans=-1
		return
	endif
	scans.starid=ScanTable.StarId
	if NS gt 1 then scans.iscan=lindgen(NS)+1 else scans.iscan=1
	bufferinfo=replicate(nightinfo(),1)
	ds_options.i=1
endif

; Set calibration factors
case SystemId of
	'Mark3'   :normfactor=pi_circle^2/2
	'NPOI'    :normfactor=pi_circle^2/8/(1-1/sqrt(2)); Changed below!
	      else:begin
		   print,'***Error(AVERAGE): unknown system: ',SystemId,'!'
		   return
		   end
endcase
if GenConfig.NumTriple gt 0 then begin
      triple_factor=fltarr(NS,GenConfig.NumTriple,max(GenConfig.NumSpecChan))+1
      triple_norm_factor=fltarr(NS,GenConfig.NumTriple,max(GenConfig.NumSpecChan))+1
endif
;
; Choose the method for computing error bars, G, N, or P (see below)
UncertaintyEstimate='N'
;
; Set bias flag
if total(abs(genconfig.v2bias)) eq 0 then !v2bias=0 else !v2bias=1
if total(abs(genconfig.tabias)) eq 0 then !tabias=0 else !tabias=1
;
; Background rate, photon rate, visibilities
;
print,'Averaging rates and squared visibilities...'
;
for i=0,GenConfig.NumOutBeam-1 do begin 
;
; Check for flagged channels so we can reduce the number of warnings
  dead_channel_index=total(reform(bgscans.rateerr(i,*) > 0),2)
  index=where(dead_channel_index eq 0,count)
  if count gt 1 then s='s' else s=''
  if count ge 1 then $
  print,'Warning(AVERAGE): ' $
	+string(count,format='(i3)')+' bad channel'+s+' in beam ' $
	+string(i+1,format='(i1)')+': '+string(index+1,format='(i2)')
  missing_bg_scans=intarr(n_elements(scantable))

for j=0,GenConfig.NumSpecChan(i)-1 do begin
for k=0,n_elements(Iscan)-1 do begin
;
  kk=Iscan(k)-1	; subtract 1 since arrays start with 0!
  StartTime=ScanTable(kk).StartTime
  StopTime=ScanTable(kk).StopTime
  sconf=scanconfig(Iscan(k),/starlist)
  kc=where(scanconfig(/starlist) eq sconf) & kc=kc(0)
;
; Check background measurement
  index=where(BGTable.ScanId eq ScanTable(kk).ScanId,bg_count)
  if bg_count gt 0 then begin
    bgr=bgscans(index).Rate(i,j)
    bgre=bgscans(index).RateErr(i,j)
    bgt=bgscans(index).Time
    bg_index=where(bgre gt 0,bg_count)
  endif
;
; Check photonrate
  PR=PhotonRate(i,j,Rec0(k):RecN(k))
  PRE=PhotonRateErr(i,j,Rec0(k):RecN(k))
  pr_index=where(PRE gt 0,NumPoint)
;
; Catch frozen channels with the back ground rate = photonrate
  if NumPoint gt 0 and bg_count gt 0 then $
  if avg(PR(pr_index))-avg(bgr(bg_index)) eq 0 then NumPoint=0
;
; Only average scans with background and sufficient number of points
;
  IF NumPoint ge MINPOINTS AND bg_count gt 0 THEN BEGIN
;
;     Average the background data
;
      bgr=bgr(bg_index)
      bgre=bgre(bg_index)
      bgwt=1/bgre^2
      bgt=bgt(bg_index)
      sum_bgwt=total(bgwt)
      scans(kk).BackgndRate(i,j)=total(bgr*bgwt)/sum_bgwt
      scans(kk).BackgndErr(i,j)=1/sqrt(sum_bgwt)
      index_l=where(bgt lt StartTime,count_l)
      index_r=where(bgt gt StopTime,count_r)
;
;     Average photonrate and squared visibilities
;
      wt=1/PRE(pr_index)^2
;     wt(*)=1
      sum_wt=total(wt)
      scans(kk).PhotonRate(i,j)=total(PR(pr_index)*wt)/sum_wt $
			       -scans(kk).BackgndRate(i,j)
      case UncertaintyEstimate of
      'G':begin
;   	  Gaussian error propagation (error of weighted mean)
          scans(kk).PhotonRateErr(i,j)=1/sqrt(sum_wt)
	  end
      'N':begin
;         Normal distribution
	  scans(kk).PhotonRateErr(i,j)=stdev(PR(pr_index))/sqrt(NumPoint)
	  end
      'P':begin
;         Poisson noise theoretical estimate
          scans(kk).PhotonRateErr(i,j)=sqrt(abs(scans(kk).PhotonRate(i,j)) $
		  /(ScanTable(kk).NumCoh*ScanTable(kk).NumIncoh*NumPoint))
	  end
      endcase
;
      bgpoints=fltarr(NumPoint)
      if (count_l gt 0) and (count_r gt 0) then begin
      	x=(findgen(NumPoint)/NumPoint*(StopTime-StartTime)+StartTime)/3600
	coeffs=poly_fit(bgt/3600,bgr,1,measure_errors=bgre)
	bgpoints=poly(x,coeffs)
      endif else begin
      	bgpoints(*)=scans(kk).BackgndRate(i,j)
      endelse
      calfactor=avg(PR(pr_index)-bgpoints)^2    ; Used until 1999-07-09
      calfactors=PR(pr_index)-bgpoints
      index=where(calfactors ne 0,NumPoint)
      if NumPoint ge MINPOINTS then begin
	calfactors=calfactors(index)
	pr_index=pr_index(index)
      endif
;
      for l=0,GenConfig.NumBaseline(i)-1 do begin
;
;	Adjustment for incoherent fluxes
	calfactor=calfactors
	for is=0,genconfig.numsid-1 do $
	  if strpos(genconfig.baselineid(l,i),genconfig.stationid(is)) eq -1 $
	    and strpos(strjoin(genconfig.baselineid(0:genconfig.numbaseline(i)-1,i)), $
		       genconfig.stationid(is)) ge 0 $
	    and scantable(kk).station(is) then $
	    calfactor=calfactor-genconfig.apdflux(is,j,i)*avg(calfactors)
	is1=where(genconfig.stationid eq strmid(genconfig.baselineid(l,i),0,3))
	is2=where(genconfig.stationid eq strmid(genconfig.baselineid(l,i),4,3))
	if genconfig.apdflux(is1,j,i) eq 0 $
	or genconfig.apdflux(is2,j,i) eq 0 then alpha=1.0 $
					   else alpha=genconfig.apdflux(is1,j,i) $
						     /genconfig.apdflux(is2,j,i)
	calfactor=calfactor^2*(4*alpha)/(alpha+1)^2
;
;	NPOI has baselines at different fringe frequencies
	if SystemId eq 'NPOI' then $
	normfactor=4.0/sinc(float(GenConfig.FringeMod(l,i))/GenConfig.NumBin)^2
;
;	Apply all corrections, i.e. bias and normalization
;	The bias correction was modified again Sep 26, 2007 because it was not
;	handling large BG rates properly in the scheme where these are zeroed
;	before computation of the bias. The two options are shown below, and
; 	for now we proceed with the one not requiring the zeroing of the BG.
	if NumPoint ge MINPOINTS then begin
          bias=(GenConfig.V2Bias(0,l,j,i,kc) $
              *(scans(kk).PhotonRate(i,j) $
		+scans(kk).BackgndRate(i,j))^(GenConfig.V2Bias(1,l,j,i,kc)))
	  bias=bias(0)*calfactor/normfactor	; use this when not zeroing BG before
;	  bias=bias(0)*PR(pr_index)^2/normfactor; must zero BG when doing bias fits
	  VSQ=VisSq(i,j,l,Rec0(k):RecN(k))-bias & VSQ=VSQ(pr_index)/calfactor
	  VSQE=VisSqErr(i,j,l,Rec0(k):RecN(k)) & VSQE=VSQE(pr_index)/calfactor
	  index=where(VSQE gt 0,count)
	endif else count=0
;
;	Average the visibilities if we have enough of them
	if count gt MINPOINTS then begin
	  wt=1/VSQE(index)^2
;         wt(*)=1	; I don't know which is better
	  sum_wt=total(wt)
	  scans(kk).VisSq(i,j,l)= $
		(total(VSQ(index)*wt)/sum_wt)*normfactor ;$
;		This code (Nov 29, 2006) removed again Sep 26, 2007
;		-GenConfig.V2Bias(0,l,j,i,kc)*(scans(kk).PhotonRate(i,j) $
;					      +scans(kk).BackgndRate(i,j)*0)^(GenConfig.V2Bias(1,l,j,i,kc))
	  case SystemId of
	  'Mark3' :scans(kk).VisSqErr(i,j,l)= $
			(stdev(VSQ(index))/sqrt(count))*normfactor
	  'NPOI'  :begin
		   case UncertaintyEstimate of
	       'G':begin
;   	           Gaussian error propagation (error of weighted mean)
		   scans(kk).VisSqErr(i,j,l)= $
		   	(1/sqrt(sum_wt))*normfactor
		   end
	       'N':begin
;                  Normal distribution
		   scans(kk).VisSqErr(i,j,l)= $
			(stdev(VSQ(index))/sqrt(NumPoint))*normfactor
		   end
	       'P':begin
;		   Estimate for Poisson noise, photonrate fluctuations neglected
		   scans(kk).VisSqErr(i,j,l)=sqrt( $
			pi_circle^4 $
			/(4*ScanTable(kk).NumCoh*ScanTable(kk).NumIncoh*count) $
			*(1/scans(kk).PhotonRate(i,j)^2 $
			  +(4/pi_circle^2)*abs(scans(kk).VisSq(i,j,l)) $
					      /scans(kk).PhotonRate(i,j)))
	           end
		   endcase
	     	   end
	  endcase
	endif else scans(kk).VisSqErr(i,j,l)=-1
;
;	Initialize E/EC variables
	scans(kk).VisSqE(i,j,l)=scans(kk).VisSq(i,j,l)
	scans(kk).VisSqEErr(i,j,l)=scans(kk).VisSqErr(i,j,l)
	scans(kk).VisSqC(i,j,l)=scans(kk).VisSq(i,j,l)
	scans(kk).VisSqCErr(i,j,l)=scans(kk).VisSqErr(i,j,l)
	scans(kk).VisSqEC(i,j,l)=scans(kk).VisSq(i,j,l)
	scans(kk).VisSqECErr(i,j,l)=scans(kk).VisSqErr(i,j,l)
;
;	Accumulate calibration factors for triples
        for tr=0,GenConfig.NumTriple-1 do begin
 	 for bl=0,2 do begin
   	  if (GenConfig.TripleBeam(bl,tr) eq i) and $
	     (GenConfig.TripleBase(bl,tr) eq l) then begin
           index=where( $
            GenConfig.TripleChan(0:GenConfig.TripleNumChan(tr)-1,bl,tr) eq j,n)
           if n gt 0 then begin
		triple_factor(kk,tr,index)= $
	        triple_factor(kk,tr,index)/(sqrt(avg(calfactor)/normfactor))
		triple_norm_factor(kk,tr,index)= $
		triple_norm_factor(kk,tr,index)*sqrt(normfactor)
	   	if scans(kk).VisSqErr(i,j,l) le 0 then $
		triple_factor(kk,tr,index) = 0
	   endif
	  endif
       	 endfor
	endfor
;
      endfor
;
  ENDIF ELSE BEGIN
;
;     We come here if there weren't enough data
      if bg_count eq 0 then begin
	if dead_channel_index(j) ne 0 then $
        print,'Warning(AVERAGE): no BackgndRate for OB=',i+1, $
  	    ', CH=',j+1,', scan=',kk+1, $
  	    ', Star: ',ScanTable(kk).starid, $
  	    format='(a,i1,a,i2,a,i3,a,a)'
        scans(kk).BackgndErr(i,j)=-1
      endif
      scans(kk).PhotonRateErr(i,j)=-1
      for l=0,GenConfig.NumBaseline(i)-1 do begin
  	scans(kk).VisSqErr(i,j,l)=-1
  	scans(kk).VisSqEErr(i,j,l)=-1
  	scans(kk).VisSqCErr(i,j,l)=-1
  	scans(kk).VisSqECErr(i,j,l)=-1
      endfor
      for t=0,GenConfig.NumTriple-1 do begin
       for l=0,2 do begin
        if (GenConfig.TripleBeam(l,t) eq i) then begin
          index=where( $
	      GenConfig.TripleChan(0:GenConfig.TripleNumChan(t)-1,l,t) eq j,n)
          if n gt 0 then begin
            triple_factor(kk,t,index)=0
	    scans(kk).TripleAmpErr(t,index)=-1
	    scans(kk).TripleAmpEErr(t,index)=-1
	    scans(kk).TripleAmpCErr(t,index)=-1
	    scans(kk).TripleAmpECErr(t,index)=-1
	    scans(kk).TriplePhaseErr(t,index)=-1
	    scans(kk).TriplePhaseCErr(t,index)=-1
          endif
         endif
        endfor
      endfor
;
  ENDELSE
;
endfor
endfor
endfor
;
; Triple products
;
print,'Averaging triple products...'
;
for t=0,GenConfig.NumTriple-1 do begin
for j=0,GenConfig.TripleNumChan(t)-1 do begin
for k=0,n_elements(Iscan)-1 do begin
;
kk=Iscan(k)-1	; subtract 1 since arrays start with 0!
sconf=scanconfig(Iscan(k),/starlist)
kc=where(scanconfig(/starlist) eq sconf) & kc=kc(0)
i_r=where(  TripleAmpErr(t,j,Rec0(k):RecN(k)) gt 0,count_r)
i_i=where(TriplePhaseErr(t,j,Rec0(k):RecN(k)) gt 0,count_i)
if count_r ne count_i then begin
  print,'***Error(AVERAGE): inconsistent triple editing!'
  return
endif
if count_r gt MINPOINTS and triple_factor(kk,t,j) ne 0 then begin
  CTR=ComplTriple(t,0,j,Rec0(k):RecN(k))
  CTI=ComplTriple(t,1,j,Rec0(k):RecN(k))
  CTER=ComplTripleErr(t,0,j,Rec0(k):RecN(k))
  CTEI=ComplTripleErr(t,1,j,Rec0(k):RecN(k))
  wtr=1/CTER(i_r)^2
  wtr(*)=1	; We are not sure complex triple errors are reliable
  wti=1/CTEI(i_i)^2
  wti(*)=1	; We are not sure complex triple errors are reliable
  sum_wtr=total(wtr)
  sum_wti=total(wti)
  scans(kk).ComplTriple(t,0,j)=total(CTR(i_r)*wtr)/sum_wtr*triple_factor(kk,t,j)
  scans(kk).ComplTriple(t,1,j)=total(CTI(i_i)*wti)/sum_wti*triple_factor(kk,t,j)
  case UncertaintyEstimate of
  'G':begin
;     Gaussian error propagation (error of weighted mean)
      scans(kk).ComplTripleErr(t,0,j)=1/sqrt(sum_wtr)*triple_factor(kk,t,j)
      scans(kk).ComplTripleErr(t,1,j)=1/sqrt(sum_wti)*triple_factor(kk,t,j)
      end
  'N':begin
;     Normal distribution
      scans(kk).ComplTripleErr(t,0,j)= $
	(stdev(CTR(i_r))/sqrt(count_r))*triple_factor(kk,t,j)
      scans(kk).ComplTripleErr(t,1,j)= $
	(stdev(CTI(i_i))/sqrt(count_i))*triple_factor(kk,t,j)
;     Based on D. Buscher's Theorem of the alignment uncertainty ellipse
      xr=complex(CTR(i_r),CTI(i_i)) $
	*exp(-i_complex*cphase(total(complex(CTR(i_r),CTI(i_i)))))
      scans(kk).ComplTripleErr(t,0,j)= $
	(stdev(float(xr))/sqrt(count_r))*triple_factor(kk,t,j)
      scans(kk).ComplTripleErr(t,1,j)= $
	(stdev(imaginary(xr))/sqrt(count_i))*triple_factor(kk,t,j)
      end
  'P':begin
;     Not available
      end
  endcase
;
; Compute amplitude and phase...
  bias=0
  tb=commonbeam(genconfig.triplebeam(*,t))
  if GenConfig.TripleBeam(0,t) eq GenConfig.TripleBeam(1,t) and $
     GenConfig.TripleBeam(0,t) eq GenConfig.TripleBeam(2,t) then $
     bias=(GenConfig.TABias(0,j,t,kc) $
         *(scans(kk).PhotonRate(tb, $
			        GenConfig.TripleChan(j,0,t)) $
	  +scans(kk).BackgndRate(tb, $
			 	 GenConfig.TripleChan(j,0,t))) $
;	 ^(GenConfig.TABias(1,j,t,kc)+3))/triple_norm_factor(kk,t)
	 ^GenConfig.TABias(1,j,t,kc))
  scans(kk).TripleAmp(t,j)=sqrt(scans(kk).ComplTriple(t,0,j)^2+ $
			        scans(kk).ComplTriple(t,1,j)^2) $
;			  -bias*triple_factor(kk,t,j)
			  -bias
; if scans(kk).starid eq 'FKV0025' and j eq 14 then stop
  if (scans(kk).ComplTriple(t,0,j) eq 0) and  $
     (scans(kk).ComplTriple(t,1,j) eq 0) then $
       scans(kk).TriplePhase(t,j)=0 $
  else scans(kk).TriplePhase(t,j)= $
		atan(scans(kk).ComplTriple(t,1,j), $
	     	     scans(kk).ComplTriple(t,0,j))
; ...and their errors
  TAUncertaintyEstimate='G'
  case TAUncertaintyEstimate of
  'G':begin
;     Gaussian error propagation, but correlations ignored
      scans(kk).TripleAmpErr(t,j)= $
        sqrt((scans(kk).ComplTriple(t,0,j)*scans(kk).ComplTripleErr(t,0,j))^2 $
            +(scans(kk).ComplTriple(t,1,j)*scans(kk).ComplTripleErr(t,1,j))^2) $
       	        /abs(scans(kk).TripleAmp(t,j))
      if scans(kk).TripleAmp(t,j) eq 0 then scans(kk).TriplePhaseErr(t,j)=-1 $
      else scans(kk).TriplePhaseErr(t,j)=scans(kk).TripleAmpErr(t,j)/ $
				     scans(kk).TripleAmp(t,j)
;     This uses the new ComplTripleErr's based on alignment of the uncertainty
;     ellipse with the triple product
      scans(kk).TripleAmpErr(t,j)=scans(kk).ComplTripleErr(t,0,j)
      scans(kk).TriplePhaseErr(t,j)=atan(scans(kk).ComplTripleErr(t,1,j), $
					 scans(kk).TripleAmp(t,j))
      end
  'N':begin
;     Normal distribution (for high-SNR case)
      scans(kk).TripleAmpErr(t,j)=(stdev(sqrt(CTR(i_r)^2+CTI(i_i)^2))/sqrt(count_r)) $
				 *triple_factor(kk,t,j)/sqrt(count_r)
      scans(kk).TriplePhaseErr(t,j)=stdev(atan(CTI(i_i),CTR(i_r)))/sqrt(count_r)
      end
  'P':begin
;     Not available
      end
  endcase
;
; Initialize derived variables
  scans(kk).TripleAmpC(t,j)=scans(kk).TripleAmp(t,j)
  scans(kk).TripleAmpCErr(t,j)=scans(kk).TripleAmpErr(t,j)
  scans(kk).TripleAmpE(t,j)=scans(kk).TripleAmp(t,j)
  scans(kk).TripleAmpEErr(t,j)=scans(kk).TripleAmpErr(t,j)
  scans(kk).TripleAmpEC(t,j)=scans(kk).TripleAmp(t,j)
  scans(kk).TripleAmpECErr(t,j)=scans(kk).TripleAmpErr(t,j)
  scans(kk).TriplePhaseC(t,j)=scans(kk).TriplePhase(t,j)
  scans(kk).TriplePhaseCErr(t,j)=scans(kk).TriplePhaseErr(t,j)
endif else begin
  scans(kk).ComplTripleErr(t,0,j)=-1
  scans(kk).ComplTripleErr(t,1,j)=-1
  scans(kk).TripleAmpErr(t,j)=-1
  scans(kk).TripleAmpCErr(t,j)=-1
  scans(kk).TripleAmpEErr(t,j)=-1
  scans(kk).TripleAmpECErr(t,j)=-1
  scans(kk).TriplePhaseErr(t,j)=-1
  scans(kk).TriplePhaseCErr(t,j)=-1
endelse
;
endfor
endfor
endfor
;
; Time, delays, NAT data
;
print,'Averaging delays...'
;
for k=0,n_elements(Iscan)-1 do begin
;
; Time
	kk=Iscan(k)-1
	t=PointTime(Rec0(k):RecN(k))
	index=where(t ge 0,count)
	if count gt 0 then scans(kk).Time=avg(t(index)) $
		else scans(kk).Time=-avg(abs(t))
;
; DelayJitter
for i=0,GenConfig.NumOutBeam-1 do begin 
for l=0,GenConfig.NumBaseline(i)-1 do begin
	v=DelayJitter(i,l,Rec0(k):RecN(k))
	ve=DelayJitterErr(i,l,Rec0(k):RecN(k))
    	index=where(ve gt 0,count)
    	if count gt MINPOINTS then begin
      		scans(kk).DelayJitter(i,l)=median(v(index))
      		scans(kk).DelayJitterErr(i,l)=1e-6
    	endif else scans(kk).DelayJitterErr(i,l)=-1e-6
endfor
endfor
;
; NATJitter, delay positions
for i=0,GenConfig.NumSid-1 do begin
;
	v=NATJitter(i,Rec0(k):RecN(k))
	ve=NATJitterErr(i,Rec0(k):RecN(k))
	index=where(ve gt 0,count)
	if count gt MINPOINTS then begin
		scans(kk).NATJitter(i)=median(v(index))
		scans(kk).NATJitterErr(i)=0.01
	endif else scans(kk).NATJitterErr(i)=-1
;
	v=NATCounts(i,Rec0(k):RecN(k))
	ve=NATCountsErr(i,Rec0(k):RecN(k))
	index=where(ve gt 0,count)
	if count gt MINPOINTS then begin
		scans(kk).NATCounts(i)=avg(v(index))
		scans(kk).NATCountsErr(i)=stdev(v(index))/sqrt(count)
	endif else scans(kk).NATCountsErr(i)=-1
;
	v=FDLPos(i,Rec0(k):RecN(k))
	ve=FDLPosErr(i,Rec0(k):RecN(k))
	index=where(ve gt 0,count)
	if count gt MINPOINTS then begin
		coeffs=polyfit(t(index)/3600,v(index),2,vft)
		scans(kk).FDLPos(i)=polynom(scans(kk).Time/3600,coeffs)
		scans(kk).FDLPosErr(i)=stdev(v(index)-vft)/sqrt(count)
	endif else begin
		scans(kk).FDLPos(i)=0
		scans(kk).FDLPosErr(i)=-1e-6
	endelse
;
	IF n_elements(GrpDelay) gt 1 and n_elements(MetroPos) gt 1 then BEGIN
	IF n_elements(GrpDelay(0,*)) ne n_elements(PointTime) $
	or n_elements(MetroPos(0,*)) ne n_elements(PointTime) then BEGIN
		if i eq 0 then $
		print,'Warning(AVERAGE): incompatible delay corrections!'
	ENDIF ELSE BEGIN
	d=FDLPos(i,Rec0(k):RecN(k))*(i ne GenConfig.RefStation-1) $
	 -GrpDelay(i,Rec0(k):RecN(k))*1d-6 $
	 -MetroPos(i,Rec0(k):RecN(k))*1d-6*(i ne GenConfig.RefStation-1)
	ferr=FDLPosErr(i,Rec0(k):RecN(k))
	gerr=GrpDelayErr(i,Rec0(k):RecN(k))
	merr=MetroPosErr(i,Rec0(k):RecN(k))
	index=where(ferr gt 0 and gerr gt 0 and merr gt 0,count)
	if count gt MINPOINTS then begin
		coeffs=polyfit(t(index)/3600,d(index),2,dft)
		scans(kk).GrpDelay(i)=polynom(scans(kk).Time/3600,coeffs)
		scans(kk).GrpDelayErr(i)=stdev(d(index)-dft)/sqrt(count)
		if i eq GenConfig.RefStation-1 then $
			scans(kk).GrpDelayErr(i)=1e-6
	endif else begin
		scans(kk).GrpDelay(i)=avg(d)
		scans(kk).GrpDelayErr(i)=-1e-6
	endelse
;		
	d=FDLPos(i,Rec0(k):RecN(k))*(i ne GenConfig.RefStation-1) $
	 -DryDelay(i,Rec0(k):RecN(k))*1d-6 $
	 -MetroPos(i,Rec0(k):RecN(k))*1d-6*(i ne GenConfig.RefStation-1)
	derr=DryDelayErr(i,Rec0(k):RecN(k))
	index=where(ferr gt 0 and derr gt 0 and merr gt 0,count)
	if count gt MINPOINTS then begin
		coeffs=polyfit(t(index)/3600,d(index),2,dft)
		scans(kk).DryDelay(i)=polynom(scans(kk).Time/3600,coeffs)
		scans(kk).DryDelayErr(i)=stdev(d(index)-dft)/sqrt(count)
		if i eq GenConfig.RefStation-1 then $
			scans(kk).DryDelayErr(i)=1e-6
	endif else begin
		scans(kk).DryDelay(i)=avg(d)
		scans(kk).DryDelayErr(i)=-1e-6
	endelse
;		
	d=FDLPos(i,Rec0(k):RecN(k))*(i ne GenConfig.RefStation-1) $
	 -WetDelay(i,Rec0(k):RecN(k))*1d-6 $
	 -MetroPos(i,Rec0(k):RecN(k))*1d-6*(i ne GenConfig.RefStation-1)
	werr=WetDelayErr(i,Rec0(k):RecN(k))
	index=where(ferr gt 0 and werr gt 0 and merr gt 0,count)
	if count gt MINPOINTS then begin
		coeffs=polyfit(t(index)/3600,d(index),2,dft)
		scans(kk).WetDelay(i)=poly(scans(kk).Time/3600,coeffs)
		scans(kk).WetDelayErr(i)=stdev(d(index)-dft)/sqrt(count)
		if i eq GenConfig.RefStation-1 then $
			scans(kk).WetDelayErr(i)=1e-6
	endif else begin
		scans(kk).WetDelay(i)=avg(d)
		scans(kk).WetDelayErr(i)=-1e-6
	endelse
	ENDELSE
;
	ENDIF
;		
endfor
;
endfor
;
; Average inch data if present
if n_elements(Sol0) ne 0 and n_elements(SolN) ne 0 then inchav
;
calcastrom
calcviscal
;
print,'Finished averaging.'
;
end
;-------------------------------------------------------------------------------
pro clearpointdata
;
; Free up memory by clearing PointData variables.
;
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
PointTime=0
FDLPos=0 & FDLPosErr=-1
DelayJitter=0 & DelayJitterErr=-1
NATJitter=0 & NATJitterErr=-1
GrpDelay=0 & GrpDelayErr=-1
DryDelay=0 & DryDelayErr=-1
WetDelay=0 & WetDelayErr=-1
MetroPos=0 & MetroPosErr=-1
GeoDelay=0 & GeoDelayErr=-1
PhotonRate=0 & PhotonRateErr=-1
VisSq=0 & VisSqErr=-1
ComplexVis=0 & ComplexVisErr=-1
ComplTriple=0 & ComplTRipleErr=-1
VisAmp=0 & VisAmpErr=-1
VisPhase=0 & VisPhaseErr=-1
TripleAmp=0 & TripleAmpErr=-1
TriplePhase=0 & TriplePhaseErr=-1
;
print,'PointData cleared.'
;
end
;-------------------------------------------------------------------------------
pro clearbiasdata
;
; Clear all bias data for a fresh average
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
genconfig.v2bias=0
genconfig.tabias=0
;
print,'Bias cleared.'
;
end
;-------------------------------------------------------------------------------
pro rewraptriple,angle
;
; Used to effectively unwrap the triple phase by rotating the complex triple,
; recomputing the phase, and subtracting the rotation angle. The angle is
; chosen so that after rotation, the entire range of phases will fit into
; the interval [-180,+180].
;
; Only the triple phase of the selected calibrator stars is unwrapped!
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if n_elements(cal_stars) eq 0 then begin
	print,'***Error(REWRAPTRIPLE): no calibrator stars defined!'
	return
endif
if cal_stars(0) eq '' then begin
	print,'***Error(REWRAPTRIPLE): no calibrator stars selected!'
	return
endif
;
if n_elements(angle) eq 0 then begin
	read,'Please enter rotation angle [degrees]: ',angle
endif
angle=angle/rad
;
; Obtain scan index corresponding to calibrator selection
list_stars,stars
mask=intarr(n_elements(scans))
for k=0,n_elements(stars)-1 do begin
	index=where(cal_stars eq stars(k),count)
	if count eq 0 then begin
		index=where(scans.StarId eq stars(k))
		mask(index)=-1
	endif
endfor
calscan_index=where(mask eq 0,calscan_count)
;
numchan=n_elements(scans(0).triplephase(0,*))
numtrpl=n_elements(scans(0).triplephase(*,0))
;
real=scans(calscan_index).compltriple(*,0,*)
imag=scans(calscan_index).compltriple(*,1,*)
real_rot=real*cos(angle)-imag*sin(angle)
imag_rot=real*sin(angle)+imag*cos(angle)
index=where(real_rot eq 0 and imag_rot eq 0,count)
if count gt 0 then real_rot(index)=1
phas_rot=atan(imag_rot,real_rot)
scans(calscan_index).triplephasec=reform(phas_rot-angle,numtrpl,numchan,calscan_count)
;
print,'Unwrapping completed.'
;
end
;-------------------------------------------------------------------------------
pro dewraptriple
;
; Unwrap the (calibrated) triple phase by subtracting a fit vs. wavelength
; derived from a stretch of phases without wraps.
; 
; Only the triple phase of the selected calibrator stars is unwrapped!
;
common FitPlotData,x_mid,x_scl,s
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if n_elements(cal_stars) eq 0 then begin
	print,'***Error(DEWRAPTRIPLE): no calibrator stars defined!'
	return
endif
if cal_stars(0) eq '' then begin
	print,'***Error(DEWRAPTRIPLE): no calibrator stars selected!'
	return
endif
;
; Obtain scan index corresponding to calibrator selection
list_stars,stars
mask=intarr(n_elements(scans))
for k=0,n_elements(stars)-1 do begin
	index=where(cal_stars eq stars(k),count)
	if count eq 0 then begin
		index=where(scans.StarId eq stars(k))
		mask(index)=-1
	endif
endfor
calscan_index=where(mask eq 0,calscan_count)
;
; Comment next two lines if you don't want all stars selected
; calscan_index=indgen(n_elements(scans))
; calscan_count=n_elements(scans)
;
numchan=n_elements(scans(0).triplephase(0,*))
;
y_fit=poly(findgen(numchan)+1,s)/RAD
;
for i=0,calscan_count-1 do begin
	scans(calscan_index(i)).triplephasec(0,*)= $
       (scans(calscan_index(i)).triplephasec(0,*)-y_fit) mod (2*pi_circle)
	index=where(scans(calscan_index(i)).triplephasec lt -pi_circle,count)
	if count gt 0 then scans(calscan_index(i)).triplephasec(0,index)= $
			   scans(calscan_index(i)).triplephasec(0,index)+2*pi_circle
	index=where(scans(calscan_index(i)).triplephasec gt +pi_circle,count)
	if count gt 0 then scans(calscan_index(i)).triplephasec(0,index)= $
			   scans(calscan_index(i)).triplephasec(0,index)-2*pi_circle
	scans(calscan_index(i)).triplephasec= $
	scans(calscan_index(i)).triplephasec+y_fit
endfor
;
print,'Unwrapping completed.'
;
end
;-------------------------------------------------------------------------------
pro manualunwrap
;
; Manually unwrap the triple phase. The user places a box in the plot window,
; and +/-360 degrees is added to the data inside it.
;
common PlotDataInfo,plotscans,plotdata_x,plotdata_y
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if ds_y.item ne 33 then begin
	print,'***Error(MANUALUNWRAP): will only unwrap TriplePhase c.!'
	return
endif
if !d.window eq -1 then begin
        print,'***Error(MANUALUNWRAP): no plot window currently open!'
        return
endif
;
icom=set_boxes(box_x,box_y)
if icom ne 4 then begin
;
	read,prompt='Add (1) or subtract (-1) 360 [degrees]? ',angle
	case angle of
		1: angle=+360.0/RAD
       	       -1: angle=-360.0/RAD
     	     else: begin
	   	   print,'***Error(MANUALUNWRAP): invalid input!'
	   	   return
	   	   end
	endcase
;
;	Start editing
	x=plotscans.x
	y=plotscans.y
	e=plotscans.ye
	ch=plotscans.ch
	tr=plotscans.tr
	pt=plotscans.pt
;
	j=where((x gt box_x(0)) and (x lt box_x(1)) and $
		(y gt box_y(2)) and (y lt box_y(1)),count)
	if count gt 0 then begin
		for i=0,count-1 do $
			scans(pt(j(i))).triplephasec(tr(j(i)),ch(j(i)))= $
			scans(pt(j(i))).triplephasec(tr(j(i)),ch(j(i)))+angle
		y(j)=y(j)+angle*RAD
		!p.color=tci(2)
		k=where(e(j) gt 0)
		oplot,x(j(k)),y(j(k))
		if ps_options.e then begin
			if count gt 1 then $
				oploterr,x(j(k)),y(j(k)),e(j(k)),3 $
			else begin
				x1=x(j(0))
				y1=y(j(0))
				e1=e(j(0))
				if e1 gt 0 then $
					oploterr,[x1,x1],[y1,y1],[e1,e1],3
			endelse
		endif
	endif
;
print,'Unwrapping completed. Hit <return> to continue.'
endif else icom=set_boxes(box_x,box_y,/clear)
;
end
;-------------------------------------------------------------------------------
pro fixearlydays
;
; Fixes wrong configuration info from the early days. Only to be called 
; immediately after reading scantable, sysconfig (and bgdata) from .con 
; files! OYSTER output .cha files must have no configuration 
; problems, and therefore this procedure is not allowed to be applied
; after reading these files.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
;
fixed=0
if SystemId eq 'NPOI' then begin
;	Date changes
	if Date eq '1995-03-18' then Date='1996-03-18'
	if Date eq '1995-03-19' then Date='1996-03-19'
	if Date eq '2001-01-07' then Date='2002-01-07'
	if Date eq '2001-01-09' then Date='2002-01-09'
	if Date eq '2001-01-12' then Date='2002-01-12'
	if genconfig.beamcombinerid eq 3 then begin
		if Date eq '2003-03-14' then Date='2003-03-15'
		if Date eq '2003-03-17' then Date='2003-03-18'
		if Date eq '2003-03-18' then Date='2003-03-19'
		if Date eq '2003-03-20' then Date='2003-03-21'
		if Date eq '2002-03-22' then Date='2003-03-23'
		if Date eq '2002-03-23' then Date='2003-03-24'
		if Date eq '2002-03-24' then Date='2003-03-25'
		if Date eq '2002-03-25' then Date='2003-03-26'
	endif
	GeoParms.Date=Date
	GeoInfo.Date=Date
	GenInfo.Date=Date
;	Date (limited) range section
	parsedate,Date,y,m,d
	if julian(y,m,d) lt julian(1996L,11L,12L) then begin
		fixwavelengths
		fixed=1
	endif
	if julian(y,m,d) lt julian(1997L,02L,11L) then begin
		fixchanwidths
		fixed=1
	endif
	if julian(y,m,d) ge julian(1999L,01L,23L) and $
	   julian(y,m,d) le julian(1999L,03L,22L) then begin
		GenConfig.StationId(0)='AC0'
		GenConfig.StationId(1)='AE0'
		GenConfig.BaselineId(0,0)='AE0-AW0'
		GenConfig.BaselineId(0,1)='AC0-AW0'
		GenConfig.BaselineId(0,2)='AC0-AE0'
		fixed=1
	endif
	if julian(y,m,d) lt julian(2000L,1L,1L) then begin
		GenConfig.SpectrometerId=['Sp1','Sp2','Sp3']
		fixed=1
	endif
;	Moved from 3-way to 6-way combiner, binner cables switched
	if julian(y,m,d) ge julian(2000L,1L,1L) and $
	   julian(y,m,d) lt julian(2001L,9L,25L) then begin
		GenConfig.BeamCombinerId=2
		GenConfig.BCInputId=[2,3,4]
		GenConfig.SpectrometerId=['Sp2','Sp3','Sp1']
		fixed=1
	endif
;	Did not use scanconfig and code for 3-way
	if julian(y,m,d) lt julian(2001L,9L,25L) then begin
		scantable.station(0:2)=1
		scantable.code=1
		fixed=1
	endif
	if julian(y,m,d) ge julian(2001L,9L,25L) and $
	   julian(y,m,d) lt julian(2002L,2L,28L) then begin
		scantable.code=1
	endif
	if julian(y,m,d) ge julian(2001L,9L,25L) and $
	   julian(y,m,d) le julian(2002L,3L,3L) then begin
		scantable.station(0:5)=1
		genconfig.spectrometerid=['Sp2','Sp3']
		genconfig.siderostatid=[6,1,2,3,5,4]
		genconfig.sidconid=[6,1,2,3,5,4]
		fixed=1
	endif
;	Date (unlimited) range section (watch this!)
	index=where(total(scantable.station(0:genconfig.numsid-1),1) le 1 $
		and scantable.code eq 1,count)
	if count gt 0 then scantable(index).code=0
;
;	Date (single) only
	if Date eq '1996-05-01' then begin
		index=where(StarTable.starid eq 'FKV0001')
		StarTable(index).starid='FKV0000'
		fixed=1
	endif
	if Date eq '1996-05-02' then begin
		index=where(startable.starid eq 'FKV0001')
		StarTable(index).starid='FKV0000'
		fixed=1
	endif
	if Date eq '1996-03-18' then begin
		GenConfig.StationId(0)='AC0'
		GenConfig.StationId(1)='AE0'
		GenConfig.StationId(2)='AW0'
		GenConfig.BaselineId(0,0)='AE0-AW0'
		GenConfig.BaselineId(0,1)='AC0-AW0'
		GenConfig.BaselineId(0,2)='AC0-AE0'
		fixed=1
	endif
	if Date eq '1996-03-19' then begin
		GenConfig.StationId(0)='AC0'
		GenConfig.StationId(1)='AE0'
		GenConfig.StationId(2)='AW0'
		GenConfig.BaselineId(0,0)='AE0-AW0'
		GenConfig.BaselineId(0,1)='AC0-AW0'
		GenConfig.BaselineId(0,2)='AC0-AE0'
		fixed=1
	endif
	if Date eq '1997-01-11' then begin
		numscan=25
		numbgscan=24
		stars=['FKV0074','FKV0100','FKV0074','FKV0100','FKV0074', $
		       'FKV0100','FKV0074','FKV0100','FKV0139','BSC1412', $
		       'FKV0139','BSC1412','FKV0139','BSC1412','FKV0139', $
		       'BSC1412','FKV0139','BSC1412','FKV0139','BSC1412', $
		       'BSC2095','FKV1137','BSC2095','FKV1137','BSC2095']
		ScanTable.starid=stars
		ScanTable.scanid=indgen(numscan)+1
		if n_elements(BGTable) ne 0 then $
			BGTable.scanid=indgen(numbgscan)+1
		fixed=1
	endif
	if Date eq '1997-01-31' then begin
		numscan=20
		numbgscan=18
		stars=['FKV0100','FKV0100','FKV0202','FKV0202','FKV0202', $
		       'BSC2095','FKV1137','BSC2095','FKV1137','BSC2095', $
		       'FKV1137','BSC2095','FKV1137','FKV0251','FKV0295', $
		       'FKV0251','FKV0295','FKV0251','FKV0237','FKV0316']
		ScanTable.starid=stars
		ScanTable.scanid=indgen(numscan)+1
		if n_elements(BGTable) ne 0 then $
			BGTable.scanid=[1,2,4,5,7,8,9,10,11,12,13,14,15,16, $
					17,18,19,20]
		fixed=1
	endif
	if Date eq '1997-02-01' then begin
		numscan=3
		numbgscan=2
		stars=['FKV0251','FKV0295','FKV0251']
		ScanTable.starid=stars
		ScanTable.scanid=indgen(numscan)+1
		if n_elements(BGTable) ne 0 then $
			BGTable.scanid=[1,2]
		fixed=1
	endif
	if Date eq '1997-02-04' then begin
		numscan=45
		numbgscan=45
		stars=['FKV0100','FKV0074','FKV0100','FKV0074','FKV0100', $
		       'FKV0074','FKV0100','FKV0074','FKV0251','FKV0295', $
		       'FKV0251','FKV0295','FKV0251','FKV0295','FKV0251', $
		       'FKV0295','FKV0251','FKV0295','FKV0251','FKV0295', $
		       'FKV0251','FKV0295','FKV0251','FKV0295','FKV0365', $
		       'FKV0379','FKV0365','FKV0379','FKV0365','FKV0379', $
		       'FKV0365','FKV0379','FKV0365','FKV0379','FKV0365', $
		       'FKV0379','FKV0365','FKV0379','FKV0365','FKV0379', $
		       'FKV0365','FKV0365','FKV0379','FKV0365','FKV0379']
		ScanTable.starid=stars
		ScanTable.scanid=indgen(numscan)+1
		if n_elements(BGTable) ne 0 then $
			BGTable.scanid=indgen(numbgscan)+1
		fixed=1
	endif
	if Date eq '1997-05-14' then begin
		ScanTable(0:4).starid= $
			['FKV0423','FKV2897','FKV0423','FKV1302','FKV1311']
	endif
	if Date eq '1998-11-18' then begin
		GenConfig.StationId(1)='E04'
		GenConfig.BaselineId(0,0)='E04-AW0'
		GenConfig.BaselineId(0,2)='AC0-E04'
;		3-way was used, but we want compatibility with 6-way
		GenConfig.BeamCombinerId=2
		GenConfig.BCInputId=[2,3,4]
		fixed=1
	endif	
	if Date eq '1998-12-16' then begin
		GenConfig.StationId(0)='E02'
		GenConfig.BaselineId(0,1)='E02-AW0'
		GenConfig.BaselineId(0,2)='E02-AE0'
;		3-way was used, but we want compatibility with 6-way
		GenConfig.BeamCombinerId=2
		GenConfig.BCInputId=[2,3,4]
		fixed=1
	endif	
	if Date eq '1998-12-18' then begin
		GenConfig.StationId(0)='E02'
		GenConfig.BaselineId(0,1)='E02-AW0'
		GenConfig.BaselineId(0,2)='E02-AE0'
;		3-way was used, but we want compatibility with 6-way
		GenConfig.BeamCombinerId=2
		GenConfig.BCInputId=[2,3,4]
		fixed=1
	endif	
	if Date eq '1999-01-08' then begin
		GenConfig.StationId(0)='E02'
		GenConfig.StationId(1)='E04'
		GenConfig.BaselineId(0,0)='E04-AW0'
		GenConfig.BaselineId(0,1)='E02-AW0'
		GenConfig.BaselineId(0,2)='E02-E04'
;		3-way was used, but we want compatibility with 6-way
		GenConfig.BeamCombinerId=2
		GenConfig.BCInputId=[2,3,4]
		fixed=1
	endif	
	if Date eq '2000-07-22' then begin
		index=scanindex([24619,25645,26809,27393,28257,29990,30762, $
       				 33164,35533])
		if index(0) ge 0 then scantable(index).code=0
		fixed=1
	endif
	if Date eq '2000-09-20' then begin
		scantable([1,5,7,9,11,13,15,17,19,44]).code=0
		fixed=1
	endif
	if Date eq '2001-03-20' then begin
		GenConfig.StationId(0)='AN0'
		GenConfig.BaselineId(0,1)='AN0-AW0'
		GenConfig.BaselineId(0,2)='AN0-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif	
	if Date eq '2001-03-22' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif	
	if Date eq '2001-03-28' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif	
	if Date eq '2001-03-30' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif	
	if Date eq '2001-03-31' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-04-09' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-05-02' then begin
		GenConfig.StationId(2)='E02'
		GenConfig.BaselineId(0,0)='AE0-E02'
		GenConfig.BaselineId(0,1)='AC0-E02'
		GenConfig.BCInputId(2)=1
		GenConfig.DelayLineId(2)=1
		fixed=1
	endif
	if Date eq '2001-05-10' then begin
		GenConfig.StationId(2)='E02'
		GenConfig.BaselineId(0,0)='AE0-E02'
		GenConfig.BaselineId(0,1)='AC0-E02'
		GenConfig.BCInputId(2)=1
		GenConfig.DelayLineId(2)=1
		fixed=1
	endif
	if Date eq '2001-05-16' then begin
		GenConfig.StationId(2)='E02'
		GenConfig.BaselineId(0,0)='AE0-E02'
		GenConfig.BaselineId(0,1)='AC0-E02'
		GenConfig.BCInputId(2)=1
		GenConfig.DelayLineId(2)=1
		fixed=1
	endif
	if Date eq '2001-05-25' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-05-26' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-05-27' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-06-01' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-06-08' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-06-10' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-06-21' then begin
		index=scanindex([18535,19824,20292,20974,21497,22157, $
				 22537,23039,23462,24099,24694,25177, $
				 25706,26703,28285,28899,29517,41077, $
				 41557])
		if index(0) ge 0 then scantable(index).code=0
		fixed=1
	endif
	if Date eq '2001-06-27' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-06-28' then begin
		GenConfig.StationId(0)='W07'
		GenConfig.BaselineId(0,1)='W07-AW0'
		GenConfig.BaselineId(0,2)='W07-AE0'
		GenConfig.BCInputId(0)=5
		GenConfig.DelayLineId(0)=5
		fixed=1
	endif
	if Date eq '2001-11-03' then begin
		scantable.station=0
		scantable.station(1)=1
		scantable.station(3)=1
		scantable(0:3).station(2)=1
		scantable(8:11).station(0)=1
		fixed=1
	endif
	if Date eq '2001-11-14' then begin
		scantable.code=0
		scantable.station=0
		stations=scantable.station
		stations([1,2,3,5,4],[0,1,2,3,4])=1
		scantable.station=stations
		fixed=1
	endif
	if Date eq '2001-11-17' then begin
		scantable.station=0
		scantable([0,6,7,13,18,19,20,21,22]).station(0)=1
		scantable([0,1,7,8,14,15,16,17]).station(1)=1
		scantable([0,2,7,9]).station(2)=1
		scantable([0,3,7,10,14,15,16,17,18,19,20,21,22]).station(3)=1
		scantable([0,5,7,12]).station(4)=1
		scantable([0,4,7,11]).station(5)=1
		scantable.code=0
		scantable([14,16,18,20,21]).code=1
		fixed=1
	endif
	if Date eq '2001-11-20' then begin
		scantable.station=0
		scantable(0).station=1
		scantable.station(3)=1
		scantable([1,2,3,4]).station(2)=1
		scantable([5,6,7,8]).station(0)=1
		scantable([9,10,11,12]).station(5)=1
		scantable([13,14]).station(4)=1
		scantable([2,4,6,8,10,12,14]).code=0
		fixed=1
	endif
	if Date eq '2001-11-21' then begin
		scantable([1,3,5,7,9]).code=0
		scantable.station=0
		scantable.station([0,2,3])=1
		fixed=1
	endif
	if Date eq '2002-01-09' then begin
		scantable([1,3,5,7,9,12,16,18]).code=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		scantable.station=1
		fixed=1
	endif
	if Date eq '2002-01-12' then begin
		scantable([1,3,5,7,9,13,15,18,20]).code=0
		scantable.station=1
		scantable(6:10).station([1,4,5])=0
		scantable(11:16).station([1,5])=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-01-25' then begin
		index=scanindex([19837,20802,21435,22180,24439,25150, $
				 28626,29882,31781,32253,33507,34209, $
				 35216,36043,38291,39045,40173,42879, $
				 44595])
		if index(0) ge 0 then scantable(index).code=0
		scantable.station=1
		index=scanindex([23608,24439,24802,25150])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-01-26' then begin
		index=scanindex([23417,27407,30948,33337,33919,34481, $
				 35026,37707,38409,39053,40280,40928, $
				 42353,44738])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex([30691,30948,33133,33337,33654,33919, $
				 34231,34481,34775,35026])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([38151,38409,38794,39053,39776,40280, $
				 40596,40928])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-01' then begin
		index=scanindex([17989,20020,22858,24640,26958,30621, $
				 32834,35115,36336,36977])
		if index(0) ge 0 then scantable(index).code=0
		scantable.station=1
		index=scanindex([24031,24640,26464,26958,27576,28268])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([29966,30621,32557,32834])
		if index(0) ge 0 then scantable(index).station(4:5)=0
		index=scanindex([36046,36336,36721,36977,37237,37468, $
				 37770,38634,39244,39541,40255,40859, $
				 41529,42312,42765])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-06' then begin
		index=scanindex([13908,16255,24259,25883,26443,28208, $
				 29478,30504,32043,32565,33350,34644, $
				 39185,39570,42371,43202,43784,44312])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex([25268,25883,26194,26443])
		if index(0) ge 0 then scantable(index).station([0,4])=0
		index=scanindex([28783,29478,30123,30504])
		if index(0) ge 0 then scantable(index).station(4:5)=0
		index=scanindex([31490,32043,32354,32565])
		if index(0) ge 0 then scantable(index).station(5)=0
		index=scanindex([38755,39185,39392,39570,39798,40065, $
				 40314,40605,40847,41078,41569])
		if index(0) ge 0 then scantable(index).station([0,1,4])=0
		index=scanindex([42011,42371])
		if index(0) ge 0 then scantable(index).station([1,4])=0
		index=scanindex([42726,43202])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([43516,43784])
		if index(0) ge 0 then scantable(index).station([1,4])=0
		index=scanindex([44089,44312])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-07' then begin
		index=scanindex([22764,30621,31379,33202,34366,36973, $
				 37670,39149,40293])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex([21725,22764])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([36412,36973,37429,37670,38022,38587, $
				 38889,39149,39706,40293,41237,41745, $
				 42025])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([38587,38889,39149,39706,40293,41237, $
				 41745,42025])
		if index(0) ge 0 then scantable(index).station(0)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-12' then begin
		scantable.station(2)=0
		scantable.station(5)=0
		index=scanindex([14086,14930,16233,17522,18731,21272, $
				 22821,24082,24800])
		if index(0) ge 0 then scantable(index).code=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-13' then begin
		index=scanindex([26313,27562,29495,30639,32634,34091, $
				 34984])
		if index(0) ge 0 then scantable(index).code=0
		scantable.station([1,2,4])=0
		index=scanindex([28903,29495,30272,30639,32055,32634])
		if index(0) ge 0 then scantable(index).station(1)=1
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-14' then begin
		index=scanindex([26348,28263,29781,32791,33502,34364, $
				 35099,36149,37224,37518,37754,38058, $
				 41773])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex([27859,28263,29208,29781])
		if index(0) ge 0 then scantable(index).station(5)=0
		index=scanindex([33166,33502])
		if index(0) ge 0 then scantable(index).station([4,5])=0
		index=scanindex([33968,34364])
		if index(0) ge 0 then scantable(index).station(0)=0
		index=scanindex([34830,35099])
		if index(0) ge 0 then scantable(index).station([1,2])=0
		index=scanindex([37224])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([37518])
		if index(0) ge 0 then scantable(index).station([0,2,3,4])=0
		index=scanindex([37754])
		if index(0) ge 0 then scantable(index).station([0,1,3,5])=0
		index=scanindex([38058])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-15' then begin
		index=scanindex([12148,13107,14171,14817,16733,23605, $
				 24791,25309,25816,26788,27292,28087, $
				 28499,29195,29560,30015,30187,30498, $
				 30711,30926,31839,32261,32518,34102, $
				 35990,36503,37364,39786,40204,42132, $
				 42805])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex([24505,24791,25102,25309])
		if index(0) ge 0 then scantable(index).station([2,5])=0
		index=scanindex([25629,25816,26788,27106,27292])
		if index(0) ge 0 then scantable(index).station(2)=0
		index=scanindex([28338,28499])
		if index(0) ge 0 then scantable(index).station(4:5)=0
		index=scanindex([29036,29195])
		if index(0) ge 0 then scantable(index).station(0)=0
		index=scanindex([29404,29560])
		if index(0) ge 0 then scantable(index).station(1:2)=0
		index=scanindex([30187])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([30498])
		if index(0) ge 0 then scantable(index).station([0,2,3,4])=0
		index=scanindex([30711])
		if index(0) ge 0 then scantable(index).station([0,1,3,5])=0
		index=scanindex([30926])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([39564,39786,40013,40204])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([41926,42132,42533,42805])
		if index(0) ge 0 then scantable(index).station([1,4])=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-20' then begin
		index=scanindex([25989,26222,26500,26758,28523,30473, $
				     36520,37640,38312,39194])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex(25989)
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex(26222)
		if index(0) ge 0 then scantable(index).station([0,2,3,4])=0
		index=scanindex(26500)
		if index(0) ge 0 then scantable(index).station([0,1,3,5])=0
		index=scanindex(26758)
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-23' then begin
		index=scanindex([12109,20339,22084,23157,23582,26100, $
				 39361,40502,42395,43030])
		if index(0) ge 0 then scantable(index).code=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-26' then begin
		index=scanindex([17738,24339,28390,28754,29116,29394, $
				 33756,37409])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex(28390)
		if index(0) ge 0 then scantable(index).station=[1,0,0,0,0,0]
		index=scanindex(28754)
		if index(0) ge 0 then scantable(index).station=[0,1,0,0,0,1]
		index=scanindex(29116)
		if index(0) ge 0 then scantable(index).station=[0,0,1,0,1,0]
		index=scanindex(29394)
		if index(0) ge 0 then scantable(index).station=[0,0,0,1,0,0]
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-27' then begin
		if scanindex(28299) ne -1 then begin
			scantable(scanindex(28299)).starid='FKV0380'
			scantable(scanindex(28299)).scanid=1
		endif
		index=scanindex([28948,29609,30003,30868,31428,33226, $
				     34383,35389,36487])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex(29609)
		if index(0) ge 0 then scantable(index).station=[1,0,0,0,0,0]
		index=scanindex(30003)
		if index(0) ge 0 then scantable(index).station=[0,1,0,0,0,1]
		index=scanindex(30868)
		if index(0) ge 0 then scantable(index).station=[0,0,0,1,0,0]
		index=scanindex(31428)
		if index(0) ge 0 then scantable(index).station=[1,1,0,1,0,1]
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-02-28' then begin
		if scanindex(24117) ne -1 then begin
			scantable(scanindex(24117)).starid='FKV0380'
			scantable(scanindex(24117)).scanid=0
			scantable(scanindex(24117)).code=1
		endif
		if scanindex(29817) ge 0 then begin
			scantable(scanindex(29817)).starid='FKV0380'
			scantable(scanindex(29817)).code=1
		endif
		index=scanindex(27314)
		if index(0) ge 0 then scantable(index).station=[1,0,0,0,0,0]
		index=scanindex(27610)
		if index(0) ge 0 then scantable(index).station=[0,1,0,0,0,1]
		index=scanindex(27839)
		if index(0) ge 0 then scantable(index).station=[0,0,0,1,0,0]
		index=scanindex(28216)
		if index(0) ge 0 then scantable(index).station=[1,1,0,1,0,1]
		index=scanindex([29817,30813,32570,32825,33148, $
				     33684])
		if index(0) ge 0 then scantable(index).station([2,4])=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-03-01' then begin
		fixed=1
		scantable.station=0
 		index=scanindex([15284,18112,20141,20592,21237,21850, $
 				 22499,23510,24975,25486,25899,26701, $
 				 27076,27862,28185,28608,28991,29275, $
 				 30576,30993,31450,32221,33732,34825, $
 				 35137,35600,36049,36404,37110,37363, $
 				 37696,38078,38630,38922,40442,41186, $
 				 41434,42030,42814,43185,43507,43853, $
 				 44963,45248,45522,45730,46410])
		if index(0) ge 0 then scantable(index).station(0)=1

 		index=scanindex([16376,18112,20592,21237,21850,22829, $
 				 23510,24975,25486,25899,26701,27327, $
 				 27862,28185,28608,28991,29600,30576, $
 				 30993,31450,32221,33924,34825,35137, $
 				 35600,36049,36607,37110,37363,37696, $
 				 38078,39413,39759,40666,41186,41434, $
 				 42030,42814,43185,44262,44490,44963, $
 				 45248,45522,45730,46097])
		if index(0) ge 0 then scantable(index).station([1,5])=1

 		index=scanindex([17699,18112,20141,20592,21237,21850, $
 				 23108,23510,24975,25486,25899,26701, $
 				 27546,27862,28185,28608,28991,30059, $
 				 30576,30993,31450,32221,34096,34825, $
 				 35137,35600,36049,36829,37110,37363, $
 				 37696,38078,38630,38922,39413,39759, $
 				 40904,41186,41434,42030,42814,43185, $
 				 43507,43853,44262,44490,44963,45248, $
 				 45522,45730,45925])
		if index(0) ge 0 then scantable(index).station(3)=1

 		index=scanindex([15284,16376,17699,18112,22499,22829, $
 				 23108,23510,27076,27327,27546,27862, $
 				 29275,29600,30059,30576,33732,33924, $
 				 34096,34825,36404,36607,36829,37110, $
 				 38630,39413,40442,40666,40904,41186, $
 				 43507,44262,45730,45925,46097,46410])
		if index(0) ge 0 then scantable(index).code=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
	endif
	if Date eq '2002-03-02' then begin
		scantable.station(2)=0
		scantable.station(4)=0
 		index=scanindex([15861,16119,18123,18361,20183,20351, $
 				 22700,22866,25037,25194,26699,26875, $
 				 28939,29120,29953,30187,31939,32219, $
 				 32985,33140,34620,34770,35232,35477, $
 				 36973,37362,38000,38208,39511,39680, $
 				 40499,40709,42936,43203,44193,44830, $
 				 44985])
		if index(0) ge 0 then scantable(index).station(0)=0

 		index=scanindex([15171,16119,17929,18361,20014,20351, $
 				 22217,22866,24812,25194,26530,26875, $
 				 28113,28266,29641,30187,31509,31688, $
 				 32786,33140,34271,34424,35003,35477, $
 				 36604,36768,37746,38208,39155,39319, $
 				 40193,40709,42406,42697,44640,44985])
		if index(0) ge 0 then scantable(index).station(1)=0

 		index=scanindex([15171,15861,17929,18123,20014,20183, $
 				 22217,22700,24812,25037,26530,26699, $
 				 29641,29953,32786,32985,35003,35232, $
 				 37746,38000,40193,40499,44640,44830])
		if index(0) ge 0 then scantable(index).station(3)=0

 		index=scanindex([15171,16119,17929,18361,20014,20351, $
 				 22217,22866,24812,25194,26530,26875, $
 				 28113,28266,29641,30187,31509,31688, $
 				 32786,33140,34271,34424,35003,35477, $
 				 36604,36768,37746,38208,39155,39319, $
 				 40193,40709,42406,42697,44640,44985])
		if index(0) ge 0 then scantable(index).station(5)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-03-15' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-03-20' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-03-21' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-03-27' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-03-31' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-04-02' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-04-03' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-04-10' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-04-12' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-04-13' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-04-14' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-04-26' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-04-28' then begin
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-05-03' then begin
		scantable.station(4)=0
		index=scanindex([14454,14782,15462,15814,16024,16490, $
				 16793,17115,17379,18979,19210,19589, $
				 19827,20213,20489,20725,21059,21300, $
				 22060,22454,22684,23232,23464,23874, $
				 24148,24793,25129,26278,26474,26953, $
				 27237,27849,28187,28507,28751,33099, $
				 33326,35546,35770,36079,36298,37489, $
				 37709,37924,38147])
		if index(0) ge 0 then scantable(index).station(0)=0
		index=scanindex([34377,34694,34951,35207,36534,36797, $
				 37036,37258])
		if index(0) ge 0 then scantable(index).station(2)=0
		index=scanindex([16490,16793,17115,17379,20213,20489, $
				 20725,21059,21300,23874,24148,24793, $
				 25129,27849,28187,28507,28751,33099, $
				 33326,35546,35770,36079,36298,37489, $
				 37709,37924,38147])
		if index(0) ge 0 then scantable(index).station(5)=0
		GenConfig.Wavelength(*,0)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
					   542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
		GenConfig.Wavelength(*,1)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
					   542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
		GenConfig.SiderostatId=[6,1,2,3,5,4]
		GenConfig.SidConId=[6,1,2,3,5,4]
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-05-04' then begin
		scantable.station(4)=0
		index=scanindex([30483,30714,31003,31225,36518,36733, $
				     37090,37317,38605,38825,39037,39261])
		if index(0) ge 0 then scantable(index).station([0,5])=0
		GenConfig.Wavelength(*,0)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
					   542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
		GenConfig.Wavelength(*,1)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
					   542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
		GenConfig.SiderostatId=[6,1,2,3,5,4]
		GenConfig.SidConId=[6,1,2,3,5,4]
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-05-08' then begin
		scantable.station(4)=0
		index=scanindex([18323,19090,20669,21181,22407,23219,24273,25476, $
				 26527,26856,27781,28218,29258,29716,30623,30935, $
				 32922,33205,33478,36025,36368,38425,39064,40083, $
				 40404,41453,41744])
		if index(0) ge 0 then scantable(index).station([0,5])=0
		GenConfig.Wavelength(*,0)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
					   542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
		GenConfig.Wavelength(*,1)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
					   542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
		GenConfig.SiderostatId=[6,1,2,3,5,4]
		GenConfig.SidConId=[6,1,2,3,5,4]
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-05-09' then begin
		scantable.station(4)=0
		index=scanindex([16114,16339,16789,18208,18579,19877,20129,21364, $
				 21672,22464,22848,23558,23852,24985,25278,26241, $
				 26528,27217,27541,28426,28769,29683,30590,31652, $
				 31876,32169,33295,34146,34871,35186,35873,36344, $
				 37347,37818,38652,38925,39706,39981,40703,41021, $
				 41735,42124,42819])
		if index(0) ge 0 then scantable(index).station([0,5])=0
		GenConfig.Wavelength(*,0)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
					   542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
		GenConfig.Wavelength(*,1)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
					   542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-05-11' then begin
		scantable.station(4)=0
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9

		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-05-12' then begin
		scantable.station([4,5])=0
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9

		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-05-15' then begin
		scantable.station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-16' then begin
		scantable.station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-18' then begin
		index=scanindex([28624,28876,29113,29510,30061,30358,31144, $
				 31372,31680,31970,32456,32686,33089,33312, $
				 33655,33872,34574,34828,35202,35439,35685, $
				 35915])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([39222,39470,39697,39929])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-19' then begin
		index=scanindex([27898,28123,30574,30770,31005,31385,31700, $
				 32041,32329,32620,32942,33225])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([37807,38024,38246,38680,38901,39111,39381, $
				 39590,39814,40035,40246,40463,40679,41122, $
				 41441,41667,41887,42364,42655])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-20' then begin
		index=scanindex([29208,29425,29908,30111,30444,30683,31289, $
				 31501,31904,32122,32736,33014,33505,34356, $
				 34542,35000,35229,35555,36031,36866,37196, $
				 37471,38098,38456,38793,39141,39386,39743, $
				 40074,40349,40727,41128,41352,41648,41936])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-22' then begin
		scantable.station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-23' then begin
		index=scanindex([28091,28404,28705,29059,29301,29766,30113, $
				 30462,30740,31453,31997,32322,32693,33004, $
				 33244,33941,34341,34956,36142,36444,36710, $
				 36931,37196,37418,37662,37877,38107,38369, $
				 38656,38881,39106,39558,39798,40144,40366, $
				 40737,40956,41174,41469,42237])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-24' then begin
		index=scanindex([27447,27680,27973,28205,28515,28740,29222, $
				 29514,29787,30025,30251,30543,30789,31047, $
				 31396,31665,31963,32327,32660])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([39897,40120,40367,40588,40804,41109,41522, $
				 41811,42084,42385])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-25' then begin
		index=scanindex([26813,27082,27317,27540,28027,28285, $
				 28523,28761,29100,29354,29570,29787, $
				 30162,30344,30558,30776,31033,31251, $
				 31471,31689,31979,32188,32452,32680, $
				 33058,33278,33518,33739,34092,34309, $
				 34538,34757,34986,35306,35510,35729, $
				 35962,36185,36409,36624,37059,37288, $
				 37520,37741,38165,38393,38643,38865, $
				 39313,39534,39751,39969])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2002-05-29' then begin
		index=scanindex([26171,26397,26810,27033,27295,27535, $
				 27921,28226,29107,29447,29872,30205, $
				 30632,30914,31232,31592,31966,32345, $
				 32629,33045,33308,33849,34076,34361, $
				 34601,34876,35092,35383,35626,35839, $
				 36069,36334,36624,36884,37106,37376, $
				 37598,38085,38305,38606,38848,39667, $
				 39986,40329,40548,40853,41079,41338, $
				 41575,41833,42088])
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
	if Date eq '2002-05-30' then begin
		index=scanindex([25372,25639,26029,26300,33635,33907, $
				 34136,34390,34611,34831,35062,35345, $
				 35586,35832,36071,36312,36579,36829, $
				 37051,37389,37640,38173,38385,38725, $
				 38946,39381,39627,39986,40280,40880, $
				 41128,41493,41760])
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
	if Date eq '2002-05-31' then begin
		index=scanindex([24822,25071,25378,25655,25886,26115, $
       				 26388,26657,27052,27293,27730,27967, $
       				 28230,28481,28843,29108,29365,29621, $
       				 29933,30166,30629,30895,31259,31518, $
       				 31763,32165,32440,32759,33023,33260, $
       				 33502,33776,34035,34393,34646,34949, $
       				 35217,35661,35923,36335,36636,37452, $
       				 37720,37985,38248,38465,38695,38934, $
       				 39162,39383,39605,39826,40050,40347, $
       				 40531,40827,41110,41657,41866,42337, $
       				 42519])
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
	if Date eq '2002-06-02' then begin
		index=scanindex([22982,23225,23463,23753,24032,24273, $
       				 24502,24771,24970,25194,25423,25864, $
       				 26118,26417,26641,27151,27355,27723, $
       				 27982,28285,28481,28767,28961,29624, $
       				 29904,30177,30425,30795,31032,31311, $
       				 31584,31812,32042,32329,32600,33046, $
       				 33324,33645,33895,34155])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([36232,36576,36798,37092,37359,37659, $
       				 37883,38156,38422])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([40005,40188,40563,40789,41048,41373, $
       				 41545,42014,42241])
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
	if Date eq '2002-06-03' then begin
		index=scanindex([23024,23251,23510,23769,23979,24203, $
       				 24490,24757,24972,25197,25423,25663, $
       				 25943,26170,26541,26755,27023,27245, $
       				 27488,27717,28082,28304,28620,28853, $
       				 29091,29312,29583,29795,30603,30833, $
       				 31135,31326,31598,31832,32078,32325, $
       				 32594,32839,33168,33413])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([35210,36938,37170,37464,37648,37928, $
       				 38171,38440,38621])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([40656,40909,41426,41717])
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
	if Date eq '2002-06-04' then begin
		index=scanindex([25167,25409,26363,26589,26808,27033, $
       				 27347,27575,27895,28208,28724,29145, $
       				 29495,29730,30323,30652,30956,31208, $
       				 31529,31765,32261,32612,32906,33129, $
       				 33463,33774,34002,34221,34493,34739, $
       				 35364,35638,36289,36725,37494,37858, $
       				 38379,38596,39515,39912,41198,41528, $
       				 41969,42211])
		if index(0) ge 0 then scantable(index).station(4)=0
		genconfig.baselineid(*,0)=['AE0-E02','AE0-AW0','AN0-AW0', $
					   'AE0-AN0','E02-AN0','E02-AW0']
		genconfig.baselineid(*,1)=['E02-AC0','AW0-AC0','W07-AW0', $
					   'W07-AC0','W07-E02','E02-AW0']
		fixed=1
	endif
	if Date eq '2002-06-05' then begin
		index=scanindex([24501,24840,25553,25800,26149,26388, $
       				 26874,27222,27575,27823,28387,28624, $
       				 30198,30536,30836,31064,31423,31682, $
       				 32228,32449,33111,33399,34107,34436, $
       				 34719,34986,35509,35837,36958,37281, $
       				 37618,37928,38298,38556,39900,40151, $
       				 40463,40763,41375,41683,42255,42482])
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
	if Date eq '2002-06-06' then begin
		index=scanindex([24415,24639,24892,25138,25470,25760, $
       				 25984,26215,26466,26684,27040,27277, $
       				 27668,27904,28297,28536,28904,29155, $
       				 29390,29624,29869,30096,30345,30569, $
       				 31029,31253,31470,31899,32117,32339, $
       				 32595,32834,33065,33287,33518,33740, $
       				 34019,34246,34552,34787,35372,35606, $
       				 35895,36128,36393,36654,36884,37106, $
       				 37837,38094,38462,38710,38958,39231, $
       				 39665,40131,40349,40675,40939,41544, $
       				 41788])
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
	if Date eq '2002-06-07' then begin
		index=scanindex([22191,22441,22743,23020,23272,23526, $
       				 23833,24032,24480,24703,25043,25221, $
       				 25439,25685,25917,26200,26384,26612, $
       				 26833,27116,27337,27671,27899,28170, $
       				 28365,28632,28867,29189,29362,29605, $
       				 29904,30128,30433,30700,31135,31436, $
       				 31867,32082,32327])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([34307,34554,35136,35594,35797,36096, $
       				 36326,36681,36921,37529,37749,38040, $
       				 38238]) 
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([41440,41635,41904,42118,42500,42694]) 
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
	if Date eq '2002-06-08' then begin
		scantable.station(4)=0
		fixed=1
	endif
	if Date eq '2002-06-10' then begin
		index=scanindex([23872,24077,24320,24570,24859,25145,25443, $
       				 25692,26071,26379,26780,27095,27764,28006, $
       				 28243,28476,29125,29455,29679,29902,31142, $
       				 31460,33206,33442,33906,34181,34689,34917, $
       				 35245]) 
		if index(0) ge 0 then scantable(index).station(4)=0
		fixed=1
	endif
        if Date eq '2002-06-23' then begin
                index=scanindex([14621,14830,15174,15470,17026,17272,17678, $
                                 17968,19688,20012,20233,21088,21289,21547, $
                                 21778,23143,23382,23600,25147,25387,25599, $
                                 26964,27194,27412,27640,28798,29054,29294])
                if index(0) ge 0 then scantable(index).station(4)=0
                GenConfig.Wavelength(*,0)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
                                           542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
                GenConfig.Wavelength(*,1)=[862.2,831.6,803.5,777.5,753.5,731.3,710.5,552.5, $
                                           542.5,656.3,533.0,524.0,515.4,598.3,585.9,574.1]*1e-9
                fixed=1
        endif

	if Date eq '2002-06-30' then begin
		scantable.station(4)=0
		fixed=1
	endif
	if Date eq '2002-07-01' then begin
		scantable.station(4)=0
		index=scanindex([25485,25840,26068,27117,27354,28570,28760, $
				 29836,33099,33327,33562,33908,35329,35671, $
				 36579,36766,37035,37271,37493,38579,38855, $
				 39629,39853,40820,41061,41434,41682]) 
		if index(0) ge 0 then scantable(index).station(5)=0
		fixed=1
	endif
	if Date eq '2002-07-06' then begin
		scantable.station(5)=0
		index=scanindex([23132,23404,23632,23851,24263,25774,26039, $
       				 26257,26494,26714,26969,27197,27416,27634, $
       				 27853,28213,28567,28786,29008,29226,29635, $
       				 29898,30192,30412,30655])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([23851,26257,27416,28786,30192])
		if index(0) ge 0 then scantable(index).station([1,2,3])=0
		index=scanindex([24263,26494,27634,29008,30412])
		if index(0) ge 0 then scantable(index).station([0,1,2])=0
		index=scanindex([25774,26714,27853,29226,30655])
		if index(0) ge 0 then scantable(index).station([0,3])=0
		index=scanindex([31895,33517,35087,37063,39750,41816])
		if index(0) ge 0 then scantable(index).station([1,2,3,4])=0
		index=scanindex([32114,33736,35309,37332,40057,42033])
		if index(0) ge 0 then scantable(index).station([0,1,2,4])=0
		index=scanindex([32335,33954,35529,37552,40293,42325])
		if index(0) ge 0 then scantable(index).station([0,3,4])=0
		index=scanindex([32555,34178,35748,37769,40585,42543])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-07-07' then begin
		scantable.station(5)=0
		index=scanindex([15318,15576,15796,16055,16275,20235,20459, $
       				 20696,20915,21134,21528,21773,21962,22179, $
       				 22398,22969,23222,23439,23658,23875,24136, $
       				 24374,24627,24845,25073,25295,25545,25771, $
       				 25991,26215,26489,26713,26933,27185,27406, $
       				 28326,28528,28715,29002,29232,29483,29705]) 
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([15796,20696,21962,23439,24627,25771,26933, $
				 29232])
		if index(0) ge 0 then scantable(index).station([1,2,3])=0
		index=scanindex([16055,20915,22179,23658,24845,25991,27185, $
				 29483])
		if index(0) ge 0 then scantable(index).station([0,1,2])=0
		index=scanindex([16275,21134,22398,23875,25073,26215,27406, $
       				 29705])
		if index(0) ge 0 then scantable(index).station([0,3])=0
		index=scanindex([16053,20694,25543,27183,29481,30612,31077, $
				 31315,31637,32245,33307,35116,37909,41705,42669])
		if index(0) ge 0 then scantable(index).code=0
		index=scanindex([30835,32471,34388,36346,38192,39943,41707])
		if index(0) ge 0 then scantable(index).station([1,2,3,4])=0
		index=scanindex([31079,32689,34614,36624,38425,40163,41944,42174])
		if index(0) ge 0 then scantable(index).station([0,1,2,4])=0
		index=scanindex([31317,32909,33309,34836,36842,38887,40382,42394])
		if index(0) ge 0 then scantable(index).station([0,3,4])=0
		index=scanindex([31639,33534,35118,37060,39133,40601,42669])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-07-08' then begin
		scantable.station(5)=0
		index=scanindex([18518,18875,19163,19471,19708,20405,20709, $
       				 20933,21208,21429,21792,22014,22233,22456, $
       				 22814,23100,23418,23636,24348,24605,24830, $
       				 25113,25605,26132,26466,26701,26923,27180, $
       				 27415,27634,27860,28091,28607,28875,29116, $
       				 29338,29597])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([19163,20933,22014,23100,24830,26466,27634, $
				 29116])
		if index(0) ge 0 then scantable(index).station([1,2,3])=0
		index=scanindex([19471,21208,22233,23418,25113,26701,27860, $
				 29338])
		if index(0) ge 0 then scantable(index).station([0,1,2])=0
		index=scanindex([19708,21429,22456,23636,25605,26923,28091, $
				 29597])
		if index(0) ge 0 then scantable(index).station([0,3])=0
		index=scanindex([32617,36231,38384,41367])
		if index(0) ge 0 then scantable(index).station([1,2,3,4])=0
		index=scanindex([32908,36510,41595])
		if index(0) ge 0 then scantable(index).station([0,1,2,4])=0
		index=scanindex([33227,36818,39123,41815])
		if index(0) ge 0 then scantable(index).station([0,3,4])=0
		index=scanindex([33445,37055,40846,42038])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-07-13' then begin
		scantable.station(4)=0
		index=scanindex([22491,24080,25800,27138,29376,31269, $
       				 32738,34289,36531,38057,39588,41196])
		if index(0) ge 0 then scantable(index).station([1,2,3,5])=0
		index=scanindex([22708,24297,26017,27357,29827,32962, $
       				 34511,36752,38288,39809,41459])
		if index(0) ge 0 then scantable(index).station([0,1,2,5])=0
		index=scanindex([22977,24639,26236,27578,28099,30107,33184, $
       				 34740,37017,38530,40027,41689])
		if index(0) ge 0 then scantable(index).station([0,3,5])=0
		index=scanindex([23197,24868,26453,27875,30326,33502, $
       				 34969,37258,38757,40263,42226])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		index=scanindex([22977,27875,33502,37017,38530,41459])
		if index(0) ge 0 then scantable(index).code=0
		fixed=1
	endif
	if Date eq '2002-07-21' then begin
		scantable.code=0
		index=scanindex([37714,39505])
		if index(0) ge 0 then scantable(index).code=1
		index=scanindex([38187,39962])
		if index(0) ge 0 then scantable(index).station([1,2,3,5])=0
		index=scanindex([38459,40183])
		if index(0) ge 0 then scantable(index).station([0,1,2,5])=0
		index=scanindex([38702,40441])
		if index(0) ge 0 then scantable(index).station([0,3,5])=0
		index=scanindex([39004,40659])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-07-23' then begin
;		Incomplete!
		index=scanindex([23244,23469,23720,24059,24289,24528, $
       				 32570,32792,33023,33296,33697,34105])
		if index(0) ge 0 then scantable(index).station(4)=1
		index=scanindex([38187,39962])
		if index(0) ge 0 then scantable(index).station([1,2,3,5])=0
		index=scanindex([38459,40183])
		if index(0) ge 0 then scantable(index).station([0,1,2,5])=0
		index=scanindex([38702,40441])
		if index(0) ge 0 then scantable(index).station([0,3,5])=0
		index=scanindex([39004,40659])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-08' then begin
		index=scanindex([17381,17607,17825,18043,18268,18489, $
       				 18735,19123,19346,19566,19785,20065, $
       				 20368,20587,20812,21033,21261,21482, $
       				 21926,22258,22482,22708,22927,23169, $
       				 23514,23751,23970,24190,24406,25036, $
       				 25426,25674,25913,26133,26355,26574])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([17825,19346,20812,22482,23970,25913, $
       				 28162,30143,31951,33731,37597,39295, $
       				 41080,42686])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([18043,19566,21033,22708,24190,26133, $
       				 28389,30365,32171,33941,37812,39511, $
       				 41303,42908])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([18268,19785,21261,22927,24406,26355, $
       				 29131,30582,32418,34279,38028,39851, $
       				 41538,43126])
		if index(0) ge 0 then scantable(index).station([0,3,4,5])=0
		index=scanindex([18489,20065,21482,23169,25036,26574, $
       				 29636,30804,32718,34628,38308,40116, $
       				 41826,43345])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-09' then begin
		scantable.station(5)=0
		index=scanindex([24974,27309,28856,30884,32655,34124, $
       				 35606,37855,39901,41434,42904])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([25196,27533,29078,31111,32873,34345, $
       				 35827,38108,40122,41651,43122])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([25451,27759,29297,31349,33095,34564, $
       				 36199,38328,40343,41873,43343])
		if index(0) ge 0 then scantable(index).station([0,3,4,5])=0
		index=scanindex([25677,28019,29576,31568,33313,34786, $
       				 36420,39110,40562,42093,43560])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-11' then begin
		index=scanindex([17209,17431,17652,17870,18089,18365, $
       				 18679,18901,19142,19360,19625,19848, $
       				 20153,20373,20591,20813,21031,21271, $
       				 21571,21792,22010,22229,22447,22674, $
       				 23092,23418,23547])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([17652,19360,20813,22229,22447,31043, $
       				 33162,34542,37933,39409,40828,42471])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([17870,19625,21031,22674,23092,31264,33389, $
       				 34763,38153,39631,41048,42691])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([18089,19848,21271,23418,23547,31483,33625, $
       				 34983,38374,39852,41273,42921])
		if index(0) ge 0 then scantable(index).station([0,3,4,5])=0
		index=scanindex([18365,19142,20591,22010,26032,30825, $
       				 32639,34320,37713,39177,40605,42252])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-12' then begin
		scantable.station(5)=0
		index=scanindex([15094,15314,15531,15753,15973,16211, $
       				 16448,16667,16888,17107,17364,17596, $
       				 17813,18032,18252,18750,18976,19198, $
       				 19425,19645,20446,20671,20941,21379, $
       				 21693,21930,22141,22361,22579,22797, $
       				 23059,23296,23513,23733,23951])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([15531,16667,17813,19198,20941,22361, $
       				 23513,25010,26479,28014,29992,31423, $
       				 33184,34923,36538,38888,40348])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([15753,16888,18032,19425,21379,22579, $
       				 23733,25226,26696,28231,30210,31641, $
       				 33402,35144,36757,39107,40565])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([15973,17107,18252,19645,21693,22797, $
       				 23951,25445,26913,28447,30428,31864, $
       				 33618,35363,36974,39338,40787])
		if index(0) ge 0 then scantable(index).station([0,3,4,5])=0
		index=scanindex([25663,27132,28667,30648,32118,33834, $
       				 35587,37191,39555,41007])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-14' then begin
		index=scanindex([17263,17487,17714,17930,18183,18451, $
       				 18866,19092,19314,19538,19756,20048, $
       				 20440,20660,20903,21124,21357,21573, $
       				 22664,22931,23148,23417,23641,23858, $
       				 24234,24492,24710,24932,25590,25807, $
       				 26300,26530,26747,26968])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([17714,19314,20903,23148,24710,26747, $
       				 28367,30018,31679,33498,35250,37390, $
       				 38879,40945,42495])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([17930,19538,21124,23417,24932,26968, $
       				 28588,30240,31912,33723,35468,37609, $
       				 39098,41167,42712])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([18183,19756,21357,23641,25590,27189, $
       				 28858,30478,32198,33966,35690,37825, $
       				 39333,41436,42937])
		if index(0) ge 0 then scantable(index).station([0,3,4,5])=0
		index=scanindex([18451,20048,21573,23858,25807,27444, $
       				 29231,30696,32249,32490,34183,35929,38045, $
       				 39710,41664,43193])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-15' then begin
		index=scanindex([13524,13796,14127,14383,14632,14929, $
       				 15341,15614,15840,16159,16474,16843, $
       				 17256,17487,17708,17927,18149,18377, $
       				 18758,18983,19203,19420,19639,19876, $
       				 20154,20382,20603,20855,21071,21303, $
       				 21628,21876,22157,22376,22603,22819, $
       				 23301,23580,23775,23991,24209,24440, $
       				 24766,25020,25244,25460,25675,25891, $
       				 26693,26941,27146,27371,27587,27882, $
       				 28219,28459,28763,28983,29203,29465])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([14127,15840,17708,19203,20603,22157, $
       				 23775,25244,27146,28763,30694,32095, $
       				 33749,35425,36960,38580,40134,41877])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([14383,16159,17927,19420,20855,22376, $
       				 23991,25460,27371,28983,30918,32313, $
       				 33969,35645,37176,38798,40352,42094])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([14632,16474,18149,19639,21071,22603, $
       				 24209,25675,27587,29203,31169,32532, $
       				 34185,35867,37402,39015,40575,42295])
		if index(0) ge 0 then scantable(index).station([0,3,4,5])=0
		index=scanindex([14929,16843,18377,19876,21303,22819, $
       				 24440,25891,27882,29465,31384,32750, $
       				 34417,36103,37621,39251,40903,42503])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-16' then begin
		index=scanindex([18094,18334,21116,21339,21528,21773, $
       				 21992,22258,22476,22782,23170,23480, $
       				 23931,24189,24501,24789,24991,25212, $
       				 25434,25654])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([21528,21773,24991,26655,30364,32000, $
       				 33780,35476,36943,38898,40666,42151])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([21992,25212,26877,30584,32228,34001, $
       				 35707,37160,39117,40886,42369])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([22258,25434,27101,30813,32458,34227, $
       				 35941,37380,39568,41108,42596])
		if index(0) ge 0 then scantable(index).station([0,3,4,5])=0
		index=scanindex([22476,25654,27328,31102,31258,32683, $
       				 34451,36172,37609,39792,41329,42818])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-17' then begin
		scantable.station([2,4])=0
		index=scanindex([16691,18843,20568,25057,26955,30061, $
       				 32445,34158,35895,37691,40035,41751])
		if index(0) ge 0 then scantable(index).station([1,3,5])=0
		index=scanindex([16946,19063,20796,25343,27333,30288, $
       				 32712,34378,36117,37949,40245,41983])
		if index(0) ge 0 then scantable(index).station([0,3])=0
		index=scanindex([17177,19284,21022,25606,27659,30510, $
       				 32934,34603,36336,38171,40483,42203])
		if index(0) ge 0 then scantable(index).station([0,1,5])=0
		fixed=1
	endif
	if Date eq '2002-08-18' then begin
		scantable.station([2,4])=0
		index=scanindex([14908,17036,18586,20572,22465,24163, $
       				 26481,26702,28324,30155,31776,33412, $
       				 35082,36662,38275,39856,42844])
		if index(0) ge 0 then scantable(index).station([1,3,5])=0
		index=scanindex([15141,17261,18807,20793,22692,24389, $
       				 26702,26922,28542,30372,31993,33632, $
       				 35299,36881,38493,40074,43062])
		if index(0) ge 0 then scantable(index).station([0,3])=0
		index=scanindex([15709,17481,19033,21094,22912,24612, $
       				 26922,27145,28762,30590,32209,33850, $
       				 35516,37101,38709,40291,43281])
		if index(0) ge 0 then scantable(index).station([0,1,5])=0
		fixed=1
	endif
	if Date eq '2002-08-23' then begin
		scantable.station([2,4])=0
		index=scanindex([13185,14787,17790,19479,21277,23174, $
       				 25133,26875,28661,30638,31365,35664, $
       				 40849,41201])
		if index(0) ge 0 then scantable(index).station([1,3,5])=0
		index=scanindex([13406,15010,18023,19703,21502,23427, $
       				 25327,27095,28904,30906,31595,35861])
		if index(0) ge 0 then scantable(index).station([0,3])=0
		index=scanindex([13626,15241,18245,19961,21733,23654, $
       				 25561,27332,29124,31139,36086])
		if index(0) ge 0 then scantable(index).station([0,1,5])=0
		fixed=1
	endif
	if Date eq '2002-08-24' then begin
		scantable.station([2,4])=0
		index=scanindex([14046,15634,17156,17224,18878,20606, $
       				 22255,23795,25391,26973,28536,30155, $
       				 31726,33280,36032,37747,39297,40998, $
       				 42587])
		if index(0) ge 0 then scantable(index).station([1,3,5])=0
		index=scanindex([14313,15852,17420,19094,20828, $
       				 22477,24018,25616,27189,28753,30373, $
       				 31948,33501,36380,37973,39518,41218, $
       				 42807])
		if index(0) ge 0 then scantable(index).station([0,3])=0
		index=scanindex([14533,16080,17638,19314,21046, $
       				 22695,24248,25834,27408,28974,30592, $
       				 32166,33724,36597,38195,39832,41437, $
       				 43032])
		if index(0) ge 0 then scantable(index).station([0,1,5])=0
		fixed=1
	endif
	if Date eq '2002-08-26' then begin
		scantable.station([2,4])=0
		index=scanindex([19674,23822,25330,27032,28657,30229, $
       				 31920,34023])
		if index(0) ge 0 then scantable(index).station([1,3,5])=0
		index=scanindex([19894,24040,25546,27315,28875,30550, $
       				 32144,34243])
		if index(0) ge 0 then scantable(index).station([0,3])=0
		index=scanindex([20116,24258,25764,27534,29095,30775, $
       			 	 32364,34463])
		if index(0) ge 0 then scantable(index).station([0,1,5])=0
		fixed=1
	endif
	if Date eq '2002-08-30' then begin
		scantable.station(4)=0
		index=scanindex([16964,18421,20236,21663,23071,24653, $
       				 26196,27563,29159,30552,32278,33745, $
       				 35472,36906,38450,39933,41934])
		if index(0) ge 0 then scantable(index).station([1,2,3,5])=0
		index=scanindex([17188,18647,20459,21884,23288,24875, $
       				 26431,27783,29378,30769,32510,33970, $
       				 35702,37126,38675,40153,42156])
		if index(0) ge 0 then scantable(index).station([0,1,2,5])=0
		index=scanindex([17406,18864,20678,22102,23505,25096, $
       				 26649,28004,29604,30990,32741,34188, $
       				 35955,37354,38918,40391,42384])
		if index(0) ge 0 then scantable(index).station([0,3,5])=0
		index=scanindex([17622,19084,20895,22327,23723,25320, $
       				 26870,28227,29824,31334,32975,34413, $
       				 36172,37578,39140,40609,42604])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-08-31' then begin
		scantable.station(4)=0
		index=scanindex([14503,15852,17255,19770,21295,22802, $
       				 24476,25954,27508,29023,30417])
		if index(0) ge 0 then scantable(index).station([1,2,3,5])=0
		index=scanindex([14727,16070,17473,19991,21526,23041, $
       				 24701,26238,27748,29248,30636])
		if index(0) ge 0 then scantable(index).station([0,1,2,5])=0
		index=scanindex([14946,16290,17696,17936,20216,21753,23384, $
       				 24948,26471,27985,29467,30855])
		if index(0) ge 0 then scantable(index).station([0,3,5])=0
		index=scanindex([15163,16514,18730,20434,21978,23612, $
       				 25183,26697,28221,29695,31134])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-09-02' then begin
		scantable.station(4)=0
		index=scanindex([22621,24336,25851,28331,29728,31698, $
       				 33966,36067,38059,40518])
		if index(0) ge 0 then scantable(index).station([1,2,3,5])=0
		index=scanindex([24554,26119,28577,30101,31977, $
       				 34295,36286,38328,40747])
		if index(0) ge 0 then scantable(index).station([0,1,2,5])=0
		index=scanindex([22922,24820,26342,28798,30411,32231, $
       				 34578,36664,38633,41070])
		if index(0) ge 0 then scantable(index).station([0,3,5])=0
		index=scanindex([23144,25044,26653,29109,30674,32526, $
       				 34966,36914,38928,41431])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-09-03' then begin
		scantable.station(4)=0
		index=scanindex([27007,28491,30935,32607,34568,37286, $
       				 39750])
		if index(0) ge 0 then scantable(index).station([1,2,3,5])=0
		index=scanindex([27225,28707,31187,32827,34824,37535, $
       				 39970])
		if index(0) ge 0 then scantable(index).station([0,1,2,5])=0
		index=scanindex([27444,28926,31405,33044,35067,37762, $
       				 40277])
		if index(0) ge 0 then scantable(index).station([0,3,5])=0
		index=scanindex([27667,29191,31661,33314,35536,37981, $
       				 40582])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-09-05' then begin
		index=scanindex([33121,33363,33608,33872,34154,35153, $
       				 35411,35630,35852,38175,38654,38893, $
       				 39151,39388,39723,40706,40955,41203, $
       				 41606,41888,42107,42563,42803,42998, $
       				 43248,43591,43821])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([24033,29041,31320,33608,35153,38893, $
       				 41203,42998])
		if index(0) ge 0 then scantable(index).station([1,2,3,4,5])=0
		index=scanindex([24270,29262,31567,33872,35411,39151, $
       				 41606,43248])
		if index(0) ge 0 then scantable(index).station([0,1,2,4,5])=0
		index=scanindex([24609,29520,31883,34154,35630,39388, $
       				 41888,43591])
		if index(0) ge 0 then scantable(index).station([0,3,4,5])=0
		index=scanindex([24987,29886,32113,35852,39723,42107, $
       				 43821])
		if index(0) ge 0 then scantable(index).station([0,1,2,3])=0
		fixed=1
	endif
	if Date eq '2002-09-13' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-09-14' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-09-15' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-09-20' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-09-21' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-09-22' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-09-23' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-09-24' then begin
		scantable.station([1,5])=0
		index=scanindex([25557,27080,27249,27337])
		if index(0) ge 0 then scantable(index).station=0
		index=scanindex([42362,42621,42841,43060,43285,43540,43806, $
       				 44110,44334,44593,44934,45165,45404,45625, $
       				 45843,46227,46464])
		if index(0) ge 0 then scantable(index).station(4)=0
		index=scanindex([25105,26406,28280,29382,30734,32118,34174, $
       				 36043,42841,44110,45404])
		if index(0) ge 0 then scantable(index).station([2,3,4])=0
		index=scanindex([25276,26640,28402,29578,30925,32347,34427, $
       				 36262,43060,44334,45625])
		if index(0) ge 0 then scantable(index).station([2,0,4])=0
		index=scanindex([25429,26767,28546,29796,31175,32581,34592, $
       				 36583,43285,44593,45843])
		if index(0) ge 0 then scantable(index).station([3,0])=0
		fixed=1
	endif
	if Date eq '2002-09-25' then begin
		scantable.station([1,5])=0
		index=scanindex([15486,20259])
		if index(0) ge 0 then scantable(index).station=0
		index=scanindex([14953,16128,17127,19578,20839,22273, $
       				 23283,24768,25799,30798,33259,34634, $
       				 36061,37523,39008,40340,41974,43789])
		if index(0) ge 0 then scantable(index).station([2,3,4])=0
		index=scanindex([15094,16329,17231,19744,21056,22375, $
       				 23491,24883,25922,31021,33476,34850, $
       				 36278,37738,39226,40557,42282,44007])
		if index(0) ge 0 then scantable(index).station([2,0,4])=0
		index=scanindex([15245,16433,17434,19901,21276,22439, $
       				 23587,25053,26100,31238,33719,35069, $
       				 36494,37954,39494,40881,42702,44223])
		if index(0) ge 0 then scantable(index).station([3,0])=0
		fixed=1
	endif
	if Date eq '2002-10-01' then begin
		scantable.station([1,5])=0
		index=scanindex([18654,20295,21644,23759,25419,28540, $
       				 30085,31990,33644,35144,36686,39077, $
       				 40986,42886,44965])
		if index(0) ge 0 then scantable(index).station([2,3,4])=0
		index=scanindex([18959,20412,21861,23983,25641,28804, $
       				 30308,32215,33874,35364,36904,39346, $
       				 41206,43134,45180,45259])
		if index(0) ge 0 then scantable(index).station([2,0,4])=0
		index=scanindex([19191,20628,22124,24342,25892,29032, $
       				 30718,32452,34099,35604,37123,39716, $
       				 41424,43361,45471])
		if index(0) ge 0 then scantable(index).station([3,0])=0
		fixed=1
	endif
	if Date eq '2002-10-09' then begin
		scantable.station([0,4,5])=0
		genconfig.baselineid(1,1)='AC0-AW0'
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-10-10' then begin
		scantable.station([0,4,5])=0
		genconfig.baselineid(1,1)='AC0-AW0'
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-10-11' then begin
		scantable.station([0,4,5])=0
		genconfig.baselineid(1,1)='AC0-AW0'
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-10-14' then begin
		scantable.station([0,4,5])=0
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2002-10-17' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-10-22' then begin
		scantable.station([0,4,5])=0
		fixed=1
	endif
	if Date eq '2002-11-07' then begin
		scantable.station(4)=0
		genconfig.spectrometerid=[1,3]
		fixed=1
	endif
	if Date eq '2003-01-03' then begin
		scantable.station(3)=0
		fixed=1
	endif
	if Date eq '2003-01-04' then begin
		genconfig.beamcombinerid=3
		scantable.station(3)=0
		GenConfig.Wavelength(*,0)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		GenConfig.Wavelength(*,1)=[849.4,820.9,793.9,768.3,744.2,722.9,701.5,683.1, $
					   664.6,648.9,617.7,603.5,590.7,577.9,566.5,556.6]*1e-9
		fixed=1
	endif
	if Date eq '2003-01-05' then begin
		scantable.station(3)=0
		genconfig.beamcombinerid=3
		fixed=1
	endif
	if Date eq '2003-01-06' then begin
		scantable.station(3)=0
		fixed=1
	endif
	if Date eq '2003-01-07' then begin
		scantable.station(3)=0
		fixed=1
	endif
endif
;
if fixed then begin
;	GenInfo[0]=GenConfig
	print,'Fixed early day configuration.'
endif
;
end
;-------------------------------------------------------------------------------
pro fixwavelengths
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(SystemId) eq 0 then begin
	print,'***Error(FIXWAVELENGTHS): SystemId undefined!'
	return
endif
;
if SystemId eq 'NPOI' then begin
for i=0,GenConfig.NumOutBeam-1 do begin
GenConfig.Wavelength(*,i)= $
	[849.4e-9, 820.9e-9, 793.9e-9, 768.3e-9, $
	 744.2e-9, 722.9e-9, 701.5e-9, 683.1e-9, $
	 664.6e-9, 648.9e-9, 631.9e-9, 617.7e-9, $
	 603.5e-9, 590.7e-9, 577.9e-9, 566.5e-9, $
	 556.6e-9, 545.2e-9, 535.2e-9, 526.7e-9, $
	 518.2e-9, 509.6e-9, 501.1e-9, 494.0e-9, $
	 485.6e-9, 478.4e-9, 472.7e-9, 465.6e-9, $
	 459.9e-9, 452.2e-9, 448.5e-9, 442.8e-9]
endfor
endif
;
print,'Fixed wavelengths.'
;
end
;-------------------------------------------------------------------------------
pro fixchanwidths
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(SystemId) eq 0 then begin
	print,'***Error(FIXCHANWIDTHS): SystemId undefined!'
	return
endif
;
if SystemId eq 'NPOI' then begin
for i=0,GenConfig.NumOutBeam-1 do begin
GenConfig.ChanWidth(*,i)= $
	[28.5e-9,28.5e-9,27.0e-9,25.6e-9, $
	 24.1e-9,21.3e-9,21.4e-9,18.4e-9, $
	 18.5e-9,15.7e-9,17.0e-9,14.2e-9, $
	 14.2e-9,12.8e-9,12.8e-9,11.4e-9, $
	  9.9e-9,11.4e-9,10.0e-9, 8.5e-9, $
	  8.5e-9, 8.6e-9, 8.5e-9, 7.1e-9, $
	  8.4e-9, 7.2e-9, 5.7e-9, 7.1e-9, $
	  5.7e-9, 7.7e-9, 3.7e-9, 5.7e-9]
endfor
endif
;
print,'Fixed chanwidths.'
;
end
;-------------------------------------------------------------------------------
pro deadcorr,remove=remove,dt=dt
;
; Apply/remove dead time [s] corrections to visibility data.
; Only works with one baseline per detector.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if system_id(systemid) ne 'NPOI' $
or produkt(genconfig.numbaseline(0:genconfig.numoutbeam-1)) ne 1 then return
;
if n_elements(remove) eq 0 then remove=0
if n_elements(dt) eq 0 then dt=200e-9
;
vissq=reform(scans.vissq)
vissqe=reform(scans.vissqe)
vissqc=reform(scans.vissqc)
vissqerr=reform(scans.vissqcerr)
index=where(vissqerr lt 0,count)
if count gt 0 then begin
	vissq(index)=0
	vissqe(index)=0
	vissqc(index)=0
endif
;
faktor=((1+dt*abs(scans.photonrate)*500*(1-vissq^2)))^2
if remove then faktor=1/faktor
scans(*).vissq(*,*)=vissq*faktor
scans(*).vissqe(*,*)=vissqe*faktor
scans(*).vissqc(*,*)=vissqc*faktor
;
for i=0,genconfig.numtriple-1 do begin
	f1=1+dt*abs(scans(*).photonrate(0,*))*500*(1-vissq(0,*,*)^2)
	f2=1+dt*abs(scans(*).photonrate(1,*))*500*(1-vissq(1,*,*)^2)
	f3=1+dt*abs(scans(*).photonrate(2,*))*500*(1-vissq(2,*,*)^2)
	faktor=f1*f2*f3
	if remove then faktor=1/faktor
	scans.tripleamp=scans.tripleamp*faktor
	scans.tripleampe=scans.tripleampe*faktor
	scans.tripleampc=scans.tripleampc*faktor
endfor
;
if remove then print,'Dead time corrections removed.' $
	  else print,'Dead time corrections applied.'
;
end
;-------------------------------------------------------------------------------
pro moiree
;
; Remove Moiree effect from AMBER MR differential phases (caused by 
; polarizers, which were removed in 2008).
;
; Note: AMBER HR data need to be filtered with a low-pass filter.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
for i=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline(i)-1 do begin
for k=0,n_elements(scans)-1 do begin
;
diffphase=reform(scans(k).diffphase(i,0:genconfig.numspecchan(i)-1,l))
scans(k).diffphasec(i,0:genconfig.numspecchan(i)-1,l)=highpass(diffphase)
vissqc=reform(scans(k).vissqc(i,0:genconfig.numspecchan(i)-1,l))
scans(k).vissqc(i,0:genconfig.numspecchan(i)-1,l)=highpass(vissqc)
;
endfor
endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro iris
;
; Remove IRIS dichroic fringes from AMBER HR amplitudes and differential phases.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
for i=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline(i)-1 do begin
for k=0,n_elements(scans)-1 do begin
;
diffphase=reform(scans(k).diffphase(i,0:genconfig.numspecchan(i)-1,l))
scans(k).diffphasec(i,0:genconfig.numspecchan(i)-1,l)=notch(diffphase)
vissqc=reform(scans(k).vissqc(i,0:genconfig.numspecchan(i)-1,l))
scans(k).vissqc(i,0:genconfig.numspecchan(i)-1,l)=notch(vissqc)
;
endfor
endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro medianob,all=all
;
; To be used for AMBER with MR or HR modes.
;
; For of TF tests, run all visibilities and phases through median filter.
; Only do this for VissqE and VissqEC, as these can easily be recreated
; with calcviscal from Vissq and VissqC (unless "all" keyword is given).
;
; Also compute a median for the calibrated triple phases. These will be
; initialized to the uncalibrated phases.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(all) eq 0 then all=0
;
; Recreate VisSqE and VisSqEC
calcviscal
;
for i=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline(i)-1 do begin
for k=0,n_elements(scans)-1 do begin
;
vissqe=reform(scans(k).vissqe(i,0:genconfig.numspecchan(i)-1,l))
vissqeerr=reform(scans(k).vissqeerr(i,0:genconfig.numspecchan(i)-1,l))
index=where(vissqeerr gt 0,count)
if count gt 0 then begin
scans(k).vissqe(i,0:genconfig.numspecchan(i)-1,l)=medianve(vissqe(index),e)
scans(k).vissqeerr(i,0:genconfig.numspecchan(i)-1,l)=e
endif
vissqec=reform(scans(k).vissqec(i,0:genconfig.numspecchan(i)-1,l))
vissqecerr=reform(scans(k).vissqecerr(i,0:genconfig.numspecchan(i)-1,l))
index=where(vissqecerr gt 0,count)
if count gt 0 then begin
scans(k).vissqec(i,0:genconfig.numspecchan(i)-1,l)=medianve(vissqec(index),e)
scans(k).vissqecerr(i,0:genconfig.numspecchan(i)-1,l)=e
endif
if all then begin
vissq=reform(scans(k).vissq(i,0:genconfig.numspecchan(i)-1,l))
vissqerr=reform(scans(k).vissqerr(i,0:genconfig.numspecchan(i)-1,l))
index=where(vissqerr gt 0,count)
if count gt 0 then begin
scans(k).vissq(i,0:genconfig.numspecchan(i)-1,l)=medianve(vissq,e)
scans(k).vissqerr(i,0:genconfig.numspecchan(i)-1,l)=e
endif
vissqc=reform(scans(k).vissqc(i,0:genconfig.numspecchan(i)-1,l))
vissqcerr=reform(scans(k).vissqcerr(i,0:genconfig.numspecchan(i)-1,l))
index=where(vissqcerr gt 0,count)
if count gt 0 then begin
scans(k).vissqc(i,0:genconfig.numspecchan(i)-1,l)=medianve(vissqc,e)
scans(k).vissqcerr(i,0:genconfig.numspecchan(i)-1,l)=e
endif
endif
;
endfor
endfor
endfor
;
for k=0,n_elements(scans)-1 do begin
for i=0,genconfig.numtriple-1 do begin
	triplephasec=reform(scans(k).triplephase(i,0:genconfig.triplenumchan(i)-1))
	triplephasecerr=reform(scans(k).triplephaseerr(i,0:genconfig.triplenumchan(i)-1))
	index=where(triplephasecerr gt 0,count)
	if count gt 0 then begin
	scans(k).triplephasec(i,0:genconfig.triplenumchan(i)-1)=medianve(triplephasec(index),e)
	scans(k).triplephasecerr(i,0:genconfig.triplenumchan(i)-1)=e
	endif
endfor
endfor
;
end
;************************************************************************Block 3
pro create_calentries,num_e
;
; Set available indicators and functions. Indicators are referenced by sequence
; number in set_caldata! Functions are also used by other fit-routines.
; Create and initialize array of cal_entry structures.
;
common BaseFunctions,numlist,functions
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
functions=[ $
	'1', $
	'x', $
	'x^2', $
	'x^3', $
	'x^4', $
;
	'L_0', $
	'L_1', $
	'L_2', $
	'L_3', $
	'L_4', $
;
	'P_0', $
	'P_1', $
	'P_2', $
	'   ', $
	'   ', $
;
	'S_5' , $
	'S_10', $
	'S_20', $
	'S_40', $
	'S_80' $
	]
num_f=n_elements(functions)
numlist=[5,5,5,5]
;
indicators=[ $
	'Wavelength', $
	'DelayJitter', $
	'TrackJitter', $
	'ZenithAngle', $
	'MirrorAngle', $
	'HourAngle', $
	'Time']
num_i=n_elements(indicators)
;
num_t=num_i*num_f	; Maximum number of terms
;
list_stars,stars
num_st=n_elements(stars)
;
num_sc=n_elements(scanconfig(scanconfig()))
;
if n_elements(num_e) eq 0 then $
	num_e=(long(total(GenConfig.NumBaseline*GenConfig.NumSpecChan)) $
	      +long(total(GenConfig.TripleNumChan*2)))*num_st
num_e=max([num_e,10])
;
cal_entries=alloc_cal_entries(num_f,num_i,num_t,num_st,num_sc,num_e)
;
end
;-------------------------------------------------------------------------------
pro set_calentries,entries
;
; Initialize cal_entries with entries.
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
;
n=n_elements(entries)
if n eq 0 then begin
	print,'***Error(SET_CALENTRIES): entries undefined!'
	return
endif
create_calentries
;
if n gt n_elements(cal_entries) then begin
	create_calentries,n
	print,'Warning(SET_CALENTRIES): Had to increase table size!'
endif
;
m1=min([n_elements(cal_entries(0).stars),n_elements(entries(0).stars)])
m2=min([n_elements(cal_entries(0).scans),n_elements(entries(0).scans)])
;
tags=tag_names(entries)
for i=0,n_tags(entries)-1 do begin
case tags(i) of
	'VARIABLE':	cal_entries(0:n-1).variable=entries.variable
	'REASON':	cal_entries(0:n-1).reason=entries.reason
	'OUTBEAM':	cal_entries(0:n-1).outbeam=entries.outbeam
	'TRIPLE':	cal_entries(0:n-1).triple=entries.triple
	'BASELINE':	cal_entries(0:n-1).baseline=entries.baseline
	'CHANNEL':	cal_entries(0:n-1).channel=entries.channel
	'STARS':	cal_entries(0:n-1).stars(0:m1-1)=entries(0:n-1).stars(0:m1-1)
	'SCANS':	cal_entries(0:n-1).scans(0:m2-1)=entries(0:n-1).scans(0:m2-1)
	'CALSTARS':	cal_entries(0:n-1).calstars(0:m1-1)=entries(0:n-1).calstars(0:m1-1)
	'INDICATORS':	cal_entries(0:n-1).indicators=entries.indicators
	'FUNCTIONS':	cal_entries(0:n-1).functions=entries.functions
	'COEFFS':	cal_entries(0:n-1).coeffs=entries.coeffs
	'X_MID':	cal_entries(0:n-1).x_mid=entries.x_mid
	'X_SCL':	cal_entries(0:n-1).x_scl=entries.x_scl
	'NUM_INDCT':	cal_entries(0:n-1).num_indct=entries.num_indct
	'NUM_FUNCN':	cal_entries(0:n-1).num_funcn=entries.num_funcn
	'NUM_COEFFS':	cal_entries(0:n-1).num_coeffs=entries.num_coeffs
	'NUM_SCANS':	cal_entries(0:n-1).num_scans=entries.num_scans
	else:		print,'***Error(SET_CALENTRIES): unknown tag (',tags(i),')!'
endcase
endfor
;
end
;-------------------------------------------------------------------------------
function calmatrix,scan_index,outbeam,channel,baseline,x_mid,x_scl
;
; Return the part of the calibration design matrix that corresponds to the 
; selected data, as reflected in scan_index.
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
;
num_indicators=n_elements(sel_indct)
num_functions=intarr(num_indicators)
num_terms=1
for i=0,num_indicators-1 do begin
	index=where(sel_funcn(i,*) ne '',count)
	num_functions(i)=count
	num_terms=num_terms*count
endfor
;
scan_count=n_elements(scan_index)
M=fltarr(scan_count,num_terms,/nozero)
;
term_index=intarr(num_indicators)
for i=0,num_terms-1 do begin
	M(*,i)=1
	for j=0,num_indicators-1 do begin
		index=where(indicators eq sel_indct(j))+1
		x=set_caldata(index(0),outbeam,channel,baseline,scan_index)
		M(*,i)=M(*,i) $
		      *evalfunction(sel_funcn(j,term_index(j)),x, $
					x_mid(j),x_scl(j))
	endfor
;
	if i lt num_terms-1 then begin
		j=num_indicators
		repeat begin
			j=j-1
			term_index(j)=term_index(j)+1
		endrep until term_index(j) lt num_functions(j)
		if (j+1) le (num_indicators-1) then $
			term_index(j+1:num_indicators-1)=0
	endif
endfor
;
return,M
;
end
;-------------------------------------------------------------------------------
pro calcoeffs,reason
;
; Compute calibration coefficients from given calibrator data and fit selection.
; Store information in one element of array cal_entries for each baseline and
; channel.
;
; The new self-calibration option (cal_options.s=1) was created for visibility
; spectra of lines, allowing the continuum to be self-calibrated to the level
; of the estimated visibility (using the continuum diameter).
;
common BaseFunctions,numlist,functions
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
reason=''
;
; Check indicator and function selection
if strlen(sel_indct(0)) eq 0 then begin
	print,'***Error(CALCOEFFS): no indicators selected!'
	return
endif
index=where(sel_funcn(*,0) ne '',count)
if count eq 0 then begin
	print,'***Error(CALCOEFFS): no base functions selected!'
	return
endif
if count lt n_elements(sel_indct) then begin
	print,'***Error(CALCOEFFS): no base functions selected for one or '+ $
		'more indicators!'
	return
endif
;
; Check calibrator star selection
if n_elements(cal_stars) eq 0 then begin
	print,'***Error(CALCOEFFS): cal_stars undefined!'
	return
endif else if cal_stars(0) eq '' then begin
	print,'***Error(CALCOEFFS): no calibration stars selected!'
	return
endif
;
; Prepare data selection via what is user-specified in the plot widgets
; Note that we use the 'y'-stream data selection!
flag=set_starsel()
if set_indexsel('y') ne 0 then begin
	print,'***Error(CALCOEFFS): invalid data selection!'
	return
endif
set_streamsel,'y',item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,scan_index
index=where(ds_ch ge genconfig.numspecchan(ds_ob),count)
if count gt 0 then begin
	print,'***Error(CALCOEFFS): invalid channel selection!'
	return
endif
index=where(ds_bl ge genconfig.numbaseline(ds_ob),count)
if count gt 0 then begin
	print,'***Error(CALCOEFFS): invalid baseline selection!'
	return
endif
scan_count=n_elements(scan_index)
;
; Obtain index into selected scans corresponding to calibrator selection
calscan_index=scan_index
for k=0,n_elements(calscan_index)-1 do begin
	index=where(cal_stars eq scantable(calscan_index(k)).starid,count)
	if count eq 0 then calscan_index(k)=-1
endfor
index=where(calscan_index ge 0,calscan_count)
if calscan_count eq 0 then begin
	print,'***Error(CALCOEFFS): There are no selected scans for this calibrator selection!'
	return
endif
calscan_index=calscan_index(where(calscan_index ge 0,calscan_count))
;
; Set/determine the scaling parameters
x_mid=fltarr(n_elements(sel_indct))
x_scl=fltarr(n_elements(sel_indct))
for i=0,n_elements(sel_indct)-1 do begin
	index=where(indicators eq sel_indct(i)) & index=index(0)+1
	x=set_caldata(index,0,0,0,calscan_index,mid,scl)
	x_mid(i)=mid
	x_scl(i)=scl
endfor
;
; Create array of cal_entry structures, if not done before
if n_elements(cal_entries) eq 0 then create_calentries
cal_reason=strmid(systime(),0,24)
;
; The loop option is equivalent to repeated calls to this function
if cal_options.l then begin
	num_b_loop=n_elements(ds_bl)
	num_c_loop=n_elements(ds_ch)
	ds_bl_bck=ds_bl
	ds_ch_bck=ds_ch
	num_p_loop=calscan_count
	if cal_options.s then num_c_loop=1 $
			 else num_p_loop=1
endif else begin
	num_b_loop=1
	num_c_loop=1
	num_p_loop=1
endelse
if cal_options.v eq 1 or cal_options.v eq 2 then num_b_loop=1
;
FOR bloop=0,num_b_loop-1 DO BEGIN
FOR ploop=0,num_p_loop-1 DO BEGIN
FOR cloop=0,num_c_loop-1 DO BEGIN
;
; Loop over selected baselines and channels of a selected output beam.
;
; Update the data selection
if cal_options.l then begin
	ds_bl=ds_bl_bck(bloop)
	ds_ch=ds_ch_bck(cloop)
	ds_pt=calscan_index(ploop)
	if cal_options.s then ds_ch=ds_ch_bck $
			 else ds_pt=calscan_index
endif else begin
	ds_pt=calscan_index
endelse
;
num_point=n_elements(ds_pt)
num_channel=n_elements(ds_ch)
num_baseline=n_elements(ds_bl) & if cal_options.v gt 0 then num_baseline=1
first=1
block=0
;
; Set up design matrix or average variable values, using selected cal data
for ib=0,num_baseline-1 do begin
for ic=0,num_channel-1 do begin
	channel=ds_ch(ic)
	baseline=ds_bl(ib)
	if cal_options.v eq 1 then begin
;	Item 31 is TripleAmp c/e
	R=set_plotdata(31,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)
	E=set_ploterr(31,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)
	endif else if cal_options.v eq 2 then begin
;	Item 33 is TriplePhaseC
	R=set_plotdata(33,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)/RAD
	E=set_ploterr(33,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)/RAD
	endif else begin
;	Item 27 is VisSq c/e
	R=set_plotdata(27,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)
	E=set_ploterr(27,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)
	endelse
;
	index=where(strpos(sel_funcn(*,0),'S_') ge 0,count)
	if count ne 0 then begin
		do_smooth=1
		if first then begin
			smooth_time=float(strmid(sel_funcn(index(0),0),2,2))/60
			i=where(indicators eq sel_indct(index(0))) & i=i(0)
			DM=set_caldata(i+1,0,0,0,indgen(n_elements(ScanTable)))
			RM=fltarr(calscan_count)
			RE=fltarr(calscan_count)
			first=0
		endif
		index=where(E le 0,count)
		if count gt 0 then E(index)=1
		wt=1/E^2
		if count gt 0 then wt(index)=0
		RM=RM+R*wt
		RE=RE+wt	; we store the weight here in RE for simplicity
	endif else begin
		do_smooth=0
		M=calmatrix(ds_pt,ds_ob,channel,baseline,x_mid,x_scl)
		if first then begin
		   nscans=n_elements(M(*,0))
		   nterms=n_elements(M(0,*))
		   if nterms gt n_elements(cal_entries(0).coeffs) then begin
			   print,'***Error(CALCOEFFS): too many terms!'
			   return
		   endif
		   DM=fltarr(nscans*num_baseline*num_channel,nterms,/nozero)
		   RM=fltarr(nscans*num_baseline*num_channel,/nozero)
		   RE=fltarr(nscans*num_baseline*num_channel,/nozero)
		   first=0
		endif
;		if cal_options.s then index=ploop else index=indgen(nscans)
		DM(block*nscans:(block+1)*nscans-1,*)=M
		RM(block*nscans:(block+1)*nscans-1)=R
		RE(block*nscans:(block+1)*nscans-1)=E
		block=block+1
	endelse
endfor
endfor
;
; Edit and weight the data
index=where(RE gt 0,count)
IF count EQ 0 THEN BEGIN
	if cal_options.s then begin
	print,'Warning(CALCOEFFS): no valid data points found! OB=',ds_ob+1, $
		', Bl=',ds_bl+1,', Pt=',ds_pt+1,format='(a,i2,a,i2,a,i3)'
	endif else begin
	print,'Warning(CALCOEFFS): no valid data points found! OB=',ds_ob+1, $
		', Bl=',ds_bl+1,', Ch=',retroparse(ds_ch+1),format='(a,i2,a,i2,a,a)'
	endelse
ENDIF ELSE BEGIN
if do_smooth then begin
;
	h=DM(scan_index)
	first_hour=min(h)-1
	last_hour=max(h)+1
	DM=DM(calscan_index)
;
	WT=RE(index)
	RM=RM(index)/WT
	DM=DM(index)
;
;
	coeffs=cal_entries(0).coeffs*0
	num_coeffs=n_elements(coeffs)
	hour_grid=first_hour $
		 +findgen(num_coeffs)/(num_coeffs-1)*(last_hour-first_hour)
	for i=0,num_coeffs-1 do begin
		exponents=((hour_grid(i)-DM)/smooth_time)^2
;		index=where(exponents gt 20,count)
;		if count gt 0 then exponents(index)=20
		index=where(exponents lt 20,count)
		if count eq 0 then index=where(exponents eq min(exponents))
		exponents=exponents(index) < 20
		WTE=WT(index)
		RME=RM(index)
		weighting=exp(-exponents)
		hwh=weighting*WTE
		coeffs(i)=total(RME*hwh)/total(hwh)
	endfor
	si=sort(DM)
	if cal_options.v eq 2 then $
	rms=sqrt(total((RM(si)-spline(hour_grid,coeffs,DM(si)))^2) $
                /n_elements(RM)) else $
	rms=sqrt(total((1-RM(si)/spline(hour_grid,coeffs,DM(si)))^2) $
                /n_elements(RM))
;
endif else begin
;
	RM=RM(index)
	RE=RE(index)
	if cal_options.s then RE(*)=1	; not yet figured out why
	DM=DM(index,*)
	if cal_options.s then begin
		v=medianve(RM,e)
		index=where(abs(RM-v) le 2*e)
		RM=RM(index)
		RE=RE(index)
		DM=DM(index,*)
	endif
	WT=1/RE
	RM=RM*WT
	for i=0,nterms-1 do DM(*,i)=DM(*,i)*WT
;
	RM=double(RM)
	DM=double(DM)
;
; 	Solve for calibration coefficients
	T=transpose(DM)
	N=T#DM
	R=T#RM
;
; 	Solution with SVD if small eigenvalues occur, direct otherwise
	if nterms gt 1 then begin
		svd,N,w,u,v
		small=where(w lt max(w)*1.0e-10,count)
		if count gt 0 then begin
			print,'Warning(CALCOEFFS):',count, $
				' eigenvalue(s) edited!'
			w(small)=0.0
			svbksb,u,w,v,R,coeffs
		endif else begin
			coeffs=invert(N)#R	
		endelse
		if cal_options.v eq 2 then $
		rms=sqrt(total(((RM-(DM#coeffs))/WT)^2) $
		         /n_elements(RM)) else $
		rms=sqrt(total((1-RM/(DM#coeffs))^2) $
		         /n_elements(RM))
	endif else begin
		coeffs=1/N*R
		if cal_options.v eq 2 then $
		rms=sqrt(total(((RM-(DM*coeffs(0)))/WT)^2) $
		         /n_elements(RM)) else $
		rms=sqrt(total((1-RM/(DM*coeffs(0)))^2) $
		         /n_elements(RM))
	endelse
	num_coeffs=n_elements(coeffs)
;
endelse
;
; Store calibration information in array of structures cal_entries.
;
index=where(cal_entries.num_coeffs gt 0,count)
if count eq n_elements(cal_entries) then begin
	print,'***Error(CALCOEFFS): structure cal_entries full!'
	return
endif
id=count
;
num_indct=n_elements(sel_indct)
num_funcn=n_elements(sel_funcn(0,*))
num_stars=n_elements(ds_stars)
num_calstars=n_elements(cal_stars)
for ib=0,num_baseline-1 do begin
for ic=0,num_channel-1 do begin
	cal_entries(id).reason=cal_reason
	cal_entries(id).variable=cal_options.v
	cal_entries(id).outbeam=ds_ob
	cal_entries(id).triple=ds_tr
	cal_entries(id).channel=ds_ch(ic)
	cal_entries(id).baseline=ds_bl(ib)
	cal_entries(id).stars(0:num_stars-1)=ds_stars
	if cal_options.s then cal_entries(id).scans(0)=ds_pt $
			 else cal_entries(id).scans(0:scan_count-1)=scan_index
	cal_entries(id).calstars(0:num_calstars-1)=cal_stars
	cal_entries(id).indicators(0:num_indct-1)=sel_indct
	cal_entries(id).x_mid(0:num_indct-1)=x_mid
	cal_entries(id).x_scl(0:num_indct-1)=x_scl
	cal_entries(id).functions(0:num_indct-1,0:num_funcn-1)=sel_funcn
	cal_entries(id).coeffs(0:num_coeffs-1)=coeffs
	cal_entries(id).num_indct=num_indct
	cal_entries(id).num_funcn=num_funcn
	cal_entries(id).num_coeffs=num_coeffs
	if cal_options.s then cal_entries(id).num_scans=1 $
			 else cal_entries(id).num_scans=scan_count
	id=id+1
endfor
endfor
;
; Diagnostic messages
if cal_options.l then begin
if cal_options.v eq 2 then trms='. (RMS [d] =' else trms='. (RMS [%] ='
if cal_options.v eq 2 then frms=RAD            else frms=100.0
if cal_options.v eq 0 then begin
if cal_options.s then $
print,'Solution calculated for OB=',ds_ob+1,', Bl=',ds_bl+1,', Pt=',ds_pt+1, $
	trms,frms*rms,')',format='(a,i2,a,i2,a,i3,a,f6.2,a)' $
else $
print,'Solution calculated for OB=',ds_ob+1,', Bl=',ds_bl+1,', Ch=',ds_ch+1, $
	trms,frms*rms,')',format='(a,i2,a,i2,a,i3,a,f6.2,a)'
endif
if cal_options.v ge 1 then $
print,'Solution calculated for TR=',ds_tr+1,', Ch=',ds_ch+1, $
	trms,frms*rms,')',format='(a,i2,a,i3,a,f6.3,a)'
endif else print,'Solution calculated.'
ENDELSE
;
ENDFOR
ENDFOR
ENDFOR
;
reason=cal_reason
;
end
;-------------------------------------------------------------------------------
pro calibrate,cal_reason,undo
;
; Calibrate scans using a selection of the calibration entries, given
; by the variable (cal_options.v) and the reason (cal_options.reason). 
; Apply only to stars listed in the entry.
; If undo flag is set (1), remove last calibration from data.
; Calibrate only valid scans, not flagged visibilities!
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if n_elements(cal_reason) eq 0 then begin
	print,'***Error(CALIBRATE): cal_reason undefined!'
	return
endif
if n_elements(undo) eq 0 then begin
	undo=0
;	print,'***Error(CALIBRATE): undo flag undefined!'
;	return
endif
if undo ne 1 and undo ne 0 then begin
	print,'***Error(CALIBRATE): undo flag has to be 1(uncal) or 0(cal)!'
	return
endif
;
entry_index=where((cal_entries.variable eq cal_options.v) and $
		  (cal_entries.reason eq cal_reason),entry_count)
if entry_count eq 0 then begin
	print,'***Error(CALIBRATE): no entries found for this selection!'
	return
endif
;
; Work on each entry
for n=0,entry_count-1 do begin
	nn=entry_index(n)
	ib=0
	ob=cal_entries(nn).outbeam
	tr=cal_entries(nn).triple
	ch=cal_entries(nn).channel
	bl=cal_entries(nn).baseline
	scan_count=cal_entries(nn).num_scans
	scan_index=cal_entries(nn).scans(0:scan_count-1)
;
	coeffs=cal_entries(nn).coeffs(0:cal_entries(nn).num_coeffs-1)
	sel_indct=cal_entries(nn).indicators(0:cal_entries(nn).num_indct-1)
	sel_funcn=cal_entries(nn).functions(0:cal_entries(nn).num_indct-1, $
					    0:cal_entries(nn).num_funcn-1)
	index=where(strpos(cal_entries(nn).functions(*,0),'S_') ge 0,count)
	if count gt 0 then begin
	   i=where(indicators eq sel_indct(index(0))) & i=i(0)
	   num_coeffs=n_elements(coeffs)
	   h=set_caldata(i+1,0,0,0,scan_index)
	   first_hour=min(h)-1
	   last_hour=max(h)+1
	   hour_grid=first_hour $
		 	 +findgen(num_coeffs)/(num_coeffs-1) $
			 *(last_hour-first_hour)
	   if n_elements(h) gt 1 then si=sort(h) else si=0
	   R=fltarr(n_elements(h))
	   R(si)=spline(hour_grid,coeffs,h(si))
	endif else begin
	   x_mid=cal_entries(nn).x_mid(0:cal_entries(nn).num_indct-1)
	   x_scl=cal_entries(nn).x_scl(0:cal_entries(nn).num_indct-1)
	   M=calmatrix(scan_index,ob,ch,bl,x_mid,x_scl)
	   if cal_entries(nn).num_coeffs eq 1 then R=M*coeffs(0) $
	  				      else R=M#coeffs
	endelse
	if cal_options.v eq 1 then begin
;	TripleAmp c & c/e
	   if undo then R=1/R
	   v=set_ploterr(29,ib,ob,tr,ch,bl,scan_index)
	   index=where(v gt 0,count)
	   if count gt 0 then begin
	     scans(scan_index(index)).TripleAmpCErr(tr,ch)=v(index)/abs(R(index))
	     v=set_plotdata(29,ib,ob,tr,ch,bl,scan_index)
	     scans(scan_index(index)).TripleAmpC(tr,ch)=v(index)/R(index)
	   endif
	   v=set_ploterr(31,ib,ob,tr,ch,bl,scan_index)
	   index=where(v gt 0,count)
	   if count gt 0 then begin
	     scans(scan_index(index)).TripleAmpECErr(tr,ch)=v(index)/abs(R(index))
	     v=set_plotdata(31,ib,ob,tr,ch,bl,scan_index)
	     scans(scan_index(index)).TripleAmpEC(tr,ch)=v(index)/R(index)
	   endif
	endif else if cal_options.v eq 2 then begin
;	TriplePhase c
	   if undo then R=-R
	   v=set_ploterr(33,ib,ob,tr,ch,bl,scan_index)/RAD
	   index=where(v gt 0,count)
	   if count gt 0 then begin
	     scans(scan_index(index)).TriplePhaseCErr(tr,ch)=v(index)
	     v=set_plotdata(33,ib,ob,tr,ch,bl,scan_index)/RAD
	     scans(scan_index(index)).TriplePhaseC(tr,ch)=v(index)-R(index)
	   endif
	endif else begin
;	VisSq c & c/e
	   if undo then R=1/R
	   v=set_ploterr(25,ib,ob,tr,ch,bl,scan_index)
	   index=where(v gt 0,count)
	   if count gt 0 then begin
	     scans(scan_index(index)).VisSqCErr(ob,ch,bl)=v(index)/abs(R(index))
	     v=set_plotdata(25,ib,ob,tr,ch,bl,scan_index)
	     scans(scan_index(index)).VisSqC(ob,ch,bl)=v(index)/R(index)
	   endif
	   v=set_ploterr(27,ib,ob,tr,ch,bl,scan_index)
	   index=where(v gt 0,count)
	   if count gt 0 then begin
	     scans(scan_index(index)).VisSqECErr(ob,ch,bl)=v(index)/abs(R(index))
	     v=set_plotdata(27,ib,ob,tr,ch,bl,scan_index)
	     scans(scan_index(index)).VisSqEC(ob,ch,bl)=v(index)/R(index)
	   endif
	endelse
;
endfor
;
stars=unique(scans(scan_index).starid)
if undo then print,'Calibration removed from stars ',stars+' ','.',format='(7a)' $
	else print,'Calibration applied  to  stars ',stars+' ','.',format='(7a)'
;
end
;-------------------------------------------------------------------------------
pro uncal
;
; Removes calibration for a given data selection. All scans.
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; Note that we use the 'y'-stream dataselection!
flag=set_starsel()
if set_indexsel('y') ne 0 then begin
	print,'***Error(RESETCAL): invalid data selection!'
	return
endif
set_streamsel,'y',item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,ds_pt
;
num_channel=n_elements(ds_ch)
num_baseline=n_elements(ds_bl)
if cal_options.v gt 0 then num_baseline=1
;
for ic=0,num_channel-1 do begin
for ib=0,num_baseline-1 do begin
	channel=ds_ch(ic)
	baseline=ds_bl(ib)
	if cal_options.v eq 1 then begin
		scans(*).TripleAmpC(ds_tr,channel)= $
			scans(*).TripleAmp(ds_tr,channel)
		scans(*).TripleAmpCErr(ds_tr,channel)= $
			scans(*).TripleAmpErr(ds_tr,channel)
		scans(*).TripleAmpEC(ds_tr,channel)= $
			scans(*).TripleAmpE(ds_tr,channel)
		scans(*).TripleAmpECErr(ds_tr,channel)= $
			scans(*).TripleAmpEErr(ds_tr,channel)
		entry_index=where((cal_entries.variable eq cal_options.v) and $
		    	(cal_entries.triple eq ds_tr) and $
		    	(cal_entries.channel eq channel),entry_count)
	endif else if cal_options.v eq 2 then begin
		scans(*).TriplePhaseC(ds_tr,channel)= $
			scans(*).TriplePhase(ds_tr,channel)
		scans(*).TriplePhaseCErr(ds_tr,channel)= $
			scans(*).TriplePhaseErr(ds_tr,channel)
		entry_index=where((cal_entries.variable eq cal_options.v) and $
		    	(cal_entries.triple eq ds_tr) and $
		    	(cal_entries.channel eq channel),entry_count)
	endif else begin
		scans(*).VisSqC(ds_ob,channel,baseline)= $
			scans(*).VisSq(ds_ob,channel,baseline)
		scans(*).VisSqCErr(ds_ob,channel,baseline)= $
			scans(*).VisSqErr(ds_ob,channel,baseline)
		scans(*).VisSqEC(ds_ob,channel,baseline)= $
			scans(*).VisSqE(ds_ob,channel,baseline)
		scans(*).VisSqECErr(ds_ob,channel,baseline)= $
			scans(*).VisSqEErr(ds_ob,channel,baseline)
		entry_index=where((cal_entries.variable eq cal_options.v) and $
		    	(cal_entries.outbeam eq ds_ob) and $
		    	(cal_entries.baseline eq baseline) and $
		    	(cal_entries.channel eq channel),entry_count)
	endelse
	index=intarr(n_elements(cal_entries))
	if entry_count gt 0 then index(entry_index)=-1
	entry_index=where(index ne -1,entry_count)
	if entry_count gt 0 then set_calentries,cal_entries(entry_index)
endfor
endfor
print,'Calibration cleared.'
;
end
;-------------------------------------------------------------------------------
pro resetcal,variable
;
; Completely removes calibration from data and re-creates cal_entries for 
; selected data type.
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(cal_options) eq 0 then initcal
if n_elements(variable) eq 0 then variable=cal_options.v
	
if variable eq 1 then begin
	scans.TripleAmpC=scans.TripleAmp
	scans.TripleAmpCErr=scans.TripleAmpErr
	scans.TripleAmpEC=scans.TripleAmpE
	scans.TripleAmpECErr=scans.TripleAmpEErr
endif else if variable eq 2 then begin
	scans.TriplePhaseC=scans.TriplePhase
	scans.TriplePhaseCErr=scans.TriplePhaseErr
endif else begin
	scans.VisSqC=scans.VisSq
	scans.VisSqCErr=scans.VisSqErr
	scans.VisSqEC=scans.VisSqE
	scans.VisSqECErr=scans.VisSqEErr
endelse
;
index=where(cal_entries.variable ne variable,count)
if count gt 0 then set_calentries,cal_entries(index) else create_calentries
;
print,'Calibration reset for data type ',cal_options.n(variable),'.', $
	format='(a,a,a)'
;
end
;-------------------------------------------------------------------------------
pro editcal,reason,command
;
; Remove calibration entries specified through reason parameter.
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
;
if n_params() ne 2 then begin
	print,'***Error(CALEDIT): you have to specify reason and command!'
	return
endif
if n_elements(reason) eq 0 then begin
	print,'***Error(CALEDIT): no reason specified!'
	return
endif
if n_elements(command) eq 0 then begin
	print,'***Error(CALEDIT): no command specified!'
	return
endif
;
if strupcase(command) eq 'UNDO' then calibrate,reason,1
;
index=where(cal_entries.reason ne reason(0),count)
if count gt 0 then begin
	set_calentries,cal_entries(index)
	print,'Removed entries; reason=',strcompress(reason)
endif else print,'No entries found for reason=',reason,'.',format='(a,a,a)'
;
end
;-------------------------------------------------------------------------------
pro initcal
;
; Create array structure cal_entries and init. arrays indicators & functions
;
common BaseFunctions,numlist,functions
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
;
if n_elements(cal_entries) eq 0 then create_calentries
;
cal_stars=''
cal_options=alloc_cal_options()
sel_indct=''
sel_funcn=strarr(1,n_elements(functions))

;
end
;-------------------------------------------------------------------------------
function calchisq,src,cal,beams,chans,ntotl
;
; For a given star and its calibrator, compute the reduced chisq of the 
; deviation of V2 of the former with respect to the latter.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
index1=where(scans.starid eq src)
index2=where(scans.starid eq cal)
;
time1=scans(index1).time
time2=scans(index2).time
;
chisq=0
ntotl=0
;
for i=0,n_elements(beams)-1 do begin
for j=0,n_elements(chans(i,*))-1 do begin
	if chans(i,j) ge 0 then begin
;
	vis1=scans(index1).vissqe(beams(i),chans(i,j),0)
	viserr1=scans(index1).vissqeerr(beams(i),chans(i,j),0)
	vis2=scans(index2).vissqe(beams(i),chans(i,j),0)
	viserr2=scans(index2).vissqeerr(beams(i),chans(i,j),0)
;
	index1e=where(viserr1 gt 0,count1)
	index2e=where(viserr2 gt 0,count2)
	if count1*count2 ne 0 then begin
		vis1e=vis1(index1e)
		viserr1e=viserr1(index1e)
		time1e=time1(index1e)
		vis2e=vis2(index2e)
		viserr2e=viserr2(index2e)
		time2e=time2(index2e)
		chisq1=((vis1e-spline(time2e,vis2e,time1e,1))/ $
			sqrt(viserr1e^2+spline(time2e,viserr2e,time1e,1)^2))^2
		n1=n_elements(chisq1)
		index=where(time1e lt time2e(0),count)
		if count gt 0 then chisq1=chisq1(count:n1-1)
		n1=n_elements(chisq1)
		n1e=n_elements(time1e)
		n2e=n_elements(time2e)
		index=where(time1e gt time2e(n2e-1),count)
		if count gt 0 then chisq1=chisq1(0:n1-1-count)
		n1=n_elements(chisq1)
		chisq2=((vis2e-spline(time1e,vis1e,time2e,1))/ $
			sqrt(viserr2e^2+spline(time1e,viserr1e,time2e,1)^2))^2
		n2=n_elements(chisq2)
		index=where(time2e lt time1e(0),count)
		if count gt 0 then chisq2=chisq2(count:n2-1)
		n2=n_elements(chisq2)
		n1e=n_elements(time1e)
		n2e=n_elements(time2e)
		index=where(time2e gt time1e(n1e-1),count)
		if count gt 0 then chisq2=chisq2(0:n2-1-count)
		n2=n_elements(chisq2)
		chisq=chisq+total(chisq1)
		ntotl=ntotl+n1
	endif
;
	endif
endfor
endfor
;
return,chisq/ntotl
;
end
;-------------------------------------------------------------------------------
function caldiameter,src,cal,beams=beams,channels=channels
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
!qiet=1
;
; Obtain diameter estimate based on visibility amplitude first
index_src=where(scans.starid eq src and scans.time gt 0,count_src)
index_cal=where(scans.starid eq cal and scans.time gt 0,count_cal)
if count_src*count_cal eq 0 then begin
	print,'***Error(CALDIAMETER): no data on either star or calibrator!'
	return,0
endif
if n_elements(beams) eq 0 then beems=indgen(GenConfig.NumOutBeam) $
			  else beems=beams-1
chans=intarr(n_elements(beems),max(GenConfig.NumSpecChan))-1
;
for i=0,n_elements(beems)-1 do begin
;
if beems(i) lt 0 or beems(i) ge GenConfig.NumOutBeam then begin
	print,'***Error(CALDIAMETER): invalid beam selection!'
	return,0
endif
if n_elements(channels) eq 0 then chans(i,0:9)=indgen(10) else begin
	s=size(channels)
	if s(0) le 1 then chans(i,0:total(channels gt 0)-1) $
			 =channels(where(channels gt 0))-1 $
		     else chans(i,0:total(channels(beems(i),*) gt 0)-1) $
			 =channels(beems(i),where(channels(beems(i),*) gt 0))-1
endelse
if max(chans) gt max(GenConfig.NumSpecChan)-1 then begin
	print,'***Error(CALDIAMETER): invalid channel selection!'
	return,0
endif
for j=0,n_elements(chans(i,where(chans(i,*) ge 0)))-1 do begin
;
	v_src=scans(index_src).vissq(beems(i),chans(i,j),0)
	ve_src=scans(index_src).vissqerr(beems(i),chans(i,j),0)
	uvr_src=sqrt(scans(index_src).uvw(beems(i),chans(i,j),0,0)^2 $
	            +scans(index_src).uvw(beems(i),chans(i,j),0,1)^2)/1e6
	v_cal=scans(index_cal).vissqe(beems(i),chans(i,j),0)
	ve_cal=scans(index_cal).vissqeerr(beems(i),chans(i,j),0)
;
	jndex_src=where(ve_src gt 0,count_src)
	jndex_cal=where(ve_cal gt 0,count_cal)
	if count_src*count_cal gt 0 then begin
		d=hb(avg(uvr_src(jndex_src)), $
                     avg(v_src(jndex_src))/avg(v_cal(jndex_cal)))
		if n_elements(d_avg) eq 0 then d_avg=d else d_avg=[d_avg,d]
	endif
;
endfor
endfor
;
if n_elements(d_avg) eq 0 then begin
	print,'***Error(CALDIAMETER): no valid data!'
	return,-1
endif
diameter=total(d_avg)/n_elements(d_avg)
if not !qiet then print,'Initial estimate: ',diameter
;
; Now do a least-squares adjustment of the diameter
;
i=where(startable.starid eq src)
;
dm=dblarr(3,3)
chifr=0.0005
dd=0.1
;
repeat begin
	startable(i).diameter=diameter
	dm(0,0)=1&dm(0,1)=startable(i).diameter&dm(0,2)=startable(i).diameter^2
	calcviscal
	chisqo=calchisq(src,cal,beems,chans,ntotl)
	startable(i).diameter=diameter-dd
	dm(1,0)=1&dm(1,1)=startable(i).diameter&dm(1,2)=startable(i).diameter^2
	calcviscal
	chisql=calchisq(src,cal,beems,chans,ntotl)
	startable(i).diameter=diameter+dd
	dm(2,0)=1&dm(2,1)=startable(i).diameter&dm(2,2)=startable(i).diameter^2
	calcviscal
	chisqh=calchisq(src,cal,beems,chans,ntotl)
	coeff=invert(dm)#[chisqo,chisql,chisqh]
	diameter=-coeff(1)/(2*coeff(2))
	startable(i).diameter=diameter
	calcviscal
	chisq=calchisq(src,cal,beems,chans,ntotl)
	dd=dd/2
	if not !qiet then print,diameter,chisqo,chisq,ntotl
endrep until 1-chisq/chisqo lt chifr
;
error=sqrt((chisq/ntotl)/coeff(2))
;
!qiet=0
return,diameter
;
end
;-------------------------------------------------------------------------------
function calvis,vis,uvw,stars
;
; Estimate squared visibilities for calibrator stars.
; Note the units: 
; u,v,w	[lambda]
; LD 	[mas]
;
; LD is the limb darkened disk diameter taken from startable.
; Coefficients A0, A1, and A2 available from startable are
; used to compute the limb darkening coeeficients.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
vis_cal=vis*0+1
s=size(vis_cal)
;
num_scans=n_elements(stars)
LD=fltarr(num_scans)
A0=fltarr(num_scans)
A1=fltarr(num_scans)
A2=fltarr(num_scans)
for i=0,n_elements(StarTable)-1 do begin
	index=where(stars eq StarTable(i).starid,count)
	if count gt 0 then begin
		A0(index)=StarTable(i).a0
		A1(index)=StarTable(i).a1
		A2(index)=StarTable(i).a2
		LD(index)=StarTable(i).diameter
	endif
endfor
if min(LD) eq 0 then $
	print,'Warning(VISEST): some diameters = 0 or not found in table!'
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
for ob=0,GenConfig.NumOutBeam-1 do begin
for ch=0,GenConfig.NumSpecChan(ob)-1 do begin
	lambda=GenConfig.Wavelength(ch,ob)*1.d9 	; lambda in nm
	u_lambda=fltarr(num_scans)
	index=where(abs(A1) ge 0.01,count)
	if count gt 0 then begin
;		Logarithmic polynomial fit coefficients
		x=alog10(lambda)
		u_lambda(index)=10^(A0+A1*x+A2*x^2)
	endif
	index=where(abs(A1) lt 0.01,count)
	if count gt 0 then begin
;		Linear polynomic fit coefficients
		x=lambda
		u_lambda(index)=A0+A1*x+A2*x^2
	endif
        alpha=1-u_lambda
        beeta=u_lambda
;	LD=sqrt(((1-u_lambda/3)/(1-7*u_lambda/15)))*UD
	LD_zero=where(LD eq 0,LD_zero_count)
	if LD_zero_count gt 0 then LD(LD_zero)=1
	for bl=0,GenConfig.NumBaseline(ob)-1 do begin
		arg=factor*sqrt(uvw(ob,ch,bl,0,*)^2+uvw(ob,ch,bl,1,*)^2)*LD
		UV_zero=where(arg eq 0,UV_zero_count)
		if UV_zero_count gt 0 then arg(UV_zero)=1
		cal=((alpha*beselj(arg,1)/arg+beeta*sqrt(pi_circle/2)* $
                      sqrt(2/(pi_circle*arg))*(sin(arg)/arg-cos(arg))/ $
                      sqrt(arg*arg*arg))/(alpha/2+beeta/3))^2
		if LD_zero_count gt 0 then cal(LD_zero)=1
		if LD_zero_count gt 0 then LD(LD_zero)=0
		if UV_zero_count gt 0 then cal(UV_zero)=1
		if s(0) eq 1 then vis_cal(*)=cal else vis_cal(ob,ch,bl,*)=cal
	endfor
endfor
endfor
;
return,vis_cal
;
end
;-------------------------------------------------------------------------------
pro calcviscal,model=model,multiple=multiple
;
; Computes estimated visibilitiy for calibrators for all averaged scans.
;
; Does all nights if more than one is loaded and multiple=1.
;
common Tables,scantable,bgtable,stationtable
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if not keyword_set(multiple) then multiple=0
;
if multiple then nights=geninfo.date else nights=genconfig.date
;
FOR night=0,n_elements(nights)-1 DO BEGIN
;
if n_elements(nights) gt 1 then $
loadnight,GenInfo(night).date, $
	  GeoInfo(night).systemid, $
	  GenInfo(night).configid
;
if checkdata([3,9]) ne 0 then return
;
; Can use the model values instead of estimation based on diameter
if n_elements(model) eq 0 then model=0
;
index=where(abs(scans.time) gt 0,count)
if count eq 0 then begin
	print,'***Error(CALCVIS): no valid scan times found!'
	return
endif
;
stars=scantable(index).StarId
;
; Compute calibration factors
if not model then cal=calvis(scans(index).VisSq,scans(index).UVW,stars) $
	     else cal=       scans(index).VisSqM
j=where(cal eq 0,count)
if count gt 0 then cal(j)=1
;
; Calibrate estimated visibilities
;
scans(index).VisSqE=scans(index).VisSq/cal
scans(index).VisSqEErr=scans(index).VisSqErr/cal
scans(index).VisSqEC=scans(index).VisSqC/cal
scans(index).VisSqECErr=scans(index).VisSqCErr/cal
;
; Do the same for the triple amplitudes
scans(index).TripleAmpE=scans(index).TripleAmp
scans(index).TripleAmpEErr=scans(index).TripleAmpErr
scans(index).TripleAmpEC=scans(index).TripleAmpC
scans(index).TripleAmpECErr=scans(index).TripleAmpCErr
for i=0,GenConfig.NumTriple-1 do begin
for j=0,2 do begin
        ob=GenConfig.TripleBeam(j,i)
        bl=GenConfig.TripleBase(j,i)
        ch=GenConfig.TripleChan(0:GenConfig.TripleNumChan(i)-1,j,i)
	scans(index).TripleAmpE(i,ch)=scans(index).TripleAmpE(i,ch) $
		/sqrt(cal(ob,ch,bl,*))
	scans(index).TripleAmpEErr(i,ch)=scans(index).TripleAmpEErr(i,ch) $
		/sqrt(cal(ob,ch,bl,*))
	scans(index).TripleAmpEC(i,ch)=scans(index).TripleAmpEC(i,ch) $
		/sqrt(cal(ob,ch,bl,*))
	scans(index).TripleAmpECErr(i,ch)=scans(index).TripleAmpECErr(i,ch) $
		/sqrt(cal(ob,ch,bl,*))
endfor
endfor
;
if not !qiet then print,'Finished visibility estimation.'
;
if n_elements(bufferinfo) gt 1 then $
storenight,11L
;
ENDFOR
;
end
;************************************************************************Block 4
pro list_stars,stars,names
;
; Return list of stars which occur in data currently loaded.
; This list is therefore not derived from the startable!
;
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
stars=''
if n_elements(scantable) eq 0 then begin
	if n_elements(scans) eq 0 then begin
		print,'***Error(LIST_STARS): neither ScanTable nor scans exist!'
		return
	endif
	starids=[scans.starid]
endif else begin
	starids=ScanTable.StarId
endelse
stars=starids(uniq(starids,sort(starids)))
if n_elements(scans) gt 0 then begin
	names=stars
	for i=0,n_elements(names)-1 do begin
		j=where(scans.starid eq stars(i),count) & j=j(0)
		if count gt 0 then $
		names(i)=scans(j).star
	endfor
endif
;
if n_params() eq 0 then begin
print,'__________________________________'
for i=0,n_elements(stars)-1 do begin
	index=where(starids eq stars(i) and scantable.code eq 1)
	ids=string(index+1,format='(i0)')+'(' $
	   +string(scantable(index).scanid,format='(i0)')+')'
	print,stars(i)+': Scans '+strjoin(ids,',')
endfor
print,'_______________***________________'
endif
;
end
;-------------------------------------------------------------------------------
pro list_bgscans
;
common StarBase,startable,notes
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([2,3]) eq -1 then return
;
list_stars,stars
;
print,'Star    | BG Scans           '
;
max_per_line=11
;
for i=0,n_elements(stars)-1 do begin
	print,'--------|------------------------------------------------------------------'
	j=where(startable.starid eq stars(i))
	index=where(ScanTable.starid eq stars(i),count)
	print,startable(j).starid,' | ',format='(a,a,$)'
	for k=0,count-1 do begin
		if k mod max_per_line eq 0 and k ne 0 then begin
			print,''
			print,'        |------------------------------------------------------------------'
			print,'        | ',format='(a,$)'
		endif
		jndex=where(BGTable.scanid eq ScanTable(index(k)).scanid,n)
		for l=0,n-1 do print,BGTable(jndex(l)).iscan,' ',format='(i3,a,$)'
		if n eq 0 then space='    ' else space=''
		print,space,'| ',format='(a,a,$)'
	endfor
	print,''
endfor
print,'--------|------------------------------------------------------------------'
;
end
;-------------------------------------------------------------------------------
pro list_scans,jndex,coherent=coherent
;
; List the observed scans. Using the index argument, the list can be
; restricted to certain indices in the scantable, or all entries corresponding
; to a specific star (e.g. index='BSC1931'). In any case, the index argument
; will contain the final list on output, *overwriting* the input variable!
;
common Tables,ScanTable,BGTable,StationTable
;
if checkdata([1]) ne 0 then return
;
help,jndex,level=-1,/full,output=o
if strmid(o(0),0,9) eq '<No name>' then do_print=1 else do_print=0
;
r=size(jndex)
if r(n_elements(r)-2) eq 7 then begin
	jndex=where(scantable.starid eq jndex,count)+1
	if count eq 0 then return
endif
if n_elements(jndex) eq 0 then jndex=indgen(n_elements(scantable))+1
if keyword_set(coherent) then begin
	cindex=where(scantable(jndex-1).code eq 1,count)
	if count eq 0 then return
	jndex=jndex(cindex)
endif
index=jndex-1
printout=strarr((n_elements(index)+1)/2,2)
printout(0:n_elements(index)-1)= $
'('+string(index+1,format='(i3)')+') '+scantable(index).starid $
     +' ('+hms((scantable(index).starttime $
               +scantable(index).stoptime)/(2*3600),/scan) $
     +', '+scanconfig(index+1) $
     +', '+string(scantable(index).code,format='(i1)')+') '
if do_print then $
print,transpose(printout),format='(2a)'
;
end
;-------------------------------------------------------------------------------
pro list_stations,stns
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
stns=''
if checkdata([8]) ne 0 then return
if n_params() eq 0 then print,GenConfig.StationId(0:GenConfig.NumSid-1) $
	else stns=GenConfig.StationId(0:GenConfig.NumSid-1)
;
end
;-------------------------------------------------------------------------------
pro list_flagreasons,class,reasons
;
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
reasons=''
if n_elements(class) eq 0 then begin
	print,'***Error(LIST_FLAGREASONS): no class defined!'
	return
endif
case class of
	'point':begin
		if n_elements(pointflagtable) eq 0 then return
		table=pointflagtable
		end
	'metro':begin
		if n_elements(inchflagtable) eq 0 then return
		table=inchflagtable
		end
	'bg'   :begin
		if n_elements(bgflagtable) eq 0 then return
		table=bgflagtable
		end
	'scan' :begin
		if n_elements(scanflagtable) eq 0 then return
		table=scanflagtable
		end
	   else:begin
		print,'***Error(LIST_FLAGREASONS): invalid class!'
		return
		end
endcase
index=where(table.time ge 0,count)
if count eq 0 then return
reasons=table(index).reason
reasons=reasons(uniq(reasons,sort(reasons)))
;
if n_params() eq 1 then print,reasons
;
end
;-------------------------------------------------------------------------------
pro list_calreasons,reasons
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
;
reasons=''
if n_elements(cal_entries) eq 0 then begin
	print,'***Error(LIST_CALREASONS): cal_entries undefined!'
	return
endif
if n_elements(cal_options) eq 0 then begin
	print,'***Error(LIST_CALREASONS): cal_options undefined!'
	return
endif
index=where(cal_entries.num_coeffs gt 0 $
	and cal_entries.variable eq cal_options.v,count)
if count eq 0 then return
reasons=cal_entries(index).reason
reasons=reasons(uniq(reasons,sort(reasons)))
;
if n_params() eq 0 then print,reasons
;
end
;-------------------------------------------------------------------------------
pro list_calentries
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
;
if n_elements(cal_entries) eq 0 then begin
	print,'***Error(LIST_CALENTRIES): no cal_entries defined!'
	return
endif
index=where(cal_entries.num_coeffs gt 0,count)
if count eq 0 then begin
	print,'***Error(LIST_CALENTRIES): cal_entries empty!'
	return
endif
entries=cal_entries(index)
print,'Number of valid entries in cal_entries:',n_elements(entries)
for i=0,n_elements(entries)-1 do begin
	print,entries(i).reason, $
		', OB=',entries(i).outbeam+1, $
		', BL=',entries(i).baseline+1, $
		', CH=',entries(i).channel+1, $
		', Stars=',entries(i).stars, $
		format='(a,a,i2,a,i2,a,i3,100a)'
endfor
print,'Listing complete.'
;
end
;-------------------------------------------------------------------------------
pro list_summary_chameleon
;
; Print summary of observations and system configuration to screen.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
print,'The following information was obtained:'
;
if checkdata([5]) eq 0 then begin
	print,'Date of observation: ',Date
endif
if checkdata([6]) eq 0 then begin
	print,'System ID: ',SystemId
endif
if checkdata([7]) eq 0 then begin
	print,'Location : Longitude=',GeoParms.Longitude,'E, Latitude=', $
		GeoParms.Latitude,', Altitude=',GeoParms.Altitude,'m', $
		format='(a,f6.1,a,f5.1,a,f5.0,a)'
endif
if checkdata([8]) eq 0 then begin
	print,'Stations :',GenConfig.StationId(0:GenConfig.NumSid-1)
	print,'Reference station :',GenConfig.StationId(GenConfig.RefStation-1)
	for i=0,GenConfig.NumOutBeam-1 do begin
		print,'--------------------------------------------------------'
		print,'OutputBeam No.',i+1, $
		      ', SpectrometerId: ',GenConfig.SpectrometerId(i)
		print,'Baselines: ', $
			GenConfig.BaselineId(0:GenConfig.NumBaseline(i)-1,i), $
			format='(a,15(a7,1x))'
		print,'Channels:    '
		print,indgen(GenConfig.NumSpecChan(i))+1, $
			format='(10(i5,1x))'
		print,'Lambda [nm]: '
		print,fix(1.d9* $
			GenConfig.Wavelength(0:GenConfig.NumSpecChan(i)-1,i)), $
			format='(10(i5,1x))'
		print,'Widths [nm]  '
		print,fix(1.d9* $
			GenConfig.ChanWidth(0:GenConfig.NumSpecChan(i)-1,i)), $
			format='(10(i5,1x))'
	endfor
	print,'--------------------------------------------------------'
	if GenConfig.NumTriple gt 0 then begin
		print,'Triple combinations:'
		tripno=strarr(3,GenConfig.NumTriple)
		tripno(0,*)='(' $
			+string(indgen(GenConfig.NumTriple)+1,format='(i2)')+') '
		print,tripno+genconfig.baselineid( $
			genconfig.triplebase(*,0:GenConfig.NumTriple-1), $
			genconfig.triplebeam(*,0:GenConfig.NumTriple-1))+'(' $
		+string(genconfig.triplebeam(*,0:GenConfig.NumTriple-1)+1, $
							  format='(i1)')+')'
		print,'--------------------------------------------------------'
	endif
endif
if checkdata([1]) eq 0 then begin
	print,'Number of scans observed: ',n_elements(ScanTable)
	print,'Observations lasted from UT ', $
		hms(ScanTable(0).starttime/3600),'  to ', $
		hms(ScanTable(n_elements(ScanTable)-1).stoptime/3600), $
		format='(a,a,a,a)'
endif
;
end
;-------------------------------------------------------------------------------
pro list_logs
;
list_scanlog
list_observerlog
list_constrictorlog
list_inchwormlog
list_oysterlog
;
end
;-------------------------------------------------------------------------------
pro list_scanlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if n_elements(ScanLog) ne 0 then begin
        print,'________________________________________________________________'
        print,'ScanLog:'
	n=strlen(ScanLog)
	maxchar=255
	for i=0,n/maxchar-1 do $
		print,strmid(ScanLog,i*maxchar,maxchar),format='(a,$)'
	print,strmid(ScanLog,i*maxchar,n mod maxchar)
        print,'_____________________________***________________________________'
endif
;
end
;-------------------------------------------------------------------------------
pro list_observerlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if n_elements(ObserverLog) ne 0 then begin
        print,'________________________________________________________________'
        print,'ObserverLog:'
	n=strlen(ObserverLog)
	maxchar=255
	for i=0,n/maxchar-1 do $
		print,strmid(ObserverLog,i*maxchar,maxchar),format='(a,$)'
	print,strmid(ObserverLog,i*maxchar,n mod maxchar)
        print,'_____________________________***________________________________'
endif
;
end
;-------------------------------------------------------------------------------
pro list_constrictorlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if n_elements(ConstrictorLog) ne 0 then begin
        print,'________________________________________________________________'
        print,'ConstrictorLog:'
	n=strlen(ConstrictorLog)
	maxchar=255
	for i=0,n/maxchar-1 do $
		print,strmid(ConstrictorLog,i*maxchar,maxchar),format='(a,$)'
	print,strmid(ConstrictorLog,i*maxchar,n mod maxchar)
        print,'_____________________________***________________________________'
endif
;
end
;-------------------------------------------------------------------------------
pro list_oysterlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if n_elements(OysterLog) ne 0 then begin
        print,'OysterLog:'
        print,OysterLog
        print,'__________***____________'
endif
;
end
;-------------------------------------------------------------------------------
pro list_inchwormlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
line=0
lines=getenv('LINES')
if lines le 0 then lines=22
;
if n_elements(InchwormLog) ne 0 then begin
        print,'________________________________________________________________'
	line=line+1
        print,'InchwormLog:'
	line=line+1
        n=strlen(InchwormLog)
        maxchar=255
	maxchar=80
        for i=0,n/maxchar-1 do begin
                print,strmid(InchwormLog,i*maxchar,maxchar),format='(a,$)'
		line=line+1
		if line mod lines eq 0 then begin
			case myhak('') of
			'q':	begin
				print,''
				print,'Quit list.'
				return
				end
			' ':    lines=22
			'\x0A': lines=1
			else:
			endcase
		endif
	endfor
        print,strmid(InchwormLog,i*maxchar,n mod maxchar)
        print,'_____________________________***________________________________'
endif
;
end
;-------------------------------------------------------------------------------
pro list_triples
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if GenConfig.NumTriple eq 0 then begin
	print,'No triples found.'
	return
endif
;
print,' #   OutputBeam    Baseline  #SpecChan'
for i=0,GenConfig.NumTriple-1 do $
	print,i,GenConfig.TripleBeam(*,i)+1, $
		GenConfig.TripleBase(*,i)+1, $
		GenConfig.TripleNumChan(i),format='(i2,1x,3(3x,3(1x,i2)))'
print,'Listing complete.'
;
end
;-------------------------------------------------------------------------------
pro print_scanforms
;
; Format the scan and bg scan listings and submit to printer.
;
common StarBase,startable,notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([1,2,3]) ne 0 then return
;
list_stars,stars
;
filename1='oyster1.txt'
openw,unit,filename1,/get_lun
;
max_per_line=11
;
printf,unit,'Star    | Diam.|  Mag.|Cal| Scans           '
for i=0,n_elements(stars)-1 do begin
	printf,unit,'--------|------|------|---|---------------------------------------------'
	j=where(startable.starid eq stars(i))
	index=where(ScanTable.starid eq stars(i) and ScanTable.code eq 1,count)
	if count gt max_per_line then n=max_per_line else n=count
	printf,unit,startable(j).starid,' | ',startable(j).diameter,' | ', $
		startable(j).mv,' | ',startable(j).bflag,' | ', $
		ScanTable(index(0:n-1)).iscan, $
		format='(a,a,f4.1,a,f4.1,a,a,a,20(i3,1x))'
	count=count-n
	line=0
	while count gt 0 do begin
	  line=line+1
	  if count gt max_per_line then n=max_per_line else n=count
	  printf,unit,'                          | ', $
	    ScanTable(index(line*max_per_line:line*max_per_line+n-1)).iscan, $
	    format='(a,20(i3,1x))'
	  count=count-n
	endwhile
endfor
printf,unit,'--------|------|------|---|---------------------------------------------'
for i=1,3 do printf,unit,''
printf,unit,'Notes:'
;
free_lun,unit
;
filename2='oyster2.txt'
openw,unit,filename2,/get_lun
;
max_per_line=9
;
printf,unit,'Star    | BG Scans           '
for i=0,n_elements(stars)-1 do begin
	printf,unit,'--------|---------------------------------------------------------------'
	j=where(startable.starid eq stars(i))
	index=where(ScanTable.starid eq stars(i),count)
	printf,unit,startable(j).starid,' | ',format='(a,a,$)'
	for k=0,count-1 do begin
		if k mod max_per_line eq 0 and k ne 0 then begin
			printf,unit,''
			printf,unit,'        | ',format='(a,$)'
		endif
		jndex=where(BGTable.scanid eq ScanTable(index(k)).scanid,n)
		for l=0,n-1 do printf,unit,BGTable(jndex(l)).iscan,' ',format='(i3,a,$)'
		if n eq 0 then space='    ' else space=''
		printf,unit,space,'| ',format='(a,a,$)'
	endfor
	printf,unit,''
endfor
printf,unit,'--------|---------------------------------------------------------------'
;
free_lun,unit
;
psfile='oyster.ps'
;
if strpos(strupcase(getenv('HOST')),'OCTANS') ne -1 then begin
	spawn,'nenscript -p'+psfile+' -fCourier9 -b'+Date+' '+filename1+' '+filename2
	spawn,'lpr '+psfile
endif else if strpos(strupcase(getenv('HOST')),'FERMION') ne -1 then begin
	spawn,'mpage -o -1 -I5 -X'+Date+' '+filename1+' '+filename2+' > '+psfile
	spawn,'lpr '+psfile
endif else if strpos(strupcase(getenv('HOST')),'GEMINI') ne -1 then begin
	spawn,'mpage -o -1 -I5 -X'+Date+' '+filename1+' '+filename2+' > '+psfile
	spawn,'lpr '+psfile
endif else begin
	spawn,'lpr '+filename1
	spawn,'lpr '+filename2
endelse
spawn,'rm -f '+psfile
spawn,'rm -f '+filename1+' '+filename2
print,'Forms submitted to printer.'
;
end
;-------------------------------------------------------------------------------
pro print_specforms
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([5,8]) ne 0 then return
;
filename='form.txt'
psfile='form.ps'
;
FOR i=1,GenConfig.NumOutBeam DO BEGIN
openw,unit,filename,/get_lun
printf,unit, $
'Ch   BG    Flg  SM   Sl   RMS   Spectrometer '+string(i,format='(i1)')
for j=1,GenConfig.NumSpecChan(i-1) do begin
printf,unit, $
'---||----|----|----|----|----|----------------------------------------||'
printf,unit,string(j,format='(i2.2)')+ $
  ' ||    |    |    |    |    |                                        ||'
endfor
printf,unit, $
'---||----|----|----|----|----|----------------------------------------||'
free_lun,unit
;
if strpos(strupcase(getenv('HOST')),'FORNAX') ne -1 then begin
	spawn,'mp -pagelength 70 -linelength 120 -subject '+Date+' ' $
		+filename+' > '+psfile
	spawn,'lp -onb -drpnpoipost '+psfile
endif else if strpos(strupcase(getenv('HOST')),'OCTANS') ne -1 then begin
	spawn,'nenscript -p'+psfile+' -fCourier9 -b'+Date+' '+filename
	spawn,'lpr '+psfile
endif else if strpos(strupcase(getenv('HOST')),'FERMION') ne -1 then begin
	spawn,'mpage -o -1 -I5 -X'+Date+' '+filename+' > '+psfile
	spawn,'lpr '+psfile
endif else if strpos(strupcase(getenv('HOST')),'GEMINI') ne -1 then begin
	spawn,'mpage -o -1 -I5 -X'+Date+' '+filename+' > '+psfile
	spawn,'lpr '+psfile
endif else spawn,'lpr '+filename
spawn,'rm -f '+psfile
spawn,'rm -f '+filename
ENDFOR
print,'Forms submitted to printer.'
;
end
;************************************************************************Block 5
pro set_visampphase
;
; Compute VisAmp and VisPhase, as well as their respective errors, from the
; complex quantities ComplexVis and ComplexVisErr for plotting
; and editing. Phases are counter-clockwise from 1+0i in radians. 
;
; The following is a citation from D F Buschers Thesis (p. 18):
; "A complex quantity [like ComplexVis] may have different variances along
; different directions in the complex plane. This can be determined for any 
; arbitrary direction from just two parameters, var_1(C) and var_2(C), which
; are defined for a complex quantity C as:
; var_1(C) = E(CC*) - E(C)E(C)*, var_2(C) = E(CC) - E(C)E(C).
; With these definitions, the variance of C along a direction at an angle t
; to the real axis is: var(C,t)=1/2(var_1(C)+Re{var_2(C)e^(-2it)}).
; Thus var_2(C) represents the part of the noise which varies with direction
; in the complex plane and var_1(C) represents the circularly symmetric term."
;
; In this program, it is assumed that ComplexVis(*,0,*,*,*) is the real part
; of ComplexVis, and ComplexVis(*,1,*,*,*) is the imaginary part. However, we
; assume that there is no correlation between the real and imaginary parts, so
; that the error of the complex visibility is also complex, with the real part
; corresponding to the error of the real part of the visibility.
;
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
; 	Compute VisAmp
	CVR=ComplexVis(*,0,*,*,*)
	CVI=ComplexVis(*,1,*,*,*)
	VisAmp(*,*,*,*)=sqrt(CVR^2+CVI^2)
;
;	Compute VisPhase, replacing undefined values
	index=where(CVR eq 0 and CVI eq 0,count)
	if count gt 0 then begin
		CVR(index)=1
		CVI(index)=1
		VisAmp(index)=2
	endif
	VisPhase(*,*,*,*)=atan(CVI,CVR)
;
;	Compute VisAmpErr. If VisAmp is zero, above replacements
;	will give a reasonable estimate.
	VisAmpErr(*,*,*,*)= $
		sqrt((CVR*ComplexVisErr(*,0,*,*,*))^2 $
		    +(CVI*ComplexVisErr(*,1,*,*,*))^2) $
		/VisAmp
;
;	Reset VisAmp replacements and flag phase of zero amplitudes
	if count gt 0 then begin
		VisAmp(index)=0
		VisPhaseErr(index)=-1
	endif
;
;	Compute VisPhaseErr for valid data points
	index=where(VisAmp gt 0,count)
	if count gt 0 then VisPhaseErr(index)=VisAmpErr(index)/VisAmp(index)
;
;	Free memory
	CVR=0
	CVI=0
;
;	Make sure all derived data is flagged where the input data is
	CVRE=ComplexVisErr(*,0,*,*,*)
	CVIE=ComplexVisErr(*,1,*,*,*)
	index=where(CVRE le 0 or CVIE le 0,count)
	if count gt 0 then begin
		VisAmpErr(index)=-1
		VisPhaseErr(index)=-1
	endif
;
end
;-------------------------------------------------------------------------------
pro set_tripleampphase
;
; Compute TripleAmp and TriplePhase, as well as their respective errors, from 
; the complex quantities ComplTriple and ComplTripleErr for plotting
; and editing. Phases are counter-clockwise from 1+0i in radians. 
; See comments in set_visampphase for further information.
;
; Note that for simplicity, we flag all data where the triple amplitude
; is exactly zero.
;
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
;	The following code should accomplish the same what procedure
;	set_visampphase does, only in a different way.
;
	TriplePhase(*)=0
	CTR=double(ComplTriple(*,0,*,*))
	CTI=double(ComplTriple(*,1,*,*))
	CTRE=double(ComplTripleErr(*,0,*,*))
	CTIE=double(ComplTripleErr(*,1,*,*))
	CTA=sqrt(CTR^2+CTI^2)
	TripleAmp(*,*,*)=CTA
	TripleAmpErr(*,*,*)=CTRE
	TriplePhase(*,*,*)=atan(CTI,CTR)
	TriplePhaseErr(*,*,*)=atan(CTIE,CTA)
	index=where(CTA gt 0,count)
	if count gt 0 then begin
;	TripleAmpErr(index)= $
;		sqrt(((CTR(index)/CTA(index))*CTRE(index))^2 $
;		    +((CTI(index)/CTA(index))*CTIE(index))^2)
;	TriplePhase(index)=atan(CTI(index),CTR(index))
;	TriplePhaseErr(index)=TripleAmpErr(index)/TripleAmp(index)
	endif else print,'Warning(SET_TRIPLEAMPPHASE): no valid data!'
	index=where(CTRE le 0 or CTIE le 0,count)
	if count gt 0 then begin
		TripleAmpErr(index)=-1
		TriplePhaseErr(index)=-1
	endif
;
;	This command effectively flags input data where the derived
;	amplitude is zero.
        index=where(TripleAmp eq 0,count)
	if count gt 0 then begin
		CTRE(index)=-1
		CTIE(index)=-1
		ComplTripleErr(*,0,*,*)=CTRE
		ComplTripleErr(*,1,*,*)=CTIE
	endif
;
end
;-------------------------------------------------------------------------------
pro set_complexvis,multiple=multiple
;
; Use squared visibility and triplephase to set the complex visibility.
; Sets baselines one triple at a time, and checks for duplicate baselines.
;
; Note: currently superseded by next procedure!
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
numscan=n_elements(scans)
;
vis=sqrt(abs(scans.vissqc))
phase=vis*0
done1=phase(*,*,*,0)
scans.complexweight=0
;
for n=0,numscan-1 do begin
done1(*)=0
;
for i=0,GenConfig.NumTriple-1 do begin
fBaseMatrix=intarr(GenConfig.NumSid,3)
fBaseFactor=fltarr(3)+1
for l=0,2 do begin
	ob=GenConfig.TripleBeam(l,i)
	bl=GenConfig.TripleBase(l,i)
	j1=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(bl,ob),0,3))
	j2=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(bl,ob),4,3))
	fBaseMatrix(j1,l)=+1
	fBaseMatrix(j2,l)=-1
	if l gt 0 then fBaseFactor(l)=(-total(fBaseMatrix(*,0)*fBaseMatrix(*,l)) gt 0)*2-1.0
endfor
for j=0,GenConfig.TripleNumChan(i)-1 do begin
	done2=0
	for l=0,2 do begin
		if not done2 then begin
		ob=GenConfig.TripleBeam(l,i)
		ch=GenConfig.TripleChan(j,l,i)
		bl=GenConfig.TripleBase(l,i)
		if not done1(ob,ch,bl) and scans(n).triplephasecerr(i,j) gt 0 then begin
			clp=0.0
			for k=0,2 do begin
				done1(GenConfig.TripleBeam(k,i), $
				      GenConfig.TripleChan(j,k,i), $
				      GenConfig.TripleBase(k,i))=1
				scans(n).complexweight(GenConfig.TripleBeam(k,i), $
						       GenConfig.TripleChan(j,k,i), $
						       GenConfig.TripleBase(k,i))=1
				for i_ob=0,GenConfig.NumOutBeam-1 do begin
				for i_bl=0,GenConfig.NumBaseline(i_ob)-1 do begin
					if GenConfig.BaselineId(i_bl,i_ob) $
					eq GenConfig.BaselineId(GenConfig.TripleBase(k,i), $
								GenConfig.TripleBeam(k,i)) $
					then begin
					done1(i_ob,GenConfig.TripleChan(j,k,i),i_bl)=1
					scans(n).complexweight(i_ob, $
							       GenConfig.TripleChan(j,k,i), $
							       i_bl)=1
					endif
				endfor
				endfor
				clp=clp+phase(GenConfig.TripleBeam(k,i), $
					      GenConfig.TripleChan(j,k,i), $
					      GenConfig.TripleBase(k,i),n)*fBaseFactor(k)
			endfor
			phase(ob,ch,bl,n)=(scans(n).triplephasec(i,j)-clp)*fBaseFactor(l)
			for i_ob=0,GenConfig.NumOutBeam-1 do begin
			for i_bl=0,GenConfig.NumBaseline(i_ob)-1 do begin
				if GenConfig.BaselineId(i_bl,i_ob) $
				eq GenConfig.BaselineId(bl,ob) then $
				phase(i_ob,ch,i_bl,n)=phase(ob,ch,bl,n)
			endfor
			endfor
			done2=1
		endif
		endif
	endfor
endfor
endfor
;
endfor
;
real_vis=cos(phase)*vis
imag_vis=sin(phase)*vis
scans.complexvis=complex(real_vis,imag_vis)
;
print,'ComplexVis set.'
;
end
;-------------------------------------------------------------------------------
pro set_complexvis,multiple=multiple
;
; Use squared visibility and triplephase to set the complex visibility for 
; imaging. For NPOI it is assumed that all spectrometers have an identical 
; channel layout, so that a loop over all triples will initialize all baselines.
; Otherwise it would be non-trivial to determine the common solution
; baselines. Uses SVD to compute solution which satisfies all closure
; phases.
;
; If the spectrometers are different, assume single triples.
;
; For data without triples, just set all phases to zero.
;
; Also copy photometry to model. 
;
; Do all nights if multiple=true.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if not keyword_set(multiple) then multiple=0
;
if multiple then nights=geninfo.date else nights=genconfig.date
;
FOR night=0,n_elements(nights)-1 DO BEGIN
;
if n_elements(nights) gt 1 then $
loadnight,GenInfo(night(0)).date, $
	  GeoInfo(night(0)).systemid, $
	  GenInfo(night(0)).configid
;
if checkdata([8,9]) ne 0 then return
numscan=n_elements(scans)
;
vis=sqrt(abs(scans.vissqc))
phase=vis*0	; to be used if there are no triples
scans.complexweight=scans.vissqcerr gt 0
;
if genconfig.numtriple ge 1 then begin
;
scans.complexweight=0
;
; Check channel and spectrometer layout
nc=genconfig.triplenumchan(0)
index=where(genconfig.triplenumchan(0:genconfig.numtriple-1) ne nc,count1)
index=where(genconfig.numspecchan(0:genconfig.numoutbeam-1) ne nc,count2)
if count1 eq 0 and count2 eq 0 then begin
	single_spec=1
endif else begin
	single_spec=0
endelse
;
baselines=unique(genconfig.baselineid(*,0:genconfig.numoutbeam-1))
;
fBaseMatrix=intarr(GenConfig.NumSid,3)
fBaseFactor=fltarr(3,genconfig.numtriple)+1
for i=0,GenConfig.NumTriple-1 do begin
	fBaseMatrix(*)=0
	for l=0,2 do begin
		ob=GenConfig.TripleBeam(l,i)
		bl=GenConfig.TripleBase(l,i)
		j1=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(bl,ob),0,3))
		j2=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(bl,ob),4,3))
		fBaseMatrix(j1,l)=+1
		fBaseMatrix(j2,l)=-1
		if l gt 0 then $
		fBaseFactor(l,i)=(-total(fBaseMatrix(*,0)*fBaseMatrix(*,l)) gt 0)*2-1.0
;		fBaseFactor(l,i)=-total(fBaseMatrix(*,0)*fBaseMatrix(*,l))	; same
	endfor
endfor
;
IF single_spec THEN BEGIN
;
for n=0,numscan-1 do begin
for j=0,nc-1 do begin
;
nr=0
;
dm=fltarr(genconfig.numtriple,n_elements(baselines))
rm=fltarr(genconfig.numtriple)
for i=0,genconfig.numtriple-1 do begin
	if scans(n).triplephasecerr(i,j) gt 0 then begin
	rm(nr)=scans(n).triplephasec(i,j)
	for l=0,2 do begin
		ob=GenConfig.TripleBeam(l,i)
		ch=GenConfig.TripleChan(j,l,i)
		bl=GenConfig.TripleBase(l,i)
		dm(nr,where(baselines eq genconfig.baselineid(bl,ob)))=fBaseFactor(l,i)
	endfor
	nr=nr+1
	endif
endfor
if nr gt 0 then begin
	dm=dm(0:nr-1,*)
	rm=rm(0:nr-1)
;	tm=transpose(dm)	; here we don't compute the normal matrix!
;	nm=tm#dm
;	ym=tm#rm
;	svdc,nm,w,u,v	; Annoyingly, this can fail!
;	svd8,nm,w,u,v
	svd8,dm,w,u,v
	small=where(w lt 0.1,count)
	if count gt 0 then w(small)=0
;	s=svsol(u,w,v,ym)
;	svb8,u,w,v,ym,s
	svb8,u,w,v,rm,s
	for ob=0,genconfig.numoutbeam-1 do begin
	for bl=0,genconfig.numbaseline(ob)-1 do begin
		phase(ob,ch,bl,n)=s(where(baselines eq genconfig.baselineid(bl,ob)))
		if scans(n).vissqcerr(ob,j,bl) gt 0 then $
		scans(n).complexweight(ob,j,bl)=1
	endfor
	endfor
endif
;
endfor
endfor
;
ENDIF ELSE BEGIN	; single_spec=1
;
phase(*,*,0,*)=scans.triplephasec
complexweight=scans.complexweight
complexweight(*)=1
index=where(scans.triplephasecerr lt 0,count)
if count gt 0 then begin
	for l=0,2 do begin
		cw=complexweight(*,*,l,*)
		cw(index)=0
		complexweight(*,*,l,*)=cw
	endfor
endif
scans.complexweight=complexweight
;
ENDELSE			; single_spec=0
;
endif	; Triples
;
real_vis=cos(phase)*vis
imag_vis=sin(phase)*vis
scans.complexvis=complex(real_vis,imag_vis)
;
; Copy also photometry into model
scans.photometrym=scans.photometry
;
if n_elements(bufferinfo) gt 1 then $
storenight,11L
;
ENDFOR
;
print,'ComplexVis set.'
;
end
;-------------------------------------------------------------------------------
pro set_triplephase
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
phase=cphase(scans.complexvis)
nc=max(genconfig.numspecchan)
ns=n_elements(scans)
;
g=genconfig
;
for i=0,genconfig.numtriple-1 do begin
fBaseMatrix=intarr(GenConfig.NumSid,3)
fBaseFactor=fltarr(3)+1
for l=0,2 do begin
	ob=GenConfig.TripleBeam(l,i)
	bl=GenConfig.TripleBase(l,i)
	j1=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(bl,ob),0,3))
	j2=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(bl,ob),4,3))
	fBaseMatrix(j1,l)=+1
	fBaseMatrix(j2,l)=-1
	if l gt 0 then fBaseFactor(l)=(-total(fBaseMatrix(*,0)*fBaseMatrix(*,l)) gt 0)*2-1.0
endfor
scans.triplephasec(i,*)=reform $
       (fBaseFactor(0)*phase(g.triplebeam(0,i),*,g.triplebase(0,i),*) $
       +fBaseFactor(1)*phase(g.triplebeam(1,i),*,g.triplebase(1,i),*) $
       +fBaseFactor(2)*phase(g.triplebeam(2,i),*,g.triplebase(2,i),*),1,nc,ns)
endfor
;
end
;-------------------------------------------------------------------------------
pro set_compltriple
;
; Discard (currently uncalibrated) ComplTriple from CHAMELEON and replace with 
; vector with phase equal to the (calibrated) TriplePhase and length equal to
; the triple amplitude.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
FOR i=0,GenConfig.NumTriple-1 DO BEGIN
;
; Real part
result=reform( $
	cos(scans(*).triplephasec(i,0:GenConfig.TripleNumChan(i)-1)) $
	*scans(*).tripleampc(i,0:GenConfig.TripleNumChan(i)-1), $
	1,1,GenConfig.TripleNumChan(i),n_elements(scans))
if GenConfig.TripleNumChan(i) eq 1 then result=reform(result,1,n_elements(scans))
scans(*).compltriple(i,0,0:GenConfig.TripleNumChan(i)-1)=result
;
; Imaginary part
result=reform( $
	sin(scans(*).triplephasec(i,0:GenConfig.TripleNumChan(i)-1)) $
	*scans(*).tripleampc(i,0:GenConfig.TripleNumChan(i)-1), $
	1,1,GenConfig.TripleNumChan(i),n_elements(scans))
if GenConfig.TripleNumChan(i) eq 1 then result=reform(result,1,n_elements(scans))
scans(*).compltriple(i,1,0:GenConfig.TripleNumChan(i)-1)=result
;
; Error mask
err=fltarr(GenConfig.TripleNumChan(i)*n_elements(scans))+1
index=where(scans(*).tripleampcerr(i,0:GenConfig.TripleNumChan(i)-1) lt 0 $
         or scans(*).triplephasecerr(i,0:GenConfig.TripleNumChan(i)-1) lt 0,count)
if count gt 0 then err(index)=-1
err=reform(err,1,1,GenConfig.TripleNumChan(i),n_elements(scans))
;
; Error of real part
; Previous computation, to be used with previous in marquardtdata
; result=reform( $
; 	sqrt(abs(sin(scans(*).triplephasec(i,0:GenConfig.TripleNumChan(i)-1)) $
; 			*scans(*).triplephasecerr(i,0:GenConfig.TripleNumChan(i)-1) $
; 			*scans(*).tripleampc(i,0:GenConfig.TripleNumChan(i)-1))^2 $
; 	   +(cos(scans(*).triplephasec(i,0:GenConfig.TripleNumChan(i)-1)) $
;		*scans(*).tripleampcerr(i,0:GenConfig.TripleNumChan(i)-1))^2), $
; 	1,1,GenConfig.TripleNumChan(i),n_elements(scans))*err
; New computation based on Buscher's Theorem
result=reform( $
	abs(scans(*).tripleampcerr(i,0:GenConfig.TripleNumChan(i)-1)), $
	1,1,GenConfig.TripleNumChan(i),n_elements(scans))*err
;
if GenConfig.TripleNumChan(i) eq 1 then result=reform(result,1,n_elements(scans))
scans(*).compltripleerr(i,0,0:GenConfig.TripleNumChan(i)-1)=result
;
; Error of imaginart part
; Previous computation, to be used with previous in marquardtdata
; result=reform( $
; 	sqrt(abs(cos(scans(*).triplephasec(i,0:GenConfig.TripleNumChan(i)-1)) $
; 			*scans(*).triplephasecerr(i,0:GenConfig.TripleNumChan(i)-1) $
; 			*scans(*).tripleampc(i,0:GenConfig.TripleNumChan(i)-1))^2 $
; 	   +(sin(scans(*).triplephasec(i,0:GenConfig.TripleNumChan(i)-1)) $
;		*scans(*).tripleampcerr(i,0:GenConfig.TripleNumChan(i)-1))^2), $
; 	1,1,GenConfig.TripleNumChan(i),n_elements(scans))*err
; New computation based on Buscher's Theorem
result=reform( $
	abs(tan(scans(*).triplephasecerr(i,0:GenConfig.TripleNumChan(i)-1)) $
		*scans(*).tripleampc(i,0:GenConfig.TripleNumChan(i)-1)), $
	1,1,GenConfig.TripleNumChan(i),n_elements(scans))*err
;
if GenConfig.TripleNumChan(i) eq 1 then result=reform(result,1,n_elements(scans))
scans(*).compltripleerr(i,1,0:GenConfig.TripleNumChan(i)-1)=result
;
ENDFOR
;
end
;-------------------------------------------------------------------------------
pro set_viserrors,use_e
;
; Compute the calibration error based on temporal fluctuations of the
; calibrator visibilities squared and triple amplitudes, and add in
; quadrature to the error of the calibrated visibilities for selected stars.
; To compute the calibration error, use the EC visibilities to compute
; chisq, and compute the additive error as sqrt((chisq-1)*e^2/z^2), where
; e is the intrinsic error, and z is the calibrated visibility (which is
; not necessarily unity if the calibrator is resolved). The latter two
; are mean quantities, i.e. the average of the values. Other means might
; be considered too (geometric, mean of the squares, etc.). The aim is to 
; produce unity chisq for the calibrator data.
;
; Derive selected program and calibrator stars from the selection made
; in the appropriate widgets.
;
; If use_e=1, use the normalized amplitudes (E) instead of the 
; calibrated normalized amplitdes (EC). This is useful if these were
; not calibrated for any reason. In that case, we "calibrate" by fitting
; polynomials to the data of each channel, before computing the chisq.
;
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; See whether uncalibrated but normalized amplitudes are to be used
if n_elements(use_e) ne 0 then use_e=use_e gt 0 else use_e=0
;
; Check if calibrator stars are defined
if n_elements(cal_stars) eq 0 then begin
	print,'***Error(SET_VISERRORS): calibrator list undefined!'
	return
endif
;
; See if calibrator stars were selected
if total(strlen(cal_stars)) eq 0 then begin
	print,'***Error(SET_VISERRORS): no calibrators selected!'
	return
endif
;
; Obtain scan index corresponding to calibrator star selection
list_stars,stars
mask=intarr(n_elements(scans))
for k=0,n_elements(stars)-1 do begin
        index=where(cal_stars eq stars(k),count)
        if count eq 0 then begin
                index=where(scans.StarId eq stars(k))
                mask(index)=-1
        endif
endfor
calscan_index=where(mask eq 0,calscan_count)
MINSCAN=2
if calscan_count lt MINSCAN then begin
	print,'***Error(SET_VISERRORS): not enough calibrator scans!'
	return
endif
;
; Check if program stars are defined
flag=set_starsel()
if set_indexsel('y') ne 0 then return
if n_elements(ds_stars) eq 0 then begin
	print,'***Error(SET_VISERRORS): program star list undefined!'
	return
endif
;
; See if program stars were selected
if total(strlen(ds_stars)) eq 0 then begin
	print,'***Error(SET_VISERRORS): no program stars selected!'
	return
endif
;
; Obtain scan index corresponding to program star selection
list_stars,stars
mask=intarr(n_elements(scans))
for k=0,n_elements(stars)-1 do begin
	index=where(ds_stars eq stars(k),count)
	if count eq 0 then begin
		index=where(scans.StarId eq stars(k))
		mask(index)=-1
	endif
endfor
scan_index=where(mask eq 0,scan_count)
;
; Determine r.m.s. of calibrator scans for each output beam and channel
for ob=0,GenConfig.NumOutBeam-1 do begin
for bl=0,GenConfig.NumBaseline(ob)-1 do begin
;
bad_ch=intarr(GenConfig.NumSpecChan(ob))
for ch=0,GenConfig.NumSpecChan(ob)-1 do begin
;
	if use_e then begin
		cv=scans(calscan_index).VisSqE(ob,ch,bl)
		ce=scans(calscan_index).VisSqEErr(ob,ch,bl)
		ct=scans(calscan_index).time/3600
		index=where(ce gt 0,count)
		if count gt MINSCAN then begin
			cv_fit=poly(ct,poly_fit(ct(index),cv(index),1))
			cv=cv/cv_fit
			ce=ce/cv_fit
		endif
	endif else begin
		cv=scans(calscan_index).VisSqEC(ob,ch,bl)
		ce=scans(calscan_index).VisSqECErr(ob,ch,bl)
	endelse
	index_ce=where(ce gt 0,count_ce)
	if count_ce gt MINSCAN then begin
;		Reset error bars first
		j=where(scans(scan_index).VisSq(ob,ch,bl) ne 0,count)
		scans(scan_index(j)).vissqcerr(ob,ch,bl)= $
		 scans(scan_index(j)).vissqerr(ob,ch,bl) $
		 *scans(scan_index(j)).vissqc(ob,ch,bl) $
		 /scans(scan_index(j)).vissq(ob,ch,bl)
		chisq=total(((cv(index_ce)-1)/ce(index_ce))^2)/(count_ce-1)
		if chisq lt 1 then chisq=1
		w=sqrt(chisq-1) $
		 *avg(scans(calscan_index(index_ce)).VisSqCErr(ob,ch,bl) $
		     /scans(calscan_index(index_ce)).VisSqC(ob,ch,bl))
		j=where(scans(scan_index).VisSqCErr(ob,ch,bl) lt 0,count)
		scans(scan_index).VisSqCErr(ob,ch,bl)= $
		 sqrt(scans(scan_index).VisSqCErr(ob,ch,bl)^2 $
		    +(scans(scan_index).VisSqC(ob,ch,bl)*w)^2)
		if count gt 0 then begin
			scans(scan_index(j)).VisSqCErr(ob,ch,bl)= $
		       -scans(scan_index(j)).VisSqCErr(ob,ch,bl)
		endif
	endif else begin
		bad_ch(ch)=ch+1
	endelse
endfor
index=where(bad_ch ne 0,count)
if count gt 0 then $
	print,'Warning(SET_VISERRORS): not enough cal scans for ob=', $
	ob+1,', bl=',bl+1,', ch=',retroparse(bad_ch(index)), $
	format='(a,i1,a,i1,a,a)'
endfor
endfor
;
; Determine r.m.s. of calibrator scans for each output triple and channel
for tr=0,GenConfig.NumTriple-1 do begin
bad_ch=intarr(GenConfig.TripleNumChan(tr))
for ch=0,GenConfig.TripleNumChan(tr)-1 do begin
;
;	Triple amplitudes
	if use_e then begin
		cv=scans(calscan_index).TripleAmpE(tr,ch)
		ce=scans(calscan_index).TripleAmpEErr(tr,ch)
		ct=scans(calscan_index).time/3600
		index=where(ce gt 0,count)
		if count gt MINSCAN then begin
			cv_fit=poly(ct,poly_fit(ct(index),cv(index),1))
			cv=cv/cv_fit
			ce=ce/cv_fit
		endif
	endif else begin
		cv=scans(calscan_index).TripleAmpEC(tr,ch)
		ce=scans(calscan_index).TripleAmpECErr(tr,ch)
	endelse
	index_ce=where(ce gt 0,count_ce)
	if count_ce gt MINSCAN then begin
;		Reset error bar first
		j=where(scans(scan_index).TripleAmp(tr,ch) ne 0,count)
		scans(scan_index(j)).TripleAmpCErr(tr,ch)= $
		 scans(scan_index(j)).TripleAmpErr(tr,ch) $
		 *scans(scan_index(j)).TripleAmpC(tr,ch) $
		 /scans(scan_index(j)).TripleAmp(tr,ch)
		chisq=total(((cv(index_ce)-1)/ce(index_ce))^2)/(count_ce-2)
		if chisq lt 1 then chisq=1
		w=sqrt(chisq-1) $
		 *avg(scans(calscan_index(index_ce)).TripleAmpCErr(tr,ch) $
		     /scans(calscan_index(index_ce)).TripleAmpC(tr,ch))
		j=where(scans(scan_index).TripleAmpCErr(tr,ch) lt 0,count)
		scans(scan_index).TripleAmpCErr(tr,ch)= $
		 sqrt(scans(scan_index).TripleAmpCErr(tr,ch)^2 $
		    +(scans(scan_index).TripleAmpC(tr,ch)*w)^2)
		if count gt 0 then begin
			scans(scan_index(j)).TripleAmpCErr(tr,ch)= $
		       -scans(scan_index(j)).TripleAmpCErr(tr,ch)
		endif
	endif else begin
		bad_ch(ch)=ch+1
	endelse
;
endfor
index=where(bad_ch ne 0,count)
if count gt 0 then $
	print,'Warning(SET_VISERRORS): not enough cal scans for tr=', $
	tr+1,', ch=',retroparse(bad_ch(index)),format='(a,i1,a,a)'
endfor
;
; Determine r.m.s. of calibrator scans for each output triple and channel
for tr=0,GenConfig.NumTriple-1 do begin
bad_ch=intarr(GenConfig.TripleNumChan(tr))
for ch=0,GenConfig.TripleNumChan(tr)-1 do begin
;
;	Closure phases
;	if use_e then begin
;		cv=scans(calscan_index).TriplePhase(tr,ch)
;		ce=scans(calscan_index).TriplePhaseErr(tr,ch)
;		ct=scans(calscan_index).time/3600
;		index=where(ce gt 0,count)
;		if count gt MINSCAN then begin
;			cv_fit=poly(ct,poly_fit(ct(index),cv(index),1))
;			cv=cv-cv_fit
;		endif
;	endif else begin
		cv=scans(calscan_index).TriplePhaseC(tr,ch)
		ce=scans(calscan_index).TriplePhaseCErr(tr,ch)
;	endelse
	index_ce=where(ce gt 0,count_ce)
	if count_ce gt MINSCAN then begin
;		Reset error bar first
		scans(scan_index).TriplePhaseCErr(tr,ch)= $
		 scans(scan_index).TriplePhaseErr(tr,ch)
		chisq=total(((cv(index_ce)-0)/ce(index_ce))^2)/(count_ce-2)
		if chisq lt 1 then chisq=1
		w=sqrt(abs(stdev(cv(index_ce))^2-median(ce(index_ce))^2/chisq))
		j=where(scans(scan_index).TriplePhaseCErr(tr,ch) lt 0,count)
		scans(scan_index).TriplePhaseCErr(tr,ch)= $
		 sqrt(scans(scan_index).TriplePhaseCErr(tr,ch)^2+w^2)
		if count gt 0 then begin
			scans(scan_index(j)).TriplePhaseCErr(tr,ch)= $
		       -scans(scan_index(j)).TriplePhaseCErr(tr,ch)
		endif
	endif else begin
		bad_ch(ch)=ch+1
	endelse
;
endfor
index=where(bad_ch ne 0,count)
if count gt 0 then $
	print,'Warning(SET_VISERRORS): not enough cal scans for tr=', $
	tr+1,', ch=',retroparse(bad_ch(index)),format='(a,i1,a,a)'
endfor
;
print,'VisErrors set for stars',' '+ds_stars,'.',format='(7a)'
;
end
;-------------------------------------------------------------------------------
pro set_minerrors,cal_stars_in
;
; Compute the calibration error based on *spectral* fluctuations of the
; calibrator visibilities squared and triple amplitudes, and add in
; quadrature to the error of the calibrated visibilities.
; To compute the calibration error, use the EC visibilities to compute
; chisq, and compute the additive error as sqrt((chisq-1)*e^2/z^2), where
; e is the intrinsic error, and z is the calibrated visibility (which is
; not necessarily unity if the calibrator is resolved). The latter two
; are mean quantities, i.e. the average of the values. Other means might
; be considered too (geometric, mean of the squares, etc.). The aim is to 
; produce unity chisq for the calibrator data.
;
; Derive selected program and calibrator stars from the selection made
; in the appropriate widgets.
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,startable,notes
;
for i=0,n_elements(geninfo.date)-1 do begin
if geoinfo(i).systemid eq 'NPOI' then begin
;
if n_elements(geninfo.date) gt 1 then $
loadnight,geninfo(i).date, $
	  geoinfo(i).systemid, $
	  geninfo(i).configid
index=where(startable.bflag eq 'C')
if n_elements(cal_stars_in) eq 0 then cal_stars=startable(index).starid $
				 else cal_stars=cal_stars_in
ds_stars=startable.starid
;
;
; Obtain scan index corresponding to calibrator star selection
list_stars,stars
mask=intarr(n_elements(scans))
for k=0,n_elements(stars)-1 do begin
        index=where(cal_stars eq stars(k),count)
        if count eq 0 then begin
                index=where(scans.StarId eq stars(k))
                mask(index)=-1
        endif
endfor
calscan_index=where(mask eq 0,calscan_count)
;
; Obtain scan index corresponding to program star selection
list_stars,stars
mask=intarr(n_elements(scans))
for k=0,n_elements(stars)-1 do begin
	index=where(ds_stars eq stars(k),count)
	if count eq 0 then begin
		index=where(scans.StarId eq stars(k))
		mask(index)=-1
	endif
endfor
scan_index=where(mask eq 0,scan_count)
;
; Determine r.m.s. of calibrator in each channel of a spectrometer
for ob=0,GenConfig.NumOutBeam-1 do begin
for bl=0,GenConfig.NumBaseline(ob)-1 do begin
;
v2=fltarr(GenConfig.NumSpecChan(ob),calscan_count)
e2=v2
;
; Compute floating calibration factor for each baseline
for sc=0,calscan_count-1 do begin
	y=scans(calscan_index(sc)).vissqc(ob,*,bl)
	e=scans(calscan_index(sc)).vissqcerr(ob,*,bl)
	e_index=where(e gt 0,count)
	if count eq 0 then continue
	y_e=y(e_index)
	e_e=e(e_index)
	ymodel=y_e*0+1
	if n_elements(ymodel) eq 1 then ymodel=fltarr(1)+ymodel
	nf=140
	f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
	for k=0,nf do r(k)=total(((y_e*f(k)-ymodel)/e_e)^2)
	k=where(r eq min(r)) & k=k(0)
	y=y*f(k)
	e=e*f(k)
	v2(*,sc)=y
	e2(*,sc)=e
endfor
;
; Compute chisq in each channel...
for ch=0,GenConfig.NumSpecChan(ob)-1 do begin
	e_index=where(e2(ch,*) gt 0,count)
	if count eq 0 then continue
	chisq=(total(((v2(ch,e_index)-1)/e2(ch,e_index))^2)/count) > 1
;
; 	...and apply corrections
	w=sqrt(chisq-1)*avg(e2(ch,e_index)/v2(ch,e_index))
	j=where(scans(scan_index).VisSqCErr(ob,ch,bl) lt 0,count)
	scans(scan_index).VisSqCErr(ob,ch,bl)= $
	 sqrt(scans(scan_index).VisSqCErr(ob,ch,bl)^2 $
	    +(scans(scan_index).VisSqC(ob,ch,bl)*w)^2)
	if count gt 0 then begin
		scans(scan_index(j)).VisSqCErr(ob,ch,bl)= $
	       -scans(scan_index(j)).VisSqCErr(ob,ch,bl)
	endif
endfor
;
endfor
endfor
;
; Do the same for triple amplitudes
for tr=0,genconfig.numtriple-1 do begin
;
ta=fltarr(genconfig.triplenumchan(tr),calscan_count)
ea=ta
;
; Compute floating calibration factor for each baseline
for sc=0,calscan_count-1 do begin
	y=scans(calscan_index(sc)).tripleampc(tr,*)
	e=scans(calscan_index(sc)).tripleampcerr(tr,*)
	e_index=where(e gt 0,count)
	if count eq 0 then continue
	y_e=y(e_index)
	e_e=e(e_index)
	ymodel=y_e*0+1
	if n_elements(ymodel) eq 1 then ymodel=fltarr(1)+ymodel
	nf=140
	f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
	for k=0,nf do r(k)=total(((y_e*f(k)-ymodel)/e_e)^2)
	k=where(r eq min(r)) & k=k(0)
	y=y*f(k)
	e=e*f(k)
	ta(*,sc)=y
	ea(*,sc)=e
endfor
;
; Compute chisq in each channel...
for ch=0,genconfig.triplenumchan(tr)-1 do begin
	e_index=where(ea(ch,*) gt 0,count)
	if count eq 0 then continue
	chisq=(total(((ta(ch,e_index)-1)/ea(ch,e_index))^2)/count) > 1
;
; 	...and apply corrections
	w=sqrt(chisq-1)*avg(ea(ch,e_index)/ta(ch,e_index))
	j=where(scans(scan_index).tripleampcerr(tr,ch) lt 0,count)
	scans(scan_index).tripleampcerr(tr,ch)= $
	 sqrt(scans(scan_index).tripleampcerr(tr,ch)^2 $
	    +(scans(scan_index).tripleampc(tr,ch)*w)^2)
	if count gt 0 then begin
		scans(scan_index(j)).tripleampcerr(tr,ch)= $
	       -scans(scan_index(j)).tripleampcerr(tr,ch)
	endif
endfor
endfor
;
if n_elements(bufferinfo) gt 1 then $
storenight,11
;
endif	; NPOI only
endfor	; Next night
;
end
;-------------------------------------------------------------------------------
pro set_maxerrors
;
; For data loaded in AMOEBA, re-compute maximum calibration errors.
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,startable,notes
;
for i=0,n_elements(geninfo.date)-1 do begin
if geoinfo(i).systemid eq 'NPOI' then begin
;
	if n_elements(geninfo.date) gt 1 then $
	loadnight,geninfo(i).date, $
		  geoinfo(i).systemid, $
		  geninfo(i).configid
	calcviscal
	index=where(startable.bflag eq 'C')
	cal_stars=startable(index).starid
	ds_stars=startable.starid
	set_viserrors,1
	if n_elements(bufferinfo) gt 1 then $
	storenight,11
;
endif
endfor
;
end
;-------------------------------------------------------------------------------
pro npoipipe,con_file
;
; Perform some of the NPOI data reduction, i.e. processing of CONSTRICTOR
; files, in an automatic fashion.
;
common Hds,path,file
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common PlotDataInfo,plotscans,plotdata_x,plotdata_y,plotdata_y_bck
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
;
if n_elements(con_file) ne 0 then begin
	hds_close
	get_data,con_file
	hds_close
endif
v_mag=fltarr(n_elements(scantable))
for i=0,n_elements(startable)-1 do begin
	index=where(scantable.starid eq startable(i).starid)
	v_mag(index)=startable(i).mv
endfor
;
; Open output file for diagnostic messages
openw,unit,file+'.txt',/get_lun
printf,unit,'----------------------------------------'
;
; Background data
medianbg
expandbg
;
; Point data plotting initialization
if init_plot('point','pt') ne 0 then return
create_pointflagtable
;
; Process station based data
for ib=0,genconfig.numsid-1 do begin
	ds_y.ib=ib
	ds_y.item=11	; NATJitter
	set_plotsel
	plotdata,1,'AUTO NATJITTER'
	y_med=median(plotscans.y(where(plotscans.ye gt 0)))
	ben_rms=[0.35,0.30,0.25,0.20,0.15,0.10,0.05]*sqrt(2)
	ben_bno=['B0','B1','B2','B3','B4','B5','B6']
	for i=0,n_elements(ben_bno)-1 do $
		if y_med lt ben_rms(i) then bno=ben_bno(i)
;
	ds_y.item=12	; NATCounts
	plotdata,1,'AUTO NATCOUNTS'
	c0=plotscans.y/10^(-v_mag/2.5) & c0=c0(where(c0 ne 0))
	c0=mean(c0)
	
;
;	Print diagnostic info
	printf,unit,'Station '+string(ib+1,format='(i1)')+':'
	printf,unit,'Seeing (Burress scale): '+bno
	printf,unit,'NAT counts for zero magnitude star: ' $
			+string(c0,format='(i5)')
	printf,unit,'----------------------------------------'
endfor
;
; Process baseline based data
for ob=0,genconfig.numoutbeam-1 do begin
	ds_y.ob=ob
;
	ds_y.ch=indgen(genconfig.numspecchan(ob))
	ds_y.item=8	; Photonrate
	set_plotsel
	plotdata,1,'AUTO PHOTONRATE'
;
	ds_y.bl=indgen(genconfig.numbaseline(ob))
	ds_y.item=10	; DelayJitter
	set_plotsel
	plotdata,1,'AUTO DELAYJITTER'
;
	ds_y.item=3	; VisSq
	plotdata,1,'AUTO VISSQ'
endfor
wait,2	; take a breath, once the script froze here, but why?
;
; Save pointflagtable
save,filename=file+'.flg',pointflagtable
;
; Average the pointdata
average
;
; Do the bias computations
if init_plot('ncal','pt') ne 0 then return
ps_options.a=1
type='V2Bias'
configs=scanconfig(/coh)
st_dir='Sel'
set_ps,date+'.ps',/color
for ob=0,genconfig.numoutbeam-1 do begin
	ds_y.ob=ob
	ds_y.ch=indgen(genconfig.numspecchan(ob))
	ds_y.bl=indgen(genconfig.numbaseline(ob))
for k=0,n_elements(startable)-1 do begin
	if startable(k).starid ne 'FKV0000' then begin
	st_sel=startable(k).starid
	s=startable(k).starid
	list_scans,s,/coh
	configs=unique(scanconfig(s))
	for i=0,n_elements(configs)-1 do begin
		y_dir.pt=configs(i)
		set_plotsel
		plotncal
	endfor
	endif
endfor
endfor
device,/close
set_screen
ps_options.a=0
;
; Average again, now that we determined the bias corrections 
average
;
; Flag incoherent scans
print,'Flagging incoherent scans...'
create_scanflagtable
index=where(scantable.code eq 0,count)
if count gt 0 then flagscandata,sc=index+1,item=24,reason='AUTO Incoh'
;
; Save flagtable
save,filename=file+'.flg',pointflagtable,scanflagtable
;
; Write to cha file
put_scandata,file+'.cha'
;
; Close diagnostic output file and print information to screen
free_lun,unit
list_summary_chameleon
spawn,'cat '+file+'.txt'
;
; Attempt to calibrate the data
initcal
cal_options.v=0	; Vissq
cal_options.l=1
sel_indct(0)='Time'
sel_funcn(0)='S_80'
;
index=where(startable.bflag eq 'C',count)
if count gt 0 then begin
	cal_stars=startable(index).starid
	print,'Found',count,' listed calibrators:',format='(a,i3,a)'
	print,cal_stars
endif else begin
	print,'Warning: no calibrators listed in startable! No cal. possible'
	return
endelse
;
; Make sure no false coherent scans are present
index=where(total(scantable.station(0:genconfig.numsid-1,*),1) eq 0,count)
if count gt 0 then scantable(index).code=0
;
if init_plot('scan','pt') ne 0 then return
st_dir='All'
; st_sel=startable(k).starid
ds_y.item=25
for ob=0,genconfig.numoutbeam-1 do begin
	ds_y.ob=ob
	ds_y.ch=indgen(genconfig.numspecchan(ob))
	ds_y.bl=indgen(genconfig.numbaseline(ob))
	configs=scanconfig(/coh)
	for i=0,n_elements(configs)-1 do begin
		y_dir.pt=configs(i)
		set_plotsel
		print,'Calibrating configuration ',configs(i)
		calcoeffs,reason
		if strlen(reason) ne 0 then begin
			calibrate,reason
;			wait,1
		endif
	endfor
endfor
;
end
;************************************************************************Block 6
pro startnpoiconfig
;
; This procedure was used to start a data base for C-term measurements.
;
; These measurements were taken by B.O'Neill as currently documented
; in file npoi/fdl.config.
storenpoiconfig,['AC0','AE0','AW0'],[2,3,4],-[18.928733,0.936589,4.801803]
storenpoiconfig,['AN0','AE0','AW0'],[5,3,4],-[11.776615,0.934349,4.799628]
storenpoiconfig,['AC0','AE0','AN0'],[2,3,1],-[18.928144,0.936048,11.473902]
storenpoiconfig,['AC0','AE0','AN0'],[2,3,4],-[18.928091,0.936350,11.539303]
storenpoiconfig,['AC0','AN0','AW0'],[2,3,4],-[18.928476,11.911872,4.802157]
storenpoiconfig,['AC0','AN0','AW0'],[2,6,4],-[18.928036,12.056784,4.801675]
storenpoiconfig,['AN0','AE0','AW0'],[2,3,4],-[11.666951,0.936485,4.801941]
storenpoiconfig,['AC0','AE0','AW0','AN0'],[2,3,4,6],-[18.926820,0.935700,4.800429,12.057755]
storenpoiconfig,['AC0','AE0','AW0','AN0'],[2,3,4,6],-[18.927108,0.935700,4.800791,12.056732]
storenpoiconfig,['AC0','AE0','AW0','AN0'],[2,3,4,6],-[18.927146,0.935700,4.800855,12.056637]
storenpoiconfig,['AC0','AE0','AW0'],[2,3,4],-[18.927484,0.935249,4.800800]
storenpoiconfig,['AW0','AN0'],[4,6],-[4.800800,12.054043]
storenpoiconfig,['AC0','AE0'],[2,3],-[18.929669,0.935811]
storenpoiconfig,['AC0','AW0'],[2,4],-[18.932131,4.800806]
storenpoiconfig,['AW0','AN0'],[4,6],-[4.800800,12.076757]
storenpoiconfig,['AW0','W07'],[4,5],-[31.310000,2.027864]
;
end
;-------------------------------------------------------------------------------
pro storenpoiconfig,stations,delaylineids,cterm
;
; Used to store C-term measurements in data base.
; Currently configured to accept only baseline/station triples as
; listed in procedure startconfig.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
systemid='NPOI'
;
numsid=n_elements(stations)
numoutbeam=1
maxnumtriple=1
numbaseline=((numsid-1)*numsid)/2
maxspecchan=32
numconfig=1
genconfig=allocgenconfig(numsid,numoutbeam,maxnumtriple,numbaseline,maxspecchan,numconfig)
;
genconfig.beamcombinerid=2
genconfig.numsid=numsid
genconfig.numoutbeam=numoutbeam
genconfig.numbaseline=numbaseline
genconfig.stationid(0:numsid-1)=stations
genconfig.delaylineid(0:numsid-1)=delaylineids
get_stationtable
genconfig.stationcoord(3,*)=cterm
l=0
for i=0,genconfig.numsid-2 do begin
for j=i+1,genconfig.numsid-1 do begin
	genconfig.baselineid(l,0)=stations(i)+'-'+stations(j)
	l=l+1
endfor
endfor
;
save,genconfig, $
	filename=!oyster_dir+'npoi/opd/'+string(systime(/julian), $
						format='(f16.8)')+'.xdr'
wait,1
;
end
;-------------------------------------------------------------------------------
pro npoiconfig
;
; Reads data base in npoi/config and solves for station and delay line
; based delays. The station part, D, can be used directly in stations.config.
; The delay line part has to have the beam combiner path contribution subtracted
; before using in fdl.config. The beam combiner contributions are listed in
; 6way.config.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
files=file_search(!oyster_dir+'npoi/opd/*.xdr')
if strlen(files(0)) eq 0 then nf=0 else nf=n_elements(files)
if nf eq 0 then return
s=''
status=dc_read_free(!oyster_dir+'npoi/stations.config',s,x,y,z,d,/col,ignore=['!'])
stations=s
ns=n_elements(stations)
cterms=fltarr(ns)
for i=0,ns-1 do cterms(i)=d(where(s eq stations(i)))
maxrow=1000
m=fltarr(maxrow,ns+6)
y=fltarr(maxrow)
n=0
for i=0,nf-1 do begin
	restore,files(i)
	for k=0,genconfig.numoutbeam-1 do begin
	for l=0,genconfig.numbaseline(k)-1 do begin
	i1=where(stations eq strmid(genconfig.baselineid(l,k),0,3))
	i2=where(stations eq strmid(genconfig.baselineid(l,k),4,3))
	m(n,i1)=+1 & m(n,i2)=-1 
	i1=where(genconfig.stationid eq strmid(genconfig.baselineid(l,k),0,3))
	i2=where(genconfig.stationid eq strmid(genconfig.baselineid(l,k),4,3))
	m(n,ns-1+genconfig.delaylineid(i1))=+1 & m(n,ns-1+genconfig.delaylineid(i2))=-1
	y(n)=genconfig.stationcoord(3,i1)-genconfig.stationcoord(3,i2)
	n=n+1
	endfor
	endfor
endfor
ct=total(abs(m),1)
; ct=ct(0:ns-1)
index=where(ct gt 0,ns6)
ns=ns6-6
m=m(*,index)
stations=stations(index(0:ns-1))
cterms=cterms(index(0:ns-1))
;
n_a_l=2
m=m(0:n-1+n_a_l,*)
y=y(0:n-1+n_a_l)
if n_a_l gt 0 then begin
	m(n+0,0)=1 & y(n+0)=cterms(0)
	m(n+1,ns)=1
endif
tm=transpose(m)
n=tm#m
r=tm#y
svdc,n,w,u,v
index=where(w/max(w) lt 1e-6,count) & if count gt 0 then w(index)=0
s=svsol(u,w,v,r)
for i=0,ns-1 do print,stations(i),s(i),format='(a,f11.6)'
for i=0,5 do print,'Delayline ID',i+1,s(ns+i),format='(a,i,f11.6)'
;
end
;-------------------------------------------------------------------------------
pro vlticonfig,uconf1,uconf2
;
; Read *-model.dat files containing OPD models for VLTI and solve for
; a new vector of paths depending on telescope, delayline, and channel
; input indices. These input files can be found on wdline, /vltdata/pointing.
; A user specified configuration (e.g. 'UT113' for UT1, DL 1, IC 3) 
; will be used to compute an improved estimate for OPL0.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
files=file_search(!oyster_dir+'vlti/opd/dl?UT???IP?-dl?UT???IP?_*-model.dat')
if strlen(files(0)) eq 0 then nf=0 else nf=n_elements(files)
if nf eq 0 then return
s=''
status=dc_read_free(!oyster_dir+'vlti/stations.config',s,x,y,z,d,/col,ignore=['!'])
stations=s
num_stations=n_elements(stations)
cterms=fltarr(num_stations)
; for i=0,num_stations-1 do cterms(i)=d(where(s eq stations(i)))
delaylines=lindgen(8)+1
inputchannels=lindgen(8)+1
num_delaylines=n_elements(delaylines)
num_inputchannels=n_elements(inputchannels)
maxrow=1000
m=fltarr(maxrow,num_stations+num_delaylines+num_inputchannels)
y=fltarr(maxrow)
n=0
for i=0,nf-1 do begin
	openr,unit,files(i),/get_lun
	l=''
	readf,unit,l
	readf,unit,l
	readf,unit,l
	free_lun,unit
	words=nameparse(l)
	y(n)=float(words(1))/1e6
	conf1=strmid(specname(files(i)),0,11)
	conf2=strmid(specname(files(i)),12,11)
;
	dl1=long(strmid(conf1,2,1))-1
	st1=long(strmid(conf1,5,1))-1
	ic1=long(strmid(conf1,10,1))-1
;
	dl2=long(strmid(conf2,2,1))-1
	st2=long(strmid(conf2,5,1))-1
	ic2=long(strmid(conf2,10,1))-1
;
	m(n,st1)=+1
	m(n,st2)=-1
;
	m(n,num_stations+dl1)=+1
	m(n,num_stations+dl2)=-1
;
	m(n,num_stations+num_delaylines+ic1)=+1
	m(n,num_stations+num_delaylines+ic2)=-1
;
	n=n+1
endfor
ct=total(abs(m),1)
nums=num_stations
numd=num_delaylines
numc=num_inputchannels
index_s=where(ct(0:nums-1) gt 0,num_stations)
index_d=where(ct(nums:nums+numd-1) gt 0,num_delaylines)
index_c=where(ct(nums+numd:nums+numd+numc-1) gt 0,num_inputchannels)
index=where(ct gt 0,num_columns)
m=m(*,index)
stations=stations(index_s)
cterms=cterms(index_s)
delaylines=delaylines(index_d)
inputchannels=inputchannels(index_c)
;
n_a_l=3
m=m(0:n-1+n_a_l,*)
y=y(0:n-1+n_a_l)
if n_a_l gt 0 then begin
	m(n+0,0)=1 & y(n+0)=cterms(0)
	m(n+1,num_stations)=1
	m(n+2,num_stations+num_delaylines)=1
endif
nrow=n
;
tm=transpose(m)
n=tm#m
r=tm#y
svdc,n,w,u,v
index=where(w/max(w) lt 1e-6,count) & if count gt 0 then w(index)=0
print,'Edited ',count,' zero eigenvalues.'
s=svsol(u,w,v,r)
for i=0,num_stations-1 do print,'Station ',stations(i),s(i)
for i=0,num_delaylines-1 do print,'Delayline ID',delaylines(i),s(num_stations+i)
for i=0,num_inputchannels-1 do print,'Input channel ID',inputchannels(i),s(num_stations+num_delaylines+i)
print,'O-C [mu]=',(y-m(0:nrow-1,*)#s)*1e6
;
if n_elements(uconf1) ne 0 and n_elements(uconf2) ne 0 then begin
	v=intarr(num_stations+num_delaylines+num_inputchannels)
	st1=strmid(uconf1,0,3)
	dl1=long(strmid(uconf1,3,1)) & dl1=where(delaylines eq dl1)
	ic1=long(strmid(uconf1,4,1)) & ic1=where(inputchannels eq ic1)
	i=where(stations eq st1)
	v(i)=1
	v(num_stations+dl1)=1
	v(num_stations+num_delaylines+ic1)=1
	st2=strmid(uconf2,0,3)
	dl2=long(strmid(uconf2,3,1)) & dl2=where(delaylines eq dl2)
	ic2=long(strmid(uconf2,4,1)) & ic2=where(inputchannels eq ic2)
	i=where(stations eq st2)
	v(i)=-1
	v(num_stations+dl2)=-1
	v(num_stations+num_delaylines+ic2)=-1
	print,'A1L=',total(v*s)
endif
;
end
;************************************************************************Block 7
pro apdrates,starid,predictedrate
;
; Propagate the stellar flux through the atmosphere and NPOI, and return
; the predicted count rate for every NPOI channel.
;
; Number of reflections by the time star light is entering the lab building:
;
; Siderostat, NAT, elevator can (N/W=2x, E=3x), Feed can (AC and AE only),
; array center (N, W only), periscope, LDL manifold (3x) + LDL pop-up (2x),
; beam compressor (2x), FDL (3x), switch yard (2x), BC (1x or 2x), 
; spectrometer (3x)
;
; Total of 22 for beams 1,2,3; 23 reflections for beams 4,5,6. Beams 1,2,3
; have one more beam splitter reflection.
;
; New info from SPIE 2006 poster:
; Feed system: 0.5-0.76
; FDL & SW: 0.72-0.84
; NAY: 0.8
; BC unused ports: 0.5
; Spectr. injection: 0.46
; Lenslet injection: 0.8
; Detector QE: 0.3
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Get predicted count rate/cm^2/s/nm
rates=stellarrate(starid,l)
l=l*1e9			; Change to lambda in nm
;
coat_t=0.99		; Transmission of coated window surfaces
unct_t=0.96		; Transmission of uncoated lenslet surfaces
s_rms=(850/l)/30	; Quality of mirror surface in units of lambda @ 850nm
p_rms=s_rms*2*!pi	; Phase rms across surface
ffld_t=exp(-p_rms^2/2)	; Far-field transmission due to mirror roughness
;
case systemid of
;
'NPOI':	begin
	aper=!pi*(12.5/2)^2	; Aperture 12.5 cm
	ci_t=1./500		; Coherent integration time 2 ms
	pickup=0.2		; That's what the NAT tracker BS takes away
	bc_t=0.25		; Beamcombiner (2 50/50) transmission
	throughput=(silver(l)*ffld_t)^22 $
	 *(1-pickup)*bc_t  $
	 *coat_t^12 $		; 3 windows (fdl passed twice), + prism & lens
	 *unct_t^5*apd(l)	; Lenslets have two lens design, + MM fiber
	f=throughput*rates*atmosphere(l)*aper*ci_t
	index=where(l gt 550 and l lt 850)
	print,'Average throughput [%]: ',avg(throughput(index))*100
	end
;
'Mark3':begin
	aper=!pi*(5.0/2)^2
	bc_t=0.5
	ci_t=1./250
	f=rates*atmosphere(l)*aper*silver(l)^7*coat_t^2*bc_t $
	 *unct_t^1*pmt(l)*ci_t
	end
;
endcase
;
; Integrate over bandpass, with proper compensation for lambda stepsize
predictedrate=fltarr(genconfig.numoutbeam,max(genconfig.numspecchan))
for ob=0,GenConfig.NumOutBeam-1 do begin
for ch=0,GenConfig.NumSpecChan(ob)-1 do begin
; 	Compute including duty cycle correction
	predictedrate(ob,ch)=total(f*generic_c(l,ch,ob))*(l(1)-l(0)) $
			    *(GenConfig.Wavelength(ch,ob)/1000e-9)
endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro natrates,starid,predictedrate
;
; Propagate the stellar flux through the atmosphere and NPOI, and return
; the predicted count rate for every NPOI channel.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Get predicted count rate/cm^2/s/nm
rate=stellarrate(starid,lambda)
lambda=lambda*1e9
;
; Interpolate onto finer grid
l=findgen(500)+400
rates=spline(lambda,rate,l)
;
ci_t=1./500		; Coherent integration 2 ms
aper=3.14*(12.5/2)^2	; Aperture 12 cm
coat_t=0.99		; Transmission of coated surfaces
unct_t=0.96		; Transmission of uncoated surfaces
pickup=0.2		; Beam splitter pickup fraction
predictedrate=rates*atmosphere(l)*aper*silver(l)^18*pickup*coat_t^10*unct_t^2*apd(l)*ci_t
;
end
;-------------------------------------------------------------------------------
pro photom,scan_ids,y=roc,e=eoc,init=init,nonl=nonl
;
; Attempt to assign relative throughputs to all stations, per spectrometer,
; based on changes in photon rate between different scan configurations.
; If the configurations don't change or there are too few configuration changes,
; let SVD assign values arbitrarily. This is OK since the results, stored in
; genconfig.reflux, are only used to put the visibility normalization at the
; time of averaging onto the same footing. 
;
; Combine all scans, which requires normalization of the stellar fluxes. The
; alternative is to do this calculation on a star-by-star basis, but then there
; could be an insufficient number of configuration changes for any given star.
;
; The data for file linear.xdr are in linear.dat, created by rflinear(misc.pro).
;
common LocalPhotom,rates
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; Obtain and store predicted count rates for all stars
if n_elements(init) eq 0 then init=0 else init=init gt 0
if n_elements(nonl) eq 0 then nonl=0 else nonl=nonl gt 0
if n_elements(rates) eq 0 or init then begin
	numstar=n_elements(startable)
	rates=fltarr(genconfig.numoutbeam,max(GenConfig.NumSpecChan),numstar)
	for i=0,numstar-1 do begin
		if startable(i).starid ne 'FKV0000' then begin
			apdrates,startable(i).starid,r
			rates(*,*,i)=r
		endif
	endfor 
endif
;
; Include non-linearity corrections
restore,!oyster_dir+'npoi/linear.xdr'
n_lin=n
q_lin=q
n_lin_ob=n_elements(q_lin(*,0,0))
n_lin_ch=n_elements(q_lin(0,*,0))
if n_lin_ob ne genconfig.numoutbeam $
or n_lin_ch ne max(genconfig.numspecchan) then begin
	n_lin=fltarr(genconfig.numoutbeam,max(genconfig.numspecchan),2)
	q_lin=n_lin+1
	for i=0,n_elements(q_lin(*,0,0))-1 do $
	for j=0,n_elements(q_lin(0,*,0))-1 do n_lin(i,j,1)=1000
endif	
;
; Process only data for specified scans; use all otherwise
if n_elements(scan_ids) eq 0 then scan_index=indgen(n_elements(scans)) $
			     else scan_index=scan_ids-1
stars=unique(scans(scan_index).starid)
numstar=n_elements(stars)
star_index=intarr(n_elements(startable))
for i=0,numstar-1 do begin
	index=where(startable.starid eq stars(i),count)
	if count gt 0 then star_index(index)=1
endfor
index=where(star_index eq 1,count)
if count eq 0 then begin
	print,'***Error(PHOTOM): none of these stars exist in the startable!'
	return
endif
rates1=rates(*,*,index)
star_index=scan_index*0-1
for i=0,numstar-1 do begin
	index=where(scans(scan_index).starid eq stars(i),count)
	if count gt 0 then star_index(index)=i
endfor
s=scantable(scan_index).station
;
roc=scans(scan_index).photonrate*0
eoc=scans(scan_index).photonrate*0
;
for ob=0,genconfig.numoutbeam-1 do begin
	sindex=intarr(genconfig.numsid)
	for i=0,genconfig.numsid-1 do $
		sindex(i)=strpos(strjoin(genconfig.baselineid(*,ob)), $
				genconfig.stationid(i))
	s_index=where(total(s,2) ne 0 and sindex ne -1)
	for j=0,genconfig.numspecchan(ob)-1 do begin
		x=extinct(scans(scan_index).za)
		n=scans(scan_index).natcounts(s_index)
		p=scans(scan_index).photonrate(ob,j)
		q=spline8(n_lin(ob,j,*),q_lin(ob,j,*),p)
		if not nonl then q(*)=1
		r=(p/x*q)/rates1(ob,j,star_index)
		re=scans(scan_index).photonrateerr(ob,j)
		index=where(re gt 0,count)
		if count gt 0 then begin
		se=s(*,index)
		dm=transpose(float(se(s_index,*)))
		tm=transpose(dm)
		nm=tm#dm
		y=tm#r(index)
		if n_elements(nm) eq 1 then begin
			sol=y/nm
		endif else begin
			svdc,nm,w,u,v
			small=where(w lt max(w)*1e-08,count)
			if count gt 0 then w(small)=0
			sol=svsol(u,w,v,y)
		endelse
		genconfig.apdflux(      *,j,ob)=0
		genconfig.apdflux(s_index,j,ob)=sol/total(sol)
		c=dm#sol
		jndex=where(c eq 0,count)
		if count gt 0 then c(jndex)=1
		roc(ob,j,index)=r(index)/c
		eoc(ob,j,index)=0.1
		if count gt 0 then eoc(ob,j,index(jndex))=-0.1
;
		tm=tm*n
		dm=transpose(tm)
		nm=tm#dm
		y=tm#p(index)
		if n_elements(nm) eq 1 then begin
			sol=y/nm
		endif else begin
			svdc,nm,w,u,v
			small=where(w lt max(w)*1e-08,count)
			if count gt 0 then w(small)=0
			sol=svsol(u,w,v,y)
		endelse
		genconfig.natflux(      *,j,ob)=0
		genconfig.natflux(s_index,j,ob)=sol
		endif
	endfor
endfor
;
end
;-------------------------------------------------------------------------------
