;*******************************************************************************
; File: functions.pro
;
; Description:
; ------------
; This file contains various functions for the CHAMELEON/AMOEBA software.
;
; Block directory:
; ----------------
; Match, utility, power, tv, time/coordinates
;
; Math:
; Block 1: fakultaet,combinations,base36,base2,eqp,produkt,kreuzprodukt,
;	   sinc,quasol,winkel,ellipse,apollonius,
;	   summe,add,box,wmean,medianve,gavg,ravg1,ravg2,gsmooth,maplevel,
; 	   signof,nonzero,cphase,cmag,dmag,hb,viscalerror,bessjy8,spline8,
;	   evalfunction,polynom,polyfit,polyres,perinom,perifit,perires,
;	   sizeof,natural,contiguous,rankindex,whereindex,whereequal,
;
; Utilities:
; Block 2: poweroftwo,powerpatch,powerspectrum,siggen,strucfunc,
;	   signalfilter,highpass,lowpass,notch
;	   blanks,stringl,myhak,elements,printindex,isnumeric,
;	   gitter,mergetable,commonbeam,
;	   ppwv,edlen,owens,oneill,silica,extinct,blackbody,
;	   v2jy,j2jy,h2jy,k2jy,l2jy,m2jy,n2jy,jy2n,q2jy,
;	   getcolor,getsymbol,greek,alphabet,
;	   histogramm,histograph,
; Block 3: hms,dms,hms2h,dms2d,wdsid,esoid,wdspos,esopos,
;	   stringparse,retroparse,nameparse,fitshparse,
;	   criparse,cri,cri_simbad,cri_vlti,vlti_station
;	   npoifile,pathname,specname,finddir,safe,breve,stringof,addline
; Block 4: set_boxes,set_points,get_ellipse,set_ellipse,set_region
;
;************************************************************************Block 1
function fakultaet,n
;
; Return factorial of n.
;
f=1L
for i=2,n do f=f*i
return,f
end
;-------------------------------------------------------------------------------
function combinations,n,i
;
; Return number of combinations with i samples taken from n elements.
;
c=1L
for j=n-i+1,n do c=c*j
return,c/fakultaet(i)
end
;-------------------------------------------------------------------------------
function base36,n
;
alfabet=['A','B','C','D','E','F','G','H','I','J','K','L','M', $
	 'N','O','P','Q','R','S','T','U','V','W','X','Y','Z']
symbols=[['0','1','2','3','4','5','6','7','8','9'],alfabet]
;
base=n_elements(symbols)
factor=alog10(double(base))
;
r=size(n(0))
if r(n_elements(r)-2) eq 7 then begin
n10=lonarr(n_elements(n))
for i=0,n_elements(n)-1 do begin
	word=n(i)
	lword=strlen(word)
	base_exp=strlen(word)
	while base_exp gt 0 do begin
		symbol=strmid(word,lword-base_exp,1)
		j=where(symbols eq symbol)
		n10(i)=n10(i)+j*base^(base_exp-1)
		base_exp=base_exp-1
	endwhile
endfor
return,n10
endif
;
m=n
n36=strarr(n_elements(m))
;
bk=base
repeat begin
	nbk=m mod bk
	n36=symbols(nbk/(bk/base))+n36
	bk=bk*base
	m=m-nbk
endrep until total(m) eq 0
;
return,n36
;
end
;-------------------------------------------------------------------------------
function base2,n
;
symbols=['0','1']
;
base=n_elements(symbols)
factor=alog10(double(base))
;
r=size(n(0))
if r(n_elements(r)-2) eq 7 then begin
n10=lonarr(n_elements(n))
for i=0,n_elements(n)-1 do begin
	word=n(i)
	lword=strlen(word)
	base_exp=strlen(word)
	while base_exp gt 0 do begin
		symbol=strmid(word,lword-base_exp,1)
		j=where(symbols eq symbol)
		n10(i)=n10(i)+j*base^(base_exp-1)
		base_exp=base_exp-1
	endwhile
endfor
return,n10
endif
;
m=n
n36=strarr(n_elements(m))
;
bk=base
repeat begin
	nbk=m mod bk
	n36=symbols(nbk/(bk/base))+n36
	bk=bk*base
	m=m-nbk
endrep until total(m) eq 0
;
return,n36
;
end
;-------------------------------------------------------------------------------
function eqp,m,n
;
; Return the probability of n counts for a Poisson process with mean m.
; Input n can be an array.
;
md=double(m)
nd=double(n)
y=dblarr(n_elements(n))
for i=0,n_elements(n)-1 do begin
	p=-md/max([nd(i),1.d0])
	if p lt -40 then a=0.d0 else a=exp(p)
	case n(i) of
      0:y(i)=a
   else:begin
	y(i)=md*a
	for k=2,n(i) do y(i)=y(i)*md/double(k)*a
	end
	endcase
endfor
;
return,y
;
end
;-------------------------------------------------------------------------------
function produkt,values
;
; Return the product of all values.
;
p=1L
for i=0,n_elements(values)-1 do p=p*values(i)
return,p
end
;-------------------------------------------------------------------------------
function kreuzprodukt,x,y,normal=normal
;
if n_elements(normal) eq 0 then normal=0 else normal=normal gt 0
v=[x(1)*y(2)-x(2)*y(1),x(2)*y(0)-x(0)*y(2),x(0)*y(1)-x(1)*y(0)]
if normal then v=v/sqrt(total(v^2))
return,v
;
end
;-------------------------------------------------------------------------------
function sinc,x_in
;
x=x_in
index=where(x eq 0,count)
if count gt 0 then x(index)=1
y=sin(!pi*x)/(!pi*x)
if count gt 0 then y(index)=1
return,y
;
end
;-------------------------------------------------------------------------------
function quasol,c
; 
; Compute the two solutions of the quadratic equation c+bx+ax^2=0, where
; [c,b,a]=c[0:2].
;
d=sqrt(c(1)^2-4*c(2)*c(0))
s1=(-c(1)+d)/(2*c(2))
s2=(-c(1)-d)/(2*c(2))
;
return,[s1,s2]
;
end
;-------------------------------------------------------------------------------
function winkel,ra1,dec1,ra2,dec2
;
; Given two positions on the sky, return the angle between them in degrees.
;
rad=180/!pi
ra1rad=ra1*15/rad
ra2rad=ra2*15/rad
dec1rad=dec1/rad
dec2rad=dec2/rad
;
return,acos(cos(dec1rad)*cos(dec2rad)*cos(ra1rad-ra2rad) $
		+sin(dec1rad)*sin(dec2rad))*rad
;
end
;-------------------------------------------------------------------------------
function ellipse,sma,smb,pa
;
; Return radius of point on ellipse at position angle pa. Origin is at
; center of ellipse; radius is largest at p=0. pa in radians. a and b 
; are the semi-major and minor axes.
;
a=float(sma)
b=float(smb)
p=float(pa)
;
esqr=1-(b/a)^2
return,sqrt(b^2/(1-esqr*cos(p)^2))
;
end
;-------------------------------------------------------------------------------
function apollonius,sma,smb,pa,sfactor
;
; pa is the position angle (positive sense) in radians.
;
a=float(sma)
b=float(smb)
p=float(pa)
factor=float(sfactor)
;
q=0
if p gt !pi/2 and p le !pi then begin
        q=1
        p=!pi-p
endif else if p gt !pi then begin
        q=2
        p=p-!pi
endif else if p lt 0 and p ge -!pi/2 then begin
        q=3
        p=-p
endif else if p lt -!pi/2 then begin
        q=4
        p=p+!pi
endif
;
a1=ellipse(a,b,!pi/2-p)
b1=sqrt(a^2+b^2-a1^2)
;
x1=b1*sqrt(1-((a*b)/(a1*b1))^2)
y1=((a*b)/(a1*b1))*b1
;
x2=x1
y2=y1*factor
;
a2=a1
b2=sqrt(x2^2+y2^2)
;
ab=a2*b2*(y2/b2)
asqplusbsq=a2^2+b2^2
;
af=sqrt((asqplusbsq+sqrt(asqplusbsq^2-4*ab^2))/2)
bf=sqrt((asqplusbsq-sqrt(asqplusbsq^2-4*ab^2))/2)
;
if x2 ne 0 then begin
        tanalphaplusbeta=y2/x2
        tanalphatimestanbeta=bf^2/af^2
        t1=tanalphaplusbeta*(1-tanalphatimestanbeta)
        t2=tanalphatimestanbeta
        alpha=atan((t1+sqrt(t1^2-4*t2))/2)
        beta =atan((t1-sqrt(t1^2-4*t2))/2)
endif else begin
        alpha=!pi/2
        beta =0
endelse
;
if factor lt 1 then paf=!pi/2-beta else paf=!pi/2-alpha
;
case q of
        0:paf=paf
        1:paf=!pi-paf
        2:paf=paf+!pi
        3:paf=-paf
        4:paf=paf-!pi
endcase
;
return,[af,bf,paf]
;
end
;-------------------------------------------------------------------------------
function summe,x,dim,sdev=sdev
;
; This function sums all elements along dimension dim (first is 0) and
; returns the remainder of the matrix. It is identical to total() in IDL,
; but should be used instead of total() under PV-WAVE since that 
; implementation does not have the dim parameter.
;
; If the input array is complex, the standard deviations (sdev) are 
; computed as the standard deviations of the radius and phase*radius,
; i.e. as the major and minor axes of the uncertainty ellipse after 
; rotation by the mean phase angle to phase=0. 
;
forward_function cphase
;
i_complex=sqrt(complex(-1,0))
;
result=size(x)
if result(n_elements(result)-2) eq 6 then c=1 else c=0
if n_elements(dim) eq 0 then begin
	xt=total(x)
	if n_elements(x) ge 2 then begin
	if c then begin
		xr=x*exp(-i_complex*cphase(xt))
		sdev=complex(stdev(float(xr)),stdev(imaginary(xr)))
;		sdev=complex(stdev(float(xr)), $
;		             atan(stdev(imaginary(xr)),abs(xt)/n_elements(x)))
	endif else sdev=stdev(x)
	endif
	return,total(x)
endif
;
n=result(1+dim)
if dim eq 0 then k=1 else k=produkt(result(1:dim))
l=k*n
result(dim+1)=1
xt=reform(make_array(size=result))
sdev=xt
m=n_elements(xt)
for i=0L,m-1 do begin
	index=lindgen(n)*k+(i mod k)+(i/k)*l
	xt(i)=total(x(index))
	if n ge 2 then begin
	if c then begin
		xr=x(index)*exp(-i_complex*cphase(xt(i)))
		sdev(i)=complex(stdev(float(xr)),stdev(imaginary(xr)))
;		sdev(i)=complex(stdev(float(xr)), $
;			 atan(stdev(imaginary(xr)),abs(xt(i))/n))
	endif else sdev(i)=stdev(x(index))
	endif
endfor
;
return,xt
;
end
;-------------------------------------------------------------------------------
function add,x,n,dim,sdev=sdev
;
; Sum every n samples along dimension dim (first is 0) of a matrix x.
; If the requested dimension is larger than the number of dimensions
; of x, it defaults to the last dimension of x.
; 
if n_elements(dim) eq 0 then dim=0
;
if n eq 1 then return,x
;
r=size(x)
dim=dim < (r(0)-1)
k=r(1+dim)/n
r0=r
r0(1+dim)=k
result=make_array(size=r0)
sdev=make_array(size=r0)
;
r1=r
r1(1+dim)=1
m=produkt(r1(1:r(0)))
if dim eq 0 then m1=1 else m1=produkt(r1(1:dim))
m2=m1*k
index=(lindgen(m)/m1)*m2+(lindgen(m) mod m1)
;
r2=r
r2(1+dim)=n
m2=produkt(r(1:dim+1))
jndex=(lindgen(m*n)/(m1*n))*m2+(lindgen(m*n) mod (m1*n))
;
for i=0L,k-1 do begin
	result(index+i*m1)=summe(reform(x(jndex+i*m1*n),r2(1:r(0))),dim,sdev=sd)
	sdev(index+i*m1)=sd
endfor
;
return,result
;
end
;-------------------------------------------------------------------------------
function box,x,j,dim
;
; Smooth values along dimension dim (first is 0) of a matrix x, over j samples.
;
if n_elements(dim) eq 0 then dim=0
;
if j eq 1 then return,x
;
result=size(x)
;
; n is the number of elements along dim
n=result(1+dim)			
;
; k is the number of elements in each of the n sub-matrices 
if dim eq 0 then k=1 else k=produkt(result(1:dim))
;
; l is the number of elements processed at a time
l=k*n
;
; Make output array
result=reform(make_array(size=result))
;
; m is the number of elements not including dim
m=n_elements(result)/n
;
for i=0L,m-1 do begin
	index=lindgen(n)*k+(i mod k)+(i/k)*l
	if j eq n_elements(index) $
		then result(index)=mean(x(index)) $
		else result(index)=smooth(x(index),j,/edge_truncate)
endfor
;
return,result
;
end
;-------------------------------------------------------------------------------
function wmean,y,e,me,chisq
;
w=1/e^2
tw=total(w)
ym=total(y*w)/tw
me=sqrt(1/tw)
chisq=total((y-ym)^2*w)/n_elements(y)
;
return,ym
;
end
;-------------------------------------------------------------------------------
function medianve,y,e
;
; Return median and error (+/-e contains 2/3 of values)
; For a Gaussian distribution, e=sigma (sigma is the standard dev.)
;
n=n_elements(y)
if n eq 0 then return,0
si=sort(y)
ys=y(si)
if n mod 2 eq 1 then v=ys((n-1)/2) else v=(ys(n/2)+ys(n/2-1))/2
if n gt 3 then e=(ys(n/2+n/3)-ys(n/2-n/3))/2
;
return,v
end
;-------------------------------------------------------------------------------
function gavg,values
;
; Return a geometric mean of all values
;
index=where(values eq 0,count)
if count gt 0 then return,0
index=where(values lt 0,count)
if count gt 0 then print,'Warning(GAVG): negative values, use absolut instead!'
return,exp(avg(alog(abs(values))))
end
;-------------------------------------------------------------------------------
function ravg1,y,w
;
; Return running average. Initialize with non-zero value of width (int).
; After initialization, w will be zero and the functions returns the
; average of the last w values of y, including the current value of y.
;
common LocalRavg1,width,num,buffer,current
;
if n_elements(w) eq 0 then w=0
;
if w gt 0 then begin
	width=w
	buffer=fltarr(width)
	num=0
	current=0.0
	w=0
endif
;
if num eq width then begin
	current=current-buffer(0)
	buffer=shift(buffer,-1)
endif
num=min([width,num+1])
current=current+y
buffer(num-1)=y
return,current/num
;
end
;-------------------------------------------------------------------------------
function ravg2,y,w
;
; Return running average.
;
common LocalRavg2,width,num,buffer,current
;
if n_elements(w) eq 0 then w=0
;
if w gt 0 then begin
	width=w
	buffer=fltarr(width)
	num=0
	current=0.0
	w=0
endif
;
if num eq width then begin
	current=current-buffer(0)
	buffer=shift(buffer,-1)
endif
num=min([width,num+1])
current=current+y
buffer(num-1)=y
return,current/num
;
end
;-------------------------------------------------------------------------------
function gsmooth,t,y,s,x,c8=c8
;
; Gaussian smoothing of y(t), with FWHM=s. Return result itself, or result
; interpolated on grid x.
;
if n_elements(c8) eq 0 then c8=0
;
if c8 then begin
	t8=double(t)
	y8=double(y)
	s8=double(s)
	n=n_elements(y8)
	r8=dblarr(n)
	status=linknload(!external_lib,'gsmooth8',t8,y8,s8,n,r8)
	r=float(r8)
endif else begin
	r=y
	a=sqrt(10)*s
	for i=0l,n_elements(t)-1 do begin
		dt=abs(t-t(i))
		index=where(dt lt a)
		w=exp(-(dt(index)/s)^2)
		r(i)=total(y(index)*w)/total(w)
	endfor
endelse
;
if n_elements(x) ne 0 then return,interpol(r,t,x) else return,r
;
end
;-------------------------------------------------------------------------------
function sockel,map_in
;
; Away from the source, the mean map level should drop to zero. Otherwise,
; the Fourier transform would be normalized incorrectly, and this function
; determined the mean baseline from looking at the means in annuli around
; the map center.
;
map=map_in
;
n=n_elements(map)
;
imsize=n_elements(map(*,0))
jmsize=n_elements(map(0,*))
;
a=float(imsize)/2
b=float(jmsize)/2
;
; Where is the source?
k=where(map eq max(map))
imax=k mod imsize
jmax=k  /  jmsize
;
; We use the map center for now
imax=imsize/2
jmax=jmsize/2
;
x=(findgen(n) mod imsize) - imax
y=(findgen(n)  /  imsize) - jmax
;
; Have the radius trace out an ellipse
r=sqrt((x/a)^2+(y/b)^2)
;
maxr=max(r)
nring=20
rstep=maxr/nring
level=fltarr(nring)
ratio=fltarr(nring)+1
;
; Find the radius outside of which the mean level stays constant
for i=0,nring-1 do begin
;
	index=where(r gt i*rstep and r lt (i+1)*rstep)
	level(i)=mean(map(index))/max(map)	; % levels
	if i gt 0 then ratio(i)=abs(level(i-1)-level(i))
;
endfor
;
i=where(ratio lt 0.01 and level lt 0.1,count)
;
if count gt 0 then return,mean(map(where(r gt i(0)*rstep))) $
	      else return,0
;
if count gt 0 then begin
	map(where(r le i(0)*rstep))=mean(map(where(r gt i(0)*rstep and r lt (i(0)+1)*rstep)))
	return,mean(map)
endif else return,0
;
end
;-------------------------------------------------------------------------------
function signof,x,y
;
if n_elements(y) eq 0 then return,(x ge 0)*2-1 $
		      else return,(x ge 0 and y ge 0)*2-1
end
;-------------------------------------------------------------------------------
function nonzero,x
;
index=where(float(x) eq 0.0,count)
if count gt 0 then x(index)=1
;
return,x
;
end
;-------------------------------------------------------------------------------
function cphase,c
;
; Return the phase of a complex variable
;
cr=float(c)
ci=imaginary(c)
index=where(cr eq 0 and ci eq 0,count)
p=atan(imaginary(c),float(c))
if count gt 0 then p(index)=0
return,p
;
end
;-------------------------------------------------------------------------------
function cmag,m1,m2
;
; Compute the combined magnitude of m1 and m2.
;
f1=fltarr(n_elements(m1))
index=where(m1 ne +100,count)
if count gt 0 then f1(index)=10.^(-m1(index)/2.5)
;
f2=fltarr(n_elements(m2))
index=where(m2 ne +100,count)
if count gt 0 then f2(index)=10.^(-m2(index)/2.5)
;
f=f1+f2
mag=fltarr(n_elements(f))+100
index=where(f ne 0,count)
if count gt 0 then mag(index)=-2.5*alog10(f(index))
if count eq 1 then return,mag(0) else return,mag
;
end
;-------------------------------------------------------------------------------
function dmag,mt,dm
;
; Compute m1 and m2 from the total magnitude and the magnitude difference.
;
ft=fltarr(n_elements(mt))
index=where(mt ne +100,count)
if count gt 0 then ft(index)=10.^(-mt(index)/2.5)
;
fr=fltarr(n_elements(dm))
index=where(dm ne +100,count)
if count gt 0 then fr(index)=10.^(-dm(index)/2.5)
;
m1=fltarr(n_elements(mt))+100
m2=fltarr(n_elements(mt))+100
index=where(fr ne 0,count)
if count gt 0 then begin
	f2=(ft*fr)/(1+fr)
	f1=f2/fr
	m1(index)=-2.5*alog10(f1)
	m2(index)=-2.5*alog10(f2)
endif
return,transpose([[m1],[m2]])
;
end
;-------------------------------------------------------------------------------
function omag,mt,m2
;
; Compute the combined magnitude of m1 and m2.
;
ft=fltarr(n_elements(mt))
index=where(mt ne +100,count)
if count gt 0 then ft(index)=10.^(-mt(index)/2.5)
;
f2=fltarr(n_elements(m2))
index=where(m2 ne +100,count)
if count gt 0 then f2(index)=10.^(-m2(index)/2.5)
;
f1=ft-f2
mag=fltarr(n_elements(f1))+100
index=where(f1 ne 0,count)
if count gt 0 then mag(index)=-2.5*alog10(f1(index))
if count eq 1 then return,mag(0) else return,mag
;
end
;-------------------------------------------------------------------------------
function v2gd,uv,v2
;
; This function returns for a given uv spacing (in millions of wavelengths)
; and calibrated squared visibility amplitude the diameter of the Gaussian.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=pi_circle/(180.*3.6)
;
return,sqrt(-alog(sqrt((v2 > 0.01) < 0.99))/0.3606737602) $
	/((uv > 0.1)*factor*pi_circle)
;
end
;-------------------------------------------------------------------------------
function hb,uv,v2
;
; This function returns for a given uv spacing (in millions of wavelengths)
; and calibrated squared visibility amplitude the diameter of the star [mas]. 
; The computation is non-iterative, since it uses polynomial fit coefficients
; determined for a grid of spacings and visibilities. The function works
; only reliably if v2 > 0.1 and v2 < 0.9.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
if n_elements(uv) eq 1 then begin
	u=v2
	u(*)=uv(0)
	uv=u
endif
;
; These coefficients are for a 3rd order fit of the d=f(v2)
; coefficients to the uv spacings.
coeff11=[10.8555,-1.52880,0.0975650,-0.00215141]
coeff21=[3.37067,-0.0641237,0.000618756,-2.44172e-06]
coeff12=[-12.2294,1.49343,-0.0793799,0.00141445]
coeff22=[-3.75824,0.0642599,-0.000620554,2.45088e-06]
coeff13=[11.6333,-1.21883,0.0499391,-0.000526415]
coeff23=[3.73860,-0.0645361,0.000624810,-2.47455e-06]
coeff14=[10.3883,-1.14758,0.0501068,-0.000630298]
coeff24=[3.23568,-0.0646144,0.000626390,-2.48441e-06]

;
; Get polynomial coefficients for given uv distance. There are two sets
; because of a discontinuity at 18 Mlambda.
d=fltarr(n_elements(uv))
;
for i=0,n_elements(uv)-1 do begin
if uv(i) le 18 then r=exp([poly(uv(i),coeff11),-poly(uv(i),coeff12), $
                        poly(uv(i),coeff13),poly(uv(i),coeff14)]) $
               else r=exp([poly(uv(i),coeff21),-poly(uv(i),coeff22), $
                        poly(uv(i),coeff23),poly(uv(i),coeff24)])
;
; Get diameter for given uv and v2
d(i)=poly(v2(i),r*[1,-1,1,-1])
endfor
;
return,d
;
; Test
uvdist=uv*1e6
arg=factor*uvdist*d
visamp=2*beselj(arg,1)/arg
vissq=visamp^2
print,'V2 should be ',vissq
;
end
;-------------------------------------------------------------------------------
function viscalerror,d,de,bl=bl,wl=wl
;
; Computes the error in the visibility caused by an error in the 
; calibrator diameter. bl [m], wl [m]
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
factor=(pi_circle^2)/(180L*3600L*1000L)
;
if n_elements(bl) eq 0 then bl=100.0  	; m
if n_elements(wl) eq 0 then wl=10e-6    ; m
uvdist=bl/wl
n=n_elements(d)
UD=d
arg=factor*uvdist*(UD-de)
visamp=2*beselj(arg,1)/arg
vissql=visamp^2
arg=factor*uvdist*(UD+de)
visamp=2*beselj(arg,1)/arg
vissqh=visamp^2
return,(abs(vissqh-vissql)/((vissqh+vissql)/2))*100
;
end
;-------------------------------------------------------------------------------
function bessjy8,arg,arg_nu
;
; Double-precision fractional bessel function.
;
;
x=double(arg)
x_nu=double(arg_nu)
n=n_elements(x)
rj=x
status=linknload(!external_lib,'bessjyidl',n,x,x_nu,rj)
;
return,rj
;
end
;-------------------------------------------------------------------------------
function spline8,x,y,t
;
x=double(x)
y=double(y)
y2=y
t=double(t)
r=t
n=n_elements(x)
m=n_elements(t)
;
status=linknload(!external_lib,'splineint',x,y,y2,t,r,n,m)
;
return,r
;
end
;-------------------------------------------------------------------------------
function evalfunction,term,x,x_mid,x_scl
;
forward_function edlen,silica,amber
;
; This is the one and only function to evaluate mathematical base functions
; used for calibration and fitting of data. See create_calentries for 
; initialization of function terms.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(x_mid) eq 0 then x_mid=0.0
if n_elements(x_scl) eq 0 then x_scl=1.0
;
xx=(x-x_mid)*x_scl
case term of
;	Polynomial
	'1'  : 	y=xx^0
	'x'  : 	y=xx
	'x^2': 	y=xx^2 
	'x^3': 	y=xx^3 
	'x^4': 	y=xx^4
	'x^5': 	y=xx^5
	'x^6': 	y=xx^6
;	Legendre polynomial
        'L_0': 	y=xx^0
        'L_1': 	y=xx
        'L_2': 	y=(3*xx^2-1)/2
        'L_3': 	y=(5*xx^3-3*xx)/2
        'L_4': 	y=(35*xx^4-30*xx^2+3)/8
        'L_5': 	y=(63*xx^5-70*xx^3+15*xx)/8
        'L_6': 	y=(231*xx^6-315*xx^4+105*xx^2-5)/16
;	Phase change [deg] due to silica and air path mismatches [microns]
	'P_0':  y=x^0	; This term only necessary if phases negative
	'P_1': 	begin
		k=360/x	; x must be wavelength[mu]
;		The refractive index of silica
		ns=silica(x)
		y=k*ns
		end
	'P_2': 	begin
		k=360/x	; x must be wavelength[mu]
;		The refractive index of air at Paranal
		na=edlen(x)
		y=k*na
		end
	 else: 	begin
	       	print,'***Error(EVALFUNCTION): unknown function: ',term,'!'
	       	return,-1
	       	end
endcase
return,y
;
end
;-------------------------------------------------------------------------------
function polynom,x,r
;
; Similar to IDL's poly, but works with our better polyfit procedure.
;
num=n_elements(x)
degree=n_elements(r)-3
s=r(0:degree)
x_mid=r(degree+1)
x_scl=r(degree+2)
;
terms=['L_0','L_1','L_2','L_3','L_4','L_5','L_6']
terms=['1','x','x^2','x^3','x^4','x^5','x^6']
m=dblarr(num,degree+1)
for i=0,degree do m(*,i)=evalfunction(terms(i),x,x_mid,x_scl)
;
return,m#s
;
end
;-------------------------------------------------------------------------------
function polyfit,x_in,y_in,degree,yft
;
; A better polynomial fit routine than IDL's own poly_fit.
; Arrays x and y are shifted to improve numerical stability.
;
if degree gt 6 then begin
	print,'***Error(POLYFIT): polynomial order too high!'
	return,0
endif
num=n_elements(x_in)
if num eq 0 then begin
	print,'***Error(POLYFIT): x undefined!'
	return,0
endif
if num ne n_elements(y_in) then begin
	print,'***Error(POLYFIT): number of elements in x and y not equal!'
	return,0
endif
;
; Make sure to remove leading empty dimensions and convert to double
x=double(reform(x_in,num))
y=double(reform(y_in,num))
;
x_mid=(min(x)+max(x))/2
x_scl=2/(max(x)-min(x))
terms=['L_0','L_1','L_2','L_3','L_4','L_5','L_6']
terms=['1','x','x^2','x^3','x^4','x^5','x^6']
m=dblarr(num,degree+1)
for i=0,degree do m(*,i)=evalfunction(terms(i),x,x_mid,x_scl)
tm=transpose(m)
n=tm#m
r=tm#y
s=invert(n)#r
r=[s,[x_mid,x_scl]]
;
yft=polynom(x,r)
return,r
;
end
;-------------------------------------------------------------------------------
function polyres,x_in,y_in,degree
;
r=polyfit(x_in,y_in,degree,yfit)
return,y_in-yfit
;
end
;-------------------------------------------------------------------------------
function perinom,x,r
;
num=n_elements(x)
m=dblarr(num,4,/nozero)
m(*,0)=1
m(*,1)=x
m(*,2)=sin(x)
m(*,3)=cos(x)
return,m#r
;
end
;-------------------------------------------------------------------------------
function perifit,x_in,y_in,period,yft
;
if n_elements(period) eq 0 then begin
	print,'***Error(PERIFIT): period not defined!'
	return,0
endif
num=n_elements(x_in)
if num eq 0 then begin
	print,'***Error(PERIFIT): x undefined!'
	return,0
endif
if num ne n_elements(y_in) then begin
	print,'***Error(PERIFIT): number of elements in x and y not equal!'
	return,0
endif
;
; Make sure to remove leading empty dimensions and convert to double
x=double(reform(x_in,num))/period*2*!pi
y=double(reform(y_in,num))
;
m=dblarr(num,4,/nozero)
m(*,0)=1
m(*,1)=x
m(*,2)=sin(x)
m(*,3)=cos(x)
;
tm=transpose(m)
n=tm#m & m=0
r=tm#y & tm=0
svd8,n,w,u,v
tol=1e-13
small=where(w lt max(w)*tol,count)
print,'SVD: will edit',count,' singular values!'
if count gt 0 then w(small)=0
svb8,u,w,v,r,s
; s=invert(n)#r
;
yft=perinom(x,s)
return,s
;
end
;-------------------------------------------------------------------------------
function perires,x,y,period
;
r=perifit(x,y,period,yfit)
return,y-yfit
;
end
;-------------------------------------------------------------------------------
function sizeof,x
;
y=x(0)
;
r=size(y) & n=n_elements(r)
;
if r(n-2) eq 0 then return,0
if r(n-2) ne 8 then y={y:y}
;
tbytes=0L
;
for i=0,n_tags(y)-1 do begin
	r=size(y.(i)) & n=n_elements(r)
	case r(n-2) of
		1:ibytes=1*r(n-1)
		2:ibytes=2*r(n-1)
		3:ibytes=4*r(n-1)
		4:ibytes=4*r(n-1)
		5:ibytes=8*r(n-1)
		6:ibytes=8*r(n-1)
		7:ibytes=1*long(total(strlen(y.(i))))
		8:ibytes=sizeof(y.(i))
		9:ibytes=16*r(n-1)
	       12:ibytes=2*r(n-1)
	       13:ibytes=4*r(n-1)
	       14:ibytes=8*r(n-1)
	       15:ibytes=8*r(n-1)
	endcase
	tbytes=tbytes+ibytes
endfor
;
return,tbytes*n_elements(x)
;
end
;-------------------------------------------------------------------------------
function natural,x,y
;
return,transpose([[reform(x)],[reform(y)]])
;
end
;-------------------------------------------------------------------------------
function contiguous,index_in,increment
;
if n_elements(increment) eq 0 then increment=1
;
index=index_in
count=n_elements(index)
if count eq 1 then return,index
gaps=index(1:count-1)-index(0:count-2)
gapi=where(gaps gt 1,ngaps)
if ngaps eq count-1 then return,-1
if ngaps ge 1 then begin
	edge_gap=gapi(0) eq 0
	while edge_gap do begin
		index=index(1:count-1)
		count=count-1
		gaps=index(1:count-1)-index(0:count-2)
		gapi=where(gaps gt 1,ngaps)
		if ngaps eq 0 then edge_gap=0 $
			      else edge_gap=gapi(0) eq 0
	endwhile
	if ngaps gt 0 then begin
	edge_gap=gapi(ngaps-1) eq count-2
	while edge_gap do begin
		index=index(0:count-2)
		count=count-1
		gaps=index(1:count-1)-index(0:count-2)
		gapi=where(gaps gt 1,ngaps)
		if ngaps eq 0 then edge_gap=0 $
			      else edge_gap=gapi(ngaps-1) eq count-2
	endwhile
	endif
	if ngaps gt 0 then begin
		blki=[-1,gapi,count-1]
		nblk=ngaps+1
		blkl=blki(1:nblk)-blki(0:nblk-1)
		iblk=where(blkl eq max(blkl)) & iblk=iblk(0)
		index=index(blki(iblk)+1:blki(iblk)+blkl(iblk))
	endif
endif
return,index
;
end
;-------------------------------------------------------------------------------
function rankindex,i
;
; i must be positive integer.
;
index=where(i lt 0,count)
jndex=where(i ge 0)
if count gt 0 then i(index)=min(i(jndex))
s=unique(i)
index=intarr(max(s)+1)
for j=0,n_elements(s)-1 do index(s(j))=j
;
return,index
;
end
;-------------------------------------------------------------------------------
function whereindex,i0,x
;
; Return the indices for each dimension of array x corresponding to element i0.
;
m=n_elements(i0)
r=size(x)
index=intarr(r(0),m)
;
for n=0,m-1 do begin
;
i=i0(n)
for k=r(0)-1,1,-1 do begin
	p=produkt(r(1:k))
	index(k,n)=fix(i/p)
	i=i-index(k,n)*p
endfor
index(0,n)=i
;
endfor
;
return,index
;
end	
;-------------------------------------------------------------------------------
function whereequal,v1,v2
;
; Return indices into v1 where v1(index) equals one of v2
;
index=lonarr(n_elements(v1))-1
for i=0L,n_elements(v2)-1 do begin
	j=where(v1 eq v2(i),count)
	if count gt 0 then index(j)=j
endfor
;
j=where(index ge 0,count)
if count eq 0 then index=-1 else index=index(j)
return,index
;
end
;************************************************************************Block 2
function poweroftwo,series
;
; Reduce to power of 2
;
n=2L^nint(alog10(n_elements(series))/alog10(2))
if n gt n_elements(series) then n=n/2
if not !quiet then print,'Number of samples used: ',n
return,series(0:n-1)
;
end
;-------------------------------------------------------------------------------
function powerpatch,t,s,t_new,limit=limit
;
; t must be in ms, type long.
;
if not keyword_set(limit) then limit=100
;
n=n_elements(t)
dt=t(1:n-1)-t(0:n-2)
t_int=long(median(dt))
print,'Sampling interval seems to be ',t_int,' ms'
index=where(dt ne t_int,icount)
print,'Found ',icount,' gaps in time series'
if icount gt 0 then begin
	dtg=t(index+1)-t(index)
	jndex=where(dtg/t_int gt limit,jcount)
	if jcount gt 0 then begin
		print,'Gap(s) too large [s]: ', $
			(t(index(jndex+1))-t(index(jndex)))/1000.0
		kndex=[0-1,index(jndex),n-1]
	endif else kndex=[0-1,n-1]
	nk=n_elements(kndex)
	n1=kndex(0:nk-2)
	n2=kndex(1:nk-1)
	dts=t(n2)-t(n1+1)
	print,'Fixable section(s) [s]: ',dts/1000.0
	i=where(dts eq max(dts)) & i=i(0)
	n1=n1(i)+1
	n2=n2(i)
endif else begin
	n1=0
	n2=n-1
endelse
t_new=t(n1)+(lindgen((t(n2)-t(n1))/t_int+1)*t_int)
return,interpol(s,t,t_new)
;
if count eq 0 then begin
	t_new=t
endif else begin
	n_new=n
	for i=0,count-1 do begin
		dt=t(index(i)+1)-t(index(i))
		num=dt/t_int
		if num lt limit then n_new=n_new+num-1
	endfor
	t_new=lonarr(n_new,/nozero)
	t_new(0:index(0))=t(0:index(0))
	k=index(0)+1
endelse
for i=0,count-1 do begin
	dt=t(index(i)+1)-t(index(i))
	num=dt/t_int
	if num lt limit then begin
		t_new(k:k+num-2)=(lindgen(num-1)+1)*t_int+t(index(i))
		k=k+num-1
	endif
	if i lt count-1 then m=index(i+1) else m=n-1
	t_new(k:k+m-index(i)-1)=t(index(i)+1:m)
	k=k+m-index(i)
;
end
;
n=n_elements(t_new)
dt=t_new(1:n-1)-t_new(0:n-2)
index=where(dt ne t_int,count)
print,'Gaps remaining after patching: ',count,'.'
if count gt 0 then begin
	index=[-1,index,n-1]
	l=index(1:count+1)-index(0:count)
	j=where(l eq max(l))
	t_new=t_new(index(j(0))+1:index(j(0)+1))
endif
;
return,interpol(s,t,t_new)
;
end
;-------------------------------------------------------------------------------
function powerspectrum,series,sampling_int,f,p,f_avg,p_avg
;
; Input series must have power-of-two number of elements.
; Compute power spectrum and return square-root of total
; power, which is equal to the standard deviation of the 
; series.
;
; Compute frequencies and the Fourier transform
;
n=n_elements(series)
m=lindgen(n)
f=m/(n*sampling_int)
p=fft(series,-1)
;
; Frequencies f above the Nyquist frequency are not physically meaningful
; The frequency interval is 1/(n*sampling_int), Nyquist frequency is 
; n/2*frequency_int=1/(2*sampling_int)=1/2 * sampling_rate. In other words,
; the power spectrum is only representative of the series up to the Nyquist
; frequency.
;
n=n/2
;
; We want units in power/Hz. Since there are n frequencies, and the sum
; over n of the power spectrum is normalized, but the frequency range is
; from 0 to (1/sampling_int)/2, the ratio of the two must be multiplied
; with the power.
; 
norm_factor=n/((1/sampling_int)/2)
p=norm_factor*2*abs(p(0:n))^2
f=f(0:n)
;
; Avoid zero power situations...
r=machar() & tiny=r.xmin
p=p>tiny
;
; Average power in constant logarithmic intervals
num=long(alog10(n)*400)
int=alog10(double(n))/(num-1)
index=nint(10^(dindgen(num)*int))
jndex=uniq(index)
num=n_elements(jndex)-1
f_avg=dblarr(num)
p_avg=f_avg
for i=0,n_elements(jndex)-2 do begin
	f_avg(i)=10^avg(alog10(f(index(jndex(i)):index(jndex(i+1))-1)))
	p_avg(i)=10^avg(alog10(p(index(jndex(i)):index(jndex(i+1))-1)))
endfor
;
; The total power (sum over all elements of p, without norm_factor)
; is equal to the variance of the series. The factor 2 (see above)
; adjusts for the fact that our sum here is only over one side of
; the FFT.
;  
return,sqrt(total(p/norm_factor))
;
;
end
;-------------------------------------------------------------------------------
function siggen,power,wavenumber_int
;
; Wavenumber_int is array [min,max] in 1/microns
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
n=n_elements(power)
w=dindgen(n)*(wavenumber_int(1)-wavenumber_int(0))/(n-1)+wavenumber_int(0)
f=c_light*w*1d6
;
p=[power,shift(reverse(power),1)]
s=complex(sqrt(p))
signal=fft(s,1)
;
return,signal
;
end
;-------------------------------------------------------------------------------
function siggen,pwr,fmax
;
; Return signal whose power spectrum is equal to pwr.
; pwr is given for frequencies from zero to fmax [Hz] in equal increments.
;
common LocalSiggen,seed
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
n=n_elements(pwr)-1
sampling_int=1/(2*fmax)
norm_factor=n/((1/sampling_int)/2)
;
p=sqrt(pwr/(2*norm_factor))*exp(2*pi_circle*i_complex*randomu(seed,n+1))
p=[p,conj(reverse(p(1:n-1)))]
s=float(fft(p,1))
return,s
;
end
;-------------------------------------------------------------------------------
pro strucfunc,series,sampling_int,dt,sf,acceleration=factor
;
if not keyword_set(factor) then factor=1.0 else factor=float(factor)
;
n=n_elements(series)
sf=dblarr(n-1)
dt=(dindgen(n-1)+1)*sampling_int
;
num_max=n/factor
;
for i=1l,n-1 do begin
;	series_shift=shift(series,-i)
;	sf(i-1)=avg((series_shift(0:n-1-i)-series(0:n-1-i))^2)
;	sf(i-1)=avg((series(i:n-1)-series(0:n-1-i))^2)
	factor=ceil((n-i)/num_max)
	index=lindgen((n-i)/factor)*factor
	sf(i-1)=avg((series(index+i)-series(index))^2)
endfor
;
!p.multi=0
!p.charsize=1.5
;
plot,dt,sf,/xlog,/ylog, $
	xtitle='Interval [s]'
;
end
;-------------------------------------------------------------------------------
function signalfilter,signal,filter
;
; Apply a filter to signal.
; The frequencies are returned by fft in this order:
; DC,min,max,-min	(even number of elements n in signal)
; DC,min,max,-max,-min	(odd number of elements n in signal)
;
; Filter should have n/2+1 elements for both even and odd cases.
;
; Compute power spectrum
p=fft(signal,-1)
n=n_elements(signal)
if n mod 2 eq 0 then begin
	p(0:n/2)=p(0:n/2)*filter(0:n/2)
	p(n/2+1:n-1)=p(n/2+1:n-1)*reverse(filter(1:n/2-1))
endif else begin
	p(0:n/2)=p(0:n/2)*filter(0:n/2)
	p(n/2+1:n-1)=p(n/2+1:n-1)*reverse(filter(1:n/2))
;	p(0:n-1)=p(0:n-1)*filter
;	p(n:n*2-1)=p(n:n*2-1)*reverse(filter)
endelse
return,float(fft(p,1))
;
end
;-------------------------------------------------------------------------------
function highpass,signal
;
n=n_elements(signal)
filter=fltarr(n/2+1)+1
filter(0:(n/2)/100)=0
return,signalfilter(signal,filter)
;
end
;-------------------------------------------------------------------------------
function lowpass,signal
;
n=n_elements(signal)
filter=fltarr(n/2+1)
filter(0:(n/2)/100)=1
return,signalfilter(signal,filter)
;
end
;-------------------------------------------------------------------------------
function notch,signal
;
n=n_elements(signal)
filter=fltarr(n/2+1)+1
p=fft(signal,-1)
p=abs(p(0:n/2))
p(0)=0
v=medianve(p,e)
index=where(p gt v+3*e,count)
if count gt 0 then begin
	notch_indices=contiguous(index)
	if notch_indices(0) ge 0 then $
	filter(notch_indices)=0
endif
return,signalfilter(signal,filter)
end
;-------------------------------------------------------------------------------
function blanks,n
;
blank=''
for i=1,n do blank=blank+' '
return,blank
;
end
;-------------------------------------------------------------------------------
function dashes,n
;
dash=''
for i=1,n do dash=dash+'-'
return,dash
;
end
;-------------------------------------------------------------------------------
function stringl,v,format=form
;
m=1024L
n=n_elements(v)
i=n/m
j=n mod m
;
if n le m then return,string(v,format=form)
;
words=strarr(n)
words(0:m-1)=string(v(0:m-1),format=form)
;
for k=1,i-1 do words(k*m:(k+1)*m-1)=string(v(k*m:(k+1)*m-1),format=form)
if j ne 0 then words(i*m:n-1)=string(v(i*m:n-1),format=form)
;
return,words
;
end
;-------------------------------------------------------------------------------
function myhak,message
;
if n_elements(message) eq 0 then $
	message='<Press Spacebar to continue, q to quit, ? for help>'
print,message,format='(a,$)'
again:
c=get_kbrd(1)
case c of
	'Q':    c='q'
	'H':    c='?'
	'h':    c='?'
	else:
endcase
case c of
'?':    begin
        print,'\x0D',$
              '------------------------------------------         '
        print,'<space>         Display next page of text.
        print,'<return>        Display next line of text.
        print,'q or Q          Quit
        print,'h, H, or ?      Display this message.
        print,'------------------------------------------         '
        goto,again
        end
else:   
endcase
;
return,c
;
end
;-------------------------------------------------------------------------------
function elements,ndim,dims
;
d=lonarr(ndim)+1 & d(ndim-1)=0
n=produkt(dims(0:ndim-1))
v=strarr(n)
for i=0,n-1 do begin
	d(ndim-1)=d(ndim-1)+1
	for j=ndim-1,0,-1 do begin
		if d(j) gt dims(j) then begin
			d(j-1)=d(j-1)+1
			d(j:ndim-1)=1
		endif
	endfor
	s=string(d)+[replicate(',',ndim-1),'']
	sc='' & for j=0,ndim-1 do sc=sc+s(j)
	v(i)=sc
end
;
return,strcompress(v,/remove_all)
;
end
;-------------------------------------------------------------------------------
function printindex,n_total,n_per_line,index_f,index_l
;
common LocalPrintIndex,iline,maxline,extraitems
;
if n_elements(index_l) eq 0 then index_l=-1
;
if index_l lt 0 then begin
	iline=0
	maxline=n_total/n_per_line
	extraitems=n_total mod n_per_line
endif
;
do_print=1
if iline lt maxline then begin
	index_f=iline*n_per_line
	index_l=(iline+1)*n_per_line-1
endif else if iline eq maxline and extraitems gt 0 then begin
	index_f=n_total-extraitems
	index_l=n_total-1
endif else begin
	do_print=0
endelse
iline=iline+1
;
return,do_print
;
end
;-------------------------------------------------------------------------------
function isnumeric,names,strict
;
; Test each element of an array of strings whether it represents a number
; or not. If strict is positive non-zero, do not even allow letters 'e', 'd',
; 'E', and 'D', which can be used in loating point numbers.
;
if n_elements(strict) eq 0 then strict=0 else strict=strict gt 0
;
result=intarr(n_elements(names))
;
for i=0,n_elements(names)-1 do begin
	ascii=byte(strcompress(names(i),/remove_all))
	index=where(ascii lt 43 or ascii gt 57 $
                 or ascii eq 44 or ascii eq 47,count)
	fp=0
	if count eq 0 then begin
		result(i)=1
	endif else if count eq 1 and not strict then begin
		case ascii(index(0)) of
			 68:fp=1
			 69:fp=1
			100:fp=1
			101:fp=1
		       else:fp=0
		endcase
		if fp eq 1 and index(0) ne 0 then result(i)=1
	endif
	jndex=where(ascii eq 43 or ascii eq 45,count)
	if count eq 2 then begin
		if fp eq 0 then result(i)=0 else begin
			if jndex(1) lt index(0) $
			or jndex(0) gt index(0) then result(i)=0
		endelse
	endif else if count gt 2 then result=0	
	index=where(ascii eq 46,count)
	if count gt 1 then result(i)=0
endfor
;
if n_elements(result) eq 1 then return,result(0) else return,result
;
end
;-------------------------------------------------------------------------------
function gitter,num_steps,value_min,value_max
;
; Return an equally spaced array of values centered on zero, i.e.
; beginning at -value_max and ending at +value_max.
;
if n_params() eq 2 then value_max=-value_min
value_step=(value_max-value_min)/(num_steps-1)
return,dindgen(num_steps)*value_step+value_min
;
end
;-------------------------------------------------------------------------------
function mergetable,table1,table2
;
; Merges two tables but does not remove duplicate entries.
;
table=replicate(table1(0),n_elements(table1)+n_elements(table2))
for i=0,n_tags(table)-1 do begin
	r=size(table.(i))
;	The following is necessary since the introduction of arrays in StarTable
	table.(i)=reform(reform([table1.(i),table2.(i)],r(1),r(0)))
endfor
return,table
;
end
;-------------------------------------------------------------------------------
function commonbeam,beams
;
ubeams=unique(beams)
counts=intarr(n_elements(ubeams))
for i=0,n_elements(ubeams)-1 do begin
	j=where(beams eq ubeams(i),count)
	counts(i)=count
endfor
i=where(counts eq max(counts))
return,ubeams(i(0))
;
end
;-------------------------------------------------------------------------------
function ppair,e
;
; Return partial air pressure [mb] for a given elevation above sea level [m]
; Note that hPa=mb!
;
; From a fit to the air pressure=alog(p)(t[C])
; http://www.engineeringtoolbox.com/air-altitude-pressure-d_462.html
coeffs=[4.6186077,-0.00011669363,-2.1845594e-09,3.8310355e-14]
;
return,exp(poly(e,coeffs))*10
;
end
;-------------------------------------------------------------------------------
function ppwv,t,h
;
; Given a temperature t[C] and relative humidity h[%], return the 
; partial water vapor pressure [mb].
; Note that hPa=mb!
;
; From a fit to the saturation water vapor pressure=alog(f)(t[C])
; http://www.engineeringtoolbox.com/relative-humidity-air-d_687.html
coeffs=[1.8092268,0.072935087,-0.00030139676,8.5339579e-07]
swvp=exp(poly(t,coeffs))
;
return,(h/100.)*swvp
;
end
;-------------------------------------------------------------------------------
function amber,lambda,p,t,f
;
; Return refractive index for air. lambda[mu], p[mb], t[C], f[mb]
; According to memo of A. Merand, Jan 2009.
; Note that hPa=mb!
;
if n_params() gt 0 and n_params() lt 4 then begin
	p=ppair(2635)		; mb
	t=12.d0      		; C
	f=ppwv(t,35)*p/1013.	; mb
endif
;
tk=t+273.15d0	; temperature in Kelvin
;
return,1.+1d-6*(1.+0.00752/lambda^2)*(77.6*p/tk+3.73d-5*f/tk^2)
;
end
;-------------------------------------------------------------------------------
function edlen,lambda,p,t,f
;
; Return refractive index for air. lambda[mu], p[mb], t[C], f[mb]
; According to Edlen, ref. K.P. Birch & M.J. Downs, 1993, Metrologia, 30, 155
; Note that hPa=mb!
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_params() gt 0 and n_params() lt 4 then begin
	p=ppair(2635)		; mb
	t=12.0d0      		; C
	f=ppwv(t,35)*p/1013.	; mb
endif
;
s=1.d0/lambda
a=8343.05d0+2406294.d0/(130.d0-s^2)+15999.d0/(38.9d0-s^2)
b=3.7345d0-0.0401d0*s^2
n=a*p/96095.43d0*(1+1d-8*(0.601d0-0.00972d0*t)*p)/(1+0.003661d0*t)-f*b*1d-2
n=n*1d-6
;
return,n+1
;
end
;-------------------------------------------------------------------------------
function owens,lambda,p,t,f
;
; Return refractive index for air. lambda[mu], p[mb], t[C], f[mb]
; According to Owens, Appl. Opt. 6, 51 (1967)
; Note that hPa=mb!
;
if n_params() gt 0 and n_params() lt 4 then begin
	p=ppair(2635)		; mb
	t=12.d0      		; C
	f=ppwv(t,35)*p/1013.	; mb
endif
;
tk=t+273.15d0	; temperature in Kelvin
;
s=1.d0/lambda
a=(2372.434d0+684255.24d0/(130.d0-s^2)+4549.4d0/(38.9d0-s^2))
b=a-(6487.31d0+58.058d0*s^2-0.7115d0*s^4+0.08851d0*s^6)
n=(a*p-b*f)/tk
n=n*1d-8
;
return,n+1
;
end
;-------------------------------------------------------------------------------
function oneill,lambda,t,r,p
;
; Return refractive index for air according to Owens, with input from the
; NPOI observerlog, i.e. t in Fahrenheit, r relative humidity in %, and
; p in inch Hg. Lambda in microns.
;
; Convert t into Celsius
t0=(t-32)*(5./9)
;
; Convert r into mb at sea level
temp=[-30.,-20,-10,0,10,20,30,40]			; Celsius
evap=[50.88,125.4,286.3,610.8,1227,2337,4243,7378]	; Pascal
e=(r/100)*interpol(evap,temp,t0)/100
;
; Convert e to height 2200 m, altitude of NPOI site
h=2.2	; km
e=e*10^(-h/2)
;
; Convert barometric pressure into mb
p0=33.8*p
;
return,owens(lambda,p0,t0,e)
;
end
;-------------------------------------------------------------------------------
function silica,lambda
;
; Return refractive index of silica. Lambda in microns.
;
w=lambda^2
ns=sqrt(1+0.6961663*w/(w-0.0684043^2) $
	 +0.4079426*w/(w-0.1162414^2) $
	 +0.8974794*w/(w-9.896161^2))
;
return,ns
;
end
;-------------------------------------------------------------------------------
function extinct,za
;
; Return extinction normalized to extinction at zenith.
; Input zenith angle is in degrees, may be array.
; From: Schlosser, Schmidt-Kaler, Milone, Chall. of Astr., p. 101
;
rad=180/!pi
secza1=1/cos(za/rad)-1
a=0.0018167
b=0.002875
c=0.000808
airmass=secza1+1-a*secza1-b*secza1^2-c*secza1^3
;
; Extinction in the visual is about 0.2 mag per airmass
k=-alog(10^(-0.2/2.5))
return,exp(-k*airmass)/exp(-k)
;
end
;-------------------------------------------------------------------------------
function wien,lambda
;
b=2.89782e-3	; m K
return,b/lambda
;
end
;-------------------------------------------------------------------------------
function blackbody,teff,lambda,jansky=jansky
;
; Return (specific) intensity B of a blackbody in units of mW m^-2 sr^-1 nm^-1.
; (These units are the ones used by Van Hamme in his limb darkening tables.) 
; Optionally, return in Jansky.
;
; B*d_lambda*dA*cos(theta)*d_Omega is the amount of energy per unit time
; of radiation having a wavelength between lambdas and lambda+d_lambda emitted 
; by a projected surface area dA*cos(theta) into the solid angle d_Omega.
;
; teff [K] 	(scalar)
; lambda [m]	(vector)
;
if n_elements(jansky) eq 0 then jansky=0
teff=float(teff)
;
unity_v=fltarr(n_elements(teff))+1
lambda_a=lambda#unity_v
flux=(1.191d-22/lambda_a^5)/(exp(1.439d-2/(lambda#teff))-1)
if jansky then flux=(flux/10e3)*(lambda_a*1e6)^2/3e-16
;
if n_elements(flux) eq 1 then return,flux(0) else return,reform(flux)
;
; Old
return,(3.742d-30/lambda^5)/(exp(1.439d-2/(lambda*teff))-1)
;
end
;-------------------------------------------------------------------------------
function efftemp,lambda,flux,cf,teff,calibrate=calibrate
;
; Return teff for each pair of (lambda,flux). Before the call,
; cf must be calibrated with a known teff (scalar) and flux @ lambda.
;
if n_elements(calibrate) eq 0 then calibrate=0
;
if calibrate then begin
	f=blackbody(teff,lambda)
	cf=flux/f
	return,cf
endif
;
teff=1.439d-2/(lambda*alog(1+1.191d-22/(lambda^5*flux*cf)))
;
return,teff
;
t=findgen(1000)*30+100
f=blackbody(t,lambda)*cf
df=abs(f-flux)
i=where(df eq min(df))
;
return,t(i)
;
end
;-------------------------------------------------------------------------------
pro ptest
;
l=1e-9*(findgen(100)*3+500)
;
t1=5000.
t2=10000.
b1=blackbody(t1,l)
b2=blackbody(t2,l)
;
d1=3.0
d2=0.4
;
c1=1.0*d1^2		; reference pixel = primary
c2=avg(b2/b1)*d2^2	; secondary
;
; Adopt tr for the reference pixel
tr=5000.
br=blackbody(tr,l)*c1
b2=blackbody(tr,l)*c2
;
; Trivial test for the reference pixel
r=efftemp(l,br,cf,tr,/cal)
; print,efftemp(l,br,cf)
;
print,avg(efftemp(l,b2,cf))
;
end
;-------------------------------------------------------------------------------
function v2jy,mag,av=av
;
; Return flux in Jy for a source of V-band magnitude.
; Using Table 7.5 of Infrared Astronomy, in Allen's Astrophysical Quantities
;
mag=float(mag)
;
if n_elements(av) eq 0 then av=0.0
return,10^(-(mag-av-8.9)/2.5)
;
end
;-------------------------------------------------------------------------------
function j2jy,mag,av=av
;
; Return flux in Jy for a source of N-band magnitude.
; Using Table 7.5 of Infrared Astronomy, in Allen's Astrophysical Quantities
;
; Optionally correct for Av using Rieke and Lebofsky 1985
;
mag=float(mag)
;
r=1.42
if n_elements(av) eq 0 then av=0.0
return,10^(-(mag-av*(1-1/r)-8.0)/2.5)
;
end
;-------------------------------------------------------------------------------
function h2jy,mag,av=av
;
; Return flux in Jy for a source of H-band magnitude.
; Using Table 7.5 of Infrared Astronomy, in Allen's Astrophysical Quantities
;
; Optionally correct for Av using Rieke and Lebofsky 1985
;
mag=float(mag)
;
r=1.22
if n_elements(av) eq 0 then av=0.0
return,10^(-(mag-av*(1-1/r)-7.6)/2.5)
;
end
;-------------------------------------------------------------------------------
function k2jy,mag,av=av
;
; Return flux in Jy for a source of K-band magnitude.
; Using Table 7.5 of Infrared Astronomy, in Allen's Astrophysical Quantities
;
; Optionally correct for Av using Rieke and Lebofsky 1985
;
mag=float(mag)
;
r=1.13
if n_elements(av) eq 0 then av=0.0
return,10^(-(mag-av*(1-1/r)-7.1)/2.5)
;
end
;-------------------------------------------------------------------------------
function l2jy,mag,av=av
;
; Return flux in Jy for a source of L-band magnitude.
; Using Table 7.5 of Infrared Astronomy, in Allen's Astrophysical Quantities
;
; Optionally correct for Av using Rieke and Lebofsky 1985
;
mag=float(mag)
;
r=1.07
if n_elements(av) eq 0 then av=0.0
return,10^(-(mag-av*(1-1/r)-6.1)/2.5)
;
end
;-------------------------------------------------------------------------------
function m2jy,mag,av=av
;
; Return flux in Jy for a source of M-band magnitude. 
; Using Table 7.5 of Infrared Astronomy, in Allen's Astrophysical Quantities
;
; Optionally correct for Av using Rieke and Lebofsky 1985
;
mag=float(mag)
;
r=1.03
if n_elements(av) eq 0 then av=0.0
return,10^(-(mag-av*(1-1/r)-5.5)/2.5)
;
end
;-------------------------------------------------------------------------------
function n2jy,mag
;
; Return flux in Jy for a source of N-band magnitude. This flux would be one
; measured by IRAS, for example.
;
; Using Table 7.5 of Infrared Astronomy, in Allen's Astrophysical Quantities
;
mag=float(mag)
;
return,10^(-(mag-3.9)/2.5)
;
end
;-------------------------------------------------------------------------------
function jy2n,flux
;
; Return N-band magnitude for a source with flux in Jy, measured by IRAS,
; for example.
;
flux=float(flux)
;
n=flux
index=where(flux ne 0,count)
if count gt 0 then n(index)=4.-2.5*alog(flux(index))/alog(10)
index=where(flux eq 0,count)
if count gt 0 then n(index)=100
return,n
;
end
;-------------------------------------------------------------------------------
function q2jy,mag
;
; Return flux in Jy for a source of Q-band magnitude.
; Using Table 7.5 of Infrared Astronomy, in Allen's Astrophysical Quantities
;
mag=float(mag)
;
return,10^(-(mag-2.5)/2.5)
;
end
;-------------------------------------------------------------------------------
function tci,i
;
; Return 24 bit integer corresponding to given Tektronix color index.
; Return input unchanged if device selected is PostScript.
;
r = bytscl([ 0,100,100,0,0,0,100,100,100,60,0,0,55,100,33,67, $
        100,75,45,17,25,50,75,100,67,40,17,17,17,45,75,90])
g = bytscl([ 0,100,0,100,0,100,0,100,50,83,100,50,0,0,33,67, $
        100,100,100,100,83,67,55,33,90,90,90,67,50,33,17,9])
b = bytscl([ 0,100,0,0,100,100,83,0,0,0,60,100,83,55,33,67, $
        33,45,60,75,83,83,83,90,45,55,67,90,100,100,100,100])
;
if !d.name eq 'PS' then return,i else return,r(i)+(g(i)*2L^8)+(b(i)*2L^16)
;
end
;-------------------------------------------------------------------------------
function getcolor,i
;
; Return index of a useful Tektronix color. Color indices start at 0 for black.
;
n=1
color=(i mod (32-n-3))+3	; Do not use reserved colors 0-2
;
; Change colors which are problematic
if !d.name eq 'PS' then begin
	if color eq  7 then color=17	; yellow is bad for paper
	if color eq 16 then color=25	; yellow is bad for paper
endif
if color eq 14 then color=31	; dark gray is bad for screen
if color eq  4 then color=28
;
return,tci(color)
;
end
;-------------------------------------------------------------------------------
function getsymbol,i
;
; Return a plot symbol other than the invisible dot. i>0.
;
symbols=[1,2,4,5,6,7]
return,symbols((i-1) mod 6)
;
end
;-------------------------------------------------------------------------------
function greek,ch
;
small_g=['alpha','beta','gamma','delta','epsilon','zeta','eta','theta','iota', $
         'kappa','lambda','mu','nu','xi','omi','pi','rho','sigma','tau', $ 
         'ups','phi','chi','psi','omega']
large_g=strupcase(small_g)
all_g=[small_g,large_g]
;
small_a=['a','b','c','d','e','f','g','h','i','j','k','l','m','n', $
	 'o','p','q','r','s','t','u','v','w','x','y','z']
large_a=strupcase(small_a)
all_a=[small_a,large_a]
;
small_l=['a','b','g','d','e','z','h','q','i','k','l','m','n','x','o','p','r', $
	 's','t','u','j','c','y','w']
large_l=strupcase(small_l)
all_l=[small_l,large_l]
;
i=strpos(all_g,ch) & i=where(i eq 0) & i=i(0)
;
if !p.font eq -1 then return,'!4'+all_a(i)+'!X'	; to use Hershey fonts
if !p.font eq +1 then return,'!M'+all_l(i)+'!X'	; to use True Type fonts
;
return,''
;
end
;-------------------------------------------------------------------------------
function alphabet,index,cap=cap
;
if n_elements(cap) eq 0 then cap=0
;
small_a=['a','b','c','d','e','f','g','h','i','j','k','l','m','n', $
	 'o','p','q','r','s','t','u','v','w','x','y','z']
large_a=strupcase(small_a)
;
if cap then return,large_a(index-1) else return,small_a(index-1)
;
end
;-------------------------------------------------------------------------------
function histogramm,x,y,min=minx,max=maxx,binsize=binsize,values=values
;
; Places values x in bins similar to histogram, but averages corresponding
; values y and returns array of averages.
;
if n_elements(minx) eq 0 then minx=min(x)
if n_elements(maxx) eq 0 then maxx=max(x)
if n_elements(binsize) eq 0 then binsize=1
;
;
f=histogram(x,min=minx,max=maxx,binsize=binsize,reverse_indices=r)
n=n_elements(f)
values=findgen(n)*binsize+minx+binsize/2
if n_elements(y) eq 0 then return,f
;
y_mean=fltarr(n)
for i=0,n-1 do begin
	if r(i) ne r(i+1) then $
	y_mean(i)=avg(y(r(r(i):r(i+1)-1)))
endfor
return,y_mean
;
for i=0,n-2 do begin
        index=where(x ge values(i) and x lt values(i+1),count)
        if count gt 0 then y_mean(i)=avg(y(index))
endfor
return,y_mean
;
end
;-------------------------------------------------------------------------------
pro histograph,y,min=miny,max=maxy,binsize=binsize,oplot=oplot,color=color
;
if n_elements(miny) eq 0 then miny=min(y)
if n_elements(maxy) eq 0 then maxy=max(y)
if n_elements(binsize) eq 0 then binsize=1
if n_elements(color) eq 0 then color=tci(1)
;
f=histogram(y,min=miny,max=maxy,binsize=binsize)
n=n_elements(f)
x=findgen(n)*binsize+miny
x2=fltarr(2*n)
x2(indgen(n)*2)=x
x2(indgen(n)*2+1)=x+binsize
f2=fltarr(2*n)
f2(indgen(n)*2)=f
f2(indgen(n)*2+1)=f
if keyword_set(oplot) then oplot,x2,f2,psym=0,color=color $
		      else plot,x2,f2,psym=0
;
end
;************************************************************************Block 3
function hms,x_in,aspro=aspro,simbad=simbad,toggl=toggl,scan=scan
;
; Return formatted string for x in hours. x may be array.
;
if n_elements(aspro) eq 0 then aspro=0
if n_elements(simbad) eq 0 then simbad=0
if n_elements(toggl) eq 0 then toggl=0
if n_elements(scan) eq 0 then scan=0
;
x=x_in
index=where(x lt 0,count)
sign=strarr(n_elements(x)) & sign(*)='+'
if count gt 0 then sign(index)='-'
sign(*)=''
x=abs(x)
hour=fix(x)
min=fix((x-hour)*60)
sec=double(((x-hour)*60-min)*60)
if aspro then begin
label= strcompress(string(hour,format='(i2.2)'))+':' $
      +strcompress(string(min,format='(i2.2)'))+':' $
      +strcompress(string(fix(sec),format='(i2.2)'))+'.' $
      +strmid(string(sec-fix(sec),format='(f6.4)'),2,4)
endif else if simbad then begin
label= strcompress(string(hour,format='(i2.2)'))+' ' $
      +strcompress(string(min,format='(i2.2)'))+' ' $
      +strcompress(string(fix(sec),format='(i2.2)'))+'.' $
      +strmid(string(sec-fix(sec),format='(f7.5)'),2,5)
endif else if toggl then begin
label= strcompress(string(hour,format='(i3.3)'))+'h ' $
      +strcompress(string(min,format='(i2.2)'))+'m ' $
      +strcompress(string(sec,format='(f4.1)'))+'s'
endif else if scan then begin
label= strcompress(string(hour,format='(i2.2)'))+'h ' $
      +strcompress(string(min,format='(i2.2)'))+'m ' $
      +strcompress(string(nint(sec),format='(i2.2)'))+'s'
endif else begin
label= strcompress(string(hour,format='(i2.2)'))+'h ' $
      +strcompress(string(min,format='(i2.2)'))+'m ' $
      +strcompress(string(sec,format='(f4.1)'))+'s'
endelse
if n_elements(label) eq 1 then return,label(0) else return,label
;
end
;-------------------------------------------------------------------------------
function dms,x_in,aspro=aspro,simbad=simbad
;
; Return formatted string for x in degrees. x may be array.
; Degrees can run -360 to 360.
;
if n_elements(aspro) eq 0 then aspro=0
if n_elements(simbad) eq 0 then simbad=0
;
x=x_in
index=where(x lt 0,count)
vorz=strarr(n_elements(x)) & vorz(*)='+'
if count gt 0 then vorz(index)='-'
x=abs(x)
deg=fix(x)
mit=fix((x-deg)*60)
sec=double(((x-deg)*60-mit)*60)
index=where(sec eq 60.0,count)
if count gt 0 then begin
	mit(index)=mit(index)+1
	sec(index)=0.0
endif
if aspro then begin
label= vorz+string(deg,format='(i2.2)')+':' $
      +strcompress(string(mit,format='(i2.2)'))+":" $
      +strcompress(string(fix(sec),format='(i2.2)'))+"." $
      +strcompress(string((sec-fix(sec))*1000,format='(i3.3)'))
endif else if simbad then begin
label= strcompress(vorz+string(deg,format='(i2.2)'))+' ' $
      +strcompress(string(mit,format='(i2.2)'))+' ' $
      +strcompress(string(fix(sec),format='(i2.2)'))+'.' $
      +strmid(string(sec-fix(sec),format='(f6.4)'),2,4)
endif else begin
label= strcompress(vorz+string(deg,format='(i3.3)'))+'d ' $
      +strcompress(string(mit,format='(i2.2)'))+"' " $
      +strcompress(string(sec,format='(f6.3)'))+'"'
endelse
if n_elements(label) eq 1 then return,label(0) else return,label
;
end
;-------------------------------------------------------------------------------
function hms2h,h_in
;
; Return floating point time in hours from hh:mm:ss.sssss input. Arrays OK.
;
h=h_in
index=where(strlen(strcompress(h,/remove_all)) eq 0,count)
if count gt 0 then h(index)='00:00:00.00000'
t=double(strmid(h,0,2))
t=t+double(strmid(h,3,2))/60
; t=t+double(strmid(h,6,strlen(h)-6))/3600
t=t+double(strmid(h,6,8))/3600
return,t
;
end
;-------------------------------------------------------------------------------
function dms2d,d_in
;
; Return floating point angle in degrees from sdd:mm:ss.sss input. Arrays OK.
;
d=d_in
ipos=strpos(d,':')
index=where(ipos eq 2,count)
if count gt 0 then d(index)='+'+d(index)
a=abs(double(strmid(d,0,3)))
a=a+double(strmid(d,4,2))/60
a=a+double(strmid(d,7,strlen(d)-7))/3600
index=where(strmid(d,0,1) eq '-',count) 
if count gt 0 then a(index)=-a(index)
return,a
;
end
;-------------------------------------------------------------------------------
function wdsid,ra,dec
;
; Return positions in WDS ID format string, i.e. +HHMMFDDMMF,
; or -HHMMFDDMMF, with the sign corresponding to the sign of the
; declination.
;
n=n_elements(ra)
;
rah=fix(ra)
ram=(ra-rah)*60
raf=(ram-fix(ram))*10
decd=fix(dec)
decm=(abs(dec-decd))*60
decf=(decm-fix(decm))*10
index=where(decd lt 0,count)
wds=strarr(n)+'+' & if count gt 0 then wds(index)='-'
wds=wds+string(rah,format='(i2.2)') $
       +string(fix(ram),format='(i2.2)') $
       +string(fix(raf),format='(i1.1)') $
       +string(abs(decd),format='(i2.2)') $
       +string(fix(decm),format='(i2.2)') $
       +string(fix(decf),format='(i1.1)')
;
return,wds
;
end
;-------------------------------------------------------------------------------
function esoid,ra,dec
;
; Return positions in ESO ID format string, i.e. +HHMMSSFFDDMMSSF,
; or -HHMMSSFFDDMMSSF, with the sign corresponding to the sign of the
; declination.
;
n=n_elements(ra)
;
rah=fix(ra)
ram=fix((ra-rah)*60)
ras=(ra-rah-ram/60.)*3600
raf=(ras-fix(ras))*100
deca=abs(dec)
decd=fix(deca)
decm=fix((deca-decd)*60)
decs=(deca-decd-decm/60.)*3600
decf=(decs-fix(decs))*10
index=where(dec lt 0,count)
wds=strarr(n)+'+' & if count gt 0 then wds(index)='-'
wds=wds+string(rah,format='(i2.2)') $
       +string(fix(ram),format='(i2.2)') $
       +string(fix(ras),format='(i2.2)') $
       +string(fix(raf),format='(i2.2)') $
       +string(abs(decd),format='(i2.2)') $
       +string(fix(decm),format='(i2.2)') $
       +string(fix(decs),format='(i2.2)') $
       +string(fix(decf),format='(i1.1)')
;
return,wds
;
end
;-------------------------------------------------------------------------------
function wdspos,id0
;
; Given a WDS ID format string, convert to RA and DEC. May contain the WDS
; catalog identifier.
;
id=id0
index=where(strpos(id,'WDS') ge 0,count)
if count gt 0 then id(index)=strmid(id(index),3,11)
;
rah=float(strmid(id,1,2))
ram=float(strmid(id,3,3))/10
decd=float(strmid(id,6,2))
decm=float(strmid(id,8,3))/10
sign=strmid(id,0,1)
f=fltarr(n_elements(id))+1
index=where(sign eq '-',count)
if count gt 0 then f(index)=-1
;
return,[[rah+ram/60],[f*(decd+decm/60)]]
;
end
;-------------------------------------------------------------------------------
function esopos,id0
;
; Given an ESO ID format string, convert to RA and DEC. May contain the OBJ
; catalog identifier.
;
id=id0
index=where(strpos(id,'OBJ') ge 0,count)
if count gt 0 then id(index)=strmid(id(index),3,16)
;
rah=float(strmid(id,1,2))
ram=float(strmid(id,3,2))
ras=float(strmid(id,5,4))/100
decd=float(strmid(id,9,2))
decm=float(strmid(id,11,2))
decs=float(strmid(id,13,3))/10
sign=strmid(id,0,1)
f=fltarr(n_elements(id))+1
index=where(sign eq '-',count)
if count gt 0 then f(index)=-1
;
return,[[rah+ram/60+ras/3600],[f*(decd+decm/60+decs/3600)]]
;
end
;-------------------------------------------------------------------------------
function stringparse,stringg
;
; Simple parser. Example: string='1,3,4-6' --> s=[1,3,4,5,6]
;
for iter=1,2 do begin
	i=0
	pos1=0
	repeat begin
		pos2=strpos(stringg,',',pos1)
		if pos2 eq -1 then pos2=strlen(stringg)
		sub=strmid(stringg,pos1,pos2-pos1)
		index=long(sub)
		if iter eq 2 then s(i)=index
		pos3=strpos(sub,'-',0)
		if pos3 ne -1 then begin
			n=long(strmid(sub,pos3+1,strlen(sub)-pos3-1))
			for k=1,n-index do begin
				i=i+1
				if iter eq 2 then s(i)=index+k
			endfor
		endif
		i=i+1
		pos1=pos2+1
	endrep until pos1 ge strlen(stringg)
	if iter eq 1 then s=intarr(i)	; First run used to set dimension
endfor
return,s
end
;-------------------------------------------------------------------------------
function retroparse,array
;
n=n_elements(array)
if n eq 0 then return,''
index=intarr(n)+1
for i=1,n_elements(array)-2 do begin
	if array(i)-array(i-1) eq 1 and $
	   array(i+1)-array(i) eq 1 then index(i)=0
endfor
stringg=string(array(0))
for i=1,n-1 do begin
	if index(i) then begin
		if index(i-1) then space=',' else space='-'
		stringg=stringg+space+string(array(i))
	endif
endfor
;
return,strcompress(stringg,/remove_all)
end
;-------------------------------------------------------------------------------
function nameparse,name_in,delimiters,posns
;
; Parse a (single) string into array of words, using delimiters. If at least
; one of the delimiters is a NULL string (i.e. ''), parse into single
; characters, but removing the remaining (single character) delimiters.
; Delimiters default to space and tab characters.
;
if n_elements(delimiters) eq 0 then delimiters=[' ',string(09b)]
si=sort(strlen(delimiters))
delimiters=delimiters(si)
;
if n_elements(name_in) eq 1 then name=name_in(0) else name=name_in
;
if delimiters(0) eq '' then begin
	words=strarr(strlen(name))
	for i=0,strlen(name)-1 do words(i)=strmid(name,i,1)
	return,words
endif
;
n=0
j=0
in_word=0
words=''
de_length=1
for i=0l,strlen(name)-1 do begin
	delimiter=0
	for k=0,n_elements(delimiters)-1 do begin
		if strmid(name,i,strlen(delimiters(k))) $
		eq delimiters(k) then begin
			delimiter=1
			de_length=strlen(delimiters(k))
			if strlen(strtrim(delimiters(k))) eq 0 then isblank=1 $
							       else isblank=0
		endif
	endfor
	if delimiter then begin
		if in_word or not isblank then begin
			n=n+1
			words=strarr(n)
			posns=intarr(n)
			if n gt 1 then words(0:n-2)=old_words
			if n gt 1 then posns(0:n-2)=old_posns
			if in_word then words(n-1)=strmid(name,j,i-j)
			posns(n-1)=j
			old_words=words
			old_posns=posns
			j=i+de_length-1
			in_word=0
		endif
	endif else begin
		if in_word eq 0 then begin
			in_word=1
			j=i+de_length-1 ; starting point for current word
		endif
	endelse
endfor
if in_word eq 1 then begin
	n=n+1
	words=strarr(n)
	posns=intarr(n)
	if n gt 1 then words(0:n-2)=old_words
	if n gt 1 then posns(0:n-2)=old_posns
	words(n-1)=strmid(name,j,i-j)
	posns(n-1)=j
endif 
return,words
;
end
;-------------------------------------------------------------------------------
function fitshparse,header,keyword_in
;
keyword=strupcase(keyword_in)
;
n=n_elements(header)
keywords=strarr(n)
values=strarr(n)
for i=0,n-1 do begin
	words=nameparse(header(i),'=')
	keywords(i)=strcompress(words(0),/remove_all)
	if n_elements(words) gt 1 then begin
		value=words(1)
		words=nameparse(value,"/")
		values(i)=strcompress(words(0),/remove_all)
		sl=strlen(values(i))
		if strpos(values(i),"'") eq 0 and $
		   strpos(values(i),"'",/reverse_search) eq sl-1 $
		then values(i)=strmid(values(i),1,sl-2)
	endif
endfor
index=where(keywords eq keyword,count)
if count eq 1 then return,values(index(0))
print,'Keyword not found!'
return,''
;
end
;-------------------------------------------------------------------------------
function criparse,request,cat_1,cat_2
;
; Parse cri request into catalog identifiers.
;
len=strlen(request)
pos=strpos(request,'-')
cat_1=strmid(request,0,pos)
cat_2=strmid(request,pos+1,len-pos-1)
if strpos(cat_2,'-') ne -1 then begin
	print,'***Error(CRIPARSE): invalid request!'
	return,-1
endif
if cat_2 eq 'hip' then cat_2='hic'
return,0
;
end
;-------------------------------------------------------------------------------
function cri,id1,request
; Do not remove: allows cri to be called recursively! (see code below)
end
;-------------------------------------------------------------------------------
function cri,id0,request,lists=lists
;
; Return id2 of star with id1 according to requested crossindex.
;
greek_alphabet=['ALF','ALP','BET','GAM','DEL','EPS','ZET','ETA','THE','TET', $
		'IOT','KAP','LAM', 'MU', 'NU', 'XI','OMI', 'PI','RHO','SIG', $
		'TAU','UPS','PHI','CHI','PSI','OME']
;
if n_params() ne 2 then begin
	print,'***Error(CRI): wrong number of parameters!'
	return,-1
endif
num_id=n_elements(id0)
if num_id eq 0 then begin
	print,'***Error(CRI): id undefined!'
	return,-1
endif else id1=id0
if n_elements(request) eq 0 then begin
	print,'***Error(CRI): request not specified!'
	return,-1
endif else request=strlowcase(request)
;
; If lists=1 search only in non-generic lists
if n_elements(lists) eq 0 then lists=0
;
; Replace HDN and BSC with HD and HR catalog identifier which are detected below
index=where(strmid(id1,0,3) eq 'HDN',count)
if count gt 0 then $
	id1(index)='HD'+string(strmid(id1(index),3,6),format='(i6.6)')
index=where(strmid(id1,0,3) eq 'BSC',count)
if count gt 0 then $
	id1(index)='HR'+string(strmid(id1(index),3,4),format='(i4.4)')
;
id2=lonarr(num_id)-1
c2=strarr(num_id)
;
; Special feature: return ID number for name
r=size(id1)
if r(n_elements(r)-2) eq 7 then begin
	if lists then $
	for i=0,num_id-1 do if strpos(id1(i),'OBJ') lt 0 then $
		id1(i)=strjoin(nameparse(id1(i),[' ','-','_']))
;	Two-letter catalog names (HD/HR), allow blank between cat and ID
	index=where(strupcase(strmid(id1,0,2)) eq 'HD',count)
	if count gt 0 and request eq 'hdn' then $
	id2(index)=long(strmid(id1(index),2,7))
	if count gt 0 and request eq 'bsc' then $
	id2(index)=cri(long(strmid(id1(index),2,7)),'hdn-bsc')
	index=where(strupcase(strmid(id1,0,2)) eq 'HR',count)
	if count gt 0 and request eq 'bsc' then $
	id2(index)=long(strmid(id1(index),2,5))
	if count gt 0 and request eq 'hdn' then $
	id2(index)=cri(long(strmid(id1(index),2,5)),'bsc-hdn')
;
	file=!oyster_dir+'starbase/vlti.hdn'
	result=file_search(file,count=fcount)
	if fcount ne 0 then begin
		hdn=0
		toes=''
		s=dc_read_fixed(file,hdn,toes,/col,ignore=['!'], $
			format='(i6,a32)')
		for i=0,num_id-1 do begin
			words=strupcase(nameparse(id1(i)))
			si=sort(words) & words=words(reverse(si))
			for j=0,n_elements(toes)-1 do begin
				l=0
				name=toes(j)
				for k=0,n_elements(words)-1 do begin
				m=strpos(strupcase(name),words(k),0)
				if m ne -1 then begin
					strput,name,blanks(strlen(words(k))),m
					l=l+1
				endif
				endfor
				if l eq n_elements(words) $
					and strlen(strcompress(toes(j),/remove_all))  $
				 	eq fix(total(strlen(words))) $
			   	then begin
					id2(i)=hdn(j)
					c2(i)='HDN'
				endif
			endfor
		endfor
	endif
	file=!oyster_dir+'starbase/vlti.bsc'
	result=file_search(file,count=fcount)
	if fcount ne 0 and total(id2 ge 0) lt num_id then begin
		bsc=0
		toes=''
		s=dc_read_fixed(file,bsc,toes,/col,ignore=['!'], $
			format='(i4,a32)')
		for i=0,num_id-1 do begin
			words=strupcase(nameparse(id1(i)))
			si=sort(words) & words=words(reverse(si))
			for j=0,n_elements(toes)-1 do begin
				l=0
				name=toes(j)
				for k=0,n_elements(words)-1 do begin
				m=strpos(strupcase(name),words(k),0)
				if m ne -1 then begin
					strput,name,blanks(strlen(words(k))),m
					l=l+1
				endif
				endfor
				if l eq n_elements(words) $
					and strlen(strcompress(toes(j),/remove_all))  $
				 	eq fix(total(strlen(words))) $
			   	then begin
					id2(i)=bsc(j)
					c2(i)='BSC'
				endif
			endfor
		endfor
	endif
	file=!oyster_dir+'starbase/vlti.hic'
	result=file_search(file,count=fcount)
	if fcount ne 0 and total(id2 ge 0) lt num_id then begin
		hic=0
		toes=''
		s=dc_read_fixed(file,hic,toes,/col,ignore=['!'], $
			format='(i6,a32)')
		for i=0,num_id-1 do begin
			words=strupcase(nameparse(id1(i)))
			si=sort(words) & words=words(reverse(si))
			for j=0,n_elements(toes)-1 do begin
				l=0
				name=toes(j)
				for k=0,n_elements(words)-1 do begin
				m=strpos(strupcase(name),words(k),0)
				if m ne -1 then begin
					strput,name,blanks(strlen(words(k))),m
					l=l+1
				endif
				endfor
				if l eq n_elements(words) $
					and strlen(strcompress(toes(j),/remove_all))  $
				 	eq fix(total(strlen(words))) $
			   	then begin
					id2(i)=hic(j)
					c2(i)='HIC'
				endif
			endfor
		endfor
	endif
	file=!oyster_dir+'starbase/vlti.usr'
	result=file_search(file,count=fcount)
	if fcount ne 0 and total(id2 ge 0) lt num_id then begin
		usr=0
		toes=''
		s=dc_read_fixed(file,usr,toes,/col,ignore=['!'], $
			format='(i4,a32)')
		for i=0,num_id-1 do begin
			words=strupcase(nameparse(id1(i)))
			si=sort(words) & words=words(reverse(si))
			for j=0,n_elements(toes)-1 do begin
				l=0
				name=toes(j)
				for k=0,n_elements(words)-1 do begin
				m=strpos(strupcase(name),words(k),0)
				if m ne -1 then begin
					strput,name,blanks(strlen(words(k))),m
					l=l+1
				endif
				endfor
				if l eq n_elements(words) $
					and strlen(strcompress(toes(j),/remove_all))  $
				 	eq fix(total(strlen(words))) $
			   	then begin
					id2(i)=usr(j)
					c2(i)='USR'
				endif
			endfor
		endfor
	endif
;	file=!oyster_dir+'starbase/vlti.usr'
;	result=file_search(file,count=fcount)
;	if fcount ne 0 and total(id2 ge 0) lt num_id  then begin
;		sid=''
;		toes=''
;		s=dc_read_free(file,usr,toes,/col,ignore=['!'])
;		for i=0,num_id-1 do begin
;			index=where(toes eq id1(i),count)
;			if count gt 0 then begin
;				id2(i)=usr(index)
;				c2(i)='USR'
;			endif
;		endfor
;	endif
	file=!oyster_dir+'starbase/toe.bsc'
	result=file_search(file,count=fcount)
	if fcount ne 0 and total(id2 ge 0) lt num_id then begin
		bsc=0
		toes=''
		s=dc_read_fixed(file,bsc,toes,/col,ignore=['!'], $
			format='(i4,a32)')
		for i=0,num_id-1 do begin
			words=strupcase(nameparse(id1(i)))
			si=sort(words) & words=words(reverse(si))
			for j=0,n_elements(toes)-1 do begin
				l=0
				name=toes(j)
				for k=0,n_elements(words)-1 do begin
				m=strpos(strupcase(name),words(k),0)
				if m ne -1 then begin
					strput,name,blanks(strlen(words(k))),m
					l=l+1
				endif
				endfor
				if l eq n_elements(words) $
					and strlen(strcompress(toes(j),/remove_all))  $
				 	eq fix(total(strlen(words))) $
					and id2(i) eq -1 $
			   	then begin
					id2(i)=bsc(j)
					c2(i)='BSC'
				endif
			endfor
		endfor
	endif
	file=!oyster_dir+'starbase/var.bsc'
	result=file_search(file,count=fcount)
	if fcount ne 0 and total(id2 ge 0) lt num_id then begin
		bsc=0
		vars=''
		s=dc_read_free(file,bsc,vars,/col,ignore=['!'])
		for i=0,num_id-1 do begin
			words=strupcase(nameparse(id1(i)))
			si=sort(words) & words=words(reverse(si))
			for j=0,n_elements(vars)-1 do begin
				l=0
				name=vars(j)
				for k=0,n_elements(words)-1 do begin
				m=strpos(strupcase(name),words(k),0)
				if m ne -1 then begin
					strput,name,blanks(strlen(words(k))),m
					l=l+1
				endif
				endfor
				if l eq n_elements(words) $
					and strlen(strcompress(vars(j),/remove_all))  $
				 	eq fix(total(strlen(words))) $
					and id2(i) eq -1 $
			   	then begin
					id2(i)=bsc(j)
					c2(i)='BSC'
				endif
			endfor
		endfor
	endif
	file=!oyster_dir+'starbase/name.bsc'
	result=file_search(file,count=fcount)
	if fcount ne 0 and total(id2 ge 0) lt num_id and not lists then begin
		bsc=0
		names=''
		s=dc_read_free(file,bsc,names,/col,ignore=['!'])
		for i=0,num_id-1 do begin
			words=reverse(strupcase(nameparse(id1(i),[' ','_'])))
			if n_elements(words) eq 2 then begin
			for j=0,n_elements(names)-1 do begin
				l=0
				name=names(j)
				if isnumeric(strmid(name,0,1)) $
					then flamsteed=fix(name) $
					else flamsteed=0
				for k=0,n_elements(words)-1 do begin
				m=strpos(name,strmid(words(k),0,3),/reverse_search)
				if flamsteed ne 0 and isnumeric(words(k)) then $
					if fix(words(k)) ne flamsteed then m=-1
				if k eq 1 and not isnumeric(words(k)) then begin
					index=where(greek_alphabet eq words(k),count)
					if count eq 0 then m=-1
				endif
				if m ne -1 then begin
					l=l+1
					for n=1,strlen(words(k)) do $
						strput,name,' ',m+n-1
				endif
				endfor	
				if l eq n_elements(words) $
					and strlen(names(j)) $
					ge fix(total(strlen(words))) $
					and id2(i) eq -1 $
				then begin
					id2(i)=bsc(j)
					c2(i)='BSC'
				endif
			endfor
			endif
		endfor
	endif
	if request eq 'bsc' then begin
		index=where(c2 eq 'HIC',count)
		if count gt 0 then id2(index)=cri(id2(index),'hic-bsc')
		index=where(c2 eq 'HDN',count)
		if count gt 0 then id2(index)=cri(id2(index),'hdn-bsc')
		index=where(c2 eq 'USR',count)
		if count gt 0 then id2(index)=-1
		return,id2 
	endif else if request eq 'hic' then begin
		index=where(c2 eq 'BSC',count)
		if count gt 0 then id2(index)=cri(id2(index),'bsc-hic')
		index=where(c2 eq 'HDN',count)
		if count gt 0 then id2(index)=cri(id2(index),'hdn-hic')
		index=where(c2 eq 'USR',count)
		if count gt 0 then id2(index)=-1
		return,id2
	endif else if request eq 'hdn' then begin
		index=where(c2 eq 'FKV',count)
		if count gt 0 then id2(index)=cri(id2(index),'fkv-hdn')
		index=where(c2 eq 'BSC',count)
		if count gt 0 then id2(index)=cri(id2(index),'bsc-hdn')
		index=where(c2 eq 'HIC',count)
		if count gt 0 then id2(index)=cri(id2(index),'hic-hdn')
		index=where(c2 eq 'USR',count)
		if count gt 0 then id2(index)=-1
		return,id2
	endif else if request eq 'fkv' then begin
		index=where(c2 eq 'BSC',count)
		if count gt 0 then id2(index)=cri(id2(index),'bsc-fkv')
		index=where(c2 eq 'HIC',count)
		if count gt 0 then id2(index)=cri(id2(index),'hic-fkv')
		index=where(c2 eq 'USR',count)
		if count gt 0 then id2(index)=-1
		return,id2
	endif else if request eq 'usr' then begin
		index=where(c2 eq 'HDN',count)
		if count gt 0 then id2(index)=-1
		index=where(c2 eq 'BSC',count)
		if count gt 0 then id2(index)=-1
		index=where(c2 eq 'HIC',count)
		if count gt 0 then id2(index)=-1
		index=where(c2 eq 'FKV',count)
		if count gt 0 then id2(index)=-1
		return,id2
	endif else if request eq 'cat' then begin
		request=c2
		return,id2
	endif

endif
		
;
if criparse(request,cat_1,cat_2) ne -1 then begin
        file=!catalogs_dir+'crossindex/'+cat_1+'_'+cat_2+'.cri'
        result=file_search(file,count=fcount)
        if fcount ne 0 then restore,filename=file else begin
                print,'***Error(CRI): file not found: ',file,'!'
                return,-1
        endelse
endif else return,-1
;
num_t=n_elements(t)
if num_id gt num_t/30L then begin
	s=sort(id1)
	id1_s=id1(s)
	id2_s=id2
	j=0L
	for i=0L,num_t-1L do begin
		while t(i).id1 gt id1_s(j) and j lt num_id-1L do j=j+1L
		while t(i).id1 eq id1_s(j) and j lt num_id-1L do begin
			id2_s(j)=t(i).id2
			j=j+1L
		endwhile
		if t(i).id1 eq id1_s(j) then id2_s(j)=t(i).id2
	endfor
	id2(s)=id2_s
endif else begin
	for i=0L,num_id-1L do begin
		index=where(t.id1 eq id1(i),count)
		if count ne 0 then id2(i)=t(index(0)).id2
	endfor
endelse
;
if n_elements(id2) eq 1 then return,id2(0) else return,id2
;
end
;-------------------------------------------------------------------------------
function cri_simbad,target0,table
;
; Query SIMBAD using either lynx or elinks (wget could be used too?) to find
; target by coordinates or identifier. In the first case, return name of star
; if found, in the second case return OYSTER StarId if found in HD catalog.
;
; Examples for target:
; '09 47 57.382 +13 16 43.66'
; 'OBJ+000556692434080'
; 'HD 100'
;
table={starid:'',name:'',hdn:0l,ra:0.d0,dec:0.d0,pmra:0.d0,pmdec:0.d0, $
	spectrum:'',mv:0.0,mj:0.0,mh:0.0,mk:0.0,bv:0.0}
;
; What text browser is available
browser='elinks -dump '
browser='lynx -dump '
;
; Lookup by coordinates
if n_elements(nameparse(target0)) gt 2 or strpos(target0,'OBJ') ge 0 then begin
	if strpos(target0,'OBJ') ge 0 then begin
		target=esopos(target0)
		words=[hms(target(0),/aspro),dms(target(1),/aspro)]
	endif else begin
		words=nameparse(target0)
	endelse
	if n_elements(words) gt 2 then begin
		ra=strjoin(words(0:2),':')
		dec=strjoin(words(3:5),':')
	endif else begin
		ra=words(0)
		dec=words(1)
	endelse
	words=nameparse(ra,':')
	ra=strjoin(words,'+')
	words=nameparse(dec,':')
	dec=strjoin(words,'+')
	if strpos(dec,'-') ge 0 then begin
		c='-'
	endif else begin
		c='%2B'
	endelse
	dec=strmid(dec,1,15)
;	Simbad4 call (using scripts)
	simbad='http://simweb.u-strasbg.fr/simbad/'
	script='sim-script?submit=submit+script&script='
	spawn,browser+'"'+simbad+script+ra+'+'+c+dec+'"',r
	j=where(strpos(r,'liste identificateurs') ge 0,count) & j=j(0)
	if count ge 1 then begin
;               SIMBAD found the target
;               We return HD, IRC, or IRAS
		k=strpos(r(j),'(')
		n=long(strmid(r(j),k+1,2))
		identifiers=r(j+1:j+n)
		k=where(strpos(identifiers,'2MASS') ge 0) & k=k(0)
		if k ge 0 then star=identifiers(k)
		k=where(strpos(identifiers,'IRAS') ge 0) & k=k(0)
		if k ge 0 then star=identifiers(k)
		k=where(strpos(identifiers,'IRC') ge 0) & k=k(0)
		if k ge 0 then star=identifiers(k)
		k=where(strpos(identifiers,'HD ') ge 0) & k=k(0)
		if k ge 0 then star=identifiers(k)
		table.name=star
;		ICRF Coordinates
		j=where(strpos(r,'coord') ge 0) & j=j(0)
		words=nameparse(r(j),[':','('])
		if strpos(words(1),'-') ge 0 then sign=-1 else sign=1
		words=nameparse(words(1),['+','-'])
		ra=strjoin(nameparse(words(0)),':')
		dec=strjoin(nameparse(words(1)),':')
		table.ra=hms2h(ra)
		table.dec=dms2d(dec)*sign
		return,star
	endif else begin
		return,''
	endelse
endif else begin
; Lookup by identifier
	words=nameparse(target0)
	if strpos(target0,'HDN') ge 0 then words=['HD',strmid(target0,3,6)]
	if strpos(target0,'HIP') ge 0 then words=['HIP',strmid(target0,3,6)]
	target=strjoin(words,'+')
;	Simbad4 call (using scripts)
	simbad='http://simweb.u-strasbg.fr/simbad/'
	script='sim-script?submit=submit+script&script=query+id+'
	spawn,browser+'"'+simbad+script+target+'"',r
	j=where(strpos(r,'liste identificateurs') ge 0,count) & j=j(0)
	if count ge 1 then begin
;               SIMBAD found the target
		table.name=target0
		k=strpos(r(j),'(')
		n=long(strmid(r(j),k+1,2))
		identifiers=r(j+1:j+n)
		k=where(strpos(identifiers,'HD ') ge 0) & k=k(0)
		if k ge 0 then begin
;			HD identifier
			words=nameparse(identifiers(k))
			table.hdn=long(words(1))
			table.starid='HDN'+string(table.hdn,format='(i6.6)')
		endif
;		Coordinates
		j=where(strpos(r,'coord') ge 0) & j=j(0)
		words=nameparse(r(j),[':','('])
		if strpos(words(1),'-') ge 0 then sign=-1 else sign=1
		words=nameparse(words(1),['+','-'])
		ra=strjoin(nameparse(words(0)),':')
		dec=strjoin(nameparse(words(1)),':')
		table.ra=hms2h(ra)
		table.dec=dms2d(dec)*sign
;		Proper motions
		j=where(strpos(r,'proper motion') ge 0) & j=j(0)
		if j(0) ge 0 then begin
		words=nameparse(r(j),[':','['])
		words=nameparse(words(1))
; 		Convert mas/year to arcsec/centennial
		if isnumeric(words(0)) then table.pmra=(double(words(0))/150) $
						      /cos(table.dec*!pi/180)
		if isnumeric(words(1)) then table.pmdec=double(words(0))/10
		endif
;		Spectral type
		j=where(strpos(r,'Spectral type') ge 0) & j=j(0)
		if j ge 0 then begin
		words=nameparse(r(j),':')
		words=nameparse(words(1))
		table.spectrum=strcompress(words(0),/remove_all)
		endif
;		Magnitudes
		j=where(strpos(r,'flux: V (Vega)') ge 0) & j=j(0)
		j=where(strpos(r,'flux: V') ge 0) & j=j(0)
		if j ge 0 then begin
			if strpos(r(j),'Vega') ge 0 then $
			table.mv=float(strmid(r(j),15,10)) else $
			table.mv=float(strmid(r(j),8,10))
		endif else table.mv=100
		j=where(strpos(r,'flux: B') ge 0) & j=j(0)
		if j ge 0 then begin
			if strpos(r(j),'Vega') ge 0 then $
			table.bv=float(strmid(r(j),15,10))-table.mv else $
			table.bv=float(strmid(r(j),8,10))-table.mv
		endif else table.bv=100
		j=where(strpos(r,'flux: J') ge 0) & j=j(0)
		if j ge 0 then begin
			if strpos(r(j),'Vega') ge 0 then $
			table.mj=float(strmid(r(j),15,10)) else $
			table.mj=float(strmid(r(j),8,10))
		endif else table.mj=100
		j=where(strpos(r,'flux: H') ge 0) & j=j(0)
		if j ge 0 then begin
			if strpos(r(j),'Vega') ge 0 then $
			table.mh=float(strmid(r(j),15,10)) else $
			table.mh=float(strmid(r(j),8,10))
		endif else table.mh=100
		j=where(strpos(r,'flux: K') ge 0) & j=j(0)
		if j ge 0 then begin
			if strpos(r(j),'Vega') ge 0 then $
			table.mk=float(strmid(r(j),15,10)) else $
			table.mk=float(strmid(r(j),8,10))
		endif else table.mk=100
		return,table.starid
	endif else begin
		return,''
	endelse
	return,star
;	The rest of this procedure is obsolete
;	
	if lynx then begin
;		Simbad3 call
		spawn,'lynx "http://simbad.u-strasbg.fr/sim-id.pl?protocol=html&Ident=' $
				+target+'" -dump',r
		tag='Basic data :'
		i=strpos(r,tag)
		j=where(i ge 0) & j=j(0)
		if j ge 0 then begin
;		SIMBAD3 found the target
		table.name=target0
		k=strpos(r(j),'--')
		star=strmid(r(j),i(j)+strlen(tag)+1,k-i(j)-strlen(tag)-1)
		if strpos(star,'HD') ge 0 then begin
			words=nameparse(star)
			table.hdn=long(words(1))
			table.starid='HDN'+string(table.hdn,format='(i6.6)')
		endif
		tag='ICRS 2000.0 coordinates'
		i=strpos(r,tag)
		j=where(i ge 0) & j=j(0)+1
		words=nameparse(r(j))
		ra=strjoin(words(0:2),':')
		dec=strjoin(words(3:5),':')
		table.ra=hms2h(ra)
		table.dec=dms2d(dec)
		tag='Proper motion'
		i=strpos(r,tag)
		j=where(i ge 0,c) & j=j(0)+1
		if c eq 1 then begin
			words=nameparse(r(j))
			pmra=float(words(0))
			pmdec=float(words(1))
; 			convert mas/year to arcsec/centennial
			table.pmra=((pmra/10)/15)/cos(table.dec*!pi/180)
			table.pmdec=pmdec/10	
		endif else begin
			pmra=0.
			pmdec=0.
		endelse
		tag='Peculiarities'
		i=strpos(r,tag)
		j=where(i ge 0) & j=j(0)
		words=nameparse(r(j),[' ',','])
		i=isnumeric(words)
		j=where(i eq 1,c)
		if c eq 2 then begin
			mb=float(words(j(0)))
			mv=float(words(j(1)))
			table.mv=mv
			table.bv=mb-mv
		endif
		tag='Spectral type'
		i=strpos(r,tag)
		j=where(i ge 0) & j=j(0)+1
		words=nameparse(r(j))
		spectrum=words(0)
		table.spectrum=spectrum
		return,star
		endif
	endif else begin
;		Simbad3 call
;		spawn,'elinks -dump "http://simbad.u-strasbg.fr/sim-id.pl?protocol=html&Ident='+target+'"',r
;		Simbad4 call
		spawn,'elinks -dump "http://simbad.u-strasbg.fr/simbad/sim-id?output.format=ASCII&Ident='+target+'"',r
		if strpos(r(0),'Identifier not found in the database') ge 0 then begin
			print,'Target not found.'
			return,''
		endif
		if strpos(r(0),'SIMBAD3') ge 0 then begin
;		SIMBAD3 found the target
		tag='Basic data :'
		i=strpos(r,tag)
		j=where(i ge 0) & j=j(0)
		table.name=target0
		k=strpos(r(j),'--')
		star=strmid(r(j),i(j)+strlen(tag)+1,k-i(j)-strlen(tag)-1)
		if strpos(target0,'HD') ge 0 then begin
			words=nameparse(target0)
			table.hdn=long(words(1))
			table.starid='HDN'+string(table.hdn,format='(i6.6)')
		endif
		tag='ICRS 2000.0 coordinates'
		i=strpos(r,tag)
		j=where(i ge 0) & j=j(0)
		words=nameparse(r(j))
		ra=strjoin(words(3:5),':')
		dec=strjoin(words(6:8),':')
		table.ra=hms2h(ra)
		table.dec=dms2d(dec)
		tag='Proper motion'
		i=strpos(r,tag)
		j=where(i ge 0,c) & j=j(0)
		if c eq 1 then begin
			words=nameparse(r(j))
			pmra=0
			pmdec=0
			for k=0,n_elements(words)-1 do begin
				if isnumeric(words(k)) and pmra eq 0 then begin
				pmra=float(words(k))
				pmdec=float(words(k+1))
; 				convert mas/year to arcsec/centennial
				table.pmra=((pmra/10)/15)/cos(table.dec*!pi/180)
				table.pmdec=pmdec/10	
				endif
			endfor
		endif else begin
			pmra=0.
			pmdec=0.
		endelse
		tag='Peculiarities'
		i=strpos(r,tag)
		j=where(i ge 0) & j=j(0)-1
		words=nameparse(r(j),[' ',','])
		i=isnumeric(words)
		j=where(i eq 1,c)
		if c eq 2 then begin
			mb=float(words(j(0)))
			mv=float(words(j(1)))
			table.mv=mv
			table.bv=mb-mv
		endif
		tag='Spectral type'
		i=strpos(r,tag)
		j=where(i ge 0) & j=j(0)
		words=nameparse(r(j))
		spectrum=words(2)
		table.spectrum=spectrum
		return,target0
		endif else begin
;		SIMBAD4 found the target
		table.name=target0
		r=strjoin(r,' ')
		k=strpos(r,'Object')
		words=nameparse(strmid(r,k+6,strlen(r)),'-')
		star=words(0)
;		Identifier
		j=strpos(r,'Identifiers')
		words=nameparse(strmid(r,j(0),strlen(r)),':')
		words=nameparse(words(1))
		j=where(strpos(words,'HD') ge 0)
		if j(0) ge 0 then begin
			table.hdn=long(words(j+1))
			table.starid='HDN'+string(table.hdn,format='(i6.6)')
		endif
;		Coordinates
		j=strpos(r,'Coordinates')
		words=nameparse(strmid(r,j(0),strlen(r)),':')
		words=nameparse(words(1))
		ra=strjoin(words(0:2),':')
		dec=strjoin(words(3:5),':')
		table.ra=hms2h(ra)
		table.dec=dms2d(dec)
;		Proper motions
		j=strpos(r,'Proper') & j=j(0)
		words=nameparse(strmid(r,j,strlen(r)),':')
		words=nameparse(words(1))
; 		Convert mas/year to arcsec/centennial
		if isnumeric(words(0)) then table.pmra=(double(words(0))/150) $
						      /cos(table.dec*!pi/180)
		if isnumeric(words(1)) then table.pmdec=double(words(0))/10
;		Spectral type
		j=strpos(r,'Spectral type')
		words=nameparse(strmid(r,j(0),strlen(r)),':')
		words=nameparse(words(1))
		table.spectrum=strcompress(words(0),/remove_all)
;		Magnitudes
		j=strpos(r,'Flux V') & j=j(0)
		if j ge 0 then words=nameparse(strmid(r,j,strlen(r)),':')
		table.mv=float(words(1))
		j=strpos(r,'Flux B') & j=j(0)
		if j ge 0 then words=nameparse(strmid(r,j,strlen(r)),':')
		table.bv=float(words(1))-table.mv
		j=strpos(r,'Flux J') & j=j(0)
		if j ge 0 then words=nameparse(strmid(r,j,strlen(r)),':')
		table.mj=float(words(1))
		j=strpos(r,'Flux H') & j=j(0)
		if j ge 0 then words=nameparse(strmid(r,j,strlen(r)),':')
		table.mh=float(words(1))
		j=strpos(r,'Flux K') & j=j(0)
		if j ge 0 then words=nameparse(strmid(r,j,strlen(r)),':')
		table.mk=float(words(1))
		return,star
		endelse
	endelse
endelse
;
end
;-------------------------------------------------------------------------------
function simbad_references,target0
;
words=nameparse(target0)
if strpos(target0,'HDN') ge 0 then words=['HD',strmid(target0,3,6)]
if strpos(target0,'HIP') ge 0 then words=['HIP',strmid(target0,3,6)]
target=strjoin(words,'+')
; Simbad4 call (using scripts)
simbad='http://simweb.u-strasbg.fr/simbad/'
script='sim-script?submit=submit+script&script=query+id+'
browser='lynx -dump '
spawn,browser+'"'+simbad+script+target+'"',r
j=where(strpos(r,'liste bibcodes') ge 0,count) & j=j(0)
words=nameparse(r(j),['(',')'])
;
return,fix(words(1))
;
end
;-------------------------------------------------------------------------------
function cri_vlti,obstarg,ra,dec
;
; Tries to find HD, HIP, or USR ID for given star, if not successful, 
; use the 'OBJ' identifier. Uses startable but restores original contents.
;
; Note that all ' ','-', and '_' are removed from the names!
;
common StarBase,StarTable,Notes
;
if n_elements(startable) ne 0 then startable_bck=startable
;
star=obstarg
if strpos(obstarg,'OBJ') lt 0 then $
star=strjoin(nameparse(obstarg,[' ','-','_']))
starid=''
;
; Remove leading CAL or SCI string
if strpos(star,'CAL') eq 0 then star=strmid(star,3)
if strpos(star,'SCI') eq 0 then star=strmid(star,3)
;
hdn=cri(star,'hdn',/lists) & hdn=hdn(0)
;
if hdn le 0 and strmid(obstarg,0,3) eq 'SAO' then $
hdn=cri(long(strmid(star,3,6)),'sao-hdn')
;
if hdn gt 0 then begin
        starid='HDN'+string(hdn,format='(i6.6)')
endif else begin
        usr=cri(star,'usr',/lists) & usr=usr(0)
        if usr gt 0 then begin
                starid='USR'+string(usr,format='(i4.4)')
        endif else begin
		if n_elements(ra) ne 0 and n_elements(dec) ne 0 then $
                starid='OBJ'+esoid(ra,dec)
        endelse
endelse
;
if n_elements(startable_bck) ne 0 then startable=startable_bck
;
return,starid
;
end
;-------------------------------------------------------------------------------
function vlti_station,name,teltype
;
case strmid(name,0,1) of
        'U':teltype=1L
        'A':teltype=4L
        'V':teltype=5L
endcase
;
case strmid(name,1,2) of
        'A0':return,01
        'A1':return,02
        'B0':return,03
        'B1':return,04
        'B2':return,05
        'B3':return,06
        'B4':return,07
        'B5':return,08
        'C0':return,09
        'C1':return,10
        'C2':return,11
        'C3':return,12
        'D0':return,13
        'D1':return,14
        'D2':return,15
        'E0':return,16
        'G0':return,17
        'G1':return,18
        'G2':return,19
        'H0':return,20
        'I1':return,21
        'J1':return,22
        'J2':return,23
        'J3':return,24
        'J4':return,25
        'J5':return,26
        'J6':return,27
        'K0':return,28
        'L0':return,29
        'M0':return,30
        'T1':return,31
        'T2':return,32
        'T3':return,33
        'T4':return,34
endcase
;
end
;-------------------------------------------------------------------------------
function vlti_stationid,stations
;
stations_1=strtrim(stations,2)
;
for i=0,n_elements(stations)-1 do begin
if strmid(stations(i),0,1) eq 'U' $
	then stations_1(i)='UT'+strmid(stations(i),1,1) $
	else stations_1(i)='A'+strmid(stations(i),0,2)
	if strlen(strcompress(stations(i),/remove_all)) eq 1 then $
	stations_1(i)='000'
endfor
;
return,stations_1
;
end
;-------------------------------------------------------------------------------
function npoifile,filename
;
; Extract date portion if one of the following standard NPOI files:
; YYYY-MM-DD.extension[.NNN.NN], with the following extensions below.
; (Raw HDS files have an additional scan.sub-scan extension.)
;
extensions=['.raw','.con','.coh','.cha','.inch']
i=0
repeat begin
	file=strmid(filename,0,strpos(filename,extensions(i)))
	i=i+1
endrep until strlen(file) ne 0 or i eq n_elements(extensions)
if strlen(file) ne 0 then begin
	while strpos(file,'/') ne -1 do $
		file=strmid(file,strpos(file,'/')+1,strlen(file))
endif else begin
	print,'Warning(NPOIFILE): non-standard extension!'
	if strpos(filename,'/') lt 0 then file=strmid(filename,0,10) $
				     else file='YYYY-MM-DD'
endelse
;
return,file
;
end
;-------------------------------------------------------------------------------
function pathname,filespec
;
; Return path in an absolute file name. Includes the last '/'.
;
path=''
if n_elements(filespec) eq 0 then filespec=''
name=filespec
while strpos(name,'/') ne -1 do begin
        path=path+strmid(name,0,strpos(name,'/')+1)
        name=strmid(name,strpos(name,'/')+1,strlen(name))
endwhile
;
if strlen(filespec) eq 0 then begin
	spawn,'pwd',path & path=path(0)
	path=path+'/'
endif
;
return,path
;
end
;-------------------------------------------------------------------------------
function specname,filespec
;
; Return name in file path.
;
n=n_elements(filespec)
if n eq 0 then return,''
names=strarr(n)
for i=0,n-1 do begin
	if strlen(filespec(i)) ne 0 then $
	names(i)=strmid(filespec(i),strlen(pathname(filespec(i))),strlen(filespec(i)))
endfor
;
if n eq 1 then return,names(0) else return,names
;
end
;-------------------------------------------------------------------------------
function finddir,path
;
if n_elements(path) eq 0 then path='.'
default_dirs=['.','..']
;
files=file_search(path+'/*')
index=where(strpos(files,':') ne -1,count)
if count eq 0 then return,default_dirs
dirs=specname(files(index))
for i=0,count-1 do dirs(i)=strmid(dirs(i),0,strlen(dirs(i))-1)
return,[default_dirs,dirs]
;
end
;-------------------------------------------------------------------------------
function safe,command
;
; Make sure no spawn command is used with 'rm *' anywhere in it.
;
if strpos(command,'rm ') ne -1 then begin
	if strpos(command,'*') ne -1 then return,0
endif
;
return,1
;
end
;-------------------------------------------------------------------------------
function breve,b
;
; Return the reversed baseline.
;
return,strmid(b,4,3)+'-'+strmid(b,0,3)
;
end
;-------------------------------------------------------------------------------
function stringof,structure
;
result=''
for i=0,n_tags(structure)-1 do result=strjoin([result,string(structure(0).(i))])
;
return,result
;
end
;-------------------------------------------------------------------------------
function addline,lines,identifier,line,before=before,after=after
;
nl=n_elements(lines)
result=strarr(nl+1)
;
index=where(strpos(lines,identifier) ge 0,count)
if count eq 0 then return,lines
if keyword_set(before) then begin
        index=index(0)
        for i=0,index-1 do result(i)=lines(i)
        result(index)=line
        result(index+1)=lines(index)
endif
if keyword_set(after) then begin
        index=index(count-1)
        for i=0,index do result(i)=lines(i)
        result(index+1)=line
endif
for i=index+2,nl do result(i)=lines(i-1)
;
return,result
;
end
;************************************************************************Block 4
function set_boxes,box_x,box_y,nbox,clear=clear
;
; Lets user place boxes in a plot for various purposes.
; Returns button status to identify user command, as well as
; the coordinates, box_x and box_y, and the number, nbox, of the boxes.
;
if n_elements(clear) eq 0 then clear=0 else clear=clear gt 0
if clear then begin
	if n_elements(nbox) eq 0 then nbox=1
	device,set_graphics_function=6
	for i=0,nbox-1 do plots,box_x(*,i),box_y(*,i),psym=0
	device,set_graphics_function=3
	return,0
endif
;
if n_params() eq 2 then begin
	boxmode=1
	numbox=1
	print,'______________________________________________________'
	print,'Place one box by clicking left mouse button to anchor '
	print,'a corner of the box. Click middle to fix size and to  '
	print,'move box around. Click left button to set the box.'
	print,'Click right button to cancel/reset and return!'
	print,'________________________***___________________________'
endif else begin
	boxmode=0
	numbox=20
	print,'_________________________________________________________'
	print,'You have a maximum of',numbox,' boxes to place! To begin:'
	print,'click left button to anchor a corner of the box. Click   '
	print,'middle button to fix size and to move box around. Click  '
	print,'left button to fix the box.'
	print,'Click right mouse button to exit to edit menu.'
	print,'_________________________***______________________________'
endelse
;
; Box corners start at top left corner and continue clockwise
box_x=fltarr(5,numbox)
box_y=fltarr(5,numbox)
nbox=-1
ierr=0
while((nbox lt numbox-1) and (ierr ne 4))do begin
cursor,x,y,/down & ierr=!err
if ierr eq 1 then begin
	nbox=nbox+1
	box_x(*,nbox)=x
	box_y(*,nbox)=y
;	Use 10 or 6 here
	device,set_graphics_function=6
	while((ierr ne 2) and (ierr ne 4)) do begin
		cursor,x,y,/change & ierr=!err
		plots,box_x(*,nbox),box_y(*,nbox),psym=0
		box_x(1:2,nbox)=x & box_y(2:3,nbox)=y
		plots,box_x(*,nbox),box_y(*,nbox),psym=0
	endwhile
	device,set_graphics_function=3
endif
if ierr eq 2 then begin
	if nbox ge 0 then begin
		device,set_graphics_function=6
		while((ierr ne 1) and (ierr ne 4)) do begin
			x0=x & y0=y
			cursor,x,y,/change & ierr=!err
			plots,box_x(*,nbox),box_y(*,nbox),psym=0
			box_x(*,nbox)=box_x(*,nbox)+x-x0 
			box_y(*,nbox)=box_y(*,nbox)+y-y0
			plots,box_x(*,nbox),box_y(*,nbox),psym=0
		endwhile
		device,set_graphics_function=3
	endif
endif
if nbox ge 0 then begin
	if box_x(1,nbox) lt box_x(0,nbox) then begin
		box_x(*,nbox)=shift(box_x(*,nbox),-2)
		box_x(3,nbox)=box_x(4,nbox)
	endif
	if box_y(2,nbox) gt box_y(1,nbox) then begin
		box_y(*,nbox)=shift(box_y(*,nbox),-2)
		box_y(4,nbox)=box_y(0,nbox)
	endif
endif
endwhile
icom=ierr
nbox=nbox+1
return,icom
;
if boxmode eq 0 then begin
	icom=cw_form('0,LIST,Delete inside boxes|Delete outside boxes|Start over again|Cancel and return,set_value=0,QUIT',title='Question?')
	icom=icom.tag0+1
	if icom.tag0 eq 3 then begin
		device,set_graphics_function=6
		for i=0,nbox do plots,box_x(*,i),box_y(*,i),psym=0
		device,set_graphics_function=3
;   		goto,boxes
	endif
endif else icom=ierr
if icom eq 4 then begin
	device,set_graphics_function=6
	for i=0,nbox do plots,box_x(*,i),box_y(*,i),psym=0
	device,set_graphics_function=3
endif
return,icom
;
end
;-------------------------------------------------------------------------------
function set_points,plotdata_x,plotdata_y,index
;
print,'___________________________'
print,'Click left button on point.'
print,'Click right button to exit!'
print,'___________***_____________'
cursor,x,y,/down
icom=!err
!err=0
dist_x=(plotdata_x-x)/(!x.crange(1)-!x.crange(0))*!d.x_size
dist_y=(plotdata_y-y)/(!y.crange(1)-!y.crange(0))*!d.y_size
dist=(dist_x^2+dist_y^2)
index=where(dist eq min(dist)) & index=index(0)
x=float(plotdata_x(index))
y=float(plotdata_y(index))
; tvcrs does not always work, especially in virtual machines...
; tvcrs,x,y,/data
; ...instead, we display a cross with oplot
device,set_graphics_function=6
ux=[-5,5]
uy=[0,0]
usersym,ux,uy
oplot,[x,!x.crange(0)],[y,!y.crange(0)],psym=8,color=tci(3)
uy=[-5,5]
ux=[0,0]
usersym,ux,uy
oplot,[x,!x.crange(0)],[y,!y.crange(0)],psym=8,color=tci(3)
device,set_graphics_function=3
return,icom
;
end
;-------------------------------------------------------------------------------
function get_ellipse,p,a1,a2,e_parms,mode,phase
;
; e_parms=fltarr(5)
; 0: center x-coordinate
; 1: center y-coordinate
; 2: major axis
; 3: minor axis
; 4: position angle
;
c=[(a1(0)+a2(0))/2,(a1(1)+a2(1))/2]
ca1=a1-c
asqr=total(ca1^2)
if mode eq 1 then begin
        cp=c-p
        cpsqr=total(cp^2)
        xsqr=total(ca1*cp)^2/asqr
        ysqr=cpsqr-xsqr
        bsqr=ysqr/(1-xsqr/asqr)
        b=sqrt(bsqr)
endif else b=e_parms(3)
;
pa=atan(ca1(0),ca1(1))
r=ellipse(sqrt(asqr),b,phase-pa)
x=r*sin(phase)+c(0)
y=r*cos(phase)+c(1)
;
e_parms(0:1)=c
e_parms(2)=sqrt(asqr)
e_parms(3)=b
e_parms(4)=pa
;
return,[[x],[y]]
end
;-------------------------------------------------------------------------------
function set_ellipse,v
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
print,'________________________________________________________'
print,'Use mouse buttons to define apparent ellipse:'
print,'Click left  button to  anchor major axis, right to fix.'
print,'Click middle button to adjust minor axis, right to fix.'
print,'Click right button to exit (major axis will dissappear).'
print,'__________________________***___________________________'
;
e_parms=dblarr(5)
;
ax=dblarr(2)
ay=dblarr(2)
;
n=90*4
phase=dindgen(n+1)*2*pi_circle/n
;
status=''
err1=0
device,set_graphics_function=6
!p.psym=0
;
WHILE ERR1 NE 4 DO BEGIN
;
!err=0
while !err ne 1 and !err ne 2 and !err ne 4 do cursor,x,y,/down
err1=!err
case status of
''  :	begin
     	case err1 of
     	1:	begin
  		status='a'
		ax(*)=x
		ay(*)=y
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		end
	2:	print,'Please anchor a first!'
	else:
	endcase
	end
'aa':	begin
  	case err1 of
	1:	begin
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		dist_x=(ax-x)/(!x.crange(1)-!x.crange(0))*!d.x_size
		dist_y=(ay-y)/(!y.crange(1)-!y.crange(0))*!d.y_size
		dist=(dist_x^2+dist_y^2)
		s=sort(dist)
		s=s([1,0])
		ax=ax(s)
		ay=ay(s)
		ax(1)=x
		ay(1)=y
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		tvcrs,ax(1),ay(1),/data
		end
	2:	begin
		status='ab'
		v=get_ellipse([x,y],[ax(0),ay(0)],[ax(1),ay(1)],e_parms,1,phase)
		plots,v(*,0),v(*,1)
		end
	else:
	endcase	
	end
'ab':	begin
	case err1 of
	1:	begin
		status='el'
		plots,v(*,0),v(*,1)
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
                dist_x=(ax-x)/(!x.crange(1)-!x.crange(0))*!d.x_size
                dist_y=(ay-y)/(!y.crange(1)-!y.crange(0))*!d.y_size
                dist=(dist_x^2+dist_y^2)
                s=sort(dist)
                s=s([1,0])
                ax=ax(s)
                ay=ay(s)
                ax(1)=x
                ay(1)=y
                plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
                tvcrs,ax(1),ay(1),/data
		v=get_ellipse([x,y],[ax(0),ay(0)],[ax(1),ay(1)],e_parms,0,phase)
		plots,v(*,0),v(*,1)
		end
	2:	begin
		plots,v(*,0),v(*,1)
		v=get_ellipse([x,y],[ax(0),ay(0)],[ax(1),ay(1)],e_parms,1,phase)
		plots,v(*,0),v(*,1)
		end
	else:
	endcase
	end
else:
endcase
if status ne '' and err1 ne 4 then begin
err2=0
while err2 ne 4 do begin
	cursor,x,y,/change & err2=!err
	case status of
	'a':	begin
		new_status='aa'
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		ax(1)=x
		ay(1)=y
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		end
	'aa':	begin
		new_status='aa'
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		ax(1)=x
		ay(1)=y
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		end
	'ab':	begin
		new_status='ab'
		plots,v(*,0),v(*,1)
		v=get_ellipse([x,y],[ax(0),ay(0)],[ax(1),ay(1)],e_parms,1,phase)
		plots,v(*,0),v(*,1)
		end
	'el':	begin
		new_status='ab'
		plots,v(*,0),v(*,1)
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		ax(1)=x
		ay(1)=y
		plots,ax,ay
		plots,total(ax)/2,total(ay)/2,psym=4
		v=get_ellipse([x,y],[ax(0),ay(0)],[ax(1),ay(1)],e_parms,0,phase)
		plots,v(*,0),v(*,1)
		end
	endcase
endwhile
status=new_status
endif
!err=err1
;
ENDWHILE
;
plots,ax,ay
device,set_graphics_function=3
return,e_parms
;
end
;-------------------------------------------------------------------------------
function set_region,image,teff
;
r=size(image)
; window,xsize=r(1),ysize=r(2),/free
; tvscl,image
;
e_parms=set_ellipse(v)
n=n_elements(v)
if n eq 0 then return,image
;
maxvy=max(v(*,1))
minvy=min(v(*,1))
maxvx=max(v(*,0))
minvx=min(v(*,0))
;
for i=0,r(2)-1 do begin
	y=float(i)/r(2)
	if y gt minvy and y lt maxvy then begin
	index=where(abs(v(*,1)-y) le 0.02)
	j0=fix(min(v(index,0))*r(1)) > 0
	j1=fix(max(v(index,0))*r(1)) < r(1)
	image(j0:j1,i)=teff
	endif
endfor
;
return,image
;
end
;-------------------------------------------------------------------------------
