;*******************************************************************************
; File: model.pro
;
; Description:
; ------------
; This file contains various functions for the AMOEBA software.
; Please note that except where stated otherwise, all Julian Day
; arguments are JD-2440000!
;
; Block directory:
; ----------------
; Block 1: num_binary,num_star,
;          componentparse,modelparse,wmcc,componentmass,
;	   topbincomp,topcomp,systemcomp,checkcomponents,checkmodel,
;
; Block 2  true2app,true2vel,
;	   mag2flux,modelfluxes,stellarfluxes,
;	   setwdparms,setrocheparms,wdmap,edstrip,gdmap,chmap,mapodize,
;	   stripvis,mapvis,imgvis,provis,rochevis,
;          componentvis,modelvis,componentflux,modelflux,
;	   modelpos,modelvel,binarypos,binaryvel
;	   modelpx,modelm,modelk,modelf,modeldm,
;	   modelchisq
;
;************************************************************************Block 1
function num_binary
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return,0
index=where(strlen(binary_model.component) gt 0,num)
return,num
;
end
;-------------------------------------------------------------------------------
function num_star
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return,0
index=where(strlen(star_model.component) gt 0,num)
return,num
;
end
;-------------------------------------------------------------------------------
function componentparse,component,comp_1,comp_2
;
; For a given component, extract the two subcomponents.
; Example: A-BC -> A,BC.
; Return status: -1: error, 0: normal completion.
;
len=strlen(component)
pos=strpos(component,'-')
comp_1=strmid(component,0,pos)
comp_2=strmid(component,pos+1,len-pos-1)
if strpos(comp_2,'-') ne -1 then begin
	print,'***Error(COMPONENTPARSE): invalid component name!'
	return,-1
endif
return,0
;
end
;-------------------------------------------------------------------------------
function modelparse,component,comp_1,comp_2
;
; For a given component, extract the two subcomponents and replace with
; stars and/or corresponding binary components as defined in model.
; Example: A-BC -> A,B-C.
; Return status: -1: error, 0: normal completion.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if componentparse(component,comp_1,comp_2) ne 0 then return,-1
;
for i=0,num_binary()-1 do begin
	if componentparse(binary_model(i).component,c1,c2) ne 0 then return,-1
	if c1+c2 eq comp_1 then comp_1=binary_model(i).component
	if c1+c2 eq comp_2 then comp_2=binary_model(i).component
endfor
;
flag=0
if strlen(comp_1) gt 1 and strpos(comp_1,'-') eq -1 then flag=1
if strlen(comp_2) gt 1 and strpos(comp_2,'-') eq -1 then flag=2
comp_12=[comp_1,comp_2]
if flag ne 0 then begin
	comp_12=comp_12(flag-1)
	print,'Error(MODELPARSE): missing binary component for '+comp_12+'!'
	return,-1
endif
;
return,0
;
end
;-------------------------------------------------------------------------------
function wmcc,component
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return,component
comp1=''
comp2=''
if strpos(component,'-') ge 0 then begin
	if componentparse(component,comp_1,comp_2) ne 0 then return,component
endif else begin
	comp_1=component
	comp_2=''
	comp2=' '
endelse
;
for i=0,strlen(comp_1)-1 do begin
	index=where(star_model.component eq strmid(comp_1,i,1),count)
	if count gt 0 then comp1=comp1+star_model(index).wmc
endfor
chars=unique(byte(comp1))
if n_elements(chars) lt strlen(comp1) then comp1=string(chars(0))
;
for i=0,strlen(comp_2)-1 do begin
	index=where(star_model.component eq strmid(comp_2,i,1),count)
	if count gt 0 then comp2=comp2+star_model(index).wmc
endfor
chars=unique(byte(comp2))
if n_elements(chars) lt strlen(comp2) then comp2=string(chars(0))
;
if strlen(comp_2) eq 0 then wmccd=comp1 else wmccd=comp1+'-'+comp2
if strlen(comp1) eq 0 or strlen(comp2) eq 0 then wmccd=component
;
return,wmccd
;
end
;-------------------------------------------------------------------------------
function componentmass,comp_1,comp_2,num_body,index_comp
;
; Compute the component masses (=sum of stellar masses)
; Components specified as returned by componentparse, e.g. 'AB', 'C'
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
components=[comp_1,comp_2]
num_body=[strlen(comp_1),strlen(comp_2)]
index_comp=intarr(max(num_body),2)
mass_sum=dblarr(2)
for k=0,1 do begin
	for j=0,num_body(k)-1 do begin
		index=where(strmid(components(k),j,1) eq star_model.component)
		mass_sum(k)=mass_sum(k)+star_model(index(0)).mass
		index_comp(j,k)=index(0)
	endfor
endfor
;
return,mass_sum
;
end
;-------------------------------------------------------------------------------
function topbincomp
;
; Return the highest ranking binary component, i.e. the component which
; contains all stars comprising the system.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
levels=strlen(binary_model.component)
index=where(max(levels) eq levels)
return,binary_model(index(0)).component
;
end
;-------------------------------------------------------------------------------
function topcomp
;
; Return the highest ranking model component, binary or single (1st).
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if strlen(topbincomp()) ne 0 then return,topbincomp()
;
return,star_model(0).component
;
end
;-------------------------------------------------------------------------------
function systemcomp
;
; Return the sum of all stellar components of a system.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
c2=''
if strlen(topbincomp()) ne 0 then begin
	if componentparse(topbincomp(),c1,c2) ne 0 then return,-1
endif else c1=topcomp()
;
return,'System '+c1+c2
;
end
;-------------------------------------------------------------------------------
function checkcomponents,component
;
; This stub of function CHECKCOMPONENTS is included here so that PV-WAVE, by 
; way of compiling it, stores the function address, which is apparently needed 
; when compiling the real thing (see the following function). Otherwise, PV-WAVE
; would complain about an unknown function CHECKCOMPONENTS when trying to 
; execute it. This does not happen, however, with recursive procedures.
;
end
;-------------------------------------------------------------------------------
function checkcomponents,component
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if strlen(component) gt 1 then begin
	index=where(binary_model.component eq component,count)
	if count ne 1 then begin
	   print,'***Error(CHECKCOMPONENTS): component defined more than once!'
	   return,-1
	endif
endif
;
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
components=[comp_1,comp_2]
for i=0,1 do begin
	if strlen(components(i)) eq 1 then begin
		index=where(star_model.component eq components(i),count)
		if count ne 1 then begin
			print,'***Error(CHECKCOMPONENTS): '+ $
				'component undefined or non-unique!'
			return,-1
		endif
	endif else if checkcomponents(components(i)) ne 0 then return,-1
endfor
;
return,0
;
end
;-------------------------------------------------------------------------------
function checkmodel
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then begin
	print,'***Error(CHECKMODEL): gen_model undefined!'
	return,-1
endif
if n_elements(star_model) eq 0 then begin
	print,'***Error(CHECKMODEL): star_model undefined!'
	return,-1
endif
if n_elements(binary_model) eq 0 then begin
	print,'***Error(CHECKDATA): binary_model undefined!'
	return,-1
endif
if total(strlen(star_model.component)) ne n_elements(star_model) then begin
	print,'***Error(CHECKMODEL): some star components composit!'
	return,-1
endif
;
if num_binary() ge 1 then if checkcomponents(topbincomp()) eq -1 then return,-1
;
return,0
;
end
;************************************************************************Block 2
function true2app,jd,o_parms,rho,theta,ra,dec,z
;
; Compute and return right ascension and declination offset of secondary
; relative to primary  for given epoch and orbital elements.
; Originally written by J. Armstrong, based on W. Heintz, Double Stars, p. 34
;
; Note: The RV of whichever component is in the vicinity of the ascending
; node is positive (away from observer) with respect to that of the other 
; component, or wrt its own RV in other parts of the orbit.
;
; Note: if the inclination is < 90, the orbit is prograde, i.e. the 
; PA increases. If i>90, the orbit is retrograde, i.e. PA decreases.
;
; Note: the periastron angle is the one of the primary. It is counted from
; the ascending node in the direction of motion of that component.
; It is changed from the primary to the secondary by adding 180 deg as the
; formulae from Heintz and Green are all assuming the secondary elements.
;
; The following statement was true until Version 6.05:
; Currently, the angle of the ascending node is 180 degrees off.
; Therefore, currently the primary is moving away from the
; observer if the secondary is at the "ascending" node.
;
; Position angle verifications: (this paragraph started in Nov 2008)
; Gamma Per: A-comp. rel. to G-comp. PA=244 deg (most of the time)
; This is according to Pourbaix 1999, and Prieur et al. 2002 (Speckle)
; and Couteau (1987, but he mentions that it is in the 4th quadrant?).
; However, the McAlister measurements give the PA as 60 deg, why?
; Mason et al. 2001 even lists PA=64 deg! Zavala confirmed 244 deg is
; consistent with NPOI observations reduced with OYSTER.
;
; Another check star is 1 Geminorum, which was measured by Hipparcos
; at rho=185 mas, theta=227 deg.
;
; Hint: if you happen to get a negative eccentricity, add 180 to the
; periastron angle and half a period to the epoch to change the sign.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
a=o_parms(0)
e=o_parms(1)
i=o_parms(2)
w=o_parms(3)
s=o_parms(4)
if n_elements(dec) ne 0 then n=precess(o_parms(5),jd2jy(jd+2440000L),ra,dec,1) $
			else n=o_parms(5)
w=(w+!pi) mod (2*!pi)	; change from primary to secondary definition
;
p=o_parms(6)
t=o_parms(7)
;
M_anom=2*pi_circle*((jd-t) mod p)/p
E_anom=M_anom+e*sin(M_anom)+e^2/2*sin(2*M_anom)
for k=0,4 do E_anom=E_anom+(M_anom-E_anom+e*sin(E_anom))/(1-e*cos(E_anom))
T_anom=2*atan(sqrt((1+e)/(1-e))*sin(E_anom/2),cos(E_anom/2))
;
; Apply apsidal motion
ws=w+(jd-t)/s_year*s
;
alpha=T_anom+ws
;
; From R.M. Green, Spherical Astronomy, p. 470
r=((a*(1-e^2))/(1+e*cos(T_anom)))
x=r*(cos(alpha)*sin(n)+sin(alpha)*cos(n)*cos(i))
y=r*(cos(alpha)*cos(n)-sin(alpha)*sin(n)*cos(i))
z=r*sin(alpha)*sin(i)
;
; From Heintz
theta=atan(sin(alpha)*cos(i),cos(alpha))+n
rho=((a*(1-e^2))/(1+e*cos(T_anom)))*sqrt(cos(alpha)^2+sin(alpha)^2*cos(i)^2)
index=where(rho lt 0,count)
if count gt 0 then begin
	theta(index)=theta(index)+pi_circle
	rho(index)=abs(rho(index))
endif
index=where(theta lt 0,count)
if count gt 0 then theta(index)=theta(index)+2*pi_circle
;
; return,[[rho*sin(theta)],[rho*cos(theta)]]
return,[[x],[y]]
;
end
;-------------------------------------------------------------------------------
function true2vel,jd,o_parms
;
; Note that if radial velocities are available for the primary only,
; one should fit M2 only.
;
; Argument of periastron taken to be of the primary component.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
e=o_parms(0)
i=o_parms(1)
w=o_parms(2)
s=o_parms(3)
p=o_parms(4)
t=o_parms(5)
m1=o_parms(6)
m2=o_parms(7)
;
M_anom=2*pi_circle*((jd-t) mod p)/p
E_anom=M_anom+e*sin(M_anom)+e^2/2*sin(2*M_anom)
for k=0,4 do E_anom=E_anom+(M_anom-E_anom+e*sin(E_anom))/(1-e*cos(E_anom))
T_anom=2*atan(sqrt((1+e)/(1-e))*sin(E_anom/2),cos(E_anom/2))
;
; K1, K2 [km/s]
p=p/s_year
k1=2*pi_circle*sin(i)/sqrt(1-e^2)*p^(-1.0/3.0)*m2*(m1+m2)^(-2.0/3.0) $
	*(a_unit/s_year)/86400000L
k2=k1*(m1/m2)
;
; Apply apsidal motion
ws=w+(jd-t)/s_year*s
v1=+k1*(cos(T_anom+ws)+e*cos(ws))
v2=-k2*(cos(T_anom+ws)+e*cos(ws))
return,[[v1],[v2]]
;
end
;-------------------------------------------------------------------------------
function mag2flux,magnitudes
;
return,10^(magnitudes/(-2.5))
;
end
;-------------------------------------------------------------------------------
function modelfluxes,model,lambda
;
; Compute the flux scaling factor based on the magnitude of a component.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; Corrections with magnitudes
if n_elements(gen_model.wavelengths) eq 1 then begin
	c1=lambda*0+mag2flux(model.magnitudes)
	c1=c1(0)
endif else if n_elements(gen_model.wavelengths) eq 2 then begin
	c1=mag2flux(polynom(lambda*1d6, $
			polyfit(gen_model.wavelengths,model.magnitudes,1)))
endif else begin
	c1=mag2flux(spline(gen_model.wavelengths,model.magnitudes,lambda*1d6))
endelse
;
return,c1
;
end
;-------------------------------------------------------------------------------
function stellarfluxes,model,lambda,ld_coeffs
;
; For a given stellar model, return the fluxes for given wavelengths lambda[m]
; as well as limb darkening coefficients according to the mode parameter.
; Note that all stellar disk fluxes are scaled with the squared diameter!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; Default flat flux distribution, uniform disk
if model.teff eq 0 then begin
	ld_coeffs=dblarr(n_elements(lambda))
	fluxes=dblarr(n_elements(lambda))+1
;
; Black body
endif else if model.teff lt 0 then begin
	ld_coeffs=dblarr(n_elements(lambda))
	fluxes=blackbody(abs(model.teff),lambda)
	if model.type eq 0 then begin
;		For uncorrelated flux, scaling with diameter doesn't make sense
		fluxes=fluxes $
		      /double(fluxes(where(abs(lambda-mean(lambda)) $
				    eq min(abs(lambda-mean(lambda))))),0)
	endif else if model.type lt 12 then begin
;		The scaling with diameter allows a more physical modeling
		fluxes=fluxes*model.diameter^2*model.ratio
	endif
;
; Limb-darkening coefficients from Kurucz model atmospheres
endif else if model.type ge 5 and model.type le 7 then begin
	ld_coeffs=kurucoeffs(model,lambda*1d9,fluxes)
	fluxes=fluxes*model.diameter^2*model.ratio
;
; Limb-darkening coefficients from Aufdenberg model atmospheres
endif else if model.type eq 8 then begin
	ld_coeffs=jasoncoeffs(model,lambda*1d9,fluxes)
	fluxes=fluxes*model.diameter^2*model.ratio
;
; Limb-darkening coefficients from Van Hamme
endif else begin
	ld_coeffs=limblinear(model.teff,model.logg,lambda*1d9,fluxes)
	fluxes=fluxes*model.diameter^2*model.ratio
endelse
;
return,fluxes
;
end
;-------------------------------------------------------------------------------
function setwdparms,model,jd,lambda
;
; Initialize parameters for the Wilson-Devinney code.
; jd is the Julian day epoch (-2440000); lambda [m] the wavelength.
; Return the parameters in form of an array.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common AuxData,parallaxes,k2,vsini
;
RAD=180/pi_circle
;
if componentparse(model.component,comp_1,comp_2) ne 0 then return,-1
star1=where(star_model.component eq comp_1) & star1=star1(0)
star2=where(star_model.component eq comp_2) & star2=star2(0)
model1=star_model(star1)
model2=star_model(star2)
;
num_parms=54
lcparms=fltarr(num_parms)
max_spots=100
spotparms=fltarr(4,2,max_spots)
;
ifrad=0			; controls output format (unused)
nref=1			; number of reflections if mref=2
mref=1			; simple treatment of reflection effect
ifsmv1=1 		; star spot control #1 (unused)
ifsmv2=0 		; star spot control #2 (unused)
icor1=1			; unknown at present
icor2=1			; unknown at present
ld=1			; linear cosine law for limb darkening
modeb=model.wdtype 	; -1: X-ray binaries
			; 0: no constraints
			; 2: detached binary
			; 3: overcontact binaries
			; 4: semi-detached binaries
			; 5: Algol-types
			; 6: double-contact
ipb=1  			; 0: compute L2 from T2; 1: decouple L from T
ifat1=0 		; 0 = Blackbody radiation, 1 = stellar atmosphere
ifat2=0 		; see ifat1
n1=10 			; number of latitude rows per hemisphere
n2=10 			; see n1, longitude elements scale with sine of latitude
bperiod=model.period	; orbital period in days
the=0. 			; semi-duration of primary eclipse (unused)
vunit=100. 		; unit for radial velocity input and output (km/s)
phstrt=0. 		; start phase
phstop=-1. 		; stop phase (do only phase phn)
phin=0.01  		; phase increment (we do only phn, however)
e=model.eccentricity	; eccentricity
per=model.periastron 	; in degrees
; 			K1, K2 [km/s]
k1=2*pi_circle*sin(model.inclination/RAD)/sqrt(1-model.eccentricity^2) $
    *(model.period/s_year)^(-1.0/3.0) $
    *model2.mass*(model1.mass+model2.mass)^(-2.0/3.0) $
    *a_unit/s_year/86400/1000
k2=k1*(model1.mass/model2.mass)
a1sini=13751.0*sqrt(1.d0-model.eccentricity^2)*k1*model.period
a2sini=13751.0*sqrt(1.d0-model.eccentricity^2)*k2*model.period
rsun=6.96e5 		; radius of sun [km]
asemima=(a1sini+a2sini)/sin(model.inclination/RAD)/rsun
f1=model1.omega 	; ratio of axial rotation rate to orbital rate
f2=model2.omega
vga=gen_model.rv/vunit	; systemic velocity (in units of vunit)
pshift=0. 		; constant phase shift
xincl=model.inclination	; orbital inclination in degrees
xincl=(90-xincl)+90	; Bug in WD: prograde and retrograde exchanged!
gr1=model1.gr 		; exponent in gravity darkening law
gr2=model2.gr 		; see gr1, convective envelopes: 0.3, radiative: 1.0
tavh=abs(model1.teff)/10000 	; effective temperature in 10000 K
tavc=abs(model2.teff)/10000	; see tavh, make sure tavh/tavc are positive
alb1=model1.albedo	; bolometric albedo
alb2=model2.albedo	; see alb1, convective envelopes: 0.5-1.0, radiative:1.0
poth=model.semimajoraxis/(model1.diameter/2)
			; omega potential = semi-major axis / radius
potc=model.semimajoraxis/(model2.diameter/2)
			; see poth, for second star
rm=model2.mass/model1.mass
			; mass ratio
xbol1=0.4       	; coefficient in bolometric limb darkening law
xbol2=0.4		; see xbol1
ybol1=0.0		; coefficient for logarithmic term
ybol2=0.0		; see ybol1
wl=lambda*1e6		; wavelength in microns
; model1.teff=abs(model1.teff)	; make sure stellarfluxes uses blackbody
; model2.teff=abs(model2.teff)	; make sure stellarfluxes uses blackbody
; hlum=poly(lambda*1e6,model1.flux_fitcoeffs)*4*pi_circle
hlum=stellarfluxes(model1,lambda,xh)
			; monochromatic luminosity
; clum=poly(lambda*1e6,model2.flux_fitcoeffs)*4*pi_circle
clum=stellarfluxes(model2,lambda,xc)
			; see hlum
tlum=hlum+clum
hlum=hlum/tlum
clum=clum/tlum
; xh=poly(lambda*1e6,model1.ld_fitcoeffs)
			; limb-darkening coefficient
; xc=poly(lambda*1e6,model2.ld_fitcoeffs)
			; see xh, for second star
yh=0.0			; see xh coefficient for logarithmic term
yc=0.0			; see yh, (unused if ld=1)
el3=0.0			; third light (unused)
zero=0.0 		; zero point for output magnitudes
factor=1.0		; scaling factor for normalized light column
;
;     In IMLC, phase = 0 when eclipse occurs and arg_periastron = 90.
;     Therefore, we borrowed the following code from MODLOG in FNL
;     to calculate phase of the periastron, which is our zero point.
;
dtr=1.745329e-2
perr=dtr*per
if e eq 0 then perr=1.570796
trc=1.570796-perr
while trc lt 0 do trc=trc+6.283185
while trc ge 6.283185 do trc=trc-6.283185
htrc=0.5*trc
if abs(1.570796-htrc) ge 7.0e-6 and abs(4.712389-htrc) ge 7.0e-6 then $
	ecan=2.0*atan(sqrt((1.0-e)/(1.0+e))*tan(htrc)) else ecan=3.141593
xmc=ecan-e*sin(ecan)
if xmc lt 0 then xmc=xmc+6.283185
phper=1.0-xmc/6.283185
pconj=(xmc+perr)/6.283185-0.25+pshift
while pconj ge 1.0 do pconj=pconj-1.0
while pconj lt 0.0 do pconj=pconj+1.0
phperi=phper+pconj	; phase of periastron
phn=(jd-model.epoch)/bperiod mod 1	; phase of binary
if e eq 0 then phn=phn+(per-90.)/360. else phn=phn+phperi
;
nsp1=0	; Number of spots on 1st star
nsp2=0	; Number of spots on 2nd star
;
; spotparms(0,0,0)=180. 	; Latitude [deg], 0 - 180 
; spotparms(1,0,0)=0.		; Longitude [deg], 0 - 360 
; spotparms(2,0,0)=20.		; Radius [deg]
; spotparms(3,0,0)=0.8		; Temperature factor
;
lcparms(0) = ifrad
lcparms(1) = nref
lcparms(2) = mref
lcparms(3) = ifsmv1
lcparms(4) = ifsmv2
lcparms(5) = icor1
lcparms(6) = icor2
lcparms(7) = ld
lcparms(8) = MODEB
lcparms(9) = IPB
lcparms(10)= IFAT1
lcparms(11)= IFAT2
lcparms(12)= N1
lcparms(13)= N2
lcparms(14)= BPERIOD
lcparms(15)= THE
lcparms(16)= VUNIT
lcparms(17)= PHN
lcparms(18)= PHSTRT
lcparms(19)= PHSTOP
lcparms(20)= PHIN
lcparms(21)= E
lcparms(22)= PER
lcparms(23)= ASEMIMA
lcparms(24)= F1
lcparms(25)= F2
lcparms(26)= VGA
lcparms(27)= PSHIFT
lcparms(28)= XINCL
lcparms(29)= GR1
lcparms(30)= GR2
lcparms(31)= tavh
lcparms(32)= tavc
lcparms(33)= alb1
lcparms(34)= alb2
lcparms(35)= poth
lcparms(36)= potc
lcparms(37)= rm
lcparms(38)= xbol1
lcparms(39)= xbol2
lcparms(40)= ybol1
lcparms(41)= ybol2
lcparms(42)= WL
lcparms(43)= HLUM
lcparms(44)= CLUM
lcparms(45)= xh
lcparms(46)= xc
lcparms(47)= yh
lcparms(48)= yc
lcparms(49)= EL3
lcparms(50)= ZERO
lcparms(51)= FACTOR
lcparms(52)= NSP1
lcparms(53)= NSP2
;
return,lcparms
;
end
;-------------------------------------------------------------------------------
function setrocheparms0,model,lambda,u,v
;
; Return array of parameters needed by the Roche code to compute model.
; This function is for the code V-1.4
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common AuxData,parallaxes,k2,vsini
;
RAD=180/pi_circle
;
rocheparms=fltarr(7)
;
rocheparms(0)=model.omega	; Rotational velocity ( < 1) in units of
				; break-up velocity
rocheparms(1)=((model.diameter/2)/parallaxes.value(0))*(a_unit/r_sun)
				; Stellar radius [Rsun] for zero rotation
rocheparms(2)=model.teff	; Effective temperature at pole for omega=1
rocheparms(3)=model.mass	; Stellar mass, needed to compute gravity
rocheparms(4)=model.tilt/RAD	; tilt of the rotation axis to LOS in degrees
				; Positive tilt moves pole southwards
rocheparms(5)=model.pa		; Pos. angle of the rotation  axis on sky,
				; from N over E
rocheparms(6)=model.diameter	; Polar angular diameter at breakup, 
				; which is 0.8913 of the diameter of the
				; non-rotating star. But acc. to more recent
				; theories, is almost (1-2%) the same, so
				; we just use the angular diameter.
;
return,rocheparms
;
end
;-------------------------------------------------------------------------------
function setrocheparms,model,lambda,u,v
;
; Return array of parameters needed by the Roche code to compute model.
; This function is for the code V-1.3
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common AuxData,parallaxes,k2,vsini
;
RAD=180/pi_circle
;
rocheparms=fltarr(9)
;
rocheparms(0)=model.omega	; rotational velocity ( < 1)  
				; in units of break-up vel.
rocheparms(1)=lambda*1e9	; wavelength in nm
rocheparms(2)=model.tilt	; tilt of the rotation axis to LOS in degrees
				; Positive tilt moves pole southwards
if model.tilt gt 90 then rocheparms(2)=-(180-model.tilt)
rocheparms(3)=atan(u,v)*180/!pi+model.pa
rocheparms(3)=atan(u,v)*180/!pi-model.pa
				; angle of baseline to rotational axis
rocheparms(4)=model.mass	; stellar mass
rocheparms(5)=((model.diameter/2)/parallaxes.value(0))*(a_unit/r_sun)
;				; stellar radius in units of solar radius
rocheparms(6)=model.teff	; effective temperature
rocheparms(7)=model.diameter	; Angular diameter
rocheparms(8)=model.pa	 	; PA of the maj. axis, from N over E
;
return,rocheparms
;
end
;-------------------------------------------------------------------------------
function wdmap,model,epoch,lambda
;
; Setup a call to the Wilson-Devinney code to get a map.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
wdparms=setwdparms(model,epoch,lambda)
;
icc=0L
maxcc=1000
ccc=lonarr(maxcc)
xcc=dblarr(maxcc)
ycc=dblarr(maxcc)
fcc=dblarr(maxcc)
;
status=linknload(!external_lib,'wilson',wdparms,icc,ccc,xcc,ycc,fcc)
i=where(finite(fcc) eq 0,count)
;
ccc=ccc(0:icc-1)
xcc=xcc(0:icc-1)
ycc=ycc(0:icc-1)
fcc=fcc(0:icc-1)
angle=(90-model.ascendingnode)/RAD
rxcc=(-ycc*sin(angle)+xcc*cos(angle))*model.semimajoraxis
rycc=(+xcc*sin(angle)+ycc*cos(angle))*model.semimajoraxis
map=alloc_map(icc,ccc,rxcc,rycc,fcc)
;
return,map
;
end
;-------------------------------------------------------------------------------
function edstrip,model,lambda,u,v
;
; Return strip brightness distribution for elliptical components.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
RAD=180/pi_circle
;
n=299	; Number of strip points
;
case model.type of 
	 9:	begin
		amajor=model.diameter/2
		bminor=amajor*model.ratio
		t=atan(u,v)-model.pa/RAD
		a=(amajor^2*cos(t)^2+bminor^2*sin(t)^2)/(amajor^2*bminor^2)
		b=(2*sin(t)*cos(t)*(bminor^2-amajor^2))/(amajor^2*bminor^2)
		c=(amajor^2*sin(t)^2+bminor^2*cos(t)^2)/(amajor^2*bminor^2)
		y=(findgen(n)-n/2)/n*2*sqrt(amajor^2*cos(t)^2+bminor^2*sin(t)^2)
		b=sqrt((b*y)^2-4*a*(c*y^2-1))
		strip=alloc_strip(n,y,b,t)
		end
endcase
;
return,strip
;
end
;-------------------------------------------------------------------------------
function gdmap,model,lambda,m_maps,ndim=ndim
;
; Obsolete
;
; Setup a call to the 1.3 Roche code to get a gravity darkened
; brightness distribution. Returns a map.
; lambda [m]
;
; This function will store equally sized maps in case
; one is called for with parameters already computed
; before.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common LocalDeane,rochemaps,n_maps,n_dim,rocheparms
;
RAD=180/pi_circle
;
if n_elements(n_dim) eq 0 then n_dim=0
if n_elements(ndim) eq 0 then ndim=128L else ndim=long(ndim)
if ndim ne n_dim then n_maps=0
n_dim=ndim
;
n_maps=0
;
parms=setrocheparms(model,lambda,0,0)
parmstring=strjoin(string(parms))
;
; If we have one or more maps stored, check whether the one is here
if n_elements(n_maps) eq 0 then n_maps=0
if n_maps gt 0 then begin
	index=where(parmstring eq rocheparms,count)
	if count eq 1 then return,rochemaps(index(0))
endif
;
rpb=0.0
;
; x=RA runs from -ndim/2 to ndim/2-1
x=float((lindgen(ndim*ndim) mod ndim) - (ndim/2))
x=-x	; RA positive towards East
; y=Dec runs from -ndim/2 to ndim/2-1
y=float((lindgen(ndim*ndim) / ndim) - (ndim/2))
f=reform(fltarr(ndim,ndim),ndim*ndim)
xs=0.0
is=0L
;
status=linknload(!external_lib,'peterson',parms,ndim,x,y,f,xs,is,rpb)
help,xs,rpb,is
x=reform(x,ndim,ndim)
y=reform(y,ndim,ndim)
f=reform(f,ndim,ndim)
save,x,y,f
;
f=reform(transpose(reform(f,ndim,ndim)),ndim*ndim)
;
; Pixel scale in units of RpB
pfactor=xs/abs(ndim/2-is)
;
; Scale in mas/RpB
rfactor=(rpb/parms(5))*(model.diameter/2)
;
; PA is here the position angle of the major axis, from N over E
angle=(-parms(8)+90)/RAD
rx=(-y*sin(angle)+x*cos(angle))*pfactor*rfactor
ry=(+x*sin(angle)+y*cos(angle))*pfactor*rfactor
map=alloc_map(ndim*ndim,intarr(ndim*ndim),rx,ry,f)
;
; Store the map
if n_elements(m_maps) eq 0 then m_maps=2
if n_maps eq 0 then begin
	rochemaps=replicate(map,m_maps)
	rocheparms=strarr(m_maps)
endif
if n_maps ge m_maps then n_maps=0
rochemaps(n_maps)=map
rocheparms(n_maps)=parmstring
n_maps=n_maps+1
;
return,map
;
end
;-------------------------------------------------------------------------------
function chmap,model,lambda,threshold=threshold
;
; Given an image cube, interpolate image at wavelength lambda [m]
; Returns a map. Sets all pixels to zero which are less than a threshold.
;
; Map coordinates are in mas.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
RAD=180/pi_circle
if n_elements(threshold) eq 0 then threshold=0.001
;
wavelengths=channel_wavelengths
;
if total(channel_images) eq 0 then retall
;
IF n_elements(channel_wavelengths) EQ 1 THEN BEGIN
image=channel_images
ENDIF ELSE BEGIN
i=where(wavelengths ge lambda,count)
if count eq 0 then begin
	image=channel_images(*,*,n_elements(wavelengths)-1)
endif else if count eq n_elements(wavelengths) then begin
	image=channel_images(*,*,0)
endif else begin
	dll=(lambda-wavelengths(i(0)-1))/(wavelengths(i(0))-wavelengths(i(0)-1))
	dlh=(wavelengths(i(0))-lambda)  /(wavelengths(i(0))-wavelengths(i(0)-1))
;
; 	Linearily interpolate images to wavelength desired
	image=channel_images(*,*,i(0)-1)*dlh+channel_images(*,*,i(0))*dll
endelse
ENDELSE
;
; Remove DC component
; image=image-sockel(image)
;
r=size(image)
imsize=r(1:2)
x=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
y=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
x=-x*images_cellsize   ; RA increases to the left (East)
y=+y*images_cellsize
;
index=where(image/max(image) ge threshold,count)
f=reform(image(index),count)
x=x(index)
y=y(index)
icc=lindgen(count)
ccc=icc	; ccc is not used here
map=alloc_map(icc,ccc,x,y,f,imsize(0),imsize(1))
;
return,map
;
end
;-------------------------------------------------------------------------------
function mapodize,radius,lambda
;
; Should be called when computing visibilities from maps in order to exclude
; emission outside the field of view. Takes radius [mas] and lambda[m],
; and uses the telescope diameter [m] stored in GenConfig to return
; apodization factor.
;
; Either radius or lambda can be arrays, but not both.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if genconfig.diameter(0) eq 0 then return,1.0
;
rad2mas=180l*3600l*1000l/!pi
; Compute Airy's disk
airy=(lambda/genconfig.diameter(0))*rad2mas
return,exp(-(2*radius/airy)^2)	; Gaussian approximation (no sidelobes)
return,sinc(radius/airy)^2	; Airy pattern itself
;
end
;-------------------------------------------------------------------------------
function stripvis,strip,u,v
;
; Compute the visibility for a strip at a single coordinate u and v[lambda]. 
; strip positions are in mas.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
uvr=sqrt(u^2+v^2)
arg=2*pi_circle*(uvr*strip.x)*MAS/RAD
;
return,total(strip.b*complex(cos(arg),sin(arg)))
;
end
;-------------------------------------------------------------------------------
function mapvis,map,u,v
;
; Compute the visibility for a map at a single coordinate u and v[lambda]. 
; Map coordinates are in mas. map.x corresponds to RA and increases towards
; East, i.e. left.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
image=reform(map.fcc,map.nx,map.ny)
dx=map.xcc(0)-map.xcc(1) & dx=dx*MAS/RAD
dy=map.ycc(map.nx)-map.ycc(map.nx-1) & dy=dy*MAS/RAD
du=1./(max(map.xcc)*MAS/RAD)
dv=1./(max(map.ycc)*MAS/RAD)
vf=fft(image)
i0=map.nx/2
j0=map.ny/2
vfs=shift(vf,i0,j0)
;
; Transform u,v coordinates into locations
ul=u/du+i0
vl=v/dv+j0
;
return,interpolate(vfs,ul,vl)
;
arg=2*pi_circle*(u*map.xcc+v*map.ycc)*MAS/RAD
;
return,total(map.fcc*complex(cos(arg),sin(arg)))
;
end
;-------------------------------------------------------------------------------
function mapvis,map,u,v
;
; Compute the visibility for a map at a single coordinate u and v[lambda]. 
; Map coordinates are in mas. map.x corresponds to RA and increases towards
; East, i.e. left.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
arg=2*pi_circle*(u*map.xcc+v*map.ycc)*MAS/RAD
;
return,total(map.fcc*complex(cos(arg),sin(arg)))
;
end
;-------------------------------------------------------------------------------
function imgvis,u,v,lambda
;
; Computes visibilities for OYSTER/PEARL "effective temperature" images.
;
; u[lambda],v[lambda],lambda[m]
;
forward_function mft
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
n=n_elements(u)
nl=n_elements(lambda)
;
; nr is the number of non-zero effective temperature regions
ir=where(rt ne 0,nr)
sf=fltarr(nl,nr)
;
for i=0,nr-1 do begin
;	This was the old call for stellar atmospheres
;	ld_coeffs=limblinear(rt(i),rg(i),lambda*1d9,fluxes)
;	Now we call a more general function for SEDs
	fluxes=stellarfluxes( $
		{teff:-rt(i),logg:rg(i), $
		 diameter:1.0,ratio:1.0, $
		 type:13}, $
		 lambda,ld_coeffs)
;	sf(*,i)=fluxes/max(fluxes)	; for Pearl Version 7.06
	sf(*,i)=fluxes
endfor
;
return,mft(cm,complexarr(n),reform(u,n),reform(v,n),(indgen(n) mod nl)+1, $
	rt,sf,/normalize)
;
; Call in Version 7.06
return,mft(cm,complexarr(n),reform(u,n),reform(v,n),rt,sf, $
	(indgen(n) mod nl)+1,/normalize)
;
end
;-------------------------------------------------------------------------------
function provis_f1,x
;
common LocalProvis,r,a
;
return,poly(x,r)
;
end
;-------------------------------------------------------------------------------
function provis_f2,x
;
common LocalProvis,r,a
;
return,beselj(a*sqrt(1-x^2),0)*provis_f1(x)
;
end
;-------------------------------------------------------------------------------
function provis,profile,mu,arg
;
; Compute the visibility of a stellar atmosphere specified with a profile
; as a function of mu=cos(theta), mu=1 at the center (see Wittkowski 2003):
; F_LD(lambda)=Int_[0,1] I_lambda(mu) J0(2pi*r_LD*B/lambda*sqrt(1-mu^2)) mu dmu
; To normalize, divide by Int_[0,1] I_lambda(mu) mu dmu.
; sqrt(1-mu^2)=sin(theta), the projected radius
;
common LocalProvis,r,a
;
f1=(mu*profile)/profile(n_elements(mu)-1)
f2=beselj(arg*sqrt(1-mu^2),0)*f1
;
return,int_tabulated(mu,f2)/int_tabulated(mu,f1)
;
; Obsolete
a=arg
r=poly_fit(mu,f1,3)
return,qromb('provis_f2',0,1,/double)/qromb('provis_f1',0,1,/double)
;
end
;-------------------------------------------------------------------------------
function rochevis,model,lambda,u,v,fluxes,init=init
;
; Prepare a call to rochevis, which computes visibilities for a Roche sphere.
; Lambda[m], u, and v should be vectors.
;
common LimbBase,limb_data
common LocalRocheVis,LDinit
common AuxData,parallaxes,k2,vsini
;
if n_elements(limb_data) eq 0 then $
restore,!atmospheres_dir+'vanhamme/limbdata.xdr'
;
rocheparms=setrocheparms0(model,lambda,u,v)
;
lam=float(lambda)
nl=n_elements(lam)
vr=fltarr(nl)
vi=vr
fluxes=fltarr(nl)
um=float(u)*lam		; convert u,v coordinates in meters
vm=float(v)*lam
vsi=0.0
;
lam=lam*1e9		; convert to nm
;
; Prepare LD data
if n_elements(LDinit) eq 0 then LDinit=1
if LDinit ne 0 then begin
	case model.model of
		'Log' :LDinit=2l
		'Sqrt':LDinit=3l
	 	else  :LDinit=1L
	endcase
endif
if n_elements(init) ne 0 then LDinit=long(init)
if LDinit gt 0 then begin
	afTeff=float(unique(limb_data.t))
	iTeffMax=n_elements(afTeff)
	afLogG=reverse(float(unique(limb_data.g)))
	iLogGMax=n_elements(afLogG)
	afWaves=float(unique(limb_data.w))
	iWavesMax=n_elements(afWaves)
	afLD=fltarr(3,iTeffMax,iLogGMax,iWavesMax)
	for i=0,n_elements(limb_data.m)-1 do begin
		index=where(limb_data.n eq limb_data.m(i))
		iT=where(afTeff eq limb_data.t(i)) & iT=iT(0)
		iG=where(afLogG eq limb_data.g(i)) & iG=iG(0)
		case LDinit of
		1: afLD(0,iT,iG,*)=limb_data.u(index)
		2: begin
		   afLD(0,iT,iG,*)=limb_data.x(index)
		   afLD(1,iT,iG,*)=limb_data.y(index)
		   end
		3: begin
		   afLD(0,iT,iG,*)=limb_data.a(index)
		   afLD(1,iT,iG,*)=limb_data.b(index)
		   end
		endcase
		afLD(2,iT,iG,*)=(limb_data.f(index)/1e0) $
			       /blackbody(afTeff(iT),afWaves/1e9)
	endfor
	aiLogG=lonarr(iTeffMax)
	for i=0,iTeffMax-1 do $
		aiLogG(i)=n_elements(where(limb_data.t eq afTeff(i)))
endif else begin
	afTeff=0.
	iTeffMax=0l
	afLogG=0.
	iLogGMax=0l
	afWaves=0.
	iWavesMax=0l
	afLD=0.
	aiLogG=0l
endelse
;
status=linknload(!external_lib,'peterson',rocheparms,nl,lam,vr,vi,fluxes, $
		um,vm, $
		LDinit,iWavesMax,iTeffMax,iLogGMax,aiLogG, $
		afLD,afLogG,afWaves,afTeff,vsi)
LDinit=0l
;
if n_elements(vsini) ne 0 then vsini.valuem=vsi
;
return,complex(vr,vi)
;
end
;-------------------------------------------------------------------------------
function componentvis,component,time,lambda,fluxes,u,v
;
; This stub of function COMPONENTVIS is included here so that PV-WAVE, by way
; of compiling it, stores the function address, which is apparently needed when
; compiling the real thing (see the following function). Otherwise, PV-WAVE
; would complain about an unknown function COMPONENTVIS when trying to execute
; it. This does not happen, however, with recursive procedures.
;
; Actually, the above is not true! The real reason is that PV-WAVE and IDL
; need to know whether f(arg) is a function or an array element reference.
;
end
;-------------------------------------------------------------------------------
function componentvis,component,time,lambda,fluxes,u,v
;
; Compute the complex visibility of a component. If double, add sub-components.
; Sub-components may themselves be binary, single stars, or both. The mode 
; parameter determines which astrophysical scenario/method is to be used for 
; the visibility computation of the components/sub-components.
;
; Input parameters:
; component: 		A, B, C,..., A-B, AB-C, AB-CD,...
; time [s]: 		dblarr(num_scan)
; lambda [m]: 		dblarr(num_wave)
; u,v [lambda]: 	dblarr(num_wave,num_scan)
;
; Output parameter:
; fluxes:		dblarr(numwave)
;
; If only one value for time and wavelength is passed, assume u and v
; are arrays corresponding to an arbitrary coverage.
;
; Note: this function actually returns a correlated flux, as complex number
; so that components can be added with the proper phase term. After addition
; of all correlated fluxes, the visibility is obtained by normalizing with the
; total flux. That is also why this function returns the total flux contribution.
;
; Modes renamed with version 6.12 (2 Aug 2007)
; 10->8; 11->9; 12->10; 14->11; 8->12; 16->13; 13->obsolete; 9->14; 17->16
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common RandomModel,RandomSeed
common AuxData,parallaxes,k2,vsini
common LocalComponentVis,old_model,new_model,old_image,new_image,old_sed,new_sed
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
; Initialize constants
factor=(pi_circle^2)/(180L*3600L*1000L)
mas2rad=pi_circle/(180L*3600L*1000L)
;
num_scan=n_elements(time)
num_wave=n_elements(lambda)
;
; Get length of component to find out whether component is single or double
len=strlen(component)
;
; Single star...................................................................
if len eq 1 then begin
;
; Find applicable model
index=where(star_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): non-unique component definition!'
	return,-1
endif
model=star_model(index)
;
; Allow rotation and stretch of all components: PA > 0 rotates counter-clockwise
; Ratio < 1 compresses E-W size of object
vr=v*cos(model.pa/RAD)+u*sin(model.pa/RAD)
ur=model.ratio*(u*cos(model.pa/RAD)-v*sin(model.pa/RAD))
;
; Get SED; all astrophysical types 12 and up compute their own fluxes
if model.type le 11 then fluxes=stellarfluxes(model,lambda,ld_coeffs) $
		    else fluxes=fltarr(num_wave)+1
;
; If an SED file (XDR format) has been specified, it overrides
if strlen(model.sed) gt 0 then begin
;	Read spectrum from XDR file: l[m], f
	restore,model.sed
	fb=f
;	Reduce resolution to half (Nyquist!) of that of the lambda array
	if n_elements(lambda) ge 2 then begin
		r=4*fix(abs(lambda(1)-lambda(0))/(l(1)-l(0)))
		if r gt 1 then fb=box(f,r)
	endif
	si=sort(lambda)
	fluxes=spline(l,fb,lambda(si))
	fluxes(si)=fluxes
endif
;
; Apply correction to fluxes specified by magnitudes
fluxes=fluxes*modelfluxes(model,lambda)
;
; Replicate fluxes to same dimensions as (u,v)
flux=dblarr(num_wave,num_scan)
for j=0,num_scan-1 do flux(*,j)=fluxes 
if n_elements(flux) eq 1 then flux=flux(0)
;
; Compute contribution due to a star spot
if model.spot(3) ne 0 then begin
	spot_model=model
	spot_model.teff=model.spot(0)*model.teff
	spot_model.diameter=model.spot(3)
	spot_fluxes=stellarfluxes(spot_model,lambda) $
		   -fluxes*(spot_model.diameter/model.diameter)^2
	xy=dblarr(2)
	xy(0)=model.spot(1)*sin(model.spot(2)/RAD)*MAS/RAD
	xy(1)=model.spot(1)*cos(model.spot(2)/RAD)*MAS/RAD
	spot_phase=complexarr(num_wave,num_scan)
	if n_elements(spot_phase) eq 1 then begin
		spot_phase=exp(2*pi_circle*i_complex*(u*xy(0)+v*xy(1)))
	endif else begin
		for j=0,num_wave-1 do $
   	    	spot_phase(j,*)=exp(2*pi_circle*i_complex $
				*(u(j,*)*xy(0)+v(j,*)*xy(1)))
	endelse
	UD=dblarr(num_wave,num_scan)
	UD(*,*)=model.spot(3)
	if n_elements(UD) eq 1 then UD=UD(0)
	arg=pi_circle*mas2rad*sqrt(u^2+v^2)*UD
	index=where(arg eq 0,count)
	if count gt 0 then arg(index)=1
	spot_amp=2*beselj(arg,1)/arg
	if count gt 0 then spot_amp(index)=1
	spot_flux=flux
	for j=0,num_scan-1 do spot_flux(*,j)=spot_fluxes 
endif else begin
	spot_amp=0
	spot_flux=0
	spot_phase=0
endelse
;
; Derive component phase from RA Dec offsets
phase=exp(2*pi_circle*i_complex*(u*model.xoff+v*model.yoff)*mas2rad)
;
case model.type of
;
;	Uncorrelated flux component
      0:begin
	return,complexarr(num_wave,num_scan)
	end
;
;	Uniform (elliptical) disk independent of wavelength, input diameter UD
;	The tmp1/tmpb elliptical code is borrowed from "difmap" by M. Shepherd.
      1:begin
	UD=dblarr(num_wave,num_scan)
	UD(*,*)=model.diameter
	if n_elements(UD) eq 1 then UD=UD(0)
	arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
	index=where(arg eq 0,count)
	if count gt 0 then arg(index)=1
	visamp=2*beselj(arg,1)/arg
	if count gt 0 then visamp(index)=1
;	save,lambda,fluxes,filename='fluxes_comp_1.xdr'
;	save,lambda,spot_fluxes,filename='fluxes_spot_1.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Uniform (elliptical) disk scales with wavelengths, input diameter LD
      2:begin
	UD=dblarr(num_wave,num_scan)
	UD1=sqrt((1-7*ld_coeffs/15)/(1-ld_coeffs/3))*model.diameter
	for j=0,num_wave-1 do UD(j,*)=UD1(j)
	if n_elements(UD) eq 1 then UD=UD(0)
	arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
	visamp=2*beselj(arg,1)/arg
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, linear law, input diameter LD
;	From: R. Hanbury Brown et al. 1974 MNRAS 167, 475
      3:begin
	alpha=dblarr(num_wave,num_scan)
	beeta=dblarr(num_wave,num_scan)
	for i=0,num_scan-1 do begin
		alpha(*,i)=1-ld_coeffs
		beeta(*,i)=ld_coeffs
	endfor
	if n_elements(alpha) eq 1 then alpha=alpha(0)
	if n_elements(beeta) eq 1 then beeta=beeta(0)
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	index=where(arg eq 0,count)
	if count gt 0 then arg(index)=1
        visamp=(alpha*beselj(arg,1)/arg+beeta*sqrt(pi_circle/2)* $
                sqrt(2/(pi_circle*arg))*(sin(arg)/arg-cos(arg))/ $
                sqrt(arg*arg*arg))/(alpha/2+beeta/3)
	if count gt 0 then visamp(index)=1
;	save,lambda,fluxes,filename='fluxes_comp_3.xdr'
;	save,lambda,spot_fluxes,filename='fluxes_spot_3.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Limb-darkened disk, logarithmic law, input diameter LD
      4:begin
	modvis=complexarr(num_wave,num_scan)
	maps=limbmaps(model,lambda*1d9)
	for j=0,num_wave-1 do begin
		fcc_total=total(maps(j).fcc)
		for i=0,num_scan-1 do begin
			modvis(j,i)=mapvis(maps(j),u(j,i),v(j,i))/fcc_total
		endfor
	endfor
	return,modvis*flux*phase+complex(spot_amp)*spot_flux*spot_phase
        end
;
;	Limb-darkened disk, linear law, input diameter LD, from Kurucz
;	Note: type number hard coded in kurugrid (limb.pro)
      5:begin
	alpha=dblarr(num_wave,num_scan)
	beeta=dblarr(num_wave,num_scan)
	for i=0,num_scan-1 do begin
		alpha(*,i)=1-ld_coeffs
		beeta(*,i)=ld_coeffs
	endfor
	if n_elements(alpha) eq 1 then alpha=alpha(0)
	if n_elements(beeta) eq 1 then beeta=beeta(0)
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	visamp=(alpha*bessjy8(arg,1)/arg $
	        +beeta*sqrt(pi_circle/2)*bessjy8(arg,1.5)/arg^1.5) $
	       /(alpha/2+beeta/3)
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, Hestroffer law, input diameter LD, from Kurucz
;	Note: type number hard coded in kurugrid (limb.pro)
      6:begin
;	arg=pi_circle*model.diameter*mas2rad*sqrt(u^2+v^2)
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	arg_nu=1+(ld_coeffs/2)
	gaama=dblarr(num_wave,num_scan)
	bessl=dblarr(num_wave,num_scan)
	delta=dblarr(num_wave,num_scan)
	for j=0,num_wave-1 do begin
		gaama(j,*)=gamma(arg_nu(j)+1)
		bessl(j,*)=bessjy8(arg(j,*),arg_nu(j))
		delta(j,*)=(arg(j,*)/2)^arg_nu(j)
	endfor
	if n_elements(gaama) eq 1 then gaama=gaama(0)
	if n_elements(bessl) eq 1 then bessl=bessl(0)
	if n_elements(delta) eq 1 then delta=delta(0)
	visamp=gaama*(bessl/delta)
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, visibility from stellar atmosphere profile
;	Kurucz models
      7:begin
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	profiles=kurufluxes(model,lambda,mu)
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		visamp=fltarr(num_uv)
		for i=0L,num_uv-1 do begin
			visamp(i)=provis(profiles,mu,arg(i))
		endfor
	endif else begin
		for j=0,num_wave-1 do begin
		for i=0,num_scan-1 do begin
			visamp(j,i)=provis(profiles(*,j),mu,arg(j,i))
		endfor
		endfor
	endelse
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, visibility from stellar atmosphere profile
;	Models from Jason Aufdenberg
      8:begin
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	profiles=jasonfluxes(model,lambda,mu)
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		visamp=fltarr(num_uv)
		for i=0L,num_uv-1 do begin
			visamp(i)=provis(profiles,mu,arg(i))
		endfor
	endif else begin
		for j=0,num_wave-1 do begin
		for i=0,num_scan-1 do begin
			visamp(j,i)=provis(profiles(*,j),mu,arg(j,i))
		endfor
		endfor
	endelse
	return,complex(visamp)*flux+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Elliptical uniform disk
;	This is a numerical version of the analytical code for mode=1
      9:begin
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		modvis=complexarr(num_uv)
		for i=0L,num_uv-1 do begin
			strip=edstrip(model,lambda,u(i),v(i))
			fcc_total=total(strip.b)
			modvis(i)=stripvis(strip,u(i),v(i))/fcc_total
		endfor
	endif else begin
		for i=0,num_scan-1 do begin
		for j=0,num_wave-1 do begin
			strip=edstrip(model,lambda(j),u(j,i),v(j,i))
			fcc_total=total(strip.b)
			modvis(j,i)=stripvis(strip,u(j,i),v(j,i))/fcc_total
		endfor
		endfor
	endelse
	return,modvis*flux*phase
        end
;
;	Gaussian elliptical disk
;	Reference: "difmap" software by M. Shepherd.
     10:begin
	GD=dblarr(num_wave,num_scan)
	GD(*,*)=model.diameter
	if n_elements(GD) eq 1 then GD=GD(0)
	uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
	index=where(uvr gt 5,count)
	if count gt 0 then uvr(index)=5
	visamp=exp(-0.3606737602*uvr*uvr)
	if count gt 0 then visamp(index)=0
; 	save,lambda,fluxes,filename='fluxes_comp_10.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Pearson disk (FFT of which is exponential)
     11:begin
	modvis=complexarr(num_wave,num_scan)
;
;	Create image with Pearson radial profile
	num=41
	map=indgen(num*num)
	xcc=float((map mod num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
	ycc=float((map  /  num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
;	Make clumpy
	nc=20
	cr=fltarr(nc)+0.1	; clump radii
	if RandomSeed(0) eq 0 then do_plot=1 else do_plot=0
	xc=randomu(RandomSeed,nc)*2-1
	yc=randomu(RandomSeed,nc)*2-1
	xc=[xc,0]
	yc=[yc,0]
	cr=[cr,0.1]	; add central clump
	nc=nc+1
	index=intarr(n_elements(map))
	for k=0,nc-1 do begin
		jndex=where(sqrt((xcc-xc(k))^2+(ycc-yc(k))^2) lt cr(k))
		index(jndex)=1
	endfor
	xcc=xcc(where(index eq 1))
	ycc=ycc(where(index eq 1))
; 	Compute radius squared
	r=sqrt(xcc^2+ycc^2)
; 	Use only circular region with r<1
	index=where(r lt 1)
	xcc=xcc(index)
	ycc=ycc(index)
	r=r(index)
	n=n_elements(r)
	if do_plot then plot,xcc,ycc,psym=3
;	Create array of to hold maps
	map=alloc_map(n,lonarr(n),fltarr(n),fltarr(n),fltarr(n))
	maps=replicate(map,num_wave)
;	Pearson width = 1.5 (1 = Lorentzian, inf. = Gaussian)
	a=[1.0,1.5,20.0]
	for j=0,num_wave-1 do begin
		fcc=fluxes(j)*pearson_funct(r*a(2)*10,a)
		maps(j).xcc=xcc*model.diameter/2
		maps(j).ycc=ycc*model.diameter/2
		maps(j).fcc=fcc
	endfor
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
		fcc_total=total(maps(0).fcc)
                for i=0L,num_uv-1 do begin
		modvis(i)=mapvis(maps(0),u(i),v(i))/fcc_total
		endfor
;	For model
	endif else begin 
		for j=0,num_wave-1 do begin
		fcc_total=total(maps(j).fcc)
		for i=0,num_scan-1 do begin
;		modvis(j,i)=mapvis(maps(j),u(j,i),v(j,i))/fcc_total
		modvis(j,i)=mapvis(maps(j),ur(j,i),vr(j,i))/fcc_total
		endfor
		endfor
	endelse
	return,modvis*flux*phase+complex(spot_amp)*spot_flux*spot_phase
        end
;
; 	Image cube (third axis: wavelength)
     12:begin
;	If diameter is specified, it is the clean beam
	if model.diameter gt 0 then begin
		vr=v
		ur=u
	endif
	if strlen(model.model) gt 0 then readimages,model.model
	if n_elements(old_model) eq 0 then old_model=''
	if n_elements(old_image) eq 0 then old_image=''
	if n_elements(old_sed) eq 0 then old_sed=''
	old_model=''	; We currently compute always
	new_model=stringof(model)
	new_image=model.model
	new_sed=model.sed
	IF new_model EQ old_model AND new_image EQ old_image $
				  AND new_sed EQ old_sed THEN BEGIN
		print,'Restoring previously saved model data...'
		restore,'vis_comp_12.xdr'
	ENDIF ELSE BEGIN
	old_model=new_model
	old_image=new_image
	old_sed=new_sed
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		map=chmap(model,lambda)
		radius=sqrt(map.xcc^2+map.ycc^2)
		map.fcc=map.fcc*mapodize(radius,lambda(0)) $
			       *modelfluxes(model,lambda(0))
		num_uv=n_elements(u)
		modvis=complexarr(num_uv)
		fcc_total=total(map.fcc)
		for i=0L,num_uv-1 do begin
			modvis(i)=mapvis(map,ur(i),vr(i))/fcc_total
		endfor
		if strlen(model.sed) eq 0 then begin
			fluxes=fcc_total*modelfluxes(model,lambda)
			flux=fluxes
		endif
	endif else begin
;		For model
		for i=0,num_scan-1 do begin
		for j=0,num_wave-1 do begin
			map=chmap(model,lambda(j))
			radius=sqrt(map.xcc^2+map.ycc^2)
			map.fcc=map.fcc*mapodize(radius,lambda(j)) $
				       *modelfluxes(model,lambda(j))
			fcc_total=total(map.fcc)
			modvis(j,i)=mapvis(map,ur(j,i),vr(j,i))/fcc_total
			if strlen(model.sed) eq 0 then begin
				fluxes(j)=fcc_total*modelfluxes(model,lambda(j))
				flux(j,i)=fluxes(j)
			endif
		endfor
		endfor
;		save,lambda,fluxes,filename='fluxes_comp_12.xdr'
	endelse
;	save,fluxes,flux,modvis,filename='vis_comp_12.xdr'
	ENDELSE
;
;	Correct for restoring beam effect if diameter is specified
	if model.diameter gt 0 then begin
		GD=dblarr(num_wave,num_scan)
		GD(*,*)=model.diameter
		vr=v*cos(model.pa/RAD)+u*sin(model.pa/RAD)
		ur=model.ratio*(u*cos(model.pa/RAD)-v*sin(model.pa/RAD))
		if n_elements(GD) eq 1 then GD=GD(0)
		uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
		index=where(uvr gt 5,count)
		if count gt 0 then uvr(index)=5
		visamp=exp(-0.3606737602*uvr*uvr)
		if count gt 0 then visamp(index)=0.01
		modvis=modvis/visamp
	endif
;
	return,modvis*flux*phase
        end
;
; 	"Effective temperature" PEARL image
     13:begin
	if model.diameter gt 0 then begin
		vr=v
		ur=u
	endif
	if strlen(model.model) gt 0 then readimage,model.model
	fluxes=fltarr(num_wave)+1
	flux=dblarr(num_wave,num_scan)+1
	modvis=complexarr(num_wave,num_scan)
	modvis(*)=imgvis(ur,vr,lambda)
;
;	Correct for restoring beam effect
	if model.diameter gt 0 then begin
		GD=dblarr(num_wave,num_scan)
		GD(*,*)=model.diameter
		if n_elements(GD) eq 1 then GD=GD(0)
		uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
		index=where(uvr gt 5,count)
		if count gt 0 then uvr(index)=5
		visamp=exp(-0.3606737602*uvr*uvr)
		if count gt 0 then visamp(index)=0.01
		modvis=modvis/visamp
	endif
;
	return,modvis*phase
        end
;
; 	Rotating star, using image from Roche code V-1.3
;	Obsolete
;    14:begin
;	modvis=complexarr(num_wave,num_scan)
;	for j=0,num_wave-1 do begin
;		map=gdmap(model,lambda(j),num_wave)
;		fcc_total=total(map.fcc)
;		for i=0,num_scan-1 do begin
;			modvis(j,i)=mapvis(map,u(j,i),v(j,i))/fcc_total
;		endfor
;	endfor
;	return,modvis*flux+complex(spot_amp)*spot_flux*spot_phase
;        end
;
; 	Rotating star, using Roche code V-1.4
     14:begin
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
		num_uv=n_elements(u)
		modvis=complexarr(num_uv)
		modvis(*)=rochevis(model,fltarr(num_uv)+lambda,u,v,fluxes)
		flux(0)=fluxes(0)
	endif else begin
		for i=0,num_scan-1 do begin
		modvis(*,i)=rochevis(model,lambda,u(*,i),v(*,i),fluxes)
		flux(*,i)=fluxes 
		endfor
	endelse
	fluxes=fluxes*modelfluxes(model,lambda)
	flux=flux*modelfluxes(model,lambda)
	return,modvis*flux*phase
        end
;
;	Ring or disk with hole, and temperature profile (Hillenbrand 1992),
;	also used by Malbet et al. (2005, "temperature gradient model").
;	T  ~ (radius/r_min)^-alpha, alpha should be -3/4
     15:begin
	num_ring=500
	r_ring=((findgen(num_ring)/num_ring)*model.width+1) $
		*(model.diameter/2)
	alpha=model.alpha
;	if alpha eq 0 then alpha=-3./4
	teff0=abs(model.teff)
	teff1=abs(model.teff)/(model.diameter/2)^alpha
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
		num_uv=n_elements(u)
		visamp=complexarr(num_uv)
	endif
	fluxes=fltarr(num_wave)
	for k=0,num_ring-1 do begin
		arg=pi_circle*r_ring(k)*2*mas2rad*sqrt(vr^2+ur^2)
		model.teff=-teff1*r_ring(k)^alpha
		f=stellarfluxes(model,lambda,ld_coeffs)*2*!pi*r_ring(k) $
		 *modelfluxes(model,lambda)
		fluxes=fluxes+f
		flux=dblarr(num_wave,num_scan)
		for j=0,num_scan-1 do flux(*,j)=f
		if n_elements(flux) eq 1 then flux=flux(0)
		visamp=visamp+flux*beselj(arg,0)
	endfor
;	save,lambda,fluxes,filename='fluxes_comp_15.xdr'
	model.teff=teff0
	return,complex(visamp)*phase
	end
;
;	Result file (*.i###) from DUSTY or other shell simulation
     16:begin
	arg=pi_circle*2*mas2rad*sqrt(vr^2+ur^2)
;	Read and decode file, last # line before data must contain wavelengths
	readshell,model.model,l_shell,r_shell,f_shell
	nl=n_elements(l_shell)
	nr=n_elements(r_shell)
	r_shell=r_shell*abs(model.diameter/2)	; Scale with diameter/2
;	Extract rows corresponding to central star
;	We assume that these have the highest intensities!
	tf=total(f_shell,1)
	index_s=where(tf/max(tf) gt 0.99,count_s)
	num_ring=nr-count_s
	visamp_shell=fltarr(num_wave,num_scan)
	if n_elements(visamp_shell) eq 1 then begin
		num_uv=n_elements(u)
		visamp_shell=complexarr(num_uv)
	endif
	fluxes=fltarr(num_wave)
	flux=dblarr(num_wave,num_scan)
;
;	V(q)/F(q)/F(0); F(q)=2pi Int(I*J0(2pi*q*x)*x*dx)
	x0=r_shell(count_s:nr-1)
	f0=dblarr(num_wave,nr)
;	Interpolate fluxes onto lambda grid
	for k=0,nr-1 do begin
		index=where(lambda ge l_shell(0) and lambda le l_shell(nl-1))
		f0(index,k)=interpol(f_shell(*,k),l_shell,lambda(index))
;		Extrapolate to smaller lambda as a constant
		index=where(lambda lt l_shell(0),count)
		if count gt 0 then f0(index,k)=f_shell(0,k)
;		Extrapolate to longer lambda as a constant
		index=where(lambda gt l_shell(nl-1),count)
		if count gt 0 then f0(index,k)=f_shell(nl-1,k)
	endfor
;	Integrate rings other than those of the central star
	f=f0(*,count_s:nr-1)
	for j=0,num_wave-1 do begin
		y0=f(j,*)*2*pi_circle*x0 & y0=reform(y0)
;		y0=gsmooth(x0,y0,15)
		y0=mapodize(x0,lambda(j))*y0*modelfluxes(model,lambda(j))
		maxrad=max(x0-x0(0)) < 1000	; Maximum radius 1"
		x1=findgen(1000)/1000*maxrad+x0(0)
		y1=interpol(y0,x0,x1)
;		y1=spline(x0,y0,x1)
;		fluxes(j)=int_tabulated(x0,y0,/double)
		fluxes(j)=int_tabulated(x1,y1,/double)
		for i=0,num_scan-1 do begin
;			y=beselj(x0*arg(j,i),0)*y0
			y=beselj(x1*arg(j,i),0)*y1
;			visamp_shell(j,i)=int_tabulated(x0,y,/double)
			visamp_shell(j,i)=int_tabulated(x1,y,/double)
			flux(j,i)=fluxes(j)
		endfor
	endfor
;
;	Add back transform of central star
	if model.diameter gt 0 then begin
		UD=dblarr(num_wave,num_scan)
		UD(*,*)=r_shell(count_s)*2
		if n_elements(UD) eq 1 then UD=UD(0)
		arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
		index=where(arg eq 0,count)
		if count gt 0 then arg(index)=1
		visamp_star=2*beselj(arg,1)/arg
		if count gt 0 then visamp_star(index)=1
;		f=total(f_shell(*,0:count_s-1),2)/count_s
;		f=interpol(f,l_shell,lambda)*!pi*(UD(0)/2)^2
		f=f0(*,0)*!pi*(UD(0)/2)^2
		f=f*modelfluxes(model,lambda)
		fluxes=fluxes+f
		flux=dblarr(num_wave,num_scan)
		for j=0,num_scan-1 do flux(*,j)=f
		if n_elements(flux) eq 1 then flux=flux(0)
		return,(complex(visamp_shell)+complex(visamp_star)*flux)*phase
	endif else begin
		return,complex(visamp_shell)*phase
	endelse
	end
;
endcase
endif
;
; Double star...................................................................
;
; Find applicable model
index=where(binary_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): no or non-unique component definition!'
	return,-1
endif
model=binary_model(index)
;
parsedate,date,y,m,d
case system_id(SystemId) of
	'NPOI':		midnight=7.0
	'Mark3':	midnight=8.0
	'Keck':		midnight=11.0
	'VLTI':		midnight=4.0
	else:		midnight=8.0
endcase
jd=(julian(y,m,d)-2440000)+time/86400
jd0=(julian(y,m,d)-2440000)+(time*0+midnight/24)
;
; Do the astrometry (i.e. compute xy) for binary components
case model.method of
;
;	Normal detached binary, positions derived from orbit
      1:begin
	o_parms=dblarr(8)
	o_parms(0)=model.semimajoraxis*MAS/RAD
	o_parms(1)=model.eccentricity
	o_parms(2)=model.inclination/RAD
	o_parms(3)=model.periastron/RAD
	o_parms(4)=model.apsidalmotion/RAD
	o_parms(5)=model.ascendingnode/RAD
	o_parms(6)=model.period
	o_parms(7)=model.epoch
	xy=true2app(jd,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	end
;
;	Normal detached binary, position from model
      2:begin
	xy=dblarr(n_elements(time),2)
	xy(*,0)=model.rho*sin(model.theta/RAD)*MAS/RAD
	xy(*,1)=model.rho*cos(model.theta/RAD)*MAS/RAD
	end
;
; 	Normal detached binary, positions from model w/orbital motion
      3:begin
	o_parms=dblarr(8)
	o_parms(0)=model.semimajoraxis*MAS/RAD
	o_parms(1)=model.eccentricity
	o_parms(2)=model.inclination/RAD
	o_parms(3)=model.periastron/RAD
	o_parms(4)=model.apsidalmotion/RAD
	o_parms(5)=model.ascendingnode/RAD
	o_parms(6)=model.period
	o_parms(7)=model.epoch
	xyO=true2app(jd, o_parms,rhoO,thetaO,gen_model.ra,gen_model.dec)
	xy0=true2app(jd0,o_parms,rho0,theta0,gen_model.ra,gen_model.dec)
	dxy=xyO-xy0
	xy=dblarr(n_elements(time),2)
	xy(*,0)=model.rho*sin(model.theta/RAD)*MAS/RAD+dxy(*,0)
	xy(*,1)=model.rho*cos(model.theta/RAD)*MAS/RAD+dxy(*,1)
	xy(*,0)=(model.rho+rhoO-rho0)*sin(model.theta/RAD+thetaO-theta0)*MAS/RAD
	xy(*,1)=(model.rho+rhoO-rho0)*cos(model.theta/RAD+thetaO-theta0)*MAS/RAD
	end
;
; 	Interacting binary, with WD code
      4:begin
	modvis=complexarr(num_wave,num_scan)
	fluxes=fltarr(num_wave)+1
	flux=dblarr(num_wave,num_scan)+1
	for i=0,num_scan-1 do begin
	for j=0,num_wave-1 do begin
	map=wdmap(model,jd(i),lambda(j))
	fcc_total=total(map.fcc)
	modvis(j,i)=mapvis(map,u(j,i),v(j,i))/fcc_total
	endfor
	endfor
	return,modvis	; Special case: this code return the visibility
        end
;
endcase
;
; Compute component visibilities (must follow astrometry because of mode 4)
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
modvis_1=componentvis(comp_1,time,lambda,fluxes1,u,v)
modvis_2=componentvis(comp_2,time,lambda,fluxes2,u,v)
;
; Compute total flux to be returned
fluxes=fluxes1+fluxes2
;
; Compute masses for center-of-mass referencing
if componentparse(component,comp_1,comp_2) eq -1 then return,-1
masses=componentmass(comp_1,comp_2,num_body,index_comp)
;
; Allocate, then compute phase offsets
phase1=complexarr(num_wave,num_scan) 
if n_elements(phase1) eq 1 then phase1=complexarr(n_elements(u))
phase2=phase1
;
; Use photometric center of system for bandwidth smearing computation
if model.component eq topbincomp() then begin
	factor1=-(1-total(fluxes1)/total(fluxes))
	factor2=+(1-total(fluxes2)/total(fluxes))
endif else begin
;	For lower level binary components, reference to center of mass
	factor1=-(1-masses(0)/total(masses))
	factor2=+(1-masses(1)/total(masses))
endelse
;
; if !owner eq 'chummel' then begin
; factor1=-(1-masses(0)/total(masses))
; factor2=+(1-masses(1)/total(masses))
; endif
;
if num_wave gt 1 then begin
	for j=0,num_wave-1 do begin
	   phase1(j,*)=exp(2*pi_circle*i_complex*factor1 $
		*(u(j,*)*xy(*,0)+v(j,*)*xy(*,1)))
	   phase2(j,*)=exp(2*pi_circle*i_complex*factor2 $
		*(u(j,*)*xy(*,0)+v(j,*)*xy(*,1)))
	endfor
endif else begin
	   phase1(*)=exp(2*pi_circle*i_complex*factor1 $
		*(u(*)*xy(0,0)+v(*)*xy(0,1)))
	   phase2(*)=exp(2*pi_circle*i_complex*factor2 $
		*(u(*)*xy(0,0)+v(*)*xy(0,1)))
endelse
;
return,modvis_1*phase1+modvis_2*phase2
;
end
;-------------------------------------------------------------------------------
function modelvis,time,lambda,fluxes,u,v
;
; Computes complex visibility for a multiple star model.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; The number of elements in the binary model corresponds to the number of
; binary components (e.g. components like A-B, A-BC, AB-CD, etc.). The
; number of elements of the star model must correspond to the number of
; real stars involved in the entire model of the (multiple) stellar system.
;
if num_binary() gt 0 then begin
;	Hierarchical model
	levels=strlen(binary_model.component)
	i=where(max(levels) eq levels)
	modvis=componentvis(binary_model(i).component,time,lambda,fluxes,u,v)
endif else if num_star() ge 1 then begin
;	Distribution of single components
	modvis=complexarr(n_elements(time),n_elements(lambda))
	fluxes=fltarr(n_elements(lambda))
;	For uvimage, which calls with num_scan=num_wave=1
	if n_elements(modvis) eq 1 then begin
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
	endif
	for i=0,num_star()-1 do begin
	modvis= $
	modvis+componentvis(star_model(i).component,time,lambda,fluxes1,u,v)
	fluxes=fluxes+fluxes1
	endfor
endif else begin
	print,'***Error(MODELVIS): model incomplete!'
	return,-1
endelse
;
return,modvis
end
;-------------------------------------------------------------------------------
function componentflux,component,epoch,lambda
;
; This stub of function COMPONENTFLUX is included here so that PV-WAVE, by way 
; of compiling it, stores the function address, which is apparently needed when
; compiling the real thing (see the following function). Otherwise, PV-WAVE
; would complain about an unknown function COMPONENTFLUX when trying to execute
; it. This does not happen, however, with recursive procedures.
;
end
;-------------------------------------------------------------------------------
function componentflux,component,epoch,lambda
;
; Input parameters:
; component: 		A, B, C,..., A-B, AB-C, AB-CD,...
; epoch [s]: 		dblarr(num_scan)
; lambda [m]: 		dblarr(num_wave)
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
; Initialize constants
factor=(pi_circle^2)/(180L*3600L*1000L)
mas2rad=pi_circle/(180L*3600L*1000L)
;
num_scan=n_elements(epoch)
num_wave=n_elements(lambda)
;
; Replicate fluxes to same dimensions as [epoch,lambda]
flux=dblarr(num_wave,num_scan)
;
; Get length of component to find out whether component is single or double
len=strlen(component)
;
; Single star...................................................................
if len eq 1 then begin
;
; Set applicable model
index=where(star_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTFLUX): non-unique component definition!'
	return,-1
endif
model=star_model(index)
;
; This initializes the limb darkening database
fluxes=stellarfluxes(model,lambda,ld_coeffs)
fluxes=fluxes*modelfluxes(model,lambda)
for i=0,num_scan-1 do flux(*,i)=fluxes 
if n_elements(flux) eq 1 then flux=flux(0)
;
case model.type of
; 	Rotating star, using Roche code V-1.4
     14:begin
	vis=rochevis(model,lambda,0,0,fluxes)
	for i=0,num_scan-1 do flux(*,i)=fluxes
	if n_elements(flux) eq 1 then flux=flux(0)
;	3.24 mas is the diameter of Vega, whose colors we use for normalization
	return,flux*(star_model(0).diameter/3.24)^2
	end
;
;	Ring or disk with hole, and temperature profile (Hillenbrand 1992),
;	also used by Malbet et al. (2005, "temperature gradient model").
;	T  ~ (radius/r_min)^-alpha, alpha should be -3/4
     15:begin
	num_ring=500
	r_ring=((findgen(num_ring)/num_ring)*model.width+1) $
		*(model.diameter/2)
	alpha=model.alpha
;	if alpha eq 0 then alpha=-3./4
	teff0=abs(model.teff)
	teff1=abs(model.teff)/(model.diameter/2)^alpha
	fluxes=fltarr(num_wave)
	for k=0,num_ring-1 do begin
		model.teff=-teff1*r_ring(k)^alpha
		f=stellarfluxes(model,lambda,ld_coeffs)*2*!pi*r_ring(k) $
		 *modelfluxes(model,lambda)
		fluxes=fluxes+f
	endfor
	model.teff=teff0
	for i=0,num_scan-1 do flux(*,i)=fluxes 
	if n_elements(flux) eq 1 then flux=flux(0)
;	save,lambda,fluxes,filename='fluxes_comp_15.xdr'
	return,flux
	end
;
   else:begin
;	save,lambda,fluxes,filename='fluxes_comp_CC.xdr'
	return,flux
	end
;
endcase
;
endif
;
; Double star...................................................................
;
; Set applicable model
index=where(binary_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): non-unique component definition!'
	return,-1
endif
model=binary_model(index)
;
case model.method of
;
; 	Interacting binary, with WD code
      4:begin
	for i=0,num_scan-1 do begin
	for j=0,num_wave-1 do begin
	map=wdmap(model,epoch(i),lambda(j))
	flux(j,i)=total(map.fcc)
	endfor
	endfor
	return,flux
        end
;
   else:
;
endcase
;
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
return,componentflux(comp_1,epoch,lambda)+componentflux(comp_2,epoch,lambda)
;
return,0
;
end
;-------------------------------------------------------------------------------
function modelflux,epoch,lambda
;
; Computes fluxes for a multiple star model.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if num_binary() gt 0 then begin
; 	Find the toplevel component
	levels=strlen(binary_model.component)
	index=where(max(levels) eq levels)
	modflux=componentflux(binary_model(index).component,epoch,lambda)
endif else if num_star() ge 1 then begin
	modflux=dblarr(num_wave,num_scan)
	for i=0,num_star()-1 do begin
	modflux= $
	modflux+componentflux(star_model(i).component,epoch,lambda)
	endfor
endif else begin
	print,'***Error(MODELFLUX): model incomplete!'
	return,-1
endelse
;
return,modflux
end
;-------------------------------------------------------------------------------
function modelpos,epoch,vellist
;
; Compute the locations of stars in a multiple system by summing up
; positional offsets generated by the binary pairs. Each binary is centered
; on the location of its center of mass. The positions are right ascension
; and declination offsets.
;
; Also return the velocities in RA and Dec [mas/day].
;
; Note: epoch = JD - 2440000 !
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
num_epoch=n_elements(epoch)
;
o_parms=dblarr(8)
poslist=dblarr(num_epoch,num_star(),2)
poslist_h=poslist
poslist_l=poslist
vellist=dblarr(num_epoch,num_star(),2)
complist=strarr(num_star())
;
epoch_l=epoch-min(binary_model.period)/1000
epoch_h=epoch+min(binary_model.period)/1000
;
FOR i=0,num_binary()-1 DO BEGIN
;
if componentparse(binary_model(i).component,comp_1,comp_2) eq -1 then return,-1
;
; Compute the component masses (=sum of stellar masses)
masses=componentmass(comp_1,comp_2,num_body,index_comp)
;
case binary_model(i).method of
;
;	Normal detached binary, positions derived from orbit
      1:begin
	o_parms(0)=binary_model(i).semimajoraxis
	o_parms(1)=binary_model(i).eccentricity
	o_parms(2)=binary_model(i).inclination/RAD
	o_parms(3)=binary_model(i).periastron/RAD
	o_parms(4)=binary_model(i).apsidalmotion/RAD
	o_parms(5)=binary_model(i).ascendingnode/RAD
	o_parms(6)=binary_model(i).period
	o_parms(7)=binary_model(i).epoch
	xy=true2app(epoch,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	xy_h=true2app(epoch_h,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	xy_l=true2app(epoch_l,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	end
;
;	Normal detached binary, position from model
      2:begin
	xy=dblarr(n_elements(epoch),2)
	xy(*,0)=binary_model(i).rho*sin(binary_model(i).theta/RAD)
	xy(*,1)=binary_model(i).rho*cos(binary_model(i).theta/RAD)
	xy_l=xy
	xy_h=xy
	end
;
endcase
;
factor=[-masses(1)/total(masses),+masses(0)/total(masses)]
for k=0,1 do begin
	for j=0,num_body(k)-1 do begin
	    poslist(*,index_comp(j,k),*)=poslist(*,index_comp(j,k),*) $
					+factor(k)*xy
	endfor
endfor
;
; Accumulate positions for epoch_h, epoch_l
for k=0,1 do begin
	for j=0,num_body(k)-1 do begin
	    poslist_h(*,index_comp(j,k),*)=poslist_h(*,index_comp(j,k),*) $
					+factor(k)*xy_h
	endfor
endfor
for k=0,1 do begin
	for j=0,num_body(k)-1 do begin
	    poslist_l(*,index_comp(j,k),*)=poslist_l(*,index_comp(j,k),*) $
					+factor(k)*xy_l
	endfor
endfor
;
ENDFOR
;
; Compute derivatives
for k=0,1 do begin
for j=0,num_star()-1 do begin
	if epoch_h(0)-epoch_l(0) gt 0 then $
	vellist(*,j,k)=(poslist_h(*,j,k)-poslist_l(*,j,k))/(epoch_h-epoch_l)
endfor
endfor
;
return,poslist
;
end
;-------------------------------------------------------------------------------
function modelvel,epoch,components
;
; Compute radial velocity of stars in a multiple system by adding up all the
; contributions from the binary pairs.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
mas2rad=pi_circle/(180L*3600L*1000L)
RAD=180/pi_circle
;
num_epoch=n_elements(epoch)
if n_elements(components) eq 0 then components=binary_model.component
;
o_parms=dblarr(8)
vellist=dblarr(num_epoch,num_star())+gen_model.rv
complist=strarr(num_star())
;
FOR ic=0,n_elements(components)-1 DO BEGIN
;
i=where(binary_model.component eq components(ic)) & i=i(0)
if componentparse(binary_model(i).component,comp_1,comp_2) eq -1 then return,-1
;
; Compute the component masses (=sum of stellar masses)
masses=componentmass(comp_1,comp_2,num_body,index_comp)
;
o_parms(0)=binary_model(i).eccentricity
o_parms(1)=binary_model(i).inclination/RAD
o_parms(2)=binary_model(i).periastron/RAD
o_parms(3)=binary_model(i).apsidalmotion/RAD
o_parms(4)=binary_model(i).period
o_parms(5)=binary_model(i).epoch
o_parms(6)=masses(0)
o_parms(7)=masses(1)
v=true2vel(epoch,o_parms)
;
for k=0,1 do begin
	for j=0,num_body(k)-1 do begin
	    vellist(*,index_comp(j,k))=vellist(*,index_comp(j,k))+v(*,k)
	endfor
endfor
;
ENDFOR
;
return,vellist
;
end
;-------------------------------------------------------------------------------
function binarypos,epoch0,component,lambda=lambda,com=com,abs=abs
;
; Return separation and position angle for the specified binary
; component. This is computed using the centers of light.
; Assume visual band is the requested wavelength.
; Epoch is the full Julian day number; if not specified, use
; today and the top binary component.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(binary_model) eq 0 then begin
	print,'***Error(BINARYPOS): no model!'
	return,-1
endif
;
if n_elements(epoch0) eq 0 then begin
        parseidldate,systime(),y,m,d
	epoch0=julian(y,m,d)
endif
epoch=epoch0-2440000L
;
if n_elements(component) eq 0 then component=topbincomp() $
			      else component=strupcase(component)
;
; Assume visual (speckle) band if not specified
; if n_elements(lambda) eq 0 then lambda=2.2e-6
if n_elements(lambda) eq 0 then lambda=550e-9
;
; Assume center-of-light if not specified otherwise
if n_elements(com) eq 0 then com=0
;
; Assume relative position, not absolute
if n_elements(abs) eq 0 then abs=0
;
rad=180/pi_circle
;
poslist=modelpos(epoch)
if componentparse(component,c1,c2) ne 0 then return,-1
c1=nameparse(c1,'')
c2=nameparse(c2,'')
index1=intarr(n_elements(c1))
index2=intarr(n_elements(c2))
flux1=fltarr(n_elements(c1))
flux2=fltarr(n_elements(c2))
mass1=fltarr(n_elements(c1))
mass2=fltarr(n_elements(c2))
for i=0,n_elements(index1)-1 do begin
	index1(i)=where(star_model.component eq c1(i))
	flux1(i)=stellarfluxes(star_model(index1(i)),lambda)
	mass1(i)=star_model(index1(i)).mass
endfor
for i=0,n_elements(index2)-1 do begin
	index2(i)=where(star_model.component eq c2(i))
	flux2(i)=stellarfluxes(star_model(index2(i)),lambda)
	mass2(i)=star_model(index2(i)).mass
endfor
;
if com then begin
	weight1=mass1
	weight2=mass2
endif else begin
	weight1=flux1
	weight2=flux2
endelse
x=total(poslist(0,index1,0)*weight1)/total(weight1) $
 -total(poslist(0,index2,0)*weight2)/total(weight2)
y=total(poslist(0,index1,1)*weight1)/total(weight1) $
 -total(poslist(0,index2,1)*weight2)/total(weight2)
x1=total(poslist(0,index1,0)*weight1)/total(weight1)
x2=total(poslist(0,index2,0)*weight2)/total(weight2)
y1=total(poslist(0,index1,1)*weight1)/total(weight1)
y2=total(poslist(0,index2,1)*weight2)/total(weight2)
if abs then begin
	x=[x1,x2]
	y=[y1,y2]
endif else begin
	x=x2-x1
	y=y2-y1
endelse
rho=sqrt(x^2+y^2)
theta=atan(x,y)*rad
index=where(theta lt 0,count)
if count gt 0 then theta(index)=theta(index)+360
;
return,[[rho],[theta]]
;
end
;-------------------------------------------------------------------------------
function binaryvel,epoch,component
;
; Epoch is the full Julian day number.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(binary_model) eq 0 then begin
	print,'***Error(BINARYVEL): no model!'
	return,-1
endif
;
if n_elements(epoch) eq 0 then begin
        parseidldate,systime(),y,m,d
	epoch=julian(y,m,d)
endif
;
vels=modelvel(epoch-2440000.d0)
if n_elements(component) eq 0 then component='A' $
			      else component=strupcase(component)
;
return,vels(where(star_model.component eq component))
;
end
;-------------------------------------------------------------------------------
function modelpx,component,error
;
; Compute the parallax [mas] from data for any input binary component.
; The error estimate is somewhat conservative in that the 
; masses are usually correlated, but this function just adds
; all stellar mass errors in quadrature.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if componentparse(strupcase(component),comp_1,comp_2) ne 0 then return,-1
mass_sum=total(componentmass(comp_1,comp_2))
i=where(binary_model.component eq strupcase(component)) & i=i(0)
parallax=binary_model(i).semimajoraxis $
	/(mass_sum*(binary_model(i).period/365.25)^2)^(1./3)
error=parallax*sqrt( $
	(binary_error(i).semimajoraxis/binary_model(i).semimajoraxis)^2 $
       +(total((star_error.mass/star_model.mass)^2)/9) $
       +(binary_error(i).period/binary_model(i).period*(2.0/3.0))^2)
return,parallax
;
end
;-------------------------------------------------------------------------------
function modelm,component,error
;
; Compute total mass of a binary component with Kepler's 3rd law with parallax.
;
common AuxData,parallaxes,k2,vsini
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; if componentparse(strupcase(component),comp_1,comp_2) ne 0 then return,-1
; mass_sum=total(componentmass(comp_1,comp_2))
i=where(binary_model.component eq strupcase(component)) & i=i(0)
mass_sum=(binary_model(i).semimajoraxis/parallaxes.value(0))^3 $
	/(binary_model(i).period/365.25)^2
error=mass_sum*sqrt( $
	(3*binary_error(i).semimajoraxis/binary_model(i).semimajoraxis)^2+ $
       	(3*binary_error(i).period/binary_model(i).period)^2+ $
	(2*parallaxes.error(0)/parallaxes.value(0))^2)
return,mass_sum
;
end
;-------------------------------------------------------------------------------
function modelk,component
;
; Compute and return velocity semi-amplitude of requested stellar component.
; Note that for systems of three or more components, the amplitude is in
; rest frame of the corresponding binary component.
; Return -1 if failed for any reason.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
RAD=180/pi_circle
;
if n_elements(binary_model) eq 0 then return,-1
pos=strpos(binary_model.component,strupcase(component))
index=where(pos ge 0,count)
if count eq 0 then return,-1
models=binary_model(index)
model=models(where(strlen(binary_model.component) eq $
	       min(strlen(binary_model.component))))
model=model(0)
pos=strpos(model.component,strupcase(component))
if componentparse(model.component,comp_1,comp_2) ne 0 then return,-1
masses=componentmass(comp_1,comp_2)
m1=masses(0)
m2=masses(1)
i=model.inclination/RAD
e=model.eccentricity
p=model.period/s_year
k1=2*pi_circle*sin(i)/sqrt(1-e^2)*p^(-1.0/3.0)*m2*(m1+m2)^(-2.0/3.0) $
	*(a_unit/s_year)/86400000L
k2=k1*(m1/m2)
;
if pos eq 0 then return,k1 else return,k2
;
end
;-------------------------------------------------------------------------------
function modelf,component
;
; Compute and return mass function.
; Return -1 if failed for any reason.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
RAD=180/pi_circle
;
if componentparse(strupcase(component),comp_1,comp_2) ne 0 then return,-1
i=where(binary_model.component eq strupcase(component)) & i=i(0)
masses=componentmass(comp_1,comp_2)
m1=masses(0)
m2=masses(1)
i=binary_model(i).inclination/RAD
;
fm=(m2^3*sin(i)^3)/(m1+m2)^2
;
return,fm
;
end
;-------------------------------------------------------------------------------
function modeldm,component,filter
;
; Compute broad band filter magnitude differences given a physical model.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
teff1=star_model(0).teff
teff2=star_model(1).teff
logg1=star_model(0).logg
logg2=star_model(1).logg
d1=star_model(0).diameter
d2=star_model(1).diameter
;
; Primary A
r=limbgrid(teff1,logg1,lambda,limbdu,fluxes)
f1=total(call_function(filter,lambda)*fluxes) $
    /total(call_function(filter,lambda))
;
; Secondary B
r=limbgrid(teff2,logg2,lambda,limbdu,fluxes)
f2=total(call_function(filter,lambda)*fluxes) $
    /total(call_function(filter,lambda))
;
; Radii
r1=d1/2
r2=d2/2
;
; Magnitudes
mag1=-2.5*alog10(f1_b*r1^2)
mag2=-2.5*alog10(f2_v*r2^2)
;
; Return predicted A-B magnitude differences
return,mag2-mag1
;
end
;-------------------------------------------------------------------------------
function modelchisq,n_deg_free
;
; Compute and return reduced Chi^2 of model. Also return number of degrees
; of freedom, assuming all model parameters are free.
; Make sure you computed the model value before calling this function!
;
modelfitparmcopy,-1,a
marquardtdata,y,ysig,ymod
;
n_deg_free=n_elements(y)-n_elements(a)
return,total(((y-ymod)/ysig)^2)/(n_deg_free)
;
end
;-------------------------------------------------------------------------------
