;*******************************************************************************
; File: plot.pro
;
; Description:
; ------------
; Container of IDL scripts specific to plotting in CHAMELEON/AMOEBA. 
;
; Block directory:
; ----------------
; Block 1: init_class,init_indexsel,init_plot,reinit_indexsel
;          set_nightsel,set_starsel,set_indexsel,set_plotsel,set_streamsel,
;	   printselerr,checkplotsel
; Block 2: set_axitems,set_sliceids,set_plotdata,set_ploterr,set_caldata
;          set_plotlabels,set_plotloops,
;	   titleplacement,testcolor,plotsymbol,wdall,set_screen,set_ps
; Block 3: set_pointflagtable,create_pointflagtable,
;	   set_inchflagtable,create_inchflagtable,
;	   create_bgflagtable,create_scanflagtable,
;          flagpointdata,flaginchdata,flagbgdata,flagscandata,
;	   unflag,resetflag
;
; Block 4: plotdata,plotuv,plotncal
;
; Block 5: init_plotorbit,thiele,innes,
;          plotorbit,plotthiele,plotellipse
;
; Block 6: init_plotvel,
;	   plotvel,plotlehmann
;
; Block 7: init_plotmag,
;	   plotmag
;
; Block 8: plotvolvox
;
;
;************************************************************************Block 1
function init_class,class,datascans
;
; Bundles a few class-specific plot initializations.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
;
if n_params() eq 2 then do_scans=1 else do_scans=0
;
if n_elements(GenConfig) eq 0 then begin
	print,'***Error(INIT_CLASS): GenConfig undefined!'
	return,-1
endif
;
nib=GenConfig.NumSid
nob=GenConfig.NumOutBeam
ntr=GenConfig.NumTriple
mch=max([GenConfig.NumSpecChan,2])
mbl=max([GenConfig.NumBaseline,2])
;
case class of
;	Pointdata
	'point' :begin
		 item=1
		 !p.psym=3
		 npt=n_elements(PointTime)
		 if do_scans and npt gt 0 then begin
		 starids=strarr(npt)
		 scanids=intarr(npt)
		 for i=0,n_elements(Iscan)-1 do begin
			starids(Rec0(i):RecN(i))=StarId(i)
			scanids(Rec0(i):RecN(i))=Iscan(i)
		 endfor
		 datascan={starid:'',time:0.d0,iscan:0}
		 datascans=replicate(datascan,npt)
		 datascans.starid=starids
		 datascans.time=PointTime
		 datascans.iscan=scanids
		 endif
		 mpt=long(total(scantable.numpoint))
		 end
	'delay' :begin
		 item=1
		 !p.psym=3
		 npt=n_elements(PointTime)
		 if do_scans and npt gt 0 then begin
		 starids=strarr(npt)
		 for i=0,n_elements(Iscan)-1 do $
			starids(Rec0(i):RecN(i))=StarId(i)
		 datascan={starid:'',time:0.d0}
		 datascans=replicate(datascan,npt)
		 datascans.starid=starids
		 datascans.time=PointTime
		 endif
		 mpt=long(total(scantable.numpoint))
		 end
;	Background scans
	'bg'    :begin
		 item=19
		 !p.psym=1
		 npt=n_elements(bgtable)
		 if do_scans and npt gt 0 then begin
		 starids=strarr(n_elements(bgtable))
		 for i=0,n_elements(starids)-1 do starids(i)= $
			scantable(long(where(scantable.scanid eq $
				bgtable(i).scanid),0)).starid 
		 datascan={starid:'',time:0.d0}
		 datascans=replicate(datascan,npt)
		 if npt gt 1 then begin
		 	datascans.starid=starids
		 	datascans.time=bgtable.time
		 endif else begin
		 	datascans.starid=starids(0)
		 	datascans.time=bgtable(0).time
		 endelse
		 endif
		 if n_elements(scantable) ne 0 then mpt=n_elements(scantable)*2 $
					       else mpt=npt*2
		 end
;	Scans (single night)
	'scan'  :begin
		 item=22
		 !p.psym=1
		 npt=n_elements(scans)
		 if do_scans and npt gt 0 then datascans=scans
		 mpt=npt
		 end
	'ncal'  :begin
		 item=22
		 !p.psym=1
		 npt=n_elements(scans)
		 if do_scans and npt gt 0 then datascans=scans
		 mpt=npt
		 end
	'astrom':begin
		 item=22
		 !p.psym=1
		 npt=n_elements(scans)
		 if do_scans and npt gt 0 then datascans=scans
		 mpt=npt
		 end
	'seeing':begin
		 item=22
		 !p.psym=1
		 npt=n_elements(scans)
		 if do_scans and npt gt 0 then datascans=scans
		 mpt=npt
		 end
	'phot'	:begin
		 item=22
		 !p.psym=1
		 npt=n_elements(scans)
		 if do_scans and npt gt 0 then datascans=scans
		 mpt=npt
		 end
	'uv'    :begin
		 item=25
		 !p.psym=3
		 npt=n_elements(scans)
		 if do_scans and npt gt 0 then datascans=scans
		 mpt=npt
		 end
;	Scans (multiple nights)
	'amoeba':begin
		 item=22
		 !p.psym=1
		 npt=n_elements(scans)
		 if do_scans and npt gt 0 then datascans=scans
		 nib=max(GenInfo.numsid)
		 nob=max(GenInfo.numoutbeam)
		 ntr=max(GenInfo.numtriple)
		 mch=max(GenInfo.numspecchan)
	 	 mbl=max(GenInfo.numbaseline)
		 mpt=npt
		 mpt=1000 	; simulated data might have this many scans
		 end
	'volvox':begin
		 item=22
		 !p.psym=1
		 npt=n_elements(scans)
		 if do_scans and npt gt 0 then datascans=scans
		 nib=max(GenInfo.numsid)
		 nob=max(GenInfo.numoutbeam)
		 ntr=max(GenInfo.numtriple)
		 mch=max(GenInfo.numspecchan)
	 	 mbl=max(GenInfo.numbaseline)
		 mpt=npt
		 mpt=1000 	; simulated data might have this many scans
		 end
	'metro' :begin
		 item=60
		 !p.psym=3
		 npt=n_elements(MetroTime)
		 if do_scans and npt gt 0 then begin
		 starids=strarr(npt) & starids(*)='CCCslew'
		 for i=0,n_elements(scantable)-1 do begin
			if SolN(i) ge 0 then $
			starids(Sol0(i):SolN(i))=scantable(i).StarId
		 endfor
		 datascan={starid:'',time:0.d0}
		 datascans=replicate(datascan,npt)
		 datascans.starid=starids
		 datascans.time=MetroTime
		 endif
		 mpt=npt
		 end
	   else :begin
		 print,'***Error(SET_INITPARMS): unknown class:',class,'!'
		 return,-1
		 end
endcase
if npt eq 0 then begin
	print,'***Error(INIT_CLASS): no data!'
	return,-1
endif
if mpt eq 0 then begin
	print,'***Error(INIT_CLASS): reload scantable from .con file!'
	return,-1
endif
return,[item,nib,nob,ntr,mch,mbl,mpt]
;
end
;-------------------------------------------------------------------------------
function init_indexsel,stream
;
; This routine is called by init_plot when initializing a minimum requirement
; selection for plotting. It also can be called to reset the data selection
; in a particular stream. Initializes all variables in SelDirectives, except 
; st_dir & st_sel. Completes initialization of DataSelInfo.
;
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
if n_elements(stream) eq 0 then begin
	print,'***Error(INIT_INDEXSEL): stream undefined!'
	return,-1
endif
if n_elements(class) eq 0 then begin
	print,'***Error(INIT_INDEXSEL): class undefined!'
	return,-1
endif
;
; Initialize directives
;
; If you change any of these, you have to change initially displayed button
; in ww_datasel too!
dir={bl:'Current',ch:'Current',pt:'All'}
;
; Initialize user input selection:
;
parms=init_class(class)
mch=parms(4)
mbl=parms(5)
mpt=parms(6)
if mch*mbl*mpt eq 0 then return,-1
sel={ch:intarr(mch),bl:intarr(mbl),pt:lonarr(mpt)}
;
; Do the data selection:
;
case stream of
	'x':i=ds_x.item
	'y':i=ds_y.item
	'z':i=ds_z.item
       else:begin
	    print,'***Error(INIT_INDEXSEL): unknown stream:',stream
	    return,-1
	    end
endcase
ds={item:i,ib:0,ob:0,tr:0,ch:-indgen(mch),bl:-indgen(mbl),pt:lindgen(mpt)}
;
; Finally, assign the items to the stream variables:
case stream of
	'x':begin
	    x_dir=dir
	    x_sel=sel
	    ds_x=ds
	    end
	'y':begin
	    y_dir=dir
	    y_sel=sel
	    ds_y=ds
	    end
	'z':begin
	    z_dir=dir
	    z_sel=sel
	    ds_z=ds
	    end
endcase
;
return,0
;
end
;-------------------------------------------------------------------------------
function init_plot,data_class,slice_class
;
; This is the first routine to be called when preparing a plot of class
; data_class/slice_class. It calls init_indexsel once for each data stream
; (x, y, and z) to complete the initialization in order to meet minimum
; selection requirements.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
; Initialize DataSelInfo first:
;
if n_elements(data_class) eq 0 then begin
	print,'***Error(INIT_PLOT): class not defined!'
	return,-1
endif 
;
if n_elements(slice_class) eq 0 then begin
	print,'***Error(INIT_PLOT): slice not defined!'
	return,-1
endif 
;
; Default axis item is Time
parms=init_class(data_class)
if parms(0) eq -1 then return,-1
i=parms(0)
ds_x={item:i}
ds_y={item:i}
ds_z={item:i}
;
class=data_class
slice=slice_class
type=''
;
ds_nights=''
ds_stars=''
;
ps_options=alloc_ps_options()
;
; Include classes here which do not allow the selection of axes
if class eq 'ncal' then begin
	ds_x.item=0
	ds_y.item=0
endif
;
; Now initialize SelDirectives:
;
; Nights:
nt_dir='All'; If you change this, change 'current' keyword in ww_plot too!
nt_sel=''
;
; Stars:
st_dir='All'; If you change this, change 'current' keyword in ww_plot too!
st_sel=''
;
if init_indexsel('x') ne 0 then return,-1
if init_indexsel('y') ne 0 then return,-1
if init_indexsel('z') ne 0 then return,-1
;
; Set a few IDL plot options
!p.charsize=1.5		; Character size
!x.style=18		; No setting X axis minimum to 0, extend axis 5%
!y.style=18		; No setting Y axis minimum to 0, extend axis 5%
!z.style=18		; No setting Z axis minimum to 0, extend axis 5%
!x.ticks=0
!y.ticks=0
!z.ticks=0
;
return,0
;
end
;-------------------------------------------------------------------------------
function reinit_indexsel
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
forward_function set_indexsel
;
ds_x_save=ds_x
x_dir_save=x_dir
x_sel_save=x_sel
if init_indexsel('x') ne 0 then return,-1
struct_assign,ds_x_save,ds_x,/nozero
if set_indexsel('x') ne 0 then return,-1
x_dir=x_dir_save
x_sel=x_sel_save
;
ds_y_save=ds_y
y_dir_save=y_dir
y_sel_save=y_sel
if init_indexsel('y') ne 0 then return,-1
struct_assign,ds_y_save,ds_y,/nozero
if set_indexsel('y') ne 0 then return,-1
y_dir=y_dir_save
y_sel=y_sel_save
;
ds_z_save=ds_z
z_dir_save=z_dir
z_sel_save=z_sel
if init_indexsel('z') ne 0 then return,-1
struct_assign,ds_z_save,ds_z,/nozero
if set_indexsel('z') ne 0 then return,-1
z_dir=z_dir_save
z_sel=z_sel_save

return,0
;
end
;-------------------------------------------------------------------------------
function set_nightsel
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
flag=0
if n_elements(nt_dir) eq 0 then flag=1
if flag then begin
	print,'***Error(SET_NIGHTSEL): nights directive undefined!'
	return,-1
endif
;
; Prepare night selection
;
if nt_dir eq 'All' then begin
	if n_elements(GenInfo) eq 0 $
		then ds_nights=checkdate() $
		else ds_nights=GenInfo.date+' ' $
			      +GeoInfo.systemid+' ' $
			      +GenInfo.configid
endif else if nt_dir eq 'Sel' then begin
	if nt_sel(0) eq '' then begin
	   print,'***Error(SET_NIGHTSEL): you have to select nights first!'
	   return,-1
	endif
	ds_nights=nt_sel
endif
;
return,0
;
end
;-------------------------------------------------------------------------------
function set_starsel
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
flag=0
if n_elements(st_dir) eq 0 then flag=1
if flag then begin
	print,'***Error(SET_STARSEL): stars directives undefined!'
	return,-1
endif
;
; Prepare star selection
;
if st_dir eq 'All' then begin
	list_stars,ds_stars
	if class eq 'metro' then ds_stars=[ds_stars,'CCCslew']
endif else if st_dir eq 'Cal' then begin
        index=where(startable.bflag eq 'C',count)
        if count gt 0 then ds_stars=startable(index).starid $
          else begin
	    print,'***Error(SET_STARSEL): no calibrators listed in startable!'
	    return,-1
	  endelse
endif else if st_dir eq 'Sel' then begin
	if st_sel(0) eq '' then begin
	   print,'***Error(SET_STARSEL): you have to select stars first!'
	   return,-1
	endif
	ds_stars=st_sel
endif
;
return,0
;
end
;-------------------------------------------------------------------------------
function set_indexsel,stream
;
; This function is called after the user has entered SelDirectives, i.e.
; directives and selections for a particular stream (x, y, or z). It prepares 
; the final data selection and stores it in DataSelInfo.
;
; Note a pecularity regarding the star and scan/point selection, since both
; effectively select data point. The star selection is applied after the scan
; selection, i.e. data for a star selected, if no scans corresponding to this
; star have been selected too, will not have data selected. In addition, this
; selection has additional directives related to the scan configuration, i.e.
; configurations which differ only by their stations used. This feature is
; required by the 6-way observing configuration.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
flag=0
if n_elements(x_dir) eq 0 then flag=1
if n_elements(y_dir) eq 0 then flag=1
if n_elements(z_dir) eq 0 then flag=1
if n_elements(x_sel) eq 0 then flag=1
if n_elements(y_sel) eq 0 then flag=1
if n_elements(z_sel) eq 0 then flag=1
if flag then begin
	print,'***Error(SET_DATASEL): some directives/selections undefined!'
	return,-1
endif
;
case stream of
	'x':begin
	    ch_dir=x_dir.ch
	    bl_dir=x_dir.bl
	    pt_dir=x_dir.pt
	    ch_sel=x_sel.ch
	    bl_sel=x_sel.bl
	    pt_sel=x_sel.pt
	    end
	'y':begin
	    ch_dir=y_dir.ch
	    bl_dir=y_dir.bl
	    pt_dir=y_dir.pt
	    ch_sel=y_sel.ch
	    bl_sel=y_sel.bl
	    pt_sel=y_sel.pt
	    end
	'z':begin
	    ch_dir=z_dir.ch
	    bl_dir=z_dir.bl
	    pt_dir=z_dir.pt
	    ch_sel=z_sel.ch
	    bl_sel=z_sel.bl
	    pt_sel=z_sel.pt
	    end
endcase
; NoTrace option not selected (default): X-selection will trace Y-selection
if stream eq 'x' and not ps_options.t then begin
	    ch_dir=y_dir.ch
	    bl_dir=y_dir.bl
	    pt_dir=y_dir.pt
	    ch_sel=y_sel.ch
	    bl_sel=y_sel.bl
	    pt_sel=y_sel.pt
endif
set_streamsel,stream,item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,ds_pt
;
if n_elements(ds_ib) eq 0 then return,-1
if n_elements(ds_ob) eq 0 then return,-1
if n_elements(ds_tr) eq 0 then return,-1
if (ds_ib lt 0) or $
   (ds_ib ge GenConfig.NumSid and GenConfig.NumSid ne 0) then begin
	print,'***Error(SET_DATASEL): invalid input beam selection!'
	return,-1
endif
if (ds_ob lt 0) or $
   (ds_ob ge GenConfig.NumOutBeam and GenConfig.NumOutBeam ne 0) then begin
	print,'***Error(SET_DATASEL): invalid output beam selection!'
	return,-1
endif
if (ds_tr lt 0) or $
   (ds_tr ge GenConfig.NumTriple and GenConfig.NumTriple ne 0) then begin
	print,'***Error(SET_DATASEL): invalid triple selection!'
	return,-1
endif
;
flag=0
if n_elements(ch_dir) eq 0 then flag=1
if n_elements(bl_dir) eq 0 then flag=1
if n_elements(pt_dir) eq 0 then flag=1
if flag then begin
	print,'***Error(SET_DATASEL): one or more directives undefined!'
	return,-1
endif
;
; Prepare channel selection
;
if ch_dir eq 'All' then begin
	ds_ch=indgen(GenConfig.NumSpecChan(ds_ob))
endif else if ch_dir eq 'Selected' then begin
	if max(ch_sel) eq 0 then begin
		print,'***Error(SET_DATASEL): no channels selected!'
		return,-1
	endif
	ds_ch=ch_sel(where(ch_sel ne 0))-1
endif else if ch_dir eq 'Next' then begin
	index=where(ch_sel ne 0,n)
	if n_elements(ds_ch) eq 0 then begin
		if n eq 0 then begin
		  print,'***Error(SET_DATASEL): '+ $
			'you have to enter channel selection first!'
		  return,-1
		endif
		ds_ch=indgen(n)
	endif else ds_ch=ds_ch+n
endif else if ch_dir eq 'Previous' then begin
	n=n_elements(ds_ch)
	if n eq 0 then begin
		print,'***Error(SET_DATASEL): no channels currently selected'
		return,-1
	endif
	ds_ch=ds_ch-n
endif
if n_elements(ds_ch) eq 0 then begin
	print,'***Error(SET_DATASEL): no channels selected!'
	return,-1
endif
;
; Prepare baseline selection
;
if bl_dir eq 'All' then begin
	ds_bl=indgen(GenConfig.NumBaseline(ds_ob))
endif else if bl_dir eq 'Selected' then begin
	if max(bl_sel) eq 0 then begin
		print,'***Error(SET_DATASEL): no baselines selected!'
		return,-1
	endif else ds_bl=bl_sel(where(bl_sel ne 0))-1
endif else if bl_dir eq 'Next' then begin
	index=where(bl_sel ne 0,n)
	if n_elements(ds_bl) eq 0 then begin
		if n eq 0 then begin
		  print,'***Error(SET_DATASEL): '+ $
			'you have to enter baseline selection first!'
		  return,-1
		endif
		ds_bl=indgen(n)
	endif else ds_bl=ds_bl+n
endif else if bl_dir eq 'Previous' then begin
	n=n_elements(ds_bl)
	if n eq 0 then begin
		print,'***Error: no baselines currently selected'
		return,-1
	endif
	ds_bl=ds_bl-n
endif
if n_elements(ds_bl) eq 0 then begin
	print,'***Error(SET_DATASEL): no baselines selected!'
	return,-1
endif
;
; Prepare point selection
;
parms=init_class(class,datascans)
num_points=n_elements(datascans)
;
if pt_dir eq 'All' then begin
	ds_pt=lindgen(num_points)
endif else if pt_dir eq 'Selected' then begin
	if max(pt_sel) eq 0 then begin
		print,'***Error(SET_DATASEL): no points/scans selected!'
		return,-1
	endif
	if max(pt_sel) gt num_points then begin
		print,'***Error(SET_DATASEL): invalid point selection!'
		return,-1
	endif
	ds_pt=pt_sel(where(pt_sel ne 0))-1
endif else if pt_dir eq 'Next' then begin
	index=where(pt_sel ne 0,n)
	if n_elements(ds_pt) eq 0 then begin
		if n eq 0 then begin
		  print,'***Error(SET_DATASEL): '+ $
			'you have to enter point/scan selection first!'
		  return,-1
		endif
		ds_pt=lindgen(n)
	endif else ds_pt=ds_pt+n
endif else if pt_dir eq 'Previous' then begin
	n=n_elements(ds_pt)
	if n eq 0 then begin
	   print,'***Error(SET_DATASEL): no points/scans currently selected'
   	   return,-1
	endif
	ds_pt=ds_pt-n
endif else begin
;	Process sub-array selection
	scanids=scanconfig(pt_dir)
	index=lonarr(num_points)
	for i=0,n_elements(scanids)-1 do begin
		j=where(datascans.iscan eq scanids(i),count)
		if count gt 0 then index(j)=1
	endfor
	if total(index) gt 0 then ds_pt=where(index ne 0)
endelse
if n_elements(ds_pt) eq 0 then begin
	print,'***Error(SET_DATASEL): no points/scans selected!'
	return,-1
endif
if stream eq 'y' then begin
	if strpos(type,'V2Bias') ge 0 then type='V2Bias'+pt_dir
	if strpos(type,'TABias') ge 0 then type='TABias'+pt_dir
	if strpos(type,'APDFlux') ge 0 then type='APDFlux'+pt_dir
	if strpos(type,'Response') ge 0 then type='Response'+pt_dir
	if strpos(type,'TrackJitter') ge 0 then type='TrackJitter'+pt_dir
endif
;
; From the scans/points selected, select those corresponding to selected stars
iscan=lindgen(n_elements(datascans))
stars=datascans(ds_pt).starid
jscan=iscan(ds_pt)
first_star=1
for is=0,n_elements(ds_stars)-1 do begin
	index=where(stars eq ds_stars(is),count)
	if count gt 0 then begin
		if first_star then begin
			pt_star=jscan(index)
			first_star=0
		endif else pt_star=[pt_star,jscan(index)]
	endif else ds_stars(is)=''
endfor
if n_elements(pt_star) gt 0 then ds_pt=pt_star
; Remove those stars from the list that wouldn't produce any data
index=where(strlen(ds_stars) gt 0,count)
if count gt 0 then ds_stars=ds_stars(index)
;
; Do this for classes with data not related to stars
; if class eq 'metro' then pt_star=ds_pt
;	
case stream of
	'x':begin
	    if n_elements(ds_bl) gt n_elements(ds_x.bl) $
	    or n_elements(ds_ch) gt n_elements(ds_x.ch) $
	    or n_elements(ds_pt) gt n_elements(ds_x.pt) then begin
		if reinit_indexsel() ne 0 then return,-1
	    endif
	    ds_x.bl=-1
	    ds_x.ch=-1
	    ds_x.pt=-1
	    ds_x.bl(0:n_elements(ds_bl)-1)=ds_bl
	    ds_x.ch(0:n_elements(ds_ch)-1)=ds_ch
	    ds_x.pt(0:n_elements(ds_pt)-1)=ds_pt
	    end
	'y':begin
	    if n_elements(ds_bl) gt n_elements(ds_y.bl) $
	    or n_elements(ds_ch) gt n_elements(ds_y.ch) $
	    or n_elements(ds_pt) gt n_elements(ds_y.pt) then begin
		if reinit_indexsel() ne 0 then return,-1
	    endif
	    ds_y.bl=-1
	    ds_y.ch=-1
	    ds_y.pt=-1
	    ds_y.bl(0:n_elements(ds_bl)-1)=ds_bl
	    ds_y.ch(0:n_elements(ds_ch)-1)=ds_ch
	    ds_y.pt(0:n_elements(ds_pt)-1)=ds_pt
	    end
	'z':begin
	    if n_elements(ds_bl) gt n_elements(ds_z.bl) $
	    or n_elements(ds_ch) gt n_elements(ds_z.ch) $
	    or n_elements(ds_pt) gt n_elements(ds_z.pt) then begin
		if reinit_indexsel() ne 0 then return,-1
	    endif
	    ds_z.bl=-1
	    ds_z.ch=-1
	    ds_z.pt=-1
	    ds_z.bl(0:n_elements(ds_bl)-1)=ds_bl
	    ds_z.ch(0:n_elements(ds_ch)-1)=ds_ch
	    ds_z.pt(0:n_elements(ds_pt)-1)=ds_pt
	    end
endcase
;
if n_elements(pt_star) eq 0 then begin
	if not !quiet then print,'***Error(SET_DATASEL): no selected scans are on these stars!'
	return,-1 
endif else return,0
;
end
;-------------------------------------------------------------------------------
pro set_plotsel
;
; Short cut for scripts to prepare a call to plotdata
;
s=set_starsel()
s=set_indexsel('x')
s=set_indexsel('y')
s=set_indexsel('z')
;
end
;-------------------------------------------------------------------------------
pro set_streamsel,stream_in,item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,ds_pt
;
; Extracts the data selection from a stream and returns it as plain arrays.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
; NoTrace option not selected (default): X-selection will trace Y-selection
if stream_in eq 'x' and not ps_options.t then stream='y' else stream=stream_in
;
case stream of
	'x':begin
	    if n_elements(ds_x) eq 0 then return
	    item=ds_x.item
	    ds_ib=ds_x.ib
	    ds_ob=ds_x.ob
	    ds_tr=ds_x.tr
	    index=where(ds_x.ch ge 0,count)
	    if count gt 0 then ds_ch=ds_x.ch(index) else ds_ch=-1
	    index=where(ds_x.bl ge 0,count)
	    if count gt 0 then ds_bl=ds_x.bl(index) else ds_bl=-1
	    index=where(ds_x.pt ge 0,count)
	    if count gt 0 then ds_pt=ds_x.pt(index) else ds_pt=-1
	    end
	'y':begin
	    if n_elements(ds_y) eq 0 then return
	    item=ds_y.item
	    ds_ib=ds_y.ib
	    ds_ob=ds_y.ob
	    ds_tr=ds_y.tr
	    index=where(ds_y.ch ge 0,count)
	    if count gt 0 then ds_ch=ds_y.ch(index) else ds_ch=-1
	    index=where(ds_y.bl ge 0,count)
	    if count gt 0 then ds_bl=ds_y.bl(index) else ds_bl=-1
	    index=where(ds_y.pt ge 0,count)
	    if count gt 0 then ds_pt=ds_y.pt(index) else ds_pt=-1
	    end
	'z':begin
	    if n_elements(ds_z) eq 0 then return
	    item=ds_z.item
	    ds_ib=ds_z.ib
	    ds_ob=ds_z.ob
	    ds_tr=ds_z.tr
	    index=where(ds_z.ch ge 0,count)
	    if count gt 0 then ds_ch=ds_z.ch(index) else ds_ch=-1
	    index=where(ds_z.bl ge 0,count)
	    if count gt 0 then ds_bl=ds_z.bl(index) else ds_bl=-1
	    index=where(ds_z.pt ge 0,count)
	    if count gt 0 then ds_pt=ds_z.pt(index) else ds_pt=-1
	    end
endcase
;
end
;-------------------------------------------------------------------------------
function checkplotsel,stream
;
; Compares plot selection to GenConfig information. If OK, returns 0.
; Returns error code if selection is incomplete or invalid.
; Note: pt selection is not checked!
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
if n_elements(stream) eq 0 then return,-1
;
if n_elements(class) eq 0 then return,1
if n_elements(slice) eq 0 then return,2
;
if n_elements(ds_nights) eq 0 then return,3
if n_elements(ds_stars) eq 0 then return,4
;
set_streamsel,stream,item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,ds_pt
;
if n_elements(ds_ib) eq 0 then return,5
if n_elements(ds_ob) eq 0 then return,6
if n_elements(ds_tr) eq 0 then return,7
if n_elements(ds_ch) eq 0 then return,8
if n_elements(ds_bl) eq 0 then return,9
;
if (ds_ib lt 0) or $
   (ds_ib ge GenConfig.NumSid and GenConfig.NumSid ne 0) then return,10
if (ds_ob lt 0) or $
   (ds_ob ge GenConfig.NumOutBeam and GenConfig.NumOutBeam ne 0) then return,11
if (ds_tr lt 0) or $
   (ds_tr ge GenConfig.NumTriple and GenConfig.NumTriple ne 0) then return,12
;
index=where((ds_ch ge 0) and (ds_ch lt GenConfig.NumSpecChan(ds_ob)),count)
num_sel=n_elements(ds_ch)
check=[3,4,5,6,7,8,21,24,25,26,27,28,29,30,31,32,33,34,35,45,46,50,51,52]
if (count lt num_sel) and ((long(where(check eq item),0) ne -1) $
			    or strpos(type,'V2Bias') ge 0 $
			    or strpos(type,'TABias') ge 0 $
			    or strpos(type,'APDFlux') ge 0 $
			    or strpos(type,'Response') ge 0 $
			    or strpos(type,'TrackJitter') ge 0 $
			    or strpos(type,'TrackOffset') ge 0) then return,13 
;
index=where((ds_bl ge 0) and (ds_bl lt GenConfig.NumBaseline(ds_ob)),count)
num_sel=n_elements(ds_bl)
check=[3,4,5,10,24,25,26,27,37,38,47,48,50,51,52,90,91]
if (count lt num_sel) and ((long(where(check eq item),0) ne -1) $
			    or strpos(type,'V2Bias') ge 0) then return,14
;
return,0
;
end
;-------------------------------------------------------------------------------
function printselerror,errno
;
case errno of
	-1:print,'***Error: stream not specified!'
	 0:return,0
	 1:print,'***Error: class not specified!'
	 2:print,'***Error: slice not specified!'
	 3:print,'***Error: nights not specified!'
	 4:print,'***Error: stars not specified!'
	 5:print,'***Error: input beam not specified!'
	 6:print,'***Error: output beam not specified!'
	 7:print,'***Error: triples not specified!'
	 8:print,'***Error: channels not specified!'
	 9:print,'***Error: baselines not specified!'
	10:print,'***Error: invalid input beam!'
	11:print,'***Error: invalid output beam!'
	12:print,'***Error: invalid triple selection!'
	13:print,'***Error: invalid channel selection!'
	14:print,'***Error: invalid baseline selection!'
      else:print,'***Error: invalid selection; unknown error code!'
endcase
return,-1
;
end
;************************************************************************Block 2
; Current master list of item ids
; Note: units are generally SI
;  1 Time (pt)		[s] since 0 UT of date
;  2 Index (pt)		An index number not necessarily related to point number
;  3 VisSq (pt)		X^2+Y^2-N
;  4 FASN (pt)
;  5 VisPhase (pt)
;  6 TripleAmp (pt)
;  7 TriplePhase (pt)
;  8 PhotonRate (pt)	/2 ms, ME/MV=0.06
;  9 FDLPath (pt)	[m] of delay added; absolute
; 10 DelayJitter (pt)	RMS ME=0.3-0.7, mean = 1-2 microns
; 11 NATJitter (pt)	sqrt(RMSX^2+RMSY^2), from residual image motion ME=0.026
; 12 NATCounts (pt)	ME/MV=0.03
; 13 GrpDelay (pt)	Correction to FDL
; 14 DryDelay (pt)	Correction to FDL
; 15 WetDelay (pt)	Correction to FDL
; 16 VacDelay (pt)	Fringe tracking delay
; 17 FDLDelay (pt)	[mu] of delay rel. to ref. delay line -sidereal d.
; 18 MetroDelay (pt)	[mu] of delay relative to reference station
; 19 Time (BG)
; 20 ScanNo (BG)
; 21 BGRate (BG)
; 22 Time (sc)		[s] since 0 UT of date
; 23 ScanNo (sc)
; 24 VisSq (sc)		<X^2+Y^2-N>/<N-D>^2
; 25 VisSq c (sc)	calibrated V^2
; 26 VisSq/e (sc)	V^2/estimated V^2
; 27 VisSq c/e (sc)	cal. V^2/est. V^2
; 28 TripleAmp (sc)
; 29 TripleAmp c (sc)	
; 30 TripleAmp/e (sc)
; 31 TripleAmp c/e (sc)
; 32 TriplePhase (sc)	Closure phase
; 33 TriplePhase c (sc)	Calibrated closure phase - n*360
; 34 PhotonRate (sc)	Rate/2 ms - background
; 34b PhotonRate c (sc) Calibrated photonrate
; 35 BackgndRate (sc)	/2 ms
; 36 FDLPath (sc) 	[m] of delay added; absolute
; 37 DelayJitter (sc)	Point-averaged delay RMS, based on CONSTRICTOR
; 38 TrackJitter (sc)   Jitter of NAT1^2+NAT2^2, based on CONSTRICTOR
; 39 FDLDelay (sc)	Delay relative to reference station
; 40 GeoDelay (sc)	Geometric delay - delay offset
; 41 uv-Radius (sc)
; 42 HourAngle 
; 43 ZenithAngle
; 44 MirrorAngle	2*angle to mirror normal
; 45 Wavelength		[mu]
; 46 Channel
; 47 BLlength
; 48 Baseline
; 49 Triple
; 50 U			[m]
; 51 V			[m]
; 52 W			[m]
; 53 FDL_O-C (sc)
; 54 Grp_O-C (sc)	FDL+group delay
; 55 Dry_O-C (sc)	FDL+group delay+dry air correction
; 56 Wet_O-C (sc)	FDL+group delay+wet air correction
; 57 VisSq m (sc)	Model
; 58 TripleAmp m (sc)	Model
; 59 TriplePhase m (sc)	Model
; 60 MetroTime (mt)	[s] since 0 UT
; 61 Par X (mt)
; 62 Par Y (mt)
; 63 Par Z (mt)
; 64 MetroPath (mt)
; 65 MetroDelay (mt)
; 66 ModelDelay (sc)
; 67 MetroDelay (sc)
; 68 Hour angle (mt)
; 69 Declination (mt)
; 70 Azimuth (mt)
; 71 Elevation (mt)
; 72 Par X C (mt)
; 73 Par Y C (mt)
; 74 Par Z C (mt)
; 75 NATCounts (sc)	Averaged total quad cell counts
; 76 BeamCounts (sc)	The sum of all NAT counts in an output beam
; 77 SpecCounts (sc)	Total photonrate in spectrometer
; 78 Azimuth (sc)  	Azimuth of star (horizon coordinates)
; 79 uv-Angle (sc)	Position angle (azimuth) of projected baseline
; 80 SI (sc)		Scintillation index
; 81 R0 (sc)		R0, Fried's parameter
; 82 T0 (sc)		Coherence time [ms]
; 83 DelayRMS (sc)	RMS of delay during scan (corr. for sidereal rate)
; 84 PhaseRMS (sc)	RMS of FINITO phase
; 85 DelayJitter (sc)	Short term delay RMS, alternative to item 37
; 86 NATOffsetXRMS	NAT or IRIS, in both cases residual image motion
; 87 NATOffsetYRMS	NAT or IRIS, in both cases residual image motion
; 88 NATOffsetXYRMS	Geometric mean of X and Y RMS
; 89 TrackJitter (sc)	Baseline value of NATOffsetXYRMS
; 90 DiffPhase (sc)
; 91 DiffPhaseC (sc)
; 92 DiffPhaseM (sc)
;-------------------------------------------------------------------------------
function set_axitems,item_ids,units,abbrs
;
; Initialize item_ids for a given plot class (i.e. selection of data for 
; a plot widget), and return string arrays containing the item names and units.
; Note: items referred to by sequence number!
; Label extensions: [units]{abbreviation for flag info}
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
if n_elements(class) eq 0 then begin
	print,'***Error(SET_AXITEMS): class not defined!'
	return,''
endif
;
if n_elements(ps_options) eq 0 then pflag=0 else pflag=ps_options.p
;
ax_items=[ $
	['Time [h] {TIME}','Time [h]'], $
	['Index {PT}','Index'], $
	['VisSq {VISSQ}','Squared visbility'], $
	['FASN {FASN}','Fringe amplitude SNR'], $
	['VisPhase [deg] {TPHAS}','Visibility phase'], $
	['TripleAmp {TAMP}','Triple amplitude'], $
	['TriplePhase [deg]','Closure phase'], $
	['PhotonRate {RATE}','Photon rate'], $
	['FDLPath [m]{FDLP}','FDL path'], $
	['DelayJitter [mu]{DJITTR}','Delay jitter'], $
	['NATJitter {TJITTR}','NAT jitter'], $
	['NATCounts {TCOUNTS}','NAT counts'], $
	['GrpDelay [mu]{GRPD}','Group delay [mu]'], $
	['DryDelay [mu]{DRYD}','Dry air disp. corr. delay [mu]'], $
	['WetDelay [mu]{WETD}','Wet air disp. corr. delay [mu]'], $
	['VacDelay [mu]{VACD}','Vacuum delay [microns]'], $
	['FDLDelay (res.) [mu]{FDLD}','FDL delay [microns]'], $
	['MetroDelay [mu]','Metrology delay [microns]'], $
	['Time [h]{TIME}','Time [h]'], $
	['ScanNo {SC}','Scan'], $
	['BGRate {RATE}','Back ground rate'], $
	['Time [h]{TIME}','Time [h]'], $
	['ScanNo {SC}','Scan'], $
	['VisSq {VISSQ}','Squared visibility'], $
	['VisSq c {VISSQ}','Squared visibility'], $
;	['Diameter (mas)','Size (mas)'], $
	['VisSq/e {VISSQ}','Squared visibility'], $
	['VisSq c/e {VISSQ}','Squared visibility'], $
	['TrplAmp {TAMP}','Triple amplitude'], $
	['TrplAmp c {TAMP}','Triple amplitude'], $
	['TrplAmp/e {TAMP}','Triple amplitude'], $
	['TrplAmp c/e {TAMP}','Triple amplitude'], $
	['TrplPhase [deg]{TPHAS}','Closure phase [deg]'], $
	['TrplPhase c [deg]{TPHAS}','Closure phase [deg]'], $
	['PhotonRate {RATE}','Photon rate'], $
	['BackgndRate {RATE}','Back ground rate'], $
	['FDLPath [m] {FDLP}','FDL path'], $
	['DelayJitter [mu]{JITTR}','Delay jitter [microns]'], $
	['TrackJitter','Angle tracking jitter'], $
	['FDLDelay [m] {FDLD}','FDL delay [m]'], $
	['GeoDelay [m]','Geometric delay [m]'], $
;	['UV-Radius [cycles/arcsec]','UV-radius [cycles/arcsec]'], $
	['UV-Radius [M'+greek('lambda')+']','Projected baseline length [M'+greek('lambda')+']'], $
	['HourAngle [h]','Hour angle'], $ 
	['ZenithAngle [deg]','Zenith angle'], $
	['MirrorAngle [deg]','Mirror angle'], $
	['Wavelength [mu]','Wavelength ['+greek('mu')+'m]'], $
	['Channel','Channel'], $
	['BLlength','Baseline length'], $
	['Baseline [m]','Baseline'], $
	['Triple','Triple'], $
	['U [M'+greek('lambda')+']','U [M'+greek('lambda')+']'], $
	['V [M'+greek('lambda')+']','V [M'+greek('lambda')+']'], $
	['W [M'+greek('lambda')+']','W [M'+greek('lambda')+']'], $
	['FDL_O-C [mu]','FDL O-C [microns]'], $
	['Grp_O-C [mu]','grp O-C [microns]'], $
	['Dry_O-C [mu]','Dry O-C [microns]'], $
	['Wet_O-C [mu]','Wet O-C [microns]'], $
	['VisSq m','Model squared visibility'], $
	['TripleAmp m','Model triple amplitude'], $
	['TriplePhase m','Model closure phase'], $
	['Time [h]','Time [h]'], $
	['Par X [mu]','X [microns]'], $
	['Par Y [mu]','Y [microns]'], $
	['Par Z [mu]','Z [microns]'], $
	['MetroPath [mu]','Metrology path [microns]'], $
	['MetroDelay [mu]','Metrology delay [microns]'], $
	['ModelDelay [mu]','Model delay [microns]'], $
	['MetroDelay [mu]','Metrology delay [microns]'], $
	['Hour angle [h]','Hour angle'], $
	['Declination [d]','Declination'], $
	['Azimuth [d]','Azimuth [deg]'], $
	['Elevation [d]','Elevation [deg]'], $
	['Par X C [microns]','X (corr.) [microns]'], $
	['Par Y C [microns]','Y (corr.) [microns]'], $
	['Par Z C [microns]','Z (corr.) [microns]'], $
	['NATCounts','NAT counts'], $
	['BeamCounts','Beam counts'], $
	['SpecCounts','SpecCounts'], $
	['Azimuth [deg]','Azimuth'], $
	['BL PA [deg]','Baseline PA [deg]'], $
	['SI','Scintillation index'], $
	['R0 ["]','R0 ["]'], $
	['T0','T0 [ms]'], $
	['DelayRMS [mu]','DelayRMS [mu]'], $
	['PhaseRMS [mu]','PhaseRMS [deg]'], $
	['DelayJitter2','DelayJitter2'], $
	['NATXRMS','NATXRMS'], $
	['NATYRMS','NATYRMS'], $
	['NATRMS','NATRMS'], $
	['TrackJitter2','Angle jitter'], $
	['DiffPhase','Differential phase [deg]'], $
	['DiffPhaseC','Differential phase [deg]'], $
	['DiffPhaseM','Model Differential phase [deg]'] $
	]
;
if pflag then ax_items=reform(ax_items(1,*)) $
	 else ax_items=reform(ax_items(0,*))
;
; Extract units and abbreviations
units=strarr(n_elements(ax_items))
abbrs=strarr(n_elements(ax_items))
for i=0,n_elements(ax_items)-1 do begin
	pos=strlen(ax_items(i))
	pos1=strpos(ax_items(i),'[')
	pos2=strpos(ax_items(i),']')
	if pos1 ne -1 and pos2 ne -1 then begin
		pos=min([pos,pos1])
		units(i)=strmid(ax_items(i),pos1,pos2-pos1+1)
	endif
	pos3=strpos(ax_items(i),'{')
	pos4=strpos(ax_items(i),'}')
	if pos3 ne -1 and pos4 ne -1 then begin
		pos=min([pos,pos3])
		abbrs(i)=strmid(ax_items(i),pos3+1,pos4-pos3-1)
	endif
	ax_items(i)=strmid(ax_items(i),0,pos)
endfor
;
; Set list of items for each class
case class of
	'point' :item_ids=[1,2,46,3,4,5,6,7,8,18,10,11,12]
	'delay' :item_ids=[1,2,46,9,17,13,14,15,16,5]
	'bg'    :item_ids=[19,20,21,46]
	'scan'  :begin
		 item_ids=[22,23,45,46,42,43,44,78, $
			   24,25,26,27,34,35,28,29,30,31, $
			   32,33]
		 if strpos(instrument_id(systemid),'NPOI') ge 0 then $
		 item_ids=[22,23,45,46,42,43,44,78, $
			   24,25,26,27,28,29,30,31, $
			   32,33,34,35,75,76,77]
		 end
	'astrom':item_ids=[22,42,39,40,66,67,53,54,55,56]
	'seeing':item_ids=[22,80,81,82,83,84,37,85,86,87,88,38,89,43,26,34,45]
	'phot'	:item_ids=[22,23,43,88,75]
	'uv'    :item_ids=[25,26,41,79,50,51]
	'metro' :item_ids=[60,68,69,70,71,61,62,63,72,73,74,64,65]
	'amoeba':begin
		 item_ids=[22,23,45,46,42,41,25,26,91,29,33]
		 if strpos(instrument_id(systemid),'NPOI') ge 0 then $
		 item_ids=[22,23,45,46,42,41,25,29,33]
		 end
	'volvox':item_ids=[22,23,53,54,55]
;	ncal produces plots with these pre-selected (x,y,z)-axes
	'ncal'  :begin
		 if strpos(type,'Response') ge 0 then begin
		 ax_items=[ $
		  '(B-V)', $
		  'Mag/count (1 bl.)', $
		  'Channel']
		 item_ids=[1,2,3]
		 endif else if strpos(type,'V2Bias') ge 0 then begin
		 ax_items=[ $
		  'Photonrate', $
		  'Squared visibility', $
		  'Channel']
		 item_ids=[1,2,3]
		 endif else if strpos(type,'TABias') ge 0 then begin
		 ax_items=[ $
		  'Photonrate', $
		  'TripleAmp', $
		  'Channel']
		 item_ids=[1,2,3]
		 endif else if strpos(type,'APDFlux') ge 0 then begin
		 ax_items=[ $
		  'Scan', $
		  'Rel. rate (O/C)', $
		  'Channel']
		 item_ids=[1,2,3]
		 endif else if strpos(type,'TrackJitter') ge 0 then begin
		 ax_items=[ $
		  'Photonrate', $
		  'TrackJitter', $
		  'Channel']
		 item_ids=[1,2,3]
		 endif else if strpos(type,'TrackOffset') ge 0 then begin
		 ax_items=[ $
		  'Offset', $
		  'Squared visibility', $
		  'Channel']
		 item_ids=[1,2,3]
		 endif
		 end
	    else:begin
		 print,'***Error(SET_AXITEMS): invalid class ',class,'!'
		 return,''
		 end
endcase
units=units(item_ids-1)
return,ax_items(item_ids-1)
;
end
;-------------------------------------------------------------------------------
pro set_sliceids,item,ib,ob,tr,ch,bl,pt
;
; Procedure to replace a single input data array indices with array of same 
; value. One of the inputs is an array already; it is the one which corresponds 
; to the selected slice.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
case slice of
	'ib':n=n_elements(ib)
	'ob':n=n_elements(ob)
	'tr':n=n_elements(tr)
	'ch':n=n_elements(ch)
	'bl':n=n_elements(bl)
	'pt':n=n_elements(pt)
endcase
;
ids=lonarr(n) 
;
ids(*)=item & item=ids
ids(*)=ib & ib=ids
ids(*)=ob & ob=ids
ids(*)=tr & tr=ids
ids(*)=ch & ch=ids
ids(*)=bl & bl=ids
ids(*)=pt & pt=ids
;
end
;-------------------------------------------------------------------------------
function set_plotdata,item,ib,ob,tr,ch,bl,pt
;
; Given data array indices, return corresponding data for item.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common Tables,ScanTable,BGTable,StationTable
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common SeeingData,scintillation,r0,t0,delayrms,phaserms,delayjitter2,natjitter2
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
ml2cpa=1e6/(RAD*3600)
if item(0) eq 3 or item(0) eq 6 then begin
case SystemId of
        'Mark3'   :normfactor=pi_circle^2/2
        'NPOI'    :normfactor=4.0/sinc(float(GenConfig.FringeMod(bl,ob)) $
				       /GenConfig.NumBin)^2
              else:normfactor=1.0
endcase
endif
ibr=GenConfig.RefStation-1
;
; Intercept some common user errors
icheck=0
if long(where([6,7,28,29,30,31,32,33] eq item),0) ne -1 $
	and GenConfig.NumTriple eq 0 then icheck=1
if icheck then begin
	print,'***Error(SET_PLOTDATA): no data to plot!'
	return,-1
endif
;
case item(0) of
;	   Time (pt)		[s] since 0 UT of date
	 1:return,abs(PointTime(pt))/3600
;	   Index (pt)		An index number
	 2:if ps_options.n then return,float(pt)+1 $
			   else return,findgen(n_elements(pt))+1
	 2:return,findgen(n_elements(pt))+1
;	   VisSq (pt)		X^2+Y^2-N
 	 3:return,VisSq(ob,ch,bl,pt)*normfactor $
 		 /nonzero(PhotonRate(ob,ch,pt))^2
;	   FASN (pt)
	 4:return,VisSq(ob,ch,bl,pt)
;	   VisPhase (pt)
	 5:return,VisPhase(ob,ch,bl,pt)*RAD
;	 6:return,TripleAmp(tr,ch,pt)
;	   TripleAmp (pt)
	 6:return,TripleAmp(tr,ch,pt)*normfactor^1.5 $
		 /nonzero(Photonrate(GenConfig.TripleBeam(0,tr), $
				     GenConfig.TripleChan(ch,0,tr),pt) $
		         *PhotonRate(GenConfig.TripleBeam(1,tr), $
				     GenConfig.TripleChan(ch,1,tr),pt) $
			 *PhotonRate(GenConfig.TripleBeam(2,tr), $
				     GenConfig.TripleChan(ch,2,tr),pt))
;	   TriplePhase (pt)
	 7:return,TriplePhase(tr,ch,pt)*RAD
;	   PhotonRate (pt)	/2 ms
	 8:return,PhotonRate(ob,ch,pt)
;	   FDLPath (pt)	[m] of delay added; absolute
	 9:return,FDLPos(ib,pt)+FDLPos(ibr,pt)*(ib ne ibr)
;	   DelayJitter (pt)	RMS
	10:return,DelayJitter(ob,bl,pt)*1000000
;	   NATJitter (pt)
	11:return,NATJitter(ib,pt)
;	   NATCounts (pt)
	12:return,NATCounts(ib,pt)
;	   GrpDelay (pt)	Correction to FDL
	13:return,GrpDelay(ib,pt)
;	   DryDelay (pt)	Correction to FDL
	14:return,DryDelay(ib,pt)
;	   WetDelay (pt)	Correction to FDL
	15:return,WetDelay(ib,pt)
;	   VacDelay (pt)	Fringe tracking delay
	16:return,(FDLPos(ib,pt)*(ib ne ibr)-GeoDelay(ib,pt))*1d6 $
	 	 -DryDelay(ib,pt)
;	   FDLDelay (pt)	[mu] of delay rel. to ref. delay line -sidereal d.
	17:return,(FDLPos(ib,pt)*(ib ne ibr)-GeoDelay(ib,pt))*1e6
;	   MetroDelay (pt)	[mu] of delay relative to reference station
	18:begin
	   if n_elements(MetroPos) le 1 then return,0
	   if ib eq ibr then return,MetroPos(ib,pt)*0 $
			else return,MetroPos(ib,pt)
	   end
;	   Time (BG)
	19:return,abs(bgscans(pt).Time/3600)
;	   ScanNo (BG)
	20:return,float(pt)+1
;	   BGRate (BG)
	21:return,bgscans(pt).Rate(ob,ch)
;	   Time (sc)		[s] since 0 UT of date
	22:return,abs(scans(pt).Time/3600)
;	   ScanNo (sc)
	23:return,float(pt)+1
;	   VisSq (sc)		<X^2+Y^2-N>/<N-D>^2
	24:return,scans(pt).VisSq(ob,ch,bl)
;	   VisSq c (sc)	calibrated V^2
	25:return,scans(pt).VisSqC(ob,ch,bl)
;	   Diameter based on VisSq c (sc)
;	25:return,hb(sqrt(scans(pt).UVW(ob,ch,bl,0)^2 $
;		         +scans(pt).UVW(ob,ch,bl,1)^2)/1e6, $
;		scans(pt).VisSqC(ob,ch,bl))
;	   VisSq/e (sc)	V^2/estimated V^2
	26:return,scans(pt).VisSqE(ob,ch,bl)
;	   VisSq c/e (sc)	cal. V^2/est. V^2
	27:return,scans(pt).VisSqEC(ob,ch,bl)
;	   TripleAmp (sc)
	28:return,scans(pt).TripleAmp(tr,ch)
;	   TripleAmp c (sc)	
	29:return,scans(pt).TripleAmpC(tr,ch)
;	   TripleAmp/e (sc)
	30:return,scans(pt).TripleAmpE(tr,ch)
;	   TripleAmp c/e (sc)
	31:return,scans(pt).TripleAmpEC(tr,ch)
;	   TriplePhase (sc)	Closure phase
	32:return,scans(pt).TriplePhase(tr,ch)*RAD
;	   TriplePhase c (sc)	Calibrated closure phase - n*360
	33:return,scans(pt).TriplePhaseC(tr,ch)*RAD
;	   PhotonRate (sc)	Rate/2 ms - background
	34:return,scans(pt).PhotonRate(ob,ch)
;	   BackgndRate (sc)	/2 ms
	35:return,scans(pt).BackgndRate(ob,ch)
;	   FDLPath (sc) 	[m] of delay added; absolute
	36:return,scans(pt).FDLPos(ib)+scans(pt).FDLPos(ibr)*(ib ne ibr)
;	   DelayJitter (sc)	Based on CONSTRICTOR
	37:return,scans(pt).DelayJitter(ob,bl)*1000000
;	   TrackJitter (sc)   Jitter of NAT1^2+NAT2^2
	38:begin
	   i=where(GenConfig.StationId eq $
	    strmid(GenConfig.BaselineId(bl,ob),0,3))
	   j=where(GenConfig.StationId eq $
	    strmid(GenConfig.BaselineId(bl,ob),4,3))
	   tjc=genconfig.trackcal(*,bl,ch,ob,0)
	   if total(abs(tjc)) eq 0 then tjc=[1.0,0.0,0.0]
	   return,(sqrt(scans(pt).NATJitter(i)^2 $
		       +scans(pt).NATJitter(j)^2)) $
		/poly(scans(pt).photonrate(ob,ch),tjc)
	   end
;	   FDLDelay (sc)	Delay relative to reference station
	39:return,scans(pt).FDLPos(ib)*(ib ne ibr)
;	   GeoDelay (sc)	Geometric delay - delay offset
	40:return,scans(pt).GeoDelay(ib)
;	   uv-Radius (sc)
;	41:return,sqrt(scans(pt).UVW(ob,ch,bl,0)^2 $
;		      +scans(pt).UVW(ob,ch,bl,1)^2)/1e6*ml2cpa
	41:return,sqrt(scans(pt).UVW(ob,ch,bl,0)^2 $
		      +scans(pt).UVW(ob,ch,bl,1)^2)/1e6
;	   HourAngle 
	42:return,scans(pt).HA
;	   ZenithAngle
	43:return,scans(pt).ZA
;	   MirrorAngle	2*angle to mirror normal
	44:return,scans(pt).MA
;	   Wavelength		[mu]
	45:return,GenConfig.Wavelength(ch,ob)*1d6
;	   Channel
	46:return,float(ch)+1
;	   BLlength
	47:return,fltarr(MB)+1
;	   Baseline
	48:return,findgen(MB)+1
;	   Triple
	49:return,findgen(MT)+1
;	   U			[m]
	50:return,scans(pt).UVW(ob,ch,bl,0)/1000000
;	   V			[m]
	51:return,scans(pt).UVW(ob,ch,bl,1)/1000000
;	   W			[m]
	52:return,scans(pt).UVW(ob,ch,bl,2)/1000000
;	   FDL_O-C (sc)
	53:return,(scans(pt).FDLPos(ib)*(ib ne ibr)-scans(pt).GeoDelay(ib))*1e6
;	   Grp_O-C (sc)	FDL+group delay
	54:return,(scans(pt).GrpDelay(ib)-scans(pt).GeoDelay(ib))*1000000
;	   Dry_O-C (sc)	FDL+group delay+dry air correction
	55:return,(scans(pt).DryDelay(ib)-scans(pt).GeoDelay(ib))*1000000
;	   Wet_O-C (sc)	FDL+group delay+wet air correction
	56:return,(scans(pt).WetDelay(ib)-scans(pt).GeoDelay(ib))*1000000
;	   VisSq m (sc)	Model
	57:return,scans(pt).VisSqM(ob,ch,bl)
;	   Diameter based on model visibility
;	57:return,hb(sqrt(scans(pt).UVW(ob,ch,bl,0)^2 $
;		         +scans(pt).UVW(ob,ch,bl,1)^2)/1e6, $
;		scans(pt).VisSqM(ob,ch,bl))
;	   TripleAmp m (sc)	Model
	58:return,scans(pt).TripleAmpM(tr,ch)
;	   TriplePhase m (sc)	Model
	59:return,scans(pt).TriplePhaseM(tr,ch)*RAD
;	   MetroTime (mt)	[s] since 0 UT
	60:return,MetroTime(pt)/3600
;	   Par X (mt)
	61:return,ParX(ib,pt)
;	   Par Y (mt)
	62:return,ParY(ib,pt)
;	   Par Z (mt)
	63:return,ParZ(ib,pt)
;	   MetroPath (mt)
	64:return,MetroPath(ib,pt)
;	   MetroDelay (mt)
	65:begin
	   if ib eq ibr then return,MetroPath(ib,pt)*0 $
			else return,MetroPath(ib,pt)-MetroPath(ibr,pt)
	   end
;	   ModelDelay (sc)
	66:return,scans(pt).ModelDelay(ib)
;	   MetroDelay (sc)
	67:return,scans(pt).MetroDelay(ib)
;	   Hour angle (mt)
	68:begin
	   hadec=sidpointing(sidmodel(ib+1),motorangle(*,ib,pt),1)
	   return,hadec(0,*)
	   end
;	   Declination (mt)
	69:begin
	   hadec=sidpointing(sidmodel(ib+1),motorangle(*,ib,pt),1)
	   return,hadec(1,*)
	   end
;	   Azimuth (mt)
	70:return,motorangle(0,ib,pt) $
		+double(double(sidmodel(ib+1)),5*sizeof(double(0)))*0
;	   Elevation (mt)
	71:return,motorangle(1,ib,pt) $
		+double(double(sidmodel(ib+1)),7*sizeof(double(0)))*0
;	   Par X C (mt)
	72:begin
	   hadec=sidpointing(sidmodel(ib+1),motorangle(*,ib,pt),1)
	   mp=mirrorpivot(sidmodel(ib+1),hadec(0,*),hadec(1,*), $
		MetroConfig.SidModel.PivotOffset(*,ib))
	   return,ParX(ib,pt)-mp(*,0)
	   end
;	   Par Y C (mt)
	73:begin
	   hadec=sidpointing(sidmodel(ib+1),motorangle(*,ib,pt),1)
	   mp=mirrorpivot(sidmodel(ib+1),hadec(0,*),hadec(1,*), $
		MetroConfig.SidModel.PivotOffset(*,ib))
	   return,ParY(ib,pt)-mp(*,1)
	   end
;	   Par Z C (mt)
	74:begin
	   hadec=sidpointing(sidmodel(ib+1),motorangle(*,ib,pt),1)
	   mp=mirrorpivot(sidmodel(ib+1),hadec(0,*),hadec(1,*), $
		MetroConfig.SidModel.PivotOffset(*,ib))
	   return,ParZ(ib,pt)-mp(*,2)
	   end
;	   NATCounts (sc)	Averaged total quad cell counts
	75:return,scans(pt).NATCounts(ib)
;	   BeamCounts (sc)	The sum of all NAT counts in an output beam
	76:return,total(scans(pt).NATCounts(*)*beamconfig(ob+1,pt+1),1)
;	   SpecCounts (sc)	Total photonrate in spectrometer
	77:return,total(scans(pt).PhotonRate(ob,0:genconfig.numspecchan(ob)-1) $
	       *(signof(scans(pt).PhotonRateErr(ob,0:genconfig.numspecchan(ob)-1))>0),2)
;	   Azimuth (sc)  	Azimuth of star (horizon coordinates)
	78:return,scans(pt).AZ/RAD
;	   uv-Angle (sc)	Position angle of projected baseline
	79:begin
	   p=-atan(avg(scans(pt).UVW(ob,ch,bl,0)),avg(scans(pt).UVW(ob,ch,bl,1)))
	   m=[[cos(p),-sin(p)],[sin(p),cos(p)]]
	   uv=transpose(m#transpose([[scans(pt).UVW(ob,ch,bl,0)], $
				     [scans(pt).UVW(ob,ch,bl,1)]]))
	   blpa=(atan(uv(*,0),uv(*,1))-p)*RAD
	   index=where(blpa lt 0,count)
	   if count gt 0 then blpa(index)=blpa(index)+180
;	   Newer, cleaner calculation, BLPA=degrees east of north
	   blpa=atan(scans(pt).UVW(ob,ch,bl,0), $
		     scans(pt).UVW(ob,ch,bl,1))*RAD
	   index=where(blpa lt 0,count)
	   if count gt 0 then blpa(index)=blpa(index)+180
	   if blpa(0) gt blpa(1) then blpa(0)=blpa(0)-180
	   return,blpa
	   end
;	   SI (sc)		Scintillation index
	80:return,scans(pt).si(ob,ch)
;	   R0 (sc) 		Fried's parameter
	81:return,scans(pt).r0
;	   T0 (sc)		Coherence time [ms]
	82:return,scans(pt).t0(ob,ch,bl)
;	   DelayRMS (sc)	RMS of residual delay over scan
	83:return,scans(pt).delayrms(ob,bl)
;	   PhaseRMS (sc)	RMS of FINITO phase
	84:return,scans(pt).phaserms(ob,bl)
;	   DelayJitter (sc)	Short term RMS of delay
	85:return,scans(pt).delayjitter2(ob,bl)
;	   NATOffsetXRMS
	86:return,scans(pt).natjitter2(ib,0)
;	   NATOffsetYRMS
	87:return,scans(pt).natjitter2(ib,1)
;	   NATOffsetXYRMS	Geometric mean of X and Y RMS
	88:return,sqrt(scans(pt).natjitter2(ib,0)*scans(pt).natjitter2(ib,1))
;	   AngleJitter		Baseline value of NATOffsetXY
	89:begin
	   i=where(GenConfig.StationId eq $
	    strmid(GenConfig.BaselineId(bl,ob),0,3))
	   j=where(GenConfig.StationId eq $
	    strmid(GenConfig.BaselineId(bl,ob),4,3))
	   return,sqrt(scans(pt).natjitter2(i,0)*scans(pt).natjitter2(i,1) $
		      +scans(pt).natjitter2(j,0)*scans(pt).natjitter2(j,1))
	   end
;	   DiffPhase (sc)
	90:return,scans(pt).DiffPhase(ob,ch,bl)*RAD
;	   DiffPhaseC (sc)
	91:return,scans(pt).DiffPhaseC(ob,ch,bl)*RAD
;	   DiffPhaseM (sc)
	92:return,scans(pt).DiffPhaseM(ob,ch,bl)*RAD
      else:begin
	   print,'***Error(SET_PLOTDATA): item not allowed:',item
	   return,-1
	   end
endcase
;
end
;-------------------------------------------------------------------------------
function set_ploterr,item,ib,ob,tr,ch,bl,pt
;
; Given data array indices, return corresponding data errors for item.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common Tables,ScanTable,BGTable,StationTable
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common SeeingData,scintillation,r0,t0,delayrms,phaserms,delayjitter2,natjitter2
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
if item(0) eq 3 or item(0) eq 6 then begin
case SystemId of
        'Mark3'   :normfactor=pi_circle^2/2
        'NPOI'    :normfactor=4.0/sinc(float(GenConfig.FringeMod(bl,ob)) $
				       /GenConfig.NumBin)^2
              else:normfactor=1.0
endcase
endif
ibr=GenConfig.RefStation-1
;
; Intercept some common user errors
icheck=0
if long(where([6,7,28,29,30,31,32,33] eq item),0) ne -1 $
	and GenConfig.NumTriple eq 0 then icheck=1
if icheck then begin
	print,'***Error(SET_PLOTDATA): no data to plot!'
	return,-1
endif
;
case item(0) of
	 1:return,(PointTime(pt) ge 0)*2-1
	 2:return,(PointTime(pt) ge 0)*2-1
;	 3:return,signof(VisSqErr(ob,ch,bl,pt),PhotonRateErr(ob,ch,pt)) $
;	    *normfactor $
;           *abs(VisSq(ob,ch,bl,pt)/nonzero(PhotonRate(ob,ch,pt))^2) $
;           *sqrt((VisSqErr(ob,ch,bl,pt)/nonzero(VisSq(ob,ch,bl,pt)))^2 $
;                 +4*(PhotonRateErr(ob,ch,pt)/nonzero(PhotonRate(ob,ch,pt)))^2)
	 3:return,VisSqErr(ob,ch,bl,pt)*normfactor $
		 /nonzero(PhotonRate(ob,ch,pt))^2
	 4:return,VisSqErr(ob,ch,bl,pt)
	 5:return,VisPhaseErr(ob,ch,bl,pt)*RAD
;	 6:return,TripleAmpErr(tr,ch,pt)
	 6:return,TripleAmpErr(tr,ch,pt)*normfactor^1.5 $
		 /nonzero(Photonrate(GenConfig.TripleBeam(0,tr), $
				     GenConfig.TripleChan(ch,0,tr),pt) $
		         *PhotonRate(GenConfig.TripleBeam(1,tr), $
				     GenConfig.TripleChan(ch,1,tr),pt) $
			 *PhotonRate(GenConfig.TripleBeam(2,tr), $
				     GenConfig.TripleChan(ch,2,tr),pt))
	 7:return,TriplePhaseErr(tr,ch,pt)*RAD
	 8:return,PhotonRateErr(ob,ch,pt)
	 9:return,FDLPosErr(ib,pt)
	10:return,DelayJitterErr(ob,bl,pt)*1000000
	11:return,NATJitterErr(ib,pt)
	12:return,NATCountsErr(ib,pt)
	13:return,GrpDelayErr(ib,pt)
	14:return,DryDelayErr(ib,pt)
	15:return,WetDelayErr(ib,pt)
	16:begin
	   sign=fltarr(n_elements(pt))+1
	   index=where(FDLPosErr(ib,pt) lt 0 $
		    or FDLPosErr(ibr,pt) lt 0 $
		    or DryDelayErr(ib,pt) lt 0,count)
	   if count gt 0 then sign(index)=-1
	   return,sqrt((FDLPosErr(ib,pt)*1d6)^2 $
		      +DryDelayErr(ib,pt)^2)*sign
	   end
	17:return,(abs(FDLPosErr(ib,pt)) $
	      *signof(FDLPosErr(ib,pt),FDLPosErr(ibr,pt)))*1e6
	18:begin
	   if n_elements(MetroPosErr) le 1 then return,0
	   return,MetroPosErr(ib,pt)
	   end
	19:return,(bgscans(pt).Time ge 0)*2-1
	20:return,(bgscans(pt).Time ge 0)*2-1
	21:return,bgscans(pt).RateErr(ob,ch)
	22:return,(scans(pt).Time ge 0)*2-1
	23:return,(scans(pt).Time ge 0)*2-1
	24:return,scans(pt).VisSqErr(ob,ch,bl)
	25:return,scans(pt).VisSqCErr(ob,ch,bl)
	250:begin
	   sigma=-genconfig.wavelength(ch,ob)/4e-6*0.1+0.4
	   return,(scans(pt).VissqC(ob,ch,bl)*sigma > 0.002) $
		*signof(scans(pt).VisSqCErr(ob,ch,bl))
	   end
	26:return,scans(pt).VisSqEErr(ob,ch,bl)
	27:return,scans(pt).VisSqECErr(ob,ch,bl)
	28:return,scans(pt).TripleAmpErr(tr,ch)
	29:return,scans(pt).TripleAmpCErr(tr,ch)
	30:return,scans(pt).TripleAmpEErr(tr,ch)
	31:return,scans(pt).TripleAmpECErr(tr,ch)
	32:return,scans(pt).TriplePhaseErr(tr,ch)*RAD
	33:return,scans(pt).TriplePhaseCErr(tr,ch)*RAD
	34:return,scans(pt).PhotonRateErr(ob,ch)
	35:return,scans(pt).BackgndErr(ob,ch)
	36:return,scans(pt).FDLPosErr(ib)
	37:return,scans(pt).DelayJitterErr(ob,bl)*1000000
	38:begin
	   i=where(GenConfig.StationId eq $
	    strmid(GenConfig.BaselineId(bl,ob),0,3))
	   j=where(GenConfig.StationId eq $
	    strmid(GenConfig.BaselineId(bl,ob),4,3))
	   sign=fltarr(n_elements(pt))+1
	   index=where(scans(pt).NATJitterErr(i) lt 0 $
		    or scans(pt).NATJitterErr(j) lt 0,count)
	   if count gt 0 then sign(index)=-1
	   return,(sqrt(scans(pt).NATJitterErr(i)^2 $
		       +scans(pt).NATJitterErr(j)^2))*sign
	   end
	39:return,abs(scans(pt).FDLPosErr(ib)) $
	      *signof(scans(pt).FDLPosErr(ib),scans(pt).FDLPosErr(ibr))
	40:return,scans(pt).GeoDelay(ib)*0+1.e-6
	41:return,scans(pt).UVW(ob,ch,bl,0)*0
	42:return,fltarr(n_elements(pt))+1.0
	43:return,fltarr(n_elements(pt))+1.0
	44:return,fltarr(n_elements(pt))+1.0
	45:return,GenConfig.WavelengthErr(ch,ob)*1d6
	46:return,fltarr(n_elements(ch))+0.1
	47:return,fltarr(n_elements(bl))
	48:return,fltarr(n_elements(bl))
	49:return,fltarr(n_elements(tr))
	50:return,scans(pt).UVW(ob,ch,bl,0)*0
	51:return,scans(pt).UVW(ob,ch,bl,0)*0
	52:return,scans(pt).UVW(ob,ch,bl,0)*0
	53:return,abs(scans(pt).FDLPosErr(ib)*1000000) $
	      *signof(scans(pt).FDLPosErr(ib),scans(pt).FDLPosErr(ibr))
	54:return,scans(pt).GrpDelayErr(ib)*1000000
	55:return,scans(pt).DryDelayErr(ib)*1000000
	56:return,scans(pt).WetDelayErr(ib)*1000000
	57:return,fltarr(n_elements(pt))
	58:return,fltarr(n_elements(pt))
	59:return,fltarr(n_elements(pt))
	60:return,(MetroTime(pt) ge 0)*2-1
	61:return,ParXErr(ib,pt)
	62:return,ParYErr(ib,pt)
	63:return,ParZErr(ib,pt)
	64:return,MetroPathErr(ib,pt)
	65:begin
	   sign=fltarr(n_elements(pt))+1
	   index=where(MetroPathErr(ib,pt) lt 0 $
                    or MetroPathErr(ibr,pt) lt 0,count)
	   if count gt 0 then sign(index)=-1
   	   return,sqrt(MetroPathErr(ib,pt)^2+MetroPathErr(ibr,pt)^2)*sign
	   end
	66:return,scans(pt).ModelDelayErr(ib)
	67:return,scans(pt).MetroDelayErr(ib)
	68:return,(fltarr(n_elements(pt))+1)*signof(motorangleerr(0,ib,pt))
	69:return,(fltarr(n_elements(pt))+1)*signof(motorangleerr(1,ib,pt))
	70:return,motorangleerr(0,ib,pt)
	71:return,motorangleerr(1,ib,pt)
	72:return,ParXErr(ib,pt)
	73:return,ParYErr(ib,pt)
	74:return,ParZErr(ib,pt)
	75:return,scans(pt).NATCountsErr(ib)
	76:return,sqrt(total(scans(pt).NATCountsErr(*)^2*beamconfig(ob+1,pt+1),1))
	77:return,sqrt(total(scans(pt).PhotonRateErr(ob,0:genconfig.numspecchan(ob)-1)^2 $
	            *(signof(scans(pt).PhotonRateErr(ob,0:genconfig.numspecchan(ob)-1))>0),2))
	78:return,fltarr(n_elements(pt))+1.0
	79:return,scans(pt).UVW(ob,ch,bl,0)*0
	80:return,scans(pt).si(ob,ch)*0+1
	81:return,scans(pt).r0*0+1
	82:return,scans(pt).t0(ob,ch,bl)*0+1
	83:return,scans(pt).delayrms(ob,bl)*0+1
	84:return,scans(pt).phaserms(ob,bl)*0+1
	85:return,scans(pt).delayjitter2(ob,bl)*0+1
	86:return,scans(pt).natjitter2(ib,0)*0+1
	87:return,scans(pt).natjitter2(ib,1)*0+1
	88:return,sqrt(scans(pt).natjitter2(ib,0)*scans(pt).natjitter2(ib,1))*0+1
	89:begin
	   i=where(GenConfig.StationId eq $
	    strmid(GenConfig.BaselineId(bl,ob),0,3))
	   j=where(GenConfig.StationId eq $
	    strmid(GenConfig.BaselineId(bl,ob),4,3))
	   return,sqrt(scans(pt).natjitter2(i,0)*scans(pt).natjitter2(i,1) $
		      +scans(pt).natjitter2(j,0)*scans(pt).natjitter2(j,1))*0+1
	   end
	90:return,scans(pt).DiffPhaseErr(ob,ch,bl)*RAD
	91:return,scans(pt).DiffPhaseCErr(ob,ch,bl)*RAD
	92:return,fltarr(n_elements(pt))
      else:begin
	   print,'***Error(SET_PLOTERR): item not allowed:',item
	   return,-1
	   end
endcase
;
end
;-------------------------------------------------------------------------------
function set_caldata,item,outbeam,channel,baseline,x_index,x_mid,x_scl
;
; Returns data of selected calibration indicators. Note that the units are not
; necessarily the same as the ones returned by set_plotdata. This function
; can also be used to set/determine the scaling parameters of the variables for
; normalization purposes. These will approximately transform the range into
; the interval [-1,1]
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
RAD=180/pi_circle
x_mid=0.0
x_scl=1.0
;
case item of
	1:begin
;	  Wavelength [mu]
	  data=fltarr(n_elements(x_index))+ $
	       float(GenConfig.Wavelength(channel,outbeam)*1.0e6)
	  x_mid=0.68
	  x_scl=6.0
	  end
	2:begin
;	  Delay jitter [mu]
	  data=scans(x_index).DelayJitter(outbeam,baseline)*1e6
	  error=scans(x_index).DelayJitterErr(outbeam,baseline)*1e6
	  index=where(error gt 0,count)
	  if count gt 0 then begin
		x_mid=(min(data(index))+max(data(index)))/2
		x_scl=2/(max(data(index))-min(data(index)))
	  endif else begin
	  	x_mid=3.0
	  	x_scl=0.7
	  endelse
	  end
	3:begin
;	  Tracking jitter [Airy disk diameter?]
	  i=where(GenConfig.StationId eq $
	   strmid(GenConfig.BaselineId(baseline,outbeam),0,3))
	  j=where(GenConfig.StationId eq $
	   strmid(GenConfig.BaselineId(baseline,outbeam),4,3))
	  data=sqrt(scans(x_index).NATJitter(i)^2+scans(x_index).NATJitter(j)^2)
	  rate=scans(x_index).photonrate(outbeam,channel)
	  tjc=genconfig.trackcal(*,baseline,channel,outbeam,0)
	  if total(abs(tjc)) eq 0 then tjc=[1.0,0.0,0.0]
	  data=data/poly(rate,tjc)
	  sign=fltarr(n_elements(x_index))+1
	  index=where(scans(x_index).NATJitterErr(i) lt 0 $
		   or scans(x_index).NATJitterErr(j) lt 0,count)
	  if count gt 0 then sign(index)=-1
	  error=sqrt(scans(x_index).NATJitter(i)^2 $
	  	    +scans(x_index).NATJitter(j)^2)*sign
	  index=where(error gt 0,count)
	  if count gt 0 then begin
		x_mid=(min(data(index))+max(data(index)))/2
		x_scl=2/(max(data(index))-min(data(index)))
	  endif else begin
	  	x_mid=0.25
	  	x_scl=30.0
	  endelse
	  end
	4:begin
;	  Zenith angle [rad]
	  data=float(scans(x_index).ZA/RAD)
	  x_mid=30/RAD
	  x_scl=2/(60/RAD)
	  x_mid=(min(data)+max(data))/2
	  x_scl=2/(max(data)-min(data))
	  end
	5:begin
;	  Mirror angle [rad]
	  data=float(scans(x_index).MA/RAD)
	  x_mid=60/RAD
	  x_scl=2/(80/RAD)
	  x_mid=(min(data)+max(data))/2
	  x_scl=2/(max(data)-min(data))
	  end
	6:begin
;    	  Hour angle [h]
	  data=float(scans(x_index).HA)
	  x_mid=0.5
	  x_scl=0.3
	  x_mid=(min(data)+max(data))/2
	  x_scl=2/(max(data)-min(data))
	  end
	7:begin
;	  Time [h]
	  data=float(abs(scans(x_index).time)/3600)
	  index=where(data gt 0,count)
	  if count gt 1 then begin
		x_mid=(min(data(index))+max(data(index)))/2
		x_scl=2/(max(data(index))-min(data(index)))
	  endif else begin
	    	x_mid=7.6
	    	x_scl=0.25
	  endelse
	  end
    else: begin
	  print,'***Error(SET_CALDATA): item not allowed: ',item
	  x_scl=0.0
	  return,-1
	  end
endcase
;
return,data
;
end
;-------------------------------------------------------------------------------
pro set_plotlabels,plot_labels,axis_index,item
; 
; Plot labels indicate the array index settings for plotted variables.
; This procedure prepares a string array containing the labels and an
; integer array indicating which labels apply to the selected axis item.
; The sliced dimension is not labeled.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
plot_labels=sindgen(6)
plot_labels(0)=' IB'
plot_labels(1)=' OB'
plot_labels(2)=' Tr'
plot_labels(3)=' Bl'
plot_labels(4)=' Ch'
plot_labels(5)=' Pt'
axis_index=intarr(6)
;
; IB
list=[9,11,12,13,14,15,16,17,18,36,39,40,41,53,54,55,56,61,62,63,64,65,66,67, $
	68,69,70,71,72,73,74,75,79,86,87,88]
index=where(list eq item,count)
if count gt 0 and slice ne 'ib' then begin
	axis_index(0)=1
	list=[13,14,15,16,17,18,39,40,53,54,55,56,65,67]
	index=where(list eq item,count)
	if count gt 0 then plot_labels(0)=plot_labels(0) $
                          +'('+string(GenConfig.RefStation,format='(i1)')+')'
endif
;
; OB
list=[3,4,5,8,10,21,24,25,26,27,34,35,37,38,50,51,52,76,77,80,82,83,84,85,89,90,91]
index=where(list eq item,count)
if count gt 0 and slice ne 'ob' then axis_index(1)=2
;
; Tr
list=[6,7,28,29,30,31,32,33,58,59]
index=where(list eq item,count)
if count gt 0 and slice ne 'tr' then axis_index(2)=3
;
; Ch
list=[3,4,5,6,7,8,21,24,25,26,27,28,29,30,31,32,33,34,35,50,51,52,80,82,90,91]
index=where(list eq item,count)
if count gt 0 and slice ne 'ch' then axis_index(4)=5
;
; Bl
list=[3,4,5,10,24,25,26,27,37,38,50,51,52,82,83,84,85,89,90,91]
index=where(list eq item,count)
if count gt 0 and slice ne 'bl' then axis_index(3)=4
;
; Pt
list=[3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,21,24,25,26,27,28,29,30,31,32, $
	33,34,35,36,37,38,39,40,41,50,51,52,53,54,55,56,57,58,59,60,61,62,63, $
	64,66,68,69,70,71,72,73,74,75,76,77,79,80,81,82,83,84,85,86,87,88,89, $
	90,91]
index=where(list eq item,count)
if count gt 0 and slice ne 'pt' then axis_index(5)=6
;
index=where(axis_index ne 0)
if index(0) ne -1 then axis_index=axis_index(index)-1 else axis_index=-1
;
; For publications, do not include the labels
if ps_options.p then axis_index=-1
;
end
;-------------------------------------------------------------------------------
pro set_plotloops,stream,num_baseline,num_channel,num_point
;
; Calling plot routines loop over channels, baselines, and points. However,
; depending on the selected data items, not all loop indices produce a different
; plot. This procedure resets loop ranges for unnecessary loops and the calling
; plot routine compares the ranges for the two axes and uses the larger one.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
;
set_streamsel,stream,item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,ds_pt
;
num_channel=n_elements(ds_ch)
num_baseline=n_elements(ds_bl)
num_point=n_elements(ds_pt)
case item of
	 1: begin
	    num_channel=1
  	    num_baseline=1
	    end
	 2: begin
	    num_channel=1
	    num_baseline=1
	    end
	 6: num_baseline=1
	 7: num_baseline=1
	 8: num_baseline=1
	 9: begin
	    num_channel=1
	    num_baseline=1
	    end
	10: begin
	    num_channel=1
	    end
        11: begin
	    num_channel=1
            num_baseline=1
            end
        12: begin
	    num_channel=1
            num_baseline=1
            end
	13: begin
	    num_channel=1
  	    num_baseline=1
	    end
	14: begin
	    num_channel=1
 	    num_baseline=1
	    end
	15: begin
	    num_channel=1
	    num_baseline=1
	    end
	16: begin
	    num_channel=1
	    num_baseline=1
	    end
	17: begin
	    num_channel=1
	    num_baseline=1
	    end
	18: begin
	    num_channel=1
	    num_baseline=1
	    end
	19: begin
	    num_channel=1
	    num_baseline=1
	    end
	20: begin
	    num_channel=1
	    num_baseline=1
	    end
	21: num_baseline=1
	22: begin
	    num_channel=1
	    num_baseline=1
	    end
	23: begin
	    num_channel=1
	    num_baseline=1
	    end
	28: num_baseline=1
	29: num_baseline=1
	30: num_baseline=1
	31: num_baseline=1
	32: num_baseline=1
	33: num_baseline=1
	34: num_baseline=1
	35: num_baseline=1
	36: begin
	    num_channel=1
	    num_baseline=1
	    end
	37: num_channel=1
	38: num_channel=1
	39: begin
	    num_channel=1
	    num_baseline=1
	    end
	40: begin
	    num_channel=1
	    num_baseline=1
	    end
	41: begin
	    num_channel=1
	    num_baseline=1
	    end
	42: begin
	    num_channel=1
	    num_baseline=1
	    end
	43: begin
	    num_channel=1
	    num_baseline=1
	    end
	44: begin
	    num_channel=1
	    num_baseline=1
	    end
	45: begin
	    num_baseline=1
	    num_point=1
	    end
	46: begin
	    num_baseline=1
	    num_point=1
	    end
	47: begin
	    num_channel=1
	    num_baseline=1
	    end
	48: begin
	    num_channel=1
	    num_baseline=1
	    end
	49: begin
	    num_channel=1
	    num_baseline=1
	    end
	53: begin
	    num_channel=1
	    num_baseline=1
	    end
	54: begin
	    num_channel=1
	    num_baseline=1
	    end
	55: begin
	    num_channel=1
	    num_baseline=1
	    end
	56: begin
	    num_channel=1
	    num_baseline=1
	    end
	60: begin
	    num_channel=1
	    num_baseline=1
	    end
	61: begin
	    num_channel=1
	    num_baseline=1
	    end
	62: begin
	    num_channel=1
	    num_baseline=1
	    end
	63: begin
	    num_channel=1
	    num_baseline=1
	    end
	64: begin
	    num_channel=1
	    num_baseline=1
	    end
	65: begin
	    num_channel=1
	    num_baseline=1
	    end
	66: begin
	    num_channel=1
	    num_baseline=1
	    end
	67: begin
	    num_channel=1
	    num_baseline=1
	    end
	68: begin
	    num_channel=1
	    num_baseline=1
	    end
	69: begin
	    num_channel=1
	    num_baseline=1
	    end
	70: begin
	    num_channel=1
	    num_baseline=1
	    end
	71: begin
	    num_channel=1
	    num_baseline=1
	    end
	72: begin
	    num_channel=1
	    num_baseline=1
	    end
	73: begin
	    num_channel=1
	    num_baseline=1
	    end
	74: begin
	    num_channel=1
	    num_baseline=1
	    end
	75: begin
	    num_channel=1
	    num_baseline=1
	    end
	76: begin
	    num_channel=1
	    num_baseline=1
	    end
	77: begin
	    num_channel=1
	    num_baseline=1
	    end
	78: begin
	    num_channel=1
	    num_baseline=1
	    end
	79: begin
	    num_channel=1
	    num_baseline=1
	    end
	80: begin
	    num_baseline=1
	    end
	81: begin
	    num_channel=1
	    num_baseline=1
	    end
	83: begin
	    num_channel=1
	    end
	84: begin
	    num_channel=1
	    end
	85: begin
	    num_channel=1
	    end
	86: begin
	    num_channel=1
	    num_baseline=1
	    end
	87: begin
	    num_channel=1
	    num_baseline=1
	    end
	88: begin
	    num_channel=1
	    num_baseline=1
	    end
	89: begin
	    num_channel=1
	    num_baseline=1
	    end
      else: 
endcase
;
; No loops are done for the sliced dimension
case slice of
	'ch':num_channel=1
	'bl':num_baseline=1
	'pt':num_point=1
	else:return
endcase
;
end
;-------------------------------------------------------------------------------
pro titleplacement,type,ds_stars,label,num_label
;
; This function works best if called with types 't', 'x', 'y', in this order!
;
common PlacementInfo,xpix_per_char,ypix_per_char, $
		     title_yoffset,title_xoffset, $
		     xtitle_xoffset,xtitle_yoffset, $
		     ytitle_xoffset,ytitle_yoffset, $
		     num_title,max_star_per_title,xpix_per_star, $
		     num_xtitle,max_xlabel_per_title,xpix_per_xlabel, $
		     num_ytitle,max_ylabel_per_title,xpix_per_ylabel
;
if !p.multi(1) gt 2 or !p.multi(2) gt 2 then mf=2.0 else mf=1.0
;
case type of
	't':begin
;	    Calculate title placement variables
	    num_star=n_elements(ds_stars)
	    xpix_per_unit_char=!d.x_ch_size
	    xpix_per_char=fix(xpix_per_unit_char*!p.charsize/mf)
	    ypix_per_unit_char=!d.y_ch_size
	    ypix_per_char=fix(ypix_per_unit_char*!p.charsize/mf)
	    xpix_per_star=(max(strlen(ds_stars))+1)*xpix_per_char
	    max_star_per_title=!d.x_vsize/xpix_per_star
	    num_title=fix(float(num_star)/max_star_per_title+1)
	    title_yoffset=!d.y_vsize-(indgen(num_title)+1)*(ypix_per_char)
	    num_star_per_title=intarr(num_title)+max_star_per_title
	    if num_title ge 2 then begin
		sub_total=total(num_star_per_title(0:num_title-2))
		num_star_per_title(num_title-1)=num_star-sub_total
	    endif else num_star_per_title(0)=num_star
	    title_xoffset=intarr(num_title)
	    for i=0,num_title-1 do title_xoffset(i)= $
		(!d.x_vsize-num_star_per_title(i)*xpix_per_star)/2
	    end
	'x':begin
;	    Calculate xtitle placement variables			
	    xpix_per_xlabel=(strlen(label)+1)*xpix_per_char
	    max_xlabel_per_title=!d.x_vsize/xpix_per_xlabel
	    num_xtitle=fix(float(num_label)/max_xlabel_per_title+1)
	    xtitle_yoffset= $
		(num_xtitle-indgen(num_xtitle))*(ypix_per_char)
	    num_xlabel_per_title= $
		intarr(num_xtitle)+max_xlabel_per_title
	    if num_xtitle ge 2 then begin
			sub_total= $
			   total(num_xlabel_per_title(0:num_xtitle-2))
			num_xlabel_per_title(num_xtitle-1)= $
			   num_label-sub_total
	    endif else num_xlabel_per_title(0)=num_label
	    xtitle_xoffset=intarr(num_xtitle)
	    for i=0,num_xtitle-1 do xtitle_xoffset(i)= $
		    (!d.x_vsize-num_xlabel_per_title(i)*xpix_per_xlabel)/2
	    end
	'y':begin
;	    Calculate ytitle placement variables			
	    xpix_per_ylabel=(strlen(label)+1)*xpix_per_char
	    max_ylabel_per_title= $
		(!d.y_vsize-ypix_per_char*(num_title+num_xtitle)) $
				 /ypix_per_char
	    num_ytitle=fix(float(num_label)/max_ylabel_per_title+1)
	    ytitle_xoffset=indgen(num_ytitle)*xpix_per_ylabel
	    num_ylabel_per_title= $
			intarr(num_ytitle)+max_ylabel_per_title
	    if num_ytitle ge 2 then begin
			sub_total= $
			   total(num_ylabel_per_title(0:num_ytitle-2))
			num_ylabel_per_title(num_ytitle-1)= $
			   num_label-sub_total
	    endif else num_ylabel_per_title(0)=num_label
	    ytitle_yoffset=intarr(num_ytitle)
	    for i=0,num_ytitle-1 do ytitle_yoffset(i)= $
		(!d.y_vsize-(num_ylabel_per_title(i)+num_title+num_xtitle) $
				*ypix_per_char)/2+num_xtitle*ypix_per_char
	    end
endcase
;
end
;-------------------------------------------------------------------------------
pro plotsymbol,psym
;
; IDL lacks, despite the high price for this software, a decent list of plot 
; symbols. This procedure is an interface to the idlastro routines vsym and 
; plotsym to remedy this embarrasing situation.
;
;
;    open/filled
;	0/10: +
;	1/11: x
;	2/12: *
;	3/13: Triangle up
;	4/14: Triangle down
;	5/15: Square
;	6/16: Diamond
;	7/17: Pentagon
;	8/18: Star
;	9/19: Circle
;
fill=psym/10 gt 0
;
case (psym mod 10) of
	0:vsym,4,/skel,rot=45,fill=fill
	1:vsym,4,/skel,fill=fill
	2:vsym,5,/skel,fill=fill
	3:plotsym,4,fill=fill
	4:plotsym,5,fill=fill
	5:plotsym,8,fill=fill
	6:vsym,4,rot=45,fill=fill
	7:vsym,5,fill=fill
	8:plotsym,3,fill=fill
	9:plotsym,0,fill=fill
endcase
;
end
; 
;-------------------------------------------------------------------------------
pro wdall
;
repeat begin
	wset,-1
	if !d.window ne -1 then wdelete
endrep until !d.window eq -1
;
end
;-------------------------------------------------------------------------------
pro set_screen,all=all
;
; Quick way to reset plot parameters to sane values.
; Does not reset ranges and symbols, as this would eliminate a very
; useful control for the user over compound plot procedures!
;
if n_elements(all) eq 0 then all=0
;
set_plot,!display
trucolor
!p.color=tci(1)
!p.font=-1
!p.charsize=1
!p.charthick=1
!x.thick=2
!y.thick=2
!x.ticks=0
!y.ticks=0
!x.ticklen=0
!y.ticklen=0
!x.tickname=''
!y.tickname=''
!x.style=0
!y.style=0
!x.type=0
!y.type=0
!x.margin=[10,3]
!y.margin=[4,2]
!p.title=''
!p.subtitle=''
!x.title=''
!y.title=''
!p.multi=0
!p.thick=1
;
if all then begin
	!p.psym=0
	!x.range=0
	!y.range=0
endif
;
end
;-------------------------------------------------------------------------------
pro set_ps,filename,color=color
;
; Starting with IDL 7.1, 24-bit images can be sent to a PostScript device,
; even though that has an indexed color model as a default. All you need
; is to make sure that a greyscale color table is loaded before calling
; TV, e.g., set_plot,'ps' & device, decomposed=0 & loadct,0 & tv, rose, /true
;
itc=2.54	; inch to cm
;
if n_elements(filename) eq 0 then filename='idl.ps'
if n_elements(color) eq 0 then color=0
;
set_plot,'ps'
device,filename=filename,color=color
;
!p.font=1
!p.charsize=1.5
device,/times
!x.thick=2
!y.thick=2
!p.thick=1
;
width=7*itc
height=5*itc
length=width
x=length/640
y=length/691
xsize=x*!xsize
ysize=y*!ysize
device,xsize=xsize,ysize=ysize, $
	xoffset=0.75*itc,yoffset=0.75*itc
if xsize gt width then device,/landscape
;
if color then tek_color
;
end
;-------------------------------------------------------------------------------
pro testcolor
;
; This routine sends a color plot to a color printer to check ink cartridges,
; etc. 
;
if strpos(strupcase(getenv('HOST')),'GEMINI') ne -1 then begin
	set_plot,'ps'
	device,color=1
;
	x=findgen(18)
	tek_color
	plot,x,color=0
	for i=0,15 do begin
		x=[i,i,i+1,i+1,i]
		y=[0,7,7,0,0]
		polyfill,x,y,color=i
		s=strcompress(string(i))
		xyouts,i,8,s,color=1
	endfor
	for i=16,31 do begin
		x=[i-16,i-16,i-15,i-15,i-16]
		y=[9,15,15,9,9]
		polyfill,x,y,color=i
		s=strcompress(string(i))
		xyouts,i-16,16,s,color=1
	endfor
;
	device,/close_file
;
	spawn,'lpr wave.ps'
	set_plot,!display
endif else print,'***Error(TESTCOLOR): routine for gemini printer only!'
;
end
;************************************************************************Block 3
pro set_pointflagtable,reason,item,inbeam,outbeam,triple,channel,baseline,time
;
; Stores new entries in pointflagtable, expanding the table if necessary.
;
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
min_incr=2000
num_flag=n_elements(item)
if n_elements(pointflagtable) eq 0 then create_pointflagtable,num_flag+min_incr
num_tble=n_elements(pointflagtable)
;
index=where(pointflagtable.time gt 0,count)
if count gt 0 then id0=index(count-1)+1 else id0=0
;
if num_flag gt num_tble-id0 then begin
	table=pointflagtable
	create_pointflagtable,num_tble+num_flag+min_incr
	pointflagtable(0:num_tble-1).reason=table.reason
	pointflagtable(0:num_tble-1).item=table.item
	pointflagtable(0:num_tble-1).inbeam=table.inbeam
	pointflagtable(0:num_tble-1).outbeam=table.outbeam
	pointflagtable(0:num_tble-1).triple=table.triple
	pointflagtable(0:num_tble-1).channel=table.channel
	pointflagtable(0:num_tble-1).baseline=table.baseline
	pointflagtable(0:num_tble-1).time=table.time
endif
;
if num_flag eq 1 then begin
	pointflagtable(id0:id0+num_flag-1).reason=reason(0)
	pointflagtable(id0:id0+num_flag-1).item=item(0)
	pointflagtable(id0:id0+num_flag-1).inbeam=inbeam(0)
	pointflagtable(id0:id0+num_flag-1).outbeam=outbeam(0)
	pointflagtable(id0:id0+num_flag-1).triple=triple(0)
	pointflagtable(id0:id0+num_flag-1).channel=channel(0)
	pointflagtable(id0:id0+num_flag-1).baseline=baseline(0)
	pointflagtable(id0:id0+num_flag-1).time=time(0)
endif else begin
	pointflagtable(id0:id0+num_flag-1).reason=reason
	pointflagtable(id0:id0+num_flag-1).item=item
	pointflagtable(id0:id0+num_flag-1).inbeam=inbeam
	pointflagtable(id0:id0+num_flag-1).outbeam=outbeam
	pointflagtable(id0:id0+num_flag-1).triple=triple
	pointflagtable(id0:id0+num_flag-1).channel=channel
	pointflagtable(id0:id0+num_flag-1).baseline=baseline
	pointflagtable(id0:id0+num_flag-1).time=time
endelse
;
end	
;-------------------------------------------------------------------------------
pro create_pointflagtable,MP
;
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([1]) ne 0 then return
;
if n_elements(MP) eq 0 then MP=long(total(scantable.NumPoint))
;
reason=strarr(MP)
item=bytarr(MP)
baseline=bytarr(MP)
triple=bytarr(MP)
channel=intarr(MP)	; changed to int for high-resolution spectrometers
inbeam=bytarr(MP)
outbeam=bytarr(MP)
time=dblarr(MP)-1	; '-1': to find out current number of edited points
pointflagtable= $
 build_pointflagtable(reason,item,baseline,triple,channel,inbeam,outbeam,time)
;
end
;-------------------------------------------------------------------------------
pro set_inchflagtable,reason,item,inbeam,time
;
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
min_incr=2000
num_flag=n_elements(item)
if n_elements(inchflagtable) eq 0 then create_inchflagtable,num_flag+min_incr
num_tble=n_elements(inchflagtable)
;
index=where(inchflagtable.time gt 0,count)
if count gt 0 then id0=index(count-1)+1 else id0=0
;
if num_flag gt num_tble-id0 then begin
	table=inchflagtable
	create_inchflagtable,num_tble+num_flag+min_incr
	inchflagtable(0:num_tble-1).reason=table.reason
	inchflagtable(0:num_tble-1).item=table.item
	inchflagtable(0:num_tble-1).inbeam=table.inbeam
	inchflagtable(0:num_tble-1).time=table.time
endif
;
inchflagtable(id0:id0+num_flag-1).reason=reason
inchflagtable(id0:id0+num_flag-1).item=item
inchflagtable(id0:id0+num_flag-1).inbeam=inbeam
inchflagtable(id0:id0+num_flag-1).time=time
;
end	
;-------------------------------------------------------------------------------
pro create_inchflagtable,MP
;
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(MP) eq 0 then MP=500
;
reason=strarr(MP)
item=bytarr(MP)
inbeam=bytarr(MP)
time=dblarr(MP)-1	; '-1': to find out current number of edited points
inchflagtable= $
 build_inchflagtable(reason,item,inbeam,time)
;
end
;-------------------------------------------------------------------------------
pro create_scanflagtable
;
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([1]) ne 0 then return
;
MP=n_elements(scantable)*GenConfig.NumOutBeam $
  *max(GenConfig.NumSpecChan)*max(GenConfig.NumBaseline)
;
reason=sindgen(MP)
item=bytarr(MP)
baseline=bytarr(MP)
triple=bytarr(MP)
channel=intarr(MP)	; changed to int for high-resolution spectrometers
inbeam=bytarr(MP)
outbeam=bytarr(MP)
time=dblarr(MP)-1	; '-1': to find out current number of edited points
scanflagtable= $
 build_scanflagtable(reason,item,baseline,triple,channel,inbeam,outbeam,time)
;
end
;-------------------------------------------------------------------------------
pro create_bgflagtable
;
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([2]) ne 0 then return
;
n=n_elements(bgtable)*GenConfig.NumOutBeam*max(GenConfig.NumSpecChan)
;
reason=strarr(n)
item=bytarr(n)
channel=intarr(n)	; changed to int for high-resolution spectrometers
outbeam=bytarr(n)
time=dblarr(n)-1	; '-1': to find out current number of edited points
bgflagtable= $
 build_bgflagtable(reason,item,channel,outbeam,time)
;
end
;-------------------------------------------------------------------------------
pro flagpointdata,flagtable,flag=flag,ib=ib,ob=ob,tr=tr,ch=ch,bl=bl,pt=pt, $
	item=item,reason=reason,notable=notable
;
; Applies (flag=1) or removes (flag=-1) flags of flag table in point data.
; Calls a C-program for faster execution.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
if checkdata([8,11]) ne 0 then return
;
if not keyword_set(flag) then flag=1L else flag=long(flag)
if(flag ne 1) and (flag ne -1) then begin
	print,'***Error(FLAGPOINTDATA): flag must be 1 (flag) or -1 (unflag)!'
	return
endif
;
numflag=n_elements(flagtable)
IF numflag EQ 0 THEN BEGIN
;
; Check flag table
if n_elements(pointflagtable) eq 0 then create_pointflagtable
index=where(pointflagtable.time ge 0,count)
id0=count
;
; Check pt
if not keyword_set(pt) then pt0=indgen(n_elements(PointTime)) $
else begin
index=intarr(n_elements(PointTime))
jndex=intarr(n_elements(PointTime))
for i=0,n_elements(pt)-1 do begin
	if pt(i) gt 0 then index(pt(i)-1)=1 else begin
		jndex(abs(pt(i))-1)=1
		index(where(jndex eq 0))=1
		index(abs(pt(i))-1)=0
	endelse
endfor
pt0=where(index eq 1)
endelse
;
; Check ib
if not keyword_set(ib) then ib0=indgen(GenConfig.NumSid) $
		       else ib0=ib-1
; Check ob
if not keyword_set(ob) then ob0=indgen(GenConfig.NumOutBeam) $
		       else ob0=ob-1
;
; Check tr
if not keyword_set(tr) then tr0=indgen(GenConfig.NumTriple) $
		       else tr0=tr-1
; Check ch
if not keyword_set(ch) then ch=indgen(max(GenConfig.NumSpecChan))+1
ch0=intarr(n_elements(ob0),max(GenConfig.NumSpecChan))-1
for i=0,n_elements(ob0)-1 do begin
        s=size(ch)
        if s(0) le 1 then ch0(i,0:total(ch gt 0)-1) $
                         =ch(where(ch gt 0))-1 $
                     else ch0(i,0:total(ch(ob0(i),*) gt 0)-1) $
                         =ch(ob0(i),where(ch(ob0(i),*) gt 0))-1
endfor
;
; Check bl
if not keyword_set(bl) then bl=indgen(max(GenConfig.NumBaseline))+1
bl0=intarr(n_elements(ob0),max(GenConfig.NumBaseline))-1
for i=0,n_elements(ob0)-1 do begin
        s=size(bl)
        if s(0) le 1 then bl0(i,0:total(bl gt 0)-1) $
                         =bl(where(bl gt 0))-1 $
                     else bl0(i,0:total(bl(ob0(i),*) gt 0)-1) $
                         =bl(ob0(i),where(bl(ob0(i),*) gt 0))-1
endfor
;
; Check reason
if not keyword_set(reason) then reason=strmid(systime(),0,24)+' MANUAL'
;
; Check notable
if not keyword_set(notable) then notable=0
;
!qiet=1
fcount=0
for i=0,n_elements(ib0)-1 do begin
for j=0,n_elements(ob0)-1 do begin
for k=0,n_elements(tr0)-1 do begin
for l=0,n_elements(ch0(j,*))-1 do begin
for m=0,n_elements(bl0(j,*))-1 do begin
	case item of
	5:	begin
		if ch0(j,l) ge 0 and bl0(j,m) ge 0 then begin
		index=where(VisPhaseErr(ob0(j),ch0(j,l),bl0(j,m),pt0) gt 0,count)
		if count gt 0 then begin
			id=id0+count-1
		   	set_pointflagtable,reason,intarr(count)+item, $
				intarr(count),ob0(j),intarr(count), $
				ch0(j,l),bl0(j,m), $
				abs(PointTime(pt0(index)))
		endif
		endif
		end
	endcase
	if count gt 0 then begin
		flagpointdata,pointflagtable(id0:id),flag=flag
		if notable then pointflagtable(id0:id).time=-1 else id0=id
	endif
	fcount=fcount+count
endfor
endfor
endfor
endfor
endfor
;
print,'Editing complete:',fcount,' primary data points.'
!qiet=0
;
return
ENDIF
;
; Subscriptions to be flagged
item=flagtable.item
inbeam=flagtable.inbeam
outbeam=flagtable.outbeam 
triple=flagtable.triple
baseline=flagtable.baseline
channel=flagtable.channel
time=flagtable.time
;
; Auxilliary information
stationid=GenConfig.StationId
baselineid=GenConfig.BaselineId
tripleOB=GenConfig.TripleBeam
tripleBl=GenConfig.TripleBase
tripleCh=GenConfig.TripleChan
tripleNumCh=GenConfig.TripleNumChan
;
; Loop information
numinbeam=GenConfig.NumSid
numoutbeam=GenConfig.NumOutBeam
numtriple=GenConfig.NumTriple
;
; Dimensioning information
maxtriple=n_elements(GenConfig.TripleChan)
maxtchannel=max(GenConfig.TripleNumChan)
maxbaseline=max(GenConfig.NumBaseline)
maxchannel=max(GenConfig.NumSpecChan)
;
numbaseline=GenConfig.NumBaseline
numchannel=GenConfig.NumSpecChan
numpoint=n_elements(PointTime)
;
flagged=0L
flagged=linknload(!external_lib,'flagpointdata',numflag,flag, $
	item,inbeam,outbeam,triple,baseline,channel,time, $
	PointTime,FDLPosErr,DelayJitterErr,NATJitterErr,NATCountsErr, $
	PhotonRateErr,VisSqErr, $
	VisAmpErr,VisPhaseErr,TripleAmpErr,TriplePhaseErr, $
	GrpDelayErr,DryDelayErr,WetDelayErr, $
	stationid,baselineid,tripleOB,tripleBl,tripleCh,tripleNumCh, $
	numinbeam,numoutbeam,numtriple,numbaseline,numchannel,numpoint, $
	maxtriple,maxtchannel,maxbaseline,maxchannel)
;
if flagged eq -1 then print,'***Error(FLAGPOINTDATA): invalid item!' $
else if not !qiet then print,'Editing complete:',flagged,' primary data points.'
;
end
;-------------------------------------------------------------------------------
pro flaginchdata,flagtable,flag=flag
;
; Applies (flag=1) or removes (flag=-1) flags of flag table in metro data.
; Calls a C-program for faster execution.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
;
if not keyword_set(flag) then flag=1L else flag=long(flag)
if(flag ne 1) and (flag ne -1) then begin
	print,'***Error(FLAGMETRODATA): flag must be 1 (flag) or -1 (unflag)!'
	return
endif
;
numflag=n_elements(flagtable)
if numflag eq 0 then begin
	print,'***Error(FLAGMETRODATA): zero number of flags!'
	return
endif
if checkdata([8,16]) ne 0 then return
;
item=flagtable.item
inbeam=flagtable.inbeam
time=flagtable.time
;
numinbeam=GenConfig.NumSid
;
numpoint=n_elements(MetroTime)
;
flagged=0L
flagged=linknload(!external_lib,'flaginchdata',numflag,flag, $
	item,inbeam,time, $
	MetroTime,ParXErr,ParYErr,ParZErr,MetroPathErr,MotorAngleErr, $
	numinbeam,numpoint)
;
if flagged eq -1 then print,'***Error(FLAGMETRODATA): invalid item!' $
else begin
	print,'Editing complete:',flagged,' primary data points.'
	for i=0,GenConfig.NumSid-1 do begin
		index=where(ParXErr(i,*) gt 0,count)
		if count gt 0 then begin
			ParX(i,*)=ParX(i,*)-ParX(i,index(0))
			ParY(i,*)=ParY(i,*)-ParY(i,index(0))
			ParZ(i,*)=ParZ(i,*)-ParZ(i,index(0))
		endif
	endfor
	print,'Re-referenced metro data.'
endelse
;
end
;-------------------------------------------------------------------------------
pro flagbgdata,flagtable,flag=flag
;
; Applies (flag=1) or removes (flag=-1) flags of flag table in BG data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if not keyword_set(flag) then flag=1L else flag=long(flag)
if(flag ne 1) and (flag ne -1) then begin
	print,'***Error(FLAGBGDATA): flag must be 1 (flag) or -1 (unflag)!'
	return
endif
;
numflag=n_elements(flagtable)
if numflag eq 0 then begin
	print,'***Error(FLAGBGDATA): zero number of flags!'
	return
endif
if checkdata([8,2,10]) ne 0 then return
;
index=where(bgscans.RateErr lt 0,count0)
;
for i=0,numflag-1 do begin
	j=where(abs(bgscans.Time) eq flagtable(i).time,count)
	if count gt 0 then begin
		ob=flagtable(i).outbeam
		ch=flagtable(i).channel
		case flagtable(i).item of
;
		19:begin
		   bgscans(j).Time=-flag*abs(bgscans(j).Time)
		   for ob=0,GenConfig.NumOutBeam-1 do begin
     		   for ch=0,GenConfig.NumSpecChan(ob)-1 do begin
		   bgscans(j).RateErr(ob,ch)= $
		  	-flag*abs(bgscans(j).RateErr(ob,ch))
		   endfor
		   endfor
		   end
		20:begin
		   bgscans(j).Time=-flag*abs(bgscans(j).Time)
		   for ob=0,GenConfig.NumOutBeam-1 do begin
     		   for ch=0,GenConfig.NumSpecChan(ob)-1 do begin
		   bgscans(j).RateErr(ob,ch)= $
		  	-flag*abs(bgscans(j).RateErr(ob,ch))
		   endfor
		   endfor
		   end
		21:bgscans(j).RateErr(ob,ch)= $
			-flag*abs(bgscans(j).RateErr(ob,ch))
		else:begin
   		   print,'***Error(FLAGBGDATA): item not allowed: ', $
			flagtable(i).item,'!'
     		   end
;
		endcase
	endif
endfor
;
index=where(bgscans.RateErr lt 0,count1)
;
print,'Editing complete:',count1-count0,' data points.'
end
;-------------------------------------------------------------------------------
pro flagscandata,flagtable,flag=flag,ib=ib,ob=ob,tr=tr,ch=ch,bl=bl,sc=sc, $
	item=item,reason=reason
;
; Applies (flag=1) or removes (flag=-1) flags of flag table in scan data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
if checkdata([8,9]) ne 0 then return
;
if not keyword_set(flag) then flag=1
if(flag ne 1L) and (flag ne -1L) then begin
	print,'***Error(FLAGSCANDATA): flag must be 1 (flag) or -1 (unflag)!'
	return
endif
;
numflag=n_elements(flagtable)
IF numflag EQ 0 THEN BEGIN
;
; Check flag table
if n_elements(scanflagtable) eq 0 then create_scanflagtable
index=where(scanflagtable.time ge 0,count)
id0=count
;
; Check sc
if not keyword_set(sc) then sc=indgen(n_elements(scans))+1
index=intarr(n_elements(scans))
jndex=intarr(n_elements(scans))
for i=0,n_elements(sc)-1 do begin
	if sc(i) gt 0 then index(sc(i)-1)=1 else begin
		jndex(abs(sc(i))-1)=1
		index(where(jndex eq 0))=1
		index(abs(sc(i))-1)=0
	endelse
endfor
sc0=where(index eq 1)
;
; Check ib
if not keyword_set(ib) then ib0=indgen(GenConfig.NumSid) $
		       else ib0=ib-1
; Check ob
if not keyword_set(ob) then ob0=indgen(GenConfig.NumOutBeam) $
		       else ob0=ob-1
;
; Check tr
if not keyword_set(tr) then tr0=indgen(GenConfig.NumTriple > 1) $
		       else tr0=tr-1
; Check ch
if not keyword_set(ch) then ch=indgen(max(GenConfig.NumSpecChan))+1
ch0=intarr(n_elements(ob0),max(GenConfig.NumSpecChan))-1
for i=0,n_elements(ob0)-1 do begin
        s=size(ch)
        if s(0) le 1 then ch0(i,0:total(ch gt 0)-1) $
                         =ch(where(ch gt 0))-1 $
                     else ch0(i,0:total(ch(ob0(i),*) gt 0)-1) $
                         =ch(ob0(i),where(ch(ob0(i),*) gt 0))-1
endfor
;
; Check bl
if not keyword_set(bl) then bl=indgen(max(GenConfig.NumBaseline))+1
bl0=intarr(n_elements(ob0),max(GenConfig.NumBaseline))-1
for i=0,n_elements(ob0)-1 do begin
        s=size(bl)
        if s(0) le 1 then bl0(i,0:total(bl gt 0)-1) $
                         =bl(where(bl gt 0))-1 $
                     else bl0(i,0:total(bl(ob0(i),*) gt 0)-1) $
                         =bl(ob0(i),where(bl(ob0(i),*) gt 0))-1
endfor
;
; Check reason
if not keyword_set(reason) then reason=strmid(systime(),0,24)+' MANUAL'
;
!qiet=1
fcount=0
for i=0,n_elements(ib0)-1 do begin
for j=0,n_elements(ob0)-1 do begin
for k=0,n_elements(tr0)-1 do begin
for l=0,n_elements(ch0(j,*))-1 do begin
for m=0,n_elements(bl0(j,*))-1 do begin
	case item of
	24:	begin
		if ch0(j,l) ge 0 and bl0(j,m) ge 0 then begin
		index=where(scans(sc0).vissqerr(ob0(j),ch0(j,l),bl0(j,m)) gt 0,count)
		if count gt 0 then begin
			id=id0+count-1
			scanflagtable(id0:id).reason=reason
			scanflagtable(id0:id).item=24
			scanflagtable(id0:id).channel=ch0(j,l)
			scanflagtable(id0:id).baseline=bl0(j,m)
			scanflagtable(id0:id).outbeam=ob0(j)
			scanflagtable(id0:id).time=abs(scans(sc0(index)).time)
		endif
		endif else count=0
		end
	endcase
	if count gt 0 then begin
		flagscandata,scanflagtable(id0:id),flag=flag
		fcount=fcount+count
		id0=id+1
	endif
endfor
endfor
endfor
endfor
endfor
;
print,'Editing complete:',fcount,' primary data points.'
!qiet=0
;
return
ENDIF
;
fcount=0
;
FOR i=0,numflag-1 do BEGIN
;
j=where(abs(scans(*).Time) eq flagtable(i).time,count)
if count gt 0 then begin
  fcount=fcount+1
  ib=flagtable(i).inbeam
  ob=flagtable(i).outbeam
  tr=flagtable(i).triple
  ch=flagtable(i).channel
  bl=flagtable(i).baseline
  case flagtable(i).item of
;
  22:begin
     scans(j).Time=-flag*abs(scans(j).Time)
     scans(j).VisSqErr=-flag*abs(scans(j).VisSqErr)
     scans(j).VisSqCErr=-flag*abs(scans(j).VisSqCErr)
     scans(j).VisSqEErr=-flag*abs(scans(j).VisSqEErr)
     scans(j).VisSqECErr=-flag*abs(scans(j).VisSqECErr)
     scans(j).PhotonRateErr=-flag*abs(scans(j).PhotonRateErr)
     scans(j).BackgndErr=-flag*abs(scans(j).BackgndErr)
     scans(j).TripleAmpErr=-flag*abs(scans(j).TripleAmpErr)
     scans(j).TripleAmpCErr=-flag*abs(scans(j).TripleAmpCErr)
     scans(j).TripleAmpEErr=-flag*abs(scans(j).TripleAmpEErr)
     scans(j).TripleAmpECErr=-flag*abs(scans(j).TripleAmpECErr)
     scans(j).TriplePhaseErr=-flag*abs(scans(j).TriplePhaseErr)
     scans(j).TriplePhaseCErr=-flag*abs(scans(j).TriplePhaseCErr)
     scans(j).FDLPosErr=-flag*abs(scans(j).FDLPosErr)
     scans(j).NATJitterErr=-flag*abs(scans(j).NATJitterErr)
     scans(j).GrpDelayErr=-flag*abs(scans(j).GrpDelayErr)
     scans(j).DryDelayErr=-flag*abs(scans(j).DryDelayErr)
     scans(j).WetDelayErr=-flag*abs(scans(j).WetDelayErr)
     scans(j).DelayJitterErr=-flag*abs(scans(j).DelayJitterErr)
     end
  23:begin
     scans(j).Time=-flag*abs(scans(j).Time)
     for ob=0,GenConfig.NumOutBeam-1 do begin
     for ch=0,GenConfig.NumSpecChan(ob)-1 do begin
       for bl=0,GenConfig.NumBaseline(ob)-1 do begin
          scans(j).VisSqErr(ob,ch,bl)=-flag*abs(scans(j).VisSqErr(ob,ch,bl))
	  scans(j).VisSqCErr(ob,ch,bl)=-flag*abs(scans(j).VisSqCErr(ob,ch,bl))
	  scans(j).VisSqEErr(ob,ch,bl)=-flag*abs(scans(j).VisSqEErr(ob,ch,bl))
	  scans(j).VisSqECErr(ob,ch,bl)=-flag*abs(scans(j).VisSqECErr(ob,ch,bl))
	endfor
       scans(j).PhotonRateErr(ob,ch)=-flag*abs(scans(j).PhotonRateErr(ob,ch))
       scans(j).BackgndErr(ob,ch)=-flag*abs(scans(j).BackgndErr(ob,ch))
     endfor
     endfor
     for tr=0,GenConfig.NumTriple-1 do begin
     for ch=0,GenConfig.TripleNumChan(tr)-1 do begin
	scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
	scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
	scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
	scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
	scans(j).TriplePhaseErr(tr)=-flag*abs(scans(j).TriplePhaseErr(tr))
	scans(j).TriplePhaseCErr(tr)=-flag*abs(scans(j).TriplePhaseCErr(tr))
     endfor
     endfor
     for ib=0,GenConfig.NumSid-1 do begin
       scans(j).FDLPosErr(ib)=-flag*abs(scans(j).FDLPosErr(ib))
       scans(j).NATJitterErr(ib)=-flag*abs(scans(j).NATJitterErr(ib))
       scans(j).GrpDelayErr(ib)=-flag*abs(scans(j).GrpDelayErr(ib))
       scans(j).DryDelayErr(ib)=-flag*abs(scans(j).DryDelayErr(ib))
       scans(j).WetDelayErr(ib)=-flag*abs(scans(j).WetDelayErr(ib))
     endfor
     for ob=0,GenConfig.NumOutBeam-1 do begin
     for bl=0,GenConfig.NumBaseline(ob)-1 do begin
        scans(j).DelayJitterErr(ob,bl)=-flag*abs(scans(j).DelayJitterErr(ob,bl))
     endfor
     endfor
     end
  24:begin
     scans(j).VisSqErr(ob,ch,bl)=-flag*abs(scans(j).VisSqErr(ob,ch,bl))
     scans(j).VisSqCErr(ob,ch,bl)=-flag*abs(scans(j).VisSqCErr(ob,ch,bl))
     scans(j).VisSqEErr(ob,ch,bl)=-flag*abs(scans(j).VisSqEErr(ob,ch,bl))
     scans(j).VisSqECErr(ob,ch,bl)=-flag*abs(scans(j).VisSqECErr(ob,ch,bl))
     for tr=0,GenConfig.NumTriple-1 do begin
     for l=0,2 do begin
       if (GenConfig.TripleBase(l,tr) eq bl) and $
	  (GenConfig.TripleBeam(l,tr) eq ob) then begin
	 k=where(GenConfig.TripleChan(0:GenConfig.TripleNumChan(tr)-1,l,tr) $
	     eq ch,count)
	 not_l=where(indgen(3) ne l)
	 not_ob=GenConfig.TripleBeam(not_l,tr)
	 not_bl=GenConfig.TripleBase(not_l,tr)
         kk=[k,k]
	 if count gt 0 then begin
	  if flag eq 1 $
          or (flag eq -1 $
          and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
           scans(j).TripleAmpErr(tr,k)=-flag*abs(scans(j).TripleAmpErr(tr,k))
           scans(j).TripleAmpCErr(tr,k)=-flag*abs(scans(j).TripleAmpCErr(tr,k))
           scans(j).TripleAmpEErr(tr,k)=-flag*abs(scans(j).TripleAmpEErr(tr,k))
           scans(j).TripleAmpECErr(tr,k)=-flag*abs(scans(j).TripleAmpECErr(tr,k))
           scans(j).TriplePhaseErr(tr,k)=-flag*abs(scans(j).TriplePhaseErr(tr,k))
           scans(j).TriplePhaseCErr(tr,k)=-flag*abs(scans(j).TriplePhaseCErr(tr,k))
	  endif
         endif
       endif
     endfor
     endfor
     end
  25:begin
     scans(j).VisSqErr(ob,ch,bl)=-flag*abs(scans(j).VisSqErr(ob,ch,bl))
     scans(j).VisSqCErr(ob,ch,bl)=-flag*abs(scans(j).VisSqCErr(ob,ch,bl))
     scans(j).VisSqEErr(ob,ch,bl)=-flag*abs(scans(j).VisSqEErr(ob,ch,bl))
     scans(j).VisSqECErr(ob,ch,bl)=-flag*abs(scans(j).VisSqECErr(ob,ch,bl))
     for tr=0,GenConfig.NumTriple-1 do begin
     for l=0,2 do begin
       if (GenConfig.TripleBase(l,tr) eq bl) and $
	  (GenConfig.TripleBeam(l,tr) eq ob) then begin
	 k=where(GenConfig.TripleChan(0:GenConfig.TripleNumChan(tr)-1,l,tr) eq ch,count)
	 not_l=where(indgen(3) ne l)
	 not_ob=GenConfig.TripleBeam(not_l,tr)
	 not_bl=GenConfig.TripleBase(not_l,tr)
         kk=[k,k]
	 if count gt 0 then begin
	  if flag eq 1 $
          or (flag eq -1 $
          and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
           scans(j).TripleAmpErr(tr,k)=-flag*abs(scans(j).TripleAmpErr(tr,k))
           scans(j).TripleAmpCErr(tr,k)=-flag*abs(scans(j).TripleAmpCErr(tr,k))
           scans(j).TripleAmpEErr(tr,k)=-flag*abs(scans(j).TripleAmpEErr(tr,k))
           scans(j).TripleAmpECErr(tr,k)=-flag*abs(scans(j).TripleAmpECErr(tr,k))
           scans(j).TriplePhaseErr(tr,k)=-flag*abs(scans(j).TriplePhaseErr(tr,k))
           scans(j).TriplePhaseCErr(tr,k)=-flag*abs(scans(j).TriplePhaseCErr(tr,k))
	  endif
         endif
       endif
     endfor
     endfor
     end
  26:begin
     scans(j).VisSqErr(ob,ch,bl)=-flag*abs(scans(j).VisSqErr(ob,ch,bl))
     scans(j).VisSqCErr(ob,ch,bl)=-flag*abs(scans(j).VisSqCErr(ob,ch,bl))
     scans(j).VisSqEErr(ob,ch,bl)=-flag*abs(scans(j).VisSqEErr(ob,ch,bl))
     scans(j).VisSqECErr(ob,ch,bl)=-flag*abs(scans(j).VisSqECErr(ob,ch,bl))
     for tr=0,GenConfig.NumTriple-1 do begin
     for l=0,2 do begin
       if (GenConfig.TripleBase(l,tr) eq bl) and $
	  (GenConfig.TripleBeam(l,tr) eq ob) then begin
	 k=where(GenConfig.TripleChan(0:GenConfig.TripleNumChan(tr)-1,l,tr) eq ch,count)
	 not_l=where(indgen(3) ne l)
	 not_ob=GenConfig.TripleBeam(not_l,tr)
	 not_bl=GenConfig.TripleBase(not_l,tr)
         kk=[k,k]
	 if count gt 0 then begin
	  if flag eq 1 $
          or (flag eq -1 $
          and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
           scans(j).TripleAmpErr(tr,k)=-flag*abs(scans(j).TripleAmpErr(tr,k))
           scans(j).TripleAmpCErr(tr,k)=-flag*abs(scans(j).TripleAmpCErr(tr,k))
           scans(j).TripleAmpEErr(tr,k)=-flag*abs(scans(j).TripleAmpEErr(tr,k))
           scans(j).TripleAmpECErr(tr,k)=-flag*abs(scans(j).TripleAmpECErr(tr,k))
           scans(j).TriplePhaseErr(tr,k)=-flag*abs(scans(j).TriplePhaseErr(tr,k))
           scans(j).TriplePhaseCErr(tr,k)=-flag*abs(scans(j).TriplePhaseCErr(tr,k))
	  endif
         endif
       endif
     endfor
     endfor
     end
  27:begin
     scans(j).VisSqErr(ob,ch,bl)=-flag*abs(scans(j).VisSqErr(ob,ch,bl))
     scans(j).VisSqCErr(ob,ch,bl)=-flag*abs(scans(j).VisSqCErr(ob,ch,bl))
     scans(j).VisSqEErr(ob,ch,bl)=-flag*abs(scans(j).VisSqEErr(ob,ch,bl))
     scans(j).VisSqECErr(ob,ch,bl)=-flag*abs(scans(j).VisSqECErr(ob,ch,bl))
     for tr=0,GenConfig.NumTriple-1 do begin
     for l=0,2 do begin
       if (GenConfig.TripleBase(l,tr) eq bl) and $
	  (GenConfig.TripleBeam(l,tr) eq ob) then begin
	 k=where(GenConfig.TripleChan(0:GenConfig.TripleNumChan(tr)-1,l,tr) eq ch,count)
	 not_l=where(indgen(3) ne l)
	 not_ob=GenConfig.TripleBeam(not_l,tr)
	 not_bl=GenConfig.TripleBase(not_l,tr)
         kk=[k,k]
	 if count gt 0 then begin
	  if flag eq 1 $
          or (flag eq -1 $
          and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
           scans(j).TripleAmpErr(tr,k)=-flag*abs(scans(j).TripleAmpErr(tr,k))
           scans(j).TripleAmpCErr(tr,k)=-flag*abs(scans(j).TripleAmpCErr(tr,k))
           scans(j).TripleAmpEErr(tr,k)=-flag*abs(scans(j).TripleAmpEErr(tr,k))
           scans(j).TripleAmpECErr(tr,k)=-flag*abs(scans(j).TripleAmpECErr(tr,k))
           scans(j).TriplePhaseErr(tr,k)=-flag*abs(scans(j).TriplePhaseErr(tr,k))
           scans(j).TriplePhaseCErr(tr,k)=-flag*abs(scans(j).TriplePhaseCErr(tr,k))
	  endif
         endif
       endif
     endfor
     endfor
     end
  28:begin
     scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
     scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
     scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
     scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
     end
  29:begin
     scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
     scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
     scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
     scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
     end
  30:begin
     scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
     scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
     scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
     scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
     end
  31:begin
     scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
     scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
     scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
     scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
     end
  32:begin
     scans(j).TriplePhaseErr(tr,ch)=-flag*abs(scans(j).TriplePhaseErr(tr,ch))
     scans(j).TriplePhaseCErr(tr,ch)=-flag*abs(scans(j).TriplePhaseCErr(tr,ch))
     end
  33:begin
     scans(j).TriplePhaseErr(tr,ch)=-flag*abs(scans(j).TriplePhaseErr(tr,ch))
     scans(j).TriplePhaseCErr(tr,ch)=-flag*abs(scans(j).TriplePhaseCErr(tr,ch))
     end
  34:begin
;    Photonrate
     scans(j).PhotonRateErr(ob,ch)=-flag*abs(scans(j).PhotonRateErr(ob,ch))
     scans(j).VisSqErr(ob,ch,*)=-flag*abs(scans(j).VisSqErr(ob,ch,*))
     scans(j).VisSqCErr(ob,ch,*)=-flag*abs(scans(j).VisSqCErr(ob,ch,*))
     scans(j).VisSqEErr(ob,ch,*)=-flag*abs(scans(j).VisSqEErr(ob,ch,*))
     scans(j).VisSqECErr(ob,ch,*)=-flag*abs(scans(j).VisSqECErr(ob,ch,*))
     for tr=0,GenConfig.NumTriple-1 do begin
     for l=0,2 do begin
       if (GenConfig.TripleBeam(l,tr) eq ob) then begin
	 k=where(GenConfig.TripleChan(0:GenConfig.TripleNumChan(tr)-1,l,tr) eq ch,count)
	 not_l=where(indgen(3) ne l)
	 not_ob=GenConfig.TripleBeam(not_l,tr)
	 not_bl=GenConfig.TripleBase(not_l,tr)
         kk=[k,k]
	 if count gt 0 then begin
	  if flag eq 1 $
          or (flag eq -1 $
          and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
           scans(j).TripleAmpErr(tr,k)=-flag*abs(scans(j).TripleAmpErr(tr,k))
           scans(j).TripleAmpCErr(tr,k)=-flag*abs(scans(j).TripleAmpCErr(tr,k))
           scans(j).TripleAmpEErr(tr,k)=-flag*abs(scans(j).TripleAmpEErr(tr,k))
           scans(j).TripleAmpECErr(tr,k)=-flag*abs(scans(j).TripleAmpECErr(tr,k))
           scans(j).TriplePhaseErr(tr,k)=-flag*abs(scans(j).TriplePhaseErr(tr,k))
           scans(j).TriplePhaseCErr(tr,k)=-flag*abs(scans(j).TriplePhaseCErr(tr,k))
	  endif
	 endif
       endif
     endfor
     endfor
     end
  36:begin
;    FDLPath
     scans(j).FDLPosErr(ib)=-flag*abs(scans(j).FDLPosErr(ib))
     for ob=0,GenConfig.NumOutBeam-1 do begin
     for bl=0,GenConfig.NumBaseline(ob)-1 do begin
       if strpos(GenConfig.BaselineId(bl,ob),GenConfig.StationId(ib)) ne -1 $
		then begin
         scans(j).DelayJitterErr(ob,bl)=-flag*abs(scans(j).DelayJitterErr(ob,bl))
	 scans(j).VisSqErr(ob,*,bl)=-flag*abs(scans(j).VisSqErr(ob,*,bl))
         scans(j).VisSqCErr(ob,*,bl)=-flag*abs(scans(j).VisSqCErr(ob,*,bl))
         scans(j).VisSqEErr(ob,*,bl)=-flag*abs(scans(j).VisSqEErr(ob,*,bl))
         scans(j).VisSqECErr(ob,*,bl)=-flag*abs(scans(j).VisSqECErr(ob,*,bl))
       endif
     endfor
     endfor
     for tr=0,GenConfig.NumTriple-1 do begin
     for l=0,2 do begin
       m=GenConfig.TripleBase(l,tr)
       n=GenConfig.TripleBeam(l,tr)
       if strpos(GenConfig.BaselineId(m,n),GenConfig.StationId(ib)) ne -1 $
	  							  then begin
       not_l=where(indgen(3) ne l)
       not_ob=GenConfig.TripleBeam(not_l,tr)
       not_bl=GenConfig.TripleBase(not_l,tr)
       for ch=0,GenConfig.TripleNumChan(tr)-1 do begin
	k=GenConfig.TripleChan(ch,l,tr)
        kk=[k,k]
	if flag eq 1 $
        or (flag eq -1 $
        and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
	 scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
	 scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
	 scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
	 scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
         scans(j).TriplePhaseErr(tr,ch)=-flag*abs(scans(j).TriplePhaseErr(tr,ch))
         scans(j).TriplePhaseCErr(tr,ch)=-flag*abs(scans(j).TriplePhaseCErr(tr,ch))
	endif
       endfor
       endif
     endfor
     endfor
     end
  37:begin
;    DelayJitter
     baseline=GenConfig.BaselineId(bl,ob)
     station1=strmid(baseline,0,3)
     station2=strmid(baseline,4,3)
     for ob=0,GenConfig.NumOutBeam-1 do begin
     for bl=0,GenConfig.NumBaseline(ob)-1 do begin
      if (strpos(GenConfig.BaselineId(bl,ob),station1) ne -1) and $
	 (strpos(GenConfig.BaselineId(bl,ob),station2) ne -1) then begin
       scans(j).DelayJitterErr(ob,bl)=-flag*abs(scans(j).DelayJitterErr(ob,bl))
       scans(j).VisSqErr(ob,*,bl)=-flag*abs(scans(j).VisSqErr(ob,*,bl))
       scans(j).VisSqCErr(ob,*,bl)=-flag*abs(scans(j).VisSqCErr(ob,*,bl))
       scans(j).VisSqEErr(ob,*,bl)=-flag*abs(scans(j).VisSqEErr(ob,*,bl))
       scans(j).VisSqECErr(ob,*,bl)=-flag*abs(scans(j).VisSqECErr(ob,*,bl))
      endif
      for tr=0,GenConfig.NumTriple-1 do begin
      for l=0,2 do begin
       if (GenConfig.TripleBase(l,tr) eq bl) and $
	  (GenConfig.TripleBeam(l,tr) eq ob) then begin
        not_l=where(indgen(3) ne l)
        not_ob=GenConfig.TripleBeam(not_l,tr)
        not_bl=GenConfig.TripleBase(not_l,tr)
        for ch=0,GenConfig.TripleNumChan(tr)-1 do begin
	 k=GenConfig.TripleChan(ch,l,tr)
         kk=[k,k]
	 if flag eq 1 $
         or (flag eq -1 $
         and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
          scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
          scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
          scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
          scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
          scans(j).TriplePhaseErr(tr,ch)=-flag*abs(scans(j).TriplePhaseErr(tr,ch))
          scans(j).TriplePhaseCErr(tr,ch)=-flag*abs(scans(j).TriplePhaseCErr(tr,ch))
	 endif
	endfor
       endif
      endfor
      endfor
     endfor
     endfor
     end
  38:begin
;    NATJitter
     baseline=GenConfig.BaselineId(bl,ob)
     station1=strmid(baseline,0,3)
     station2=strmid(baseline,4,3)
     scans(j).NATJitterErr(where(GenConfig.StationId eq station1))= $
      -flag*abs(scans(j).NATJitterErr(where(GenConfig.StationId eq station1)))
     scans(j).NATJitterErr(where(GenConfig.StationId eq station2))= $
      -flag*abs(scans(j).NATJitterErr(where(GenConfig.StationId eq station2)))
     for ob=0,GenConfig.NumOutBeam-1 do begin
     for bl=0,GenConfig.NumBaseline(ob)-1 do begin
      if (strpos(GenConfig.BaselineId(bl,ob),station1) ne -1) and $
	 (strpos(GenConfig.BaselineId(bl,ob),station2) ne -1) then begin
       scans(j).VisSqErr(ob,*,bl)=-flag*abs(scans(j).VisSqErr(ob,*,bl))
       scans(j).VisSqCErr(ob,*,bl)=-flag*abs(scans(j).VisSqCErr(ob,*,bl))
       scans(j).VisSqEErr(ob,*,bl)=-flag*abs(scans(j).VisSqEErr(ob,*,bl))
       scans(j).VisSqECErr(ob,*,bl)=-flag*abs(scans(j).VisSqECErr(ob,*,bl))
      endif
      for tr=0,GenConfig.NumTriple-1 do begin
      for l=0,2 do begin
       if (GenConfig.TripleBase(l,tr) eq bl) and $
	  (GenConfig.TripleBeam(l,tr) eq ob) then begin
        not_l=where(indgen(3) ne l)
        not_ob=GenConfig.TripleBeam(not_l,tr)
        not_bl=GenConfig.TripleBase(not_l,tr)
        for ch=0,GenConfig.TripleNumChan(tr)-1 do begin
	 k=GenConfig.TripleChan(ch,l,tr)
         kk=[k,k]
	 if flag eq 1 $
         or (flag eq -1 $
         and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
          scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
          scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
          scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
          scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
          scans(j).TriplePhaseErr(tr,ch)=-flag*abs(scans(j).TriplePhaseErr(tr,ch))
          scans(j).TriplePhaseCErr(tr,ch)=-flag*abs(scans(j).TriplePhaseCErr(tr,ch))
	 endif
	endfor
       endif
      endfor
      endfor
     endfor
     endfor
     end
  39:begin
;    FDLDelay
     scans(j).FDLPosErr(ib)=-flag*abs(scans(j).FDLPosErr(ib))
     scans(j).GrpDelayErr(ib)=-flag*abs(scans(j).GrpDelayErr(ib))
     scans(j).DryDelayErr(ib)=-flag*abs(scans(j).DryDelayErr(ib))
     scans(j).WetDelayErr(ib)=-flag*abs(scans(j).WetDelayErr(ib))
     for ob=0,GenConfig.NumOutBeam-1 do begin
     for bl=0,GenConfig.NumBaseline(ob)-1 do begin
       if strpos(GenConfig.BaselineId(bl,ob),GenConfig.StationId(ib)) ne -1 $
		then begin
         scans(j).DelayJitterErr(ob,bl)=-flag*abs(scans(j).DelayJitterErr(ob,bl))
	 scans(j).VisSqErr(ob,*,bl)=-flag*abs(scans(j).VisSqErr(ob,*,bl))
         scans(j).VisSqCErr(ob,*,bl)=-flag*abs(scans(j).VisSqCErr(ob,*,bl))
         scans(j).VisSqEErr(ob,*,bl)=-flag*abs(scans(j).VisSqEErr(ob,*,bl))
         scans(j).VisSqECErr(ob,*,bl)=-flag*abs(scans(j).VisSqECErr(ob,*,bl))
       endif
     endfor
     endfor
     for tr=0,GenConfig.NumTriple-1 do begin
     for bl=0,2 do begin
       m=GenConfig.TripleBase(bl,tr)
       n=GenConfig.TripleBeam(bl,tr)
       if strpos(GenConfig.BaselineId(m,n),GenConfig.StationId(ib)) ne -1 $
	  							  then begin
       not_l=where(indgen(3) ne bl)
       not_ob=GenConfig.TripleBeam(not_l,tr)
       not_bl=GenConfig.TripleBase(not_l,tr)
       for ch=0,GenConfig.TripleNumChan(tr)-1 do begin
	k=GenConfig.TripleChan(ch,bl,tr)
        kk=[k,k]
	if flag eq 1 $
        or (flag eq -1 $
        and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
	 scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
	 scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
	 scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
	 scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
         scans(j).TriplePhaseErr(tr,ch)=-flag*abs(scans(j).TriplePhaseErr(tr,ch))
         scans(j).TriplePhaseCErr(tr,ch)=-flag*abs(scans(j).TriplePhaseCErr(tr,ch))
	endif
       endfor
       endif
     endfor
     endfor
     end
  53:begin
;    FDL_O-C
     scans(j).FDLPosErr(ib)=-flag*abs(scans(j).FDLPosErr(ib))
     scans(j).GrpDelayErr(ib)=-flag*abs(scans(j).GrpDelayErr(ib))
     scans(j).DryDelayErr(ib)=-flag*abs(scans(j).DryDelayErr(ib))
     scans(j).WetDelayErr(ib)=-flag*abs(scans(j).WetDelayErr(ib))
     for ob=0,GenConfig.NumOutBeam-1 do begin
     for bl=0,GenConfig.NumBaseline(ob)-1 do begin
       if strpos(GenConfig.BaselineId(bl,ob),GenConfig.StationId(ib)) ne -1 $
		then begin
         scans(j).DelayJitterErr(ob,bl)=-flag*abs(scans(j).DelayJitterErr(ob,bl))
	 scans(j).VisSqErr(ob,*,bl)=-flag*abs(scans(j).VisSqErr(ob,*,bl))
         scans(j).VisSqCErr(ob,*,bl)=-flag*abs(scans(j).VisSqCErr(ob,*,bl))
         scans(j).VisSqEErr(ob,*,bl)=-flag*abs(scans(j).VisSqEErr(ob,*,bl))
         scans(j).VisSqECErr(ob,*,bl)=-flag*abs(scans(j).VisSqECErr(ob,*,bl))
       endif
     endfor
     endfor
     for tr=0,GenConfig.NumTriple-1 do begin
     for bl=0,2 do begin
       m=GenConfig.TripleBase(bl,tr)
       n=GenConfig.TripleBeam(bl,tr)
       if strpos(GenConfig.BaselineId(m,n),GenConfig.StationId(ib)) ne -1 $
	  							  then begin
       not_l=where(indgen(3) ne bl)
       not_ob=GenConfig.TripleBeam(not_l,tr)
       not_bl=GenConfig.TripleBase(not_l,tr)
       for ch=0,GenConfig.TripleNumChan(tr)-1 do begin
	k=GenConfig.TripleChan(ch,bl,tr)
        kk=[k,k]
	if flag eq 1 $
        or (flag eq -1 $
        and produkt(scans(j).VisSqErr(not_ob,kk,not_bl) gt 0) ne 0) then begin
	 scans(j).TripleAmpErr(tr,ch)=-flag*abs(scans(j).TripleAmpErr(tr,ch))
	 scans(j).TripleAmpCErr(tr,ch)=-flag*abs(scans(j).TripleAmpCErr(tr,ch))
	 scans(j).TripleAmpEErr(tr,ch)=-flag*abs(scans(j).TripleAmpEErr(tr,ch))
	 scans(j).TripleAmpECErr(tr,ch)=-flag*abs(scans(j).TripleAmpECErr(tr,ch))
         scans(j).TriplePhaseErr(tr,ch)=-flag*abs(scans(j).TriplePhaseErr(tr,ch))
         scans(j).TriplePhaseCErr(tr,ch)=-flag*abs(scans(j).TriplePhaseCErr(tr,ch))
	endif
       endfor
       endif
     endfor
     endfor
     end
  54:scans(j).GrpDelayErr(ib)=-flag*abs(scans(j).GrpDelayErr(ib))
  55:scans(j).DryDelayErr(ib)=-flag*abs(scans(j).DryDelayErr(ib))
  56:scans(j).WetDelayErr(ib)=-flag*abs(scans(j).WetDelayErr(ib))
else:begin
     print,'***Error(FLAGSCANDATA): item not allowed: ',flagtable(i).item,'!'
     fcount=fcount-1
     end
  endcase
endif
ENDFOR
;
if not !qiet then print,'Editing complete:',fcount,' primary data points.'
;
; Overwrite copy of data in AMOEBA buffer
set_compltriple
if n_elements(bufferinfo) gt 1 then $
storenight,11
;
end
;-------------------------------------------------------------------------------
pro unflag,class,reason
;
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
if n_elements(class) eq 0 then begin
	print,"***Error(UNFLAG): you have to specify class!"
	return
endif
if n_elements(reason) eq 0 then begin
	print,'***Error(UNFLAG): you have to specify REASON!'
	return
endif
case class of
	'point':begin
		if checkdata([17,11]) ne 0 then return
                index=where(pointflagtable.reason eq reason(0) $
			and pointflagtable.time gt 0,count)
                print,'Found ',count,' entries.'
                if count gt 0 then begin
                        table=pointflagtable(index)
                        flagpointdata,table,flag=-1
                        pointflagtable(index).time=-1
			pointflagtable= $
			pointflagtable(reverse(sort(pointflagtable.time)))
                endif
                end
	'metro':begin
		if checkdata([20,16]) ne 0 then return
                index=where(inchflagtable.reason eq reason(0) $
			and inchflagtable.time gt 0,count)
                print,'Found ',count,' entries.'
                if count gt 0 then begin
                        table=inchflagtable(index)
                        flaginchdata,table,flag=-1
                        inchflagtable(index).time=-1
			inchflagtable= $
			inchflagtable(reverse(sort(inchflagtable.time)))
                endif
                end
	'bg'   :begin
		if checkdata([18,10]) ne 0 then return
		index=where(bgflagtable.reason eq reason(0) $
			and bgflagtable.time gt 0,count)
		print,'Found ',count,' entries.'
		if count ne 0 then begin
			table=bgflagtable(index)
			flagbgdata,table,flag=-1
			bgflagtable(index).time=-1
			bgflagtable= $
			bgflagtable(reverse(sort(bgflagtable.time)))
		endif
		end
	'scan' :begin
		if checkdata([19,9]) ne 0 then return
		index=where(scanflagtable.reason eq reason(0) $
			and scanflagtable.time gt 0,count)
		print,'Found ',count,' entries.'
		if count ne 0 then begin
			table=scanflagtable(index)
			flagscandata,table,flag=-1L
			scanflagtable(index).time=-1
			scanflagtable= $
			scanflagtable(reverse(sort(scanflagtable.time)))
		endif
		end
	else   :print,'***Error(UNFLAG): class not valid: ',class
endcase
end
;-------------------------------------------------------------------------------
pro resetflag,class
;
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(class) eq 0 then begin
	print,"***Error(RESETFLAG): you have to specify class!"
	return
endif
;
case class of
	'scan' :begin
		if checkdata([9]) ne 0 then return
		scans.time=abs(scans.time)
		scans.vissqerr=abs(scans.vissqerr)
		scans.vissqeerr=abs(scans.vissqeerr)
		scans.vissqcerr=abs(scans.vissqcerr)
		scans.vissqecerr=abs(scans.vissqecerr)
		scans.compltripleerr=abs(scans.compltripleerr)
		scans.tripleamperr=abs(scans.tripleamperr)
		scans.tripleampeerr=abs(scans.tripleampeerr)
		scans.tripleampcerr=abs(scans.tripleampcerr)
		scans.tripleampecerr=abs(scans.tripleampecerr)
		scans.triplephaseerr=abs(scans.triplephaseerr)
		scans.triplephasecerr=abs(scans.triplephasecerr)
		scans.photonrateerr=abs(scans.photonrateerr)
		scans.backgnderr=abs(scans.backgnderr)
		scans.fdlposerr=abs(scans.fdlposerr)
		scans.natjittererr=abs(scans.natjittererr)
		scans.delayjittererr=abs(scans.delayjittererr)
		scans.grpdelayerr=abs(scans.grpdelayerr)
		scans.drydelayerr=abs(scans.drydelayerr)
		scans.wetdelayerr=abs(scans.wetdelayerr)
		if n_elements(bufferinfo) gt 1 then $
		storenight,11
		end
	'bg'   :begin
		if checkdata([10]) ne 0 then return
		bgscans.time=abs(bgscans.time)
		bgscans.rateerr=abs(bgscans.rateerr)
		end
	'point':begin
		if checkdata([11]) ne 0 then return
		pointtime=abs(pointtime)
		vissqerr=abs(vissqerr)
		photonrateerr=abs(photonrateerr)
		if n_elements(compltriple) ne 0 then begin
			compltripleerr=abs(compltripleerr)
			tripleamperr=abs(tripleamperr)
			triplephaseerr=abs(triplephaseerr)
		endif
		fdlposerr=abs(fdlposerr)
		natjittererr=abs(natjittererr)
		delayjittererr=abs(delayjittererr)
		if n_elements(grpdelay) ne 0 then begin
			grpdelayerr=abs(grpdelayerr)
			drydelayerr=abs(drydelayerr)
			wetdelayerr=abs(wetdelayerr)
		endif
		end
	'metro':begin
		if checkdata([16]) ne 0 then return
		metrotime=abs(metrotime)
		parxerr=abs(parxerr)
		paryerr=abs(paryerr)
		parzerr=abs(parzerr)
		totaldelayerr=abs(totaldelayerr)
		end
	else:	print,'***Error(RESETFLAG): unknown class:',class,'!'
endcase
;
print,'Flags reset.'
;		
end
;************************************************************************Block 4
pro plotdata,edit,reason,overplot=overplot,ms1=ms1,ms2=ms2
;
; Main plotting routine. Plots data (2D, 3D), all classes, all slices.
;
; Input parms: 
; edit=0: no editing; edit=1: auto; edit=2: zero
; reason: Flag reason (string).
;
; Some information on symbols and colors:
; !p.psym=0: no symbol, connect points with lines
;	  1: +
;	  2: *
;	  3: .
;	  4: Diamond
;	  5: Triangle
;	  6: Square
;	  7: x
; tek_color
; !p.color=0: black
;	   1: white
;	   2: red
;	   3: green
;	   4: blue
;	   .
;	   .
;         31: <last color index>
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common PlotDataInfo,plotscans,plotdata_x,plotdata_y,plotdata_y_bck
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common StarBase,startable,notes
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common PlacementInfo,xpix_per_char,ypix_per_char, $
		     title_yoffset,title_xoffset, $
		     xtitle_xoffset,xtitle_yoffset, $
		     ytitle_xoffset,ytitle_yoffset, $
		     num_title,max_star_per_title,xpix_per_star, $
		     num_xtitle,max_xlabel_per_title,xpix_per_xlabel, $
		     num_ytitle,max_ylabel_per_title,xpix_per_ylabel
;
; If overplot=0 then open a new window
do_multi_nights=0
if n_elements(overplot) eq 0 then overplot=0 else do_multi_nights=1
;
; Do we want to edit?
if n_params() eq 0 then edit=0
if (edit lt 0) or (edit gt 2) then begin
	print,'***Error(PLOTDATA): invalid editing mode: ',edit,'!'
	return
endif
;
; NoTrace option not selected (default): X-selection will trace Y-selection
if not ps_options.t then begin
	xitem=ds_x.item
	ds_x=ds_y
	ds_x.item=xitem
endif
;
; Check data selection
if printselerror(checkplotsel('x')) ne 0 then return
if printselerror(checkplotsel('y')) ne 0 then return
if printselerror(checkplotsel('z')) ne 0 then return
;
; Set label indices
ax_items=set_axitems(item_ids,units,abbrs)
ax_items=ax_items+units
set_plotlabels,plot_xlabels,xlabel_index,ds_x.item
set_plotlabels,plot_ylabels,ylabel_index,ds_y.item
set_plotlabels,plot_zlabels,zlabel_index,ds_z.item
;
; Initial loop ranges
set_plotloops,'x',num_b_x,num_c_x,num_p_x
set_plotloops,'y',num_b_y,num_c_y,num_p_y
;
; NoTrace option not selected (default): loops controlled by Y-selection
if not ps_options.t then begin
	num_b_x=1
	num_c_x=1
	num_p_x=1
endif
;
; Initialize index counter
numarr=intarr(6,num_b_x*num_c_x*num_p_x*num_b_y*num_c_y*num_p_y)
num_plot=0
;
; Some options conflict
if ps_options.g and ps_options.d then begin
	print,'***Error(PLOT): incompatible options (Select Image or 3D)!'
	return
endif
;
; Check existence of data and prepare a structure array (datascans) holding 
; star and time information for the selected data scans/points. It is used 
; to select the data for a particular star and to provide time stamps for 
; editing. 
parms=init_class(class,datascans)
do_info=1
;
; Determine whether automatic scaling is requested
if total(abs(!x.range)) eq 0 then xrange_auto=1 else xrange_auto=0
if total(abs(!y.range)) eq 0 then yrange_auto=1 else yrange_auto=0
if total(abs(!z.range)) eq 0 then zrange_auto=1 else zrange_auto=0
;
; For option All in 1 we have to go through the data twice to find the
; number of plots in a window (num_plot) and the total range of data values.
; However, we now go through it twice anyway to see how many plots we have.
if ps_options.a or ps_options.g then num_run=2 else num_run=2
;
; Initialize the data selection for each stream
set_streamsel,'x',d,xinbeam,xoutbeam,xtriple,x_ds_ch,x_ds_bl,x_ds_pt
set_streamsel,'y',d,yinbeam,youtbeam,ytriple,y_ds_ch,y_ds_bl,y_ds_pt
xitem=ds_x.item
yitem=ds_y.item
zitem=ds_z.item
;
; Check for polar mode az-el plots
if ps_options.v and xitem eq 43 and yitem eq 78 then polar=1 else polar=0
;
; Various checks for editing mode
if edit ne 0 then begin
num_run=1
do_info=0
case class of
	'point':begin
		if n_elements(pointflagtable) eq 0 then create_pointflagtable
		index=where(pointflagtable.time ge 0,count)
		end
	'delay':begin
		if n_elements(pointflagtable) eq 0 then create_pointflagtable
		index=where(pointflagtable.time ge 0,count)
		end
	'scan' :begin
		if n_elements(scanflagtable) eq 0 then create_scanflagtable
		index=where(scanflagtable.time ge 0,count)
		end
	'phot' :begin
		if n_elements(scanflagtable) eq 0 then create_scanflagtable
		index=where(scanflagtable.time ge 0,count)
		end
	'bg'   :begin
		if n_elements(bgflagtable) eq 0 then create_bgflagtable
		index=where(bgflagtable.time ge 0,count)
		end
	'metro':begin
		if n_elements(inchflagtable) eq 0 then create_inchflagtable
		index=where(inchflagtable.time ge 0,count)
		end
	else:	begin
		print,'***Error(PLOTDATA): no editing allowed in this class!'
		return
		end
endcase
id0=count
id=id0
if slice ne 'pt' then begin
	print,'***Error(PLOTDATA): no automatic editing with slice other than pt!'
	return
endif
timestamp=strmid(systime(),0,24)
if n_elements(reason) eq 0 then begin
	reason=''
	read,'Please enter reason: ',reason
endif
reason=timestamp+' '+reason+'_'+abbrs(ds_y.item-1)
print,'Begin editing, REASON='+reason
endif
;
; Loops start here!
FOR run_no=1,num_run DO BEGIN
plot_no=0	; Current (sub)plot number; first plot will be no. 1
;
IF run_no eq 2 AND num_plot gt 0 THEN BEGIN
;
num_pt_x=n_elements(unique(numarr(0,0:num_plot-1)))
num_ch_x=n_elements(unique(numarr(1,0:num_plot-1)))
num_bl_x=n_elements(unique(numarr(2,0:num_plot-1)))
num_pt_y=n_elements(unique(numarr(3,0:num_plot-1)))
num_ch_y=n_elements(unique(numarr(4,0:num_plot-1)))
num_bl_y=n_elements(unique(numarr(5,0:num_plot-1)))
;
; Set number of plots in x and y
case class of
	'point' :begin
		 num_v=num_bl_x*num_bl_y
		 num_h=num_ch_x*num_ch_y
;		 Successive division method
		 if num_v eq 1 or num_h eq 1 then begin
		 	num_t=num_h*num_v
			num_v=num_t
			num_h=1
		 	while num_v/num_h ge 2 do begin
				num_h=num_h+1
				num_v=num_t/num_h
		 		if num_v*num_h lt num_t then num_v=num_v+1
		 	endwhile
		 endif
		 end
	'delay' :begin
		 if slice eq 'ch' then begin
			num_p=num_pt_x*num_pt_y
			num_h=long(sqrt(num_p))
			if num_h*num_h ne num_p then begin
				if num_p-num_h*num_h gt num_h then begin
					num_h=num_h+1
					num_v=num_h
				endif else num_v=num_h+1
			endif else num_v=num_h
		 endif else begin
			num_h=num_bl_x*num_bl_y
			num_v=num_ch_x*num_ch_y
		 endelse
		 end
	'bg'    :begin
;		 Square method
		 num_c=num_ch_x*num_ch_y
		 num_h=long(sqrt(num_c))
		 if num_h*num_h ne num_c then begin
			if num_c-num_h*num_h gt num_h then begin
				num_h=num_h+1
				num_v=num_h
			endif else num_v=num_h+1
		 endif else num_v=num_h
		 end
	'scan'  :begin
;		 Successive division method
		 num_v=num_bl_x*num_bl_y
		 num_h=num_ch_x*num_ch_y*num_pt_x*num_pt_y
		 if num_plot lt (num_v*num_h) then begin
			num_h=1
			num_v=num_plot
		 endif
		 if num_v eq 1 or num_h eq 1 then begin
		 	num_t=num_h*num_v
		 	num_v=num_t
		 	num_h=1
		 	while num_v/num_h ge 2 do begin
				num_h=num_h+1
				num_v=num_t/num_h
		 		if num_v*num_h lt num_t then num_v=num_v+1
		 	endwhile
		 endif
		 end
	'astrom':begin
		 num_h=num_bl_x*num_bl_y
		 num_v=num_ch_x*num_ch_y
		 end
	'seeing':begin
		 num_h=num_bl_x*num_bl_y
		 num_v=num_ch_x*num_ch_y
		 end
	'phot'	:begin
		 num_h=1
		 num_v=1
		 end
	'metro' :begin
;		 Successive division method
		 num_h=num_bl_x*num_bl_y
		 num_v=num_ch_x*num_ch_y
		 num_t=num_h*num_v
		 while num_v/num_h ge 2 do begin
			num_h=num_h+1
			num_v=num_t/num_h
		 	if num_v*num_h lt num_t then num_v=num_v+1
		 endwhile
		 end
	'amoeba':begin
;		 Successive division method
		 num_v=num_bl_x*num_bl_y
		 num_h=num_ch_x*num_ch_y*num_pt_x*num_pt_y
		 if num_v eq 1 or num_h eq 1 then begin
			num_t=num_h*num_v
			num_v=num_t
			num_h=1
		 	while num_v/num_h ge 2 do begin
				num_h=num_h+1
				num_v=num_t/num_h
		 		if num_v*num_h lt num_t then num_v=num_v+1
		 	endwhile
		 endif else if num_h/num_v gt 3 then begin
			num_v=num_v+2
			num_h=num_t/num_v
			if num_v*num_h lt num_t then num_h=num_h+1
		 endif
		 end
	'volvox':begin
;		 Successive division method
		 num_v=num_bl_x*num_bl_y
		 num_h=num_ch_x*num_ch_y*num_pt_x*num_pt_y
		 if num_v eq 1 or num_h eq 1 then begin
			num_t=num_h*num_v
			num_v=num_t
			num_h=1
		 	while num_v/num_h ge 2 do begin
				num_h=num_h+1
				num_v=num_t/num_h
		 		if num_v*num_h lt num_t then num_v=num_v+1
		 	endwhile
		 endif
		 end
	else	:begin
		 print,'***Error(PLOTDATA): this class not handled!'
		 return
		 end
endcase
; !p.multi=[#plots remaining,#plot columns,#plot rows,#plots stacked in Z,
;           0:left to right and top to bottom;1:top to bottom and left to right]
if ps_options.a $
or ps_options.d $
or ps_options.g then !p.multi=[0,1,1,0,1] $
		else !p.multi=[0,num_h,num_v,0,1] 
;
; For multiple plots, IDL reduces the charsize by a factor of 2
!p.charsize=1.0
if !p.multi(1)*!p.multi(2) le 1 then !p.charsize=1.5
if !p.multi(1) gt 2 or !p.multi(2) gt 2 then begin
	!p.charsize=2.0
	mf=2.0 
endif else mf=1.0
;
; Open a plot window for plotting and manual editing on the screen
if (edit eq 0) and (not ps_options.d) $
	       and (not ps_options.g) $
	       and (!d.name ne 'PS') then begin
	if overplot eq 0 then $
	window,/free,title=Date+', Class:'+class,xsize=!xsize,ysize=!ysize, $
		xpos=!dxsize-!xsize,ypos=!dysize-!ysize
endif else if (edit eq 0) and (not ps_options.d) $
              	 	  and (not ps_options.g) $
	      		  and (!d.name eq 'PS') then begin
	device,color=ps_options.c
	tek_color
	if ps_options.p then begin
		if mf eq 1 then !p.charsize=1.5 else !p.charsize=2.0
	endif
endif
if !d.name eq 'PS' and ps_options.c then frame_color=0 else frame_color=1
;
; Do not maintain info about plotted data in some cases, e.g. multiple plots
if !p.multi(1)*!p.multi(2) gt 1 then do_info=0
if ps_options.d or ps_options.g then do_info=0
;
; Calculate star title placement variables
titleplacement,'t',ds_stars
stars=ds_stars
for i=0,n_elements(stars)-1 do $
	stars(i)=startable(where(startable.starid eq stars(i))).name
titleplacement,'t',stars
;
ENDIF
; !p.multi=[0,2,4,0]
;
for pt_x=0,num_p_x-1 do begin
for ch_x=0,num_c_x-1 do begin
for bl_x=0,num_b_x-1 do begin
for pt_y=0,num_p_y-1 do begin
for ch_y=0,num_c_y-1 do begin
for bl_y=0,num_b_y-1 do begin
;
;	Set x-axis data
	if ps_options.t then begin
		ch=ch_x
		bl=bl_x
		pt=pt_x
	endif else begin
		ch=ch_y
		bl=bl_y
		pt=pt_y
	endelse
	xchan=x_ds_ch(ch)
	xbase=x_ds_bl(bl)
	xpoint=x_ds_pt(pt)
	case slice of
		'ch':xchan=x_ds_ch
		'bl':xbase=x_ds_bl
		'pt':xpoint=x_ds_pt
	        else:
	endcase
;	If triple data is plotted, OB and CH need to be derived from TR selection
	triple_index=[6,7,28,29,30,31,32,33,58,59]
	index=where(yitem eq triple_index,tr_count)
	if tr_count gt 0 then begin
		xoutbeam=genconfig.triplebeam(0,xtriple)
		xchan=genconfig.triplechan(xchan,0,xtriple)
	endif
	xdata=dblarr(n_elements(xchan)*n_elements(xbase)*n_elements(xpoint),/nozero)
	xdata(*)=set_plotdata(xitem,xinbeam,xoutbeam,xtriple,xchan,xbase,xpoint)
	xdata_err=xdata
	xdata_err(*)=set_ploterr(xitem,xinbeam,xoutbeam,xtriple,xchan,xbase,xpoint)
;
;	Set y-axis data
	ychan=y_ds_ch(ch_y)
	ybase=y_ds_bl(bl_y)
	ypoint=y_ds_pt(pt_y)
	case slice of
		'ch':ychan=y_ds_ch
		'bl':ybase=y_ds_bl
		'pt':ypoint=y_ds_pt
	        else:
	endcase
	index=where(yitem eq triple_index,tr_count)
	if tr_count gt 0 then begin
		youtbeam=genconfig.triplebeam(0,ytriple)
		ychan=genconfig.triplechan(ychan,0,ytriple)
	endif
	ydata=dblarr(n_elements(ychan)*n_elements(ybase)*n_elements(ypoint),/nozero)
	ydata(*)=set_plotdata(yitem,yinbeam,youtbeam,ytriple,ychan,ybase,ypoint)
	ydata_err=ydata
	ydata_err(*)=set_ploterr(yitem,yinbeam,youtbeam,ytriple,ychan,ybase,ypoint)
;
;	Set z-axis data
	index=where(yitem eq triple_index,tr_count)
	if tr_count gt 0 then begin
		youtbeam=genconfig.triplebeam(0,ytriple)
		ychan=genconfig.triplechan(ychan,0,ytriple)
	endif
	zdata=dblarr(n_elements(ychan)*n_elements(ybase)*n_elements(ypoint),/nozero)
	zdata(*)=set_plotdata(zitem,yinbeam,youtbeam,ytriple,ychan,ybase,ypoint)
;
;	Set ids for all plotted points
	item=yitem
	inbeam=yinbeam
	outbeam=youtbeam
	triple=ytriple
	chan=ychan
	base=ybase
	point=ypoint
	set_sliceids,item,inbeam,outbeam,triple,chan,base,point
;
; 	Automatic flagging; once complete, skip the plotting
	if (edit eq 1 or edit eq 2) then begin
	if class eq 'point' or class eq 'delay' then begin
		MINPOINTS=10
		mv=fltarr(n_elements(Iscan))	; for statistical information
		me=fltarr(n_elements(Iscan))	; for statistical information
		plotscans={y:mv,ye:me}
;		Process on a scan-by-scan basis
	    	FOR n=0,n_elements(Iscan)-1 DO BEGIN
		index=where(point ge Rec0(n) and point le RecN(n),count)
		IF count GT 0 THEN BEGIN
		y=ydata(index)
		e=ydata_err(index)
		p=index
		if edit eq 1 then begin		 	; Edit outliers
		   index=where(e ge 0,count)
; 		   Do not edit scans with few points (B. Zavala)
;                  if count le MINPOINTS then count=0 (removed: CAH 2011-12-02)
		   if count gt MINPOINTS then begin
			   ye=y(index)
;			   sig=stddev(ye)
			   med=medianve(ye,sig)
			   mv(n)=med
			   me(n)=sig
			   plotscans.y(n)=med
			   plotscans.ye(n)=sig
			   index_outlier=where(abs(y-med) gt (3*sig) $
					and e ge 0,count1)
			   scan_ok=0
			   if sig gt 0 then begin
			   case item(0) of
; 			   Flag all if certain constraint is not met
;				Photonrate
			    8: if sig/med lt 0.1 then scan_ok=1
;				DelayJitter
			   10: if med lt 2.0 and sig lt 1.0 then scan_ok=1
;				NATJitter
			   11: if sig lt 0.05 then scan_ok=1
; 				NATCounts
			   12: if sig/med lt 0.1 then scan_ok=1
;			   Default is to use index_outlier
			   else: index=index_outlier
			   endcase
			   endif
			   if scan_ok then index=index_outlier
			   if index(0) ne -1 then count=n_elements(index) $
					     else count=0
		   endif
		endif else if edit eq 2 then begin	; Edit zero values
		   index=where(y eq 0,count)
		endif
		if count gt 0 then begin
		   if count eq 1 then index=index(0)
		   set_pointflagtable,reason,item(p(index)), $
			inbeam(p(index)),outbeam(p(index)),triple(p(index)), $
			chan(p(index)),base(p(index)), $
			abs(PointTime(point(p(index))))
		   id=id+count
		endif
		ENDIF
	    	ENDFOR
		if edit eq 1 then begin
;			Temporarily inserted for diagnostics
;			index=where(me gt 0,count)
;			if count gt 0 then begin
;			print,'Median of all values: ',median(mv(index))
;			print,'Median of all median errors: ',median(me(index))
;			print,'Median of normalized errors: ',median(me(index) $
;								    /mv(index))
;			endif
		endif
	endif else if class eq 'metro' then begin
		MINPOINTS=5
	    	for n=0,n_elements(Jscan)-1 do begin
		index=where(point ge Sol0(n) and point le SolN(n),count)
		if count gt 0 then begin
		y=ydata(index)
		e=ydata_err(index)
		p=index
		if edit eq 1 then begin		 	; Edit outliers
		   index=where(e ge 0,count)
		   if count gt MINPOINTS then begin
			   ye=y(index)
			   med=medianve(ye,sig)
;			   sig=stddev(ye)
			   index=where(abs(y-med) gt (5*sig),count)
		   endif
		endif else if edit eq 2 then begin	; Edit zero values
		   index=where(y eq 0,count)
		endif
		if count gt 0 then begin
		   if count eq 1 then index=index(0)
		   set_inchflagtable,reason,item(p(index)),inbeam(p(index)), $
			abs(MetroTime(point(p(index))))
		   id=id+count
		endif
		endif
	    	endfor
	endif else begin	; all other classes
		index=where(ydata_err gt 0,count)
		if count gt 0 then begin
		x=xdata(index)
		y=ydata(index)
		e=ydata_err(index)
		p=index
		if edit eq 1 then begin
		   	wt=1./e^2
			wt=wt/avg(wt)
			case n_elements(wt) of
		  		1:coeffs=y
		  		2:coeffs=poly_fit(x,y,1,measure_errors=e,yfit=yft)
		  		3:coeffs=poly_fit(x,y,2,measure_errors=e,yfit=yft)
		  	     else:coeffs=poly_fit(x,y,3,measure_errors=e,yfit=yft)
			endcase
		   	sigma=stddev(y-yft)
		   	index=where(abs((y-yft)/sigma) gt 2,count)
	 	endif else index=where(y eq 0,count)
		if count gt 0 then begin
			n=id+count-1
			if count eq 1 then index=index(0)
			case class of
			'scan':begin
			       scanflagtable(id:n).reason=reason
			       scanflagtable(id:n).item=item(p(index))
			       scanflagtable(id:n).baseline=base(p(index))
			       scanflagtable(id:n).triple=triple(p(index))
			       scanflagtable(id:n).channel=chan(p(index))
			       scanflagtable(id:n).outbeam=outbeam(p(index))
			       scanflagtable(id:n).inbeam=inbeam(p(index))
			       scanflagtable(id:n).time= $
				abs(scans(point(p(index))).time)
			       end
			'bg'  :begin
			       bgflagtable(id:n).reason=reason
			       bgflagtable(id:n).item=item(p(index))
			       bgflagtable(id:n).channel=chan(p(index))
			       bgflagtable(id:n).outbeam=outbeam(p(index))
			       bgflagtable(id:n).time= $
				abs(bgscans(point(p(index))).time)
			       end
			endcase
			id=n+1
		endif
		endif 
	endelse
	goto,SKIPPLOT
	endif
;
;	If there is no valid data, skip the rest
	index=where(xdata_err ge 0 and ydata_err ge 0,count)
	if ps_options.f eq 0 and count eq 0 then begin
		if not do_multi_nights then goto,SKIPPLOT
		goto,SKIPPLOT	; better?
	endif
	if ps_options.f ne 0 then index=lindgen(n_elements(ydata))

;       We have valid data, so increment the plot number counter
	plot_no=plot_no+1
	numarr(*,plot_no-1)=[pt_x,ch_x,bl_x,pt_y,ch_y,bl_y]
;
;       First, we have to find out about the ranges
	count=n_elements(index)
	jndex=lindgen(count)
	if not xrange_auto and yrange_auto then $
		jndex=where(xdata(index) ge !x.range(0) $
			and xdata(index) le !x.range(1),count)
	if not yrange_auto and xrange_auto then $
		jndex=where(ydata(index) ge !y.range(0) $
			and ydata(index) le !y.range(1),count)
	if count eq 0 then jndex=lindgen(n_elements(index))
	xmin=min(xdata(index(jndex)))
	xmax=max(xdata(index(jndex)))
	ymin=min(ydata(index(jndex)))
	ymax=max(ydata(index(jndex)))
	zmin=min(zdata(index(jndex)))
	zmax=max(zdata(index(jndex)))
;
;       Here we got the range information and can skip the rest if plotting
;       will occur in the second run through the data (All in 1 option).
	if run_no eq 1 then begin
		num_plot=num_plot+1
		if ps_options.g then goto,SKIPPLOT
		if plot_no eq 1 then begin
			axmax=xmax
			axmin=xmin
			aymax=ymax
			aymin=ymin
			azmax=zmax
			azmin=zmin

			p_xdata=xdata
			p_ydata=ydata
			p_xdata_err=xdata_err
			p_ydata_err=ydata_err

			p_item=item
			p_inbeam=inbeam
			p_outbeam=outbeam
			p_triple=triple
			p_chan=chan
			p_base=base
			p_point=point
			p_time=dblarr(n_elements(point),/nozero)
			p_time(*)=datascans(point).time
			p_star=strarr(n_elements(point))
			p_star(*)=datascans(point).starid
		endif else begin
			axmax=max([axmax,xmax])
			axmin=min([axmin,xmin])
			aymax=max([aymax,ymax])
			aymin=min([aymin,ymin])
			azmax=max([azmax,zmax])
			azmin=min([azmin,zmin])

			p_xdata=[p_xdata,xdata]
			p_ydata=[p_ydata,ydata]
			p_xdata_err=[p_xdata_err,xdata_err]
			p_ydata_err=[p_ydata_err,ydata_err]

			p_item=[p_item,item]
			p_inbeam=[p_inbeam,inbeam]
			p_outbeam=[p_outbeam,outbeam]
			p_triple=[p_triple,triple]
			p_chan=[p_chan,chan]
			p_base=[p_base,base]
			p_point=[p_point,point]
			p_time=[p_time,datascans(point).time]
			p_star=[p_star,datascans(point).starid]
			
		endelse
		goto,SKIPPLOT
	endif
;
;       Set the ranges for plotting
	if ps_options.a and not ps_options.g then begin
		if n_elements(axmax) eq 0 and not do_multi_nights then begin
			window,/free
			xyouts,0.4,0.5,'No data',/normal,charsize=2
			return
		endif
		xmax=axmax
		xmin=axmin
		ymax=aymax
		ymin=aymin
		zmax=azmax
		zmin=azmin
	endif
	if xmax eq xmin then begin
		xmax=xmax+1
		xmin=xmin-1
	endif
	if ymax eq ymin then begin
		ymax=ymax+1
		ymin=ymin-1
	endif
	if zmax eq zmin then begin
		zmax=zmax+1
		zmin=zmin-1
	endif
	if class eq 'bg' then ymin=-0.1
	if xrange_auto then !x.range=[xmin,xmax] else !x.style=1
	if yrange_auto then !y.range=[ymin,ymax] else !y.style=1
	if zrange_auto then !z.range=[zmin,zmax] else !z.style=1
;
;	Prepare the axis labels
	xlabels=plot_xlabels
	xlabels(0)=plot_xlabels(0)+strcompress(string(xinbeam(0)+1))
	xlabels(1)=plot_xlabels(1)+strcompress(string(xoutbeam(0)+1))
	xlabels(2)=plot_xlabels(2)+strcompress(string(xtriple(0)+1))
	xlabels(3)=plot_xlabels(3)+strcompress(string(xbase(0)+1))
	xlabels(4)=plot_xlabels(4)+strcompress(string(xchan(0)+1))
	xlabels(5)=plot_xlabels(5)+strcompress(string(xpoint(0)+1))
	xlabel=''
	if xlabel_index(0) ne -1 then begin
	   xlabels=xlabels(xlabel_index)
	   for i=0,n_elements(xlabels)-1 do xlabel=xlabel+xlabels(i)
	endif 
	ylabels=plot_ylabels
	ylabels(0)=plot_ylabels(0)+strcompress(string(yinbeam(0)+1))
	ylabels(1)=plot_ylabels(1)+strcompress(string(youtbeam(0)+1))
	ylabels(2)=plot_ylabels(2)+strcompress(string(ytriple(0)+1))
	ylabels(3)=plot_ylabels(3)+strcompress(string(ybase(0)+1))
	ylabels(4)=plot_ylabels(4)+strcompress(string(ychan(0)+1))
	ylabels(5)=plot_ylabels(5)+strcompress(string(ypoint(0)+1))
	ylabel=''
	if ylabel_index(0) ne -1 then begin
	   ylabels=ylabels(ylabel_index)
	   for i=0,n_elements(ylabels)-1 do ylabel=ylabel+ylabels(i)
	endif 
	zlabels=plot_zlabels
	zlabels(0)=plot_zlabels(0)+strcompress(string(yinbeam(0)+1))
	zlabels(1)=plot_zlabels(1)+strcompress(string(youtbeam(0)+1))
	zlabels(2)=plot_zlabels(2)+strcompress(string(ytriple(0)+1))
	zlabels(3)=plot_zlabels(3)+strcompress(string(ybase(0)+1))
	zlabels(4)=plot_zlabels(4)+strcompress(string(ychan(0)+1))
	zlabels(5)=plot_zlabels(5)+strcompress(string(ypoint(0)+1))
	zlabel=''
	if zlabel_index(0) ne -1 then begin
	   zlabels=zlabels(zlabel_index)
	   for i=0,n_elements(zlabels)-1 do zlabel=zlabel+zlabels(i)
	endif 
;
	!p.color=tci(frame_color)
	if xlabel eq '' then num_xlabel=0 else num_xlabel=num_plot
	if ylabel eq '' then num_ylabel=0 else num_ylabel=num_plot
	if zlabel eq '' then num_zlabel=0 else num_zlabel=num_plot
	xtitle=ax_items(long(where(item_ids eq xitem(0)),0))
	ytitle=ax_items(long(where(item_ids eq yitem(0)),0))
	ztitle=ax_items(long(where(item_ids eq zitem(0)),0))
	if not ps_options.a and not ps_options.g then begin
		xcomma=','
		ycomma=','
		zcomma=','
		if !p.multi(2) gt 2 then begin
			xtitle=ytitle+' vs '+xtitle
			ytitle=''
			ycomma=''
		endif
		if num_xlabel gt 0 then xtitle=xtitle+xcomma+xlabel
		if num_ylabel gt 0 then ytitle=ytitle+ycomma+ylabel
		if num_zlabel gt 0 then ztitle=ztitle+zcomma+zlabel
	endif else if plot_no eq 1 then begin
		titleplacement,'x',ds_stars,xlabel,num_xlabel
		titleplacement,'y',ds_stars,ylabel,num_ylabel
	endif
;
;	Plot frame
;	Image 
	if ps_options.g then begin
		if plot_no eq 1 then begin
		xpix=n_elements(ydata)
		ypix=num_plot
		y_image=bytarr(xpix,num_plot)
		y_image_err=fltarr(xpix,num_plot)
		rebin_xfactor=max([!xsize/xpix,1])
		rebin_yfactor=min([20,!ysize/ypix])
		xmargin=[50,20]
		ymargin=[50,(num_title+2)*ypix_per_char]
		xsize=!xsize+total(xmargin)
		ysize=rebin_yfactor*ypix+total(ymargin)
		window,/free,title='Class:'+class,xsize=xsize,ysize=ysize, $
			xpos=!dxsize-!xsize,ypos=!dysize-!ysize

		xpix_t=float(xpix)
		while xpix_t gt !xsize do xpix_t=xpix_t/2
		xfactor=(xsize-total(xmargin))/xpix_t

		plot,indgen(num_plot), $
		     indgen(num_plot),/nodata,/device, $
		     xtitle=ytitle+' image',ytitle='',title='', $
		     xrange=[0,xfactor*xpix/rebin_xfactor], $
		     yrange=[0,num_plot],xminor=-1, $
		     xstyle=1,yticks=num_plot, $
		     ystyle=1,ytickformat='(i2)', $
		     xticklen=-10.0/ysize, $
		     yticklen= -5.0/xsize, $
		     position=[xmargin(0)-1,ymargin(0)-1, $
			       xsize-xmargin(1), $
			       ysize-ymargin(1)]

		titleplacement,'t',ds_stars
		for is=0,n_elements(ds_stars)-1 do begin
			title_index=is/max_star_per_title
			xyouts,title_xoffset(title_index) $
				+(is mod max_star_per_title)*xpix_per_star, $
       				title_yoffset(title_index), $
				ds_stars(is), $
       				/device,size=!p.charsize/mf,color=tci((is mod 30)+2)
		endfor
		endif
;	3D
	endif else if ps_options.d then begin
		if plot_no eq 1 then begin
		pos=[0.1,0.1,0.9,0.9,0.1,0.9]
		plot,xdata,ydata,/nodata,/t3d,position=pos, $
			xtitle=xtitle,ytitle=ytitle,charsize=2.0
		xscale=!x.s
		yscale=!y.s
		t3d,/xzexch
		xrange=!x.range
		!x.range=!z.range
		plot,zdata,ydata,/nodata,/t3d,/noerase,pos=pos,xtitle=ztitle, $
			charsize=2.0
		zscale=!x.s
		!x.range=xrange
		t3d,/xzexch
		t3d,/yzexch
		yrange=!y.range
		!y.range=!z.range
		plot,xdata,zdata,/nodata,/t3d,/noerase,pos=pos
		t3d,/yzexch
		!y.range=yrange
		!x.s=xscale
		!y.s=yscale
		!z.s=zscale
		endif
;	All in 1
	endif else if ps_options.a then begin
		if plot_no eq 1 then begin
		if num_xlabel eq 0 then ymrgn=3 else ymrgn=4
		pos=[ $
		 num_ytitle*xpix_per_ylabel+9*xpix_per_char, $
		 num_xtitle*ypix_per_char+ymrgn*ypix_per_char, $
		 !d.x_vsize-3*xpix_per_char, $
		 !d.y_vsize-(num_title+1)*ypix_per_char]
		!x.title=xtitle
		!y.title=ytitle
		if not overplot then $
		plot,xdata,ydata,/nodata,position=pos,/device,title='', $
			xtype=!x.type,ytype=!y.type
		if not ps_options.p and not overplot then $
		xyouts,1.0-float(ypix_per_char)/!d.x_size,!y.window(1), $
			GenConfig.Date+' '+GenConfig.configid,/normal, $
			orientation=-90,size=!p.charsize/mf
		x1=!x.crange(0) & x2=!x.crange(1)
           	xrange=x2-x1 & x1=x1-0.1*xrange & x2=x2+0.1*xrange
;          	oplot,[x1,x2],[0.0,0.0],psym=0,linestyle=0,thick=1
		endif
;	Standard
	endif else begin
		if max([abs(!y.range(1)),abs(!y.range(0))]) gt 10000 or $
		   max([abs(!y.range(1)),abs(!y.range(0))]) lt 1./10 then $
			xmrgn=14 else xmrgn=10
		if abs((!y.range(1)+!y.range(0))/(!y.range(1)-!y.range(0))) gt 10000 then $
			xmrgn=17
		if !d.name eq 'PS' then xmrgn=xmrgn-6
		!x.title=xtitle
		!y.title=ytitle
		if polar then begin
		!x.range=0
		!y.range=0
		plot,fltarr(360)+90,findgen(360)/180*!pi,xmargin=[xmrgn,3],title='', $
			ymargin=[4,1+(num_title/mf)], $
			xtype=!x.type,ytype=!y.type,ytick_get=yv,/polar,psym=0
		endif else begin
		if ps_options.p and num_plot gt 1 then $
		!p.title='('+alphabet(plot_no)+')' else !p.title=''
		plot,xdata,ydata,/nodata,xmargin=[xmrgn,3], $
			ymargin=[3,1+(num_title/mf)], $
			xtype=!x.type,ytype=!y.type,ytick_get=yv
		endelse
		if plot_no eq 1 and not ps_options.p then $
		xyouts,1.0-float(ypix_per_char)/!d.x_size,!y.window(1), $
			GenConfig.Date+' '+GenConfig.configid,/normal, $
			orientation=-90,size=!p.charsize/mf
		x1=!x.crange(0) & x2=!x.crange(1)
           	xrange=x2-x1 & x1=x1-0.1*xrange & x2=x2+0.1*xrange
           	oplot,[x1,x2],[0.0,0.0],psym=0,linestyle=0,thick=1
;		Some plot specific axis labeling
		if SystemId eq 'NPOI' then begin
			case yitem(0) of
			11: 	begin
				ben_rms=[0.35,0.30,0.25,0.20,0.15,0.10,0.05]*sqrt(2)
				ben_bno=['B0','B1','B2','B3','B4','B5','B6']
				for l=0,n_elements(ben_rms)-1 do $
				if ben_rms(l) gt !y.crange(0) and ben_rms(l) lt !y.crange(1) $
				then xyouts,!x.crange(1)-0.05*(!x.crange(1)-!x.crange(0)), $
					    ben_rms(l),ben_bno(l)
				end
			else:
			endcase
		endif
	endelse
;
	plotted=0
;
;	Fill in and plot image here
	if ps_options.g then begin
		if ps_options.f eq 0 then begin
			index=where(ydata_err ge 0,count)
			if count eq 0 then goto,NOIMAGE
			maxv=max(ydata(index))
			minv=min(ydata(index))
		endif else begin
			maxv=max(ydata)
			minv=min(ydata)
		endelse
;		Use color indices 32 to 255 (grey scales in ct0)
		if maxv-minv eq 0 then begin
			maxv=1.0
			minv=-1.0
		endif
		y_image(*,plot_no-1)=byte((ydata-minv)/(maxv-minv)*223+32)
		y_image_err(*,plot_no-1)=ydata_err
;
;		Display if image array complete
		if plot_no eq num_plot then begin
			if ps_options.f eq 0 then begin
				index=where(y_image_err lt 0,count)
				if count gt 0 then y_image(index)=0
			endif
			while xpix gt !xsize do begin
			y_image=y_image(where((indgen(xpix) mod 2) ne 0,count),*)
			xpix=count
			endwhile
				
			tv,rebin(y_image, $
				 xpix*rebin_xfactor, $
				 ypix*rebin_yfactor,/sample), $
					xmargin(0),ymargin(0)
		endif
		plotted=1
		goto,SKIPPLOT
	endif
;
;       Now we plot one star after the other....................................
;	if class eq 'scan' and ps_options.a then !p.psym=0
	dobgrms=1
	for is=0,n_elements(ds_stars)-1 do begin
;
		ic=where(startable.starid eq ds_stars(is)) & ic=ic(0)
		ds_star=startable(ic).name
		if strlen(strcompress(ds_star,/remove_all)) eq 0 $
				      then ds_star=startable(ic).var
		if strlen(strcompress(ds_star,/remove_all)) eq 0 $
				      then ds_star=ds_stars(is)
		if systemid eq 'NPOI' then ds_star=ds_stars(is)
;
;		We use different plot symbols for different stars
		if class eq 'amoeba' and ps_options.a then begin
			!p.psym=getsymbol(is+1)
		endif
;
;		Find out which scans/points correspond to current star
		stars=datascans(point).starid
		index=where(stars eq ds_stars(is),count)
		if count eq 0 then goto,NODATA
;
;		Set data
		x=xdata(index)
		y=ydata(index)
		z=zdata(index)
		x_err=xdata_err(index)
		y_err=ydata_err(index)
		e_index=index
;
;            	Edit the data
		if (ps_options.f eq 0) or (edit eq 1) then begin
			x_index=where(x_err ge 0,count)
			if count eq 0 then goto,NODATA
			xe=x(x_index)
			ye=y(x_index)
			ze=z(x_index)
			xe_err=x_err(x_index)
			ye_err=y_err(x_index)
			e_index=e_index(x_index)
			y_index=where(ye_err ge 0,count)
			if count eq 0 then goto,NODATA
			xe=xe(y_index)
			ye=ye(y_index)
			ze=ze(y_index)
			ye_err=ye_err(y_index)
			xe_err=xe_err(y_index)
			e_index=e_index(y_index)
		endif else begin
			xe=x
			ye=y
			ze=z
			xe_err=x_err
			ye_err=y_err
		endelse
;
;		Sort the data if neccessary; note that suffix e is reused!
		check=[1,2,19,20,22,23]
		if long(where(check eq xitem),0) eq -1 $
	 		and not ps_options.n then si=sort(xe) $
			else si=lindgen(count)
		x=xe(si)
		y=ye(si)
		z=ze(si)
		x_err=xe_err(si)
		y_err=ye_err(si)
		e_index=e_index(si)
;
;		Set the default plot color
		!p.color=tci(frame_color)
;
;		Add star to star title
		if n_elements(ds_stars) gt 1 $
			and (not ps_options.a or class eq 'seeing') then $
		!p.color=getcolor(ic)
		title_index=is/max_star_per_title
		if not ps_options.p then $
		xyouts,title_xoffset(title_index) $
			+(is mod max_star_per_title)*xpix_per_star, $
;		       title_yoffset(title_index),ds_stars(is), $
		       title_yoffset(title_index),ds_star, $
		       /device,size=!p.charsize/mf
;		Plot a line with symbols used under the star labels
		xl=!x.crange(0)
		xh=!x.crange(1)
		xr=xh-xl
		yl=!y.crange(0)
		yh=!y.crange(1)
		yr=yh-yl
;		oplot,[xl,xl]+xr/20+is*(float(xpix_per_star)/!d.x_size)*xr,[yh,yh]-0.03*yr, $
;			color=tci(15)
;
;		Add data selection title for All in 1 option
		if ps_options.a then $
		!p.color=getcolor(plot_no-1)
		if class eq 'seeing' then !p.color=tci(frame_color)
		if ps_options.a and (not ps_options.d) then begin
			title_index=(plot_no-1)/max_ylabel_per_title
			if num_ylabel eq 0 then title_index=0
			xyouts,ytitle_xoffset(title_index), $
			       ytitle_yoffset(title_index)+ $
			        ((plot_no-1) mod max_ylabel_per_title) $
					*ypix_per_char, $
			       ylabel,/device,size=!p.charsize
			title_index=(plot_no-1)/max_xlabel_per_title
			if num_xlabel eq 0 then title_index=0
			xyouts,xtitle_xoffset(title_index)+ $
				((plot_no-1) mod max_xlabel_per_title) $
					*xpix_per_xlabel, $
			       xtitle_yoffset(title_index), $
			       xlabel,/device,size=!p.charsize
		endif
		if class eq 'seeing' then !p.color=getcolor(ic)
;	
;		Plot the data
		if n_elements(x) gt 100 then !p.psym=3
		if ps_options.e then psym=3 else psym=!p.psym
		if ps_options.l then psym=-abs(psym) else psym=abs(psym)
		if ps_options.d then plots,x,y,z,psym=psym,/t3d else begin
;		Custom option for scaling visibilities
		if ps_options.m and ps_options.v then begin
			mitem=0
			if yitem eq 25 then mitem=57
			if yitem eq 29 then mitem=58
			if mitem ne 0 then begin
			ymodel=set_plotdata(mitem,yinbeam,youtbeam, $
					    ytriple,ychan,ybase,ypoint)
			ymodel=ymodel(e_index)
			if n_elements(ymodel) eq 1 then ymodel=fltarr(1)+ymodel
			nf=140
			f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
			for i=0,nf do r(i)=total(((y*f(i)-ymodel)/y_err)^2)
			i=where(r eq min(r)) & i=i(0)
			y=y*f(i)
			endif
		endif else if ps_options.v then begin
;		Nothing here yet
		endif
		if polar then oplot,x,y,psym=psym,/polar $
			 else oplot,x,y,psym=psym
;		oploterr modifies the input variable!
		y_err_bck=y_err
		if ps_options.e then begin
			if n_elements(x) gt 1 $
				then oploterror,x,y,x*0,y_err $
				else oploterror,[x(0),x(0)],[y(0),y(0)], $
						[0,0],[y_err(0),y_err(0)]
		endif
		y_err=y_err_bck
;
;		Display the rms for back ground plots
		if class eq 'bg' and not ps_options.a and dobgrms then begin
			if ps_options.f then index=indgen(n_elements(ydata)) $
					else index=where(ydata_err gt 0)
			if n_elements(index) gt 1 then begin
			rms='RMS(%) = '+string(stddev(ydata(index)) $
					      /avg(ydata(index))*100, $
						format='(f5.1)')
			xyouts,!x.crange(0)+0.04*(!x.crange(1)-!x.crange(0)), $
			       !y.crange(0)+0.05*(!y.crange(1)-!y.crange(0)), $
			       rms,color=tci(1),size=!p.charsize/mf
			dobgrms=0
			endif
		endif
;
;		Plot a model if requested
		if ps_options.m then begin
		if n_elements(star_model) eq 0 then begin
;			print,'***Error(PLOTDATA): no model!' 
;		endif else if gen_model(0).starid ne ds_stars(is) then begin
;			print,'***Error(PLOTDATA): no model for this star!' 
		endif else begin
			mitem=0
			if yitem eq 25 then mitem=57
			if yitem eq 29 then mitem=58
			if yitem eq 33 then mitem=59
			if yitem eq 91 then mitem=92
			if mitem ne 0 then begin
			ymodel=set_plotdata(mitem,yinbeam,youtbeam, $
					    ytriple,ychan,ybase,ypoint)
			ymodel=ymodel(e_index)
			if n_elements(ymodel) eq 1 then ymodel=fltarr(1)+ymodel
			oplot,x,ymodel,psym=-3,color=tci(frame_color),thick=2
			if keyword_set(ms1) then begin
				ds=scans
				scans=ms1
				ymodel=set_plotdata(mitem,yinbeam,youtbeam, $
						    ytriple,ychan,ybase,ypoint)
				ymodel=ymodel(e_index)
				scans=ds
				if n_elements(ymodel) eq 1 then ymodel=fltarr(1)+ymodel
				oplot,x,ymodel,psym=-3,linestyle=2,color=tci(frame_color)
			endif
			if keyword_set(ms2) then begin
				ds=scans
				scans=ms2
				ymodel=set_plotdata(mitem,yinbeam,youtbeam, $
						    ytriple,ychan,ybase,ypoint)
				ymodel=ymodel(e_index)
				scans=ds
				if n_elements(ymodel) eq 1 then ymodel=fltarr(1)+ymodel
				oplot,x,ymodel,psym=-3,linestyle=1,color=tci(frame_color)
			endif
			endif else begin
				print,'***Error: Must plot calibrated data with model!'
				return
			endelse
		endelse
		endif
;		
;		Plot flagged data (ps_options.f=1)
		if ps_options.f then begin
		!p.color=tci(2)
		index=where(x_err lt 0 or y_err lt 0,count)
		if count ne 0 then begin
			oplot,x(index),y(index),psym=psym
			if ps_options.e then begin
				if n_elements(index) gt 1 then begin	
				      oploterror,x(index), $
				      y(index),x(index)*0,abs(y_err(index))
				endif else begin
				      x1=x(index(0))
				      y1=y(index(0))
				      e1=abs(y_err(index(0)))
				      oploterror,[x1,x1],[y1,y1],[0,0],[e1,e1]
				endelse
			endif
		endif
		endif
		endelse	; else clause for non-3D plotting
		plotted=1
	NODATA:
	endfor	; End of star plotting loop.....................................
;
	NOIMAGE:
	if not plotted then plot_no=plot_no-1
	SKIPPLOT:
endfor
endfor
endfor
endfor
endfor
endfor
;
ENDFOR	; End of all plotting
;
; Display alert if no data was found to be valid
if plot_no eq 0 and edit eq 0 then begin
	print,'No data.'
;	window,/free
;	xyouts,0.4,0.5,'No data',/normal,charsize=2
endif
;
; Set PlotDataInfo variables
if do_info and plot_no gt 0 then begin
	plotscans={x:p_xdata,y:p_ydata,xe:p_xdata_err,ye:p_ydata_err, $
		    item:p_item,star:p_star,time:p_time, $
		    ib:p_inbeam,ob:p_outbeam,tr:p_triple, $
		    ch:p_chan,bl:p_base,pt:p_point}
	plotdata_x=p_xdata
	plotdata_y=p_ydata
	plotdata_y_bck=plotscans.y
endif
;
; Apply the flags to the data
if edit ne 0 then begin
	if id gt id0 then begin
		case class of
			'point':flagpointdata,pointflagtable(id0:id-1),flag=1
			'metro':flaginchdata,inchflagtable(id0:id-1),flag=1
			'delay':flagpointdata,pointflagtable(id0:id-1),flag=1
			'scan' :flagscandata,scanflagtable(id0:id-1),flag=1
			'bg'   :flagbgdata,bgflagtable(id0:id-1),flag=1
		endcase
	endif else print,'No data flagged.'
endif
;
; Reset plot ranges, colors, ...
if xrange_auto then !x.range=0
if yrange_auto then !y.range=0
if zrange_auto then !z.range=0
!p.color=tci(1)
!x.title=''
!y.title=''
;
end
;-------------------------------------------------------------------------------
pro plotncal,skip_plot=skip_plot
;
; This procedure now comprises three different calibrations of channels 
; (Response), baselines (Bias), and stations (APDFlux). It is intended as
; a container for system calibrations, and therefore does not offer axis
; plot selection. It also stores calibration information in the configuration
; variable GenConfig.
;
; This procedure is aware of different scan configurations related to switching
; in and out stations in the NPOI 6-way observing mode. It initializes the
; corresponding information in genconfig.config, using function scanconfig.
;
common StarBase,startable,notes
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common PlotDataInfo,plotscans,plotdata_x,plotdata_y,plotdata_y_bck
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common PlacementInfo,xpix_per_char,ypix_per_char, $
		     title_yoffset,title_xoffset, $
		     xtitle_xoffset,xtitle_yoffset, $
		     ytitle_xoffset,ytitle_yoffset, $
		     num_title,max_star_per_title,xpix_per_star, $
		     num_xtitle,max_xlabel_per_title,xpix_per_xlabel, $
		     num_ytitle,max_ylabel_per_title,xpix_per_ylabel
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; We added skip_plot so that we can include ncal in a pipeline (in prep.)
if n_elements(skip_plot) eq 0 then skip_plot=0 else skip_plot=1
;
; Check availability of data
if checkdata([1,9]) ne 0 then return
;
; Check data selection
if printselerror(checkplotsel('y')) ne 0 then return
;
; Extract selection of y stream
set_streamsel,'y',item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,ds_pt
;
; Extract scan configuration
sconf=''
if strpos(type,'V2Bias')      ne -1 then sconf=strmid(type, 6,strlen(type) -6)
if strpos(type,'TABias')      ne -1 then sconf=strmid(type, 6,strlen(type) -6)
if strpos(type,'APDFlux')     ne -1 then sconf=strmid(type, 7,strlen(type) -7)
if strpos(type,'Response')    ne -1 then sconf=strmid(type, 8,strlen(type) -8)
if strpos(type,'TrackJitter') ne -1 then sconf=strmid(type,11,strlen(type)-11)
if strpos(type,'TrackOffset') ne -1 then sconf=strmid(type,11,strlen(type)-11)
if strlen(sconf) eq 0 then sconf=scanconfig()
sconf=sconf(0)
genconfig.config=''
genconfig.config(0:n_elements(scanconfig(/starlist))-1)=scanconfig(/starlist)
configs=strarr(n_elements(scanconfig(/starlist)))
for i=0,n_elements(scanconfig(/starlist))-1 do begin
	words=nameparse(genconfig.config(i))
	configs(i)=words(0)
endfor
cf=where(sconf eq configs)
if cf(0) lt 0 and strpos(type,'APDFlux') eq -1 then begin
	print,'Warning: fit results will not be stored!'
	print,'Please select one of these in Points drop-down menu: ' $
		+strjoin(scanconfig(/coh),' ')+'!'
endif
genconfig_checksum=''
for i=0,n_tags(genconfig)-1 do genconfig_checksum=genconfig_checksum $
				 +strjoin(string(genconfig.(i)),/single)
;
; For 'Response', normalize to 2 stations per detector
if strpos(type,'Response') ge 0 then begin
	f_response=1.0
	i=scanconfig(sconf)
	if i(0) ne 0 then begin
	n=0
	sids=genconfig.stationid $
		(where(scantable(i(0)).station(0:genconfig.numsid-1) ne 0))
	for i=0,n_elements(sids)-1 do $
	if strpos(strjoin(genconfig.baselineid(*,ds_ob)),sids(i)) ge 0 then n=n+1
	f_response=2.0/n
	endif
endif
;
; Initialize loop ranges
num_c=n_elements(ds_ch)
num_b=n_elements(ds_bl) 
if strpos(type,'TABias') ge 0 then num_b=1
if strpos(type,'APDFlux') ge 0 then num_b=1
if strpos(type,'Response') ge 0 then num_b=1
;
; Initialize index counter
numarr=intarr(2,num_b*num_c)
num_plot=0
;
; For option All in 1 we have to go through the data twice to find the
; number of plots in a window (num_plot) and the total range of data values.
; However, we now go through it twice anyway to see how many plots we have.
if ps_options.a then num_run=2 else num_run=2
;
; Determine whether automatic scaling is requested
if total(abs(!x.range)) eq 0 then xrange_auto=1 else xrange_auto=0
if total(abs(!y.range)) eq 0 then yrange_auto=1 else yrange_auto=0
if total(abs(!z.range)) eq 0 then zrange_auto=1 else zrange_auto=0
;
; For squared visibility bias plot, choose logarithmic scale
if strpos(type,'V2Bias') ge 0  and not !v2bias then begin
	!x.type=1
	!y.type=1
endif else if strpos(type,'TABias') ge 0  and not !tabias then begin
	!x.type=1
	!y.type=1
endif else begin
	!x.type=0
	!y.type=0
endelse
xrange_bck=!x.range
yrange_bck=!y.range
if !x.type eq 1 and not xrange_auto then !x.range=10^!x.range
if !y.type eq 1 and not yrange_auto then !y.range=10^!y.range
;
RAD=180/pi_circle
;
; Loops start here!
FOR run_no=1,num_run DO BEGIN
plot_no=0	; Current (sub)plot number; first plot will be no. 1
;
IF run_no eq 2 AND num_plot gt 0 THEN BEGIN
;
num_v=n_elements(unique(numarr(0,0:num_plot-1)))
num_h=n_elements(unique(numarr(1,0:num_plot-1)))
if num_v eq 1 or num_h eq 1 then begin
	num_t=num_h*num_v
	num_v=num_t
	num_h=1
	while num_v/num_h ge 2 do begin
		num_h=num_h+1
		num_v=num_t/num_h
 		if num_v*num_h lt num_t then num_v=num_v+1
	endwhile
endif
; !p.multi=[#plots remaining,#plot columns,#plot rows,#plots stacked in Z,
;           0:left to right and top to bottom;1:top to bottom and left to right]
if ps_options.a or ps_options.d then !p.multi=0 $
				else !p.multi=[0,num_h,num_v,0,1] 
;
; For multiple plots, IDL reduces the charsize by a factor of 2
!p.charsize=1.0
if !p.multi(1)*!p.multi(2) le 1 then !p.charsize=1.5
if !p.multi(1) gt 2 or !p.multi(2) gt 2 then begin
	!p.charsize=2.0
	mf=2.0 
endif else mf=1.0
if !p.multi(1) gt 4 or !p.multi(2) gt 4 then !p.charsize=1.0
;
; Open a plot window for plotting
if not ps_options.d and (!d.name ne 'PS') then begin
	window,/free,title=Date+', Class:'+class,xsize=!xsize,ysize=!ysize, $
		xpos=!dxsize-!xsize,ypos=!dysize-!ysize
endif else if not ps_options.d and (!d.name eq 'PS') then begin
	device,color=ps_options.c
	tek_color
	if ps_options.p then begin
		if mf eq 1 then !p.charsize=1.5 else !p.charsize=2.0
	endif
endif else begin
	print,'***Error(PLOTNCAL): invalid selection and/or device name!'
	return
endelse
if !d.name eq 'PS' and ps_options.c then frame_color=0 else frame_color=1
;
; Calculate star title placement variables
titleplacement,'t',ds_stars
;
ENDIF
;
for ich=0,num_c-1 do begin
for ibl=0,num_b-1 do begin
;
; 	Initialize data arrays
	x=fltarr(n_elements(scans))
	y=fltarr(n_elements(scans))
	e=fltarr(n_elements(scans))
	z=fltarr(n_elements(scans))
	t=fltarr(n_elements(scans))
	s=strarr(n_elements(scans))
;
	ch=ds_ch(ich)
	bl=ds_bl(ibl)
;
; 	Set the data
	for st=0,n_elements(ds_stars)-1 do begin
		index=where(scans(ds_pt).starid eq ds_stars(st),count)
		if count ne 0 then begin
		if strpos(type,'Response') ge 0 then begin
			j=where(startable.starid eq ds_stars(st),count)
			if count ne 0 then begin
			s(st)=ds_stars(st)
			x(st)=startable(j).bv
			tt=scans(ds_pt(index)).time
			if x(st) lt 100 then begin
			pr=scans(ds_pt(index)).photonrate(ds_ob,ch) $
			  /cos(scans(ds_pt(index)).za/RAD)
			pr=pr*f_response
			pre=scans(ds_pt(index)).photonrateerr(ds_ob,ch) $
			  /cos(scans(ds_pt(index)).za/RAD)
			pre=pre*f_response
			index=where(pr gt 0 and pre gt 0,count)
			if count gt 1 then begin
				sigma=stdev(pr(index),mean)
			endif else if count eq 1 then begin
				sigma=pre(index)
				mean=pr(index)
			endif
			if count gt 0 then begin
				y(st)=2.5*alog10(mean)+startable(j).mv
				e(st)=2.5*sigma/mean/alog(10)
				t(st)=avg(tt(index))
			endif
			endif
			endif
		endif else if strpos(type,'TrackJitter') ge 0 then begin
			jndex=where(scantable(ds_pt(index)).code eq 1,count)
			if count gt 0 then begin
			p=scans(ds_pt(index(jndex))).photonrate(ds_ob,ch)
			pe=scans(ds_pt(index(jndex))).photonrateerr(ds_ob,ch)
	   		i=where(GenConfig.StationId eq $
	    		 strmid(GenConfig.BaselineId(bl,ds_ob),0,3))
	   		j=where(GenConfig.StationId eq $
			 strmid(GenConfig.BaselineId(bl,ds_ob),4,3))
			tj=(sqrt(scans(ds_pt(index(jndex))).NATJitter(i)^2 $
				+scans(ds_pt(index(jndex))).NATJitter(j)^2))
			sign=fltarr(n_elements(ds_pt(index(jndex))))+1
			endex=where(scans(ds_pt(index(jndex))).NATJitterErr(i) lt 0 $
				 or scans(ds_pt(index(jndex))).NATJitterErr(j) lt 0,count)
			if count gt 0 then sign(endex)=-1
			tje=(sqrt(scans(ds_pt(index(jndex))).NATJitterErr(i)^2 $ 
				 +scans(ds_pt(index(jndex))).NATJitterErr(j)^2))*sign
			kndex=where(pe gt 0 and tje gt 0,count)
			if count gt 0 then begin
				s(ds_pt(index(jndex(kndex))))=ds_stars(st)
				x(ds_pt(index(jndex(kndex))))=p(kndex)
				y(ds_pt(index(jndex(kndex))))=tj(kndex)
				e(ds_pt(index(jndex(kndex))))=tje(kndex)
			endif
			endif
		endif else if strpos(type,'TrackOffset') ge 0 then begin
			jndex=where(scantable(ds_pt(index)).code eq 1,count)
			if count gt 0 then begin
			v=scans(ds_pt(index(jndex))).vissqec(ds_ob,ch,bl)
			ve=scans(ds_pt(index(jndex))).vissqecerr(ds_ob,ch,bl)
			o=scans(ds_pt(index(jndex))).trackoffset(ds_ob,bl)
			oe=o*0+1
			kndex=where(ve gt 0 and oe gt 0,count)
			if count gt 0 then begin
				s(ds_pt(index(jndex(kndex))))=ds_stars(st)
				x(ds_pt(index(jndex(kndex))))=o(kndex)
				y(ds_pt(index(jndex(kndex))))=v(kndex)
				e(ds_pt(index(jndex(kndex))))=oe(kndex)
			endif
			endif
		endif else if strpos(type,'V2Bias') ge 0 then begin
;			Photonrate must include back ground
			jndex=where(scantable(ds_pt(index)).code eq 0,count)
			if count gt 0 then begin
			v=scans(ds_pt(index(jndex))).vissq(ds_ob,ch,bl)
			ve=scans(ds_pt(index(jndex))).vissqerr(ds_ob,ch,bl)
			p=scans(ds_pt(index(jndex))).photonrate(ds_ob,ch) $
			 +scans(ds_pt(index(jndex))).backgndrate(ds_ob,ch)
			pe=scans(ds_pt(index(jndex))).photonrateerr(ds_ob,ch)
			if not !v2bias then $
			kndex=where(v gt 0 and ve gt 0 $
				and p gt 0 and pe gt 0,count) else $
			kndex=where(ve gt 0 and pe gt 0,count)
			if count gt 1 then sigma=stdev(v(kndex)) else $
			if count eq 1 then sigma=ve(kndex)
			if count gt 0 then begin
				s(ds_pt(index(jndex(kndex))))=ds_stars(st)
				x(ds_pt(index(jndex(kndex))))=p(kndex)
				y(ds_pt(index(jndex(kndex))))=v(kndex)
				e(ds_pt(index(jndex(kndex))))=ve(kndex)
			endif
			endif
		endif else if strpos(type,'TABias') ge 0 then begin
;			Photonrate must include back ground
			jndex=where(scantable(ds_pt(index)).code eq 0,count)
			if count gt 0 then begin
			v=scans(ds_pt(index(jndex))).tripleamp(ds_tr,ch)
			ve=scans(ds_pt(index(jndex))).tripleamperr(ds_tr,ch)
			tb=commonbeam(genconfig.triplebeam(*,ds_tr))
			p=scans(ds_pt(index(jndex))).photonrate(tb, $
					genconfig.triplechan(ch,0,ds_tr)) $
			 +scans(ds_pt(index(jndex))).backgndrate(tb, $
					genconfig.triplechan(ch,0,ds_tr))
			pe=scans(ds_pt(index(jndex))).photonrateerr(tb, $
					    genconfig.triplechan(ch,0,ds_tr))
			if not !tabias then $
			kndex=where(v gt 0 and ve gt 0 $
				and p gt 0 and pe gt 0,count) else $
			kndex=where(ve gt 0 and pe gt 0,count)
			if count gt 1 then sigma=stdev(v(kndex)) else $
			if count eq 1 then sigma=ve(kndex)
			if count gt 0 then begin
				s(ds_pt(index(jndex(kndex))))=ds_stars(st)
				x(ds_pt(index(jndex(kndex))))=p(kndex)
				y(ds_pt(index(jndex(kndex))))=v(kndex)
				e(ds_pt(index(jndex(kndex))))=ve(kndex)
			endif
			endif
		endif
		endif
	endfor
	if strpos(type,'APDFlux') ge 0 then begin
		photom,ds_pt+1,y=rrates,e=erates,nonl=ps_options.v
		y=rrates(ds_ob,ch,*)
		e=erates(ds_ob,ch,*)
		x=scans(ds_pt).iscan
		s=scans(ds_pt).starid
	endif
;
;	Concatenate arrays
	index=where(strlen(s) gt 0,count)
	if count eq 0 then goto,SKIPPLOT
	x=x(index)
	y=y(index)
	e=e(index)
	s=s(index)
;
;	If there is no valid data, skip the rest
	index=where(e gt 0,count)
	if count eq 0 then goto,SKIPPLOT
;
;       We have valid data, so increment the plot number counter
	plot_no=plot_no+1
	numarr(*,plot_no-1)=[ibl,ich]
;
;       First, we have to find out about the ranges
	xmin=min(x(index))
	ymin=min(y(index))
	zmin=min(z(index))
	xmax=max(x(index))
	ymax=max(y(index))
	zmax=max(z(index))
;
;       Here we got the range information and can skip the rest if plotting
;       will occur in the second run through the data (All in 1 option).
	if run_no eq 1 then begin
		num_plot=num_plot+1
		if plot_no eq 1 then begin
			axmax=xmax
			axmin=xmin
			aymax=ymax
			aymin=ymin
			azmax=zmax
			azmin=zmin
		endif else begin
			axmax=max([axmax,xmax])
			axmin=min([axmin,xmin])
			aymax=max([aymax,ymax])
			aymin=min([aymin,ymin])
			azmax=max([azmax,zmax])
			azmin=min([azmin,zmin])
		endelse
		goto,SKIPPLOT
	endif
;
;  	Set the ranges for plotting
        if ps_options.a then begin
                if n_elements(axmax) eq 0 then begin
			window,/free
                        xyouts,0.4,0.5,'No data',/normal,charsize=2
                        return
                endif
		xmax=axmax
		xmin=axmin
		ymax=aymax
		ymin=aymin
		zmax=azmax
		zmin=azmin
	endif
	if xrange_auto then !x.range=[xmin,xmax] else !x.style=1
	if strpos(type,'V2Bias'     ) ge 0 then $
	if yrange_auto then !y.range=[ymin,ymax] else !y.style=1
	if strpos(type,'TABias'     ) ge 0 then $
	if yrange_auto then !y.range=[ymin,ymax] else !y.style=1
	if strpos(type,'APDFlux'    ) ge 0 then $
	if yrange_auto then !y.range=[ymin,ymax] else !y.style=1
	if strpos(type,'Response'   ) ge 0 then $
	if yrange_auto then !y.range=[ymax,ymin] else !y.style=1
	if strpos(type,'TrackJitter') ge 0 then $
	if yrange_auto then !y.range=[ymin,ymax] else !y.style=1
	if strpos(type,'TrackOffset') ge 0 then $
	if yrange_auto then !y.range=[ymin,ymax] else !y.style=1
	if zrange_auto then !z.range=[zmin,zmax]

	!p.color=tci(frame_color)
	psym=1
	num_xlabel=0 
	num_ylabel=num_plot
	num_zlabel=num_plot
	ax_items=set_axitems(item_ids)
	xtitle=ax_items(0)
	ytitle=ax_items(1)
	ztitle=ax_items(2)
	xlabel=''
	ylabel='OB '+strcompress(string(ds_ob+1))+ $
	      ' CH '+strcompress(string(ch+1))
	if strpos(type,'V2Bias') ge 0 then $
		ylabel=ylabel+' BL '+strcompress(string(bl+1))
	if strpos(type,'TABias') ge 0 then $
		ylabel='TR '+strcompress(string(ds_tr+1)) $
		     +' CH '+strcompress(string(ch+1))
	if strpos(type,'TrackJitter') ge 0 then begin
		xlabel='OB '+strcompress(string(ds_ob+1))+ $
		      ' CH '+strcompress(string(ch+1))
		ylabel='OB '+strcompress(string(ds_ob+1))+ $
		      ' BL '+strcompress(string(bl+1))
		num_xlabel=num_plot
	endif
	if strpos(type,'TrackOffset') ge 0 then begin
		xlabel='OB '+strcompress(string(ds_ob+1))+ $
		      ' BL '+strcompress(string(bl+1))
		ylabel=ylabel+' BL '+strcompress(string(bl+1))
		num_xlabel=num_plot
	endif
	zlabel='CH '+strcompress(string(ch+1))

	if not ps_options.a then begin
		xcomma=', '
		ycomma=', '
		zcomma=', '
		if !p.multi(2) gt 2 then begin
			xtitle=ytitle+' vs '+xtitle
			ytitle=''
			ycomma=''
		endif
		if num_xlabel gt 0 then xtitle=xtitle+xcomma+xlabel
		if num_ylabel gt 0 then ytitle=ytitle+ycomma+ylabel
		if num_zlabel gt 0 then ztitle=ztitle+zcomma+zlabel
	endif else if plot_no eq 1 then begin
		titleplacement,'x',ds_stars,xlabel,num_xlabel
		titleplacement,'y',ds_stars,ylabel,num_ylabel
	endif
;
;	Plot frame
;	3D
	if ps_options.d then begin
		if plot_no eq 1 then begin
		pos=[0.1,0.1,0.9,0.9,0.1,0.9]
		plot,x(where(e gt 0)),y(where(e gt 0)),/nodata,/t3d,position=pos, $
			xtitle=xtitle,ytitle=ytitle
		xscale=!x.s
		yscale=!y.s
		t3d,/xzexch
		xrange=!x.range
		!x.range=!z.range
		plot,z,y,/nodata,/t3d,/noerase,pos=pos, $
				xtitle=ztitle
		zscale=!x.s
		!x.range=xrange
		t3d,/xzexch
		t3d,/yzexch
		yrange=!y.range
		!y.range=!z.range
		plot,x,z,/nodata,/t3d,/noerase,pos=pos
		t3d,/yzexch
		!y.range=yrange
		!x.s=xscale
		!y.s=yscale
		!z.s=zscale
		endif
;	All in 1
	endif else if ps_options.a then begin
		if plot_no eq 1 then begin
		if num_xlabel eq 0 then ymrgn=3 else ymrgn=4
		if max([abs(ymax),abs(ymin)]) gt 10000 then $ 
			xmrgn=12 else xmrgn=9
		pos=[ $
		 num_ytitle*xpix_per_ylabel+7*xpix_per_char, $
		 num_xtitle*ypix_per_char+ymrgn*ypix_per_char, $
		 !xsize-3*xpix_per_char, $
		 !ysize-(num_title+1)*ypix_per_char]
		if ps_options.p then pos(0)=9*xpix_per_char
		if !d.name eq 'PS' then begin
		plot,x(where(e gt 0)),y(where(e gt 0)),/nodata,/device, $
			xtitle=xtitle,ytitle=ytitle, $
			xtype=!x.type,ytype=!y.type, $
			xmargin=[xmrgn,3],ymargin=[4,1+(num_title/mf)]
		endif else begin
		plot,x(where(e gt 0)),y(where(e gt 0)), $
			/nodata,position=pos,/device, $
			xtitle=xtitle,ytitle=ytitle,xtype=!x.type,ytype=!y.type
		endelse
		if not ps_options.p then $
		xyouts,1.0-float(ypix_per_char)/!d.x_size,!y.window(1), $
			Date,/normal,orientation=-90,size=!p.charsize/mf
		endif
		!p.psym=getsymbol(ich+1)
;	Standard
	endif else begin
		if max([abs(ymax),abs(ymin)]) gt 10000 then $ 
			xmrgn=12 else xmrgn=9
                !x.title=xtitle
                !y.title=ytitle
		plot,x(where(e gt 0)),y(where(e gt 0)), $
			/nodata,xmargin=[xmrgn,3], $
			ymargin=[4,1+(num_title/mf)], $
			xtype=!x.type,ytype=!y.type,ystyle=!y.style
		if plot_no eq 1 and not ps_options.p then $
		xyouts,1.0-float(ypix_per_char)/!d.x_size,!y.window(1), $
			Date,/normal,orientation=-90,size=!p.charsize/mf
	endelse
;
;       Now we plot one star after the other
	for st=0,n_elements(ds_stars)-1 do begin
;
;		Add star to star title
		if n_elements(ds_stars) gt 1 and not ps_options.a then $
			!p.color=getcolor(st) else !p.color=tci(frame_color)
		title_index=st/max_star_per_title
		if not ps_options.p then $
		xyouts,title_xoffset(title_index) $
			+(st mod max_star_per_title)*xpix_per_star, $
		       title_yoffset(title_index),ds_stars(st), $
		       /device,size=!p.charsize/mf
;
;		Add data selection title for All in 1 option
		if ps_options.a then !p.color=getcolor(plot_no-1)
		if ps_options.a and (not ps_options.d) $
				and (not ps_options.p) $
				and (!d.name ne 'PS') $
		then begin
			title_index=(plot_no-1)/max_ylabel_per_title
			if num_ylabel eq 0 then title_index=0
			xyouts,ytitle_xoffset(title_index), $
			       ytitle_yoffset(title_index)+ $
			        ((plot_no-1) mod max_ylabel_per_title) $
					*ypix_per_char, $
			       ylabel,/device,size=!p.charsize
			title_index=(plot_no-1)/max_xlabel_per_title
			if num_xlabel eq 0 then title_index=0
			xyouts,xtitle_xoffset(title_index)+ $
				((plot_no-1) mod max_xlabel_per_title) $
					*xpix_per_xlabel, $
			       xtitle_yoffset(title_index), $
			       xlabel,/device,size=!p.charsize
		endif
;	
;		Plot the data
;		if ps_options.e then psym=3 else psym=abs(!p.psym)
		if strpos(type,'APDFlux') ge 0 then begin
			index=where(s eq ds_stars(st) and e gt 0,count)
			if ps_options.l then psym=-psym
			if count gt 0 then oplot,x(index),y(index),psym=psym
		endif else if strpos(type,'V2Bias') ge 0 then begin
			index=where(s eq ds_stars(st) and e gt 0,count)
			if count gt 0 then begin
			xs=x(index)
			ys=y(index)
			es=e(index)
			zs=xs*0+float(ch)
			if ps_options.d then plots,xs,ys,zs,/t3d,psym=psym $
			else begin
				oplot,xs,ys,psym=psym
;				IDL refuses to plot a single error bar!
				if ps_options.e then begin
				index=where(es ge ys,count)
;				If err. bar ext. to neg., clip to plot
				if count gt 0 then es(index)=ys(index)-1e-5
				oploterror,[xs,xs],[ys,ys],[xs,xs]*0,[es,es]
				endif
			endelse
			endif
		endif else if strpos(type,'TrackJitter') ge 0 then begin
			index=where(s eq ds_stars(st) and e gt 0,count)
			if count gt 0 then begin
			xs=x(index)
			ys=y(index)
			es=e(index)
			zs=xs*0+float(ch)
			if ps_options.d then plots,xs,ys,zs,/t3d,psym=psym $
			else begin
				if ps_options.l then psym=-abs(psym)
				oplot,xs,ys,psym=psym
;				IDL refuses to plot a single error bar!
				if ps_options.e then begin
				index=where(es ge ys,count)
;				If err. bar ext. to neg., clip to plot
				if count gt 0 then es(index)=ys(index)-1e-5
				oploterror,[xs,xs],[ys,ys],[xs,xs]*0,[es,es]
				endif
			endelse
			endif
		endif else if strpos(type,'TrackOffset') ge 0 then begin
			index=where(s eq ds_stars(st) and e gt 0,count)
			if count gt 0 then begin
			xs=x(index)
			ys=y(index)
			es=e(index)
			zs=xs*0+float(ch)
			if ps_options.d then plots,xs,ys,zs,/t3d,psym=psym $
			else begin
				if ps_options.l then psym=-abs(psym)
				si=sort(xs)
				oplot,xs(si),ys(si),psym=psym
;				IDL refuses to plot a single error bar!
				if ps_options.e then begin
				index=where(es ge ys,count)
;				If err. bar ext. to neg., clip to plot
				if count gt 0 then es(index)=ys(index)-1e-5
				oploterror,[xs,xs],[ys,ys],[xs,xs]*0,[es,es]
				endif
			endelse
			endif
		endif else if strpos(type,'TABias') ge 0 then begin
			index=where(s eq ds_stars(st) and e gt 0,count)
			if count gt 0 then begin
			xs=x(index)
			ys=y(index)
			es=e(index)
			zs=xs*0+float(ch)
			if ps_options.d then plots,xs,ys,zs,/t3d,psym=psym $
			else begin
				oplot,xs,ys,psym=psym
;				IDL refuses to plot a single error bar!
				if ps_options.e then begin
				index=where(es ge ys,count)
;				If err. bar ext. to neg., clip to plot
				if count gt 0 then es(index)=ys(index)-1e-5
				oploterror,[xs,xs],[ys,ys],[xs,xs]*0,[es,es]
				endif
			endelse
			endif
		endif else begin
		z(st)=float(ch)
		if e(st) gt 0 then begin
		if ps_options.d then plots,[x(st)],[y(st)],[z],/t3d,psym=psym $
		else begin
			oplot,[x(st)],[y(st)],psym=psym
;			IDL refuses to plot a single error bar!
			if ps_options.e then $
				oploterror,[x(st),x(st)],[y(st),y(st)], $
					[x(st),x(st)]*0,[e(st),e(st)]
		endelse
		endif
		endelse
	endfor
;
;	Do a linear fit and print results on screen
	index=where(e gt 0,count)
	if count ge 1 then begin
	if strpos(type,'Response') ge 0 then begin
		if count eq 2 then begin
			slope=(y(index(1))-y(index(0))) $
			     /(x(index(1))-x(index(0)))
			intcp=y(index(0))-slope*x(index(0))
			r=[intcp,slope]
			yft=y(index)
			sig=0.0
		endif else r=poly_fit(x(index),y(index),1,yft,ybd,sig,mat)
		cf=where(sconf eq genconfig.config)
		if cf(0) ge 0 then genconfig.response(*,ch,ds_ob,cf)=r
		xrange=!x.crange(1)-!x.crange(0)
		yrange=!y.crange(0)-!y.crange(1)
		xmin=!x.crange(0)+xrange*0.05
		ymin=!y.crange(0)-yrange*0.05
		xyouts,xmin,ymin,'SM='+string(r(0),format='(f4.2)'), $
				color=tci(frame_color),charsize=2.0/mf
		xs=x(index) & si=sort(xs) & xs=xs(si) & ys=yft(si)
		oplot,xs,ys,psym=0,color=tci(frame_color)
	endif else if strpos(type,'V2Bias') ge 0 then begin
		r=[0,0]
		if !x.type eq 1 and !y.type eq 1 then begin
		if max(alog10(x(index)))-min(alog10(x(index))) lt 0.01 then begin
;			Force slope=-1
			slope=-1.0
			intcp=avg(alog10(y(index)))-slope*avg(alog10(x(index)))
			r=[intcp,slope]
			yft=poly(alog10(x(index)),r)
		endif else begin
		if count eq 2 then begin
			slope=(alog10(y(index(1)))-alog10(y(index(0)))) $
			     /(alog10(x(index(1)))-alog10(x(index(0))))
			intcp=alog10(y(index(0)))-slope*alog10(x(index(0)))
			r=[intcp,slope]
			yft=alog10(y(index))
		endif else begin
;			Fit with slope -1
			r=poly_fit(alog10(x(index)),alog10(y(index)*x(index)),0,yft)
			r=[r(0),-1]
			yft=poly(alog10(x(index)),r)
;			General linear fit
			r=poly_fit(alog10(x(index)),alog10(y(index)),1,yft)
		endelse
		endelse
		r=reform(r)
		r(0)=10^r(0)
		for st=0,n_elements(ds_stars)-1 do begin
			cf=where(sconf+' '+ds_stars(st) eq genconfig.config,ncf)
			for icf=0,ncf-1 do genconfig.v2bias(*,bl,ch,ds_ob,cf(icf))=r
		endfor
		xs=x(index) & si=sort(xs) & xs=xs(si) & ys=10^yft(si)
		xmin=10^float(nint(!x.crange(0))+0.02)
		ymin=10^float(nint(!y.crange(0))+0.05)
		if ps_options.p then begin
			xmin=max(xs)*2
			if ich ge 2 then s=-1 else s=+1
			ymin=min(ys)*(1+0.5*s*(((ich+1) mod 2)-0.5))
		endif
		if not ps_options.a and not ps_options.p then $
		xyouts,xmin,ymin,'V!U2!N ='+string(r(0),format='(f6.2)')+ $
		       	'P!E'+string(r(1),format='(f5.2)'), $
				color=tci(frame_color),charsize=2.0/mf
		oplot,xs,ys,psym=0,color=tci(frame_color)
		endif
	endif else if strpos(type,'TABias') ge 0 then begin
		r=[0,0]
		if !x.type eq 1 and !y.type eq 1 then begin
		if max(alog10(x(index)))-min(alog10(x(index))) lt 0.01 then begin
;			Force slope=-1
			slope=-1.0
			intcp=avg(alog10(y(index)))-slope*avg(alog10(x(index)))
			r=[intcp,slope]
			yft=poly(alog10(x(index)),r)
		endif else begin
		if count eq 2 then begin
			slope=(alog10(y(index(1)))-alog10(y(index(0)))) $
			     /(alog10(x(index(1)))-alog10(x(index(0))))
			intcp=alog10(y(index(0)))-slope*alog10(x(index(0)))
			r=[intcp,slope]
			yft=alog10(y(index))
		endif else begin
;			General linear fit
			r=poly_fit(alog10(x(index)),alog10(y(index)),1,yft)
			if r(1) lt -3 or r(1) gt 0 then begin
;			Fit with slope -1
			r=poly_fit(alog10(x(index)),alog10(y(index)*x(index)),0,yft)
			r=[r(0),-1]
			yft=poly(alog10(x(index)),r)
			endif
		endelse
		endelse
		r=reform(r)
		r(0)=10^r(0)
		for st=0,n_elements(ds_stars)-1 do begin
			cf=where(sconf+' '+ds_stars(st) eq genconfig.config,ncf)
			for icf=0,ncf-1 do genconfig.tabias(*,ch,ds_tr,cf(icf))=r
		endfor
		xs=x(index) & si=sort(xs) & xs=xs(si) & ys=10^yft(si)
		xmin=10^float(nint(!x.crange(0))+0.02)
		ymin=10^float(nint(!y.crange(0))+0.05)
		xyouts,xmin,ymin,'V!U2!N ='+string(r(0),format='(f5.2)')+ $
		       	'P!E'+string(r(1),format='(f5.2)'), $
				color=tci(frame_color),charsize=2.0/mf
		oplot,xs,ys,psym=0,color=tci(frame_color)
		endif
	endif else if strpos(type,'TrackJitter') ge 0 then begin
		pmax=max(x(index))
		r=[1,0,0]
		if pmax gt 1 then begin
			p=findgen(long(pmax))
			r=poly_fit(x(index),y(index),2,yft)
			oplot,p,poly(p,r),psym=0,color=tci(1)
		endif
		cf=where(sconf eq genconfig.config)
		if cf(0) ge 0 then genconfig.trackcal(*,bl,ch,ds_ob,cf)=r
	endif else if strpos(type,'TrackOffset') ge 0 and not !offsetcal then begin
		a=[0.5,0,32]
		r=curvefit(x(index),y(index),e(index)*0+1,a, $
			function_name='funct_gauss', $
			fita=[1,1,1],/noderiv)
		si=sort(x(index))
		oplot,x(index(si)),r(si),psym=0,color=tci(frame_color)
		genconfig.offsetcal(*,bl,ch,ds_ob)=a(1:2)
	endif
	endif
;
SKIPPLOT:
endfor
endfor
ENDFOR
;
title1='OB: '+string(ds_ob+1)
title2=' Bl: '+retroparse(ds_bl+1)
title3=' Ch: '+retroparse(ds_ch+1)
title=title1+title2+title3
xyouts,0.1,0.05,title,/normal
;
; Display alert if no data was found to be valid
if plot_no eq 0 then begin
;	window,/free
;	xyouts,0.4,0.5,'No data',/normal,charsize=2
	print,'No data to plot.'
endif
;
genconfig_checksum1=''
for i=0,n_tags(genconfig)-1 do genconfig_checksum1=genconfig_checksum1 $
				 +strjoin(string(genconfig.(i)),/single)
if genconfig_checksum ne genconfig_checksum1 then print,'Fit results stored.'
;
; Reset plot variables
if xrange_auto then !x.range=0 else !x.range=xrange_bck
if yrange_auto then !y.range=0 else !y.range=yrange_bck
if zrange_auto then !z.range=0
!x.type=0
!y.type=0
!p.multi=0
!p.color=tci(1)
!x.title=''
!y.title=''
end
;-------------------------------------------------------------------------------
pro plotuv,overplot=overplot,ms1=ms1
;
; This procedure makes uv-coordinate based plots.
;
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Tables,ScanTable,BGTable,StationTable
common StarBase,StarTable,Notes
common PlacementInfo,xpix_per_char,ypix_per_char, $
		     title_yoffset,title_xoffset, $
		     xtitle_xoffset,xtitle_yoffset, $
		     ytitle_xoffset,ytitle_yoffset, $
		     num_title,max_star_per_title,xpix_per_star, $
		     num_xtitle,max_xlabel_per_title,xpix_per_xlabel, $
		     num_ytitle,max_ylabel_per_title,xpix_per_ylabel
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common LocalPlotUV,model_x,model_y
;
; Check data selection
if printselerror(checkplotsel('x')) ne 0 then return
if printselerror(checkplotsel('y')) ne 0 then return
;
; If overplot=0 then open a new window
do_multi_nights=0
if n_elements(overplot) eq 0 then overplot=0 else do_multi_nights=1
;
if n_elements(ds_stars) eq 0 then begin
	print,'Warning(PLOTUV): no stars selected, using all stars!'
	list_stars,ds_stars
endif
;
set_streamsel,'y',item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,ds_pt
ax_items=set_axitems(item_ids,units)
ax_items=ax_items+units
; pt=indgen(n_elements(scans))
pt=ds_pt
if n_elements(pt) eq 1 then pt=[pt,pt]
;
; Determine number of plots
num_stars=n_elements(ds_stars)
num_h=long(sqrt(num_stars))
if num_h*num_h ne num_stars then begin
	if num_stars-num_h*num_h gt num_h then begin
		num_h=num_h+1
		num_v=num_h
	endif else num_v=num_h+1
endif else num_v=num_h
; !p.multi=[#plots remaining,#plot columns,#plot rows,#plots stacked in Z,
;           0:left to right and top to bottom;1:top to bottom and left to right]
if ps_options.a then !p.multi=0 else !p.multi=[0,num_h,num_v,0,1]
;
; For multiple plots, IDL reduces the charsize by a factor of 2
!p.charsize=1.5
if !p.multi(1)*!p.multi(2) le 1 then !p.charsize=1.5
if !p.multi(1) gt 2 or !p.multi(2) gt 2 then begin
	!p.charsize=2.0
	mf=2.0 
endif else mf=1.0
;
; Open a plot window
!xsize=512
!ysize=!xsize*(0.25/0.27)
if !d.name ne 'PS' then begin
	if overplot eq 0 or !d.x_vsize le 4 then begin
	window,/free,title=Date+', Class: uv',xsize=!xsize,ysize=!ysize, $
		xpos=!dxsize-!xsize,ypos=!dysize-!ysize
	endif
endif else if !d.name eq 'PS' then begin
	device,color=ps_options.c
	tek_color
	if ps_options.p then begin
		if mf eq 1 then !p.charsize=1.5 else !p.charsize=2.0
	endif
endif else begin
	print,'***Error(PLOTUV): invalid device name!'
	return
endelse
model_x=0
model_y=0
model_z=0
if !d.name eq 'PS' and ps_options.c then frame_color=0 else frame_color=1
;
; For option All in 1 we have to go through the data twice to find the
; number of plots in a window (num_plot) and the total range of data values.
if ps_options.a then num_run=2 else num_run=1
;
; Compute some auxilliary variables
titleplacement,'t',' '
;
; All spectrometers or just one?
if ps_options.o then begin
	f_ob=0
	l_ob=GenConfig.NumOutBeam-1
endif else begin
	f_ob=ds_ob
	l_ob=ds_ob
endelse
;
; Compute more auxilliary plot variables
if ps_options.o then num_xlabel=produkt(GenConfig.NumSpecChan) $
	        else num_xlabel=n_elements(ds_ch)
titleplacement,'x',dummy,' ',0
titleplacement,'y',dummy,' ',0
;
if ds_x.item ge 50 and ds_y.item ge 50 then do_uv=1 else do_uv=0
;
; Determine whether automatic scaling is requested
if total(abs(!x.range)) eq 0 then xrange_auto=1 else xrange_auto=0
if total(abs(!y.range)) eq 0 then yrange_auto=1 else yrange_auto=0
;
FOR run_no=1,num_run DO BEGIN
first=1
;
for is=0,n_elements(ds_stars)-1 do begin
;
;       First, we have to find out about the ranges
	if run_no eq 1 then begin
	if not ps_options.a then first=1
	for ob=f_ob,l_ob do begin
		if ps_options.o then begin
;			l_bl=GenConfig.NumBaseline(ob)-1
;			l_ch=GenConfig.NumSpecChan(ob)-1
			l_bl=n_elements(ds_bl)-1
			l_ch=n_elements(ds_ch)-1
		endif else begin
			l_bl=n_elements(ds_bl)-1
			l_ch=n_elements(ds_ch)-1
		endelse
		for ib=0,l_bl do begin
		for ic=0,l_ch do begin
		   if ps_options.o then begin
;			ibase=ib
;			ichan=ic
			ibase=ds_bl(ib)
			ichan=ds_ch(ic)
		   endif else begin
			ibase=ds_bl(ib)
			ichan=ds_ch(ic)
		   endelse
		   itriple=0
	   	   udata=set_plotdata(ds_x.item,ds_ib,ob,itriple,ichan,ibase,pt)
		   vdata=set_plotdata(ds_y.item,ds_ib,ob,itriple,ichan,ibase,pt)
;       	   For editing, select VisSqCErr
		   vis_err=set_ploterr(25,ds_ib,ob,itriple,ichan,ibase,pt)
		   index=where(scans(pt).starid eq ds_stars(is),count)
		   if count gt 0 then begin
		        u=udata(index)
		        v=vdata(index)
		        e=vis_err(index)
		        if ps_options.f then begin
				count=n_elements(e)
				index=indgen(count) 
			endif else index=where(e ge 0,count)
			if count gt 0 then begin
				if first then begin
					umax=max(u(index))
					umin=min(u(index))
					vmax=max(v(index))
					vmin=min(v(index))
					first=0
				endif else begin
					umax=max([umax,max(u(index))])
					umin=min([umin,min(u(index))])
					vmax=max([vmax,max(v(index))])
					vmin=min([vmin,min(v(index))])
				endelse
			endif
		   endif else begin
			print,'***Warning(PLOTUV): no data for star: ', $
				ds_stars(is)
		   endelse
		endfor
		endfor
	endfor
	if ps_options.a then goto,SKIPPLOT
	endif
;
        if n_elements(umin) eq 0 then begin
		if n_elements(ds_stars) eq 1 and not do_multi_nights then begin
			if !d.name ne 'PS' then begin
				window,/free
				xyouts,0.4,0.5,'No data',/normal,charsize=2
			endif else print,'No data'
		endif
		if not do_multi_nights then goto,SKIPPLOT
	endif
;
;	Plot the frame
	!p.color=tci(frame_color)
	if ps_options.o eq 0 then begin
		title0='OB: '+string(ob,format='(i1)')
		title1='Bl: ' 
		for i=0,n_elements(ds_bl)-1 do $
			title1=title1+strcompress(string(ds_bl(i)+1))
		title1='Bl: '+retroparse(ds_bl+1)
		title2='Ch: '+retroparse(ds_ch+1)
	endif else begin
		title0='OB: '
		for i=f_ob,l_ob do $
			title0=title0+strcompress(string(i+1))
;		title1='All Bl'
;		title2='All Ch'
		title1='Bl: ' 
		for i=0,n_elements(ds_bl)-1 do $
			title1=title1+strcompress(string(ds_bl(i)+1))
		title2='Ch: '+retroparse(ds_ch+1)
	endelse
;
	if do_uv then begin
		dmax=max([umax,vmax,-umin,-vmin])
		dmin=-dmax
		umin=dmin
		umax=dmax
		vmin=dmin
		vmax=dmax
	endif
;
	if xrange_auto then !x.range=[umin,umax]
;	uv-coverage is plotted as projected onto the sky, i.e.
;	the u coordinate increases to the left (east)
	if ds_x.item eq 50 and systemid ne 'Keck' then !x.range=shift(!x.range,1)
	if yrange_auto then !y.range=[vmin,vmax]
;
	if is eq 0 or not ps_options.a and overplot eq 0 then $
	plot,udata,vdata,/nodata, $
	 	xtitle=ax_items(long(where(item_ids eq ds_x.item),0)), $
	 	ytitle=ax_items(long(where(item_ids eq ds_y.item),0))
;
	if is eq 0 and not ps_options.p then $
	xyouts,0.1,!y.window(1)+float(ypix_per_char)/!d.y_size/mf, $
		title0+', '+title1+', '+title2,/normal,size=!p.charsize/mf
;
	if is eq 0 and not ps_options.p then $
	xyouts,1.0-float(ypix_per_char)/!d.x_size,!y.window(1), $
		Date,/normal,orientation=-90,size=!p.charsize/mf
;
	if is eq 0 and do_uv then begin
		n=17+3
		c=4
		l=fltarr(n)+c	; units of charactersize
		if ds_x.item eq 50 and ds_y.item eq 51 then sense=0 else sense=1
		if startable(where(startable.starid eq ds_stars(is))).dec lt 0 $
			then sense=sense-0.5 lt 0
		p=pi_circle*sense-signof(sense-0.5)*findgen(n)/(n-5)*pi_circle
		p(n-3)=pi_circle*(1-sense) & l(n-3)=c-0.2*c
		p(n-2)=pi_circle*(1-sense) & l(n-2)=c+0.2*c
		p(n-1)=p(n-4) & l(n-1)=l(n-4)
		x=l*sin(p)
		y=l*cos(p)
		usersym,x,y,thick=2
		if do_uv and not ps_options.p then oplot,[0,0],[0,0],psym=8
		l=0.5	; units of charactersize
		n=45
		p=findgen(n+1)*2*pi_circle/n
		x=l*sin(p)
		y=l*cos(p)
		usersym,x,y,/fill
		if systemid eq 'Keck' then !p.psym=8
;		if systemid eq 'VLTI' then !p.psym=8
		if n_elements(ds_ch) eq 1 or n_elements(ds_pt) eq 1 then !p.psym=8
	endif
	if not do_uv then !p.psym=1
;
	if ps_options.a then !p.color=getcolor(is)
	num_char_per_label=max(strlen(ds_stars)+1)
	max_num_xlabel=fix((!x.window(1)-!x.window(0)) $
		      /(num_char_per_label*float(xpix_per_char)/!d.x_size))-1
	if not ps_options.p then $
	xyouts,!x.window(0)+(2+(is mod max_num_xlabel)*num_char_per_label) $
			    *float(xpix_per_char)/!d.x_size, $
       	       !y.window(1)-(2+is/max_num_xlabel) $
			    *float(ypix_per_char)/!d.y_size, $
		ds_stars(is),/normal,size=!p.charsize/mf
	if ps_options.l then !p.psym=-abs(!p.psym) else !p.psym=abs(!p.psym)
;
; 	Now that we have the range, we plot the data for this star
	for ob=f_ob,l_ob do begin
		for ib=0,n_elements(ds_bl)-1 do begin
		nch=0
		for ic=0,n_elements(ds_ch)-1 do begin
		   ibase=ds_bl(ib)
		   ichan=ds_ch(ic)
		   itriple=0
		   udata=set_plotdata(ds_x.item,ds_ib,ob,itriple,ichan,ibase,pt)
		   vdata=set_plotdata(ds_y.item,ds_ib,ob,itriple,ichan,ibase,pt)
;       	   For editing, select VisSqCErr
		   vis_err=set_ploterr(25,ds_ib,ob,itriple,ichan,ibase,pt)
		   index=where(scans(pt).starid eq ds_stars(is))
		   u=udata(index)
		   v=vdata(index)
		   e=vis_err(index)
;            	   Edit the data
		   index=where(e ge 0,count)
		   if count gt 0 then begin
			if ps_options.a then !p.color=getcolor(is) $
					else !p.color=getcolor(ob)
			if overplot gt 0 then !p.color=getcolor(overplot+1)
			oplot,u(index),v(index)
			if do_uv then oplot,-u(index),-v(index)
			if ps_options.e and ds_y.item eq 25 then begin
				if count gt 1 then $
				oploterror,u(index),v(index),u(index)*0,e(index) else $
				oploterror,[u(index),u(index)], $
					 [v(index),v(index)], $
					 [u(index),u(index)]*0, $
					 [e(index),e(index)]
			endif
		   endif
		   if ps_options.f eq 1 then begin
			index=where(e lt 0,count)
			if count ne 0 then begin
			!p.color=tci(2)
			oplot,u(index),v(index)
			if do_uv then oplot,-u(index),-v(index)
			if ps_options.e and ds_y.item eq 25 then begin
				if count gt 1 then $
				oploterror,u(index),v(index),u(index)*0,e(index) else $
				oploterror,[u(0),u(0)],[v(0),v(0)],[u(0),u(0)]*0,[e(0),e(0)]
			endif
			endif
		   endif
		   if ps_options.m then begin
		   if n_elements(star_model) eq 0 then $
			print,'***Error(PLOTUV): no model!' $
		   else if gen_model(0).starid ne ds_stars(is) then $
			print,'***Error(PLOTUV): no model for this star!' $
		   else begin
			mitem=0
			if ds_y.item eq 25 then mitem=57
			if mitem ne 0 then begin
			ymodel=set_plotdata(mitem,ds_ib,ob,itriple,ichan,ibase,pt)
			if n_elements(ymodel) eq 1 then ymodel=fltarr(1)+ymodel
			index=where(scans(pt).starid eq ds_stars(is))
			jndex=where(vis_err(index) gt 0,count)
			if count gt 0 then begin
				nch=nch+1
				x=udata(index(jndex))
				y=ymodel(index(jndex))
				su=uniq(x)
				x=x(su)
				y=y(su)
				t=scans(pt).time
				si=sort(x)
				x=x(si)
				y=y(si)
;				Overplot all model values of this bl/ch
;				oplot,x,y,psym=-7,color=tci(frame_color)
				model_x=[model_x,x]
				model_y=[model_y,y]
				if keyword_set(ms1) then begin
				ds=scans
				scans=ms1
				zmodel=set_plotdata(mitem,ds_ib,ob, $
						    itriple,ichan,ibase,pt)
				if n_elements(zmodel) eq 1 then $
					zmodel=fltarr(1)+zmodel
				z=zmodel(index(jndex))
				scans=ds
				z=z(si)
				model_z=[model_z,z]
				endif
			endif
			endif
		   endelse
		   endif
		endfor
		endfor
	endfor
;	Plot accumulated model data to get data points connected
	if ps_options.m and n_elements(model_x) gt 1 and not do_uv then begin
	x=model_x(1:n_elements(model_x)-1)
	y=model_y(1:n_elements(model_y)-1)
	if keyword_set(ms1) then z=model_z(1:n_elements(model_z)-1)
	if nch gt 1 and ib gt 1 then begin
		si=sort(x)
		oplot,x(si),y(si),psym=0,color=tci(frame_color)
		if keyword_set(ms1) then $
			oplot,x(si),z(si),psym=0,color=tci(frame_color), $
				psym=0,color=tci(frame_color),linestyle=2
	endif else if nch gt 1 then begin
;		Multi-channel, group by channel
		npt=n_elements(x)/nch
		for i=0,npt-1 do begin
			oplot,x(indgen(nch)*npt+i),y(indgen(nch)*npt+i), $
				psym=0,color=tci(frame_color)
			if keyword_set(ms1) then $
			oplot,x(indgen(nch)*npt+i),z(indgen(nch)*npt+i), $
				psym=0,color=tci(frame_color),linestyle=2
		endfor
	endif else begin
;		Single channel
		si=sort(x)
		oplot,x(si),y(si),psym=7*0,color=tci(frame_color)
		if keyword_set(ms1) then $
		oplot,x(si),z(si),psym=4*0,color=tci(frame_color),linestyle=2
	endelse
	model_x=0
	model_y=0
	endif
	SKIPPLOT:
endfor
ENDFOR
;
; Reset plot ranges, colors, ...
if xrange_auto then !x.range=0
if yrange_auto then !y.range=0
!p.multi=0
!p.color=tci(1)
!x.title=''
!y.title=''
;
end
;************************************************************************Block 5
function init_plotorbit
;
common PlotOrbit,plotorbit_options
;
plotorbit_options=alloc_plotorbit_options()
;
!p.psym=0
;
end
;-------------------------------------------------------------------------------
function thiele
;
; Use Thiele-Innes method to estimate orbital parameters once the apparent
; ellipse has been fitted to the positions.
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
o_parms=dblarr(8)
RAD=180/pi_circle
;
; Check inputs
num_scan=n_elements(positions)
if num_scan eq 0 then begin
	print,'***Error(THIELE): no data!'
	return,o_parms
endif
if n_elements(e_parms) eq 0 then begin
	print,'***Error(THIELE): ellipse parms undefined!'
	return,o_parms
endif
if total(e_parms) eq 0 then begin
	print,'***Error(THIELE): ellipse parms zero!'
	return,o_parms
endif
;
; Set and sort data
jd=positions.jd
theta=positions.theta
rho=positions.rho
s=sort(jd)
jd=jd(s)
theta=theta(s)
rho=rho(s)
;
; Determine orbital sense
dtheta=theta-shift(theta,1)
index=where(dtheta lt -pi_circle,count)
if count gt 0 then dtheta(index)=dtheta(index)+2*pi_circle
index=where(dtheta ge 0,count1)
index=where(dtheta lt 0,count2)
if count1 gt count2 then begin
	o_parms(7)=1 
	print,'Orbit is PROGRADE.'
endif else begin
	o_parms(7)=-1
	print,'Orbit is RETROGRADE.'
endelse
;
; Compute cumulative area a of fit-ellipse
n=3600
p=dindgen(n)*2*pi_circle/n
if o_parms(7) eq -1 then p=2*pi_circle-p
r=ellipse(e_parms(2),e_parms(3),p-e_parms(4))
x=r*sin(p)+e_parms(0)
y=r*cos(p)+e_parms(1)
a=dblarr(n)
for i=1,n-1 do $
	a(i)=sqrt(total(crossp([x(i),y(i),0.0],[x(i-1),y(i-1),0.0])^2))/2+a(i-1)
p=atan(x,y)
index=where(p lt 0,count)
if count gt 0 then p(index)=p(index)+2*pi_circle
r=sqrt(x^2+y^2)
e_area=e_parms(2)*e_parms(3)*pi_circle
;
; Use Kepler's second law to estimate period
; Compute cummulative area for observed positions...
area=dblarr(num_scan)
for i=0,num_scan-1 do begin
	dp=abs(theta(i)-p)
	area(i)=a(where(dp eq min(dp)))
endfor
; ...We combine pairs to estimate the period...
nc=combinations(num_scan,2)
djd=dblarr(nc)
da=dblarr(nc)
; ...determine optimum mean spacing of measures
j=1
for i=0,num_scan-2 do begin
	djd(i)=jd(i+j)-jd(i)
	da(i)=area(i+j)-area(i)
	if da(i) lt 0 then da(i)=da(i)+e_area
endfor
index=where(djd ne 0)
da_m=median(da(index))
limit=0.01	; Binary radius must sweep more than (limit*100)% of area 
;		  between m.
j=fix(limit/(da_m/e_area)+1)
optimum=3	; Hand-tweaked, for now...
j_max=max([j,fix(e_area/da_m/optimum)])
;
; ...Obtain estimates for all allowed pairs...
djd(*)=0
da(*)=0
n=0
break=0
while j lt num_scan and not break do begin
	for i=0,num_scan-j-1 do begin
		djd(n)=jd(i+j)-jd(i)
		da(n)=area(i+j)-area(i)
		if da(n) lt 0 then da(n)=da(n)+e_area
		n=n+1
	endfor
	if j eq j_max then break=1
	j=j+1
endwhile
index=where(djd ne 0 and da ne 0,count)
da=da(index) & djd=djd(index)
o_parms(5)=median(e_area/da*djd)
;
; First improvement of period by unwrapping phases
new_da=da
for i=0,count-1 do begin
	new_da(i)=da(i)+fix(djd(i)/o_parms(5))*e_area
	period=e_area/new_da(i)*djd(i)
	if period/o_parms(5) lt 0.1 then new_da(i)=0
endfor
index=where(new_da ne 0)
period=e_area/new_da(index)*djd(index)
o_parms(5)=median(period)
;
; Second improvement by computing cummulative areas
; for i=0,count-1 do begin
;	new_da(i)=da(i)+fix(djd(i)/o_parms(5))*e_area
;	period=e_area/new_da(i)*djd(i)
;	if period/o_parms(5) lt 0.1 then new_da(i)=new_da(i)-e_area
; endfor
; cda=new_da
; cdjd=djd
; for i=0L,n_elements(new_da)-1 do cda(i)=total(new_da(0:i))
; for i=0L,n_elements(djd)-1 do cdjd(i)=total(djd(0:i))
; coeffs=poly_fit(cdjd,cda,1)
; o_parms(5)=e_area/coeffs(1)
;
; Use Kepler's second law to compute epochs for phases p
t=a/e_area*o_parms(5)
dp=abs(theta(0)-p)
index=where(dp eq min(dp))
a0=a(index) & a0=a0(0)
tjd=dblarr(num_scan)
for i=0,num_scan-1 do begin
	dp=abs(theta(i)-p)
	index=where(dp eq min(dp))
	tjd(i)=jd(0)+((jd(i)-jd(0)) mod o_parms(5)) $
		-abs(a(index)-a0)/e_area*o_parms(5)
endfor
jd0=double(median(tjd))
dp=abs(theta(0)-p)
index=where(dp eq min(dp))
t=t-t(index(0))+jd0
;
; Find periastron and compute eccentricty
p_w=atan(e_parms(0),e_parms(1))+pi_circle
dp=abs(p_w-p)
p_index=where(dp eq min(dp)) & p_index=p_index(0)
ap=r(p_index)
o_parms(1)=1/(1+ap/sqrt(total(e_parms(0:1)^2)))
;
; Get epoch from array t
o_parms(6)=t(p_index)
;
; Compute point R with true anomaly = 90
t_90=o_parms(6)+o_parms(5)/2/pi_circle $
	*(acos(o_parms(1))-o_parms(1)*sqrt(1-o_parms(1)^2))
dt=abs((t-t_90) mod o_parms(5))
r_index=where(dt eq min(dt)) & r_index=r_index(0)
;
; Compute Thiele-Innes constants
X1=x(p_index)/(1-o_parms(1))
Y1=y(p_index)/(1-o_parms(1))
X2=x(r_index)/(1-o_parms(1)^2)
Y2=y(r_index)/(1-o_parms(1)^2)
;
; Compute argument of periastron
wpw=atan((X1-Y2)/(X2+Y1))
wmw=atan((X1+Y2)/(X2-Y1))
if (X1-Y2)*sin(wpw) lt 0 then wpw=wpw+pi_circle
if (X1+Y2)*sin(wmw) gt 0 then wmw=wmw+pi_circle
o_parms(3)=((wpw+wmw)/2) mod (2*pi_circle)
o_parms(3)=(o_parms(3)+pi_circle) mod (2*pi_circle)	; change to primary
;
; Compute argument of ascending node
o_parms(4)=(wpw-wmw)/2
if o_parms(4) lt 0 then o_parms(4)=o_parms(4)+2*pi_circle
;
; If e near 0, set w=0 and re-do epoch
if o_parms(1) lt 0.001 then begin
	o_parms(6)=o_parms(6)-o_parms(3)/2/pi_circle*o_parms(5)
	o_parms(3)=0
endif
;
; Compute inclination
o_parms(2)=atan(sqrt(-(X1+y2)*sin(wpw)/(X1-Y2)/sin(wmw)))*2
;
; Compute semi-major axis 
o_parms(0)=(X1-Y2)/sin(wpw)/(1+cos(o_parms(2)))
;
print,'__________________________________'
print,'Thiele-Innes estimates: '
print,'Semi-major axis = ',o_parms(0)
print,'Eccentricity =    ',o_parms(1)
print,'Inclination =     ',o_parms(2)*RAD
print,'Periastron =      ',o_parms(3)*RAD
print,'Ascending node =  ',o_parms(4)*RAD
print,'Period =          ',o_parms(5)
print,'Epoch =           ',o_parms(6)
print,'_______________***________________'
;
return,o_parms
end
;-------------------------------------------------------------------------------
function innes,t_parms,prograde=prograde
;
; Given the orbital period, epoch of periastron, and the 5 fit parameters
; of an apparent ellipse fit, compute the 7 orbital elements.
; The analysis follows closely R.M. Green, Spherical Astronomy, pp.470
;
; Center x =        t_parms(0)
; Center y =        t_parms(1)
; Semi-major axis = t_parms(2) 
; Semi-major axis = t_parms(3)
; Position angle =  t_parms(4) [rad]
; Period =          t_parms(5)
; Epoch =           t_parms(6)
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(t_parms) eq 0 then $
	t_parms=[e_parms,binary_model(0).period,binary_model(0).epoch+2440000]
o_parms=t_parms
RAD=180/pi_circle
;
; Check inputs
if n_elements(t_parms) eq 0 then begin
	print,'***Error(INNES): orbit parms undefined!'
	return,o_parms
endif
;
if n_elements(prograde) eq 0 then begin
	jd=positions.jd
	theta=positions.theta
	rho=positions.rho
	s=sort(jd)
	jd=jd(s)
	theta=theta(s)
	rho=rho(s)
;
; 	Determine orbital sense
	dtheta=theta-shift(theta,1)
	index=where(dtheta lt -pi_circle,count)
	if count gt 0 then dtheta(index)=dtheta(index)+2*pi_circle
	index=where(dtheta ge 0,count1)
	index=where(dtheta lt 0,count2)
	if count1 gt count2 then begin
		print,'Orbit is PROGRADE.'
		prograde=1
	endif else begin
		print,'Orbit is RETROGRADE.'
		prograde=0
	endelse
endif
;
; Determine eccentricity
ecc=sqrt(t_parms(2)^2-t_parms(3)^2)/t_parms(2)
alpha=atan(-t_parms(0),-t_parms(1))
beeta=t_parms(4)-alpha
cp=sqrt(t_parms(3)^2/(1-(ecc*cos(beeta))^2))
ca=sqrt(t_parms(0)^2+t_parms(1)^2)
o_parms(1)=ca/cp
;
; Determine coordinates (x1,y1) of periastron P
ap=cp-ca
x1=ap*sin(alpha)
y1=ap*cos(alpha)
;
; Compute eccentric anomaly for the point on the orbit where T_anom=90
E_anom=acos(o_parms(1))
;
; Compute time for this point
t=o_parms(6)+o_parms(5)/(2*pi_circle)*(E_anom-o_parms(1)*sqrt(1-o_parms(1)^2))
;
; Use equal area law to find coordinates (x2,y2) of point R
e_area=t_parms(2)*t_parms(3)*pi_circle
r_area=(t-o_parms(6))/o_parms(5)*e_area
x2p=x1
y2p=y1
dalpha=2*pi_circle/3600
if not prograde then dalpha=-dalpha
alpha2=alpha
a=0
repeat begin
	alpha2=alpha2+dalpha
	r=ellipse(t_parms(2),t_parms(3),alpha2-t_parms(4))
	x2=r*sin(alpha2)+t_parms(0)
	y2=r*cos(alpha2)+t_parms(1)
	da=sqrt(total(crossp([x2,y2,0.0],[x2p,y2p,0.0])^2))/2
	a=a+da
	x2p=x2
	y2p=y2
endrep until a gt r_area
;
; Compute Thiele-Innes constants
X1=x1/(1-o_parms(1))
Y1=y1/(1-o_parms(1))
X2=x2/(1-o_parms(1)^2)
Y2=y2/(1-o_parms(1)^2)
;
; Compute argument of periastron
wpw=atan((X1-Y2),(X2+Y1))
wmw=atan((X1+Y2),(X2-Y1))
if (X1-Y2)*sin(wpw) lt 0 then wpw=wpw+pi_circle
if (X1+Y2)*sin(wmw) gt 0 then wmw=wmw+pi_circle
o_parms(3)=((wpw+wmw)/2) mod (2*pi_circle)
;
; Compute argument of ascending node
o_parms(4)=(wpw-wmw)/2
if o_parms(4) lt 0 then o_parms(4)=o_parms(4)+2*pi_circle
;
; If e near 0, set w=0 and re-do epoch
if o_parms(1) lt 0.001 then begin
	o_parms(6)=o_parms(6)-o_parms(3)/2/pi_circle*o_parms(5)
	o_parms(3)=0
endif
;
; Compute inclination
o_parms(2)=atan(sqrt(-(X1+Y2)*sin(wpw)/(X1-Y2)/sin(wmw)))*2
;
; Compute semi-major axis 
o_parms(0)=(X1-Y2)/sin(wpw)/(1+cos(o_parms(2)))
;
print,'__________________________________'
print,'Thiele-Innes estimates: '
print,'Semi-major axis = ',o_parms(0)
print,'Eccentricity =    ',o_parms(1)
print,'Inclination =     ',o_parms(2)*RAD
print,'Periastron =      ',o_parms(3)*RAD
print,'Ascending node =  ',o_parms(4)*RAD
print,'Period =          ',o_parms(5)
print,'Epoch =           ',o_parms(6)
print,'_______________***________________'
;
return,o_parms
end
;-------------------------------------------------------------------------------
pro plotorbit,overplot=overplot
;
; Plots multiple star orbits.
;
; Some information on symbols and colors:
; !p.psym=0: no symbol, connect points with lines
;         1: +
;         2: *
;         3: .
;         4: Diamond
;         5: Triangle
;         6: Square
;         7: x
; tek_color
; !p.color=0: black
;          1: white
;          2: red
;          3: green
;          4: blue
;          .
;          .
;         31: <last color index>
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common PlotOrbit,plotorbit_options
common PlotorbitDataInfo,plotscans,plotdata_x,plotdata_y
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(overplot) eq 0 then overplot=0
;
; Check data
if n_elements(positions) eq 0 then begin
	print,'***Error(PLOTORBIT): no data!'
	return
endif
;
; Set character size
!p.charsize=1.5
;
; Set number of plots
!p.multi=0
;
; Open a plot window for plotting on the screen
if n_elements(gen_model) ne 0 then star=gen_model.starid else star=' '
if not overplot then begin
if !d.name ne 'PS' then begin
	window,/free,title=star,xsize=!xsize,ysize=!ysize, $
		xpos=!dxsize-!xsize,ypos=!dysize-!ysize
endif else begin
        device,color=plotorbit_options.c
	tek_color
endelse
endif
if !d.name eq 'PS' and plotorbit_options.c then frame_color=0 else frame_color=1
;
; Determine whether automatic scaling is requested
if total(abs(!x.range)) eq 0 then xrange_auto=1 else xrange_auto=0
if total(abs(!y.range)) eq 0 then yrange_auto=1 else yrange_auto=0
!x.style=2
!y.style=2
!x.margin=[8.0,2.0]
!y.margin=[6.0,4.0]/!d.y_ch_size*!d.x_ch_size
;
; Set data
index=where(positions.component eq plotorbit_options.component,count)
if count eq 0 then begin
	print,'***Error(PLOTORBIT): no data for this component!'
	return
endif
positions_save=positions
positions=positions(index)
rho=positions.rho
theta=positions.theta
if plotorbit_options.j then $
	theta=precess(theta,positions.jy,gen_model.ra,gen_model.dec,-1)
ra=rho*sin(theta)
dec=rho*cos(theta)
;
; Prepare PlotorbitDataInfo
plotscans=positions
plotdata_x=ra
plotdata_y=dec
;
; Set and equalize ranges
if xrange_auto then !x.range=[max(ra),min(ra)] else !x.style=1
if yrange_auto then !y.range=[min(dec),max(dec)] else !y.style=1
;xrange=!x.range(0)-!x.range(1)
;yrange=!y.range(1)-!y.range(0)
;drange=xrange-yrange
;if drange lt 0 then begin
;	!x.range(0)=!x.range(0)-drange/2
;	!x.range(1)=!x.range(1)+drange/2
;endif else begin
;	!y.range(0)=!y.range(0)-drange/2
;	!y.range(1)=!y.range(1)+drange/2
;endelse
;
; Plot frame
!p.color=tci(frame_color)
plot,[ra,ra],[dec,dec],/nodata,xrange=!x.range,yrange=!y.range, $
	xtitle='Right Ascension offset [mas]', $
	ytitle='Declination offset [mas]', $
	title='';wmcc(plotorbit_options.component)
;
; Plot dotted cross lines
oplot,!x.crange,[0,0],linestyle=1,psym=0
oplot,[0,0],!y.crange,linestyle=1,psym=0
;
; Plot data
if plotorbit_options.e then begin
	rhom=positions.rhom
	them=positions.thetam
	if plotorbit_options.j then $
		them=precess(them,positions.jy,gen_model.ra,gen_model.dec,-1)
	ram=rhom*sin(them)
	decm=rhom*cos(them)
	n=180
	p=findgen(n+1)*2*pi_circle/n
	for i=0,n_elements(positions)-1 do begin
;		Note: we do not precess the error ellipse pa!
		r=ellipse(positions(i).emajor,positions(i).eminor, $
			p+positions(i).pa+pi_circle/2)
		x=r*cos(p)+ra(i)
		y=r*sin(p)+dec(i)
;		Plot uncertainty ellipse and line connecting to orbit
		if positions(i).rho gt 0 then begin
			oplot,x,y,psym=0,thick=2
			oplot,[ra(i),ram(i)],[dec(i),decm(i)],psym=0,color=tci(2)
		endif else if plotorbit_options.f then begin
			oplot,x,y,psym=0,color=tci(2)
			oplot,[ra(i),ram(i)],[dec(i),decm(i)],color=tci(2),psym=0
		endif
	endfor
endif else begin
	l=(!y.crange(1)-!y.crange(0))/5
	l=0.5*2	; units of charactersize
	n=45
	p=findgen(n+1)*2*pi_circle/n
	x=l*sin(p)
	y=l*cos(p)
	usersym,x,y,/fill
	index=where(positions.rho gt 0,count)
;	Use psym=8 for user symbol
	psym=8
	if plotorbit_options.t then psym=-psym
	if count gt 0 then oplot,ra(index),dec(index),psym=psym
	if plotorbit_options.f then begin
		index=where(positions.rho lt 0,count)
		if count gt 0 then oplot,ra(index),dec(index),psym=psym,color=tci(2)
	endif
endelse
;
; Code for user requests
;
chris=0
if chris then begin
xyouts,ra(2)+15,dec(2)-2,'1998.2',charsize=1.3
xyouts,ra(6)+15,dec(6),'1999.1',charsize=1.3
xyouts,ra(12)+5,dec(12)+3,'2000.8',charsize=1.3
xyouts,ra(13)-2,dec(13),'2002.0',charsize=1.3
xyouts,ra(15)-2,dec(15),'2003.0',charsize=1.3
xyouts,ra(16)+5,dec(16)-5,'2004.2',charsize=1.3
endif
;
ken=0
if ken then begin
for year=1983,2005 do begin
RAD=180/pi_circle
rt=binarypos(julian(year,1,2),'B-C')
x0=rt(0)*sin(rt(1)/RAD)
y0=rt(0)*cos(rt(1)/RAD)
rt=binarypos(julian(year,1,1),'B-C')
x1=rt(0)*sin(rt(1)/RAD)
y1=rt(0)*cos(rt(1)/RAD)
rt=binarypos(julian(year,1,3),'B-C')
x2=rt(0)*sin(rt(1)/RAD)
y2=rt(0)*cos(rt(1)/RAD)
yy0=2.0/sqrt(1.+((y1-y2)/(x2-x1))^2)
xx0=yy0*((y1-y2)/(x2-x1))
y=yy0+y0
x=xx0+x0
oplot,[x,x0],[y,y0],psym=0
y=-yy0+y0
x=-xx0+x0
oplot,[x,x0],[y,y0],psym=0
;
if year mod 5 eq 0 then begin
yy0=7.0/sqrt(1.+((y1-y2)/(x2-x1))^2)
xx0=yy0*((y1-y2)/(x2-x1))
y=yy0+y0
x=xx0+x0
if year le 1995 or year eq 2005 then xyouts,x,y,string((year-1900) mod 100,format='(i2.2)')
y=-yy0+y0
x=-xx0+x0
if year ge 1996 and year lt 2005 then xyouts,x,y,string((year-1900) mod 100,format='(i2.2)')
endif
endfor
endif
;
beate=0
if beate then begin
RAD=180/pi_circle
rt=binarypos(julian(2009,7,13))
x0=rt(0)*sin(rt(1)/RAD)
y0=rt(0)*cos(rt(1)/RAD)
oplot,[x0,x0],[y0,y0],psym=4
rt=binarypos(jy2jd(1991.2489))
rt=[1260,264.]
x0=rt(0)*sin(rt(1)/RAD)
y0=rt(0)*cos(rt(1)/RAD)
; oplot,[x0,x0],[y0,y0],psym=5
xyouts,-840,-260,'NACO',charsize=1
xyouts,-1120,-100,'Hp',charsize=1
endif
;
; Plot model
if plotorbit_options.o then plotthiele,plotorbit_options.component
if plotorbit_options.l then plotellipse
;
; Reset plot variables
if xrange_auto then !x.range=0
if yrange_auto then !y.range=0
!p.color=tci(1)
;
; Restore full set of position data
positions=positions_save
;
end
;-------------------------------------------------------------------------------
pro plotthiele,component,linestyle=linestyle
;
; Please note that for multiple systems measured at different epochs,
; no unique orbit can be plotted.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common PlotOrbit,plotorbit_options
;
RAD=180/pi_circle
;
if n_elements(component) eq 0 then begin
	print,'***Error(PLOTTHIELE): component undefined!'
	return
endif
if n_elements(binary_model) eq 0 then begin
	print,'***Error(PLOTTHIELE): no model!'
	return
endif
if n_elements(linestyle) eq 0 then linestyle=0
;
i=where(component eq binary_model.component,count) & i=i(0)
if count eq 0 then begin
	print,'***Error(PLOTTHIELE): component not in model!'
	return
endif
;
p=binary_model(i).period
if n_elements(positions) ge 3 then t=median(positions.jd)-p/2 $
			      else t=avg(positions.jd)-p/2
; The epoch should be close to the median observing epoch
t=binary_model(i).epoch	; If you want to plot the periastron...
;
num=360*4
jd=dindgen(num+1)/num*p+t
;
; Assume reddest wavelengths for photocenter
lambda=fltarr(1)+gen_model.wavelengths(n_elements(gen_model.wavelengths)-1)*1e-6
lambda=2.2e-6
lambda=fltarr(1)+550e-9
;
p=modelpos(jd)
pos1=dblarr(num+1,2)
if componentparse(component,comp_1,comp_2) ne 0 then return
tflux=0.d0
for k=0,strlen(comp_1)-1 do begin
	j=where(strmid(comp_1,k,1) eq star_model.component) & j=j(0)
	flux=stellarfluxes(star_model(j),lambda) $
	    *modelfluxes(star_model(j),lambda) & flux=total(flux)
	pos1=pos1+reform(p(*,j,*)*flux,num+1,2)
	tflux=tflux+flux
endfor
pos1=pos1/tflux
pos2=dblarr(num+1,2)
tflux=0.d0
for k=0,strlen(comp_2)-1 do begin
	j=where(strmid(comp_2,k,1) eq star_model.component) & j=j(0)
	flux=stellarfluxes(star_model(j),lambda) $
	    *modelfluxes(star_model(j),lambda) & flux=total(flux)
	pos2=pos2+reform(p(*,j,*)*flux,num+1,2)
	tflux=tflux+flux
endfor
pos2=pos2/tflux
dx=pos2(*,0)-pos1(*,0)
dy=pos2(*,1)-pos1(*,1)
;
if !d.name eq 'PS' and plotorbit_options.c then frame_color=0 else frame_color=1
!p.color=tci(frame_color)
if plotorbit_options.j then begin
	rho=sqrt(dx^2+dy^2)
	the=atan(dx,dy)
	the=precess(the,jd2jy(jd+2440000L),gen_model.ra,gen_model.dec,-1)
	dx=rho*sin(the)
	dy=rho*cos(the)
endif
oplot,dx,dy,psym=0,linestyle=linestyle
;
; Plot epoch of periastron
oplot,[dx(0),0],[dy(0),0],psym=0
;
return
;
; For AAS photo release
oplot,dx(40:1800),dy(40:1800),psym=0
;
; Kepler triangles
polyfill,[0,dx(1756:1800),0],[0,dy(1756:1800),0],color=tci(7)
; polyfill,[0,dx(856:900),0],[0,dy(856:900),0],color=tci(7)
polyfill,[0,dx(966:1010),0],[0,dy(966:1010),0],color=tci(7)
;
; Arrow stem
oplot,dx(0:30),dy(0:30),psym=0,thick=2,color=tci(2)
;
; Arrow head
x=[0,-0.9,4.9,-0.9,0]*0.2
y=[0,2.3,0,-2.3,0]*0.2
angle=atan(dy(29)-dy(30),dx(29)-dx(30))
xr=+x*cos(angle)+y*sin(angle)
yr=-x*sin(angle)+y*cos(angle)
usersym,xr,yr,/fill
oplot,[dx(30),dx(30)],[dy(30),dy(30)],psym=8,color=tci(2)
;
; Stellar disks
l=0.4
n=45
p=findgen(n+1)*2*pi_circle/n
x=l*sin(p)
y=l*cos(p)
polyfill,x,y,color=tci(5)
polyfill,x+dx(1800),y+dy(1800),color=tci(5)
; usersym,x,y,/fill
; oplot,[dx(1800),dx(1800)],[dy(1800),dy(1800)],psym=8,color=tci(5)
; oplot,[0,0],[0,0],psym=8,color=tci(5)
;
end
;-------------------------------------------------------------------------------
pro plotellipse
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if !d.window eq -1 and !d.name eq 'X' then begin
        print,'***Error(PLOTELLIPSE): no plot window currently open!'
        return
endif
if n_elements(e_parms) eq 0 then begin
	print,'***Error(PLOTELLIPSE): e_parms undefined!'
	return
endif
;
n=90
p=(findgen(n+1)/n)*2*pi_circle
;
r=ellipse(e_parms(2),e_parms(3),p-e_parms(4))
x=r*sin(p)+e_parms(0)
y=r*cos(p)+e_parms(1)
oplot,x,y
;
end
;************************************************************************Block 6
function init_plotvel
;
common PlotVel,plotvel_options
;
plotvel_options=alloc_plotvel_options()
;
!p.psym=1
;
end
;-------------------------------------------------------------------------------
pro plotvel,overplot=overplot
;
; Plots multiple star velocities.
;
; Some information (obsolete) on symbols and colors:
; !p.psym=0: no symbol, connect points with lines
;         1: +
;         2: *
;         3: .
;         4: Diamond
;         5: Triangle
;         6: Square
;         7: x
; New procedure for plot symbols: plotsymbol
;    open/filled
;	0/10: +
;	1/11: x
;	2/12: *
;	3/13: Triangle up
;	4/14: Triangle down
;	5/15: Square
;	6/16: Diamond
;	7/17: Pentagon
;	8/18: Star
;	9/19: Circle
;
; tek_color
; !p.color=0: black
;          1: white
;          2: red
;          3: green
;          4: blue
;          .
;          .
;         31: <last color index>
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common PlotVel,plotvel_options
common PlotVelDataInfo,plotscans,plotdata_x,plotdata_y
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common PlotVelLocal,pu,xu,yu,pl,xl,yl
;
if n_elements(overplot) eq 0 then overplot=0
;
; Check data
if n_elements(velocities) eq 0 then begin
	print,'***Error(PLOTVEL): no data!'
	return
endif
;
; Prepare PlotvelDataInfo
plotscans=velocities
;
; Set character size
!p.charsize=1.5
;
; Set margins, number of plots, and axis styles
!x.margin=[8.0,2.0]
!y.margin=[6.0,4.0]/!d.y_ch_size*!d.x_ch_size
if plotvel_options.r and plotvel_options.o then begin
	!p.multi=[0,1,2]
	ymargin1=[!y.margin(0)*0,!y.margin(1)]
	ymargin2=[!y.margin(0),!y.margin(1)*0]
	xstyle1=6
	xstyle2=2
	yfactor=1.5
endif else begin
	!p.multi=0
	ymargin1=!y.margin
	ymargin2=!y.margin
	xstyle1=2
	xstyle2=2
	yfactor=1
endelse
!y.style=1
;
; Open a plot window for plotting on the screen
if n_elements(gen_model) ne 0 then star=gen_model.starid else star=' '
if not overplot then begin
if !d.name ne 'PS' then begin
	window,/free,title=star,xsize=!xsize,ysize=!ysize, $
		xpos=!dxsize-!xsize,ypos=!dysize-!ysize
endif else begin
        device,color=plotvel_options.c
	tek_color
endelse
endif
if !d.name eq 'PS' and plotvel_options.c then frame_color=0 else frame_color=1
!p.color=tci(frame_color)
;
; Determine whether automatic scaling is requested
if total(abs(!x.range)) eq 0 then xrange_auto=1 else xrange_auto=0
if total(abs(!y.range)) eq 0 then yrange_auto=1 else yrange_auto=0
;
; Set data
index=where(velocities.component eq plotvel_options.component,count)
if count eq 0 then begin
	print,'***Error(PLOTVEL): no data for this component!'
	return
endif
t=velocities(index).jd
y=velocities(index).value
e=velocities(index).error
v=velocities(index).valuem
p=velocities(index).symbol
c=velocities(index).color;*(!d.name ne 'PS')
jndex=where(c eq 1,count)
if count gt 0 then c(jndex)=frame_color
plotscans=plotscans(index)
if plotvel_options.p and num_binary() gt 0 then begin
;	Compute orbital phase using lowest hierarchical level binary component
	ci=strpos(binary_model.component,plotvel_options.component)
	i=where(ci ge 0)
	components=binary_model(i).component
	periods=binary_model(i).period
	si=sort(strlen(components))
	components=components(si)
	periods=periods(si)
	j=where(periods ne 0) & j=j(0)
	i=where(binary_model.component eq components(j)) & i=i(0)
	x=(velocities(index).jd-binary_model(i).epoch)/binary_model(i).period $
		mod 1
	index=where(x lt 0,count)
	if count gt 0 then x(index)=x(index)+1
	xtitle='Phase'
;	Remove systemic velocity caused by top binary component
	s=modelvel(t,topbincomp())
	s=s(*,where(star_model.component eq plotvel_options.component))
	r=componentparse(topbincomp(),c1,c2)
	if strlen(c1) eq 1 and c1 eq plotvel_options.component then s=s*0
	if strlen(c2) eq 1 and c2 eq plotvel_options.component then s=s*0
	v=v-s
	y=y-s
endif else begin
	x=velocities(index).jd
	xtitle='Epoch'
endelse
ytitle1='Velocity [km/s]'
ytitle2='Residual velocity [km/s]'
title=wmcc(plotvel_options.component)
if plotvel_options.b then title=''
;
plotdata_x=x
plotdata_y=y
;
; Set and equalize ranges
if xrange_auto then !x.range=[min(x),max(x)]
if yrange_auto then !y.range=[min(y),max(y)]
;
; Set symbol
if plotvel_options.e then !p.psym=3 else !p.psym=1
;
; Plot data
index=where(e gt 0,count)
if count gt 0 then begin
	if not overplot then begin
	plot,x(index),y(index), $
		xtitle=xtitle,ytitle=ytitle1,title=title, $
		ymargin=ymargin1,xstyle=xstyle1,/nodata,xticklen=0.04
		pu=!p
		xu=!x
		yu=!y
	endif else begin
		!p=pu
		!x=xu
		!y=yu
	endelse
	for i=0,count-1 do begin
		plotsymbol,p(index(i))
		oplot,[x(index(i)),x(index(i))], $
		      [y(index(i)),y(index(i))], $
		      psym=8,color=tci(c(index(i)))
		if plotvel_options.p then $
		oplot,[x(index(i)),x(index(i))]+1, $
		      [y(index(i)),y(index(i))], $
		      psym=8,color=tci(c(index(i)))
	if plotvel_options.e then begin
		!p.color=tci(c(index(i)))
		oploterror,[x(index(i)),x(index(i))], $
		      [y(index(i)),y(index(i))], $
		      [x(index(i)),x(index(i))]*0, $
		      [e(index(i)),e(index(i))]
		if plotvel_options.p then $
		oploterror,[x(index(i)),x(index(i))]+1, $
		      [y(index(i)),y(index(i))], $
		      [x(index(i)),x(index(i))]*0, $
		      [e(index(i)),e(index(i))]
	endif
	endfor
endif
if plotvel_options.f then begin
	index=where(e le 0,count)
	if count gt 0 then begin
		oplot,x(index),y(index),color=tci(2)
		if plotvel_options.e then $
			!p.color=tci(2)
			oploterror,x(index),y(index),x(index)*0,e(index)
	endif
endif
;
; Plot dashed line corresponding to rv (systemic velocity)
if n_elements(gen_model) ne 0 then $
oplot,!x.crange,[gen_model.rv,gen_model.rv],linestyle=1, $
	color=tci(frame_color),psym=0
;
!p.color=tci(frame_color)
;
; Plot model
if plotvel_options.o then plotlehmann,plotvel_options.component
if plotvel_options.o then begin
;
;	Plot residuals
	if plotvel_options.r then begin
	if not overplot then $
	axis,xaxis=1,xtickname=replicate(' ',6),xticklen=0.04
	r=y-v
	index=where(e gt 0,count)
	if count gt 0 then begin
	if not overplot then begin
	plot,x(index),r(index),ymargin=ymargin2,xstyle=xstyle2,yrange=[0,0], $
		ytitle=ytitle2,xtitle=xtitle,/nodata,xticklen=0.04
		pl=!p
		xl=!x
		yl=!y
	endif else begin
		!p=pl
		!x=xl
		!y=yl
	endelse
	for i=0,count-1 do begin
		plotsymbol,p(index(i))
		oplot,[x(index(i)),x(index(i))], $
		      [r(index(i)),r(index(i))], $
		      psym=8,color=tci(c(index(i)))
		if plotvel_options.p then $
		oplot,[x(index(i)),x(index(i))]+1, $
		      [r(index(i)),r(index(i))], $
		      psym=8,color=tci(c(index(i)))
	endfor
	if plotvel_options.e then begin
		oploterror,x(index),r(index),x(index)*0,e(index)
		if plotvel_options.p then $
		oploterror,x(index)+1,r(index),x(index)*0,e(index)
	endif
	oplot,!x.crange,[0,0],color=tci(frame_color),psym=0
	endif
	endif
endif
;
; Reset plot variables
if xrange_auto then !x.range=0
if yrange_auto then !y.range=0
;
end
;-------------------------------------------------------------------------------
pro plotlehmann,component
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common PlotVel,plotvel_options
;
RAD=180/pi_circle
;
if n_elements(component) eq 0 then begin
	print,'***Error(PLOTLEHMANN): component undefined!'
	return
endif
if n_elements(binary_model) eq 0 then begin
	print,'***Error(PLOTLEHMANN): no model!'
	return
endif
if n_elements(linestyle) eq 0 then !p.linestyle=0
;
i=where(component eq star_model.component,count) & i=i(0)
if count eq 0 then begin
	print,'***Error(PLOTLEHMANN): component not in model!'
	return
endif
;
; Compute orbital phase using lowest hierarchical level binary component
ci=strpos(binary_model.component,plotvel_options.component)
i=where(ci ge 0)
components=binary_model(i).component
periods=binary_model(i).period
si=sort(strlen(components))
components=components(si)
periods=periods(si)
j=where(periods ne 0) & j=j(0)
i=where(binary_model.component eq components(j)) & i=i(0)
;
p=binary_model(i).period
if n_elements(velocities) ge 3 then t=median(velocities.jd)-p/2 $
			       else t=avg(velocities.jd)-p/2
; The epoch should be close to the median observing epoch
t=binary_model(i).epoch	; If you want to plot the periastron...
;
num=360
jd=dindgen(num)/num*p+t
;
x=(jd-binary_model(i).epoch)/binary_model(i).period mod 1
index=where(x lt 0,count)
if count gt 0 then x(index)=x(index)+1
;
v=modelvel(jd)
;
; Remove systemic velocity caused by top binary component
s=modelvel(jd,topbincomp())
r=componentparse(topbincomp(),c1,c2)
if strlen(c1) eq 1 and c1 eq plotvel_options.component then s=s*0
if strlen(c2) eq 1 and c2 eq plotvel_options.component then s=s*0
v=v-s
;
index=where(star_model.component eq component)
if plotvel_options.p then oplot,x,v(*,index),psym=0 $
		     else oplot,jd,v(*,index),psym=0
;
end
;************************************************************************Block 7
function init_plotmag
;
common PlotMag,plotmag_options
;
plotmag_options=alloc_plotmag_options()
;
!p.psym=1
;
end
;-------------------------------------------------------------------------------
pro plotmag
;
; Plots multiple star magnitudes.
;
; Some information on symbols and colors:
; !p.psym=0: no symbol, connect points with lines
;         1: +
;         2: *
;         3: .
;         4: Diamond
;         5: Triangle
;         6: Square
;         7: x
; tek_color
; !p.color=0: black
;          1: white
;          2: red
;          3: green
;          4: blue
;          .
;          .
;         31: <last color index>
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common PlotMag,plotmag_options
common PlotMagDataInfo,plotscans,plotdata_x,plotdata_y
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; Check data
if n_elements(magnitudes) eq 0 then begin
	print,'***Error(PLOTMAG): no data!'
	return
endif
;
; Prepare PlotvelDataInfo
plotscans=magnitudes
;
; Set character size
!p.charsize=1.5
;
; Set margins, number of plots, and axis styles
!x.margin=[8.0,2.0]
!y.margin=[6.0,4.0]/!d.y_ch_size*!d.x_ch_size
if plotmag_options.o then begin
	!p.multi=[0,1,2]
	ymargin1=[!y.margin(0)*0,!y.margin(1)]
	ymargin2=[!y.margin(0),!y.margin(1)*0]
	xstyle1=6
	xstyle2=2
	yfactor=1.5
endif else begin
	!p.multi=0
	ymargin1=!y.margin
	ymargin2=!y.margin
	xstyle1=2
	xstyle2=2
	yfactor=1
endelse
;
; Open a plot window for plotting
if n_elements(gen_model) ne 0 then star=gen_model.starid else star=' '
if !d.name ne 'PS' then begin
	window,/free,title=star,xsize=!xsize,ysize=!ysize, $
		xpos=!dxsize-!xsize,ypos=!dysize-!ysize
endif else begin
        width=17.78
        factor=width/640
        device,xsize=factor*!xsize,ysize=factor*yfactor*!ysize, $
                xoffset=1,yoffset=1
endelse
;
; Determine whether automatic scaling is requested
if total(abs(!x.range)) eq 0 then xrange_auto=1 else xrange_auto=0
if total(abs(!y.range)) eq 0 then yrange_auto=1 else yrange_auto=0
!x.style=2
!y.style=2
!x.margin=[8.0,2.0]
!y.margin=[6.0,4.0]/!d.y_ch_size*!d.x_ch_size
;
; Set data
index=where((magnitudes.component eq plotmag_options.component) $
        and (magnitudes.filter eq plotmag_options.filter),count)
if count eq 0 then begin
	print,'***Error(PLOTMAG): no data for this component!'
	return
endif
y=magnitudes(index).value
e=magnitudes(index).error
if plotmag_options.p then begin
	if num_binary() eq 0 then begin
		print,'***Error(PLOTMAG): no model!'
		return
	endif
	i=where(min(binary_model.period) eq binary_model.period)
	x=(magnitudes(index).jd-binary_model(i).epoch)/binary_model(i).period $
		mod 1
	index=where(x lt 0,count)
	if count gt 0 then x(index)=x(index)+1
	xtitle='Phase'
endif else begin
	x=magnitudes(index).jd
	xtitle='Epoch'
endelse
ytitle1='Magnitude, filter='+plotmag_options.filter
ytitle2='Residual magnitude, filter='+plotmag_options.filter
title='Component '+plotmag_options.component
;
plotdata_x=x
plotdata_y=y
;
; Set and equalize ranges
if xrange_auto then !x.range=[min(x),max(x)]
if yrange_auto then !y.range=[max(y),min(y)]
;
; Set symbol
if plotmag_options.e then !p.psym=3 else !p.psym=1
;
; Plot data
index=where(e gt 0,count)
if count gt 0 then begin
	plot,x(index),y(index),color=tci(1),xtitle=xtitle,ytitle=ytitle1,title=title
	if plotmag_options.e then oploterror,x(index),y(index),x(index)*0,e(index)
endif
if plotmag_options.f then begin
	index=where(e le 0,count)
	if count gt 0 then begin
		oplot,x(index),y(index),color=tci(2)
		if plotmag_options.e then $
			!p.color=tci(2)
			oploterror,x(index),y(index),x(index)*0,e(index)
	endif
endif
;
; Plot model
if plotmag_options.o then begin
	index=where((magnitudes.component eq plotmag_options.component) $
		and (magnitudes.filter eq plotmag_options.filter))
	v=magnitudes(index).valuem
	si=sort(x)
	oplot,x(si),v(si,0),psym=0
;
;	Plot residuals
	xs=!x.s & ys=!y.s
	xc=!x.crange & yc=!y.crange
	r=y-v
	index=where(e gt 0,count)
	plot,x(index),r(index),ymargin=ymargin2,xstyle=xstyle2,yrange=[0,0], $
		ytitle=ytitle2,xtitle=xtitle
	if plotmag_options.e then oploterror,x(index),r(index),x(index)*0,e(index)
	oplot,!x.crange,[0,0],color=tci(1),psym=0
	!x.s=xs & !y.s=ys
	!x.crange=xc & !y.crange=yc
endif
;
; Reset plot variables
if xrange_auto then !x.range=0
if yrange_auto then !y.range=0
!p.color=tci(1)
;
end
;************************************************************************Block 1
pro plotvolvox
;
common StarBase,StarTable,Notes
;
table=startable
index=where(startable.starid ne 'FKV0000',count)
;
rename_starids,'fkv-bsc'
rename_starids,'bsc-hip'
read_catalogs
ra0=startable(index).ra*15l*3600000l
dec0=startable(index).dec*3600000l
;
startable=table
ra=startable(index).ra*15l*3600000l
rae=startable(index).rae*15l*3600000l
dec=startable(index).dec*3600000l
dece=startable(index).dece*3600000l
;
dra=ra-ra0-avg(ra-ra0)
ddec=dec-dec0
;
if !d.name ne 'PS' then window,/free
!p.charsize=1.5
plot,dra,ddec,psym=3,xtitle='RA offset [mas]',ytitle='Dec offset [mas]', $
	title='Current - Hipparcos'
oplot,!x.crange,[0,0],psym=0,linestyle=1
oplot,[0,0],!y.crange,psym=0,linestyle=1
for i=0,count-1 do begin
	oplot,[dra(i)-rae(i)/2,dra(i)+rae(i)/2], $
	      [ddec(i),ddec(i)],psym=0
	oplot,[dra(i),dra(i)], $
	      [ddec(i)-dece(i)/2,ddec(i)+dece(i)/2],psym=0
endfor
if !d.name eq 'PS' then begin
	device,/close
	set_plot,!display
endif
;
end
;-------------------------------------------------------------------------------
