/* ---------------------------------------------------------------------
   Steps along the time series for the array phase screen realization.
   This does the division down from Coarse Steps to every other point -
   2Ro.

   This is gen-3.  It calculates iN2RoIWindow points (2-Ro spacing) on
   the initial call and uses iN2RoNWindow-1 on each subsequent request
   of the next (double) column of values.  Requests for Phase Screen
   values are made through the "xxxNeededRo" process and passed up
   stream until they can be satisfied. Each layer keeps what values it
   needs to calculate the next step at its level of resolution as well
   as the values not yet retrieved by the layer below ("xxxTakenRo").
   Each layer keeps a full set of values it needs for it's calculations.
   Here, that means we carry all the necessary 2-Ro tabulated values
   plus the iNAdvance Coarse Steps out ahead.
   ---------------------------------------------------------------------*/

#include <stddef.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include <fstream>

#include "npoi.h"

extern Seed S;
extern Array A;

int Get2RoStep(PhScreen * PS) {

  int iND1 = PS->iNDiam-1, iND = PS->iNDiam, iNA = A.iNArray, 
    i2NWI = PS->iN2RoIWindow, i2NW = PS->iN2RoNWindow;
  
  /* Ask for all the 2Ro values needed by Ro and the leading Coarse Step
     values, too. */
  PS->i2RoNeededRo = PS->iRoNeededRo;
  if (PS->iRoNeededRo < PS->iN2RoIWindow * 2) 
    PS->i2RoNeededRo = PS->iN2RoIWindow * 2;
  PS->i2RoNeededRo = ((PS->i2RoNeededRo/iND1) + PS->iNAdvance) * iND1;
  
  /* Send it up to make sure it's ready. */
  if (PS->i2RoNeededRo > PS->iCSLastCalcRo || PS->iCSLastCalcRo == 0) {
    if (GetCoarseStep(PS) == EXIT_FAILURE) return (EXIT_FAILURE);

  
  }
  /* Now, satisfy Ro's needs */
  while (PS->iRoNeededRo > PS->i2RoLastCalcRo) {

    /* Generate the ndim random variables for the first iN2RoIWindow time 
       steps of the time series.  This is the first window. */
    
    int i2RoC = PS->i2RoLastCalcRo;

    /* i2NWI = 1 should never happen! */
    if (i2RoC == 0 || i2NWI == 1) { 
      int nshort = 2 * PS->iNSteps - 1, newdim = PS->aiN2Ro[0], ix, iy;

      for (int iA = 0; iA < iNA; iA++) { // over the array

	/* Initial window.  Only Coarse Step values are in place, move
	   them in first. There should be "nshort" of them and their
	   locations in a 1-Ro resolved grid are stored in PS->aiRoXI
	   and PS->aiRoYI. */
	for (int is = 0; is < PS->iNSteps; is++) {
	  for (int jb = 0; jb < 2; jb++) { // Coarse Steps brackett mirrors
	    ix = is * iND1;
	    iy = jb * iND1;
	    int ix2 = ix/2;
	    int iy2 = iy/2;
	    PS->af2RoZayx[iA][iy2][ix2] = PS->afCSZayx[iA][jb][is];
	  }
	}
	
	/* Gaussian random variates, properly correleated and weighted */
	float afzp[newdim], afyp[newdim];
	for (int i = 0; i < newdim; i++) {
	  float x = gasdev(&S.lidum);
	  afyp[i]=x/sqrt(PS->ad2RoIFW[i]) ;
	}
	for (int i = 0; i < newdim; i++) {
	  afzp[i]=0. ;
	  for (int j = 0; j < newdim; j++){
	    afzp[i] += PS->ad2RoIFOth[i][j]*afyp[j] ;
	  }
	}

	/* Calculate the offsets to these based on the values already in
	   place. */
	float fdz0 = PS->af2RoZayx[iA][0][0]; // should be zero already!
	for (int inew = 0; inew < newdim; inew++) {
	  float fdum = 0.;
	  for (int is = 0; is < PS->iNSteps; is++) {
	    for (int jb = 0; jb < 2; jb++) {
	      if (is+jb > 0) {
		int i = is*2 + jb -1;
		iy = jb * iND1/2;
		ix = is * iND1/2;
		fdum += PS->ad2RoIEt[inew][i]*(PS->af2RoZayx[iA][iy][ix]-fdz0);
	      }
	    }
	  }
	  /* These were stored in Ro units */
	  ix = PS->ai2RoXI[inew];
	  iy = ((ix-(ix/2)*2)*iND + PS->ai2RoYI[inew])/2;
	  ix = ix/2;
	  PS->af2RoZayx[iA][iy][ix] = afzp[inew] + fdz0 - fdum;
	}
      }

      /* Tell GetCoarseStep: "Thank You" */
      PS->iCSLastTakenRo = (PS->iNSteps - 1) * iND1; // Units of Ro
      PS->i2RoLastCalcRo = PS->iN2RoIWindow * 2 - 1; // Ditto
    }
    
    /* This should never happen.  Complain. */

    else if (PS->i2RoLastCalcRo > 0 && PS->i2RoLastCalcRo < (i2NWI-1)*2) {
      cout << "The i2RoLastCalc index is in the middle of the"
	   << " initial 2Ro window!\n  This should never happen!!" 
	   << endl ;
      return (EXIT_FAILURE);
    }
    
    /* Do single time steps from now on, averaging over iN2RoWindow-1
       steps.*/
    
    else {
      
      /* Check whether there is room for the next value.  This does 2
         columns at a time. */
      if ((PS->i2RoLastCalcRo + 2 * PS->iNAdvance - PS->iRoFirstRo + 
	   2)/2 > PS->i2RoZMaxX) {
	cout << "Overfilled 2Ro Step Internal Storage!"
	     << "  Single Step Mode"  << endl
	     << "2RoLastCalcRo = " << PS->i2RoLastCalcRo << endl
	     << "2RoFirstRo = " << PS->i2RoFirstRo << endl
	     << "RoNeededRo = " << PS->iRoNeededRo << endl
	     << "Bailing Out!" << endl;
	return EXIT_FAILURE;
      }
      
      for (int iA = 0; iA < iNA; iA++) { 

	/* Prepare for the single step  calculation */
	int iXstart = PS->i2RoLastCalcRo - i2NW * 2 + 3 - PS->i2RoFirstRo;
	int iXstart2 = iXstart/2; // in 2Ro units
	iXstart = iXstart2 * 2; // start at beginning of 2Ro set
	int nshort = PS->aiN2Ro[1] - iND, inew = iND;
	int icase = (PS->i2RoLastCalcRo + 2) % iND1/2;
	
	/* Account for case where Coarse Step values are in the column
           that is being added. */
	if (icase == 0) { 
	  nshort += 2;
	  inew -= 2;
	}
	
	/* Bring in the next two values from Coarse Step if
           appropriate. */
	if (icase == 0) {
	  if (!iA) PS->iCSLastTakenRo += iND1; // Thank You, but only once!
	  int ix2 = (PS->iCSLastTakenRo - PS->i2RoFirstRo)/2; // for here
	  int ixcs = (PS->iCSLastTakenRo - PS->iCSFirstRo)/iND1; // for there
	  PS->af2RoZayx[iA][0][ix2] = PS->afCSZayx[iA][0][ixcs];
	  PS->af2RoZayx[iA][iND/2][ix2] = PS->afCSZayx[iA][1][ixcs];
	}
	
	float afzp[inew], afyp[inew];
	for (int i = 0; i < inew; i++) {
	  afyp[i]=gasdev(&S.lidum)/sqrt(PS->ad2RoFW[icase][i]) ;
	}
	for (int i = 0; i < inew; i++) {
	  afzp[i]=0. ;
	  for (int j = 0; j < inew; j++){
	    afzp[i] += PS->ad2RoFOth[icase][i][j]*afyp[j] ;
	  }
	}
	
	/* And, finally, add in the offsets */
	float fdz0 = PS->af2RoZayx[iA][0][iXstart2] ;
	for (int in = 0; in < inew; in++) {
	  float fdum = 0.;
	  for (int iold = 1; iold < nshort; iold++) {
	    int ix = PS->ai2RoX[icase][iold] + iXstart;
	    int iy = PS->ai2RoY[icase][iold];
	    /* Convert to 2Ro storage locations */
	    iy = ((ix - (ix/2)*2)*iND + iy)/2;
	    ix = ix/2;
	    fdum += PS->ad2RoEt[icase][in][iold-1]*
	      (PS->af2RoZayx[iA][iy][ix]-fdz0);
	  }
	  int ix = PS->ai2RoX[icase][nshort + in] + iXstart;
	  int iy = PS->ai2RoY[icase][nshort + in];
	  iy = ((ix - (ix/2)*2)*iND + iy)/2;
	  ix = ix/2;
	  PS->af2RoZayx[iA][iy][ix] = afzp[in] + fdz0 - fdum;
	}

      }
      PS->i2RoLastCalcRo += 2 ;
    }
  }
  
  /* HouseKeeping */
  /* Should we dump some stored values? */
  if (PS->i2RoLastTakenRo - i2NW*2 - iND1 - PS->i2RoFirstRo > iND1) { 
    /* Apparently we should! */
    //cout << "**** 2Ro Housekeeping! ***" << endl;
    for (int ia = 0; ia < iNA; ia++) {
      for (int iy = 0; iy < iND; iy++) {
	for (int is = iND1/2; is < PS->i2RoZMaxX; is++) {
	  PS->af2RoZayx[ia][iy][is-iND1/2] = PS->af2RoZayx[ia][iy][is];
	}
	for (int is = 0; is < iND1/2; is++) {
	  PS->af2RoZayx[ia][iy][PS->i2RoZMaxX-is] = 0.; // Keep things clean
	}
      }
    }
    PS->i2RoFirstRo += iND1;
  }
  return EXIT_SUCCESS;
}

