/* ---------------------------------------------------------------------
   Steps along the time series for the array phase screen realization.
   This does the lowest resolution steps, the Coarse Steps.  All array
   members are explicitly included.

   This is a "gen-3" calculation.  It does iNSteps steps, iND-1 in width
   and iNB = 2 high on the first calculation.  Then, when that last time
   step or beyond is requested it uses the single step method (but does
   iNA columns at a time) as many times as necessary until it has
   equaled or passed the requested time step.  Note that
   PS->iCSLastCalcRo is in Ro units.
   ---------------------------------------------------------------------*/

#include <stddef.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include <fstream>

#include "npoi.h"

extern Seed S;
extern Array A;

int GetCoarseStep(PhScreen * PS) {

  int IfWrite = 0; // debug!


  /* Things everyone needs. */
    int iNA = A.iNArray, iNS = PS->iNSteps, iNB = 2, // NBracket = 2
      iND = PS->iNDiam, iND1 = iND - 1, iNAB = iNA*iNB, iNS1 = iNS - 1; 

  while (PS->i2RoNeededRo > PS->iCSLastCalcRo) {

    /* Generate the ndim random variables for the first iNSteps of the
       time series.  This is the first window and hence works of the
       basic Covariance Matricies, directly. */

    int iCLC = PS->iCSLastCalcRo, iCLS = iCLC/iND1;

    if (iCLC == 0 || iNS == 1) {
      int ndim = iNS * iNAB - 1 ;
      float afzp[ndim], afyp[ndim];
      for (int i = 0; i < ndim; i++) {
	afyp[i]=sqrt(PS->adCSW[i])* gasdev(&S.lidum);
      }
      for (int i = 0; i < ndim; i++) {
	afzp[i]=0. ;
	for (int j = 0; j < ndim; j++){
	  afzp[i] += PS->adCSOth[i][j]*afyp[j] ;
	}
      }

      //dimension [a][y][x]
      PS->afCSZayx[0][0][0] = 0. ; // This defines z=0 for the whole shebang
      for (int ia = 0; ia < iNA; ia++) {
	for (int jy = 0; jy < iNB; jy++) {
	  for (int kx = 0; kx < iNS; kx++) {
	    int ijk = ia + (jy + kx * iNB) * iNA - 1 ;
	    if (ijk >= 0) PS->afCSZayx[ia][jy][kx] = afzp[ijk] ;
	  }
	}
      }
      PS->iCSLastCalcRo += (PS->iNSteps -1)* iND1 ;

      /* The rest of the handshaking should default to zero, which is
	 right. */
    }

    /* This should never happen.  Complain. */

    else if (iCLC > 0 && iCLC < iNS) {
      cout << "The iCoarseLastCalc index is in the middle of the"
	   << " initial Coarse window!\n  This should never happen!!" 
	   << endl ;
      return (EXIT_FAILURE);
    }
    

    /* Do single time steps from now on, taking account of the previous
       iNSteps-1 steps.*/

    else {

      float afzp[iNAB], afyp[iNAB];

      /* Check whether there is room for the next value. */
      if ((PS->iCSLastCalcRo - PS->iCSFirstRo + 1)/iND > PS->iCSZMaxX) {
	cout << "Overfilled Coarse Step Internal Storage!"
	     << "  Single Step Mode"  << endl
	     << "CSLastCalcRo = " << PS->iCSLastCalcRo << endl
	     << "CSFirstRo = " << PS->iCSFirstRo << endl
	     << "2RoNeededRo = " << PS->i2RoNeededRo << endl
	     << "Bailing Out!" << endl;
	return EXIT_FAILURE;
      }

      /* Get the new gaussian random variates, reweighted and correlated
         among themselves */
      for (int iab = 0; iab < iNAB; iab++) {
	afyp[iab]=gasdev(&S.lidum)/sqrt(PS->adCSFW[iab]) ;
      }
      for (int iab = 0; iab < iNAB; iab++) {
	afzp[iab]=0. ;
	for (int jab = 0; jab < iNAB; jab++){
	  afzp[iab] += PS->adCSFOth[iab][jab]*afyp[jab] ;
	}
      }

      int iCLS1 = (iCLC - PS->iCSFirstRo)/iND1 + 1;
      /* Then add in the offsets and place in the next columns.  In the
         process the (0,0), "fdz0", value has to be taken out and then
         put back in. */
      float fdz0 = PS->afCSZayx[0][0][iCLS1 - iNS1] ;

      /* ia, ib index the new [ia]ib] columns being added. */
      for (int ia = 0; ia < iNA; ia++) {
	for (int ib = 0; ib < iNB; ib++) {
	  int iab = ia + ib * iNA;
	  float fdum = 0. ;
	  /* ka, kb, ks go back and pick up the previously claculated
             iNS-1 sets of Z[ka][kb][ks], skipping the first one which
             is locally zeroed and pass them through FIxEt. */
	  for (int ka = 0; ka < iNA; ka++) {
	    for (int kb = 0; kb < iNB; kb++) {
	      for (int ks = 0; ks < iNS1; ks++) { 
		int js = ks + iCLS1 - iNS1;
		int kabs = ka + (kb + ks * iNB) * iNA - 1;
		if (kabs >= 0) {
		  fdum +=PS->adCSEt[iab][kabs]*(PS->afCSZayx[ka][kb][js]-fdz0);
		}
	      }
	    }
	  }
	  PS->afCSZayx[ia][ib][iCLS1] = afzp[iab] + fdz0 - fdum;
	}
      }
      PS->iCSLastCalcRo += iND1 ;
    }
  }

  /* Housekeeping */
  /* Should we dump some stored values? */
  if (PS->iCSLastTakenRo - iNS * iND1 - PS->iCSFirstRo > iND1) { 
    /* Apparently we should! */
    //cout << "*** Coarse Step Housekeeping! ****" << endl;
    int iCL = PS->iCSLastCalcRo;
    for (int ia = 0; ia < iNA; ia++) {
      for (int ib = 0; ib < iNB; ib++) {
	for (int is = 1; is < PS->iCSZMaxX; is++) {
	  PS->afCSZayx[ia][ib][is-1] = PS->afCSZayx[ia][ib][is];
	}
	PS->afCSZayx[ia][ib][PS->iCSZMaxX-1] = 0.; // Keep things clean
      }
    }
    PS->iCSFirstRo += iND1;
  }
  return EXIT_SUCCESS;
}

