/* ---------------------------------------------------------------------
   Steps along the time series for the array phase screen realization.
   For the moment it is the finest we subdivide for the 2-D calculation.

   This is a "gen-3" calculation.  It takes the necessary values from the
   2Ro calculation and fills in all the remaining values for Ro resolution
   on the initial calculation.  Then, when an Ro beyond that is requested
   it uses the single step method (but does 2Ro columns at a time) as many
   times as necessary until it has calculated the requested time step.
   ---------------------------------------------------------------------*/

#include <stddef.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include <fstream>

#include "npoi.h"

extern Seed S;
extern Array A;

int GetRoStep(PhScreen * PS) {

  int IfWrite = 0; // debug

  int iNA = A.iNArray, iNWI = PS->iNEffWindowI, iNW = PS->iNEffWindow, 
    iND = PS->iNDiam, iND1 = iND-1;

  /* Ask for all the values needed including the 2Ro leading values. */
  PS->iRoNeededRo = ((PS->iSubDNeededRo + 1)/2)*2;
  if (PS->iNEffWindowI > PS->iRoNeededRo) PS->iRoNeededRo = PS->iNEffWindowI;  
  PS->iRoNeededRo += PS->iNAdvance * 2 ;

  /* Send it up to make sure it's ready. */
  if (PS->iRoNeededRo > PS->i2RoLastCalcRo || PS->i2RoLastCalcRo == 0) {
    if (Get2RoStep(PS) == EXIT_FAILURE) return (EXIT_FAILURE);
  }


  /* Now, satisfy SubD's needs */
  while (PS->iRoLastCalcRo < PS->iSubDNeededRo) {

    /* Generate the ndim random variables for the first iNWindow time 
       steps of the time series.  This is the first window. */
    
    int iRoC = PS->iRoLastCalcRo;
    
    if (iRoC == 0 || iNWI == 1) {
      int newdim = PS->iNEffWindowI/2 * iND, 
	nshort = PS->aiNRo[0] - newdim, ix, iy;

      for (int iA = 0; iA < iNA; iA++) { // over the array

	/* Initial window.  By now Coarse and 2Ro values are in place,
	   move them in first. There should be "nshort" of them and their
	   locations in a 1-Ro resolved grid are stored in PS->aiRoXI and
	   PS->aiRoYI. */
	for (int i = 0; i < nshort; i++) {
	  ix = PS->aiRoXI[i];
	  iy = PS->aiRoYI[i];
	  int ix2 = ix/2;
	  int iy2 = (iy + iND * (ix - ix2 * 2)) / 2;
	  PS->afRoZayx[iA][iy][ix] = PS->af2RoZayx[iA][iy2][ix2];
	}
	
	/* Tell Get2RoStep: "Thank You" */
	PS->i2RoLastTakenRo = PS->iNEffWindowI - 2 + PS->iNAdvance * 2;
	
	/* Gaussian random variates, properly correleated and weighted */
	float afzp[newdim], afyp[newdim];
	for (int i = 0; i < newdim; i++) {
	  afyp[i]=gasdev(&S.lidum)/sqrt(PS->adRoIFW[i]) ;
	}
	for (int i = 0; i < newdim; i++) {
	  afzp[i]=0. ;
	  for (int j = 0; j < newdim; j++){
	    afzp[i] += PS->adRoIFOth[i][j]*afyp[j] ;
	  }
	}
	
	/* Calculate the offsets to these based on the values already in
	   place. */
	float fdz0 = PS->afRoZayx[0][0][0]; // should be zero already!
	for (int inew = 0; inew < newdim; inew++) {
	  float fdum = 0.;
	  for (int iold = 1; iold < nshort; iold++) {
	    ix = PS->aiRoXI[iold-1];
	    iy = PS->aiRoYI[iold-1];
	    fdum += PS->adRoIEt[inew][iold]*(PS->afRoZayx[iA][iy][ix]-fdz0);
	  }
	  ix = PS->aiRoXI[nshort + inew];
	  iy = PS->aiRoYI[nshort + inew];
	  PS->afRoZayx[iA][iy][ix] = afzp[inew] + fdz0 - fdum;
	}
      }
      PS->iRoLastCalcRo = PS->iNEffWindowI - 1;
    }
    
    /* This should never happen.  Complain. */
    
    else if (PS->iRoLastCalcRo > 0 && PS->iRoLastCalcRo < iNW) {
      cout << "The iRoLastCalc index is in the middle of the"
	   << " initial Ro window!\n  This should never happen!!" 
	   << endl ;
      return (EXIT_FAILURE);
    }
    
    /* Do single time steps from now on, averaging over iNWindow-1 steps.*/

    else {

      /* Check whether there is room for the next value.  This does 2
         columns at a time. */
      if (PS->iRoLastCalcRo + 2 * PS->iNAdvance - PS->iRoFirstRo + 2 > 
	  PS->iRoZMaxX) {
	cout << "Overfilled Ro Step Internal Storage!"
	     << "  Single Step Mode"  << endl
	     << "RoLastCalcRo = " << PS->iRoLastCalcRo << endl
	     << "RoFirstRo = " << PS->iRoFirstRo << endl
	     << "SubDNeededRo = " << PS->iSubDNeededRo << endl
	     << "Bailing Out!" << endl;
	return EXIT_FAILURE;
      }

      for (int iA = 0; iA < iNA; iA++) { // over the array

	/* Bring in the next two columns from Get2RoStep */
	int inext = PS->iRoLastCalcRo + PS->iNAdvance * 2 + 1;
	int ix2 = inext/2;
	for (int iy2 = 0; iy2 < iND; iy2++) {
	  int iy = iy2 * 2;
	  int ix = ix2 * 2;
	  if (iy > iND) {
	    ix++;
	    iy -= iND;
	  }
	  ix = ix - PS->iRoFirstRo;
	  int ix2d = ix2 - PS->i2RoFirstRo/2;
	  PS->afRoZayx[iA][iy][ix] = PS->af2RoZayx[iA][iy2][ix2d];
	}
	

	/* Prepare for the offset calculation */
	int iXstart = PS->iRoLastCalcRo - PS->iNEffWindow + 1 
	  - PS->iRoFirstRo;
	int nshort = PS->aiNRo[1] - iND, 
	  newdim = iND; // Here also doing 2 Ro columns at a time

	int inew = iND ; float afzp[inew], afyp[inew];
	for (int i = 0; i < inew; i++) {
	  afyp[i]=gasdev(&S.lidum)/sqrt(PS->adRoFW[i]) ;
	}
	for (int i = 0; i < inew; i++) {
	  afzp[i]=0. ;
	  for (int j = 0; j < inew; j++){
	    afzp[i] += PS->adRoFOth[i][j]*afyp[j] ;
	  }
	}
	
	/* And, finally, add in the offsets */
	float fdz0 = PS->afRoZayx[0][0][iXstart] ;
	for (int inew = 0; inew < newdim; inew++) {
	  float fdum = 0.;
	  for (int iold = 0; iold < nshort; iold++) {
	    int ix = PS->aiRoX[iold] + iXstart;
	    int iy = PS->aiRoY[iold];
	    fdum += PS->adRoEt[inew][iold]*(PS->afRoZayx[iA][iy][ix]-fdz0);
	  }
	  int ix = PS->aiRoX[nshort + inew] + iXstart;
	  int iy = PS->aiRoY[nshort + inew];
	  PS->afRoZayx[iA][iy][ix] = afzp[inew] + fdz0 - fdum;
	}
      }

      /* Say "Thank You" */
      PS->i2RoLastTakenRo += 2;
      PS->iRoLastCalcRo += 2; // Step it!
    }
  }

  /* HouseKeeping */
  /* Should we dump some stored values? */
  if (PS->iRoLastTakenRo - PS->iNEffWindow - PS->iRoFirstRo > iND1) { 
    /* Apparently we should! */
    //cout << "*** Ro Housekeeping! ****" << endl;
    int iRLC = PS->iRoLastCalcRo;
    for (int ia = 0; ia < iNA; ia++) {
      for (int iy = 0; iy < iND; iy++) {
	for (int is = iND1; is < PS->iRoZMaxX; is++) {
	  PS->afRoZayx[ia][iy][is-iND1] = PS->afRoZayx[ia][iy][is];
	}
	for (int is = 0; is < iND1; is++) {
	PS->afRoZayx[ia][iy][PS->iRoZMaxX-is] = 0.; // Keep things clean
	}
      }
    }
    PS->iRoFirstRo += iND1;
  }
  return EXIT_SUCCESS;
}
