/* ---------------------------------------------------------------------
   This generates the wavefront map for a finite, "multi-Ro", aperture
   at some time offset, fTOff, from the nominal end of the dither,
   iT * PS->fDelT.  The original calculation done with a zero Ro model
   (version 2.2) should be reproduced when the map dimensions are set
   such that AP.iNx * AP.iNy <= 1.

   The routine formerly of this name (now called SubdivideRo) now just
   generates the wavefront map on the fixed Ro/8 grid.  This routine
   controls the interpolations to evaluate the piston values at the
   various time offsets for the points specified AP.afXYOffset.

   "iA" is the index for which aperture is involved.  

   "* afWF" is the vector of pointers to the array for the wavefront 
      offset map for the iA aperture.  Units are nm.

   "* afDWFX" and "afDWFY" are the vectors of pointers to the arrays for
      the wavefront slopes (a freebe or nearly so when evaluating the
      splines).  Units are mu/m. 
----------------------------------------------------------------------*/

#include <stddef.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include <fstream>

#include "npoi.h"

extern Spline SP, SPINIT; 
extern PhScreen PS;
extern Array A;
extern Seed S; // Needed to pass around the RNG seed.
extern Aperture AP;

int GetWaveFront(int iA, float fTOff, int iT, float* afWF, float* afDWFX, 
		 float* afDWFY) {


  /* A call with zero signals the beginning of a new simulation.  It
     resets the variables and otherwise cleans things up.  It also calls
     up through the construction tree so that the initializations for
     the new simulation can take place.

     But it does not do any of the pretabulations, since those were done
     the first time */

  static float fDRORo = PS.fDelR/PS.fRo, h = PS.fRo/PS.iNSubdivide;

  /* Sanity check.  "fTOff" must always be negative.  iT represents the
     time index *at the end* of the time interval! */

  if (fTOff > 0.) {
    cout << "Time offset from the time index, ""iT"", MUST BE <= 0!" 
	 << endl;
    return EXIT_FAILURE;
  }

  /* We can pretabluate a lot about the "y" sampling - and will. */
  /* Need "constant" dimensions - *KEEP THAT IN MIND* */
  static int kyi[100]; static float wtll[100]; double fy[100], dx;
  if (AP.iNxy > 100) 
    cout << "In GetWaveFront: AP.iNxy = " << AP.iNxy 
	 <<" exceeds internal storage limit of 100. " << endl
	 << "This is a warning, but you're going to get into trouble...!"
	 << endl;

  if (iT == 0) {
    /* Reset everything.  Sets WFNeededRo = 1. */
    if (Reset(&PS, &A) == EXIT_FAILURE) return(EXIT_FAILURE);
    if (SubdivideRo(&PS) == EXIT_FAILURE) return(EXIT_FAILURE);
    /* Pretabulate some stuff for the "y" linear interpolation */
    for (int n = 0; n < AP.iNxy; n++) {
      fy[n] = (AP.afXYOffset[n][1] + (PS.iNDiam-1) / 2 * PS.fRo)/h; // Ro's
      wtll[n] = modf(fy[n], &dx);
      kyi[n] = (int) dx ;
    }
    return(EXIT_SUCCESS);
  }

  // Check to see if you need to move the screen into a new Ro interval
  /* One beyond outside Ro */
  PS.iWFNeededRo = (int) ((AP.fDiameter + PS.fDelR * iT)/PS.fRo) + 1; 
  
  if (PS.iWFNeededRo > PS.iSubDLastCalcRo) {
    if (SubdivideRo(&PS) == EXIT_FAILURE) return(EXIT_FAILURE);
  }

  /* Feed off the array created by Subdivide. */
  {int ix, jx, ky, ly; float wti, wtj, wtk, wtl, x, y, syl, syk, sxi, sxj;
  float xOff =((float) (iT + fTOff)) * PS.fWVel * PS.fDelT;
  for (int iNxy = 0; iNxy < AP.iNxy; iNxy++) {
    /* Get locations in meters.  Offset relative to center of mirrors
       and relative to stored data window. */
    x = (xOff + AP.afXYOffset[iNxy][0] + AP.fDiameter / 2. - 
      PS.iSubDFirstRo * PS.fRo)/h;
    /* Ro indices */
    ix = (int) x ;
    jx = ix + 1 ;
    wtj = (x - ix) ;
    wti = 1.-wtj;
    ky = kyi[iNxy];
    ly = ky + 1;
    wtl = wtll[iNxy];
    wtk = 1.-wtl;
    sxi = wtk * PS.afSubDZayx[iA][ky][ix] + wtl * PS.afSubDZayx[iA][ly][ix];
    sxj = wtk * PS.afSubDZayx[iA][ky][jx] + wtl * PS.afSubDZayx[iA][ly][jx];
    syk = wti * PS.afSubDZayx[iA][ky][ix] + wtj * PS.afSubDZayx[iA][ky][jx];
    syl = wti * PS.afSubDZayx[iA][ly][ix] + wtj * PS.afSubDZayx[iA][ly][jx];
    afWF[iNxy] = wti * sxi + wtj * sxj;
    afDWFX[iNxy] = (sxj-sxi) / h;
    afDWFY[iNxy] = (syl-syk) / h;
  }
  } // scope variables

  /* Thank Subdivide */
  /* Unlike others, this accounts for the trailing edge of the mirror (plus
     a little).  We don't free up values until the apertures have
     completely moved off that Ro interval. */
  PS.iSubDLastTakenRo = (int) ((iT-1) * fDRORo);

  return (EXIT_SUCCESS);

}


