/* Initializes the matrices that will generate the correlated random
   variables for the initial window at a 2Ro spacing and the matrices to
   step the window along, 2Ro's at a time. */

#include <stddef.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include <fstream>

#include "npoi.h"
extern PhScreen PS;
extern Array A;
extern Seed S;
extern Aperture AP;

int Init2RoArray() {

  //cout << "In Init2RoArray" << endl;
  int IfWrite = 0; // for diagnostic output
  /* Set up the covariance matrix for the initial window.  That involves
     knowing where the CoarseStep values are for this Aperture and then
     evaluating what the values are at the 2Ro spacing in the initial
     window, first.  */

  int iND = PS.iNDiam, iND1 = iND-1; // useful abbreviations
  /* As before, we load the positions of the known points first, then
     the points being added.  This allows the conditional matrices to be
     easily defined. */

  /* Set up Coarse Steps locations, units of Ro, x axis along wind
     direction. */
  for (int ix = 0; ix < PS.iNSteps; ix++) {
    for (int iy = 0; iy < 2; iy++) {
      PS.aiCoarseX[iy][ix] = ix * iND1 ; 
      PS.aiCoarseY[iy][ix] = iy * iND1 ;
    }
  }
  

  /**************************/
  /* For the Initial Window */
  /**************************/

  /* All the 2Ro points, excluding those that will be filled in by known
     CoarseStep values. */ 
  {int iN2Ro0=0; /* accumulates to == the number of 2Ro points exclusive 
		  of those at CoarseStep points. */
  for (int ix = 0; ix < PS.iN2RoIWindow * 2; ix++) {
    for (int iy = 0; iy < iND; iy++) {
      int jxy = iy + ix * iND; //running index
      /* Use if even and if not a CoarseStep point, ie., not if ix is an
	 integral number of iNDiameters and iy is not at the bottom or
	 top. */
      if ((jxy/2)*2 == jxy) { 
	if (((ix/iND1)*iND1 != ix) || ((iy/iND1)*iND1 != iy)){
	  PS.ai2RoXI[iN2Ro0] = ix;
	  PS.ai2RoYI[iN2Ro0] = iy; 
	  iN2Ro0++; // increment only if used
	}
      }
    }
  }
  PS.aiN2Ro[0] = iN2Ro0; // Number of 2Ro values needed initially
  } // iN2Ro0 scope
  

  /* addum[][] now replaces PS.ad2RoICov[][] because we don't think its
     ever needed!  We killed it in npoi.h, too!! */
  { int i2NSteps = 2 * PS.iNSteps;
  int ndim = PS.aiN2Ro[0] + i2NSteps - 1; // finite aperture case if got here
  double addum[ndim][ndim];
  /* */

  /* Note: we always explicitly skip the first (CoarseStep) point since
     the calculation is zeroed to it, and it's position is by definition
     (0,0). */
  int ixa,ixb,iya,iyb; double dxa, dya, dxb, dyb;
  for (int ia = 1; ia <= ndim; ia++) { 
    if (ia < i2NSteps) {
      int ia1 = ia/2;
      int ia2 = ia - ia1*2;
      ixa = PS.aiCoarseX[ia2][ia1];
      iya = PS.aiCoarseY[ia2][ia1];
    }
    else {
      ixa = PS.ai2RoXI[ia-i2NSteps];
      iya = PS.ai2RoYI[ia-i2NSteps];
    }
    dxa = ixa;
    dya = iya;
    double dxa2ya2p = StructFnRo(sqrt(dxa*dxa+dya*dya)) ;
    for (int jb = 1; jb <= ndim; jb++) { 
      if (jb < i2NSteps) {
	int jb1 = jb/2;
	int jb2 = jb - jb1*2;
	ixb = PS.aiCoarseX[jb2][jb1];
	iyb = PS.aiCoarseY[jb2][jb1];
      }
      else {
	ixb = PS.ai2RoXI[jb-i2NSteps];
	iyb = PS.ai2RoYI[jb-i2NSteps];
      }
      if (ia == jb) {
	//PS.ad2RoICov[ia-1][ia-1] = dxa2ya2p ;
	addum[ia-1][ia-1] = dxa2ya2p ;
      }
      else {
	double dxb = ixb ;
	double dyb = iyb ;
	double dxb2yb2p = StructFnRo(sqrt(dxb*dxb+dyb*dyb)) ;
	double dxab = dxa - dxb ;
	double dyab = dya - dyb ;
	double dxab2yab2p = StructFnRo(sqrt(dxab*dxab+dyab*dyab)) ;
	addum[ia-1][jb-1] = (dxa2ya2p+dxb2yb2p-dxab2yab2p)/2. ;
	//PS.ad2RoICov[ia-1][jb-1] = addum[ia-1][jb-1] ;
      }
    }
  }


  /* Now, diagonalize the matrix. */

  int nshort = i2NSteps - 1 ; // M*(N-1)-1 in paper notation
  int M = ndim - nshort;

  double ** ap2RoCov, ** ap2RoOth;
  ap2RoCov=(double **) malloc((unsigned) ndim*sizeof(double*)) ;
  for (int i=0; i < ndim; i++) ap2RoCov[i] = addum[i]; // Set pointers for call
  ap2RoOth=(double **) malloc((unsigned) ndim*sizeof(double*)) ;
  for (int i=0; i < ndim; i++) ap2RoOth[i] = PS.ad2RoIOth[i]; // and for return

  //cout << "Initial: DiagCov" << endl;
  if (DiagonalCov(ap2RoCov, ndim, IfWrite, ap2RoOth, PS.ad2RoIW) 
      == EXIT_FAILURE) return EXIT_FAILURE;
  
  
  /* Calculate the "E" and "F" matricies needed to fill in the 2Ro
     points not already calculated as CoarseSteps. */
  
  /* This is for initial calculation.  Hence, only CoarseStep values are
     known.  So the "thin" dimension is actually "thick". */

  /* Set pointers for matrices going in and coming back. */
  { double **apOth, **apFOth, **apF, **apEt; 
  apOth=(double **) malloc((unsigned) ndim*sizeof(double*));
  for (int i=0; i < ndim; i++) apOth[i] = PS.ad2RoIOth[i]; // Orthogonal in
  apFOth=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apFOth[i] = PS.ad2RoIFOth[i]; // Orthogonal in
  apF=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apF[i] = PS.ad2RoIF[i]; // Orthogonal in
  apEt=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apEt[i] = PS.ad2RoIEt[i]; // Orthogonal in

  if (FIxEt(apOth, PS.ad2RoIW, ndim, nshort, IfWrite, apF, apFOth, apEt, 
	    PS.ad2RoIFW) == EXIT_FAILURE) return (EXIT_FAILURE);
  }
  } // storage scoping


  /**************************************************/
  /* Now, single stepping after the initial window. */
  /**************************************************/

  /* All the 2Ro points as far back as iN2RoNWindow-1, including those
     that were filled in by Coarse Step values, are used *and* the
     iNAdvance pairs of Coarse Step values leading the calculation.  The
     latter are inserted before the last two (Ro) columns of 2Ro points.

     For the last two (Ro) column we need to be careful.  To deal with
     the situation where the first of these two columns contains known
     Coarse Steps, but otherwise unknown 2Ro values, we hand load the
     y=0 and iNDiam values, the locations where Coarse Steps could be,
     first.  Then the number of values to be solved for is just iNDiam
     or iNDiam-2, depending on whether Coarse Step values actually
     appear in that column.  The PS.ai2RoX,Y[][] vectors will allow you
     to map out the results, regardless of the order stored.

     Sadly, since the distance between the main body of 2Ro points and
     the "Advance" Coarse Step points varies, we have to provide for up
     to 6 (=iNDiam/2) specific cases.  We will take case = 0 to be when
     we are filling in a pair of columns which contain two known Coarse
     Step values, as described above.  */

  int iNCases = PS.iNDiam/2; 
  {int iN2Ro1, jxy;
  for (int icase = 0; icase < iNCases; icase++) {
    IfWrite = 0;
    iN2Ro1=0;
    
    /* First PS.iN2RoNWindow-1 sets of known 2Ro values. */
    for (int ix = 0; ix < (PS.iN2RoNWindow-1) * 2; ix++) {
      for (int iy = 0; iy < iND; iy++) {
	jxy = iy + ix * iND; // running index
	if ((jxy/2)*2 == jxy) { 
	  PS.ai2RoX[icase][iN2Ro1] = ix;
	  PS.ai2RoY[icase][iN2Ro1] = iy; 
	  iN2Ro1++; // increment only if used
	}
      }
    }
    
    /* Next, the iNAdvance pairs of Coarse Step values "leading" the
       calculation.  icase = 0 is when the two columns to be filled in
       have known Coarse Step values in the first column.  Then, the
       Advance Coarse Step values are farthest from the unknowns. 
       
       This is set up for the general case, assuming the index of the
       first point, the lower left, of the known 2Ro values is at (0,0)
       (and is the point to which the calculation is "zeroed"). */

    for (int ix = 0; ix < PS.iNAdvance; ix++) {
      int ixRo = (PS.iN2RoNWindow - icase - 1) * 2 + (ix+1) * iND1 ;
      for (int iy = 0; iy < 2; iy++) {
	PS.ai2RoX[icase][iN2Ro1] = ixRo ;
	PS.ai2RoY[icase][iN2Ro1] = iy * iND1; 
	iN2Ro1++; // increment only if used
      }
    }
    
    /* Now, rearrange the (possible) pair of coarse step points to be
       first for the last pair of Ro columns. */
    {int ix = PS.iN2RoNWindow * 2 - 2 ; // next to last Ro column
    PS.ai2RoX[icase][iN2Ro1] = ix;
    PS.ai2RoY[icase][iN2Ro1] = 0; 
    iN2Ro1++; 
    PS.ai2RoX[icase][iN2Ro1] = ix;
    PS.ai2RoY[icase][iN2Ro1] = iND1; 
    iN2Ro1++; 
    
    if (iND > 3) { // need to fill in the middle of this column
      for (int iy = 2; iy < iND - 2; iy++) {
	if ((iy/2)*2 == iy) { // can do this with iy only on even columns
	  PS.ai2RoX[icase][iN2Ro1] = ix;
	  PS.ai2RoY[icase][iN2Ro1] = iy;; 
	  iN2Ro1++; 
	}
      }
    }
    } // scope of ix
    
    {int ix = PS.iN2RoNWindow * 2 - 1; // last Ro column
    for (int iy = 0; iy < iND ; iy++) {
      jxy = iy + ix * iND; // running index
      if ((jxy/2)*2 == jxy) { 
	PS.ai2RoX[icase][iN2Ro1] = ix;
	PS.ai2RoY[icase][iN2Ro1] = iy; 
	iN2Ro1++; 
      }
    }
    PS.aiN2Ro[1] = iN2Ro1; // Number of 2Ro values currently known or needed
    } // scope of ix

  } // end iNCases
  } // variables scope
  
  /**********************************/
  /*  Now to fill in the matrices.  */
  /**********************************/

  {  int ndim = PS.aiN2Ro[1]-1; 
  double addum[ndim][ndim]; int iNDo2 = PS.iNDiam/2;

  /* Note: we explicitly skip the first point since the calculation is
     zeroed to it, and it's position is by definition (0,0). */
  double dxa, dya, dxb, dyb, dxa2ya2p, dxb2yb2p, dxab2yab2p, dxab, dyab;
  for (int icase = 0; icase < iNDo2; icase++) {
    for (int ia = 1; ia < PS.aiN2Ro[1]; ia++) { 
      dxa = PS.ai2RoX[icase][ia];
      dya = PS.ai2RoY[icase][ia];
      dxa2ya2p = StructFnRo(sqrt(dxa*dxa+dya*dya)) ;
      for (int jb = ia; jb < PS.aiN2Ro[1]; jb++) { 
	if (ia == jb) {
	  addum[ia-1][ia-1] = dxa2ya2p ;
	}
	else {
	  dxb = PS.ai2RoX[icase][jb];
	  dyb = PS.ai2RoY[icase][jb];
	  dxb2yb2p = StructFnRo(sqrt(dxb*dxb+dyb*dyb)) ;
	  dxab = dxa - dxb ;
	  dyab = dya - dyb ;
	  dxab2yab2p = StructFnRo(sqrt(dxab*dxab+dyab*dyab)) ;
	  addum[ia-1][jb-1] = (dxa2ya2p+dxb2yb2p-dxab2yab2p)/2. ;
	  addum[jb-1][ia-1] = addum[ia-1][jb-1]; // symmetric
	}
      }
    }
    
    /* Diagonalize the matrix.  We use the Jacobi Transformation method,
       which is not the fastest but reliable. */
    
    double ** ap2RoCov, ** ap2RoOth, ad2RoOth[ndim][ndim], ad2RoW[ndim];
    ap2RoCov=(double **) malloc((unsigned) ndim*sizeof(double*)) ;
    for (int i=0; i < ndim; i++) ap2RoCov[i] = addum[i]; // Set pointers 
    ap2RoOth=(double **) malloc((unsigned) ndim*sizeof(double*)) ;
    for (int i=0; i < ndim; i++) ap2RoOth[i] = ad2RoOth[i]; //  return
    
    if (DiagonalCov(ap2RoCov, ndim, IfWrite, ap2RoOth, ad2RoW) 
	== EXIT_FAILURE) return EXIT_FAILURE;
    
    
    /* Calculate the "E" and "F" matricies needed to fill in the 2Ro
       points not already calculated as CoarseSteps. */
    
    int nshort = ndim - PS.iNDiam ; // M*(N-1)-1 in paper notation
    if (icase == 0) nshort = nshort + 2; // Coarse Steps known
    int M = ndim - nshort ;
    
    /* Set pointers for matrices going in and coming back. */
    double **apOth, **apFOth, **apF, **apEt; 
    apOth=(double **) malloc((unsigned) ndim*sizeof(double*));
    for (int i=0; i < ndim; i++) apOth[i] = ad2RoOth[i]; // Orthogonal in
    apFOth=(double **) malloc((unsigned) M*sizeof(double*));
    for (int i=0; i < M; i++) apFOth[i] = PS.ad2RoFOth[icase][i];// Orthgnl out
    apF=(double **) malloc((unsigned) M*sizeof(double*));
    for (int i=0; i < M; i++) apF[i] = PS.ad2RoF[icase][i]; // F out
    apEt=(double **) malloc((unsigned) M*sizeof(double*));
    for (int i=0; i < M; i++) apEt[i] = PS.ad2RoEt[icase][i]; // FixEt out
    
    if (FIxEt(apOth, ad2RoW, ndim, nshort, IfWrite, apF, apFOth, apEt, 
	      PS.ad2RoFW[icase]) == EXIT_FAILURE) return (EXIT_FAILURE);
  } // end icase
  }

  return EXIT_SUCCESS;
}
