/* Initializes the Aperture specification for Finite Ro calculations. 

   NOTE: The Phase Screen structure must already be in place.  We need to
   know Ro and the (minimum) Ro window size. */

#include <stddef.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include <fstream>

#include "npoi.h"
Aperture AP; extern Aperture AP;
extern Array A;
extern PhScreen PS;

int InitAperture() {

  char cdum[100]; int iFbefore = 0;
  
  AP.iNxyMax = 100; // force this for the time being

  /* Make sure the Phase Screen info is already in.  We do this by checking
     for a finite Ro size, since it can't be zero. */
  if (!PS.fRo) {
    cout << "The call to InitAperture needs to be after the call to "
	 << "InitPhaseScreen!" << endl;
    return EXIT_FAILURE;
  }
  
  int IfWrite = 0; // for diagnostic output
  /* See if there's an Aperture sampling specification file. */
  ifstream fin("aperture") ; 
  if (fin.is_open()) {
    int k;
    fin >> cdum >> AP.iNxy 
        >> cdum >> AP.fDiameter ;
    if (AP.iNxy > AP.iNxyMax) {
      cout << "Aperture offset index exceeds the current maximum of "
	   << AP.iNxyMax << endl;
      return (EXIT_FAILURE) ;
    }
    for (int xy =0; xy < AP.iNxy; xy++) {
      fin >> k ; // index, let it override the ordering
      fin >> cdum >> AP.afXYOffset[k][0] ; // x offset
      fin >> cdum >> AP.afXYOffset[k][1] ; // y offset (meters) ;
      if (fin.eof()) { 
        cout << "APERTURE: offset dimension not correct!!"
             << endl ;
        return (EXIT_FAILURE) ;
      }
      if (xy != k && iFbefore == 0) { 
        cout << "warning (only once): "
	     << "Aperture offset indices are out of sequence.  "
             << endl ;
	cout << "I'll assume you know what you're doing, but they'd "
	     << "better all be there!" << endl;
	iFbefore = 1 ;
      }
    }
    fin.close() ;
  } 
  else {  // Initialize to some test value
    /* Initialize Aperture Structure.  Wavefront sampled over an aperture at
       most at 100 points.  Currently set for a single sample in the center
       of 10 cm mirrors. */
    AP.iNxy = 1;
    AP.fDiameter = 0.1 ; // meters
    for (int xy = 0; xy < AP.iNxy; xy++) {
      for (int k = 0; k < 2; k++) {
	AP.afXYOffset[xy][k] = 0.; // meters [ixy][dx,dy]
      }
    }
    /* In this case, write out what we've used. */
    ofstream fout("aperture") ;
    if (fout.is_open()) {
      fout << "NumOffsets " << AP.iNxy 
	   << " Diameter " << AP.fDiameter << endl ;
      for (int i = 0; i < AP.iNxy; i++) {
	fout << i ;
	fout << " dx " << AP.afXYOffset[i][0] 
	     << " dy " << AP.afXYOffset[i][1] << endl ;
      } 
    }
    fout.close() ;
  }

  /* Check if (x,y) offsets are all inside the aperture. */
  int iFerror = 0;
  for (int xy = 0; xy < AP.iNxy; xy++) {
    float x = AP.afXYOffset[xy][0];
    float y = AP.afXYOffset[xy][1];
    float r = AP.fDiameter/2;
    if (x*x + y*y > r*r) {
      cout << "InitAperture Calling!: Offset pair " << xy << ":(" 
	   << x << ", " << y 
	   << ") is outside the aperture, r = " << r << endl;
      iFerror = 1;
    }
  }
  if (iFerror) return (EXIT_FAILURE) ; 
  return EXIT_SUCCESS;
}
