/* Initializes the matrices that will generate the correlated random
   variables for the initial window at an Ro spacing and the matrices to
   step the window along, 2Ro's at a time. */

#include <stddef.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include <fstream>

#include "npoi.h"
extern PhScreen PS;
extern Array A;
extern Aperture AP;

int InitRoArray() {

  int IfWrite = 0; // for diagnostic output
  /* Set up the covariance matrix for the initial window.  In this case
     we are completely imbedded in 2Ro values and there are no special
     cases.  For convenience we do two whole columns at once.  This
     eliminates odd/even considerations (doing two), but gets very
     expensive for large mirrors (the whole columns). */

  int iND = PS.iNDiam, iND1 = iND-1; // useful abbreviations
  /* As before, we load the positions of the known points first, then
     the points being added.  This allows the conditional matrices to be
     easily defined. */

  /**************************/
  /* For the Initial Window */
  /**************************/

  /* First all the known, 2Ro, points. At even running index.  This
     includes the 2Ro points leading the Ro resolution area. */
  { int iNRo0 = 0, iNAll2Ro = PS.iNEffWindowI + PS.iNAdvance * 2;
  for (int ix = 0; ix < iNAll2Ro; ix++) {
    for (int iy = 0; iy < iND; iy++) {
      int jxy = iy + ix * iND; //running index
      if ((jxy/2)*2 == jxy) { 
	  PS.aiRoXI[iNRo0] = ix;
	  PS.aiRoYI[iNRo0] = iy; 
	  iNRo0++; // increment only if used
	}
      }
    }

  /* Then the unknown points in between - at odd running index. */
  int M=0;
  for (int ix = 0; ix < PS.iNEffWindowI; ix++) {
    for (int iy = 0; iy < iND; iy++) {
      int jxy = iy + ix * iND; //running index
      if ((jxy/2)*2 != jxy) { 
	PS.aiRoXI[iNRo0] = ix;
	PS.aiRoYI[iNRo0] = iy; 
	M++; iNRo0++; // increment only if used
      }
    }
  }


  PS.aiNRo[0] = iNRo0; // Number of Ro values needed initially
  int ndim = iNRo0 - 1; // finite aperture case if got here
  double addum[ndim][ndim];
  /* */
  /* Note: we always explicitly skip the first (CoarseStep) point since
     the calculation is zeroed to it, and it's position is by definition
     (0,0). */
  int ixa,ixb,iya,iyb; double dxa, dya, dxb, dyb;
  for (int ia = 1; ia <= ndim; ia++) { 
    dxa = PS.aiRoXI[ia];
    dya = PS.aiRoYI[ia];
    double dxa2ya2p = StructFnRo(sqrt(dxa*dxa+dya*dya)) ;
    for (int jb = 1; jb <= ndim; jb++) { 
      dxb = PS.aiRoXI[jb];
      dyb = PS.aiRoYI[jb];
      if (ia == jb) {
	addum[ia-1][ia-1] = dxa2ya2p ;
      }
      else {
	double dxb2yb2p = StructFnRo(sqrt(dxb*dxb+dyb*dyb)) ;
	double dxab = dxa - dxb ;
	double dyab = dya - dyb ;
	double dxab2yab2p = StructFnRo(sqrt(dxab*dxab+dyab*dyab)) ;
	addum[ia-1][jb-1] = (dxa2ya2p+dxb2yb2p-dxab2yab2p)/2. ;
      }
    }
  }


  /* Now, diagonalize the matrix. */

  int nshort = ndim - M ; // M*(N-1)-1 in paper notation

  double ** apCov, ** apOth, adIOth[ndim][ndim], adIW[ndim];
  apCov=(double **) malloc((unsigned) ndim*sizeof(double*)) ;
  for (int i=0; i < ndim; i++) apCov[i] = addum[i]; // Set pointers for call
  apOth=(double **) malloc((unsigned) ndim*sizeof(double*)) ;
  for (int i=0; i < ndim; i++) apOth[i] = adIOth[i]; // and for return

  //cout << "Initial: DiagCov" << endl;
  if (DiagonalCov(apCov, ndim, IfWrite, apOth, adIW) 
      == EXIT_FAILURE) return EXIT_FAILURE;
  
  /* Calculate the "E" and "F" matricies needed to fill in the 2Ro
     points not already calculated as CoarseSteps. */
  
  /* Set pointers for matrices going in and coming back. */
  double **apFOth, **apF, **apEt; 
  apFOth=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apFOth[i] = PS.adRoIFOth[i]; 
  apF=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apF[i] = PS.adRoIF[i]; 
  apEt=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apEt[i] = PS.adRoIEt[i]; 

  if (FIxEt(apOth, adIW, ndim, nshort, IfWrite, apF, apFOth, apEt, 
	    PS.adRoIFW) == EXIT_FAILURE) return (EXIT_FAILURE);
  } // M, iNRo scope  


  /********************************************/
  /* Now, the steps after the initialization. */
  /********************************************/


  /* First all the known points before the columns to be filled.  This
     is the area at full Ro resolution */
  { int iNRo1 = 0;
  for (int ix = 0; ix < PS.iNEffWindow; ix++) {
    for (int iy = 0; iy < iND; iy++) {
      PS.aiRoX[iNRo1] = ix;
      PS.aiRoY[iNRo1] = iy; 
      iNRo1++; // increment only if used
    }
  }

  /* Next, the 2Ro points in the 2*(iNAdvance+1) columns beyond the Ro
     resolved area. */
  int iNAll2Ro = PS.iNEffWindow + 2 * (PS.iNAdvance + 1);
  for (int ix = PS.iNEffWindow; ix < iNAll2Ro; ix++) {
    for (int iy = 0; iy < iND; iy++) {
      int jxy = iy + ix * iND; //running index
      if ((jxy/2)*2 == jxy) { 
	PS.aiRoX[iNRo1] = ix;
	PS.aiRoY[iNRo1] = iy; 
	iNRo1++; // increment only if used
      }
    }
  }


  /* Then the unknown points in the two columns off the end of the fully Ro
     resolved region - at odd running index. */
  int M=0;
  for (int ix =  PS.iNEffWindow; ix < PS.iNEffWindow+2; ix++) {
    for (int iy = 0; iy < iND; iy++) {
      int jxy = iy + ix * iND; //running index
      if ((jxy/2)*2 != jxy) { 
	PS.aiRoX[iNRo1] = ix;
	PS.aiRoY[iNRo1] = iy; 
	M++; iNRo1++; // increment only if used
      }
    }
  }


  PS.aiNRo[1] = iNRo1; // Number of Ro values needed initially

  int ndim = iNRo1 - 1; // finite aperture case if got here
  double addum[ndim][ndim];
  /* */

  /* Note: we always explicitly skip the first (CoarseStep) point since
     the calculation is zeroed to it, and it's position is by definition
     (0,0). */
  int ixa,ixb,iya,iyb; double dxa, dya, dxb, dyb;
  for (int ia = 1; ia <= ndim; ia++) { 
    dxa = PS.aiRoX[ia];
    dya = PS.aiRoY[ia];
    double dxa2ya2p = StructFnRo(sqrt(dxa*dxa+dya*dya)) ;
    for (int jb = 1; jb <= ndim; jb++) { 
      dxb = PS.aiRoX[jb];
      dyb = PS.aiRoY[jb];
      if (ia == jb) {
	addum[ia-1][ia-1] = dxa2ya2p ;
      }
      else {
	double dxb2yb2p = StructFnRo(sqrt(dxb*dxb+dyb*dyb)) ;
	double dxab = dxa - dxb ;
	double dyab = dya - dyb ;
	double dxab2yab2p = StructFnRo(sqrt(dxab*dxab+dyab*dyab)) ;
	addum[ia-1][jb-1] = (dxa2ya2p+dxb2yb2p-dxab2yab2p)/2. ;
      }
    }
  }

  /* Now, diagonalize the matrix. */

  int nshort = ndim - M ; // M*(N-1)-1 in paper notation

  double ** apCov, ** apOth, adNOth[ndim][ndim], adNW[ndim];
  apCov=(double **) malloc((unsigned) ndim*sizeof(double*)) ;
  for (int i=0; i < ndim; i++) apCov[i] = addum[i]; // Set pointers for call
  apOth=(double **) malloc((unsigned) ndim*sizeof(double*)) ;
  for (int i=0; i < ndim; i++) apOth[i] = adNOth[i]; // and for return

  //cout << "Initial: DiagCov" << endl;
  if (DiagonalCov(apCov, ndim, IfWrite, apOth, adNW) 
      == EXIT_FAILURE) return EXIT_FAILURE;
  
  
  /* Calculate the "E" and "F" matricies needed to fill in the 2Ro
     points not already calculated as CoarseSteps. */
  
  /* Set pointers for matrices going in and coming back. */
  double **apFOth, **apF, **apEt; 
  apFOth=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apFOth[i] = PS.adRoFOth[i];
  apF=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apF[i] = PS.adRoF[i]; 
  apEt=(double **) malloc((unsigned) M*sizeof(double*));
  for (int i=0; i < M; i++) apEt[i] = PS.adRoEt[i]; 

  if (FIxEt(apOth, adNW, ndim, nshort, IfWrite, apF, apFOth, apEt, 
	    PS.adRoFW) == EXIT_FAILURE) return (EXIT_FAILURE);
  } // M, iNRo1 scope

  return EXIT_SUCCESS;
}
