/* Sets up the data structures needed for the NPOI simulations */

// RNG Seed
struct Seed { 
  Seed (long int LS = 0L) {lseed=LS;} ;
  long int lseed; // Random Number Generator seed
  long int lidum; //Continuing argument for Random Number Generator
} ; 

// Array geometry.  No default initialization (no constructor)
//  Array();
struct Array {
  int iNArray; // How many elements in the Array 
  float afx[6]; // x_i (E-W) positions of elements (meters)
  float afy[6]; // y_i (N-S) positions of elements (meters)
} ;

// Phase Screen parameters.
//  PhScreen();
struct PhScreen {
  float fWVelX; // X component of Screen (wind) velocity
  float fWVelY; // Y component of Screen (wind) velocity
  float fWVel; // Total velocity
  float fDelT; // Sample time spacing
  float fDelR; // Sample spacing = WVel * fDelT
  float fRo; // Ro at 550nm
  float fDelTRo; // Time to move one Ro - calculated from above.
  int iNAdvance; /* How far should this level of resolution be ahead of
                    the next finer resolution.  For the moment, this is
                    set = 3 in InitArray.  Going beyond that affects the
                    dimensions, below. */
  int iNSubdivide; /* Expect =4, to provide Wave Fronts on Ro/4 grid. */


  int iNDiam; // How many Ro's to bracket the mirrors - <= 13
  double fOutline; /* Distance between the outer two points bracketing
                      mirrors. */

  /* To get an idea of what we're doing, below is the case for one 0.3m
     mirror.  First, you want the mirror to be cleanly straddled by
     calculated Phase Screen points.  That means a full width of the
     path of 4Ro (assuming Ro = 0.1m) and that in turn means 5 points to
     evaluate along a column (perpendicular to the wind direction).  In
     this case the basic layout of the initialization calculation has
     the Phase Screen evaluated up to 1-Ro resolution as follows:

             4o.x.o.x.o x o   o   o   o
             3.x.x.x.x x x x   
           Y 2x.x.x.x.x x x                  <- wind direction
             1.x.x.x.x x x x   
             0o.x.o.x.o x o   o   o   o
              0         1         2
              0123456789012345678901234
	                X

      In this, "o" represents Coarse Step values, "x" the 2-Ro values
      and "." the 1-Ro values as evaluated by the initial window
      calculations.  (X,Y) coordinates are in units of Ro.  The
      algorithm specifies that iNSteps=7 (of iNDiam-1 Ro units),
      iN2RoI=7 (of 2Ro units) and iNEffWindowI=8 (Ro units).  Note that
      the next points added at each level are well ahead (iNAdvance in
      their respective units) of where the next higher level of
      resolution is encountered.  That keeps the statistical properties
      of the grid intact with reasonable computational effort at each
      level while subdividing. */

  /* Coarse Steps */

  int iNSteps; // How many Diameter steps to carry (for array) <= 13

  /* Definition of Top (coarse) Window.  Assumes no more than 6
     telescopes, no more than 13 (coarse) time steps.  For a finite
     aperture a wavefront is calculated at each side of the mirror.
     Thus, in this case there will be at most 2 points for no more than
     6 telescopes for no more than 13 time steps.  Since we zero
     everything to one of the sample points, the dimensions below are
     2*6*13 - 1 = 155 */
  double adCSCov[155][155]; // Covariance matrix for initial window
  double adCSOth[155][155]; // Orthogonal transformation, diagonalizing Cov
  double adCSW[155]; // Eigenvalues for above

  /* Next, pretabulated matricies for time stepping the initial window
     along, one step at a time.  Again, these include the possibility
     two parallel tracks just outside a finite aperture.  And, again,
     this assumes one of the points on one of the aperture's tracks is
     zeroed for the calculation.  Hence, up to 12 new points need to be
     calculated (2*6) and up to 2*6*12 - 1 = 143 old points are held
     fixed. */
  double adCSEt[12][143]; // Stores F(inverse)*E(transpose), really
  double adCSF[12][12]; // Inverse of the cov mtx for the array (next step)
  double adCSFOth[12][12]; // Orthogonal transformation, as above
  double adCSFW[12]; // Eigenvalues, too.

  /* afCSZayx[a][y][x] holds the currently interesting values of the
     Wavefront offsets for the Coarse Steps.  a = iNArray, y = iNBracket
     and x = iNSteps.  These have maximum values of a = 6, y = 2 and x =
     20, where we've added some to x just in case. */
  float afCSZayx[6][2][20]; // Course Step Wavefront values.

  /* location of Coarse Step pts in units of Ro */
  int aiCoarseX[2][13], aiCoarseY[2][13]; 

  /* 2-Ro Steps */ 
  /* Only when iNDiam >= 3! */
  /* In general we next need to step down from step sizes of iNDiam-1
     for the coarest step to 2Ro steps.  We'll do this in one step, but
     caution that this can be expensive.  It should be reconsidered if
     you need to go at all beyond 11Ro mirrors. 

     The scaling is easy for even numbers of intervals per coarse step
     (iNDiam odd) but complicated for odd numbers of intervals (iNDiam
     even), so for the moment iNDiam is forced to be odd in
     InitAperture.

     In the philosophy of reducing the reach of the calculation as we
     increase the detail, we limit the dimension along the wind
     direction to be iNWindow long, and kept at least iNAdvance
     (currently = 3) complete 2Ro's ahead of where the 1Ro grid spacing
     has been generated.  The first attempt at calculating a 2Ro column
     into a coarse step block less than iNAdvance from the end will
     cause the next coarse step to be calculated.  All coarse steps
     ahead of the window as well as those in the window will be included
     in establishing the statistical biases as will previously
     calculated 2Ro values in the window. */

  /* Since some CoarseStep values will always be available, even the
     initial step at this stage of subdivision will be at least
     partially "conditional".  We will use all CoarseStep values for
     this Array element, up to 26 - 1 = 25 (2x13 for iNDiam = 3
     mirrors), and in the initial step expect to calculate up to iNDiam
     * (iNEffWindow/2 + NAdvance).  This can be up to 13*11 = 143.
     Adding that to the CoarseStep number the maximum dimension of this
     array will be 168. The size of the 2Ro windows, initial and in
     normal, are in PS.iN2Ro(I,N)Window, in 2Ro units along x. */
  int iN2RoIWindow, iN2RoNWindow;

  /* All the 2Ro points: (X,Y) indices == locations in Ro units.
     Initially ("I") excluding those that will be filled in by known
     CoarseStep values.  For later steps we have the case where next
     (known) Coarse Step values are at a variable distance from the
     points we are calculating and can even take displace two of the
     values we might want to calculate.  We handle this by putting the
     top and bottom indices of the first of those two (Ro) rows in
     before the remaining 2Ro positions.  Choosing between these two
     cases is then just a matter of saving out iNDiam or iNDiam-2 values
     for solving.

     Note that There are up to iNDiam/2 = 6 possible relative positions,
     which we must account for explicitly (sigh).  We choose the timing
     of these such that the zeroth case is the one where the 2Ro colums
     being filled are the ones with the *known* Coarse Step values, top
     and bottom.

     The number of positions needed, ie, "M", in each kind of array is
     given in PS.aiN2Ro[0,1]. */
  int ai2RoXI[143], ai2RoYI[143], ai2RoX[6][143], ai2RoY[6][143],
    aiN2Ro[2];

  /* af2RoZayx[a][y][x] holds the currently interesting values of the
     Wavefront offsets for the 2Ro steps.  a = iNArray, y = iNDiam
     and x = iN2Ro.  These have maximum values of a = 6, y = 13 and x =
     11. We have increased the x limit to 20 for safety. */
  float af2RoZayx[6][13][20]; //  Wavefront values at 2Ro level.

  /* Initial 2Ro Step */ 
  /* Don't need to save the covarience matrix! */
  //double ad2RoICov[168][168]; // Covariance matrix for initial window
  double ad2RoIOth[168][168]; // Orthogonal transformation, diagonalizing Cov
  double ad2RoIW[168]; // Eigenvalues for above
  /* Since this will always be a conditional calculation, we set it up
     directly for the initial window. In this case up to 26 - 1 Coarse
     Step values are known and we need up to 143 new 2Ro values. */
  double ad2RoIEt[143][25]; // Stores F(inverse)*E(transpose), really
  double ad2RoIF[143][143]; // Inverse of the cov mtx for the array (next step)
  double ad2RoIFOth[143][143]; // Orthogonal transformation, as above
  double ad2RoIFW[143]; // Eigenvalues, too.

  /* Stepping by 2Ro */
  /* For the single 2Ro steps there will be two possible cases,
     depending on whether the two column set of new values includes two,
     already calculated, Coarse Step values.  In each step we will use
     all 2Ro values already known in the (iN2Ro-1)*2 columns already
     known and the two Coarse Step values in this double column, if
     relevant.  The iNAdvance*2 Coarse Step values are always included.
     So we will have up to (iNEffWindow/2 + iNAdvance -1 = 10) * (iNDiam
     = 13) 2Ro values plus 2*3 = 6 leading Coarse Step values for a
     total of 136 - 1 = 135.  From those we need to calculate iNDiam <=
     13 new 2Ro values.  The generating covariance matrix for thus has a
     dimension of 148.  This is the normal case*/

  /* You only need one "generating" cov matrix for both cases, and you
     don't need to save it! */
  double ad2RoNOth[148][148]; // Orthogonal transformation, diagonalizing Cov
  double ad2RoNW[148]; // Eigenvalues for above

  /* Single step case: calculate NDiam new values for the next 2 Cols.*/
  /* "n" for normal */
  double ad2RoEt[6][13][135]; // Stores F(inverse)*E(transpose), really
  double ad2RoF[6][13][13]; // Inverse of the cov mtx for the array (next step)
  double ad2RoFOth[6][13][13]; // Orthogonal transformation, as above
  double ad2RoFW[6][13]; // Eigenvalues, too.


  /* Ro Steps */
  /* Used only if iNDiam >= 3! */
  int iNWindow; // Number in each Window (ie, n-1 values from before).
  int iNEffWindow; // Generalize above to finite mirror case
  int iNEffWindowI; // to initialize

  /* Compared to the above, this is relatively simple.  The initial step
     uses all known 2Ro values to fill in every other value.  By then,
     the Coarse Step values have been meshed with the initial 2Ro
     values, so no special logic is involved.  After the initial step,
     single stepping is done two columns at a time, thereby meshing
     seamlessly with the 2Ro values already calculated.  Again, no
     special logic is required. */

  /* afRoZayx[a][y][x] holds the currently interesting values of the
     Wavefront offsets for the Ro steps.  a = iNArray, y = iNDiam and x
     = 2iNDiam.  These have maximum values of a = 6, y = 13 and x =
     25. */
  float afRoZayx[6][13][50]; //  Wavefront values at Ro level.  dmWas 30

  /* Initial Ro Step */
  /* Maximum dimensions are: y = 13, known points and = 13 unknown
     points for every other point in two columns of a iNDiam = 13
     mirror, and x = 8 (iNEffWindow/2) 2Ro steps for the same case.  To
     the known points we need to add iNAdvance * iNdiam leading 2Ro
     points, for a total of 13 * (8 + 3)= 143 (-1) known points and 13 *
     8 = 104 unknowns.  Again, the generating covariance matrix, of
     dimension 247 - 1 = 246, does not need to be saved.  */

  double adRoIEt[104][142]; // Stores F(inverse)*E(transpose), really
  double adRoIF[104][104]; // Inverse of the cov mtx (next step)
  double adRoIFOth[104][104]; // Orthogonal transformation, as above
  double adRoIFW[104]; // Eigenvalues, too.

  /* The actual dimension of the whole arrays for initializing and
     single stepping are stored in iNRo[0,1], respectively. Note, these
     include the position of the first point, so these, minus one, are
     the matrix dimensions.  Also, storage for the (X,Y) locations in Ro
     units. */
  int aiRoXI[247], aiRoYI[247], aiRoX[143], aiRoY[143], aiNRo[2];

  /* Single Stepping Ro */
  /* The above is a bit of an overkill at this level of division.
     Instead we will opt to include only iNAdvance 2Ro columns before
     and after the 2Ro columns that need to be filled, at most.  All Ro
     level values are known in the columns before and there will
     therefore be 2 * iNAdvance * iNdiam = 6 * 13 = 78 values known at
     every Ro point.  To these must be added the known 2Ro points in the
     two columns to be filled and the 2 * iNAdvance columns ahead of
     that column, (iNAdvance + 1) * iNdiam = 4 * 13 = 52 for a total of
     130 known points.  Finally there are iNDiam unknowns in the 2Ro
     columns in question = 13.  So the generating covariance matrix has
     dimensions or 143 - 1 = 142.  (X,Y) storage is given above. */

  double adRoEt[13][129]; // Stores F(inverse)*E(transpose), really
  double adRoF[13][13]; // Inverse of the cov mtx for the array (next step)
  double adRoFOth[13][13]; // Orthogonal transformation, as above
  double adRoFW[13]; // Eigenvalues, too.

  /* SubDivide Ro */
  /* High Resolution ("Subdivided") Wave Front map. */
  /* For the moment all SubdivideRo does is transfer the Ro resolution
     map from afRoZayx (from GetRoStep) to afZayx where GetWaveFront can
     access it. */
  int iNSubDWindowI, iNSubDWindow; // Initial and incremental windows.

  /* afSubDZayx[a][y][x] contains the part of the wavefront map
     currently needed to continue the calculation, tabulated at the
     "iNSubdivide" level. Here a = 6 is the number of array elements, x
     = 120 comes from needing at least 2 full mirror diameters of data
     (plus a little) so that one roles values off the end, one Coarse
     Step at a time.  Finally, y = 49, which is iNDiam Ro values.  We
     fill this in with values from the Ro store, as needed.  In microns,
     Z[0][0][0]=0.*/
  /* NOTE: This will get substantially modified when we start
     superresolving the Ro tabulation. */
  float afSubDZayx[6][49][120]; // Wavefront map in current use at Ro level

  /* Handshaking among the screen building routines. */
  /* Indicators to the stepping routines of what is needed, what has
     been calculated and what has been already taken.  This will force
     them to do their next step, prevent them from dumping values too
     soon and let them dump when needed. */
  int i2RoNeededRo, iRoNeededRo, iSubDNeededRo, 
    iWFNeededRo;// Needed now
  /* last Ro calculated */
  int iSubDLastCalcRo, iRoLastCalcRo, i2RoLastCalcRo, iCSLastCalcRo; 
  /* first col. currently stored */
  int iWFFirstRo, iSubDFirstRo, iRoFirstRo, i2RoFirstRo, 
    iCSFirstRo; 
  /* Anything here or earlier can be dumped.*/
  int iSubDLastTakenRo, iRoLastTakenRo, 
    i2RoLastTakenRo, iCSLastTakenRo; 
  /* Maximum sizes provided for internal storage. Set in InitPScreen */
  int iCSZMaxX; // in iNDiam-1 units
  int i2RoZMaxX; // in 2-Ro units
  int iRoZMaxX; // in Ro
  int iSubDZMaxX; // The array at its finest and best.

  /* This sets up spline interpolation to get the wave front at any
     point in or near the Ro interval.  iZ[iA][Y][X]. */
  float afIZ[6][49][49]; // For subdivided points (carry 4 from previous).
  float afIDZX[6][49][49]; // For spline derivatives at the subdivided pts.
  float afIDZY[6][49][49]; // For spline derivatives at the subdivided pts.

  /* Wavefront of the array elements at tabulated points. */
  float afWF[6]; // This is it, the Wave Front.
  float afDWFX[6]; // And it's derivatives, it's almost free with splines.
  float afDWFY[6]; // 
} ;

// Spline derivatives and interpolation
struct Spline {
  /* Storage for Spline interpolation of the Ro/8 wavefront (in afIZ[][]
     array). */
  int iNSpline; // Index for interval
  float fH; // Size of interval, assumed constant (=Ro/8)
  float afDM[50]; // Max Dimension is just a guess
  float afDP[50];
  float afW[50];
  float afUI[50];
  float afB[50];
};

// Finite-Ro Apertures
struct Aperture {
  /* Storage for where the wavefront needs to be specified in the
     apertures.  These points are completely arbitrary, and are defined
     as offsets from the nominal aperture centers given in Array.  If
     iNxy <= 1, sample with no offset.  You initiate a real "multi-Ro"
     calculation only when iNxy > 1. */
  int iNxy; // number of (x,y) offset pairs
  int iNxyMax; // for sanity checks
  float fDiameter; // Aperture diameter (meters)
  float afXYOffset[100][2]; // (x, y) offsets from center (meters)
};

// Scintillation parameters and pretabulated matricies
struct Scintillation {
  /* This heavily uses the PhaseScreen parameters, since it needs to be
     current to the same time step.  However, there is a lot of
     geometry, so the flat file (and default) parameters are
     extensive. */

  /* And not yet implemented, as you can see. */
};  
  
// Prototypes

int SubdivideRo(PhScreen*) ;
int GetRoStep(PhScreen *);
int Get2RoStep(PhScreen *);
int GetCoarseStep(PhScreen *);
// above just for time being!
int FIxEt(double **, double *, int, int, int, double **,
	      double **, double **, double *) ;
int DiagonalCov(double **, int, int, double **, double * ) ;
int GetMultiRoData(int) ;
int GetWaveFront(int, float, int, float*, float*, float*) ;
int Init(long int) ;
int InitAperture() ;
int InitArray() ;
int InitCoarseArray() ;
int Init2RoArray() ;
int InitRoArray() ;
int InitGeometry() ;
int InitPhaseScreen() ;
int InitSpline() ;
int Reset(PhScreen*, Array*) ;
double StructFn(double, double) ;
double StructFnRo(double) ;
int sderup(Spline*) ;
int sderiv(Spline*, float*, float*) ;
float splintrp(float*, float*, int, float, float, float, float*) ;
float ran1(long*) ;
float gammln(float) ;
float poidev(float, long*) ;
float gasdev(long*) ;
// void jacobi(float **, int, float *, float **, int *) ;
void jacobi(double **, int, double *, double **, int *) ;
void gaussj(double **, int, double **, int) ;

// Defines
  #define TWOPI 6.28318530717958647692
  #define PI 3.14159265358979323846
  #define PI2 1.57079632679489661923 // PI/2
  #define PI4 0.78539816339744830962 // PI/4
  #define DEGRAD 180. / PI

// External Variables 
extern int NumPS; // Number of Phase Screens (turbulent layers) above Array 

