/* ---------------------------------------------------------
   Evaluates spline first derivatives at tabular points.

   Assumes parabolic run out.

   Assumes spline coefficients are pretabulated on tabular points.

   Assumes h[i] (del x) = const.  Then the middle coeff for the y's is
   identically 0.  We therefore kill do[i] and store do[1] in dm[1] and 
   do[N] in dp[N]. 
   --------------------------------------------------------- */

#include <stddef.h>
#include <stdlib.h>

#include <fstream>

#include "npoi.h"

int sderiv(Spline * Sp, float * y, float * dy) {
  int nx1 = Sp->iNSpline - 1; 
  int IfWrite = 0; float Dy[20];
  dy[0] = (Sp->afDM[0] * y[0] + Sp->afDP[0] * y[1]) / Sp->afUI[0];
  if (IfWrite) Dy[0] = Sp->afDM[0] * y[0] + Sp->afDP[0] * y[1]; // end IfWrite
  for (int j = 1; j < nx1; j++) {
    int jm=j-1;
    dy[j] = (Sp->afDM[j] * y[jm] + Sp->afDP[j] * y[j+1] - 
      Sp->afB[j] * dy[jm]) / Sp->afUI[j];
    if (IfWrite) Dy[j] = Sp->afDM[j] * y[jm] + 
		   Sp->afDP[j] * y[j+1]; // end IfWrite
  }
  dy[nx1] = (Sp->afDM[nx1] * y[nx1-1] + Sp->afDP[nx1] * y[nx1] - 
	     Sp->afB[nx1] * dy[nx1-1]) / Sp->afUI[nx1];
  if (IfWrite) Dy[nx1] = Sp->afDM[nx1] * y[nx1-1] + 
		 Sp->afDP[nx1] * y[nx1]; // end IfWrite
  for (int j = nx1-1; j >= 0; j--) {
    dy[j] = dy[j] - Sp->afW[j] * dy[j+1];
  }

  if (IfWrite) {
    cout << "i   Dy:" << endl;
    for (int i = 0; i <= nx1; i++) cout << i << " " << Dy[i] << endl;
  } // end IfWrite

  return(EXIT_SUCCESS);
}

/* ----------------------------------------------------------
   Pretabulate Spline derivative coefficients.

   Assumes parabolic runout and h[i]=const=h.  Lots of things simplify then,
   see "sderiv" comments. 

   Note, nx and h are part of the Spline structure, allowing us to pretab
   for different scales and numbers of points. 
   ----------------------------------------------------------*/

int sderup(Spline * Sp) {
  int nx1 = Sp->iNSpline - 1;
  Sp->afB[0] = 0.;
  Sp->afUI[0] = 2.;
  Sp->afW[0] = 1.;
  Sp->afDM[0] = -4. / Sp->fH;
  Sp->afDP[0] = 4. / Sp->fH;
  for (int j = 1; j < nx1; j++) {
    Sp->afB[j] = 0.5;
    Sp->afUI[j] = 2. - Sp->afW[j-1] * 0.5;
    Sp->afW[j] = 0.5 / Sp->afUI[j];
    Sp->afDM[j] = -1.5 / Sp->fH;
    Sp->afDP[j] = 1.5 / Sp->fH;
  }
  Sp->afB[nx1] = 2.;
  Sp->afW[nx1] = 0.;
  Sp->afUI[nx1] = 2. - Sp->afW[nx1-1] * 2.;
  Sp->afDM[nx1] = -4. / Sp->fH;
  Sp->afDP[nx1] = 4. / Sp->fH;

  int IfWrite = 0;
  if (IfWrite) {
    cout << " i   B   W   UI   DM   DP  h= " << Sp->fH << endl;
    for (int i = 0; i <= nx1; i++) {
      cout << i << " " 
	   << Sp->afB[i] << " "
	   << Sp->afW[i] << " "
	   << Sp->afUI[i] << " "
	   << Sp->afDM[i] << " "
	   << Sp->afDP[i] << endl;
    }
  } // end IfWrite

  return(EXIT_SUCCESS);
}

/* --------------------------------------------------------
   Interpolate the function (and derivative) using already derived cubic
   spline coefficients.

   Assumes external logic has already deduced the correct interval and 
   has provided the lower index (ie, [i, i+1]) as well as the fractional
   location of the value (ie, fp = (x - x[i])/h).

   The function returns the interpolated value, or zero if it fails.
   ---------------------------------------------------------*/

float splintrp(float *y, float *dy, int i, float h, float fp, 
	       float fm, float *cdy) {

  if (h == 0.) {
    cout << "H=0 in spline interpolation!" << endl;
    return(0.);
  }
  float fm2 = fm * fm;
  float fp2 = fp * fp;
  float Ax = fm2 * (1. + 2. * fp);
  float Bx = fp2 * (2. * fm + 1.);
  float Cx = h * fm2 * fp;
  float Dx = - h * fp2 * fm;
  float Apx = - 6. * fm * fp / h;
  //float Bpx = -Apx;
  float Cpx = fm * (1. - 3. * fp);
  float Dpx = fp * (1. - 3. * fm);
  *cdy = Apx * (y[i] - y[i+1]) + Cpx * dy[i] + Dpx * dy[i+1];
  return(Ax * y[i] + Bx * y[i+1] + Cx * dy[i] + Dx * dy[i+1]);
}
