/** ========================================================
  atmosphere.cpp

  This is a wrapper for Deane's atmosphere code.
  ========================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "memory.h"
#include "config.hpp"
#include "atmosphere.hpp"
#include "../atmos/npoi.h"
#include "display.hpp"

#define NUM_STATIONS 6
#define BUFFER_SIZE 5000

/*  the interface to deane's code */

int Init(long int) ;
int GetWaveFront(int iA, float fTOff, int iT, float* afWF, float* afDWFX, 
		 float* afDWFY) ;
float *power ( int iPts, float fDeltaT, float *pfData ) ;

/** =============================
    atmosphere constructor
    =============================*/
atmosphere::atmosphere ( config *pConfig )
{
  FILE *pF ;
  int i ;
  float *pfStnX, *pfStnY, fApDia, *pfApX, *pfApY, fVelX, fVelY, fR0 ;
  printf ( "atmosphere: Initialization begins.\n" ) ;

  /* Get the station configuration from the configuration class */

  iStations = pConfig->getStations() ;
  pfStnX = alloc1df ( iStations ) ;
  pfStnY = alloc1df ( iStations ) ;
  pConfig->getStationList ( pfStnX, pfStnY ) ;

  /* and write the array geometry file */

  pF = fopen ( "geometry", "w" ) ;
  if ( pF == NULL )
    {
      printf ( "cannot write array geometry file\n" ) ;
      exit (1) ;
    }
  fprintf ( pF, "NumInArray %2d\n", iStations ) ;
  for ( i = 0; i < iStations ; i++ )
    fprintf ( pF, "%2d x %12.6f y %12.6f\n", i, pfStnX[i], pfStnY[i] ) ;
  fclose ( pF ) ;
  free (pfStnX) ;
  free (pfStnY) ;


  /* Get the aperture configuration:  Diameter, number and sample position */

  fApDia  = pConfig->getApSize() ;
  iNumPts = pConfig->getNumApPts() ;
  pfApX   = alloc1df ( iNumPts ) ;
  pfApY   = alloc1df ( iNumPts ) ;
  pConfig->getApPtList ( pfApX, pfApY ) ;

  /* and write the aperture configuration file */

  pF = fopen ( "aperture", "w" ) ;
  if ( pF == NULL )
    {
      printf ( "cannot write aperture file\n" ) ;
      exit (1) ;
    }
  fprintf ( pF, "NumOffsets %3d  Diameter %7.3f\n", iNumPts, fApDia ) ;
  for ( i = 0; i < iNumPts ; i++ )
    fprintf ( pF, "%2d dx %12.6f dy %12.6f\n", 
	     i, fApDia*pfApX[i], fApDia*pfApY[i] ) ;
  fclose ( pF ) ;
  free (pfApX) ;
  free (pfApY) ;

  /* file "screen" contains a description of the atmosphere and sets the 
     position of the points that need to be calculated */

  pConfig->getAtmos( &fVelX, &fVelY, &fR0) ;
  fDeltaT = fR0/sqrt(fVelX*fVelX + fVelY*fVelY) ;
  pF = fopen ( "screen", "w" ) ;
  if ( pF == NULL )
    {
      printf ( "cannot write atmosphere screen file\n" ) ;
      exit (1) ;
    }
  fprintf ( pF, "NWindow  10  WindVelX %4.1f WindVelY %4.1f DelTime %5.3f Ro %4.2f\n",
	    fVelX,fVelY, fDeltaT, fR0 ) ;
  fclose ( pF ) ;

  fLastTime  = 0. ;


  iOff = 0 ;   /* set this variable to turn off all atmospheric motion */

  /* Allocate storage for the wavefront. */

  pfWF   = alloc2df(iStations, iNumPts) ;
  pfDWFx = alloc2df(iStations, iNumPts) ;
  pfDWFy = alloc2df(iStations, iNumPts) ;

  /* Initialize the random number generator.
   * and the phase screen.  */

  lseed = 0 ;
  if (Init(lseed) == EXIT_FAILURE)
    {
      printf ( "Init failed to Initialize - Exiting\n" ) ;
      exit(1) ;
    }

  /* Call the atmosphere once, with time=0 for initialization */

  GetWaveFront(0, 0.0, 0, pfWF[0], pfDWFx[0], pfDWFy[0] ) ;

  printf ( "atmosphere initialized\n" ) ;
}

/**================================================
   atmosphere::testAtmos

   test the phases returned from the atmos simulator
   ===================================================*/
void atmosphere::testAtmos (void)
{
  int iPoints = 5000 ;         /* number of time steps */
  float fTimeStep = 0.010 ;   /* Time step, in seconds */
  float **pfData, *pfTime, *pfPower, *pfX ;
  int iAp, i, iStn, iApPts, iPt ;
  display *pPlot ;

  printf ("\n testAtmos:\n" ) ;

  printf ("                fDeltaT = %9.6f\n", fDeltaT ) ;
  printf ( "Atmosphere consists of %d points in each of %d stations\n",
	   iNumPts, iStations ) ;
  printf ( " time step = %6.3f\n", fTimeStep) ;

  /* set up data storage for plots */

  pfData = alloc2df( iStations, iPoints ) ;
  pfTime = alloc1df( iPoints ) ;
  pfX    = alloc1df( iPoints ) ;

  iStn = iStations ;
  if ( iStn > 4 ) 
    iStn = 4 ;
  iApPts = iNumPts ;
  if ( iApPts > 4 )
    iApPts = 4 ;

  for ( i = 0 ; i < iPoints ; i++ )
    {
      pfTime[i] = fTimeStep*(i+1) ;
      printf ( "Time =%7.4f   ", pfTime[i] ) ;
      for ( iAp = 0 ; iAp < iStn ; iAp++ )
	{
	  getWF( iAp, pfTime[i], pfWF[iAp] ) ;
	  printf ( "Stn%2d=%8.3f  ", iAp, pfWF[iAp][0] ) ;
	  pfData[iAp][i] = pfWF[iAp][0] - pfWF[0][0] ;
	}
      /* now for the points within an aperture */

      printf ( " #=#=# " ) ;
      for ( iPt = 1 ; iPt < iApPts ; iPt++ )
	printf ( "Pt%2d=%8.3f  ", iPt, pfWF[iAp-1][iPt] ) ;
      printf ( "  um\n" ) ;
    }
  /* plot the time series */

  for ( i = 1 ; i < iStn ; i++ )
    {
      pPlot = new display ( XY ) ;
      pPlot->setTitles( "Time (seconds)", "Delay (microns)",
			"Atmospheric Time Series" ) ;
      pPlot->range( 0.0, 0.0, 0.0, 0.0 ) ;
      pPlot->plotXY ( iPoints, pfTime, pfData[i] ) ;
    }
  /* and the power spectrum */

  for ( i =1 ; i < iPoints/2 ; i++ )
    {
      pfTime[i] = float(i)/(fTimeStep*float(iPoints)) ;
      pfX[i] = log10( pfTime[i] ) ;
    }
  for ( i = 1 ; i < iStn ; i++ )
    {
      pfPower = power ( iPoints, iPoints*fTimeStep, pfData[i] ) ;
      pPlot = new display ( XY ) ;
      pPlot->setTitles( "log Frequency", "log power", 
			"Atmospheric Power Spectrum" ) ;
      pPlot->range( 0.0, 0.0, 0.0, 0.0 ) ;
      pPlot->plotXY ( iPoints/2-1, pfX+1, pfPower+1 ) ;
      /*
      pPlot->plotXY ( iPoints/2-1, pfTime+1, pfPower+1 ) ;
      */
    }
}

float *power ( int iPts, float fDeltaT, float *pfData )
{
  float fFreq, *pfRes ;
  float fX, fY, fMin, fMax ;
  float fDeltaF, fMean ;
  int iD, iF ;

  /* subtract off the mean value */

  fMean = 0. ;
  for ( iD = 0 ; iD < iPts ; iD++ )
    fMean += pfData[iD] ;
  fMean /= float(iPts) ;

  /* Nyquest sample the FT */

  fDeltaF = 1.0/fDeltaT ;

  printf ( "mean value of time series = %6.3f\n", fMean ) ;
  printf ( "total time interval       = %6.3f\n", fDeltaT ) ;
  printf ( "frequency interval        = %6.3f  ", fDeltaF ) ;
  printf ( "     max freq = %6.3f\n", iPts*fDeltaF/2. ) ;

  pfRes = alloc1df ( iPts/2 ) ;
  for ( iF = 1 ; iF < iPts/2 ; iF++ )
    {
      fX = 0. ;
      fY = 0. ;
      fFreq = fDeltaF * iF ;
      for (  iD = 0 ; iD < iPts ; iD++ )
	{
	  fX += (pfData[iD]-fMean) * cos( 2.*PI*iF*iD/iPts) ;
	  fY += (pfData[iD]-fMean) * sin( 2.*PI*iF*iD/iPts) ;
	}
      pfRes[iF] = log10(fX*fX + fY*fY) ;
    }
  /* Calculate the range of values */

  fMin = pfRes[1] ;
  fMax = pfRes[1] ;
  for ( iF = 2 ; iF < iPts/2 ; iF++ )
    if( pfRes[iF] > fMax )
      fMax = pfRes[iF] ;
    else if ( pfRes[iF] < fMin )
      fMin = pfRes[iF] ;
  printf ( " Range of power -- %8.6f to %8.6f.  %d points\n", fMin, fMax, iPts ) ;

  return ( pfRes ) ;
}

/** ==========================================
    atmosphere::getWF

    This is my interface to Deane's atmosphere code.  ALL ACCESS TO
    THE ATMOSPHERE SHOULD BE DONE THROUGH CALLS TO getWF.

    returns the wavefront for this aperture at the specified time.

    The atmosphere simulator works in units of fDeltaT = 0.002 sec
    and nanometers.

    ****NOTE**** This version relies entirely on the wavefront storage
    and interpolation inside the atmosphere simulation.
    ==========================================*/
void atmosphere::getWF( int iAp, double dfTime, float *pfWFront )
{
  float fTOff ;
  int iT, i ;

  /* The atmosphere gets confused if we take a step that is too large
     so we take small steps until we have passed the time of interest,
     then backstep to the time we want.  */

  while ( fLastTime < dfTime )
    {
      fLastTime += fDeltaT ;

      /* Convert the time into an integer part and a fractional part */

      iT = 1 + int( fLastTime / fDeltaT ) ;
      if ( GetWaveFront(iAp, 0.0, iT, pfWFront, pfDWFx[iAp], pfDWFy[iAp] )
	   == EXIT_FAILURE )
	exit(EXIT_FAILURE) ;
    }

  iT = 1 + int( dfTime / fDeltaT ) ;
  fTOff = float(dfTime/fDeltaT - double(iT)) ;

  if ( GetWaveFront(iAp, fTOff, iT, pfWFront, pfDWFx[iAp], pfDWFy[iAp] )
       == EXIT_FAILURE )
    exit(EXIT_FAILURE) ;

  /* if desired, turn off the atmosphere */

  if ( iOff )
    for ( i = 0; i < iNumPts ;i++ )
      pfWFront[i] = 2.*iAp ;
}

/** ==========================================
    atmosphere::piston

    returns the mean piston for this aperture
    at the specified time

    Input Time is in seconds.
    Output Delay is in micrometers.
    ==========================================*/
float atmosphere::piston( int iAp, double dfTime )
{
  float fPiston ;
  int i ;

  getWF( iAp, dfTime, pfWF[iAp] ) ;

  /* The piston is the phase averaged over the wavefront */

  fPiston = 0. ;
  for ( i = 0 ; i < iNumPts ; i++ )
    fPiston += pfWF[iAp][i] ;
  fPiston /= float(iNumPts) ;

  return(fPiston) ; 
}
