/*===============================================================
  simulation of the beam combiner -- Version II

  This is a complete simulation of the NPOI beam combiner

  The wavelengths are in microns.
  Times should be in seconds.
  ================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "memory.h"
#include "dmmath.h"

#include "config.hpp"
#include "atmosphere.hpp"
#include "fdl.hpp"
#include "combiner.hpp"


float poidev(float fMean, long* lSeed) ;

static int bVerbose = 0 ;
static int bNoisey  = 1 ;



/** ========================================================
    combiner --- constructor

    The beam combiner class handles:

    the description of the beam combiner,
    the mixing of the beams,
    the calculation of the intensities at the detector
    The sampling of the data.

    This version handles one spectrograph of the NPOI 6-way beam
    combiner.

    To calculate the intensity at each time during the fringe
    sampling, it needs to have knowledge of the atmosphere and FDL
    delays for each of the stations that contributes to the detector.

    Future upgrades include an integration over the aperture to
    include, tip-tilt as well as higher order aberrations effects,
    scintillation and the effect of visibility amplitude reduction due
    to tracking off the center of the fringe packet.

    13 Feburary 2002   mozurk    function created
    ========================================================*/
combiner::combiner (  config *pConfig, fdl **pFdl, atmosphere *pA,
		      int *piStationList )
{
  int i, iW ;
  float fStroke ;

  printf ( "combiner: initialization begins.\n" ) ;

  iNumBins     = pConfig->getBins() ;
  iNumStations = pConfig->getStations() ;
  iNumApPoints = pConfig->getNumApPts()  ;
  printf ( "  Number of bins     = %4d\n", iNumBins  ) ;
  printf ( "  Number of Stations = %4d. Station list\n", iNumStations ) ;
  printf ( "  Number of Aper Pts = %4d\n", iNumApPoints  ) ;

  /* general data sampling description */

  fSTime    = 0.002  ;   /* NPOI sample time default = 2 ms */
  fClock    = 0.0000002 ; /* NPOI time resolution = 200 nanoseconds */
  iNumTicks = (int)(fSTime/fClock) ;  /* number of ticks per stroke */

  fTimeDelay = 0.006 ;

  /* beam combiner description */

  piStn = alloc1di ( iNumStations ) ;
  for ( i = 0 ; i < iNumStations ; i++ )
    {
      piStn[i] = piStationList[i] ;
      printf ( " fdl %3d == station %3d\n", i, piStationList[i] ) ;
    }

  /* Create a list of wavelengths -- linear in wavenumber! */

  iNumWaves    = pConfig->getWaves() ;
  printf ( "  Number of Wavelns  = %4d\n", iNumWaves ) ;
  pfWave = alloc1df ( iNumWaves ) ;
  pConfig->getWavelengths( pfWave );
  printf ( "wavelength list for %d channels\n", iNumWaves ) ;
  for ( i = 0 ; i < iNumWaves ; i++ )
    printf ( " channel %3d  wavelength = %6.3f\n", i, pfWave[i] ) ;

  /* allocate storage for fringe pattern calculation */

  pfD   = alloc2df ( 2, iNumStations ) ;
  pfWF  = alloc3df ( 2, iNumStations, iNumApPoints ) ;
  pfDelay= alloc1df ( iNumWaves ) ;
  pfData = alloc2df ( iNumWaves, iNumBins ) ;

  /* Initialize the detectors */

  pfDark = alloc1df ( iNumWaves ) ;
  for ( iW = 0 ; iW < iNumWaves ; iW++ )
    pfDark[iW]  = 0.0 ; 

  /* Initialize the star */

  pfI0   = alloc2df ( iNumWaves, iNumStations ) ;
  pfVis  = alloc3df ( iNumWaves, iNumStations, iNumStations ) ;
  printf ( "combiner: memory allocated\n" ) ;

  pConfig->getStarInfo( pfI0, pfVis ) ;

  pFDL = pFdl ;
  pAtm = pA ;

  /* set the mapping of time into bins */

  fStroke = pConfig->getStroke() ;
  setBinMap (fStroke) ;

  pfBin = alloc3df ( 2, iNumWaves, iNumBins+1 ) ;
  setBinDurations(fStroke) ;

  printf ( "combiner: initialization complete\n" ) ;
}

/** ============================================================
    setBinDurations

    Makes a table of the start and end times for each bin on both the
    up-stroke and on the down-stroke.  These duration tables will be
    used for integrating across the bins.

    pfBin[iDir][iW][iB] is the start time for bin iB at wavelength iW.
    iDir = 0 for up stroke and 1 for down stroke.

    iB = iNumBins gives the end time for the last bin.
    ============================================================*/
void combiner::setBinDurations(float fStroke)
{
  int iWave, iBin ;
  int iTicksPerBin, iFirstTick, iTick ;

  printf ( "setBinDurations:  Max wavelength = %6.3f microns\n", pfWave[0] ) ;
  printf ( "                  stroke length  = %6.3f microns\n", fStroke ) ;

  /* Create dead time at the start and end of the binning time so that
     the bins at all wavelengths are the same length in phase.  This
     digitization is needed to make the data look the real thing. */

  /* the duration of a tick in seconds.  This is the size of the
     digitization noise.  */

  for ( iWave = 0 ; iWave < iNumWaves ; iWave++ )
    {
      iTicksPerBin = (int)((pfWave[iWave]/fStroke)*iNumTicks/(2*iNumBins)) ;
      iFirstTick = iNumTicks/4 - iNumBins*iTicksPerBin/2 ;
      iTick = iFirstTick ;
      for ( iBin = 0 ; iBin < iNumBins+1 ; iBin++ )
	{
	  pfBin[0][iWave][iBin] = fSTime * iTick / iNumTicks ;
	  pfBin[1][iWave][iBin] = fSTime - pfBin[0][iWave][iBin] ;
	  iTick += iTicksPerBin ;
	}
    }
  /* Display the bin sizes */

  printf ( " Start times in microseconds for the bins.  " ) ;
  printf ( " Up stroke data.  Wavelengths in nm ---->\n" ) ;
  printf ( " bin " ) ;
  for ( iWave = 0 ; iWave < iNumWaves ; iWave++ )
    printf ( " %3d", int(1000.*pfWave[iWave]) ) ;
  printf ( "\n" ) ;
  for ( iBin = 0 ; iBin < iNumBins+1 ; iBin++ )
    {
      printf ( "bin%2d", iBin ) ;
      for ( iWave = 0 ; iWave < iNumWaves ; iWave++ )
	printf ( "%4d", int(1.E6*pfBin[0][iWave][iBin]) ) ;
      printf ( "\n" ) ;
    }
}

/** ============================================================
    setBinMap

    Set up the map needed for binning the data.

    piBinMap[i][j] is the bin for wavelength j at clock tick i that
    gets the signal.  Bins run from 0 to iNumBins-1.  Wavelengths run
    from 0 to iNumWaves-1.  If the bin number stored in piBinMap is
    out of range, the data is discarded.

    This version of setBinMap sets bins for the upstroke only
    ============================================================*/
void combiner::setBinMap (float fMaxWave )
{
  int iWave, iBin ;
  int iTicksPerBin, iFirstTick, iTick, i, iHalf ;

  printf ( "setBinMap:  Max wavelength = %6.3f microns\n", fMaxWave ) ;

  /* Create dead time at the start and end of the binning time so that
     the bins at all wavelengths are the same length in phase */

  iHalf = iNumTicks/2 ;
  printf ( "number of clock ticks per stroke = %d\n", iNumTicks ) ;
  piBinMap = alloc2di( iNumTicks, iNumWaves ) ;

  for ( iWave = 0 ; iWave < iNumWaves ; iWave++ )
    {
      /* do the up stroke */

      iTicksPerBin = (int)((pfWave[iWave]/fMaxWave)*iHalf/iNumBins) ;
      iFirstTick = (iHalf - iNumBins*iTicksPerBin)/2 ;
      iTick = 0 ;
      printf ( "Channel %3d ticks per bin = %4d  first tick = %4d\n",
	       iWave, iTicksPerBin, iFirstTick ) ;
      while ( iTick < iFirstTick )
	piBinMap[iTick++][iWave] = -1 ;
      for ( iBin = 0 ; iBin < iNumBins ; iBin++ )
	for ( i = 0 ; i < iTicksPerBin ; i++ )
	  piBinMap[iTick++][iWave] = iBin ;
      while ( iTick < iHalf )
	piBinMap[iTick++][iWave] = -1 ;

      /* Now do the down stroke -- flipped about the middle. */

      for ( iTick = 0 ; iTick < iHalf ; iTick++ )
	piBinMap[iNumTicks-iTick-1][iWave] = piBinMap[iTick][iWave] ;

    }
  printf ( "Bin map set for upstroke and downstroke\n" ) ;
}

/** ============================================================
    getData

    This function executes one "sample-time" and returns the
    accumulated data.  The sample time is the minimum possible time
    interval over which the fringe parameters can be estimated.  It is
    the rate at which data is stored on disk.

    fTime is the time the data is delivered to the fringe processer.
    The data is stored in piData, which is also returned.
    ============================================================*/
int **combiner::getData( float fTime, int **piData )
{
  int iBin, iWave ;
  static long int lSeed = 239671953857 ;

  if ( bVerbose )
    printf ( "combiner::getData begins.  Time data is delivered = %9.6f seconds.\n", fTime ) ;

  /* Calculate the expected signal for each datum and put it in  pfData */

  getI(fTime-fTimeDelay, pfData) ;

  /* add noise and convert to integer */

  for ( iBin = 0 ; iBin < iNumBins ; iBin++ )
    for ( iWave = 0 ; iWave < iNumWaves ; iWave++ )
      if ( bNoisey )
	piData[iWave][iBin] = int(poidev( pfData[iWave][iBin], &lSeed )) ;
      else
	piData[iWave][iBin] = int(pfData[iWave][iBin]+0.5) ;
  /*
  showData( piData ) ;
  */
  return ( piData  ) ;
}

/** ============================================================
    getI

    This function is the model of the fringe combination process.
    It sets the values of pfI[iWave] for the current time.

    The intensity is given by 

    \sum_a I_0(a) ( 1 + \sum_{i,j} V_i,j cos ( 2 pi d/wave ) )

    NOTE: there is no integration over the bandpass of each filter!

    important internal variables 

    pfI0    -- the station intensities for each wavelength (constant for now)
    pfDelay -- the station delay at this time tick
    pfVis   -- the Visibility amplitude for each baseline/wavelength
    pfDark  -- backgrund (dark) count for each wavelength

    pfI     -- the intensity(signal) at each wavelength
    ============================================================*/
void combiner::getI( float fTime0, float **pfData ) 
{
  int i, j, iAp, iDir, iWave, iPts, iBin ;
  float fData, fT0, fT1, fDeltaT ;

  printf ( "getI: begins for time %9.6f\n", fTime0 ) ;
  if ( bVerbose )
    printf ( "numBins = %3d  numWaves = %3d\n", iNumBins, iNumWaves ) ;
  for ( iBin = 0 ; iBin < iNumBins ; iBin++ )
    for ( iWave = 0 ; iWave < iNumWaves ; iWave++ )
      {
	/* calculate the counts on this detector */

	pfData[iWave][iBin] = 0. ;

	/* Loop over up- and down- strokes */
	
	for ( iDir = 0 ; iDir < 2 ; iDir++ )
	  {
	    /* get the start-time and the length of the bin */

	    fT0 = fTime0 + pfBin[iDir][iWave][iBin] ;
	    fT1 = fTime0 + pfBin[iDir][iWave][iBin+1] ;
	    fDeltaT = (fT1 - fT0)*(1-2*iDir) ;

	    /* Add up the DC signal level for this bin. */

	    pfData[iWave][iBin] += pfDark[iWave] ;
	    for ( i = 0 ; i < iNumStations ; i++ )
	      pfData[iWave][iBin] += pfI0[iWave][i] ;

	    /* Get the delay at the start and end of each bin.  This is a
	       three dimensional array.  WF[t][ap][point] there t is the
	       time, either the start or the end of the integration, ap is
	       the aperture number and point is the point in the wavefront. */

	    for ( iAp = 0 ; iAp < iNumStations ; iAp++ )
	      {
		/* First, we get the atmospheric delay values. */

		pAtm->getWF( iAp, fT0, pfWF[0][iAp] ) ;
		pAtm->getWF( iAp, fT1, pfWF[1][iAp] ) ;

		/* then the delay line values. */

		pfD[0][iAp]  = pFDL[iAp]->readMetrology(fT0) ;
		pfD[1][iAp]  = pFDL[iAp]->readMetrology(fT1) ;
		for ( iPts = 0 ; iPts < iNumApPoints ; iPts++ )
		  {
		    /* Dispersion gets added to only the atmospheric delay */

		    pfWF[0][iAp][iPts] *=   1.+ n_dryAir( pfWave[iWave] ) ;
		    pfWF[1][iAp][iPts] *=   1.+ n_dryAir( pfWave[iWave] ) ;

		    /* now differrence the two. */

		    pfWF[0][iAp][iPts] -= pfD[0][iAp] ;
		    pfWF[1][iAp][iPts] -= pfD[1][iAp] ;
		  }
	      }

	    /* Integrate the intensity over the aperture and the
	       duration of the bin.  We will assume the delay varies
	       linearly with time so the integration over time gives a
	       sinc function depending only on the delay at the start
	       and end times. */

	    for ( i = 1 ; i < iNumStations ; i++ )
	      for ( j = 0 ; j < i ; j++ )
		{
		  /* calculte the intensity at each point in the aperture */

		  fData = 0 ;
		  for ( iPts = 0; iPts < iNumApPoints ; iPts++ )
		    fData =  fData 
		      + sinc(PI*(pfWF[1][i][iPts]-pfWF[0][i][iPts])/pfWave[iWave] )
		      *  cos(PI*(pfWF[1][i][iPts]+pfWF[0][i][iPts])/pfWave[iWave] ) ;
		  /*
		  if ( bVerbose )
		    {
		      printf ( "prev pfData = %12.4f    fData = %12.4f.",
			       pfData[iWave][iBin], fData ) ;
		      printf ( "  V=%5.2f  I1 = %8.0f  iNumApPoints = %3d  ",
			       pfVis[iWave][i][j], sqrt(pfI0[iWave][i]*pfI0[iWave][j]),
			       iNumApPoints ) ;
		    }
		  */
		  pfData[iWave][iBin] += 2.*sqrt(pfI0[iWave][i]*pfI0[iWave][j])
		    * pfVis[iWave][i][j] * fData / iNumApPoints ;
		  /*
		  if ( bVerbose )
		    printf ( "curr pfData = %12.4f\n", pfData[iWave][iBin] ) ;
		  */
		}
	    /*
	    if ( bVerbose )
	      printf ( "prev pfData = %12.4f\n", pfData[iWave][iBin] ) ;
	    */
	    if ( bVerbose )
	      if ( iDir == 0 )
		printf ( "wave = %3d  bin = %3d dir = %2d time int %9.6f to %9.6f  ",
			 iWave, iBin, iDir, fT0, fT1 ) ;
	      else
		printf ( "   dir = %2d time int %9.6f to %9.6f  Sig= %9.3f\n",
			 iDir, fT0, fT1, pfData[iWave][iBin] ) ;
	  }
	pfData[iWave][iBin] *= fDeltaT ;
      }
  printf ( "getI: ends\n" ) ;
}

void combiner::showBinMap ( void )
{
  int iTick, iWave ;

  for ( iTick = 0 ; iTick < iNumTicks ; iTick++ )
    {
      for ( iWave = 0 ; iWave < iNumWaves ; iWave++ )
	printf ( "%3d", piBinMap[iTick][iWave] ) ;
      printf ( "\n" ) ;
    }
} 

void combiner::showData ( int **piData )
{
  int iWave, iBin ;

  printf ( " bins-->\n" ) ;
  for ( iWave = 0 ; iWave < iNumWaves ; iWave++ )
    {
      for ( iBin = 0 ; iBin < iNumBins ; iBin++ )
	printf ( "%3d", piData[iWave][iBin] ) ;
      printf ( "\n" ) ;
    }
}

