/*===============================================================
  config.cpp    Simulation of the beam combiner -- Version II

  Configuration

  This class deals with all the configuration details need to make the
  simulation work properly.
  ================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "memory.h"
#include "dmmath.h"
#include "config.hpp"

static int bVerbose = 0 ;

/** ========================================================
    config --- constructor

    16 July 2002   mozurkewich   function created
    ========================================================*/
config::config ( configType eType )
{
  if ( bVerbose ) 
    printf ( "config: constructor begins\n" ) ;

  eConfigType = eType ;

  iNumWaves =   32 ;
  if ( eType == NPOI )
    {
      iNumStations = 6 ;
      iNumBins  =   64; 
      fApDia    = 0.12 ;
      iNumApPts =    7 ;
      fStroke   = 0.900 ;
      piModAmp  = alloc1di( iNumStations ) ;
      piModAmp[0] = 0 ;
      piModAmp[1] = 4 ;
      piModAmp[2] = 1 ;
      piModAmp[3] = 6 ;
      piModAmp[4] = 2 ;
      piModAmp[5] = 3 ;
    }
  else if ( eType == TEST )
    {
      iNumStations = 5 ;
      iNumBins  =   64; 
      fApDia    = 0.12 ;
      iNumApPts =    7 ;
      fStroke   = 0.900 ;
      piModAmp  = alloc1di( iNumStations ) ;
      piModAmp[0] = 0 ;
      piModAmp[1] = 4 ;
      piModAmp[2] = 1 ;
      piModAmp[3] = 6 ;
      piModAmp[4] = 2 ;
    }
  else
    {
      printf ( "Invalid configuration type\n" ) ;
      exit(1) ;
    }

  pfWave = alloc1df( iNumWaves) ;

  pfWave[ 0] = 0.8522 ;
  pfWave[ 1] = 0.8220 ;
  pfWave[ 2] = 0.7942 ;
  pfWave[ 3] = 0.7685 ;
  pfWave[ 4] = 0.7448 ;
  pfWave[ 5] = 0.7228 ;
  pfWave[ 6] = 0.7023 ;
  pfWave[ 7] = 0.6832 ;
  pfWave[ 8] = 0.6654 ;
  pfWave[ 9] = 0.6487 ;
  pfWave[10] = 0.6330 ;
  pfWave[11] = 0.6183 ;
  pfWave[12] = 0.6045 ;
  pfWave[13] = 0.5914 ;
  pfWave[14] = 0.5791 ;
  pfWave[15] = 0.5675 ;
  pfWave[16] = 0.5565 ;
  pfWave[17] = 0.5461 ;
  pfWave[18] = 0.5362 ;
  pfWave[19] = 0.5268 ;
  pfWave[20] = 0.5179 ;
  pfWave[21] = 0.5094 ;
  pfWave[22] = 0.5013 ;
  pfWave[23] = 0.4937 ;
  pfWave[24] = 0.4863 ;
  pfWave[25] = 0.4793 ;
  pfWave[26] = 0.4727 ;
  pfWave[27] = 0.4663 ;
  pfWave[28] = 0.4602 ;
  pfWave[29] = 0.4544 ;
  pfWave[30] = 0.4488 ;
  pfWave[31] = 0.4435 ;

  if ( bVerbose )
    printf ( "configuration initialized\n" ) ;
}



int config::getBins( void)
{
  return (iNumBins) ;
}

int config::getWaves( void)
{
  return (iNumWaves) ;
}

int config::getStations( void)
{
  return (iNumStations) ;
}

void config::getStationList ( float *pfX, float *pfY )
{
  if ( iNumStations == 6 )
    {
      pfX[0] =   7. * sin(PI*60./180.) ;
      pfX[1] =  17. * sin(PI*60./180.) ;
      pfX[2] =  27. * sin(PI*60./180.) ;
      pfX[3] =  -7. * sin(PI*60./180.) ;
      pfX[4] = -17. * sin(PI*60./180.) ;
      pfX[5] = -27. * sin(PI*60./180.) ;

      pfY[0] =  7. * cos(PI*60./180.) ;
      pfY[1] = 17. * cos(PI*60./180.) ;
      pfY[2] = 27. * cos(PI*60./180.) ;
      pfY[3] =  7. * cos(PI*60./180.) ;
      pfY[4] = 17. * cos(PI*60./180.) ;
      pfY[5] = 27. * cos(PI*60./180.) ;
    }
  else if ( iNumStations == 5 )
    {
      pfX[0] =  0. ;
      pfX[1] =  1.0 ;
      pfX[2] =  3.0 ;
      pfX[3] = 10.0 ;
      pfX[4] = 30.0 ;
      pfY[0] =  0. ;
      pfY[1] =  0.0 ;
      pfY[2] =  0.0 ;
      pfY[3] =  0.0 ;
      pfY[4] =  0.0 ;
    }
 else if ( iNumStations == 4 )
    {
      pfX[0] =  0. ;
      pfX[1] = 10.0 ;
      pfX[2] =  0.0 ;
      pfX[3] = 10.0 ;
      pfY[0] =  0. ;
      pfY[1] =  0.0 ;
      pfY[2] = 10.0 ;
      pfY[3] = 10.0 ;
    }
  else
    {
      printf ( "getStationList:  invalid number of Stations\n" ) ;
      exit(1) ;
    }
}

float config::getApSize( void)
{
  return (fApDia) ;
}

int config::getNumApPts( void)
{
  return (iNumApPts) ;
}

float config::getStroke( void)
{
  return (fStroke) ;
}

void config::getApPtList( float *pfX, float *pfY )
{
  int i ;
  if ( iNumApPts == 1 )
    {
      pfX[0] =  0.00 ;
      pfY[0] =  0.00 ;
    }
  else
    {
      /* points in a ring around the central point */

      pfX[0] =  0.00 ;
      pfY[0] =  0.00 ;
      for ( i = 1 ; i < iNumApPts ; i++ )
	{
	  pfX[i] =  0.375*cos(2*PI*i/float(iNumApPts)) ;
	  pfY[i] =  0.375*sin(2*PI*i/float(iNumApPts)) ;
	}
    }
}


void config::getWavelengths ( float *pfWaveLength )
{
  int i ;

  for ( i = 0 ; i < iNumWaves ; i++ )
    pfWaveLength[i] = pfWave[i] ;
}

void config::getModAmps ( int *piAmps )
{
  int i ;

  for ( i = 0 ; i < iNumStations ; i++ )
    piAmps[i] = piModAmp[i] ;
}

void config::getStarInfo( float **pfI0, float ***pfVis )
{
  int iW, i, j ;

  for ( iW = 0 ; iW < iNumWaves ; iW++ )
    {
      for ( i = 0 ; i < iNumStations ; i++ )
	{
	  /* Set count rates.  Units are counts per Second per
             wavelength from each Aperture. Choices are uniform
	     or more signal in the red.  */

	  pfI0[iW][i] = 500000. ;
	  pfI0[iW][i] = 300.*(36.- (float)iW) ;

	  /* 16 channels have signal, but the HeNe Channel never has counts */

	  if ( (iW > 18)||(iW == 10) )
	    pfI0[iW][i] = 0. ;

	  /*  The H_Alpha filter replaces four additional channels near H_alpha
	      with bluer channels */
	
	  if ( (iW>7)&&(iW<13) )
	    pfI0[iW][i] = 0. ;
	  if ( iW > 20 )
	    pfI0[iW][i] = 0. ;
	 

	  /* Set the visibilities.  Only baselines [i][j] with i<j exist.
	     zero the others */

	  for ( j = 0 ; j < iNumStations ; j++ )
	    {
	      pfVis[iW][i][j] = 0. ;
	      for ( j = 0 ; j < iNumStations ; j++ )
		{
		  pfVis[iW][i][j] = 0. ;
		  /*
		    if ( j < i )
		    pfVis[iW][i][j] = 0.75 ;
		  */
		}
	    }
	}
      /* That's too confusing, so for now, keep all the flux but only
         a couple of baselines */

      pfVis[iW][1][0] = .95 ;
      if ( iNumStations > 2 )
	pfVis[iW][2][0] = .95 ;
      if ( iNumStations > 3 )
	pfVis[iW][3][0] = .95 ;
    }
}

/**==============================================================
   getAtmos

   return the atmosphere parameters

   fVelX  ===  wind velocity in X direction (meters/second)
   fVelY  ===  wind velocity in Y direction (meters/second)
   fR0    ===  Fried's parameter (meters )
   ==============================================================*/
void config::getAtmos( float *pfVelX, float *pfVelY, float *pfR0)
{
  *pfVelX = 0. ;
  *pfVelY = 20. ;
  *pfR0   = 0.05 ;
}
