/** ===============================================================
    file display.cpp

    display stuff with pgplot

    ===============================================================*/
#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <math.h>

#include "display.hpp"



/** =================================================================
    display   constructor for a gray scale plot


    =================================================================*/
display::display ( plotType ePlot, int iNumWaves, int iNumBins) 
{
  eType = ePlot ;

  if ( eType != GRAY_SCALE )
    {
      printf ( "bad plot designation!\n" )  ;
      exit(1) ;
    }

  iSelect = cpgopen( "/xserv" ) ;
  cpgpap ( 4., 2. ) ;
  pfData = alloc1df ( iNumBins*iNumWaves+1 );
  cpgask(0) ;

  fWhite = 1.0 ; 
  fBlack = 0.0 ;
  iFrame = 0 ;
  tr[0] = 0. ;
  tr[1] = 1. ;
  tr[2] = 0. ;
  tr[3] = 0. ;
  tr[4] = 0. ;
  tr[5] = 1. ;

}


/** =================================================================
    display   constructor for a single frame at a time X-Y Plot
    =================================================================*/
display::display ( plotType ePlot )
{
  eType = ePlot ;

  if ( eType != XY )
    {
      printf ( "bad plot designation!\n" )  ;
      exit(1) ;
    }

  /* Set the default titles */

  strncpy ( pcXLabel, "X", 120 ) ;
  strncpy ( pcYLabel, "Y",   120 ) ;
  strncpy ( pcTitle,  "Title", 120 ) ;

  iAutoRange = 0 ;
  iSelect = cpgopen( "/xserv" ) ;
  cpgpap ( 10., 0.4 ) ;
  cpgask(0) ;
}

void display::range( float fXmin, float fXmax, float fYmin, float fYmax )
{
  if ( fXmin == fXmax )
    {
      iAutoRange = 1;
      printf ( "setting autorange to 1\n" ) ;
    }
  else
    iAutoRange = 0 ;
}

void display::plotXY ( int iPoints, float *pfX, float *pfY )
{
  int i ;
  float pfYRange[2] ;
  float pfBar1[2] = { 1., 1. } ;
  float pfBar2[2] = { 4., 4. } ;
  float pfBar3[2] = { 6., 6. } ;
  float fXmax, fXmin ;

  if ( iAutoRange )
    {
      /* Determine the max and min values */

      fXmax = pfX[0] ;
      fXmin = pfX[0] ;
      for ( i = 1 ; i < iPoints ; i++ )
	{
	  if ( fXmax < pfX[i] )
	    fXmax = pfX[i] ;
	  else if ( fXmin > pfX[i] )
	    fXmin = pfX[i] ;
	}
      pfYRange[1] = pfY[0] ;
      pfYRange[0] = pfY[0] ;
      for ( i = 1 ; i < iPoints ; i++ )
	{
	  if ( pfYRange[1] < pfY[i] )
	    pfYRange[1] = pfY[i] ;
	  else if ( pfYRange[0] > pfY[i] )
	    pfYRange[0] = pfY[i] ;
	}
    }
  else
    {
      /* autoRange is off.  used these guesses */

      fXmin = 0. ;
      fXmax = pfX[iPoints-1]+1 ;
      pfYRange[0] = 0.0 ;
      pfYRange[1] = 0.2 ;
    }

  cpgslct ( iSelect ) ;
  cpgsch ( 2.0 ) ;
  cpgsci ( 1 ) ;
  cpgenv( fXmin, fXmax, pfYRange[0], pfYRange[1], 0, 1) ;
  cpglab( pcXLabel, pcYLabel, pcTitle ) ;
  cpgline ( iPoints, pfX, pfY ) ;
  if ( !iAutoRange )
    {
      cpgline ( 2, pfBar1, pfYRange ) ;
      cpgline ( 2, pfBar2, pfYRange ) ;
      cpgline ( 2, pfBar3, pfYRange ) ;
    }

}


/** =================================================================
    display   constructor for a chart-recorder plot


    =================================================================*/
display::display ( plotType ePlot, int iLines, float fxMin, float fxMax, float fyMin, float fyMax )
{
  int i ;
  eType = ePlot ;
  char pcTitle[80] ;

  if ( eType != CHART )
    {
      printf ( "Bad plot designation!\n" )  ;
      exit(1) ;
    }
  iFirst = 1 ;
  iSelect = cpgopen( "/xserv" ) ;
  cpgpap ( 12., 0.3 ) ;
  cpgask(0) ;
  cpgsci ( 1 ) ;
  sprintf ( pcTitle, "Chart" ) ; 
  cpgsch ( 2.0 ) ;
  cpgenv( fxMin, fxMax, fyMin, fyMax, 0, 1) ;

  iNumLines = iLines ;
  piPoint = alloc1di ( iNumLines ) ;
  for ( i = 0 ; i < iNumLines ; i++ )
    piPoint[i] = 0 ;
  iBuffSize = 2000 ;
  pfX = alloc2df ( iNumLines, iBuffSize ) ;
  pfY = alloc2df ( iNumLines, iBuffSize ) ;

}

void display::chart ( int iLine, float fX, float fY )
{
  cpgslct ( iSelect ) ;

  if ( iFirst )
    {
      cpgsci(1 ) ;
      iFirst = 0 ;
      cpglab( pcXLabel, pcYLabel, pcTitle ) ;
    }
  cpgsci ( iLine+2 ) ;
  pfX[iLine][piPoint[iLine]] = fX ;
  pfY[iLine][piPoint[iLine]] = fY ;
  piPoint[iLine]++ ;
  cpgline ( piPoint[iLine], pfX[iLine], pfY[iLine] ) ;

}

/*-----------------------------------------------------------------
  displayFrame

  Makes a one frame display of a single data frame

  x-axis is wavelength 
  y-axis is bin number

  this is a gray scale plot and auto scales the intensities

  ------------------------------------------------------------------*/
void display::displayFrame(  int **piData )
{
  char pcTitle[80] ;
  int i, j, iPix ;
  int iNumBins = 64 ;
  int iNumWaves = 32 ;

  /* set the axis ranges */

  cpgslct( iSelect) ;


  cpgsci ( 1 ) ;
  sprintf ( pcTitle, "FRAME  %3d", iFrame++) ; 
  cpgsch(2.0) ;
  cpgenv( 0.0, (float)(iNumWaves+1), 0.0, (float)(iNumBins+1), 1, 1) ;
  cpglab( pcXLabel, pcYLabel, pcTitle ) ;
  fWhite = piData[0][0] ;
  fBlack = piData[0][0] ;

  pfData[0] = 0. ;
  iPix = 0 ;
  for ( i = 0 ; i < iNumBins ; i++ )
    for ( j = 0 ; j < iNumWaves ; j++ )
      {
	pfData[iPix] = piData[j][i] ;
	if ( pfData[iPix] < fBlack )
	  fBlack = pfData[iPix] ;
	if ( pfData[iPix] > fWhite )
	  fWhite = pfData[iPix] ;
	iPix++ ;
      }	  

  /* force black to be zero counts */

  fBlack = 0. ;
  cpggray( pfData, iNumWaves, iNumBins, 1, iNumWaves, 1,
	   iNumBins, fWhite, fBlack, &tr[0] ) ;
}


void display::setTitles( char *pcX, char *pcY, char *pcTop )
{
  strncpy ( pcXLabel, pcX,   120 ) ;
  strncpy ( pcYLabel, pcY,   120 ) ;
  strncpy ( pcTitle,  pcTop, 120 ) ;
}
