/*------------------------------------------------
    file dmMath.c

 Mathematics

 special functions:
 sinc

  linear algebra:
  gauss-jordan elimination


----------------------------------------------------*/
#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <math.h>
#include "memory.h"
#include "dmmath.h"




/*-----------------------------------------------------------
  gaussj

  Linear equation solution by Gauss-Jordan elimination
  Numerical recipes in C

  modified by dm to use c-type array indexing.

  A[0..n-1][0..n-1] is the input matrix.
  B[0..n-1][0..m-1] is input containing the right-hand side vectors.

  This routine solves the 'k' sets of 'n' linear equations

  sum_j A[i][j] X[j] = B[i][k]

  for X[j]. for each k.  

  On output,

  A is replaced by its matrix inverse and 
  B is replaced by the corresponding set of solution vectors.
  -----------------------------------------------------------*/
void gaussj(float **a, int n, float **b, int m )
{
  int bVerbose = 0 ;
  int *indxc, *indxr, *ipiv ;
  int i, icol, irow, j, k, l, ll ;
  float big, temp, pivinv, dum;


  if (bVerbose) 
    printf ( "gaussj: begins\n" ) ;

  /* allocate memory for bookkeeping arrays */

  indxc = alloc1di(n) ;
  indxr = alloc1di(n) ;
  ipiv  = alloc1di(n) ;

  for ( j = 0 ; j < n ; j++ )
    {
      ipiv[j] = 0 ;
    }
  if ( bVerbose )
    printf (  "memory allocated and initialized\n" ) ;

  /* here starts the main loop over the columns to be reduced */

  for ( i = 0 ; i < n ; i++ )
    {
      big = 0.0 ;

      /* The outer loop in search of a pivot point */

      for ( j = 0 ; j < n ; j++ )
	{
	  if ( ipiv[j] != 1. )
	    for ( k = 0 ; k < n ; k++ )
	      {
		if ( ipiv[k] == 0)
		  {
		    if ( fabs(a[j][k]) >= big )
		      {
			big = fabs(a[j][k]) ;
			irow = j ;
			icol = k ;
		      }
		  }
		else if ( ipiv[k] > 1 )
		  nrerror("gaussj: Singular Matrix-1\n" ) ;
	      }
	}
      ++(ipiv[icol]) ;

      /* We now have the pivot element, so we interchange rows, if needed,
	 to put the pivot element on the diagonal.  The columns are not
	 physically interchanged, only relabeled:  indxc[i], the column
	 of the i-th pivot element, is the i-th column that is reduced
	 while indxr[i] is the row in which that pivot element was 
	 originally located.  If indxr[i] != indxc[i], there is an implied
	 column interchange.  With this form or bookkeeping, the solution
	 b's will end up in the correct order, and the inverse matrix
	 will be scrambled by columns.  */

      if ( irow != icol )
	{
	  for ( l = 0 ; l < n ; l++ )
	    SWAP(a[irow][l],a[icol][l]) ;
	  for ( l = 0 ; l < m ; l++ )
	    SWAP(b[irow][l],b[icol][l]) ;

	}

      /* We are now ready to divide the pivot row by the pivot element
	 (was at irow, icol, now at icol,icol) */

      indxr[i] =  irow ; 
      indxc[i] =  icol ;
      if ( a[icol][icol] == 0.0 )
	nrerror("gaussj: Singular matrix-2\n" ) ;

      pivinv = 1.0 / a[icol][icol] ;
      /*dm bug.  Next line should not be there */
      /*
      a[icol][icol] = 1.0 ; 
      */
      for ( l = 0 ; l < n ; l++ )
	a[icol][l] *= pivinv ;
      for ( l = 0 ; l < m ; l++ )
	b[icol][l] *= pivinv ;

      /* next, reduce the rows, except the pivot row */

      for ( ll = 0 ; ll < n ; ll++ )
	if ( ll != icol )
	  {
	    dum = a[ll][icol] ;
	    /*dm bug.  Next line should not be there */
	    /*
	    a[ll][icol] = 0.0 ;
	    */
	    for ( l = 0 ; l < n ; l++ )
	      a[ll][l] -= a[icol][l]*dum ;
	    for ( l = 0 ; l < m ; l++ )
	      b[ll][l] -= b[icol][l]*dum ;
	  }
    }

  /* This is the end of the main loop over columns of the reduction.
     It only remains to unscramble the solution in view of the column
     interchanges.  We do this by interchanging pairs of columns in
     the reverse order that the premutation was built up. */

  for ( l=n-1 ; l >=0 ; l-- )
    {
      if ( indxr[l] != indxc[l] )
	{
	  for(k=0 ; k< n ; k++ )
	    SWAP(a[k][indxr[l]],a[k][indxc[l]]) ;
	}
    }

  free1di(ipiv)  ;
  free1di(indxr) ;
  free1di(indxc) ;
}


/*-----------------------------------------------------------
  dgaussj

  Double Precision Linear equation solution by
  Gauss-Jordan elimination.
  Numerical recipes in C

  modified by dm to use c-type array indexing.

  A[0..n-1][0..n-1] is the input matrix.
  B[0..n-1][0..m-1] is input containing the right-hand side vectors.

  This routine solves the 'k' sets of 'n' linear equations

  sum_j A[i][j] X[j] = B[i][k]

  for X[j]. for each k.  

  On output,

  A is replaced by its matrix inverse and 
  B is replaced by the corresponding set of solution vectors.
  -----------------------------------------------------------*/
void dgaussj(double **a, int n, double **b, int m )
{
  int *indxc, *indxr, *ipiv ;
  int i, icol, irow, j, k, l, ll ;
  double big, dum, pivinv, temp ;

  /* allocate memory for bookkeeping arrays */

  indxc = alloc1di(n) ;
  indxr = alloc1di(n) ;
  ipiv  = alloc1di(n) ;

  for ( j = 0 ; j < n ; j++ )
    ipiv[j] = 0 ;

  /* here starts the main loop over the columns to be reduced */

  for ( i = 0 ; i < n ; i++ )
    {
      big = 0.0 ;

      /* The outer loop in search of a pivot point */

      for ( j = 0 ; j < n ; j++ )
	{
	  if ( ipiv[j] != 1. )
	    for ( k = 0 ; k < n ; k++ )
	      {
		if ( ipiv[k] == 0)
		  {
		    if ( ABS(a[j][k]) >= big )
		      {
			big = ABS(a[j][k]) ;
			irow = j ;
			icol = k ;
		      }
		  }
		else if ( ipiv[k] > 1 )
		  nrerror("dgaussj: Singular Matrix-1\n" ) ;
	      }
	}
      ++(ipiv[icol]) ;

      /* We now have the pivot element, so we interchange rows, if needed,
	 to put the pivot element on the diagonal.  The columns are not
	 physically interchanged, only relabeled:  indxc[i], the column
	 of the i-th pivot element, is the i-th column that is reduced
	 while indxr[i] is the row in which that pivot element was 
	 originally located.  If indxr[i] != indxc[i], there is an implied
	 column interchange.  With this form or bookkeeping, the solution
	 b's will end up in the correct order, and the inverse matrix
	 will be scrambled by columns.  */

      if ( irow != icol )
	{
	  for ( l = 0 ; l < n ; l++ )
	    SWAP(a[irow][l],a[icol][l]) ;
	  for ( l = 0 ; l < m ; l++ )
	    SWAP(b[irow][l],b[icol][l]) ;
	}

      /* We are now ready to divide the pivot row by the pivot element
	 (was at irow, icol, now at icol,icol) */

      indxr[i] =  irow ; 
      indxc[i] =  icol ;
      if ( a[icol][icol] == 0.0 )
	nrerror("dgaussj: Singular matrix-2\n" ) ;

      pivinv = 1.0 / a[icol][icol] ;
      /*dm bug.  Next line should not be there */
      /*
      a[icol][icol] = 1.0 ; 
      */
      for ( l = 0 ; l < n ; l++ )
	a[icol][l] *= pivinv ;
      for ( l = 0 ; l < m ; l++ )
	b[icol][l] *= pivinv ;

      /* next, reduce the rows, except the pivot row */

      for ( ll = 0 ; ll < n ; ll++ )
	if ( ll != icol )
	  {
	    dum = a[ll][icol] ;
	    /*dm bug.  Next line should not be there */
	    /*
	    a[ll][icol] = 0.0 ;
	    */
	    for ( l = 0 ; l < n ; l++ )
	      a[ll][l] -= a[icol][l]*dum ;
	    for ( l = 0 ; l < m ; l++ )
	      b[ll][l] -= b[icol][l]*dum ;
	  }
    }

  /* This is the end of the main loop over columns of the reduction.
     It only remains to unscramble the solution in view of the column
     interchanges.  We do this by interchanging pairs of columns in
     the reverse order that the premutation was built up. */

  for ( l=n-1 ; l >=0 ; l-- )
    {
      if ( indxr[l] != indxc[l] )
	{
	  for(k=0 ; k< n ; k++ )
	    SWAP(a[k][indxr[l]],a[k][indxc[l]]) ;
	}
    }

  free1di(ipiv)  ;
  free1di(indxr) ;
  free1di(indxc) ;
}



/*----------------------------------------------------------------------
    fourn

    Replaces data by its ndim-dimensional discrete Fourier transform,
    if iSign is input as 1.  nn[1..ndim] is an integer array containing
    the lengths of each dimension (number of complex values), which MUST
    all be powers of two.  data is a real array of length twice the product
    of these lengths, in which the data are stored in a multi-dimensional
    complex array: real and imaginary parts of each element are in concecutive 
    locations, and the rightmost index of the array increases most rapidly as
    one proceeds along data.  For a two-dimensional array, this is equivalent
    to storing the array by rows.  If iSign is input at -1, data is replaced
    by its inverse transform times the product of the lengths of all dimensions.

    Does a multi-dimensional fast fourier transform of a complex function.

    float  *data     pointer to the address one before the start of the data
                     data[1] is the first real, data[2] the first imag
                     data[3] the second real,  etc.
    int ndim         the number of dimensions.
    long nn[]        the length of each dimension, again, starts at 1.
    int iSign        1 for a forward transform -1 for an inverse transform
-------------------------------------------------------------------------*/
void fourn ( float *data, int *nn , int ndim, int iSign )
{
  int idim ;
  signed int i1, i2, i3, i2rev, i3rev, ip1, ip2, ip3, ifp1, ifp2 ;
  signed int ibit, k1, k2, n, nprev, nrem, ntot ;
  float temp, tempi, tempr ;
  double theta, wi, wpi, wpr, wr, wtemp ;

  /* Compute total number of complex values */

  ntot = 1 ;
  for ( idim = 1 ; idim <= ndim ; idim++ )
    {
      ntot *= nn[idim] ;
    }

  printf ( "fourn: %d-D FFT dimension = %d", ndim, nn[1] ) ;
  for ( i1 = 1 ; i1 < ndim ; i1++ )
    printf ( "X %d", nn[i1+1] ) ;

  /* Main loop over the dimensions */

  nprev = 1 ;
  for ( idim = ndim ; idim > 0 ; idim-- )
    {
      n = nn[idim] ;
      nrem = ntot / (n*nprev) ;
      ip1 = nprev << 1 ;
      ip2 = ip1*n ;
      ip3 = ip2 * nrem ;
      i2rev= 1 ;

      /* This is the bit reversal section of the routine */

      for ( i2=1 ; i2<=ip2; i2+=ip1 )
	{
	  if ( i2 < i2rev )
	    {
	      for ( i1 = i2 ; i1 <= i2 + ip1 - 2 ; i1 += 2 )
		{
		  for ( i3 = i1 ; i3 <= ip3; i3 += ip2)
		    {
		      i3rev = i2rev + i3 - i2 ;
		      SWAP ( data[i3],   data[i3rev] ) ;
		      SWAP ( data[i3+1], data[i3rev+1] ) ;
		    }
		}
	    }
	  ibit = ip2 >> 1 ;
	  while ( ibit >= ip1 && i2rev > ibit )
	    {
	      i2rev -= ibit ;
	      ibit >>= 1 ;
	    }
	  i2rev += ibit ;
	}
      ifp1 = ip1 ;

      /* the Danielson-Lanczos section */

      while ( ifp1 < ip2 )
	{
	  ifp2 = ifp1 << 1;

	  /* Initialize for the Trig recurrence */

	  theta = iSign * 2 * PI / (ifp2/ip1) ;
	  wtemp = sin ( 0.5 * theta ) ;
	  wpr = -2.0 * wtemp*wtemp ;
	  wpi = sin ( theta ) ;
	  wr = 1.0 ;
	  wi = 0.0 ;
	  for ( i3 = 1 ; i3 <= ifp1 ; i3 += ip1 )
	    {
	      for ( i1 = i3 ; i1 <= i3+ip1-2 ; i1 += 2 )
		{
		  for ( i2 = i1 ; i2 <= ip3 ; i2 += ifp2 )
		    {
		      /* Danielson-Lanczos formula: */

		      k1 = i2 ;
		      k2 = k1 + ifp1 ;
		      tempr = (float) wr * data[k2]   - (float) wi * data[k2+1] ;
		      tempi = (float) wr * data[k2+1] + (float) wi * data[k2]   ;
		      data[k2]   = data[k1]   - tempr ;
		      data[k2+1] = data[k1+1] - tempi ;
		      data[k1]   += tempr ;
		      data[k1+1] += tempi ;
		    }
		}
	      /* trig recurrence */

	      wr = (wtemp=wr)*wpr - wi*wpi + wr ;
	      wi = wi*wpr + wtemp*wpi + wi ;
	    }
	  ifp1 = ifp2 ;
	}
      nprev *= n ;
    }
  /* normalizing.  dm added this code. */

  if ( iSign == -1 )
    {
      printf ( " normalizing  " ) ;

      for ( i1 = 1 ; i1 < 2*ntot+1 ; i1++ )
	data[i1] /= ntot ;
    }
  printf ( " done!\n" ) ;

}


/*----------------------------------------------------------------------
  dfourn --- double precision version of fourn

    Replaces data by its ndim-dimensional discrete Fourier transform,
    if iSign is input as 1.  nn[1..ndim] is an integer array containing
    the lengths of each dimension (number of complex values), which MUST
    all be powers of two.  data is a real array of length twice the product
    of these lengths, in which the data are stored in a multi-dimensional
    complex array: real and imaginary parts of each element are in concecutive 
    locations, and the rightmost index of the array increases most rapidly as
    one proceeds along data.  For a two-dimensional array, this is equivalent
    to storing the array by rows.  If iSign is input at -1, data is replaced 
    by its inverse transform.

    The normalization is done by dividing the output of the inverse FT (iSign=-1)
    by the product of the lengths of all dimensions. 

    Does a multi-dimensional fast fourier transform of a complex function.

    double *data     pointer to the address one before the start of the data
                     data[1] is the first real, data[2] the first imag
                     data[3] the second real,  etc.
    int ndim         the number of dimensions.
    long nn[]        the length of each dimension, again, starts at 1.
    int iSign        1 for a forward transform -1 for an inverse transform
-------------------------------------------------------------------------*/
void dfourn ( double *data, int *nn , int ndim, int iSign )
{
  int idim ;
  signed int i1, i2, i3, i2rev, i3rev, ip1, ip2, ip3, ifp1, ifp2 ;
  signed int ibit, k1, k2, n, nprev, nrem, ntot ;
  double temp, tempi, tempr ;
  double theta, wi, wpi, wpr, wr, wtemp ;

  /* Compute total number of complex values */

  ntot = 1 ;
  for ( idim = 1 ; idim <= ndim ; idim++ )
    {
      ntot *= nn[idim] ;
    }

  printf (  "dfourn: %d-D FFT dimension = %d", ndim, nn[1] ) ;
  for ( i1 = 1 ; i1 < ndim ; i1++ )
     printf (  "X %d", nn[i1+1] ) ;

  /* Main loop over the dimensions */

  nprev = 1 ;
  for ( idim = ndim ; idim > 0 ; idim-- )
    {
      n = nn[idim] ;
      nrem = ntot / (n*nprev) ;
      ip1 = nprev << 1 ;
      ip2 = ip1*n ;
      ip3 = ip2 * nrem ;
      i2rev= 1 ;

      /* This is the bit reversal section of the routine */

      for ( i2=1 ; i2<=ip2; i2+=ip1 )
	{
	  if ( i2 < i2rev )
	    {
	      for ( i1 = i2 ; i1 <= i2 + ip1 - 2 ; i1 += 2 )
		{
		  for ( i3 = i1 ; i3 <= ip3; i3 += ip2)
		    {
		      i3rev = i2rev + i3 - i2 ;
		      SWAP ( data[i3],   data[i3rev] ) ;
		      SWAP ( data[i3+1], data[i3rev+1] ) ;
		    }
		}
	    }
	  ibit = ip2 >> 1 ;
	  while ( ibit >= ip1 && i2rev > ibit )
	    {
	      i2rev -= ibit ;
	      ibit >>= 1 ;
	    }
	  i2rev += ibit ;
	}
      ifp1 = ip1 ;

      /* the Danielson-Lanczos section */

      while ( ifp1 < ip2 )
	{
	  ifp2 = ifp1 << 1;

	  /* Initialize for the Trig recurrence */

	  theta = iSign * 2 * PI / (ifp2/ip1) ;
	  wtemp = sin ( 0.5 * theta ) ;
	  wpr = -2.0 * wtemp*wtemp ;
	  wpi = sin ( theta ) ;
	  wr = 1.0 ;
	  wi = 0.0 ;
	  for ( i3 = 1 ; i3 <= ifp1 ; i3 += ip1 )
	    {
	      for ( i1 = i3 ; i1 <= i3+ip1-2 ; i1 += 2 )
		{
		  for ( i2 = i1 ; i2 <= ip3 ; i2 += ifp2 )
		    {
		      /* Danielson-Lanczos formula: */

		      k1 = i2 ;
		      k2 = k1 + ifp1 ;
		      tempr = (double) wr * data[k2]   - (double) wi * data[k2+1] ;
		      tempi = (double) wr * data[k2+1] + (double) wi * data[k2]   ;
		      data[k2]   = data[k1]   - tempr ;
		      data[k2+1] = data[k1+1] - tempi ;
		      data[k1]   += tempr ;
		      data[k1+1] += tempi ;
		    }
		}
	      /* trig recurrence */

	      wr = (wtemp=wr)*wpr - wi*wpi + wr ;
	      wi = wi*wpr + wtemp*wpi + wi ;
	    }
	  ifp1 = ifp2 ;
	}
      nprev *= n ;
    }
  /* normalizing */

  if ( iSign == -1 )
    {
      printf (  " normalizing  " ) ;

      for ( i1 = 1 ; i1 < 2*ntot+1 ; i1++ )
	data[i1] /= ntot ;
    }
  printf (  " done!\n" ) ;

}


/*----------------------------------------
  sinc(x) =  sin(x)/x 

  The numerical approximation near the
  origin should be good to about a part 
  in 10^20 (i.e, also good for double prec).
  ----------------------------------------*/
float sinc ( float fX )
{
  float fZ, fXsq ;

  fXsq = PI*PI*fX*fX ;
  if ( ( fX > -1.e-3 ) && ( fX < 1.e-3 ) )
    fZ = 1. - (fXsq/6.)*(1. - (fXsq/20.)*(1. - (fXsq/42.)*(1. - fXsq/72.))) ;
  else
    fZ = sin ( PI * fX ) / (PI*fX) ;

  return ( fZ ) ;
}


 
/**********************************************************
  returns (n-1) where n is the index of refraction of air, given the
  wavelength in microns.
 
  This is from Allen, so may not be very accurate
 **********************************************************/
float n_dryAir( float fWave )
{
  double dW, dN ;
 
  dW = 1./(fWave * fWave ) ;
  dN = 64.328 + 29498.1/(146.-dW) + 255.4/(41.-dW) ;
  dN = 1.e-6*dN ;
 
  return ( dN ) ;
}
