/*

  FDL simulation modeled after the NPOI.

  This is a very simple model.

  The delay line is assumed to work perfectly, with the exception of a
  time delay between receiving and implementing a command.  The FDL
  reads its interface at 800 Hz.  It takes only one command -- change
  velocity by dV.  This command takes 100 uSec to be implemented and
  runs the FDL at a constant velocity until it receives the next
  command.

  all times are in Seconds
  ================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include "memory.h"

#include "fdl.hpp"

static int bVerbose = 0 ; 

fdl::fdl ( float fStroke, float fDelay0, float fServoTimeDelay )
{
  fLastTime = 0 ;
  fTimeDelay = fServoTimeDelay ;
  iLast = 0 ;
  iBufferSize = 10000 ;
  pfTime  = alloc1df( iBufferSize ) ;
  pfDelay = alloc1df( iBufferSize ) ;

  fStrokeTime = 0.002 ;
  fStrokeAmp  = fStroke ;

  /* fTime0 and fDeltaTime are used to convert the time, in seconds
     to record number */

  fTime0 = fTimeDelay ;
  fDeltaTime = 0.00125 ;
  fVelocity  = 0. ;
  pfTime[0]  = fTime0 ;
  pfDelay[0] = fDelay0 ;

  printf ( "fdl: initialization complete for stroke = %f\n", fStroke ) ;
}

/** ==============================
    fdl::move

    Change the delay line velocity 
    units are microns per second
    ==============================*/
void fdl::move( float fTime, float fV )
{
  int iRec ;
  if ( bVerbose ) 
    printf ( "set FDL  cmd time =%9.6f  deltaV = %5.1f\n", fTime, fV/1000. ) ;

  /* Find where this point belongs in the buffer.  Beware roundoff.  */

  iRec = int( (fTime - fTime0)/fDeltaTime) + 1 ;

  if ( iRec <= iLast )
    {
      printf ( " FDL: move comand arrives too late!!\n" ) ;
      printf ( "    Req time = %9.6f\n",   fTime ) ;
      printf ( "    Last Record is %4d\n", iLast ) ;
      printf ( "    for time = %9.6f\n",   fTime0 + iRec*fDeltaTime ) ;
      exit(1) ;
    }
  else if ( iRec > iBufferSize )
    {
      printf ( " FDL buffer full!!\n" ) ;
      exit(1) ;
    }
  while ( iLast < iRec )
    {
      iLast++ ;
      pfTime[iLast]  = pfTime[iLast-1]  + fDeltaTime ;
      pfDelay[iLast] = pfDelay[iLast-1] + fVelocity * fDeltaTime ;
      if ( bVerbose )
	printf ( "set FDL i=%4d  t=%6.3f  pos=%6.3f  V= %6.3f\n",
		 iLast, pfTime[iLast], 0.001*pfDelay[iLast], 0.001*fVelocity ) ;
    }
  fVelocity = fV;
}

/** =====================================================
    readMetrology

    return the delay line position at the requested time
    delay is the FDL cart position plus the stroke
    =====================================================*/
float fdl::readMetrology ( double dfTime )
{
  int iRec ;
  float fRec ;
  float fStroke ;

  /* work out the stroke offset.  fStroke runs from 0 to 1 during the stroke */

  fStroke = (float)(dfTime /(double)fStrokeTime) ;
  fStroke -= (int)fStroke ;
  if ( fStroke < 0.5 )
    fStroke = fStrokeAmp*( -0.5 + 2.*fStroke ) ;
  else
    fStroke = fStrokeAmp*(  1.5 - 2.*fStroke ) ;


  fRec = (dfTime - (double)fTime0)/fDeltaTime ;
  iRec = (int)fRec ;
  fRec -= iRec ;

  while ( iLast < iRec+1 )
    {
      iLast++ ;
      pfTime[iLast]  = pfTime[iLast-1]  + fDeltaTime ;
      pfDelay[iLast] = pfDelay[iLast-1] + fVelocity * fDeltaTime ;
      if ( bVerbose )
	printf ( "readMetrology: adding record %3d for time %6.4f, delay = %7.3f\n",
		 iLast, pfTime[iLast], pfDelay[iLast] ) ;
    }
  /* return linear interpolation of the delay between stored times. */

  return ( fStroke + pfDelay[iRec] + fRec*(pfDelay[iRec+1]-pfDelay[iRec]) ) ;
}


/** =================================================
    servo

    Implement a servo.  For now this is a simple
    PID servo with only the proportional term.

    It also has a time delay.  The unit of time is seconds.

    fTime is the time corresponding to the END of data taking.  So the
    data integration ends no sooner than fTime.  For the Huggins
    Fringe boards, the data comes off the board 4 ms after the end of
    the integration (at fTime + 3) and the processing is finished no
    sooner than 6 ms after the integration ends (fTime + 4).  The real
    system combines data from several integrations so the time delay
    is even longer.
    =================================================*/
float fdl::servo ( float fTime, float fDelay )
{
  float fTimeDelay = 0.002 ;
  float fGain =  30.0 ;
  float fdV ;

  /* calculate the servo response */

  fdV = fGain * fDelay  ;

  /* and apply it to the delay line */

  if ( bVerbose )
    printf ( "fdl::move changing fdl velocity to %9.3f um/sec at time %6.3f\n",
	     fVelocity+fdV, fTime+fTimeDelay ) ;

  move( fTime+fTimeDelay, fdV ) ;
  return ( fdV ) ;
}

