/*
  fringeMath.cpp

  All the routines necessary for calculating the fringe positions from
  the raw data.

  David Mozurkewich -- 21 February 2002
  ================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "dmmath.h"
#include "memory.h"
#include "fringeMath.hpp"

int bVerbose = 0 ;

fringeMath::fringeMath ( int iB, int iF, int iW, float *pfW )
{
  int i ;

  iNumBins  = iB ;
  iNumWaves = iW ;
  iMaxFreq  = iF+1 ;
  iNumDelays = 61 ;
  fDeltaD = 0.25 ;

  pfWave     = alloc1df( iW ) ;
  pfX        = alloc2df( iMaxFreq, iW ) ;
  pfY        = alloc2df( iMaxFreq, iW ) ;
  pfN        = alloc1df( iW ) ;
  pfGD       = alloc2df( iMaxFreq, iNumDelays ) ;
  pfMaxAmp   = alloc1df( iMaxFreq) ;
  pfGroupDelay = alloc1df( iMaxFreq) ;


  for ( i = 0 ; i < iW ; i++ )
    pfWave[i] = pfW[i] ;

  fDelay0 = -0.5*fDeltaD*float(iNumDelays) ;

  printf ( "fringeMath:  initialization complete\n" ) ;
} ;

/** =================================================
    displayData

    =================================================*/
void fringeMath::displayData ( float fTime, float fDelay )
{
  static FILE *pFile=NULL ;

  if ( pFile==NULL )
    pFile = fopen ( "dataFile", "w" ) ;

  printf (        " T= %6.3f   piston error = %7.3f\n", fTime, fDelay ) ;
  fprintf( pFile, " T= %6.3f   piston error = %7.3f\n", fTime, fDelay ) ;
  fflush ( pFile ) ;
}

/** =================================================
    calcXYN

    Calculate N as well as X and Y for all fringe frequencies.
    =================================================*/
float fringeMath::calcXYN(int **piData )
{
  float fTotal = 0. ;
  int iW, i, k ;
  for ( iW = 0 ; iW < iNumWaves ; iW++ )
    {
      pfN[iW] = 0 ;
      for ( i = 0 ; i < iNumBins ; i++ )
	pfN[iW] += piData[iW][i] ;
      fTotal += pfN[iW] ;
      for ( k = 0 ; k < iMaxFreq ; k++ )
	{
	  pfX[k][iW] = 0 ;
	  pfY[k][iW] = 0 ;
	  for ( i = 0 ; i < iNumBins ; i++ )
	    {
	      pfX[k][iW] += piData[iW][i] * cos ( 2.*PI*i*k/iNumBins) ;
	      pfY[k][iW] += piData[iW][i] * sin ( 2.*PI*i*k/iNumBins) ;
	    }
	}
    }
  return ( fTotal) ;
}


/** =================================================
    calcGD

    Calculate the group delay for the fringe at frequency iF.

    This is the amplitude as a function of
    delay used for finding the fringe tracking error signal.

    After the group delay is calculated, return the delay
    corresponding to the peak value.

    return value is the delay of the peak.
  =================================================*/
float fringeMath::calcGD(int iF, float *pfSNR )
{
  int i, j, iMax ;
  float fImag, fReal ;
  float fArg, fMax, fDelay ;
  fMax = 0. ;
  iMax = 0 ;
  for ( i = 0 ; i < iNumDelays ; i++ )
    {
      fDelay = fDelay0 + fDeltaD * float(i) ;
      fReal = 0. ;
      fImag = 0. ;
      for ( j = 0 ; j < iNumWaves ; j++ )
	{
	  fArg = 2.*PI*fDelay/pfWave[j] ;
	  fReal += pfX[iF][j]*cos(fArg) - pfY[iF][j]*sin(fArg) ;
	  fImag += pfX[iF][j]*sin(fArg) + pfY[iF][j]*cos(fArg) ;
	}
      pfGD[iF][i] = fReal*fReal + fImag*fImag ;
      if ( pfGD[iF][i] > fMax )
	{
	  iMax = i ;
	  fMax = pfGD[iF][i] ;
	}
    }
  *pfSNR = fMax ; 
  pfMaxAmp[iF] = fMax ;
  pfGroupDelay[iF] = fDelay0 + fDeltaD*(float)iMax  ;

  return ( pfGroupDelay[iF] ) ;
}  


int fringeMath::getSpectrum( int iF, float **pfX, float **pfY )
{
  int i ;
  if ( bVerbose )
    printf ( "getSpectrum:  begins\n") ;
  if (( *pfX == NULL )&&(*pfY == NULL) )
    {
      printf ( "   allocating memory for arrays\n" ) ;
      *pfX = alloc1df( iNumDelays ) ;
      *pfY = alloc1df( iNumDelays ) ;
    }
  else if ( ( *pfX == NULL) ||(*pfY == NULL) )
    {
      printf ( " fringeMath::getSpectrum: ERROR!\n" ) ;
      printf ( "             only one pointer initialized\n" ) ;
      exit(1) ;
    }
  if ( bVerbose )
    printf ( "  ready to copy values to output\n" ) ;
  for ( i = 0; i < iNumDelays ; i++ )
    {
      (*pfX)[i] = fDelay0 + fDeltaD * float(i) ;
      (*pfY)[i] = pfGD[iF][i] ;
    }
  if ( bVerbose )
    printf ( "getSpectrum:  done\n" ) ;
  return ( iNumDelays ) ;
}
