/*

  Fast Steering Mirror (FSM) simulation modeled after how the NPOI
  should work.

  This is a very simple model.

  The FSM is assumed to work perfectly, with the exception of a time
  delay between receiving and implementing a command.  The FSM reads
  its interface at 500 Hz.  It takes only one command -- change
  voltages by dVtip and dVtilt.  The nominal voltage is 500V and the
  rails are at 0 and 1000 Volts.  It take 2 ms to implement a move.

  all times are in Seconds
  ================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include "memory.h"
#include "dmmath.h"
#include "fsm.hpp"

static int bVerbose = 0 ; 

fsm::fsm ( float fTheta )
{
  fUpRail   = 10. ;
  fDownRail = -10. ;
  fCenter = 0.5*(fUpRail + fDownRail) ;
  fRange = 0.5 * ( fUpRail - fDownRail) ;

  fTimeDelay = 0.001 ;
  iBufferSize = 10000 ;

  fTime0 = 0.0005 ;
  fDeltaTime = 0.001 ;

  pfTime = alloc1df( iBufferSize ) ;
  pfTip  = alloc1df( iBufferSize ) ;
  pfTilt = alloc1df( iBufferSize ) ;

  iNumRecs = 0 ;        /* number of records already stored (with values) */
  pfTime[0]  = fTime0 ;
  pfTip[0]   = fCenter ;
  pfTilt[0]  = fCenter ;

  printf ( "fsm: initialization complete\n" ) ;
}

/** ==============================
    fsm::move

    Change the fsm position
    units are seconds and microradians ( should be in volts )

    For now, the FSM is assumed to be perfect.  It moves to where it
    was told to move in fDeltaTime and stays there until the next time
    is it asked to move.

    The motion starts the next full clock tick after fTimeDelay
    seconds after the command was received.

    The time sent to the move command is the time the error signal was
    calculated.  The value sent to this routine should have all time
    delays associated with taking the data and with converting the
    data into an error signal.  The time delays due to the mechanics
    should be applied in this function.
    ==============================*/
void fsm::move( float fTime, float fDeltaTip, float fDeltaTilt )
{
  int iRec ;
  if ( bVerbose ) 
    printf ( "set FSM  cmd time =%9.6f  deltatip-tilt = %6.2f %6.2f\n",
	     fTime, fDeltaTip, fDeltaTilt ) ;

  /* Find where this point belongs in the buffer.  The command is
     delayed by fTimeDelay before it is applied to the mirror. */

  iRec = int( (fTime + fTimeDelay - fTime0)/fDeltaTime) + 1 ;

  if ( iRec <= iNumRecs )
    {
      printf ( " FSM: move comand arrives too late!!\n" ) ;
      printf ( "    Req time = %6.3f is in record %3d\n", fTime, iRec ) ;
      printf ( "    records filled up to number %d\n", iNumRecs ) ;
      return ;
    }
  else if ( iRec > iBufferSize )
    {
      printf ( " FSM buffer full!!\n" ) ;
      exit(1) ;
    }
  while ( iNumRecs < iRec )
    {
      iNumRecs++ ;
      pfTime[iNumRecs]  = pfTime[iNumRecs-1] + fDeltaTime ;
      pfTip[iNumRecs]   =  pfTip[iNumRecs-1] ;
      pfTilt[iNumRecs]  =  pfTilt[iNumRecs-1] ;
      if ( bVerbose )
	printf ( "set FSM i=%4d  t=%6.3f tip=%6.3f  tilt= %6.3f\n",
		 iNumRecs, pfTime[iNumRecs], pfTip[iNumRecs], pfTilt[iNumRecs] ) ;
    }
  iNumRecs++ ;
  pfTime[iNumRecs] =  pfTime[iNumRecs-1] + fDeltaTime ;
  pfTip[iNumRecs]  =  pfTip[iNumRecs-1]  + fDeltaTip ;
  pfTilt[iNumRecs] =  pfTilt[iNumRecs-1] + fDeltaTilt ;

  /*
  pfTip[iNumRecs]   =  MAX(MIN(pfTip[iNumRecs-1] + fDeltaTip, fUpRail), fDownRail) ;
  pfTilt[iNumRecs]  =  MAX(MIN(pfTilt[iNumRecs-1]+ fDeltaTilt,fUpRail), fDownRail) ;
  */

}

/** =====================================================
    getTip

    return the FSM tip position at the requested time
    =====================================================*/
float fsm::getTip ( float fTime )
{
  int iRec ;
  float fRec ;

  fRec = (fTime - fTime0)/fDeltaTime ;
  iRec = (int)fRec ;
  fRec -= iRec ;

  while ( iNumRecs < iRec+1 )
    {
      iNumRecs++ ;
      pfTime[iNumRecs] = pfTime[iNumRecs-1]  + fDeltaTime ;
      pfTip[iNumRecs]  = pfTip[iNumRecs-1]  ;
      if ( bVerbose )
	printf ( "getTip: adding record %3d for time %6.4f, tip = %7.3f\n",
		 iNumRecs, pfTime[iNumRecs], pfTip[iNumRecs] ) ;
    }
  /* return linear interpolation of the tip between stored times. */

  return (  pfTip[iRec] + fRec*(pfTip[iRec+1]-pfTip[iRec]) ) ;
}
/** =====================================================
    getTilt

    return the FSM tilt position at the requested time
    =====================================================*/
float fsm::getTilt ( float fTime )
{
  int iRec ;
  float fRec ;

  fRec = (fTime - fTime0)/fDeltaTime ;
  iRec = (int)fRec ;
  fRec -= iRec ;

  while ( iNumRecs < iRec+1 )
    {
      iNumRecs++ ;
      pfTime[iNumRecs] = pfTime[iNumRecs-1]  + fDeltaTime ;
      pfTilt[iNumRecs]  = pfTilt[iNumRecs-1]  ;
      if ( bVerbose )
	printf ( "getTilt: adding record %3d for time %6.4f, tilt = %7.3f\n",
		 iNumRecs, pfTime[iNumRecs], pfTilt[iNumRecs] ) ;
    }
  /* return linear interpolation of the tilt between stored times. */

  return (  pfTilt[iRec] + fRec*(pfTilt[iRec+1]-pfTilt[iRec]) ) ;
}


/** =================================================
    servo

    Implement a servo.  For now this is a simple
    PID servo with only the proportional term.

    It also has a time delay.  The unit of time is seconds.

    fTime is the time when the error signal arrives at the FSM.  The
    mirror starts moving fTimeDelay later and continues for fDeltaTime
    The time delay is applied in the move command.

    =================================================*/
void fsm::servoTipTilt ( float fTime, float fErrorX, float fErrorY )
{
  float fGain = 0.3 ;

  float fVx, fVy ;
  /* calculate the servo response */

  fVx = fGain * fErrorX  ;
  fVy = fGain * fErrorY  ;

  /* and apply it to the FSM */

  if ( bVerbose )
    {
      printf ( "fsm::servo:                                           ") ;
      printf ( "              servo gain = %6.3f\n", fGain );
      printf ( "fsm::servo:                                           ") ;
      printf ( "              errorX = %9.3f  errorY = %9.3f\n", fErrorX, fErrorY );
      printf ( "fsm::servo:                                           ") ;
      printf ( "              changing fsm tip/tilt by %7.3f,%7.3f uRad at time %6.3f\n",
	       fVx, fVy, fTime ) ;
    }

  move( fTime, fVx, fVy ) ;
}
